/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGINTERESTOBJECT_H
#define SKGINTERESTOBJECT_H
/** @file
 * This file defines classes SKGInterestObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgobjectbase.h"
#include "skgaccountobject.h"
#include "skgerror.h"
#include "skgbankmodeler_export.h"

class SKGInterestObject;

/**
 * This class is interest option for an account
 */
class SKGBANKMODELER_EXPORT SKGInterestObject : public SKGObjectBase
{
    Q_OBJECT;
    /**
     * This enumerate defines how to compute value date
     */
    Q_ENUMS(ValueDateMode);
public:
    /**
     * This enumerate defines how to compute value date
     */
    enum ValueDateMode {FIFTEEN = 0,
                        J0,
                        J1,
                        J2,
                        J3,
                        J4,
                        J5
                       };

    /**
     * This enumerate defines how to compute interest
     */
    enum InterestMode {FIFTEEN24 = 0,
                       DAYS360,
                       DAYS365
                      };

    /**
     * Constructor
     * @param iDocument the document containing the object
     * @param iID the identifier in @p iTable of the object
     */
    explicit SKGInterestObject(SKGDocument* iDocument = NULL, int iID = 0);

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    SKGInterestObject(const SKGInterestObject& iObject);

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    SKGInterestObject(const SKGObjectBase& iObject);

    /**
     * Operator affectation
     * @param iObject the object to copy
     */
    virtual const SKGInterestObject& operator= (const SKGObjectBase& iObject);

    /**
     * Destructor
     */
    virtual ~SKGInterestObject();

    /**
     * Get the parent account
     * @param oAccount the parent account
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getAccount(SKGAccountObject& oAccount) const;

    /**
     * Set the parent account
     * @param iAccount the parent account
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setAccount(const SKGAccountObject& iAccount);

    /**
     * Set the quantity for the date of this unit
     * @param iValue the quantity
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setRate(double iValue);

    /**
     * Get the quantity for the date of this unit
     * @return the quantity
     */
    virtual double getRate() const;

    /**
     * Set date of this value
     * @param iDate the date
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setDate(const QDate& iDate);

    /**
     * Get date of this value
     * @return the date
     */
    virtual QDate getDate() const;

    /**
     * Set income value date mode
     * @param iMode the mode
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setIncomeValueDateMode(SKGInterestObject::ValueDateMode iMode);

    /**
     * Get income value date mode
     * @return the income value date mode
     */
    virtual SKGInterestObject::ValueDateMode getIncomeValueDateMode() const;

    /**
     * Set expenditue value date mode
     * @param iMode the mode
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setExpenditueValueDateMode(SKGInterestObject::ValueDateMode iMode);

    /**
     * Get expenditue value date mode
     * @return the expenditue value date mode
     */
    virtual SKGInterestObject::ValueDateMode getExpenditueValueDateMode() const;

    /**
     * Set interest computation mode
     * @param iMode the mode
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setInterestComputationMode(SKGInterestObject::InterestMode iMode);

    /**
     * Get interest computation mode
     * @return the interest computation mode
     */
    virtual SKGInterestObject::InterestMode getInterestComputationMode() const;

protected:
    /**
     * Get where clause needed to identify objects.
     * For this class, the whereclause is based on date + unit
     * @return the where clause
     */
    virtual QString getWhereclauseId() const;
};
/**
 * Declare the class
 */
Q_DECLARE_TYPEINFO(SKGInterestObject, Q_MOVABLE_TYPE);
#endif
