/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a plugin for delete operation.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgdeleteplugin.h"
#include "skgtraces.h"
#include "skgerror.h"
#include "skgobjectbase.h"
#include "skgtransactionmng.h"
#include "skgmainpanel.h"

#include <QtGui/QWidget>

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <kaboutdata.h>
#include <kmessagebox.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGDeletePluginFactory, registerPlugin<SKGDeletePlugin>();)
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGDeletePluginFactory("skg_delete", "skg_delete"))

SKGDeletePlugin::SKGDeletePlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
    SKGTRACEIN(10, "SKGDeletePlugin::SKGDeletePlugin");
}

SKGDeletePlugin::~SKGDeletePlugin()
{
    SKGTRACEIN(10, "SKGDeletePlugin::~SKGDeletePlugin");
    m_currentDocument = NULL;
    m_deleteAction = NULL;
}

bool SKGDeletePlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
    SKGTRACEIN(10, "SKGDeletePlugin::setupActions");
    Q_UNUSED(iArgument);

    m_currentDocument = iDocument;

    setComponentData(KGlobal::mainComponent());
    setXMLFile("../skg_delete/skg_delete.rc");

    //Menu
    m_deleteAction = new KAction(KIcon("edit-delete"), i18nc("Verb, delete an item", "Delete"), this);
    connect(m_deleteAction, SIGNAL(triggered(bool)), this, SLOT(actionDelete()));
    actionCollection()->addAction(QLatin1String("edit_delete"), m_deleteAction);
    m_deleteAction->setShortcut(Qt::Key_Delete);

    if(SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("edit_delete", m_deleteAction);
    return true;
}

void SKGDeletePlugin::refresh()
{
    SKGTRACEIN(10, "SKGDeletePlugin::refresh");

    if(m_deleteAction && SKGMainPanel::getMainPanel()) m_deleteAction->setEnabled(SKGMainPanel::getMainPanel()->getSelectedObjects().count() > 0);
}

QString SKGDeletePlugin::title() const
{
    return i18nc("Verb, delete an item", "Delete");
}

int SKGDeletePlugin::getOrder() const
{
    return 5;
}

void SKGDeletePlugin::actionDelete()
{
    SKGError err;
    SKGTRACEINRC(10, "SKGDeletePlugin::actionDelete", err);
    if(SKGMainPanel::getMainPanel() && m_currentDocument) {
        {
            SKGObjectBase::SKGListSKGObjectBase selection = SKGMainPanel::getMainPanel()->getSelectedObjects();
            int nb = selection.count();
            SKGBEGINPROGRESSTRANSACTION(*m_currentDocument, i18nc("Verb, delete an item", "Delete"), err, nb);
            for(int i = 0; !err && i < nb; ++i) {
                err = selection[i].remove();
                if(err && err.getReturnCode() == ERR_FORCEABLE) {
                    QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));
                    int rc = KMessageBox::questionYesNo(SKGMainPanel::getMainPanel(),
                                                        err.getFullMessage() % '\n' % i18nc("Question",  "Do you want to force the deletion ?"),
                                                        i18nc("Question",  "Do you want to force the deletion ?"),
                                                        KStandardGuiItem::yes(), KStandardGuiItem::no(),
                                                        "forcedelete"
                                                       );
                    QApplication::restoreOverrideCursor();
                    if(rc == KMessageBox::Yes) err = selection[i].remove(true, true);
                }
                if(!err) err = m_currentDocument->stepForward(i + 1);
            }
        }

        //
        KMessageBox::enableMessage("forcedelete");

        //status bar
        if(!err) err = SKGError(0, i18nc("Successful message after an user action", "Objects deleted."));
    }

    //Display error
    SKGMainPanel::displayErrorMessage(err);
}

#include "skgdeleteplugin.moc"
