/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a plugin for debug.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgdebugpluginwidget.h"
#include "skgmainpanel.h"
#include "skgdocument.h"
#include "skgtransactionmng.h"
#include "skgservices.h"
#include "skgtraces.h"

#include <kross/core/manager.h>
#include <kross/core/action.h>

#include <QDomDocument>

SKGDebugPluginWidget::SKGDebugPluginWidget(SKGDocument* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::SKGDebugPluginWidget");
    if(!iDocument) return;

    ui.setupUi(this);

    //Set icons
    ui.kSQLPushButton->setIcon(KIcon("system-run"));
    ui.kSQLTransactionPushButton->setIcon(KIcon("system-run"));
    ui.kRefreshViewsAndIndexes->setIcon(KIcon("view-refresh"));

    //Fill combo box
    ui.kExplainCmb->addItem(KIcon("system-run"), i18nc("Execute an SQL query", "Execute"));
    ui.kExplainCmb->addItem(KIcon("help-hint"), i18nc("Explain an SQL query", "Explain"));
    ui.kExplainCmb->addItem(KIcon("games-hint"), i18nc("Explain the SQL query plan", "Explain query plan"));
    foreach(const QString & s, Kross::Manager::self().interpreters()) {
        ui.kExplainCmb->addItem(KIcon("media-playback-start"), i18nc("Execute script", "Execute script [%1]", s), s);
    }
    ui.kInput->setVisible(false);

    //Set level trace
    ui.kTraceLevel->setValue(SKGTraces::SKGLevelTrace);

    //Set profiling mode
    ui.kEnableProfilingChk->setCheckState(SKGTraces::SKGPerfo ? Qt::Checked : Qt::Unchecked);

    //Init debug page
    QStringList tables;
    ui.kSQLInput->addItem("SELECT * FROM sqlite_master;");
    iDocument->getDistinctValues("sqlite_master", "name", "type in ('table', 'view')", tables);
    int nb = tables.count();
    for(int i = 0; i < nb; ++i) {
        ui.kSQLInput->addItem("SELECT * FROM " % tables[i] % ';');
    }
    ui.kSQLInput->addItem("ANALYZE;");
    ui.kSQLInput->addItem("PRAGMA integrity_check;");
    for(int i = 0; i < nb; ++i) {
        ui.kSQLInput->addItem("PRAGMA table_info(" % tables[i] % ");");
        ui.kSQLInput->addItem("PRAGMA index_list(" % tables[i] % ");");
    }

    iDocument->getDistinctValues("sqlite_master", "name", "type='index'", tables);
    nb = tables.count();
    for(int i = 0; i < nb; ++i) {
        ui.kSQLInput->addItem("PRAGMA index_info(" % tables[i] % ");");
    }
}

SKGDebugPluginWidget::~SKGDebugPluginWidget()
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::~SKGDebugPluginWidget");
}

QString SKGDebugPluginWidget::getState()
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    root.setAttribute("explain", ui.kExplainCmb->currentIndex());
    root.setAttribute("enableProfiling", ui.kEnableProfilingChk->checkState() == Qt::Checked ? "Y" : "N");
    root.setAttribute("levelTraces", ui.kTraceLevel->value());
    root.setAttribute("sqlOrder", ui.kSQLInput->currentText());

    return doc.toString();
}

void SKGDebugPluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    QString explain = root.attribute("explain");
    QString enableProfiling = root.attribute("enableProfiling");
    QString levelTraces = root.attribute("levelTraces");
    QString sqlOrder = root.attribute("sqlOrder");
    QString sqlResult = root.attribute("sqlResult");

    if(!explain.isEmpty()) ui.kExplainCmb->setCurrentIndex(SKGServices::stringToInt(explain == "Y" ? "1" : explain));
    if(!enableProfiling.isEmpty()) ui.kEnableProfilingChk->setCheckState(enableProfiling == "Y" ? Qt::Checked : Qt::Unchecked);
    if(!levelTraces.isEmpty()) ui.kTraceLevel->setValue(SKGServices::stringToInt(levelTraces));
    ui.kSQLInput->setText(sqlOrder);
    ui.kSQLResult->setPlainText(sqlResult);
}

void SKGDebugPluginWidget::onExecuteSqlOrderInTransaction()
{
    onExecuteSqlOrder(true);
}

void SKGDebugPluginWidget::onExecuteSqlOrder(bool iInTransaction)
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::onExecuteSqlOrder");
    SKGError err;
    int exp = ui.kExplainCmb->currentIndex();
    if(exp > 2) {
        //Script execution
        ui.kSQLResult->clear();
        QString text = ui.kInput->toPlainText();
        Kross::Action action(this, "MyScript");
        action.setCode(text.toAscii());
        // skgresult.setText(skgdocument.getUniqueIdentifier())
        // skgerror=skgdocument.sendMessage("Hello")
        // skgerror=skgdocument.sendMessage("Hello")
        // skgmainpanel.closeAllOtherPages(skgmainpanel.currentPage())
        action.setInterpreter(ui.kExplainCmb->itemData(exp).toString());

        action.addObject(ui.kSQLResult, "skgresult");
        action.addObject(getDocument(), "skgdocument");
        //action.addObject(&err, "skgerror");
        action.addObject(SKGMainPanel::getMainPanel(), "skgmainpanel");

        // Finally execute the scripting code.
        //qScriptRegisterMetaType(action.engine(), SKGErrorToScriptValue, SKGErrorFromScriptValue);
        action.trigger();
    } else {
        //SQL execution
        QString text = ui.kSQLInput->currentText();
        if(exp == 1)  text = "EXPLAIN " % text;
        else if(exp == 2)  text = "EXPLAIN QUERY PLAN " % text;
        QString oResult;
        double time = SKGServices::getMicroTime();
        if(iInTransaction) {
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Display an SQL command from the debug plugin", "SQL command from debug plugin"), err);
            err = getDocument()->dumpSelectSqliteOrder(text, oResult);
        } else {
            QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
            err = getDocument()->dumpSelectSqliteOrder(text, oResult);
            QApplication::restoreOverrideCursor();
        }
        time = SKGServices::getMicroTime() - time;

        oResult += i18nc("Display the execution time needed by an SQL query", "\nExecution time: %1 ms", SKGServices::doubleToString(time));

        if(!err) {
            ui.kSQLResult->setPlainText(oResult);
        } else {
            ui.kSQLResult->setPlainText(err.getFullMessageWithHistorical());
        }
    }
}

void SKGDebugPluginWidget::onTraceLevelModified()
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::onTraceLevelModified");
    SKGTraces::SKGLevelTrace = ui.kTraceLevel->value();
}

void SKGDebugPluginWidget::onModeChanged()
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::onModeChanged");
    int exp = ui.kExplainCmb->currentIndex();
    ui.kInput->setVisible(exp > 2);
    ui.kSQLInput->setVisible(exp < 3);
}

void SKGDebugPluginWidget::onProfilingModeChanged()
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::onProfilingModeChanged");
    SKGTraces::SKGPerfo = (ui.kEnableProfilingChk->checkState() == Qt::Checked);
}

void SKGDebugPluginWidget::onRefreshViewsAndIndexes()
{
    SKGTRACEIN(10, "SKGDebugPluginWidget::onProfilingModeChanged");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    SKGError err;
    err = getDocument()->refreshViewsIndexesAndTriggers();
    if(err) {
        ui.kSQLResult->setPlainText(err.getFullMessageWithHistorical());
    }
    QApplication::restoreOverrideCursor();
}

#include "skgdebugpluginwidget.moc"
