/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to manage scheduled operations
 *
 * @author Stephane MANKOWSKI
 */
#include "skgscheduledplugin.h"
#include "skgscheduledpluginwidget.h"
#include "skgscheduledboardwidget.h"
#include "skgscheduled_settings.h"
#include "skgrecurrentoperationobject.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"
#include "skgoperationobject.h"
#include "skgdocumentbank.h"

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <kaboutdata.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGScheduledPluginFactory, registerPlugin<SKGScheduledPlugin>();)
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGScheduledPluginFactory("skrooge_scheduled", "skrooge_scheduled"))

SKGScheduledPlugin::SKGScheduledPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
    SKGTRACEIN(10, "SKGScheduledPlugin::SKGScheduledPlugin");
}

SKGScheduledPlugin::~SKGScheduledPlugin()
{
    SKGTRACEIN(10, "SKGScheduledPlugin::~SKGScheduledPlugin");
    m_currentBankDocument = NULL;
    m_scheduleOperationAction = NULL;
}

bool SKGScheduledPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
    SKGTRACEIN(10, "SKGScheduledPlugin::setupActions");
    Q_UNUSED(iArgument);
    m_currentBankDocument = qobject_cast<SKGDocumentBank*>(iDocument);
    if(m_currentBankDocument == NULL) return false;

    KComponentData data = SKGScheduledPluginFactory::componentData();
    const_cast<KAboutData*>(data.aboutData())->setProgramName(ki18n("%1").subs(KGlobal::mainComponent().aboutData()->programName()));
    setComponentData(data);
    setXMLFile("skrooge_scheduled.rc");

    //Create yours actions here
    m_scheduleOperationAction = new KAction(KIcon("skrooge_schedule"), i18nc("Verb, create a scheduled operation", "Schedule"), this);
    connect(m_scheduleOperationAction, SIGNAL(triggered(bool)), this, SLOT(actionScheduleOperation()));
    actionCollection()->addAction(QLatin1String("schedule_operation"), m_scheduleOperationAction);
    m_scheduleOperationAction->setShortcut(Qt::CTRL + Qt::Key_I);

    if(SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("schedule_operation", m_scheduleOperationAction);
    return true;
}

void SKGScheduledPlugin::refresh()
{
    SKGTRACEIN(10, "SKGScheduledPlugin::refresh");
    if(SKGMainPanel::getMainPanel()) {
        SKGObjectBase::SKGListSKGObjectBase selection = SKGMainPanel::getMainPanel()->getSelectedObjects();

        if(selection.count() > 0) {
            bool onOperation = (selection.at(0).getRealTable() == "operation" &&  selection.at(0).getTable() != "v_operation_consolidated");
            if(m_scheduleOperationAction) m_scheduleOperationAction->setEnabled(onOperation);
        } else {
            if(m_scheduleOperationAction) m_scheduleOperationAction->setEnabled(false);
        }

        //Automatic insert
        if(m_currentBankDocument && m_currentBankDocument->getDatabase() != NULL) {
            QString doc_id = m_currentBankDocument->getUniqueIdentifier();
            if(m_docUniqueIdentifier != doc_id && m_currentBankDocument->getParameter("SKG_DB_BANK_VERSION") >= "0.5") {
                m_docUniqueIdentifier = doc_id;

                SKGError err;
                //Read Setting
                bool check_on_open = skgscheduled_settings::check_on_open();

                if(check_on_open) {
                    SKGBEGINTRANSACTION(*m_currentBankDocument, i18nc("Noun, name of the user action", "Insert recurrent operations"), err);
                    int nbi = 0;
                    err = SKGRecurrentOperationObject::process(m_currentBankDocument, nbi);
                }
                //Display error
                if(SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
            }
        }
    }
}

void SKGScheduledPlugin::close()
{
    SKGTRACEIN(10, "SKGScheduledPlugin::close");
}

int SKGScheduledPlugin::getNbDashboardWidgets()
{
    return 1;
}

QString SKGScheduledPlugin::getDashboardWidgetTitle(int iIndex)
{
    if(iIndex == 0) return i18nc("Noun, the title of a section", "Scheduled operations");
    return "";
}

SKGWidget* SKGScheduledPlugin::getDashboardWidget(int iIndex)
{
    if(iIndex == 0) return new SKGScheduledBoardWidget(m_currentBankDocument);
    return NULL;
}

SKGTabPage* SKGScheduledPlugin::getWidget()
{
    SKGTRACEIN(10, "SKGScheduledPlugin::getWidget");
    return new SKGScheduledPluginWidget(m_currentBankDocument);
}

QWidget* SKGScheduledPlugin::getPreferenceWidget()
{
    SKGTRACEIN(10, "SKGScheduledPlugin::getPreferenceWidget");
    QWidget* widget = new QWidget();
    ui.setupUi(widget);
    return widget;
}

KConfigSkeleton* SKGScheduledPlugin::getPreferenceSkeleton()
{
    return skgscheduled_settings::self();
}

QString SKGScheduledPlugin::title() const
{
    return i18nc("Noun", "Scheduled operations");
}

QString SKGScheduledPlugin::icon() const
{
    return "chronometer";
}

QString SKGScheduledPlugin::toolTip() const
{
    return i18nc("Noun", "Operations scheduled management");
}

int SKGScheduledPlugin::getOrder() const
{
    return 20;
}

QStringList SKGScheduledPlugin::tips() const
{
    QStringList output;
    output.push_back(i18nc("Description of a tips", "<p>... you can schedule operations or templates.</p>"));
    return output;
}

bool SKGScheduledPlugin::isInContext() const
{
    return true;
}

bool SKGScheduledPlugin::isEnabled() const
{
    return true;
}

SKGError SKGScheduledPlugin::savePreferences() const
{
    SKGError err;
    if(m_currentBankDocument) {
        //Read Setting
        if(skgscheduled_settings::create_template()) {
            SKGObjectBase::SKGListSKGObjectBase recurrents;
            err = m_currentBankDocument->getObjects("v_recurrentoperation", "(select count(1) from operation where operation.id=rd_operation_id and t_template='N')=1", recurrents);
            int nb = recurrents.count();
            if(nb) {
                SKGBEGINPROGRESSTRANSACTION(*m_currentBankDocument, i18nc("Noun, name of the user action", "Conversion schedule"), err, nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Migration of existing schedule in template mode
                    SKGRecurrentOperationObject recOp = recurrents.at(i);
                    SKGOperationObject operationObj;

                    if(err.isSucceeded()) recOp.getParentOperation(operationObj);

                    SKGOperationObject operationObjOrig = operationObj;
                    if(err.isSucceeded()) err = operationObjOrig.duplicate(operationObj , operationObjOrig.getDate(), true);

                    if(err.isSucceeded()) err = recOp.setParentOperation(operationObj);
                    if(err.isSucceeded()) err = recOp.save();
                    if(err.isSucceeded()) err = recOp.load();

                    if(err.isSucceeded()) err = operationObjOrig.setAttribute("r_recurrentoperation_id", SKGServices::intToString(recOp.getID()));
                    if(err.isSucceeded()) err = operationObjOrig.save();

                    if(err.isSucceeded()) err = m_currentBankDocument->stepForward(i + 1);
                }
                if(err.isSucceeded()) m_currentBankDocument->sendMessage(i18nc("An information message",  "All scheduled operations have been converted in template"), true);
            }
        }
    }
    return err;
}

void SKGScheduledPlugin::actionScheduleOperation()
{
    SKGError err;
    SKGTRACEINRC(10, "SKGScheduledPlugin::actionScheduleOperation", err);

    //Get Selection
    if(SKGMainPanel::getMainPanel()) {
        SKGObjectBase::SKGListSKGObjectBase selection = SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb = selection.count();
        if(nb && m_currentBankDocument) {
            SKGBEGINPROGRESSTRANSACTION(*m_currentBankDocument, i18nc("Noun, name of the user action", "Operation schedule"), err, nb);
            for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                SKGOperationObject operationObj = selection.at(i);
                bool isTemplate = operationObj.isTemplate();

                SKGOperationObject operationObjOrig;
                if(!isTemplate && skgscheduled_settings::create_template()) {
                    //The selected operation is not a template and settings is set to create one
                    operationObjOrig = operationObj;
                    if(err.isSucceeded()) err = operationObjOrig.duplicate(operationObj , operationObjOrig.getDate(), true);
                    if(err.isSucceeded()) m_currentBankDocument->sendMessage(i18nc("An information message",  "A template has been created"), true);
                }

                SKGRecurrentOperationObject recOp;
                err = operationObj.addRecurrentOperation(recOp);
                if(err.isSucceeded()) err = recOp.warnEnabled(skgscheduled_settings::remind_me());
                if(err.isSucceeded()) err = recOp.setWarnDays(skgscheduled_settings::remind_me_days());
                if(err.isSucceeded()) err = recOp.autoWriteEnabled(skgscheduled_settings::auto_write());
                if(err.isSucceeded()) err = recOp.setAutoWriteDays(skgscheduled_settings::auto_write_days());
                if(err.isSucceeded()) err = recOp.timeLimit(skgscheduled_settings::nb_times());
                if(err.isSucceeded()) err = recOp.setTimeLimit(skgscheduled_settings::nb_times_val());
                if(err.isSucceeded()) err = recOp.setPeriodIncrement(skgscheduled_settings::once_every());
                if(err.isSucceeded()) err = recOp.setPeriodUnit((SKGRecurrentOperationObject::PeriodUnit) SKGServices::stringToInt(skgscheduled_settings::once_every_unit()));
                if(err.isSucceeded() && !isTemplate) err = recOp.setDate(recOp.getNextDate());
                if(err.isSucceeded()) err = recOp.save();
                if(!isTemplate && skgscheduled_settings::create_template()) {
                    if(err.isSucceeded()) err = recOp.load();
                    if(err.isSucceeded()) err = operationObjOrig.setAttribute("r_recurrentoperation_id", SKGServices::intToString(recOp.getID()));
                    if(err.isSucceeded()) err = operationObjOrig.save();
                }
                if(err.isSucceeded()) err = m_currentBankDocument->stepForward(i + 1);
            }
        }

        //status bar
        if(err.isSucceeded())  err = SKGError(0, i18nc("Successful message after an user action", "Operation scheduled."));
        else err.addError(ERR_FAIL, i18nc("Error message",  "Operation schedule failed"));

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage(err);
    }
}
#include "skgscheduledplugin.moc"
