/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin to generate report.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgreportpluginwidget.h"
#include "skgmainpanel.h"
#include "skgbankincludes.h"
#include "skgtraces.h"
#include "skgcolorbutton.h"

#include <klocale.h>
#include <kmenu.h>

#include <QtGui/QGraphicsScene>

/**
 * The size of the forecast
 */
static const int FORECASTMAX = 400;

SKGReportPluginWidget::SKGReportPluginWidget(SKGDocumentBank* iDocument, bool iMinimmumMode)
    : SKGTabPage(iDocument), m_nbLevelLines(0), m_nbLevelColumns(0)
{
    SKGTRACEIN(10, "SKGReportPluginWidget::SKGReportPluginWidget");
    if(!iDocument) return;

    m_timer.setSingleShot(true);
    connect(&m_timer, SIGNAL(timeout()), this, SLOT(refreshActivated()));

    ui.setupUi(this);

    ui.kTitle->hide();
    ui.kPeriod->setCurrentIndex(1);
    ui.kInterval->setCurrentIndex(2);

    ui.kSetupReport->setIcon(KIcon("configure"));
    ui.kLineUp->setIcon(KIcon("list-add"));
    ui.kLineDown->setIcon(KIcon("list-remove"));
    ui.kColUp->setIcon(KIcon("list-add"));
    ui.kColDown->setIcon(KIcon("list-remove"));

    ui.kLineAdd->setIcon(KIcon("arrow-right"));
    ui.kLineRemove->setIcon(KIcon("edit-delete"));

    ui.kMode->addItem(KIcon("skg-chart-pie"), i18nc("Noun, the numerical sum of a list of values", "Sum"));
    ui.kMode->addItem(KIcon("skg-chart-line"), i18nc("Noun", "History"));

    ui.kForecastCmb->addItem(i18nc("Noun", "None"));
    ui.kForecastCmb->addItem(KIcon("chronometer"), i18nc("Noun", "Schedule"));
    ui.kForecastCmb->addItem(KIcon("applications-education-mathematics"), i18nc("Noun", "Moving average"));
    ui.kForecastCmb->addItem(KIcon("applications-education-mathematics"), i18nc("Noun", "Weighted moving average"));
    ui.kForecastCmb->addItem(KIcon("skrooge_budget"), i18nc("Noun", "Budget"));

    //Build contextual menu
    KMenu* tableMenu = ui.kTableWithGraph->getTableContextualMenu();
    if(tableMenu) {
        tableMenu->addSeparator();
        QStringList overlayopen;
        overlayopen.push_back("skg_open");
        m_openReportAction = tableMenu->addAction(KIcon("view-statistics", NULL, overlayopen), i18nc("Verb", "Open report..."));
        if(m_openReportAction) m_openReportAction->setEnabled(false);
    }

    KMenu* graphMenu = ui.kTableWithGraph->getGraphContextualMenu();
    if(graphMenu) {
        graphMenu->addSeparator();
        graphMenu->addAction(m_openReportAction);
    }

    connect(m_openReportAction, SIGNAL(triggered(bool)), this, SLOT(onOpenReport()));

    //Init comboboxes
    m_attsForColumns << "d_date" << "d_DATEWEEK" << "d_DATEMONTH" << "d_DATEQUARTER" << "d_DATESEMESTER" << "d_DATEYEAR";
    m_attsForLines << "#NOTHING#" <<
                   "t_ACCOUNTTYPE" << "t_ACCOUNT" <<
                   "t_REALCATEGORY" <<
                   "t_PAYEE" <<
                   "t_TYPEEXPENSENLS" <<
                   "t_mode" <<
                   "t_status" <<
                   "t_UNITTYPE" << "t_UNIT" <<
                   "t_REALREFUND";

    //Adding properties
    QStringList properties;
    iDocument->getDistinctValues("parameters", "t_name", "t_uuid_parent like '%-operation'", properties);
    int nb = properties.count();
    for(int i = 0; i < nb; ++i)
        m_attsForLines.push_back("p_" + properties.at(i));
    m_attsForColumns += m_attsForLines;


    if(iDocument) {
        foreach(const QString & att, m_attsForColumns)
        ui.kColumns->addItem(iDocument->getIcon(att), iDocument->getDisplay(att));

        foreach(const QString & att, m_attsForLines) {
            ui.kLines->addItem(iDocument->getIcon(att), iDocument->getDisplay(att));
        }

        ui.kColumns->setCurrentIndex(2);
    }

    connect(ui.kTableWithGraph, SIGNAL(selectionChanged()), this, SLOT(onSelectionChanged()));

    //Refresh
    connect(ui.kColumns, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kLines, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kMode, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kPeriod, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kTimeline, SIGNAL(valueChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kInterval, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kDateBegin, SIGNAL(dateChanged(QDate)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kDateEnd, SIGNAL(dateChanged(QDate)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kNbIntervals, SIGNAL(valueChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kIncomes, SIGNAL(stateChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kExpenses, SIGNAL(stateChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kTransfers, SIGNAL(stateChanged(int)) , this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kForecastCmb, SIGNAL(currentIndexChanged(int)) , this, SLOT(refresh()), Qt::QueuedConnection);
    connect(ui.kForecastValue, SIGNAL(valueChanged(int)) , this, SLOT(refresh()), Qt::QueuedConnection);

    connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)));

    if(iMinimmumMode) {
        ui.kTableWithGraph->getShowWidget()->setState("\"graph\"");
        setCurrentMode(-1);
    } else {
        setCurrentMode(0);
    }
}

SKGReportPluginWidget::~SKGReportPluginWidget()
{
    SKGTRACEIN(10, "SKGReportPluginWidget::~SKGReportPluginWidget");
    m_openReportAction = NULL;
}

int SKGReportPluginWidget::getCurrentMode()
{
    return m_mode;
}

void SKGReportPluginWidget::onBtnModeClicked()
{
    QWidget* sender = static_cast<QWidget*>(this->sender());

    int currentMode = getCurrentMode();
    int newMode = 0;

    if(sender == ui.kSetupReport) newMode = 0;

    if(currentMode == newMode) newMode = -1;
    setCurrentMode(newMode);
}

void SKGReportPluginWidget::setCurrentMode(int iMode)
{
    m_mode = iMode;

    ui.setupWidget->setVisible(m_mode == 0);
    ui.kTableWithGraph->setFilterVisibility(m_mode == 0);

    if(m_mode == 0) {
        ui.kSetupReport->setChecked(true);
    }
}

QString SKGReportPluginWidget::getState()
{
    SKGTRACEIN(10, "SKGReportPluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root;
    if(m_lastState.hasChildNodes()) {
        doc = m_lastState;
        root = doc.documentElement();
    } else {
        root = doc.createElement("parameters");
        doc.appendChild(root);
    }

    root.setAttribute("columns", m_attsForColumns.value(ui.kColumns->currentIndex()));

    QString lineString;
    int nb = m_attsForLinesAdded.count();
    for(int i = 0; i < nb; ++i) {
        lineString += m_attsForLinesAdded.at(i);
        if(i < nb - 1) lineString += OBJECTSEPARATOR;
    }

    root.setAttribute("lines", lineString);
    root.setAttribute("lines2", m_attsForLines.value(ui.kLines->currentIndex()));
    root.setAttribute("mode", SKGServices::intToString(ui.kMode->currentIndex()));
    root.setAttribute("period", SKGServices::intToString(ui.kPeriod->currentIndex()));
    if(ui.kPeriod->currentIndex() == 4) {
        root.setAttribute("date_begin", SKGServices::intToString(ui.kDateBegin->date().toJulianDay()));
        root.setAttribute("date_end", SKGServices::intToString(ui.kDateEnd->date().toJulianDay()));
    }
    root.setAttribute("interval", SKGServices::intToString(ui.kInterval->currentIndex()));
    root.setAttribute("nb_intervals", SKGServices::intToString(ui.kNbIntervals->value()));
    root.setAttribute("timeline", SKGServices::intToString(ui.kTimeline->value()));
    root.setAttribute("incomes", ui.kIncomes->isChecked() ? "Y" : "N");
    root.setAttribute("expenses", ui.kExpenses->isChecked() ? "Y" : "N");
    root.setAttribute("transfers", ui.kTransfers->isChecked() ? "Y" : "N");
    root.setAttribute("currentPage", SKGServices::intToString(getCurrentMode()));
    root.setAttribute("tableAndGraphState", ui.kTableWithGraph->getState());
    root.setAttribute("nbLevelLines", SKGServices::intToString(m_nbLevelLines));
    root.setAttribute("nbLevelColumns", SKGServices::intToString(m_nbLevelColumns));
    root.setAttribute("forecast", SKGServices::intToString(ui.kForecastCmb->currentIndex()));
    root.setAttribute("forecastValue", SKGServices::intToString(ui.kForecastValue->value()));

    if(m_operationWhereClause.length()) root.setAttribute("operationWhereClause", m_operationWhereClause);

    return doc.toString();
}

void SKGReportPluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGReportPluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    QString columns = root.attribute("columns");
    QString lines = root.attribute("lines");
    QString lines2 = root.attribute("lines2");
    QString mode = root.attribute("mode");
    QString period = root.attribute("period");
    QString interval = root.attribute("interval");
    QString nb_interval = root.attribute("nb_intervals");
    QString timeline = root.attribute("timeline");
    QString date_begin = root.attribute("date_begin");
    QString date_end = root.attribute("date_end");
    QString incomes = root.attribute("incomes");
    QString expenses = root.attribute("expenses");
    QString transfers = root.attribute("transfers");
    QString currentPage = root.attribute("currentPage");
    QString forecast = root.attribute("forecast");
    QString forecastValue = root.attribute("forecastValue");
    QString tableAndGraphState = root.attribute("tableAndGraphState");
    QString title = root.attribute("title");
    QString title_icon = root.attribute("title_icon");
    m_operationWhereClause = root.attribute("operationWhereClause");
    QString nbLevelLinesString = root.attribute("nbLevelLines");
    QString nbLevelColumnsString = root.attribute("nbLevelColumns");

    //Default values
    if(nbLevelLinesString.isEmpty()) nbLevelLinesString = '0';
    if(nbLevelColumnsString.isEmpty()) nbLevelColumnsString = '0';
    if(columns.isEmpty()) columns = m_attsForColumns.at(2);
    if(lines2.isEmpty()) lines2 = m_attsForLines.at(0);
    if(mode.isEmpty()) mode = '0';
    if(period.isEmpty()) period = '1';
    if(interval.isEmpty()) interval = '2';
    if(nb_interval.isEmpty()) nb_interval = '1';
    if(timeline.isEmpty()) timeline = '1';
    if(incomes.isEmpty()) incomes = 'Y';
    if(expenses.isEmpty()) expenses = 'Y';
    if(transfers.isEmpty()) transfers = 'Y';
    if(currentPage.isEmpty()) currentPage = '0';
    if(forecast.isEmpty()) forecast = '0';
    if(forecastValue.isEmpty()) forecastValue = '0';

    m_nbLevelLines = SKGServices::stringToInt(nbLevelLinesString);
    m_nbLevelColumns = SKGServices::stringToInt(nbLevelColumnsString);
    ui.kColumns->setCurrentIndex(m_attsForColumns.indexOf(columns));
    ui.kLines->setCurrentIndex(m_attsForLines.indexOf(lines2));
    m_attsForLinesAdded.clear();
    if(!lines.isEmpty()) m_attsForLinesAdded = lines.split(OBJECTSEPARATOR);
    ui.kMode->setCurrentIndex(SKGServices::stringToInt(mode));
    ui.kPeriod->setCurrentIndex(SKGServices::stringToInt(period));
    ui.kInterval->setCurrentIndex(SKGServices::stringToInt(interval));
    ui.kTimeline->setValue(SKGServices::stringToInt(timeline));
    ui.kNbIntervals->setValue(SKGServices::stringToInt(nb_interval));
    if(!date_begin.isEmpty()) ui.kDateBegin->setDate(QDate::fromJulianDay(SKGServices::stringToInt(date_begin)));
    if(!date_end.isEmpty()) ui.kDateEnd->setDate(QDate::fromJulianDay(SKGServices::stringToInt(date_end)));
    ui.kIncomes->setChecked(incomes != "N");
    ui.kExpenses->setChecked(expenses != "N");
    ui.kTransfers->setChecked(transfers != "N");
    setCurrentMode(SKGServices::stringToInt(currentPage));
    ui.kForecastCmb->setCurrentIndex(SKGServices::stringToInt(forecast));
    ui.kForecastValue->setValue(SKGServices::stringToInt(forecastValue));

    refresh();

    ui.kTableWithGraph->setState(tableAndGraphState);
    if(!title.isEmpty()) {
        ui.kTitle->setComment("<html><body><b>" + SKGServices::stringToHtml(title) + "</b></body></html>");
        ui.kTitle->show();
    } else {
        ui.kTitle->hide();
    }
    if(!title_icon.isEmpty()) ui.kTitle->setPixmap(KIcon(title_icon).pixmap(22, 22), KTitleWidget::ImageLeft);
    if(!m_operationWhereClause.isEmpty()) {
        //We keep a copy of given state in case of bookmark
        m_lastState = doc;
        dataModified("", 0);
    }
}

QString SKGReportPluginWidget::getDefaultStateAttribute()
{
    return "SKGREPORT_DEFAULT_PARAMETERS";
}

QWidget* SKGReportPluginWidget::zoomableWidget() const
{
    return NULL;
}

void SKGReportPluginWidget::getWhereClauseAndTitleForSelection(int row, int column, QString& oWc, QString& oTitle)
{
    //Build where clause and title
    QString currentComment = ui.kTitle->comment();
    if(!currentComment.isEmpty()) {
        currentComment.remove("<html><body><b>");
        currentComment.remove("</b></body></html>");
        oTitle = SKGServices::htmlToString(currentComment) + ". ";
    }

    SKGTableWithGraph::DisplayAdditionalFlag mode = ui.kTableWithGraph->getAdditionalDisplayMode();

    //Condition on line attribute
    oTitle += i18nc("Noun, a list of items", "Sub operations with ");
    SKGColorButton* btn = qobject_cast<SKGColorButton*>(ui.kTableWithGraph->table()->cellWidget(row, 0));

    QString attLine;
    QStringList listAtt = m_attsForLinesAdded;
    if(listAtt.count() == 0 || ui.kLines->currentIndex() > 0) listAtt.push_back(m_attsForLines.at(ui.kLines->currentIndex()));
    int nb = listAtt.count();
    for(int i = 0; i < nb; ++i) {
        QString att = listAtt.at(i);
        if(att == "#NOTHING#") att = "";
        if(att.startsWith(QLatin1String("p_"))) {
            QString propertyName = att.right(att.length() - 2);
            att = "IFNULL((SELECT t_value FROM parameters WHERE t_uuid_parent=v_operation_consolidated.id||'-operation' AND t_name='" + propertyName + "'), '#NULL#')";
        }
        if(att.isEmpty()) att = '\'' + i18nc("Noun", "All") + '\'';

        if(!attLine.isEmpty()) attLine += "||'" + OBJECTSEPARATOR + "'||";
        attLine += att;
    }

    oWc = attLine;
    QString lineVal = (btn ? btn->text() : ui.kTableWithGraph->table()->item(row, 0)->data(1).toString());
    QString title = ui.kTableWithGraph->table()->horizontalHeaderItem(0)->text();
    if(lineVal.isEmpty() && row == ui.kTableWithGraph->table()->rowCount() - 1) {
        //This is the last sum
        oWc = "";
        if(!attLine.isEmpty()) oWc = attLine + " NOT LIKE '%#NULL#%'";
    } else {
        //This is not the last sum
        if(lineVal.isEmpty()) oWc += " IS NULL OR " + attLine;
        oWc += " = '" + SKGServices::stringToSqlString(lineVal) + "' OR " +
               attLine + " like '" + SKGServices::stringToSqlString(lineVal) + OBJECTSEPARATOR + "%'";
        oWc = '(' + oWc + ')';
        oTitle += i18nc("Noun",  "'%1' with '%2'", title, lineVal);
    }

    //Condition on column attribute
    int nbCol = ui.kTableWithGraph->getNbColumns();
    QString att = m_attsForColumns[ui.kColumns->currentIndex()];
    if(att == "#NOTHING#") att = "";
    if(att.startsWith(QLatin1String("p_"))) {
        QString propertyName = att.right(att.length() - 2);
        att = "IFNULL((SELECT t_value FROM parameters WHERE t_uuid_parent=v_operation_consolidated.id||'-operation' AND t_name='" + propertyName + "'), '#NULL#')";
    }
    if(!att.isEmpty()) {
        if(column != 0 && column < nbCol) {
            if(!oWc.isEmpty()) {
                oWc += " AND ";
                oTitle += i18nc("Noun",  " and ");
            }

            oWc += '(' + att;
            QString val = ui.kTableWithGraph->table()->horizontalHeaderItem(column)->text();
            if(val.isEmpty()) {
                oWc += " IS NULL OR " + att + "=''";
                oTitle += i18nc("Noun",  "'%1' are empty", ui.kColumns->currentText());
            } else {
                oWc += " = '" + SKGServices::stringToSqlString(val) + "' OR " +
                       att + " like '" + SKGServices::stringToSqlString(val) + OBJECTSEPARATOR + "%'";
                oTitle += i18nc("Noun",  "'%1' with '%2'", ui.kColumns->currentText(), val);
            }
            oWc += ')';
        } else {
            oWc = '(' + oWc + ") AND " + att + " NOT LIKE '%#NULL#%'";
        }
    }

    //Condition on other attribute
    if(!oWc.isEmpty()) {
        oWc += " AND ";
        oTitle += i18nc("Noun",  " and ");
    }
    oWc += getConsolidatedWhereClause();

    QString during = ui.kPeriod->text();
    if(ui.kPeriod->currentIndex() == 4) {
        during += SKGServices::dateToSqlString(QDateTime(ui.kDateBegin->date())) + ' ' + SKGServices::dateToSqlString(QDateTime(ui.kDateEnd->date()));
    } else if(ui.kPeriod->currentIndex() > 0) {
        if(ui.kNbIntervals->isVisible()) during += ' ' + SKGServices::intToString(ui.kNbIntervals->value()) + ' ' ;
        during += ui.kInterval->text();
    }

    QStringList types;
    if(ui.kIncomes->isChecked()) types.push_back(i18nc("Noun",  "incomes"));
    if(ui.kExpenses->isChecked()) types.push_back(i18nc("Noun",  "expenses"));
    if(ui.kTransfers->isChecked()) types.push_back(i18nc("Noun",  "transfers"));

    oTitle += i18nc("Noun",  "during '%1' for '%2'", during, types.join(" "));
}

void SKGReportPluginWidget::onDoubleClick(int row, int column)
{
    _SKGTRACEIN(10, "SKGReportPluginWidget::onDoubleClick");

    QString wc;
    QString title;
    getWhereClauseAndTitleForSelection(row, column, wc, title);

    //Open
    if(QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier) {
        //Call debug plugin
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);
        root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE " + wc);

        SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Debug plugin"), -1, doc.toString());
    } else {
        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent(getDocument()->getParameter("SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS"));
        QDomElement root = doc.documentElement();
        if(root.isNull()) {
            root = doc.createElement("parameters");
            doc.appendChild(root);
        }

        root.setAttribute("operationTable", "v_operation_consolidated");
        root.setAttribute("operationWhereClause", wc);
        root.setAttribute("title", title);
        root.setAttribute("title_icon", "view-statistics");
        root.setAttribute("currentPage", "-1");

        SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
    }
}

void SKGReportPluginWidget::onOpenReport()
{
    SKGError err;
    SKGTRACEINRC(10, "SKGReportPluginWidget::onOpenReport", err);
    QList<QTableWidgetItem *> selection = ui.kTableWithGraph->table()->selectedItems();
    int nb = selection.count();
    if(nb) {
        QString wc;
        QString title;
        for(int i = 0; i < nb; ++i) {
            QString wc2;
            QString title2;
            getWhereClauseAndTitleForSelection(selection.at(i)->row(), selection.at(i)->column(), wc2, title2);
            if(!wc2.isEmpty()) {
                if(!wc.isEmpty()) wc = '(' + wc + ") OR (" + wc2 + ')';
                else wc = wc2;
            }
            if(!title2.isEmpty()) {
                if(!title.isEmpty()) title = i18n("(%1) or (%2)", title, title2);
                else title = title2;
            }
        }

        //Call report plugin
        QDomDocument doc("SKGML");
        doc.setContent(getState());
        QDomElement root = doc.documentElement();
        root.setAttribute("operationWhereClause", wc);
        root.setAttribute("title", title);
        root.setAttribute("title_icon", "view-statistics");
        SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge report plugin"), -1, doc.toString());
    }
}

void SKGReportPluginWidget::onOneLevelMore()
{
    _SKGTRACEIN(10, "SKGTableWithGraph::onOneLevelMore");
    if(sender() == ui.kLineUp) ++m_nbLevelLines;
    else ++m_nbLevelColumns;
    refresh();
}

void SKGReportPluginWidget::onOneLevelLess()
{
    _SKGTRACEIN(10, "SKGTableWithGraph::onOneLevelLess");
    if(sender() == ui.kLineDown) --m_nbLevelLines;
    else --m_nbLevelColumns;
    refresh();
}

void SKGReportPluginWidget::onAddLine()
{
    _SKGTRACEIN(10, "SKGTableWithGraph::onAddLine");
    m_attsForLinesAdded.push_back(m_attsForLines.value(ui.kLines->currentIndex()));
    ui.kLines->setCurrentIndex(0);
    refresh();
}

void SKGReportPluginWidget::onRemoveLine()
{
    _SKGTRACEIN(10, "SKGTableWithGraph::onRemoveLine");
    if(m_attsForLinesAdded.count()) m_attsForLinesAdded.pop_back();
    refresh();
}

QString SKGReportPluginWidget::getConsolidatedWhereClause(QString* oWhereClausForPreviousData, QString*  oWhereClausForForecastData)
{
    //Build where clause
    QString wc;

    QString strfFormat;
    QString sqlInterval;
    double val = ui.kNbIntervals->value();
    double one = 1;
    QDate a = QDate::currentDate();
    QDate b = a;

    switch(ui.kInterval->currentIndex()) {
    case 0:
        // Interval is in days
        strfFormat = "'%Y-%m-%d'";
        sqlInterval = "DAY";
        break;
    case 1:
        // Interval is in weeks
        strfFormat = "'%Y-%W'";
        sqlInterval = "DAY";
        val *= 7;
        one *= 7;
        break;
    case 2:
        // Interval is in months
        strfFormat = "'%Y-%m'";
        sqlInterval = "MONTH";
        break;
    case 3:
        // Interval is in years
        strfFormat = "'%Y'";
        sqlInterval = "YEAR";
        break;
    }

    switch(ui.kPeriod->currentIndex()) {
    case 1:
        // Current Interval
        switch(ui.kInterval->currentIndex()) {
        case 0:
            // Interval is in days
            break;
        case 1:
            // Interval is in weeks
            a = a.addDays(1 - a.dayOfWeek());
            b = a.addDays(7 - 1);
            break;
        case 2:
            // Interval is in months
            a = a.addDays(1 - a.day());
            b = a.addMonths(1).addDays(-1);
            break;
        case 3:
            // Interval is in years
            a = a.addDays(1 - a.day()).addMonths(1 - a.month());
            b = a.addYears(1).addDays(-1);
            break;
        }
        ui.kDateBegin->setDate(a);
        ui.kDateEnd->setDate(b);
        wc = "strftime(" + strfFormat + ",d_date) = strftime(" + strfFormat + ",'now')";
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData = "strftime(" + strfFormat + ",d_date) < strftime(" + strfFormat + ",'now')";
        if(oWhereClausForForecastData) *oWhereClausForForecastData = "strftime(" + strfFormat + ",d_date) > strftime(" + strfFormat + ",'now')";
        break;
    case 2:
        // Previous Interval
        switch(ui.kInterval->currentIndex()) {
        case 0:
            // Interval is in days
            b = b.addDays(-1);
            a = b.addDays(-val + 1);
            break;
        case 1:
            // Interval is in weeks
            b = b.addDays(-a.dayOfWeek());
            a = b.addDays(-val + 1);
            break;
        case 2:
            // Interval is in months
            b = b.addDays(-b.day());
            a = a.addDays(1 - a.day()).addMonths(-val);
            break;
        case 3:
            // Interval is in years
            b = b.addDays(-b.day()).addMonths(-b.month());
            a = a.addMonths(1 - a.month()).addDays(1 - a.day()).addYears(-val);
            break;
        }
        ui.kDateBegin->setDate(a);
        ui.kDateEnd->setDate(b);
        wc = "strftime(" + strfFormat + ",d_date)>=strftime(" + strfFormat + ",date('now','start of month', '-" + SKGServices::intToString(val) + ' ' + sqlInterval + "'))";
        if(ui.kForecastCmb->currentIndex() != 1) wc += " AND strftime(" + strfFormat + ",d_date)<=strftime(" + strfFormat + ",date('now','start of month', '-" + SKGServices::intToString(one) + ' ' + sqlInterval + "'))";  //For forecast based on scheduled operations
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData = "strftime(" + strfFormat + ",d_date) < strftime(" + strfFormat + ",date('now','start of month', '-" + SKGServices::intToString(val) + ' ' + sqlInterval + "'))";
        if(oWhereClausForForecastData) *oWhereClausForForecastData = "strftime(" + strfFormat + ",d_date) > strftime(" + strfFormat + ",date('now','start of month', '-" + SKGServices::intToString(one) + ' ' + sqlInterval + "'))";
        break;
    case 3:
        //Last Interval
        switch(ui.kInterval->currentIndex()) {
        case 0:
            // Interval is in days
            a = a.addDays(-val);
            break;
        case 1:
            // Interval is in weeks
            a = a.addDays(-val);
            break;
        case 2:
            // Interval is in months
            a = a.addMonths(-val);
            break;
        case 3:
            // Interval is in years
            a = a.addYears(-val);
            break;
        }
        a = a.addDays(1);
        ui.kDateBegin->setDate(a);
        ui.kDateEnd->setDate(b);
        wc = "d_date > date('now','-" + SKGServices::intToString(val) + ' ' + sqlInterval + "')";
        if(ui.kForecastCmb->currentIndex() != 1) wc += " AND d_date<=date('now')";  //For forecast based on scheduled operations
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData = "d_date <= date('now','-" + SKGServices::intToString(val) + ' ' + sqlInterval + "')";
        if(oWhereClausForForecastData) *oWhereClausForForecastData = "d_date > date('now')";
        break;
    case 4:
        // Custom Date
        wc = "d_date>='" + SKGServices::dateToSqlString(QDateTime(ui.kDateBegin->date())) + '\'';
        if(ui.kForecastCmb->currentIndex() != 1) wc += "AND d_date<='" + SKGServices::dateToSqlString(QDateTime(ui.kDateEnd->date())) + '\'';  //For forecast based on scheduled operations
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData = "d_date<'" + SKGServices::dateToSqlString(QDateTime(ui.kDateBegin->date())) + '\'';
        if(oWhereClausForForecastData) *oWhereClausForForecastData = "d_date>'" + SKGServices::dateToSqlString(QDateTime(ui.kDateEnd->date())) + '\'';
        break;
    case 5:
        // Timeline
        val = 13 - ui.kTimeline->value();
        switch(ui.kInterval->currentIndex()) {
        case 0:
            // Interval is in days
            a = a.addDays(1 - val);
            b = a.addDays(1);
            break;
        case 1:
            // Interval is in weeks
            a = a.addDays(8 - 7 * val - a.dayOfWeek());
            b = a.addDays(6);
            break;
        case 2:
            // Interval is in months
            a = a.addDays(1 - a.day()).addMonths(1 - val);
            b = a.addMonths(1).addDays(-1);
            break;
        case 3:
            // Interval is in years
            a = a.addDays(1 - a.day()).addMonths(1 - a.month()).addYears(1 - val);
            b = a.addYears(1).addDays(-1);
            break;
        }
        ui.kDateBegin->setDate(a);
        ui.kDateEnd->setDate(b);
        wc = "d_date>='" + SKGServices::dateToSqlString(QDateTime(ui.kDateBegin->date())) + '\'';
        if(ui.kForecastCmb->currentIndex() != 1) wc += "AND d_date<='" + SKGServices::dateToSqlString(QDateTime(ui.kDateEnd->date())) + '\'';  //For forecast based on scheduled operations
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData = "d_date<'" + SKGServices::dateToSqlString(QDateTime(ui.kDateBegin->date())) + '\'';
        if(oWhereClausForForecastData) *oWhereClausForForecastData = "d_date>'" + SKGServices::dateToSqlString(QDateTime(ui.kDateEnd->date())) + '\'';
        break;
    default:
        // Take all dates
        a = a.addYears(-1);
        ui.kDateBegin->setDate(a);
        ui.kDateEnd->setDate(b);
        wc = "1=1";
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData = "1=0";
        if(oWhereClausForForecastData) *oWhereClausForForecastData = "d_date > date('now')";
        break;
    }

    wc = "((" + wc + ") OR d_date='0000') AND d_date!='0000-00-00'";
    if(oWhereClausForPreviousData) *oWhereClausForPreviousData = "((" + *oWhereClausForPreviousData + ") OR d_date='0000-00-00')";

    QString operationTypes;
    if(ui.kIncomes->isChecked() && !ui.kExpenses->isChecked()) {
        operationTypes = "t_TYPEEXPENSE='+'";
    } else if(ui.kExpenses->isChecked() && !ui.kIncomes->isChecked()) {
        operationTypes = "t_TYPEEXPENSE='-'";
    }
    if(operationTypes.length() > 0) {
        QString condition = " AND " + operationTypes;
        wc += condition;
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData += condition;
    }

    if(ui.kTransfers->isChecked() == false) {
        QString condition = " AND i_group_id=0";
        wc += condition;
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData += condition;
    }

    if(m_operationWhereClause.length()) {
        QString condition = " AND (" + m_operationWhereClause + ')';
        wc += condition;
        if(oWhereClausForPreviousData) *oWhereClausForPreviousData += condition;
    }

    return wc;
}

void SKGReportPluginWidget::onSelectionChanged()
{
    if(m_openReportAction) m_openReportAction->setEnabled(ui.kTableWithGraph->table()->selectedItems().count() > 0);
}

void SKGReportPluginWidget::refresh()
{
    int p = ui.kPeriod->currentIndex();

    ui.kForecastValue->setEnabled(ui.kForecastCmb->currentIndex() > 0);
    ui.kLineRemove->setEnabled(m_attsForLinesAdded.count());

    //Check dates
    ui.kDateSelect->setVisible(p > 0);
    ui.kDateSelect->setEnabled(p == 4);
    ui.kTimeline->setVisible(p == 5);

    QDate d1 = ui.kDateBegin->date();
    QDate d2 = ui.kDateEnd->date();
    if(d1 > d2) {
        ui.kDateBegin->setDate(d2);
        ui.kDateEnd->setDate(d1);
    }

    //Check income & expense
    if(!ui.kIncomes->isChecked() && !ui.kExpenses->isChecked()) {
        if(sender() == ui.kIncomes) ui.kExpenses->setChecked(true);
        else ui.kIncomes->setChecked(true);
    }

    bool current = (p == 1);
    bool previous = (p == 2);
    bool last = (p == 3);
    bool timeline = (p == 5);

    ui.kInterval->setVisible(current || previous || last || timeline);
    ui.kNbIntervals->setVisible(last || previous);
    ui.kForecastFrm->setEnabled(!timeline);
    if(timeline) ui.kForecastCmb->setCurrentIndex(0);

    m_timer.start(300);
}

void SKGReportPluginWidget::refreshActivated()
{
    dataModified("", 0);
}

void SKGReportPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)

{
    SKGTRACEIN(10, "SKGReportPluginWidget::dataModified");
    Q_UNUSED(iIdTransaction);

    //Refresh panel
    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled(db != NULL);
    if(db != NULL) {
        //Check if needed
        if(iTableName == "operation" || iTableName.isEmpty()) {
            SKGError err;
            SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
            if(doc) {
                //Check if update is needed
                QString ParametersUsed = getState() + ';' + SKGServices::intToString(doc->getTransactionToProcess());
                if(ParametersUsed == m_previousParametersUsed) {
                    SKGTRACEL(10) << "Same parameters. Refresh ignored" << endl;
                    return;
                }
                m_previousParametersUsed = ParametersUsed;

                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

                //Fill line and create title
                QString lineString;
                QString title;
                int nb = m_attsForLinesAdded.count();
                for(int i = 0; i < nb; ++i) {
                    lineString += doc->getDisplay(m_attsForLinesAdded.at(i));
                    if(i < nb - 1) lineString += OBJECTSEPARATOR;
                }
                if(ui.kLines->currentIndex() > 0 && nb) {
                    lineString += OBJECTSEPARATOR;
                }
                title = lineString;
                if(ui.kLines->currentIndex() > 0 || nb == 0) {
                    title += ui.kLines->text();
                }
                ui.kLineLabel->setVisible(nb > 0);
                ui.kLineLabel->setText(lineString);

                //Get parameters
                int col = ui.kColumns->currentIndex();
                int line = ui.kLines->currentIndex();

                if(col >= 0 && line >= 0) {
                    //Mode history ?
                    bool modeHistory = (ui.kMode->currentIndex() > 0);

                    //Get condition (must be done before forecast on scheduled operations)
                    int nbVirtualColumn = 0;
                    QString conditionPrevious;
                    QString conditionForecast;
                    QString condition = getConsolidatedWhereClause(&conditionPrevious, &conditionForecast);
                    int forecastMode = ui.kForecastCmb->currentIndex();

                    //Compute forecast on scheduled operations
                    if(forecastMode == 1 || forecastMode == 4) {
                        //Create operations
                        int nbDays = ui.kDateBegin->date().daysTo(QDate::currentDate());
                        nbDays = (nbDays * ui.kForecastValue->value()) / FORECASTMAX;
                        QDate lastDate = QDate::currentDate().addDays(nbDays);
                        if(forecastMode == 1) {
                            //Create scheduled operations
                            doc->beginTransaction("#INTERNAL#");
                            int nbInserted = 0;
                            SKGRecurrentOperationObject::process(doc, nbInserted, lastDate);
                        } else {
                            //Create budgeted operations
                            SKGObjectBase::SKGListSKGObjectBase budgets;
                            if(err.isSucceeded()) err = doc->getObjects("v_budget",
                                                            "t_PERIOD>STRFTIME('%Y-%m', date('now')) AND "
                                                            "t_PERIOD<=STRFTIME('%Y-%m', '" + SKGServices::dateToSqlString(QDateTime(lastDate)) + "')", budgets);
                            int nb = budgets.count();
                            if(nb) {
                                //Get most active account
                                SKGStringListList list;
                                err = doc->executeSelectSqliteOrder("SELECT account.id FROM account, operation WHERE operation.rd_account_id=account.id AND d_date>date('now', '-3 month') GROUP BY (t_name) ORDER BY COUNT(1) DESC LIMIT 1", list);
                                if(err.isSucceeded() && list.count() == 2) {
                                    SKGAccountObject account(doc, SKGServices::stringToInt(list.at(1).at(0)));
                                    err = account.load();

                                    //Get unit
                                    SKGUnitObject unit(doc);
                                    if(err.isSucceeded()) err = unit.setName(doc->getPrimaryUnit().Name);
                                    if(err.isSucceeded()) err = unit.setSymbol(doc->getPrimaryUnit().Symbol);
                                    if(err.isSucceeded()) err = unit.load();


                                    doc->beginTransaction("#INTERNAL#", nb);
                                    for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                                        SKGBudgetObject budget = budgets.at(i);

                                        SKGCategoryObject cat;
                                        budget.getCategory(cat);

                                        SKGOperationObject op;
                                        if(err.isSucceeded()) err = account.addOperation(op);
                                        if(err.isSucceeded()) err = op.setDate(QDate(budget.getYear(), qMax(budget.getMonth(), 1), 1));
                                        if(err.isSucceeded()) err = op.setUnit(unit);
                                        if(err.isSucceeded()) err = op.save();

                                        SKGSubOperationObject sop;
                                        if(err.isSucceeded()) err = op.addSubOperation(sop);
                                        if(err.isSucceeded()) err = sop.setCategory(cat);
                                        if(err.isSucceeded()) err = sop.setQuantity(budget.getBudgetedAmount());
                                        if(err.isSucceeded()) err = sop.save();

                                        if(err.isSucceeded()) err = doc->stepForward(i + 1);
                                    }
                                }
                            }
                        }
                        ui.kForecastValue->setToolTip(SKGServices::dateToSqlString(QDateTime(lastDate)));
                    } else {
                        ui.kForecastValue->setToolTip("");
                        conditionForecast = "1=0";
                    }

                    //Execute sql order
                    SKGStringListList table;
                    QString tableName = "v_operation_consolidated";
                    QString attCol = m_attsForColumns[col ];
                    if(attCol == "#NOTHING#") attCol = "";
                    if(attCol.startsWith(QLatin1String("p_"))) {
                        QString propertyName = attCol.right(attCol.length() - 2);
                        attCol = "IFNULL((SELECT t_value FROM parameters WHERE t_uuid_parent=v_operation_consolidated.id||'-operation' AND t_name='" + propertyName + "'), '#NULL#')";
                    }

                    QStringList listAtt = m_attsForLinesAdded;
                    if(listAtt.count() == 0 || line > 0) listAtt.push_back(m_attsForLines.at(line));
                    QString attLine;
                    int nb = listAtt.count();
                    for(int i = 0; i < nb; ++i) {
                        QString att = listAtt.at(i);
                        if(att == "#NOTHING#") att = "";
                        if(att.startsWith(QLatin1String("p_"))) {
                            QString propertyName = att.right(att.length() - 2);
                            att = "IFNULL((SELECT t_value FROM parameters WHERE t_uuid_parent=v_operation_consolidated.id||'-operation' AND t_name='" + propertyName + "'), '#NULL#')";
                        }
                        if(att.isEmpty()) att = '\'' + i18nc("Noun", "All") + '\'';

                        if(!attLine.isEmpty()) attLine += "||'" + OBJECTSEPARATOR + "'||";
                        attLine += att;
                    }

                    if(modeHistory && !attCol.isEmpty()) {
                        tableName = "(SELECT ";
                        if(attCol != "d_date") tableName += "(CASE WHEN " + conditionPrevious + " THEN '0000' ELSE d_date END) AS d_date, ";
                        tableName += "(CASE WHEN " + conditionPrevious + " THEN '0000' ELSE " + attCol + "||(CASE WHEN " + conditionForecast + " THEN '999' ELSE '' END) END) AS " + m_attsForColumns[col ] + ",* FROM v_operation_consolidated) as v_operation_consolidated";
                    }

                    //Remove #NULL# columns and lines
                    if(!attLine.isEmpty()) condition = '(' + condition + ") AND " + attLine + " NOT LIKE '%#NULL#%'";
                    if(!attCol.isEmpty()) condition = '(' + condition + ") AND " + attCol + " NOT LIKE '%#NULL#%'";

                    if(err.isSucceeded()) err = doc->getConsolidatedView(tableName, attCol, attLine, "f_REALCURRENTAMOUNT", "TOTAL", condition, table);
                    IFSKGTRACEL(10) {
                        QStringList dump = SKGServices::tableToDump(table, SKGServices::DUMP_TEXT);
                        int nbl = dump.count();
                        for(int i = 0; i < nbl; ++i) {
                            SKGTRACE << dump[i] << endl;
                        }
                    }

                    if(forecastMode == 1 || forecastMode == 4) {
                        //Rollback create operations
                        doc->endTransaction(false);

                        //Compute nb date in futur
                        QStringList line1 = table.at(0);
                        int nbCol = line1.count();
                        for(int i = 0; i < nbCol; ++i) {
                            QString title = line1.at(i);
                            if(title.endsWith(QLatin1String("999"))) {
                                title = title.left(title.count() - 3);
                                line1.replace(i, title);
                                ++nbVirtualColumn;
                            }
                        }
                        table.replace(0, line1);
                    }

                    if(err.isSucceeded()) {
                        if(table.count()) {
                            //Change title
                            QStringList line1 = table.at(0);
                            if(line1.count()) {
                                line1.replace(0, title);
                                table.replace(0, line1);
                            }
                        }

                        //Compute forecast on average
                        int nbLines = table.count();
                        if(nbLines && (forecastMode == 2 || forecastMode == 3)) {
                            QStringList newLine = table.at(0);
                            int nbVals = newLine.count() - 1;
                            int pourcent = ui.kForecastValue->value();
                            if(nbVals >= 3 && pourcent > 0) {
                                //Compute nb value to add
                                nbVirtualColumn = pourcent * nbVals / FORECASTMAX;

                                //Build header
                                for(int i = 0; i < nbVirtualColumn; ++i) {
                                    newLine.push_back(i18nc("Noun", "N %1", (i + 1)));
                                }
                                table.replace(0, newLine);

                                //Build values
                                for(int j = 1; j < nbLines; ++j) {
                                    QStringList newLine = table.at(j);
                                    for(int i = 0; i < nbVirtualColumn; ++i) {
                                        //Moving average
                                        double nv = 0;
                                        double weight = 0;
                                        for(int k = 0; k < nbVirtualColumn; ++k) {
                                            if(forecastMode == 2) {
                                                //Simple mode
                                                nv += SKGServices::stringToDouble(newLine.at(nbVals + i - k));
                                                ++weight;
                                            } else {
                                                //Weighted mode
                                                nv += (nbVirtualColumn - k) * SKGServices::stringToDouble(newLine.at(nbVals + i - k));
                                                weight += nbVirtualColumn - k;
                                            }
                                        }
                                        newLine.push_back(SKGServices::doubleToString(nv / weight));
                                    }
                                    table.replace(j, newLine);
                                }
                            }
                        }

                        //Create grouped by lines table
                        int nbLevelLineMax = 0;
                        {
                            SKGStringListList groupedByLineTable = table;
                            {
                                int nbCols = -1;
                                QString previousLineName = "###";
                                if(groupedByLineTable.count()) {
                                    nbCols = groupedByLineTable.at(0).count();
                                }
                                for(int i = 1; nbCols && i < groupedByLineTable.count(); ++i) {  //Dynamically modified
                                    QStringList line = groupedByLineTable.at(i);
                                    QString val = line.at(0);

                                    //Rebuild val for the number of level
                                    QStringList vals = val.split(OBJECTSEPARATOR);
                                    int nbvals = vals.count();
                                    if(nbvals > m_nbLevelLines + 1) {
                                        //Rebuild val
                                        val = "";
                                        for(int k = 0; k <= m_nbLevelLines; ++k) {
                                            val += vals[k];
                                            if(k != m_nbLevelLines) val += OBJECTSEPARATOR;
                                        }
                                    }
                                    nbLevelLineMax = qMax(nbLevelLineMax, nbvals - 1);

                                    if(val == previousLineName) {
                                        //Current line is merged with previous one
                                        QStringList newLine;
                                        newLine.push_back(val);
                                        for(int k = 1; k < nbCols; ++k) {
                                            QString valstring1 = line.at(k);
                                            QString valstring2 = groupedByLineTable.at(i - 1).at(k);

                                            if(!valstring1.isEmpty() || !valstring2.isEmpty()) {
                                                double sum2 = 0;
                                                if(!valstring1.isEmpty()) sum2 += SKGServices::stringToDouble(valstring1);
                                                if(!valstring2.isEmpty()) sum2 += SKGServices::stringToDouble(valstring2);
                                                newLine.push_back(SKGServices::doubleToString(sum2));
                                            } else {
                                                newLine.push_back("");
                                            }
                                        }

                                        groupedByLineTable.replace(i - 1, newLine);

                                        //Remove current line
                                        groupedByLineTable.removeAt(i);
                                        --i;
                                    } else {
                                        //Current line is just modified
                                        QStringList newLine;
                                        newLine.push_back(val);
                                        for(int k = 1; k < nbCols; ++k) {
                                            newLine.push_back(line.at(k));
                                        }
                                        groupedByLineTable.replace(i, newLine);

                                        previousLineName = val;
                                    }
                                }
                            }
                            table = groupedByLineTable;
                        }

                        //Create grouped by columns table
                        int nbLevelColMax = 0;
                        {
                            SKGStringListList groupedByColTable = table;
                            int nbLines = groupedByColTable.count();
                            if(nbLines) {
                                QString previousColumnName = "###";
                                for(int i = 1; nbLines && i < groupedByColTable.at(0).count(); ++i) {  //Dynamically modified
                                    QString val = groupedByColTable.at(0).at(i);

                                    //Rebuild val for the number of level
                                    QStringList vals = val.split(OBJECTSEPARATOR);
                                    int nbvals = vals.count();
                                    if(nbvals > m_nbLevelColumns + 1) {
                                        //Rebuild val
                                        val = "";
                                        for(int k = 0; k <= m_nbLevelColumns; ++k) {
                                            val += vals[k];
                                            if(k != m_nbLevelColumns) val += OBJECTSEPARATOR;
                                        }
                                    }
                                    nbLevelColMax = qMax(nbLevelColMax, nbvals - 1);

                                    if(val == previousColumnName) {
                                        //Current column is merged with previous one
                                        QStringList newLine = groupedByColTable.at(0);
                                        newLine.removeAt(i);
                                        groupedByColTable.replace(0, newLine);

                                        for(int k = 1; k < nbLines; ++k) {
                                            newLine = groupedByColTable.at(k);
                                            QString valstring1 = newLine.at(i);
                                            QString valstring2 = newLine.at(i - 1);

                                            if(!valstring1.isEmpty() || !valstring2.isEmpty()) {
                                                double sum2 = 0;
                                                if(!valstring1.isEmpty()) sum2 += SKGServices::stringToDouble(valstring1);
                                                if(!valstring2.isEmpty()) sum2 += SKGServices::stringToDouble(valstring2);
                                                newLine.replace(i - 1, SKGServices::doubleToString(sum2));
                                            } else {
                                                newLine.removeAt(i - 1);
                                            }
                                            newLine.removeAt(i);
                                            groupedByColTable.replace(k, newLine);
                                        }

                                        //Remove current line
                                        --i;
                                    } else {
                                        //Current column is just modified
                                        QStringList newLine = groupedByColTable.at(0);
                                        newLine.replace(i, val);
                                        groupedByColTable.replace(0, newLine);

                                        previousColumnName = val;
                                    }
                                }
                            }
                            table = groupedByColTable;
                        }

                        //Create history (must be done after groups)
                        if(modeHistory) table = SKGReportPluginWidget::getHistorizedTable(table);

                        //Set data
                        SKGServices::SKGUnitInfo primaryUnit = doc->getPrimaryUnit();
                        SKGServices::SKGUnitInfo secondaryUnit = doc->getSecondaryUnit();

                        SKGTableWithGraph::DisplayAdditionalFlag dmode = SKGTableWithGraph::NONE;
                        if(m_attsForColumns.at(col).startsWith(QLatin1String("d_"))) dmode |= SKGTableWithGraph::AVERAGE | SKGTableWithGraph::LIMITS;
                        if(!modeHistory) dmode |= SKGTableWithGraph::SUM;
                        ui.kTableWithGraph->setData(table, primaryUnit, secondaryUnit, dmode, nbVirtualColumn);

                        //Enable/Disable buttons
                        m_nbLevelLines = qMin(nbLevelLineMax, m_nbLevelLines);
                        ui.kLineDown->setEnabled(m_nbLevelLines > 0);
                        ui.kLineUp->setEnabled(m_nbLevelLines < nbLevelLineMax);

                        m_nbLevelColumns = qMin(nbLevelColMax, m_nbLevelColumns);
                        ui.kColDown->setEnabled(m_nbLevelColumns > 0);
                        ui.kColUp->setEnabled(m_nbLevelColumns < nbLevelColMax);
                    }
                }
                QApplication::restoreOverrideCursor();

            }

            //Display error
            SKGMainPanel::getMainPanel()->displayErrorMessage(err);
        }
    }
}

SKGStringListList SKGReportPluginWidget::getHistorizedTable(const SKGStringListList& iTable)
{
    //Build history
    SKGStringListList historizedTable;

    historizedTable.push_back(iTable.at(0));

    QStringList newLine;
    int nblines = iTable.count();
    int nbCols = 0;
    if(nblines) {
        nbCols = iTable.at(0).count();
    }
    for(int i = 1; i < nblines; ++i) {
        QStringList newLine;
        newLine.push_back(iTable.at(i).at(0));

        double sum = 0;
        for(int j = 1; j < nbCols; ++j) {
            sum += SKGServices::stringToDouble(iTable.at(i).at(j));
            newLine.push_back(SKGServices::doubleToString(sum));
        }
        historizedTable.push_back(newLine);
    }

    return historizedTable;
}

#include "skgreportpluginwidget.moc"
