/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to track operations.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgpayeepluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgobjectmodel.h"
#include "skgdocumentbank.h"
#include "skgpayeeobject.h"
#include "skgtransactionmng.h"

#include <KAction>

#include <QDomDocument>
#include <QKeyEvent>

SKGPayeePluginWidget::SKGPayeePluginWidget(SKGDocument* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEIN(1, "SKGPayeePluginWidget::SKGPayeePluginWidget");
    if(!iDocument) return;

    ui.setupUi(this);

    ui.kNameLbl->setText(i18n("%1:", iDocument->getDisplay("t_name")));
    ui.kAddressLabel->setText(i18n("%1:", iDocument->getDisplay("t_address")));

    ui.kAddButton->setIcon(KIcon("list-add"));
    ui.kModifyButton->setIcon(KIcon("dialog-ok-apply"));
    ui.kDeleteUnusedButton->setIcon(KIcon("edit-delete"));

    ui.kView->setModel(new SKGObjectModel(static_cast<SKGDocumentBank*>(getDocument()), "v_payee_display", "1=0", this, "", false));

    //Add registered global action in contextual menu
    if(SKGMainPanel::getMainPanel()) {
        ui.kView->getView()->insertGlobalAction("edit_delete");
        ui.kView->getView()->insertGlobalAction();
        ui.kView->getView()->insertGlobalAction("edit_find");
        ui.kView->getView()->insertGlobalAction("open_report");

        ui.kView->getView()->resizeColumnToContents(0);
    }
    connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection);
    connect(ui.kView->getView(), SIGNAL(clickEmptyArea()), this, SLOT(cleanEditor()));
    connect(ui.kView->getView(), SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onDoubleClick()));
    connect(ui.kView->getView(), SIGNAL(selectionChangedDelayed()), this, SLOT(onSelectionChanged()));

    //Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter(this);

    dataModified("", 0);
}

SKGPayeePluginWidget::~SKGPayeePluginWidget()
{
    SKGTRACEIN(1, "SKGPayeePluginWidget::~SKGPayeePluginWidget");
}

bool SKGPayeePluginWidget::eventFilter(QObject *object, QEvent *event)
{
    if(event && event->type() == QEvent::KeyPress) {
        QKeyEvent *keyEvent = (QKeyEvent *) event;
        if((keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter) && object == this) {
            if(QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAddButton->isEnabled()) {
                ui.kAddButton->click();
            } else if(QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kModifyButton->isEnabled()) {
                ui.kModifyButton->click();
            }
        }
    }

    return false;
}

void SKGPayeePluginWidget::onSelectionChanged()
{
    SKGTRACEIN(10, "SKGPayeePluginWidget::onSelectionChanged");

    int nbSelect = ui.kView->getView()->getNbSelectedObjects();
    if(nbSelect == 1) {
        SKGPayeeObject obj(ui.kView->getView()->getFirstSelectedObject());

        ui.kNameInput->setText(obj.getName());
        ui.kAddressEdit->setText(obj.getAddress());
    } else if(nbSelect > 1) {
        ui.kNameInput->setText(NOUPDATE);
        ui.kAddressEdit->setText(NOUPDATE);
    }

    onEditorModified();
    Q_EMIT selectionChanged();
}

QString SKGPayeePluginWidget::getState()
{
    SKGTRACEIN(10, "SKGPayeePluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);
    root.setAttribute("view", ui.kView->getState());
    return doc.toString();
}

void SKGPayeePluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGPayeePluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();
    ui.kView->setState(root.attribute("view"));
}

QString SKGPayeePluginWidget::getDefaultStateAttribute()
{
    return "SKGPAYEE_DEFAULT_PARAMETERS";
}

QWidget* SKGPayeePluginWidget::mainWidget() const
{
    return ui.kView->getView();
}

void SKGPayeePluginWidget::onEditorModified()
{
    _SKGTRACEIN(10, "SKGPayeePluginWidget::onEditorModified");
    int nb = getNbSelectedObjects();
    ui.kModifyButton->setEnabled(!ui.kNameInput->text().isEmpty() && nb >= 1);
    ui.kAddButton->setEnabled(!ui.kNameInput->text().isEmpty());
}

void SKGPayeePluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
    SKGTRACEIN(10, "SKGPayeePluginWidget::dataModified");
    Q_UNUSED(iIdTransaction);

    if(iTableName == "payee" || iTableName.isEmpty()) {
        //Set completions
        SKGMainPanel::fillWithDistinctValue(ui.kNameInput, getDocument(), "payee", "t_name", "", true);
        SKGMainPanel::fillWithDistinctValue(ui.kAddressEdit, getDocument(), "payee", "t_address", "", true);
    }
}

void SKGPayeePluginWidget::onAddPayee()
{
    SKGError err;
    _SKGTRACEINRC(10, "SKGPayeePluginWidget::onAddPayee", err);

    QString name = ui.kNameInput->text();
    SKGPayeeObject payee;
    {
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Payee creation '%1'", name)  , err);

        if(err.isSucceeded()) err = SKGPayeeObject::createPayee(static_cast<SKGDocumentBank*>(getDocument()), name, payee);
        if(err.isSucceeded()) err = payee.setAddress(ui.kAddressEdit->text());
        if(err.isSucceeded()) err = payee.save();
    }
    //status bar
    if(err.isSucceeded()) {
        err = SKGError(0, i18nc("Successful message after an user action", "Payee '%1' created", name));
        ui.kView->getView()->selectObject(payee.getUniqueID());
    } else err.addError(ERR_FAIL, i18nc("Error message", "Payee creation failed"));

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGPayeePluginWidget::onModifyPayee()
{
    SKGError err;
    _SKGTRACEINRC(10, "SKGPayeePluginWidget::onModifyPayee", err);

    //Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();

    int nb = selection.count();

    {
        SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Payee update")  , err, nb);
        //Is it a massive modification of payees to merge them ?
        if(nb > 1 && ui.kNameInput->text() != NOUPDATE && !ui.kNameInput->text().startsWith(QLatin1String("="))) {
            getDocument()->sendMessage(i18nc("Information message", "You tried to modify all names of selected payees. Payees have been merged."));

            //Do the merge
            SKGPayeeObject payeeObj1 = selection[0];
            for(int i = 1; err.isSucceeded() && i < nb; ++i) {
                SKGPayeeObject payeeObj = selection[i];
                err = payeeObj1.merge(payeeObj);
            }

            //Change selection for the rest of the operation
            selection.clear();
            selection.push_back(payeeObj1);
            nb = 1;
        }

        for(int i = 0; err.isSucceeded() && i < nb; ++i) {
            //Modification of object
            SKGPayeeObject payee = selection[i];
            err = payee.setName(ui.kNameInput->text());
            if(err.isSucceeded()) err = payee.setAddress(ui.kAddressEdit->text());
            if(err.isSucceeded()) err = payee.save();
        }
        //status bar
        if(err.isSucceeded()) err = SKGError(0, i18nc("Successful message after an user action", "Payee updated"));
        else err.addError(ERR_FAIL, i18nc("Error message", "Payee update failed"));
    }

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGPayeePluginWidget::onDoubleClick()
{
    _SKGTRACEIN(10, "SKGPayeePluginWidget::onDoubleClick");

    //Get selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
    if(selection.count() == 1) {
        SKGPayeeObject payee(selection.at(0));
        QString name = payee.getName();

        //Build where clause and title
        QString wc = "t_PAYEE = '" + SKGServices::stringToSqlString(name) + '\'';
        QString title = i18nc("Noun, a list of items", "Operations assigned to payee '%1'",  name);

        //Open
        if(QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier) {
            //Call debug plugin
            QDomDocument doc("SKGML");
            QDomElement root = doc.createElement("parameters");
            doc.appendChild(root);
            root.setAttribute("sqlOrder", "SELECT * from v_operation_display WHERE " + wc);

            SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Debug plugin"), -1, doc.toString());
        } else {
            //Call operation plugin
            QDomDocument doc("SKGML");
            doc.setContent(getDocument()->getParameter("SKGOPERATION_DEFAULT_PARAMETERS"));
            QDomElement root = doc.documentElement();
            if(root.isNull()) {
                root = doc.createElement("parameters");
                doc.appendChild(root);
            }

            root.setAttribute("operationTable", "v_operation_display");
            root.setAttribute("operationWhereClause", wc);
            root.setAttribute("title", title);
            root.setAttribute("title_icon", "skrooge_payee");
            root.setAttribute("currentPage", "-1");

            SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
        }
    }
}

void SKGPayeePluginWidget::cleanEditor()
{
    if(getNbSelectedObjects() == 0) {
        ui.kNameInput->setText("");
        ui.kAddressEdit->setText("");
    }
}

void SKGPayeePluginWidget::activateEditor()
{
    ui.kNameInput->setFocus();
}

bool SKGPayeePluginWidget::isEditor()
{
    return true;
}

void SKGPayeePluginWidget::onDeleteUnused()
{
    KAction* act = SKGMainPanel::getMainPanel()->getGlobalAction("edit_delete_unused_payees");
    if(act) act->trigger();
}

#include "skgpayeepluginwidget.moc"


