/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGOPERATIONPLUGIN_H
#define SKGOPERATIONPLUGIN_H
/** @file
* This file is Skrooge plugin for operation management.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skginterfaceplugin.h"
#include "ui_skgoperationpluginwidget_pref.h"

#include <QObject>
#include <QVariantList>
#include <kgenericfactory.h>

class SKGMainPanel;
class KAction;
class SKGDocumentBank;

/**
 * This file is Skrooge plugin for operation management
 */
class SKGOperationPlugin : public SKGInterfacePlugin
{
    Q_OBJECT
    Q_INTERFACES(SKGInterfacePlugin);

public:
    /**
     * Default Constructor
     */
    SKGOperationPlugin(QObject* iParent, const QVariantList& iArg);

    /**
     * Default Destructor
     */
    virtual ~SKGOperationPlugin();

    /**
     * Called to initialise the plugin
     * @param iDocument the main document
     * @param iArgument the arguments
     * @return true if the plugin is compatible with the document
     */
    virtual bool setupActions(SKGDocument* iDocument, const QStringList& iArgument);

    /**
     * Must be modified to close properly the plugin.
     */
    virtual void close();

    /**
     * Must be modified to refresh widgets after a modification.
     */
    virtual void refresh();

    /**
     * The number of dashboard widgets of the plugin.
     * @return The number of dashboard widgets of the plugin
     */
    virtual int getNbDashboardWidgets();

    /**
     * Get a dashboard widget title of the plugin.
     * @param iIndex the index of the widget
     * @return The title
     */
    virtual QString getDashboardWidgetTitle(int iIndex);

    /**
     * Get a dashboard widget of the plugin.
     * @param iIndex the index of the widget
     * @return The dashboard widget of the plugin
     */
    virtual SKGWidget* getDashboardWidget(int iIndex);

    /**
     * The preference widget of the plugin.
     * @return The preference widget of the plugin
     */
    virtual QWidget* getPreferenceWidget();

    /**
     * The preference skeleton of the plugin.
     * @return The preference skeleton of the plugin
     */
    virtual KConfigSkeleton* getPreferenceSkeleton();

    /**
     * The context widget of the plugin.
     * @return The context widget of the plugin
     */
    virtual SKGTabPage* getWidget();

    /**
     * The title of the plugin.
     * @return The title of the plugin
     */
    virtual QString title() const;

    /**
     * The icon of the plugin.
     * @return The icon of the plugin
     */
    virtual QString icon() const;

    /**
     * The toolTip of the plugin.
     * @return The toolTip of the plugin
     */
    virtual QString toolTip() const;

    /**
     * The tips list of the plugin.
     * @return The tips list of the plugin
     */
    virtual QStringList tips() const;

    /**
     * Must be implemented to set the position of the plugin.
     * @return integer value between 0 and 999 (default = 999)
     */
    virtual int getOrder() const;

    /**
     * Must be implemented to know if a plugin must be display in context chooser.
     * @return true of false (default = false)
     */
    virtual bool isInContext() const;

    /**
     * The advice list of the plugin.
     * @return The advice list of the plugin
     */
    virtual QList<SKGInterfacePlugin::SKGAdvice> advices() const;

    /**
     * Must be implemented to execute the automatic correction for the advice.
     * @param iAdviceIdentifier the identifier of the advice
     * @param iSolution the identifier of the possible solution
     * @return an object managing the error. MUST return ERR_NOTIMPL if iAdviceIdentifier is not known
     *   @see SKGError
     */
    virtual SKGError executeAdviceCorrection(const QString& iAdviceIdentifier, int iSolution) const;

    /**
     * @brief Open highlighted operations
     **/
    static void openHighLights();

private Q_SLOTS:
    void actionDuplicate();
    void actionCreateTemplate();
    void actionSwitchToPointed();
    void actionSwitchHighLight();
    void actionOpenHighLights();
    void actionOpenLastModified();
    void actionOpenSubOperations();
    void actionOpenDuplicate();
    void actionGroupOperation();
    void actionUngroupOperation();
    void actionMergeSubOperations();

private:
    Q_DISABLE_COPY(SKGOperationPlugin);

    KAction *m_duplicateAction;
    KAction *m_createTemplateAction;
    KAction *m_switchToPointedAction;
    KAction *m_switchHighLight;
    KAction *m_openHighLights;
    KAction *m_openLastModified;
    KAction *m_openSuboperations;
    KAction *m_openDuplicate;
    KAction *m_groupOperation;
    KAction *m_ungroupOperation;
    KAction *m_mergeOperationAction;

    SKGDocumentBank* m_currentBankDocument;

    Ui::skgoperationplugin_pref ui;
};

#endif // SKGOPERATIONPLUGIN_H
