/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin to generate categories.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgcategoriespluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgobjectmodel.h"
#include "skgsortfilterproxymodel.h"
#include "skgdocumentbank.h"
#include "skgcategoryobject.h"
#include "skgtransactionmng.h"

#include <klocale.h>
#include <KAction>

#include <QDomDocument>
#include <QtGui/QWidget>
#include <QGraphicsScene>
#include <QHeaderView>
#include <QKeyEvent>

SKGCategoriesPluginWidget::SKGCategoriesPluginWidget(SKGDocumentBank* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEIN(10, "SKGCategoriesPluginWidget::SKGCategoriesPluginWidget");
    if(!iDocument) return;

    ui.setupUi(this);

    ui.kNameLbl->setText(i18n("%1:", iDocument->getDisplay("t_name")));

    ui.kAddCategoryButton->setIcon(KIcon("list-add"));
    ui.kModifyCategoryButton->setIcon(KIcon("dialog-ok-apply"));
    ui.kDeleteUnusedButton->setIcon(KIcon("edit-delete"));

    m_objectModel = new SKGObjectModel(static_cast<SKGDocumentBank*>(getDocument()), "v_category_display", "1=0", this, "r_category_id", false);
    SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel(this);
    modelproxy->setSourceModel(m_objectModel);
    modelproxy->setSortRole(Qt::UserRole);
    modelproxy->setDynamicSortFilter(true);
    ui.kCategoriesView->setModel(modelproxy);

    //Add registered global action in contextual menu
    if(SKGMainPanel::getMainPanel()) {
        ui.kCategoriesView->insertGlobalAction("edit_delete");
        ui.kCategoriesView->insertGlobalAction();
        ui.kCategoriesView->insertGlobalAction("edit_find");
        ui.kCategoriesView->insertGlobalAction("open_report");

        ui.kCategoriesView->resizeColumnToContents(0);
        ui.kCategoriesView->header()->setStretchLastSection(false);
    }

    connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection);
    dataModified("", 0);

    connect(ui.kCategoriesView, SIGNAL(selectionChangedDelayed()), this, SLOT(onSelectionChanged()));

    connect(m_objectModel, SIGNAL(beforeReset()), ui.kCategoriesView, SLOT(saveSelection()));
    connect(m_objectModel, SIGNAL(afterReset()), ui.kCategoriesView, SLOT(resetSelection()));

    //Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter(this);
}

SKGCategoriesPluginWidget::~SKGCategoriesPluginWidget()
{
    SKGTRACEIN(10, "SKGCategoriesPluginWidget::~SKGCategoriesPluginWidget");
    m_objectModel = NULL;
}

bool SKGCategoriesPluginWidget::eventFilter(QObject *object, QEvent *event)
{
    if(event && event->type() == QEvent::KeyPress) {
        QKeyEvent *keyEvent = (QKeyEvent *) event;
        if((keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter) && object == this) {
            if(QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAddCategoryButton->isEnabled()) {
                ui.kAddCategoryButton->click();
            } else if(QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kModifyCategoryButton->isEnabled()) {
                ui.kModifyCategoryButton->click();
            }
        }
    }

    return false;
}

void SKGCategoriesPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
    SKGTRACEIN(10, "SKGCategoriesPluginWidget::dataModified");
    Q_UNUSED(iIdTransaction);

    if(iTableName == "category" || iTableName.isEmpty()) {
        //Correction bug 2299394 vvv
        if(ui.kCategoriesView->isAutoResized()) ui.kCategoriesView->resizeColumnsToContentsDelayed();
        //Correction bug 2299394 ^^^

        //Set completions
        SKGMainPanel::fillWithDistinctValue(ui.kNameInput, getDocument(), "category", "t_name", "", true);

        onEditorModified();
    }
}

void SKGCategoriesPluginWidget::onSelectionChanged()
{
    SKGTRACEIN(10, "SKGCategoriesPluginWidget::onSelectionChanged");

    int nbSelect = ui.kCategoriesView->getNbSelectedObjects();
    if(nbSelect == 1) {
        SKGCategoryObject obj(ui.kCategoriesView->getFirstSelectedObject());

        ui.kNameInput->setText(obj.getName());
    } else if(nbSelect > 1) {
        ui.kNameInput->setText(NOUPDATE);
    }

    onEditorModified();
    Q_EMIT selectionChanged();
}

QString SKGCategoriesPluginWidget::getState()
{
    SKGTRACEIN(10, "SKGCategoriesPluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    //Memorize table settings
    root.setAttribute("view", ui.kCategoriesView->getState());
    return doc.toString();
}

void SKGCategoriesPluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGCategoriesPluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    if(m_objectModel) {
        m_objectModel->setFilter("1=1");
    }

    //!!! Must be done here after setFilter
    ui.kCategoriesView->setState(root.attribute("view"));

    //Correction bug 2299394 vvv
    if(ui.kCategoriesView->isAutoResized()) ui.kCategoriesView->resizeColumnsToContentsDelayed();
    //Correction bug 2299394 ^^^
}

QString SKGCategoriesPluginWidget::getDefaultStateAttribute()
{
    return "SKGCATEGORIES_DEFAULT_PARAMETERS";
}

QWidget* SKGCategoriesPluginWidget::mainWidget() const
{
    return ui.kCategoriesView;
}

void SKGCategoriesPluginWidget::onEditorModified()
{
    _SKGTRACEIN(10, "SKGCategoriesPluginWidget::onEditorModified");
    int nb = getNbSelectedObjects();
    ui.kModifyCategoryButton->setEnabled(!ui.kNameInput->text().isEmpty() && nb >= 1);
    ui.kAddCategoryButton->setEnabled(!ui.kNameInput->text().isEmpty() && nb <= 1);
}

void SKGCategoriesPluginWidget::onAddCategory()
{
    SKGError err;
    _SKGTRACEINRC(10, "SKGCategoriesPluginWidget::onAddCategory", err);
    SKGCategoryObject cat;
    QString name = ui.kNameInput->text();
    {
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();

        int nb = selection.count();
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Category creation '%1'", name)  , err);

        if(nb == 1) {
            SKGCategoryObject parent = selection[0];
            name = parent.getFullName() + OBJECTSEPARATOR + name;
        }
        if(err.isSucceeded()) err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), name, cat);

    }

    //status bar
    if(err.isSucceeded()) {
        ui.kCategoriesView->selectObject(cat.getUniqueID());
        err = SKGError(0, i18nc("Successful message after an user action", "Category '%1' created", name));
    } else err.addError(ERR_FAIL, i18nc("Error message", "Category creation failed"));

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGCategoriesPluginWidget::onUpdateCategory()
{
    SKGError err;
    _SKGTRACEINRC(10, "SKGCategoriesPluginWidget::onUpdateCategory", err);
    //Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();

    int nb = selection.count();
    {
        SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Category update")  , err, nb);
        if(nb > 1 && ui.kNameInput->text() != NOUPDATE && !ui.kNameInput->text().startsWith(QLatin1String("="))) {
            getDocument()->sendMessage(i18nc("Information message", "You tried to modify all names of selected categories. Categories have been merged."));

            //Do the merge
            SKGCategoryObject catObj1 = selection[0];
            for(int i = 1; err.isSucceeded() && i < nb; ++i) {
                SKGCategoryObject catObj = selection[i];
                err = catObj1.merge(catObj);
            }

            //Change selection for the rest of the operation
            selection.clear();
            selection.push_back(catObj1);
            nb = 1;
        }

        for(int i = 0; err.isSucceeded() && i < nb; ++i) {

            //Modification of object
            SKGCategoryObject cat = selection[i];
            err = cat.setName(ui.kNameInput->text());
            if(err.isSucceeded()) err = cat.save();
        }
        //status bar
        if(err.isSucceeded()) err = SKGError(0, i18nc("Successful message after an user action", "Category updated"));
        else err.addError(ERR_FAIL, i18nc("Error message", "Category update failed"));
    }

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGCategoriesPluginWidget::onDeleteUnused()
{
    KAction* act = SKGMainPanel::getMainPanel()->getGlobalAction("edit_delete_unused_categories");
    if(act) act->trigger();
}

void SKGCategoriesPluginWidget::onDoubleClick()
{
    _SKGTRACEIN(10, "SKGCategoriesPluginWidget::onDoubleClick");

    //Get selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
    if(selection.count() == 1) {
        SKGCategoryObject cat(selection.at(0));
        QString name = cat.getFullName();

        //Build where clause and title
        QString wc = "t_REALCATEGORY";
        QString title;
        if(name.isEmpty()) {
            wc += " IS NULL OR t_REALCATEGORY=''";
            wc = '(' + wc + ')';
            title = i18nc("Noun, a list of items", "Sub operations with empty category");
        } else {
            wc += " = '" + SKGServices::stringToSqlString(name) + "' OR t_REALCATEGORY like '" + SKGServices::stringToSqlString(name) + OBJECTSEPARATOR + "%'";
            wc = '(' + wc + ')';
            title = i18nc("Noun, a list of items", "Sub operations with category containing '%1'",  name);
        }

        //Open
        if(QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier) {
            //Call debug plugin
            QDomDocument doc("SKGML");
            QDomElement root = doc.createElement("parameters");
            doc.appendChild(root);
            root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE " + wc);

            SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Debug plugin"), -1, doc.toString());
        } else {
            //Call operation plugin
            QDomDocument doc("SKGML");
            doc.setContent(getDocument()->getParameter("SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS"));
            QDomElement root = doc.documentElement();
            if(root.isNull()) {
                root = doc.createElement("parameters");
                doc.appendChild(root);
            }

            root.setAttribute("operationTable", "v_operation_consolidated");
            root.setAttribute("operationWhereClause", wc);
            root.setAttribute("title", title);
            root.setAttribute("title_icon", "skrooge_category");
            root.setAttribute("currentPage", "-1");

            SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
        }
    }
}

void SKGCategoriesPluginWidget::cleanEditor()
{
    if(getNbSelectedObjects() == 0) {
        ui.kNameInput->setText("");
    }
}
void SKGCategoriesPluginWidget::activateEditor()
{
    ui.kNameInput->setFocus();
}

bool SKGCategoriesPluginWidget::isEditor()
{
    return true;
}

#include "skgcategoriespluginwidget.moc"
