/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to manage budgets.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgbudgetpluginwidget.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgbudgetobject.h"
#include "skgbudgetruleobject.h"
#include "skgcategoryobject.h"
#include "skgtransactionmng.h"
#include "skgtraces.h"
#include "skgdocument.h"

#include <QDomDocument>
#include <QKeyEvent>

SKGBudgetPluginWidget::SKGBudgetPluginWidget(SKGDocument* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEIN(1, "SKGBudgetPluginWidget::SKGBudgetPluginWidget");
    if(!iDocument) return;

    ui.setupUi(this);

    ui.kPeriodLbl->setText(i18n("%1:", iDocument->getDisplay("t_period")));
    ui.kYearLbl->setText(i18n("%1:", iDocument->getDisplay("i_year")));
    ui.kMonthLbl->setText(i18n("%1:", iDocument->getDisplay("i_month")));
    ui.kAmountLabel->setText(i18n("%1:", iDocument->getDisplay("f_value")));
    ui.kCategoryLabel->setText(i18n("%1:", iDocument->getDisplay("t_category")));
    ui.kYearCheck->setText(i18n("%1:", iDocument->getDisplay("i_year")));
    ui.kMonthCheck->setText(i18n("%1:", iDocument->getDisplay("i_month")));
    ui.kCategoryCheck->setText(i18n("%1:", iDocument->getDisplay("t_category")));
    ui.kPeriodLbl2->setText(i18n("%1:", iDocument->getDisplay("t_period")));
    ui.kAmountLabel2->setText(i18n("%1:", iDocument->getDisplay("f_value")));
    ui.kCategoryTransferCheck->setText(i18n("%1:", iDocument->getDisplay("t_category")));

    ui.kYear->setValue(QDate::currentDate().year());
    ui.kYearAuto->setValue(QDate::currentDate().year());
    ui.kYearAutoBase->setValue(QDate::currentDate().year());
    ui.kMonth->setValue(QDate::currentDate().month());
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Monthly"));
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Yearly"));
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Individual"));


    ui.kView->getShowWidget()->addItem("all", i18nc("Noun, budget items to display", "All"), "",
                                       "",
                                       "current;currentYear;currentMonth;previousYear;previousMonth");
    ui.kView->getShowWidget()->addSeparator();
    ui.kView->getShowWidget()->addItem("current", i18nc("Noun, budget items to display", "Current"), "",
                                       "t_PERIOD>=STRFTIME('%Y-%m', date('now')) OR t_PERIOD=STRFTIME('%Y', date('now'))",
                                       "currentMonth",  //Check when checked
                                       "",       //Uncheck when checked
                                       "",       //Check when unchecked
                                       "all"     //Uncheck when unchecked
                                      );
    ui.kView->getShowWidget()->addItem("currentYear", i18nc("Noun, budget items to display", "Current year"), "view-calendar-month",
                                       "t_PERIOD LIKE STRFTIME('%Y', date('now'))||'%'",
                                       "",
                                       "",
                                       "",
                                       "all"
                                      );
    ui.kView->getShowWidget()->addItem("currentMonth", i18nc("Noun, budget items to display", "Current month"), "view-calendar-week",
                                       "t_PERIOD=STRFTIME('%Y-%m', date('now'))",
                                       "",
                                       "",
                                       "",
                                       "all;current"
                                      );
    ui.kView->getShowWidget()->addSeparator();
    ui.kView->getShowWidget()->addItem("previousYear", i18nc("Noun, budget items to display", "Previous year"), "view-calendar-month",
                                       "t_PERIOD LIKE STRFTIME('%Y', date('now','-1 Year'))||'%'",
                                       "",
                                       "",
                                       "",
                                       "all"
                                      );
    ui.kView->getShowWidget()->addItem("previousMonth", i18nc("Noun, budget items to display", "Previous month"), "view-calendar-week",
                                       "t_PERIOD=STRFTIME('%Y-%m', date('now','-1 Month'))",
                                       "",
                                       "",
                                       "",
                                       "all"
                                      );
    ui.kView->getShowWidget()->setDefaultState("current");

    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "All"), SKGBudgetRuleObject::ALL);
    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "Negative"), SKGBudgetRuleObject::NEGATIVE);
    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "Positive"), SKGBudgetRuleObject::POSITIVE);

    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Next"), SKGBudgetRuleObject::NEXT);
    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Current"), SKGBudgetRuleObject::CURRENT);
    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Current year"), SKGBudgetRuleObject::YEAR);

    SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
    if(doc) {
        ui.kUnitCmb->addItem("%");
        ui.kUnitCmb->addItem(doc->getPrimaryUnit().Symbol);
        ui.kUnit->setText(doc->getPrimaryUnit().Symbol);

        //Bind operation view
        m_objectModel = new SKGObjectModel(doc, "v_budget_display", "1=0", this, "", false);
        ui.kView->setModel(m_objectModel);
    }

    //Add registered global action in contextual menu
    if(SKGMainPanel::getMainPanel()) {
        ui.kView->getView()->insertGlobalAction("edit_delete");
    }
    connect(ui.kView->getView(), SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onDoubleClick()));
    connect(ui.kView->getView(), SIGNAL(selectionChangedDelayed()), this, SLOT(onSelectionChanged()));
    connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection);

    //Add Standard KDE Icons to buttons to Operations
    ui.kModifyBtn->setIcon(KIcon("dialog-ok-apply"));
    ui.kAddBtn->setIcon(KIcon("list-add"));
    ui.kRulesBtn->setIcon(KIcon("skrooge_transfer"));
    ui.kProcessBtn->setIcon(KIcon("system-run"));

    ui.kManualBtn->setIcon(KIcon("user-properties"));
    ui.kAutoBtn->setIcon(KIcon("games-solve"));

    SKGMainPanel::fillWithDistinctValue(ui.kCategoryEdit, getDocument(), "category", "t_fullname", "");
    SKGMainPanel::fillWithDistinctValue(ui.kCategoryRule, getDocument(), "category", "t_fullname", "");
    SKGMainPanel::fillWithDistinctValue(ui.kCategoryTransfer, getDocument(), "category", "t_fullname", "");

    setCurrentMode(0);

    //Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter(this);
}

SKGBudgetPluginWidget::~SKGBudgetPluginWidget()
{
    SKGTRACEIN(1, "SKGBudgetPluginWidget::~SKGBudgetPluginWidget");
    m_objectModel = NULL;
}

bool SKGBudgetPluginWidget::eventFilter(QObject *object, QEvent *event)
{
    if(event && event->type() == QEvent::KeyPress) {
        QKeyEvent *keyEvent = (QKeyEvent *) event;
        if((keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter) && object == this) {
            if(QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAddBtn->isEnabled()) {
                ui.kAddBtn->click();
            } else if(QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kModifyBtn->isEnabled()) {
                ui.kModifyBtn->click();
            }
        }
    }

    return false;
}

QString SKGBudgetPluginWidget::getState()
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    root.setAttribute("currentPage", SKGServices::intToString(getCurrentMode()));
    if(m_objectModel && m_objectModel->getRealTable() == "budget") {
        root.setAttribute("view", ui.kView->getState());
        root.setAttribute("viewRule", m_viewRule);
    } else {
        root.setAttribute("view", m_viewBudget);
        root.setAttribute("viewRule", ui.kView->getState());
    }

    return doc.toString();
}

void SKGBudgetPluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    QString currentPage = root.attribute("currentPage");
    if(currentPage.isEmpty()) currentPage = '0';

    setCurrentMode(SKGServices::stringToInt(currentPage));

    m_viewBudget = root.attribute("view");
    m_viewRule = root.attribute("viewRule");
    if(m_objectModel && m_objectModel->getRealTable() == "budget") {
        ui.kView->setState(m_viewBudget);
    } else {
        ui.kView->setState(m_viewRule);
    }
}

QString SKGBudgetPluginWidget::getDefaultStateAttribute()
{
    return "SKGBUDGET_DEFAULT_PARAMETERS";
}

QWidget* SKGBudgetPluginWidget::mainWidget() const
{
    return ui.kView->getView();
}

void SKGBudgetPluginWidget::refresh()
{
    SKGTRACEIN(1, "SKGBudgetPluginWidget::refresh");

    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled(db != NULL);
    if(db != NULL) {
        //Refresh yours widgets here
    }
}

void SKGBudgetPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::dataModified");
    Q_UNUSED(iIdTransaction);

    //Refresh widgets
    if(iTableName == "category" || iTableName.isEmpty()) {
        //Set type category
        SKGMainPanel::fillWithDistinctValue(ui.kCategoryEdit, getDocument(), "category", "t_fullname", "");
        SKGMainPanel::fillWithDistinctValue(ui.kCategoryRule, getDocument(), "category", "t_fullname", "");
        SKGMainPanel::fillWithDistinctValue(ui.kCategoryTransfer, getDocument(), "category", "t_fullname", "");
    }
}

int SKGBudgetPluginWidget::getCurrentMode()
{
    return m_mode;
}

void SKGBudgetPluginWidget::setCurrentMode(int iMode)
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::setCurrentMode");
    if(!m_objectModel) return;
    if(iMode == 2 && m_mode != 2) {
        ui.kView->getShowWidget()->setEnabled(false);
        m_viewBudget = ui.kView->getState();
        m_objectModel->setFilter("");
        m_objectModel->setTable("v_budgetrule_display");
        ui.kView->setState(m_viewRule);
    } else if(iMode != 2 && m_mode == 2) {
        ui.kView->getShowWidget()->setEnabled(true);
        m_viewRule = ui.kView->getState();
        m_objectModel->setTable("v_budget_display");
        ui.kView->setState(m_viewBudget);
    }
    m_mode = iMode;

    ui.SKGManualSection->hide();
    ui.SKGAutoSection->hide();
    ui.SKGRuleSection->hide();
    ui.SKGEditionButtonsSection->hide();
    ui.kManualBtn->setChecked(false);
    ui.kAutoBtn->setChecked(false);
    ui.kRulesBtn->setChecked(false);

    if(m_mode == 0) {
        ui.SKGManualSection->show();
        ui.SKGEditionButtonsSection->show();
        ui.kManualBtn->setChecked(true);
    } else if(m_mode == 1) {
        ui.SKGAutoSection->show();
        ui.SKGEditionButtonsSection->show();
        ui.kAutoBtn->setChecked(true);
    } else if(m_mode == 2) {
        ui.SKGRuleSection->show();
        ui.SKGEditionButtonsSection->show();
        ui.kRulesBtn->setChecked(true);
    }

    onCreatorModified();
}

void SKGBudgetPluginWidget::onBtnModeClicked()
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::onBtnModeClicked");
    QWidget* sender = static_cast<QWidget*>(this->sender());

    int currentMode = getCurrentMode();
    int newMode = 0;

    if(sender == ui.kManualBtn) newMode = 0;
    else if(sender == ui.kAutoBtn) newMode = 1;
    else if(sender == ui.kRulesBtn) newMode = 2;

    if(currentMode == newMode) newMode = -1;
    setCurrentMode(newMode);
}

void SKGBudgetPluginWidget::onProcessRules()
{
    SKGError err;
    _SKGTRACEINRC(10, "SKGBudgetPluginWidget::onProcessRules", err);

    {
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Process budget rules")  , err);
        err = SKGBudgetRuleObject::processAllRules(static_cast<SKGDocumentBank*>(getDocument()));
    }
    //status bar
    if(err.isSucceeded()) {
        err = SKGError(0, i18nc("Successful message after an user action", "Budget rules processed"));
    } else err.addError(ERR_FAIL, i18nc("Error message", "Budget rules failed"));

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGBudgetPluginWidget::onAddClicked()
{
    SKGError err;
    _SKGTRACEINRC(10, "SKGBudgetPluginWidget::onAddClicked", err);

    if(getCurrentMode() == 2) {
        //Creation of a rule
        QStringList uniqueIDs;
        {
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget rule creation")  , err);
            SKGBudgetRuleObject budgetRule(getDocument());
            err = updateBudgetRule(budgetRule);
            uniqueIDs.push_back(budgetRule.getUniqueID());
        }
        //status bar
        if(err.isSucceeded()) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget rule created"));
            ui.kView->getView()->selectObjects(uniqueIDs);
        } else err.addError(ERR_FAIL, i18nc("Error message", "Budget rule creation failed"));
    } else {
        //Creation of a budget
        QStringList uniqueIDs;
        {
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget creation")  , err);
            if(getCurrentMode() == 0) {
                //Manual creation
                int mode = ui.kPeriod->currentIndex();
                if(mode == 0) {  //Monthly
                    for(int m = 1; err.isSucceeded() && m <= 12; ++m) {
                        SKGBudgetObject budget(getDocument());
                        err = updateBudget(budget, m);
                        uniqueIDs.push_back(budget.getUniqueID());
                    }
                } else if(mode == 1) {  //Yearly
                    SKGBudgetObject budget(getDocument());

                    err = updateBudget(budget, 0);
                    uniqueIDs.push_back(budget.getUniqueID());
                } else { //Individual
                    SKGBudgetObject budget(getDocument());
                    err = updateBudget(budget);
                    uniqueIDs.push_back(budget.getUniqueID());
                }
            } else {
                //Automatic creation
                err = SKGBudgetObject::createAutomaticBudget((SKGDocumentBank*) getDocument(), ui.kYearAuto->value(), ui.kYearAutoBase->value(),
                        ui.kRemovePrevious->isChecked(), ui.kBalancingMonthly->isChecked(), ui.kBalancingAnnual->isChecked());
            }
        }

        //status bar
        if(err.isSucceeded()) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget created"));
            ui.kView->getView()->selectObjects(uniqueIDs);
        } else err.addError(ERR_FAIL, i18nc("Error message", "Budget creation failed"));
    }

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGBudgetPluginWidget::onUpdateClicked()
{
    SKGError err;
    SKGTRACEINRC(10, "SKGBudgetPluginWidget::onUpdateClicked", err);
    //Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
    if(getCurrentMode() == 2) {
        {
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget rule update"), err);
            SKGBudgetRuleObject rule = selection.at(0);
            err = updateBudgetRule(rule);
        }

        //status bar
        if(err.isSucceeded()) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget rule updated"));
        } else err.addError(ERR_FAIL, i18nc("Error message",  "Budget rule update failed"));
    } else {
        {
            int nb = selection.count();
            SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget update"), err, nb);
            for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                SKGBudgetObject budget = selection.at(i);
                int mode = ui.kPeriod->currentIndex();
                if(mode == 1) {  //Yearly
                    err = updateBudget(budget, 0);
                } else { //Individual
                    err = updateBudget(budget);
                }

                if(err.isSucceeded()) err = getDocument()->stepForward(i + 1);
            }
        }

        //status bar
        if(err.isSucceeded()) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget updated"));
        } else err.addError(ERR_FAIL, i18nc("Error message",  "Budget update failed"));
    }


    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

SKGError SKGBudgetPluginWidget::updateBudget(SKGBudgetObject& iBudget, int iMonth)
{
    SKGError err;
    if(err.isSucceeded() && ui.kYear->isEnabled()) err = iBudget.setYear(ui.kYear->value());
    if(err.isSucceeded() && ui.kMonth->isEnabled()) err = iBudget.setMonth(iMonth != -1 ? iMonth : ui.kMonth->value());

    SKGCategoryObject cat;
    QString catName = ui.kCategoryEdit->text().trimmed();
    if(err.isSucceeded()) err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catName, cat, true);
    if(err.isSucceeded()) err = iBudget.setCategory(cat);
    if(err.isSucceeded()) err = iBudget.enableSubCategoriesInclusion(ui.kIncludingSubCategories->isChecked());

    double val = ui.kAmountEdit->value();
    //Is the sign forced ?
    if(ui.kAmountEdit->sign() == 0) {
        //No
        SKGObjectBase cat2(cat.getDocument(), "v_category_display", cat.getID());

        //Are we able to find to sign with the category ?
        if(cat2.getAttribute("t_TYPEEXPENSE") == "-") val = -val;
    }
    if(err.isSucceeded())  err = iBudget.setBudgetedAmount(val);

    if(err.isSucceeded()) err = iBudget.save();
    return err;
}

SKGError SKGBudgetPluginWidget::updateBudgetRule(SKGBudgetRuleObject& iRule)
{
    SKGError err;
    SKGCategoryObject cat;
    QString catName = ui.kCategoryRule->text().trimmed();
    if(err.isSucceeded()) err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catName, cat, true);
    SKGCategoryObject catchange;
    QString catchangeName = ui.kCategoryTransfer->text().trimmed();
    if(err.isSucceeded()) err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catchangeName, catchange, true);
    if(err.isSucceeded()) err = iRule.enableCategoryCondition(ui.kCategoryCheck->isChecked());
    if(err.isSucceeded()) err = iRule.setBudgetCategory(cat);
    if(err.isSucceeded()) err = iRule.enableYearCondition(ui.kYearCheck->isChecked());
    if(err.isSucceeded()) err = iRule.setBudgetYear(ui.kYearRule->value());
    if(err.isSucceeded()) err = iRule.enableMonthCondition(ui.kMonthCheck->isChecked());
    if(err.isSucceeded()) err = iRule.setBudgetMonth(ui.kMonthRule->value());
    if(err.isSucceeded()) {
        bool absolute = (ui.kUnitCmb->currentIndex() == 1);
        double val = ui.kAmountEdit2->value();
        if(!absolute) val = qMin(qMax(double(0), val), double(100));
        err = iRule.setQuantity(val, absolute);
    }
    if(err.isSucceeded()) err = iRule.setCondition((SKGBudgetRuleObject::Condition) ui.kConditionCmb->itemData(ui.kConditionCmb->currentIndex()).toInt());
    if(err.isSucceeded()) err = iRule.enableCategoryChange(ui.kCategoryTransferCheck->isChecked());
    if(err.isSucceeded()) err = iRule.setTransfer((SKGBudgetRuleObject::Mode) ui.kModeCmb->itemData(ui.kModeCmb->currentIndex()).toInt(), catchange);
    if(err.isSucceeded()) err = iRule.save();
    return err;
}
void SKGBudgetPluginWidget::open(const SKGBudgetObject& iBudget)
{
    _SKGTRACEIN(10, "SKGBudgetPluginWidget::open");

    //Build where clause and title
    QString y = SKGServices::intToString(iBudget.getYear());
    QString m = SKGServices::intToString(iBudget.getMonth());
    if(m.length() == 1) m = '0' + m;

    QString wc = "i_IDCATEGORY IN (SELECT bc.id_category FROM budgetcategory bc WHERE bc.id=" + SKGServices::intToString(iBudget.getID()) + ") AND "
                 "STRFTIME('%Y', d_date)='" + y + '\'';
    if(m != "00") wc += " AND STRFTIME('%m', d_date)='" + m + '\'';
    QString title = i18nc("Noun, a list of items", "Operations assigned to budget");

    //Open
    if(QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier) {
        //Call debug plugin
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);
        root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE " + wc);

        SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Debug plugin"), -1, doc.toString());
    } else {
        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent(iBudget.getDocument()->getParameter("SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS"));
        QDomElement root = doc.documentElement();
        if(root.isNull()) {
            root = doc.createElement("parameters");
            doc.appendChild(root);
        }

        root.setAttribute("operationTable", "v_operation_consolidated");
        root.setAttribute("operationWhereClause", wc);
        root.setAttribute("title", title);
        root.setAttribute("title_icon", "skrooge_budget");
        root.setAttribute("currentPage", "-1");

        SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
    }
}

void SKGBudgetPluginWidget::onDoubleClick()
{
    _SKGTRACEIN(10, "SKGBudgetPluginWidget::onDoubleClick");
    if(!m_objectModel) return;

    //Get selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
    if(selection.count() == 1 && m_objectModel->getRealTable() == "budget") {
        SKGBudgetObject budget(selection.at(0));
        SKGBudgetPluginWidget::open(budget);
    }
}

void SKGBudgetPluginWidget::onCreatorModified()
{
    bool test = !ui.kAmountEdit->text().isEmpty() && !ui.kYear->text().isEmpty();
    ui.kAddBtn->setEnabled(test || getCurrentMode() != 0);
    ui.kModifyBtn->setEnabled((test && ui.kPeriod->currentIndex() != 0 && getCurrentMode() == 0 && getNbSelectedObjects())
                              || (getCurrentMode() == 2 && getNbSelectedObjects() == 1));

    bool monthCondition = (ui.kPeriod->currentIndex() == 2 || getCurrentMode() == 2);
    ui.kMonthLbl->setVisible(monthCondition);
    ui.kMonth->setVisible(monthCondition);
}

void SKGBudgetPluginWidget::onSelectionChanged()
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::onSelectionChanged");
    if(!m_objectModel) return;

    SKGObjectBase::SKGListSKGObjectBase objs = getSelectedObjects();
    int nb = objs.count();
    int mode = getCurrentMode();
    if(nb) {
        if(m_objectModel->getRealTable() == "budget") {
            SKGBudgetObject budget = objs.at(0);
            ui.kYear->setValue(budget.getYear());
            ui.kMonth->setValue(budget.getMonth());
            ui.kAmountEdit->setValue(budget.getBudgetedAmount());
            ui.kCategoryEdit->setText(budget.getAttribute("t_CATEGORY"));
            ui.kPeriod->setCurrentIndex(budget.getMonth() == 0 ? 1 : 2); //Set yearly or individual
            ui.kIncludingSubCategories->setChecked(budget.isSubCategoriesInclusionEnabled());

            if(mode > 0) setCurrentMode(0);
        } else {
            SKGBudgetRuleObject rule = objs.at(0);
            ui.kYearCheck->setChecked(rule.isYearConditionEnabled());
            ui.kYearRule->setValue(rule.getBudgetYear());
            ui.kMonthCheck->setChecked(rule.isMonthConditionEnabled());
            ui.kMonthRule->setValue(rule.getBudgetMonth());
            ui.kCategoryCheck->setChecked(rule.isCategoryConditionEnabled());
            ui.kCategoryRule->setText(rule.getAttribute("t_CATEGORYCONDITION"));
            ui.kCategoryTransferCheck->setChecked(rule.isCategoryChangeEnabled());
            ui.kCategoryTransfer->setText(rule.getAttribute("t_CATEGORY"));
            ui.kUnitCmb->setCurrentIndex(rule.isAbolute() ? 1 : 0);
            ui.kAmountEdit2->setValue(rule.getQuantity());
            ui.kModeCmb->setCurrentIndex(ui.kModeCmb->findData(rule.getTransferMode()));
            ui.kConditionCmb->setCurrentIndex(ui.kConditionCmb->findData(rule.getCondition()));
        }
    }

    ui.kPeriod->setEnabled(nb <= 1);
    ui.kYear->setEnabled(nb <= 1);
    ui.kMonth->setEnabled(nb <= 1);

    onCreatorModified();
    Q_EMIT selectionChanged();
}

void SKGBudgetPluginWidget::activateEditor()
{
    if(getCurrentMode() == -1) setCurrentMode(0);
    ui.kAmountEdit->setFocus();
}

bool SKGBudgetPluginWidget::isEditor()
{
    return true;
}

#include "skgbudgetpluginwidget.moc"


