/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGSHOW_H
#define SKGSHOW_H
/** @file
 * A widget to select what to show.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include <qtoolbutton.h>
#include <qmap.h>
#include <qtimer.h>

#include "skgbasegui_export.h"

class KMenu;
class QAction;

/**
 * This file is a widget to select what to show
 */
class SKGBASEGUI_EXPORT SKGShow : public QToolButton
{
    Q_OBJECT
    /**
     * State of the object
     */
    Q_PROPERTY(QString state READ getState WRITE setState USER true);

public:
    /**
     * Default Constructor
     * @param parent the parent
     */
    SKGShow(QWidget *parent = 0);

    /**
     * Default Destructor
     */
    virtual ~SKGShow();

    /**
     * Get the current state
     * @return a string containing all activated item identifiers (separated by ;)
     */
    virtual QString getState();

    /**
     * Set the current state
     * @param iState a string containing all activated item identifiers (separated by ;)
     */
    virtual void setState(const QString& iState);

    /**
     * Set the default state
     * @param iState a string containing all activated item identifiers (separated by ;)
     */
    virtual void setDefaultState(const QString& iState);

    /**
     * Get the current where clause
     * @return a where clause string
     */
    virtual QString getWhereClause();

    /**
     * @brief Add an item to the menu
     *
     * @param iIdentifier unique identifier of the item
     * @param iText text
     * @param iIcon icon Defaults to "".
     * @param iWhereClose icon Defaults to "".
     * @param iListIdToCheckWhenChecked list of item identifiers (separated by ;) to check when checked Defaults to "".
     * @param iListIdToUncheckWhenChecked list of item identifiers (separated by ;) to uncheck when unchecked Defaults to "".
     * @param iListIdToCheckWhenUnchecked list of item identifiers (separated by ;) to check when checked Defaults to "".
     * @param iListIdToUncheckWhenUnchecked list of item identifiers (separated by ;) to uncheck when unchecked Defaults to "".
     * @return void
     **/
    virtual void addItem(const QString& iIdentifier, const QString& iText, const QString& iIcon = "",
                         const QString& iWhereClose = "",
                         const QString& iListIdToCheckWhenChecked = "",
                         const QString& iListIdToUncheckWhenChecked = "",
                         const QString& iListIdToCheckWhenUnchecked = "",
                         const QString& iListIdToUncheckWhenUnchecked = ""
                        );

    /**
     * @brief Add a separator
     *
     * @return void
     **/
    virtual void addSeparator();

Q_SIGNALS:
    /**
     * @brief Emitted when an item is changed
     *
     * @return void
     **/
    void stateChanged();

private Q_SLOTS:
    void itemChanged();

private:
    Q_DISABLE_COPY(SKGShow);
    QAction* getAction(const QString& iIdentifier);

    KMenu* m_menu;
    QTimer m_timer;
    QString m_defaultState;

    QMap<QAction*, QString> m_check_to_check;
    QMap<QAction*, QString> m_uncheck_to_check;
    QMap<QAction*, QString> m_check_to_uncheck;
    QMap<QAction*, QString> m_uncheck_to_uncheck;
    QMap<QAction*, QString> m_whereclause;
};

#endif // SKGSHOW_H
