/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file defines classes SKGImportMt940.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportmt940.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgbankincludes.h"
#include "skgobjectbase.h"
#include "skgimportexportmanager.h"

#include <QFile>

#include <klocale.h>

SKGError SKGImportMt940::importFile(SKGImportExportManager* iImporter, SKGDocumentBank* iDocument)
{
    if(!iImporter || !iDocument) return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));
    SKGError err;
    SKGTRACEINRC(2, "SKGImportMt940::importFile", err);
    QString iFileName = iImporter->getFileName();
    SKGTRACEL(10) << "Input filename=" << iFileName << endl;
    //Begin transaction
    if(iDocument) {
        err = iDocument->beginTransaction("#INTERNAL#", 2);
        if(err.isSucceeded()) {
            //Open file
            if(err.isSucceeded()) {
                QFile file(iFileName);
                if(!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
                    err.setReturnCode(ERR_INVALIDARG);
                    err.setMessage(i18nc("Error message",  "Open file '%1' failed", iFileName));
                } else {

                    //Read lines
                    QStringList lines;
                    {
                        QTextStream stream(&file);
                        if(!iImporter->getCodec().isEmpty()) stream.setCodec(iImporter->getCodec().toAscii().constData());
                        while(!stream.atEnd()) {
                            //Read line
                            QString line = stream.readLine().trimmed();
                            if(!line.isEmpty())  lines.push_back(line);
                        }
                    }
                    //close file
                    file.close();

                    //Step 1 done
                    if(err.isSucceeded()) err = iDocument->stepForward(1);

                    //Read lines
                    bool importActivated = false;
                    bool in86 = false;
                    SKGAccountObject account;
                    SKGOperationObject operation;
                    SKGUnitObject unit;

                    int nb = lines.count();
                    if(err.isSucceeded()) err = iDocument->beginTransaction("#INTERNAL#", nb);
                    for(int i = 0; i < nb && err.isSucceeded(); ++i) {
                        //Read line
                        QString line = lines.at(i);
                        if(!line.isEmpty()) {
                            if(line.startsWith(QLatin1String(":20:"))) {
                                importActivated = true;
                                in86 = false;
                            } else if(importActivated) {
                                if(line.startsWith(QLatin1String(":25:"))) {
                                    //Account
                                    QStringList vals = SKGServices::splitCSVLine(line.right(line.count() - 4), '/');
                                    QString name = "MT940";
                                    QString number = name;
                                    if(vals.count() == 2) {
                                        name = vals.at(0);
                                        number = vals.at(1);
                                    } else if(vals.count() == 1) {
                                        name = vals.at(0);
                                    }

                                    //Search if account is already existing
                                    SKGObjectBase::SKGListSKGObjectBase listAccount;
                                    err = iDocument->getObjects("v_account", "t_number='" + number + '\'', listAccount);
                                    if(err.isSucceeded()) {
                                        if(listAccount.count() == 1) {
                                            //Yes ! Only one account found
                                            account = listAccount.at(0);
                                            err = iDocument->sendMessage(i18nc("An information message",  "Using account '%1' for import", account.getName()));
                                        } else {
                                            if(listAccount.count() > 1) err = iDocument->sendMessage(i18nc("An information message",  "More than one possible account found."));

                                            SKGBankObject bank(iDocument);
                                            if(err.isSucceeded()) err = bank.setName(name);
                                            if(err.isSucceeded()) err = bank.setNumber(name);
                                            if(err.isSucceeded() && bank.load().isFailed()) err = bank.save();
                                            if(err.isSucceeded()) err = bank.addAccount(account);
                                            if(err.isSucceeded()) err = account.setName(name);
                                            if(err.isSucceeded()) err = account.setNumber(number);
                                            if(err.isSucceeded()) err = account.setType(SKGAccountObject::CURRENT);
                                            if(err.isSucceeded() && account.load().isFailed()) err = account.save();
                                            if(err.isSucceeded()) err = iDocument->sendMessage(i18nc("An information message",  "Default account '%1' created for import", name));
                                        }
                                    }

                                    in86 = false;
                                } else if(line.startsWith(QLatin1String(":28C:"))) {
                                    in86 = false;
                                } else if(line.startsWith(QLatin1String(":60F:")) || line.startsWith(QLatin1String(":60M:"))) {
                                    //Example C040802EUR16,40
                                    if(account.getNbOperation() > 1) {
                                        err = iDocument->sendMessage(i18nc("An information message", "The initial balance of '%1' has not been set because some operations are already existing", account.getName()));
                                    } else {
                                        //Set initial balance
                                        QString val = line.right(line.count() - 5);

                                        err = SKGUnitObject::createCurrencyUnit(iDocument, val.mid(7, 3), unit);
                                        if(err.isSucceeded()) err = account.setInitialBalance((val[0] == 'C' ? 1.0 : -1.0) * SKGServices::stringToDouble(val.right(val.count() - 10)), unit);
                                        if(err.isSucceeded()) err = account.save();
                                        if(err.isSucceeded()) err = iDocument->sendMessage(i18nc("An information message", "The initial balance of '%1' has been set with OFX file content", account.getName()));
                                    }

                                    in86 = false;
                                } else if(line.startsWith(QLatin1String(":62F:")) || line.startsWith(QLatin1String(":62M:"))) {
                                    in86 = false;
                                } else if(line.startsWith(QLatin1String(":61:"))) {
                                    //Example :61:0712280103D000000000200,00FMSCNONREF
                                    QString val = line.right(line.count() - 4);
                                    QDate date = QDate::fromString(val.left(6), "yyMMdd");
                                    if(date.year() < 1970) date = date.addYears(100);

                                    int index = (val[10] == 'R' ? 11 : 10);
                                    double sign = (val[index] == 'C' ? 1.0 : -1.0);
                                    ++index;

                                    if(val[index] == 'R')  ++index;

                                    QString amountString;
                                    while(true) {
                                        if((val[index] >= '0' && val[index] <= '9') || val[index] == '-' || val[index] == ',') {
                                            amountString += val[index];
                                            ++index;
                                        } else {
                                            break;
                                        }
                                    }

                                    err = account.addOperation(operation);
                                    if(err.isSucceeded()) err = operation.setDate(date);
                                    if(err.isSucceeded()) err = operation.setUnit(unit);
                                    if(err.isSucceeded()) err = operation.setAttribute("t_imported", "T");
                                    //if(err.isSucceeded()) err = operation.setImportID("MT940");
                                    if(err.isSucceeded()) err = operation.save();

                                    SKGSubOperationObject subop;
                                    err = operation.addSubOperation(subop);
                                    if(err.isSucceeded()) err = subop.setQuantity(sign * SKGServices::stringToDouble(amountString));
                                    if(err.isSucceeded()) err = subop.save();

                                    in86 = false;
                                } else if(line.startsWith(QLatin1String(":NS:")) && !operation.exist()) {
                                    QString val = line.right(line.count() - 4);
                                    QString comment = account.getComment();
                                    comment += (comment.isEmpty() || val.isEmpty() ? "" : " ") + val;
                                    if(err.isSucceeded()) err = account.setComment(comment);
                                    if(err.isSucceeded()) err = account.save();
                                } else if(line.startsWith(QLatin1String(":86:")) || line.startsWith(QLatin1String(":NS:"))) {
                                    QString val = line.right(line.count() - 4);
                                    QString comment = operation.getComment();
                                    comment += (comment.isEmpty() || val.isEmpty() ? "" : " ") + val;
                                    if(err.isSucceeded()) err = operation.setComment(comment);
                                    if(err.isSucceeded()) err = operation.save();

                                    in86 = true;
                                } else if(in86) {
                                    QString val = line;
                                    QString comment = operation.getComment();
                                    comment += (comment.isEmpty() || val.isEmpty() ? "" : " ") + val;
                                    if(err.isSucceeded()) err = operation.setComment(comment);
                                    if(err.isSucceeded()) err = operation.save();
                                }
                            }
                        }
                        if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                    }
                    if(err.isSucceeded()) err = iDocument->endTransaction(true);
                    else  iDocument->endTransaction(false);

                    //Step 2 done
                    if(err.isSucceeded()) err = iDocument->stepForward(2);
                }
            }
        }
        if(err.isSucceeded()) err = iDocument->endTransaction(true);
        else  iDocument->endTransaction(false);
    }

    return err;
}

#include "skgimportmt940.moc"
