/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file defines classes SKGImportGsb.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportgsb.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgbankincludes.h"
#include "skgobjectbase.h"
#include "skgimportexportmanager.h"
#include "skgpayeeobject.h"

#include <klocale.h>
#include <kfilterdev.h>

#include <QDomDocument>

SKGError SKGImportGsb::importFile(SKGImportExportManager* iImporter, SKGDocumentBank* iDocument)
{
    if(!iImporter || !iDocument) return SKGError(ERR_ABORT, i18nc("Error message", "Invalid parameters"));
    SKGError err;
    SKGTRACEINRC(2, "SKGImportGsb::importFile", err);
    QString iFileName = iImporter->getFileName();
    SKGTRACEL(10) << "Input filename=" << iFileName << endl;

    //Initialisation
    //Open file
    QIODevice* file = KFilterDev::deviceForFile(iFileName, "application/x-gzip");
    if(!file->open(QIODevice::ReadOnly)) {
        err.setReturnCode(ERR_INVALIDARG);
        err.setMessage(i18nc("Error message",  "Open file '%1' failed", iFileName));
    } else {
        QDomDocument doc;

        //Set the file without uncompression
        QString errorMsg;
        int errorLine = 0;
        int errorCol = 0;
        bool contentOK = doc.setContent(file->readAll(), &errorMsg, &errorLine, &errorCol);
        file->close();

        //Get root
        QDomElement docElem = doc.documentElement();

        if(!contentOK) {
            err.setReturnCode(ERR_ABORT);
            err.setMessage(i18nc("Error message",  "%1-%2: '%3'", errorLine, errorCol, errorMsg));
        } else {
            //Check version
            QDomElement general = docElem.firstChildElement("General");
            if(general.isNull()) {
                err.setReturnCode(ERR_ABORT);
                err.setMessage(i18nc("Error message",  "Bad version of Grisbi document. Version must be >= 0.6.0"));
                contentOK = false;
            }
        }

        if(!contentOK) {
            err.addError(ERR_INVALIDARG, i18nc("Error message",  "Invalid XML content in file '%1'", iFileName));
        } else {
            err = iDocument->beginTransaction("#INTERNAL#", 9);

            QMap<QString, SKGUnitObject> mapIdUnit;
            QMap<QString, SKGBankObject> mapIdBank;
            QMap<QString, SKGAccountObject> mapIdAccount;
            QMap<QString, SKGCategoryObject> mapIdCategory;
            QMap<QString, SKGOperationObject> mapIdOperation;
            QMap<QString, SKGPayeeObject> mapIdPayee;
            QMap<QString, QString> mapIdMode;

            //Step 1-Create units
            if(err.isSucceeded()) {
                QDomNodeList currencyList = docElem.elementsByTagName("Currency");
                int nb = currencyList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get account object
                    QDomElement currency = currencyList.at(i).toElement();

                    //Creation of the units
                    SKGUnitObject unitObj(iDocument);
                    if(err.isSucceeded()) err = unitObj.setName(getAttribute(currency,  "Na"));
                    if(err.isSucceeded()) err = unitObj.setSymbol(getAttribute(currency,  "Co"));
                    if(err.isSucceeded()) err = unitObj.setNumberDecimal(SKGServices::stringToInt(getAttribute(currency,  "Fl")));
                    if(err.isSucceeded()) err = unitObj.save();

                    mapIdUnit[getAttribute(currency,  "Nb")] = unitObj;

                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(1);

            //Step 2-Create banks
            if(err.isSucceeded()) {
                QDomNodeList bankList = docElem.elementsByTagName("Bank");
                int nb = bankList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get account object
                    QDomElement bank = bankList.at(i).toElement();

                    //Creation of the banks
                    SKGBankObject bankObj(iDocument);
                    if(err.isSucceeded()) err = bankObj.setName(getAttribute(bank,  "Na"));
                    if(err.isSucceeded()) err = bankObj.setNumber(getAttribute(bank,  "BIC"));
                    if(err.isSucceeded()) err = bankObj.save();

                    mapIdBank[getAttribute(bank,  "Nb")] = bankObj;

                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(2);

            //Step 3-Create accounts
            SKGBankObject bankDefault(iDocument);
            if(err.isSucceeded()) err = bankDefault.setName("GRISBI");
            if(err.isSucceeded()) err = bankDefault.save();
            if(err.isSucceeded()) {
                QDomNodeList accountList = docElem.elementsByTagName("Account");
                int nb = accountList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get account object
                    QDomElement account = accountList.at(i).toElement();

                    //Creation of the account
                    SKGAccountObject accountObj;
                    SKGBankObject bank = mapIdBank[getAttribute(account,  "Bank")];
                    if(!bank.exist()) bank = bankDefault;
                    if(err.isSucceeded()) err = bank.addAccount(accountObj);
                    if(err.isSucceeded()) err = accountObj.setName(getAttribute(account,  "Name"));
                    if(err.isSucceeded()) err = accountObj.setNumber(getAttribute(account,  "Bank_account_number"));
                    if(err.isSucceeded()) err = accountObj.setComment(getAttribute(account,  "Comment"));
                    if(err.isSucceeded()) err = accountObj.setAgencyAddress(getAttribute(account,  "Owner_address"));
                    if(err.isSucceeded()) err = accountObj.setClosed(getAttribute(account,  "Closed_account") != "0");
                    if(err.isSucceeded()) err = accountObj.save();
                    if(err.isSucceeded()) {
                        accountObj.setClosed(false); //To be sure that the modification is possible. NOT SAVED
                        err = accountObj.setInitialBalance(SKGServices::stringToDouble(getAttribute(account,  "Initial_balance")), mapIdUnit[getAttribute(account,  "Currency")]);
                    }

                    mapIdAccount[getAttribute(account,  "Number")] = accountObj;

                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(3);

            //Step 4-Get payees
            if(err.isSucceeded()) {
                QDomNodeList partyList = docElem.elementsByTagName("Party");
                int nb = partyList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get payee object
                    QDomElement party = partyList.at(i).toElement();
                    SKGPayeeObject payeeObject;
                    err = SKGPayeeObject::createPayee(iDocument, getAttribute(party,  "Na"), payeeObject);
                    mapIdPayee[getAttribute(party,  "Nb")] = payeeObject;

                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(4);

            //Step 5-Get payement mode
            if(err.isSucceeded()) {
                QDomNodeList paymentList = docElem.elementsByTagName("Payment");
                int nb = paymentList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get mode object
                    QDomElement payment = paymentList.at(i).toElement();
                    mapIdMode[getAttribute(payment,  "Number")] = getAttribute(payment,  "Name");

                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(5);

            //Step 6-Create categories
            if(err.isSucceeded()) {
                QDomNodeList categoryList = docElem.elementsByTagName("Category");
                int nb = categoryList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get account object
                    QDomElement category = categoryList.at(i).toElement();

                    //Creation of the categories
                    SKGCategoryObject catObj(iDocument);
                    if(err.isSucceeded()) err = catObj.setName(getAttribute(category,  "Na"));
                    if(err.isSucceeded()) err = catObj.save();

                    mapIdCategory[getAttribute(category,  "Nb")] = catObj;

                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(6);

            //Step 7-Create subcategories
            if(err.isSucceeded()) {
                QDomNodeList categoryList = docElem.elementsByTagName("Sub_category");
                int nb = categoryList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get account object
                    QDomElement category = categoryList.at(i).toElement();

                    //Creation of the subcategories
                    SKGCategoryObject catParentObj = mapIdCategory[getAttribute(category,  "Nbc")];
                    SKGCategoryObject catObj;
                    if(err.isSucceeded()) err = catParentObj.addCategory(catObj);
                    if(err.isSucceeded()) err = catObj.setName(getAttribute(category,  "Na"));
                    if(err.isSucceeded()) err = catObj.save();

                    QString id = SKGServices::intToString(10000 * SKGServices::stringToInt(getAttribute(category,  "Nbc")) + SKGServices::stringToInt(getAttribute(category,  "Nb")));
                    mapIdCategory[id] = catObj;

                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(7);

            //Step 8-Create transaction
            if(err.isSucceeded()) {
                QDomNodeList transactionList = docElem.elementsByTagName("Transaction");
                int nb = transactionList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get account object
                    QDomElement transaction = transactionList.at(i).toElement();

                    //Creation of the operation
                    SKGAccountObject account = mapIdAccount[getAttribute(transaction,  "Ac")];
                    SKGOperationObject opObj;
                    if(err.isSucceeded()) err = account.addOperation(opObj);
                    if(err.isSucceeded()) err = opObj.setDate(QDate::fromString(getAttribute(transaction,  "Dt"), "MM/dd/yyyy"));
                    if(err.isSucceeded()) err = opObj.setUnit(mapIdUnit[ getAttribute(transaction,  "Cu")]);
                    if(err.isSucceeded()) err = opObj.setPayee(mapIdPayee[ getAttribute(transaction,  "Pa")]);
                    if(err.isSucceeded()) err = opObj.setMode(mapIdMode[getAttribute(transaction,  "Pn")]);
                    if(err.isSucceeded()) err = opObj.setNumber(SKGServices::stringToInt(getAttribute(transaction,  "Pc")));
                    if(err.isSucceeded()) err = opObj.setComment(getAttribute(transaction,  "No"));
                    if(err.isSucceeded()) err = opObj.setImported(true);
                    if(err.isSucceeded()) err = opObj.setImportID("GSB-" + getAttribute(transaction,  "Nb"));
                    if(err.isSucceeded()) err = opObj.setStatus(getAttribute(transaction,  "Re") == "0" ? SKGOperationObject::NONE : SKGOperationObject::CHECKED);
                    if(err.isSucceeded()) err = opObj.setGroupOperation(mapIdOperation[getAttribute(transaction,  "Trt")]);
                    if(err.isSucceeded()) err = opObj.save();

                    SKGSubOperationObject subObj;
                    if(err.isSucceeded()) err = opObj.addSubOperation(subObj);
                    QString id = SKGServices::intToString(10000 * SKGServices::stringToInt(getAttribute(transaction,  "Ca")) + SKGServices::stringToInt(getAttribute(transaction,  "Sca")));
                    if(err.isSucceeded()) err = subObj.setCategory(mapIdCategory[id]);
                    if(err.isSucceeded()) err = subObj.setComment(getAttribute(transaction,  "No"));
                    if(err.isSucceeded()) err = subObj.setQuantity(SKGServices::stringToDouble(getAttribute(transaction,  "Am")));
                    if(err.isSucceeded()) err = subObj.save();

                    mapIdOperation[getAttribute(transaction,  "Nb")] = opObj;

                    if(err.isSucceeded() && i % 100 == 0) err = iDocument->executeSqliteOrder("ANALYZE");
                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(8);

            //Step 9-Create scheduled transaction
            if(err.isSucceeded()) {
                QDomNodeList scheduledList = docElem.elementsByTagName("Scheduled");
                int nb = scheduledList.count();
                err = iDocument->beginTransaction("#INTERNAL#", nb);
                for(int i = 0; err.isSucceeded() && i < nb; ++i) {
                    //Get account object
                    QDomElement transaction = scheduledList.at(i).toElement();

                    //Creation of the operation
                    SKGAccountObject account = mapIdAccount[getAttribute(transaction,  "Ac")];
                    SKGOperationObject opObj;
                    bool isClosed = account.isClosed();
                    if(isClosed) err = account.setClosed(false);
                    if(err.isSucceeded()) err = account.addOperation(opObj);
                    if(err.isSucceeded() && isClosed) err = account.setClosed(true);
                    QDate firstDate = QDate::fromString(getAttribute(transaction,  "Dt"), "MM/dd/yyyy");
                    if(err.isSucceeded()) err = opObj.setDate(firstDate);
                    if(err.isSucceeded()) err = opObj.setUnit(mapIdUnit[ getAttribute(transaction,  "Cu")]);
                    if(err.isSucceeded()) err = opObj.setPayee(mapIdPayee[ getAttribute(transaction,  "Pa")]);
                    if(err.isSucceeded()) err = opObj.setMode(mapIdMode[getAttribute(transaction,  "Pn")]);
                    if(err.isSucceeded()) err = opObj.setNumber(SKGServices::stringToInt(getAttribute(transaction,  "Pc")));
                    if(err.isSucceeded()) err = opObj.setComment(getAttribute(transaction,  "No"));
                    if(err.isSucceeded()) err = opObj.setImported(true);
                    if(err.isSucceeded()) err = opObj.setImportID("GSB-" + getAttribute(transaction,  "Nb"));
                    if(err.isSucceeded()) err = opObj.setTemplate(true);
                    if(err.isSucceeded()) err = opObj.save();

                    SKGSubOperationObject subObj;
                    if(err.isSucceeded()) err = opObj.addSubOperation(subObj);
                    QString id = SKGServices::intToString(10000 * SKGServices::stringToInt(getAttribute(transaction,  "Ca")) + SKGServices::stringToInt(getAttribute(transaction,  "Sca")));
                    if(err.isSucceeded()) err = subObj.setCategory(mapIdCategory[id]);
                    if(err.isSucceeded()) err = subObj.setComment(getAttribute(transaction,  "No"));
                    if(err.isSucceeded()) err = subObj.setQuantity(SKGServices::stringToDouble(getAttribute(transaction,  "Am")));
                    if(err.isSucceeded()) err = subObj.save();

                    QString Tra = getAttribute(transaction,  "Tra");
                    if(Tra != "0") {
                        //This is a transfer
                        SKGOperationObject opObj2;
                        if(err.isSucceeded()) err = opObj.duplicate(opObj2, opObj.getDate(), true);
                        if(err.isSucceeded()) err = opObj2.setImported(true);
                        if(err.isSucceeded()) err = opObj2.setImportID("GSB-" + getAttribute(transaction,  "Nb") + "_TR");
                        if(err.isSucceeded()) err = opObj2.save();

                        SKGObjectBase::SKGListSKGObjectBase subObjs2;
                        if(err.isSucceeded()) err = opObj2.getSubOperations(subObjs2);
                        if(err.isSucceeded() && subObjs2.count()) {
                            SKGSubOperationObject subObj2 = subObjs2.at(0);
                            err = subObj2.setQuantity(-subObj.getQuantity());
                            err = subObj2.save();
                        }

                        //Group operations
                        if(err.isSucceeded()) err = opObj.setGroupOperation(opObj2);
                        if(err.isSucceeded()) err = opObj.save();
                    }

                    //Create the schedule
                    SKGRecurrentOperationObject recuObj;
                    if(err.isSucceeded()) err = opObj.addRecurrentOperation(recuObj);
                    if(err.isSucceeded()) err = recuObj.setAutoWriteDays(0);
                    if(err.isSucceeded()) err = recuObj.autoWriteEnabled(getAttribute(transaction,  "Au") != "0");
                    if(err.isSucceeded()) {
                        //text_frequency [] = { _("Once"), _("Weekly"), _("Monthly"), _("two months"), ("trimester"), _("Yearly"), _("Custom"), NULL };
                        int occu = 1;
                        SKGRecurrentOperationObject::PeriodUnit period = SKGRecurrentOperationObject::DAY;
                        QString Pe = getAttribute(transaction,  "Pe");
                        if(Pe == "0") {
                            period = SKGRecurrentOperationObject::MONTH;
                            if(err.isSucceeded()) err = recuObj.timeLimit(true);
                            if(err.isSucceeded()) err = recuObj.setTimeLimit(1);

                        } else if(Pe == "1") {
                            occu = 7;
                            period = SKGRecurrentOperationObject::DAY;
                        } else if(Pe == "2") {
                            period = SKGRecurrentOperationObject::MONTH;
                        } else if(Pe == "3") {
                            occu = 2;
                            period = SKGRecurrentOperationObject::MONTH;
                        } else if(Pe == "4") {
                            occu = 3;
                            period = SKGRecurrentOperationObject::MONTH;
                        } else if(Pe == "5") {
                            period = SKGRecurrentOperationObject::YEAR;
                        } else if(Pe == "6") {
                            //text_frequency_user [] = { _("Days"), _("Weeks"), _("Months"), _("Years"), NULL };
                            occu = SKGServices::stringToInt(getAttribute(transaction,  "Pep"));
                            QString Pei = getAttribute(transaction,  "Pei");
                            if(Pei == "0") period = SKGRecurrentOperationObject::DAY;
                            else if(Pei == "1") {
                                period = SKGRecurrentOperationObject::DAY;
                                occu *= 7;
                            } else if(Pei == "2") period = SKGRecurrentOperationObject::MONTH;
                            else period = SKGRecurrentOperationObject::YEAR;
                        }

                        if(err.isSucceeded()) err = recuObj.setPeriodUnit(period);
                        if(err.isSucceeded()) err = recuObj.setPeriodIncrement(occu);

                        QString Dtl = getAttribute(transaction,  "Dtl");
                        if(err.isSucceeded() && !Dtl.isEmpty()) {
                            if(err.isSucceeded()) err = recuObj.timeLimit(true);
                            if(err.isSucceeded()) err = recuObj.setTimeLimit(QDate::fromString(Dtl, "MM/dd/yyyy"));
                        }
                    }
                    if(err.isSucceeded()) err = recuObj.save();

                    if(err.isSucceeded() && i % 100 == 0) err = iDocument->executeSqliteOrder("ANALYZE");
                    if(err.isSucceeded()) err = iDocument->stepForward(i + 1);
                }

                if(err.isSucceeded()) err = iDocument->endTransaction(true);
                else  iDocument->endTransaction(false);
            }
            if(err.isSucceeded()) err = iDocument->stepForward(9);

            if(err.isSucceeded()) err = iDocument->endTransaction(true);
            else  iDocument->endTransaction(false);

            if(err.isSucceeded()) err = iDocument->executeSqliteOrder("ANALYZE");
        }
    }

    delete file;
    return err;
}


QString SKGImportGsb::getAttribute(const QDomElement& iElement, const QString& iAttribute)
{
    QString val = iElement.attribute(iAttribute);
    if(val == "(null)") val = "";
    return val;
}

#include "skgimportgsb.moc"
