/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGDOCUMENTBANK_H
#define SKGDOCUMENTBANK_H
/** @file
 * This file defines classes SKGDocumentBank.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgdocument.h"
#include "skgdefinebank.h"
#include "skgbankmodeler_export.h"

class SKGUnitObject;
class SKGUnitValueObject;

/**
 * This class manages skg bank documents
 */
class SKGBANKMODELER_EXPORT SKGDocumentBank : public SKGDocument
{
    Q_OBJECT;
    Q_CLASSINFO("D-Bus Interface", "org.skrooge.SKGDocumentBank");

public:
    /**
    * Constructor
    */
    explicit SKGDocumentBank();

    /**
    * Destructor
    */
    virtual ~SKGDocumentBank();

public Q_SLOTS:
    /**
     * Close the current transaction.
     * A transaction is needed to modify the SKGDocument.
     * This transaction is also used to manage the undo/redo.
     * @see beginTransaction
     * @param succeedded : true to indicate that current transaction has been successfully executed
     *                   : false to indicate that current transaction has failed
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError endTransaction(bool succeedded);

    /**
     * dump the document in the std output.
     * It's useful for debug.
     * @param iMode to select what you want to dump.
     * @code
     * document->dump (DUMPUNIT|DUMPPARAMETERS);
     * @endcode
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError dump(int iMode = DUMPBANKOBJECT);

    /**
     * Create or modify an account
     * @param iName account name
     * @param iNumber account number
     * @param iBankName name of the bank
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError addOrModifyAccount(const QString& iName, const QString& iNumber, const QString& iBankName);

    /**
     * Create or modify the value of an unit
     * @param iUnitName unit name
     * @param iDate date
     * @param iValue unit value for the date @p iDate
     * @param oValue this output unit value
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError addOrModifyUnitValue(const QString& iUnitName, const QDate& iDate, double iValue, SKGUnitValueObject* oValue = NULL);

    /**
     * Get formated money
     * @param iValue value
     * @param iUnit unit
     * @return formated value in red or black
     */
    virtual QString formatMoney(double iValue, SKGServices::SKGUnitInfo iUnit) const;

    /**
        * Get formated percentage
        * @param iValue value
        * @return formated value in red or black
        */
    virtual QString formatPercentage(double iValue) const;

    /**
     * Get Primary unit. WARNING: This value can be not uptodated in a transaction.
     * @return Primary unit.
     */
    virtual SKGServices::SKGUnitInfo getPrimaryUnit();

    /**
     * Get Secondary unit. WARNING: This value can be not uptodated in a transaction.
     * @return Secondary unit.
     */
    virtual SKGServices::SKGUnitInfo getSecondaryUnit();

    /**
     * Refresh all views and indexes in the database
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError refreshViewsIndexesAndTriggers();

    /**
     * Return the number version of views, indexes and triggers
     * @return
     */
    virtual QString getViewsIndexesAndTriggersVersion();

    /**
     * Get display schemas
    * @param iRealTable the real table name
     * @return list of schemas
     */
    virtual QList<SKGDocument::SKGModelTemplate> getDisplaySchemas(const QString& iRealTable) const;

    /**
     * Get the display string for any modeler object (table, attribute)
     * @param iString the name of the object (example: v_operation, v_unit.t_name)
     * @return the display string
     */
    virtual QString getDisplay(const QString& iString) const;

    /**
     * Get the real attribute
     * @param iString the name of the attribute (something like t_BANK)
     * @return the real attribute (something like bank.rd_bank_id.t_name)
     */
    virtual QString getRealAttribute(const QString& iString) const;

    /**
     * Get the icon for attribute
     * @param iString the name of the attribute
     * @return the icon
     */
    virtual QIcon getIcon(const QString& iString) const;

    /**
     * Get the attribute type
     * @param iAttributeName the name of an attribute
     * @return the type
     */
    virtual SKGServices::AttributeType getAttributeType(const QString& iAttributeName) const;

    /**
     * Get the file extension for this kind of document (must be overwritten)
     * @return file extension (like skg)
     */
    virtual QString getFileExtension() const;

    /**
     * Get the header of the file (useful for magic mime type).
     * @return document header
     */
    virtual QString getDocumentHeader() const;

    /**
     * Get budget report
     * @param iMonth the month
     * @return the html report
     */
    virtual QString getBudget(const QString& iMonth);

    /**
     * Get 5 main category report
     * @param iMonth the month
     * @param iChart to add chart
     * @return the html report
     */
    virtual QString get5MainCategories(const QString& iMonth, bool iChart = true);

    /**
     * Get 5 main variation of categories report
     * @param iMonth the month
     * @param iPreviousMonth the previous month
     * @param iOnlyIssues only "Expenses increased" and "Incomes decreased"
     * @param oCategoryList to get the category for each variation
     * @return the list of variation string
     */
    QStringList get5MainCategoriesVariationList(const QString& iMonth, const QString& iPreviousMonth, bool iOnlyIssues = false, QStringList* oCategoryList = NULL);

    /**
     * Get 5 main variation of categories report
     * @param iMonth the month
     * @param iPreviousMonth the previous month
     * @param iOnlyIssues only "Expenses increased" and "Incomes decreased"
     * @return the html report
     */
    QString get5MainCategoriesVariation(const QString& iMonth, const QString& iPreviousMonth, bool iOnlyIssues = false);

protected:
    /**
     * Get impacted tables if one object of @p iTable is modifier.
     * @param iTable name of a table
     * @return impacted tables
     */
    virtual QStringList getImpactedTable(const QString& iTable) const;

    /**
     * Migrate the current SKGDocument to the latest version of the data model.
     * WARNING: This method must be used in a transaction.
     * @see beginTransaction
     * @param oMigrationDone to know if a migration has been done or not.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError migrate(bool& oMigrationDone);

    /**
     * Compute balance of each operation.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError computeBalances();
    /**
     * Compute the budgets categories links.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError computeBudgetCategoriesLinks();

private Q_SLOTS:
    virtual void refreshCache(const QString& iTable);

private:
    Q_DISABLE_COPY(SKGDocumentBank);

    QString m_5mainVariations_inputs;
    QStringList m_5mainVariations_cache;
    QStringList m_5mainVariationsCat_cache;
};
#endif
