/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGBUDGETOBJECTT_H
#define SKGBUDGETOBJECTT_H
/** @file
 * This file defines classes SKGBudgetObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgobjectbase.h"
#include "skgerror.h"
#include "skgbankmodeler_export.h"

class SKGCategoryObject;
class SKGDocumentBank;

/**
 * This class is a budget
 */
class SKGBANKMODELER_EXPORT SKGBudgetObject : public SKGObjectBase
{
    Q_OBJECT;
public:
    /**
    * Constructor
    * @param iDocument the document containing the object
    * @param iID the identifier in @p iTable of the object
    */
    explicit SKGBudgetObject(SKGDocument* iDocument = NULL, int iID = 0);

    /**
     * Destructor
     */
    virtual ~SKGBudgetObject();

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    SKGBudgetObject(const SKGBudgetObject& iObject);

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    SKGBudgetObject(const SKGObjectBase& iObject);

    /**
     * Operator affectation
     * @param iObject the object to copy
     */
    virtual const SKGBudgetObject& operator= (const SKGObjectBase &iObject);

    /**
     * Set year of the budget
     * @param iYear the year
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setYear(int iYear);

    /**
     * Get year of the budget
     * @return year of the budget
     */
    virtual int getYear() const;

    /**
     * Set month of the budget
     * @param iMonth the month
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setMonth(int iMonth);

    /**
     * Get month of the budget
     * @return month of the budget
     */
    virtual int getMonth() const;

    /**
     * Set the category
     * @param iCategory the category
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setCategory(const SKGCategoryObject& iCategory);

    /**
     * Remove the category
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError removeCategory();

    /**
     * Get the category
     * @param oCategory the category
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getCategory(SKGCategoryObject& oCategory) const;

    /**
     * Enable / disable the inclusion of sub categories
     * @param iEnable condition
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError enableSubCategoriesInclusion(bool iEnable);

    /**
     * To know if sub categories inclusion is enabled or disabled
     * @return condition
     */
    virtual int isSubCategoriesInclusionEnabled() const;

    /**
     * Set budgeted amount of the budget
     * @param iAmount the budgeted amount
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setBudgetedAmount(double iAmount);

    /**
     * Get budgeted amount of the budget
     * @return budgeted amount of the budget
     */
    virtual double getBudgetedAmount() const;

    /**
     * Get delta
     * @return delta
     */
    virtual double getDelta() const;

    /**
     * Process all rules
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError process();

    /**
     * Create automatically budget items based on existing operations
     * @param iDocument the document where to create*
     * @param iYear year of budget
     * @param iBaseYear year of the base for computation
     * @param iRemovePreviousBudget remove existing budget for @param iYear
     * @param iBalancingMonthly create a balancing budget for each month
     * @param iBalancingAnnual create a balancing budget for each @param iYear
     * @return an object managing the error
     *   @see SKGError
     */
    static SKGError createAutomaticBudget(SKGDocumentBank* iDocument, int iYear, int iBaseYear,
                                          bool iRemovePreviousBudget = false, bool iBalancingMonthly = false, bool iBalancingAnnual = false);

private:
    virtual QString getWhereclauseId() const;
};
/**
 * Declare the class
 */
Q_DECLARE_TYPEINFO(SKGBudgetObject, Q_MOVABLE_TYPE);
#endif
