/***************************************************************************
*   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
***************************************************************************/
/** @file
* This file defines classes SKGBudgetObject.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
*/
#include "skgbudgetobject.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"
#include "skgcategoryobject.h"
#include "skgbudgetruleobject.h"

SKGBudgetObject::SKGBudgetObject(SKGDocument* iDocument, int iID)
    : SKGObjectBase(iDocument, "v_budget", iID)
{
}

SKGBudgetObject::~SKGBudgetObject()
{
}

SKGBudgetObject::SKGBudgetObject(const SKGBudgetObject& iObject)
    : SKGObjectBase(iObject)
{
}

SKGBudgetObject::SKGBudgetObject(const SKGObjectBase& iObject)
{
    if(iObject.getRealTable() == "budget") {
        copyFrom(iObject);
    } else {
        *this = SKGObjectBase(iObject.getDocument(), "v_budget", iObject.getID());
    }
}

const SKGBudgetObject& SKGBudgetObject::operator= (const SKGObjectBase & iObject)
{
    copyFrom(iObject);
    return *this;
}

QString SKGBudgetObject::getWhereclauseId() const
{
    //Could we use the id
    QString output = SKGObjectBase::getWhereclauseId();
    if(output.isEmpty()) {
        output = "i_year=" + getAttribute("i_year") + " AND "
                 "i_month=" + getAttribute("i_month") + " AND "
                 "rc_category_id=" + (getAttribute("rc_category_id").isEmpty() ? "0" : getAttribute("rc_category_id"));
    }
    return output;
}

SKGError SKGBudgetObject::setBudgetedAmount(double iAmount)
{
    SKGError err = setAttribute("f_budgeted", SKGServices::doubleToString(iAmount));
    if(err.isSucceeded()) err = setAttribute("f_budgeted_modified", SKGServices::doubleToString(iAmount));
    return err;
}

double SKGBudgetObject::getBudgetedAmount() const
{
    return SKGServices::stringToDouble(getAttribute("f_budgeted"));
}

double SKGBudgetObject::getDelta() const
{
    return SKGServices::stringToDouble(getAttribute("f_DELTABEFORETRANSFER"));
}

SKGError SKGBudgetObject::setYear(int iYear)
{
    return setAttribute("i_year", SKGServices::intToString(iYear));
}

int SKGBudgetObject::getYear() const
{
    return SKGServices::stringToInt(getAttribute("i_year"));
}

SKGError SKGBudgetObject::setMonth(int iMonth)
{
    return setAttribute("i_month", SKGServices::intToString(iMonth));
}

int SKGBudgetObject::getMonth() const
{
    return SKGServices::stringToInt(getAttribute("i_month"));
}

SKGError SKGBudgetObject::setCategory(const SKGCategoryObject& iCategory)
{
    return setAttribute("rc_category_id", SKGServices::intToString(iCategory.getID()));;
}

SKGError SKGBudgetObject::getCategory(SKGCategoryObject& oCategory) const
{
    return getDocument()->getObject("v_category", "id=" + getAttribute("rc_category_id"), oCategory);

}

SKGError SKGBudgetObject::enableSubCategoriesInclusion(bool iEnable)
{
    return setAttribute("t_including_subcategories", iEnable ? "Y" : "N");
}

int SKGBudgetObject::isSubCategoriesInclusionEnabled() const
{
    return (getAttribute("t_including_subcategories") == "Y");
}

SKGError SKGBudgetObject::removeCategory()
{
    return setAttribute("rc_category_id", "0");
}

SKGError SKGBudgetObject::process()
{
    SKGError err;
    SKGTRACEINRC(10, "SKGBudgetObject::process", err);
    //Main values
    int m = getMonth();
    int y = getYear();
    double budgeted = getBudgetedAmount();
    double transferred = SKGServices::stringToDouble(getAttribute("f_transferred"));

    //Get budgets rules ordered
    SKGObjectBase::SKGListSKGObjectBase budgetsRules;
    QString sql = "(t_year_condition='N' OR i_year=" + SKGServices::intToString(y) + ") AND "
                  "(t_month_condition='N' OR i_month=" + SKGServices::intToString(m) + ") AND "
                  "(t_category_condition='N' OR rc_category_id=" + getAttribute("rc_category_id") + ") "
                  "ORDER BY t_absolute DESC, id";
    err = getDocument()->getObjects("v_budgetrule", sql, budgetsRules);

    int nb = budgetsRules.count();
    if(err.isSucceeded() && nb) {
        err = getDocument()->beginTransaction("#INTERNAL#", nb);
        for(int i = 0; err.isSucceeded() && i < nb; ++i) {
            SKGBudgetRuleObject rule = budgetsRules.at(0);

            //Do we have something to do
            SKGBudgetRuleObject::Condition cond = rule.getCondition();
            double delta = getDelta();
            double quantity = rule.getQuantity();
            if(delta) {
                if(cond == SKGBudgetRuleObject::ALL || (delta < 0 && cond == SKGBudgetRuleObject::NEGATIVE) || (delta > 0 && cond == SKGBudgetRuleObject::POSITIVE)) {
                    //Compute value to transfer
                    double value = (rule.isAbolute() ? (cond == SKGBudgetRuleObject::NEGATIVE ? qMax(delta, quantity) : qMin(delta, quantity)) : quantity * delta / 100.0);

                    //Get impacted budget
                    SKGBudgetObject impactedBudget;
                    SKGBudgetRuleObject::Mode mode = rule.getTransferMode();
                    if(mode == SKGBudgetRuleObject::NEXT) {
                        //Next
                        impactedBudget = *this;
                        err = impactedBudget.resetID();
                        if(m == 0) {
                            //Yearly budget
                            ++y;
                        } else {
                            //Monthly budget
                            ++m;
                            if(m == 13) {
                                m = 1;
                                ++y;
                            }
                        }
                        if(err.isSucceeded()) err = impactedBudget.setYear(y);
                        if(err.isSucceeded()) err = impactedBudget.setMonth(m);
                    } else if(mode == SKGBudgetRuleObject::YEAR) {
                        //Year
                        if(err.isSucceeded()) err = impactedBudget.setMonth(0);
                    }

                    //Change category
                    if(err.isSucceeded() && rule.isCategoryChangeEnabled()) {
                        SKGCategoryObject transferCategory;
                        rule.getTransferCategory(transferCategory);
                        err = impactedBudget.setCategory(transferCategory);
                    }

                    if(err.isSucceeded() && impactedBudget.exist()) {
                        err = impactedBudget.load();
                        QString newBudget = SKGServices::doubleToString(budgeted - value);
                        if(err.isSucceeded()) err = impactedBudget.setAttribute("f_budgeted_modified", newBudget);
                        if(err.isSucceeded()) err = impactedBudget.save();

                        transferred += value;
                        if(err.isSucceeded()) err = setAttribute("f_transferred", SKGServices::doubleToString(transferred));
                        if(err.isSucceeded()) err = save();
                    }
                    //SKGBudgetObject budget(iDocument, id);
                }
            }
            if(err.isSucceeded()) err = getDocument()->stepForward(i + 1);
        }

        if(err.isSucceeded()) err = getDocument()->endTransaction(true);
        else  getDocument()->endTransaction(false);
    }

    return err;
}

SKGError SKGBudgetObject::createAutomaticBudget(SKGDocumentBank* iDocument, int iYear, int iBaseYear,
        bool iRemovePreviousBudget, bool iBalancingMonthly, bool iBalancingAnnual)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGBudgetObject::createAutomaticBudget", err);
    QString baseYear = SKGServices::intToString(iBaseYear);
    int fistMonth = 0;
    if(iDocument) {
        SKGStringListList listTmp;
        err = iDocument->executeSelectSqliteOrder(
                  "SELECT MIN(STRFTIME('%m', d_date)) FROM operation WHERE i_group_id = 0 AND STRFTIME('%Y', d_date) = '" + baseYear +
                  "' AND t_template='N'",
                  listTmp);
        if(listTmp.count() == 2)  fistMonth = SKGServices::stringToInt(listTmp.at(1).at(0));
    }
    if(err.isSucceeded() && iDocument) {
        SKGStringListList listTmp;
        QString sql = "SELECT t_REALCATEGORY, COUNT(TOT),"
                      "100*COUNT(TOT)/((CASE WHEN STRFTIME('%Y', date('now'))<>'" + baseYear + "' THEN 12 ELSE STRFTIME('%m', date('now'))-1 END)-" +
                      SKGServices::intToString(fistMonth) + "+1) AS CPOUR,"
                      "ROUND(TOTAL(TOT)/COUNT(TOT)), MAX(MONTH), TOTAL(TOT) FROM ("
                      "SELECT t_REALCATEGORY, STRFTIME('%m', d_date) AS MONTH, TOTAL(f_REALCURRENTAMOUNT) AS TOT "
                      "FROM v_operation_consolidated WHERE i_group_id = 0 AND d_DATEYEAR = '" + baseYear + "' AND d_DATEMONTH<STRFTIME('%Y-%m', date('now')) "
                      "GROUP BY t_REALCATEGORY, d_DATEMONTH"
                      ") GROUP BY t_REALCATEGORY ORDER BY COUNT(TOT) DESC, (MAX(TOT)-MIN(TOT))/ABS(ROUND(TOTAL(TOT)/COUNT(TOT))) ASC, ROUND(TOTAL(TOT)/COUNT(TOT)) ASC";
        err = iDocument->executeSelectSqliteOrder(sql, listTmp);

        double sumBudgeted = 0;
        double sumOps = 0;

        int nbval = listTmp.count();
        if(err.isSucceeded() && iDocument) {
            int step = 0;
            err = iDocument->beginTransaction("#INTERNAL#", nbval - 1 + 4);

            //Remove previous budgets
            if(iRemovePreviousBudget) {
                if(err.isSucceeded()) err = iDocument->executeSqliteOrder("DELETE FROM budget WHERE i_year=" + SKGServices::intToString(iYear));
                ++step;
                if(err.isSucceeded()) err = iDocument->stepForward(step);
            }

            //Create budgets
            for(int i = 1; err.isSucceeded() && i < nbval; ++i) {  //Ignore header
                //Get values
                QString catName = listTmp.at(i).at(0);
                int count = SKGServices::stringToInt(listTmp.at(i).at(1));
                int countPourcent = SKGServices::stringToInt(listTmp.at(i).at(2));
                double amount = SKGServices::stringToDouble(listTmp.at(i).at(3));
                int month = SKGServices::stringToInt(listTmp.at(i).at(4));
                sumOps += SKGServices::stringToDouble(listTmp.at(i).at(5));

                if(!catName.isEmpty() && (countPourcent > 85 || count == 1)) {
                    SKGCategoryObject cat;;
                    err = iDocument->getObject("v_category", "t_fullname = '" + SKGServices::stringToSqlString(catName) + '\'', cat);
                    for(int m = fistMonth; err.isSucceeded() && m <= (count == 1 ? fistMonth : 12); ++m) {
                        SKGBudgetObject budget(iDocument);
                        err = budget.setBudgetedAmount(amount);
                        if(err.isSucceeded()) err = budget.setYear(iYear);
                        if(err.isSucceeded()) err = budget.setMonth(count == 1 ? month : m);
                        if(err.isSucceeded()) err = budget.setCategory(cat);
                        if(err.isSucceeded()) err = budget.save();

                        sumBudgeted += amount;
                    }
                }
                ++step;
                if(err.isSucceeded()) err = iDocument->stepForward(step);
            }

            //Monthly balancing
            if(err.isSucceeded() && iBalancingMonthly) {
                for(int m = 1; err.isSucceeded() && m <= 12; ++m) {
                    SKGStringListList listTmp;
                    err = iDocument->executeSelectSqliteOrder("SELECT TOTAL(f_budgeted) FROM budget WHERE i_year=" + SKGServices::intToString(iYear) + " AND i_month=" + SKGServices::intToString(m), listTmp);
                    if(err.isSucceeded() && listTmp.count() == 2) {
                        SKGBudgetObject budget(iDocument);
                        err = budget.setBudgetedAmount(-SKGServices::stringToDouble(listTmp.at(1).at(0)));
                        if(err.isSucceeded()) err = budget.setYear(iYear);
                        if(err.isSucceeded()) err = budget.setMonth(m);
                        if(err.isSucceeded()) err = budget.save();
                    }

                }
            }
            ++step;
            if(err.isSucceeded()) err = iDocument->stepForward(step);

            //Annual balancing
            if(err.isSucceeded() && sumOps != sumBudgeted && iBalancingAnnual) {
                SKGBudgetObject budget(iDocument);
                err = budget.setBudgetedAmount(sumOps - sumBudgeted);
                if(err.isSucceeded()) err = budget.setYear(iYear);
                if(err.isSucceeded()) err = budget.setMonth(0);
                if(err.isSucceeded()) err = budget.save();
            }
            ++step;
            if(err.isSucceeded()) err = iDocument->stepForward(step);

            //Analyze
            if(err.isSucceeded()) err = iDocument->executeSqliteOrder("ANALYZE");;
            ++step;
            if(err.isSucceeded()) err = iDocument->stepForward(step);

            if(err.isSucceeded()) err = iDocument->endTransaction(true);
            else  iDocument->endTransaction(false);
        }
    }
    return err;
}

#include "skgbudgetobject.moc"
