/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file defines classes SKGObjectModel.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectmodel.h"
#include "skgdocumentbank.h"
#include "skgaccountobject.h"
#include "skgcategoryobject.h"
#include "skgpayeeobject.h"
#include "skgoperationobject.h"
#include "skgtrackerobject.h"
#include "skgrecurrentoperationobject.h"
#include "skgunitobject.h"
#include "skgunitvalueobject.h"
#include "skgtransactionmng.h"
#include "skgtraces.h"
#include "skgmainpanel.h"

#include <kicon.h>
#include <kstandarddirs.h>
#include <klocalizedstring.h>
#include <klocale.h>
#include <kconfiggroup.h>

#include <QApplication>
#include <QDir>
#include <QMimeData>
#include <QFont>
#include <qcolor.h>

SKGObjectModel::SKGObjectModel ( SKGDocumentBank* iDocument,
                                 const QString& iTable,
                                 const QString& iWhereClause,
                                 QObject *parent,
                                 const QString& iParentAttribute,
                                 bool iResetOnCreation )
        : SKGObjectModelBase ( iDocument, iTable, iWhereClause, parent, iParentAttribute, false )
{
    SKGTRACEIN ( 1, "SKGObjectModel::SKGObjectModel" );


    isResetRealyNeeded=iResetOnCreation;
    refresh();
}

void SKGObjectModel::buidCache()
{
    SKGObjectModelBase::buidCache();
    operationTable= ( getRealTable() =="operation" );
    payeeTable= ( getRealTable() =="payee" );
    trackerTable= ( getRealTable() =="refund" );
    recurrentoperationTable= ( getRealTable() =="recurrentoperation" );
    accountTable= ( getRealTable() =="account" );
    unitTable= ( getRealTable() =="unit" );
    unitvalueTable= ( getRealTable() =="unitvalue" );
    suboperationTable= ( getTable() =="v_operation_consolidated" );
    ruleTable= ( getRealTable() =="rule" );
    categoryTable= ( getRealTable() =="category" );
    interestTable= ( getRealTable() =="interest" );

    if ( unitvalueTable ) {
        SKGUnitValueObject unitValObject=getObject ( index ( 0,0 ) );
        SKGUnitObject unitObject;
        unitValObject.getUnit ( unitObject );
        SKGUnitObject parent;
        unitObject.getUnit ( parent );
        if ( parent.exist() ) {
            cacheUnit.Name=parent.getName();
            cacheUnit.Symbol=parent.getSymbol();
            cacheUnit.Value=1;
            cacheUnit.NbDecimal=parent.getNumberDecimal();
        }
        else
        {
            cacheUnit.Name="";
            cacheUnit.Symbol="";
            cacheUnit.Value=1;
            cacheUnit.NbDecimal=2;
        }
        if (cacheUnit.Symbol.isEmpty()) cacheUnit.Symbol=' ';

        //Bug 209672 vvvv
        if (unitObject.getType()==SKGUnitObject::INDEX) {
            cacheUnit.Symbol=' ';
            cacheUnit.Name=' ';
        }
        //Bug 209672 ^^^^

    }

    if ( operationTable ) {
        //Read Setting
        KSharedConfigPtr config=KSharedConfig::openConfig ();
        KConfigGroup pref=config->group ( "skrooge_operation" );
        fontFutureOperationsColor = qVariantFromValue (pref.readEntry ( "fontFutureColor",QColor ( Qt::gray ) ));
        fontNotValidatedOperationsColor = qVariantFromValue (pref.readEntry ( "fontNotValidatedColor",QColor ( Qt::blue ) ));
        fontSubOperationsColor = qVariantFromValue (pref.readEntry ( "fontSubOperationColor",QColor ( Qt::green ) ));
    }

    if ( recurrentoperationTable ) {
        //Read Setting
        KSharedConfigPtr config=KSharedConfig::openConfig ();
        KConfigGroup pref=config->group ( "skrooge_scheduled" );
        fontDisabledScheduleColor = qVariantFromValue (pref.readEntry ( "fontFutureColor",QColor ( Qt::gray ) ));
    }

    if ( operationTable || recurrentoperationTable || accountTable || ruleTable) {
        icon_favorite=KIcon ( "rating" );
    }

    if ( operationTable || recurrentoperationTable ) {
        icon_skrooge_transfer=KIcon ( "skrooge_transfer" );
        icon_skrooge_split=KIcon ( "skrooge_split" );
        icon_imported=KIcon ( "utilities-file-archiver" );
        {
            QStringList overlay;
            overlay.push_back("dialog-ok");
            icon_imported_checked=KIcon ( "utilities-file-archiver", NULL, overlay);
        }

        icon_recurrent=KIcon ( "chronometer" );
        {
            QStringList overlay;
            overlay.push_back ( "rating" );
            icon_recurrent_master=KIcon ( "chronometer", NULL, overlay );
        }
    }

    if (unitTable) {

        icon_skrooge_much_more=KIcon ( "skrooge_much_more" );
        icon_skrooge_much_less=KIcon ( "skrooge_much_less" );
        icon_skrooge_more=KIcon ( "skrooge_more" );
        icon_skrooge_less=KIcon ( "skrooge_less" );
    }

    if (ruleTable) {
        icon_search=KIcon ( "edit-find");
        icon_update=KIcon ( "view-refresh");
        icon_alarm=KIcon ( "dialog-warning");
    }
    icon_closed=KIcon ( "dialog-close");

    if (categoryTable) {
        icon_category=KIcon ( "skrooge_category");
        {
            QStringList overlay;
            overlay.push_back ( "list-remove" );
            icon_category_moins=KIcon ( "skrooge_category", NULL, overlay );
        }
        {
            QStringList overlay;
            overlay.push_back ( "list-add" );
            icon_category_plus=KIcon ( "skrooge_category", NULL, overlay );
        }
    }
    fontRedColor = qVariantFromValue (qVariantFromValue ( QColor ( Qt::red ) ));
}

SKGObjectModel::~SKGObjectModel()
{
    SKGTRACEIN ( 1, "SKGObjectModel::~SKGObjectModel" );
}

QVariant SKGObjectModel::headerData ( int section, Qt::Orientation orientation, int role ) const
{
    _SKGTRACEIN ( 10, "SKGObjectModelBase::headerData" );

    if ( orientation==Qt::Horizontal ) {
        if ( role == Qt::DisplayRole ) {
            QString att;
            if ( section>=0 && section< listAttibutes.count() ) att=listAttibutes[section];
            else att=SKGServices::intToString ( section );

            if ( att=="t_bookmarked" || att=="i_NBRECURRENT" || att=="t_status" || att=="t_close" || att=="t_imported" ) return "";
        }
    }
    return SKGObjectModelBase::headerData ( section, orientation, role );
}

QVariant SKGObjectModel::data ( const QModelIndex &index, int role ) const
{
    if ( !index.isValid() ) return QVariant();
    _SKGTRACEIN ( 10, "SKGObjectModel::data" );

    switch ( role ) {
    case Qt::DisplayRole:
    case Qt::EditRole:
    case Qt::UserRole: {
        SKGObjectBase* obj = getObjectPointer ( index );
        QString att=listAttibutes[index.column() ];
        QString val=obj->getAttribute ( att );
        if ( ( ( operationTable || recurrentoperationTable || trackerTable || accountTable || ruleTable) &&
                ( att=="t_bookmarked" || att=="t_status" || att=="i_NBRECURRENT" || att=="t_imported" ) )
                || ( att=="t_close" )
                || ( att=="t_action_type")
           ) {
            if ( role==Qt::UserRole ) {
                if ( ruleTable && att=="t_action_type" ) {
                    SKGObjectBase* obj = getObjectPointer ( index );
                    if (obj) {
                        QString val=obj->getAttribute ( att );
                        if ( val=="S" ) return i18nc("Noun, a search", "Search");
                        else if ( val=="U" ) return i18nc("Noun, a modification", "Update");
                        else return i18nc("Noun, an alarm", "Alarm");
                    }
                }
                return val;
            } else return "";
        } else if ( accountTable && att=="t_type" ) {
            if ( val=="C" ) return i18nc ( "Noun, a type of account", "Current" );
            else if ( val=="D" ) return i18nc ( "Noun, a type of account","Credit card" );
            else if ( val=="A" ) return i18nc ( "Noun, a type of account","Assets" );
            else if ( val=="I" ) return i18nc ( "Noun, a type of account", "Investment" );
            else return i18nc ( "Noun", "Other" );
        } else if ( unitTable && att=="t_type" ) {
            if ( val=="1" ) return i18nc ( "Noun", "Primary currency" );
            else if ( val=="2" ) return i18nc ( "Noun", "Secondary currency" );
            else if ( val=="C" ) return i18nc ( "Noun", "Currency" );
            else if ( val=="S" ) return i18nc ( "Noun", "Share" );
            else if ( val=="I" ) return i18nc ( "Noun, example CAC40, DOWJONES", "Index" );
            else return i18nc ( "Noun","Object" );
        } else if ( interestTable && att=="t_expenditure_value_date_mode" ) {
            if ( val=="0" ) return i18nc ( "Noun","Day -0" );
            else if ( val=="1" ) return i18nc ( "Noun","Day -1" );
            else if ( val=="2" ) return i18nc ( "Noun","Day -2" );
            else if ( val=="3" ) return i18nc ( "Noun","Day -3" );
            else if ( val=="4" ) return i18nc ( "Noun","Day -4" );
            else if ( val=="5" ) return i18nc ( "Noun","Day -5" );
            else return i18nc ( "Noun","Fifteen" );
        } else if ( interestTable && att=="t_income_value_date_mode" ) {
            if ( val=="0" ) return i18nc ( "Noun","Day +0" );
            else if ( val=="1" ) return i18nc ( "Noun","Day +1" );
            else if ( val=="2" ) return i18nc ( "Noun","Day +2" );
            else if ( val=="3" ) return i18nc ( "Noun","Day +3" );
            else if ( val=="4" ) return i18nc ( "Noun","Day +4" );
            else if ( val=="5" ) return i18nc ( "Noun","Day +5" );
            else return i18nc ( "Noun","Fifteen" );
        } else if ( getAttributeType ( index.column() ) ==SKGServices::FLOAT ) {
            double dval=SKGServices::stringToDouble ( val );
            if ( role == Qt::DisplayRole ) {
                if ( att.endsWith(QLatin1String("_INCOME")) ||
                        att.endsWith(QLatin1String("_EXPENSE")) ||
                        (categoryTable && (att=="f_REALCURRENTAMOUNT" ||
                                           att=="f_SUMCURRENTAMOUNT"))) {
                    double dval=SKGServices::stringToDouble ( val );
                    if ( dval==0 ) return "";
                }

                SKGServices::SKGUnitInfo unit;
                unit.Symbol="";
                unit.NbDecimal=2;
                if ( !att.contains("QUANTITY")) {
                    unit= static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit();
                    if ( unitvalueTable && !cacheUnit.Symbol.isEmpty() ) {
                        unit=cacheUnit;
                    }
                } else {
                    if (att=="f_QUANTITYOWNED") {
                        unit.NbDecimal=SKGServices::stringToInt(obj->getAttribute ("i_nbdecimal"));
                    } else {
                        unit.Symbol=obj->getAttribute ("t_UNIT");
                    }
                }

                //Bug 209672 vvvv
                if (unitTable) {
                    if (obj->getAttribute ("t_type")=="I") unit.Symbol=' ';
                }
                //Bug 209672 ^^^

                if (QString::compare(att, "f_rate", Qt::CaseInsensitive)==0) {
                    unit.Symbol='%';
                    unit.NbDecimal=2;
                } else if (att=="f_coef") {
                    unit.Symbol="";
                    unit.NbDecimal=2;
                }

                if (unit.Symbol.length()==0) unit.Symbol=' ';

                return KGlobal::locale()->formatMoney ( dval, unit.Symbol, unit.NbDecimal );
            } else {
                return dval;
            }
        } else if ( getAttributeType ( index.column() ) ==SKGServices::INTEGER ) {
            if ( recurrentoperationTable && att=="i_nb_times" ) {
                QString t_times=obj->getAttribute ( "t_times" );
                if ( t_times!="Y" ) return QChar(8734);
            } else if ( (att=="i_number" || att=="i_NBOPERATIONS" || att=="i_SUMNBOPERATIONS") && val=="0" ) return "";

            return SKGServices::stringToInt ( val );
        }

        break;
    }
    case Qt::DecorationRole: {
        //decoration
        SKGObjectBase* obj = getObjectPointer ( index );
        QString att=listAttibutes[index.column() ];
        if ( operationTable || recurrentoperationTable ) {
            if ( att=="t_mode" ) {
                if ( obj->getAttribute ( "i_group_id" ) !="0" ) return icon_skrooge_transfer;
            } else if ( att=="t_bookmarked") {
                if (obj->getAttribute ( "t_bookmarked" ) =="Y" )  return icon_favorite;
            } else if ( att=="t_CATEGORY" ) {
                if ( SKGServices::stringToInt ( obj->getAttribute ( "i_NBSUBCATEGORY" ) ) >1 ) return icon_skrooge_split;
            } else if ( att=="i_NBRECURRENT" && operationTable ) {
                if ( obj->getAttribute ( "i_NBRECURRENT" ) !="0" ) return icon_recurrent_master;
                else if ( obj->getAttribute ( "r_recurrentoperation_id" ) !="0" ) return icon_recurrent;
            } else if ( att=="t_imported") {
                QString impStatus=obj->getAttribute ( "t_imported" );
                if (impStatus=="Y") return icon_imported;
                else if (impStatus=="P")  return icon_imported_checked;
            } else if (att=="t_REFUND" || att=="t_REALREFUND") {
                SKGObjectBase* obj = getObjectPointer ( index );
                QString trackerName;
                if ( obj ) trackerName=obj->getAttribute(att);
                if (!trackerName.isEmpty()) {
                    SKGTrackerObject tracker=SKGObjectBase(getDocument(), "refund"); //Better performance if v_refund is not used
                    tracker.setName(trackerName);
                    tracker.load();

                    if (tracker.isClosed()) return icon_closed;
                }
            }
        } else if ( ruleTable ) {
            if (att=="t_action_type") {
                QString val=obj->getAttribute ( att );
                if ( val=="S" ) return icon_search;
                else if ( val=="U" ) return icon_update;
                else return icon_alarm;
            } else if ( att=="t_bookmarked") {
                if (obj->getAttribute ( "t_bookmarked" ) =="Y" )  return icon_favorite;
            }
        } else if ( unitTable ) {
            if ( att=="f_CURRENTAMOUNT" ) {
                SKGUnitObject unit=*obj;
                double annualchange=365*unit.getDailyChange();
                if ( annualchange>=15 ) return icon_skrooge_much_more;
                else if ( annualchange<=-15 ) return icon_skrooge_much_less;
                else if ( annualchange>=5 ) return icon_skrooge_more;
                else if ( annualchange<=-5 ) return icon_skrooge_less;
            }
        } else if ( accountTable ) {
            if (att=="t_BANK" ) {
                QString t_icon=obj->getAttribute ( "t_icon" );
                if ( t_icon.isEmpty() ) t_icon=obj->getAttribute ( "t_ICON" );
                if ( !t_icon.isEmpty() ) {
                    QDir dirLogo ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/" ) ) );
                    return qVariantFromValue ( QIcon ( dirLogo.absoluteFilePath ( t_icon ) ) );
                }
            } else if ( att=="t_bookmarked") {
                if (obj->getAttribute ( "t_bookmarked" ) =="Y" )  return icon_favorite;
            }
        } else if ( categoryTable) {
            if (index.column() == 0 )
            {
                QString t_TYPEEXPENSE=obj->getAttribute ( "t_TYPEEXPENSE" );
                if ( t_TYPEEXPENSE=="-" ) return icon_category_moins;
                else if ( t_TYPEEXPENSE=="+" ) return icon_category_plus;
                return icon_category;
            }
        }
        break;
    }
    case Qt::TextColorRole: {
        //Text color
        SKGObjectBase* obj = getObjectPointer ( index );
        QString att=listAttibutes[index.column() ];
        if ( recurrentoperationTable ) {
            if ( obj->getAttribute ( "i_nb_times" )=="0") return  fontDisabledScheduleColor;
            else if ( att=="d_date" && SKGServices::stringToTime ( obj->getAttribute ( "d_date" ) ).date() <=QDate::currentDate() ) return fontRedColor;
        } else if ( operationTable ) {
            if ( SKGServices::stringToTime ( obj->getAttribute ( "d_date" ) ).date() >QDate::currentDate() ) return fontFutureOperationsColor;
            else if (getAttributeType(index.column())!=SKGServices::FLOAT) {
                if ( obj->getAttribute ( "t_imported" )=="P") return  fontNotValidatedOperationsColor;
                else if (suboperationTable) return fontSubOperationsColor;
            }
        }
        break;
    }
    case Qt::TextAlignmentRole: {
        //Text alignment
        if (recurrentoperationTable) {
            QString att=listAttibutes[index.column() ];

            if (att=="i_auto_write_days" || att=="i_warn_days" || att=="i_nb_times" ) {
                return (int) (Qt::AlignVCenter|Qt::AlignLeft);
            }
        }
        break;
    }

    case Qt::CheckStateRole: {
        //CheckState
        SKGObjectBase* obj = getObjectPointer ( index );
        if ( obj ) {
            QString att=listAttibutes[index.column() ];
            if ( operationTable && att=="t_status" ) {
                QString t_status=obj->getAttribute ( "t_status" );
                if ( t_status=="P" ) return Qt::PartiallyChecked;
                else if ( t_status=="Y" ) return Qt::Checked;
                else if ( t_status=="N" ) return Qt::Unchecked;
            } else if ( att=="t_close" ) {
                QString t_close=obj->getAttribute ( "t_close" );
                if ( t_close=="Y" ) return Qt::Checked;
                else return Qt::Unchecked;
            } else  if ( recurrentoperationTable && att=="i_auto_write_days" ) {
                QString t_auto_write=obj->getAttribute ( "t_auto_write" );
                if ( t_auto_write=="Y" ) return Qt::Checked;
                else return Qt::Unchecked;
            } else  if ( recurrentoperationTable && att=="i_warn_days" ) {
                QString t_auto_write=obj->getAttribute ( "t_warn" );
                if ( t_auto_write=="Y" ) return Qt::Checked;
                else return Qt::Unchecked;
            } else  if ( recurrentoperationTable && att=="i_nb_times" ) {
                QString t_times=obj->getAttribute ( "t_times" );
                if ( t_times=="Y" ) return Qt::Checked;
                else return Qt::Unchecked;
            }
        }
        break;
    }
    case Qt::ToolTipRole: {
        //Tooltip
        QString toolTipString;
        QString att=listAttibutes[index.column() ];
        if (getAttributeType ( index.column() ) ==SKGServices::FLOAT) {
            SKGObjectBase* obj = getObjectPointer ( index );
            if ( obj ) {

                //Add secondary unit
                if (!att.contains("QUANTITY")) {
                    SKGServices::SKGUnitInfo secondaryUnit= static_cast<SKGDocumentBank*>(getDocument())->getSecondaryUnit();
                    if ( !secondaryUnit.Symbol.isEmpty() ) {
                        double val=SKGServices::stringToDouble ( obj->getAttribute ( att ) );
                        if ((!att.endsWith(QLatin1String("_INCOME")) && !att.endsWith(QLatin1String("_EXPENSE"))) ||val!=0) {
                            if ( secondaryUnit.Value )  toolTipString=KGlobal::locale()->formatMoney ( val/secondaryUnit.Value, secondaryUnit.Symbol, secondaryUnit.NbDecimal );

                        }

                    }
                }

                //Add original amount
                QString originalAmount=obj->getProperty ( "SKG_OP_ORIGINAL_AMOUNT" );
                if ( !originalAmount.isEmpty() ) {
                    if ( !toolTipString.isEmpty() ) toolTipString+="\n\n";
                    double val1=SKGServices::stringToDouble ( obj->getAttribute ( "f_CURRENTAMOUNT" ) );
                    double val2=SKGServices::stringToDouble ( originalAmount );
                    double gain= ( val2!=0 ? 100.0* ( val1-val2 ) /val2 : 0 );
                    SKGServices::SKGUnitInfo primaryUnit=static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit();
                    toolTipString+=i18nc ( "Noun","Original amount=%1 (%2 %)",
                                           KGlobal::locale()->formatMoney ( val2, primaryUnit.Symbol, primaryUnit.NbDecimal ),
                                           gain );
                }

                //Add balance
                if (operationTable) {
                    SKGOperationObject op=*obj;
                    if (!op.isTemplate()) {
                        if ( !toolTipString.isEmpty() ) toolTipString+="\n\n";
                        SKGServices::SKGUnitInfo primaryUnit=static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit();
                        toolTipString+=i18nc ( "Noun","Account balance=%1",
                                               KGlobal::locale()->formatMoney ( op.getBalance(), primaryUnit.Symbol, primaryUnit.NbDecimal));
                    }
                }
            }
        } else if ( operationTable || recurrentoperationTable) {
            if ( att=="t_imported") {
                SKGOperationObject op=*(getObjectPointer ( index ));
                if (recurrentoperationTable) {
                    SKGRecurrentOperationObject rop=*(getObjectPointer ( index ));
                    rop.getParentOperation(op);
                }
                toolTipString=op.getImportID();
            } else if (att=="t_REFUND" || att=="t_REALREFUND") {
                SKGObjectBase* obj = getObjectPointer ( index );
                QString trackerName;
                if ( obj ) trackerName=obj->getAttribute(att);
                if (!trackerName.isEmpty()) {
                    SKGTrackerObject tracker=SKGObjectBase(getDocument(), "v_refund");
                    tracker.setName(trackerName);
                    tracker.load();
                    SKGServices::SKGUnitInfo unit= static_cast<SKGDocumentBank*>(getDocument())->getPrimaryUnit();
                    toolTipString=KGlobal::locale()->formatMoney ( tracker.getCurrentAmount(), unit.Symbol, unit.NbDecimal );
                }
            } else if ( att=="t_CATEGORY" ) {
                SKGOperationObject op=*(getObjectPointer ( index ));
                if (recurrentoperationTable) {
                    SKGRecurrentOperationObject rop=*(getObjectPointer ( index ));
                    rop.getParentOperation(op);
                }
                if ( SKGServices::stringToInt ( op.getAttribute ( "i_NBSUBCATEGORY" ) ) >1 ) {
                    SKGObjectBase::SKGListSKGObjectBase subOps;
                    op.getSubOperations(subOps);
                    foreach(const SKGObjectBase& subOp, subOps) {
                        toolTipString+=subOp.getDisplayName()+'\n';
                    }
                }
            } else if ( att=="t_mode" ) {
                SKGOperationObject op=*(getObjectPointer ( index ));
                if (recurrentoperationTable) {
                    SKGRecurrentOperationObject rop=*(getObjectPointer ( index ));
                    rop.getParentOperation(op);
                }
                if ( op.getAttribute ( "i_group_id" )!= "0" ) {
                    SKGOperationObject gop;
                    op.getGroupOperation(gop);

                    SKGObjectBase::SKGListSKGObjectBase gOps;
                    op.getGroupedOperations(gOps);
                    foreach(const SKGOperationObject& gOp, gOps) {
                        SKGAccountObject account;
                        gOp.getParentAccount(account);
                        toolTipString+=account.getDisplayName()+'\n';
                    }
                }
            }

            if ( operationTable && toolTipString.isEmpty() ) {
                SKGOperationObject op=*(getObjectPointer ( index ));
                if (op.getStatus()==SKGOperationObject::POINTED)
                {
                    toolTipString=i18nc("A tool tip", "This operation is pointed but not checked yet.");
                    toolTipString+='\n';
                    toolTipString+=i18nc("A tool tip", "You can use the reconciliation mode to validate pointed operations.");
                    toolTipString+='\n';
                    toolTipString+=i18nc("A tool tip", "Click to switch back status.");
                    toolTipString+='\n';
                    toolTipString+=i18nc("A tool tip", "Ctrl+click to force checked status.");
                }
                else if (op.getStatus()==SKGOperationObject::CHECKED)
                {
                    toolTipString=i18nc("A tool tip", "This operation is already checked.");
                }
                else if (op.getStatus()==SKGOperationObject::NONE)
                {
                    toolTipString=i18nc("A tool tip", "This operation is not pointed yet.");
                    toolTipString+='\n';
                    toolTipString+=i18nc("A tool tip", "Click to set pointed status.");
                    toolTipString+='\n';
                    toolTipString+=i18nc("A tool tip", "Ctrl+click to force checked status.");
                }
            }
        } else if ( ruleTable && att=="t_action_type" ) {
            SKGObjectBase* obj = getObjectPointer ( index );
            if (obj) {
                QString val=obj->getAttribute ( att );
                if ( val=="S" ) toolTipString=i18nc("Noun, a search", "Search");
                else if ( val=="U" ) toolTipString=i18nc("Noun, a modification", "Update");
                else toolTipString=i18nc("Noun, an alarm", "Alarm");
            }
        }

        QString toolTipStringBase=SKGObjectModelBase::data(index, role).toString();
        if (!toolTipStringBase.isEmpty())
        {
            if (!toolTipString.isEmpty()) toolTipString+="\n\n";
            toolTipString+=toolTipStringBase;
        }
        return toolTipString;
        break;
    }
    default: {
    }
    }
    return SKGObjectModelBase::data ( index, role );
}

bool SKGObjectModel::setData ( const QModelIndex & index, const QVariant& value, int role )
{
    if ( !index.isValid() ) return false;

    if ( role == Qt::CheckStateRole ) {
        SKGError err;
        if ( accountTable ) {
            SKGAccountObject obj=getObject ( index );
            SKGBEGINTRANSACTION ( *getDocument(),
                                  ( ( Qt::CheckState ) value.toInt() == Qt::Checked ? i18nc("Noun, name of the user action", "Close account '%1'", obj.getName() ) : i18nc("Noun, name of the user action", "Open account '%1'", obj.getName() ) ), err );
            err=obj.setClosed ( ( Qt::CheckState ) value.toInt() == Qt::Checked );
            if ( err.isSucceeded() ) err=obj.save();
        } else if ( trackerTable ) {
            SKGTrackerObject obj=getObject ( index );
            SKGBEGINTRANSACTION ( *getDocument(),
                                  ( ( Qt::CheckState ) value.toInt() == Qt::Checked ? i18nc("Noun, name of the user action", "Close tracker '%1'", obj.getName() ) : i18nc("Noun, name of the user action", "Open tracker '%1'", obj.getName() ) ), err );
            err=obj.setClosed ( ( Qt::CheckState ) value.toInt() == Qt::Checked );
            if ( err.isSucceeded() ) err=obj.save();
        } else if ( operationTable ) {
            SKGOperationObject obj=getObject ( index );
            SKGBEGINTRANSACTION ( *getDocument(), i18nc("Noun, name of the user action", "Change operation status" ), err );
            SKGOperationObject::OperationStatus statusinitial=obj.getStatus();
            SKGOperationObject::OperationStatus t_status=statusinitial;
            if ( QApplication::keyboardModifiers() &Qt::ControlModifier ) {
                //2747379: NONE ==> CHECKED, POINTED ==> CHECKED, CHECKED ==> CHECKED
                t_status= SKGOperationObject::CHECKED ;
                //t_status= ( t_status==SKGOperationObject::POINTED ? SKGOperationObject::NONE : ( t_status==SKGOperationObject::CHECKED ? SKGOperationObject::POINTED : SKGOperationObject::NONE ) );
            } else {
                //2747379: NONE ==> POINTED, POINTED ==> NONE, CHECKED ==> POINTED
                t_status= ( t_status==SKGOperationObject::NONE ? SKGOperationObject::POINTED : ( t_status==SKGOperationObject::POINTED ? SKGOperationObject::NONE : SKGOperationObject::POINTED ) );
                //t_status=(t_status==SKGOperationObject::POINTED ? SKGOperationObject::CHECKED : (t_status==SKGOperationObject::CHECKED ? SKGOperationObject::CHECKED : SKGOperationObject::POINTED ));
            }
            if ( t_status!=statusinitial ) {
                err=obj.setStatus ( t_status );
                if ( err.isSucceeded() ) err=obj.save();
            }
        } else if ( recurrentoperationTable ) {
            QString att=listAttibutes[index.column() ];

            //Get the real object, not the object from the view
            SKGObjectBase* objtmp=getObjectPointer ( index );
            if ( objtmp ) {
                SKGRecurrentOperationObject obj=SKGRecurrentOperationObject ( objtmp->getDocument(), objtmp->getID() );

                SKGBEGINTRANSACTION ( *getDocument(), i18nc("Noun, name of the user action", "Recurrent operation update" ), err );
                if ( att=="i_warn_days" ) {
                    err=obj.warnEnabled ( !obj.isWarnEnabled() );
                    if ( err.isSucceeded() ) err=obj.save();
                } else if ( att=="i_auto_write_days" ) {
                    err=obj.autoWriteEnabled ( !obj.isAutoWriteEnabled() );
                    if ( err.isSucceeded() ) err=obj.save();
                } else if ( att=="i_nb_times" ) {
                    err=obj.timeLimit ( !obj.hasTimeLimit() );
                    if ( err.isSucceeded() ) err=obj.save();
                }
            }
        }

        SKGMainPanel::displayErrorMessage ( err );
        return err.isSucceeded();
    }
    return QAbstractItemModel::setData ( index, value, role );
}

Qt::ItemFlags SKGObjectModel::flags ( const QModelIndex &index ) const
{
    _SKGTRACEIN ( 10, "SKGObjectModel::flags" );

    Qt::ItemFlags flags=SKGObjectModelBase::flags ( index );

    if ( categoryTable || payeeTable ) {
        if ( index.isValid() ) flags|=Qt::ItemIsDragEnabled | Qt::ItemIsDropEnabled;
        else flags|=Qt::ItemIsDropEnabled;
    } else if ( index.isValid() && operationTable ) {
        QString att=listAttibutes[index.column() ];
        if ( att=="t_status" && !suboperationTable ) flags|=Qt::ItemIsUserCheckable;
    } else if ( index.isValid() && recurrentoperationTable ) {
        QString att=listAttibutes[index.column() ];
        if ( att=="i_warn_days" || att=="i_auto_write_days" || att=="i_nb_times" ) flags|=Qt::ItemIsUserCheckable;
    } else if ( index.isValid() ) {
        QString att=listAttibutes[index.column() ];
        if ( att=="t_close" ) flags|=Qt::ItemIsUserCheckable;
    }

    return flags;
}

Qt::DropActions SKGObjectModel::supportedDragActions() const
{
    if ( categoryTable || payeeTable ) return Qt::MoveAction;
    return SKGObjectModelBase::supportedDragActions();
}

Qt::DropActions SKGObjectModel::supportedDropActions() const
{
    if ( categoryTable || payeeTable ) return Qt::MoveAction;
    return SKGObjectModelBase::supportedDropActions();
}

bool SKGObjectModel::dropMimeData ( const QMimeData *data,
                                    Qt::DropAction action,
                                    int row, int column,
                                    const QModelIndex &parent )
{
    if ( SKGObjectModelBase::dropMimeData ( data, action, row, column, parent ) ) return true;
    if ( action == Qt::IgnoreAction ) return true;
    if ( !data || !(data->hasFormat ( "application/skg.category.ids" ) || data->hasFormat ( "application/skg.payee.ids" )) ) return false;
    if ( column > 0 ) return false;

    SKGError err;
    if (data->hasFormat ( "application/skg.category.ids" ))
    {
        QByteArray encodedData = data->data ( "application/skg.category.ids" );
        QDataStream stream ( &encodedData, QIODevice::ReadOnly );
        QStringList newItems;

        SKGCategoryObject parentCategory;
        if ( parent.isValid() ) parentCategory=getObject ( parent );
        {
            SKGBEGINTRANSACTION ( *getDocument(), i18nc("Noun, name of the user action", "Move category" ), err );

            while ( !stream.atEnd() && err.isSucceeded() ) {
                int o_id;
                QString o_table;
                stream >> o_table;
                stream >> o_id;

                SKGCategoryObject child ( getDocument(), o_id );
                err=child.load();
                if ( err.isSucceeded() ) {
                    if ( parent.isValid() ) err=child.setParentCategory ( parentCategory );
                    else err=child.removeParentCategory();
                }
                if ( err.isSucceeded() ) err=child.save();
            }
        }
    }
    else
    {
        QByteArray encodedData = data->data ( "application/skg.payee.ids" );
        QDataStream stream ( &encodedData, QIODevice::ReadOnly );
        QStringList newItems;

        if ( parent.isValid() )
        {
            SKGBEGINTRANSACTION ( *getDocument(), i18nc("Noun, name of the user action", "Merge payee" ), err );
            SKGPayeeObject parentPayee=getObject ( parent );
            while ( !stream.atEnd() && err.isSucceeded() ) {
                int o_id;
                QString o_table;
                stream >> o_table;
                stream >> o_id;

                SKGPayeeObject child ( getDocument(), o_id );
                err=child.load();
                SKGObjectBase::SKGListSKGObjectBase ops;
                if ( err.isSucceeded() ) err=child.getOperations(ops);
                int nb=ops.count();
                for (int i=0; err.isSucceeded() && i<nb; ++i)
                {
                    SKGOperationObject op=ops.at(i);
                    err=op.setPayee(parentPayee);
                    if (err.isSucceeded()) err=op.save(true, false);
                }

                if ( err.isSucceeded() ) err=child.remove();
            }
        }
    }
    SKGMainPanel::displayErrorMessage ( err );
    return err.isSucceeded();
}

void SKGObjectModel::dataModified ( const QString& iTableName, int iIdTransaction )
{
    if ( getRealTable() ==iTableName || iTableName.isEmpty() || getRealTable() =="doctransaction" ) {
        SKGTRACEIN ( 1, "SKGObjectModel::dataModified" );
        if ( iTableName=="category" ) {
            //Full refresh
            isResetRealyNeeded=true;
            refresh();
        } else {
            SKGObjectModelBase::dataModified ( iTableName, iIdTransaction );
        }
    } else SKGObjectModelBase::dataModified ( iTableName, iIdTransaction );
}


#include "skgobjectmodel.moc"

