/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A table with graph with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtablewithgraph.h"
#include "skgservices.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skggraphicsscene.h"

#include <kmenu.h>
#include <kfiledialog.h>
#include <kstringhandler.h>

#include <QtGui/QHeaderView>
#include <QtGui/QGraphicsLineItem>
#include <QDomDocument>
#include <QRegExp>
#include <QPrinter>
#include <QPainter>
#include <QTableWidgetItem>
#include <QDesktopServices>
#include <QTimer>
#include <QScriptEngine>

#include <math.h>

/**
  * Data identifier for value
  */
#define DATA_VALUE 12
/**
  * Data identifier for color
  */
#define DATA_COLOR_H 11
/**
  * Data identifier for color
  */
#define DATA_COLOR_S 12
/**
  * Data identifier for color
  */
#define DATA_COLOR_V 13
/**
  * Data identifier for color
  */
#define DATA_COLOR_A 14
/**
  * Box size
  */
#define BOX_SIZE 120
/**
  * Box margin
  */
#define BOX_MARGIN 10

Qt::SortOrder SKGTableWithGraph::sortOrder=Qt::AscendingOrder;
int SKGTableWithGraph::sortColumn=0;

SKGTableWithGraph::SKGTableWithGraph ( QWidget *parent )
        : QWidget ( parent ), scene ( NULL ), additionalInformation ( NONE ), nbVirtualColumns ( 0 ),
        selectable ( true ), toolBarVisible ( true ), limitVisible ( true ), linearRegressionVisible(true),
        mainMenu ( NULL ),
        indexSum ( -1 ), indexAverage ( -1 ), indexMin ( -1 ), indexLinearRegression ( -1 )
{
    ui.setupUi ( this );

    timer = new QTimer ( this );
    timer->setSingleShot ( true );
    connect ( timer, SIGNAL ( timeout() ), this, SLOT ( refresh() ) );

    timerRedraw = new QTimer ( this );
    timerRedraw->setSingleShot ( true );
    connect ( timerRedraw, SIGNAL ( timeout() ), this, SLOT ( redrawGraph() ) );

    //Build contextual menu
    ui.kTable->setContextMenuPolicy ( Qt::CustomContextMenu );
    connect ( ui.kTable, SIGNAL ( customContextMenuRequested ( const QPoint & ) ),this,SLOT ( showMenu ( const QPoint& ) ) );

    mainMenu = new KMenu ( ui.kTable );

    QMenu* exp=mainMenu->addMenu ( i18nc ( "Verb, action to export items in another format","Export" ) );

    QAction* actCSV = exp->addAction ( KIcon ( "text-csv" ), i18nc ( "Noun, user action", "Export CSV..." ) );
    connect ( actCSV, SIGNAL ( triggered ( bool ) ), this, SLOT ( onExportCSV() ) );

    QAction* actTXT = exp->addAction ( KIcon ( "text-plain" ), i18nc ( "Noun, user action", "Export TXT..." ) );
    connect ( actTXT, SIGNAL ( triggered ( bool ) ), this, SLOT ( onExportTXT() ) );

    actShowToolBar = getGraphContextualMenu()->addAction ( i18nc ( "Noun, user action", "Show bottom tool bar" ) );
    if ( actShowToolBar )
    {
        actShowToolBar->setCheckable ( true );
        actShowToolBar->setChecked ( true );
        connect ( actShowToolBar, SIGNAL ( triggered ( bool ) ), this, SLOT ( onSwitchToolBarVisibility() ) );
    }

    actShowLimits = getGraphContextualMenu()->addAction ( i18nc ( "Noun, user action", "Show limits and average" ) );
    if ( actShowLimits )
    {
        actShowLimits->setCheckable ( true );
        actShowLimits->setChecked ( true );
        connect ( actShowLimits, SIGNAL ( triggered ( bool ) ), this, SLOT ( onSwitchLimits() ) );
    }

    actShowLinearRegression = getGraphContextualMenu()->addAction ( i18nc ( "Noun, user action", "Show linear regression" ) );
    if ( actShowLinearRegression )
    {
        actShowLinearRegression->setCheckable ( true );
        actShowLinearRegression->setChecked ( true );
        connect ( actShowLinearRegression, SIGNAL ( triggered ( bool ) ), this, SLOT ( onSwitchLinearRegression() ) );
    }

    //Set headers parameters
    QHeaderView* verticalHeader=ui.kTable->verticalHeader();
    if ( verticalHeader )  verticalHeader->hide();

    ui.kTable->setSortingEnabled ( false ); //sort must be enable for refresh method
    QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
    if ( horizontalHeader )
    {
        horizontalHeader->setResizeMode ( QHeaderView::ResizeToContents );
        horizontalHeader->show();
        horizontalHeader->setSortIndicatorShown ( true );
        horizontalHeader->setSortIndicator ( sortColumn, sortOrder );
        connect ( horizontalHeader, SIGNAL ( sortIndicatorChanged ( int, Qt::SortOrder ) ), this, SLOT ( refresh() ) );
    }
    ui.kTable->verticalHeader()->setDefaultSectionSize ( ui.kTable->verticalHeader()->minimumSectionSize() );

    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-bar-stacked" ), i18nc ( "Noun, a type of graph, with bars stacked upon each other","Stack" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-bar" ), i18nc ( "Noun, a type of graph, with bars placed besides each other","Histogram" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-pie" ), i18nc ( "Noun, a type of graph that looks like a sliced pie","Pie" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-ring" ), i18nc ( "Noun, a type of graph that looks like concentric slices of a pie (a la filelight)","Concentric pie" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-scatter" ), i18nc ( "Noun, a type of graph with only points","Point" ) );
    ui.kDisplayMode->addItem ( KIcon ( "skg-chart-line" ), i18nc ( "Noun, a type of graph with only lines","Line" ) );

    connect ( ui.kDisplayMode, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( redrawGraphDelayed() ), Qt::QueuedConnection );
    connect ( ui.kAllPositive, SIGNAL ( stateChanged ( int ) ), this, SLOT ( redrawGraphDelayed() ), Qt::QueuedConnection );
    connect ( ui.graphicView, SIGNAL ( resized() ), this, SLOT ( redrawGraphDelayed() ), Qt::QueuedConnection );
}


SKGTableWithGraph::~SKGTableWithGraph()
{
    if ( scene )
    {
        delete scene;
        scene=NULL;
    }

    mainMenu=NULL;
    timer=NULL;
    timerRedraw=NULL;
    actShowToolBar=NULL;
    actShowLimits=NULL;
    actShowLinearRegression=NULL;
}

void SKGTableWithGraph::setToolBarVisible ( bool iVisibility )
{
    toolBarVisible=iVisibility;
    ui.kGraphTypeFrm->setVisible ( toolBarVisible );
    if ( actShowToolBar ) actShowToolBar->setChecked ( toolBarVisible );
}

bool SKGTableWithGraph::isToolBarVisible() const
{
    return toolBarVisible;
}

void SKGTableWithGraph::onSwitchToolBarVisibility()
{
    setToolBarVisible ( !isToolBarVisible() );
}

void SKGTableWithGraph::onSwitchLimits()
{
    limitVisible=!limitVisible;
    redrawGraphDelayed();
}

void SKGTableWithGraph::onSwitchLinearRegression()
{
    linearRegressionVisible=!linearRegressionVisible;
    redrawGraphDelayed();
}

KMenu* SKGTableWithGraph::getTableContextualMenu() const
{
    return mainMenu;
}

KMenu* SKGTableWithGraph::getGraphContextualMenu() const
{
    return ui.graphicView->getContextualMenu();
}

void SKGTableWithGraph::showMenu ( const QPoint& pos )
{
    if ( mainMenu ) mainMenu->popup ( ui.kTable->mapToGlobal ( pos ) );
}

QTableWidget* SKGTableWithGraph::table() const
{
    return ui.kTable;
}

SKGStringListList SKGTableWithGraph::getTable()
{
    //Build table
    SKGStringListList table;

    //Get header names
    int nb=ui.kTable->columnCount();
    QStringList cols;
    for ( int i=0; i<nb; i++ )
    {
        cols.append ( ui.kTable->horizontalHeaderItem ( i )->text() );
    }
    table.append ( cols );

    //Get content
    int nb2=ui.kTable->rowCount();
    for ( int i=0; i<nb2; i++ )
    {
        QStringList row;
        for ( int j=0; j<nb; j++ )
        {
            row.append ( ui.kTable->item ( i,j )->text() );
        }
        table.append ( row );
    }
    return table;
}
QString SKGTableWithGraph::getState()
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    if ( isGraphVisible() && isTableVisible() ) root.setAttribute ( "splitterState", QString ( ui.splitter->saveState().toHex() ) );
    root.setAttribute ( "graphMode", SKGServices::intToString ( ( int ) getGraphType() ) );
    root.setAttribute ( "allPositive", ui.kAllPositive->checkState() ==Qt::Checked ? "Y" : "N" );
    root.setAttribute ( "filter", ui.kFilterEdit->text() );
    root.setAttribute ( "bottomToolBarVisible", isToolBarVisible() ? "Y" : "N" );
    root.setAttribute ( "limitVisible", limitVisible ? "Y" : "N" );
    root.setAttribute ( "linearRegressionVisible", linearRegressionVisible ? "Y" : "N" );

    QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
    root.setAttribute ( "sortOrder", SKGServices::intToString ( horizontalHeader->sortIndicatorOrder() ) );
    root.setAttribute ( "sortColumn", SKGServices::intToString ( horizontalHeader->sortIndicatorSection() ) );
    root.setAttribute ( "graphicViewState", ui.graphicView->getState() );

    return doc.toString();
}

void SKGTableWithGraph::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::setState" );
    QDomDocument doc ( "SKGML" );
    if ( doc.setContent ( iState ) )
    {
        QDomElement root = doc.documentElement();

        QString splitterState=root.attribute ( "splitterState" );
        if ( !splitterState.isEmpty() ) ui.splitter->restoreState ( QByteArray::fromHex ( splitterState.toAscii() ) );
        QString graphMode=root.attribute ( "graphMode" );
        if ( !graphMode.isEmpty() )
        {
            setGraphType ( SKGTableWithGraph::HISTOGRAM );
            ui.kDisplayMode->setCurrentIndex ( 1 );
            setGraphType ( ( SKGTableWithGraph::GraphType ) SKGServices::stringToInt ( graphMode ) );
        }

        QString allPositive=root.attribute ( "allPositive" );
        if ( !allPositive.isEmpty() ) ui.kAllPositive->setCheckState ( ( allPositive=="Y" ? Qt::Checked : Qt::Unchecked ) );
        ui.kFilterEdit->setText ( root.attribute ( "filter" ) );

        QString bottomToolBarVisible=root.attribute ( "bottomToolBarVisible" );
        if ( !bottomToolBarVisible.isEmpty() ) setToolBarVisible ( bottomToolBarVisible=="Y" );

        QString limitVisibleString=root.attribute ( "limitVisible" );
        if ( !limitVisibleString.isEmpty() )
        {
            limitVisible= ( limitVisibleString=="Y" );
            if ( actShowLimits ) actShowLimits->setChecked ( limitVisible );
        }

        QString linearRegressionVisibleString=root.attribute ( "linearRegressionVisible" );
        if ( !linearRegressionVisibleString.isEmpty() )
        {
            linearRegressionVisible= ( linearRegressionVisibleString=="Y" );
            if ( actShowLinearRegression ) actShowLinearRegression->setChecked ( linearRegressionVisible );
        }

        QString sortOrder=root.attribute ( "sortOrder" );
        QString sortColumn=root.attribute ( "sortColumn" );
        if ( sortOrder.isEmpty() ) sortOrder='0';
        if ( sortColumn.isEmpty() ) sortColumn='0';
        ui.kTable->setColumnCount ( SKGServices::stringToInt ( sortColumn ) +1 );
        ui.kTable->horizontalHeader()->setSortIndicator ( SKGServices::stringToInt ( sortColumn ), ( Qt::SortOrder ) SKGServices::stringToInt ( sortOrder ) );

        QString graphicViewState=root.attribute ( "graphicViewState" );
        if ( !graphicViewState.isEmpty() ) ui.graphicView->setState ( graphicViewState );
    }
    else
    {
        //vvv Correction bug 2278703:Zoom fit when opening report tab
        setGraphType ( SKGTableWithGraph::HISTOGRAM );
        setGraphType ( SKGTableWithGraph::STACK );
        //^^^ Correction bug 2278703:Zoom fit when opening report tab
    }
}

void SKGTableWithGraph::onFilterModified()
{
    if ( timer ) timer->start ( 300 );
}

void SKGTableWithGraph::setData ( const SKGStringListList& iData, SKGServices::SKGUnitInfo iPrimaryUnit,
                                  SKGServices::SKGUnitInfo iSecondaryUnit,
                                  DisplayAdditionalFlag iAdditionalInformation,
                                  int iNbVirtualColumn )
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::setData" );
    data=iData;
    primaryUnit=iPrimaryUnit;
    secondaryUnit=iSecondaryUnit;
    additionalInformation=iAdditionalInformation;
    nbVirtualColumns=iNbVirtualColumn;

    refresh();
}

SKGTableWithGraph::DisplayAdditionalFlag SKGTableWithGraph::getAdditionnalDisplayMode() const
{
    return additionalInformation;
}

bool SKGTableWithGraph::listSort ( const QStringList &s1, const QStringList &s2 )
{
    if ( sortColumn>=s1.count() ) sortColumn=s1.count()-1;
    if ( sortColumn>=0 )
    {
        QString v1=s1.at ( sortColumn );
        QString v2=s2.at ( sortColumn );
        if ( sortColumn==0 )
        {
            int v=KStringHandler::naturalCompare ( v1, v2, Qt::CaseInsensitive );
            return ( sortOrder ? v>0 : v<0 );
        }

        double vd1=SKGServices::stringToDouble ( v1 );
        double vd2=SKGServices::stringToDouble ( v2 );
        return ( sortOrder ? vd1>vd2 : vd1<vd2 );
    }
    else return false;
}

void SKGTableWithGraph::refresh()
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::refresh" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    SKGStringListList groupedTable=data;
    int nbCols=-1;
    if ( groupedTable.count() ) nbCols=groupedTable.at ( 0 ).count();
    int nbRealCols=nbCols;

    //Create filtered table
    {
        QStringList words=SKGServices::splitCSVLine ( ui.kFilterEdit->text(), ' ', true );
        int nbwords=words.count();

        for ( int i=groupedTable.count()-1; i>=1; --i ) //The first line is not filtered because it's the title
        {
            QStringList line=groupedTable.at ( i );
            if ( nbCols )
            {
                QString val=line.at ( 0 );

                //Filtered
                bool ok=true;
                for ( int w=0; ok && w<nbwords; ++w )
                {
                    QString word=words.at ( w );
                    ok=val.contains ( word , Qt::CaseInsensitive );

                }
                if ( !ok ) groupedTable.removeAt ( i );
            }
        }
    }

    //Compute sums line
    int nblines=groupedTable.count();
    if ( additionalInformation&SUM )
    {
        QStringList sums;
        sums.push_back ( i18nc ( "Noun, the numerical sum of a list of values","Sum" ) );
        for ( int i=1; i<nbCols; ++i )
        {
            double sum=0;
            for ( int j=1; j<nblines; ++j )
            {
                QString valstring=groupedTable.at ( j ).at ( i );
                if ( !valstring.isEmpty() ) sum+=SKGServices::stringToDouble ( valstring );
            }
            sums.push_back ( SKGServices::doubleToString ( sum ) );
        }
        groupedTable.push_back ( sums );
        nblines++;
    }

    //Compute sum and average column
    indexSum=-1;
    indexAverage=-1;
    indexMin=-1;
    indexLinearRegression=-1;
    if ( nbCols>2 && ( additionalInformation&SUM || additionalInformation&AVERAGE || additionalInformation&LIMITS ) )
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::refresh-computation" );
        //Add title
        QStringList newLine=groupedTable.at ( 0 );
        if ( additionalInformation&SUM )
        {
            indexSum=newLine.count();
            newLine.push_back ( i18nc ( "Noun, the numerical sum of a list of values","Sum" ) );
        }
        if ( additionalInformation&AVERAGE )
        {
            indexAverage=newLine.count();
            newLine.push_back ( i18nc ( "Noun, the numerical average of a list of values","Average" ) );
        }
        if ( additionalInformation&LIMITS )
        {
            indexMin=newLine.count();
            newLine.push_back ( i18nc ( "Noun, the minimum value of a list of values","Min" ) );
            newLine.push_back ( i18nc ( "Noun, the maximum value of a list of values","Max" ) );
        }

        indexLinearRegression=newLine.count();
        newLine.push_back ( i18nc ( "Noun","Linear regression" ) );

        groupedTable.replace ( 0, newLine );

        for ( int i=1; i<nblines; ++i )
        {
            QStringList newLine=groupedTable.at ( i );
            double sumy=0;
            double sumy2=0;
            double sumx=0;
            double sumx2=0;
            double sumxy=0;
            double min=10e20;
            double max=-10e20;
            int nbVals=0;
            for ( int j=1; j<nbCols-nbVirtualColumns; ++j )
            {
                QString valstring=newLine.at ( j );
                if ( !valstring.isEmpty() )
                {
                    double v=SKGServices::stringToDouble ( valstring );
                    sumx+=j;
                    sumx2+=j*j;
                    sumy+=v;
                    sumy2+=v*v;
                    sumxy+=j*v;
                    min=qMin ( min, v );
                    max=qMax ( max, v );
                    ++nbVals;
                }
            }
            if ( additionalInformation&SUM ) newLine.push_back ( SKGServices::doubleToString ( sumy ) );
            if ( additionalInformation&AVERAGE ) newLine.push_back ( SKGServices::doubleToString ( sumy/nbVals ) );
            if ( additionalInformation&LIMITS )
            {
                newLine.push_back ( SKGServices::doubleToString ( min ) );
                newLine.push_back ( SKGServices::doubleToString ( max ) );
            }

            double s2x=sumx2/nbVals-sumx*sumx/ ( nbVals*nbVals );
            double sxy=sumxy/nbVals- ( sumx/nbVals ) * ( sumy/nbVals );

            double a=sxy/s2x;
            double b=sumy/nbVals-a*sumx/nbVals;
            newLine.push_back ( "y="+SKGServices::doubleToString ( a ) +"*x+"+SKGServices::doubleToString ( b ) );

            groupedTable.replace ( i, newLine );
        }
        ++nbCols;
        if ( additionalInformation&SUM ) ++nbCols;
        if ( additionalInformation&AVERAGE ) ++nbCols;
        if ( additionalInformation&LIMITS ) nbCols+=2;
    }

    //Sort lines
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::refresh-sort" );
        //Set parameters for static method
        QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
        sortOrder=horizontalHeader->sortIndicatorOrder();
        sortColumn=horizontalHeader->sortIndicatorSection();

        //Extract title and sums
        if ( groupedTable.count() )
        {
            QStringList fist=groupedTable.takeFirst();
            QStringList last;
            if ( additionalInformation&SUM && groupedTable.count() ) last=groupedTable.takeLast();

            //Sort
            qSort ( groupedTable.begin(), groupedTable.end(), listSort );

            //Add title and sums
            groupedTable.insert ( 0, fist );
            if ( additionalInformation&SUM ) groupedTable.push_back ( last );
        }
    }

    IFSKGTRACEL ( 10 )
    {
        QStringList dump=SKGServices::tableToDump ( groupedTable, SKGServices::DUMP_TEXT );
        int nbl=dump.count();
        for ( int i=0; i<nbl; ++i )
        {
            SKGTRACE << dump[i] << endl;
        }
    }

    //Fill table
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::refresh-fill" );

        int nbRealColumns=nbRealCols-nbVirtualColumns;
        KLocale* locale=KGlobal::locale();
        ui.kTable->hide();
	QHeaderView* horizontalHeader=ui.kTable->horizontalHeader();
	if ( horizontalHeader ) horizontalHeader->setResizeMode ( QHeaderView::Fixed ); //Needed to improve performances of setHorizontalHeaderLabels
        ui.kTable->clear();
        ui.kTable->setRowCount ( nblines-1 );
        ui.kTable->setColumnCount ( nbCols );
        for ( int i=0; i<nblines; ++i )
        {
            QStringList line=groupedTable.at ( i );
            if ( i==0 )
            {
		SKGTRACEIN ( 10, "SKGTableWithGraph::refresh-sethorizontalheaderlabels" );
                //Set header
                ui.kTable->setHorizontalHeaderLabels ( line );
            }
            else
            {
                for ( int j=0; j<nbCols; ++j )
                {
                    QString val=line.at ( j );

                    QTableWidgetItem* item;
                    if ( j==0 )
                    {
                        item=new QTableWidgetItem ( val );
                        if ( indexLinearRegression!=-1 ) item->setData ( DATA_VALUE, line.at ( indexLinearRegression ) );
                    }
                    else
                    {
                        if ( !val.isEmpty() )
                        {
                            if ( j==indexLinearRegression )
                            {
                                item=new QTableWidgetItem ( val );
                            }
                            else
                            {
                                double vald=SKGServices::stringToDouble ( val );

                                item=new QTableWidgetItem ( locale->formatMoney ( vald, primaryUnit.Name, primaryUnit.NbDecimal ) );
                                if ( !secondaryUnit.Name.isEmpty() && secondaryUnit.Value )
                                {
                                    item->setToolTip ( locale->formatMoney ( vald/secondaryUnit.Value, secondaryUnit.Name, secondaryUnit.NbDecimal ) );
                                }

                                item->setData ( DATA_VALUE, vald );
                                item->setTextAlignment ( Qt::AlignRight );
                                if ( vald<0 ) item->setForeground ( QBrush ( QColor ( Qt::red ) ) );
                            }
                        }
                        else
                        {
                            item=new QTableWidgetItem ( "" );
                            item->setData ( DATA_VALUE, "" );
                        }
                        item->setFlags ( (j>=nbRealColumns && j!=indexSum) || ui.kTable->horizontalHeaderItem(j)->text()=="0000" ? Qt::NoItemFlags : Qt::ItemIsEnabled|Qt::ItemIsSelectable );
                    }
                    ui.kTable->setItem ( i-1, j, item );
                }
            }
        }

	//Refresh graphic view
	redrawGraph();

	if ( horizontalHeader ) horizontalHeader->setResizeMode ( QHeaderView::ResizeToContents );
	ui.kTable->show();
    }

    QApplication::restoreOverrideCursor();
}

void SKGTableWithGraph::onSelectionChanged ( QTableWidgetItem* current, QTableWidgetItem* previous )
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onSelectionChanged" );

    //Unset color on previous selection
    if ( previous )
    {
        QAbstractGraphicsShapeItem * graphicItem=dynamic_cast<QAbstractGraphicsShapeItem *> ( ( QGraphicsItem* ) previous->data ( 1 ).toLongLong() );
        if ( graphicItem )
        {
            graphicItem->setBrush ( QBrush ( QColor::fromHsv ( graphicItem->data ( DATA_COLOR_H ).toInt(),
                                             graphicItem->data ( DATA_COLOR_S ).toInt(),
                                             graphicItem->data ( DATA_COLOR_V ).toInt(),
                                             graphicItem->data ( DATA_COLOR_A ).toInt() ) ) );
            graphicItem->setSelected ( false );
        }
        else
        {
            QGraphicsLineItem * graphicItem=dynamic_cast<QGraphicsLineItem *> ( ( QGraphicsItem* ) previous->data ( 1 ).toLongLong() );
            if ( graphicItem )
            {
                QPen pen=graphicItem->pen();
                pen.setColor ( QColor::fromHsv ( graphicItem->data ( DATA_COLOR_H ).toInt(),
                                                 graphicItem->data ( DATA_COLOR_S ).toInt(),
                                                 graphicItem->data ( DATA_COLOR_V ).toInt(),
                                                 graphicItem->data ( DATA_COLOR_A ).toInt() ) );
                graphicItem->setPen ( pen );
                graphicItem->setSelected ( false );
            }
        }
    }

    //Set highlight color on current selection
    if ( current )
    {
        QAbstractGraphicsShapeItem * graphicItem=dynamic_cast<QAbstractGraphicsShapeItem *> ( ( QGraphicsItem* ) current->data ( 1 ).toLongLong() );
        if ( graphicItem )
        {
            graphicItem->setBrush ( QBrush ( QApplication::palette().color ( QPalette::Highlight ) ) );
            graphicItem->setSelected ( true );
        }
        else
        {
            QGraphicsLineItem * graphicItem=dynamic_cast<QGraphicsLineItem *> ( ( QGraphicsItem* ) current->data ( 1 ).toLongLong() );
            if ( graphicItem )
            {
                QPen pen=graphicItem->pen();
                pen.setColor ( QApplication::palette().color ( QPalette::Highlight ) );
                graphicItem->setPen ( pen );
                graphicItem->setSelected ( false );
            }
        }
    }
}

void SKGTableWithGraph::onDoubleClickGraph()
{
    if ( scene )
    {
        //Get selection
        QList<QGraphicsItem *> selectedGraphItems=scene->selectedItems();
        if ( selectedGraphItems.count() )
        {
            emit cellDoubleClicked ( selectedGraphItems[0]->data ( 1 ).toInt(), selectedGraphItems[0]->data ( 2 ).toInt() );
        }
    }
}

void SKGTableWithGraph::onDoubleClick ( int row, int column )
{
    emit cellDoubleClicked ( row, column );
}

void SKGTableWithGraph::onSelectionChangedInGraph()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onSelectionChangedInGraph" );
    if ( scene )
    {
        //Get selection
        QList<QGraphicsItem *> selectedGraphItems=scene->selectedItems();
        if ( selectedGraphItems.count() )
        {
            ui.kTable->setCurrentCell ( selectedGraphItems[0]->data ( 1 ).toInt(), selectedGraphItems[0]->data ( 2 ).toInt() );
        }
    }
}

void SKGTableWithGraph::redrawGraphDelayed()
{
    if ( timerRedraw ) timerRedraw->start ( 300 );
}

void SKGTableWithGraph::redrawGraph()
{
    SKGTRACEIN ( 10, "SKGTableWithGraph::redrawGraph" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    ui.graphicView->hide();
    ui.kTable->hide();

    //Recreate scene
    if ( scene )
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::redrawGraph-remove scene" );
        scene->clear();
        delete scene;
    }


    scene=new SKGGraphicsScene();
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::redrawGraph-create scene" );
        //Get current selection
        int crow=ui.kTable->currentRow();
        int ccolumn=ui.kTable->currentColumn();

        //Get nb columns and rows
        int nbRows=ui.kTable->rowCount()- ( additionalInformation&SUM ? 1 : 0 ); //Sum
        int nbColumns=getNbColumns ( false );
        int nbRealColumns=nbColumns-nbVirtualColumns;

        //Get graphic mode
        GraphType mode= ( GraphType ) ui.kDisplayMode->currentIndex();
        bool inPositive=false;
        if ( mode==STACK || mode==HISTOGRAM || mode==POINT || mode==LINE )
        {
            ui.kAllPositive->show();
            inPositive= ( ui.kAllPositive->checkState() ==Qt::Checked );
        }
        else ui.kAllPositive->hide();

        //Compute y limits
        double minLimit=0;
        double maxLimit=0;
        int nbLevel=0;
        SKGTRACEL ( 30 ) << "mode=" << mode << endl;
        SKGTRACEL ( 30 ) << "nb rows        =" << nbRows << endl;
        SKGTRACEL ( 30 ) << "nb columns     =" << nbColumns << endl;
        SKGTRACEL ( 30 ) << "selected row   =" << crow << endl;
        SKGTRACEL ( 30 ) << "selected column=" << ccolumn << endl;
        if ( mode==STACK )
        {
            //STACK
            for ( int x=0; x<nbRows; ++x )
            {
                double sumPositive=0;
                double sumNegative=0;
                for ( int y=0; y<nbColumns; ++y )
                {
                    QTableWidgetItem* tableItem=ui.kTable->item ( x,y );
                    if ( tableItem )
                    {
                        double val=tableItem->data ( DATA_VALUE ).toDouble();
                        if ( inPositive || val>0 )
                        {
                            sumPositive+=fabs ( val );
                        }
                        else
                        {
                            sumNegative+=val;
                        }
                    }
                }

                minLimit=qMin ( minLimit, sumNegative );
                maxLimit=qMax ( maxLimit, sumPositive );
            }
        }
        else if ( mode==HISTOGRAM || mode==POINT || mode==LINE )
        {
            //HISTOGRAM or POINTS or LINES
            for ( int x=0; x<nbRows; ++x )
            {
                for ( int y=0; y<nbColumns; ++y )
                {
                    QTableWidgetItem* tableItem=ui.kTable->item ( x,y );
                    if ( tableItem )
                    {
                        double val=tableItem->data ( DATA_VALUE ).toDouble();
                        if ( inPositive )
                        {
                            maxLimit=qMax ( maxLimit, fabs ( val ) );
                        }
                        else
                        {
                            minLimit=qMin ( minLimit, val );
                            maxLimit=qMax ( maxLimit, val );
                        }
                    }
                }
            }
        }
        else if ( mode==CONCENTRICPIE )
        {
            for ( int x=0; x<nbRows; ++x )
            {
                QString xname=ui.kTable->item ( x, 0 )->text();
                QStringList vals=xname.split ( OBJECTSEPARATOR );
                int nbvals=vals.count();
                nbLevel=qMax ( nbLevel, nbvals-1 );
            }
        }
        SKGTRACEL ( 30 ) << "minLimit=" << minLimit << endl;
        SKGTRACEL ( 30 ) << "maxLimit=" << maxLimit << endl;

        //Compute
        double width=10;
        double maxX=0;
        double margin=0;
        if ( nbRows )
        {
            QRect vSize=ui.graphicView->rect();
            if ( mode==STACK )
            {
                width= ( maxLimit-minLimit ) /nbRows;
                width=width* ( ( double ) vSize.width() ) / ( ( double ) vSize.height() );
                margin= ( maxLimit-minLimit ) *0.1;
                maxX=width*nbRows+margin;
            }
            else if ( mode==HISTOGRAM || mode==POINT || mode==LINE )
            {
                width= ( maxLimit-minLimit ) / ( nbRows* ( nbColumns-1 ) );
                width=width* ( ( double ) vSize.width() ) / ( ( double ) vSize.height() );
                margin= ( maxLimit-minLimit ) *0.1;
                maxX=width* ( nbColumns-1 ) * ( nbRows+1 ) ;
            }
        }

        SKGTRACEL ( 30 ) << "width=" << width << endl;
        SKGTRACEL ( 30 ) << "maxX=" << maxX << endl;
        SKGTRACEL ( 30 ) << "margin=" << margin << endl;

        //Initialise pen
        QPen dotPen=QPen ( Qt::SolidLine ); //WARNING: Qt::DotLine is very bad for performances
        dotPen.setColor ( Qt::gray );
        dotPen.setWidth ( margin/40 );

        QPen blackPen=QPen ( Qt::SolidLine ); //WARNING: Qt::DotLine is very bad for performances
        blackPen.setColor ( Qt::black );
        blackPen.setWidth ( margin/40 );

	//Set rect
        int nbColInMode2= ( nbColumns>2 ? sqrt ( nbColumns-1 ) +.5 : 1 );
        scene->setItemIndexMethod ( QGraphicsScene::NoIndex );
        SKGTRACEL ( 10 ) << "Items:" <<nbRows << "x" << ( nbColumns-1 ) << "=" << nbRows* ( nbColumns-1 ) << endl;

        //Redraw scene
        double x0=0;
        double x1=0;
        double ymin=0;
        double ymax=0;
        for ( int x=0; x<nbRows; ++x )
        {
            QString xname=ui.kTable->item ( x, 0 )->text();
            double yPlus=0;
            double yMoins=0;
            int jprevious=-1;
            for ( int j=1; j<nbColumns; ++j )
            {
                //Get column name
                QString yname=ui.kTable->horizontalHeaderItem ( j )->text();

                //Get cell value
                QTableWidgetItem* tableItem=ui.kTable->item ( x,j );
                if ( tableItem )
                {
                    double val=tableItem->data ( DATA_VALUE ).toDouble();
                    QString valstring=tableItem->text();

                    int color_h= ( 240+359*x/nbRows ) %360; //First color is blue to be compliant with min (red) and max (green)
                    int color_s=255-155* ( nbColumns-1-j ) /nbColumns;
                    int color_v=255-155* ( nbColumns-1-j ) /nbColumns;
                    int color_a=200;
                    if ( j>=nbRealColumns )
                    {
                        //Jaune
                        color_h=60;
                        color_s=255;
                        color_v=255;
                    }

                    QColor color;
                    if ( x==crow && j==ccolumn ) color=QApplication::palette().color ( QPalette::Highlight );
                    else color=QColor::fromHsv ( color_h,color_s,color_v );
                    color.setAlpha ( color_a );

                    QBrush brush ( color );

                    QGraphicsItem* graphItem=NULL;
                    if ( mode==STACK )
                    {
                        //STACK
                        if ( val>0 || inPositive )
                        {
                            graphItem=scene->addRect ( width*x, -yPlus, width, -fabs ( val ), QPen(), brush );
                            yPlus+=fabs ( val );
                            if ( yPlus>ymax ) ymax=yPlus;
                        }
                        else
                        {
                            graphItem=scene->addRect ( width*x, -yMoins, width, -val, QPen(), brush );
                            yMoins+=val;
                            if ( yMoins<ymin ) ymin=yMoins;
                        }
                    }
                    else if ( mode==HISTOGRAM )
                    {
                        //HISTOGRAM
                        if ( j==1 ) x0=width* ( ( j-1 ) * ( nbRows+1 ) +x ) +width;
                        if ( j==nbColumns-nbVirtualColumns-1 ) x1=width* ( ( j-1 ) * ( nbRows+1 ) +x ) +width;
                        if ( inPositive )
                        {
                            graphItem=scene->addRect ( width* ( ( j-1 ) * ( nbRows+1 ) +x ) +width/2, 0, width, -fabs ( val ), QPen(), brush );
                            if ( fabs ( val ) >ymax ) ymax=fabs ( val );
                        }
                        else
                        {
                            graphItem=scene->addRect ( width* ( ( j-1 ) * ( nbRows+1 ) +x ) +width/2, 0, width, -val, QPen(), brush );
                            if ( val>ymax ) ymax=val;
                            if ( val<ymin ) ymin=val;
                        }
                    }
                    else if ( mode==POINT )
                    {
                        //POINTS
                        double radius=margin/8;
                        double xmin=width* ( j-1 ) * ( nbRows+1 )-radius;
                        if ( j==1 ) x0=xmin+radius;
                        if ( j==nbColumns-nbVirtualColumns-1 ) x1=xmin+radius;

                        if ( inPositive )
                        {
                            switch ( x%3 )
                            {
                            case 0:
                                graphItem=scene->addEllipse ( xmin, -fabs ( val )-radius, 2*radius, 2*radius, QPen(), brush );
                                break;
                            case 1:
                                graphItem=scene->addRect ( xmin, -fabs ( val )-radius, 2*radius, 2*radius, QPen(), brush );
                                break;
                            case 2:
                                QPolygonF polygon;
                                polygon << QPointF ( xmin, -fabs ( val )-radius ) << QPointF ( xmin+2*radius, -fabs ( val ) +radius )
                                << QPointF ( xmin+2*radius, -fabs ( val )-radius ) << QPointF ( xmin, -fabs ( val ) +radius );
                                graphItem=scene->addPolygon ( polygon, QPen(), brush );
                                break;
                            }
                            if ( fabs ( val ) >ymax ) ymax=fabs ( val );
                        }
                        else
                        {
                            switch ( x%3 )
                            {
                            case 0:
                                graphItem=scene->addEllipse ( xmin, -val-radius, 2*radius, 2*radius, QPen(), brush );
                                break;
                            case 1:
                                graphItem=scene->addRect ( xmin, -val-radius, 2*radius, 2*radius, QPen(), brush );
                                break;
                            case 2:
                                QPolygonF polygon;
                                polygon << QPointF ( xmin, -val-radius ) << QPointF ( xmin+2*radius, -val+radius )
                                << QPointF ( xmin+2*radius, -val-radius ) << QPointF ( xmin, -val+radius );
                                graphItem=scene->addPolygon ( polygon, QPen(), brush );
                                break;
                            }
                            if ( val>ymax ) ymax=val;
                            if ( val<ymin ) ymin=val;
                        }

                    }
                    else if ( mode==LINE )
                    {
                        //LINES
                        //Empty cells are ignored
                        if ( j==1 ) x0=width* ( j-1 ) * ( nbRows+1 );
                        if ( j==nbColumns-nbVirtualColumns-1 ) x1=width* ( j-1 ) * ( nbRows+1 );

                        if ( !valstring.isEmpty() )
                        {
                            if ( jprevious==-1 ) jprevious=j;

                            //Create pen
                            QPen pen=QPen ( color );
                            pen.setWidth ( margin/4 );
                            pen.setCapStyle ( Qt::RoundCap );
                            pen.setJoinStyle ( Qt::RoundJoin );

                            QTableWidgetItem* tableItem2=ui.kTable->item ( x, jprevious );
                            if ( tableItem2 )
                            {
                                QString val2string=tableItem->text();
                                if ( !val2string.isEmpty() )
                                {
                                    double val2=tableItem2->data ( DATA_VALUE ).toDouble();

                                    if ( inPositive )
                                    {
                                        graphItem=scene->addLine ( width* ( jprevious-1 ) * ( nbRows+1 )- ( jprevious==j ? width/20 : 0 ), -fabs ( val2 ), width* ( j-1 ) * ( nbRows+1 ), -fabs ( val ), pen );
                                        if ( fabs ( val ) >ymax ) ymax=fabs ( val );
                                    }
                                    else
                                    {
                                        graphItem=scene->addLine ( width* ( jprevious-1 ) * ( nbRows+1 )- ( jprevious==j ? width/20 : 0 ), -val2, width* ( j-1 ) * ( nbRows+1 ), -val, pen );
                                        if ( val>ymax ) ymax=val;
                                        if ( val<ymin ) ymin=val;
                                    }
                                    jprevious=j;
                                }
                            }
                        }
                    }
                    else if ( mode==PIE || mode==CONCENTRICPIE )
                    {
                        //PIE
                        //Compute absolute sum of the column
                        val=fabs ( val );
                        double sumColumn=0;
                        double previousSum=0;
                        for ( int x2=0; x2<nbRows; ++x2 )
                        {
                            QTableWidgetItem* tableItem=ui.kTable->item ( x2,j );
                            if ( tableItem )
                            {
                                double absVal=fabs ( tableItem->data ( DATA_VALUE ).toDouble() );
                                sumColumn+=absVal;
                                if ( x2<x ) previousSum+=absVal;
                            }
                        }

                        if ( sumColumn )
                        {
                            int nbvals=xname.split ( OBJECTSEPARATOR ).count();
                            double pas=0;
                            double p=0;
                            if ( mode==CONCENTRICPIE )
                            {
                                pas=100/ ( nbLevel+1 );
                                p=pas* ( nbLevel+1-nbvals );
                            }

                            QPainterPath path;
                            path.moveTo ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +BOX_SIZE/2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +BOX_SIZE/2 );
                            path.arcTo ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +BOX_MARGIN+p/2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +BOX_MARGIN+p/2, BOX_SIZE-2*BOX_MARGIN-p, BOX_SIZE-2*BOX_MARGIN-p, 360*previousSum/sumColumn, 360*val/sumColumn );
                            if ( mode==CONCENTRICPIE && nbvals<=nbLevel+1 && nbvals>1 )
                            {
                                p=pas* ( nbLevel+1-nbvals+1 );
                                path.moveTo ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +BOX_SIZE/2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +BOX_SIZE/2 );
                                path.arcTo ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +BOX_MARGIN+p/2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +BOX_MARGIN+p/2, BOX_SIZE-2*BOX_MARGIN-p, BOX_SIZE-2*BOX_MARGIN-p, 360*previousSum/sumColumn, 360*val/sumColumn );
                            }
                            path.closeSubpath();
                            graphItem=scene->addPath ( path, mode==CONCENTRICPIE ? QPen ( Qt::white ) : QPen(), brush );
                        }
                    }

                    if ( graphItem )
                    {
                        graphItem->setZValue ( 5 );
                        graphItem->setToolTip ( xname+'\n'+yname+'\n'+valstring+'\n'+tableItem->toolTip() );
                        bool isSelect= ( isSelectable() && tableItem->flags() &Qt::ItemIsEnabled );
                        graphItem->setFlag ( QGraphicsItem::ItemIsSelectable, isSelect );
                        if ( isSelect ) graphItem->setCursor ( Qt::PointingHandCursor );
                        graphItem->setData ( 1, x );
                        graphItem->setData ( 2, j );
                        graphItem->setData ( DATA_COLOR_H, color_h );
                        graphItem->setData ( DATA_COLOR_S, color_s );
                        graphItem->setData ( DATA_COLOR_V, color_v );
                        graphItem->setData ( DATA_COLOR_A, color_a );

                        tableItem->setData ( 1, ( qlonglong )  graphItem );
                    }
                }
                else
                {
                    SKGTRACE << "WARNING: cell " << x << "," << j << " null" << endl;
                }
            }
        }

        //Draw axis
        double scaleText=0;
        if ( nbRows )
        {
            if ( mode==PIE || mode==CONCENTRICPIE )
            {
                //PIE
                int nbRowInMode2= ( ( int ) ( ( nbColumns-1 ) /nbColInMode2 ) );
                if ( nbRowInMode2*nbColInMode2<nbColumns-1 ) ++nbRowInMode2;
                for ( int i=0; i<=nbColInMode2; ++i )
                {
                    QGraphicsLineItem* item=scene->addLine ( BOX_SIZE*i, 0, BOX_SIZE*i, BOX_SIZE*nbRowInMode2 );
                    item->setPen ( dotPen );
                    item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                }
                for ( int i=0; i<=nbRowInMode2; ++i )
                {
                    QGraphicsLineItem* item=scene->addLine ( 0, BOX_SIZE*i, BOX_SIZE*nbColInMode2, BOX_SIZE*i );
                    item->setPen ( dotPen );
                    item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                }

                for ( int j=1; j<nbColumns; ++j )
                {
                    //Get column name
                    QString yname=ui.kTable->horizontalHeaderItem ( j )->text();

                    QGraphicsTextItem* textItem=scene->addText ( yname );
                    textItem->setPos ( BOX_SIZE* ( ( j-1 ) %nbColInMode2 ) +2, BOX_SIZE*floor ( ( j-1 ) /nbColInMode2 ) +2 );
                    textItem->scale ( .5, .5 );
                    textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );

                }
            }
            else
            {
                //STACK & HISTOGRAMM
                QGraphicsLineItem* item;

                //Lines x
                double ystep=computeStepSize ( ymax-ymin, 10 );
                if ( ystep>0 )
                {
                    KLocale* locale=KGlobal::locale();
                    for ( double y=ystep; y<ymax+margin; y=y+ystep )
                    {
                        item=scene->addLine ( -margin, -y, maxX, -y );
                        item->setPen ( dotPen );
                        item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                        item->setZValue ( 1 );

                        QGraphicsTextItem* textItem=scene->addText ( locale->formatMoney ( y, primaryUnit.Name, primaryUnit.NbDecimal ) );
                        textItem->setPos ( -margin, -y );
                        if ( scaleText==0 )
                        {
                            QRectF rect=textItem->boundingRect();
                            scaleText=0.8*qMin ( margin/rect.width(), ystep/rect.height() );
                        }
                        textItem->scale ( scaleText, scaleText );
                        textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                    }
                    for ( double y=-ystep; y>ymin-margin; y=y-ystep )
                    {
                        item=scene->addLine ( -margin, -y, maxX, -y );
                        item->setPen ( dotPen );
                        item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                        item->setZValue ( 1 );

                        QGraphicsTextItem* textItem=scene->addText ( locale->formatMoney ( y, primaryUnit.Name, primaryUnit.NbDecimal ) );
                        textItem->setPos ( -margin, -y );
                        if ( scaleText==0 )
                        {
                            QRectF rect=textItem->boundingRect();
                            scaleText=0.8*qMin ( margin/rect.width(), ystep/rect.height() );
                        }
                        textItem->scale ( scaleText, scaleText );
                        textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                    }
                }

                if ( mode==HISTOGRAM || mode==POINT || mode==LINE )
                {
                    //Line y
                    double xstep=width* ( nbRows+1 );
                    int jstep=qMax ( computeStepSize ( nbColumns, 20 ), 1.0 );
                    for ( int j=1; j<nbColumns; j+=jstep )
                    {
                        QString yname=ui.kTable->horizontalHeaderItem ( j )->text();
                        double x=xstep+ ( j-2 ) *xstep;

                        QGraphicsTextItem* textItem=scene->addText ( yname );
                        textItem->setPos ( x, 0 );
                        textItem->scale ( scaleText, scaleText );
                        textItem->rotate ( 90 );
                        textItem->moveBy ( textItem->boundingRect().height() *scaleText, 0 );
                        textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );

                        QGraphicsLineItem* item=scene->addLine ( x, -ymin+margin, x, -ymax-margin );
                        item->setPen ( dotPen );
                        item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                    }
                }

                //Axis y
                item=scene->addLine ( 0, -ymin+margin, 0, -ymax-margin, blackPen );
                item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                addArrow ( QPointF ( 0, -ymax-margin ), margin/2, 45, 90 );
                item->setZValue ( 2 );

                //Axis x
                item=scene->addLine ( -margin, 0, maxX, 0, blackPen );
                item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                addArrow ( QPointF ( maxX, 0 ), margin/2, 45, 180 );
                item->setZValue ( 2 );
            }
        }

        //Draw Average
        bool lineCondition=limitVisible && !inPositive && ( mode==HISTOGRAM || mode==POINT || mode==LINE ) && scaleText && nbRows==1;
        int averageCol=getAverageColumnIndex();
        if ( lineCondition && averageCol!=-1 )
        {
            QTableWidgetItem* tableItem=ui.kTable->item ( 0, averageCol );
            if ( tableItem )
            {
                double y=tableItem->data ( DATA_VALUE ).toDouble();
                KLocale* locale=KGlobal::locale();
                QGraphicsLineItem* item=scene->addLine ( 0, -y, maxX, -y );
                dotPen.setColor ( Qt::blue );
                item->setPen ( dotPen );
                item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                item->setZValue ( 1 );

                QGraphicsTextItem* textItem=scene->addText ( locale->formatMoney ( y, primaryUnit.Name, primaryUnit.NbDecimal ) );
                textItem->setPos ( maxX, -y );
                textItem->setDefaultTextColor ( Qt::blue );
                textItem->scale ( scaleText, scaleText );
                textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );
            }
        }

        //Draw Min & Max limites
        int minCol=getMinColumnIndex();
        if ( lineCondition && minCol!=-1 )
        {
            //Min
            {
                QTableWidgetItem* tableItem=ui.kTable->item ( 0, minCol );
                if ( tableItem )
                {
                    double y=tableItem->data ( DATA_VALUE ).toDouble();
                    KLocale* locale=KGlobal::locale();
                    QGraphicsLineItem* item=scene->addLine ( 0, -y, maxX, -y );
                    dotPen.setColor ( Qt::red );
                    item->setPen ( dotPen );
                    item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                    item->setZValue ( 1 );

                    QGraphicsTextItem* textItem=scene->addText ( locale->formatMoney ( y, primaryUnit.Name, primaryUnit.NbDecimal ) );
                    textItem->setPos ( maxX, -y );
                    textItem->setDefaultTextColor ( Qt::red );
                    textItem->scale ( scaleText, scaleText );
                    textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                }
            }

            //Max
            {
                QTableWidgetItem* tableItem=ui.kTable->item ( 0, minCol+1 );
                if ( tableItem )
                {
                    double y=tableItem->data ( DATA_VALUE ).toDouble();
                    KLocale* locale=KGlobal::locale();
                    QGraphicsLineItem* item=scene->addLine ( 0, -y, maxX, -y );
                    dotPen.setColor ( Qt::green );
                    item->setPen ( dotPen );
                    item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                    item->setZValue ( 1 );

                    QGraphicsTextItem* textItem=scene->addText ( locale->formatMoney ( y, primaryUnit.Name, primaryUnit.NbDecimal ) );
                    textItem->setPos ( maxX, -y );
                    textItem->setDefaultTextColor ( Qt::green );
                    textItem->scale ( scaleText, scaleText );
                    textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                }
            }
        }

        //Draw Linear Regression
        bool lineCondition2=linearRegressionVisible && !inPositive && ( mode==HISTOGRAM || mode==POINT || mode==LINE ) && scaleText && nbRows==1;
        if ( lineCondition2 && indexLinearRegression!=-1 )
        {
            QTableWidgetItem* tableItem=ui.kTable->item ( 0, indexLinearRegression );
            if ( tableItem )
            {
                QString f=tableItem->text();

                QScriptEngine myEngine;
                QString f0=f;
                f0=f0.remove ( "y=" );
                f0=f0.replace ( 'x', '1' );
                f0=f0.replace ( ',','.' ); // Replace comma by a point in case of typo
                double y0 = myEngine.evaluate ( f0 ).toNumber();

                QString f1=f;
                f1=f1.remove ( "y=" );
                f1=f1.replace ( 'x', SKGServices::intToString ( nbRealColumns-1 ) );
                f1=f1.replace ( ',','.' ); // Replace comma by a point in case of typo
                double y1 = myEngine.evaluate ( f1 ).toNumber();

                QGraphicsLineItem* item=scene->addLine ( x0, -y0, x1, -y1 );
                dotPen.setColor ( Qt::darkYellow );
                item->setPen ( dotPen );
                item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
                item->setZValue ( 1 );

                /*QGraphicsTextItem* textItem=scene->addText ( f );
                textItem->setPos ( maxX, -y1 );
                textItem->setDefaultTextColor ( Qt::darkYellow );
                textItem->scale ( scaleText, scaleText );
                textItem->setFlag ( QGraphicsItem::ItemIsSelectable, false );*/
            }
        }
    }
    {
        SKGTRACEIN ( 10, "SKGTableWithGraph::redrawGraph-add scene to view" );
        scene->setSceneRect ( scene->itemsBoundingRect() );
        ui.graphicView->setScene ( scene );
        ui.graphicView->show();
        ui.kTable->show();
        ui.graphicView->onZoomOriginal();

        //Add selection event on scene
        connect ( scene, SIGNAL ( selectionChanged() ), this, SLOT ( onSelectionChangedInGraph() ), Qt::QueuedConnection );
        connect ( scene, SIGNAL ( doubleClicked() ), this, SLOT ( onDoubleClickGraph() ), Qt::QueuedConnection );
    }
    QApplication::restoreOverrideCursor();
}

int SKGTableWithGraph::getNbColumns ( bool iWithComputed ) const
{
    int nbColumns=ui.kTable->columnCount();
    if ( !iWithComputed )
    {
        if ( indexMin!=-1 ) nbColumns-=2;
        if ( indexAverage!=-1 ) --nbColumns;
        if ( indexSum!=-1 ) --nbColumns;
        if ( indexLinearRegression!=-1 ) --nbColumns;
    }
    return nbColumns;
}

int SKGTableWithGraph::getAverageColumnIndex() const
{
    return indexAverage;
}

int SKGTableWithGraph::getMinColumnIndex() const
{
    return indexMin;
}

double SKGTableWithGraph::computeStepSize ( double iRange, double iTargetSteps )
{
    // Calculate an initial guess at step size
    double tempStep = iRange / iTargetSteps;
    // Get the magnitude of the step size
    double mag = floor ( log10 ( tempStep ) );
    double magPow = pow ( ( double ) 10.0, mag );
    // Calculate most significant digit of the new step size
    double magMsd = ( ( int ) ( tempStep / magPow + .5 ) );
    // promote the MSD to either 1, 2, or 5
    if ( magMsd > 5.0 ) magMsd = 10.0;
    else if ( magMsd > 2.0 ) magMsd = 5.0;
    else if ( magMsd > 1.0 ) magMsd = 2.0;
    return magMsd * magPow;
}

void SKGTableWithGraph::addArrow ( const QPointF& iPeak, double iSize, double iArrowAngle, double iDegree )
{
    if ( scene )
    {
        QPolygonF pol;
        double radian=3.14*iArrowAngle/360;
        pol << QPointF ( 0, 0 ) << QPointF ( iSize*cos ( radian ), iSize*sin ( radian ) ) << QPointF ( iSize*cos ( radian ), -iSize*sin ( radian ) ) << QPointF ( 0, 0 );
        QGraphicsPolygonItem * item=scene->addPolygon ( pol, QPen ( Qt::black, iSize/20 ), QBrush ( Qt::black ) );
        item->rotate ( iDegree );
        item->moveBy ( iPeak.x(), iPeak.y() );
        item->setFlag ( QGraphicsItem::ItemIsSelectable, false );
        item->setZValue ( 2 );
    }
}

void SKGTableWithGraph::onExportCSV()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onExportCSV" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "*.csv|"+i18nc ( "File format", "CSV Files" ) , this );
    if ( fileName.isEmpty() ) return;

    {
        SKGError err;

        //Write file
        QFile file ( fileName );
        if ( !file.open ( QIODevice::WriteOnly | QIODevice::Text ) )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18nc ( "Error message",  "Save file [%1] failed",fileName ) );
        }
        else
        {
            QTextStream out ( &file );
            QStringList dump=SKGServices::tableToDump ( getTable(), SKGServices::DUMP_CSV );
            int nbl=dump.count();
            for ( int i=0; i<nbl; ++i )
            {
                out << dump[i] << endl;
            }
        }

        //Close file
        file.flush();
        file.close();

    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGTableWithGraph::onExportTXT()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onExportTXT" );
    QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///IMPEXP" ), "*.txt|"+i18nc ( "File format", "Text document" ) , this );
    if ( fileName.isEmpty() ) return;

    {
        SKGError err;

        //Write file
        QFile file ( fileName );
        if ( !file.open ( QIODevice::WriteOnly | QIODevice::Text ) )
        {
            err.setReturnCode ( ERR_INVALIDARG );
            err.setMessage ( i18nc ( "Error message",  "Save file [%1] failed",fileName ) );
        }
        else
        {
            QTextStream out ( &file );
            QStringList dump=SKGServices::tableToDump ( getTable(), SKGServices::DUMP_TEXT );
            int nbl=dump.count();
            for ( int i=0; i<nbl; ++i )
            {
                out << dump[i] << endl;
            }
        }

        //Close file
        file.flush();
        file.close();

    }
    QDesktopServices::openUrl ( QUrl ( fileName ) );
}

void SKGTableWithGraph::setGraphType ( SKGTableWithGraph::GraphType iType ) const
{
    ui.kDisplayMode->setCurrentIndex ( ( int ) iType );
}

SKGTableWithGraph::GraphType SKGTableWithGraph::getGraphType () const
{
    return ( SKGTableWithGraph::GraphType ) ui.kDisplayMode->currentIndex();
}

void SKGTableWithGraph::setGraphVisible ( bool iVisible ) const
{
    ui.graph_widget->setVisible ( iVisible );
}

bool SKGTableWithGraph::isGraphVisible() const
{
    return ui.graph_widget->isVisible();
}

void SKGTableWithGraph::setTableVisible ( bool iVisible ) const
{
    ui.table_widget->setVisible ( iVisible );
}

bool SKGTableWithGraph::isTableVisible() const
{
    return ui.table_widget->isVisible();
}

void SKGTableWithGraph::setGraphTypeSelectorVisible ( bool iVisible ) const
{
    ui.kGraphTypeFrm->setVisible ( iVisible );
    if ( actShowToolBar ) actShowToolBar->setVisible ( iVisible );
}

bool SKGTableWithGraph::isGraphTypeSelectorVisible() const
{
    return ui.kGraphTypeFrm->isVisible();
}

void SKGTableWithGraph::showTable()
{
    setTableVisible ( true );
    setGraphVisible ( false );
}


void SKGTableWithGraph::showGraph()
{
    setGraphVisible ( true );
    setTableVisible ( false );
}


void SKGTableWithGraph::showTableAndGraph()
{
    setGraphVisible ( true );
    setTableVisible ( true );
}

void SKGTableWithGraph::setSelectable ( bool iSelectable )
{
    selectable=iSelectable;
}

bool SKGTableWithGraph::isSelectable() const
{
    return selectable;
}

#include "skgtablewithgraph.moc"

