/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file implements classes SKGCategoryObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgcategoryobject.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"

#include <klocale.h>

SKGCategoryObject::SKGCategoryObject ( const SKGDocument* iDocument, int iID ) : SKGNamedObject ( iDocument, "v_category", iID )
{
}

SKGCategoryObject::~SKGCategoryObject()
{
}

SKGCategoryObject::SKGCategoryObject ( const SKGCategoryObject& iObject )
                : SKGNamedObject ( iObject )
{
}

SKGCategoryObject::SKGCategoryObject ( const SKGObjectBase& iObject )

{
        if ( iObject.getRealTable() =="category" ) {
                copyFrom ( iObject );
        } else {
                *this=SKGNamedObject::SKGNamedObject ( iObject.getDocument(), "v_category", iObject.getID() );
        }
}

const SKGCategoryObject& SKGCategoryObject::operator= ( const SKGObjectBase& iObject )
{
        copyFrom ( iObject );
        return *this;
}

SKGError SKGCategoryObject::setName ( const QString& iName )
{
        SKGError err;
        if ( iName.contains ( OBJECTSEPARATOR ) ) err=SKGError ( ERR_FAIL,  i18nc("Error message", "The name can not contain [%1]" , QString(OBJECTSEPARATOR) ) );
        else err=SKGNamedObject::setName ( iName );
        return err;
}

QString SKGCategoryObject::getWhereclauseId() const
{
        //Could we use the id
        QString output = SKGObjectBase::getWhereclauseId();
        if ( output.isEmpty() ) {
                if ( ! ( getAttribute ( "t_name" ).isEmpty() ) ) {
                        output = "t_name='"+getAttribute ( "t_name" ) +'\'';
                }
                if ( ! ( getAttribute ( "r_category_id" ).isEmpty() ) ) {
                        if ( !output.isEmpty() ) output+=" AND ";
                        output += "r_category_id="+getAttribute ( "r_category_id" );
                }
        }
        return output;
}

QString SKGCategoryObject::getFullName() const
{
        return getAttribute ( "t_fullname" );
}

SKGError SKGCategoryObject::createPathCategory ( const SKGDocumentBank* iDocument,
                const QString& iFullPath,
                SKGCategoryObject& oCategory,
                bool iSendPopupMessageOnCreation )
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGCategoryObject::createPathCategory", err );

        //Check if category is already existing
	if(iFullPath.isEmpty())
	{
	  oCategory=SKGCategoryObject(NULL, 0);
	}
	else
	{
	    SKGObjectBase::getObject ( iDocument, "v_category", "t_fullname='"+SKGServices::stringToSqlString ( iFullPath ) +'\'', oCategory );
	    if ( oCategory.getID() ==0 ) {
		    //No, we have to create it
		    //Search category separator
		    int posSeparator=iFullPath.lastIndexOf ( OBJECTSEPARATOR );
		    if ( posSeparator==-1 ) {
			    //Not found, so it's only a single category
			    //Search if object already exist
			    //We don't catch the error because it's just to know
			    //if the object exists
			    SKGObjectBase::getObject ( iDocument, "v_category", "t_name='"+SKGServices::stringToSqlString ( iFullPath ) +
						      "' AND (r_category_id=0 OR r_category_id IS NULL)", oCategory );

			    if ( oCategory.getID() ==0 ) {
				    //Doesn't exist, we have to create it
				    oCategory = SKGCategoryObject ( ( SKGDocumentBank* ) iDocument );
				    err=oCategory.setName ( iFullPath );
				    if ( err.isSucceeded() ) err=oCategory.save();
			    }
		    } else {
			    //Get first and second parts of the branch
			    QString first = iFullPath.mid ( 0,posSeparator );
			    QString second = iFullPath.mid ( posSeparator+strlen ( OBJECTSEPARATOR ), iFullPath.length()-posSeparator-strlen ( OBJECTSEPARATOR ) );

			    //Get first category
			    SKGCategoryObject FirstCategory;
			    err=SKGCategoryObject::createPathCategory ( iDocument, first, FirstCategory );

			    if ( err.isSucceeded() ) {
				    //Get second category
				    err=FirstCategory.addCategory ( oCategory );

				    //Add second under first
				    if ( err.isSucceeded() ) err = oCategory.setName ( second );

				    //save
				    if ( err.isSucceeded() ) err = oCategory.save();
			    }
		    }

		    if ( err.isSucceeded() && iSendPopupMessageOnCreation && iDocument ) {
			    ( ( SKGDocumentBank* ) iDocument )->sendMessage (  i18nc( "Information message", "Category [%1] has been created" , iFullPath ), true );
		    }
	    }
	}

        return err;
}

SKGError SKGCategoryObject::addCategory ( SKGCategoryObject& oCategory )
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGCategoryObject::addCategory", err );
	if ( getID() == 0 )  err=SKGError ( ERR_FAIL,  i18nc("Error message", "%1 failed because linked object is not yet saved in the database." , QString("SKGCategoryObject::addCategory") ) );
        else {
                oCategory = SKGCategoryObject ( ( SKGDocumentBank* ) getDocument() );
                err = oCategory.setAttribute ( "r_category_id", SKGServices::intToString ( getID() ) );
        }
        return err;
}

SKGError SKGCategoryObject::setParentCategory ( const SKGCategoryObject& iCategory )
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGCategoryObject::setParentCategory", err );
	if ( iCategory.getID() ==0 ) err=SKGError ( ERR_FAIL,  i18nc("Error message", "%1 failed because linked object is not yet saved in the database." , QString("SKGCategoryObject::setParentCategory") ) );
        else {
                //Check if it's a loop
                SKGCategoryObject current=iCategory;
                do {
                        if ( current==*this ) {
                                err=SKGError ( ERR_FAIL,  i18nc("Error message", "You cannot create a loop." ) );
                        } else {
                                SKGCategoryObject parent;
                                current.getParentCategory ( parent );
                                current=parent;
                        }
                } while ( err.isSucceeded() && current.getID() !=0 );

                if ( err.isSucceeded() ) err = setAttribute ( "r_category_id", SKGServices::intToString ( iCategory.getID() ) );
        }
        return err;
}

SKGError SKGCategoryObject::removeParentCategory()
{
        return setAttribute ( "r_category_id", "" );
}

SKGError SKGCategoryObject::getParentCategory ( SKGCategoryObject& oCategory ) const
{
        SKGError err;
        QString parent_id=getAttribute ( "r_category_id" );
        if ( !parent_id.isEmpty() ) err=getObject ( getDocument(), "v_category", "id="+parent_id , oCategory );
        return err;
}

SKGError SKGCategoryObject::getRootCategory ( SKGCategoryObject& oCategory ) const
{
        SKGError err;
        SKGCategoryObject parent;
        err=getParentCategory ( parent );
        if ( err.isSucceeded() ) {
                if ( !parent.exist() ) {
                        //No parent
                        oCategory=*this;
                } else {
                        //Parent exist
                        err=parent.getRootCategory ( oCategory );
                }
        }
        return err;
}


SKGError SKGCategoryObject::getCategories ( SKGListSKGObjectBase& oCategoryList ) const
{
        return getObjects ( getDocument(), "v_category",
                            "r_category_id=" + SKGServices::intToString ( getID() ),
                            oCategoryList );
}
#include "skgcategoryobject.moc"
