/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGSERVICES_H
#define SKGSERVICES_H
/** @file
 * This file is part of Skrooge and defines classes SKGDocument.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgerror.h"
#include "skgdefine.h"

#include <QStringList>
#include <QHash>
#include <QDateTime>

/**
 * @def SQLLITEERROR
 * Must be used to generate a SKGError on an error coming from sqlite's API
 * Example of usage:
 * @code
 * int rc=QSqlDatabase_close ( currentDatabase );
 * if(rc!=SQLITE_OK) err=SKGError(SQLLITEERROR+rc, QSqlDatabase_errmsg(currentDatabase));
 * @see SKGError
 * @endcode

 */
#define SQLLITEERROR 10000

class QSqlDatabase;
class SKGDocument;
class QTextStream;

/**
 * A map of strings ==> SKGQStringQStringMap
 */
typedef QHash<QString, QString> SKGQStringQStringMap;

/**
 * the iterator for SKGQStringQStringMap
 */
typedef QHash<QString, QString>::Iterator SKGQStringQStringMapIterator;

/**
 * A list of QStringList ==> SKGStringListList
 */
typedef QList<QStringList> SKGStringListList;

/**
 * the iterator for SKGStringListList
 */
typedef QList<QStringList>::Iterator SKGStringListListIterator;

/**
 * A list of int ==> SKGIntList
 */
typedef QList<int> SKGIntList;

/**
 * the iterator for SKGIntList
 */
typedef QList<int>::Iterator SKGIntListIterator;

/**
 * Describe an attribute
 */
struct SKGAttributeInfo {
        QString name; 			/**< The name of the attribute */
        QString type; 			/**< The type of the attribute */
        bool notnull; 			/**< To know if the attribute can be null or not */
        QString defaultvalue; 		/**< The default value of the attribute */
} ;

/**
 * A list of SKGAttributeInfo ==> SKGAttributesList
 */
typedef QList<SKGAttributeInfo> SKGAttributesList;

/**
 * the iterator for SKGAttributesList
 */
typedef QList<SKGAttributeInfo>::Iterator SKGAttributesListIterator;

/**
* This class containing various static services
*/
class KDE_EXPORT SKGServices : public QObject
{
        Q_OBJECT;
        Q_ENUMS(DumpMode);

public:
        /**
         * This structure defines dump modes
         */
        enum DumpMode {
                DUMP_CSV, /**< To dump in CSV mode */
                DUMP_TEXT  /**< To dump in DUMP_TEXT mode */
        };

        /**
         * Get environment variable
         * @param iAttribute name of the variable
         * @return value of the variable
         */
        static QString getenv(const QString& iAttribute);

        /**
        * Convert a integer into a QString
        * @param iNumber the integer to convert
        * @return the converted QString
        */
        static QString intToString(int iNumber);

        /**
         * Convert a QString into an integer
         * @param iNumber the QString to convert
         * @return the converted integer
         */
        static int stringToInt(const QString& iNumber);

        /**
         * Convert a QString into a csv QString
         * @param iNumber the QString to convert
         * @return the converted csv string
         */
        static QString stringToCsv(const QString& iNumber);

        /**
         * Convert a QStringList into a csv QString
         * @param iList the QStringList to convert
         * @return the converted csv string
         */
        static QString stringsToCsv(const QStringList& iList);

        /**
         * Convert a double into a QString
         * @param iNumber the double to convert
         * @return the converted QString
         */
        static QString doubleToString(double iNumber);

        /**
         * Convert a QString into an double
         * @param iNumber the QString to convert
         * @return the converted double
         */
        static double stringToDouble(const QString& iNumber);

        /**
         * Convert a QDateTime into a QString ready for sql order.
        * The output format is "%Y-%m-%d %H-%M-%S"
        * @param iDateTime the time
        * @return the converted QString
        */
        static QString timeToString(const QDateTime& iDateTime);

        /**
         * Convert a QString to QDateTime.
         * @param iDateString the time. The format must be "%Y-%m-%d %H-%M-%S or "%Y-%m-%d"
         * @return the converted time_t
         */
        static QDateTime stringToTime(const QString& iDateString);

        /**
         * Convert a QDateTime into a QString ready for sql order.
         * The output format is "%Y-%m-%d"
         * @param iDateTime the time
         * @return the converted QString
         */
        static QString dateToSqlString(const QDateTime& iDateTime);

        /**
         * Format a date.
         * The output format is "YYYY-MM-DD"
         * @param iDate the QString representing the input date
         * @param iFormat the format of the input date. Must be in the following list:
         * "YYYYMMDD", "YYYY-MM-DD"
         * "MM/DD/YY", "MM-DD-YY", "DD/MM/YY", "DD-MM-YY"
         * "MM/DD/YYYY", "MM-DD-YYYY", "DD/MM/YYYY", "DD-MM-YYYY"
         * @return the converted QString
         */
        static QString dateToSqlString(const QString& iDate, const QString& iFormat);

        /**
         * Get the format of a list of string dates
         * @param iDates a list of string dates
         * @return the supported format of all of these dates.
         */
        static QString getDateFormat(const QStringList& iDates);

        /**
         * Return string ready for sql order
         * @param iString the string
         * @return the escaped string
         */
        static QString stringToSqlString(const QString& iString);

        /**
         * Split a csv line
         * @param iString the csv line
         * @param iSeparator the csv separator
         * @param iCoteDefineBlock to indicate if character " define a block
         * @return the list of all values in this csv line
         */
        static QStringList splitCSVLine(const QString& iString, const QChar& iSeparator=';', bool iCoteDefineBlock=true);

        /**
         * To dump a table
         * @param iTable A table
         * @param iMode dump mode
         * @return the dump of the table
         */
        static QStringList tableToDump(const SKGStringListList& iTable, DumpMode iMode);

        /**
         * Get the table name corresponding to a table or a view
         * @param iTable table or view
         * @return table name corresponding
         */
        static QString getRealTable(const QString& iTable);

        /**
         * Copy a sqlite database from memory to file or from file to memory.
         * It's used to do a load or a save
         * @param iFileDb the sqlite pointer corresponding to iFileDbFile
         * @param iMemoryDb the sqlite pointer corresponding to  memory database
         * @param iFromFileToMemory
         *        true: the copy is done from iFileDb to iMemoryDb (needed of a load)
         *        false: the copy is done from iMemoryDb to iFileDb (needed of a save)
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError copySqliteDatabase(QSqlDatabase* iFileDb, QSqlDatabase* iMemoryDb, bool iFromFileToMemory);

        /**
         * Execute a sqlite order
         * @param iDocument the SKGDocument document
         * @param iSqlOrder the sql order
         * @param iLastId to retrieve the id of the lest created object
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError executeSqliteOrder(const SKGDocument* iDocument, const QString& iSqlOrder, int* iLastId=NULL);

        /**
         * Execute a sqlite order
         * @param iDb a database pointer
         * @param iSqlOrder the sql order
         * @param iLastId to retrieve the id of the lest created object
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError executeSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, int* iLastId=NULL);

        /**
        * Execute a select sqlite order and return the result in @p oResult
        * @param iDocument the SKGDocument document
        * @param iSqlOrder the sql order
        * @param oResult the result of the select. It's a vector of vector of QString
        * @return An object managing the error
        *   @see SKGError
        */
        static SKGError executeSelectSqliteOrder(const SKGDocument* iDocument, const QString& iSqlOrder, SKGStringListList& oResult);

        /**
        * Execute a select sqlite order and return the result in @p oResult
        * @param iDb A database pointer
        * @param iSqlOrder the sql order
        * @param oResult the result of the select. It's a vector of vector of QString
        * @return An object managing the error
        *   @see SKGError
        */
        static SKGError executeSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, SKGStringListList& oResult);

        /**
        * dump a select sqlite order
        * @param iDocument the SKGDocument document
        * @param iSqlOrder the sql order
        * @param oStream the output stream, NULL for std output (cout)
        * @param iMode dump mode
        * @return An object managing the error
        *   @see SKGError
        */
        static SKGError dumpSelectSqliteOrder(const SKGDocument* iDocument, const QString& iSqlOrder, QTextStream* oStream=NULL, DumpMode iMode=DUMP_TEXT);

        /**
        * dump a select sqlite order
        * @param iDb A database pointer
        * @param iSqlOrder the sql order
        * @param oStream the output stream, NULL for std output (cout)
        * @param iMode dump mode
        * @return An object managing the error
        *   @see SKGError
        */
        static SKGError dumpSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, QTextStream* oStream=NULL, DumpMode iMode=DUMP_TEXT);
        /**
         * dump a select sqlite order
         * @param iDocument the SKGDocument document
         * @param iSqlOrder the sql order
         * @param oResult the output
         * @param iMode dump mode
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError dumpSelectSqliteOrder(const SKGDocument* iDocument, const QString& iSqlOrder, QString& oResult, DumpMode iMode=DUMP_TEXT);

        /**
         * dump a select sqlite order
         * @param iDb A database pointer
         * @param iSqlOrder the sql order
         * @param oResult the output
         * @param iMode dump mode
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError dumpSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, QString& oResult, DumpMode iMode=DUMP_TEXT);

        /**
         * dump a select sqlite order
         * @param iDocument the SKGDocument document
         * @param iSqlOrder the sql order
         * @param oResult the output
         * @param iMode dump mode
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError dumpSelectSqliteOrder(const SKGDocument* iDocument, const QString& iSqlOrder, QStringList& oResult, DumpMode iMode=DUMP_TEXT);

        /**
         * dump a select sqlite order
         * @param iDb A database pointer
         * @param iSqlOrder the sql order
         * @param oResult the output
         * @param iMode dump mode
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError dumpSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, QStringList& oResult, DumpMode iMode=DUMP_TEXT);

        /**
         * Retrieve all distinct values of an attribute of a table
         * @param iDocument the document where to look for
         * @param iTable the table where to look for
         * @param iAttribute the attribute wanted (only one)
         * @param iWhereClause a whereclause
         * @param oResult the output result
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError getDistinctValues(const SKGDocument* iDocument, const QString& iTable, const QString& iAttribute, const QString& iWhereClause, QStringList& oResult);

        /**
         * Retrieve all distinct values of an attribute of a table
         * @param iDocument the document where to look for
         * @param iTable the table where to look for
         * @param iAttribute the attribute wanted (only one)
         * @param oResult the output result
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError getDistinctValues(const SKGDocument* iDocument, const QString& iTable, const QString& iAttribute, QStringList& oResult);

        /**
         * Retrieve description of each attribute of a table
         * @param iDocument the document where to look for
         * @param iTable the table where to look for
         * @param oResult the output result
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError getAttributesDescription(const SKGDocument* iDocument, const QString& iTable, SKGAttributesList& oResult);

        /**
         * Retrieve list of attributes
         * @param iDocument the document where to look for
         * @param iTable the table where to look for
         * @param oResult the output result
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError getAttributesList(const SKGDocument* iDocument, const QString& iTable, QStringList& oResult);

        /**
         * Retrieve list of tables
         * @param iDocument the document where to look for
         * @param oResult the output result
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError getTablesList(const SKGDocument* iDocument, QStringList& oResult);

        /**
         * Return the current time in millisecond
         * @return current
         */
        static double getMicroTime();

        /**
         * Encrypt or decrypt a file
         * @param iFileSource source file
         * @param iFileTarget target file
         * @param iPassword password
         * @param iEncrypt true to encryot, false to decrypt
         * @return An object managing the error
         *   @see SKGError
         */
        static SKGError cryptFile(const QString& iFileSource, const QString& iFileTarget,
                                  const QString& iPassword, bool iEncrypt=true);

        /**
         * To enable, disable sql traces
         */
        static int SKGSqlTraces;

private:
        Q_DISABLE_COPY(SKGServices);
        SKGServices();
        ~SKGServices();
        static SKGError lastCallbackError;
};
#endif
