/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is part of Skrooge and implements classes SKGAccountObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgaccountobject.h"
#include "skgbankobject.h"
#include "skgoperationobject.h"
#include "skgunitobject.h"
#include "skgdocumentbank.h"

SKGAccountObject::SKGAccountObject(const SKGDocument* iDocument, int iID): SKGNamedObject(iDocument, "v_account", iID) {}

SKGAccountObject::~SKGAccountObject() {}

SKGAccountObject::SKGAccountObject(const SKGAccountObject& iObject)
                : SKGNamedObject(iObject)
{}

SKGAccountObject::SKGAccountObject(const SKGNamedObject& iObject)
                : SKGNamedObject(iObject.getDocument(), "v_account", iObject.getID())
{
        if (iObject.getRealTable()=="account") {
                copyFrom(iObject);
        } else {
                *this=SKGNamedObject::SKGNamedObject(iObject.getDocument(), "v_account", iObject.getID());
        }
}

SKGAccountObject::SKGAccountObject(const SKGObjectBase& iObject)
{
        if (iObject.getRealTable()=="account") {
                copyFrom(iObject);
        } else {
                *this=SKGNamedObject::SKGNamedObject(iObject.getDocument(), "v_account", iObject.getID());
        }
}

const SKGAccountObject& SKGAccountObject::operator= (const SKGObjectBase& iObject)
{
        copyFrom(iObject);
        return *this;
}

SKGError SKGAccountObject::setBank(const SKGBankObject& iBank)
{
        return setAttribute("rd_bank_id", SKGServices::intToString(iBank.getID()));
}

SKGError SKGAccountObject::getBank(SKGBankObject& oBank) const
{
        SKGError err = getObject(getDocument(), "v_bank", "id=" + getAttribute("rd_bank_id"), oBank);
        return err;
}

SKGError SKGAccountObject::setNumber(const QString& iNumber)
{
        return setAttribute("t_number", iNumber);
}

QString SKGAccountObject::getNumber() const
{
        return getAttribute("t_number");
}

SKGError SKGAccountObject::setComment(const QString& iComment)
{
        return setAttribute("t_comment", iComment);
}

QString SKGAccountObject::getComment() const
{
        return getAttribute("t_comment");
}

SKGError SKGAccountObject::setAgencyNumber(const QString& iNumber)
{
        return setAttribute("t_agency_number", iNumber);
}

QString SKGAccountObject::getAgencyNumber() const
{
        return getAttribute("t_agency_number");
}

SKGError SKGAccountObject::setAgencyAddress(const QString& iAddress)
{
        return setAttribute("t_agency_address", iAddress);
}

QString SKGAccountObject::getAgencyAddress() const
{
        return getAttribute("t_agency_address");
}

SKGError SKGAccountObject::addOperation(SKGOperationObject& oOperation)
{
        SKGError err;
        if (getID() == 0) err=SKGError(ERR_FAIL, tr("%1 failed because linked object is not yet saved in the database.").arg("SKGAccountObject::addOperation"));
        else {
                oOperation = SKGOperationObject(getDocument());
                err = oOperation.setParentAccount(*this);
        }
        return err;
}

double SKGAccountObject::getCurrentAmount() const
{
        return SKGServices::stringToDouble(getAttribute("f_CURRENTAMOUNT"));
}

double SKGAccountObject::getAmount(const QDate& iDate) const
{
        double output=0;
        SKGStringListList listTmp;
        SKGError err=SKGServices::executeSelectSqliteOrder(getDocument(),"SELECT TOTAL(f_QUANTITY), rc_unit_id FROM v_operation  WHERE "
                        "d_date<='"+SKGServices::dateToSqlString(QDateTime(iDate))+ "' AND rd_account_id="+SKGServices::intToString(getID())+
                        " GROUP BY rc_unit_id",
                        listTmp);
        int nb=listTmp.count();
        for (int i=1; err.isSucceeded() && i<nb ;++i) {
                QString quantity=listTmp.at(i).at(0);
                QString unitid=listTmp.at(i).at(1);

                double coef=1;
                QString val=getDocument()->getCachedValue("unitvalue-"+unitid);
                if (!val.isEmpty()) {
                        //Yes
                        coef=SKGServices::stringToDouble(val);
                } else {
                        //No
                        SKGUnitObject unit(getDocument(), SKGServices::stringToInt(unitid));
                        coef=unit.getAmount(iDate);
                }

                output+=coef*SKGServices::stringToDouble(quantity);
        }
        return output;
}

SKGError SKGAccountObject::setType(SKGAccountObject::AccountType iType)
{
        SKGError err;
        if (err.isSucceeded()) err=setAttribute("t_type", (iType==CURRENT ? "C" : (iType==CREDITCARD ? "D" : (iType==ACTIF ? "A" : (iType==INVESTMENT ? "I" : "O")))));
        return err;
}

SKGAccountObject::AccountType SKGAccountObject::getType() const
{
        QString typeString=getAttribute("t_type");
        return (typeString=="C" ? CURRENT : (typeString=="D" ? CREDITCARD : (typeString=="A" ? ACTIF : (typeString=="I" ? INVESTMENT : OTHER))));
}

SKGError SKGAccountObject::setClosed(bool iClosed)
{
        return setAttribute("t_close", iClosed ? "Y" :"N");
}

bool SKGAccountObject::isClosed() const
{
        return (getAttribute("t_close")=="Y" ? true : false);
}

SKGError SKGAccountObject::getUnit(SKGUnitObject& oUnit) const
{
        SKGObjectBase::SKGListSKGObjectBase units;
        SKGError err=SKGObjectBase::getObjects(getDocument(), "v_unit",
                                               "t_type IN ('1', '2', 'C') AND (SELECT count(1) from v_operation_display where rc_unit_id=v_unit.id AND rd_account_id="+SKGServices::intToString(getID())+")>0 ORDER BY t_type", units);
        int nb=units.count();
        if (nb) oUnit=units.at(0);
        return err;
}
#include "skgaccountobject.moc"
