/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationplugin.h"
#include "skgoperationpluginwidget.h"
#include "skgtraces.h"
#include "skgoperationobject.h"
#include "skgsuboperationobject.h"
#include "skgtransactionmng.h"
#include "skgmainpanel.h"
#include "skrooge_operation.h"

#include <kactioncollection.h>
#include <kstandardaction.h>

#include <QDomDocument>

K_PLUGIN_FACTORY(SKGOperationPluginFactory, registerPlugin<SKGOperationPlugin>();)
K_EXPORT_PLUGIN(SKGOperationPluginFactory("skrooge_operation", "skrooge_operation"))

SKGOperationPlugin::SKGOperationPlugin(QObject* iParent, const QVariantList& /*iArg*/) : SKGInterfacePlugin(iParent)
{
        SKGTRACEIN(10, "SKGOperationPlugin::SKGOperationPlugin");
}

SKGOperationPlugin::~SKGOperationPlugin()
{
        SKGTRACEIN(10, "SKGOperationPlugin::~SKGOperationPlugin");
        parent=NULL;
        currentBankDocument=NULL;
}

void SKGOperationPlugin::setupActions(SKGMainPanel* iParent, SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGOperationPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;
        parent=iParent;

        setComponentData( SKGOperationPluginFactory::componentData() );
        setXMLFile("skrooge_operation.rc");

        //Menu
        duplicateAction = new KAction(KIcon("skrooge_duplicate"), i18n("&Duplicate"), this);
        connect(duplicateAction, SIGNAL(triggered(bool)), this, SLOT(actionDuplicate()));
        actionCollection()->addAction( QLatin1String("edit_duplicate_operation"), duplicateAction );
        duplicateAction->setShortcut(Qt::CTRL+Qt::Key_D);

        iParent->registedGlobalAction("edit_duplicate_operation", duplicateAction);

        switchToPointedAction = new KAction(KIcon("dialog-ok"), i18n("&Point"), this);
        connect(switchToPointedAction, SIGNAL(triggered(bool)), this, SLOT(actionSwitchToPointed()));
        actionCollection()->addAction( QLatin1String("edit_point_selected_operation"), switchToPointedAction );
        switchToPointedAction->setShortcut(Qt::CTRL+Qt::Key_R);

        iParent->registedGlobalAction("edit_point_selected_operation", switchToPointedAction);

        switchToCheckedAction = new KAction(KIcon("dialog-ok-apply"), i18n("Pointed to &checked"), this);
        connect(switchToCheckedAction, SIGNAL(triggered(bool)), this, SLOT(actionSwitchToChecked()));
        actionCollection()->addAction( QLatin1String("edit_switch_to_checked"), switchToCheckedAction );
        switchToCheckedAction->setShortcut(Qt::CTRL+Qt::ALT+Qt::Key_R);

        iParent->registedGlobalAction("edit_switch_to_pointed", switchToCheckedAction);

        fastEditionAction = new KAction(KIcon("games-solve"), i18n("&Fast edition"), this);
        actionCollection()->addAction( QLatin1String("fast_edition"), fastEditionAction );
        fastEditionAction->setEnabled(false);
        fastEditionAction->setShortcut(Qt::Key_F10);

        iParent->registedGlobalAction("fast_edition", fastEditionAction);

        switchBookmark = new KAction(KIcon("rating"), i18n("Switch &bookmark"), this);
        connect(switchBookmark, SIGNAL(triggered(bool)), this, SLOT(actionSwitchBookmark()));
        actionCollection()->addAction( QLatin1String("edit_switch_bookmark"), switchBookmark );
        switchBookmark->setShortcut(Qt::CTRL+Qt::Key_B);

        iParent->registedGlobalAction("edit_switch_bookmark", switchBookmark);

        splitOperationAction = new KAction(KIcon("skrooge_split"), i18n("&Split"), this);
        connect(splitOperationAction, SIGNAL(triggered(bool)), this, SLOT(actionSplitOperation()));
        actionCollection()->addAction( QLatin1String("edit_split_operation"), splitOperationAction );
        splitOperationAction->setShortcut(Qt::CTRL+Qt::Key_Slash);

        iParent->registedGlobalAction("edit_split_operation", splitOperationAction);

        openBookmarks = new KAction(KIcon("rating"), i18n("Open &bookmarks"), this);
        connect(openBookmarks, SIGNAL(triggered(bool)), this, SLOT(actionOpenBookmarks()));
        actionCollection()->addAction( QLatin1String("view_open_bookmarks"), openBookmarks );
        openBookmarks->setShortcut(Qt::CTRL+Qt::SHIFT+Qt::Key_B);

        iParent->registedGlobalAction("edit_open_bookmarks", openBookmarks);
}

void SKGOperationPlugin::refresh()
{
        SKGTRACEIN(10, "SKGOperationPlugin::refresh");
        SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();

        switchToCheckedAction->setEnabled(currentBankDocument->getDatabase()!=NULL);

        if (selection.count()>0) {
                bool onOperation=(selection.at(0).getRealTable()=="operation");
                duplicateAction->setEnabled(onOperation);
                switchBookmark->setEnabled(onOperation);
                switchToPointedAction->setEnabled(onOperation);
                splitOperationAction->setEnabled(onOperation && selection.count()==1);
        } else {
                duplicateAction->setEnabled(false);
                switchBookmark->setEnabled(false);
                splitOperationAction->setEnabled(false);
                switchToPointedAction->setEnabled(false);
        }

        int nb=0;
        SKGObjectBase::getNbObjects(currentBankDocument, "operation", "t_status='P'", nb);
        switchToCheckedAction->setEnabled(nb);
}

void SKGOperationPlugin::close()
{
        SKGTRACEIN(10, "SKGOperationPlugin::close");
}

SKGTabWidget* SKGOperationPlugin::getWidget()
{
        SKGTRACEIN(10, "SKGOperationPlugin::getWidget");
        return new SKGOperationPluginWidget(parent, (SKGDocumentBank*) currentBankDocument, fastEditionAction);
}

QWidget* SKGOperationPlugin::getPreferenceWidget()
{
        SKGTRACEIN(10, "SKGMonthlyPlugin::getPreferenceWidget");
        QWidget* widget=new QWidget();
        ui.setupUi(widget);
        return widget;
}

KConfigSkeleton* SKGOperationPlugin::getPreferenceSkeleton()
{
        return skrooge_operation::self();
}

SKGError SKGOperationPlugin::savePreferences() const
{
        return SKGError();
}

QString SKGOperationPlugin::title() const
{
        return i18n("Operations");
}

QString SKGOperationPlugin::icon() const
{
        return "view-pim-tasks";
}

QString SKGOperationPlugin::statusTip () const
{
        return i18n("Operations management (creation, update ...)");
}

QString SKGOperationPlugin::toolTip () const
{
        return i18n("Operations management");
}


int SKGOperationPlugin::getOrder() const
{
        return 15;
}

QStringList SKGOperationPlugin::tips() const
{
        QStringList output;
        output.push_back(i18n("<p>... you can press <strong>+</strong>, <strong>-</strong>, <strong>CTRL +</strong> or <strong>CTRL -</strong> to quickly change dates.</p>"));
        output.push_back(i18n("<p>... you can update many operations in one shot.</p>"));
        output.push_back(i18n("<p>... you can double click on an operation to show or edit sub operations.</p>"));
        output.push_back(i18n("<p>... you can duplicate an operation including complex operation (splitted, grouped, ...).</p>"));
        return output;
}

bool SKGOperationPlugin::isInContext() const
{
        return true;
}

void SKGOperationPlugin::actionSwitchToChecked()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGOperationPlugin::actionSwitchToChecked",err);

        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        SKGObjectBase::SKGListSKGObjectBase list;
        err=SKGObjectBase::getObjects(currentBankDocument, "operation", "t_status='P'", list);
        int nb=list.count();
        if (err.isSucceeded()) {
                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18n("Switch to checked"), err, nb);
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGOperationObject op=list[i];
                        err=op.setStatus(SKGOperationObject::CHECKED);
                        if (err.isSucceeded()) err=op.save();
                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i+1);
                }
        }
        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Operation checked."));
        else err.addError(ERR_FAIL, i18n("Switch failed"));

        QApplication::restoreOverrideCursor();

        //Display error
        parent->displayErrorMessage(err);
}

void SKGOperationPlugin::actionSwitchToPointed()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGOperationPlugin::actionSwitchToPointed",err);
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
        int nb=selection.count();
        {
                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18n("Switch to pointed"), err, nb);

                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGOperationObject operationObj=selection.at(i);
                        if (err.isSucceeded()) err=operationObj.setStatus(SKGOperationObject::POINTED);
                        if (err.isSucceeded()) err=operationObj.save();

                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i+1);
                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18n("Operation pointed."));
        else err.addError(ERR_FAIL, i18n("Switch failed"));

        //Display error
        parent->displayErrorMessage(err);
}

void SKGOperationPlugin::actionSwitchBookmark()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGOperationPlugin::actionSwitchBookmark",err);
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
        int nb=selection.count();
        {
                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18n("Operation bookmark"), err, nb);

                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGOperationObject operationObj=selection.at(i);
                        if (err.isSucceeded()) err=operationObj.bookmark(!operationObj.isBookmarked());
                        if (err.isSucceeded()) err=operationObj.save();

                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i+1);
                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded())  err=SKGError(0, i18n("Operation bookmarked."));
        else err.addError(ERR_FAIL, i18n("Operation bookmark failed"));

        //Display error
        parent->displayErrorMessage(err);
}

void SKGOperationPlugin::actionDuplicate()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGOperationPlugin::actionDuplicate",err);
        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
        int nb=selection.count();
        {
                SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18n("Duplicate operation"), err, nb);

                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                for (int i=0; err.isSucceeded() && i<nb; ++i) {
                        SKGOperationObject operationObj=selection.at(i);
                        SKGOperationObject dup;
                        if (err.isSucceeded()) err=operationObj.duplicate(dup);

                        if (err.isSucceeded()) err=currentBankDocument->stepForward(i+1);
                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) {
                err=SKGError(0, i18n("Operation duplicated."));
                //TODO ui.kOperationView->selectObject(operation.getUniqueID());
        } else err.addError(ERR_FAIL, i18n("Duplicate operation failed"));

        //Display error
        parent->displayErrorMessage(err);
}

void SKGOperationPlugin::actionOpenBookmarks()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGOperationPlugin::actionOpenBookmarks",err);
        QString wc="t_bookmarked='Y'";
        QString title=i18n("Operations bookmarked");

        //Call operation plugin
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);
        root.setAttribute("operationTable", "v_operation_consolidated");
        root.setAttribute("operationWhereClause", wc);
        root.setAttribute("title", title);
        root.setAttribute("title_icon", "view-statistics");

        parent->setNewTabContent(parent->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
}

void SKGOperationPlugin::actionSplitOperation()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGOperationPlugin::actionSplitOperation",err);

        //Get Selection
        SKGObjectBase::SKGListSKGObjectBase selection=parent->getSelectedObjects();
        int nb=selection.count();
        if (nb==1) {
                SKGBEGINTRANSACTION(*currentBankDocument, i18n("Operation split"), err);

                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                SKGOperationObject operationObj=selection.at(0);
                SKGSubOperationObject subOp;
                err=operationObj.addSubOperation(subOp);
                if (err.isSucceeded()) err=subOp.save();
                if (err.isSucceeded()) SKGOperationPluginWidget::openOperation(operationObj, parent);
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded())  err=SKGError(0, i18n("Operation splitted."));
        else err.addError(ERR_FAIL, i18n("Operation split failed"));

        //Display error
        parent->displayErrorMessage(err);
}
#include "skgoperationplugin.moc"
