/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file is part of Skrooge and implements classes SKGError.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
*/
#include "skgerror.h"

SKGError::SKGError()
                :rc(0), message(""), previousError(NULL)
{}

SKGError::SKGError(int iRc, const QString& iMessage)
                :rc(iRc), message(iMessage), previousError(NULL)
{}

SKGError::SKGError(const SKGError& iError)
{
        rc = iError.rc;
        message = iError.message;
        if (iError.previousError != NULL) previousError = new SKGError(*iError.previousError);
        else previousError = NULL;
}

SKGError::~SKGError()
{
        if (previousError) {
                delete previousError;
                previousError = NULL;
        }
}

const SKGError& SKGError::operator= (const SKGError& iError)
{
        if (previousError) {
                delete previousError;
                previousError = NULL;
        }

        rc = iError.rc;
        message = iError.message;
        if (iError.previousError != NULL) previousError = new SKGError(*iError.previousError);

        return *this;
}

bool SKGError::isFailed() const
{
        return (rc > 0);
}

bool SKGError::isSucceeded() const
{
        return (rc <= 0);
}

bool SKGError::isWarning() const
{
        return (rc < 0);
}

int SKGError::getReturnCode() const
{
        return rc;
}

void SKGError::setReturnCode(int iReturnCode)
{
        rc = iReturnCode;
}

void SKGError::setMessage(const QString & iMessage)
{
        message = iMessage;
}

QString SKGError::getMessage() const
{
        return message;
}

QString SKGError::getFullMessage() const
{
        QString output = "[";
        output += (rc == 0 ? "SUC" : (rc > 0 ? "ERR" : "WAR"));
        output += '-';

        QString tmp;
        tmp.setNum(rc);
        output += tmp;
        output += ']';
        if (message.length()) output += ": " + message;
        return output;
}

QString SKGError::getFullMessageWithHistorical() const
{
        QString output = getFullMessage();
        if (previousError) output += '\n' + previousError->getFullMessageWithHistorical();
        return output;
}

int SKGError::getHistoricalSize() const
{
        int output = 0;
        if (previousError) output += 1 + previousError->getHistoricalSize();
        return output;
}

void SKGError::addError(int iRc, const QString & iMessage)
{
        SKGError* pe = new SKGError(*this);
        setReturnCode(iRc);
        setMessage(iMessage);
        if (previousError) delete previousError;
        previousError = pe;
}

void SKGError::addError(const SKGError& iError)
{
        addError(iError.getReturnCode(), iError.getMessage());
}

SKGError* SKGError::getPreviousError() const
{
        return previousError;
}

