/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A table view for skrooge.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtableview.h"
#include "skgtraces.h"
#include "skgobjectmodelbase.h"
#include "skgservices.h"
#include "skgdocument.h"
#include "skgerror.h"
#include "skgtransactionmng.h"
#include "skgobjectbase.h"
#include "skgmainpanel.h"

#include <kicon.h>
#include <kmenu.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kmessagebox.h>

#include <QDomDocument>
#include <QHeaderView>
#include <QSortFilterProxyModel>
#include <QBasicTimer>
#include <QEvent>
#include <QMouseEvent>
#include <QScrollBar>
#include <QApplication>
#include <QPrinter>
#include <QPainter>
#include <QDesktopServices>

SKGTableView::SKGTableView(QWidget *parent)
                : QTableView(parent),
                autoResize(true), actAutoResize(NULL),
                smoothScrolling(false), actSmoothScrolling(NULL),
                document(NULL)
{
        QHeaderView* hori=horizontalHeader();
        hori->setContextMenuPolicy(Qt::CustomContextMenu);
        connect(hori,SIGNAL(customContextMenuRequested(const QPoint & ) ),this,SLOT(showHeaderMenu( const QPoint& )));
        header_menu = new KMenu(this);
        header_menu->addTitle(i18n("Columns"));

        setContextMenuPolicy(Qt::ActionsContextMenu);

        // hori->setResizeMode(QHeaderView::ResizeToContents);
        hori->setMovable(true);

        verticalHeader()->hide();
        verticalHeader()->setDefaultSectionSize(verticalHeader()->minimumSectionSize());
        horizontalHeader()->setResizeMode(QHeaderView::Fixed);
        setWordWrap(false);

        //By default Smooth scrolling is activated
        switchSmoothScrolling();

        connect(horizontalHeader(), SIGNAL(sectionMoved (int, int, int)), this, SLOT(moveSection()), Qt::QueuedConnection);
        connect(horizontalHeader(), SIGNAL(geometriesChanged()), this, SLOT(setupHeaderMenu()), Qt::QueuedConnection);
}

SKGTableView::~SKGTableView()
{
        document=NULL;
}

bool SKGTableView::isAutoResized()
{
        return autoResize;
}

bool SKGTableView::isSmoothScrolling()
{
        return smoothScrolling;
}

QString SKGTableView::getState()
{
        SKGTRACEIN(10, "SKGTableView::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);


        SKGObjectModelBase* model=(SKGObjectModelBase*) this->model();
        QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *>(model);
        if (proxyModel) model=(SKGObjectModelBase*) proxyModel->sourceModel();

        QHeaderView* hHeader=horizontalHeader();
        if (hHeader && model) {
                root.setAttribute("sortOrder", SKGServices::intToString((int) hHeader->sortIndicatorOrder()));
                root.setAttribute("sortColumn", model->getAttribute(hHeader->sortIndicatorSection()));

                //Memorize order
                int nb=hHeader->count();
                if (nb) {
                        QString columns;
                        QString columnsSize;
                        QString columnsVisibility;
                        for (int i=0; i<nb; ++i) {
                                int idx=hHeader->logicalIndex(i);
                                if (i) columns+=';';
                                columns+=model->getAttribute(idx);

                                if (i) columnsSize+=';';
                                columnsSize+=SKGServices::intToString(hHeader->sectionSize (idx));

                                if (i) columnsVisibility+=';';
                                columnsVisibility+=(hHeader->isSectionHidden(idx) ? "N" : "Y");
                        }
                        root.setAttribute("columns", columns);
                        if (!autoResize) root.setAttribute("columnsSize", columnsSize);
                        root.setAttribute("columnsVisibility", columnsVisibility);
                        root.setAttribute("columnsAutoResize", autoResize ? "Y" : "N");
                        root.setAttribute("smoothScrolling", smoothScrolling ? "Y" : "N");
                }
        }
        root.setAttribute("showGrid", showGrid() ? "Y" : "N");
        root.setAttribute("alternatingRowColors", alternatingRowColors() ? "Y" : "N");
        return doc.toString();
}

void SKGTableView::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGTableView::setState");
        QDomDocument doc("SKGML");

        QString viewState=iState;
        if (viewState.isEmpty() && document) {
                //Get default state
                viewState=document->getParameter(parameterName);
        }

        SKGObjectModelBase* model=(SKGObjectModelBase*) this->model();
        QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *>(model);
        if (proxyModel) model=(SKGObjectModelBase*) proxyModel->sourceModel();

        if (doc.setContent(viewState)) {
                QDomElement root = doc.documentElement();

                QString sortOrder=root.attribute ( "sortOrder");
                QString sortColumn=root.attribute ( "sortColumn");
                QString columns=root.attribute("columns");
                QString columnsSize=root.attribute("columnsSize");
                QString columnsVisibility=root.attribute("columnsVisibility");
                QString columnsAutoResize=root.attribute("columnsAutoResize");
                QString smoothScrollings=root.attribute("smoothScrolling");
                QString showGrid=root.attribute("showGrid");
                QString alternatingRowColors=root.attribute("alternatingRowColors");

                //Set column order
                if (!columns.isEmpty()) {
                        QStringList listAtt=SKGServices::splitCSVLine(columns, ';');
                        QStringList sizes=SKGServices::splitCSVLine(columnsSize, ';');
                        QStringList visibilities=SKGServices::splitCSVLine(columnsVisibility, ';');

                        int nb=listAtt.count();
                        int nbvisibilities=visibilities.count();
                        int nbsizes=sizes.count();
                        for (int i=0; i<nb; ++i) {
                                if (nbvisibilities==nb) {
                                        listAtt[i]=listAtt[i]+"|"+visibilities[i];
                                        if (nbsizes==nb) {
                                                listAtt[i]=listAtt[i]+"|"+sizes[i];
                                        }
                                }

                        }
                        if (model) model->setSupportedAttributes(listAtt);
                }

                //Set autoResize
                if (!columnsAutoResize.isEmpty()) {
                        autoResize=(columnsAutoResize=="Y");
                        horizontalHeader()->setResizeMode(autoResize ? QHeaderView::Fixed : QHeaderView::Interactive);
                } else resizeColumnsToContents();

                //Set smoothScrolling
                if (!smoothScrollings.isEmpty()) {
                        smoothScrolling=(smoothScrollings=="N"); //The next line will set the right value
                        switchSmoothScrolling();
                }

                //Set sort
                if (!sortOrder.isEmpty() && !sortColumn.isEmpty()) {
                        int index=SKGServices::splitCSVLine(columns, ';').indexOf(sortColumn);
                        if (index==-1) index=model->getIndexAttribute(sortColumn);
                        if (index==-1) index=0;
                        this->sortByColumn(index, (Qt::SortOrder) SKGServices::stringToInt(sortOrder) );
                }

                //Set showGrid
                if (!showGrid.isEmpty()) setShowGrid(showGrid=="Y");

                //Set alternatingRowColors
                if (!alternatingRowColors.isEmpty()) setAlternatingRowColors(alternatingRowColors=="Y");
        } else {
                QStringList listAtt;
                if (model) model->setSupportedAttributes(listAtt);
        }
}

void SKGTableView::showHeaderMenu(const QPoint& pos)
{
        header_menu->popup(horizontalHeader()->mapToGlobal(pos));
}

void SKGTableView::setDefaultSaveParameters(const SKGDocument* iDocument, const QString& iParameterName)
{
        document=(SKGDocument*) iDocument;
        parameterName=iParameterName;
}

void SKGTableView::moveSection()
{
        setupHeaderMenu(false);
}

void SKGTableView::setupHeaderMenu(bool iRefreshColumnSize)
{
        SKGObjectModelBase* model=(SKGObjectModelBase*) this->model();
        QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *>(model);
        if (proxyModel) model=(SKGObjectModelBase*) proxyModel->sourceModel();

        header_menu->clear();

        //Processing
        QMenu* columns=header_menu->addMenu(i18n("Columns"));

        // Set right click menu
        if (model) {
                QList<SKGObjectModelBase::SKGModelTemplate> schemas=model->getSchemas();
                int nbSchemas=schemas.count();
                if (nbSchemas) {
                        QMenu* viewAppearanceMenu=columns->addMenu(KIcon("view-file-columns"), i18n("View appearance"));

                        for (int i=0; i<nbSchemas; ++i) {
                                SKGObjectModelBase::SKGModelTemplate schema=schemas.at(i);
                                QAction* act = viewAppearanceMenu->addAction(schema.name);
                                if (!schema.icon.isEmpty()) act->setIcon(KIcon(schema.icon));
                                act->setData(schema.schema);

                                connect(act, SIGNAL(triggered(bool)), this, SLOT(changeSchema()));
                        }
                }
        }

        QAction* actResize = columns->addAction(KIcon("zoom-fit-width"), i18n("Resize to content"));
        connect(actResize, SIGNAL(triggered(bool)), this, SLOT(resizeColumnsToContents()));

        actAutoResize = columns->addAction(i18n("Auto resize"));
        actAutoResize->setCheckable(true);
        actAutoResize->setChecked(autoResize);
        connect(actAutoResize, SIGNAL(triggered(bool)), this, SLOT(switchAutoResize()));

        if (model && model->supportedDragActions()==Qt::IgnoreAction) {
                actSmoothScrolling = header_menu->addAction(i18n("Smooth scrolling"));
                actSmoothScrolling->setCheckable(true);
                actSmoothScrolling->setChecked(smoothScrolling);
                connect(actSmoothScrolling, SIGNAL(triggered(bool)), this, SLOT(switchSmoothScrolling()));
        } else {
                //Disable smooth scrolling
                smoothScrolling=true;
                switchSmoothScrolling();
        }

        QAction* actShowGrid = header_menu->addAction(KIcon("view-file-columns"),  i18n("Show grid"));
        actShowGrid->setCheckable(true);
        actShowGrid->setChecked(showGrid());
        connect(actShowGrid, SIGNAL(triggered(bool)), this, SLOT(setShowGrid(bool)));

        QAction* actAlternatingRowColors = header_menu->addAction(i18n("Alternate row colors"));
        actAlternatingRowColors->setCheckable(true);
        actAlternatingRowColors->setChecked(alternatingRowColors());
        connect(actAlternatingRowColors, SIGNAL(triggered(bool)), this, SLOT(setAlternatingRowColors(bool)));

        if (document) {
                QAction* actDefault = header_menu->addAction(KIcon("document-save"), i18n("Save parameters"));
                connect(actDefault, SIGNAL(triggered(bool)), this, SLOT(saveDefaultClicked()));
        }

        columns->addSeparator();

        if (model) {
                QHeaderView* hHeader=horizontalHeader();
                int nbcol=hHeader->count();
                for (int i=0; i<nbcol; ++i) {
                        int idx=hHeader->logicalIndex(i);
                        QString col = model->headerData(idx,Qt::Horizontal,Qt::UserRole).toString();
                        QStringList values=col.split("|");

                        if (iRefreshColumnSize) {
                                if (values.count()>1) hHeader->setSectionHidden(idx, values.at(1)=="N");
                                if (values.count()>2) {
                                        int s=SKGServices::stringToInt(values.at(2));
                                        if (s>0) hHeader->resizeSection(idx, s);
                                }
                        }
                        QAction* act = columns->addAction(values.at(0));
                        act->setCheckable(true);
                        act->setChecked(!hHeader->isSectionHidden(idx));
                        act->setIcon(model->headerData(idx,Qt::Horizontal,Qt::DecorationRole).value<QIcon>());
                        act->setData(idx);

                        connect(act, SIGNAL(triggered(bool)), this, SLOT(showHideColumn()));
                }
        }
        header_menu->addSeparator();

        QMenu* exp=header_menu->addMenu(i18n("Export"));

        QAction* actPDF = exp->addAction(KIcon("application-pdf"), i18n("Export PDF..."));
        connect(actPDF, SIGNAL(triggered(bool)), this, SLOT(onExportPDF()));

        QAction* actCSV = exp->addAction(KIcon("text-csv"), i18n("Export CSV..."));
        connect(actCSV, SIGNAL(triggered(bool)), this, SLOT(onExportCSV()));

        QAction* actTXT = exp->addAction(KIcon("text-plain"), i18n("Export TXT..."));
        connect(actTXT, SIGNAL(triggered(bool)), this, SLOT(onExportTXT()));
}

void SKGTableView::saveDefaultClicked()
{
        SKGError err;
        SKGBEGINTRANSACTION(*document, i18n("Save default parameters"), err);
        err=document->setParameter(parameterName, getState());
}

void SKGTableView::switchAutoResize()
{
        autoResize=actAutoResize->isChecked();
        horizontalHeader()->setResizeMode(autoResize ? QHeaderView::Fixed : QHeaderView::Interactive);
        if (autoResize) resizeColumnsToContents();
}

void SKGTableView::switchSmoothScrolling()
{
        if (actSmoothScrolling) smoothScrolling=actSmoothScrolling->isChecked();
        else smoothScrolling=!smoothScrolling;

        QAbstractScrollArea *scrollArea = dynamic_cast<QAbstractScrollArea*>(this);
        if (scrollArea) {
                QWidget *viewport = scrollArea->viewport();
                if (smoothScrolling) {
                        viewport->installEventFilter(this);
                        scrollArea->installEventFilter(this);

                        state = SKGTableView::Steady;

                        setVerticalScrollMode(QAbstractItemView::ScrollPerPixel);
                        setHorizontalScrollMode(QAbstractItemView::ScrollPerPixel);
                } else {
                        viewport->removeEventFilter(this);
                        scrollArea->removeEventFilter(this);

                        setVerticalScrollMode(QAbstractItemView::ScrollPerItem);
                        setHorizontalScrollMode(QAbstractItemView::ScrollPerItem);
                }
        }
}

void SKGTableView::showHideColumn()
{
        QAction* sender=static_cast<QAction*>(this->sender());
        if (sender) {
                QHeaderView* hHeader=horizontalHeader();

                int idx=sender->data().toInt();
                hHeader->setSectionHidden (idx, !hHeader->isSectionHidden(idx));
        }
}

void SKGTableView::changeSchema()
{
        QStringList list;

        QAction* sender=static_cast<QAction*>(this->sender());
        if (sender)  list=SKGServices::splitCSVLine(sender->data().toString(), ';');

        SKGObjectModelBase* model=(SKGObjectModelBase*) this->model();
        QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *>(model);
        if (proxyModel) model=(SKGObjectModelBase*) proxyModel->sourceModel();
        if (model) {
                //Reset column oder
                QHeaderView* hHeader=horizontalHeader();
                int nbcol=hHeader->count();
                for (int i=0; i<nbcol; ++i) {
                        int idx=hHeader->visualIndex(i);
                        if (idx!=i) hHeader->moveSection(idx,i);
                }

                model->setSupportedAttributes(list);
                model->refresh();
                hHeader->setSortIndicator(0, Qt::AscendingOrder);

                resizeColumnsToContents();

                setupHeaderMenu();
        }

}

void SKGTableView::saveSelection()
{
        selection.clear();

        QItemSelectionModel *selModel=selectionModel();
        if (selModel) {
                SKGObjectModelBase* model=(SKGObjectModelBase*) this->model();
                QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *>(model);
                if (proxyModel) model=(SKGObjectModelBase*) proxyModel->sourceModel();

                QModelIndexList indexes=selModel->selectedRows();
                foreach(const QModelIndex& index, indexes) {
                        QModelIndex idxs=(proxyModel ? proxyModel->mapToSource(index) : index);
                        SKGObjectBase obj=model->getObject(idxs);
                        selection.push_back(obj.getUniqueID());
                }
        }
}

void SKGTableView::setAlternatingRowColors(bool enable)
{
        QTableView::setAlternatingRowColors(enable);
}

void SKGTableView::selectObject(const QString& iUniqueID)
{
        QItemSelectionModel *selModel=selectionModel();
        if (selModel) {
                selModel->clearSelection();

                SKGObjectModelBase* model=(SKGObjectModelBase*) this->model();
                QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *>(model);
                if (proxyModel) model=(SKGObjectModelBase*) proxyModel->sourceModel();

                int nbRows=model->rowCount(QModelIndex());
                if (nbRows) {
                        for (int i=0; i<nbRows; ++i) {
                                QModelIndex index=model->index(i, 0, QModelIndex());
                                SKGObjectBase obj=model->getObject(index);
                                if (obj.getUniqueID()==iUniqueID) {
                                        QModelIndex idxs=(proxyModel ? proxyModel->mapFromSource(index) : index);
                                        selModel->select(idxs, QItemSelectionModel::Select|QItemSelectionModel::Rows);
                                        scrollTo( idxs);
                                        break;
                                }
                        }
                }
        }
}

void SKGTableView::resetSelection()
{
        QItemSelectionModel *selModel=selectionModel();
        if (selModel) {
                SKGObjectModelBase* model=(SKGObjectModelBase*) this->model();
                QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *>(model);
                if (proxyModel) model=(SKGObjectModelBase*) proxyModel->sourceModel();

                int nbRows=model->rowCount(QModelIndex());
                if (nbRows) {
                        foreach(const QString& sel, selection) {
                                for (int i=0; i<nbRows; ++i) {
                                        QModelIndex index=model->index(i, 0, QModelIndex());
                                        SKGObjectBase obj=model->getObject(index);
                                        if (obj.getUniqueID()==sel) {
                                                QModelIndex idxs=(proxyModel ? proxyModel->mapFromSource(index) : index);
                                                selModel->select(idxs, QItemSelectionModel::Select|QItemSelectionModel::Rows);
                                                break;
                                        }
                                }
                        }
                }
        }
}

static QPoint scrollOffset(QWidget *widget)
{
        int x = 0, y = 0;

        QAbstractScrollArea *scrollArea = dynamic_cast<QAbstractScrollArea*>(widget);
        if (scrollArea) {
                x = scrollArea->horizontalScrollBar()->value();
                y = scrollArea->verticalScrollBar()->value();
        }

        return QPoint(x, y);
}

static void setScrollOffset(QWidget *widget, const QPoint &p)
{
        QAbstractScrollArea *scrollArea = dynamic_cast<QAbstractScrollArea*>(widget);
        if (scrollArea) {
                scrollArea->horizontalScrollBar()->setValue(p.x());
                scrollArea->verticalScrollBar()->setValue(p.y());
        }
}

static QPoint deaccelerate(const QPoint &speed, int a = 1, int max = 64)
{
        int x = qBound(-max, speed.x(), max);
        int y = qBound(-max, speed.y(), max);
        x = (x == 0) ? x : (x > 0) ? qMax(0, x - a) : qMin(0, x + a);
        y = (y == 0) ? y : (y > 0) ? qMax(0, y - a) : qMin(0, y + a);
        return QPoint(x, y);
}

bool SKGTableView::eventFilter(QObject *object, QEvent *event)
{
        QEvent::Type type = event->type();
        if (type != QEvent::MouseButtonPress &&
                        type != QEvent::MouseButtonRelease &&
                        type != QEvent::MouseMove)
                return false;

        QMouseEvent *mouseEvent = dynamic_cast<QMouseEvent*>(event);
        if (!mouseEvent || mouseEvent->modifiers() != Qt::NoModifier)
                return false;

        QWidget *viewport = dynamic_cast<QWidget*>(object);
        if (!viewport || this->ignored.removeAll(event))
                return false;

        bool consumed = false;
        switch (this->state) {

        case SKGTableView::Steady:
                if (mouseEvent->type() == QEvent::MouseButtonPress)
                        if (mouseEvent->buttons() == Qt::LeftButton) {
                                consumed = true;
                                this->state = SKGTableView::Pressed;
                                this->pressPos = mouseEvent->pos();
                                this->offset = scrollOffset(this);
                        }
                break;

        case SKGTableView::Pressed:
                if (mouseEvent->type() == QEvent::MouseButtonRelease) {
                        consumed = true;
                        this->state = SKGTableView::Steady;

                        QMouseEvent *event1 = new QMouseEvent(QEvent::MouseButtonPress,
                                                              this->pressPos, Qt::LeftButton,
                                                              Qt::LeftButton, Qt::NoModifier);
                        QMouseEvent *event2 = new QMouseEvent(*mouseEvent);

                        this->ignored << event1;
                        this->ignored << event2;
                        QApplication::postEvent(object, event1);
                        QApplication::postEvent(object, event2);
                }
                if (mouseEvent->type() == QEvent::MouseMove) {
                        consumed = true;
                        this->state = SKGTableView::ManualScroll;
                        this->dragPos = QCursor::pos();
                        if (!this->ticker.isActive())
                                this->ticker.start(20, this);
                }
                break;

        case SKGTableView::ManualScroll:
                if (mouseEvent->type() == QEvent::MouseMove) {
                        consumed = true;
                        QPoint delta = mouseEvent->pos() - this->pressPos;
                        setScrollOffset(this, this->offset - delta);
                }
                if (mouseEvent->type() == QEvent::MouseButtonRelease) {
                        consumed = true;
                        this->state = SKGTableView::AutoScroll;
                }
                break;

        case SKGTableView::AutoScroll:
                if (mouseEvent->type() == QEvent::MouseButtonPress) {
                        consumed = true;
                        this->state = SKGTableView::Stop;
                        this->speed = QPoint(0, 0);
                }
                if (mouseEvent->type() == QEvent::MouseButtonRelease) {
                        consumed = true;
                        this->state = SKGTableView::Steady;
                        this->speed = QPoint(0, 0);
                }
                break;

        case SKGTableView::Stop:
                if (mouseEvent->type() == QEvent::MouseButtonRelease) {
                        consumed = true;
                        this->state = SKGTableView::Steady;
                }
                if (mouseEvent->type() == QEvent::MouseMove) {
                        consumed = true;
                        this->state = SKGTableView::ManualScroll;
                        this->dragPos = QCursor::pos();
                        if (!this->ticker.isActive())
                                this->ticker.start(20, this);
                }
                break;

        default:
                break;
        }

        return consumed;
}

void SKGTableView::timerEvent(QTimerEvent *event)
{
        int count = 0;
        if (this->state == SKGTableView::ManualScroll) {
                count++;
                this->speed = QCursor::pos() - this->dragPos;
                this->dragPos = QCursor::pos();
        }

        if (this->state == SKGTableView::AutoScroll) {
                count++;
                this->speed = deaccelerate(this->speed);
                QPoint p = scrollOffset(this);
                setScrollOffset(this, p - this->speed);
                if (this->speed == QPoint(0, 0))
                        this->state = SKGTableView::Steady;
        }

        if (!count) this->ticker.stop();

        QTableView::timerEvent(event);
}

void SKGTableView::mousePressEvent ( QMouseEvent * event )
{
        if (event->button()==Qt::LeftButton && !(this->indexAt(event->pos()).isValid())) {
                clearSelection();
        }
        QTableView::mousePressEvent(event);
}

SKGStringListList SKGTableView::getTable()
{
        //Build table
        SKGStringListList table;

        //Get header names
        QAbstractItemModel* model=this->model();
        int nb=model->columnCount();
        QStringList cols;
        for (int i=0; i<nb; i++) {
                cols.append(model->headerData(i,Qt::Horizontal, Qt::UserRole).toString());
        }
        table.append(cols);

        //Get content
        int nb2=model->rowCount();
        for (int i=0; i<nb2; i++) {
                QStringList row;
                for (int j=0; j<nb; j++) {
                        row.append(model->data(model->index(i,j), Qt::UserRole).toString());
                }
                table.append(row);
        }
        return table;
}

void SKGTableView::onExportCSV()
{
        _SKGTRACEIN(10, "SKGTableView::onExportCSV");
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.csv|"+i18n("CSV Files") , this);
        if (fileName.isEmpty()) return;
        {
                SKGError err;

                //Write file
                QFile file(fileName);
                if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
                        err.setReturnCode(ERR_INVALIDARG);
                        err.setMessage(tr("Save file [%1] failed").arg(fileName));
                } else {
                        QTextStream out(&file);
                        QStringList dump=SKGServices::tableToDump(getTable(), SKGServices::DUMP_CSV);
                        int nbl=dump.count();
                        for (int i=0; i<nbl; ++i) {
                                out << dump[i] << endl;
                        }
                }

                //Close file
                file.close();

        }
        QDesktopServices::openUrl(QUrl(fileName));
}

void SKGTableView::onExportTXT()
{
        _SKGTRACEIN(10, "SKGTableView::onExportTXT");
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "*.txt|"+i18n("Text document") , this);
        if (fileName.isEmpty()) return;

        SKGError err;

        //Write file
        QFile file(fileName);
        if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
                err.setReturnCode(ERR_INVALIDARG);
                err.setMessage(tr("Save file [%1] failed").arg(fileName));
        } else {
                QTextStream out(&file);
                QStringList dump=SKGServices::tableToDump(getTable(), SKGServices::DUMP_TEXT);
                int nbl=dump.count();
                for (int i=0; i<nbl; ++i) {
                        out << dump[i] << endl;
                }
        }

        //Close file
        file.close();

        QDesktopServices::openUrl(QUrl(fileName));
}

void SKGTableView::onExportPDF()
{
        _SKGTRACEIN(10, "SKGTableView::onExportPDF");
#ifndef QT_NO_PRINTER
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "application/pdf" , this);
        if (fileName.isEmpty()) return;

        {
                QImage image(this->size(), QImage::Format_ARGB32);
                QPainter painter(&image);
                this->render(&painter);

                {
                        QPrinter printer(QPrinter::HighResolution);
                        printer.setOutputFileName(fileName);
                        QPainter painter(&printer);

                        QRect rect = painter.viewport();
                        QSize size = image.size();
                        size.scale(rect.size(), Qt::KeepAspectRatio);
                        painter.setViewport(rect.x(), rect.y(), size.width(), size.height());
                        painter.setWindow(image.rect());
                        painter.drawImage(0, 0, image);
                }
        }
        QDesktopServices::openUrl(QUrl(fileName));
#endif
}
#include "skgtableview.moc"
