/*  Sjaak, a program for playing chess
 *  Copyright (C) 2011, 2014  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef BOARD_H
#define BOARD_H

#include "compilerdef.h"
#include "bitboard.h"
#include "pieces.h"
#include "piece_types.h"
#include "move.h"
#include "hashkey.h"
#include "squares.h"

/* Possible rule flags */
#define RF_FORCE_CAPTURE      0x00000001
#define RF_MULTI_CAPTURE      0x00000002

#define RF_KEEP_CAPTURE       0x00000004  /* Captured pieces go to your hand */
#define RF_RETURN_CAPTURE     0x00000008  /* Captured pieces go back to their owner's hand */
#define RF_USE_CAPTURE        (RF_KEEP_CAPTURE | RF_RETURN_CAPTURE)

#define RF_KING_TABOO         0x00000010  /* Kings cannot face eachother along a ray */
#define RF_KING_TRAPPED       0x00000020  /* Kings are trapped in a "palace" */
#define RF_KING_CORNERDRIVE   0x00000040  /* Kings can drive eachother from the corners */
#define RF_KING_DUPLECHECK    0x00000080  /* It's check if all kings are under attack, if there is more than one. */

#define RF_ALLOW_DROPS        0x00000100  /* The game allows drop moves */
#define RF_FORCE_DROPS        0x00000200  /* Drops are forced if possible */
#define RF_GATE_DROPS         0x00000400  /* Drops work as S-chess gates */
#define RF_USE_DROPS          0x00000700  /* Game uses drops */

#define RF_ALLOW_PICKUP       0x00000800  /* The player is allowed to take pieces in-hand */

#define RF_PROMOTE_IN_PLACE   0x00001000  /* Promotions can be done in-place, without moving a piece */

#define RF_SPECIAL_IS_INIT    0x00002000  /* Special moves are only initial moves */

#define RF_USE_HOLDINGS       (RF_USE_DROPS | RF_USE_CAPTURE | RF_ALLOW_PICKUP)  /* Game uses holdings in some way */

#define RF_USE_SHAKMATE       0x00010000  /* The checking sequence prior to mate needs particular pieces */
#define RF_USE_BARERULE       0x00020000  /* Shatranj-style baring rule */
#define RF_USE_CHASERULE      0x00040000  /* Xiangqi-style chase rule is in effect */

#define RF_CAPTURE_ANY_FLAG   0x00100000  /* There is a "capture the flag" victory condition. */
#define RF_CAPTURE_ALL_FLAG   0x00200000  /* There is a "capture the flag" victory condition. */
#define RF_CAPTURE_THE_FLAG   (RF_CAPTURE_ANY_FLAG | RF_CAPTURE_ALL_FLAG)  /* There is a "capture the flag" victory condition. */

/* Board state */
#define BF_CHECK              0x0001      /* Whether the side to move is in-check or not */
#define BF_WSHAK              0x0002      /* Whether a "shak" was given or not */
#define BF_BSHAK              0x0004      /* Whether a "shak" was given or not */

template <typename kind>
struct unmake_info_t {
   bitboard_t<kind> init;
   uint64_t pawn_hash;
   uint64_t hash;
   int8_t fifty_counter;

   int8_t ep;
   int8_t ep_capture;

   uint8_t board_flags;

   piece_t pickup_piece[4];
#ifdef DEBUGMODE
   move_t move;
#endif
};

template <typename kind>
struct board_t {
   bitboard_t<kind> bbc[NUM_SIDES];
   bitboard_t<kind> bbp[MAX_PIECE_TYPES];
   bitboard_t<kind> flag[NUM_SIDES];  /* Flag bitboard, for "capture the flag" */
   bitboard_t<kind> royal;
   bitboard_t<kind> init;
   int8_t piece[8 * sizeof(kind)];

   /* Piece holdings.
    * These are indexed by [piece type][side to move]
    * We actually have a rule flag to specify whether we're interested in these or not, so we can skip a chunk
    * of code for variants where we're not.
    */
   int8_t holdings[MAX_PIECE_TYPES][NUM_SIDES];

   /* Hash key */
   uint64_t hash;
   uint64_t pawn_hash;

   /* Rule flags, to change the behaviour of the move generator or the evaluation function */
   uint32_t rule_flags;

   /* Record the board state, in check, castle status */
   uint8_t board_flags;

   /* En-passant target square and capture location. */
   int8_t ep;
   int8_t ep_capture;

   /* Half-move clock (50-move counter) */
   int8_t fifty_counter;

   /* Side to move */
   side_t side_to_move;

   /* Description of all piece types */
   piece_description_t<kind> *piece_types;

   bool check() const { return (board_flags & BF_CHECK); }
   void check(bool chk) {
      board_flags &= ~BF_CHECK;
      board_flags |= uint8_t(chk);
      if (!chk) board_flags &= ~(BF_WSHAK << side_to_move);
   }

   void shak() {
      board_flags |= (BF_WSHAK << side_to_move);
   }
   bool have_shak() {
      return (board_flags & (BF_WSHAK << side_to_move)) != 0;
   }

   static void clear_board(board_t<kind> *board)
   {
      piece_description_t<kind> *ps = board->piece_types;
      uint32_t rule_flags = board->rule_flags;
      bitboard_t<kind> flag[NUM_SIDES];
      flag[WHITE] = board->flag[WHITE];
      flag[BLACK] = board->flag[BLACK];

      memset(board, 0, sizeof *board);

      board->piece_types = ps;
      board->rule_flags = rule_flags;
      board->flag[WHITE] = flag[WHITE];
      board->flag[BLACK] = flag[BLACK];
   }

   void put_piece(int type, side_t side, int square)
   {
      bbc[side].set(square);
      bbp[type].set(square);
      piece[square] = type;
      if (piece_types->piece_flags[type] & PF_ROYAL)
         royal.set(square);
      hash ^= piece_key[type][side][square];
      if (type == piece_types->pawn_index[side]) pawn_hash ^= piece_key[type][side][square];
   }

   void clear_piece(int type, side_t side, int square)
   {
      bbc[side].reset(square);
      bbp[type].reset(square);
      royal.reset(square);
      init.reset(square);
      hash ^= piece_key[type][side][square];
      if (type == piece_types->pawn_index[side]) pawn_hash ^= piece_key[type][side][square];
   }

   void put_new_piece(int type, side_t side, int square)
   {
      put_piece(type, side, square);
      init.set(square);
   }

   int8_t get_piece(int square) const { return piece[square]; }
   side_t get_side(int square) const {
      if (bbc[WHITE].test(square)) return WHITE;
      if (bbc[BLACK].test(square)) return BLACK;
      return NONE;
   }

   bitboard_t<kind> get_occupied() const { return bbc[WHITE] | bbc[BLACK]; }

   int piece_count(int piece, side_t side) const { return (bbc[side] & bbp[piece]).popcount(); }

   int locate_least_valued_piece(bitboard_t<kind> mask) const
   {
      int *perm = piece_types->val_perm;

      for (int n=0; n<piece_types->num_piece_types; n++) {
         if (!(bbp[perm[n]] & mask).is_empty())
            return (bbp[perm[n]] & mask).bitscan();
      }

      return -1;
   }

   void makemove(move_t move, unmake_info_t<kind> *ui)
   {
      int n;

      /* First: backup information for unmake */
      ui->init = init;
      ui->hash = hash;
      ui->fifty_counter = fifty_counter;
      ui->ep = ep;
      ui->ep_capture = ep_capture;
      ui->board_flags = board_flags;
#ifdef DEBUGMODE
      ui->move = move;
#endif

      /* Second: resolve all pickups */
      n = get_move_pickups(move);
      for (int c=0; c<n; c++) {
         uint16_t p  = get_move_pickup(move, c);
         int square  = decode_pickup_square(p);
         int piece   = get_piece(square);
         side_t side = get_side(square);

         ui->pickup_piece[c] = piece_for_side(piece, side);
         clear_piece(piece, side, square);
      }

      /* Third: resolve all swaps */
      n = get_move_swaps(move);
      for (int c=0; c<n; c++) {
         uint16_t p  = get_move_swap(move, c);
         int from    = decode_swap_from(p);
         int to      = decode_swap_to(p);
         int piece   = get_piece(from);
         side_t side = get_side(from);

         clear_piece(piece, side, from);
         put_piece(piece, side, to);
      }

      /* Fourth: resolve all drops */
      n = get_move_drops(move);
      for (int c=0; c<n; c++) {
         uint16_t p  = get_move_drop(move, c);
         int square  = decode_drop_square(p);
         int piece   = decode_drop_piece(p);
         side_t side = decode_drop_side(p);

         put_piece(piece, side, square);
      }

      /* Fifth: update holdings */
      if (expect((rule_flags & RF_USE_HOLDINGS) && get_move_holdings(move), false)) {
         uint16_t p  = get_move_holding(move);
         int count   = decode_holding_count(p);
         int piece   = decode_holding_piece(p);
         side_t side = decode_holding_side(p);
         if (count < 0)
            hash ^= hold_key[piece][side][holdings[piece][side]];
         holdings[piece][side] += count;
         if (count > 0)
            hash ^= hold_key[piece][side][holdings[piece][side]];
      }

      /* Sixth: update status bits */
      ep_capture = ep = 0;
      if (move & MOVE_SET_ENPASSANT) {
         ep = (get_move_from(move) + get_move_to(move))/2;
         ep_capture = get_move_to(move);
      }

      /* Seventh: flip side to move */
      if (expect((move & MOVE_KEEP_TURN) == 0, true)) {
         side_to_move = next_side[side_to_move];
         hash ^= side_to_move_key;
      }

      /* Finally: update 50-move clock */
      fifty_counter++;
      if (move & MOVE_RESET50) fifty_counter = 0;


      /* Assume we're not in check */
      check(false);
   }

   void unmakemove(move_t move, unmake_info_t<kind> *ui)
   {
      int n;

      /* First: flip side to move */
      if (expect((move & MOVE_KEEP_TURN) == 0, true))
         side_to_move = next_side[side_to_move];

      /* Second: reverse all drops */
      n = get_move_drops(move);
      for (int c=0; c<n; c++) {
         uint16_t p  = get_move_drop(move, c);
         int square  = decode_drop_square(p);
         int piece   = decode_drop_piece(p);
         side_t side = decode_drop_side(p);

         clear_piece(piece, side, square);
      }

      /* Third: reverse all swaps */
      n = get_move_swaps(move);
      for (int c=0; c<n; c++) {
         uint16_t p  = get_move_swap(move, c);
         int to      = decode_swap_from(p);
         int from    = decode_swap_to(p);
         int piece   = get_piece(from);
         side_t side = get_side(from);

         clear_piece(piece, side, from);
         put_piece(piece, side, to);
      }

      /* Fourth: reverse all pickups */
      n = get_move_pickups(move);
      for (int c=0; c<n; c++) {
         uint16_t p  = get_move_pickup(move, c);
         int square  = decode_pickup_square(p);
         int piece   = neutral_piece(ui->pickup_piece[c]);
         side_t side = side_for_piece(ui->pickup_piece[c]);

         put_piece(piece, side, square);
      }

      /* Fifth: update holdings */
      if (expect((rule_flags & RF_USE_HOLDINGS) && get_move_holdings(move), false)) {
         uint16_t p  = get_move_holding(move);
         int count   = decode_holding_count(p);
         int piece   = decode_holding_piece(p);
         side_t side = decode_holding_side(p);
         holdings[piece][side] -= count;
         assert(holdings[piece][side] >= 0);
      }

      /* Finally: restore backedup information */
      init = ui->init;
      hash = ui->hash;
      fifty_counter = ui->fifty_counter;
      ep = ui->ep;
      ep_capture = ui->ep_capture;
      board_flags = ui->board_flags;
   }

   void print(FILE* file = stdout, bool ansi = true) const
   {
      bitboard_t<kind> occ = get_occupied();
      int c, n;

      if (file != stdout) ansi = false;

      for (c=bitboard_t<kind>::board_ranks-1; c>=0; c--) {
         if (ansi) fprintf(file, "%2s", rank_names[c]);
         if (ansi) fprintf(file, "\033[1m");
         for (n=0; n<bitboard_t<kind>::board_files; n++) {
            int square = bitboard_t<kind>::pack_rank_file(c, n);
            int piece = get_piece(square);
            // 46/45 works for purple/cyan
            // 43/41 works for red/yellow
            if (ansi) fprintf(file, "%s", ((c^n)&1) ? "\033[46m" : "\033[45m");
            if (occ.test(square)) {
               bool white = bbc[WHITE].test(square);
               if (ansi) fprintf(file, "%s", white ? "\033[37m" : "\033[30m");
               fprintf(file, "%-2s", piece_types->piece_abbreviation[piece][1-white]);
            } else {
               if (ansi)
                  fprintf(file, "  ");
               else
                  fprintf(file, "%c ", ((c^n)&1) ? '+' : '.');
            }
         }
         if (ansi) fprintf(file, "\033[0m");
         if (side_to_move == BLACK && c == bitboard_t<kind>::board_ranks-1) fprintf(file, "*");
         if (side_to_move == WHITE && c == 0) fprintf(file, "*");
         fprintf(file, "\n");
      }
      if (ansi) {
         fprintf(file, "  ");
         for (n=0; n<bitboard_t<kind>::board_files; n++)
            fprintf(file, "%s ", file_names[n]);
         fprintf(file, "\n");
      }

      if (rule_flags & RF_USE_HOLDINGS) {
         for (c=0; c<NUM_SIDES; c++) {
            fprintf(file, "%s holdings [ ", c ? "Black" : "White");
            for (n=0; n<piece_types->num_piece_types; n++)
               if (holdings[n][c])
                  fprintf(file, "%s: %02d ", piece_types->piece_abbreviation[n][c], holdings[n][c]);
            fprintf(file, "]\n");
         }
      }
   }

   void print_demo_board(bitboard_t<kind> xmark, bitboard_t<kind> omark, bool ansi = true) const
   {
      bitboard_t<kind> occ = get_occupied();
      int c, n;

      for (c=bitboard_t<kind>::board_ranks-1; c>=0; c--) {
         if (ansi) printf("%2s", rank_names[c]);
         if (ansi) printf("\033[1m");
         for (n=0; n<bitboard_t<kind>::board_files; n++) {
            int square = bitboard_t<kind>::pack_rank_file(c, n);
            int piece = get_piece(square);
            // 46/45 works for purple/cyan
            // 43/41 works for red/yellow
            if (ansi) printf("%s", ((c^n)&1) ? "\033[46m" : "\033[45m");
            if (occ.test(square)) {
               bool white = bbc[WHITE].test(square);
               if (ansi) printf("%s", white ? "\033[37m" : "\033[30m");
               printf("%-2s", piece_types->piece_abbreviation[piece][1-white]);
            } else {
               char cc = ' ';
               if (omark.test(square)) cc ='*';
               if (xmark.test(square)) cc ='+';
               if (cc != ' ' && ansi) printf("%s", side_to_move == WHITE ? "\033[37m" : "\033[30m");
               if (ansi)
                  printf("%c ", cc);
               else
                  printf("%c%c", ((c^n)&1) ? '+' : '.', cc);
            }
         }
         if (ansi) printf("\033[0m");
         if (side_to_move == BLACK && c == bitboard_t<kind>::board_ranks-1) printf("*");
         if (side_to_move == WHITE && c == 0) printf("*");
         printf("\n");
      }
      if (ansi) {
         printf("  ");
         for (n=0; n<bitboard_t<kind>::board_files; n++)
            printf("%s ", file_names[n]);
         printf("\n");
      }

      if (rule_flags & RF_USE_HOLDINGS) {
         for (c=0; c<NUM_SIDES; c++) {
            printf("%s holdings [ ", c ? "Black" : "White");
            for (n=0; n<piece_types->num_piece_types; n++)
               if (holdings[n][c])
                  printf("%s: %02d ", piece_types->piece_abbreviation[n][c], holdings[n][c]);
            printf("]\n");
         }
      }
   }

   void print_bitboards() const
   {
      int c, n;

      if (side_to_move == WHITE)
         printf("White to move\n");
      else
         printf("Black to move\n");

      printf("White pieces\tBlack pieces\tUnmoved pieces\tRoyal\t\tep\t\tepc\n");
      bitboard_t<kind> epbb, epcbb;
      if (ep) epbb.set(ep);
      if (ep_capture) epcbb.set(ep_capture);
      for (c=bitboard_t<kind>::board_ranks-1; c>=0; c--) {
         printf("%s", bbc[0].rank_string(c));
         printf("\t");
         printf("%s", bbc[1].rank_string(c));
         printf("\t");
         printf("%s", init.rank_string(c));
         printf("\t");
         printf("%s", royal.rank_string(c));
         printf("\t");
         printf("%s", epbb.rank_string(c));
         printf("\t");
         printf("%s", epcbb.rank_string(c));
         printf("\n");
      }

      if (!piece_types)
         return;

      if (!flag[WHITE].is_empty() || !flag[BLACK].is_empty()) {
         printf("\nWhite flags\tBlack flags\n");
         for (c=bitboard_t<kind>::board_ranks-1; c>=0; c--) {
            printf("%s", flag[WHITE].rank_string(c));
            printf("\t");
            printf("%s", flag[BLACK].rank_string(c));
            printf("\n");
         }
      }

      for (n=0; n<piece_types->num_piece_types; n+=7) {
         printf("\n");
         int k;
         for (k=0; k<7; k++) {
            if (n+k >= piece_types->num_piece_types) break;
            char *s = piece_types->piece_name[n+k];
            printf("%*s", -bitboard_t<kind>::board_files-2, s);
         }
         printf("\n");
         for (c=bitboard_t<kind>::board_ranks-1; c>=0; c--) {
            for (k=0; k<7; k++) {
               if (n+k >= piece_types->num_piece_types) break;
               printf("%s", bbp[n+k].rank_string(c));
               printf("  ");
            }
            printf("\n");
         }
      }

      if (rule_flags & RF_USE_HOLDINGS) {
         for (c=0; c<NUM_SIDES; c++) {
            printf("%s holdings [ ", c ? "Black" : "White");
            for (n=0; n<piece_types->num_piece_types; n++)
               if (holdings[n][c])
                  printf("%s: %02d ", piece_types->piece_abbreviation[n][c], holdings[n][c]);
            printf("]\n");
         }
      }
   }
};


#endif
