/*  Sjaak, a program for playing chess
 *  Copyright (C) 2011, 2014  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef MOVEGEN_H
#define MOVEGEN_H

#include <ctype.h>
#include <algorithm>
#include "assert.h"
#include "bitboard.h"
#include "pieces.h"
#include "board.h"
#include "move.h"
#include "movelist.h"
#include "aligned_malloc.h"

template<typename kind>
struct movegen_t {
   /* Leapers and asymmetric leapers. */
   bitboard_t<kind> leaper[MAX_LEAPER_TYPES][sizeof(kind)*8];
   bitboard_t<kind> aleaper[NUM_SIDES][MAX_LEAPER_TYPES][sizeof(kind)*8];
   uint8_t leaper_description[MAX_LEAPER_TYPES]; // (n,m) encoded in two nibbles
   int number_of_leapers;
   int number_of_aleapers;

   /* Stepper descriptions */
   /* TODO: make stepper description and stepper_step depend on the
    * side-to-move, so we can re-use steppers for both sides, as for
    * asymmetric leapers.
    */
   uint32_t stepper_description[MAX_STEPPER_TYPES][NUM_SIDES]; // 8 directions, with repeat counts (0-15) for each->32 bits
   bitboard_t<kind> stepper_step[MAX_STEPPER_TYPES][NUM_SIDES][sizeof(kind)*8];
   bitboard_t<kind> step_mask[8];
   int inverse_step[8];
   int step_shift[8];
   int number_of_steppers;

   /* Castling move bitboards */
   bitboard_t<kind> short_castle_mask[2];
   bitboard_t<kind> short_castle_free[2];
   bitboard_t<kind> short_castle_safe[2];
   bitboard_t<kind> long_castle_mask[2];
   bitboard_t<kind> long_castle_free[2];
   bitboard_t<kind> long_castle_safe[2];
   int short_castle_king_dest[2];
   int long_castle_king_dest[2];

   /* Normal slider tables */
   bitboard_t<kind> **horizontal_slider_move;
   bitboard_t<kind> **vertical_slider_move;
   bitboard_t<kind> **horizontal_hopper_move;
   bitboard_t<kind> **vertical_hopper_move;
   move_flag_t slider_flags[MAX_SLIDER_TYPES];
   int number_of_sliders;
   move_flag_t hopper_flags[MAX_HOPPER_TYPES];
   int number_of_hoppers;

   /* Super piece */
   bitboard_t<kind> super[sizeof(kind)*8];
   bitboard_t<kind> super_slider[sizeof(kind)*8];
   bitboard_t<kind> super_hopper[sizeof(kind)*8];
   bitboard_t<kind> super_leaper[sizeof(kind)*8];
   bitboard_t<kind> super_stepper[sizeof(kind)*8];

   void initialise() {
      /* Clear leaper/stepper descriptions */
      number_of_leapers = 0;
      number_of_aleapers = 0;
      number_of_steppers = 1;
      number_of_sliders = 0;
      number_of_hoppers = 0;

      memset(leaper, 0, sizeof leaper);
      memset(aleaper, 0, sizeof aleaper);

      memset(step_mask, 0, sizeof step_mask);
      memset(stepper_step, 0, sizeof stepper_step);

      for (int n = 0; n<2; n++) {
         short_castle_mask[n].clear();
         short_castle_free[n].clear();
         short_castle_safe[n].clear();
         long_castle_mask[n].clear();
         long_castle_free[n].clear();
         long_castle_safe[n].clear();
      }

      /* Free tables if previously allocated */
      if (horizontal_slider_move) aligned_free(horizontal_slider_move);
      if (vertical_slider_move)   aligned_free(vertical_slider_move  );
      if (horizontal_hopper_move) aligned_free(horizontal_hopper_move);
      if (vertical_hopper_move)   aligned_free(vertical_hopper_move  );

      horizontal_slider_move = NULL;
      vertical_slider_move   = NULL;
      horizontal_hopper_move = NULL;
      vertical_hopper_move   = NULL;

      /* Bitshifts for steppers */
      /* Bitshifts for all directions: N   NE  E   SE    S   SW    W   NW */
      step_shift[0] = bitboard_t<kind>::board_files;        // N
      step_shift[1] = bitboard_t<kind>::board_files+1;      // NE
      step_shift[2] = 1;                                    // E
      step_shift[3] =-bitboard_t<kind>::board_files+1;      // SE
      step_shift[4] =-bitboard_t<kind>::board_files;        // S
      step_shift[5] =-bitboard_t<kind>::board_files-1;      // SW
      step_shift[6] =-1;                                    // W
      step_shift[7] = bitboard_t<kind>::board_files-1;      // NW

      /* N NE E SE S SW W NW
       * 0 1  2 3  4 5  6 7
       * S SW W NW N NE E SE
       * 4 5  6 7  0 1  2 3
       */
      for (int n = 0; n<8; n++) 
         inverse_step[n] = (n+4)&7;

      step_mask[0] = ~ bitboard_t<kind>::board_north_edge;
      step_mask[1] = ~(bitboard_t<kind>::board_east_edge | bitboard_t<kind>::board_north_edge);
      step_mask[2] = ~ bitboard_t<kind>::board_east_edge;
      step_mask[3] = ~(bitboard_t<kind>::board_east_edge | bitboard_t<kind>::board_south_edge);
      step_mask[4] = ~ bitboard_t<kind>::board_south_edge;
      step_mask[5] = ~(bitboard_t<kind>::board_west_edge | bitboard_t<kind>::board_south_edge);;
      step_mask[6] = ~ bitboard_t<kind>::board_west_edge;
      step_mask[7] = ~(bitboard_t<kind>::board_west_edge | bitboard_t<kind>::board_north_edge);
   }

   void initialise_slider_tables()
   {
      int board_files = bitboard_t<kind>::board_files;
      int board_ranks = bitboard_t<kind>::board_ranks;
      int board_size = board_files * board_ranks;
      int file, rank, square;
      int occ_mask, occ;
      int n;

      /* Allocate memory for tables */
      size_t file_table_size;
      size_t rank_table_size;
      size_t file_table_start;
      size_t rank_table_start;
      size_t file_table_offset;
      size_t rank_table_offset;
      uint8_t *memory;

      /* Determine size of tables: rank attacks, so index by file */
      rank_table_size  = board_files*sizeof(bitboard_t<kind> *);
      if (rank_table_size & 15) rank_table_size += 16 - (rank_table_size & 15);
      rank_table_start = rank_table_size;
      rank_table_size += board_files*(1<<board_files)*sizeof(bitboard_t<kind>);
      rank_table_offset = (1<<board_files)*sizeof(bitboard_t<kind>);

      /* Determine size of tables: file attacks, so index by rank */
      file_table_size  = board_ranks*sizeof(bitboard_t<kind> *);
      if (file_table_size & 15) file_table_size += 16 - (file_table_size & 15);
      file_table_start = file_table_size;
      file_table_size += board_ranks*(1<<board_ranks)*sizeof(bitboard_t<kind>);
      file_table_offset = (1<<board_ranks)*sizeof(bitboard_t<kind>);

      /* Free tables if previously allocated */
      if (horizontal_slider_move) aligned_free(horizontal_slider_move);
      if (vertical_slider_move)   aligned_free(vertical_slider_move  );
      if (horizontal_hopper_move) aligned_free(horizontal_hopper_move);
      if (vertical_hopper_move)   aligned_free(vertical_hopper_move  );

      /* Allocate tables for horizontal (rank) attacks */
      memory = (uint8_t *)aligned_malloc(rank_table_size, 16);
      assert(memory);
      memset(memory, 0, rank_table_size);
      horizontal_slider_move = (bitboard_t<kind> **)memory;
      for(n = 0; n<board_files; n++) {
         horizontal_slider_move[n] = (bitboard_t<kind> *)(memory + rank_table_start + n*rank_table_offset);
      }

      memory = (uint8_t *)aligned_malloc(rank_table_size, 16);
      assert(memory);
      memset(memory, 0, rank_table_size);
      horizontal_hopper_move = (bitboard_t<kind> **)memory;
      for(n = 0; n<board_files; n++) {
         horizontal_hopper_move[n] = (bitboard_t<kind> *)(memory + rank_table_start + n*rank_table_offset);
      }

      /* Allocate tables for vertical (file) attacks */
      memory = (uint8_t *)aligned_malloc(file_table_size, 16);
      assert(memory);
      memset(memory, 0, file_table_size);
      vertical_slider_move = (bitboard_t<kind> **)memory;
      for(n = 0; n<board_ranks; n++) {
         vertical_slider_move[n] = (bitboard_t<kind> *)(memory + file_table_start + n*file_table_offset);
      }

      memory = (uint8_t *)aligned_malloc(file_table_size, 16);
      assert(memory);
      memset(memory, 0, file_table_size);
      vertical_hopper_move = (bitboard_t<kind> **)memory;
      for(n = 0; n<board_ranks; n++) {
         vertical_hopper_move[n] = (bitboard_t<kind> *)(memory + file_table_start + n*file_table_offset);
      }

      /* Rank attacks */
      for (file = 0; file<board_files; file++) {
         for (occ = 0; occ < 1<<board_files; occ++) {
            /* Left of slider position, rook and cannon moves, rook
             * attacks.
             */
            for (n=file-1; n>=0; n--) {
               horizontal_slider_move[file][occ] |= bitboard_t<kind>::board_file[n];//bitboard_t<kind>::square_bitboards[n];
               if ( occ & (1 << n) )
                  break;
            }
            n--;
            /* Cannon attacks */
            for (; n>=0; n--) {
               horizontal_hopper_move[file][occ] |= bitboard_t<kind>::board_file[n];//bitboard_t<kind>::square_bitboards[n];
               if ( occ & (1 << n) )
                  break;
            }

            /* Right of slider position */
            for (n=file+1; n<board_files; n++) {
               horizontal_slider_move[file][occ] |= bitboard_t<kind>::board_file[n];//bitboard_t<kind>::square_bitboards[n];

               if ( occ & (1 << n) )
                  break;
            }
            n++;
            /* Cannon attacks */
            for (; n<board_files; n++) {
               horizontal_hopper_move[file][occ] |= bitboard_t<kind>::board_file[n];//bitboard_t<kind>::square_bitboards[n];
               if ( occ & (1 << n) )
                  break;
            }
         }
      }

      /* File attacks */
      for (rank = 0; rank < board_ranks; rank++) {
         for (occ = 0; occ < 1<<board_ranks; occ++) {
            /* South of slider position, rook and cannon moves, rook
             * attacks.
             */
            for (n=rank-1; n>=0; n--) {
               vertical_slider_move[rank][occ] |= bitboard_t<kind>::board_rank[n];//bitboard_t<kind>::square_bitboards[board_files*n];
               if ( occ & (1 << n) )
                  break;
            }
            n--;
            /* Cannon attacks */
            for (; n>=0; n--) {
               vertical_hopper_move[rank][occ] |= bitboard_t<kind>::board_rank[n];//bitboard_t<kind>::square_bitboards[board_files*n];
               if ( occ & (1 << n) )
                  break;
            }

            /* North of slider position */
            for (n=rank+1; n<board_ranks; n++) {
               vertical_slider_move[rank][occ] |= bitboard_t<kind>::board_rank[n];//bitboard_t<kind>::square_bitboards[board_files*n];
               if ( occ & (1 << n) )
                  break;
            }
            n++;
            /* Cannon attacks */
            for (; n<board_ranks; n++) {
               vertical_hopper_move[rank][occ] |= bitboard_t<kind>::board_rank[n];//bitboard_t<kind>::square_bitboards[board_files*n];
               if ( occ & (1 << n) )
                  break;
            }
         }
      }

      /* Initialise superpiece attacks to a full board */
      for (n=0; n<board_size; n++) {
         super_slider[n] = super_hopper[n] = super_leaper[n] = super[n] = bitboard_t<kind>::board_all;
      }
   }

   void initialise_super_tables(void)
   {
      int board_files = bitboard_t<kind>::board_files;
      int board_ranks = bitboard_t<kind>::board_ranks;
      int board_size = board_files * board_ranks;
      int n, c;

      /* Initialise stepper masks */
      for (int c = 1; c<number_of_steppers; c++) {
         for(n=0; n<board_size; n++) {
            bitboard_t<kind> stepper;
            stepper.set(n);

            stepper_step[c][WHITE][n] = generate_stepper_move_bitboard(make_stepper_index(c), WHITE, bitboard_t<kind>::board_empty, stepper);
            stepper_step[c][BLACK][n] = generate_stepper_move_bitboard(make_stepper_index(c), BLACK, bitboard_t<kind>::board_empty, stepper);
         }
      }

      for(n=0; n<board_size; n++)
         super_stepper[n].clear();

      /* The super-stepper can reverse-step as well as step. This is needed
       * for cases where we need to do a reverse lookup.
       */
      for (int c = 1; c<number_of_steppers; c++) {
         for (int side=0; side<2; side++) {
            for(n=0; n<board_size; n++) {
               bitboard_t<kind> bb = stepper_step[c][side][n];
               super_stepper[n] |= bb;
               while(!bb.is_empty()) {
                  int s = bb.bitscan();
                  bb.reset(s);

                  super_stepper[s].set(n);
               }
            }
         }
      }

      for(n=0; n<board_size; n++) {
         super_stepper[n] &= bitboard_t<kind>::board_all;

         super_leaper[n].clear();
         for (c=0; c<number_of_leapers; c++) {
            super_leaper[n] |= leaper[c][n];
            super_leaper[n] |= aleaper[WHITE][c][n];
            super_leaper[n] |= aleaper[BLACK][c][n];
         }
         super_leaper[n] &= bitboard_t<kind>::board_all;

         super_slider[n].clear();
         for (c=0; c<number_of_sliders; c++) {
            super_slider[n] |= generate_slider_move_bitboard(slider_flags[c], WHITE, n, bitboard_t<kind>::board_empty);
         }
         super_slider[n] &= bitboard_t<kind>::board_all;

         super_hopper[n].clear();
         for (c=0; c<number_of_hoppers; c++) {
            super_hopper[n] |= generate_slider_move_bitboard(hopper_flags[c]>>4, WHITE, n, bitboard_t<kind>::board_empty);
         }
         super_hopper[n] &= bitboard_t<kind>::board_all;

         super[n] = super_hopper[n] | super_leaper[n] | super_slider[n] | super_stepper[n];
      }
   }


   /* Move generator, per piece type */
   bitboard_t<kind> generate_leaper_move_bitboard(move_flag_t flags, side_t side, int square, bitboard_t<kind> occ) const {
      assert(is_leaper(flags));
      bitboard_t<kind> moves;
      int index = get_leaper_index(flags);

      moves = is_aleaper(flags) ? aleaper[side][index][square]
                                :  leaper      [index][square];

      /* Simple leaper? */
      if (is_simple_leaper(flags))
         return moves;

      /* Double-step leaper */
      if (is_double_leaper(flags)) {
         bitboard_t<kind> bb = moves & ~occ;
         index = get_leaper_index2(flags);
         while (!bb.is_empty()) {
            int square = bb.bitscan();
            bb.reset(square);
            moves |= leaper[index][square];
         }
      }

      /* Masked leaper */
      if (is_masked_leaper(flags)) {
         index = get_leaper_indexm(flags);
         moves &= leaper[index][square];
      }

      return moves;
   }

   bitboard_t<kind> generate_slider_move_bitboard(move_flag_t flags, side_t side, int square, bitboard_t<kind> occ) const {
      assert(is_slider(flags));
      bitboard_t<kind> moves;
      int file = unpack_file(square);
      int rank = unpack_rank(square);
      int diag = occ.diagonal_nr[square];
      int anti = occ.anti_diagonal_nr[square];
      int index;

      if (flags & MF_SLIDER_H) {
         index = occ.get_rank(rank);
         moves |= horizontal_slider_move[file][index] & bitboard_t<kind>::board_rank[rank];
      }

      if (flags & MF_SLIDER_V) {
         index = occ.get_file(file);
         moves |= vertical_slider_move[rank][index] & bitboard_t<kind>::board_file[file];
      }

      if (flags & MF_SLIDER_D) {
         bitboard_t<kind> mask = bitboard_t<kind>::board_diagonal[diag];
         index = (occ & mask).fill_south().get_rank(0);

         moves |= horizontal_slider_move[file][index] & mask;
      }

      if (flags & MF_SLIDER_A) {
         bitboard_t<kind> mask = bitboard_t<kind>::board_antidiagonal[anti];
         index = (occ & mask).fill_south().get_rank(0);

         moves |= horizontal_slider_move[file][index] & mask;
      }

      return moves;
   }

   bitboard_t<kind> generate_hopper_move_bitboard(move_flag_t flags, side_t side, int square, bitboard_t<kind> occ) const {
      assert(is_hopper(flags));
      bitboard_t<kind> moves;
      int file = unpack_file(square);
      int rank = unpack_rank(square);
      int diag = occ.diagonal_nr[square];
      int anti = occ.anti_diagonal_nr[square];
      int index;

      //moves = generate_slider_move_bitboard(flags>>4, side, square, occ);
      //return generate_slider_move_bitboard(flags>>4, side, square, occ&~moves) ^ moves;

      if (flags & MF_HOPPER_H) {
         index = occ.get_rank(rank);
         moves |= horizontal_hopper_move[file][index] & bitboard_t<kind>::board_rank[rank];
      }

      if (flags & MF_HOPPER_V) {
         index = occ.get_file(file);
         moves |= vertical_hopper_move[rank][index] & bitboard_t<kind>::board_file[file];
      }

      if (flags & MF_HOPPER_D) {
         bitboard_t<kind> mask = bitboard_t<kind>::board_diagonal[diag];
         index = (occ & mask).fill_south().get_rank(0);

         moves |= horizontal_hopper_move[file][index] & mask;
      }

      if (flags & MF_HOPPER_A) {
         bitboard_t<kind> mask = bitboard_t<kind>::board_antidiagonal[anti];
         index = (occ & mask).fill_south().get_rank(0);

         moves |= horizontal_hopper_move[file][index] & mask;
      }

      return moves;
   }

   bitboard_t<kind> generate_stepper_move_bitboard(move_flag_t flags, side_t side, bitboard_t<kind> occ, bitboard_t<kind> steppers) const
   {
      bitboard_t<kind> moves;

      /* Check for single stepper moves, which are generated in parallel */
      int si = get_stepper_index(flags);
      int d;
      for (d=0; d<8; d++) {
         int c = (stepper_description[si][side] >> (d*4)) & 15;
         bitboard_t<kind> dmoves = steppers;

         if (c == 0) continue;

         /* We have a repetition count, so we do a number of steps one after the other.
          * This can effectively duplicate a slider.
          */
         for ( ; c>0; c--) {
            dmoves &= step_mask[d];
            dmoves = dmoves.sshift(step_shift[d]);
            moves |= dmoves;
            dmoves &= ~occ;
         }
      }

      return moves;
   }


   bitboard_t<kind> generate_super_attacks_for_squares(bitboard_t<kind> squares, const bitboard_t<kind> super[sizeof(kind)*8]) const
   {
      bitboard_t<kind> attacks;

      while (!squares.is_empty()) {
         int square = squares.bitscan();
         squares.reset(square);
         attacks |= super[square];
      }

      return attacks;
   }

   /* Generate an attack bitboard for all attackers within a specified mask */
   inline bitboard_t<kind> generate_attack_bitboard(const board_t<kind> *board, const bitboard_t<kind> test_squares, const bitboard_t<kind> source_mask, side_t side_to_move) const
   {
      piece_description_t<kind> *piece_types;
      move_flag_t *piece_capture_flags;
      bitboard_t<kind> own, enemy, own_movers;
      bitboard_t<kind> occupied;
      bitboard_t<kind> attacked;
      int n;

      piece_types = board->piece_types;
      piece_capture_flags = piece_types->piece_capture_flags;

      /* Bookkeeping: we keep a pointer to the next move in the move list, and
       * update the number of moves in the list at the end of this function
       */
      own = board->bbc[side_to_move];
      enemy = board->bbc[next_side[side_to_move]];

      occupied = own | enemy | test_squares;

      own_movers = own & source_mask;

      bitboard_t<kind> possible_attackers = own_movers;

      for (n=0; n<piece_types->num_piece_types && !possible_attackers.is_empty(); n++) {
         if ((possible_attackers & board->bbp[n]).is_empty()) continue;
         possible_attackers &= ~board->bbp[n];

         bitboard_t<kind> bb = own_movers & board->bbp[n];

         /* Steppers */
         if (is_stepper(piece_capture_flags[n])) {
            int si = get_stepper_index(piece_capture_flags[n]);
            int d;
            for (d=0; d<8; d++) {
               int c = (stepper_description[si][side_to_move] >> (d*4)) & 15;
               bitboard_t<kind> captures = bb;
               /* We have a repetition count, so we do a number of steps one after the other.
                * This can effectively duplicate a slider.
                */
               for ( ; c>0; c--) {
                  captures &= step_mask[d];
                  captures = captures.sshift(step_shift[d]);
                  captures &= piece_types->prison[side_to_move][n];
                  attacked |= captures;
                  captures &= ~occupied;
               }
            }
         }

         /* Sliders and leapers */
         if (piece_capture_flags[n] & MF_HOPSLIDELEAP)
         while (!bb.is_empty()) {
            move_flag_t capture_flags = piece_capture_flags[n];
            int from = bb.bitscan();
            bb.reset(from);

            if (is_leaper(capture_flags)) attacked |= generate_leaper_move_bitboard(capture_flags, side_to_move, from, occupied);
            if (is_slider(capture_flags)) attacked |= generate_slider_move_bitboard(capture_flags, side_to_move, from, occupied);
            if (is_hopper(capture_flags)) attacked |= generate_hopper_move_bitboard(capture_flags, side_to_move, from, occupied);
            attacked &= piece_types->prison[side_to_move][n];
         }
      }

      return attacked;
   }

   inline bitboard_t<kind> generate_move_bitboard_for_flags(move_flag_t flags, int square, const bitboard_t<kind> occupied, side_t side_to_move) const
   {
      bitboard_t<kind> attacked;

      /* Steppers */
      if (is_stepper(flags)) {
         bitboard_t<kind> bb = bitboard_t<kind>::square_bitboards[square];
         int si = get_stepper_index(flags);
         for (int d=0; d<8; d++) {
            int c = (stepper_description[si][side_to_move] >> (d*4)) & 15;
            bitboard_t<kind> captures = bb;
            /* We have a repetition count, so we do a number of steps one after the other.
             * This can effectively duplicate a slider.
             */
            for ( ; c>0; c--) {
               captures &= step_mask[d];
               captures = captures.sshift(step_shift[d]);
               attacked |= captures;
               captures &= ~occupied;
            }
         }
      }

      /* Sliders and leapers */
      if (flags & MF_HOPSLIDELEAP) {
         if (is_leaper(flags)) attacked |= generate_leaper_move_bitboard(flags, side_to_move, square, occupied);
         if (is_slider(flags)) attacked |= generate_slider_move_bitboard(flags, side_to_move, square, occupied);
         if (is_hopper(flags)) attacked |= generate_hopper_move_bitboard(flags, side_to_move, square, occupied);
      }

      return attacked;
   }

   inline bitboard_t<kind>
   generate_move_bitboard_from_squares_for_flags(move_flag_t flags, bitboard_t<kind> squares, const bitboard_t<kind> occupied, side_t side_to_move) const
   {
      bitboard_t<kind> attacked;
      while (!squares.is_empty()) {
         int square = squares.bitscan();
         squares.reset(square);
         attacked |= generate_move_bitboard_for_flags(flags, square, occupied, side_to_move);
      }
      return attacked;
   }

   /* Generate an attack bitboard for all attackers within a specified mask */
   inline bitboard_t<kind> generate_moves_bitboard(const board_t<kind> *board, bitboard_t<kind> test_squares, bitboard_t<kind> source_mask, side_t side_to_move) const
   {
      piece_description_t<kind> *piece_types;
      move_flag_t *piece_move_flags;
      bitboard_t<kind> own, enemy, own_movers;
      bitboard_t<kind> occupied;
      bitboard_t<kind> attacked;
      int n;

      piece_types = board->piece_types;
      piece_move_flags = piece_types->piece_move_flags;

      /* Bookkeeping: we keep a pointer to the next move in the move list, and
       * update the number of moves in the list at the end of this function
       */
      own = board->bbc[side_to_move];
      enemy = board->bbc[next_side[side_to_move]];

      occupied = own | enemy | test_squares;

      own_movers = own & source_mask;

      bitboard_t<kind> possible_attackers = own_movers;

      for (n=0; n<piece_types->num_piece_types && !possible_attackers.is_empty(); n++) {
         if ((possible_attackers & board->bbp[n]).is_empty()) continue;
         possible_attackers &= ~board->bbp[n];

         bitboard_t<kind> bb = own_movers & board->bbp[n];

         /* Steppers */
         if (is_stepper(piece_move_flags[n])) {
            int si = get_stepper_index(piece_move_flags[n]);
            int d;
            for (d=0; d<8; d++) {
               int c = (stepper_description[si][side_to_move] >> (d*4)) & 15;
               bitboard_t<kind> captures = bb;
               /* We have a repetition count, so we do a number of steps one after the other.
                * This can effectively duplicate a slider.
                */
               for ( ; c>0; c--) {
                  captures &= step_mask[d];
                  captures = captures.sshift(step_shift[d]);
                  captures &= piece_types->prison[side_to_move][n];
                  attacked |= captures;
                  captures &= ~occupied;
               }
            }
         }

         /* Sliders and leapers */
         if (piece_move_flags[n] & MF_HOPSLIDELEAP)
         while (!bb.is_empty()) {
            move_flag_t capture_flags = piece_move_flags[n];
            int from = bb.bitscan();
            bb.reset(from);

            if (is_leaper(capture_flags)) attacked |= generate_leaper_move_bitboard(capture_flags, side_to_move, from, occupied);
            if (is_slider(capture_flags)) attacked |= generate_slider_move_bitboard(capture_flags, side_to_move, from, occupied);
            if (is_hopper(capture_flags)) attacked |= generate_hopper_move_bitboard(capture_flags, side_to_move, from, occupied);
         }

         attacked &= piece_types->prison[side_to_move][n];
      }

      return attacked;
   }

   /* Returns TRUE or FALSE depending on whether to given side may still
    * castle or not in the given position.
    */
   inline bool may_castle(const board_t<kind> *board, side_t side) const
   {
      if (board->did_castle(side))
         return false;

      if ((board->init & board->royal & board->bbc[side]).is_empty())
         return false;

      if ((board->init & short_castle_mask[side] & board->bbc[side]).is_empty())
         return false;

      if ((board->init & long_castle_mask[side] & board->bbc[side]).is_empty())
         return false;

      return true;
   }


   move_flag_t define_slider(const char *movestr) {
      const char *s = movestr;
      move_flag_t flags = 0;

      while (*s && isspace(*s)) s++;

      if (*s == '\0')
         return 0;

      int shift = 0;

      if (strstr(s, "slide") == s)
         shift = 0;

      if (strstr(s, "hop") == s)
         shift = 4;

      while (*s) {
         switch (*s) {
            case 'H':
               flags |= MF_SLIDER_H << shift;
               break;
            case 'V':
               flags |= MF_SLIDER_V << shift;
               break;
            case 'D':
               flags |= MF_SLIDER_D << shift;
               break;
            case 'A':
               flags |= MF_SLIDER_A << shift;
               break;
            case ')':
               break;
            default:
               break;
         }
         s++;
      }

      if (shift == 0) {
         slider_flags[number_of_sliders] = flags;
         number_of_sliders++;
      } else {
         hopper_flags[number_of_hoppers] = flags;
         number_of_hoppers++;
      }

      return flags;
   }

   /* Find a leaper with a particular move pattern */
   int find_leaper_description(int n, int m) const
   {
      int index = -1;
      int c;

      for (c=0; c<number_of_leapers; c++) {
         if (leaper_description[c] == (n | (m<<4))) { index = c; break; }
         if (leaper_description[c] == (m | (n<<4))) { index = c; break; }
      }

      return index;
   }


#define update_leaper_bb(bb,n,m)                                  \
   if ( (x+n) >= 0 && (y+m) >= 0 &&                               \
        (x+n) < board_files && (y+m) < board_ranks) {             \
      int dest_sqr = bitboard_t<kind>::pack_rank_file(y+m, x+n);  \
      bb.set(dest_sqr);                                           \
   }
#define update_leaper(leaper_bb,n,m) update_leaper_bb(leaper_bb[number_of_leapers][sqr], n, m)

   move_flag_t define_asymmetric_leaper(const char *movestr) {
      int board_files = bitboard_t<kind>::board_files;
      int board_ranks = bitboard_t<kind>::board_ranks;
      int size = board_files * board_ranks;
      for (int n = 0; n < size; n++) {
         aleaper[0][number_of_leapers][n].clear();
         aleaper[1][number_of_leapers][n].clear();
      }

      const char *s = movestr;
      char op = ' ';

      int index, index2, indexm, index_flags;
      index = index2 = indexm = index_flags = 0;
      uint8_t description = 0;

      s++;
      while (*s) {
         int n, m;

         while (*s && s[-1] != '(') s++;
         if(!*s) break;
         sscanf(s, "%d", &n);

         while (*s && s[-1] != ',') s++;
         if(!*s) break;
         sscanf(s, "%d", &m); s++;

         switch (op) {
            case '|':   /* Define a leaper with more than one type of move */
            case ' ':
               index = find_leaper_description(n, m);
               if (index == -1 || op == '|') {
                  leaper_description[number_of_leapers] = n | (m<<4);
                  index = number_of_leapers;
               }
               index_flags |= 1;
               break;

            case '+':   /* A compound leaper, with two steps one after the other */
               index2 = find_leaper_description(n, m);
               if (index2 == -1) {
                  number_of_leapers++;
                  if (number_of_leapers >= MAX_LEAPER_TYPES)
                     return -1;
                  index2 = number_of_leapers;
                  leaper_description[number_of_leapers] = n | (m<<4);
               }
               index_flags |= 2;
               break;

            case '&':   /* A compound leaper, with a mask (used to implement "lame leapers") */
               /* Define a new type of leaper for the mask.
                * FIXME: check if this type was already defined and re-use.
                */
               indexm = find_leaper_description(n, m);
               if (indexm == -1) {
                  number_of_leapers++;
                  if (number_of_leapers >= MAX_LEAPER_TYPES)
                     return -1;
                  indexm = number_of_leapers;
                  leaper_description[number_of_leapers] = n | (m<<4);
               }
               index_flags |= 4;
               break;
         }

         if (op == '|')
            leaper_description[number_of_leapers] = 0xff; /* Invalid */

         /* Dimensions of the board */
         int w = board_files;
         int h = board_ranks;

         int x, y, dx, dy;
         for (y=0; y<h; y++) {
            for (x=0; x<w; x++) {
               int sqr = x + y*w;
               /* White */
               update_leaper(aleaper[WHITE], n, m);

               /* Black */
               update_leaper(aleaper[BLACK], n, -m);
            }
         }

         s++;
         while (*s && *s == ')') s++;
         op = s[0];
      }

      if (index == number_of_leapers || index_flags > 1)
         number_of_leapers++;

      return (index | (index2 << 4) | (indexm << 8) | (index_flags << 12))<<16 | MF_LEAPER_ASYMM | MF_IS_LEAPER;
   }

   move_flag_t define_symmetric_leaper(const char *movestr) {
      int board_files = bitboard_t<kind>::board_files;
      int board_ranks = bitboard_t<kind>::board_ranks;
      int size = board_files * board_ranks;
      for (int n = 0; n < size; n++) {
         aleaper[0][number_of_leapers][n].clear();
         aleaper[1][number_of_leapers][n].clear();
      }

      const char *s = movestr;
      char op = ' ';

      int index, index2, indexm, index_flags;
      index = index2 = indexm = index_flags = 0;
      uint8_t description = 0;

      while (*s && *s != ' ') s++;
      s++;
      while (*s) {
         int n, m;

         while (*s && s[-1] != '(') s++;
         while (*s && *s == '(') s++;
         if(!*s) break;
         sscanf(s, "%d", &n);

         while (*s && s[-1] != ',') s++;
         if(!*s) break;
         sscanf(s, "%d", &m); s++;

         switch (op) {
            case '|':   /* Define a leaper with more than one type of move */
            case ' ':
               index = find_leaper_description(n, m);
               if (index == -1 || op == '|') {
                  leaper_description[number_of_leapers] = n | (m<<4);
                  index = number_of_leapers;
               }
               index_flags |= 1;
               break;

            case '+':   /* A compound leaper, with two steps one after the other */
               index2 = find_leaper_description(n, m);
               if (index2 == -1) {
                  number_of_leapers++;
                  if (number_of_leapers >= MAX_LEAPER_TYPES)
                     return -1;
                  index2 = number_of_leapers;
                  leaper_description[number_of_leapers] = n | (m<<4);
               }
               index_flags |= 2;
               break;

            case '&':   /* A compound leaper, with a mask (used to implement "lame leapers") */
               /* Define a new type of leaper for the mask.
                * FIXME: check if this type was already defined and re-use.
                */
               indexm = find_leaper_description(n, m);
               if (indexm == -1) {
                  number_of_leapers++;
                  if (number_of_leapers >= MAX_LEAPER_TYPES)
                     return -1;
                  indexm = number_of_leapers;
                  leaper_description[number_of_leapers] = n | (m<<4);
               }
               index_flags |= 4;
               break;
         }

         if (op == '|')
            leaper_description[number_of_leapers] = 0xff; /* Invalid */

         /* Dimensions of the board */
         int w = board_files;
         int h = board_ranks;

         int x, y, dx, dy;
         for (y=0; y<h; y++) {
            for (x=0; x<w; x++) {
               int sqr = bitboard_t<kind>::pack_rank_file(y, x);

               update_leaper(leaper, n, m);
               update_leaper(leaper, n,-m);
               update_leaper(leaper,-n, m);
               update_leaper(leaper,-n,-m);

               update_leaper(leaper, m, n);
               update_leaper(leaper,-m, n);
               update_leaper(leaper, m,-n);
               update_leaper(leaper,-m,-n);
            }
         }

         s++;
         while (*s && *s == ')') s++;
         op = s[0];
      }

      if (index == number_of_leapers || index_flags > 1)
         number_of_leapers++;

      return (index | (index2 << 4) | (indexm << 8) | (index_flags << 12))<<16 | MF_IS_LEAPER;
   }


   bitboard_t<kind> make_leaper_bitboard(int sqr, int n, int m) const
   {
      int board_files = bitboard_t<kind>::board_files;
      int board_ranks = bitboard_t<kind>::board_ranks;

      int x = unpack_file(sqr);
      int y = unpack_rank(sqr);

      bitboard_t<kind> leaper;

      update_leaper_bb(leaper, n, m);
      update_leaper_bb(leaper, n,-m);
      update_leaper_bb(leaper,-n, m);
      update_leaper_bb(leaper,-n,-m);

      update_leaper_bb(leaper, m, n);
      update_leaper_bb(leaper,-m, n);
      update_leaper_bb(leaper, m,-n);
      update_leaper_bb(leaper,-m,-n);

      return leaper;
   }
#undef update_leaper
#undef update_leaper_bb


   move_flag_t define_stepper(const char *movestr) {
      const char *s = movestr;

      /* Dimensions of the board */
      int w = bitboard_t<kind>::board_files;
      int h = bitboard_t<kind>::board_ranks;

      if (number_of_steppers >= MAX_STEPPER_TYPES)
         return 0;

      if (!movestr)
         return 0;

      while (*s && isspace(*s)) s++;

      if (*s == '\0')
         return 0;

      if (strstr(s, "step ") != s)
         return 0;
      s+=5;

      stepper_description[number_of_steppers][WHITE] = 0;
      stepper_description[number_of_steppers][BLACK] = 0;
      while (*s) {
         int count = 1;
         int shift = 0;

         if (isdigit(*s)) {
            sscanf(s, "%d", &count);
            assert(count < 16);
            assert(count >= 0);
            s++;
         }

         if (strstr(s, "NE") == s) shift = 1;
         else if (strstr(s, "NW") == s) shift = 7;
         else if (strstr(s, "SE") == s) shift = 3;
         else if (strstr(s, "SW") == s) shift = 5;
         else if (strstr(s, "N") == s)  shift = 0;
         else if (strstr(s, "E") == s)  shift = 2;
         else if (strstr(s, "S") == s)  shift = 4;
         else if (strstr(s, "W") == s)  shift = 6;

         stepper_description[number_of_steppers][WHITE] |= count << (4*shift);
         stepper_description[number_of_steppers][BLACK] |= count << (4*inverse_step[shift]);
         
         while(*s && *s != ',')
            s++;
         if (*s) s++;
         while (*s && isspace(*s)) s++;
      }

      /* Calculate single step bitboards, mainly used for mobility calculation, since normal single steppers are
       * generated using bulk shifts.
       * TODO
       */
      number_of_steppers++;

      return make_stepper_index(number_of_steppers-1);
   }

   /* TODO: Betza/Mueller notation for piece movement describes moves and
    * captures in one go, not individually.
    */
   move_flag_t define_betza(const char *movestr) {
      const char *s = movestr;
      move_t flags = 0;

      while (*s) {
         const char *atom = s;
         /* Find first upper-case character */
         while (*atom && (islower(*atom) || isspace(*atom))) atom++;

         if (atom != s) {
            /* TODO: modifiers */
         }
         /* Repeated atoms = riders.
          * The only ones we implement are WW and FF
          */
         char a = atom[0];
         if (atom[1] == atom[0] || atom[1] == '0') {
            switch (*atom) {
               case 'W':
                  a = 'R';
                  break;

               case 'F':
                  a = 'B';
                  break;

               default:
                  return 0;
            }
            atom++;
         }
         switch (a) {
            case 'K':   /* King = FW */
               flags |= define_piece_move("leap (1,0)|(1,1)");
               break;

            case 'Q':   /* Queen = RB */
               flags |= define_piece_move("slide (H,V,A,D)");
               break;

            case 'R':   /* Rook = WW */
               flags |= define_piece_move("slide (H,V)");
               break;

            case 'B':   /* Bishop = FF */
               flags |= define_piece_move("slide (A,D)");
               break;

            case ' ':
            case 'O':   /* No move, or castling */
               break;

            case 'W':   /* Wazir = (1,0) */
               flags |= define_piece_move("leap (1,0)");
               break;

            case 'F':   /* Ferz = (1,1) */
               flags |= define_piece_move("leap (1,1)");
               break;

            case 'D':   /* Dabbabah = (2,0) */
               flags |= define_piece_move("leap (2,0)");
               break;

            case 'N':   /* Knight = (2,1) */
               flags |= define_piece_move("leap (2,1)");
               break;

            case 'A':   /* Alfil = (2, 2) */
               flags |= define_piece_move("leap (2,2)");
               break;

            case 'H':   /* Threeleaper = (3, 0) */
               flags |= define_piece_move("leap (3,0)");
               break;

            case 'C':   /* Camel = (3, 1) */
            case 'L':
               flags |= define_piece_move("leap (3,1)");
               break;

            case 'Z':   /* Zebra = (3, 2) */
            case 'J':
               flags |= define_piece_move("leap (3,2)");
               break;

            case 'G':   /* (3, 3) leaper */
               flags |= define_piece_move("leap (3,3)");
               break;

            default:
               return 0;
         }
         s = atom + 1;
      }

      return flags;
   }

   move_flag_t define_piece_move(const char *movestr) {
      if (!movestr) return 0;
      const char *s = movestr;
      while (isspace(*s)) s++;
      if (s[0] == '\0') return 0;

      /* What type of mover is this? */
      if (strstr(s, "none ") == s)
         return 0;
      if (strstr(s, "slide ") == s)
         return define_slider(s);
      if (strstr(s, "hop ") == s)
         return define_slider(s);
      if (strstr(s, "step ") == s)
         return define_stepper(s);
      if (strstr(s, "aleap ") == s)
         return define_asymmetric_leaper(s);
      if (strstr(s, "leap ") == s)
         return define_symmetric_leaper(s);

      /* TODO: try to interpret a Betza-like move description and translate
       * it to what is used internally.
       */
      return 0;
   }

   /* Deduce castle flags from king positions and destinations and rook locations. */
   void deduce_castle_flags(side_t side, int king_from, int king_to, int rook_from)
   {
      /* King-side or queen side? */
      bool king_side = (unpack_file(king_to) >= bitboard_t<kind>::board_files/2);
      int rook_to = king_side ? (king_to - 1) : (king_to + 1);
      int c, c_first, c_last;

      /* It is not enough that the king and rook have a clear path
       * between them: the path to the destination squares needs to be cleared
       * as well.
       * This is implied in normal chess, but not in FRC.
       */
      if (king_side) {
         short_castle_mask[side] = bitboard_t<kind>::square_bitboards[king_from] | bitboard_t<kind>::square_bitboards[rook_from];
         short_castle_free[side].clear();

         /* The path of the King */
         c_first = std::min(king_from, king_to);
         c_last  = std::max(king_from, king_to);
         for (c = c_first; c <= c_last; c++) {
            short_castle_free[side] |= bitboard_t<kind>::square_bitboards[c];
         }
         short_castle_safe[side] = short_castle_free[side] | bitboard_t<kind>::square_bitboards[king_from];

         /* The path of the Rook */
         c_first = std::min(rook_to, rook_from);
         c_last  = std::max(rook_to, rook_from);
         for (c = c_first; c <= c_last; c++) {
            short_castle_free[side] |= bitboard_t<kind>::square_bitboards[c];
         }

         /* Make sure the king and rook are not marked on the "free" bitboard.
          * Makes no difference for normal chess, but does affect FRC.
          */
         short_castle_free[side] &= ~short_castle_mask[side];
         short_castle_king_dest[side] = king_to;
      } else {
         long_castle_mask[side] = bitboard_t<kind>::square_bitboards[king_from] | bitboard_t<kind>::square_bitboards[rook_from];
         long_castle_free[side].clear();

         /* The path of the King */
         c_last  = std::max(king_from, king_to);
         c_first = std::min(king_from, king_to);
         for (c = c_last; c >= c_first; c--) {
            long_castle_free[side] |= bitboard_t<kind>::square_bitboards[c];
         }
         long_castle_safe[side] = long_castle_free[side];

         /* The path of the Rook */
         c_first = std::min(rook_from+1, rook_to);
         c_last  = std::max(rook_from+1, rook_to);
         for (c = c_first; c <= c_last; c++) {
            long_castle_free[side] |= bitboard_t<kind>::square_bitboards[c];
         }

         /* Make sure the king and rook are not marked on the "free" bitboard.
          * Makes no difference for normal chess, but does affect FRC.
          */
         long_castle_free[side] &= ~long_castle_mask[side];
         long_castle_king_dest[side] = king_to;
      }
   }

   bitboard_t<kind> get_all_attackers(const board_t<kind> *board, bitboard_t<kind> mask, int square) const
   {
      bitboard_t<kind> occupied           = board->get_occupied() & mask;
      bitboard_t<kind> possible_attackers = occupied & super[square];
      occupied.set(square);

      bitboard_t<kind> attacked;
      bitboard_t<kind> attacker;

      for (int n=0; n<board->piece_types->num_piece_types && !possible_attackers.is_empty(); n++) {
         move_flag_t capture_flags = board->piece_types->piece_capture_flags[n];
         for (side_t side = WHITE; side<=BLACK; side++) {
            bitboard_t<kind> bb = possible_attackers & board->bbp[n] & board->bbc[side];

            if (bb.is_empty()) continue;
            possible_attackers ^= bb;

            /* Steppers */
            if (is_stepper(capture_flags)) {
               int si = get_stepper_index(board->piece_types->piece_capture_flags[n]);
               int d;
               for (d=0; d<8; d++) {
                  int max_c = (stepper_description[si][side] >> (d*4)) & 15;
                  bitboard_t<kind> captures = bb & super_stepper[square];
                  /* We have a repetition count, so we do a number of steps one after the other.
                   * This can effectively duplicate a slider.
                   */
                  for (int c = 1; c<=max_c && !captures.is_empty(); c++) {
                     captures &= step_mask[d];
                     captures = captures.sshift(step_shift[d]);
                     captures &= board->piece_types->prison[side][n];
                     if (captures.test(square)) {
                        attacker.set(square - c*step_shift[d]);
                        break;
                     }
                     captures &= ~occupied;
                  }
               }
            }

            /* Sliders and leapers */
            if (is_leaper(capture_flags)) {
               bitboard_t<kind> bp = bb;
               while (!(bp & super_leaper[square]).is_empty()) {
                  int s = (bp & super_leaper[square]).bitscan();
                  bp.reset(s);
                  attacked = generate_leaper_move_bitboard(capture_flags, side, s, occupied);
                  attacked &= board->piece_types->prison[side][n];

                  if (attacked.test(square)) {
                     attacker.set(s);
                     bb.reset(s);
                  }
               }
            }
#if 0
            if (is_slider(capture_flags)) {
               bitboard_t<kind> bp = bb;
               while (!(bp & super_slider[square]).is_empty()) {
                  int s = (bp & super_slider[square]).bitscan();
                  bp.reset(s);
                  attacked = generate_slider_move_bitboard(capture_flags, side, s, occupied);
                  attacked &= board->piece_types->prison[side][n];

                  if (attacked.test(square)) {
                     attacker.set(s);
                     bb.reset(s);
                  }
               }
            }
#endif
            if (is_hopper(capture_flags)) {
               bitboard_t<kind> bp = bb;
               while (!(bp & super_hopper[square]).is_empty()) {
                  int s = (bp & super_hopper[square]).bitscan();
                  bp.reset(s);
                  attacked = generate_hopper_move_bitboard(capture_flags, side, s, occupied);
                  attacked &= board->piece_types->prison[side][n];

                  if (attacked.test(square)) {
                     attacker.set(s);
                     bb.reset(s);
                  }
               }
            }
         }
      }

      /* Find sliders */
      move_flag_t cf[] = { MF_SLIDER_V, MF_SLIDER_H, MF_SLIDER_D, MF_SLIDER_A };
      for (int i=0; i<sizeof cf / sizeof *cf; i++) {
         bitboard_t<kind> sliders;
         for (int piece = 0; piece<board->piece_types->num_piece_types; piece++) {
            if (board->piece_types->piece_capture_flags[piece] & cf[i])
               sliders |= board->bbp[piece];
         }

         sliders &= super_slider[square] & mask;
         if (sliders.is_empty()) continue;

         attacker |= sliders & generate_slider_move_bitboard(cf[i], WHITE, square, occupied);
      }

      return attacker;
   }

   bool player_in_check(const board_t<kind> *board, side_t side) const
   {
      bitboard_t<kind> royal = board->royal & board->bbc[side];
      bitboard_t<kind> empty;

      /* If there are no royal pieces, then there is no check */
      if (royal.is_empty()) return false;

      /* If there is more than one king, we can never be in check - unless
       * the rules say we are when all kings are under attack.
       */
      if (!royal.onebit() && !(board->rule_flags & RF_KING_DUPLECHECK)) return false;

      move_flag_t *capture_flags = board->piece_types->piece_capture_flags;
      bitboard_t<kind> sup[4];
      bitboard_t<kind> mask[4];
      bitboard_t<kind> kmask;

      bitboard_t<kind> bb = royal;
      while (!bb.is_empty()) {
         int square = bb.bitscan();
         bb.reset(square);
         sup[0] |= super_slider[square];
         sup[1] |= super_leaper[square];
         sup[2] |= super_stepper[square];
         sup[3] |= super_hopper[square];
         kmask   = super[square] & board->bbc[next_side[side]];
      }
      if (royal.onebit() && kmask.is_empty()) return false;
      for (int n=0; n<4; n++)
         sup[n] &= board->bbc[next_side[side]];
      for (int n = 0; n<board->piece_types->num_piece_types; n++) {
         if (is_slider(capture_flags[n]))  mask[0] |= sup[0] & board->bbp[n];
         if (is_leaper(capture_flags[n]))  mask[1] |= sup[1] & board->bbp[n];
         if (is_stepper(capture_flags[n])) mask[2] |= sup[2] & board->bbp[n];
         if (is_hopper(capture_flags[n]))  mask[3] |= sup[3] & board->bbp[n];
      }

      /* Mask out pieces that occur in more than one mask: we only need to
       * test them once, afterall.
       */
      mask[3] &= ~(mask[0] | mask[1] | mask[2]);
      mask[2] &= ~(mask[0] | mask[1]);
      mask[1] &= ~(mask[0]);

      bitboard_t<kind> attacked_squares;
      
      /* TODO: we can do one better, at least for sliders and normal
       * leapers (lame leapers are more tricky): when generating the attack
       * bitboard, first generate appropriate attacks from the target
       * square and intersect with the piece type. This allows us to test
       * against all pieces of the particular type in one go, and we avoid
       * some possible false positives.
       */
      for (int n=0; n<4; n++) {
         if (mask[n].is_empty()) continue;
         attacked_squares |= generate_attack_bitboard(board, empty, mask[n], next_side[side]);
         if ((attacked_squares & royal) == royal) return true;
      }

      if (expect(board->rule_flags & RF_KING_TABOO, false)) {
         bitboard_t<kind> other_king = board->royal & kmask;
         if (!other_king.is_empty()) {
            int square = other_king.bitscan();
            bitboard_t<kind> occ = board->get_occupied();

            attacked_squares |= generate_slider_move_bitboard(MF_SLIDER_V, next_side[side], square, occ);

         }
         return (attacked_squares & royal) == royal;
      }
      return false;
   }

   bool was_checking_move(board_t<kind> *board, side_t side, move_t lastmove) const
   {
      side_t oside = next_side[side];
      bitboard_t<kind> royal = board->royal & board->bbc[side];
      bitboard_t<kind> empty;

      if (royal.is_empty()) return false;

      if (royal.onebit()) {
         bitboard_t<kind> move_bb;
         int king = royal.bitscan();
         int from = get_move_from(lastmove);
         int to   = get_move_to(lastmove);

         move_bb.set(from);
         move_bb.set(to);

         if (is_castle_move(lastmove)) {
            move_bb.set(get_castle_move_from2(lastmove));
            move_bb.set(get_castle_move_to2(lastmove));
         }

         if (is_capture_move(lastmove))
            move_bb.set(get_move_capture_square(lastmove));

         if ((super[king] & move_bb).is_empty()) {
            assert(!player_in_check(board, side));
            return false;
         } else {
            bitboard_t<kind> mask;
            move_bb.reset(to);
            if (is_castle_move(lastmove)) {
               mask.set(get_castle_move_to2(lastmove));
               move_bb.reset(get_castle_move_to2(lastmove));
            }

            int king_file = unpack_file(king);
            int king_rank = unpack_rank(king);
            int king_diag = bitboard_t<kind>::diagonal_nr[king];
            int king_anti = bitboard_t<kind>::anti_diagonal_nr[king];
            while (!move_bb.is_empty()) {
               int square = move_bb.bitscan();
               int file = unpack_file(square);
               int rank = unpack_rank(square);
               int diag = bitboard_t<kind>::diagonal_nr[square];
               int anti = bitboard_t<kind>::anti_diagonal_nr[square];
               move_bb.reset(square);

               if (file == king_file) mask |= bitboard_t<kind>::board_file[file];
               if (rank == king_rank) mask |= bitboard_t<kind>::board_rank[rank];
               if (diag == king_diag) mask |= bitboard_t<kind>::board_diagonal[diag];
               if (anti == king_anti) mask |= bitboard_t<kind>::board_antidiagonal[anti];
            }

            //bitboard_t<kind> sliders;
            //for (int n = 0; n<board->piece_types->num_piece_types; n++) {
            //   if (board->piece_types->piece_capture_flags[n] & (MF_HOPSLIDE|MF_STEPPER))
            //      sliders |= board->bbp[n];
            //}
            //mask &= sliders;
            mask.set(to);

            /* Possible lame leapers */
            mask |= super_leaper[king] & board->bbc[oside];

            bitboard_t<kind> attacked_squares = generate_attack_bitboard(board, empty, mask, oside);
            if (attacked_squares.test(king)) return true;
            return false;
         }
      }

      return player_in_check(board, side);
   }

   bitboard_t<kind> get_pinned_pieces(const board_t<kind> *board, side_t side) const
   {
      bitboard_t<kind> royal = board->royal & board->bbc[side];
      bitboard_t<kind> pinned;
      bitboard_t<kind> potential_pins;

      /* If there is more than one king, or no king at all - ignore pins.
       */
      if (!royal.onebit()) return pinned;

      int king = royal.bitscan();
      potential_pins = board->bbc[side] & super[king];

      for (int n = 0; n<board->piece_types->num_piece_types; n++) {
         bitboard_t<kind> atk = board->bbp[n] & board->bbc[next_side[side]] & super[king];
         move_flag_t atk_flags = board->piece_types->piece_capture_flags[n];

         while(!atk.is_empty()) {
            int attacker = atk.bitscan();
            atk.reset(attacker);

            /* Sliders */
            if (is_slider(atk_flags)) {
               bitboard_t<kind> bb = potential_pins &
                                     bitboard_t<kind>::board_between[king][attacker] &
                                     generate_slider_move_bitboard(atk_flags, next_side[side], attacker, potential_pins);
               if (bb.onebit()) pinned |= bb;
            }

            /* Hoppers */
            if (is_hopper(atk_flags)) {
               bitboard_t<kind> bb = potential_pins &
                                     bitboard_t<kind>::board_between[king][attacker] &
                                     generate_slider_move_bitboard(atk_flags>>4, next_side[side], attacker, potential_pins);
               if (bb.twobit()) pinned |= bb;
            }

            /* TODO: multi-steppers */

            /* Lame leapers */
            if (is_leaper(atk_flags) && is_masked_leaper(atk_flags) && is_double_leaper(atk_flags)) {
               bitboard_t<kind> occ = board->get_occupied();
               bitboard_t<kind> atk = generate_leaper_move_bitboard(atk_flags, next_side[side], attacker, occ);

               if (!atk.test(king)) {
                  int index = get_leaper_index(atk_flags);
                  potential_pins &= leaper[index][attacker];
                  while (!potential_pins.is_empty()) {
                     int square = potential_pins.bitscan();
                     potential_pins.reset(square);

                     occ.reset(square);
                     atk = generate_leaper_move_bitboard(atk_flags, next_side[side], attacker, occ);
                     if (atk.test(king)) {
                        pinned.set(square);
                        break;
                     }
                  }
               }
            }
         }
      }

      return pinned;
   }


   template<bool special>
   void generate_stepper_moves_mask_for_piece(movelist_t *movelist, const board_t<kind> *board,
      int piece, move_flag_t move_flags, uint8_t piece_flags, piece_description_t<kind> *piece_types,
      bitboard_t<kind> from_bb, bitboard_t<kind> destination_mask, bitboard_t<kind> occupied,
      bitboard_t<kind> promotion_zone, bitboard_t<kind> optional_promotion_zone,
      side_t side_to_move, uint32_t allowed_promotion_pieces) const
   {
      move_t move;

      /* Check for stepper moves, which are generated in parallel */
      if (is_stepper(move_flags)) {
         int si = get_stepper_index(move_flags);
         for (int d=0; d<8; d++) {   /* Loop over all directions */
            int max_c = (stepper_description[si][side_to_move] >> (d*4)) & 15;
            bitboard_t<kind> moves = from_bb;
            /* We have a repetition count, so we do a number of steps one after the other.
             * This can effectively duplicate a slider.
             */
            for (int c = 1; c<=max_c; c++) {
               moves &= step_mask[d];
               moves = moves.sshift(step_shift[d]);
               moves &= ~occupied;
               moves &= board->piece_types->prison[side_to_move][neutral_piece(piece)];

               /* Scan all bits */
               bitboard_t<kind> bb = moves & destination_mask;
               while (!bb.is_empty()) {
                  int to = bb.bitscan();
                  int from = to - c*step_shift[d];
                  bb.reset(to);

                  /* Check for promotions
                   * When the piece moves into the promotion zone, it will get promoted to one of the allowed
                   * promotion pieces, which can be different for each piece type (and further restricted, for
                   * instance during Q-search).
                   * Promotion to a royal piece is only allowed if the number of royal pieces a player has is
                   * smaller than the maximum number of royal pieces.
                   */
                  if (promotion_zone.test(to) || promotion_zone.test(from)) {
                     uint16_t c = allowed_promotion_pieces;
                     while (c) {
                        int tpiece = bitscan16(c);
                        c ^= 1<<tpiece;
                        if (piece_types->piece_maximum[tpiece][side_to_move] == 128 ||
                            board->piece_count(tpiece, side_to_move) < piece_types->piece_maximum[tpiece][side_to_move]) {
                           tpiece = piece_for_side(tpiece, side_to_move);
                           move = encode_normal_promotion(piece, from, to, tpiece);
                           movelist->push(move);
                        }
                     }
                     /* If promotions are optional, we also encode a normal move */
                     if (optional_promotion_zone.test(to)) {
                        move = encode_normal_move(piece, from, to);
                        if (special && c>1 && piece_flags & PF_SET_EP)
                           move |= MOVE_SET_ENPASSANT;
                        if (piece_flags & PF_NORET)
                           move |= MOVE_RESET50;
                        movelist->push(move);
                     }
                  } else {
                     move = encode_normal_move(piece, from, to);
                     if (special && c>1 && piece_flags & PF_SET_EP)
                        move |= MOVE_SET_ENPASSANT;
                     if (piece_flags & PF_NORET)
                        move |= MOVE_RESET50;
                     movelist->push(move);
                  }
               }
            }
         }
      }
   }

   template<bool capture_to_holdings>
   void generate_stepper_captures_mask_for_piece(movelist_t *movelist, const board_t<kind> *board,
      int piece, move_flag_t move_flags, uint8_t piece_flags, piece_description_t<kind> *piece_types,
      bitboard_t<kind> from_bb, bitboard_t<kind> destination_mask, bitboard_t<kind> occupied,
      bitboard_t<kind> enemy, bitboard_t<kind> ep_capture,
      bitboard_t<kind> promotion_zone, bitboard_t<kind> optional_promotion_zone,
      side_t side_to_move, uint32_t allowed_promotion_pieces) const
   {
      move_t move;

      /* Check for stepper moves, which are generated in parallel */
      if (is_stepper(move_flags)) {
         int si = get_stepper_index(move_flags);
         for (int d=0; d<8; d++) {   /* Loop over all directions */
            int max_c = (stepper_description[si][side_to_move] >> (d*4)) & 15;
            bitboard_t<kind> captures = from_bb;
            /* We have a repetition count, so we do a number of steps one after the other.
             * This can effectively duplicate a slider.
             */
            for (int c = 1; c<=max_c; c++) {
               captures &= step_mask[d];
               captures = captures.sshift(step_shift[d]);
               captures &= board->piece_types->prison[side_to_move][neutral_piece(piece)];

               /* Scan all bits */
               bitboard_t<kind> bb = captures & (enemy | ep_capture) & destination_mask;
               while (!bb.is_empty()) {
                  int to = bb.bitscan();
                  int from = to - c*step_shift[d];
                  int ptaken = piece_for_side(board->get_piece(to), next_side[side_to_move]);
                  bb.reset(to);

                  /* Check for promotions
                   * When the piece moves into the promotion zone, it will get promoted to one of the allowed
                   * promotion pieces, which can be different for each piece type (and further restricted, for
                   * instance during Q-search).
                   * Promotion to a royal piece is only allowed if the number of royal pieces a player has is
                   * smaller than the maximum number of royal pieces.
                   */
                  if (promotion_zone.test(to) || promotion_zone.test(from)) {
                     uint16_t c = allowed_promotion_pieces;
                     while (c) {
                        int tpiece = bitscan16(c);
                        c ^= 1<<tpiece;
                        if (piece_types->piece_maximum[tpiece][side_to_move] == 128 ||
                            board->piece_count(tpiece, side_to_move) < piece_types->piece_maximum[tpiece][side_to_move]) {
                           tpiece = piece_for_side(tpiece, side_to_move);
                           move = encode_capture_promotion(piece, from, to, ptaken, tpiece);
                           if (capture_to_holdings) {
                              int pstore = piece_types->demotion[board->get_piece(to)];
                              side_t store_side = NONE;
                              if (board->rule_flags & RF_KEEP_CAPTURE)   store_side = side_to_move;
                              if (board->rule_flags & RF_RETURN_CAPTURE) store_side = next_side[side_to_move];
                              assert(store_side != NONE);
                              move = add_move_store(move, piece_for_side(pstore, store_side), 1);
                           }
                           movelist->push(move);
                        }
                     }
                     /* If promotions are optional, we also encode a normal move */
                     if (optional_promotion_zone.test(to)) {
                        if (ep_capture.test(to)) {
                           int ptaken = piece_for_side(board->get_piece(board->ep_capture), next_side[side_to_move]);
                           move = encode_en_passant_capture(piece, from, to, ptaken, board->ep_capture);
                        } else {
                           move = encode_normal_capture(piece, from, to, ptaken);
                        }
                        if (capture_to_holdings) {
                           int pstore = piece_types->demotion[board->get_piece(to)];
                           side_t store_side = NONE;
                           if (board->rule_flags & RF_KEEP_CAPTURE)   store_side = side_to_move;
                           if (board->rule_flags & RF_RETURN_CAPTURE) store_side = next_side[side_to_move];
                           assert(store_side != NONE);
                           move = add_move_store(move, piece_for_side(pstore, store_side), 1);
                        }
                        movelist->push(move);
                     }
                  } else {
                     if (ep_capture.test(to)) {
                        int ptaken = piece_for_side(board->get_piece(board->ep_capture), next_side[side_to_move]);
                        move = encode_en_passant_capture(piece, from, to, ptaken, board->ep_capture);
                     } else {
                        move = encode_normal_capture(piece, from, to, ptaken);
                     }
                     if (capture_to_holdings) {
                        int pstore = piece_types->demotion[board->get_piece(to)];
                        side_t store_side = NONE;
                        if (board->rule_flags & RF_KEEP_CAPTURE)   store_side = side_to_move;
                        if (board->rule_flags & RF_RETURN_CAPTURE) store_side = next_side[side_to_move];
                        assert(store_side != NONE);
                        move = add_move_store(move, piece_for_side(pstore, store_side), 1);
                     }
                     movelist->push(move);
                  }
               }
               captures &= ~occupied;
            }
         }
      }
   }


   template<bool generate_drops, bool capture_to_holdings>
   void generate_moves_mask(movelist_t *movelist, const board_t<kind> *board, bitboard_t<kind> source_mask, bitboard_t<kind> destination_mask, side_t side_to_move, uint32_t allowed_promotion_pieces, uint32_t allowed_drop_pieces) const
   {
      piece_description_t<kind> *piece_types;
      move_flag_t *piece_capture_flags;
      move_flag_t *piece_move_flags;
      move_flag_t *special_move_flags;
      bitboard_t<kind> own, enemy, own_movers;
      bitboard_t<kind> occupied;
      bitboard_t<kind> attacked;
      move_t move;
      int n;

      piece_types = board->piece_types;
      piece_capture_flags = piece_types->piece_capture_flags;
      piece_move_flags = piece_types->piece_move_flags;
      special_move_flags = piece_types->piece_special_move_flags;

      /* Bookkeeping: we keep a pointer to the next move in the move list, and
       * update the number of moves in the list at the end of this function
       */
      own = board->bbc[side_to_move];
      enemy = board->bbc[next_side[side_to_move]];

      occupied = own | enemy;

      own_movers = own & source_mask;
      bitboard_t<kind> movers = own_movers;

      /* Generate drops */
      if (generate_drops && allowed_drop_pieces && (board->rule_flags & (RF_ALLOW_DROPS | RF_FORCE_DROPS))) {
         bool dropped = false;
         for (n=0; n<piece_types->num_piece_types; n++) {
            if (board->holdings[n][side_to_move] && (allowed_drop_pieces & (1 << n))) {
               dropped = true;

               int piece = piece_for_side(n, side_to_move);
               bitboard_t<kind> drops = destination_mask & ~occupied & piece_types->drop_zone[side_to_move][n];

               if (piece_types->piece_flags[n] & PF_DROPONEFILE) {
                  for (int f = 0; f<bitboard_t<kind>::board_files; f++) {
                     if (!(own & board->bbp[n] & bitboard_t<kind>::board_file[f]).is_empty())
                        drops &= ~bitboard_t<kind>::board_file[f];
                  }
               }

               while (!drops.is_empty()) {
                  int to = drops.bitscan();
                  drops.reset(to);
                  move = encode_drop_move(piece, to);
                  move = add_move_retrieve(move, piece, 1);
                  movelist->push(move);
               }
            }
         }

         /* Break out early if drops are possible and are forced if possible; no other moves are legal. */
         if (dropped && (board->rule_flags & RF_FORCE_DROPS))
            goto done;
      }

      /* Generate lifts */
      if (generate_drops && allowed_drop_pieces && !board->check() && (board->rule_flags & RF_ALLOW_PICKUP)) {
         for (n=0; n<piece_types->num_piece_types; n++) {
            if (piece_types->piece_flags[n] & PF_ROYAL) continue;
            bitboard_t<kind> lift = own_movers & board->bbp[n];

            if (!lift.is_empty() && (allowed_drop_pieces & (1 << n))) {
               int piece = piece_for_side(n, side_to_move);

               while (!lift.is_empty()) {
                  int from = lift.bitscan();
                  lift.reset(from);
                  move = encode_pickup_move(piece, from);
                  move = add_move_store(move, piece, 1);
                  movelist->push(move);
               }
            }
         }
      }


      /* Now generate moves for all pieces; only scan our own pieces. This mainly helps variants with different
       * armies.
       * We generate all moves for a particular piece-type first.
       */
      for (n=0; n<piece_types->num_piece_types && !movers.is_empty(); n++) {
         if ((movers & board->bbp[n]).is_empty()) continue;
         movers &= ~board->bbp[n];

         bitboard_t<kind> special_zone = piece_types->special_zone[side_to_move][n];
         if (board->rule_flags & RF_SPECIAL_IS_INIT) special_zone &= board->init;

         bitboard_t<kind> ep_capture;

         if (piece_types->piece_flags[n] & PF_TAKE_EP && board->ep)
            ep_capture.set(board->ep);

         bitboard_t<kind> promotion_zone = piece_types->promotion_zone[side_to_move][n];
         bitboard_t<kind> optional_promotion_zone = piece_types->optional_promotion_zone[side_to_move][n];

         bitboard_t<kind> bb = own_movers & board->bbp[n];
         int piece = piece_for_side(n, side_to_move);

         /* In-place promotions */
         if ((board->rule_flags & RF_PROMOTE_IN_PLACE) && !(bb & optional_promotion_zone).is_empty()) {
            bitboard_t<kind> bp = bb & optional_promotion_zone;
            while (!bp.is_empty()) {
               int square = bp.bitscan();
               bp.reset(square);

               uint16_t c = piece_types->piece_promotion_choice[n] & allowed_promotion_pieces;
               while (c) {
                  int tpiece = bitscan16(c);
                  c ^= 1<<tpiece;
                  if (piece_types->piece_maximum[tpiece][side_to_move] == 128 ||
                        board->piece_count(tpiece, side_to_move) < piece_types->piece_maximum[tpiece][side_to_move]) {
                     tpiece = piece_for_side(tpiece, side_to_move);
                     move = encode_normal_promotion(piece, square, square, tpiece);
                     movelist->push(move);
                  }
               }
            }
         }

         /* Generate stepper moves, in parallel */
         generate_stepper_moves_mask_for_piece<false>(movelist, board,
                  piece, piece_move_flags[n], piece_types->piece_flags[n], piece_types,
                  bb & ~special_zone, destination_mask, occupied, promotion_zone, optional_promotion_zone,
                  side_to_move, piece_types->piece_promotion_choice[n] & allowed_promotion_pieces);
         generate_stepper_moves_mask_for_piece<true>(movelist, board,
                  piece, special_move_flags[n], piece_types->piece_flags[n], piece_types,
                  bb & special_zone, destination_mask, occupied, promotion_zone, optional_promotion_zone,
                  side_to_move, piece_types->piece_promotion_choice[n] & allowed_promotion_pieces);
         generate_stepper_captures_mask_for_piece<capture_to_holdings>(movelist, board,
                  piece, piece_capture_flags[n], piece_types->piece_flags[n], piece_types,
                  bb, destination_mask, occupied, enemy, ep_capture, promotion_zone, optional_promotion_zone,
                  side_to_move, piece_types->piece_promotion_choice[n] & allowed_promotion_pieces);

         /* Castling
          * Because of the hassle when doing legality testing, we explicitly test whether castling is allowed in
          * the current position by testing for attacks on any of the critical squares. This is a hassle and
          * potentially slow, but only if castling may be possible in the current position.
          */
         if (expect(piece_types->piece_flags[n] & PF_CASTLE, false) && !(board->init&bb).is_empty()) {
            if ((board->init & short_castle_mask[side_to_move]) == short_castle_mask[side_to_move]) {
               if (short_castle_free[side_to_move].is_empty() ||
                   ((occupied & short_castle_free[side_to_move]).is_empty() &&
                    !(destination_mask & short_castle_free[side_to_move]).is_empty())) {
                  bitboard_t<kind> test             = short_castle_safe[side_to_move];
                  bitboard_t<kind> mask             = generate_super_attacks_for_squares(test, super);
                  bitboard_t<kind> attacked_squares = generate_attack_bitboard(board, test, mask, next_side[side_to_move]);
                  if ((attacked_squares & short_castle_safe[side_to_move]).is_empty()) {
                     int from1 = (short_castle_mask[side_to_move] &  bb).bitscan();
                     int from2 = (short_castle_mask[side_to_move] & ~bb).bitscan();
                     int piece2 = piece_for_side(board->get_piece(from2), side_to_move);
                     int to1 = short_castle_king_dest[side_to_move];
                     move = encode_castle_move(piece, from1, to1, piece2, from2, to1-1);
                     movelist->push(move);
                  }
               }
            }

            if ((board->init & long_castle_mask[side_to_move]) == long_castle_mask[side_to_move]) {
               if ((long_castle_free[side_to_move]).is_empty() ||
                   ((occupied & long_castle_free[side_to_move]).is_empty() &&
                    !(destination_mask & long_castle_free[side_to_move]).is_empty())) {
                  bitboard_t<kind> test             = long_castle_safe[side_to_move];
                  bitboard_t<kind> mask             = generate_super_attacks_for_squares(test, super);
                  bitboard_t<kind> attacked_squares = generate_attack_bitboard(board, test, mask, next_side[side_to_move]);
                  if ((attacked_squares & long_castle_safe[side_to_move]).is_empty()) {
                     int from1 = (long_castle_mask[side_to_move] &  bb).bitscan();
                     int from2 = (long_castle_mask[side_to_move] & ~bb).bitscan();
                     int piece2 = piece_for_side(board->get_piece(from2), side_to_move);
                     int to1 = long_castle_king_dest[side_to_move];
                     move = encode_castle_move(piece, from1, to1, piece2, from2, to1+1);
                     movelist->push(move);
                  }
               }
            }
         }

         /* Now determine slider and leaper moves for this piece type - if it has any */
         if ( (piece_move_flags[n] | piece_capture_flags[n] | special_move_flags[n]) & (MF_SLIDER|MF_HOPPER|MF_IS_LEAPER) ) {
            while (!bb.is_empty()) {
               bitboard_t<kind> moves;
               bitboard_t<kind> captures;
               int from = bb.bitscan();
               bb.reset(from);

               move_flag_t move_flags = piece_move_flags[n];
               move_flag_t capture_flags = piece_capture_flags[n];
               if (special_zone.test(from))
                  move_flags = special_move_flags[n];

               move_flags    &= (MF_SLIDER | MF_HOPPER | MF_LEAPER_FLAGS);
               capture_flags &= (MF_SLIDER | MF_HOPPER | MF_LEAPER_FLAGS);

               if (is_leaper(move_flags)) moves |= generate_leaper_move_bitboard(move_flags, side_to_move, from, occupied);
               if (is_slider(move_flags)) moves |= generate_slider_move_bitboard(move_flags, side_to_move, from, occupied);
               if (is_hopper(move_flags)) moves |= generate_hopper_move_bitboard(move_flags, side_to_move, from, occupied);
               moves &= piece_types->prison[side_to_move][n];

               /* Optimise the common case where pieces move the same way
                * they capture.
                */
               if (capture_flags == move_flags) {
                  captures = moves;
               } else {
                  if (is_leaper(capture_flags)) captures |= generate_leaper_move_bitboard(capture_flags, side_to_move, from, occupied);
                  if (is_slider(capture_flags)) captures |= generate_slider_move_bitboard(capture_flags, side_to_move, from, occupied);
                  if (is_hopper(capture_flags)) captures |= generate_hopper_move_bitboard(capture_flags, side_to_move, from, occupied);
                  captures &= piece_types->prison[side_to_move][n];
               }

               /* Mask out occupied squares from normal moves, only capture enemy pieces */
               moves &= ~occupied;
               captures &= enemy;

               moves &= destination_mask;
               captures &= destination_mask;

               /* Serialise moves
                * We separate out the promotion moves and options and
                * serialise those in a separate loop.
                */
               bitboard_t<kind> pmoves    = moves    & (promotion_zone & ~optional_promotion_zone);
               bitboard_t<kind> pcaptures = captures & (promotion_zone & ~optional_promotion_zone);
               moves     ^= pmoves;
               captures  ^= pcaptures;
               pmoves    |= moves & optional_promotion_zone;
               pcaptures |= captures & optional_promotion_zone;
               /* Also include moves that originate in the promotion zone. */
               if (promotion_zone.test(from)) {
                  pmoves    |= moves;
                  pcaptures |= captures;
               }

               while (!moves.is_empty()) {
                  int to = moves.bitscan();
                  moves.reset(to);
                  move = encode_normal_move(piece, from, to);
                  movelist->push(move);
               }
               while (!captures.is_empty()) {
                  int to = captures.bitscan();
                  int ptaken = piece_for_side(board->get_piece(to), next_side[side_to_move]);
                  captures.reset(to);
                  move = encode_normal_capture(piece, from, to, ptaken);
                  if (capture_to_holdings) {
                     int pstore = piece_types->demotion[board->get_piece(to)];
                     side_t store_side = NONE;
                     if (board->rule_flags & RF_KEEP_CAPTURE)   store_side = side_to_move;
                     if (board->rule_flags & RF_RETURN_CAPTURE) store_side = next_side[side_to_move];
                     assert(store_side != NONE);
                     move = add_move_store(move, piece_for_side(pstore, store_side), 1);
                  }
                  movelist->push(move);
               }

               /* Promotions */
               while (!pmoves.is_empty()) {
                  int to = pmoves.bitscan();
                  pmoves.reset(to);
                  uint16_t c = piece_types->piece_promotion_choice[n] & allowed_promotion_pieces;
                  while (c) {
                     int tpiece = bitscan16(c);
                     c ^= 1<<tpiece;
                     if (piece_types->piece_maximum[tpiece][side_to_move] == 128 ||
                           board->piece_count(tpiece, side_to_move) < piece_types->piece_maximum[tpiece][side_to_move]) {
                        tpiece = piece_for_side(tpiece, side_to_move);
                        move = encode_normal_promotion(piece, from, to, tpiece);
                        movelist->push(move);
                     }
                  }
               }
               while (!pcaptures.is_empty()) {
                  int to = pcaptures.bitscan();
                  int ptaken = piece_for_side(board->get_piece(to), next_side[side_to_move]);
                  pcaptures.reset(to);
                  uint16_t c = piece_types->piece_promotion_choice[n] & allowed_promotion_pieces;
                  while (c) {
                     int tpiece = bitscan16(c);
                     c ^= 1<<tpiece;
                     if (piece_types->piece_maximum[tpiece][side_to_move] == 128 ||
                           board->piece_count(tpiece, side_to_move) < piece_types->piece_maximum[tpiece][side_to_move]) {
                        tpiece = piece_for_side(tpiece, side_to_move);
                        move = encode_capture_promotion(piece, from, to, ptaken, tpiece);
                        if (capture_to_holdings) {
                           int pstore = piece_types->demotion[board->get_piece(to)];
                           side_t store_side = NONE;
                           if (board->rule_flags & RF_KEEP_CAPTURE)   store_side = side_to_move;
                           if (board->rule_flags & RF_RETURN_CAPTURE) store_side = next_side[side_to_move];
                           assert(store_side != NONE);
                           move = add_move_store(move, piece_for_side(pstore, store_side), 1);
                        }
                        movelist->push(move);
                     }
                  }
               }
            }
         }
      }

   done:
      return;
   }

   inline void generate_moves_mask_wrap(movelist_t *movelist, const board_t<kind> *board, bitboard_t<kind> source_mask, bitboard_t<kind> destination_mask, side_t side_to_move, uint32_t allowed_promotion_pieces, uint32_t allowed_drop_pieces, bool quiesc_only = false) const
   {
      if ((board->rule_flags & RF_USE_DROPS)) {
         if (!quiesc_only) {
            if (board->rule_flags & RF_USE_CAPTURE)
               generate_moves_mask<true, true>(movelist, board, source_mask, destination_mask, side_to_move, allowed_promotion_pieces, allowed_drop_pieces);
            else
               generate_moves_mask<true, false>(movelist, board, source_mask, destination_mask, side_to_move, allowed_promotion_pieces, allowed_drop_pieces);
         } else {
            bitboard_t<kind> oking = board->royal & board->bbc[next_side[side_to_move]];
            bitboard_t<kind> king_zone;
            if (oking.onebit())
               king_zone = bitboard_t<kind>::neighbour_board[oking.bitscan()];

            if (board->rule_flags & RF_USE_CAPTURE) {
               generate_moves_mask<false, true>(movelist, board, source_mask, destination_mask, side_to_move, allowed_promotion_pieces, allowed_drop_pieces);
               generate_moves_mask<true, true>(movelist, board, source_mask, king_zone, side_to_move, allowed_promotion_pieces, allowed_drop_pieces);
            } else {
               generate_moves_mask<false, false>(movelist, board, source_mask, destination_mask, side_to_move, allowed_promotion_pieces, allowed_drop_pieces);
               generate_moves_mask<true, false>(movelist, board, source_mask, king_zone, side_to_move, allowed_promotion_pieces, allowed_drop_pieces);
            }
         }
      } else
         generate_moves_mask<false, false>(movelist, board, source_mask, destination_mask, side_to_move, allowed_promotion_pieces, allowed_drop_pieces);
   }


   /* Generate pseudo-legal check evasions. This is mainly useful in that
    * it removes most illegal moves. It may not catch all of them though,
    * depending on rules and piece moves in a particular variant.
    */
   bool generate_evasions(movelist_t *movelist, const board_t<kind> *board, side_t side_to_move) const
   {
      assert(board->check());

      bitboard_t<kind> destination = bitboard_t<kind>::board_all;
      bitboard_t<kind> origin = bitboard_t<kind>::board_all;
      bitboard_t<kind> attacker, bb, kings, pinned, occ, destest;
      kings = board->royal & board->bbc[side_to_move];
      occ   = board->get_occupied();

      /* FIXME: if there are multiple kings we get duplicate moves in the
       * movelist.
       */
      if (!kings.onebit()) return false;

      movelist->num_moves = 0;

      /* Evasions */
      generate_moves_mask_wrap(movelist, board, kings, destination, side_to_move, 0xffffffff, 0);

      /* Identify attacking pieces */
      bb = kings;
      while (!bb.is_empty()) {
         int king = bb.bitscan();
         bb.reset(king);
         attacker |= get_all_attackers(board, occ, king);
      }
      assert(!attacker.is_empty());

      /* Captures of attacking pieces */
      generate_moves_mask_wrap(movelist, board, origin^kings, attacker, side_to_move, 0xffffffff, 0);
      destest |= attacker;

      /* En-passant captures */
      if (attacker.test(board->ep_capture)) {
         bitboard_t<kind> bb; bb.set(board->ep);
         generate_moves_mask_wrap(movelist, board, origin^kings, bb, side_to_move, 0xffffffff, 0);
      }

      /* Interpose */
      bb = kings;
      pinned = get_pinned_pieces(board, side_to_move);
      while (!bb.is_empty()) {
         int king = bb.bitscan();
         bb.reset(king);

         bitboard_t<kind> bp = attacker;
         while (!bp.is_empty()) {
            int square = bp.bitscan();
            bp.reset(square);

            bitboard_t<kind> destination = bitboard_t<kind>::board_between[king][square] & ~occ;
            destest |= destination;
            if (!destination.is_empty())
               generate_moves_mask_wrap(movelist, board, origin^(kings | pinned), destination, side_to_move, 0xffffffff, 0xffffffff);
         }
      }

      /* Promotion to king, if promotion to king is allowed */
      for (int n = 0; n<board->piece_types->num_piece_types; n++) {
         if (!(board->piece_types->piece_promotion_choice[n] &
            board->piece_types->royal_pieces))
            continue;

         if ((board->bbp[n] & board->bbc[side_to_move]).is_empty())
            continue;

         bitboard_t<kind> from_mask = board->bbp[n]&(~pinned);
         bitboard_t<kind> to_mask   = destination & (~destest | board->piece_types->promotion_zone[side_to_move][n]);

         if (!to_mask.is_empty() && !from_mask.is_empty())
            generate_moves_mask_wrap(movelist, board, from_mask, to_mask, side_to_move, 0xffffffff & board->piece_types->royal_pieces, 0);
      }

      return true;
   }

   /* Add gating moves to the move list.
    * Intended for Seirawan chess.
    */
   void generate_gate_moves(movelist_t *movelist, const board_t<kind> *board, side_t side_to_move) const
   {
      bitboard_t<kind> king = board->royal & board->bbc[side_to_move];
      bitboard_t<kind> rank = bitboard_t<kind>::board_north_edge;
      int n_last, n;

      if (side_to_move == WHITE)
         rank = bitboard_t<kind>::board_south_edge;
      rank &= board->init;

      /* We only care about pieces that have not yet moved */
      if (rank.is_empty())
         return;

      bitboard_t<kind> pinned = get_pinned_pieces(board, side_to_move);
      pinned &= rank;

      if (!pinned.is_empty()) {
         /* Filter out moves of pinned pieces that are not along the same rank:
          * they are illegal because they expose the king to check.
          */
         n_last = movelist->num_moves-1;
         n = 0;
         while (n<movelist->num_moves) {
            move_t move = movelist->move[n];
            bitboard_t<kind> from = bitboard_t<kind>::square_bitboards[get_move_from(move)];
            bitboard_t<kind> to   = bitboard_t<kind>::square_bitboards[get_move_to(move)];
            if (!(pinned & from).is_empty() && (rank & to).is_empty()) {
               movelist->move[n] = movelist->move[n_last];
               movelist->move[n_last] = move;
               movelist->num_moves--;
               n_last--;
            }
            n++;
         }
      }

      /* Go through the move list and add appropriate gating moves */
      n_last = movelist->num_moves;
      for (n=0; n<n_last; n++) {
         move_t base = movelist->move[n];
         int from = get_move_from(base);
         bitboard_t<kind> bb_from = bitboard_t<kind>::square_bitboards[from];

         if (!(bb_from & rank).is_empty()) {
            for (int n=0; n<board->piece_types->num_piece_types; n++) {
               if (board->holdings[n][side_to_move] == 0) continue;
               int tpiece = piece_for_side(n, side_to_move);
               movelist->push(add_move_gate(base, tpiece, from) | MOVE_RESET50);
            }
         }

         if (is_castle_move(base)) {
            int from = get_castle_move_from2(base);
            bitboard_t<kind> bb_from = bitboard_t<kind>::square_bitboards[from];

            if (!(bb_from & rank).is_empty()) {
               for (int n=0; n<board->piece_types->num_piece_types; n++) {
                  if (board->holdings[n][side_to_move] == 0) continue;
                  int tpiece = piece_for_side(n, side_to_move);
                  movelist->push(add_move_gate(base, tpiece, from) | MOVE_RESET50);
               }
            }
         }
      }
   }


   void generate_moves(movelist_t *movelist, const board_t<kind> *board, side_t side_to_move, bool quiesc_only = false) const
   {
      bitboard_t<kind> destination = bitboard_t<kind>::board_all;
      bitboard_t<kind> origin = bitboard_t<kind>::board_all;

      /* If we are in check, then only generate moves in/to the area that can be reached by a superpiece standing
       * in the location of the king(s). These will be the only candidates for resolving the check, all other
       * moves will be pruned anyway.
       */
      if (board->check()) {
         if (generate_evasions(movelist, board, side_to_move))
            goto finalise;
         else {
            bitboard_t<kind> royal = board->royal & board->bbc[side_to_move];
            assert(!royal.is_empty());

            destination = generate_super_attacks_for_squares(royal, super);
            quiesc_only = false;
         }
      } else if (quiesc_only) {
         destination = board->bbc[next_side[side_to_move]];
      }

      movelist->num_moves = 0;
      generate_moves_mask_wrap(movelist, board, origin, destination, side_to_move, 0xffffffff, 0xffffffff, quiesc_only);

      if (quiesc_only) {
         for (int n=0; n<board->piece_types->num_piece_types; n++) {
            destination = board->piece_types->promotion_zone[side_to_move][n] & ~board->bbc[next_side[side_to_move]];
            origin      = board->bbp[n] & board->bbc[side_to_move];
            if (destination.is_empty()) continue;
            if (origin.is_empty()) continue;

            generate_moves_mask_wrap(movelist, board, origin, destination, side_to_move, 0xffffffff, 0, quiesc_only);
         }
      }

finalise:
      if ( (board->rule_flags & RF_GATE_DROPS) &&
            !(board->init & board->bbc[side_to_move] & (bitboard_t<kind>::board_south_edge | bitboard_t<kind>::board_north_edge)).is_empty()) {
         generate_gate_moves(movelist, board, side_to_move);
      }
      return;
   }

   void generate_chase_candidates(movelist_t *movelist, const board_t<kind> *board, side_t side_to_move) const
   {
      assert(board->rule_flags & RF_USE_CHASERULE);
      bitboard_t<kind> destination = board->bbc[next_side[side_to_move]];
      bitboard_t<kind> origin = bitboard_t<kind>::board_all;
      bitboard_t<kind> self = bitboard_t<kind>::board_north;
      bitboard_t<kind> other = bitboard_t<kind>::board_north;

      if (side_to_move == BLACK) {
         self = bitboard_t<kind>::board_north;
         other = bitboard_t<kind>::board_south;
      }

      for (int n = 0; n<board->piece_types->num_piece_types; n++) {
         if (board->piece_types->royal_pieces & (1 << n))      origin &= ~board->bbp[n];
         if (board->piece_types->defensive_pieces & (1 << n))  origin &= ~board->bbp[n];
         if (board->piece_types->pawn_pieces & (1 << n))       origin &= ~(board->bbp[n] & self);
         if (board->piece_types->pawn_pieces & (1 << n))       destination &= ~(board->bbp[n] & other);
      }

      movelist->num_moves = 0;
      generate_moves_mask_wrap(movelist, board, origin, destination, side_to_move, 0xffffffff, 0xffffffff);
   }

};

#endif
