/////////////////////////////////////////////////////////////////////////////
// File:        baseobj.h
// Author:      Cesar Mauri Loba (cesar at crea-si dot com)
// Copyright:   (C) 2010 Cesar Mauri Loba - CREA Software Systems
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
/////////////////////////////////////////////////////////////////////////////
#ifndef SPCORE_BASEOBJ_H
#define SPCORE_BASEOBJ_H

#include <assert.h>
#include <boost/detail/atomic_count.hpp>
#include <boost/intrusive_ptr.hpp>
namespace spcore {

/*
	Smart pointers.

	Here we use the instrusive_ptr from boost library which we expose
	under the alias name "SmartPtr"

	For convenience, although is ugly, we use define's to declare
	the alias instead of the following idiom:

	template <typename T>
	struct SmartPtr { typedef boost::intrusive_ptr<T> type; };

	Doing so to use the smart pointer we should write something like:

	SmartPtr<MyType>::type

	which complicates the syntax besides other problems. See:

	http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2003/n1449.pdf

	TODO: implement it using the template alias feature of the new
	C++0x standard when it become more widespread.		
*/	

#define SmartPtr boost::intrusive_ptr
#define smartptr_static_cast boost::static_pointer_cast
#define smartptr_dynamic_cast boost::dynamic_pointer_cast

///**
//Base class for all other classes. It makes sure client code do not use delete
//*/
class IBaseObject {
  protected:
    inline virtual ~IBaseObject();

    inline IBaseObject();


  public:
    inline virtual void AddRef() const;

    inline virtual void Release() const;


  protected:
    // Delete current object
    inline virtual void Destroy() const;


	// Assignement operator.
	IBaseObject& operator=( const IBaseObject& ) {
		// Does nothing. The reference counter should never be copied
		return *this;
	}

  private:
	// Copy constructor. Disabled
	IBaseObject ( const IBaseObject& );

    mutable boost::detail::atomic_count m_refCount;

};
inline IBaseObject::~IBaseObject() {
}

inline IBaseObject::IBaseObject() : m_refCount(1) {
}

inline void IBaseObject::AddRef() const {
  	assert (m_refCount> 0);
  	++m_refCount;
}

inline void IBaseObject::Release() const {
  	if (--m_refCount== 0) Destroy();
}

// Delete current object. Abstract to force implementation
// in subclasses to ensure deletion is carried out within module
inline void IBaseObject::Destroy() const {
  	delete this;
}


} // namespace spcore

namespace boost
{
	inline void intrusive_ptr_add_ref(const ::spcore::IBaseObject * p) {
		p->AddRef();
	}
	inline void intrusive_ptr_release(const ::spcore::IBaseObject * p) {
		p->Release();
	}
};
#endif
