/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <libconfig.h>
#include <assert.h>
#include <math.h>
#include "siril.h"
#include "proto.h"
#include "conversion.h"
#include "callbacks.h"
#include "single_image.h"

static const char *keywords[] = {
	"working-directory",	// WD
	"camera-raw",			// CR
	"libraw-settings",		// LIBRAW
	"prepro-formula",		// Preprocessing formula used
	"registration-method"	// REG
};
enum token_index { WD = 0, CR = 1, LIBRAW = 2, PREPRO = 3, REG = 4, NOTOK };

int round_to_int(double x) {
	assert(x >= INT_MIN-0.5);
	assert(x <= INT_MAX+0.5);
	if (x >= 0)
		return (int) (x+0.5);
	return (int) (x-0.5);
}

WORD round_to_WORD(double x) {
	if (x <= 0)
		return (WORD)0;
	if (x > USHRT_MAX_DOUBLE)
		return USHRT_MAX;
	return (WORD)(x+0.5);
}

BYTE round_to_BYTE(double x) {
	if (x <= 0)
		return (BYTE)0;
	if (x > UCHAR_MAX_DOUBLE)
		return UCHAR_MAX;
	return (BYTE)(x+0.5);
}


/* returns TRUE if fit has 3 layers */
gboolean isrgb(fits *fit){
	return (fit->naxis == 3);
}

/* returns TRUE if str ends with ending, case insensitive */
gboolean ends_with(const char *str, const char *ending) {
	if (!str || str[0] == '\0') return FALSE;
	if (!ending || ending[0] == '\0') return TRUE;
	int ending_len = strlen(ending);
	int str_len = strlen(str);
	if (ending_len > str_len) return FALSE;
	return !strncasecmp(str+str_len-ending_len, ending, ending_len);
}

/* searches for an extension '.something' in filename from the end, and returns
 * the index of the first '.' found */
int get_extension_index(const char *filename) {
	int i;
	if (filename == NULL || filename[0] == '\0') return -1;
	i = strlen(filename)-1;
	do {
		if (filename[i] == '.')
			return i;
		i--;
	} while (i > 0);
	return -1;
}

const char *get_filename_ext(const char *filename) {
    const char *dot = strrchr(filename, '.');
    if(!dot || dot == filename) return "";
    return dot + 1;
}

// tests whether the given file is either regular or a symlink
// Return value is 1 if file is readable (not actually opened to verify)
int is_readable_file(const char *filename) {
	struct stat sts;
	if (stat(filename, &sts)) return 0;
	if (S_ISREG (sts.st_mode) || S_ISLNK(sts.st_mode)) return 1;
	return 0;
}

/* Tests if filename is the canonical name of a known file type
 * `type' is set according to the result of the test,
 * `realname' (optionnal) is set according to the found file name.
 * If filename contains an extension, only this file name is tested, else all
 * extensions are tested for the file name. */
int stat_file(const char *filename2, image_type *type, char *realname){
	/* FIXME: these variables should not be size-limiter, and realname
	 * should be a char ** to be allocated in the function */
	char basename[256], filename[256], extension[6];
	int separator_index, len, i, extension_len;
	*type=TYPEUNDEF;	// default value

	/* check for an extension in filename and isolate it, including the . */
	if (!filename2 || filename2[0] == '\0') return 1;
	len = strlen(filename2);
	separator_index = get_extension_index(filename2);
	extension_len = len - separator_index;
	if (separator_index == -1 || extension_len < 4 || extension_len > 5) {
		strncpy(basename, filename2, 255);
		basename[255] = '\0';
		extension[0] = '\0';
	} else {
		strncpy(basename, filename2, separator_index);
		basename[separator_index] = '\0';
		strncpy(extension, filename2+separator_index, extension_len);
		extension[extension_len] = '\0';
	}
	/* if filename2 had an extension, we only test for it */
	if (extension[0] != '\0') {
		if (is_readable_file(filename2)){
			if (realname) strcpy(realname, filename2);
			*type = get_type_for_extension_name(extension);
			return 0;
		}
		return 1;
	}

	/* else, we can test various file extensions */
	/* first we test lowercase, then uppercase */
	int k;
	for (k=0; k<2; k++) {
		i = 0;
		while (supported_extensions[i]) {
			char *str = strdup(supported_extensions[i]);
			if (k==1) {
				char *newstr = convtoupper(str);
				free(str);
				str = newstr;
			}
			snprintf(filename, 255, "%s%s", basename, str);
			if (str) free(str);
			if (is_readable_file(filename)){
				*type = get_type_for_extension_name(supported_extensions[i]);
				if (realname) strcpy(realname, filename);
				return 0;
			}
			i++;
		}
	}
	return 1;
}

/******************************** CONFIGURATION FILE **********************************
 *                       We use libconf to generate this file                         *
 * ************************************************************************************/

int readinitfile(){
	config_t config;
	const char *str;
	
	if (!com.initfile) return 1;
	
	config_init(&config);
	
	if (! config_read_file(&config, com.initfile)) return 1;
	siril_log_message("Loading init file: %s\n", com.initfile);
	
	/* Working directory */
	if (config_lookup_string(&config, keywords[WD], &str)) {
		if (changedir(str)) {
			siril_log_message("Reverting current working directory to startup directory, the saved directory is not available anymore\n");
			set_GUI_CWD();
			return(writeinitfile());
		}
	}
	
	/* Camera-raw setting */
	if (config_lookup_string(&config, keywords[CR], &str)) {
		if (strlen(str) < 4) {
			strcpy(com.raw, str);
			initialize_default_extension();
		}
	}
	
	/* Libraw setting */
	config_setting_t *setting = config_lookup(&config, keywords[LIBRAW]);
	
	if (setting){
		config_setting_lookup_bool(setting, "cfa", &com.raw_set.cfa);
		int pattern, inter;
		config_setting_lookup_int(setting, "pattern", &pattern);
		com.raw_set.bayer_pattern = pattern;
		config_setting_lookup_int(setting, "inter", &inter);
		com.raw_set.bayer_inter = inter;
		config_setting_lookup_float(setting, "mul_0", &com.raw_set.mul[0]);
		config_setting_lookup_float(setting, "mul_2", &com.raw_set.mul[2]);
		config_setting_lookup_float(setting, "bright", &com.raw_set.bright);
		config_setting_lookup_int(setting, "auto", &com.raw_set.auto_mul);
		config_setting_lookup_int(setting, "cam_wb", &com.raw_set.use_camera_wb);
		config_setting_lookup_int(setting, "auto_wb", &com.raw_set.use_auto_wb);
		config_setting_lookup_int(setting, "user_qual", &com.raw_set.user_qual);
		config_setting_lookup_float(setting, "gamm_0", &com.raw_set.gamm[0]);
		config_setting_lookup_float(setting, "gamm_1", &com.raw_set.gamm[1]);
		config_setting_lookup_int(setting, "user_black", &com.raw_set.user_black);
	}
	
	/* Prepro setting */
	config_lookup_int(&config, keywords[PREPRO], &com.preproformula);
	
	/* Registration setting */
	config_lookup_int(&config, keywords[REG], &com.reg_methods);
				
	return 0;
}

int writeinitfile(){
	config_t config;
	config_setting_t *root, *setting, *libraw_group;
 	
	/* We test if the file exists. If not we create it
	 * ----> Libconf don't do that(???!!!) */
	FILE *initfile;
	initfile = fopen(com.initfile, "w+");
	if (initfile==NULL) {
		perror("opening init file");
		return 1;
	}
	fclose(initfile);
	/* ******************************* */
	
	config_init(&config);
	if (! config_read_file(&config, com.initfile)) {
		fprintf(stderr, "%s:%d - %s\n", config_error_file(&config), config_error_line(&config),
			config_error_text(&config));
		config_destroy(&config);
		return 1;
	}
	root = config_root_setting(&config);
	setting = config_setting_get_member(root, keywords[WD]);
	if (!setting)
		setting = config_setting_add(root, keywords[WD], CONFIG_TYPE_STRING);
	config_setting_set_string(setting, com.wd);

	setting = config_setting_get_member(root, keywords[CR]);
	if (!setting)
		setting = config_setting_add(root, keywords[CR], CONFIG_TYPE_STRING);
	config_setting_set_string(setting, com.raw);
	
	/* Libraw settings */
	config_setting_remove(root, keywords[LIBRAW]);	// Need to remove before to re-create: if not, config_setting_add returns NULL
	libraw_group = config_setting_add(root, keywords[LIBRAW], CONFIG_TYPE_GROUP);
	
	setting = config_setting_add(libraw_group, "cfa", CONFIG_TYPE_BOOL);
	config_setting_set_bool(setting, com.raw_set.cfa);
	setting = config_setting_add(libraw_group, "pattern", CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.raw_set.bayer_pattern);
	setting = config_setting_add(libraw_group, "inter", CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.raw_set.bayer_inter);
	setting = config_setting_add(libraw_group, "mul_0", CONFIG_TYPE_FLOAT);
	config_setting_set_float(setting, com.raw_set.mul[0]);
	setting = config_setting_add(libraw_group, "mul_2", CONFIG_TYPE_FLOAT);
	config_setting_set_float(setting, com.raw_set.mul[2]);
	setting = config_setting_add(libraw_group, "bright", CONFIG_TYPE_FLOAT);
	config_setting_set_float(setting, com.raw_set.bright);
	setting = config_setting_add(libraw_group, "auto", CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.raw_set.auto_mul);
	setting = config_setting_add(libraw_group, "cam_wb", CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.raw_set.use_camera_wb);
	setting = config_setting_add(libraw_group, "auto_wb", CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.raw_set.use_auto_wb);
	setting = config_setting_add(libraw_group, "user_qual", CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.raw_set.user_qual);
	setting = config_setting_add(libraw_group, "gamm_0", CONFIG_TYPE_FLOAT);
	config_setting_set_float(setting, com.raw_set.gamm[0]);
	setting = config_setting_add(libraw_group, "gamm_1", CONFIG_TYPE_FLOAT);
	config_setting_set_float(setting, com.raw_set.gamm[1]);
	setting = config_setting_add(libraw_group, "user_black", CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.raw_set.user_black);

	setting = config_setting_get_member(root, keywords[PREPRO]);
	if (!setting)
		setting = config_setting_add(root, keywords[PREPRO], CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.preproformula);
	
	setting = config_setting_get_member(root, keywords[REG]);
	if (!setting)
		setting = config_setting_add(root, keywords[REG], CONFIG_TYPE_INT);
	config_setting_set_int(setting, com.reg_methods);

	if (!config_write_file(&config, com.initfile)) {
		fprintf(stderr, "Error while writing file.\n");
		config_destroy(&config);
		return 1;
	}
	config_destroy(&config);
	
	return 0;
}

int checkinitfile(){
	char *home;
	struct stat sts;
	char filename[255];

	// try to read the file given on command line
	if (com.initfile && !readinitfile()) {
		return 0;
	}

	// no file given on command line, set initfile to default location
	if ((home = getenv("HOME")) == NULL) {
		fprintf(stderr, "Could not get the environment variable $HOME, no config file.\n");
		return 1;
	}
	com.initfile = malloc(strlen(home) + 20);
	sprintf(com.initfile, "%s/.siril/siril.cfg", home);
	if(readinitfile()) {
		set_GUI_CWD();
		// if that fails, check and create the default ini file
		snprintf(filename, 255, "%s/.siril", home);
		if (stat (filename, &sts) != 0) {
			if(errno == ENOENT){
				if (mkdir(filename, 0755)){
					fprintf(stderr,"Could not create dir %s, please check\n", filename);
					return 1;
				}
				return (writeinitfile());
			}
		}

		if (!(S_ISDIR (sts.st_mode))){
			fprintf(stderr,"There is a file named %s, that is not a directory.\
					Remove or rename it first\n", filename);
			exit (1);
		}
		return (writeinitfile());
	}
	return 0;
}
/***************************************************************************************/

/* Try to change the CWD to the argument, absolute or relative.
 * If success, the new CWD is written to com.wd */
int changedir(const char *dir){
	char str[256];
	if (dir == NULL || dir[0] == '\0') return 1;
	if (!chdir(dir)) {
		/* do we need to search for sequences in the directory now? We still need to
		 * press the check seq button to display the list, and this is also done there. */
		/* check_seq();
		update_sequence_list();*/
		if (dir[0] == '/') {
			if (com.wd) free(com.wd);
			com.wd = strdup(dir);
			if (!com.wd) return 1;
		} else {
			// dir can be a relative path
			com.wd = realloc(com.wd, PATH_MAX);
			if (!com.wd) return 1;
			com.wd = getcwd(com.wd, PATH_MAX);
		}
		siril_log_message("Setting CWD (Current Working Directory) to `%s'\n", com.wd);
		gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(gtk_builder_get_object(builder, "OpenDial")), com.wd);
		set_GUI_CWD();

		snprintf(str, 255, "%s v%s - %s", PACKAGE, VERSION, dir);
		gtk_window_set_title(GTK_WINDOW(gtk_builder_get_object(builder, "main_window")), str);
		return 0;
	}
	siril_log_message("Could not change directory.\n");
	return 1;
}

/* This method populates the sequence combo box with the sequences found in the CWD.
 * If only one sequence is found, or if a sequence whose name matches the
 * possibly NULL argument is found, it is automatically selected, which triggers
 * its loading */
int update_sequences_list(const char *sequence_name_to_select) {
	GtkComboBoxText *seqcombo;
	DIR *dir;
	struct dirent *file;
	char *suf;
	char filename[256];
	int number_of_loaded_sequences = 0;
	int index_of_seq_to_load = -1;

	if((dir = opendir(com.wd)) == NULL){
		fprintf(stderr, "working directory cannot be opened.\n");
		com.wd[0] = '\0';
		return 1;
	}

	// clear the previous list
	seqcombo = GTK_COMBO_BOX_TEXT(gtk_builder_get_object(builder, "comboboxtext1"));
	gtk_combo_box_text_remove_all(seqcombo);

	while ((file=readdir(dir)) != NULL){
		if ((suf=strstr(file->d_name, ".seq")) && strlen(suf)==4){
			sequence *seq = readseqfile(file->d_name);
			if (seq != NULL) {
				if (seq->type == SEQ_REGULAR) {
					strncpy(filename, file->d_name, 255);
				} else if (seq->type == SEQ_SER) {
					int idx = strlen(file->d_name)-4;
					strncpy(filename, file->d_name, idx);
					strcpy(filename+idx, ".ser");
				}
#if defined(HAVE_FFMPEG) || defined(HAVE_FFMS2)
				else if (seq->type == SEQ_AVI) {
					int idx = strlen(file->d_name)-4;
					strncpy(filename, file->d_name, idx);
					strcpy(filename+idx, ".");
					strcpy(filename+idx+1, seq->ext);
				}
#endif
				free_sequence(seq, TRUE);
				gtk_combo_box_text_append_text(seqcombo, filename);
				if (sequence_name_to_select && !strcmp(filename, sequence_name_to_select)) {
					index_of_seq_to_load = number_of_loaded_sequences;
				}
				++number_of_loaded_sequences;
			}
		}
	}
	closedir(dir);
	if (!number_of_loaded_sequences) {
		fprintf(stderr, "No valid sequence found in CWD.\n");
		return -1;
	} else {
		fprintf(stdout, "Loaded %d sequence(s)\n", number_of_loaded_sequences);
	}

	if (number_of_loaded_sequences > 1 && index_of_seq_to_load < 0) {
		//g_signal_handlers_block_by_func(GTK_WIDGET(seqcombo), on_seqproc_entry_changed, NULL);
		gtk_combo_box_popup(GTK_COMBO_BOX(seqcombo));
		//g_signal_handlers_unblock_by_func(GTK_WIDGET(seqcombo), on_seqproc_entry_changed, NULL);
	}
	else if (index_of_seq_to_load >= 0)
		gtk_combo_box_set_active(GTK_COMBO_BOX(seqcombo), index_of_seq_to_load);
	else gtk_combo_box_set_active(GTK_COMBO_BOX(seqcombo), 0);
	return 0;
}

void update_used_memory() {
	unsigned long size,resident,share,text,lib,data,dt;
	const char* statm_path = "/proc/self/statm";
	FILE *f = fopen(statm_path,"r");
	static int page_size_in_k = 0;
	if (page_size_in_k == 0) {
		page_size_in_k = getpagesize()/1024;
	}
	if(!f){
		perror(statm_path);
		return;
	}
	if(7 != fscanf(f,"%ld %ld %ld %ld %ld %ld %ld",
				&size,&resident,&share,&text,&lib,&data,&dt))
	{
		perror(statm_path);
		fclose(f);
		return;
	}
	fclose(f);
	set_GUI_MEM(resident*page_size_in_k);
}

#if 0
/* returns true if the command theli is available */
gboolean theli_is_available() {
	int retval = system("theli > /dev/null");
	if (WIFEXITED(retval))
		return 0 == WEXITSTATUS(retval); // 0 if it's available, 127 if not
	return FALSE;
}
#endif

/* expands the ~ in filenames */
void expand_home_in_filename(char *filename, int size) {
	if (filename[0] == '~' && filename[1] == '\0') strcat(filename, "/");
	int len = strlen(filename);
	if (len < 2) return;		// not very necessary now with the first line
	if (filename[0] == '~' && filename[1] == '/') {
		char *homepath = getenv("HOME");
		int j, homelen = strlen(homepath);
		if (len + homelen > size-1) {
			siril_log_message("file name is too long, not expanding it\n");
			return;
		}
		for (j=len; j>0; j--)		// edit in place
			filename[j+homelen-1] = filename[j];
		// the -1 above is tricky: it's the removal of the ~ character from
		// the original string
		strncpy(filename, homepath, homelen);
	}
}

WORD get_normalized_value(fits *fit) {
	image_find_minmax(fit, 1);			// recomputing is needed for dark, flat & offset fits
	if (fit->maxi <= UCHAR_MAX)
		return UCHAR_MAX;
	return USHRT_MAX;
}

/* This function reads a text file and displays it in the
 * show_data_dialog */
void read_and_show_textfile(char *path, char *title){
	char line[64] = "";
	char txt[1024] = "";
	
	FILE *f = fopen(path, "r");
	if (!f) {
		show_dialog("File not found", "Error", "gtk-dialog-error");
		return;
	}
	while (fgets(line, sizeof(line), f)!=NULL)
		strcat(txt,line);
	show_data_dialog(txt, title);
	fclose(f);
}

/* Exchange the two parameters of the function 
 * Usefull in Dynamic PSF (PSF.c) */
void swap_param(double *a, double *b){
	double tmp;
	tmp=*a;
	*a=*b;
	*b=tmp;
}

// in-place quick sort, of array a of size n
void quicksort_d (double *a, int n) {
	if (n < 2)
		return;
	double p = a[n / 2];
	double *l = a;
	double *r = a + n - 1;
	while (l <= r) {
		if (*l < p) {
			l++;
			continue;
		}
		if (*r > p) {
			r--;
			continue; // we need to check the condition (l <= r) every time we change the value of l or r
		}
		double t = *l;
		*l++ = *r;
		*r-- = t;
	}
	quicksort_d(a, r - a + 1);
	quicksort_d(l, a + n - l);
}

// in-place quick sort, of array a of size n
void quicksort_s (WORD *a, int n) {
	if (n < 2)
		return;
	WORD p = a[n / 2];
	WORD *l = a;
	WORD *r = a + n - 1;
	while (l <= r) {
		if (*l < p) {
			l++;
			continue;
		}
		if (*r > p) {
			r--;
			continue; // we need to check the condition (l <= r) every time we change the value of l or r
		}
		WORD t = *l;
		*l++ = *r;
		*r-- = t;
	}
	quicksort_s(a, r - a + 1);
	quicksort_s(l, a + n - l);
}

double get_median_value_from_sorted_word_data(WORD *data, int size){
	double median = 0;
	
	switch (size%2) {
		case 0:
			median = 0.5*(data[(size/2)-1]+data[size/2]);
		break;
		default:
			median = data[(size-1)/2];
		break;
	}
	return median;
}

/* return the sigma of a set of data. In the same time it computes
 * the mean value */
double get_standard_deviation(WORD *data, int size){
	int i;
	double sigma, mean;
	
	mean = sigma = 0.0;
	if (size < 2) return 0.0;
	for (i=0; i<size; ++i) {
		mean += (double)data[i];
	}
	mean /= size;
	for (i=0; i<size; ++i)
		sigma += pow((double)data[i]-mean, 2);
	sigma /= (size - 1);
	sigma = sqrt(sigma);
	return sigma;
}

/* returns a new string in uppercase */
char *convtoupper(char *str){
	char *newstr, *p;
	p = newstr = strdup(str);
	while(*p){
		*p=toupper(*p);
		++p;
	}
	
	return newstr;
}

/* This function gets full path without filename
 * This is the same function than dirname from liben.h except that the
 * argument is constant */
const char * extract_path(const char *str){
	size_t filelen, pathlen;
	const char *p;
	char *inpfile = NULL;
	
	p = strrchr(str, '/');
	
	if(p == NULL ) {
		p = str;
	} else {
		p--;	// exclude '/' from path.
	}
	
	filelen = strlen(p);
	pathlen = strlen(str);
	inpfile = malloc(pathlen-filelen+1);
	strncpy(inpfile, str, pathlen-filelen+1);
	return inpfile;
}
