/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <math.h>
#include <gsl/gsl_integration.h>
#include <gsl/gsl_sf_erf.h>
#include <gsl/gsl_statistics.h>
#include <fitsio.h>
#include <complex.h>
#include <limits.h>
#include <assert.h>
#include <libgen.h>

#include "siril.h"
#include "proto.h"
#include "callbacks.h"
#include "colors.h"
#include "histogram.h"
#include "single_image.h"
#include "gradient.h"
#include "PSF_list.h"
#include "opencv.h"
#include "Def_Math.h"
#include "Def_Wavelet.h"

/* this file contains all functions for image processing */

int threshlo(fits *fit, int level){
	int i, layer;
	WORD *buf;

	for (layer=0; layer<fit->naxes[2]; ++layer){
		buf = fit->pdata[layer];
		for (i=0;i<fit->rx*fit->ry;++i){
			*buf= max(level, *buf);
			buf++;
		}
	}
	return 0;
}

int threshhi(fits *fit, int level){
	int i, layer;
	WORD *buf;

	for (layer=0; layer<fit->naxes[2]; ++layer){
		buf = fit->pdata[layer];
		for (i=0;i<fit->rx*fit->ry;++i){
			*buf = min(level, *buf);
			buf++;
		}
	}
	return 0;
}

int nozero(fits *fit, int level){
	int i, layer;
	WORD *buf;

	for (layer=0; layer<fit->naxes[2]; ++layer){
		buf = fit->pdata[layer];
		for (i=0;i<fit->rx*fit->ry;++i){
			if(*buf==0)
				*buf=level;
			buf++;
		}
	}
	return 0;
}

/* equivalent to (map simple_operation a), with simple_operation being
 * (lambda (pixel) (oper pixel scalar))
 * oper is a for addition, s for substraction (i for difference) and so on. */
int soper(fits *a, float scalar, char oper){
	WORD *gbuf;
	int i, layer;

	for (layer=0; layer<a->naxes[2]; ++layer){
		gbuf=a->pdata[layer];
		switch (oper){
			case OPER_ADD:
				for (i=0;i<a->rx * a->ry;++i){
					gbuf[i] = gbuf[i] + scalar < USHRT_MAX ? gbuf[i] + scalar : USHRT_MAX;
				}
				break;
			case OPER_SUB:
				for (i=0;i<a->rx * a->ry;++i){
					if (scalar <= *gbuf) {
						gbuf[i] -= scalar;
					} else {
						gbuf[i] = 0;
					}
				}
				break;
			case OPER_MUL:
				for (i=0;i<a->rx * a->ry; ++i){
					gbuf[i] = gbuf[i] * scalar < USHRT_MAX ? gbuf[i] * scalar : USHRT_MAX;
				}
				break;
			case OPER_DIV:
				for (i=0;i<a->rx * a->ry; ++i){
					gbuf[i] /= scalar;
				}
				break;
		}
	}
	return 0;
}

/* applies operation of image a with image b, for all their layers:
 * a = a oper b
 * returns 0 on success */
int imoper(fits *a, fits *b, char oper){
	WORD *gbuf, *buf;
	int i, j, layer;

	if (a->rx != b->rx || a->ry != b->ry){
		siril_log_message("imoper: images don't have the same size (w = %d|%d, h = %d|%d)\n",
				a->rx, b->rx, a->ry, b->ry);
		return 1;
	}
	for (layer=0; layer<a->naxes[2]; ++layer){
		buf = b->pdata[layer];
		gbuf = a->pdata[layer];
		switch (oper){
			case OPER_ADD:
				for (i=0;i<b->rx;++i){
					for (j=0;j<b->ry;++j){
						if ((*buf+*gbuf) > USHRT_MAX){		
							*(gbuf++)=USHRT_MAX;			// avoid clipping
							buf++;	
							}
						else
						*(gbuf++) += *(buf++);
					}
				}
				break;
			case OPER_SUB:
				for (i=0;i<b->rx;++i){
					for (j=0;j<b->ry;++j){
						if (*buf <= *gbuf){
							*(gbuf++) -= *(buf++);
						}
						else {
							*(gbuf++) = 0; buf++;
						}
					}
				}
				break;
			case OPER_MUL:
				for (i=0;i<b->rx;++i){
					for (j=0;j<b->ry;++j){
						*(gbuf++) *= *(buf++);
					}
				}
				break;
			case OPER_DIV:
				for (i=0;i<b->rx;++i){
					for (j=0;j<b->ry;++j){
						if (*buf == 0) {
							*(gbuf++) = USHRT_MAX; buf++;
						} else {
							*(gbuf++) /= *(buf++);
						}
					}
				}
				break;
		}
	}
	return 0;
}

/* This function applies a subtraction but contrary to Sub in imoper
 * it will use int type format (signed 32-bit). Indeed, pixels values of both
 * fits are very close: if WORD is used, many pixels will be out of bounds
 */
int sub_background(fits* image, fits* background, int layer){
	int *pxl_image, *pxl_bkg, min=0.;
	WORD *image_buf	= image->pdata[layer];
	WORD *bkg_buf	= background->pdata[layer];
	int i;
	if ((image->rx)!=(background->rx) || ((image->ry)!=(background->ry))) {
		char *msg=siril_log_message("Images don't have the same size (w = %d|%d, h = %d|%d)\n",
				image->rx, background->rx, image->ry, background->ry);
		show_dialog(msg, "Error", "gtk-dialog-error");
		return 1;
	}
	size_t ndata = image->rx*image->ry;
	
	// First step we convert data, apply the subtraction and search for minimum
	pxl_image	= malloc(sizeof (int) * ndata);
	pxl_bkg		= malloc(sizeof (int) * ndata);
	for (i=0; i < ndata; i++) {
		pxl_image[i]	= (int)image_buf[i];
		pxl_bkg[i]		= (int)bkg_buf[i];
		pxl_image[i]-=pxl_bkg[i];
		min=min(pxl_image[i], min);
	}
	image_buf = image->pdata[layer];
	// Second we apply an offset to the result and re-convert the data
	for (i=0; i < ndata; i++) {
		pxl_image[i]+=abs(min);
		if (pxl_image[i]>USHRT_MAX) image_buf[i]=USHRT_MAX; //avoid clipping
		else image_buf[i] = (WORD)pxl_image[i];
	}

	// We free memory
	free(pxl_image);
	free(pxl_bkg);
	return 0;
}

int addmax(fits *a, fits *b){
	WORD *gbuf[3]={a->pdata[RLAYER],a->pdata[GLAYER],a->pdata[BLAYER]};
	WORD *buf[3] ={b->pdata[RLAYER],b->pdata[GLAYER],b->pdata[BLAYER]};
	gint i, layer;

	if (a->rx != b->rx || a->ry != b->ry || a->naxes[2] != b->naxes[2]){
		siril_log_message("addmax: images don't have the same size (w = %d|%d, h = %d|%d, layers = %d|%d)\n",
				a->rx, b->rx, a->ry, b->ry, a->naxes[2], b->naxes[2]);
		return 1;
	}
	assert(a->naxes[2] <= 3);
	for (layer=0; layer<a->naxes[2]; ++layer){
		for (i=0; i<a->ry*a->rx; ++i){
			if (buf[layer][i] > gbuf[layer][i])
				gbuf[layer][i]=buf[layer][i];
		}
	}
	siril_log_message("addmax was applied\n");
	return 0;
}

int fmul(fits *a, int layer, float coeff){
	WORD *buf;
	int i;
	
	if (coeff < 0.0) return 1;
	buf=a->pdata[layer];
	for (i=0; i<a->rx * a->ry; ++i){
		buf[i] = buf[i] * coeff < USHRT_MAX ? buf[i] * coeff : USHRT_MAX;
	}
	return 0;
}

/* If fdiv is ok, function returns 0. If overflow, fdiv returns 1*/
int fdiv(fits *a, fits *b, float coef){
	WORD *gbuf, *buf;
	int i, layer;
	int retvalue = 0;
	double temp;

	if (a->rx!=b->rx || a->ry!=b->ry || a->naxes[2] != b->naxes[2]){
		fprintf(stderr, "Wrong size or channel count: %d=%d? / %d=%d?\n",
				a->rx,b->rx,a->ry,b->ry);
		return -1;
	}
	for (layer=0; layer<a->naxes[2]; ++layer){
		buf=b->pdata[layer];
		gbuf=a->pdata[layer];
		for (i=0;i<b->rx*b->ry;++i){
			if (buf[i]==0) buf[i]=1;		// avoid division by 0
			temp=((double)coef * ((double)gbuf[i]/(double)buf[i]));
			if (temp > USHRT_MAX_DOUBLE) retvalue=1;
			gbuf[i]=round_to_WORD(temp);
		}
	}
	return retvalue;
}

/* normalized division a/b, stored in a, with max value equal to the original
 * max value of a, for each layer. */
int ndiv(fits *a, fits *b) {
	double *div;
	int layer, i, nb_pixels;
	if (a->rx != b->rx || a->ry != b->ry || a->naxes[2] != b->naxes[2]){
		fprintf(stderr, "Wrong size or channel count: %d=%d? / %d=%d?, %d=%d?\n",
				a->rx, b->rx, a->ry, b->ry, (int)a->naxes[2], (int)b->naxes[2]);
		return 1;
	}
	nb_pixels = a->rx* a->ry;
	div = malloc(nb_pixels * sizeof(double));

	for (layer=0; layer<a->naxes[2]; ++layer) {
		double max = 0, norm;
		for (i = 0; i < nb_pixels; ++i) {
			if (!b->pdata[layer][i]) div[i] = (double)a->pdata[layer][i];
			else div[i] = (double)a->pdata[layer][i] / (double)b->pdata[layer][i];
			max = max(div[i], max);
		}
		norm = max / (double)a->max[layer];
		for (i = 0; i < nb_pixels; ++i) {
			a->pdata[layer][i] = round_to_WORD(div[i] / norm);
		}
	}

	free(div);
	return 0;
}

#ifdef HAVE_OPENCV
int unsharp(fits *fit, double sigma, double amount, gboolean verbose){
	struct timeval t_start, t_end;

	if (sigma <= 0.0) return 1;
	if (verbose) {		
		siril_log_color_message("Unsharp: processing...\n", "red");	
		gettimeofday (&t_start, NULL);
	}
	unsharp_filter(fit, sigma, amount);	

	if (verbose) {		
		gettimeofday (&t_end, NULL);
		show_time(t_start, t_end);
	}
	return 0;
}

#else
double gaussienne(double sigma, int size, double *gauss){
	double s2, s;
	int i,j,n;

	n=size/2;
	s2=sigma*sigma;
	s=(double)0;
	for (i=0;i<size;++i){
		for (j=0;j<size;++j){
			s+=gauss[i*size+j]=exp(-((double)(i-n)*(i-n) + (double)(j-n)*(j-n))/2/s2);
			//~ fprintf(stderr,"%d:%d %f %f \n", i, j, s, gauss[i*size+j]);
		}
	}
	return s;
}

int unsharp(fits *fit, double sigma, double mult, gboolean verbose){
	// if fabs(mult) > 0.01, unsharp computes the unsharp mask
	// else unsharp computes a gaussian filter

	double normalize,g, *gauss/*, *gaussbuf*/;
	WORD *buf, *gbuf;
	int size, ss2, stride, i, j, k, l, layer;
	struct timeval t_start, t_end;

	if (verbose) {		
		siril_log_color_message("Unsharp: processing...\n", "red");	
		gettimeofday (&t_start, NULL);
	}

	//	fprintf(stderr,"gfitrx: %d mult :%f\n", gfit.rx, mult);
	//~ size=(int)(4*sigma);
	size = (int)(2*(((sigma - 0.8) / 0.3) + 1));	// heuristic. Just to be homogeneous with opencv
	if (!(size%2))
		++size;

	ss2=size/2;
	gauss=malloc(size*size*sizeof(double));
	if (!gauss){
		perror("unsharp.c: Alloc error for gauss");
		return -1;
	}
	//	fprintf(stderr,"size: %d sigma:%f\n", size, sigma);

	normalize=gaussienne(sigma, size, gauss);
	//	fprintf(stderr,"gfitrx: %d ss2:%d\n", gfit.rx, ss2);
	wfit[4].data=(WORD *)calloc(1, fit->rx*fit->ry*sizeof(WORD));

	wfit[4].rx=fit->rx;
	wfit[4].ry=fit->ry;
	wfit[4].lo=fit->lo;
	wfit[4].hi=fit->hi;
	stride=wfit[4].rx-size;
	for (layer=0; layer<fit->naxes[2]; ++layer){
		memcpy(wfit[4].data, fit->pdata[layer], fit->rx*fit->ry*sizeof(WORD));

		buf=fit->pdata[layer]+ss2+ss2*fit->rx;
		//	fprintf(stderr,"gfitrx: %d ss2:%d\n", gfit.rx, ss2);
		for (i=ss2;i<fit->ry-ss2;++i){
			for (j=ss2;j<fit->rx-ss2;++j){
				g=(double)0;
				gbuf=wfit[4].data+(i-ss2)*fit->rx+j-ss2;
				//~ gaussbuf=gauss;
				for (k=0;k<size;++k){
					for(l=0;l<size;++l){
						g+=(*gbuf++)*(gauss[k*size+l]);
					}
					gbuf+=stride;
				}
				*(buf++)=g/normalize;
			}
			buf+=ss2+ss2;
		}

		buf=fit->pdata[layer];
		gbuf=wfit[4].data;
		if (fabs(mult)>0.0){
			for (i=0; i<fit->rx * fit->ry; i++){
				//~ double tmp = gbuf[i] + mult * (gbuf[i] - buf[i]);
				double tmp = gbuf[i] * (1.0 + mult) + buf[i] * (- mult);
				if (tmp < 0.0) buf[i] = 0;
				else if (tmp > USHRT_MAX_DOUBLE) buf[i] = USHRT_MAX;
				else buf[i] = (WORD) tmp;
			}
		}
	}
	free(gauss);
	clearfits(&wfit[4]);
	if (verbose) {		
		gettimeofday (&t_end, NULL);
		show_time(t_start, t_end);
	}
	return 0;
}
#endif

// inplace cropping of the image in fit
int crop(fits *fit, rectangle *bounds){
	WORD *from, *to;
	int stridefrom, i, j, layer;
	int newnbdata;
	struct timeval t_start, t_end;
	
	siril_log_color_message("Crop: processing...\n", "red");
	gettimeofday (&t_start, NULL);

	newnbdata = bounds->w * bounds->h;
	for (layer=0; layer<fit->naxes[2]; ++layer){
		from = fit->pdata[layer] + (fit->ry - bounds->y - bounds->h) * fit->rx + bounds->x;
		fit->pdata[layer] = fit->data+layer*newnbdata;
		to = fit->pdata[layer];
		stridefrom = fit->rx - bounds->w;

		for (i=0; i<bounds->h; ++i){
			for (j=0; j<bounds->w; ++j){
				*to++ = *from++;
			}
			from+= stridefrom;
		}
	}
	fit->rx = fit->naxes[0] = bounds->w;
	fit->ry = fit->naxes[1] = bounds->h;

	clear_stars_list();

	gettimeofday (&t_end, NULL);
	show_time(t_start, t_end);
	
	return 0;
}

/* takes the image in gfit, copies it in a temporary fit to shift it, and copy it back into gfit */
/* TODO: it can be done in the same, thus avoiding to allocate, it just needs to care
 * about the sign of sx and sy to avoid data overwriting in the same allocated space. */
int shift(int sx, int sy){
	int x, y, nx, ny, i, ii, layer;
	fits tmpfit;
	copyfits(&(gfit), &tmpfit, CP_ALLOC|CP_FORMAT, 0);
	i=0;
	/* the loop is the same than in composit() */
	for (y=0;y<gfit.ry;++y){
		for (x=0;x<gfit.rx;++x){
			nx=(x-sx);
			ny=(y-sy);
			//printf("x=%d y=%d sx=%d sy=%d i=%d ii=%d\n",x,y,shiftx,shifty,i,ii);
			if (nx >= 0 && nx < gfit.rx && ny >= 0 && ny < gfit.ry){
				ii=ny*gfit.rx+nx;
				//printf("shiftx=%d shifty=%d i=%d ii=%d\n",shiftx,shifty,i,ii);
				if (ii > 0 && ii < gfit.rx * gfit.ry){
					for (layer=0; layer<gfit.naxes[2]; ++layer){
						tmpfit.pdata[layer][i] = gfit.pdata[layer][ii];
					}
				}
			}
			++i;
		}
	}

	for (layer=0; layer<gfit.naxes[2]; ++layer){
		memcpy(gfit.pdata[layer], tmpfit.pdata[layer], gfit.rx * gfit.ry * sizeof(WORD));
	}
	free(tmpfit.data);

	return 0;
}

#if 0
int rshift2(char *genname, char *outname, int number, char *shiftfile){
	int j,shiftx,shifty,count,n;
	char line[256];
	FILE *sf=NULL;

	if (shiftfile!=NULL){
		sf=fopen(shiftfile,"r");
		if (sf==NULL){
			siril_log_message("rshift2: could not open shift file %s\n", shiftfile);
			return 0;
		}
		fgets(line,255,sf);
		while(line[0]=='#'){
			fgets(line,255,sf);
		}
	}
	shiftx=shifty=0;

	for (j=1;j<=number;++j){
		if(sf!=NULL){
			count=sscanf(line,"%d %d %d",&n,&shiftx,&shifty);
			if(count!=3){
				siril_log_message("rshift2: format error in shift file %s\n", shiftfile);
				return 0;
			}
			fgets(line,255,sf);
		}
		else {
			shiftx=0;
			shifty=0;
			n=j;
		}
		fprintf(stderr,"rshift2 %d %d %d\n", n, shiftx,shifty);
		siril_log_message("Processing image %s%d"SIRIL_FITS_EXTENSION, genname,n);

		buildfilename(genname,n);
		if(readfits(com.formname, &(gfit), com.formname)){
			fprintf(stderr,"missed %s com.formanme",com.formname);
			return 1;
		}
		shift(shiftx, shifty);
		buildfilename(outname,n);
		savefits(com.formname,&gfit);
	}
	if(sf!=NULL){
		fclose(sf);
	}
	return 0;
}
#endif

/* code borrowed from Lynkeos */
double quality(fftw_complex *spectrum, unsigned int rx, unsigned int ry,
		unsigned short down, unsigned short up )
{
	unsigned short x, y;
	double q = 0.0;
	unsigned long d2 = down*down, u2 = up*up;
	unsigned long n = 0;
	double lum = (creal(spectrum[0]))/(double)rx/(double)ry;
	short dx , dy;
	long f2;
	fftw_complex s;

	for( y = 0; y < ry; y++ )
	{
		for ( x = 0; x < rx/2 +1; x++ )
		{
			dx=x;
			dy=y;
			if ( dy >= rx/2 )
				dy -= rx;
			f2 = dx*dx + dy*dy;
			if ( f2 > d2 && f2 < u2 )
			{
				s = spectrum[y*(rx/2+1)+x];
				q += sqrt( creal(s) * creal(s) + cimag (s) * cimag (s) );
				n++;
				//fprintf(stderr,"5x %d y %d q %e lum %e f2 %ld d2 %lu u2 %lu\n",
				//x,y,q, lum,f2,d2,u2);
			}
		}
	}

	return( q/lum/(double)n );
}

/* code borrowed from Lynkeos */
/* this function will be removed after validation of the new entropy below */
double old_entropy(WORD *image, unsigned int rx, unsigned int ry ) {
	// Padded width for in place transform
	double e = 0, bmax = 0;
	unsigned long x, y;

	// Compute the quadratic pixel sum
	for( y = 0; y < ry; y++ )
		for( x = 0; x < rx; x++ )
			bmax += image[y*rx+x] * image[y*rx+x];

	bmax = sqrt(bmax);

	// Compute the entropy
	for( y = 0; y < ry; y++ )
	{
		for( x = 0; x < rx; x++ )
		{
			double b = image[y*rx+x]/bmax;
			if ( b > 0.0 )
				e -= b * log(b);
		}
	}
	return( e );
}

/* This entropy function computes the entropy for the image in gfit for its
 * layer 'layer', in the area designated by area which must be non-NULL.
 * An optional stats parameter can be used to provide the background and sigma
 * value, and when it is given, the entropy will only be computed for pixels
 * with values above background + 1 * sigma. It must be NULL otherwise.
 */
double entropy(fits *fit, int layer, rectangle *area, stats *opt_stats) {
	double e = 0.0, bmax = 0.0, threshold = 0.0;
	int i, j, stridebuf;
	WORD *buf;
	stridebuf = fit->rx - area->w;
	if (opt_stats && opt_stats->median >= 0.0 && opt_stats->sigma >= 0.0)
		threshold = opt_stats->median + 1 * opt_stats->sigma;

	// Compute the quadratic pixel sum
	buf = fit->pdata[layer] + (fit->ry - area->y - area->h) * fit->rx + area->x;
	for (i=0; i<area->h; ++i){
		for (j=0; j<area->w; ++j){
			double b = (double)(*buf);
			if (b > threshold)
				bmax += b * b;
			buf++;
		}
	}
	bmax = sqrt(bmax);

	// Compute the entropy
	buf = fit->pdata[layer] + (fit->ry - area->y - area->h) * fit->rx + area->x;
	for (i=0; i<area->h; ++i) {
		for (j=0; j<area->w; ++j) {
			double b = (double)(*buf);
			double bm = b / bmax;
			if (b > threshold /*&& bm > 0.0*/)	/* can't be 0 */
				e -= bm * log(bm);
			buf++;
		}
		buf += stridebuf;
	}

	return e;
}

int loglut(fits *fit, int dir){
	// This function maps fit with a log LUT
	int i, layer;
	gdouble normalisation, temp;
	WORD *buf[3]={fit->pdata[RLAYER],fit->pdata[GLAYER],fit->pdata[BLAYER]};
	assert(fit->naxes[2] <= 3);

	normalisation=USHRT_MAX_DOUBLE / log(USHRT_MAX_DOUBLE);

	for (i = 0; i < fit->ry * fit->rx; i++) {
		for (layer=0; layer<fit->naxes[2]; ++layer){
			temp=buf[layer][i]+1;
			if (dir == LOG)
				buf[layer][i]=normalisation*log(temp);
			else
				buf[layer][i]=exp(temp/normalisation);
		}
	}
	return 0;
}

double contrast(fits* fit, int layer) {
	stats *stat = statistics(fit, layer, &com.selection);
	int i;
	WORD *buf = fit->pdata[layer];
	double contrast = 0.0;
	double mean = stat->mean;
	
	free(stat);
	stat=NULL;
	for (i=0; i<fit->rx*fit->ry; i++)
		contrast += SQR((double)buf[i]-mean);
	contrast /= fit->rx*fit->ry;
	return contrast;
}

int ddp(fits *a,int level, float coeff, float sigma){
	copyfits(a,&wfit[0], CP_ALLOC|CP_COPYA|CP_FORMAT, 0);
	unsharp (&wfit[0], sigma, 0, FALSE);
	soper(&wfit[0], level, OPER_ADD);
	nozero(&wfit[0], 1);
	fdiv(a, &wfit[0], level);
	soper(a, coeff, OPER_MUL);
	return 0;
}

int visu(fits *fit, int low, int high) {
	if (low < 0 || low > USHRT_MAX || high < 1 || high > USHRT_MAX)
		return 1;
	if (single_image_is_loaded() && com.cvport < com.uniq->nb_layers) {
		com.uniq->layers[com.cvport].hi = high;
		com.uniq->layers[com.cvport].lo = low;
	} else if (sequence_is_loaded() && com.cvport < com.seq.nb_layers) {
		com.seq.layers[com.cvport].hi = high;
		com.seq.layers[com.cvport].lo = low;
	}
	else return 1;
	set_cutoff_sliders_values();
	redraw(com.cvport, REMAP_ONLY);
	redraw_previews();
	return 0;
}

/* fill an image or selection with the value 'level' */
int fill(fits *fit, int level, rectangle *arearg){
	WORD *buf;
	int stridebuf, i, j, layer;
	rectangle area;
	
	if (arearg) {
		memcpy(&area, arearg, sizeof(rectangle));
	} else {
		if (com.selection.h && com.selection.w) {
			memcpy(&area, &com.selection, sizeof(rectangle));
		} else {
			area.w = fit->rx; area.h = fit->ry;
			area.x = 0; area.y = 0;
		}		
	}
	for (layer=0; layer<fit->naxes[2]; ++layer){
		buf = fit->pdata[layer] + (fit->ry - area.y - area.h) * fit->rx + area.x;
		stridebuf = fit->rx - area.w;
		for (i=0; i<area.h; ++i){
			for (j=0; j<area.w; ++j){
				*buf++ = level;
			}
			buf+= stridebuf;
		}
	}
	return 0;
}

int off(fits *fit, int level){
	WORD *buf[3]={fit->pdata[RLAYER],fit->pdata[GLAYER],fit->pdata[BLAYER]};
	int i, layer;
	assert(fit->naxes[2] <= 3);
	if (level == 0) return 0;
	if (level < -USHRT_MAX) level = -USHRT_MAX;
	else if (level > USHRT_MAX) level = USHRT_MAX;
	for (i=0; i<fit->rx * fit->ry; ++i) {
		for (layer=0; layer<fit->naxes[2]; ++layer){
			WORD val = buf[layer][i];
			if ((level < 0 && val < -level))
				buf[layer][i] = 0;
			else if (level > 0 && val > USHRT_MAX - level)
				buf[layer][i] = USHRT_MAX;
			else buf[layer][i] = val + level;
		}
	}
	return 0;
}

void mirrorx(fits *fit, gboolean verbose){
	int line, axis, line_size;
	WORD *swapline, *src, *dst;
	struct timeval t_start, t_end;

	if (verbose) {
		siril_log_color_message("Horizontal mirror: processing...\n", "red");
		gettimeofday (&t_start, NULL);
	}
	
	line_size = fit->rx * sizeof(WORD);
	swapline = malloc(line_size);

	for (axis = 0; axis < fit->naxes[2]; axis++) {
		for (line = 0; line < fit->ry/2; line++) {
			src = fit->pdata[axis] + line * fit->rx;
			dst = fit->pdata[axis] + (fit->ry - line - 1) * fit->rx;

			memcpy(swapline, src, line_size);
			memcpy(src, dst, line_size);
			memcpy(dst, swapline, line_size);
		}
	}
	free(swapline);
	if (verbose) {
		gettimeofday (&t_end, NULL);
		show_time(t_start, t_end);
	}
}

void mirrory(fits *fit, gboolean verbose){
	struct timeval t_start, t_end;

	if (verbose) {
		siril_log_color_message("Vertical mirror: processing...\n", "red");
		gettimeofday (&t_start, NULL);
	}

	fits_flip_top_to_bottom(fit);
	fits_rotate_pi(fit);
	
	if (verbose) {
		gettimeofday (&t_end, NULL);
		show_time(t_start, t_end);	
	}
}

/* this method rotates the image 180 degrees, useful after german mount flip.
 * fit->rx, fit->ry, fit->naxes[2] and fit->pdata[*] are required to be assigned correctly */
void fits_rotate_pi(fits *fit) {
	int i, line, axis, line_size;
	WORD *line1, *line2, *src, *dst, swap;

	line_size = fit->rx * sizeof(WORD);
	line1 = malloc(line_size);
	line2 = malloc(line_size);

	for (axis = 0; axis < fit->naxes[2]; axis++) {
		for (line = 0; line < fit->ry/2; line++) {
			src = fit->pdata[axis] + line * fit->rx;
			dst = fit->pdata[axis] + (fit->ry - line - 1) * fit->rx;

			memcpy(line1, src, line_size);
			for (i=0; i<fit->rx/2; i++) {
				swap = line1[i];
				line1[i] = line1[fit->rx-i-1];
				line1[fit->rx-i-1] = swap;
			}
			memcpy(line2, dst, line_size);
			for (i=0; i<fit->rx/2; i++) {
				swap = line2[i];
				line2[i] = line2[fit->rx-i-1];
				line2[fit->rx-i-1] = swap;
			}
			memcpy(src, line2, line_size);
			memcpy(dst, line1, line_size);
		}
		if (fit->ry & 1) {
			/* swap the middle line */
			src = fit->pdata[axis] + line * fit->rx;
			for (i=0; i<fit->rx/2; i++) {
				swap = src[i];
				src[i] = src[fit->rx-i-1];
				src[fit->rx-i-1] = swap;
			}
		}
	}
	free(line1);
	free(line2);
}

/* This function fills the data in the lrgb image with LRGB information from l, r, g and b
 * images. Layers are not aligned, images need to be all of the same size.
 * It may be used in the command line, currently unused. */
int lrgb(fits *l, fits *r, fits *g, fits *b, fits *lrgb){
	//
	// Combines l r g and b components into resulting lrgb
	// We transform each pixel from RGB to HSI,
	// then take I from the luminance l fits and
	// immediately step back to RGB to the working copy
	//
	guint x, y;
	gdouble rr, gg, bb, h, s, i/*, ps3, dps3, qps3, dpi*/;
	gint maxi;
	WORD *pr, *pg, *pb, *dr, *dg, *db, *pl;

	//
	// some stats used to normalize
	//
	image_find_minmax(r, 0);
	image_find_minmax(g, 0);
	image_find_minmax(b, 0);
	maxi = max(r->maxi, max(g->maxi, b->maxi));
	image_find_minmax(l, 0);
	//
	// initialize pointers
	//
	pr=r->data;
	pg=g->data;
	pb=b->data;
	pl=l->data;
	dr=lrgb->pdata[RLAYER];
	dg=lrgb->pdata[GLAYER];
	db=lrgb->pdata[BLAYER];
	//
	// some trigo constants
	// we stick to h in radians, not in degrees
	//
	//dpi=2*M_PI;
	//ps3=M_PI/3;
	//dps3=2*M_PI;
	//dps3=2*M_PI/3;
	//qps3=4*M_PI/3;
	//
	// Main loop
	//
	fprintf(stderr,"HSI->RGB %d %d\n", r->ry, r->rx);
	for (y = 0; y < r->ry; y++) {
		for (x = 0; x < r->rx; x++) {
			//
			// First normalize rgb to [0 1]
			//
			rr=(double)(*pr++)/maxi;
			gg=(double)(*pg++)/maxi;
			bb=(double)(*pb++)/maxi;

			rgb_to_hsl(rr,gg,bb,&h,&s,&i);
			//
			// replace luminance
			//
			i=*pl++/(double)l->maxi;
			//
			// and back to RGB
			hsl_to_rgb(h,s,i,&rr,&gg,&bb);
			//
			// now denormalize and store
			//
			*dr++=(WORD)(rr*maxi);
			*dg++=(WORD)(gg*maxi);
			*db++=(WORD)(bb*maxi);
		}
	}
	return 0;
}

/* In this function, offset has already been subtracted to the flat
 * and the dark. */
int preprocess(fits *brut, fits *offset, fits *dark, fits *flat, float level) {
	struct timeval t_start, t_end;
	GtkWidget *CCD_formula = lookup_widget("formula_button_1");
	gboolean use_ccd_formula = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CCD_formula));
	
	gettimeofday (&t_start, NULL);
	
	if(use_ccd_formula && (com.preprostatus & USE_OFFSET))
		imoper(brut, offset, OPER_SUB);

	if(com.preprostatus & USE_DARK)
		imoper(brut, dark, OPER_SUB);

	if(com.preprostatus & USE_FLAT) {
#ifdef USE_FLAT_AUTOLEVEL		// is it needed anymore ?
		ndiv(brut, flat);
#else
		if (fdiv(brut, flat, level) > 0)
			siril_log_message("Overflow detected, change level value in settings: %0.2lf is too high.\n", level);
#endif
	}
	gettimeofday (&t_end, NULL);
	show_time(t_start, t_end);
	return 0;
}

/* returns 1 on error */
int seqpreprocess(void){
	int i;
	float normalisation = 1.f;
	char source_filename[256], dest_filename[256], msg[256];
	fits *dark, *offset, *flat;
	GtkWidget *autobutton = lookup_widget("checkbutton_auto_evaluate");
	gboolean autolevel = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(autobutton));
	GtkWidget *CCD_formula = lookup_widget("formula_button_1");
	gboolean use_ccd_formula = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CCD_formula));
	
	if (single_image_is_loaded()) {
		dark = com.uniq->dark;
		offset = com.uniq->offset;
		flat = com.uniq->flat;
	}
	else if (sequence_is_loaded()) {
		dark = com.seq.dark;
		offset = com.seq.offset;
		flat = com.seq.flat;
	}
	else return 1;

	if (use_ccd_formula && com.preprostatus & USE_DARK &&
			com.preprostatus & USE_OFFSET) {
		progress_bar_set_text("Substracting offset to dark...");
		imoper(dark, offset, OPER_SUB);
	}

	if (com.preprostatus & USE_OFFSET &&
			com.preprostatus & USE_FLAT) {
		progress_bar_set_text("Substracting offset to flat...");
		imoper(flat, offset, OPER_SUB);
	}
	
	if (com.preprostatus & USE_FLAT) {
		if (autolevel) {
			/* TODO: evaluate the layer to apply but gnerally RLAYER is a good choice.
			* Indeed, if it is image from APN, CFA picture are in black & white */
			stats *stat = statistics(flat, RLAYER, NULL);
			normalisation = stat->mean;
			siril_log_message("Normalisation value auto evaluated: %.2lf\n", normalisation);
			free(stat);
		}
		else {	
			GtkEntry *norm_entry = GTK_ENTRY(gtk_builder_get_object(builder, "entry_flat_norm"));
			normalisation = atof (gtk_entry_get_text (norm_entry));
		}
	}

	if (single_image_is_loaded()) {
		snprintf(msg, 255, "Pre-processing image %s", com.uniq->filename);
		progress_bar_set_text(msg);
		progress_bar_set_percent(0.5);

		preprocess(com.uniq->fit, offset, dark, flat, normalisation);
		
		snprintf(dest_filename, 255, "%s%s", com.uniq->ppprefix, basename(com.uniq->filename));
		dest_filename[255] = '\0';
		snprintf(msg, 255, "Saving image %s", com.uniq->filename);
		progress_bar_set_text(msg);
		savefits(dest_filename, com.uniq->fit);
	}
	else {	// sequence
		for (i=0; i<com.seq.number; i++) {
			seq_get_image_filename(&com.seq, i, source_filename);
			snprintf(msg, 255, "Loading and pre-processing image %d/%d (%s)",
					i+1, com.seq.number, source_filename);
			msg[255] = '\0';
			progress_bar_set_text(msg);
			progress_bar_set_percent((double)(i+1)/(double)com.seq.number);
			if (seq_read_frame(&com.seq, i, &(wfit[4]))) {
				snprintf(msg, 255, "Could not read one of the raw files: %s."
						" Aborting preprocessing.",
						source_filename);
				progress_bar_set_text(msg);
				return 1;
			}
			preprocess(&(wfit[4]), offset, dark, flat, normalisation);
			snprintf(dest_filename, 255, "%s%s", com.seq.ppprefix, source_filename);
			dest_filename[255] = '\0';
			snprintf(msg, 255, "Saving image %d/%d (%s)",
					i+1, com.seq.number, dest_filename);
			progress_bar_set_text(msg);
			savefits(dest_filename, &(wfit[4]));
		}
	}
	return 0;
}

void initialize_preprocessing() {
	char str[17];
	
	sprintf(str, "formula_button_%d", com.preproformula);
	GtkToggleButton *formula_button = GTK_TOGGLE_BUTTON(lookup_widget(str));

	gtk_toggle_button_set_active(formula_button, TRUE);
}

/* computes the background value using the histogram and/or median value.
 * The argument layer can be -1 for automatic setting (= green for RGB) */
double background(fits* fit, int reqlayer, rectangle *selection){
	int layer=RLAYER;
	double bg;
	
	if (reqlayer >= 0) layer = reqlayer;
	else if (isrgb(&gfit)) layer=GLAYER;		//GLAYER is better to evaluate background
	gsl_histogram* histo = computeHisto(fit, layer);// histogram in full image

	bg = gsl_histogram_max_bin(histo);

	gsl_histogram_free(histo);
	stats* stat = statistics(fit, layer, selection);
	if (fabs(bg-stat->median)>(10))	//totaly arbitrary. Allow to see a background at 0 when a planet take plenty of room.
		bg = stat->median;
//	printf("layer = %d\n", layer);
	free(stat);
	stat=NULL;
	return bg;
}

/* Based on Jean-Luc Starck and Fionn Murtagh (1998), Automatic Noise
 * Estimation from the Multiresolution Support, Publications of the 
 * Royal Astronomical Society of the Pacific, vol. 110, pp. 193–199. */
double backgroundnoise(fits* fit, int reqlayer){
	int layer=RLAYER, n, k;
	unsigned int i, ndata;
	double sigma0, sigma, mean;
	WORD *buf;
	
	if (reqlayer >= 0) layer = reqlayer;
	else if (isrgb(&gfit)) layer=GLAYER;		//GLAYER is better to evaluate background
	ndata = fit->rx * fit->ry;
	
	copyfits(fit, &wfit[0], CP_ALLOC|CP_FORMAT|CP_COPYA, 0);
	get_wavelets_plan(&wfit[0], 2, 0);
	
	stats *stat = statistics(&wfit[0], layer, NULL);
	sigma0 = stat->sigma;
	sigma = sigma0;
	mean = stat->mean;
	buf = wfit[0].pdata[layer];
	double *array1 = calloc(ndata, sizeof(double));
	double *array2 = calloc(ndata, sizeof(double));
	double *set = array1, *subset = array2;
	
	for (i=0; i<ndata; i++)
		set[i] = (double)buf[i];
	n = 0;
	do {
		sigma0 = sigma;
		for (i=0, k=0; i<ndata; i++) {
			if (fabs(set[i] - mean) < 3.0 * sigma0) {
				subset[k] = set[i];
				k++;
			}
		}
		sigma = gsl_stats_sd(subset, 1, k);
		set = subset;
        (set == array1) ? (subset = array2) : (subset = array1);
		ndata = k;
		n++;
	} while (fabs(sigma-sigma0)/sigma > 0.0001 && n < 10);
	free(array1);
	free(array2);
	free(stat);
	clearfits(&wfit[0]);
	
	return sigma*2.35482;
}

stats* statistics(fits *fit, int layer, rectangle *selection){
	double sigma, mean=0.0, avgdev=0.0, max=0.0, min=0.0, sum=0.0, median=0.0;
	double nbdata = fit->rx * fit->ry;
	gsl_histogram* histo;
	size_t i;
	int hist_size;
	stats* stat = malloc(sizeof(stats));

	if (selection && selection->h > 0 && selection->w > 0){
		histo = computeHisto_Selection(fit, layer, selection);
		nbdata = selection->h * selection->w;
	}
	else histo = computeHisto(fit, layer);
	hist_size = gsl_histogram_bins(histo);
	
	/* Calcul of sigma */
	sigma=gsl_histogram_sigma(histo);
	
	/* Calcul of the median */
	/* Get the maximum non null element */
	for (i=hist_size-1; i>0; i--){
		if (gsl_histogram_get(histo,i)){
			max=i;
			break;	//we get out of the loop
		}
	}
	/* Get the minimum non null element */
	for (i=0;i<hist_size;i++){
		if (gsl_histogram_get(histo,i)){
			min=i;
			break;	//we get out of the loop
		}
	}
	/* Get the median value */
	for (i=0;i<hist_size;i++){
		sum+=gsl_histogram_get(histo,i);
		if (sum>nbdata/2.){
			median=i;
			break;	//we get out of the loop
		}
	}
	/* Calcul of the Mean and the Average Absolute Deviation from the Median */
	int k;
	for (i=0, k=0; i<hist_size;i++) {
		double pxl = gsl_histogram_get(histo,i);
		mean += pxl * (double) i;
		if (i>0 && i < USHRT_MAX) {				// we reject hot and cold pixels
			avgdev += fabs((double)pxl - (double)median) * (double) i;
			k=k+i;
		}
	}
	mean /= nbdata;
	avgdev /= (double)k;
	gsl_histogram_free(histo);
	switch (layer){
		case 0:
			if (fit->naxes[2]==1) strcpy(stat->layername,"B&W");
			else strcpy(stat->layername,"Red");
			break;
		case 1:
			strcpy(stat->layername,"Green");
			break;
		case 2:
			strcpy(stat->layername,"Blue");
			break;
	}
	stat->mean = mean;
	stat->avgdev = avgdev;
	stat->median = median;
	stat->sigma = sigma;
	stat->min = min;
	stat->max = max;
	return stat;
}

void show_FITS_header(fits *fit){
	if (fit->header)
		show_data_dialog(fit->header, "FITS Header");
}

#ifdef HAVE_OPENCV
/* These functions do not more than resize_gaussian and rotate_image
 * except for console outputs. 
 * Indeed, siril_log_message seems not working in a cpp file */
int verbose_resize_gaussian(fits *image, int toX, int toY, int interpolation) {
	int retvalue;
	char *str_inter;
	struct timeval t_start, t_end;
	
	switch (interpolation) {
	case 0:
		str_inter = strdup("Nearest-Neighbor");
		break;
	default:
	case 1:
		str_inter = strdup("Bilinear");
		break;
	case 2:
		str_inter = strdup("Pixel Area Relation");
		break;
	case 3:
		str_inter = strdup("Bicubic");
		break;
	case 4:
		str_inter = strdup("Lanczos4");
		break;
	}

	siril_log_color_message("Resample (%s interpolation): processing...\n", "red", str_inter);
	gettimeofday (&t_start, NULL);
	
	retvalue = resize_gaussian(&gfit, toX, toY, interpolation);
	
	gettimeofday (&t_end, NULL);
	show_time(t_start, t_end);
	
	return retvalue;
}

int verbose_rotate_image(fits *image, double angle, int interpolation, int cropped) {
	int layer;
	char *str_inter;
	struct timeval t_start, t_end;
	
	switch (interpolation) {
	case 0:
		str_inter = strdup("Nearest-Neighbor");
		break;
	default:
	case 1:
		str_inter = strdup("Bilinear");
		break;
	case 2:
		str_inter = strdup("Pixel Area Relation");
		break;
	case 3:
		str_inter = strdup("Bicubic");
		break;
	case 4:
		str_inter = strdup("Lanczos4");
		break;
	}

	siril_log_color_message("Rotation (%s interpolation, angle=%g): processing...\n", "red", str_inter, angle);
	gettimeofday (&t_start, NULL);
	
	for (layer = 0; layer < com.uniq->nb_layers; layer ++)
		rotate_image(&gfit, angle, layer, interpolation, 1);
	
	gettimeofday (&t_end, NULL);
	show_time(t_start, t_end);
	
	return 0;
}

#endif

/* Algorithm written in order to display the gaussian equalized image : 
 * http://www.ifa.hawaii.edu/users/ishida/gauss_equal.pro */

double gauss_int(double value) {
	return gsl_sf_erf_Q(-value);
}

double bisect_pdf(double p, double up, double low) {
	if (p < 0.0 || p > 1.0) return -1.0;
	double mid = low + (up - low) * p;
	double z=gauss_int(mid);
	int count = 1;
	while ((fabs(up - low) > (mid * 1.0e-6)) && (count < 100)) {
		if (z > p)
			up = mid;
		else
			low = mid;
		mid = (up + low)/2.;
		z = gauss_int(mid);
		count ++;
	}
	return mid;
}

/* The GAUSS_CVF function computes the cutoff value V in a standard 
 * Gaussian (normal) distribution with a mean of 0.0 and a variance of 
 * 1.0 such that the probability that a random variable X is greater 
 * than V is equal to a user-supplied probability P . */
double gauss_cvf(double p){
	double cutoff = 0.0, below, up;
	int adjust;
	
	if (p > 1.0 || p < 0.0) return -1.0;
	if (p==0.0) return 1.0e12;
	if (p==1.0) return -1.0e12;
	
	if (p > 0.5) {
		p = 1.0 - p;
		adjust = 1;
	}
	else adjust = 0;
	below = 0.0;
	up = 1.0;
	
	while (gauss_int(up) < (1.0 - p)) {
		below = up;
		up = 2 * up;
	}
	cutoff = bisect_pdf(1.0 - p, up, below);
	if (adjust) {
		p = 1.0 - p;
		return -cutoff;
	}
	else 
		return cutoff;
}

/* This function computes wavelets with the number of Nbr_Plan and
 * extracts plan "Plan" in fit parameters */

int get_wavelets_plan(fits *fit, int Nbr_Plan, int Plan) {
	char File_Name_Imag[250]="rawdata";
	char File_Name_Transform[256][3];
	int i;
	wave_transf_des Wavelet[3];
	float *Imag = f_vector_alloc (fit->ry*fit->rx);
	
	snprintf(File_Name_Transform[0], 255, "r_%s.wave", File_Name_Imag);
	wavelet_transform_file (Imag, File_Name_Transform[0], TO_PAVE_BSPLINE, Nbr_Plan, fit->pdata[RLAYER]);
	wave_io_read (File_Name_Transform[0], &Wavelet[0]);
 	if (isrgb(fit)){
		snprintf(File_Name_Transform[1], 255, "g_%s.wave", File_Name_Imag);
		wavelet_transform_file (Imag, File_Name_Transform[1], TO_PAVE_BSPLINE, Nbr_Plan, fit->pdata[GLAYER]);
		wave_io_read (File_Name_Transform[1], &Wavelet[1]);
		snprintf(File_Name_Transform[2], 255, "b_%s.wave", File_Name_Imag);
		wavelet_transform_file (Imag, File_Name_Transform[2], TO_PAVE_BSPLINE, Nbr_Plan, fit->pdata[BLAYER]);
		wave_io_read (File_Name_Transform[2], &Wavelet[2]);
	}
   
	int Nl = Wavelet[0].Nbr_Ligne;
	int Nc = Wavelet[0].Nbr_Col;
		
	pave_2d_extract_plan (Wavelet[0].Pave.Data, Imag, Nl, Nc, Plan);
	reget_rawdata (Imag, Nl, Nc, fit->pdata[RLAYER]);
	if (isrgb(&gfit)){
		reget_rawdata (Imag, Nl, Nc, fit->pdata[GLAYER]);
		pave_2d_extract_plan (Wavelet[1].Pave.Data, Imag, Nl, Nc, Plan);
		reget_rawdata (Imag, Nl, Nc, fit->pdata[BLAYER]);
		pave_2d_extract_plan (Wavelet[2].Pave.Data, Imag, Nl, Nc, Plan);
	}

	/* Free */
	free ((char *) Imag);
	int layer = fit->naxes[2];
	for (i=0; i < layer; i++) {
		wave_io_free (&Wavelet[i]);
		remove(File_Name_Transform[i]);
	}
	return 0;
}

double distance_between_2_points (point a, point b) {
	double dx, dy, distance;

	dx = b.x-a.x;
	dy = b.y-a.y;
	distance = sqrt(dx*dx+dy*dy);
		
	return distance;
}

int find_hot_pixels(fits *fit, WORD threshold, char *filename) {
	int x, y, count=0;
	WORD *buf = fit->pdata[RLAYER];
	FILE* cosme_file = NULL;
	strcat(filename, ".lst");
	cosme_file = fopen(filename, "w");
	
	for (y=0; y<fit->ry; y++) {
		for(x=0; x<fit->rx; x++) {
			if (buf[x + y * fit->rx] > threshold) {
				fprintf(cosme_file, "P %d %d\n", x, y);
				count++;
			}
		}
	}
	fclose(cosme_file);
	return count;
}

/* The function smoothes an image using the median filter with the
 * ksize x ksize aperture. Each channel of a multi-channel image is 
 * processed independently. In-place operation is supported. */
int median_filter(fits *fit, int ksize, double modulation, int iterations) {
	assert(ksize%2==1 && ksize > 1);
	int i, x, y, xx, yy, layer, iter=0;
	int nx = fit->rx;
	int ny = fit->ry;
	int radius = (ksize-1)/2;
	double norm = (double)get_normalized_value(fit);
	struct timeval t_start, t_end;

	siril_log_color_message("Median Filter: processing...\n", "red");
	gettimeofday (&t_start, NULL);
	
	do {
		if (iterations != 1) siril_log_message("Iteration #%d...\n", iter+1);	
		for (layer=0; layer<com.uniq->nb_layers; layer++) {
		/* FILL image upside-down */
			WORD **image = malloc(fit->ry*sizeof(WORD *));
			for (i=0; i<fit->ry; i++)
				image[fit->ry-i-1] = fit->pdata[layer] + i*fit->rx;
		
			for (y=0; y<ny; y++){
				for (x=0; x<nx; x++){
					WORD *data = calloc(ksize*ksize, sizeof(WORD));
					i=0;
					for (yy=y-radius; yy<=y+radius; yy++){
						for (xx=x-radius; xx<=x+radius; xx++){
							WORD tmp;
							if (xx<0 && yy>=0) {
								if (yy >= ny) tmp=image[ny-1][00];
								else tmp=image[yy][0];
							}
							else if (xx>0 && yy<=0) {
								if (xx >= nx) tmp=image[00][nx-1];
								else tmp=image[0][xx];
							}
							else if (xx<=0 && yy<=0) {
								tmp=image[0][0];
							}
							else {
								if (xx >= nx && yy >=ny) tmp=image[ny-1][nx-1];
								else if (xx >= nx && yy < ny) tmp=image[yy][nx-1];
								else if (xx < nx && yy >= ny) tmp=image[ny-1][xx];
								else tmp = image[yy][xx];
							}
							data[i++] = tmp;
						}
					}
					quicksort_s(data, ksize*ksize);
					WORD median = round_to_WORD(get_median_value_from_sorted_word_data(data, ksize*ksize));
					double pixel = modulation * (median/norm);
					pixel += (1.0 - modulation) * ((double)image[y][x]/norm);
					image[y][x] = round_to_WORD(pixel * norm);
					free(data);
				}
			}
			free(image);
		}
		iter++;
	} while (iter < iterations);
	gettimeofday (&t_end, NULL);
	show_time(t_start, t_end);
	return 0;
}
