/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#include "siril.h"
#include "proto.h"
#include "callbacks.h"
#include "PSF.h"
#include "registration.h"	// for update_reg_interface
#include "stacking.h"	// for update_stack_interface

static GtkListStore *list_store = NULL;

enum {
	COLUMN_IMNAME,		// string
	COLUMN_SHIFTX,		// int
	COLUMN_SHIFTY,		// int
	COLUMN_SELECTED,	// gboolean
	COLUMN_FWHM,		// converted to string, not pure double
	COLUMN_CURRENT,		// int weight, current file loaded, display IMNAME in bold
	COLUMN_REFERENCE,	// background color depending on the image being reference
	COLUMN_INDEX,		// int
	N_COLUMNS
};

void get_list_store() {
	if (list_store == NULL)
		list_store = GTK_LIST_STORE(gtk_builder_get_object(builder, "liststore1"));
}

void add_image_to_sequence_list(sequence *seq, int index, int layer) {
	static GtkTreeSelection *selection = NULL;
	GtkTreeIter iter;
	char imname[256], fwhm_str[20];
	int shiftx = -1, shifty = -1;

	get_list_store();
	if (!selection)
		selection = GTK_TREE_SELECTION(gtk_builder_get_object(builder, "treeview-selection1"));
	if (seq == NULL) {
		gtk_list_store_clear(list_store);
		return;		// just clear the list
	}
	if (seq->regparam && seq->regparam[layer]) {
		shiftx = seq->regparam[layer][index].shiftx;
		shifty = seq->regparam[layer][index].shifty;

		if (seq->regparam[layer][index].fwhm) {
			// Is it certain that FWHMX is > than FWHMY? The minimization seems to imply this.
			if (seq->regparam[layer][index].fwhm->FWHMX > seq->regparam[layer][index].fwhm->FWHMY)
				sprintf(fwhm_str, "%.3g", seq->regparam[layer][index].fwhm->FWHMX);
			else sprintf(fwhm_str, "%.3g", seq->regparam[layer][index].fwhm->FWHMY);
		} else if (seq->regparam[layer][index].entropy) {
			sprintf(fwhm_str, "%g", seq->regparam[layer][index].entropy);
		} else sprintf(fwhm_str, "N/A");
	} else sprintf(fwhm_str, "N/A");

	gtk_list_store_append (list_store, &iter);
	gtk_list_store_set (list_store, &iter,
			COLUMN_IMNAME, seq_get_image_filename(seq, index, imname),
			COLUMN_SHIFTX, shiftx,
			COLUMN_SHIFTY, shifty,
			COLUMN_SELECTED, seq->imgparam[index].incl,
			COLUMN_FWHM, fwhm_str,
			COLUMN_CURRENT, index == seq->current ? 800 : 400,
			// weight value is 400 by default "normal":
			// http://developer.gnome.org/gtk3/3.5/GtkCellRendererText.html#GtkCellRendererText--weight
			COLUMN_REFERENCE, index == seq->reference_image ? "Beige" : "WhiteSmoke",
			COLUMN_INDEX, index,
			-1);
	/* see example at http://developer.gnome.org/gtk3/3.5/GtkListStore.html */
	if (index == seq->current) {
		gtk_tree_selection_select_iter(selection, &iter);
	}
}

/* called on sequence loading (set_seq) and on layer tab change */
void fill_sequence_list(sequence *seq, int layer) {
	int i;
	if (seq == NULL) return;
	add_image_to_sequence_list(NULL, 0, 0);	// clear  
	if (seq->number == 0) return;
	for (i=0; i<seq->number; i++) {
		add_image_to_sequence_list(seq, i, layer);
	}
}

void show_seqlist(GtkWidget *widget, gboolean show) {
	/*gboolean current_shrink;
	gtk_container_child_get(GTK_CONTAINER(widget),
			gtk_paned_get_child1(GTK_PANED(widget)),
			"shrink", &current_shrink, NULL);
	gtk_container_child_set(GTK_CONTAINER(widget),
			gtk_paned_get_child1(GTK_PANED(widget)),
			"shrink", !current_shrink, NULL);*/
	gtk_paned_set_position(GTK_PANED(widget), show ? 160 : 0);
	/*if (current_shrink) {
	  gtk_paned_set_position(GTK_PANED(widget), 0);
	  gtk_widget_set_size_request(gtk_paned_get_child2(GTK_PANED(widget)), 0, -1);
	  }*/
}

void on_toggle_show_seqlist_toggled(GtkToggleButton *togglebutton, gpointer user_data) {
	static GtkWidget *paned = NULL;
	if (!paned)
		paned = lookup_widget("paned1");
	show_seqlist(paned, gtk_toggle_button_get_active(togglebutton));
}

void on_seqlist_image_selection_toggled(GtkCellRendererToggle *cell_renderer,
		gchar *path, gpointer user_data) {
	gint *index = gtk_tree_path_get_indices(gtk_tree_path_new_from_string(path));
	if (!index) return;
	if (index[0] >= com.seq.number) return;
	fprintf(stdout, "toggle selection index = %d\n", index[0]);
	/*if (gtk_cell_renderer_toggle_get_active(cell_renderer) ==
			com.seq.imgparam[index[0]].incl) {
		fprintf(stdout, "mismatch in selection toggle, what should I do?\n");
		return;
	}*/

	sequence_list_change_selection(path, !com.seq.imgparam[index[0]].incl);
	siril_log_message("%s image %d in sequence %s\n",
			com.seq.imgparam[index[0]].incl ? "excluding" : "including",
			index[0], com.seq.seqname);

	com.seq.imgparam[index[0]].incl = !com.seq.imgparam[index[0]].incl;
	if (com.seq.imgparam[index[0]].incl)
		com.seq.selnum++;
	else 	com.seq.selnum--;
	adjust_exclude(index[0], TRUE);	// check or uncheck excluded checkbox in seq tab
	update_reg_interface(FALSE);
	writeseqfile(&com.seq);
	redraw(com.cvport, REMAP_NONE);
}

/* double click on an image -> open it */
void on_treeview1_row_activated(GtkTreeView *tree_view, GtkTreePath *path,
		GtkTreeViewColumn *column, gpointer user_data) {
	gint *index = gtk_tree_path_get_indices(path);
	if (!index) return;
	if (index[0] >= com.seq.number) return;
	fprintf(stdout, "loading image %d\n", index[0]);
	seq_load_image(&com.seq, index[0], &gfit, TRUE);
}

/****************** modification of the list store (tree model) ******************/

void sequence_list_change_selection(gchar *path, gboolean new_value) {
	GtkTreeIter iter;
	get_list_store();
	gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(list_store), &iter, path);
	gtk_list_store_set(list_store, &iter, COLUMN_SELECTED, new_value, -1);
}

void sequence_list_change_selection_index(int index) {
	sequence_list_change_selection(
			gtk_tree_path_to_string(gtk_tree_path_new_from_indices(index, -1)),
			com.seq.imgparam[index].incl);
}

void sequence_list_change_current() {
	GtkTreeIter iter;
	gboolean valid;
	gint row_count = 0;

	get_list_store();
	valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), &iter);
	while (valid) {
		gtk_list_store_set(list_store, &iter,
				COLUMN_CURRENT, (row_count == com.seq.current) ? 800 : 400,
				-1);
		row_count++;
		valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store), &iter);
	}
}

void sequence_list_change_reference() {
	GtkTreeIter iter;
	gboolean valid;
	gint row_count = 0;

	get_list_store();
	valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), &iter);
	while (valid) {
		gtk_list_store_set(list_store, &iter,
				COLUMN_REFERENCE,
				(row_count == com.seq.reference_image) ? "Beige" : "WhiteSmoke",
				-1);
		row_count++;
		valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store), &iter);
	}
}

void clear_sequence_list() {
	get_list_store();
	gtk_list_store_clear(list_store);
}
