/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/time.h>
#include <dirent.h>
#include <limits.h>
#include <ctype.h>
#include <assert.h>
#include <math.h>
#include <libgen.h>

#include "siril.h"
#include "proto.h"
#include "callbacks.h"
#include "ser.h"
#ifdef HAVE_FFMPEG
#include "avi.h"
#elif defined(HAVE_FFMS2)
#include "films.h"
#endif
#include "single_image.h"
#include "histogram.h"
#include "PSF.h"
#include "PSF_list.h"	// clear_stars_list
#include "registration.h"	// for update_reg_interface
#include "stacking.h"	// for update_stack_interface

/* name is sequence filename, with or without .seq extension */
sequence * readseqfile(const char *name){
	char line[512], *scanformat;
	char filename[512], *seqfilename;
	int i, nbsel;
	FILE *seqfile;
	int allocated=0;
	int current_layer = -1;
	sequence *seq;

	if (!name) return NULL;
	siril_log_message("Reading sequence file `%s'.\n", name);

	if(!ends_with(name, ".seq")){
		seqfilename = malloc(strlen(name) + 6);	/* 6 stands for a max length of 4 + '.' + '\0' */
		sprintf(seqfilename, "%s.seq", name);
	} else {
		seqfilename = strdup(name);
	}

	if ((seqfile = fopen(seqfilename, "r")) == NULL) {
		perror("fopen sequence file");
		siril_log_message("Reading sequence failed, file cannot be opened: %s.\n", seqfilename);
		free(seqfilename);
		return NULL;
	}

	seq = calloc(1, sizeof(sequence));
	initialize_sequence(seq, TRUE);
	i=0;
	while (fgets(line, 511, seqfile)) {
		switch (line[0]) {
			case '#':
				continue;
			case 'S':
				/* The double quote as sequence name is a sequence with no name.
				 * Such sequences don't exist anymore. */
				assert(line[2] != '"');
				if (line[2] == '\'')	/* new format, quoted string */
					scanformat = "'%511[^']' %d %d %d %d %d";
				else scanformat = "%511s %d %d %d %d %d";

				if(sscanf(line+2, scanformat,
							filename, &seq->beg, &seq->number,
							&seq->selnum, &seq->fixed,
							&seq->reference_image) != 6 ||
						allocated != 0){
					fprintf(stderr,"readseqfile: sequence file format error: %s\n",line);
					goto error;
				}
				if (seq->number == 0) {
					fprintf(stderr, "readseqfile: sequence is empty?\n");
					goto error;
				}
				seq->seqname = strdup(filename);
				seq->imgparam = calloc(seq->number, sizeof(imgdata));
				allocated = 1;
				break;

			case 'L':
				/* for now, the L line stores the number of layers for each image. */
				if (line[1] == ' ') {
					if (sscanf(line+2, "%d", &seq->nb_layers) != 1) {
						fprintf(stderr,"readseqfile: sequence file format error: %s\n",line);
						goto error;
					}
					seq->regparam = calloc(seq->nb_layers, sizeof(regdata*));
					seq->layers = calloc(seq->nb_layers, sizeof(layer_info));
				} else if (line[1] >= '0' && line[1] <= '9') {
					/* in the future, wavelength and name of each layer will be added here */
				}
				break;
			case 'I':
				/* previously was the simple image description with no line key */
				if (sscanf(line+2, "%d %d", &(seq->imgparam[i].filenum),
							&(seq->imgparam[i].incl)) != 2) {
					fprintf(stderr,"readseqfile: sequence file format error: %s\n",line);
					goto error;
				}
				++i;
				break;
			case 'R':
				/* registration info */
				current_layer = line[1] - '0';
				if (current_layer < 0 || current_layer > 9) {
					fprintf(stderr,"readseqfile: sequence file format error: %s\n",line);
					goto error;
				}
				if (seq->regparam[current_layer] == NULL) {
					seq->regparam[current_layer] = calloc(seq->number, sizeof(regdata));
					if (seq->regparam[current_layer] == NULL) {
						fprintf(stderr, "readseqfile: could not allocate registration data\n");
						goto error;
					}
					i = 0;
				}
				if (i >= seq->number) {
					fprintf(stderr, "\nreadseqfile ERROR: out of array bounds in reg info!\n\n");
				} else {
					if (sscanf(line+3,"%d %d %lg",
								&(seq->regparam[current_layer][i].shiftx),
								&(seq->regparam[current_layer][i].shifty),
								&(seq->regparam[current_layer][i].quality)) != 3) {
						fprintf(stderr,"readseqfile: sequence file format error: %s\n",line);
						goto error;
					}
					++i;
				}
				break;
			case 'T':
				/* sequence type (several files or a single file) */
				if (line[1] == 'S') {
					seq->type = SEQ_SER;
#ifdef HAVE_FFMS2
					seq->ext = "ser";
#endif
					if (seq->ser_file) break;
					seq->ser_file = malloc(sizeof(struct ser_struct));
					ser_init_struct(seq->ser_file);
					seqfilename[strlen(seqfilename)-1] = 'r';
					if (ser_open_file(seqfilename, seq->ser_file)) {
						free(seq->ser_file);
						seq->ser_file = NULL;
						goto error;
					}
					else ser_display_info(seq->ser_file);
				}
#if defined(HAVE_FFMPEG)
				else if (!strncasecmp(file->d_name + fnlen - 4, ".avi", 4)) {
					struct avi_struct *avi_file = malloc(sizeof(struct avi_struct));
					if (avi_open_file(file->d_name, avi_file))
						continue;
					new_seq = calloc(1, sizeof(sequence));
					initialize_sequence(new_seq, TRUE);
					new_seq->seqname = strndup(file->d_name, fnlen-4);
					new_seq->beg = 0;
					new_seq->end = avi_file->frame_count-1;
					new_seq->number = avi_file->frame_count;
					new_seq->type = SEQ_AVI;
					new_seq->avi_file = avi_file;
					new_seq->ext = "avi";
					sequences[nb_seq] = new_seq;
					nb_seq++;
					fprintf(stdout, "Found a AVI sequence (number %d)\n", nb_seq);
					progress_bar_set_percent(-1.0);	// pulse
				}
#elif defined(HAVE_FFMS2)
				else if (line[1] == 'A') {
					seq->type = SEQ_AVI;
					if (seq->film_file) break;
					seq->film_file = malloc(sizeof(struct film_struct));
					int i = 0, nb_film = get_nb_film_ext_supported();
					char *backup_name = strdup(seqfilename);
					while (i < nb_film) {
						int len_ext = strlen(supported_film[i].extension);
						/* test for extension in lowercase */
						strncpy(seqfilename + strlen(seqfilename)-3, supported_film[i].extension, len_ext);
						if (access(seqfilename, F_OK) != -1) break;
						else {
							/* reinitialize seqfilename if no match: need to do it because of extensions with length of 4 */
							strcpy(seqfilename, backup_name);
							/* test for extension in uppercase */
							strncpy(seqfilename + strlen(seqfilename)-3, convtoupper(supported_film[i].extension), len_ext);
							if (access(seqfilename, F_OK) != -1) break;
							/* reinitialize seqfilename if no match: need to do it because of extensions with length of 4 */
							strcpy(seqfilename, backup_name);
						}
						i++;
					}
					free(backup_name);
					if (film_open_file(seqfilename, seq->film_file)) {
						free(seq->film_file);
						seq->film_file = NULL;
						goto error;
					}
					else {
						film_display_info(seq->film_file);
						seq->ext = strdup(get_filename_ext(seq->film_file->filename));
					}
				}
				else seq->ext = "fit";
#endif
				break;
		}
	}
	fclose(seqfile);
	seq->end = seq->imgparam[seq->number-1].filenum;
	seq->current = -1;
	for (i=0, nbsel=0; i<seq->number; i++)
		if (seq->imgparam[i].incl)
			nbsel++;
	if (nbsel != seq->selnum) {
		siril_log_message("fixing the selection number in the .seq file (%d) to the actual value (%d) (not saved)\n", seq->selnum, nbsel);
		seq->selnum = nbsel;
	}
	update_used_memory();
	free(seqfilename);
	return seq;
error:
	fclose(seqfile);
	if (seq->seqname)
		free(seq->seqname);
	free(seq);
	siril_log_message("Could not load sequence\n");
	update_used_memory();
	free(seqfilename);
	return NULL;
}

int read_single_sequence(char *realname, int imagetype) {
	int retval=3;		// needs to return 3 if ok !!!
#ifdef HAVE_FFMS2
	const char *ext;
#endif
	char *name = strdup(realname);
	
	if (!changedir(extract_path(realname))) 
		writeinitfile();
	if (check_only_one_film_seq(realname)) retval = 1;
	else {
		switch (imagetype) {
			case TYPESER:
				name[strlen(name)-1] = 'q';
				break;
#ifdef HAVE_FFMS2
			case TYPEAVI:
				ext = get_filename_ext(realname);
				int len = strlen(ext);
				strncpy(name+strlen(name)-len, "seq", len);
				break;
#endif
			default:
				retval = 1;
		}
		if (!set_seq(basename(name))) {
			control_window_switch_to_tab(IMAGE_SEQ);
			GtkComboBoxText *combo_box_text = GTK_COMBO_BOX_TEXT(lookup_widget("comboboxtext1"));
			gtk_combo_box_text_remove_all(combo_box_text);
			gtk_combo_box_text_append(combo_box_text, 0, basename(realname));
			g_signal_handlers_block_by_func(GTK_COMBO_BOX(combo_box_text), on_seqproc_entry_changed, NULL);
			gtk_combo_box_set_active(GTK_COMBO_BOX(combo_box_text), 0);
			g_signal_handlers_unblock_by_func(GTK_COMBO_BOX(combo_box_text), on_seqproc_entry_changed, NULL);
		}
		else retval = 1;
	}
	free(name);
	return retval;
}

/* sets the maximum value for the spin button and display the initial file name */
int seqsetnum(int image_number) {
	GtkSpinButton *spin;
	GtkAdjustment *adj;
	if (com.seq.number <= 0 || image_number >= com.seq.number) return 1;
	spin = GTK_SPIN_BUTTON(gtk_builder_get_object(builder, "imagenumber_spin"));
	adj = gtk_spin_button_get_adjustment(spin);

	gtk_adjustment_set_upper(adj, (gdouble)com.seq.number-1);
	gtk_adjustment_set_value(adj, (gdouble)image_number);	// 0 is default
	display_image_number(image_number);
	//on_imagenumberspin_output(GTK_SPIN_BUTTON(spin), NULL);	// redraw the real file number instead of 0
	return 0;
}

int check_seq() {
	// in the current working directory, looks for sequences of fits files
	// and builds the corresponding sequence files
	// called when changing wd with name==NULL
	// 	or when an explicit root name is given in the GUI
	//	or when searching for sequences
	char *basename;
	int curidx, retval = 1, fixed;
	DIR *dir;
	struct dirent *file;
	sequence *sequences[40];
	int i, nb_seq = 0;

	if (!com.wd) {
		siril_log_message("Current working directory is not set, aborting.\n");
		return 1;
	}
	if ((dir = opendir(com.wd)) == NULL) {
		fprintf(stderr, "working directory cannot be opened.\n");
		free(com.wd);
		com.wd = NULL;
		return 1;
	}

	while ((file = readdir(dir)) != NULL) {
		sequence *new_seq;
		int fnlen = strlen(file->d_name);
		if (!strncasecmp(file->d_name + fnlen - 4, ".ser", 4)) {
			struct ser_struct *ser_file = malloc(sizeof(struct ser_struct));
			ser_init_struct(ser_file);
			if (ser_open_file(file->d_name, ser_file))
				continue;
			new_seq = calloc(1, sizeof(sequence));
			initialize_sequence(new_seq, TRUE);
			new_seq->seqname = strndup(file->d_name, fnlen-4);
			new_seq->beg = 0;
			new_seq->end = ser_file->frame_count-1;
			new_seq->number = ser_file->frame_count;
			new_seq->type = SEQ_SER;
			new_seq->ser_file = ser_file;
			sequences[nb_seq] = new_seq;
			nb_seq++;
			fprintf(stdout, "Found a SER sequence (number %d)\n", nb_seq);
			progress_bar_set_percent(-1.0);	// pulse
		}
#ifdef HAVE_FFMPEG
		else if (!strncasecmp(file->d_name + fnlen - 4, ".avi", 4)) {
			struct avi_struct *avi_file = malloc(sizeof(struct avi_struct));
			if (avi_open_file(file->d_name, avi_file))
				continue;
			new_seq = calloc(1, sizeof(sequence));
			initialize_sequence(new_seq, TRUE);
			new_seq->seqname = strndup(file->d_name, fnlen-4);
			new_seq->beg = 0;
			new_seq->end = avi_file->frame_count-1;
			new_seq->number = avi_file->frame_count;
			new_seq->type = SEQ_AVI;
			new_seq->avi_file = avi_file;
			sequences[nb_seq] = new_seq;
			nb_seq++;
			fprintf(stdout, "Found a AVI sequence (number %d)\n", nb_seq);
			progress_bar_set_percent(-1.0);	// pulse
		}
#elif defined HAVE_FFMS2
		else if (!check_for_film_extensions(get_filename_ext(file->d_name))) {
			struct film_struct *film_file = malloc(sizeof(struct film_struct));
			if (film_open_file(file->d_name, film_file)) {
				free(film_file);
				continue;
			}
			new_seq = calloc(1, sizeof(sequence));
			initialize_sequence(new_seq, TRUE);
			int len = strlen(get_filename_ext(file->d_name));
			new_seq->seqname = strndup(file->d_name, fnlen-(len+1));
			new_seq->beg = 0;
			new_seq->end = film_file->frame_count-1;
			new_seq->number = film_file->frame_count;
			new_seq->type = SEQ_AVI;
			new_seq->film_file = film_file;
			sequences[nb_seq] = new_seq;
			nb_seq++;
			fprintf(stdout, "Found a AVI sequence (number %d)\n", nb_seq);
			progress_bar_set_percent(-1.0);	// pulse
		}
#endif

		else if (!strncmp(file->d_name + fnlen - SIRIL_FITS_EXT_LEN,
					SIRIL_FITS_EXTENSION,
					SIRIL_FITS_EXT_LEN)) {
			if (!get_index_and_basename(file->d_name, &basename, &curidx, &fixed)) {
				int current_seq = -1;
				/* search in known sequences if we already have it */
				for (i=0; i<nb_seq; i++) {
					if (!strcmp(sequences[i]->seqname, basename)) {
						current_seq = i;
					}
				}
				/* not found */
				if (current_seq == -1) {
					if (nb_seq == 40) {
						fprintf(stderr, "too many sequences\n");
						continue;
					}
					new_seq = calloc(1, sizeof(sequence));
					initialize_sequence(new_seq, TRUE);
					new_seq->seqname = basename;
					new_seq->beg = INT_MAX;
					new_seq->end = 0;
					new_seq->fixed = fixed;
					sequences[nb_seq] = new_seq;
					current_seq = nb_seq;
					nb_seq++;
					fprintf(stdout, "Found a sequence (number %d) with base name"
							" \"%s\", looking for first and last indexes.\n",
							nb_seq, basename);
					progress_bar_set_percent(-1.0);	// pulse
				}
				if (curidx < sequences[current_seq]->beg)
					sequences[current_seq]->beg = curidx;
				if (curidx > sequences[current_seq]->end)
					sequences[current_seq]->end = curidx;
			}
		}
	}
	closedir(dir);
	if (nb_seq > 0) {
		for (i=0; i<nb_seq; i++) {
			if (sequences[i]->beg != sequences[i]->end) {
				char msg[200];
				sprintf(msg, "sequence %d, found: %d to %d",
						i+1, sequences[i]->beg, sequences[i]->end);
				progress_bar_set_text(msg);
				if (!buildseqfile(sequences[i]) && retval)
					retval = 0;	// at least one succeeded to be created
			}
			free_sequence(sequences[i], TRUE);
		}
		return retval;
	}
	return 1;	// no sequence found
}

/* Check for on film sequence of the name passed in arguement
 * Returns 0 if OK */
int check_only_one_film_seq(char* name) {
	int retval = 1;
	DIR *dir;
	sequence *new_seq = NULL;

	if (!com.wd) {
		siril_log_message("Current working directory is not set, aborting.\n");
		return 1;
	}
	if ((dir = opendir(com.wd)) == NULL) {
		fprintf(stderr, "working directory cannot be opened.\n");
		free(com.wd);
		com.wd = NULL;
		return 1;
	}
	
	int fnlen = strlen(name);
	if (!strncasecmp(name + fnlen - 4, ".ser", 4)) {
		struct ser_struct *ser_file = malloc(sizeof(struct ser_struct));
		ser_init_struct(ser_file);
		if (ser_open_file(name, ser_file)) return 1;
			
		new_seq = calloc(1, sizeof(sequence));
		initialize_sequence(new_seq, TRUE);
		new_seq->seqname = strndup(name, fnlen-4);
		new_seq->beg = 0;
		new_seq->end = ser_file->frame_count-1;
		new_seq->number = ser_file->frame_count;
		new_seq->type = SEQ_SER;
		new_seq->ser_file = ser_file;
	}
#ifdef HAVE_FFMS2
	else if (!check_for_film_extensions(get_filename_ext(name))) {
		struct film_struct *film_file = malloc(sizeof(struct film_struct));
		if (film_open_file(name, film_file)) {
			free(film_file);
			return 1;
		}
		new_seq = calloc(1, sizeof(sequence));
		initialize_sequence(new_seq, TRUE);
		int len = strlen(get_filename_ext(name));
		new_seq->seqname = strndup(name, fnlen-len-1);
		new_seq->beg = 0;
		new_seq->end = film_file->frame_count-1;
		new_seq->number = film_file->frame_count;
		new_seq->type = SEQ_AVI;
		new_seq->film_file = film_file;
		fprintf(stdout, "Found a AVI sequence\n");
	}
#endif
	if (!new_seq) return 0;
	if (new_seq->beg != new_seq->end) {
			if (!buildseqfile(new_seq) && retval)
				retval = 0;
		}
		free_sequence(new_seq, TRUE);
	return retval;
}

/* Saves the sequence in the seqname.seq file. */
int writeseqfile(sequence *seq){
	char *filename;
	FILE *seqfile;
	int i,j;

	if (!seq->seqname || seq->seqname[0] == '\0') return 1;
	filename = malloc(strlen(seq->seqname)+5);
	sprintf(filename, "%s.seq", seq->seqname);
	seqfile = fopen(filename, "w+");
	if (seqfile == NULL) {
		siril_log_message("Writing sequence file: cannot open %s for writing\n", filename);
		free(filename);
		return 1;
	}
	siril_log_message("Writing sequence file %s\n", filename);
	free(filename);

	fprintf(seqfile,"#Siril sequence file. Contains list of files (images), selection, and registration data\n");
	fprintf(seqfile,"#S 'sequence_name' start_index nb_images nb_selected fixed_len reference_image\n");
	fprintf(stderr,"S '%s' %d %d %d %d %d\n", 
			seq->seqname, seq->beg, seq->number, seq->selnum, seq->fixed, seq->reference_image);
	fprintf(seqfile,"S '%s' %d %d %d %d %d\n", 
			seq->seqname, seq->beg, seq->number, seq->selnum, seq->fixed, seq->reference_image);
	if (seq->type != SEQ_REGULAR) {
		/* sequence type, not needed for regular, S for ser, A for avi */
		fprintf(stderr, "T%c\n", seq->type == SEQ_SER ? 'S' : 'A');
		fprintf(seqfile, "T%c\n", seq->type == SEQ_SER ? 'S' : 'A');
	}

	/* TODO: FILL nb_layers */
	fprintf(stderr, "L %d\n", seq->nb_layers);
	fprintf(seqfile, "L %d\n", seq->nb_layers);

	for(i=0; i < seq->number; ++i){
		fprintf(seqfile,"I %d %d\n", seq->imgparam[i].filenum, 
				seq->imgparam[i].incl);
		//fprintf(stderr,"I %d %d\n", seq->imgparam[i].filenum, 
		//		seq->imgparam[i].incl);
	}

	for(j=0; j < seq->nb_layers; j++) {
		if (seq->regparam[j]) {
			for (i=0; i < seq->number; ++i) {
				fprintf(stderr, "R%d %d %d %g\n", j,
						seq->regparam[j][i].shiftx,
						seq->regparam[j][i].shifty,
						seq->regparam[j][i].quality);
				fprintf(seqfile, "R%d %d %d %g\n", j,
						seq->regparam[j][i].shiftx,
						seq->regparam[j][i].shifty,
						seq->regparam[j][i].quality);
			}
		}
	}
	fclose(seqfile);
	return 0;
}

gboolean existseq(const char *name){
	char *filename;
	struct stat sts;
	if (!name || name[0] == '\0') return FALSE;
	filename = malloc(strlen(name)+5);
	sprintf(filename, "%s.seq", name);
	if(stat(filename, &sts)==0){
		free(filename);
		return TRUE;
	}
	free(filename);
	return FALSE;
}

/* try to create the sequence file for the newly found sequence */
int buildseqfile(sequence *seq) {
	image_type imagetype;
	int i;
	char *filename;
	imgdata *oldparam;

	if (seq->end <= 0 || !seq->seqname || seq->seqname[0] == '\0') return 1;
	if (existseq(seq->seqname)) {
		fprintf(stderr,"seqfile '%s.seq' already exists, not recomputing\n", seq->seqname);
		return 0;
	}

	filename = malloc(strlen(seq->seqname)+20);
	if (seq->type == SEQ_REGULAR) {
		get_possible_image_filename(seq, seq->beg, filename);
		// check if the sequence begins at first_index
		if (stat_file(filename, &imagetype, NULL) || imagetype != TYPEFITS) {
			siril_log_message("The sequence %s doesn't start at the frame number %d"
					" with the specified fixed size index (%d). Cannot load.\n",
					seq->seqname, seq->beg, seq->fixed);
			free(filename);
			return 1;
		}
	}

	int alloc_size = 30;
	//seq->number = 0;
	// fill in one pass: realloc needed
	if (seq->type == SEQ_REGULAR) {
		if (seq->end - seq->beg < 111)
			alloc_size = seq->end - seq->beg + 1;	// last index IS included
	} else alloc_size = seq->end - seq->beg + 1;		// always continuous
	oldparam = seq->imgparam;
	if ((seq->imgparam = realloc(seq->imgparam, alloc_size*sizeof(imgdata))) == NULL) {
		fprintf(stderr, "Could not reallocate image parameters structure in sequence\n");
		if (oldparam) free(oldparam);
		free(filename);
		return 2;
	}
	for (i = seq->beg; i <= seq->end; i++) {
		if (seq->type == SEQ_REGULAR) {
			get_possible_image_filename(seq, i, filename);
			if (!stat_file(filename, &imagetype, NULL) && imagetype == TYPEFITS) {
				if (seq->number+1 > alloc_size-1) {
					alloc_size += 25;
					oldparam = seq->imgparam;
					if (!(seq->imgparam = realloc(seq->imgparam, alloc_size*sizeof(imgdata)))) {
						fprintf(stderr, "Could not reallocate image parameters structure in sequence\n");
						if (oldparam) free(oldparam);
						free(filename);
						return 2;
					}
				}
				seq->imgparam[seq->number].filenum=i;
				seq->imgparam[seq->number].incl=SEQUENCE_DEFAULT_INCLUDE;
				seq->imgparam[seq->number].eval = 0.0;
				seq->number++;
			}
		} else {
			seq->imgparam[i].filenum=i;
			seq->imgparam[i].incl=SEQUENCE_DEFAULT_INCLUDE;
			seq->imgparam[i].eval = 0.0;
		}
	}
#if SEQUENCE_DEFAULT_INCLUDE == TRUE
	seq->selnum = seq->number;
#else
	seq->selnum = 0;
#endif
	writeseqfile(seq);

	siril_log_message("sequence found: %s %d->%d\n", seq->seqname, seq->beg, seq->end);
	free(filename);
	return 0;
}

/* load a sequence */
int set_seq(const char *name){
	sequence *seq;
	int image_to_load;
	
	if ((seq = readseqfile(name)) == NULL) {
		fprintf(stderr, "could not load sequence %s\n", name);
		return 1;
	}
	free_image_data();
	if (seq->reference_image != -1)
		image_to_load = seq->reference_image;
	else image_to_load = 0;

	if (seq_read_frame(seq, image_to_load, &gfit)) {
		fprintf(stderr, "could not load first image from sequence\n");
		free(seq);
		return 1;
	}

	/* initialize sequence-related runtime data */
	seq->current = image_to_load;

	if (seq->nb_layers == -1) {	// not init yet, first loading of the sequence
		seq->nb_layers = gfit.naxes[2];
		seq->regparam = calloc(seq->nb_layers, sizeof(regdata *));
		seq->layers = calloc(seq->nb_layers, sizeof(layer_info));
		writeseqfile(seq);
	}
	/* Sequence is stored in com.seq for now */
	free_sequence(&com.seq, FALSE);
	memcpy(&com.seq, seq, sizeof(sequence));

	if (seq->nb_layers > 1)
		show_rgb_window();
	else hide_rgb_window();
	init_layers_hi_and_lo_values(MIPSLOHI); // set some hi and lo values in seq->layers,
	set_cutoff_sliders_max_values();// update min and max values for contrast sliders
	set_cutoff_sliders_values();	// update values for contrast sliders for this image
	seqsetnum(image_to_load);	// set limits for spin button and display loaded filenum
	set_layers_for_assign();	// set default layers assign and populate combo box
	set_layers_for_registration();	// set layers in the combo box for registration
	fill_sequence_list(seq, 0);	// display list of files in the sequence
	set_output_filename_to_sequence_name();
	sliders_mode_set_state(MINMAX);		// We force MINMAX
	initialize_display_mode();

	/* initialize image-related runtime data */
	set_cutoff_sliders_values();	// update values for contrast sliders for this image
	set_display_mode();		// display the display mode in the combo box
	display_filename();		// display filename in gray window
	adjust_exclude(image_to_load, FALSE);	// check or uncheck excluded checkbox
	adjust_refimage(image_to_load);	// check or uncheck reference image checkbox
	set_prepro_button_sensitiveness(); // enable or not the preprobutton
	update_reg_interface(TRUE);	// change the registration prereq message
	update_stack_interface();	// get stacking info and enable the Go button
	adjust_reginfo();		// change registration displayed/editable values
	update_gfit_histogram_if_needed();

	/* redraw and display image */
	show_main_gray_window();
	close_tab();	//close Green and Blue Tab if a 1-layer sequence is loaded
	adjust_vport_size_to_image();	// resize viewports to the displayed image size
	redraw(com.cvport, REMAP_ALL);

	update_used_memory();
	return 0;
}

/* name_buf must be allocated to 256 characters */
char *seq_get_image_filename(sequence *seq, int index, char *name_buf) {
	switch (seq->type) {
		case SEQ_REGULAR:
			return fit_sequence_get_image_filename(seq, index, name_buf, TRUE);
		case SEQ_SER:
			if (!name_buf || index < 0 || index > seq->end) {
				return NULL;
			}
			snprintf(name_buf, 255, "%d from %s.ser", index, seq->seqname);
			name_buf[255] = '\0';
			return name_buf;
#if defined(HAVE_FFMPEG) || defined(HAVE_FFMS2)
		case SEQ_AVI:
			if (!name_buf || index < 0 || index > seq->end) {
				return NULL;
			}
			//snprintf(name_buf, 255, "%d from %s.avi", index, seq->seqname);
			snprintf(name_buf, 255, "%s_%d", seq->seqname, index);
			name_buf[255] = '\0';
			return name_buf;
#endif
		case SEQ_INTERNAL:
			snprintf(name_buf, 255, "%s_%d", seq->seqname, index);
			name_buf[255] = '\0';
			return name_buf;
	}
	return NULL;
}

int seq_read_frame(sequence *seq, int index, fits *dest) {
	char filename[256];
	assert(index < seq->number);
	switch (seq->type) {
		case SEQ_REGULAR:
			fit_sequence_get_image_filename(seq, index, filename, TRUE);
			if (readfits(filename, dest, NULL)) {
				siril_log_message("could not load image %d from sequence %s\n",
						index, seq->seqname); 
				return 1;
			}
			break;
		case SEQ_SER:
			assert(seq->ser_file);
			if (ser_read_frame(seq->ser_file, index, dest)) {
				siril_log_message("could not load frame %d from SER sequence %s\n",
						index, seq->seqname); 
				return 1;
			}
			break;
#if defined(HAVE_FFMPEG)
		case SEQ_AVI:
			assert(seq->avi_file);
			if (avi_read_frame(seq->avi_file, index, dest)) {
				siril_log_message("could not load frame %d from AVI sequence %s\n",
						index, seq->seqname); 
				return 1;
			}
			// should dest->maxi be set to 255 here?
			break;
#elif defined(HAVE_FFMS2)
		case SEQ_AVI:
			assert(seq->film_file);
			if (film_read_frame(seq->film_file, index, dest)) {
				siril_log_message("could not load frame %d from AVI sequence %s\n",
						index, seq->seqname); 
				return 1;
			}
			// should dest->maxi be set to 255 here?
			break;
#endif
		case SEQ_INTERNAL:
			assert(seq->internal_fits);
			copyfits(seq->internal_fits[index], dest, CP_FORMAT, -1);
			dest->data = seq->internal_fits[index]->data;
			dest->pdata[0] = seq->internal_fits[index]->pdata[0];
			dest->pdata[1] = seq->internal_fits[index]->pdata[1];
			dest->pdata[2] = seq->internal_fits[index]->pdata[2];
			break;
	}
	image_find_minmax(dest, 0);
	return 0;
}

/* same as seq_read_frame above, but creates an image the size of the selection
 * rectangle only. layer is set to the layer number in the read partial frame.
 * The partial image result is only one-channel deep, so it cannot be used to
 * have a partial RGB image. */
int seq_read_frame_part(sequence *seq, int layer, int index, fits *dest, const rectangle *area) {
	char filename[256];
	fits tmp_fit;
	memset(&tmp_fit, 0, sizeof(fits));
	switch (seq->type) {
		case SEQ_REGULAR:
			fit_sequence_get_image_filename(seq, index, filename, TRUE);
			if (readfits_partial(filename, layer, dest, area)) {
				siril_log_message("Could not load partial image %d from sequence %s\n",
						index, seq->seqname); 
				return 1;
			}
			break;
		case SEQ_SER:
			assert(seq->ser_file);
			/* that could be optimized for SER files, image parts
			 * can be read since we know how they are stored. */
			if (ser_read_frame(seq->ser_file, index, &tmp_fit)) {
				siril_log_message("could not load frame %d from SER sequence %s\n",
						index, seq->seqname); 
				return 1;
			}
			extract_region_from_fits(&tmp_fit, layer, dest, area);
			clearfits(&tmp_fit);
			break;
#if defined(HAVE_FFMPEG)
		case SEQ_AVI:
			assert(seq->avi_file);
			if (avi_read_frame(seq->avi_file, index, &tmp_fit)) {
				siril_log_message("could not load frame %d from AVI sequence %s\n",
						index, seq->seqname); 
				return 1;
			}
			extract_region_from_fits(&tmp_fit, layer, dest, area);
			clearfits(&tmp_fit);
			break;
#elif defined(HAVE_FFMS2)
		case SEQ_AVI:
			assert(seq->film_file);
			if (film_read_frame(seq->film_file, index, &tmp_fit)) {
				siril_log_message("could not load frame %d from AVI sequence %s\n",
						index, seq->seqname); 
				return 1;
			}
			extract_region_from_fits(&tmp_fit, layer, dest, area);
			clearfits(&tmp_fit);
			break;
#endif
		case SEQ_INTERNAL:
			assert(seq->internal_fits);
			extract_region_from_fits(seq->internal_fits[index], 0, dest, area);
			break;
	}
	return 0;
}

/* TODO: cut that method in two, with an internal func taking a filename and a fits * */
/* if load_it is true, dest is assumed to be gfit */
int seq_load_image(sequence *seq, int index, fits *dest, gboolean load_it) {
	seq->current = index;
	clear_stars_list();
	clear_histograms();
	// what else needs to be cleaned?
	if (load_it) {
		set_cursor_waiting(TRUE);
		if (seq_read_frame(seq, index, dest)) {
			set_cursor_waiting(FALSE);
			return 1;
		}
		set_fwhm_star_as_star_list(seq);// display the fwhm star if possible
		init_layers_hi_and_lo_values(MIPSLOHI); // If MIPS-LO/HI exist we load these values. If not it is min/max
		sliders_mode_set_state(com.sliders);
		set_cutoff_sliders_max_values();// update min and max values for contrast sliders
		set_cutoff_sliders_values();	// update values for contrast sliders for this image			
		set_display_mode();		// display the display mode in the combo box
		redraw(com.cvport, REMAP_ALL);	// redraw and display image
		redraw_previews();		// redraw registration preview areas
		display_filename();		// display filename in gray window
		adjust_reginfo();		// change registration displayed/editable values
		calculate_fwhm(com.vport[com.cvport]);
		update_gfit_histogram_if_needed();
		set_cursor_waiting(FALSE);
	}
	/* change the displayed value in the spin button to have the real file number
	 * instead of the index of the adjustment */
	display_image_number(index);
	sequence_list_change_current();
	adjust_exclude(index, FALSE);	// check or uncheck excluded checkbox
	adjust_refimage(index);	// check or uncheck reference image checkbox
	update_used_memory();
	return 0;
}

/* if FWHM was calculated on the sequence, a minimisation exists for all
 * images, and when switching to a new image, it should be set as the only item
 * in the star list, in order to be displayed.
 * A special care is required in PSF_list.c:clear_stars_list(), to not free this data. */
void set_fwhm_star_as_star_list(sequence *seq) {
	assert(seq->regparam);
	int layer = get_registration_layer(seq);
	/* we chose here the first layer that has been allocated, which doesn't
	 * mean it contains data for all images. Handle with care. */
	if (seq->regparam && layer >= 0 && layer < seq->nb_layers && seq->regparam[layer] &&
			seq->regparam[layer][seq->current].fwhm && !com.stars) {
		com.stars = malloc(2 * sizeof(Param_GAUSS *));
		com.stars[0] = seq->regparam[layer][seq->current].fwhm;
		com.stars[1] = NULL;
		com.star_is_seqdata = TRUE;
	}
}

/* Rebuilds the file name of an image in a sequence.
 * The file name is stored in name_buffer, which must be allocated 256 bytes
 * The index is the index in the sequence, not the number appearing in the file name
 * Return value: NULL on error, name_buffer on success.
 */
char *fit_sequence_get_image_filename(sequence *seq, int index, char *name_buffer, gboolean add_fits_ext) {
	char format[20];
	if (index < 0 || index > seq->number || name_buffer == NULL)
		return NULL;
	if (seq->fixed <= 1){
		sprintf(format, "%%s%%d");
	} else {
		sprintf(format, "%%s%%.%dd", seq->fixed);
	}
	if (add_fits_ext)
		strcat(format, SIRIL_FITS_EXTENSION);
	snprintf(name_buffer, 255, format,
			seq->seqname, seq->imgparam[index].filenum);
	name_buffer[255] = '\0';
	return name_buffer;
}

/* Returns a filename for an image that could be in a sequence, but the sequence structure
 * has not been fully initialized yet. Only beg, end, fixed and seqname are used.
 */
char *get_possible_image_filename(sequence *seq, int image_number, char *name_buffer) {
	char format[20];
	if (image_number < seq->beg || image_number > seq->end || name_buffer == NULL)
		return NULL;
	if (seq->fixed <= 1){
		sprintf(format, "%%s%%d"SIRIL_FITS_EXTENSION);
	} else {
		sprintf(format, "%%s%%.%dd"SIRIL_FITS_EXTENSION, seq->fixed);
	}
	sprintf(name_buffer, format, seq->seqname, image_number);
	return name_buffer;
}

/* splits a filename in a base name and an index number, if the file name ends with .fit
 * it also computes the fixed length if there are zeros in the index */
int	get_index_and_basename(const char *filename, char **basename, int *index, int *fixed){
	char *buffer;
	int i, fnlen, first_zero, digit_idx;

	*index = -1;		// error values
	*fixed = 0;
	first_zero = -1;
	*basename = NULL;
	fnlen = strlen(filename);
	if (fnlen < strlen(SIRIL_FITS_EXTENSION)+2) return -1;
	if (!ends_with(filename, SIRIL_FITS_EXTENSION)) return -1;
	i = fnlen-strlen(SIRIL_FITS_EXTENSION)-1;
	if (!isdigit(filename[i])) return -1;
	digit_idx = i;

	buffer = strdup(filename);
	buffer[fnlen-strlen(SIRIL_FITS_EXTENSION)] = '\0';		// for atoi()
	do {
		if (buffer[i] == '0' && first_zero < 0)
			first_zero = i;
		if (buffer[i] != '0' && first_zero > 0)
			first_zero = -1;
		i--;
	} while (i >= 0 && isdigit(buffer[i]));
	i++;
	if (i == 0) {
		free(buffer);
		return -1;	// no base name, only number
	}
	if (first_zero >= 0)
		*fixed = digit_idx - i + 1;
	//else *fixed = 0;
	*index = atoi(buffer+i);
	if (*basename == NULL) {	// don't copy it if we already have it
		*basename = malloc(i * sizeof(char) + 1);
		strncpy(*basename, buffer, i);
		(*basename)[i] = '\0';
	}
	//fprintf(stdout, "from filename %s, base name is %s, index is %d\n", filename, *basename, *index);
	free(buffer);
	return 0;
}

/* sets default values for the sequence */
void initialize_sequence(sequence *seq, gboolean is_zeroed) {
	int i;
	if (!is_zeroed) {
		memset(seq, 0, sizeof(sequence));
	}
	seq->nb_layers = -1;		// uninit value
	seq->reference_image = -1;	// uninit value
	seq->type = SEQ_REGULAR;
	for (i=0; i<PREVIEW_NB; i++) {
		seq->previewX[i] = -1;
		seq->previewY[i] = -1;
	}
}

/* call this to close a sequence. Second arg must be FALSE for com.seq
 * WARNING: the data is not reset to NULL, if seq is to be reused,
 * initialize_sequence() must be called on it right after free_sequence()
 * (= do it for com.seq) */
void free_sequence(sequence *seq, gboolean free_seq_too) {
	static GtkComboBoxText *cbbt_layers = NULL;
	int i;
		
	if (cbbt_layers == NULL)
		cbbt_layers = GTK_COMBO_BOX_TEXT(gtk_builder_get_object(
					builder, "comboboxreglayer"));
	gtk_combo_box_text_remove_all(cbbt_layers);
	
	if (seq == NULL) return;
	if (seq->nb_layers > 0 && seq->regparam) {
		for (i=0; i<seq->nb_layers; i++) {
			if (seq->regparam[i]) {
				int j;
				for (j=0; j < seq->number; j++) {
					if (seq->regparam[i][j].fwhm)
						free(seq->regparam[i][j].fwhm);
				}
				free(seq->regparam[i]);
			}
		}
		free(seq->regparam);
	}
	if (seq->seqname)	free(seq->seqname);
	if (seq->layers)	free(seq->layers);
	if (seq->imgparam)	free(seq->imgparam);

	if (seq->ser_file) {
		ser_close_file(seq->ser_file);	// frees the data too
		free(seq->ser_file);
	}
#if defined(HAVE_FFMPEG)
	if (seq->avi_file) {
		avi_close_file(seq->avi_file);	// frees the data too
		free(seq->avi_file);
	}
#elif defined(HAVE_FFMS2)
	if (seq->film_file) {
		film_close_file(seq->film_file);	// frees the data too
		free(seq->film_file);
	}
#endif
	if (seq->internal_fits) {
		/* the fits in internal_fits should still be referenced somewhere */
		free(seq->internal_fits);
	}
	if (free_seq_too)	free(seq);
}

void sequence_free_preprocessing_data(sequence *seq) {
	// free opened files
	if (seq->ppprefix) {
		free(seq->ppprefix);
		seq->ppprefix = NULL;
	}
	if (seq->offset) {
		clearfits(seq->offset);
		free(seq->offset);
		seq->offset = NULL;
	}
	if (seq->dark) {
		clearfits(seq->dark);
		free(seq->dark);
		seq->dark = NULL;
	}
	if (seq->flat) {
		clearfits(seq->flat);
		free(seq->flat);
		seq->flat = NULL;
	}
}

gboolean sequence_is_loaded() {
	return (com.seq.seqname != NULL && com.seq.imgparam != NULL);
}

/* Start a processing on all images of the sequence seq, on layer layer if it applies.
 * The see coment in siril.h for help on process format.
 */
int sequence_processing(sequence *seq, sequence_proc process, int layer) {
	int i;
	float cur_nb, nb_frames;
	fits fit;
	rectangle area;

	if (!com.selection.w || !com.selection.h) {
		siril_log_message("No selection was made for a selection-based sequence processing\n");
		return 1;
	}
	memcpy(&area, &com.selection, sizeof(rectangle));
	memset(&fit, 0, sizeof(fits));
	check_or_allocate_regparam(seq, layer);

	/*if (selected_images_only)
		nb_frames = (float)seq->selnum;
	else*/
	nb_frames = (float)seq->number;

	for (i=0, cur_nb=0.f; i<seq->number; ++i) {
		//if (selected_images_only && !seq->imgparam[i].incl)
		//	continue;
		/* opening the image */
		if (seq_read_frame_part(seq, layer, i, &fit, &area))
			return 1;
		/* processing the image */
		if (process(seq, layer, i, &fit) < 0)
			return 1;
		cur_nb += 1.f;
		progress_bar_set_percent(cur_nb/nb_frames);
	}
	return 0;
}

/* Computes FWHM for a sequence image and store data in the sequence imgdata.
 * seq_layer is the corresponding layer in the raw image from the sequence.
 */
int seqprocess_fwhm(sequence *seq, int seq_layer, int frame_no, fits *fit) {
	rectangle area;
	area.x = area.y = 0;
	area.w = fit->rx; area.h = fit->ry;
	assert(seq_layer < seq->nb_layers);
	Param_GAUSS *result = Get_Minimisation(fit, 0, &area);
	if (result) {
		seq->regparam[seq_layer][frame_no].fwhm = result;
		return 0;
	} else {
		seq->regparam[seq_layer][frame_no].fwhm = NULL;
		return 1;
	}
}

void do_fwhm_sequence_processing(sequence *seq, int layer) {
	int i, retval;
	siril_log_message("Starting sequence processing of PSF\n");
	progress_bar_set_text("Computing PSF on selected star");
	retval = sequence_processing(seq, &seqprocess_fwhm, layer);
	siril_log_message("Finished sequence processing of PSF\n");
	if (retval) {
		progress_bar_set_text("Failed to compute PSF for the sequence. Ready.");
		set_cursor_waiting(FALSE);
		return;
	}
	// update the list
	if (seq->type != SEQ_INTERNAL)
		fill_sequence_list(seq, layer);
	// set the coordinates of the detected star, not known by the processing
	for (i=0; i<seq->number; i++) {
		Param_GAUSS *star = seq->regparam[layer][i].fwhm;
		if (star) {
			// same code as in add_star(), set position of star using selection coords
			star->xpos = star->x0 + com.selection.x;
			star->ypos = com.selection.y + com.selection.h - star->y0;
			fprintf(stdout, "star image %d: %g, %g\n", i, star->xpos, star->ypos);
		}
	}
	set_fwhm_star_as_star_list(seq);
	progress_bar_set_text("Finished computing PSF for the sequence. Ready.");
}

int seqprocess_planetary(sequence *seq, int seq_layer, int frame_no, fits *fit) {
	stats *stat;
	/* use the full image region because here we already have a frame part */
	rectangle area;
	area.x = area.y = 0;
	area.w = fit->rx; area.h = fit->ry;

	assert(seq_layer < seq->nb_layers);
	/* the area for statistics: NULL or area, it's the same since area is
	 * full image, but NULL is a faster */
	stat = statistics(fit, 0, NULL);
	seq->regparam[seq_layer][frame_no].entropy = entropy(fit, 0, &area, stat);
	fprintf(stdout, "frame %04d entropy: %g\n", frame_no, seq->regparam[seq_layer][frame_no].entropy);
	/*seq->regparam[seq_layer][frame_no].contrast = contrast(fit, 0);
	fprintf(stdout, "Frame %04d entropy: %g\tcontrast: %g\n", frame_no,
			seq->regparam[seq_layer][frame_no].entropy,
			seq->regparam[seq_layer][frame_no].contrast);*/
	return 0;
}

/* requires seq->nb_layers and seq->number to be already set */
void check_or_allocate_regparam(sequence *seq, int layer) {
	assert(layer < seq->nb_layers);
	if (!seq->regparam && seq->nb_layers > 0) {
		seq->regparam = calloc(seq->nb_layers, sizeof(regdata*));
		seq->layers = calloc(seq->nb_layers, sizeof(layer_info));
	}
	if (seq->regparam && !seq->regparam[layer] && seq->number > 0) {
		seq->regparam[layer] = calloc(seq->number, sizeof(regdata));
	}
}

/* internal sequence are a set of 1-layer images already loaded elsewhere, and
 * directly referenced as fits *.
 * This is used in LRGV composition.
 * The returned sequence does not contain any reference to files, and thus has
 * to be populated with internal_sequence_set() */
sequence *create_internal_sequence(int size) {
	int i;
	sequence *seq = calloc(1, sizeof(sequence));
	initialize_sequence(seq, TRUE);
	seq->type = SEQ_INTERNAL;
	seq->number = size;
	seq->selnum = size;
	seq->nb_layers = 1;	
	seq->internal_fits = calloc(size, sizeof(fits *));
	seq->seqname = strdup("internal sequence");
	seq->imgparam = calloc(size, sizeof(imgdata));
	for (i = 0; i < size; i++) {
		seq->imgparam[i].filenum = i;
		seq->imgparam[i].incl = 1;
	}
	check_or_allocate_regparam(seq, 0);
	return seq;
}

void internal_sequence_set(sequence *seq, int index, fits *fit) {
	assert(seq);
	assert(seq->internal_fits);
	assert(index < seq->number);
	seq->internal_fits[index] = fit;
}

