#define PLANETARY_REG_USES_IMAGE_PART
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <complex.h>
#include <fftw3.h>
#include <gtk/gtk.h>
#include <assert.h>

#include "registration.h"
#include "stacking.h"
#include "siril.h"
#include "callbacks.h"
#include "proto.h"
#include "PSF.h"

/* callback for the selected area event */
void _reg_selected_area_callback() { update_reg_interface(TRUE); }


static struct registration_method *reg_methods[4];

struct registration_method *new_reg_method(char *name, registration_function f, selection_type s) {
	struct registration_method *reg = malloc(sizeof (struct registration_method));
	reg->name = strdup(name);
	reg->method_ptr = f;
	reg->sel = s;
	return reg;
}

void initialize_registration_methods() {
	GtkComboBoxText *regcombo;
	int i = 0;
	reg_methods[i++] = new_reg_method("DFT translation (planetary/deep-sky)", &register_shift_dft, REQUIRES_SQUARED_SELECTION);
	reg_methods[i++] = new_reg_method("PSF translation (deep-sky)", &register_shift_fwhm, REQUIRES_ANY_SELECTION);
	//if (theli_is_available())
	//	reg_methods[i++] = new_reg_method("theli", register_theli, REQUIRES_NO_SELECTION);
	reg_methods[i] = NULL;

	/* fill comboboxregmethod */
	regcombo = GTK_COMBO_BOX_TEXT(gtk_builder_get_object(builder, "comboboxregmethod"));
	gtk_combo_box_text_remove_all(regcombo);
	i = 0;
	while (reg_methods[i] != NULL) {
		gtk_combo_box_text_append_text(regcombo, reg_methods[i]->name);
		siril_log_message("Added a registration method: %s\n", reg_methods[i]->name);
		i++;
	}
	if (i > 0) {
		gtk_combo_box_set_active(GTK_COMBO_BOX(regcombo), com.reg_methods);
	}

	/* register to the new area selected event */
	register_selection_update_callback(_reg_selected_area_callback);
}

struct registration_method *get_selected_registration_method() {
	GtkComboBox *regcombo = GTK_COMBO_BOX(gtk_builder_get_object(builder, "comboboxregmethod"));
	int index = gtk_combo_box_get_active(regcombo);
	return reg_methods[index];
}

/* register images: calculate shift in images to be aligned with the reference image;
 * images are not modified, only shift parameters are saved in regparam in the sequence.
 * layer is the layer on which the registration will be done, green by default (set in siril_init())
 * -
 * wfit[0] is used to load the reference image, wfit[4] is used to load other images
 */
int register_shift_dft(struct registration_args *args) {
	int i, x, y, size;
	//double qual;
	fftw_complex *ref, *img, *in, *out, *convol;
	fftw_plan p, q;
	int shiftx, shifty, shift;
	int ret;
	int plan;
	fftw_complex *cbuf, *ibuf, *obuf;
	float nb_frames, cur_nb;
	int ref_image;
	regdata *current_regdata;
	static volatile gboolean running = FALSE, abort_process = FALSE;
	char *tmpmsg, tmpfilename[256];
#ifdef PLANETARY_REG_USES_IMAGE_PART
	rectangle full_area;	// the area to use after getting image_part
	int j;
	double e_min = DBL_MAX;
	int e_index=-1;
	stats *stat;
#else
	int debx, deby;
	WORD *ubuf;	// pointer on original image data
	fftw_complex *dbuf;
#endif

	if (running) {
		abort_process = TRUE;
		return 1;
	} else {
		abort_process = FALSE;
		running = TRUE;
	}

	/* the selection needs to be squared for the DFT */
	assert(args->selection.w == args->selection.h);
	size = args->selection.w;
#ifdef PLANETARY_REG_USES_IMAGE_PART
	full_area.x = full_area.y = 0;
	full_area.h = full_area.w = size;
#else
	debx = args->selection.x;
	deby = args->selection.y;
#endif

	if (args->process_all_frames)
		nb_frames = (float)args->seq->number;
	else nb_frames = (float)args->seq->selnum;

	if (!args->seq->regparam) {
		fprintf(stderr, "regparam should have been created before\n");
		return -1;
	}
	if (args->seq->regparam[args->layer]) {
		siril_log_message("Recomputing already existing registration for this layers\n");
		current_regdata = args->seq->regparam[args->layer];
	} else {
		current_regdata = calloc(args->seq->number, sizeof(regdata));
		if (current_regdata == NULL) {
			siril_log_message("Error allocating registration data\n");
			return -2;
		}
	}

	/* loading reference frame */
	if (args->seq->reference_image == -1)
		ref_image = 0;
	else ref_image = args->seq->reference_image;

	progress_bar_set_text("Register DFT: loading and processing reference frame");
#ifdef PLANETARY_REG_USES_IMAGE_PART
	ret = seq_read_frame_part(args->seq, args->layer, ref_image, &(wfit[0]), &args->selection);

	/* the area for statistics: NULL or area, it's the same
	 * since area is full image, but NULL is a faster */
	stat = statistics(&(wfit[0]), 0, NULL);
	current_regdata[ref_image].entropy = entropy(&(wfit[0]), 0, &full_area, stat);
	e_min = current_regdata[ref_image].entropy;
	e_index = ref_image;
#else
	ret = seq_read_frame(args->seq, ref_image, &(wfit[0]));
#endif
	if (ret){
		siril_log_message("Register: could not load first image to register, aborting.\n");
		return ret;
	}

	ref = fftw_malloc(sizeof(fftw_complex) * size * size);
	img = fftw_malloc(sizeof(fftw_complex) * size * size);
	in = fftw_malloc(sizeof(fftw_complex) * size * size);
	out = fftw_malloc(sizeof(fftw_complex) * size * size);
	convol = fftw_malloc(sizeof(fftw_complex) * size * size);

	if (nb_frames > 200.f)
		plan = FFTW_MEASURE;
	else plan = FFTW_ESTIMATE;

	p = fftw_plan_dft_2d(size, size, ref, out, FFTW_FORWARD, plan);
	q = fftw_plan_dft_2d(size, size, convol, out, FFTW_BACKWARD, plan);

	/* copying image selection into the fftw data */
#ifdef PLANETARY_REG_USES_IMAGE_PART
	j = 0;
	for (y=0; y<size; y++) {
		for (x=0; x<size; x++) {
			ref[j] = (double)wfit[0].data[j];
			j++;
		}
	}
#else
	dbuf = ref;
	ubuf = wfit[0].pdata[args->layer] + debx + deby * wfit[0].rx;
	for (y=0; y<size; ++y) {
		for (x=0; x<size; x++) {
			*dbuf++ = (double)*ubuf++;
		}
		ubuf += wfit[0].rx-size;
	}
#endif

	fftw_execute_dft(p, ref, in); /* repeat as needed */
	//qual = quality(in, size, size, 0, (WORD)size);
	current_regdata[ref_image].shiftx = 0;
	current_regdata[ref_image].shifty = 0;
	//current_regdata[ref_image].quality = qual;

	for (i=0, cur_nb=0.f; i<args->seq->number; ++i){
		if (abort_process) break;
		if (i == ref_image) continue;
		if (!args->process_all_frames && !args->seq->imgparam[i].incl) continue;

		seq_get_image_filename(args->seq, i, tmpfilename);
		tmpmsg = siril_log_message("Register: processing image %s\n", tmpfilename);
		tmpmsg[strlen(tmpmsg)-1] = '\0';
		progress_bar_set_text(tmpmsg);
#ifdef PLANETARY_REG_USES_IMAGE_PART
		if (!(ret = seq_read_frame_part(args->seq, args->layer, i, &(wfit[4]), &args->selection))) {
			/* entropy, used in stacking for image filtering */
			/* the area for statistics: NULL or area, it's the same
			 * since area is full image, but NULL is a faster */
			stat = statistics(&(wfit[4]), 0, NULL);		// layer is 0 in frame_part
			current_regdata[i].entropy = entropy(&(wfit[4]), 0, &full_area, stat);
			if (current_regdata[i].entropy < e_min && args->seq->imgparam[i].incl) {
				e_min = current_regdata[i].entropy;
				e_index = i;
			}

			j = 0;
			for (y=0; y<size; y++) {
				for (x=0; x<size; x++) {
					img[j] = (double)wfit[4].data[j];
					j++;
				}
			}
#else
		if (!(ret = seq_read_frame(args->seq, i, &(wfit[4])))) {
			dbuf = img;
			ubuf = wfit[4].pdata[args->layer] + debx + deby * wfit[4].rx;

			for(y=0;y<size;++y){
				for(x=0;x<size;x++){
					*dbuf++=(double)*ubuf++;
				}
				ubuf+=wfit[4].rx-size;
			}
#endif
			fftw_execute_dft(p,img,out); /* repeat as needed */
			/* originally, quality is computed with the quality
			 * function working on the fft space. Now we use entropy
			 * instead.
			 * qual=quality(out,size,size,0,(WORD)size);
			 */
			cbuf=convol;
			ibuf=in;
			obuf=out;
			for (x=0;x<size*size;x++){
				*cbuf++ = *ibuf++ * conj(*obuf++);
			}

			fftw_execute_dft(q,convol,ref); /* repeat as needed */
			shift=0;
			for (x=1; x<size*size; ++x){
				if(creal(ref[x]) > creal(ref[shift])){
					shift=x;
					// break or get last value?
				}
			}
			shifty = shift/size;
			shiftx = shift%size;
			if (shifty > size/2){
				shifty -= size;
			}
			if (shiftx > size/2){
				shiftx -= size;
			}
			
			current_regdata[i].shiftx = shiftx;
			current_regdata[i].shifty = shifty;
			//current_regdata[i].quality = qual;
			current_regdata[i].quality = 0.0;

			/* shiftx and shifty are the x and y values for translation that
			 * would make this image aligned with the reference image.
			 * WARNING: the y value is counted backwards, since the FITS is
			 * stored down from up.
			 */
			fprintf(stderr,"reg: file %d, shiftx=%d shifty=%d entropy=%g\n",
					args->seq->imgparam[i].filenum,
					current_regdata[i].shiftx,
					current_regdata[i].shifty,
					current_regdata[i].entropy);
			cur_nb += 1.f;
			progress_bar_set_percent(cur_nb/nb_frames);
		}
		else {
			//report_fits_error(ret, error_buffer);
			if (current_regdata == args->seq->regparam[args->layer])
				args->seq->regparam[args->layer] = NULL;
			free(current_regdata);
			return ret;
		}
	}

	fftw_destroy_plan(p);
	fftw_destroy_plan(q);
	fftw_free(in);
	fftw_free(out);
	fftw_free(ref);
	fftw_free(img);
	fftw_free(convol);
	args->seq->regparam[args->layer] = current_regdata;
	update_used_memory();
	siril_log_message("Registration finished.\n");
#ifdef PLANETARY_REG_USES_IMAGE_PART
	siril_log_color_message("Best frame: #%d with quality=%g.\n", "bold", e_index, e_min);
	free(stat);
#endif
	running = FALSE; 
	if (abort_process) {
		abort_process = FALSE;
		return 1;
	}
	abort_process = FALSE;
	return 0;
}

/* register images: calculate shift in images to be aligned with the reference image;
 * images are not modified, only shift parameters are saved in regparam in the sequence.
 * layer is the layer on which the registration will be done, green by default (set in siril_init())
 * -
 */
int register_shift_fwhm(struct registration_args *args) {
	int frame, ref_image;
	float nb_frames, cur_nb;
	double reference_xpos, reference_ypos;
	double fwhm_min = DBL_MAX;
	int fwhm_index=-1;
	regdata *current_regdata;
	/* First and longest step: get the minimization data on one star for all
	 * images to register, which provides FWHM but also star coordinates */
	// TODO: detect that it was already computed, and don't do it again
	do_fwhm_sequence_processing(args->seq, args->layer);

	/* Intermediate step: allocate registration data */
	if (!args->seq->regparam) {
		fprintf(stderr, "regparam should have been created before\n");
		return -1;
	}
	if (args->seq->regparam[args->layer]) {
		//siril_log_message("Recomputing (and overwriting) already existing registration for this layer\n");
		// ^ not necessarily, it can be allocated in do_fwhm_sequence_processing above.
		current_regdata = args->seq->regparam[args->layer];
	} else {
		current_regdata = calloc(args->seq->number, sizeof(regdata));
		if (current_regdata == NULL) {
			siril_log_message("Error allocating registration data\n");
			return -2;
		}
	}

	if (args->process_all_frames)
		nb_frames = (float)args->seq->number;
	else nb_frames = (float)args->seq->selnum;

	/* loading reference frame */
	if (args->seq->reference_image == -1)
		ref_image = 0;
	else ref_image = args->seq->reference_image;
	if (!current_regdata[ref_image].fwhm) {
		siril_log_message("Registration PSF: failed to compute PSF for reference frame at least\n");
		if (current_regdata != args->seq->regparam[args->layer])
			free(current_regdata);
		return -1;
	}
	reference_xpos = current_regdata[ref_image].fwhm->xpos;
	reference_ypos = current_regdata[ref_image].fwhm->ypos;
	
	fwhm_min = current_regdata[ref_image].fwhm->FWHMX;
	fwhm_index = ref_image;

	/* Second step: align image by aligning star coordinates together */
	for (frame=0, cur_nb=0.f; frame<args->seq->number; frame++) {
		double tmp;
		if (!args->process_all_frames && !args->seq->imgparam[frame].incl) continue;
		/* quality is not computed in this registration method */
		current_regdata[frame].quality = 0.0;
		if (frame == ref_image || !current_regdata[frame].fwhm) {
			current_regdata[frame].shiftx = 0;
			current_regdata[frame].shifty = 0;
			continue;
		}
		if (current_regdata[frame].fwhm->FWHMX < fwhm_min && current_regdata[frame].fwhm->FWHMX > 0.0) {
			fwhm_min = current_regdata[frame].fwhm->FWHMX;
			fwhm_index = frame;
		}
		tmp = reference_xpos - current_regdata[frame].fwhm->xpos;
		current_regdata[frame].shiftx = round_to_int(tmp);
		tmp = current_regdata[frame].fwhm->ypos - reference_ypos;
		current_regdata[frame].shifty = round_to_int(tmp);

		/* shiftx and shifty are the x and y values for translation that
		 * would make this image aligned with the reference image.
		 * WARNING: the y value is counted backwards, since the FITS is
		 * stored down from up.
		 */
		fprintf(stderr,"reg: file %d, shiftx=%d shifty=%d\n",
				args->seq->imgparam[frame].filenum,
				current_regdata[frame].shiftx,
				current_regdata[frame].shifty);
		cur_nb += 1.f;
		progress_bar_set_percent(cur_nb/nb_frames);
	}

	args->seq->regparam[args->layer] = current_regdata;
	update_used_memory();
	siril_log_message("Registration finished.\n");
	siril_log_color_message("Best frame: #%d with fwhm=%.3g.\n", "bold", fwhm_index, fwhm_min);
	return 0;
}

void on_comboboxregmethod_changed (GtkComboBox *box, gpointer user_data) {
	int reg = gtk_combo_box_get_active(box);
	
	com.reg_methods = reg;
	writeinitfile();
}

int get_registration_layer(sequence *seq) {
	int reglayer;
	if (!sequence_is_loaded()) return -1;
	/* finding allocated registration layer */
/*	if (seq->nb_layers == 3) {
		if (seq->regparam[GLAYER])
			reglayer = GLAYER;
		else if (seq->regparam[BLAYER])
			reglayer = BLAYER;
		else if (seq->regparam[RLAYER])
			reglayer = RLAYER;
		else reglayer = -1;
	}
	else if (seq->regparam[RLAYER])
		reglayer = RLAYER;
	else reglayer = -1; */
	reglayer = gtk_combo_box_get_active(GTK_COMBO_BOX(gtk_builder_get_object(builder, "comboboxreglayer")));
	return reglayer;
}	

/* Selects the "register all" or "register selected" according to the number of
 * selected images, if argument is false.
 * Verifies that enough images are selected and an area is selected.
 */
void update_reg_interface(gboolean dont_change_reg_radio) {
	static GtkWidget *go_register = NULL;
	static GtkLabel *labelreginfo = NULL;
	static GtkToggleButton *reg_all = NULL, *reg_sel = NULL;
	int nb_images_reg;	/* the number of images to register */

	if (!go_register) {
		go_register = lookup_widget("goregister_button");
		reg_all = GTK_TOGGLE_BUTTON(gtk_builder_get_object(builder, "regallbutton"));
		reg_sel = GTK_TOGGLE_BUTTON(gtk_builder_get_object(builder, "regselbutton"));
		labelreginfo = GTK_LABEL(gtk_builder_get_object(builder, "labelregisterinfo"));
	}

	if (!dont_change_reg_radio) {
		if (com.seq.selnum < com.seq.number)
			gtk_toggle_button_set_active(reg_sel, TRUE);
		else 	gtk_toggle_button_set_active(reg_all, TRUE);
	}

	if (gtk_toggle_button_get_active(reg_all))
		nb_images_reg = com.seq.number;
	else nb_images_reg = com.seq.selnum;
	if (nb_images_reg > 1 && com.selection.w > 0 && com.selection.h > 0) {
		gtk_widget_set_sensitive(go_register, TRUE);
		gtk_label_set_text(labelreginfo, "");
	} else {
	       	gtk_widget_set_sensitive(go_register, FALSE);
		if (nb_images_reg <= 1 && com.selection.w <= 0 && com.selection.h <= 0)
			gtk_label_set_text(labelreginfo, "Select an area in image first, and select images in the sequence.");
		else if (nb_images_reg <= 1)
			gtk_label_set_text(labelreginfo, "Select images in the sequence.");
		else gtk_label_set_text(labelreginfo, "Select an area in image first.");
	}
}

/* try to maximize the area within the image size (based on gfit) */
void compute_squared_selection(rectangle *area) {
	//fprintf(stdout, "function entry: %d,%d,\t%dx%d\n", area->x, area->y, area->w, area->h);
	if (area->x >= 0 && area->x + area->w <= gfit.rx &&
			area->y >= 0 && area->y + area->h <= gfit.ry)
		return;

	if (area->x < 0) {
		area->x++;
		if (area->x + area->w > gfit.rx) {
			/* reduce area */
			area->w -= 2;
			area->h -= 2;
			area->y++;
		}
	} else if (area->x + area->w > gfit.rx) {
		area->x--;
		if (area->x < 0) {
			/* reduce area */
			area->x++;
			area->w -= 2;
			area->h -= 2;
			area->y++;
		}
	}

	if (area->y < 0) {
		area->y++;
		if (area->y + area->h > gfit.ry) {
			/* reduce area */
			area->h -= 2;
			area->w -= 2;
			area->x++;
		}
	} else if (area->y + area->h > gfit.ry) {
		area->y--;
		if (area->y < 0) {
			/* reduce area */
			area->x++;
			area->w -= 2;
			area->h -= 2;
			area->y++;
		}
	}

	return compute_squared_selection(area);
}

void get_the_registration_area(struct registration_args *reg_args, struct registration_method *method) {
	int max;
	switch (method->sel) {
		case REQUIRES_NO_SELECTION:
			break;
		case REQUIRES_ANY_SELECTION:
			memcpy(&reg_args->selection, &com.selection, sizeof(rectangle));
			break;
		case REQUIRES_SQUARED_SELECTION:
			/* Passed arguments are X,Y of the center of the square and the size of
			 * the square. */
			if (com.selection.w > com.selection.h)
				max = com.selection.w;
			else max = com.selection.h;

			reg_args->selection.x = com.selection.x + com.selection.w/2 - max/2;
			reg_args->selection.w = max;
			reg_args->selection.y = com.selection.y + com.selection.h/2 - max/2;
			reg_args->selection.h = max;
			compute_squared_selection(&reg_args->selection);

			/* save it back to com.selection do display it properly */
			memcpy(&com.selection, &reg_args->selection, sizeof(rectangle));
			fprintf(stdout, "final area: %d,%d,\t%dx%d\n",
					reg_args->selection.x, reg_args->selection.y,
					reg_args->selection.w, reg_args->selection.h);
			redraw(com.cvport, REMAP_NONE);
			break;
	}
}

/* callback for 'Go register' button */
void on_seqregister_button_clicked(GtkButton *button, gpointer user_data) {
	struct registration_args reg_args;
	struct registration_method *method;
	char *msg;
	GtkToggleButton *regall;
	GtkComboBox *cbbt_layers;
	struct timeval t_start, t_end;

	if (com.selection.w <= 0 && com.selection.h <= 0) {
		msg = siril_log_message("All prerequisites are not filled for registration. Select a rectangle first.\n");
		show_dialog(msg, "Warning", "gtk-dialog-warning");
		return ;
	}

	control_window_switch_to_tab(OUTPUT_LOGS);
	/* getting the selected registration method */
	method = get_selected_registration_method();

	/* filling the arguments for registration */
	reg_args.seq = &com.seq;
	regall = GTK_TOGGLE_BUTTON(lookup_widget("regallbutton"));
	reg_args.process_all_frames = gtk_toggle_button_get_active(regall);
	/* getting the selected registration layer from the combo box. The value is the index
	 * of the selected line, and they are in the same order than layers so there should be
	 * an exact matching between the two */
	cbbt_layers = GTK_COMBO_BOX(gtk_builder_get_object(builder, "comboboxreglayer"));
	reg_args.layer = gtk_combo_box_get_active(cbbt_layers);
	get_the_registration_area(&reg_args, method);

	msg = siril_log_color_message("Registration: processing using method: %s\n", "red", method->name);
	msg[strlen(msg)-1] = '\0';
	gettimeofday (&t_start, NULL);
	set_cursor_waiting(TRUE);
	progress_bar_set_text(msg);
	progress_bar_set_percent(0.0);

	if (!method->method_ptr(&reg_args)) {
		writeseqfile(&com.seq);		// shouldn't it be the same seq file?
		fill_sequence_list(&com.seq, com.cvport);
		set_layers_for_registration();	// update display of available reg data
	}
	progress_bar_set_text("Registration complete");
	progress_bar_set_percent(1.0);
	update_stack_interface();
	set_cursor_waiting(FALSE);
	gettimeofday (&t_end, NULL);
	show_time(t_start, t_end);
}

