/*
 * This file is part of Siril, an astronomy image processor.
 * Copyright (C) 2005-2011 Francois Meyer (dulle at free.fr)
 * Copyright (C) 2012-2014 team free-astro (see more in AUTHORS file)
 * Reference site is http://free-astro.vinvin.tf/index.php/Siril
 *
 * Siril is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siril is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Siril. If not, see <http://www.gnu.org/licenses/>.
 *
 * Useful links about OpenCV:
 * http://docs.opencv.org/modules/core/doc/intro.html
 * http://docs.opencv.org/modules/imgproc/doc/geometric_transformations.html#resize
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#ifdef HAVE_OPENCV
# include <math.h>

#include "siril.h"
#include "opencv.h"
#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>

using namespace cv;

/* resizes image to the sizes toX * toY, and stores it back in image */
int resize_gaussian(fits *image, int toX, int toY, int interpolation) {
	assert(image->data);
	assert(image->rx);
	int type =  CV_16U;
	if (image->naxes[2] != 1) 
		type = CV_16UC3;

	Mat in(image->ry, image->rx, type, image->data);
	Mat out(toY, toX, type);
	resize(in, out, out.size(), 0, 0, interpolation);
	image->rx = toX;
	image->naxes[0] = toX;
	image->ry = toY;
	image->naxes[1] = toY;
	WORD *newdata = (WORD*) realloc(image->data, toX * toY * sizeof(WORD) * image->naxes[2]);
	if (!newdata) {
		free(newdata);
		return 1;
	}
	image->data = newdata;
	memcpy(image->data, out.data, toX * toY * sizeof(WORD) * image->naxes[2]);
	if (image->naxes[2] == 1) {
		image->pdata[0] = image->data;
		image->pdata[1] = image->data;
		image->pdata[2] = image->data;
	}
	else {
		image->pdata[0] = image->data;
		image->pdata[1] = image->data + (toX * toY);
		image->pdata[2] = image->data + (toX * toY) * 2;	
	}
	in.release();
	return 0;
}

/* Rotate an image with the angle "angle"
 * TODO allow users to choose between a cropped or not cropped result
 * cropped = 0 : no crop
 * cropped = 1 : crop
 * For now the result is always cropped
 */

int rotate_image(fits *image, double angle, int layer, int interpolation, int cropped) {
	assert(image->data);
	assert(image->rx);
	int ndata = image->rx * image->ry;
	Mat in(image->ry, image->rx, CV_16U, image->data + ndata * layer);
	Mat out;
	
	Point2f pt(in.cols/2.0, in.rows/2.0);		// We take the center of the image. Should we pass this in function parameters ?
	Mat r = getRotationMatrix2D(pt, angle, 1.0);
	warpAffine(in, out, r, Size(in.cols, in.rows), INTER_LINEAR);
	memcpy(image->data + ndata * layer, out.data, in.cols*in.rows*sizeof(WORD));
	image->pdata[layer] = image->data + ndata * layer;
	image->rx = out.cols;
	image->ry = out.rows;
	in.release();
	return 0;
}

int unsharp_filter(fits* image, double sigma, double amount) {
	assert(image->data);
	assert(image->rx);
	int type =  CV_16U;
	if (image->naxes[2] != 1) 
		type = CV_16UC3;

	Mat in(image->ry, image->rx, type, image->data);
	Mat out, contrast;
	GaussianBlur(in, out, Size(), sigma);
	if (fabs(amount) > 0.0) {
		Mat sharpened = in * (1 + amount) + out * (- amount);
		out = sharpened.clone();
		sharpened.release();
	}
	
	memcpy(image->data, out.data, image->rx * image->ry * sizeof(WORD) * image->naxes[2]);
	if (image->naxes[2] == 1) {
		image->pdata[0] = image->data;
		image->pdata[1] = image->data;
		image->pdata[2] = image->data;
	}
	else {
		image->pdata[0] = image->data;
		image->pdata[1] = image->data + (image->rx * image->ry);
		image->pdata[2] = image->data + (image->rx * image->ry) * 2;	
	}
	in.release();
	return 0;
}

#endif
