#ifndef INCLUDED_MODBASE_
#define INCLUDED_MODBASE_

#include <iomanip>
#include <unordered_map>

#include "../error/error.h"
#include "../typedefs/typedefs.h"
#include "../scenario/scenario.h"

class ModBase
{
    friend std::ostream &operator<<(std::ostream &, ModBase const &);

    std::string const d_id;

    uint16_t d_cost = 0;
    uint16_t d_nr;
    SizeVect d_count;
    uint16_t d_falsePositives;

    uint16_t d_configuration = 0;   // config == ALL -> valid

    static uint16_t s_nr;

    public:
        ModBase(std::string const &id);
        virtual ~ModBase();

        double sensitivity(size_t idx) const;
        double specificity(double age) const;
        uint16_t cost() const;
        uint16_t nr() const;
        void resetCounters(size_t nRounds);
        void count(size_t round);
        void falsePositive();
        std::string const &id() const;
        double const *dose() const;
        double dose(uint16_t idx) const;

        void roundHeader(std::ostream &out) const;
        void count(std::ostream &out, size_t round) const;

    protected:
                                // ptr to the parameter line handling function
        typedef Err (*Handler)(ModBase *obj, std::istream &in);
        typedef std::unordered_map<std::string, Handler> HandlerMap;

        enum ConfigBits
        {
            COST        =   1 << 0,
            SENSITIVITY =   1 << 1,
            SPECIFICITY =   1 << 2,
            DOSE        =   1 << 3,
            BETA        =   1 << 4,
        };

        struct Specificity              // used by Mammo and Tomo
        {
            uint16_t    d_beginAge;
            uint16_t    d_endAge;
            double      d_value;
        };
        typedef std::vector<Specificity> SpecVect;

        uint16_t configure(HandlerMap const &handlers, Scenario::Range &range,
                           char const *id, SpecVect const *specPtr = 0);
        uint16_t addConfigLine(HandlerMap const &handlers, ConfigSrc src, 
                               char const *id,
                               Scenario::LineIter const &lineIter,
                               SpecVect const *specPtr);

        static Err costHandler(ModBase *obj, std::istream &in);
        
        Err doubleHandler(double *dest, std::istream &in, ConfigBits type);
        Err specificities(SpecVect *dest, std::istream &in);

        uint16_t configured() const;
        bool setBit(ConfigBits bit);    // true: now set, false: already set

        static void insertBirads(std::ostream &out, char const *label, 
                                 double const *value);
        static void insertSpecificity(std::ostream &out,
                                      SpecVect const &value);

    private:
        virtual double const *vDose() const = 0;
        virtual double vDose(uint16_t idx) const = 0;
        virtual void vInsert(std::ostream &out) const = 0;
        virtual double vSensitivity(size_t idx) const = 0;
        virtual double vSpecificity(double age) const = 0;
};

inline uint16_t ModBase::configured() const
{
    return d_configuration;
}

inline void ModBase::count(size_t round)
{
    ++d_count[round];
}

inline void ModBase::count(std::ostream &out, size_t round) const
{
    out << ',' << std::setw(7) << d_count[round];
}

inline uint16_t ModBase::nr() const
{
    return d_nr;
}

inline uint16_t ModBase::cost() const
{
    return d_cost;
}

inline void ModBase::falsePositive()
{
    ++d_falsePositives;
}

inline std::string const &ModBase::id() const
{
    return d_id;
}

inline double const *ModBase::dose() const
{
    return vDose();
}

inline double ModBase::dose(uint16_t idx) const
{
    return vDose(idx);
}


inline double ModBase::sensitivity(size_t idx) const
{
    return vSensitivity(idx);
}

inline double ModBase::specificity(double age) const
{
    return vSpecificity(age);
}

inline void ModBase::roundHeader(std::ostream &out) const
{
    out << ", #" << d_id;
}    

#endif

