#ifndef INCLUDED_SCENARIO_
#define INCLUDED_SCENARIO_

// iterations:             1
// seed:                   1
// nWomen:                 100000
// discountReferenceAge:   50

#include <unordered_set>
#include <unordered_map>

#include <bobcat/config>

#include "../typedefs/typedefs.h"
#include "../enums/enums.h"
#include "../error/error.h"
#include "../options/options.h"

class ConfFile;

class Scenario
{
    enum PairID
    {
//        DISCOUNTAGE,
        ITERATIONS,
        N_WOMEN,
        SEED
    };

    Error &d_error;
    Options &d_options;
    ConfFile const &d_confFile;     // return ConfFile's parameters unless
                                    // overruled by Scenario

    mutable FBB::Config d_scenario; // scenario provided 
                                    // configuration parameters

    bool d_spread;              // true: apply variations to configuration
                                //       parameters

    SeedType d_seedType;        // fixed, increasing or random (default)

//    size_t d_discountAge;
    size_t d_iterations;
    size_t d_nWomen;
    size_t d_seed;

    StringVect d_labels;            // text of 'label:' lines

    static char const s_spread[];

    static char const s_generator[];
    static std::unordered_map<std::string, SeedType> s_generatorMap;

    static std::pair<size_t, char const *> s_sizeDefault[];
    static std::unordered_set<std::string> s_availableFiles;

    public:
        typedef FBB::Config::const_iterator         const_iterator;
        typedef FBB::Config::CIVect::const_iterator IDiterator;
        typedef std::pair<IDiterator, IDiterator>   IDpair;

        struct Single
        {
            const_iterator  iter;
            ConfigSrc       src;
        };
        struct Range
        {
            IDpair          pair;
            ConfigSrc       src;
        };            

        Scenario(Error &error, Options &d_options, ConfFile const &confFile,
                 std::istream &stream, uint16_t lineNr);

        void setValues();                       // set the scenario 
                                                // configuration values

                                                // 0: not found, 
        size_t value(char const *id) const;     // string::npos: error

        Range lines(char const *id) const;         
        Single find(char const *id) const;
        const_iterator end() const;             // d_confFile.end()

        bool spread() const;        // true: apply variations to configuration
                                    //       parameters
    
        size_t iterations() const;
        size_t seed() const;
        SeedType seedType() const;
        size_t nWomen() const;
//        size_t discountAge() const;

        std::ostream &error(ConfigSrc type, uint16_t lineNr, Err value) const;
        Error &error() const;

        std::ofstream parametersFile(size_t idx) const;
        std::ofstream dataFile(size_t idx) const;
        std::ofstream roundsFile(size_t idx) const;
        std::ofstream sensitivityFile() const;

        Options const &options() const;

        void writeParameters(std::ostream &out, size_t iter) const;

        StringVect const &labels() const;

    private:
        void alterOptions();

        void setSpread();                   // set variableSpred 
        void setSeedType();                 // set the random generator seed
                                            //  type 

        size_t setValue(PairID id);

        static std::ofstream outStream(std::string const &fname, size_t idx);
        static std::string replaceDollar(std::string const &settingsFile, 
                                         size_t idx);
        static std::ofstream open(std::string const &fname);
};

inline StringVect const &Scenario::labels() const
{
    return d_labels;
}

inline Options const &Scenario::options() const
{
    return d_options;
}
    
inline Error &Scenario::error() const
{
    return d_error;
}

inline std::ostream &Scenario::error(ConfigSrc type, uint16_t lineNr, 
                                     Err value) const
{
    return d_error(type, lineNr, value);
}

inline bool Scenario::spread() const
{
    return d_spread;
}

inline size_t Scenario::iterations() const
{
    return d_iterations;
}

inline size_t Scenario::seed() const
{
    return d_seed;
}

inline SeedType Scenario::seedType() const
{
    return d_seedType;
}

inline size_t Scenario::nWomen() const
{
    return d_nWomen;
}

//inline size_t Scenario::discountAge() const
//{
//    return d_discountAge;
//}

inline std::ofstream Scenario::dataFile(size_t idx) const
{
    return outStream(d_options.dataFile(), idx);
}

inline std::ofstream Scenario::roundsFile(size_t idx) const
{
    return outStream(d_options.roundsFile(), idx);
}

inline std::ofstream Scenario::sensitivityFile() const
{
    return outStream(d_options.sensitivityFile(), 0);
}

inline std::ofstream Scenario::parametersFile(size_t idx) const
{
    return outStream(d_options.parametersFile(), idx);
}

#endif






