/* DataDesc/ddt_parse.c -- automatic parsing of data structures             */

/* Copyright (c) 2004, 2005, 2006, 2007, 2009, 2010. The SimGrid Team.
 * All rights reserved.                                                     */

/* This program is free software; you can redistribute it and/or modify it
 * under the terms of the license (GNU LGPL) which comes with this package. */

#include <ctype.h>              /* isdigit */

#include "xbt/ex.h"
#include "datadesc_private.h"
#include "ddt_parse.yy.h"

XBT_LOG_NEW_DEFAULT_SUBCATEGORY(xbt_ddt_parse, xbt_ddt,
                                "Parsing C data structures to build XBT data description");

typedef struct s_type_modifier {
  short is_long;
  int is_unsigned:1;
  int is_short:1;

  int is_struct:1;
  int is_union:1;
  int is_enum:1;

  int is_ref:1;

  int is_dynar:2;
  int is_matrix:2;
} s_type_modifier_t, *type_modifier_t;

typedef struct s_field {
  xbt_datadesc_type_t type;
  char *type_name;
  char *name;
  s_type_modifier_t tm;
} s_identifier_t;

extern char *xbt_ddt_parse_text;       /* text being considered in the parser */

/* local functions */
static void parse_type_modifier(type_modifier_t type_modifier)
{
  XBT_IN();
  do {
    if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_STAR) {
      /* This only used when parsing 'short *' since this function returns when int, float, double,... is encountered */
      XBT_DEBUG("This is a reference");
      type_modifier->is_ref++;

    } else if (!strcmp(xbt_ddt_parse_text, "unsigned")) {
      XBT_DEBUG("This is an unsigned");
      type_modifier->is_unsigned = 1;

    } else if (!strcmp(xbt_ddt_parse_text, "short")) {
      XBT_DEBUG("This is short");
      type_modifier->is_short = 1;

    } else if (!strcmp(xbt_ddt_parse_text, "long")) {
      XBT_DEBUG("This is long");
      type_modifier->is_long++; /* handle "long long" */

    } else if (!strcmp(xbt_ddt_parse_text, "struct")) {
      XBT_DEBUG("This is a struct");
      type_modifier->is_struct = 1;

    } else if (!strcmp(xbt_ddt_parse_text, "union")) {
      XBT_DEBUG("This is an union");
      type_modifier->is_union = 1;

    } else if (!strcmp(xbt_ddt_parse_text, "enum")) {
      XBT_DEBUG("This is an enum");
      type_modifier->is_enum = 1;

    } else if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_EMPTY) {
      XBT_DEBUG("Pass space");

    } else {
      XBT_DEBUG("Done with modifiers (got %s)", xbt_ddt_parse_text);
      break;
    }

    xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
    if ((xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_WORD) &&
        (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_STAR)) {
      XBT_DEBUG("Done with modifiers (got %s,%d)", xbt_ddt_parse_text,
             xbt_ddt_parse_tok_num);
      break;
    }
  } while (1);
  XBT_OUT();
}

static void print_type_modifier(s_type_modifier_t tm)
{
  int i;

  XBT_IN();
  if (tm.is_unsigned)
    printf("(unsigned) ");
  if (tm.is_short)
    printf("(short) ");
  for (i = 0; i < tm.is_long; i++)
    printf("(long) ");

  if (tm.is_struct)
    printf("(struct) ");
  if (tm.is_enum)
    printf("(enum) ");
  if (tm.is_union)
    printf("(union) ");

  for (i = 0; i < tm.is_ref; i++)
    printf("(ref) ");
  XBT_OUT();
}

static void change_to_fixed_array(xbt_dynar_t dynar, long int size)
{
  s_identifier_t former, array;
  memset(&array, 0, sizeof(array));

  XBT_IN();
  xbt_dynar_pop(dynar, &former);
  array.type_name = (char *) xbt_malloc(strlen(former.type->name) + 48);
  XBT_DEBUG("Array specification (size=%ld, elm='%s'), change pushed type",
         size, former.type_name);
  sprintf(array.type_name, "%s%s%s%s[%ld]",
          (former.tm.is_unsigned ? "u " : ""),
          (former.tm.is_short ? "s " : ""),
          (former.tm.is_long ? "l " : ""), former.type_name, size);
  free(former.type_name);

  array.type = xbt_datadesc_array_fixed(array.type_name, former.type, size);   /* redeclaration are ignored */
  array.name = former.name;

  xbt_dynar_push(dynar, &array);
  XBT_OUT();
}

static void change_to_ref(xbt_dynar_t dynar)
{
  s_identifier_t former, ref;
  memset(&ref, 0, sizeof(ref));

  XBT_IN();
  xbt_dynar_pop(dynar, &former);
  ref.type_name = (char *) xbt_malloc(strlen(former.type->name) + 2);
  XBT_DEBUG("Ref specification (elm='%s'), change pushed type",
         former.type_name);
  sprintf(ref.type_name, "%s*", former.type_name);
  free(former.type_name);

  ref.type = xbt_datadesc_ref(ref.type_name, former.type);     /* redeclaration are ignored */
  ref.name = former.name;

  xbt_dynar_push(dynar, &ref);
  XBT_OUT();
}

static void change_to_ref_pop_array(xbt_dynar_t dynar)
{
  s_identifier_t former, ref;
  memset(&ref, 0, sizeof(ref));

  XBT_IN();
  xbt_dynar_pop(dynar, &former);
  ref.type = xbt_datadesc_ref_pop_arr(former.type);    /* redeclaration are ignored */
  ref.type_name = (char *) strdup(ref.type->name);
  ref.name = former.name;

  free(former.type_name);

  xbt_dynar_push(dynar, &ref);
  XBT_OUT();
}

static void change_to_dynar_of(xbt_dynar_t dynar,
                               xbt_datadesc_type_t subtype)
{
  s_identifier_t former, ref;
  memset(&ref, 0, sizeof(ref));

  XBT_IN();
  xbt_dynar_pop(dynar, &former);
  ref.type = xbt_datadesc_dynar(subtype, NULL);        /* redeclaration are ignored */
  ref.type_name = (char *) strdup(ref.type->name);
  ref.name = former.name;

  free(former.type_name);

  xbt_dynar_push(dynar, &ref);
  XBT_OUT();
}

static void change_to_matrix_of(xbt_dynar_t dynar,
                                xbt_datadesc_type_t subtype)
{
  s_identifier_t former, ref;
  memset(&ref, 0, sizeof(ref));

  XBT_IN();
  xbt_dynar_pop(dynar, &former);
  ref.type = xbt_datadesc_matrix(subtype, NULL);       /* redeclaration are ignored */
  ref.type_name = (char *) strdup(ref.type->name);
  ref.name = former.name;

  free(former.type_name);

  xbt_dynar_push(dynar, &ref);
  XBT_OUT();
}

static void add_free_f(xbt_dynar_t dynar, void_f_pvoid_t free_f)
{
  s_identifier_t former, ref;
  memset(&ref, 0, sizeof(ref));

  XBT_IN();
  xbt_dynar_pop(dynar, &former);
  memcpy(former.type->extra, free_f, sizeof(free_f));
  xbt_dynar_push(dynar, &former);
  XBT_OUT();
}

static void parse_statement(char *definition,
                            xbt_dynar_t identifiers,
                            xbt_dynar_t fields_to_push)
{
  char buffname[512];

  s_identifier_t identifier;

  int expect_id_separator = 0;

  XBT_IN();
  memset(&identifier, 0, sizeof(identifier));

  xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
  if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_RA) {
    XBT_OUT();
    THROWF(mismatch_error, 0, "End of the englobing structure or union");
  }

  if (XBT_LOG_ISENABLED(xbt_ddt_parse, xbt_log_priority_debug)) {
    int colon_pos;
    for (colon_pos = xbt_ddt_parse_col_pos;
         definition[colon_pos] != ';'; colon_pos++);
    definition[colon_pos] = '\0';
    XBT_DEBUG("Parse the statement \"%s%s;\" (col_pos=%d)",
           xbt_ddt_parse_text,
           definition + xbt_ddt_parse_col_pos, xbt_ddt_parse_col_pos);
    definition[colon_pos] = ';';
  }

  if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_WORD)
    PARSE_ERROR
        ("Unparsable symbol: found a typeless statement (got '%s' instead)",
         xbt_ddt_parse_text);

        /**** get the type modifier of this statement ****/
  parse_type_modifier(&identifier.tm);

  /*  FIXME: This does not detect recursive definitions at all? */
  if (identifier.tm.is_union || identifier.tm.is_enum
      || identifier.tm.is_struct)
    PARSE_ERROR
        ("Unimplemented feature: XBT_DEFINE_TYPE cannot handle recursive type definition yet");

        /**** get the base type, giving "short a" the needed love ****/
  if (!identifier.tm.is_union &&
      !identifier.tm.is_enum &&
      !identifier.tm.is_struct &&
      (identifier.tm.is_short || identifier.tm.is_long
       || identifier.tm.is_unsigned) && strcmp(xbt_ddt_parse_text, "char")
      && strcmp(xbt_ddt_parse_text, "float")
      && strcmp(xbt_ddt_parse_text, "double")
      && strcmp(xbt_ddt_parse_text, "int")) {

    /* bastard user, they omited "int" ! */
    identifier.type_name = (char *) strdup("int");
    XBT_DEBUG("the base type is 'int', which were omited (you vicious user)");
  } else {
    identifier.type_name = (char *) strdup(xbt_ddt_parse_text);
    XBT_DEBUG("the base type is '%s'", identifier.type_name);
    xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
  }

        /**** build the base type for latter use ****/
  if (identifier.tm.is_union) {
    PARSE_ERROR
        ("Unimplemented feature: XBT_DEFINE_TYPE cannot handle union yet (get callback from annotation?)");

  } else if (identifier.tm.is_enum) {
    PARSE_ERROR
        ("Unimplemented feature: XBT_DEFINE_TYPE cannot handle enum yet");

  } else if (identifier.tm.is_struct) {
    sprintf(buffname, "struct %s", identifier.type_name);
    identifier.type = xbt_datadesc_struct(buffname);   /* Get created when does not exist */

  } else if (identifier.tm.is_unsigned) {
    if (!strcmp(identifier.type_name, "int")) {
      if (identifier.tm.is_long == 2) {
        identifier.type = xbt_datadesc_by_name("unsigned long long int");
      } else if (identifier.tm.is_long) {
        identifier.type = xbt_datadesc_by_name("unsigned long int");
      } else if (identifier.tm.is_short) {
        identifier.type = xbt_datadesc_by_name("unsigned short int");
      } else {
        identifier.type = xbt_datadesc_by_name("unsigned int");
      }

    } else if (!strcmp(identifier.type_name, "char")) {
      identifier.type = xbt_datadesc_by_name("unsigned char");

    } else {                    /* impossible, gcc parses this shit before us */
      THROW_IMPOSSIBLE;
    }

  } else if (!strcmp(identifier.type_name, "float")) {
    /* no modificator allowed by gcc */
    identifier.type = xbt_datadesc_by_name("float");

  } else if (!strcmp(identifier.type_name, "double")) {
    if (identifier.tm.is_long)
      PARSE_ERROR("long double not portable and thus not handled");

    identifier.type = xbt_datadesc_by_name("double");

  } else {                      /* signed integer elemental */
    if (!strcmp(identifier.type_name, "int")) {
      if (identifier.tm.is_long == 2) {
        identifier.type = xbt_datadesc_by_name("signed long long int");
      } else if (identifier.tm.is_long) {
        identifier.type = xbt_datadesc_by_name("signed long int");
      } else if (identifier.tm.is_short) {
        identifier.type = xbt_datadesc_by_name("signed short int");
      } else {
        identifier.type = xbt_datadesc_by_name("int");
      }

    } else if (!strcmp(identifier.type_name, "char")) {
      identifier.type = xbt_datadesc_by_name("char");

    } else {
      XBT_DEBUG("Base type is a constructed one (%s)", identifier.type_name);
      if (!strcmp(identifier.type_name, "xbt_matrix_t")) {
        identifier.tm.is_matrix = 1;
      } else if (!strcmp(identifier.type_name, "xbt_dynar_t")) {
        identifier.tm.is_dynar = 1;
      } else {
        identifier.type = xbt_datadesc_by_name(identifier.type_name);
        if (!identifier.type)
          PARSE_ERROR("Unknown base type '%s'", identifier.type_name);
      }
    }
  }
  /* Now identifier.type and identifier.name speak about the base type.
     Stars are not eaten unless 'int' was omitted.
     We will have to enhance it if we are in fact asked for array or reference.

     Dynars and matrices also need some extra love (prodiged as annotations)
   */

        /**** look for the symbols of this type ****/
  for (expect_id_separator = 0; (       /*(xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_EMPTY) && FIXME */
                                  (xbt_ddt_parse_tok_num !=
                                   XBT_DDT_PARSE_TOKEN_SEMI_COLON));
       xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump()) {

    if (expect_id_separator) {
      if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_COLON) {
        expect_id_separator = 0;
        continue;

      } else if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_LB) {
        /* Handle fixed size arrays */
        xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
        if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_RB) {
          PARSE_ERROR
              ("Unimplemented feature: XBT_DEFINE_TYPE cannot deal with [] constructs (yet)");

        } else if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_WORD) {
          char *end;
          long int size = strtol(xbt_ddt_parse_text, &end, 10);

          if (end == xbt_ddt_parse_text || *end != '\0') {
            /* Not a number. Get the constant value, if any */
            int *storage = xbt_dict_get_or_null(xbt_dd_constants,
                                                xbt_ddt_parse_text);
            if (storage) {
              size = *storage;
            } else {
              PARSE_ERROR
                  ("Unparsable size of array. Found '%s', expected number or known constant. Need to use xbt_datadesc_set_const(), huh?",
                   xbt_ddt_parse_text);
            }
          }

          /* replace the previously pushed type to an array of it */
          change_to_fixed_array(identifiers, size);

          /* eat the closing bracket */
          xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
          if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_RB)
            PARSE_ERROR("Unparsable size of array");
          XBT_DEBUG("Fixed size array, size=%ld", size);
          continue;
        } else {
          PARSE_ERROR("Unparsable size of array");
        }
        /* End of fixed size arrays handling */

      } else if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_WORD) {
        /* Handle annotation */
        s_identifier_t array;
        char *keyname = NULL;
        char *keyval = NULL;
        memset(&array, 0, sizeof(array));
        if (strcmp(xbt_ddt_parse_text, "XBT_ANNOTE"))
          PARSE_ERROR
              ("Unparsable symbol: Expected 'XBT_ANNOTE', got '%s'",
               xbt_ddt_parse_text);

        xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
        if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_LP)
          PARSE_ERROR
              ("Unparsable annotation: Expected parenthesis, got '%s'",
               xbt_ddt_parse_text);

        while ((xbt_ddt_parse_tok_num =
                xbt_ddt_parse_lex_n_dump()) ==
               XBT_DDT_PARSE_TOKEN_EMPTY);

        if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_WORD)
          PARSE_ERROR
              ("Unparsable annotation: Expected key name, got '%s'",
               xbt_ddt_parse_text);
        keyname = (char *) strdup(xbt_ddt_parse_text);

        while ((xbt_ddt_parse_tok_num =
                xbt_ddt_parse_lex_n_dump()) ==
               XBT_DDT_PARSE_TOKEN_EMPTY);

        if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_COLON)
          PARSE_ERROR
              ("Unparsable annotation: expected ',' after the key name, got '%s'",
               xbt_ddt_parse_text);

        while ((xbt_ddt_parse_tok_num =
                xbt_ddt_parse_lex_n_dump()) ==
               XBT_DDT_PARSE_TOKEN_EMPTY);

        /* get the value */

        if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_WORD)
          PARSE_ERROR
              ("Unparsable annotation: Expected key value, got '%s'",
               xbt_ddt_parse_text);
        keyval = (char *) strdup(xbt_ddt_parse_text);

        while ((xbt_ddt_parse_tok_num =
                xbt_ddt_parse_lex_n_dump()) ==
               XBT_DDT_PARSE_TOKEN_EMPTY);

        /* Done with parsing the annotation. Now deal with it by replacing previously pushed type with the right one */

        XBT_DEBUG("Anotation: %s=%s", keyname, keyval);
        if (!strcmp(keyname, "size")) {
          if (!identifier.tm.is_ref)
            PARSE_ERROR
                ("Size annotation for a field not being a reference");
          identifier.tm.is_ref--;

          if (!strcmp(keyval, "1")) {
            change_to_ref(identifiers);
            free(keyval);
          } else {
            char *p;
            int fixed = 1;
            for (p = keyval; *p != '\0'; p++)
              if (!isdigit(*p))
                fixed = 0;
            if (fixed) {
              change_to_fixed_array(identifiers, atoi(keyval));
              change_to_ref(identifiers);
              free(keyval);

            } else {
              change_to_ref_pop_array(identifiers);
              xbt_dynar_push(fields_to_push, &keyval);
            }
          }
        } else if (!strcmp(keyname, "subtype")) {
          xbt_datadesc_type_t subtype = xbt_datadesc_by_name(keyval);
          if (identifier.tm.is_matrix) {
            change_to_matrix_of(identifiers, subtype);
            identifier.tm.is_matrix = -1;
          } else if (identifier.tm.is_dynar) {
            change_to_dynar_of(identifiers, subtype);
            identifier.tm.is_dynar = -1;
          } else {
            PARSE_ERROR
                ("subtype annotation only accepted for dynars and matrices, but passed to '%s'",
                 identifier.type_name);
          }
          free(keyval);
        } else if (!strcmp(keyname, "free_f")) {
          int *storage = xbt_dict_get_or_null(xbt_dd_constants, keyval);
          if (!storage)
            PARSE_ERROR
                ("value for free_f annotation of field %s is not a known constant",
                 identifier.name);
          if (identifier.tm.is_matrix == -1) {
            add_free_f(identifiers, *(void_f_pvoid_t *) storage);
            identifier.tm.is_matrix = 0;
          } else if (identifier.tm.is_dynar == -1) {
            add_free_f(identifiers, *(void_f_pvoid_t *) storage);
            identifier.tm.is_dynar = 0;
          } else {
            PARSE_ERROR
                ("free_f annotation only accepted for dynars and matrices which subtype is already declared (field %s)",
                 identifier.name);
          }
          free(keyval);
        } else {
          free(keyval);
          PARSE_ERROR("Unknown annotation type: '%s'", keyname);
        }
        free(keyname);

        /* Get all the multipliers */
        while (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_STAR) {

          xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();

          if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_WORD)
            PARSE_ERROR
                ("Unparsable annotation: Expected field name after '*', got '%s'",
                 xbt_ddt_parse_text);

          keyval = xbt_malloc(strlen(xbt_ddt_parse_text) + 2);
          sprintf(keyval, "*%s", xbt_ddt_parse_text);

          /* ask caller to push field as a multiplier */
          xbt_dynar_push(fields_to_push, &keyval);

          /* skip blanks after this block */
          while ((xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump())
                 == XBT_DDT_PARSE_TOKEN_EMPTY);
        }

        if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_RP)
          PARSE_ERROR
              ("Unparsable annotation: Expected parenthesis, got '%s'",
               xbt_ddt_parse_text);

        continue;

        /* End of annotation handling */
      } else {
        PARSE_ERROR
            ("Unparsable symbol: Got '%s' instead of expected comma (',')",
             xbt_ddt_parse_text);
      }
    } else if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_COLON) {
      PARSE_ERROR("Unparsable symbol: Unexpected comma (',')");
    }

    if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_STAR) {
      identifier.tm.is_ref++;   /* We indeed deal with multiple references with multiple annotations */
      continue;
    }

    /* found a symbol name. Build the type and push it to dynar */
    if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_WORD) {

      identifier.name = (char *) strdup(xbt_ddt_parse_text);
      XBT_DEBUG("Found the identifier \"%s\"", identifier.name);

      xbt_dynar_push(identifiers, &identifier);
      XBT_DEBUG("Dynar_len=%lu", xbt_dynar_length(identifiers));
      expect_id_separator = 1;
      continue;
    }

    PARSE_ERROR
        ("Unparasable symbol (maybe a def struct in a def struct or a parser bug ;)");
  }

  if (identifier.tm.is_matrix > 0)
    PARSE_ERROR("xbt_matrix_t field without 'subtype' annotation");
  if (identifier.tm.is_dynar > 0)
    PARSE_ERROR("xbt_dynar_t field without 'subtype' annotation");

  XBT_OUT();
}

static xbt_datadesc_type_t parse_struct(char *definition)
{

  xbt_ex_t e;

  char buffname[32];
  static int anonymous_struct = 0;

  xbt_dynar_t identifiers;
  s_identifier_t field;
  unsigned int iter;
  int done;

  xbt_dynar_t fields_to_push;
  char *name;

  volatile xbt_datadesc_type_t struct_type;

  XBT_IN();
  identifiers = xbt_dynar_new(sizeof(s_identifier_t), NULL);
  fields_to_push = xbt_dynar_new(sizeof(char *), NULL);

  /* Create the struct descriptor */
  if (xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_WORD) {
    struct_type = xbt_datadesc_struct(xbt_ddt_parse_text);
    XBT_VERB("Parse the struct '%s'", xbt_ddt_parse_text);
    xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
  } else {
    sprintf(buffname, "anonymous struct %d", anonymous_struct++);
    XBT_VERB("Parse the anonymous struct nb %d", anonymous_struct);
    struct_type = xbt_datadesc_struct(buffname);
  }

  if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_LA)
    PARSE_ERROR
        ("Unparasable symbol: Expecting struct definition, but got %s instead of '{'",
         xbt_ddt_parse_text);

  /* Parse the identifiers */
  done = 0;
  do {
    TRY {
      parse_statement(definition, identifiers, fields_to_push);
    }
    CATCH(e) {
      if (e.category != mismatch_error)
        RETHROW;
      xbt_ex_free(e);
      done = 1;
    }

    XBT_DEBUG("This statement contained %lu identifiers",
           xbt_dynar_length(identifiers));
    /* append the identifiers we've found */
    xbt_dynar_foreach(identifiers, iter, field) {
      if (field.tm.is_ref)
        PARSE_ERROR
            ("Not enough XBT_ANNOTATE to deal with all dereferencing levels of %s (%d '*' left)",
             field.name, (int)field.tm.is_ref);

      XBT_VERB("Append field '%s' to %p", field.name, (void *) struct_type);
      xbt_datadesc_struct_append(struct_type, field.name, field.type);
      free(field.name);
      free(field.type_name);

    }
    xbt_dynar_reset(identifiers);
    XBT_DEBUG("struct_type=%p", (void *) struct_type);

    /* Make sure that all fields declaring a size push it into the cbps */
    xbt_dynar_foreach(fields_to_push, iter, name) {
      XBT_DEBUG("struct_type=%p", (void *) struct_type);
      if (name[0] == '*') {
        XBT_VERB("Push field '%s' as a multiplier into size stack of %p",
              name + 1, (void *) struct_type);
        xbt_datadesc_cb_field_push_multiplier(struct_type, name + 1);
      } else {
        XBT_VERB("Push field '%s' into size stack of %p",
              name, (void *) struct_type);
        xbt_datadesc_cb_field_push(struct_type, name);
      }
      free(name);
    }
    xbt_dynar_reset(fields_to_push);
  } while (!done);
  xbt_datadesc_struct_close(struct_type);

  /* terminates */
  if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_RA)
    PARSE_ERROR
        ("Unparasable symbol: Expected '}' at the end of struct definition, got '%s'",
         xbt_ddt_parse_text);

  xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();

  xbt_dynar_free(&identifiers);
  xbt_dynar_free(&fields_to_push);
  XBT_OUT();
  return struct_type;
}

static xbt_datadesc_type_t parse_typedef(char *definition)
{

  s_type_modifier_t tm;

  xbt_datadesc_type_t typedef_desc = NULL;

  XBT_IN();
  memset(&tm, 0, sizeof(tm));

  /* get the aliased type */
  parse_type_modifier(&tm);

  if (tm.is_struct) {
     parse_struct(definition);
  }

  parse_type_modifier(&tm);

  if (tm.is_ref)
    PARSE_ERROR
        ("XBT_DEFINE_TYPE cannot handle reference without annotation");

  /* get the aliasing name */
  if (xbt_ddt_parse_tok_num != XBT_DDT_PARSE_TOKEN_WORD)
    PARSE_ERROR
        ("Unparsable typedef: Expected the alias name, and got '%s'",
         xbt_ddt_parse_text);

  /* (FIXME: should) build the alias */
  PARSE_ERROR
      ("Unimplemented feature: XBT_DEFINE_TYPE cannot handle typedef yet");

  XBT_OUT();
  return typedef_desc;
}


/**
 * xbt_datadesc_parse:
 *
 * Create a datadescription from the result of parsing the C type description
 */
xbt_datadesc_type_t
xbt_datadesc_parse(const char *name, const char *C_statement)
{

  xbt_datadesc_type_t res = NULL;
  char *definition;
  int semicolon_count = 0;
  int def_count, C_count;

  XBT_IN();
  /* reput the \n in place for debug */
  for (C_count = 0; C_statement[C_count] != '\0'; C_count++)
    if (C_statement[C_count] == ';' || C_statement[C_count] == '{')
      semicolon_count++;
  definition = (char *) xbt_malloc(C_count + semicolon_count + 1);
  for (C_count = 0, def_count = 0; C_statement[C_count] != '\0'; C_count++) {
    definition[def_count++] = C_statement[C_count];
    if (C_statement[C_count] == ';' || C_statement[C_count] == '{') {
      definition[def_count++] = '\n';
    }
  }
  definition[def_count] = '\0';

  /* init */
  XBT_VERB("_xbt_ddt_type_parse(%s) -> %d chars", definition, def_count);
  xbt_ddt_parse_pointer_string_init(definition);

  /* Do I have a typedef, or a raw struct ? */
  xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();

  if ((xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_WORD)
      && (!strcmp(xbt_ddt_parse_text, "struct"))) {
    xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
    res = parse_struct(definition);

  } else if ((xbt_ddt_parse_tok_num == XBT_DDT_PARSE_TOKEN_WORD)
             && (!strcmp(xbt_ddt_parse_text, "typedef"))) {
    xbt_ddt_parse_tok_num = xbt_ddt_parse_lex_n_dump();
    res = parse_typedef(definition);

  } else {
    XBT_ERROR
        ("Failed to parse the following symbol (not a struct neither a typedef) :\n%s",
         definition);
    xbt_abort();
  }

  xbt_ddt_parse_pointer_string_close();
  XBT_VERB("end of _xbt_ddt_type_parse()");
  free(definition);
  /* register it under the name provided as symbol */
  if (strcmp(res->name, name)) {
    XBT_ERROR
        ("In XBT_DEFINE_TYPE, the provided symbol (here %s) must be the C type name (here %s)",
         name, res->name);
    xbt_abort();
  }
  xbt_ddt_parse_lex_destroy();
  XBT_OUT();
  return res;
}

xbt_dict_t xbt_dd_constants;
/** \brief Declare a constant to the parsing mecanism. See the "\#define and fixed size array" section */
void xbt_datadesc_set_const(const char *name, int value)
{
  int *stored = xbt_new(int, 1);
  *stored = value;

  xbt_dict_set(xbt_dd_constants, name, stored, NULL);
}
