/* Copyright (c) 2006-2019. The SimGrid Team. All rights reserved.          */

/* This program is free software; you can redistribute it and/or modify it
 * under the terms of the license (GNU LGPL) which comes with this package. */

#ifndef SIMGRID_S4U_COMM_HPP
#define SIMGRID_S4U_COMM_HPP

#include <simgrid/forward.h>
#include <simgrid/s4u/Activity.hpp>

#include <atomic>
#include <string>
#include <vector>

namespace simgrid {
namespace s4u {
/** @brief Communication async
 *
 * Represents all asynchronous communications, that you can test or wait onto.
 */
class XBT_PUBLIC Comm : public Activity {
  Mailbox* mailbox_                   = nullptr;
  kernel::actor::ActorImpl* sender_   = nullptr;
  kernel::actor::ActorImpl* receiver_ = nullptr;
  double rate_                        = -1;
  void* dst_buff_                     = nullptr;
  size_t dst_buff_size_               = 0;
  void* src_buff_                     = nullptr;
  size_t src_buff_size_               = sizeof(void*);
  std::string tracing_category_       = "";
  std::atomic_int_fast32_t refcount_{0};
  /* FIXME: expose these elements in the API */
  bool detached_                                                          = false;
  int (*match_fun_)(void*, void*, kernel::activity::CommImpl*)            = nullptr;
  void (*clean_fun_)(void*)                                               = nullptr;
  void (*copy_data_function_)(kernel::activity::CommImpl*, void*, size_t) = nullptr;

  Comm() : Activity() {}

public:
#ifndef DOXYGEN
  friend XBT_PUBLIC void intrusive_ptr_release(Comm* c);
  friend XBT_PUBLIC void intrusive_ptr_add_ref(Comm* c);
  friend Mailbox; // Factory of comms
#endif

  virtual ~Comm();

  static xbt::signal<void(Actor const&)> on_sender_start;
  static xbt::signal<void(Actor const&)> on_receiver_start;
  static xbt::signal<void(Actor const&)> on_completion;

  /*! take a vector s4u::CommPtr and return when one of them is finished.
   * The return value is the rank of the first finished CommPtr. */
  static int wait_any(std::vector<CommPtr> * comms) { return wait_any_for(comms, -1); }
  /*! Same as wait_any, but with a timeout. If the timeout occurs, parameter last is returned.*/
  static int wait_any_for(std::vector<CommPtr>* comms_in, double timeout);

  /*! take a vector s4u::CommPtr and return when all of them is finished. */
  static void wait_all(std::vector<CommPtr>* comms);
  /*! take a vector s4u::CommPtr and return the rank of the first finished one (or -1 if none is done). */
  static int test_any(std::vector<CommPtr> * comms);

  Comm* start() override;
  Comm* wait() override;
  Comm* wait_for(double timeout) override;
  Comm* cancel() override;
  bool test() override;

  /** Start the comm, and ignore its result. It can be completely forgotten after that. */
  Comm* detach();
  /** Start the comm, and ignore its result. It can be completely forgotten after that. */
  Comm* detach(void (*clean_function)(void*))
  {
    clean_fun_ = clean_function;
    return detach();
  }

  /** Sets the maximal communication rate (in byte/sec). Must be done before start */
  CommPtr set_rate(double rate);

  /** Specify the data to send.
   *
   * This is way will get actually copied over to the receiver.
   * That's completely unrelated from the simulated size (given with @ref Activity::set_remaining()):
   * you can send a short buffer in your simulator, that represents a very large message
   * in the simulated world, or the opposite.
   */
  CommPtr set_src_data(void* buff);
  /** Specify the size of the data to send. Not to be mixed with @ref Activity::set_remaining()
   *
   * That's the size of the data to actually copy in the simulator (ie, the data passed with Activity::set_src_data()).
   * That's completely unrelated from the simulated size (given with @ref Activity::set_remaining()):
   * you can send a short buffer in your simulator, that represents a very large message
   * in the simulated world, or the opposite.
   */
  CommPtr set_src_data_size(size_t size);
  /** Specify the data to send and its size. Don't mix the size with @ref Activity::set_remaining()
   *
   * This is way will get actually copied over to the receiver.
   * That's completely unrelated from the simulated size (given with @ref Activity::set_remaining()):
   * you can send a short buffer in your simulator, that represents a very large message
   * in the simulated world, or the opposite.
   */
  CommPtr set_src_data(void* buff, size_t size);

  /** Specify where to receive the data.
   *
   * That's a buffer where the sent data will be copied */
  CommPtr set_dst_data(void** buff);
  /** Specify the buffer in which the data should be received
   *
   * That's a buffer where the sent data will be copied  */
  CommPtr set_dst_data(void** buff, size_t size);

  CommPtr set_tracing_category(const std::string& category);

  /** Retrieve the mailbox on which this comm acts */
  Mailbox* get_mailbox();
  /** Retrieve the size of the received data. Not to be mixed with @ref Activity::set_remaining()  */
  size_t get_dst_data_size();

  Actor* get_sender();

#ifndef DOXYGEN
  XBT_ATTRIB_DEPRECATED_v324("Please use Comm::wait_for()") void wait(double t) override { wait_for(t); }
#endif
};
} // namespace s4u
} // namespace simgrid

#endif /* SIMGRID_S4U_COMM_HPP */
