/* cunit - A little C Unit facility                                         */

/* Copyright (c) 2005-2017. The SimGrid Team. All rights reserved.          */

/* This program is free software; you can redistribute it and/or modify it
 * under the terms of the license (GNU LGPL) which comes with this package. */

/* This is partially inspirated from the OSSP ts (Test Suite Library)       */
/* At some point we should use https://github.com/google/googletest instead */

#include "src/internal_config.h"
#include <cstdio>

#include <xbt/ex.hpp>
#include "xbt/sysdep.h"         /* bvprintf */
#include "xbt/cunit.h"
#include "xbt/dynar.h"

#define STRLEN 1024

/* collection of all suites */
static xbt_dynar_t _xbt_test_suites = nullptr;
/* global statistics */
static int _xbt_test_nb_tests = 0;
static int _xbt_test_test_failed = 0;
static int _xbt_test_test_ignore = 0;
static int _xbt_test_test_expect = 0;

static int _xbt_test_nb_units = 0;
static int _xbt_test_unit_failed = 0;
static int _xbt_test_unit_ignore = 0;
static int _xbt_test_unit_disabled = 0;

static int _xbt_test_nb_suites = 0;
static int _xbt_test_suite_failed = 0;
static int _xbt_test_suite_ignore = 0;
static int _xbt_test_suite_disabled = 0;

/* Context */
xbt_test_unit_t _xbt_test_current_unit = nullptr;

/* test suite test log */
typedef struct s_xbt_test_log {
  char *text;
  const char *file;
  int line;
} *xbt_test_log_t;

static void xbt_test_log_dump(xbt_test_log_t log)
{
  if (log)
    fprintf(stderr, "      log %p(%s:%d)=%s\n", log, log->file, log->line, log->text);
  else
    fprintf(stderr, "      log=nullptr\n");
}

/* test suite test check */
typedef struct s_xbt_test_test {
  char *title;
  int failed;
  int expected_failure;
  int ignored;
  const char *file;
  int line;
  xbt_dynar_t logs;
} *xbt_test_test_t;

static void xbt_test_test_dump(xbt_test_test_t test)
{
  if (test) {
    xbt_test_log_t log;
    unsigned int it_log;
    fprintf(stderr, "    test %p(%s:%d)=%s (%s)\n", test, test->file, test->line, test->title,
            test->failed ? "failed" : "not failed");
    xbt_dynar_foreach(test->logs, it_log, log)
        xbt_test_log_dump(log);
  } else
    fprintf(stderr, "    test=nullptr\n");
}

/* test suite test unit */
struct s_xbt_test_unit {
  int enabled;
  char *name;
  char *title;
  ts_test_cb_t func;
  const char *file;
  int line;
  xbt_dynar_t tests;            /* of xbt_test_test_t */

  int nb_tests;
  int test_failed;
  int test_ignore;
  int test_expect;
};

static void xbt_test_unit_dump(xbt_test_unit_t unit)
{
  if (unit) {
    fprintf(stderr, "  UNIT %s: %s (%s)\n", unit->name, unit->title, (unit->enabled ? "enabled" : "disabled"));
    if (unit->enabled) {
      xbt_test_test_t test;
      unsigned int it_test;
      xbt_dynar_foreach(unit->tests, it_test, test)
          xbt_test_test_dump(test);
    }
  } else {
    fprintf(stderr, "  unit=nullptr\n");
  }
}

/* test suite */
struct s_xbt_test_suite {
  int enabled;
  const char *name;
  char *title;
  xbt_dynar_t units;            /* of xbt_test_unit_t */

  int nb_tests;
  int nb_units;
  int test_failed;
  int test_ignore;
  int test_expect;
  int unit_failed;
  int unit_ignore;
  int unit_disabled;
};

/* destroy test suite */
static void xbt_test_suite_free(void *s)
{
  xbt_test_suite_t suite = *(xbt_test_suite_t *) s;

  if (suite == nullptr)
    return;
  xbt_dynar_free(&suite->units);
  free(suite->title);
  free(suite);
}

static void xbt_test_unit_free(void *unit)
{
  xbt_test_unit_t u = *(xbt_test_unit_t *) unit;
  /* name is static */
  free(u->title);
  xbt_dynar_free(&u->tests);
  free(u);
}

static void xbt_test_test_free(void *test)
{
  xbt_test_test_t t = *(xbt_test_test_t *) test;
  free(t->title);
  xbt_dynar_free(&(t->logs));
  free(t);
}

static void xbt_test_log_free(void *log)
{
  xbt_test_log_t l = *(xbt_test_log_t *) log;
  free(l->text);
  free(l);
}

/** @brief create test suite */
xbt_test_suite_t xbt_test_suite_new(const char *name, const char *fmt, ...)
{
  xbt_test_suite_t suite = xbt_new0(struct s_xbt_test_suite, 1);
  va_list ap;

  if (_xbt_test_suites == nullptr)
    _xbt_test_suites = xbt_dynar_new(sizeof(xbt_test_suite_t), xbt_test_suite_free);

  va_start(ap, fmt);
  suite->title = bvprintf(fmt, ap);
  suite->units = xbt_dynar_new(sizeof(xbt_test_unit_t), &xbt_test_unit_free);
  va_end(ap);
  suite->name = name;
  suite->enabled = 1;

  xbt_dynar_push(_xbt_test_suites, &suite);

  return suite;
}

/** @brief retrieve a testsuite from name, or create a new one */
xbt_test_suite_t xbt_test_suite_by_name(const char *name, const char *fmt, ...)
{
  xbt_test_suite_t suite;
  char *bufname;
  va_list ap;

  if (_xbt_test_suites) {
    unsigned int it_suite;
    xbt_dynar_foreach(_xbt_test_suites, it_suite, suite)
      if (not strcmp(suite->name, name))
        return suite;
  }

  va_start(ap, fmt);
  bufname = bvprintf(fmt, ap);
  va_end(ap);
  suite = xbt_test_suite_new(name, bufname, nullptr);
  free(bufname);

  return suite;
}

void xbt_test_suite_dump(xbt_test_suite_t suite)
{
  if (suite) {
    fprintf(stderr, "TESTSUITE %s: %s (%s)\n", suite->name, suite->title, suite->enabled ? "enabled" : "disabled");
    if (suite->enabled) {
      xbt_test_unit_t unit;
      unsigned int it_unit;
      xbt_dynar_foreach(suite->units, it_unit, unit)
          xbt_test_unit_dump(unit);
    }
  } else {
    fprintf(stderr, "TESTSUITE IS NULL!\n");
  }
}

/* add test case to test suite */
void xbt_test_suite_push(xbt_test_suite_t suite, const char *name, ts_test_cb_t func, const char *fmt, ...)
{
  xbt_test_unit_t unit;
  va_list ap;

  xbt_assert(suite);
  xbt_assert(func);
  xbt_assert(fmt);

  unit = xbt_new0(struct s_xbt_test_unit, 1);
  va_start(ap, fmt);
  unit->title = bvprintf(fmt, ap);
  va_end(ap);
  unit->name = (char *) name;
  unit->func = func;
  unit->file = nullptr;
  unit->line = 0;
  unit->enabled = 1;
  unit->tests = xbt_dynar_new(sizeof(xbt_test_test_t), xbt_test_test_free);

  xbt_dynar_push(suite->units, &unit);
}

/* run test one suite */
static int xbt_test_suite_run(xbt_test_suite_t suite, int verbosity)
{
  xbt_test_unit_t unit;
  xbt_test_test_t test;
  xbt_test_log_t log;

  if (suite == nullptr)
    return 0;

  /* suite title pretty-printing */
  char suite_title[81];
  int suite_len = strlen(suite->title);

  xbt_assert(suite_len < 68, "suite title \"%s\" too long (%d should be less than 68", suite->title, suite_len);

  suite_title[0] = ' ';
  for (int i = 1; i < 79; i++)
    suite_title[i] = '=';
  suite_title[79]  = '\n';
  suite_title[80]  = '\0';

  snprintf(suite_title + 40 - (suite_len + 4) / 2, 81 - (40 - (suite_len + 4) / 2), "[ %s ]", suite->title);
  suite_title[40 + (suite_len + 5) / 2] = '=';
  if (not suite->enabled)
    snprintf(suite_title + 70, 11, " DISABLED ");
  fprintf(stderr, "\n%s\n", suite_title);

  if (suite->enabled) {
    /* iterate through all tests */
    unsigned int it_unit;
    xbt_dynar_foreach(suite->units, it_unit, unit) {
      /* init unit case counters */
      unit->nb_tests = 0;
      unit->test_ignore = 0;
      unit->test_failed = 0;
      unit->test_expect = 0;

      /* display unit title */
      char* cp = bprintf(" Unit: %s ......................................"
                         "......................................",
                         unit->title);
      cp[70] = '\0';
      fprintf(stderr, "%s", cp);
      free(cp);

      /* run the test case function */
      _xbt_test_current_unit = unit;
      if (unit->enabled)
        unit->func();

      /* iterate through all performed tests to determine status */
      unsigned int it_test;
      xbt_dynar_foreach(unit->tests, it_test, test) {
        if (test->ignored) {
          unit->test_ignore++;
        } else {
          unit->nb_tests++;

          if (test->failed && not test->expected_failure)
            unit->test_failed++;
          if (not test->failed && test->expected_failure)
            unit->test_failed++;
          if (test->expected_failure)
            unit->test_expect++;
        }
      }
      /* Display whether this unit went well */
      if (unit->test_failed > 0 || unit->test_expect || (verbosity && unit->nb_tests > 0)) {
        /* some tests failed (or were supposed to), so do detailed reporting of test case */
        if (unit->test_failed > 0) {
          fprintf(stderr, ".. failed\n");
        } else if (unit->nb_tests) {
          fprintf(stderr, "...... ok\n");       /* successful, but show about expected */
        } else {
          fprintf(stderr, ".... skip\n");       /* shouldn't happen, but I'm a bit lost with this logic */
        }
        xbt_dynar_foreach(unit->tests, it_test, test) {
          const char* file = (test->file != nullptr ? test->file : unit->file);
          int line         = (test->line != 0 ? test->line : unit->line);
          const char* resname;
          if (test->ignored)
            resname = " SKIP";
          else if (test->expected_failure) {
            if (test->failed)
              resname = "EFAIL";
            else
              resname = "EPASS";
          } else {
            if (test->failed)
              resname = " FAIL";
            else
              resname = " PASS";
          }
          fprintf(stderr, "      %s: %s [%s:%d]\n", resname, test->title, file, line);

          if ((test->expected_failure && not test->failed) || (not test->expected_failure && test->failed)) {
            unsigned int it_log;
            xbt_dynar_foreach(test->logs, it_log, log) {
              file = (log->file != nullptr ? log->file : file);
              line = (log->line != 0 ? log->line : line);
              fprintf(stderr, "             %s:%d: %s\n", file, line, log->text);
            }
          }
        }
        fprintf(stderr, "    Summary: %d of %d tests failed", unit->test_failed, unit->nb_tests);
        if (unit->test_ignore) {
          fprintf(stderr, " (%d tests ignored)\n", unit->test_ignore);
        } else {
          fprintf(stderr, "\n");
        }
      } else if (not unit->enabled) {
        fprintf(stderr, " disabled\n"); /* no test were run */
      } else if (unit->nb_tests) {
        fprintf(stderr, "...... ok\n"); /* successful */
      } else {
        fprintf(stderr, ".... skip\n"); /* no test were run */
      }

      /* Accumulate test counts into the suite */
      suite->nb_tests += unit->nb_tests;
      suite->test_failed += unit->test_failed;
      suite->test_ignore += unit->test_ignore;
      suite->test_expect += unit->test_expect;

      _xbt_test_nb_tests += unit->nb_tests;
      _xbt_test_test_failed += unit->test_failed;
      _xbt_test_test_ignore += unit->test_ignore;
      _xbt_test_test_expect += unit->test_expect;

      /* What's the conclusion of this test anyway? */
      if (unit->nb_tests) {
        suite->nb_units++;
        if (unit->test_failed)
          suite->unit_failed++;
      } else if (not unit->enabled) {
        suite->unit_disabled++;
      } else {
        suite->unit_ignore++;
      }
    }
  }
  _xbt_test_nb_units += suite->nb_units;
  _xbt_test_unit_failed += suite->unit_failed;
  _xbt_test_unit_ignore += suite->unit_ignore;
  _xbt_test_unit_disabled += suite->unit_disabled;

  if (suite->nb_units) {
    _xbt_test_nb_suites++;
    if (suite->test_failed)
      _xbt_test_suite_failed++;
  } else if (not suite->enabled) {
    _xbt_test_suite_disabled++;
  } else {
    _xbt_test_suite_ignore++;
  }

  /* print test suite summary */
  if (suite->enabled) {
    int first = 1; /* for result pretty printing */

    fprintf(stderr," =====================================================================%s\n",
            (suite->nb_units ? (suite->unit_failed ? "== FAILED" : "====== OK") :
                               (suite->unit_disabled ? " DISABLED" : "==== SKIP")));
    fprintf(stderr, " Summary: Units: %.0f%% ok (%d units: ", suite->nb_units
            ? ((1 - (double) suite->unit_failed / (double) suite->nb_units) * 100.0) : 100.0, suite->nb_units);

    if (suite->nb_units != suite->unit_failed) {
      fprintf(stderr, "%s%d ok", (first ? "" : ", "), suite->nb_units - suite->unit_failed);
      first = 0;
    }
    if (suite->unit_failed) {
      fprintf(stderr, "%s%d failed", (first ? "" : ", "), suite->unit_failed);
      first = 0;
    }
    if (suite->unit_ignore) {
      fprintf(stderr, "%s%d ignored", (first ? "" : ", "), suite->unit_ignore);
      first = 0;
    }
    if (suite->unit_disabled) {
      fprintf(stderr, "%s%d disabled", (first ? "" : ", "), suite->unit_disabled);
    }
    fprintf(stderr, ")\n          Tests: %.0f%% ok (%d tests: ", suite->nb_tests
            ? ((1 - (double) suite->test_failed / (double) suite->nb_tests) * 100.0) : 100.0, suite->nb_tests);

    first = 1;
    if (suite->nb_tests != suite->test_failed) {
      fprintf(stderr, "%s%d ok", (first ? "" : ", "), suite->nb_tests - suite->test_failed);
      first = 0;
    }
    if (suite->test_failed) {
      fprintf(stderr, "%s%d failed", (first ? "" : ", "), suite->test_failed);
      first = 0;
    }
    if (suite->test_ignore) {
      fprintf(stderr, "%s%d ignored", (first ? "" : "; "), suite->test_ignore);
      first = 0;
    }
    if (suite->test_expect) {
      fprintf(stderr, "%s%d expected to fail", (first ? "" : "; "), suite->test_expect);
    }
    fprintf(stderr, ")\n");
  }
  return suite->unit_failed;
}

static void apply_selection(char *selection)
{
  /* for the parsing */
  char *sel = selection;
  int done = 0;
  char dir[STRLEN]; /* the directive */
  /* iterators */
  unsigned int it_suite;
  xbt_test_suite_t suite;
  xbt_test_unit_t unit;
  unsigned int it_unit;

  char suitename[STRLEN];
  char unitname[STRLEN];

  if (not selection || selection[0] == '\0')
    return;

  /* First apply the selection */
  while (not done) {
    int enabling = 1;

    char *p = strchr(sel, ',');
    if (p) {
      snprintf(dir, STRLEN, "%.*s", (int)(p - sel), sel);
      sel = p + 1;
    } else {
      snprintf(dir, STRLEN, "%s", sel);
      done = 1;
    }

    if (dir[0] == '-') {
      enabling = 0;
      memmove(dir, dir + 1, strlen(dir));
    }
    if (dir[0] == '+') {
      enabling = 1;
      memmove(dir, dir + 1, strlen(dir));
    }

    p = strchr(dir, ':');
    if (p) {
      snprintf(suitename, STRLEN, "%.*s", (int)(p - dir), dir);
      snprintf(unitname, STRLEN, "%s", p + 1);
    } else {
      snprintf(suitename, STRLEN, "%s", dir);
      unitname[0] = '\0';
    }

    /* Deal with the specific case of 'all' pseudo serie */
    if (not strcmp("all", suitename)) {
      xbt_assert(unitname[0] == '\0', "The 'all' pseudo-suite does not accept any unit specification\n");

      xbt_dynar_foreach(_xbt_test_suites, it_suite, suite) {
        xbt_dynar_foreach(suite->units, it_unit, unit) {
          unit->enabled = enabling;
        }
        suite->enabled = enabling;
      }
    } else {
      unsigned int it;
      for (it = 0; it < xbt_dynar_length(_xbt_test_suites); it++) {
        xbt_test_suite_t thissuite =
            xbt_dynar_get_as(_xbt_test_suites, it, xbt_test_suite_t);
        if (not strcmp(suitename, thissuite->name)) {
          /* Do not disable the whole suite when we just want to disable a child */
          if (enabling || (unitname[0] == '\0'))
            thissuite->enabled = enabling;

          if (unitname[0] == '\0') {
            xbt_dynar_foreach(thissuite->units, it_unit, unit) {
              unit->enabled = enabling;
            }
          } else {              /* act on one child only */
            unsigned int it2_unit;
            /* search it, first (we won't reuse it for external loop which gets broken) */
            for (it2_unit = 0;
                 it2_unit < xbt_dynar_length(thissuite->units);
                 it2_unit++) {
              xbt_test_unit_t thisunit = xbt_dynar_get_as(thissuite->units, it2_unit, xbt_test_unit_t);
              if (not strcmp(thisunit->name, unitname)) {
                thisunit->enabled = enabling;
                break;
              }
            }                   /* search relevant unit */
            xbt_assert(it2_unit != xbt_dynar_length(thissuite->units),
                "Suite '%s' has no unit of name '%s'. Cannot apply the selection\n", suitename, unitname);
          }                     /* act on childs (either all or one) */

          break;                /* found the relevant serie. We are happy */
        }
      }                         /* search relevant series */
      xbt_assert(it != xbt_dynar_length(_xbt_test_suites),
                 "No suite of name '%s' found. Cannot apply the selection\n", suitename);
    }
  }
}

void xbt_test_dump(char *selection)
{
  apply_selection(selection);

  if (_xbt_test_suites) {
    unsigned int it_suite;
    xbt_test_suite_t suite;

    xbt_dynar_foreach(_xbt_test_suites, it_suite, suite)
        xbt_test_suite_dump(suite);
  } else {
    printf(" No suite defined.");
  }
}

int xbt_test_run(char *selection, int verbosity)
{
  apply_selection(selection);

  if (_xbt_test_suites) {
    unsigned int it_suite;
    xbt_test_suite_t suite;
    int first = 1;

    /* Run all the suites */
    xbt_dynar_foreach(_xbt_test_suites, it_suite, suite)
      xbt_test_suite_run(suite, verbosity);

    /* Display some more statistics */
    fprintf(stderr, "\n\n TOTAL: Suites: %.0f%% ok (%d suites: ",_xbt_test_nb_suites
            ? ((1 - (double) _xbt_test_suite_failed / (double) _xbt_test_nb_suites) * 100.0)
            : 100.0, _xbt_test_nb_suites);
    if (_xbt_test_nb_suites != _xbt_test_suite_failed) {
      fprintf(stderr, "%d ok", _xbt_test_nb_suites - _xbt_test_suite_failed);
      first = 0;
    }
    if (_xbt_test_suite_failed) {
      fprintf(stderr, "%s%d failed", (first ? "" : ", "), _xbt_test_suite_failed);
      first = 0;
    }

    if (_xbt_test_suite_ignore) {
      fprintf(stderr, "%s%d ignored", (first ? "" : ", "), _xbt_test_suite_ignore);
    }
    fprintf(stderr, ")\n        Units:  %.0f%% ok (%d units: ", _xbt_test_nb_units
            ? ((1 - (double) _xbt_test_unit_failed / (double) _xbt_test_nb_units) * 100.0) : 100.0, _xbt_test_nb_units);
    first = 1;
    if (_xbt_test_nb_units != _xbt_test_unit_failed) {
      fprintf(stderr, "%d ok", _xbt_test_nb_units - _xbt_test_unit_failed);
      first = 0;
    }
    if (_xbt_test_unit_failed) {
      fprintf(stderr, "%s%d failed", (first ? "" : ", "), _xbt_test_unit_failed);
      first = 0;
    }
    if (_xbt_test_unit_ignore) {
      fprintf(stderr, "%s%d ignored", (first ? "" : ", "), _xbt_test_unit_ignore);
    }
    fprintf(stderr, ")\n        Tests:  %.0f%% ok (%d tests: ", _xbt_test_nb_tests
            ? ((1 - (double) _xbt_test_test_failed / (double) _xbt_test_nb_tests) * 100.0) : 100.0, _xbt_test_nb_tests);
    first = 1;
    if (_xbt_test_nb_tests != _xbt_test_test_failed) {
      fprintf(stderr, "%d ok", _xbt_test_nb_tests - _xbt_test_test_failed);
      first = 0;
    }
    if (_xbt_test_test_failed) {
      fprintf(stderr, "%s%d failed", (first ? "" : ", "), _xbt_test_test_failed);
      first = 0;
    }
    if (_xbt_test_test_ignore) {
      fprintf(stderr, "%s%d ignored", (first ? "" : ", "), _xbt_test_test_ignore);
      first = 0;
    }
    if (_xbt_test_test_expect) {
      fprintf(stderr, "%s%d expected to fail", (first ? "" : ", "), _xbt_test_test_expect);
    }

    fprintf(stderr, ")\n");
  } else {
    fprintf(stderr, "No unit to run!\n");
    _xbt_test_unit_failed++;
  }
  return _xbt_test_unit_failed;
}

void xbt_test_exit()
{
  xbt_dynar_free(&_xbt_test_suites);
}

/* annotate test case with test */
void _xbt_test_add(const char *file, int line, const char *fmt, ...)
{
  xbt_test_unit_t unit = _xbt_test_current_unit;
  xbt_assert(unit);

  va_list ap;
  xbt_test_test_t test = xbt_new0(struct s_xbt_test_test, 1);
  va_start(ap, fmt);
  test->title = bvprintf(fmt, ap);
  va_end(ap);
  test->failed = 0;
  test->expected_failure = 0;
  test->ignored = 0;
  test->file = file;
  test->line = line;
  test->logs = xbt_dynar_new(sizeof(xbt_test_log_t), xbt_test_log_free);
  xbt_dynar_push(unit->tests, &test);
}

/* annotate test case with log message and failure */
void _xbt_test_fail(const char *file, int line, const char *fmt, ...)
{
  xbt_test_unit_t unit = _xbt_test_current_unit;
  xbt_assert(unit);
  xbt_assert(xbt_dynar_length(_xbt_test_current_unit->tests),
      "Test failed even before being declared (broken unit: %s)", unit->title);

  va_list ap;
  xbt_test_log_t log = xbt_new(struct s_xbt_test_log, 1);
  va_start(ap, fmt);
  log->text = bvprintf(fmt, ap);
  va_end(ap);
  log->file = file;
  log->line = line;

  xbt_test_test_t test = xbt_dynar_getlast_as(unit->tests, xbt_test_test_t);
  xbt_dynar_push(test->logs, &log);

  test->failed = 1;
}

void xbt_test_exception(xbt_ex_t e)
{
  _xbt_test_fail(e.throwPoint().file, e.throwPoint().line, "Exception %s raised: %s", xbt_ex_catname(e.category), e.what());
}

void xbt_test_expect_failure()
{
  xbt_assert(xbt_dynar_length(_xbt_test_current_unit->tests),
      "Cannot expect the failure of a test before declaring it (broken unit: %s)", _xbt_test_current_unit->title);
  xbt_test_test_t test = xbt_dynar_getlast_as(_xbt_test_current_unit->tests, xbt_test_test_t);
  test->expected_failure = 1;
}

void xbt_test_skip()
{
  xbt_assert(xbt_dynar_length(_xbt_test_current_unit->tests),
      "Test skipped even before being declared (broken unit: %s)", _xbt_test_current_unit->title);
  xbt_test_test_t test = xbt_dynar_getlast_as(_xbt_test_current_unit->tests, xbt_test_test_t);
  test->ignored = 1;
}

/* annotate test case with log message only */
void _xbt_test_log(const char *file, int line, const char *fmt, ...)
{
  xbt_test_unit_t unit = _xbt_test_current_unit;
  xbt_assert(unit);
  xbt_assert(xbt_dynar_length(_xbt_test_current_unit->tests),
      "Test logged into even before being declared (broken test unit: %s)", unit->title);

  va_list ap;
  xbt_test_log_t log = xbt_new(struct s_xbt_test_log, 1);
  va_start(ap, fmt);
  log->text = bvprintf(fmt, ap);
  va_end(ap);
  log->file = file;
  log->line = line;

  xbt_test_test_t test = xbt_dynar_getlast_as(unit->tests, xbt_test_test_t);
  xbt_dynar_push(test->logs, &log);
}

#ifdef SIMGRID_TEST
XBT_TEST_SUITE("cunit", "Testsuite mechanism autotest");

XBT_TEST_UNIT("expect", test_expected_failure, "expected failures")
{
  xbt_test_add("Skipped test");
  xbt_test_skip();

  xbt_test_add("%s %s", "EXPECTED", "FAILURE");
  xbt_test_expect_failure();
  xbt_test_log("%s %s", "Test", "log");
  xbt_test_fail("EXPECTED FAILURE");
}
#endif                          /* SIMGRID_TEST */
