/* Copyright (c) 2018, 2019 Evan Nemerson <evan@nemerson.com>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#define SIMDE_TESTS_CURRENT_ISAX avx2
#if !defined(__clang__) && (defined(__linux__) || defined(__linux) || defined(__gnu_linux__)) && !defined(_GNU_SOURCE)
  #define _GNU_SOURCE 1  // for MAP_ANONYMOUS
#endif
#include <simde/x86/avx2.h>
#include <test/x86/test-avx.h>
#if !defined(HEDLEY_MSVC_VERSION)
  #include <sys/mman.h>
#endif

static int
test_simde_mm256_abs_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C( -27), INT8_C(  88), INT8_C(-122), INT8_C(  -6),
                           INT8_C( -23), INT8_C( 108), INT8_C(-103), INT8_C(  32),
                           INT8_C(  43), INT8_C( 116), INT8_C(  -6), INT8_C( -98),
                           INT8_C( -62), INT8_C( -87), INT8_C(  90), INT8_C(  82),
                           INT8_C(  86), INT8_C(   8), INT8_C(-126), INT8_C( -22),
                           INT8_C( -80), INT8_C(-125), INT8_C(  -5), INT8_C(-101),
                           INT8_C(  36), INT8_C( 114), INT8_C( -51), INT8_C(  59),
                           INT8_C( -97), INT8_C( 124), INT8_C(  25), INT8_C(  90)),
      simde_mm256_set_epi8(INT8_C(  27), INT8_C(  88), INT8_C( 122), INT8_C(   6),
                           INT8_C(  23), INT8_C( 108), INT8_C( 103), INT8_C(  32),
                           INT8_C(  43), INT8_C( 116), INT8_C(   6), INT8_C(  98),
                           INT8_C(  62), INT8_C(  87), INT8_C(  90), INT8_C(  82),
                           INT8_C(  86), INT8_C(   8), INT8_C( 126), INT8_C(  22),
                           INT8_C(  80), INT8_C( 125), INT8_C(   5), INT8_C( 101),
                           INT8_C(  36), INT8_C( 114), INT8_C(  51), INT8_C(  59),
                           INT8_C(  97), INT8_C( 124), INT8_C(  25), INT8_C(  90)) },
    { simde_mm256_set_epi8(INT8_C( 111), INT8_C(  46), INT8_C( -44), INT8_C(  36),
                           INT8_C( -79), INT8_C( 101), INT8_C(   0), INT8_C(   2),
                           INT8_C( -69), INT8_C(  31), INT8_C( -68), INT8_C( -82),
                           INT8_C( -45), INT8_C( 120), INT8_C(  39), INT8_C(  46),
                           INT8_C(  66), INT8_C(  30), INT8_C(-106), INT8_C( 118),
                           INT8_C(  61), INT8_C(  98), INT8_C( -61), INT8_C(  98),
                           INT8_C(  49), INT8_C( -12), INT8_C(-117), INT8_C(-115),
                           INT8_C(  63), INT8_C( -92), INT8_C(-102), INT8_C(-110)),
      simde_mm256_set_epi8(INT8_C( 111), INT8_C(  46), INT8_C(  44), INT8_C(  36),
                           INT8_C(  79), INT8_C( 101), INT8_C(   0), INT8_C(   2),
                           INT8_C(  69), INT8_C(  31), INT8_C(  68), INT8_C(  82),
                           INT8_C(  45), INT8_C( 120), INT8_C(  39), INT8_C(  46),
                           INT8_C(  66), INT8_C(  30), INT8_C( 106), INT8_C( 118),
                           INT8_C(  61), INT8_C(  98), INT8_C(  61), INT8_C(  98),
                           INT8_C(  49), INT8_C(  12), INT8_C( 117), INT8_C( 115),
                           INT8_C(  63), INT8_C(  92), INT8_C( 102), INT8_C( 110)) },
    { simde_mm256_set_epi8(INT8_C(  64), INT8_C( -84), INT8_C(  54), INT8_C(-102),
                           INT8_C( -69), INT8_C(  12), INT8_C(-119), INT8_C( -19),
                           INT8_C(  19), INT8_C( -55), INT8_C( -11), INT8_C(-117),
                           INT8_C( -68), INT8_C( -51), INT8_C(  26), INT8_C(  72),
                           INT8_C( -15), INT8_C( 108), INT8_C( -66), INT8_C( -24),
                           INT8_C( -97), INT8_C( -48), INT8_C(  75), INT8_C(  35),
                           INT8_C(  48), INT8_C( -25), INT8_C( -43), INT8_C(   2),
                           INT8_C( -75), INT8_C(  28), INT8_C(-108), INT8_C( -43)),
      simde_mm256_set_epi8(INT8_C(  64), INT8_C(  84), INT8_C(  54), INT8_C( 102),
                           INT8_C(  69), INT8_C(  12), INT8_C( 119), INT8_C(  19),
                           INT8_C(  19), INT8_C(  55), INT8_C(  11), INT8_C( 117),
                           INT8_C(  68), INT8_C(  51), INT8_C(  26), INT8_C(  72),
                           INT8_C(  15), INT8_C( 108), INT8_C(  66), INT8_C(  24),
                           INT8_C(  97), INT8_C(  48), INT8_C(  75), INT8_C(  35),
                           INT8_C(  48), INT8_C(  25), INT8_C(  43), INT8_C(   2),
                           INT8_C(  75), INT8_C(  28), INT8_C( 108), INT8_C(  43)) },
    { simde_mm256_set_epi8(INT8_C(   8), INT8_C( -54), INT8_C(  -1), INT8_C(-128),
                           INT8_C( 118), INT8_C( -15), INT8_C( 125), INT8_C(  76),
                           INT8_C(  47), INT8_C(  33), INT8_C(  69), INT8_C(  21),
                           INT8_C(-116), INT8_C(  34), INT8_C(  36), INT8_C(  31),
                           INT8_C( -32), INT8_C( -84), INT8_C(  23), INT8_C( -76),
                           INT8_C(  82), INT8_C(-115), INT8_C(  74), INT8_C(-110),
                           INT8_C( -46), INT8_C( 125), INT8_C( -52), INT8_C( -99),
                           INT8_C(  30), INT8_C(-106), INT8_C(  66), INT8_C(   5)),
      simde_mm256_set_epi8(INT8_C(   8), INT8_C(  54), INT8_C(   1), INT8_C(-128),
                           INT8_C( 118), INT8_C(  15), INT8_C( 125), INT8_C(  76),
                           INT8_C(  47), INT8_C(  33), INT8_C(  69), INT8_C(  21),
                           INT8_C( 116), INT8_C(  34), INT8_C(  36), INT8_C(  31),
                           INT8_C(  32), INT8_C(  84), INT8_C(  23), INT8_C(  76),
                           INT8_C(  82), INT8_C( 115), INT8_C(  74), INT8_C( 110),
                           INT8_C(  46), INT8_C( 125), INT8_C(  52), INT8_C(  99),
                           INT8_C(  30), INT8_C( 106), INT8_C(  66), INT8_C(   5)) },
    { simde_mm256_set_epi8(INT8_C( 122), INT8_C(  42), INT8_C(-121), INT8_C(-106),
                           INT8_C( 122), INT8_C(  -8), INT8_C(  81), INT8_C(-109),
                           INT8_C( 124), INT8_C(  32), INT8_C(  63), INT8_C( -21),
                           INT8_C( -51), INT8_C( -42), INT8_C(   1), INT8_C( -78),
                           INT8_C(  74), INT8_C(   8), INT8_C(  25), INT8_C(  10),
                           INT8_C( 113), INT8_C( -75), INT8_C( -32), INT8_C( 126),
                           INT8_C( -87), INT8_C(  67), INT8_C(  78), INT8_C( -64),
                           INT8_C(   7), INT8_C( -40), INT8_C( -46), INT8_C( -59)),
      simde_mm256_set_epi8(INT8_C( 122), INT8_C(  42), INT8_C( 121), INT8_C( 106),
                           INT8_C( 122), INT8_C(   8), INT8_C(  81), INT8_C( 109),
                           INT8_C( 124), INT8_C(  32), INT8_C(  63), INT8_C(  21),
                           INT8_C(  51), INT8_C(  42), INT8_C(   1), INT8_C(  78),
                           INT8_C(  74), INT8_C(   8), INT8_C(  25), INT8_C(  10),
                           INT8_C( 113), INT8_C(  75), INT8_C(  32), INT8_C( 126),
                           INT8_C(  87), INT8_C(  67), INT8_C(  78), INT8_C(  64),
                           INT8_C(   7), INT8_C(  40), INT8_C(  46), INT8_C(  59)) },
    { simde_mm256_set_epi8(INT8_C(  10), INT8_C( 120), INT8_C(  81), INT8_C(-105),
                           INT8_C(  73), INT8_C( -95), INT8_C(  79), INT8_C( -86),
                           INT8_C( -93), INT8_C( -54), INT8_C( -43), INT8_C( -88),
                           INT8_C(  59), INT8_C( -27), INT8_C(  12), INT8_C(  10),
                           INT8_C(  73), INT8_C( -48), INT8_C( 112), INT8_C(  27),
                           INT8_C(-113), INT8_C( -31), INT8_C( -56), INT8_C( -96),
                           INT8_C(  48), INT8_C( -94), INT8_C(-111), INT8_C(  60),
                           INT8_C(-116), INT8_C( -77), INT8_C( -70), INT8_C(  17)),
      simde_mm256_set_epi8(INT8_C(  10), INT8_C( 120), INT8_C(  81), INT8_C( 105),
                           INT8_C(  73), INT8_C(  95), INT8_C(  79), INT8_C(  86),
                           INT8_C(  93), INT8_C(  54), INT8_C(  43), INT8_C(  88),
                           INT8_C(  59), INT8_C(  27), INT8_C(  12), INT8_C(  10),
                           INT8_C(  73), INT8_C(  48), INT8_C( 112), INT8_C(  27),
                           INT8_C( 113), INT8_C(  31), INT8_C(  56), INT8_C(  96),
                           INT8_C(  48), INT8_C(  94), INT8_C( 111), INT8_C(  60),
                           INT8_C( 116), INT8_C(  77), INT8_C(  70), INT8_C(  17)) },
    { simde_mm256_set_epi8(INT8_C(  61), INT8_C( -57), INT8_C( -99), INT8_C(   0),
                           INT8_C(  98), INT8_C(-121), INT8_C(  67), INT8_C( -20),
                           INT8_C(  44), INT8_C(  53), INT8_C(-128), INT8_C(  44),
                           INT8_C( 127), INT8_C(  53), INT8_C(-127), INT8_C(  58),
                           INT8_C(  35), INT8_C(  83), INT8_C( -56), INT8_C(  22),
                           INT8_C(  -4), INT8_C(  -6), INT8_C(  -7), INT8_C( 121),
                           INT8_C( -22), INT8_C( -32), INT8_C( -52), INT8_C( 124),
                           INT8_C( -93), INT8_C(  55), INT8_C( -23), INT8_C( -62)),
      simde_mm256_set_epi8(INT8_C(  61), INT8_C(  57), INT8_C(  99), INT8_C(   0),
                           INT8_C(  98), INT8_C( 121), INT8_C(  67), INT8_C(  20),
                           INT8_C(  44), INT8_C(  53), INT8_C(-128), INT8_C(  44),
                           INT8_C( 127), INT8_C(  53), INT8_C( 127), INT8_C(  58),
                           INT8_C(  35), INT8_C(  83), INT8_C(  56), INT8_C(  22),
                           INT8_C(   4), INT8_C(   6), INT8_C(   7), INT8_C( 121),
                           INT8_C(  22), INT8_C(  32), INT8_C(  52), INT8_C( 124),
                           INT8_C(  93), INT8_C(  55), INT8_C(  23), INT8_C(  62)) },
    { simde_mm256_set_epi8(INT8_C(  71), INT8_C( -58), INT8_C(  24), INT8_C( 117),
                           INT8_C(   2), INT8_C( -31), INT8_C( -86), INT8_C( 101),
                           INT8_C(   3), INT8_C(  63), INT8_C(   2), INT8_C( -30),
                           INT8_C( -33), INT8_C(  51), INT8_C(  60), INT8_C(  81),
                           INT8_C( -91), INT8_C( -73), INT8_C(  66), INT8_C(  67),
                           INT8_C(  72), INT8_C(  -7), INT8_C(  44), INT8_C( -32),
                           INT8_C( -80), INT8_C( 101), INT8_C( -98), INT8_C(  89),
                           INT8_C(  89), INT8_C(  94), INT8_C( 109), INT8_C(-109)),
      simde_mm256_set_epi8(INT8_C(  71), INT8_C(  58), INT8_C(  24), INT8_C( 117),
                           INT8_C(   2), INT8_C(  31), INT8_C(  86), INT8_C( 101),
                           INT8_C(   3), INT8_C(  63), INT8_C(   2), INT8_C(  30),
                           INT8_C(  33), INT8_C(  51), INT8_C(  60), INT8_C(  81),
                           INT8_C(  91), INT8_C(  73), INT8_C(  66), INT8_C(  67),
                           INT8_C(  72), INT8_C(   7), INT8_C(  44), INT8_C(  32),
                           INT8_C(  80), INT8_C( 101), INT8_C(  98), INT8_C(  89),
                           INT8_C(  89), INT8_C(  94), INT8_C( 109), INT8_C( 109)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_abs_epi8(test_vec[i].a);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_abs_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(  9101), INT16_C( 13664), INT16_C( 14007), INT16_C( 17440),
                            INT16_C( 21201), INT16_C(-16892), INT16_C(-22702), INT16_C(-11875),
                            INT16_C(  9352), INT16_C( 21001), INT16_C( 15464), INT16_C( 27994),
                            INT16_C( 12104), INT16_C(-22404), INT16_C(-21433), INT16_C( -4031)),
      simde_mm256_set_epi16(INT16_C(  9101), INT16_C( 13664), INT16_C( 14007), INT16_C( 17440),
                            INT16_C( 21201), INT16_C( 16892), INT16_C( 22702), INT16_C( 11875),
                            INT16_C(  9352), INT16_C( 21001), INT16_C( 15464), INT16_C( 27994),
                            INT16_C( 12104), INT16_C( 22404), INT16_C( 21433), INT16_C(  4031)) },
    { simde_mm256_set_epi16(INT16_C( 20289), INT16_C( -8788), INT16_C(  5917), INT16_C(-28916),
                            INT16_C(-21540), INT16_C( -8179), INT16_C(-25392), INT16_C(  2609),
                            INT16_C( 12609), INT16_C(-11367), INT16_C(   -70), INT16_C( 28633),
                            INT16_C(-21576), INT16_C(-23753), INT16_C(-11797), INT16_C(-17346)),
      simde_mm256_set_epi16(INT16_C( 20289), INT16_C(  8788), INT16_C(  5917), INT16_C( 28916),
                            INT16_C( 21540), INT16_C(  8179), INT16_C( 25392), INT16_C(  2609),
                            INT16_C( 12609), INT16_C( 11367), INT16_C(    70), INT16_C( 28633),
                            INT16_C( 21576), INT16_C( 23753), INT16_C( 11797), INT16_C( 17346)) },
    { simde_mm256_set_epi16(INT16_C( 11563), INT16_C(-31585), INT16_C( 24583), INT16_C(  2918),
                            INT16_C(  5705), INT16_C( 31274), INT16_C(-12388), INT16_C( 31454),
                            INT16_C(  5008), INT16_C( 10123), INT16_C( 28874), INT16_C(-27636),
                            INT16_C(  1380), INT16_C(-32687), INT16_C( 24141), INT16_C( 11570)),
      simde_mm256_set_epi16(INT16_C( 11563), INT16_C( 31585), INT16_C( 24583), INT16_C(  2918),
                            INT16_C(  5705), INT16_C( 31274), INT16_C( 12388), INT16_C( 31454),
                            INT16_C(  5008), INT16_C( 10123), INT16_C( 28874), INT16_C( 27636),
                            INT16_C(  1380), INT16_C( 32687), INT16_C( 24141), INT16_C( 11570)) },
    { simde_mm256_set_epi16(INT16_C(-28981), INT16_C(-21254), INT16_C( 12206), INT16_C( 17751),
                            INT16_C(  4887), INT16_C( 27025), INT16_C( 20436), INT16_C( -3143),
                            INT16_C(  5806), INT16_C( 19398), INT16_C( 23890), INT16_C( -1841),
                            INT16_C( -1212), INT16_C(  -418), INT16_C(  2804), INT16_C(-24086)),
      simde_mm256_set_epi16(INT16_C( 28981), INT16_C( 21254), INT16_C( 12206), INT16_C( 17751),
                            INT16_C(  4887), INT16_C( 27025), INT16_C( 20436), INT16_C(  3143),
                            INT16_C(  5806), INT16_C( 19398), INT16_C( 23890), INT16_C(  1841),
                            INT16_C(  1212), INT16_C(   418), INT16_C(  2804), INT16_C( 24086)) },
    { simde_mm256_set_epi16(INT16_C(-32227), INT16_C( 26559), INT16_C( 32468), INT16_C(  9282),
                            INT16_C( 10212), INT16_C(  7157), INT16_C(-18109), INT16_C(-13716),
                            INT16_C(  3356), INT16_C( -6654), INT16_C(  3548), INT16_C(-31612),
                            INT16_C( -3226), INT16_C(-30156), INT16_C(-15323), INT16_C(  8689)),
      simde_mm256_set_epi16(INT16_C( 32227), INT16_C( 26559), INT16_C( 32468), INT16_C(  9282),
                            INT16_C( 10212), INT16_C(  7157), INT16_C( 18109), INT16_C( 13716),
                            INT16_C(  3356), INT16_C(  6654), INT16_C(  3548), INT16_C( 31612),
                            INT16_C(  3226), INT16_C( 30156), INT16_C( 15323), INT16_C(  8689)) },
    { simde_mm256_set_epi16(INT16_C( 14337), INT16_C(-20237), INT16_C(  7001), INT16_C( 29027),
                            INT16_C( -3029), INT16_C( 12894), INT16_C(-24482), INT16_C( -8195),
                            INT16_C( -7637), INT16_C(-26436), INT16_C( 15950), INT16_C(  5319),
                            INT16_C( 22977), INT16_C(  -593), INT16_C(-29639), INT16_C( 23312)),
      simde_mm256_set_epi16(INT16_C( 14337), INT16_C( 20237), INT16_C(  7001), INT16_C( 29027),
                            INT16_C(  3029), INT16_C( 12894), INT16_C( 24482), INT16_C(  8195),
                            INT16_C(  7637), INT16_C( 26436), INT16_C( 15950), INT16_C(  5319),
                            INT16_C( 22977), INT16_C(   593), INT16_C( 29639), INT16_C( 23312)) },
    { simde_mm256_set_epi16(INT16_C(  4249), INT16_C( -3888), INT16_C( 15630), INT16_C(-11095),
                            INT16_C(-21648), INT16_C(-10947), INT16_C( -1651), INT16_C(  5821),
                            INT16_C( 25032), INT16_C( 26383), INT16_C(-18726), INT16_C(-14746),
                            INT16_C(  9694), INT16_C(-29231), INT16_C( 18526), INT16_C(-12816)),
      simde_mm256_set_epi16(INT16_C(  4249), INT16_C(  3888), INT16_C( 15630), INT16_C( 11095),
                            INT16_C( 21648), INT16_C( 10947), INT16_C(  1651), INT16_C(  5821),
                            INT16_C( 25032), INT16_C( 26383), INT16_C( 18726), INT16_C( 14746),
                            INT16_C(  9694), INT16_C( 29231), INT16_C( 18526), INT16_C( 12816)) },
    { simde_mm256_set_epi16(INT16_C(  6410), INT16_C(  4746), INT16_C( 16873), INT16_C(-29607),
                            INT16_C( 21314), INT16_C(-32512), INT16_C(-23052), INT16_C( 20594),
                            INT16_C( -1613), INT16_C( 26993), INT16_C( 28325), INT16_C(   406),
                            INT16_C(-19031), INT16_C(  6060), INT16_C(-29650), INT16_C(  8164)),
      simde_mm256_set_epi16(INT16_C(  6410), INT16_C(  4746), INT16_C( 16873), INT16_C( 29607),
                            INT16_C( 21314), INT16_C( 32512), INT16_C( 23052), INT16_C( 20594),
                            INT16_C(  1613), INT16_C( 26993), INT16_C( 28325), INT16_C(   406),
                            INT16_C( 19031), INT16_C(  6060), INT16_C( 29650), INT16_C(  8164)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_abs_epi16(test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_abs_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(  596456800), INT32_C(  917980192), INT32_C( 1389477380), INT32_C(-1487744611),
                            INT32_C(  612913673), INT32_C( 1013476698), INT32_C(  793290876), INT32_C(-1404571583)),
      simde_mm256_set_epi32(INT32_C(  596456800), INT32_C(  917980192), INT32_C( 1389477380), INT32_C( 1487744611),
                            INT32_C(  612913673), INT32_C( 1013476698), INT32_C(  793290876), INT32_C( 1404571583)) },
    { simde_mm256_set_epi32(INT32_C( 1329716652), INT32_C(  387813132), INT32_C(-1411588083), INT32_C(-1664087503),
                            INT32_C(  826397593), INT32_C(   -4558887), INT32_C(-1413962953), INT32_C( -773080002)),
      simde_mm256_set_epi32(INT32_C( 1329716652), INT32_C(  387813132), INT32_C( 1411588083), INT32_C( 1664087503),
                            INT32_C(  826397593), INT32_C(    4558887), INT32_C( 1413962953), INT32_C(  773080002)) },
    { simde_mm256_set_epi32(INT32_C(  757826719), INT32_C( 1611074406), INT32_C(  373914154), INT32_C( -811828514),
                            INT32_C(  328214411), INT32_C( 1892324364), INT32_C(   90472529), INT32_C( 1582116146)),
      simde_mm256_set_epi32(INT32_C(  757826719), INT32_C( 1611074406), INT32_C(  373914154), INT32_C(  811828514),
                            INT32_C(  328214411), INT32_C( 1892324364), INT32_C(   90472529), INT32_C( 1582116146)) },
    { simde_mm256_set_epi32(INT32_C(-1899254534), INT32_C(  799950167), INT32_C(  320301457), INT32_C( 1339356089),
                            INT32_C(  380521414), INT32_C( 1565718735), INT32_C(  -79364514), INT32_C(  183804394)),
      simde_mm256_set_epi32(INT32_C( 1899254534), INT32_C(  799950167), INT32_C(  320301457), INT32_C( 1339356089),
                            INT32_C(  380521414), INT32_C( 1565718735), INT32_C(   79364514), INT32_C(  183804394)) },
    { simde_mm256_set_epi32(INT32_C(-2112002113), INT32_C( 2127832130), INT32_C(  669260789), INT32_C(-1186739604),
                            INT32_C(  219997698), INT32_C(  232555652), INT32_C( -211383756), INT32_C(-1004199439)),
      simde_mm256_set_epi32(INT32_C( 2112002113), INT32_C( 2127832130), INT32_C(  669260789), INT32_C( 1186739604),
                            INT32_C(  219997698), INT32_C(  232555652), INT32_C(  211383756), INT32_C( 1004199439)) },
    { simde_mm256_set_epi32(INT32_C(  939634931), INT32_C(  458846563), INT32_C( -198495650), INT32_C(-1604395011),
                            INT32_C( -500459332), INT32_C( 1045304519), INT32_C( 1505885615), INT32_C(-1942398192)),
      simde_mm256_set_epi32(INT32_C(  939634931), INT32_C(  458846563), INT32_C(  198495650), INT32_C( 1604395011),
                            INT32_C(  500459332), INT32_C( 1045304519), INT32_C( 1505885615), INT32_C( 1942398192)) },
    { simde_mm256_set_epi32(INT32_C(  278524112), INT32_C( 1024382121), INT32_C(-1418668739), INT32_C( -108194115),
                            INT32_C( 1640523535), INT32_C(-1227176346), INT32_C(  635342289), INT32_C( 1214172656)),
      simde_mm256_set_epi32(INT32_C(  278524112), INT32_C( 1024382121), INT32_C( 1418668739), INT32_C(  108194115),
                            INT32_C( 1640523535), INT32_C( 1227176346), INT32_C(  635342289), INT32_C( 1214172656)) },
    { simde_mm256_set_epi32(INT32_C(  420090506), INT32_C( 1105824857), INT32_C( 1396867328), INT32_C(-1510715278),
                            INT32_C( -105682575), INT32_C( 1856307606), INT32_C(-1247209556), INT32_C(-1943134236)),
      simde_mm256_set_epi32(INT32_C(  420090506), INT32_C( 1105824857), INT32_C( 1396867328), INT32_C( 1510715278),
                            INT32_C(  105682575), INT32_C( 1856307606), INT32_C( 1247209556), INT32_C( 1943134236)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_abs_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_add_epi8 (SIMDE_MUNIT_TEST_ARGS) {
  struct {
    int8_t a[32];
    int8_t b[32];
    int8_t r[32];
  } test_vec[] = {
    { { -INT8_C(  31),  INT8_C(  62), -INT8_C(  76),  INT8_C(   4), -INT8_C(  86),  INT8_C(  17),  INT8_C( 123),  INT8_C( 110),
         INT8_C(  62), -INT8_C(  91), -INT8_C( 124),  INT8_C( 104),  INT8_C(  48), -INT8_C(  40), -INT8_C( 103), -INT8_C(  47),
         INT8_C(  10), -INT8_C(  34), -INT8_C(  78),  INT8_C(  22),  INT8_C(  48),  INT8_C(  43),  INT8_C(  12),  INT8_C(  72),
         INT8_C(  27), -INT8_C( 105),  INT8_C(  29),  INT8_C(  82), -INT8_C(  83),  INT8_C(  94),  INT8_C(  31), -INT8_C( 114) },
      { -INT8_C(  99), -INT8_C(  44), -INT8_C( 109),  INT8_C(  71), -INT8_C(  27),  INT8_C(  14), -INT8_C(  75),  INT8_C(  36),
        -INT8_C(  77),  INT8_C(  58), -INT8_C( 116), -INT8_C(  29),  INT8_C(  18),  INT8_C(  38), -INT8_C(  76),  INT8_C(  28),
         INT8_C(   4),  INT8_C( 102),  INT8_C(  51),  INT8_C(  52), -INT8_C( 111),  INT8_C(  63),  INT8_C( 124), -INT8_C(  84),
        -INT8_C(  42), -INT8_C( 103), -INT8_C(   1), -INT8_C( 125), -INT8_C(   9),  INT8_C(  30),  INT8_C(  17), -INT8_C( 108) },
      {  INT8_C( 126),  INT8_C(  18),  INT8_C(  71),  INT8_C(  75), -INT8_C( 113),  INT8_C(  31),  INT8_C(  48), -INT8_C( 110),
        -INT8_C(  15), -INT8_C(  33),  INT8_C(  16),  INT8_C(  75),  INT8_C(  66), -INT8_C(   2),  INT8_C(  77), -INT8_C(  19),
         INT8_C(  14),  INT8_C(  68), -INT8_C(  27),  INT8_C(  74), -INT8_C(  63),  INT8_C( 106), -INT8_C( 120), -INT8_C(  12),
        -INT8_C(  15),  INT8_C(  48),  INT8_C(  28), -INT8_C(  43), -INT8_C(  92),  INT8_C( 124),  INT8_C(  48),  INT8_C(  34) } },
    { { -INT8_C(  14), -INT8_C(  92), -INT8_C(  37), -INT8_C(  40), -INT8_C(  78), -INT8_C( 111), -INT8_C(   4),  INT8_C( 101),
        -INT8_C(  53), -INT8_C( 120),  INT8_C(  73), -INT8_C(  35), -INT8_C(  82), -INT8_C(   3), -INT8_C(   7), -INT8_C(  78),
         INT8_C( 100),  INT8_C(  44), -INT8_C(  26), -INT8_C(  11),  INT8_C( 107),  INT8_C(  98), -INT8_C(  94),  INT8_C(  65),
        -INT8_C(   5), -INT8_C(  95), -INT8_C(  60), -INT8_C(  13), -INT8_C(  65), -INT8_C(  42), -INT8_C( 121), -INT8_C(  78) },
      {  INT8_C( 122),  INT8_C(  99), -INT8_C( 118),  INT8_C(  45), -INT8_C(  12), -INT8_C( 122), -INT8_C( 110), -INT8_C(  65),
         INT8_C(  14), -INT8_C(  37), -INT8_C( 100), -INT8_C(  67), -INT8_C(  39), -INT8_C( 107),  INT8_C( 111),  INT8_C(  61),
        -INT8_C(  62),  INT8_C(  86),  INT8_C(  50),  INT8_C(  45), -INT8_C(  72), -INT8_C(  44),  INT8_C( 111), -INT8_C(  76),
         INT8_C( 117),  INT8_C(  51), -INT8_C(  89),  INT8_C(  53),  INT8_C(   9),  INT8_C(  46), -INT8_C(  25), -INT8_C( 124) },
      {  INT8_C( 108),  INT8_C(   7),  INT8_C( 101),  INT8_C(   5), -INT8_C(  90),  INT8_C(  23), -INT8_C( 114),  INT8_C(  36),
        -INT8_C(  39),  INT8_C(  99), -INT8_C(  27), -INT8_C( 102), -INT8_C( 121), -INT8_C( 110),  INT8_C( 104), -INT8_C(  17),
         INT8_C(  38), -INT8_C( 126),  INT8_C(  24),  INT8_C(  34),  INT8_C(  35),  INT8_C(  54),  INT8_C(  17), -INT8_C(  11),
         INT8_C( 112), -INT8_C(  44),  INT8_C( 107),  INT8_C(  40), -INT8_C(  56),  INT8_C(   4),  INT8_C( 110),  INT8_C(  54) } },
    { { -INT8_C( 111),  INT8_C( 113), -INT8_C(  79), -INT8_C( 123), -INT8_C(   9),  INT8_C(  67),  INT8_C(  68),  INT8_C(   5),
         INT8_C(  31), -INT8_C(  32), -INT8_C(  62), -INT8_C(   8),  INT8_C( 118),  INT8_C(  50),  INT8_C(  53),  INT8_C(  56),
        -INT8_C( 120),  INT8_C( 103),  INT8_C( 101),  INT8_C(  64),  INT8_C(  60), -INT8_C(  44), -INT8_C(  12), -INT8_C(  79),
         INT8_C(   8), -INT8_C( 101), -INT8_C(  26),  INT8_C(  17), -INT8_C(  54), -INT8_C(  51), -INT8_C( 107),  INT8_C(  91) },
      {  INT8_C(  62),  INT8_C(  70), -INT8_C(  31),  INT8_C(  53), -INT8_C( 118),  INT8_C(  37),  INT8_C(  59), -INT8_C(  87),
         INT8_C(   6), -INT8_C(   3), -INT8_C(  95),  INT8_C( 124),  INT8_C(  47), -INT8_C(  42), -INT8_C(  76), -INT8_C(  73),
         INT8_C(  61),  INT8_C(  25), -INT8_C(   8),  INT8_C( 121), -INT8_C(  18), -INT8_C(  20),  INT8_C(  43), -INT8_C(  10),
        -INT8_C( 120),  INT8_C(  17),  INT8_C(   7),  INT8_C(  82), -INT8_C(  33), -INT8_C(  99), -INT8_C(  83),  INT8_C(  29) },
      { -INT8_C(  49), -INT8_C(  73), -INT8_C( 110), -INT8_C(  70), -INT8_C( 127),  INT8_C( 104),      INT8_MAX, -INT8_C(  82),
         INT8_C(  37), -INT8_C(  35),  INT8_C(  99),  INT8_C( 116), -INT8_C(  91),  INT8_C(   8), -INT8_C(  23), -INT8_C(  17),
        -INT8_C(  59),      INT8_MIN,  INT8_C(  93), -INT8_C(  71),  INT8_C(  42), -INT8_C(  64),  INT8_C(  31), -INT8_C(  89),
        -INT8_C( 112), -INT8_C(  84), -INT8_C(  19),  INT8_C(  99), -INT8_C(  87),  INT8_C( 106),  INT8_C(  66),  INT8_C( 120) } },
    { { -INT8_C(  29), -INT8_C( 114),  INT8_C(  83),  INT8_C( 109), -INT8_C(  76), -INT8_C( 114),  INT8_C(  22), -INT8_C(  70),
        -INT8_C( 117), -INT8_C(  73),  INT8_C(  54), -INT8_C(  69), -INT8_C( 115), -INT8_C(  22),  INT8_C( 114), -INT8_C(  53),
         INT8_C(   3),  INT8_C( 106),  INT8_C(  68), -INT8_C(  15),  INT8_C(  87),  INT8_C( 111), -INT8_C(  25), -INT8_C(  33),
        -INT8_C( 127), -INT8_C(  17),  INT8_C(  49),  INT8_C(  96), -INT8_C( 116), -INT8_C(  34),  INT8_C( 125),  INT8_C( 111) },
      {  INT8_C( 109), -INT8_C(  48), -INT8_C(  35),  INT8_C(  33),  INT8_C(  94), -INT8_C(  13), -INT8_C(  37), -INT8_C(  22),
        -INT8_C(  85),  INT8_C(  17), -INT8_C(  91),  INT8_C(  56), -INT8_C(   5),  INT8_C(  23),  INT8_C(   3), -INT8_C(   2),
        -INT8_C( 126),  INT8_C(  72), -INT8_C(  16), -INT8_C(  39), -INT8_C(  73), -INT8_C(  41), -INT8_C(  72),  INT8_C(  56),
        -INT8_C(  58), -INT8_C(  23), -INT8_C( 104),  INT8_C(  82), -INT8_C(  57),  INT8_C(  22), -INT8_C(  62),  INT8_C(  52) },
      {  INT8_C(  80),  INT8_C(  94),  INT8_C(  48), -INT8_C( 114),  INT8_C(  18), -INT8_C( 127), -INT8_C(  15), -INT8_C(  92),
         INT8_C(  54), -INT8_C(  56), -INT8_C(  37), -INT8_C(  13), -INT8_C( 120),  INT8_C(   1),  INT8_C( 117), -INT8_C(  55),
        -INT8_C( 123), -INT8_C(  78),  INT8_C(  52), -INT8_C(  54),  INT8_C(  14),  INT8_C(  70), -INT8_C(  97),  INT8_C(  23),
         INT8_C(  71), -INT8_C(  40), -INT8_C(  55), -INT8_C(  78),  INT8_C(  83), -INT8_C(  12),  INT8_C(  63), -INT8_C(  93) } },
    { { -INT8_C(  26), -INT8_C(  97),  INT8_C(  85),  INT8_C(  69), -INT8_C( 110),  INT8_C(  48),  INT8_C(  47),  INT8_C(  61),
         INT8_C(  65), -INT8_C(  44),  INT8_C( 118),  INT8_C(  60), -INT8_C(  21),  INT8_C( 121),  INT8_C(  59),  INT8_C( 109),
        -INT8_C(  63),  INT8_C(  43),  INT8_C(  70),  INT8_C( 121),  INT8_C(   2), -INT8_C(   2), -INT8_C(  79), -INT8_C(  55),
        -INT8_C(  25),  INT8_C(  74),  INT8_C(  27), -INT8_C(  81),  INT8_C(  96), -INT8_C(  35), -INT8_C(  29),  INT8_C(  70) },
      {  INT8_C( 124),  INT8_C(  57), -INT8_C( 117),  INT8_C(  15),  INT8_C( 105), -INT8_C(  70),  INT8_C(  76), -INT8_C(  85),
        -INT8_C( 114), -INT8_C(  62), -INT8_C(  25),  INT8_C( 122),  INT8_C(  60),  INT8_C(  34), -INT8_C(  25), -INT8_C(   3),
         INT8_C(  77),  INT8_C(  46),  INT8_C( 118),  INT8_C(  80),  INT8_C(  44),  INT8_C(  40),  INT8_C(  25),  INT8_C(  20),
         INT8_C( 114),  INT8_C(  52), -INT8_C(  61), -INT8_C(  46),  INT8_C(  18), -INT8_C(  90),  INT8_C(  24), -INT8_C( 114) },
      {  INT8_C(  98), -INT8_C(  40), -INT8_C(  32),  INT8_C(  84), -INT8_C(   5), -INT8_C(  22),  INT8_C( 123), -INT8_C(  24),
        -INT8_C(  49), -INT8_C( 106),  INT8_C(  93), -INT8_C(  74),  INT8_C(  39), -INT8_C( 101),  INT8_C(  34),  INT8_C( 106),
         INT8_C(  14),  INT8_C(  89), -INT8_C(  68), -INT8_C(  55),  INT8_C(  46),  INT8_C(  38), -INT8_C(  54), -INT8_C(  35),
         INT8_C(  89),  INT8_C( 126), -INT8_C(  34), -INT8_C( 127),  INT8_C( 114), -INT8_C( 125), -INT8_C(   5), -INT8_C(  44) } },
    { { -INT8_C(  33), -INT8_C(  92), -INT8_C(  99),  INT8_C(  73),  INT8_C(  94), -INT8_C(  22), -INT8_C(  12), -INT8_C(  19),
        -INT8_C(  84), -INT8_C(  37),  INT8_C( 103), -INT8_C(  24), -INT8_C(   2),  INT8_C(  78), -INT8_C(  26),  INT8_C(  75),
         INT8_C( 124),  INT8_C(  92), -INT8_C( 101), -INT8_C(  87), -INT8_C( 124), -INT8_C(  76), -INT8_C(  67), -INT8_C(  10),
        -INT8_C(  23),      INT8_MIN, -INT8_C(  56), -INT8_C(   5),  INT8_C(  38), -INT8_C(  31), -INT8_C( 119),  INT8_C(   6) },
      { -INT8_C( 123),  INT8_C(  39),  INT8_C(  79), -INT8_C(  29),  INT8_C(  17),  INT8_C(  67), -INT8_C(  48), -INT8_C(  67),
         INT8_C(  30),  INT8_C(  55), -INT8_C(  90),  INT8_C(  28), -INT8_C( 122), -INT8_C( 116),  INT8_C( 104),  INT8_C(   2),
        -INT8_C(  24),  INT8_C(   3), -INT8_C(  85),  INT8_C( 109), -INT8_C(  72),  INT8_C( 104),  INT8_C(  99), -INT8_C(  95),
        -INT8_C(  24),  INT8_C(  44), -INT8_C( 100),  INT8_C(  15),  INT8_C(  13),  INT8_C(  37),  INT8_C(  21), -INT8_C( 110) },
      {  INT8_C( 100), -INT8_C(  53), -INT8_C(  20),  INT8_C(  44),  INT8_C( 111),  INT8_C(  45), -INT8_C(  60), -INT8_C(  86),
        -INT8_C(  54),  INT8_C(  18),  INT8_C(  13),  INT8_C(   4), -INT8_C( 124), -INT8_C(  38),  INT8_C(  78),  INT8_C(  77),
         INT8_C( 100),  INT8_C(  95),  INT8_C(  70),  INT8_C(  22),  INT8_C(  60),  INT8_C(  28),  INT8_C(  32), -INT8_C( 105),
        -INT8_C(  47), -INT8_C(  84),  INT8_C( 100),  INT8_C(  10),  INT8_C(  51),  INT8_C(   6), -INT8_C(  98), -INT8_C( 104) } },
    { {  INT8_C(  76),  INT8_C( 100),  INT8_C( 117),  INT8_C(  93), -INT8_C(  89),  INT8_C(  70),  INT8_C(  27), -INT8_C(  59),
         INT8_C( 125), -INT8_C(  63), -INT8_C(  30),  INT8_C(   3),  INT8_C(  77),  INT8_C(  74),  INT8_C(   6),  INT8_C(  53),
         INT8_C(  77), -INT8_C(  79), -INT8_C(  94),  INT8_C(   5),  INT8_C(  26),  INT8_C(   6), -INT8_C(  90),  INT8_C(   2),
         INT8_C(  50),  INT8_C(  66),  INT8_C(  17),  INT8_C(  63),  INT8_C( 104),  INT8_C(  38), -INT8_C(  47), -INT8_C(  76) },
      { -INT8_C( 118),  INT8_C(  70),  INT8_C(  18),  INT8_C(  49), -INT8_C( 116),  INT8_C(  45), -INT8_C(   9),  INT8_C(  10),
        -INT8_C(  18), -INT8_C(  39),  INT8_C(  13),  INT8_C(  59),  INT8_C(  35),  INT8_C(  19),  INT8_C( 112),  INT8_C( 112),
        -INT8_C(  59),  INT8_C(  19),  INT8_C( 118), -INT8_C(  33),  INT8_C(  25),  INT8_C(  28), -INT8_C(  31),  INT8_C(  75),
         INT8_C(  95), -INT8_C(  13), -INT8_C( 118), -INT8_C(  57),  INT8_C(  25),  INT8_C(  91),  INT8_C( 123), -INT8_C(  92) },
      { -INT8_C(  42), -INT8_C(  86), -INT8_C( 121), -INT8_C( 114),  INT8_C(  51),  INT8_C( 115),  INT8_C(  18), -INT8_C(  49),
         INT8_C( 107), -INT8_C( 102), -INT8_C(  17),  INT8_C(  62),  INT8_C( 112),  INT8_C(  93),  INT8_C( 118), -INT8_C(  91),
         INT8_C(  18), -INT8_C(  60),  INT8_C(  24), -INT8_C(  28),  INT8_C(  51),  INT8_C(  34), -INT8_C( 121),  INT8_C(  77),
        -INT8_C( 111),  INT8_C(  53), -INT8_C( 101),  INT8_C(   6), -INT8_C( 127), -INT8_C( 127),  INT8_C(  76),  INT8_C(  88) } },
    { { -INT8_C(  95), -INT8_C( 115), -INT8_C(  43),  INT8_C(  46), -INT8_C(  70), -INT8_C(  52),  INT8_C(  56), -INT8_C(  88),
        -INT8_C(  91),  INT8_C(  69), -INT8_C(  29), -INT8_C(  56),  INT8_C(  89),  INT8_C(  84),  INT8_C(  57),  INT8_C(  30),
         INT8_C( 103), -INT8_C(  81), -INT8_C(   3),      INT8_MIN, -INT8_C(  53), -INT8_C(  34), -INT8_C(  53),  INT8_C(  42),
        -INT8_C(  47),  INT8_C(  85), -INT8_C(  15), -INT8_C(  21), -INT8_C(  80),  INT8_C( 109), -INT8_C( 113),  INT8_C(  81) },
      { -INT8_C(   6),  INT8_C( 100),      INT8_MAX, -INT8_C(  75),  INT8_C(  49), -INT8_C(  73),  INT8_C(  93), -INT8_C(  42),
        -INT8_C(   3),  INT8_C(  65), -INT8_C(  97),  INT8_C(  86), -INT8_C( 107), -INT8_C(  40),  INT8_C( 116), -INT8_C(   4),
        -INT8_C( 121),  INT8_C( 113),  INT8_C( 124),  INT8_C(  82),  INT8_C(  79),  INT8_C(  71),  INT8_C( 125),  INT8_C(  33),
        -INT8_C( 100),  INT8_C( 110),  INT8_C(  12),  INT8_C(  76), -INT8_C(  37), -INT8_C( 101), -INT8_C(  99), -INT8_C(  42) },
      { -INT8_C( 101), -INT8_C(  15),  INT8_C(  84), -INT8_C(  29), -INT8_C(  21), -INT8_C( 125), -INT8_C( 107),  INT8_C( 126),
        -INT8_C(  94), -INT8_C( 122), -INT8_C( 126),  INT8_C(  30), -INT8_C(  18),  INT8_C(  44), -INT8_C(  83),  INT8_C(  26),
        -INT8_C(  18),  INT8_C(  32),  INT8_C( 121), -INT8_C(  46),  INT8_C(  26),  INT8_C(  37),  INT8_C(  72),  INT8_C(  75),
         INT8_C( 109), -INT8_C(  61), -INT8_C(   3),  INT8_C(  55), -INT8_C( 117),  INT8_C(   8),  INT8_C(  44),  INT8_C(  39) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi8(test_vec[i].b);
    simde__m256i r = simde_mm256_add_epi8(a, b);
    simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_add_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  struct {
    int16_t a[16];
    int16_t b[16];
    int16_t r[16];
  } test_vec[] = {
    { {  INT16_C( 16774),  INT16_C( 17194),  INT16_C( 10451),  INT16_C(  3235),  INT16_C( 31051), -INT16_C( 23344),  INT16_C( 16496),  INT16_C( 29568),
        -INT16_C(   901), -INT16_C(  4920), -INT16_C(  5814), -INT16_C( 23726),  INT16_C( 13824),  INT16_C( 17418),  INT16_C( 19901),  INT16_C( 17354) },
      { -INT16_C(  2930),  INT16_C( 24966),  INT16_C( 10524),  INT16_C( 26734),  INT16_C( 16034),  INT16_C(  4876), -INT16_C( 29570), -INT16_C(  1658),
         INT16_C( 20104), -INT16_C( 11291),  INT16_C( 14135),  INT16_C( 14454), -INT16_C( 32658),  INT16_C( 11132),  INT16_C( 18125),  INT16_C( 23662) },
      {  INT16_C( 13844), -INT16_C( 23376),  INT16_C( 20975),  INT16_C( 29969), -INT16_C( 18451), -INT16_C( 18468), -INT16_C( 13074),  INT16_C( 27910),
         INT16_C( 19203), -INT16_C( 16211),  INT16_C(  8321), -INT16_C(  9272), -INT16_C( 18834),  INT16_C( 28550), -INT16_C( 27510), -INT16_C( 24520) } },
    { { -INT16_C(  2758),  INT16_C( 22461),  INT16_C( 11038), -INT16_C( 15937), -INT16_C( 13463), -INT16_C(  6188),  INT16_C( 23127), -INT16_C(  7968),
        -INT16_C( 14936), -INT16_C(  8269),  INT16_C( 10749),  INT16_C( 27415), -INT16_C( 27479),  INT16_C( 30358),  INT16_C(  1242),  INT16_C(  5586) },
      { -INT16_C( 28423),  INT16_C(  6252),  INT16_C( 11195),  INT16_C(  9689), -INT16_C( 21002),  INT16_C( 19724), -INT16_C(  4857), -INT16_C( 20691),
        -INT16_C(  8014), -INT16_C( 20594), -INT16_C( 23031), -INT16_C( 19942), -INT16_C( 20422),  INT16_C(  5161), -INT16_C(  1099), -INT16_C( 20951) },
      { -INT16_C( 31181),  INT16_C( 28713),  INT16_C( 22233), -INT16_C(  6248),  INT16_C( 31071),  INT16_C( 13536),  INT16_C( 18270), -INT16_C( 28659),
        -INT16_C( 22950), -INT16_C( 28863), -INT16_C( 12282),  INT16_C(  7473),  INT16_C( 17635), -INT16_C( 30017),  INT16_C(   143), -INT16_C( 15365) } },
    { { -INT16_C( 27253),  INT16_C( 18374), -INT16_C( 24640), -INT16_C( 18836),  INT16_C( 30796),  INT16_C( 21252),  INT16_C( 12645),  INT16_C(  6146),
        -INT16_C( 28398),  INT16_C(  7111), -INT16_C(  7625),  INT16_C( 29134), -INT16_C(  2158),  INT16_C( 18309), -INT16_C( 20494),  INT16_C( 32502) },
      { -INT16_C( 17340),  INT16_C(  1477),  INT16_C( 12636), -INT16_C( 22341), -INT16_C( 16471),  INT16_C(  4092), -INT16_C(   271),  INT16_C(   807),
        -INT16_C(  4465), -INT16_C( 14818), -INT16_C(  4912),  INT16_C( 25399), -INT16_C( 16925), -INT16_C( 10582), -INT16_C( 24468), -INT16_C( 20396) },
      {  INT16_C( 20943),  INT16_C( 19851), -INT16_C( 12004),  INT16_C( 24359),  INT16_C( 14325),  INT16_C( 25344),  INT16_C( 12374),  INT16_C(  6953),
         INT16_C( 32673), -INT16_C(  7707), -INT16_C( 12537), -INT16_C( 11003), -INT16_C( 19083),  INT16_C(  7727),  INT16_C( 20574),  INT16_C( 12106) } },
    { {  INT16_C(  6493), -INT16_C( 17995),  INT16_C( 29002), -INT16_C(  3231),  INT16_C( 23856),  INT16_C(  8450),  INT16_C( 10588), -INT16_C(  5340),
         INT16_C( 17176), -INT16_C(  5966), -INT16_C(  5841),  INT16_C(  4939), -INT16_C(  2394),  INT16_C(  4841),  INT16_C( 15766), -INT16_C(  3133) },
      {  INT16_C( 30806), -INT16_C( 24404),  INT16_C(  3817),  INT16_C(  6803), -INT16_C( 27029), -INT16_C( 14533),  INT16_C( 24767), -INT16_C( 10317),
         INT16_C( 26019), -INT16_C( 11584),  INT16_C(  2894), -INT16_C(  2587), -INT16_C( 12799), -INT16_C( 26617), -INT16_C( 13813),  INT16_C( 24971) },
      { -INT16_C( 28237),  INT16_C( 23137), -INT16_C( 32717),  INT16_C(  3572), -INT16_C(  3173), -INT16_C(  6083), -INT16_C( 30181), -INT16_C( 15657),
        -INT16_C( 22341), -INT16_C( 17550), -INT16_C(  2947),  INT16_C(  2352), -INT16_C( 15193), -INT16_C( 21776),  INT16_C(  1953),  INT16_C( 21838) } },
    { {  INT16_C( 14403),  INT16_C( 11265), -INT16_C( 27322), -INT16_C( 20154), -INT16_C( 32213), -INT16_C(  5511),  INT16_C( 11490), -INT16_C( 31294),
        -INT16_C( 32111), -INT16_C(  8361),  INT16_C( 15757), -INT16_C( 28716), -INT16_C(  9205),  INT16_C(  5927), -INT16_C( 19802), -INT16_C(  5768) },
      {  INT16_C( 31466),  INT16_C( 12310),  INT16_C( 23567),  INT16_C( 15074),  INT16_C( 23518), -INT16_C( 16348), -INT16_C(  6521),  INT16_C(  6213),
        -INT16_C( 25240), -INT16_C(  2313), -INT16_C( 13094), -INT16_C(  6779), -INT16_C( 21336),  INT16_C( 20220),  INT16_C( 30046),  INT16_C( 18744) },
      { -INT16_C( 19667),  INT16_C( 23575), -INT16_C(  3755), -INT16_C(  5080), -INT16_C(  8695), -INT16_C( 21859),  INT16_C(  4969), -INT16_C( 25081),
         INT16_C(  8185), -INT16_C( 10674),  INT16_C(  2663),  INT16_C( 30041), -INT16_C( 30541),  INT16_C( 26147),  INT16_C( 10244),  INT16_C( 12976) } },
    { {  INT16_C( 20207), -INT16_C(   391),  INT16_C( 23466), -INT16_C( 30408),  INT16_C( 23734),  INT16_C( 15689), -INT16_C( 28861), -INT16_C( 21675),
         INT16_C( 19756),  INT16_C(  1697),  INT16_C(  9753), -INT16_C( 15893), -INT16_C(  5934),  INT16_C( 12559),  INT16_C( 18269),  INT16_C( 19578) },
      { -INT16_C(  3179),  INT16_C( 16458), -INT16_C( 32177),  INT16_C(  1481),  INT16_C(  4830),  INT16_C(  8515), -INT16_C( 26463), -INT16_C( 12851),
         INT16_C( 28389), -INT16_C(   301), -INT16_C( 16491),  INT16_C( 26559), -INT16_C( 12377),  INT16_C(  1176),  INT16_C(  4630), -INT16_C( 21424) },
      {  INT16_C( 17028),  INT16_C( 16067), -INT16_C(  8711), -INT16_C( 28927),  INT16_C( 28564),  INT16_C( 24204),  INT16_C( 10212),  INT16_C( 31010),
        -INT16_C( 17391),  INT16_C(  1396), -INT16_C(  6738),  INT16_C( 10666), -INT16_C( 18311),  INT16_C( 13735),  INT16_C( 22899), -INT16_C(  1846) } },
    { { -INT16_C( 26106),  INT16_C( 21996), -INT16_C( 19172), -INT16_C(  1446), -INT16_C( 25145),  INT16_C( 26908), -INT16_C(  5834),  INT16_C(  6966),
         INT16_C(  2647), -INT16_C(  5094), -INT16_C(  9783),  INT16_C( 28756), -INT16_C(  4952), -INT16_C( 16524), -INT16_C( 15105),  INT16_C(  1387) },
      {  INT16_C( 22366),  INT16_C( 31322), -INT16_C( 19444), -INT16_C( 11404), -INT16_C( 28590), -INT16_C( 30660),  INT16_C( 29561), -INT16_C( 11869),
        -INT16_C( 17027),  INT16_C( 18109),  INT16_C(  4503),  INT16_C( 16310),  INT16_C( 11006), -INT16_C(   514),  INT16_C( 27118),  INT16_C( 19458) },
      { -INT16_C(  3740), -INT16_C( 12218),  INT16_C( 26920), -INT16_C( 12850),  INT16_C( 11801), -INT16_C(  3752),  INT16_C( 23727), -INT16_C(  4903),
        -INT16_C( 14380),  INT16_C( 13015), -INT16_C(  5280), -INT16_C( 20470),  INT16_C(  6054), -INT16_C( 17038),  INT16_C( 12013),  INT16_C( 20845) } },
    { {  INT16_C( 23744), -INT16_C( 13114),  INT16_C( 14864),  INT16_C( 25248), -INT16_C(  9013),  INT16_C( 17642), -INT16_C( 29105), -INT16_C( 13291),
        -INT16_C( 11445), -INT16_C(  7662), -INT16_C( 14108), -INT16_C(  7646),  INT16_C(  8434), -INT16_C(  7969), -INT16_C(  7798),  INT16_C( 18988) },
      { -INT16_C(  3523),  INT16_C( 19991), -INT16_C( 18643), -INT16_C(  1872), -INT16_C( 25709), -INT16_C(  7364),  INT16_C( 21033),  INT16_C( 29871),
        -INT16_C( 15835),  INT16_C(  2391),  INT16_C( 31114),  INT16_C( 32236), -INT16_C( 13415),  INT16_C(  9053), -INT16_C( 30035), -INT16_C(  5522) },
      {  INT16_C( 20221),  INT16_C(  6877), -INT16_C(  3779),  INT16_C( 23376),  INT16_C( 30814),  INT16_C( 10278), -INT16_C(  8072),  INT16_C( 16580),
        -INT16_C( 27280), -INT16_C(  5271),  INT16_C( 17006),  INT16_C( 24590), -INT16_C(  4981),  INT16_C(  1084),  INT16_C( 27703),  INT16_C( 13466) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_add_epi16(a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_add_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  struct {
    int32_t a[8];
    int32_t b[8];
    int32_t r[8];
  } test_vec[] = {
    { { -INT32_C(   162005335), -INT32_C(   622453682), -INT32_C(  1422873298),  INT32_C(   732145866), -INT32_C(  1831844680), -INT32_C(  1222554996), -INT32_C(   916290153), -INT32_C(   253014202) },
      { -INT32_C(  1679408307), -INT32_C(  1921659553),  INT32_C(  1144563066),  INT32_C(    91216973), -INT32_C(  1399308512),  INT32_C(   543406473), -INT32_C(  2048211394),  INT32_C(  1651889429) },
      { -INT32_C(  1841413642),  INT32_C(  1750854061), -INT32_C(   278310232),  INT32_C(   823362839),  INT32_C(  1063814104), -INT32_C(   679148523),  INT32_C(  1330465749),  INT32_C(  1398875227) } },
    { {  INT32_C(  2013092632), -INT32_C(  1576766936),  INT32_C(  1709587735),  INT32_C(   963269913),  INT32_C(   518324884), -INT32_C(    96581189),  INT32_C(   612313103),  INT32_C(   377943294) },
      {  INT32_C(  2022605647),  INT32_C(   203068149), -INT32_C(   395247153), -INT32_C(   350102442), -INT32_C(  1710684194),  INT32_C(  1603553104),  INT32_C(  1854083952),  INT32_C(  1451493639) },
      { -INT32_C(   259269017), -INT32_C(  1373698787),  INT32_C(  1314340582),  INT32_C(   613167471), -INT32_C(  1192359310),  INT32_C(  1506971915), -INT32_C(  1828570241),  INT32_C(  1829436933) } },
    { { -INT32_C(  2117201268),  INT32_C(  1955457445),  INT32_C(  1079902186), -INT32_C(  1188331813), -INT32_C(   715967355), -INT32_C(   332077188),  INT32_C(    22722554),  INT32_C(  1280892608) },
      { -INT32_C(  1764940047), -INT32_C(   116761841),  INT32_C(   893019994),  INT32_C(  1810785766),  INT32_C(   373375642),  INT32_C(   604140841), -INT32_C(   316318675),  INT32_C(   725253434) },
      {  INT32_C(   412825981),  INT32_C(  1838695604),  INT32_C(  1972922180),  INT32_C(   622453953), -INT32_C(   342591713),  INT32_C(   272063653), -INT32_C(   293596121),  INT32_C(  2006146042) } },
    { { -INT32_C(  1279195228), -INT32_C(  1129460638),  INT32_C(   435283763), -INT32_C(   427433908),  INT32_C(  1274856994),  INT32_C(  1752170043), -INT32_C(  1789487782), -INT32_C(  1228894190) },
      { -INT32_C(    93683049), -INT32_C(  2118772914),  INT32_C(  1251715326), -INT32_C(  1439620984),  INT32_C(   569716198), -INT32_C(  2037750484),  INT32_C(   203153658),  INT32_C(   130276464) },
      { -INT32_C(  1372878277),  INT32_C(  1046733744),  INT32_C(  1686999089), -INT32_C(  1867054892),  INT32_C(  1844573192), -INT32_C(   285580441), -INT32_C(  1586334124), -INT32_C(  1098617726) } },
    { { -INT32_C(  1409208994),  INT32_C(  1110292548), -INT32_C(   393426848), -INT32_C(   829243928),  INT32_C(   401639403), -INT32_C(   425886996), -INT32_C(   889996966), -INT32_C(   204360043) },
      {  INT32_C(   664785891), -INT32_C(   345387893),  INT32_C(  2111042965), -INT32_C(  1639160397), -INT32_C(   642433812),  INT32_C(   280974006), -INT32_C(  1596280309),  INT32_C(  1267968872) },
      { -INT32_C(   744423103),  INT32_C(   764904655),  INT32_C(  1717616117),  INT32_C(  1826562971), -INT32_C(   240794409), -INT32_C(   144912990),  INT32_C(  1808690021),  INT32_C(  1063608829) } },
    { {  INT32_C(   158478974), -INT32_C(  1795892225), -INT32_C(  2079209519),  INT32_C(   421682476),  INT32_C(  1341314969),  INT32_C(   895463721), -INT32_C(   858441372),  INT32_C(  1662544357) },
      { -INT32_C(  1704097125),  INT32_C(   925786470),  INT32_C(  1438335017),  INT32_C(   930012573), -INT32_C(   578395980),  INT32_C(  1980950034),  INT32_C(    71493663), -INT32_C(   328705455) },
      { -INT32_C(  1545618151), -INT32_C(   870105755), -INT32_C(   640874502),  INT32_C(  1351695049),  INT32_C(   762918989), -INT32_C(  1418553541), -INT32_C(   786947709),  INT32_C(  1333838902) } },
    { {  INT32_C(  1267193317),  INT32_C(  1602401590), -INT32_C(  1816838667), -INT32_C(   825613542), -INT32_C(  1767157628),  INT32_C(  1443675702), -INT32_C(   145076570), -INT32_C(  1897676119) },
      { -INT32_C(   824612201),  INT32_C(   355293984), -INT32_C(  1297554792), -INT32_C(  1971293690), -INT32_C(   115332157), -INT32_C(  1890636567),  INT32_C(   596028026),  INT32_C(    78735980) },
      {  INT32_C(   442581116),  INT32_C(  1957695574),  INT32_C(  1180573837),  INT32_C(  1498060064), -INT32_C(  1882489785), -INT32_C(   446960865),  INT32_C(   450951456), -INT32_C(  1818940139) } },
    { { -INT32_C(   187528492),  INT32_C(  2097872869), -INT32_C(   399527198), -INT32_C(   395137243), -INT32_C(   991849765),  INT32_C(   945041854),  INT32_C(  1197267675),  INT32_C(   424348996) },
      {  INT32_C(  2098011544), -INT32_C(      321763), -INT32_C(   270062902), -INT32_C(  1244177958), -INT32_C(  1451640597), -INT32_C(   991834647), -INT32_C(   334742104), -INT32_C(   486189237) },
      {  INT32_C(  1910483052),  INT32_C(  2097551106), -INT32_C(   669590100), -INT32_C(  1639315201),  INT32_C(  1851476934), -INT32_C(    46792793),  INT32_C(   862525571), -INT32_C(    61840241) } }
  };


  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_add_epi32(a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_add_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  struct {
    int64_t a[4];
    int64_t b[4];
    int64_t r[4];
  } test_vec[] = {
    { {  INT64_C( 6349431231190787759), -INT64_C(  254720912029419480), -INT64_C( 1585114888304005479), -INT64_C( 1993816771242646234) },
      {  INT64_C( 1718517911670765936),  INT64_C( 2128417586150406366), -INT64_C( 5915363177252979306), -INT64_C( 4600475057435654898) },
      {  INT64_C( 8067949142861553695),  INT64_C( 1873696674120986886), -INT64_C( 7500478065556984785), -INT64_C( 6594291828678301132) } },
    { { -INT64_C( 5470999611667550195), -INT64_C( 9064397034655497070),  INT64_C( 1368083854048468167),  INT64_C( 6718945478753745967) },
      {  INT64_C( 3132092573034315757),  INT64_C( 5341117986440919217),  INT64_C( 4118987204768407570), -INT64_C( 1675067501828407790) },
      { -INT64_C( 2338907038633234438), -INT64_C( 3723279048214577853),  INT64_C( 5487071058816875737),  INT64_C( 5043877976925338177) } },
    { {  INT64_C( 3979532279430715765),  INT64_C(  335787494049348827), -INT64_C( 1668789448351202924), -INT64_C(  385608049042419774) },
      {  INT64_C( 5828296139942220324),  INT64_C( 4591233640581478336), -INT64_C( 2785840211528068457), -INT64_C( 3331470012021497532) },
      { -INT64_C( 8638915654336615527),  INT64_C( 4927021134630827163), -INT64_C( 4454629659879271381), -INT64_C( 3717078061063917306) } },
    { { -INT64_C( 3290620525541536369), -INT64_C( 1080797550094253291), -INT64_C( 2810608374329123313),  INT64_C( 7121710342746510679) },
      { -INT64_C( 4244998137797193421), -INT64_C( 1798196463016764481),  INT64_C( 8705188840942103139),  INT64_C(  708843972859818144) },
      { -INT64_C( 7535618663338729790), -INT64_C( 2878994013111017772),  INT64_C( 5894580466612979826),  INT64_C( 7830554315606328823) } },
    { { -INT64_C( 5181843299269526110), -INT64_C( 8809360683665548033),  INT64_C( 8565929875506776903),  INT64_C( 5736108927119678651) },
      {  INT64_C( 5453497739975387389), -INT64_C( 6486905996753797037),  INT64_C( 9005490340975441324), -INT64_C( 6897921832086805687) },
      {  INT64_C(  271654440705861279),  INT64_C( 3150477393290206546), -INT64_C(  875323857227333389), -INT64_C( 1161812904967127036) } },
    { { -INT64_C( 1213926134046595696), -INT64_C( 2826187797014754619),  INT64_C(  724891255407629523),  INT64_C( 6159312156450889015) },
      {  INT64_C( 1853046892828626987),  INT64_C( 5451329691363056547), -INT64_C( 8240199362465037608),  INT64_C( 1523872447569830923) },
      {  INT64_C(  639120758782031291),  INT64_C( 2625141894348301928), -INT64_C( 7515308107057408085),  INT64_C( 7683184604020719938) } },
    { {  INT64_C( 5935625214915008241),  INT64_C( 8607162423740308398),  INT64_C( 5453775861294584267), -INT64_C( 1632482330388195751) },
      {  INT64_C( 8546019858262733976),  INT64_C( 5566467293344283178),  INT64_C( 6840156418577748321), -INT64_C( 2155261274678556364) },
      { -INT64_C( 3965099000531809399), -INT64_C( 4273114356624960040), -INT64_C( 6152811793837219028), -INT64_C( 3787743605066752115) } },
    { { -INT64_C(   41444076634421970),  INT64_C( 8884486176460245628),  INT64_C( 1062298490825980762), -INT64_C( 8700866135333380333) },
      { -INT64_C( 6311257025808602730), -INT64_C( 7671331955779620109), -INT64_C( 4352814043531794769), -INT64_C( 5119924125424365294) },
      { -INT64_C( 6352701102443024700),  INT64_C( 1213154220680625519), -INT64_C( 3290515552705814007),  INT64_C( 4625953812951805989) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi64(test_vec[i].b);
    simde__m256i r = simde_mm256_add_epi64(a, b);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_alignr_epi8_case0(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C(  57), INT8_C( -47), INT8_C( -81), INT8_C( -95),
                           INT8_C(-100), INT8_C( -69), INT8_C( -75), INT8_C(-127),
                           INT8_C(  -8), INT8_C( -83), INT8_C( -94), INT8_C(-113),
                           INT8_C( 107), INT8_C( -39), INT8_C( -99), INT8_C(  46),
                           INT8_C( -61), INT8_C( -72), INT8_C(  -5), INT8_C(-126),
                           INT8_C( 113), INT8_C(-120), INT8_C(  -9), INT8_C( 117),
                           INT8_C( 121), INT8_C( -14), INT8_C( -58), INT8_C(-118),
                           INT8_C( -70), INT8_C( -35), INT8_C(-100), INT8_C(  48)),
      simde_mm256_set_epi8(INT8_C( -98), INT8_C(  80), INT8_C( -16), INT8_C(  73),
                           INT8_C( 126), INT8_C(-115), INT8_C(  31), INT8_C(  80),
                           INT8_C( -75), INT8_C( 117), INT8_C( -13), INT8_C( -70),
                           INT8_C(-101), INT8_C( -10), INT8_C(-111), INT8_C(  96),
                           INT8_C( -77), INT8_C(   0), INT8_C(  70), INT8_C(-121),
                           INT8_C(  78), INT8_C(  54), INT8_C(  38), INT8_C(  18),
                           INT8_C( -12), INT8_C(  68), INT8_C(  53), INT8_C(  45),
                           INT8_C(  77), INT8_C( 103), INT8_C( 118), INT8_C( 116)),
      simde_mm256_set_epi8(INT8_C( -39), INT8_C( -99), INT8_C(  46), INT8_C( -98),
                           INT8_C(  80), INT8_C( -16), INT8_C(  73), INT8_C( 126),
                           INT8_C(-115), INT8_C(  31), INT8_C(  80), INT8_C( -75),
                           INT8_C( 117), INT8_C( -13), INT8_C( -70), INT8_C(-101),
                           INT8_C( -35), INT8_C(-100), INT8_C(  48), INT8_C( -77),
                           INT8_C(   0), INT8_C(  70), INT8_C(-121), INT8_C(  78),
                           INT8_C(  54), INT8_C(  38), INT8_C(  18), INT8_C( -12),
                           INT8_C(  68), INT8_C(  53), INT8_C(  45), INT8_C(  77)) },
    { simde_mm256_set_epi8(INT8_C(   8), INT8_C(  70), INT8_C(  96), INT8_C(  42),
                           INT8_C(  73), INT8_C( -91), INT8_C( -93), INT8_C( -58),
                           INT8_C( -35), INT8_C( -14), INT8_C( 118), INT8_C(-128),
                           INT8_C( -97), INT8_C(  97), INT8_C(  22), INT8_C(   4),
                           INT8_C(  24), INT8_C( 124), INT8_C( -32), INT8_C( -48),
                           INT8_C( -74), INT8_C(  95), INT8_C(  95), INT8_C( -38),
                           INT8_C(  54), INT8_C( 120), INT8_C( -65), INT8_C( -96),
                           INT8_C( -44), INT8_C( -78), INT8_C( -95), INT8_C(-111)),
      simde_mm256_set_epi8(INT8_C(  66), INT8_C(  48), INT8_C(  60), INT8_C(  -4),
                           INT8_C(  33), INT8_C(  67), INT8_C(  97), INT8_C(  65),
                           INT8_C(  80), INT8_C(-109), INT8_C(  88), INT8_C( -23),
                           INT8_C( 107), INT8_C( -65), INT8_C( -65), INT8_C(  62),
                           INT8_C(  90), INT8_C( 118), INT8_C(  25), INT8_C( 105),
                           INT8_C( 109), INT8_C( -45), INT8_C(   4), INT8_C(-107),
                           INT8_C(  55), INT8_C(-101), INT8_C( -65), INT8_C( 121),
                           INT8_C( -76), INT8_C(  55), INT8_C(   6), INT8_C(  86)),
      simde_mm256_set_epi8(INT8_C(  97), INT8_C(  22), INT8_C(   4), INT8_C(  66),
                           INT8_C(  48), INT8_C(  60), INT8_C(  -4), INT8_C(  33),
                           INT8_C(  67), INT8_C(  97), INT8_C(  65), INT8_C(  80),
                           INT8_C(-109), INT8_C(  88), INT8_C( -23), INT8_C( 107),
                           INT8_C( -78), INT8_C( -95), INT8_C(-111), INT8_C(  90),
                           INT8_C( 118), INT8_C(  25), INT8_C( 105), INT8_C( 109),
                           INT8_C( -45), INT8_C(   4), INT8_C(-107), INT8_C(  55),
                           INT8_C(-101), INT8_C( -65), INT8_C( 121), INT8_C( -76)) },
    { simde_mm256_set_epi8(INT8_C(  -1), INT8_C(  93), INT8_C( 114), INT8_C(  49),
                           INT8_C( 120), INT8_C( -70), INT8_C(-112), INT8_C(  19),
                           INT8_C(  56), INT8_C( -12), INT8_C( -77), INT8_C( -85),
                           INT8_C( -59), INT8_C(  80), INT8_C( -47), INT8_C( -28),
                           INT8_C( -33), INT8_C(  73), INT8_C( -24), INT8_C(-106),
                           INT8_C(-122), INT8_C(-111), INT8_C(  -3), INT8_C(  57),
                           INT8_C(  67), INT8_C( 107), INT8_C( -68), INT8_C(-101),
                           INT8_C(  -9), INT8_C(   5), INT8_C(-124), INT8_C( -69)),
      simde_mm256_set_epi8(INT8_C( -67), INT8_C(  65), INT8_C( -72), INT8_C( -90),
                           INT8_C( -44), INT8_C(   2), INT8_C(  39), INT8_C( -45),
                           INT8_C(  51), INT8_C( -30), INT8_C( -47), INT8_C(   1),
                           INT8_C( -91), INT8_C( -40), INT8_C( -73), INT8_C(  33),
                           INT8_C(-117), INT8_C( -31), INT8_C(-117), INT8_C(  60),
                           INT8_C( -36), INT8_C(-120), INT8_C( -10), INT8_C( -12),
                           INT8_C( -25), INT8_C(-103), INT8_C(  48), INT8_C( -50),
                           INT8_C( 101), INT8_C(  93), INT8_C( -31), INT8_C( -33)),
      simde_mm256_set_epi8(INT8_C(  80), INT8_C( -47), INT8_C( -28), INT8_C( -67),
                           INT8_C(  65), INT8_C( -72), INT8_C( -90), INT8_C( -44),
                           INT8_C(   2), INT8_C(  39), INT8_C( -45), INT8_C(  51),
                           INT8_C( -30), INT8_C( -47), INT8_C(   1), INT8_C( -91),
                           INT8_C(   5), INT8_C(-124), INT8_C( -69), INT8_C(-117),
                           INT8_C( -31), INT8_C(-117), INT8_C(  60), INT8_C( -36),
                           INT8_C(-120), INT8_C( -10), INT8_C( -12), INT8_C( -25),
                           INT8_C(-103), INT8_C(  48), INT8_C( -50), INT8_C( 101)) },
    { simde_mm256_set_epi8(INT8_C( 114), INT8_C( -28), INT8_C( -81), INT8_C(-126),
                           INT8_C(-114), INT8_C( -94), INT8_C(-101), INT8_C(  52),
                           INT8_C(  30), INT8_C(  66), INT8_C(-105), INT8_C(  50),
                           INT8_C(  48), INT8_C(  15), INT8_C(   1), INT8_C(-106),
                           INT8_C( -36), INT8_C(  61), INT8_C( -18), INT8_C(   8),
                           INT8_C(  89), INT8_C( -23), INT8_C(  58), INT8_C( -29),
                           INT8_C( 110), INT8_C(  57), INT8_C( -39), INT8_C(   1),
                           INT8_C( -60), INT8_C(  60), INT8_C( -80), INT8_C( -20)),
      simde_mm256_set_epi8(INT8_C( -60), INT8_C(  69), INT8_C( -69), INT8_C( 107),
                           INT8_C(   2), INT8_C( -33), INT8_C(  14), INT8_C(  77),
                           INT8_C(  15), INT8_C(  63), INT8_C(  63), INT8_C( -53),
                           INT8_C(  -9), INT8_C(  76), INT8_C(   7), INT8_C( -76),
                           INT8_C( -23), INT8_C(-107), INT8_C(  84), INT8_C(  37),
                           INT8_C( -18), INT8_C(  47), INT8_C(  58), INT8_C(  31),
                           INT8_C(  69), INT8_C(  12), INT8_C(  46), INT8_C( 101),
                           INT8_C( -38), INT8_C( 117), INT8_C(-113), INT8_C( -36)),
      simde_mm256_set_epi8(INT8_C(  15), INT8_C(   1), INT8_C(-106), INT8_C( -60),
                           INT8_C(  69), INT8_C( -69), INT8_C( 107), INT8_C(   2),
                           INT8_C( -33), INT8_C(  14), INT8_C(  77), INT8_C(  15),
                           INT8_C(  63), INT8_C(  63), INT8_C( -53), INT8_C(  -9),
                           INT8_C(  60), INT8_C( -80), INT8_C( -20), INT8_C( -23),
                           INT8_C(-107), INT8_C(  84), INT8_C(  37), INT8_C( -18),
                           INT8_C(  47), INT8_C(  58), INT8_C(  31), INT8_C(  69),
                           INT8_C(  12), INT8_C(  46), INT8_C( 101), INT8_C( -38)) },
    { simde_mm256_set_epi8(INT8_C(  13), INT8_C( -32), INT8_C( -51), INT8_C(  49),
                           INT8_C(  15), INT8_C(  15), INT8_C( -66), INT8_C( -38),
                           INT8_C(  69), INT8_C(   1), INT8_C( -50), INT8_C(-103),
                           INT8_C(  72), INT8_C( -87), INT8_C(-104), INT8_C( -58),
                           INT8_C(  63), INT8_C(-120), INT8_C(  36), INT8_C( -61),
                           INT8_C( 106), INT8_C(  97), INT8_C(-128), INT8_C(  26),
                           INT8_C(-121), INT8_C( -27), INT8_C( 107), INT8_C(-112),
                           INT8_C(  57), INT8_C(  39), INT8_C(  98), INT8_C( 126)),
      simde_mm256_set_epi8(INT8_C(-110), INT8_C(  50), INT8_C( 102), INT8_C( -85),
                           INT8_C( -62), INT8_C(  74), INT8_C( 102), INT8_C(  36),
                           INT8_C( -55), INT8_C(-104), INT8_C(  11), INT8_C( -70),
                           INT8_C( -20), INT8_C(-104), INT8_C( -37), INT8_C( 122),
                           INT8_C( -88), INT8_C(  70), INT8_C(  43), INT8_C( -34),
                           INT8_C(  14), INT8_C(  55), INT8_C( -41), INT8_C(  33),
                           INT8_C(  53), INT8_C(-102), INT8_C(  56), INT8_C(  26),
                           INT8_C( -79), INT8_C( -30), INT8_C(  73), INT8_C( -53)),
      simde_mm256_set_epi8(INT8_C( -87), INT8_C(-104), INT8_C( -58), INT8_C(-110),
                           INT8_C(  50), INT8_C( 102), INT8_C( -85), INT8_C( -62),
                           INT8_C(  74), INT8_C( 102), INT8_C(  36), INT8_C( -55),
                           INT8_C(-104), INT8_C(  11), INT8_C( -70), INT8_C( -20),
                           INT8_C(  39), INT8_C(  98), INT8_C( 126), INT8_C( -88),
                           INT8_C(  70), INT8_C(  43), INT8_C( -34), INT8_C(  14),
                           INT8_C(  55), INT8_C( -41), INT8_C(  33), INT8_C(  53),
                           INT8_C(-102), INT8_C(  56), INT8_C(  26), INT8_C( -79)) },
    { simde_mm256_set_epi8(INT8_C(-103), INT8_C(  23), INT8_C( -90), INT8_C(  43),
                           INT8_C(  -8), INT8_C(  54), INT8_C( -93), INT8_C(  64),
                           INT8_C(-116), INT8_C( -76), INT8_C( -53), INT8_C(  35),
                           INT8_C(  51), INT8_C( -86), INT8_C(   8), INT8_C(  43),
                           INT8_C( -60), INT8_C(-113), INT8_C(  44), INT8_C(-102),
                           INT8_C(  77), INT8_C(  95), INT8_C(  57), INT8_C( 124),
                           INT8_C( -91), INT8_C( -87), INT8_C( -54), INT8_C( -83),
                           INT8_C( 108), INT8_C(  76), INT8_C(  90), INT8_C( -49)),
      simde_mm256_set_epi8(INT8_C( -43), INT8_C(  -5), INT8_C(  94), INT8_C(  76),
                           INT8_C(  -9), INT8_C( 110), INT8_C( -82), INT8_C( 112),
                           INT8_C( -25), INT8_C(  11), INT8_C( -46), INT8_C( -57),
                           INT8_C( -13), INT8_C(-127), INT8_C(  10), INT8_C(-113),
                           INT8_C(  55), INT8_C(  85), INT8_C(  75), INT8_C(  61),
                           INT8_C(  49), INT8_C( -14), INT8_C(  44), INT8_C(-128),
                           INT8_C(-102), INT8_C( -17), INT8_C(  64), INT8_C( -73),
                           INT8_C(  14), INT8_C(  19), INT8_C(  -2), INT8_C(   7)),
      simde_mm256_set_epi8(INT8_C( -86), INT8_C(   8), INT8_C(  43), INT8_C( -43),
                           INT8_C(  -5), INT8_C(  94), INT8_C(  76), INT8_C(  -9),
                           INT8_C( 110), INT8_C( -82), INT8_C( 112), INT8_C( -25),
                           INT8_C(  11), INT8_C( -46), INT8_C( -57), INT8_C( -13),
                           INT8_C(  76), INT8_C(  90), INT8_C( -49), INT8_C(  55),
                           INT8_C(  85), INT8_C(  75), INT8_C(  61), INT8_C(  49),
                           INT8_C( -14), INT8_C(  44), INT8_C(-128), INT8_C(-102),
                           INT8_C( -17), INT8_C(  64), INT8_C( -73), INT8_C(  14)) },
    { simde_mm256_set_epi8(INT8_C( -32), INT8_C( -82), INT8_C(  79), INT8_C(  79),
                           INT8_C( 127), INT8_C(  16), INT8_C(-118), INT8_C( 113),
                           INT8_C( -52), INT8_C( -25), INT8_C(-124), INT8_C( -77),
                           INT8_C( 114), INT8_C(-118), INT8_C( -84), INT8_C(  73),
                           INT8_C( -12), INT8_C( -67), INT8_C(-127), INT8_C( -69),
                           INT8_C(  30), INT8_C(  14), INT8_C( -47), INT8_C(-119),
                           INT8_C( -36), INT8_C( -97), INT8_C( -89), INT8_C(  24),
                           INT8_C( -41), INT8_C( 117), INT8_C(-102), INT8_C(  54)),
      simde_mm256_set_epi8(INT8_C( -66), INT8_C( -50), INT8_C(  39), INT8_C(  30),
                           INT8_C(  53), INT8_C( 108), INT8_C( -71), INT8_C( -62),
                           INT8_C(  58), INT8_C( -69), INT8_C( -39), INT8_C( -28),
                           INT8_C(  58), INT8_C(  54), INT8_C(-117), INT8_C( -49),
                           INT8_C( 126), INT8_C( -71), INT8_C( -39), INT8_C(-120),
                           INT8_C(  88), INT8_C(  41), INT8_C( 127), INT8_C( 111),
                           INT8_C( -32), INT8_C( -18), INT8_C(  12), INT8_C(  97),
                           INT8_C(  83), INT8_C( -96), INT8_C(  13), INT8_C(-101)),
      simde_mm256_set_epi8(INT8_C(-118), INT8_C( -84), INT8_C(  73), INT8_C( -66),
                           INT8_C( -50), INT8_C(  39), INT8_C(  30), INT8_C(  53),
                           INT8_C( 108), INT8_C( -71), INT8_C( -62), INT8_C(  58),
                           INT8_C( -69), INT8_C( -39), INT8_C( -28), INT8_C(  58),
                           INT8_C( 117), INT8_C(-102), INT8_C(  54), INT8_C( 126),
                           INT8_C( -71), INT8_C( -39), INT8_C(-120), INT8_C(  88),
                           INT8_C(  41), INT8_C( 127), INT8_C( 111), INT8_C( -32),
                           INT8_C( -18), INT8_C(  12), INT8_C(  97), INT8_C(  83)) },
    { simde_mm256_set_epi8(INT8_C(-105), INT8_C(   9), INT8_C( -52), INT8_C(  54),
                           INT8_C( -11), INT8_C( -39), INT8_C( -64), INT8_C( -79),
                           INT8_C(  88), INT8_C(  52), INT8_C( -95), INT8_C( -70),
                           INT8_C(  46), INT8_C( 121), INT8_C( -22), INT8_C( 113),
                           INT8_C(  26), INT8_C( 104), INT8_C( -52), INT8_C( 109),
                           INT8_C(   5), INT8_C(  87), INT8_C(  90), INT8_C(  -7),
                           INT8_C( -35), INT8_C(-112), INT8_C(  20), INT8_C(-128),
                           INT8_C( -21), INT8_C( -67), INT8_C(  37), INT8_C( -35)),
      simde_mm256_set_epi8(INT8_C(  66), INT8_C( -23), INT8_C(  45), INT8_C( -38),
                           INT8_C( -94), INT8_C( 119), INT8_C( -23), INT8_C(  70),
                           INT8_C( -75), INT8_C(  25), INT8_C(  79), INT8_C(-101),
                           INT8_C(-122), INT8_C(-107), INT8_C( -10), INT8_C(-109),
                           INT8_C( -96), INT8_C(  45), INT8_C(  31), INT8_C( -60),
                           INT8_C( 100), INT8_C( 115), INT8_C(  79), INT8_C(  38),
                           INT8_C( -35), INT8_C(  57), INT8_C(  13), INT8_C(  93),
                           INT8_C(  54), INT8_C( -61), INT8_C( 101), INT8_C(-113)),
      simde_mm256_set_epi8(INT8_C( 121), INT8_C( -22), INT8_C( 113), INT8_C(  66),
                           INT8_C( -23), INT8_C(  45), INT8_C( -38), INT8_C( -94),
                           INT8_C( 119), INT8_C( -23), INT8_C(  70), INT8_C( -75),
                           INT8_C(  25), INT8_C(  79), INT8_C(-101), INT8_C(-122),
                           INT8_C( -67), INT8_C(  37), INT8_C( -35), INT8_C( -96),
                           INT8_C(  45), INT8_C(  31), INT8_C( -60), INT8_C( 100),
                           INT8_C( 115), INT8_C(  79), INT8_C(  38), INT8_C( -35),
                           INT8_C(  57), INT8_C(  13), INT8_C(  93), INT8_C(  54)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_alignr_epi8(test_vec[i].a, test_vec[i].b, 3);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_alignr_epi8_case1(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C(  57), INT8_C( -47), INT8_C( -81), INT8_C( -95),
                           INT8_C(-100), INT8_C( -69), INT8_C( -75), INT8_C(-127),
                           INT8_C(  -8), INT8_C( -83), INT8_C( -94), INT8_C(-113),
                           INT8_C( 107), INT8_C( -39), INT8_C( -99), INT8_C(  46),
                           INT8_C( -61), INT8_C( -72), INT8_C(  -5), INT8_C(-126),
                           INT8_C( 113), INT8_C(-120), INT8_C(  -9), INT8_C( 117),
                           INT8_C( 121), INT8_C( -14), INT8_C( -58), INT8_C(-118),
                           INT8_C( -70), INT8_C( -35), INT8_C(-100), INT8_C(  48)),
      simde_mm256_set_epi8(INT8_C( -98), INT8_C(  80), INT8_C( -16), INT8_C(  73),
                           INT8_C( 126), INT8_C(-115), INT8_C(  31), INT8_C(  80),
                           INT8_C( -75), INT8_C( 117), INT8_C( -13), INT8_C( -70),
                           INT8_C(-101), INT8_C( -10), INT8_C(-111), INT8_C(  96),
                           INT8_C( -77), INT8_C(   0), INT8_C(  70), INT8_C(-121),
                           INT8_C(  78), INT8_C(  54), INT8_C(  38), INT8_C(  18),
                           INT8_C( -12), INT8_C(  68), INT8_C(  53), INT8_C(  45),
                           INT8_C(  77), INT8_C( 103), INT8_C( 118), INT8_C( 116)),
      simde_mm256_set_epi8(INT8_C( -75), INT8_C(-127), INT8_C(  -8), INT8_C( -83),
                           INT8_C( -94), INT8_C(-113), INT8_C( 107), INT8_C( -39),
                           INT8_C( -99), INT8_C(  46), INT8_C( -98), INT8_C(  80),
                           INT8_C( -16), INT8_C(  73), INT8_C( 126), INT8_C(-115),
                           INT8_C(  -9), INT8_C( 117), INT8_C( 121), INT8_C( -14),
                           INT8_C( -58), INT8_C(-118), INT8_C( -70), INT8_C( -35),
                           INT8_C(-100), INT8_C(  48), INT8_C( -77), INT8_C(   0),
                           INT8_C(  70), INT8_C(-121), INT8_C(  78), INT8_C(  54)) },
    { simde_mm256_set_epi8(INT8_C(   8), INT8_C(  70), INT8_C(  96), INT8_C(  42),
                           INT8_C(  73), INT8_C( -91), INT8_C( -93), INT8_C( -58),
                           INT8_C( -35), INT8_C( -14), INT8_C( 118), INT8_C(-128),
                           INT8_C( -97), INT8_C(  97), INT8_C(  22), INT8_C(   4),
                           INT8_C(  24), INT8_C( 124), INT8_C( -32), INT8_C( -48),
                           INT8_C( -74), INT8_C(  95), INT8_C(  95), INT8_C( -38),
                           INT8_C(  54), INT8_C( 120), INT8_C( -65), INT8_C( -96),
                           INT8_C( -44), INT8_C( -78), INT8_C( -95), INT8_C(-111)),
      simde_mm256_set_epi8(INT8_C(  66), INT8_C(  48), INT8_C(  60), INT8_C(  -4),
                           INT8_C(  33), INT8_C(  67), INT8_C(  97), INT8_C(  65),
                           INT8_C(  80), INT8_C(-109), INT8_C(  88), INT8_C( -23),
                           INT8_C( 107), INT8_C( -65), INT8_C( -65), INT8_C(  62),
                           INT8_C(  90), INT8_C( 118), INT8_C(  25), INT8_C( 105),
                           INT8_C( 109), INT8_C( -45), INT8_C(   4), INT8_C(-107),
                           INT8_C(  55), INT8_C(-101), INT8_C( -65), INT8_C( 121),
                           INT8_C( -76), INT8_C(  55), INT8_C(   6), INT8_C(  86)),
      simde_mm256_set_epi8(INT8_C( -93), INT8_C( -58), INT8_C( -35), INT8_C( -14),
                           INT8_C( 118), INT8_C(-128), INT8_C( -97), INT8_C(  97),
                           INT8_C(  22), INT8_C(   4), INT8_C(  66), INT8_C(  48),
                           INT8_C(  60), INT8_C(  -4), INT8_C(  33), INT8_C(  67),
                           INT8_C(  95), INT8_C( -38), INT8_C(  54), INT8_C( 120),
                           INT8_C( -65), INT8_C( -96), INT8_C( -44), INT8_C( -78),
                           INT8_C( -95), INT8_C(-111), INT8_C(  90), INT8_C( 118),
                           INT8_C(  25), INT8_C( 105), INT8_C( 109), INT8_C( -45)) },
    { simde_mm256_set_epi8(INT8_C(  -1), INT8_C(  93), INT8_C( 114), INT8_C(  49),
                           INT8_C( 120), INT8_C( -70), INT8_C(-112), INT8_C(  19),
                           INT8_C(  56), INT8_C( -12), INT8_C( -77), INT8_C( -85),
                           INT8_C( -59), INT8_C(  80), INT8_C( -47), INT8_C( -28),
                           INT8_C( -33), INT8_C(  73), INT8_C( -24), INT8_C(-106),
                           INT8_C(-122), INT8_C(-111), INT8_C(  -3), INT8_C(  57),
                           INT8_C(  67), INT8_C( 107), INT8_C( -68), INT8_C(-101),
                           INT8_C(  -9), INT8_C(   5), INT8_C(-124), INT8_C( -69)),
      simde_mm256_set_epi8(INT8_C( -67), INT8_C(  65), INT8_C( -72), INT8_C( -90),
                           INT8_C( -44), INT8_C(   2), INT8_C(  39), INT8_C( -45),
                           INT8_C(  51), INT8_C( -30), INT8_C( -47), INT8_C(   1),
                           INT8_C( -91), INT8_C( -40), INT8_C( -73), INT8_C(  33),
                           INT8_C(-117), INT8_C( -31), INT8_C(-117), INT8_C(  60),
                           INT8_C( -36), INT8_C(-120), INT8_C( -10), INT8_C( -12),
                           INT8_C( -25), INT8_C(-103), INT8_C(  48), INT8_C( -50),
                           INT8_C( 101), INT8_C(  93), INT8_C( -31), INT8_C( -33)),
      simde_mm256_set_epi8(INT8_C(-112), INT8_C(  19), INT8_C(  56), INT8_C( -12),
                           INT8_C( -77), INT8_C( -85), INT8_C( -59), INT8_C(  80),
                           INT8_C( -47), INT8_C( -28), INT8_C( -67), INT8_C(  65),
                           INT8_C( -72), INT8_C( -90), INT8_C( -44), INT8_C(   2),
                           INT8_C(  -3), INT8_C(  57), INT8_C(  67), INT8_C( 107),
                           INT8_C( -68), INT8_C(-101), INT8_C(  -9), INT8_C(   5),
                           INT8_C(-124), INT8_C( -69), INT8_C(-117), INT8_C( -31),
                           INT8_C(-117), INT8_C(  60), INT8_C( -36), INT8_C(-120)) },
    { simde_mm256_set_epi8(INT8_C( 114), INT8_C( -28), INT8_C( -81), INT8_C(-126),
                           INT8_C(-114), INT8_C( -94), INT8_C(-101), INT8_C(  52),
                           INT8_C(  30), INT8_C(  66), INT8_C(-105), INT8_C(  50),
                           INT8_C(  48), INT8_C(  15), INT8_C(   1), INT8_C(-106),
                           INT8_C( -36), INT8_C(  61), INT8_C( -18), INT8_C(   8),
                           INT8_C(  89), INT8_C( -23), INT8_C(  58), INT8_C( -29),
                           INT8_C( 110), INT8_C(  57), INT8_C( -39), INT8_C(   1),
                           INT8_C( -60), INT8_C(  60), INT8_C( -80), INT8_C( -20)),
      simde_mm256_set_epi8(INT8_C( -60), INT8_C(  69), INT8_C( -69), INT8_C( 107),
                           INT8_C(   2), INT8_C( -33), INT8_C(  14), INT8_C(  77),
                           INT8_C(  15), INT8_C(  63), INT8_C(  63), INT8_C( -53),
                           INT8_C(  -9), INT8_C(  76), INT8_C(   7), INT8_C( -76),
                           INT8_C( -23), INT8_C(-107), INT8_C(  84), INT8_C(  37),
                           INT8_C( -18), INT8_C(  47), INT8_C(  58), INT8_C(  31),
                           INT8_C(  69), INT8_C(  12), INT8_C(  46), INT8_C( 101),
                           INT8_C( -38), INT8_C( 117), INT8_C(-113), INT8_C( -36)),
      simde_mm256_set_epi8(INT8_C(-101), INT8_C(  52), INT8_C(  30), INT8_C(  66),
                           INT8_C(-105), INT8_C(  50), INT8_C(  48), INT8_C(  15),
                           INT8_C(   1), INT8_C(-106), INT8_C( -60), INT8_C(  69),
                           INT8_C( -69), INT8_C( 107), INT8_C(   2), INT8_C( -33),
                           INT8_C(  58), INT8_C( -29), INT8_C( 110), INT8_C(  57),
                           INT8_C( -39), INT8_C(   1), INT8_C( -60), INT8_C(  60),
                           INT8_C( -80), INT8_C( -20), INT8_C( -23), INT8_C(-107),
                           INT8_C(  84), INT8_C(  37), INT8_C( -18), INT8_C(  47)) },
    { simde_mm256_set_epi8(INT8_C(  13), INT8_C( -32), INT8_C( -51), INT8_C(  49),
                           INT8_C(  15), INT8_C(  15), INT8_C( -66), INT8_C( -38),
                           INT8_C(  69), INT8_C(   1), INT8_C( -50), INT8_C(-103),
                           INT8_C(  72), INT8_C( -87), INT8_C(-104), INT8_C( -58),
                           INT8_C(  63), INT8_C(-120), INT8_C(  36), INT8_C( -61),
                           INT8_C( 106), INT8_C(  97), INT8_C(-128), INT8_C(  26),
                           INT8_C(-121), INT8_C( -27), INT8_C( 107), INT8_C(-112),
                           INT8_C(  57), INT8_C(  39), INT8_C(  98), INT8_C( 126)),
      simde_mm256_set_epi8(INT8_C(-110), INT8_C(  50), INT8_C( 102), INT8_C( -85),
                           INT8_C( -62), INT8_C(  74), INT8_C( 102), INT8_C(  36),
                           INT8_C( -55), INT8_C(-104), INT8_C(  11), INT8_C( -70),
                           INT8_C( -20), INT8_C(-104), INT8_C( -37), INT8_C( 122),
                           INT8_C( -88), INT8_C(  70), INT8_C(  43), INT8_C( -34),
                           INT8_C(  14), INT8_C(  55), INT8_C( -41), INT8_C(  33),
                           INT8_C(  53), INT8_C(-102), INT8_C(  56), INT8_C(  26),
                           INT8_C( -79), INT8_C( -30), INT8_C(  73), INT8_C( -53)),
      simde_mm256_set_epi8(INT8_C( -66), INT8_C( -38), INT8_C(  69), INT8_C(   1),
                           INT8_C( -50), INT8_C(-103), INT8_C(  72), INT8_C( -87),
                           INT8_C(-104), INT8_C( -58), INT8_C(-110), INT8_C(  50),
                           INT8_C( 102), INT8_C( -85), INT8_C( -62), INT8_C(  74),
                           INT8_C(-128), INT8_C(  26), INT8_C(-121), INT8_C( -27),
                           INT8_C( 107), INT8_C(-112), INT8_C(  57), INT8_C(  39),
                           INT8_C(  98), INT8_C( 126), INT8_C( -88), INT8_C(  70),
                           INT8_C(  43), INT8_C( -34), INT8_C(  14), INT8_C(  55)) },
    { simde_mm256_set_epi8(INT8_C(-103), INT8_C(  23), INT8_C( -90), INT8_C(  43),
                           INT8_C(  -8), INT8_C(  54), INT8_C( -93), INT8_C(  64),
                           INT8_C(-116), INT8_C( -76), INT8_C( -53), INT8_C(  35),
                           INT8_C(  51), INT8_C( -86), INT8_C(   8), INT8_C(  43),
                           INT8_C( -60), INT8_C(-113), INT8_C(  44), INT8_C(-102),
                           INT8_C(  77), INT8_C(  95), INT8_C(  57), INT8_C( 124),
                           INT8_C( -91), INT8_C( -87), INT8_C( -54), INT8_C( -83),
                           INT8_C( 108), INT8_C(  76), INT8_C(  90), INT8_C( -49)),
      simde_mm256_set_epi8(INT8_C( -43), INT8_C(  -5), INT8_C(  94), INT8_C(  76),
                           INT8_C(  -9), INT8_C( 110), INT8_C( -82), INT8_C( 112),
                           INT8_C( -25), INT8_C(  11), INT8_C( -46), INT8_C( -57),
                           INT8_C( -13), INT8_C(-127), INT8_C(  10), INT8_C(-113),
                           INT8_C(  55), INT8_C(  85), INT8_C(  75), INT8_C(  61),
                           INT8_C(  49), INT8_C( -14), INT8_C(  44), INT8_C(-128),
                           INT8_C(-102), INT8_C( -17), INT8_C(  64), INT8_C( -73),
                           INT8_C(  14), INT8_C(  19), INT8_C(  -2), INT8_C(   7)),
      simde_mm256_set_epi8(INT8_C( -93), INT8_C(  64), INT8_C(-116), INT8_C( -76),
                           INT8_C( -53), INT8_C(  35), INT8_C(  51), INT8_C( -86),
                           INT8_C(   8), INT8_C(  43), INT8_C( -43), INT8_C(  -5),
                           INT8_C(  94), INT8_C(  76), INT8_C(  -9), INT8_C( 110),
                           INT8_C(  57), INT8_C( 124), INT8_C( -91), INT8_C( -87),
                           INT8_C( -54), INT8_C( -83), INT8_C( 108), INT8_C(  76),
                           INT8_C(  90), INT8_C( -49), INT8_C(  55), INT8_C(  85),
                           INT8_C(  75), INT8_C(  61), INT8_C(  49), INT8_C( -14)) },
    { simde_mm256_set_epi8(INT8_C( -32), INT8_C( -82), INT8_C(  79), INT8_C(  79),
                           INT8_C( 127), INT8_C(  16), INT8_C(-118), INT8_C( 113),
                           INT8_C( -52), INT8_C( -25), INT8_C(-124), INT8_C( -77),
                           INT8_C( 114), INT8_C(-118), INT8_C( -84), INT8_C(  73),
                           INT8_C( -12), INT8_C( -67), INT8_C(-127), INT8_C( -69),
                           INT8_C(  30), INT8_C(  14), INT8_C( -47), INT8_C(-119),
                           INT8_C( -36), INT8_C( -97), INT8_C( -89), INT8_C(  24),
                           INT8_C( -41), INT8_C( 117), INT8_C(-102), INT8_C(  54)),
      simde_mm256_set_epi8(INT8_C( -66), INT8_C( -50), INT8_C(  39), INT8_C(  30),
                           INT8_C(  53), INT8_C( 108), INT8_C( -71), INT8_C( -62),
                           INT8_C(  58), INT8_C( -69), INT8_C( -39), INT8_C( -28),
                           INT8_C(  58), INT8_C(  54), INT8_C(-117), INT8_C( -49),
                           INT8_C( 126), INT8_C( -71), INT8_C( -39), INT8_C(-120),
                           INT8_C(  88), INT8_C(  41), INT8_C( 127), INT8_C( 111),
                           INT8_C( -32), INT8_C( -18), INT8_C(  12), INT8_C(  97),
                           INT8_C(  83), INT8_C( -96), INT8_C(  13), INT8_C(-101)),
      simde_mm256_set_epi8(INT8_C(-118), INT8_C( 113), INT8_C( -52), INT8_C( -25),
                           INT8_C(-124), INT8_C( -77), INT8_C( 114), INT8_C(-118),
                           INT8_C( -84), INT8_C(  73), INT8_C( -66), INT8_C( -50),
                           INT8_C(  39), INT8_C(  30), INT8_C(  53), INT8_C( 108),
                           INT8_C( -47), INT8_C(-119), INT8_C( -36), INT8_C( -97),
                           INT8_C( -89), INT8_C(  24), INT8_C( -41), INT8_C( 117),
                           INT8_C(-102), INT8_C(  54), INT8_C( 126), INT8_C( -71),
                           INT8_C( -39), INT8_C(-120), INT8_C(  88), INT8_C(  41)) },
    { simde_mm256_set_epi8(INT8_C(-105), INT8_C(   9), INT8_C( -52), INT8_C(  54),
                           INT8_C( -11), INT8_C( -39), INT8_C( -64), INT8_C( -79),
                           INT8_C(  88), INT8_C(  52), INT8_C( -95), INT8_C( -70),
                           INT8_C(  46), INT8_C( 121), INT8_C( -22), INT8_C( 113),
                           INT8_C(  26), INT8_C( 104), INT8_C( -52), INT8_C( 109),
                           INT8_C(   5), INT8_C(  87), INT8_C(  90), INT8_C(  -7),
                           INT8_C( -35), INT8_C(-112), INT8_C(  20), INT8_C(-128),
                           INT8_C( -21), INT8_C( -67), INT8_C(  37), INT8_C( -35)),
      simde_mm256_set_epi8(INT8_C(  66), INT8_C( -23), INT8_C(  45), INT8_C( -38),
                           INT8_C( -94), INT8_C( 119), INT8_C( -23), INT8_C(  70),
                           INT8_C( -75), INT8_C(  25), INT8_C(  79), INT8_C(-101),
                           INT8_C(-122), INT8_C(-107), INT8_C( -10), INT8_C(-109),
                           INT8_C( -96), INT8_C(  45), INT8_C(  31), INT8_C( -60),
                           INT8_C( 100), INT8_C( 115), INT8_C(  79), INT8_C(  38),
                           INT8_C( -35), INT8_C(  57), INT8_C(  13), INT8_C(  93),
                           INT8_C(  54), INT8_C( -61), INT8_C( 101), INT8_C(-113)),
      simde_mm256_set_epi8(INT8_C( -64), INT8_C( -79), INT8_C(  88), INT8_C(  52),
                           INT8_C( -95), INT8_C( -70), INT8_C(  46), INT8_C( 121),
                           INT8_C( -22), INT8_C( 113), INT8_C(  66), INT8_C( -23),
                           INT8_C(  45), INT8_C( -38), INT8_C( -94), INT8_C( 119),
                           INT8_C(  90), INT8_C(  -7), INT8_C( -35), INT8_C(-112),
                           INT8_C(  20), INT8_C(-128), INT8_C( -21), INT8_C( -67),
                           INT8_C(  37), INT8_C( -35), INT8_C( -96), INT8_C(  45),
                           INT8_C(  31), INT8_C( -60), INT8_C( 100), INT8_C( 115)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_alignr_epi8(test_vec[i].a, test_vec[i].b, 10);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_alignr_epi8_case2(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C( 113), INT8_C( -69), INT8_C(  23), INT8_C( -66),
                           INT8_C( 115), INT8_C( -83), INT8_C( -66), INT8_C( -71),
                           INT8_C(  28), INT8_C(  74), INT8_C(  -4), INT8_C(  16),
                           INT8_C(  16), INT8_C( -20), INT8_C(  79), INT8_C( -49),
                           INT8_C(  13), INT8_C(  59), INT8_C( -99), INT8_C(  73),
                           INT8_C(  19), INT8_C(  31), INT8_C( -64), INT8_C( -94),
                           INT8_C( 111), INT8_C( 121), INT8_C( -91), INT8_C( -99),
                           INT8_C(-120), INT8_C(  24), INT8_C(-116), INT8_C( -29)),
      simde_mm256_set_epi8(INT8_C(  19), INT8_C( 117), INT8_C(  83), INT8_C(  54),
                           INT8_C( -84), INT8_C(-110), INT8_C( 115), INT8_C(  25),
                           INT8_C( 122), INT8_C( -20), INT8_C(  49), INT8_C( -87),
                           INT8_C(   9), INT8_C( -79), INT8_C(  28), INT8_C( -93),
                           INT8_C( -97), INT8_C(  38), INT8_C(  85), INT8_C( -70),
                           INT8_C(  96), INT8_C(   7), INT8_C( -55), INT8_C(  -8),
                           INT8_C(  94), INT8_C(  69), INT8_C(  20), INT8_C(  64),
                           INT8_C(  54), INT8_C(  88), INT8_C( -92), INT8_C( -89)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C( 113), INT8_C( -69), INT8_C(  23),
                           INT8_C( -66), INT8_C( 115), INT8_C( -83), INT8_C( -66),
                           INT8_C( -71), INT8_C(  28), INT8_C(  74), INT8_C(  -4),
                           INT8_C(  16), INT8_C(  16), INT8_C( -20), INT8_C(  79),
                           INT8_C(   0), INT8_C(  13), INT8_C(  59), INT8_C( -99),
                           INT8_C(  73), INT8_C(  19), INT8_C(  31), INT8_C( -64),
                           INT8_C( -94), INT8_C( 111), INT8_C( 121), INT8_C( -91),
                           INT8_C( -99), INT8_C(-120), INT8_C(  24), INT8_C(-116)) },
    { simde_mm256_set_epi8(INT8_C(  -7), INT8_C( -66), INT8_C(-120), INT8_C( 103),
                           INT8_C( -48), INT8_C( -55), INT8_C(-109), INT8_C(  52),
                           INT8_C( 103), INT8_C(  30), INT8_C( -21), INT8_C(-107),
                           INT8_C(  14), INT8_C(  -9), INT8_C(  61), INT8_C(-114),
                           INT8_C(-120), INT8_C(-116), INT8_C( -34), INT8_C( -96),
                           INT8_C( 124), INT8_C(  29), INT8_C( -49), INT8_C(  74),
                           INT8_C( -82), INT8_C(  66), INT8_C(  24), INT8_C(  47),
                           INT8_C(  -7), INT8_C( -96), INT8_C(  99), INT8_C( -53)),
      simde_mm256_set_epi8(INT8_C(  77), INT8_C(  25), INT8_C(  25), INT8_C(  69),
                           INT8_C( -34), INT8_C(   2), INT8_C(  12), INT8_C(  -5),
                           INT8_C( -49), INT8_C( -58), INT8_C( -38), INT8_C( -80),
                           INT8_C(  88), INT8_C(  48), INT8_C(  49), INT8_C(  73),
                           INT8_C( 121), INT8_C(  39), INT8_C( -20), INT8_C(  28),
                           INT8_C( 121), INT8_C(   0), INT8_C( -34), INT8_C(  24),
                           INT8_C( -45), INT8_C(-121), INT8_C( -12), INT8_C(  37),
                           INT8_C(  64), INT8_C(  31), INT8_C(  31), INT8_C(  41)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(  -7), INT8_C( -66), INT8_C(-120),
                           INT8_C( 103), INT8_C( -48), INT8_C( -55), INT8_C(-109),
                           INT8_C(  52), INT8_C( 103), INT8_C(  30), INT8_C( -21),
                           INT8_C(-107), INT8_C(  14), INT8_C(  -9), INT8_C(  61),
                           INT8_C(   0), INT8_C(-120), INT8_C(-116), INT8_C( -34),
                           INT8_C( -96), INT8_C( 124), INT8_C(  29), INT8_C( -49),
                           INT8_C(  74), INT8_C( -82), INT8_C(  66), INT8_C(  24),
                           INT8_C(  47), INT8_C(  -7), INT8_C( -96), INT8_C(  99)) },
    { simde_mm256_set_epi8(INT8_C(  -1), INT8_C(  20), INT8_C( -95), INT8_C( 112),
                           INT8_C(  49), INT8_C(-103), INT8_C( 108), INT8_C( -46),
                           INT8_C( -85), INT8_C( -99), INT8_C(  78), INT8_C( 125),
                           INT8_C( -66), INT8_C( -37), INT8_C(  94), INT8_C( 112),
                           INT8_C(-126), INT8_C( -37), INT8_C(-124), INT8_C( -71),
                           INT8_C( -21), INT8_C( -79), INT8_C(  -2), INT8_C(  34),
                           INT8_C( -56), INT8_C(  22), INT8_C(-102), INT8_C(  53),
                           INT8_C( -24), INT8_C( -87), INT8_C(  16), INT8_C( -85)),
      simde_mm256_set_epi8(INT8_C(  66), INT8_C( -80), INT8_C(  10), INT8_C( -95),
                           INT8_C( -33), INT8_C(  65), INT8_C(-115), INT8_C( -97),
                           INT8_C( -96), INT8_C(  17), INT8_C( 123), INT8_C( -81),
                           INT8_C( -25), INT8_C( -92), INT8_C( 110), INT8_C(  74),
                           INT8_C(  58), INT8_C( 126), INT8_C(  24), INT8_C( -87),
                           INT8_C(-124), INT8_C(  -7), INT8_C( -55), INT8_C( 112),
                           INT8_C(  10), INT8_C(-117), INT8_C( -16), INT8_C(  34),
                           INT8_C( -23), INT8_C( 105), INT8_C(  32), INT8_C(-122)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(  -1), INT8_C(  20), INT8_C( -95),
                           INT8_C( 112), INT8_C(  49), INT8_C(-103), INT8_C( 108),
                           INT8_C( -46), INT8_C( -85), INT8_C( -99), INT8_C(  78),
                           INT8_C( 125), INT8_C( -66), INT8_C( -37), INT8_C(  94),
                           INT8_C(   0), INT8_C(-126), INT8_C( -37), INT8_C(-124),
                           INT8_C( -71), INT8_C( -21), INT8_C( -79), INT8_C(  -2),
                           INT8_C(  34), INT8_C( -56), INT8_C(  22), INT8_C(-102),
                           INT8_C(  53), INT8_C( -24), INT8_C( -87), INT8_C(  16)) },
    { simde_mm256_set_epi8(INT8_C(  25), INT8_C( -64), INT8_C( 105), INT8_C( -19),
                           INT8_C(-100), INT8_C( 119), INT8_C(  53), INT8_C(-128),
                           INT8_C(  50), INT8_C(-115), INT8_C(  36), INT8_C( 126),
                           INT8_C(   7), INT8_C( -21), INT8_C(-106), INT8_C(-120),
                           INT8_C( 127), INT8_C( -66), INT8_C(  -1), INT8_C( -93),
                           INT8_C(  -5), INT8_C(  40), INT8_C(  78), INT8_C(-119),
                           INT8_C(  82), INT8_C( -17), INT8_C(  18), INT8_C(  47),
                           INT8_C(   8), INT8_C(-108), INT8_C(-114), INT8_C(  65)),
      simde_mm256_set_epi8(INT8_C(-113), INT8_C( 113), INT8_C(-118), INT8_C( -40),
                           INT8_C(-102), INT8_C( -62), INT8_C(  64), INT8_C( -33),
                           INT8_C(  77), INT8_C( -83), INT8_C( -74), INT8_C(  19),
                           INT8_C(  71), INT8_C(  17), INT8_C( -17), INT8_C( -35),
                           INT8_C(  93), INT8_C( -41), INT8_C(  11), INT8_C(   0),
                           INT8_C( -41), INT8_C( -66), INT8_C( -72), INT8_C( 117),
                           INT8_C( 100), INT8_C(-122), INT8_C(  24), INT8_C(-112),
                           INT8_C(  87), INT8_C(  37), INT8_C( -17), INT8_C( -51)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(  25), INT8_C( -64), INT8_C( 105),
                           INT8_C( -19), INT8_C(-100), INT8_C( 119), INT8_C(  53),
                           INT8_C(-128), INT8_C(  50), INT8_C(-115), INT8_C(  36),
                           INT8_C( 126), INT8_C(   7), INT8_C( -21), INT8_C(-106),
                           INT8_C(   0), INT8_C( 127), INT8_C( -66), INT8_C(  -1),
                           INT8_C( -93), INT8_C(  -5), INT8_C(  40), INT8_C(  78),
                           INT8_C(-119), INT8_C(  82), INT8_C( -17), INT8_C(  18),
                           INT8_C(  47), INT8_C(   8), INT8_C(-108), INT8_C(-114)) },
    { simde_mm256_set_epi8(INT8_C( -71), INT8_C(  29), INT8_C( -63), INT8_C( -99),
                           INT8_C(  16), INT8_C(  67), INT8_C(-118), INT8_C(  12),
                           INT8_C( -46), INT8_C( 113), INT8_C(-128), INT8_C(-114),
                           INT8_C( -51), INT8_C( -75), INT8_C(  97), INT8_C( -73),
                           INT8_C(  24), INT8_C( -63), INT8_C(  20), INT8_C( -26),
                           INT8_C( -44), INT8_C(  71), INT8_C( 109), INT8_C( 113),
                           INT8_C(  36), INT8_C( 100), INT8_C(  42), INT8_C( -55),
                           INT8_C(  12), INT8_C( -66), INT8_C( -91), INT8_C(  82)),
      simde_mm256_set_epi8(INT8_C(  81), INT8_C(  93), INT8_C(  76), INT8_C(  23),
                           INT8_C( -32), INT8_C(  31), INT8_C(  25), INT8_C( -76),
                           INT8_C(  38), INT8_C( 114), INT8_C(  74), INT8_C( -11),
                           INT8_C(  55), INT8_C( -33), INT8_C( -78), INT8_C( -96),
                           INT8_C( 106), INT8_C(   1), INT8_C(  32), INT8_C( -46),
                           INT8_C( 124), INT8_C(  89), INT8_C( -80), INT8_C( -22),
                           INT8_C(-117), INT8_C( -99), INT8_C( -39), INT8_C( -63),
                           INT8_C(  12), INT8_C( 106), INT8_C( 120), INT8_C(  64)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C( -71), INT8_C(  29), INT8_C( -63),
                           INT8_C( -99), INT8_C(  16), INT8_C(  67), INT8_C(-118),
                           INT8_C(  12), INT8_C( -46), INT8_C( 113), INT8_C(-128),
                           INT8_C(-114), INT8_C( -51), INT8_C( -75), INT8_C(  97),
                           INT8_C(   0), INT8_C(  24), INT8_C( -63), INT8_C(  20),
                           INT8_C( -26), INT8_C( -44), INT8_C(  71), INT8_C( 109),
                           INT8_C( 113), INT8_C(  36), INT8_C( 100), INT8_C(  42),
                           INT8_C( -55), INT8_C(  12), INT8_C( -66), INT8_C( -91)) },
    { simde_mm256_set_epi8(INT8_C(  91), INT8_C( -47), INT8_C( -11), INT8_C(  95),
                           INT8_C(-101), INT8_C(  71), INT8_C( -99), INT8_C( 119),
                           INT8_C(  53), INT8_C( -40), INT8_C(  93), INT8_C( -71),
                           INT8_C(  53), INT8_C( 101), INT8_C(  91), INT8_C(  38),
                           INT8_C( 101), INT8_C(  10), INT8_C(   7), INT8_C(  37),
                           INT8_C( -66), INT8_C(   4), INT8_C( -96), INT8_C( -47),
                           INT8_C( -72), INT8_C(-123), INT8_C( -18), INT8_C( -16),
                           INT8_C(  57), INT8_C(  -1), INT8_C(-122), INT8_C(  93)),
      simde_mm256_set_epi8(INT8_C( 125), INT8_C( -97), INT8_C( -63), INT8_C(  47),
                           INT8_C(  76), INT8_C( -11), INT8_C(  32), INT8_C(  96),
                           INT8_C( 121), INT8_C( -93), INT8_C(  72), INT8_C( -40),
                           INT8_C( -11), INT8_C( -76), INT8_C(  51), INT8_C(-122),
                           INT8_C( 108), INT8_C( -61), INT8_C(  -9), INT8_C( -38),
                           INT8_C( -65), INT8_C( -53), INT8_C( -36), INT8_C(  43),
                           INT8_C( -52), INT8_C( -14), INT8_C( -72), INT8_C(  79),
                           INT8_C(   4), INT8_C(  96), INT8_C( -88), INT8_C(  40)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(  91), INT8_C( -47), INT8_C( -11),
                           INT8_C(  95), INT8_C(-101), INT8_C(  71), INT8_C( -99),
                           INT8_C( 119), INT8_C(  53), INT8_C( -40), INT8_C(  93),
                           INT8_C( -71), INT8_C(  53), INT8_C( 101), INT8_C(  91),
                           INT8_C(   0), INT8_C( 101), INT8_C(  10), INT8_C(   7),
                           INT8_C(  37), INT8_C( -66), INT8_C(   4), INT8_C( -96),
                           INT8_C( -47), INT8_C( -72), INT8_C(-123), INT8_C( -18),
                           INT8_C( -16), INT8_C(  57), INT8_C(  -1), INT8_C(-122)) },
    { simde_mm256_set_epi8(INT8_C(-125), INT8_C(  50), INT8_C(   9), INT8_C(-114),
                           INT8_C(  12), INT8_C(-115), INT8_C(  20), INT8_C( -81),
                           INT8_C(  65), INT8_C(  -9), INT8_C(   1), INT8_C(  -7),
                           INT8_C(  48), INT8_C(  81), INT8_C( -58), INT8_C( -29),
                           INT8_C(-102), INT8_C(  98), INT8_C( 126), INT8_C(  49),
                           INT8_C(  95), INT8_C(  45), INT8_C( -60), INT8_C( -92),
                           INT8_C(  -7), INT8_C( -56), INT8_C( -47), INT8_C( -85),
                           INT8_C( -93), INT8_C(  33), INT8_C( -91), INT8_C( -34)),
      simde_mm256_set_epi8(INT8_C(  88), INT8_C( -22), INT8_C( -14), INT8_C(  72),
                           INT8_C(-106), INT8_C( -75), INT8_C( -40), INT8_C( -29),
                           INT8_C(  71), INT8_C(  52), INT8_C( -19), INT8_C( -85),
                           INT8_C(  47), INT8_C( -19), INT8_C( -33), INT8_C( -99),
                           INT8_C( -95), INT8_C(-103), INT8_C( 113), INT8_C(  75),
                           INT8_C(  16), INT8_C(-125), INT8_C(  51), INT8_C( -20),
                           INT8_C( -87), INT8_C(  79), INT8_C(  50), INT8_C( -45),
                           INT8_C( -77), INT8_C( -89), INT8_C(  25), INT8_C( -53)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(-125), INT8_C(  50), INT8_C(   9),
                           INT8_C(-114), INT8_C(  12), INT8_C(-115), INT8_C(  20),
                           INT8_C( -81), INT8_C(  65), INT8_C(  -9), INT8_C(   1),
                           INT8_C(  -7), INT8_C(  48), INT8_C(  81), INT8_C( -58),
                           INT8_C(   0), INT8_C(-102), INT8_C(  98), INT8_C( 126),
                           INT8_C(  49), INT8_C(  95), INT8_C(  45), INT8_C( -60),
                           INT8_C( -92), INT8_C(  -7), INT8_C( -56), INT8_C( -47),
                           INT8_C( -85), INT8_C( -93), INT8_C(  33), INT8_C( -91)) },
    { simde_mm256_set_epi8(INT8_C( 108), INT8_C( -54), INT8_C(  79), INT8_C( -34),
                           INT8_C(  79), INT8_C( 114), INT8_C(  95), INT8_C( -13),
                           INT8_C( -32), INT8_C( 106), INT8_C(   6), INT8_C(-103),
                           INT8_C( -17), INT8_C( 126), INT8_C( -76), INT8_C( 100),
                           INT8_C( 103), INT8_C( -73), INT8_C(-120), INT8_C(  89),
                           INT8_C(  92), INT8_C(   1), INT8_C(  24), INT8_C( -42),
                           INT8_C( -87), INT8_C(-120), INT8_C( -69), INT8_C( 118),
                           INT8_C( -71), INT8_C(  35), INT8_C(  31), INT8_C( -27)),
      simde_mm256_set_epi8(INT8_C( 106), INT8_C(  34), INT8_C( -93), INT8_C( -41),
                           INT8_C(  92), INT8_C(  50), INT8_C(-109), INT8_C(  24),
                           INT8_C(  49), INT8_C( -59), INT8_C( -13), INT8_C(  39),
                           INT8_C( 111), INT8_C( -38), INT8_C(  26), INT8_C(  -9),
                           INT8_C(-106), INT8_C(-118), INT8_C( 104), INT8_C( -38),
                           INT8_C( 127), INT8_C( -21), INT8_C(  72), INT8_C(-105),
                           INT8_C( 100), INT8_C(  65), INT8_C(  70), INT8_C(-111),
                           INT8_C(  93), INT8_C(-113), INT8_C( -69), INT8_C( -81)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C( 108), INT8_C( -54), INT8_C(  79),
                           INT8_C( -34), INT8_C(  79), INT8_C( 114), INT8_C(  95),
                           INT8_C( -13), INT8_C( -32), INT8_C( 106), INT8_C(   6),
                           INT8_C(-103), INT8_C( -17), INT8_C( 126), INT8_C( -76),
                           INT8_C(   0), INT8_C( 103), INT8_C( -73), INT8_C(-120),
                           INT8_C(  89), INT8_C(  92), INT8_C(   1), INT8_C(  24),
                           INT8_C( -42), INT8_C( -87), INT8_C(-120), INT8_C( -69),
                           INT8_C( 118), INT8_C( -71), INT8_C(  35), INT8_C(  31)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_alignr_epi8(test_vec[i].a, test_vec[i].b, 17);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_alignr_epi8_case3(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C(  57), INT8_C( -47), INT8_C( -81), INT8_C( -95),
                           INT8_C(-100), INT8_C( -69), INT8_C( -75), INT8_C(-127),
                           INT8_C(  -8), INT8_C( -83), INT8_C( -94), INT8_C(-113),
                           INT8_C( 107), INT8_C( -39), INT8_C( -99), INT8_C(  46),
                           INT8_C( -61), INT8_C( -72), INT8_C(  -5), INT8_C(-126),
                           INT8_C( 113), INT8_C(-120), INT8_C(  -9), INT8_C( 117),
                           INT8_C( 121), INT8_C( -14), INT8_C( -58), INT8_C(-118),
                           INT8_C( -70), INT8_C( -35), INT8_C(-100), INT8_C(  48)),
      simde_mm256_set_epi8(INT8_C( -98), INT8_C(  80), INT8_C( -16), INT8_C(  73),
                           INT8_C( 126), INT8_C(-115), INT8_C(  31), INT8_C(  80),
                           INT8_C( -75), INT8_C( 117), INT8_C( -13), INT8_C( -70),
                           INT8_C(-101), INT8_C( -10), INT8_C(-111), INT8_C(  96),
                           INT8_C( -77), INT8_C(   0), INT8_C(  70), INT8_C(-121),
                           INT8_C(  78), INT8_C(  54), INT8_C(  38), INT8_C(  18),
                           INT8_C( -12), INT8_C(  68), INT8_C(  53), INT8_C(  45),
                           INT8_C(  77), INT8_C( 103), INT8_C( 118), INT8_C( 116)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  57), INT8_C( -47),
                           INT8_C( -81), INT8_C( -95), INT8_C(-100), INT8_C( -69),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -61), INT8_C( -72),
                           INT8_C(  -5), INT8_C(-126), INT8_C( 113), INT8_C(-120)) },
    { simde_mm256_set_epi8(INT8_C(   8), INT8_C(  70), INT8_C(  96), INT8_C(  42),
                           INT8_C(  73), INT8_C( -91), INT8_C( -93), INT8_C( -58),
                           INT8_C( -35), INT8_C( -14), INT8_C( 118), INT8_C(-128),
                           INT8_C( -97), INT8_C(  97), INT8_C(  22), INT8_C(   4),
                           INT8_C(  24), INT8_C( 124), INT8_C( -32), INT8_C( -48),
                           INT8_C( -74), INT8_C(  95), INT8_C(  95), INT8_C( -38),
                           INT8_C(  54), INT8_C( 120), INT8_C( -65), INT8_C( -96),
                           INT8_C( -44), INT8_C( -78), INT8_C( -95), INT8_C(-111)),
      simde_mm256_set_epi8(INT8_C(  66), INT8_C(  48), INT8_C(  60), INT8_C(  -4),
                           INT8_C(  33), INT8_C(  67), INT8_C(  97), INT8_C(  65),
                           INT8_C(  80), INT8_C(-109), INT8_C(  88), INT8_C( -23),
                           INT8_C( 107), INT8_C( -65), INT8_C( -65), INT8_C(  62),
                           INT8_C(  90), INT8_C( 118), INT8_C(  25), INT8_C( 105),
                           INT8_C( 109), INT8_C( -45), INT8_C(   4), INT8_C(-107),
                           INT8_C(  55), INT8_C(-101), INT8_C( -65), INT8_C( 121),
                           INT8_C( -76), INT8_C(  55), INT8_C(   6), INT8_C(  86)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   8), INT8_C(  70),
                           INT8_C(  96), INT8_C(  42), INT8_C(  73), INT8_C( -91),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  24), INT8_C( 124),
                           INT8_C( -32), INT8_C( -48), INT8_C( -74), INT8_C(  95)) },
    { simde_mm256_set_epi8(INT8_C(  -1), INT8_C(  93), INT8_C( 114), INT8_C(  49),
                           INT8_C( 120), INT8_C( -70), INT8_C(-112), INT8_C(  19),
                           INT8_C(  56), INT8_C( -12), INT8_C( -77), INT8_C( -85),
                           INT8_C( -59), INT8_C(  80), INT8_C( -47), INT8_C( -28),
                           INT8_C( -33), INT8_C(  73), INT8_C( -24), INT8_C(-106),
                           INT8_C(-122), INT8_C(-111), INT8_C(  -3), INT8_C(  57),
                           INT8_C(  67), INT8_C( 107), INT8_C( -68), INT8_C(-101),
                           INT8_C(  -9), INT8_C(   5), INT8_C(-124), INT8_C( -69)),
      simde_mm256_set_epi8(INT8_C( -67), INT8_C(  65), INT8_C( -72), INT8_C( -90),
                           INT8_C( -44), INT8_C(   2), INT8_C(  39), INT8_C( -45),
                           INT8_C(  51), INT8_C( -30), INT8_C( -47), INT8_C(   1),
                           INT8_C( -91), INT8_C( -40), INT8_C( -73), INT8_C(  33),
                           INT8_C(-117), INT8_C( -31), INT8_C(-117), INT8_C(  60),
                           INT8_C( -36), INT8_C(-120), INT8_C( -10), INT8_C( -12),
                           INT8_C( -25), INT8_C(-103), INT8_C(  48), INT8_C( -50),
                           INT8_C( 101), INT8_C(  93), INT8_C( -31), INT8_C( -33)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  -1), INT8_C(  93),
                           INT8_C( 114), INT8_C(  49), INT8_C( 120), INT8_C( -70),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -33), INT8_C(  73),
                           INT8_C( -24), INT8_C(-106), INT8_C(-122), INT8_C(-111)) },
    { simde_mm256_set_epi8(INT8_C( 114), INT8_C( -28), INT8_C( -81), INT8_C(-126),
                           INT8_C(-114), INT8_C( -94), INT8_C(-101), INT8_C(  52),
                           INT8_C(  30), INT8_C(  66), INT8_C(-105), INT8_C(  50),
                           INT8_C(  48), INT8_C(  15), INT8_C(   1), INT8_C(-106),
                           INT8_C( -36), INT8_C(  61), INT8_C( -18), INT8_C(   8),
                           INT8_C(  89), INT8_C( -23), INT8_C(  58), INT8_C( -29),
                           INT8_C( 110), INT8_C(  57), INT8_C( -39), INT8_C(   1),
                           INT8_C( -60), INT8_C(  60), INT8_C( -80), INT8_C( -20)),
      simde_mm256_set_epi8(INT8_C( -60), INT8_C(  69), INT8_C( -69), INT8_C( 107),
                           INT8_C(   2), INT8_C( -33), INT8_C(  14), INT8_C(  77),
                           INT8_C(  15), INT8_C(  63), INT8_C(  63), INT8_C( -53),
                           INT8_C(  -9), INT8_C(  76), INT8_C(   7), INT8_C( -76),
                           INT8_C( -23), INT8_C(-107), INT8_C(  84), INT8_C(  37),
                           INT8_C( -18), INT8_C(  47), INT8_C(  58), INT8_C(  31),
                           INT8_C(  69), INT8_C(  12), INT8_C(  46), INT8_C( 101),
                           INT8_C( -38), INT8_C( 117), INT8_C(-113), INT8_C( -36)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( 114), INT8_C( -28),
                           INT8_C( -81), INT8_C(-126), INT8_C(-114), INT8_C( -94),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -36), INT8_C(  61),
                           INT8_C( -18), INT8_C(   8), INT8_C(  89), INT8_C( -23)) },
    { simde_mm256_set_epi8(INT8_C(  13), INT8_C( -32), INT8_C( -51), INT8_C(  49),
                           INT8_C(  15), INT8_C(  15), INT8_C( -66), INT8_C( -38),
                           INT8_C(  69), INT8_C(   1), INT8_C( -50), INT8_C(-103),
                           INT8_C(  72), INT8_C( -87), INT8_C(-104), INT8_C( -58),
                           INT8_C(  63), INT8_C(-120), INT8_C(  36), INT8_C( -61),
                           INT8_C( 106), INT8_C(  97), INT8_C(-128), INT8_C(  26),
                           INT8_C(-121), INT8_C( -27), INT8_C( 107), INT8_C(-112),
                           INT8_C(  57), INT8_C(  39), INT8_C(  98), INT8_C( 126)),
      simde_mm256_set_epi8(INT8_C(-110), INT8_C(  50), INT8_C( 102), INT8_C( -85),
                           INT8_C( -62), INT8_C(  74), INT8_C( 102), INT8_C(  36),
                           INT8_C( -55), INT8_C(-104), INT8_C(  11), INT8_C( -70),
                           INT8_C( -20), INT8_C(-104), INT8_C( -37), INT8_C( 122),
                           INT8_C( -88), INT8_C(  70), INT8_C(  43), INT8_C( -34),
                           INT8_C(  14), INT8_C(  55), INT8_C( -41), INT8_C(  33),
                           INT8_C(  53), INT8_C(-102), INT8_C(  56), INT8_C(  26),
                           INT8_C( -79), INT8_C( -30), INT8_C(  73), INT8_C( -53)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  13), INT8_C( -32),
                           INT8_C( -51), INT8_C(  49), INT8_C(  15), INT8_C(  15),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  63), INT8_C(-120),
                           INT8_C(  36), INT8_C( -61), INT8_C( 106), INT8_C(  97)) },
    { simde_mm256_set_epi8(INT8_C(-103), INT8_C(  23), INT8_C( -90), INT8_C(  43),
                           INT8_C(  -8), INT8_C(  54), INT8_C( -93), INT8_C(  64),
                           INT8_C(-116), INT8_C( -76), INT8_C( -53), INT8_C(  35),
                           INT8_C(  51), INT8_C( -86), INT8_C(   8), INT8_C(  43),
                           INT8_C( -60), INT8_C(-113), INT8_C(  44), INT8_C(-102),
                           INT8_C(  77), INT8_C(  95), INT8_C(  57), INT8_C( 124),
                           INT8_C( -91), INT8_C( -87), INT8_C( -54), INT8_C( -83),
                           INT8_C( 108), INT8_C(  76), INT8_C(  90), INT8_C( -49)),
      simde_mm256_set_epi8(INT8_C( -43), INT8_C(  -5), INT8_C(  94), INT8_C(  76),
                           INT8_C(  -9), INT8_C( 110), INT8_C( -82), INT8_C( 112),
                           INT8_C( -25), INT8_C(  11), INT8_C( -46), INT8_C( -57),
                           INT8_C( -13), INT8_C(-127), INT8_C(  10), INT8_C(-113),
                           INT8_C(  55), INT8_C(  85), INT8_C(  75), INT8_C(  61),
                           INT8_C(  49), INT8_C( -14), INT8_C(  44), INT8_C(-128),
                           INT8_C(-102), INT8_C( -17), INT8_C(  64), INT8_C( -73),
                           INT8_C(  14), INT8_C(  19), INT8_C(  -2), INT8_C(   7)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(-103), INT8_C(  23),
                           INT8_C( -90), INT8_C(  43), INT8_C(  -8), INT8_C(  54),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -60), INT8_C(-113),
                           INT8_C(  44), INT8_C(-102), INT8_C(  77), INT8_C(  95)) },
    { simde_mm256_set_epi8(INT8_C( -32), INT8_C( -82), INT8_C(  79), INT8_C(  79),
                           INT8_C( 127), INT8_C(  16), INT8_C(-118), INT8_C( 113),
                           INT8_C( -52), INT8_C( -25), INT8_C(-124), INT8_C( -77),
                           INT8_C( 114), INT8_C(-118), INT8_C( -84), INT8_C(  73),
                           INT8_C( -12), INT8_C( -67), INT8_C(-127), INT8_C( -69),
                           INT8_C(  30), INT8_C(  14), INT8_C( -47), INT8_C(-119),
                           INT8_C( -36), INT8_C( -97), INT8_C( -89), INT8_C(  24),
                           INT8_C( -41), INT8_C( 117), INT8_C(-102), INT8_C(  54)),
      simde_mm256_set_epi8(INT8_C( -66), INT8_C( -50), INT8_C(  39), INT8_C(  30),
                           INT8_C(  53), INT8_C( 108), INT8_C( -71), INT8_C( -62),
                           INT8_C(  58), INT8_C( -69), INT8_C( -39), INT8_C( -28),
                           INT8_C(  58), INT8_C(  54), INT8_C(-117), INT8_C( -49),
                           INT8_C( 126), INT8_C( -71), INT8_C( -39), INT8_C(-120),
                           INT8_C(  88), INT8_C(  41), INT8_C( 127), INT8_C( 111),
                           INT8_C( -32), INT8_C( -18), INT8_C(  12), INT8_C(  97),
                           INT8_C(  83), INT8_C( -96), INT8_C(  13), INT8_C(-101)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -32), INT8_C( -82),
                           INT8_C(  79), INT8_C(  79), INT8_C( 127), INT8_C(  16),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -12), INT8_C( -67),
                           INT8_C(-127), INT8_C( -69), INT8_C(  30), INT8_C(  14)) },
    { simde_mm256_set_epi8(INT8_C(-105), INT8_C(   9), INT8_C( -52), INT8_C(  54),
                           INT8_C( -11), INT8_C( -39), INT8_C( -64), INT8_C( -79),
                           INT8_C(  88), INT8_C(  52), INT8_C( -95), INT8_C( -70),
                           INT8_C(  46), INT8_C( 121), INT8_C( -22), INT8_C( 113),
                           INT8_C(  26), INT8_C( 104), INT8_C( -52), INT8_C( 109),
                           INT8_C(   5), INT8_C(  87), INT8_C(  90), INT8_C(  -7),
                           INT8_C( -35), INT8_C(-112), INT8_C(  20), INT8_C(-128),
                           INT8_C( -21), INT8_C( -67), INT8_C(  37), INT8_C( -35)),
      simde_mm256_set_epi8(INT8_C(  66), INT8_C( -23), INT8_C(  45), INT8_C( -38),
                           INT8_C( -94), INT8_C( 119), INT8_C( -23), INT8_C(  70),
                           INT8_C( -75), INT8_C(  25), INT8_C(  79), INT8_C(-101),
                           INT8_C(-122), INT8_C(-107), INT8_C( -10), INT8_C(-109),
                           INT8_C( -96), INT8_C(  45), INT8_C(  31), INT8_C( -60),
                           INT8_C( 100), INT8_C( 115), INT8_C(  79), INT8_C(  38),
                           INT8_C( -35), INT8_C(  57), INT8_C(  13), INT8_C(  93),
                           INT8_C(  54), INT8_C( -61), INT8_C( 101), INT8_C(-113)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(-105), INT8_C(   9),
                           INT8_C( -52), INT8_C(  54), INT8_C( -11), INT8_C( -39),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  26), INT8_C( 104),
                           INT8_C( -52), INT8_C( 109), INT8_C(   5), INT8_C(  87)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_alignr_epi8(test_vec[i].a, test_vec[i].b, 26);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_and_si256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi64x(INT64_C( 8722470578646828517), INT64_C(  891261850847437783),
                             INT64_C( 8698554819020653857), INT64_C(-7282900013878242954)),
      simde_mm256_set_epi64x(INT64_C(-8128142018056442141), INT64_C( 5559182722028422309),
                             INT64_C( 2093267872519066825), INT64_C(-7117023562774970023)),
      simde_mm256_set_epi64x(INT64_C(  648519197013312737), INT64_C(  866420841735143557),
                             INT64_C( 1730587322060899329), INT64_C(-7482378910948097712)) },
    { simde_mm256_set_epi64x(INT64_C(-2297219683620407228), INT64_C(-2314825045857877411),
                             INT64_C(-2223407797787304327), INT64_C( 5408595704702705619)),
      simde_mm256_set_epi64x(INT64_C( 1902387556947256757), INT64_C(-4636290958455233996),
                             INT64_C( -193279292138890017), INT64_C( 2387678637527501964)),
      simde_mm256_set_epi64x(INT64_C(    1867272746704900), INT64_C(-6944527661819330028),
                             INT64_C(-2233693047608222631), INT64_C(   72674428659436672)) },
    { simde_mm256_set_epi64x(INT64_C(-8083909718117301567), INT64_C(   11995607010100125),
                             INT64_C(-6068617776224060223), INT64_C(-6387203967446836987)),
      simde_mm256_set_epi64x(INT64_C(-8320376883848651160), INT64_C(-4950145821323384534),
                             INT64_C(-7969688999974624617), INT64_C(  659904372446782737)),
      simde_mm256_set_epi64x(INT64_C(-8322647438183611840), INT64_C(    2406350531494152),
                             INT64_C(-9131628786599059327), INT64_C(   74330855942160641)) },
    { simde_mm256_set_epi64x(INT64_C(-7862557356832127783), INT64_C(-5197238245936512816),
                             INT64_C(-1440736387308233171), INT64_C( -422437923560182700)),
      simde_mm256_set_epi64x(INT64_C( 4501573497311276896), INT64_C( 1568099047173454230),
                             INT64_C( 6784671475384752865), INT64_C(-5901872067663085826)),
      simde_mm256_set_epi64x(INT64_C( 1324204786773460032), INT64_C( 1568098471546732688),
                             INT64_C( 5476791399028365857), INT64_C(-6196932668584612780)) },
    { simde_mm256_set_epi64x(INT64_C(  -83457062575009429), INT64_C(-7222721162513873213),
                             INT64_C( 8275972355230696496), INT64_C( 5685146925209815999)),
      simde_mm256_set_epi64x(INT64_C( 7621095561231011691), INT64_C(-1384347240916299959),
                             INT64_C( 8784701942784527649), INT64_C(-6329984144489188000)),
      simde_mm256_set_epi64x(INT64_C( 7549018173429252459), INT64_C(-8592431562369268159),
                             INT64_C( 8126746635764630560), INT64_C(  586910516468318496)) },
    { simde_mm256_set_epi64x(INT64_C( 5973184558080946927), INT64_C(-1786695518880322601),
                             INT64_C(  564422817571527071), INT64_C( 4038585732338755869)),
      simde_mm256_set_epi64x(INT64_C(-8901168232869945121), INT64_C( 8118630853720063073),
                             INT64_C( -228868271804772649), INT64_C(-6456700929251086932)),
      simde_mm256_set_epi64x(INT64_C(   27024505729917135), INT64_C( 6926573216261613633),
                             INT64_C(  346814025888696471), INT64_C( 2306177340255840524)) },
    { simde_mm256_set_epi64x(INT64_C( 4967668340414178010), INT64_C(-2410168209476403592),
                             INT64_C(-3019436090811439415), INT64_C(-6965119139859890192)),
      simde_mm256_set_epi64x(INT64_C(-5120337331222163918), INT64_C(-1589564432494918546),
                             INT64_C( 5292723257474752308), INT64_C( 2511807878775255697)),
      simde_mm256_set_epi64x(INT64_C(   67729921108361746), INT64_C(-3999169530918599576),
                             INT64_C( 4616337787987166720), INT64_C(  167381957966049936)) },
    { simde_mm256_set_epi64x(INT64_C(-6179811667909625694), INT64_C(-2471055444546593648),
                             INT64_C( 7540412455883833292), INT64_C( 6654843089135720963)),
      simde_mm256_set_epi64x(INT64_C( -939588147635733509), INT64_C(-1340596046637757449),
                             INT64_C(-1662948605324253370), INT64_C(  817158485966988858)),
      simde_mm256_set_epi64x(INT64_C(-6758530821969135454), INT64_C(-3664509346923870064),
                             INT64_C( 7540157231680104260), INT64_C(  599541701488411138)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_and_si256(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_andnot_si256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
   { simde_mm256_set_epi32(INT32_C( 1296069903), INT32_C( -401713319), INT32_C( -398512257), INT32_C( 1831595067),
                            INT32_C(  -79935516), INT32_C(-1180021826), INT32_C( 1322035843), INT32_C( 1532358492)),
      simde_mm256_set_epi32(INT32_C( 1994603249), INT32_C(-1155877896), INT32_C( 1480474617), INT32_C( 1055447888),
                            INT32_C( 1247599905), INT32_C( -290225910), INT32_C(  -75325456), INT32_C( 1443351725)),
      simde_mm256_set_epi32(INT32_C(  849545968), INT32_C(  319856800), INT32_C(  268439680), INT32_C(  314577728),
                            INT32_C(    4233217), INT32_C( 1175552000), INT32_C(-1325263504), INT32_C(   67240097)) },
    { simde_mm256_set_epi32(INT32_C(  751085853), INT32_C(  464664830), INT32_C( 1171943738), INT32_C(  238039385),
                            INT32_C(  447621112), INT32_C( -944668210), INT32_C(  -26835475), INT32_C( -497913063)),
      simde_mm256_set_epi32(INT32_C( -346681351), INT32_C( 1057375625), INT32_C(-1640028951), INT32_C( 1365052187),
                            INT32_C( 1146265967), INT32_C( -337636497), INT32_C(-2119055512), INT32_C(-1077772260)),
      simde_mm256_set_epi32(INT32_C(-1022228768), INT32_C(  604259585), INT32_C(-1708849983), INT32_C( 1364003330),
                            INT32_C( 1146126343), INT32_C(  675287585), INT32_C(   26296832), INT32_C(  494960644)) },
    { simde_mm256_set_epi32(INT32_C( 1652916374), INT32_C(  430611066), INT32_C(-1135996673), INT32_C( 1059116837),
                            INT32_C(  642663426), INT32_C(-2076745619), INT32_C(-1526373034), INT32_C(-1938949753)),
      simde_mm256_set_epi32(INT32_C(   21997480), INT32_C( 2006437191), INT32_C(-1106118986), INT32_C( -552923366),
                            INT32_C( 1259738142), INT32_C(-1272739174), INT32_C(  256185924), INT32_C(  828965431)),
      simde_mm256_set_epi32(INT32_C(   21636904), INT32_C( 1712669957), INT32_C(   34727424), INT32_C(-1073018854),
                            INT32_C( 1225789468), INT32_C(  805341842), INT32_C(  171971072), INT32_C(  822084144)) },
    { simde_mm256_set_epi32(INT32_C( 1850172884), INT32_C(  347371552), INT32_C(-2124471172), INT32_C(-1122146912),
                            INT32_C(-2084330368), INT32_C(-1474045155), INT32_C(-1097896595), INT32_C( -157279178)),
      simde_mm256_set_epi32(INT32_C( 1150160686), INT32_C( 1294066543), INT32_C(-1264650397), INT32_C( -958193094),
                            INT32_C( 1906124487), INT32_C(-1106662757), INT32_C(-1281709058), INT32_C(-1981986383)),
      simde_mm256_set_epi32(INT32_C(    8915498), INT32_C( 1224835919), INT32_C(  880857859), INT32_C( 1122107930),
                            INT32_C( 1880883783), INT32_C(  369631362), INT32_C(   17859218), INT32_C(  157106561)) },
    { simde_mm256_set_epi32(INT32_C( 1083224354), INT32_C(-2047292862), INT32_C(-1063111811), INT32_C(  679392193),
                            INT32_C(-1649939198), INT32_C(  165669772), INT32_C(-1686920475), INT32_C( -669365464)),
      simde_mm256_set_epi32(INT32_C(    1910614), INT32_C( 1742547828), INT32_C( 1824050734), INT32_C( 1245292821),
                            INT32_C( 1373149632), INT32_C(-1705573089), INT32_C(-2120011139), INT32_C( 1659478998)),
      simde_mm256_set_epi32(INT32_C(     853588), INT32_C( 1644503348), INT32_C(  739822594), INT32_C( 1107363860),
                            INT32_C( 1079513280), INT32_C(-1845492717), INT32_C(    8392728), INT32_C(  585212118)) },
    { simde_mm256_set_epi32(INT32_C( -215087200), INT32_C(  439485206), INT32_C(   52069439), INT32_C( 1985678052),
                            INT32_C(  236126003), INT32_C( 1543880509), INT32_C(-1076073317), INT32_C(  911942733)),
      simde_mm256_set_epi32(INT32_C(  289944943), INT32_C( 1791469709), INT32_C(-1356404241), INT32_C(-2084942503),
                            INT32_C(-1405988531), INT32_C(   52512230), INT32_C( -191001241), INT32_C(   11336758)),
      simde_mm256_set_epi32(INT32_C(    4206671), INT32_C( 1623566473), INT32_C(-1406899776), INT32_C(-2120202983),
                            INT32_C(-1608515508), INT32_C(   52445378), INT32_C( 1073840484), INT32_C(   10803250)) },
    { simde_mm256_set_epi32(INT32_C( -255265247), INT32_C(  884745637), INT32_C(-1256896434), INT32_C( 1712403469),
                            INT32_C(-1168205511), INT32_C( 1417172277), INT32_C(    6510726), INT32_C( 1896753771)),
      simde_mm256_set_epi32(INT32_C( 1154369104), INT32_C( 1355111439), INT32_C(  107073276), INT32_C( 1081379485),
                            INT32_C( -791672049), INT32_C(-1430371667), INT32_C( 1251404018), INT32_C(  686255615)),
      simde_mm256_set_epi32(INT32_C(   67502160), INT32_C( 1078024202), INT32_C(   39880368), INT32_C(    6589072),
                            INT32_C( 1082130950), INT32_C(-1434058104), INT32_C( 1251254384), INT32_C(  148981140)) },
    { simde_mm256_set_epi32(INT32_C( 1670586553), INT32_C( 1338043318), INT32_C(-2053008274), INT32_C( 1109326069),
                            INT32_C(  858036123), INT32_C(  201080868), INT32_C( -768148447), INT32_C( 1651890892)),
      simde_mm256_set_epi32(INT32_C( 1950737481), INT32_C(-1368157265), INT32_C(  504476696), INT32_C(-1751057511),
                            INT32_C(-1576762410), INT32_C( 1167387907), INT32_C(  398094763), INT32_C( 1151527447)),
      simde_mm256_set_epi32(INT32_C(  340051008), INT32_C(-1607266295), INT32_C(  437264400), INT32_C(-1784675576),
                            INT32_C(-2147483580), INT32_C( 1140894979), INT32_C(   92799370), INT32_C(   75636755)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_andnot_si256(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_adds_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C(-119), INT8_C( -56), INT8_C(  53), INT8_C(-117),
                           INT8_C(  25), INT8_C(  -8), INT8_C( -23), INT8_C( -22),
                           INT8_C( 105), INT8_C(  21), INT8_C( -22), INT8_C(-123),
                           INT8_C( -78), INT8_C(   1), INT8_C( 124), INT8_C(  72),
                           INT8_C( -56), INT8_C( 100), INT8_C( -58), INT8_C(  52),
                           INT8_C( 106), INT8_C(-112), INT8_C( -69), INT8_C(-107),
                           INT8_C(  71), INT8_C( -50), INT8_C(  48), INT8_C( -51),
                           INT8_C( -22), INT8_C( 119), INT8_C(  38), INT8_C(-116)),
      simde_mm256_set_epi8(INT8_C( -61), INT8_C(  39), INT8_C( 108), INT8_C( 116),
                           INT8_C(   8), INT8_C(  57), INT8_C( -68), INT8_C(-111),
                           INT8_C(  66), INT8_C(-114), INT8_C(  86), INT8_C(  70),
                           INT8_C(  95), INT8_C(  18), INT8_C(  95), INT8_C(  15),
                           INT8_C( -45), INT8_C(  91), INT8_C(  47), INT8_C( 120),
                           INT8_C(-127), INT8_C(  -1), INT8_C(  68), INT8_C(-116),
                           INT8_C(  47), INT8_C(  11), INT8_C( -19), INT8_C( -17),
                           INT8_C(-114), INT8_C(  40), INT8_C( -43), INT8_C( -57)),
      simde_mm256_set_epi8(INT8_C(-128), INT8_C( -17), INT8_C( 127), INT8_C(  -1),
                           INT8_C(  33), INT8_C(  49), INT8_C( -91), INT8_C(-128),
                           INT8_C( 127), INT8_C( -93), INT8_C(  64), INT8_C( -53),
                           INT8_C(  17), INT8_C(  19), INT8_C( 127), INT8_C(  87),
                           INT8_C(-101), INT8_C( 127), INT8_C( -11), INT8_C( 127),
                           INT8_C( -21), INT8_C(-113), INT8_C(  -1), INT8_C(-128),
                           INT8_C( 118), INT8_C( -39), INT8_C(  29), INT8_C( -68),
                           INT8_C(-128), INT8_C( 127), INT8_C(  -5), INT8_C(-128)) },
    { simde_mm256_set_epi8(INT8_C(  39), INT8_C(  31), INT8_C( -61), INT8_C( -35),
                           INT8_C(  33), INT8_C(  40), INT8_C( 126), INT8_C( 109),
                           INT8_C(   9), INT8_C( 100), INT8_C(-107), INT8_C(  27),
                           INT8_C( -15), INT8_C(  43), INT8_C(-116), INT8_C(-111),
                           INT8_C( -44), INT8_C(-120), INT8_C(  17), INT8_C( -18),
                           INT8_C( -10), INT8_C( -86), INT8_C(   7), INT8_C(  61),
                           INT8_C( -46), INT8_C( -53), INT8_C(  -3), INT8_C(  98),
                           INT8_C( -75), INT8_C( -98), INT8_C(  77), INT8_C( 102)),
      simde_mm256_set_epi8(INT8_C(-107), INT8_C(-113), INT8_C(   9), INT8_C(  35),
                           INT8_C(  62), INT8_C( -77), INT8_C(   0), INT8_C(   9),
                           INT8_C(  -7), INT8_C(  23), INT8_C( -88), INT8_C( -35),
                           INT8_C(  78), INT8_C(   7), INT8_C(  79), INT8_C( -98),
                           INT8_C( -54), INT8_C(  87), INT8_C( 123), INT8_C(  91),
                           INT8_C( 116), INT8_C(  67), INT8_C( 110), INT8_C(  43),
                           INT8_C( -68), INT8_C( -85), INT8_C( -23), INT8_C( -82),
                           INT8_C(  66), INT8_C(-119), INT8_C( -51), INT8_C(  79)),
      simde_mm256_set_epi8(INT8_C( -68), INT8_C( -82), INT8_C( -52), INT8_C(   0),
                           INT8_C(  95), INT8_C( -37), INT8_C( 126), INT8_C( 118),
                           INT8_C(   2), INT8_C( 123), INT8_C(-128), INT8_C(  -8),
                           INT8_C(  63), INT8_C(  50), INT8_C( -37), INT8_C(-128),
                           INT8_C( -98), INT8_C( -33), INT8_C( 127), INT8_C(  73),
                           INT8_C( 106), INT8_C( -19), INT8_C( 117), INT8_C( 104),
                           INT8_C(-114), INT8_C(-128), INT8_C( -26), INT8_C(  16),
                           INT8_C(  -9), INT8_C(-128), INT8_C(  26), INT8_C( 127)) },
    { simde_mm256_set_epi8(INT8_C( -96), INT8_C(  -2), INT8_C(-104), INT8_C(  30),
                           INT8_C( -98), INT8_C(-110), INT8_C( -27), INT8_C(   8),
                           INT8_C(  36), INT8_C( -65), INT8_C(  59), INT8_C(  66),
                           INT8_C( -66), INT8_C( -58), INT8_C( 111), INT8_C(  36),
                           INT8_C(  35), INT8_C(  -4), INT8_C( -85), INT8_C(  66),
                           INT8_C( 120), INT8_C(  62), INT8_C(  85), INT8_C(  -9),
                           INT8_C(  -9), INT8_C( -49), INT8_C(  90), INT8_C( -80),
                           INT8_C( -89), INT8_C( -62), INT8_C(-127), INT8_C(-100)),
      simde_mm256_set_epi8(INT8_C( -67), INT8_C(  11), INT8_C( -39), INT8_C(  56),
                           INT8_C( -35), INT8_C( 114), INT8_C(  34), INT8_C( -29),
                           INT8_C(  -3), INT8_C( 101), INT8_C( 115), INT8_C(  44),
                           INT8_C(  36), INT8_C( -77), INT8_C(  98), INT8_C( 105),
                           INT8_C(  91), INT8_C(  26), INT8_C(   9), INT8_C(   8),
                           INT8_C(  79), INT8_C(  -2), INT8_C( -60), INT8_C(  45),
                           INT8_C(-118), INT8_C( -81), INT8_C(  63), INT8_C( -54),
                           INT8_C( -51), INT8_C(  41), INT8_C(  33), INT8_C( -19)),
      simde_mm256_set_epi8(INT8_C(-128), INT8_C(   9), INT8_C(-128), INT8_C(  86),
                           INT8_C(-128), INT8_C(   4), INT8_C(   7), INT8_C( -21),
                           INT8_C(  33), INT8_C(  36), INT8_C( 127), INT8_C( 110),
                           INT8_C( -30), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                           INT8_C( 126), INT8_C(  22), INT8_C( -76), INT8_C(  74),
                           INT8_C( 127), INT8_C(  60), INT8_C(  25), INT8_C(  36),
                           INT8_C(-127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                           INT8_C(-128), INT8_C( -21), INT8_C( -94), INT8_C(-119)) },
    { simde_mm256_set_epi8(INT8_C( -17), INT8_C( -49), INT8_C(  22), INT8_C( -45),
                           INT8_C(  54), INT8_C(  22), INT8_C( -24), INT8_C( 117),
                           INT8_C( -78), INT8_C( 103), INT8_C(-105), INT8_C(  94),
                           INT8_C(-107), INT8_C(   6), INT8_C(-126), INT8_C(  89),
                           INT8_C(  17), INT8_C(   8), INT8_C(  15), INT8_C( -34),
                           INT8_C(  29), INT8_C(  90), INT8_C(  96), INT8_C( -78),
                           INT8_C( -87), INT8_C( -25), INT8_C(  62), INT8_C(  48),
                           INT8_C(-125), INT8_C( -71), INT8_C( -72), INT8_C(  85)),
      simde_mm256_set_epi8(INT8_C( -54), INT8_C(  53), INT8_C( -73), INT8_C( -38),
                           INT8_C(-124), INT8_C(  23), INT8_C(  96), INT8_C( -55),
                           INT8_C(  34), INT8_C( 108), INT8_C( -62), INT8_C(-104),
                           INT8_C(  81), INT8_C( -26), INT8_C(  -9), INT8_C( -47),
                           INT8_C( -73), INT8_C(  27), INT8_C( -45), INT8_C(-119),
                           INT8_C(-120), INT8_C(-121), INT8_C(  67), INT8_C( -46),
                           INT8_C(  17), INT8_C( 101), INT8_C(  16), INT8_C(-101),
                           INT8_C( -58), INT8_C(  77), INT8_C( -17), INT8_C( -35)),
      simde_mm256_set_epi8(INT8_C( -71), INT8_C(   4), INT8_C( -51), INT8_C( -83),
                           INT8_C( -70), INT8_C(  45), INT8_C(  72), INT8_C(  62),
                           INT8_C( -44), INT8_C( 127), INT8_C(-128), INT8_C( -10),
                           INT8_C( -26), INT8_C( -20), INT8_C(-128), INT8_C(  42),
                           INT8_C( -56), INT8_C(  35), INT8_C( -30), INT8_C(-128),
                           INT8_C( -91), INT8_C( -31), INT8_C( 127), INT8_C(-124),
                           INT8_C( -70), INT8_C(  76), INT8_C(  78), INT8_C( -53),
                           INT8_C(-128), INT8_C(   6), INT8_C( -89), INT8_C(  50)) },
    { simde_mm256_set_epi8(INT8_C(  99), INT8_C(  94), INT8_C( -71), INT8_C(  34),
                           INT8_C(-112), INT8_C(  96), INT8_C( -20), INT8_C( -14),
                           INT8_C(  85), INT8_C( 101), INT8_C(-125), INT8_C( -64),
                           INT8_C(  54), INT8_C( -63), INT8_C( -68), INT8_C( -91),
                           INT8_C(  62), INT8_C( -51), INT8_C(  27), INT8_C(  22),
                           INT8_C( -58), INT8_C(  12), INT8_C(   9), INT8_C( -36),
                           INT8_C(  40), INT8_C( -72), INT8_C( -90), INT8_C(  -3),
                           INT8_C( -94), INT8_C( -54), INT8_C(  87), INT8_C( -43)),
      simde_mm256_set_epi8(INT8_C(  99), INT8_C( -12), INT8_C(  40), INT8_C( -99),
                           INT8_C(  48), INT8_C(-109), INT8_C(  57), INT8_C( 112),
                           INT8_C( -67), INT8_C( -21), INT8_C(  14), INT8_C(  22),
                           INT8_C( -61), INT8_C( -64), INT8_C(  99), INT8_C( -14),
                           INT8_C( 121), INT8_C( 116), INT8_C( 125), INT8_C( -80),
                           INT8_C( -90), INT8_C( -41), INT8_C(  82), INT8_C( 106),
                           INT8_C( -62), INT8_C( -13), INT8_C( -52), INT8_C( -14),
                           INT8_C( -76), INT8_C(-109), INT8_C( 112), INT8_C( -92)),
      simde_mm256_set_epi8(INT8_C( 127), INT8_C(  82), INT8_C( -31), INT8_C( -65),
                           INT8_C( -64), INT8_C( -13), INT8_C(  37), INT8_C(  98),
                           INT8_C(  18), INT8_C(  80), INT8_C(-111), INT8_C( -42),
                           INT8_C(  -7), INT8_C(-127), INT8_C(  31), INT8_C(-105),
                           INT8_C( 127), INT8_C(  65), INT8_C( 127), INT8_C( -58),
                           INT8_C(-128), INT8_C( -29), INT8_C(  91), INT8_C(  70),
                           INT8_C( -22), INT8_C( -85), INT8_C(-128), INT8_C( -17),
                           INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128)) },
    { simde_mm256_set_epi8(INT8_C( -49), INT8_C( 105), INT8_C(-105), INT8_C(  27),
                           INT8_C( -85), INT8_C(  38), INT8_C(-122), INT8_C(  31),
                           INT8_C( -23), INT8_C(  94), INT8_C(  33), INT8_C( 112),
                           INT8_C( 117), INT8_C(  95), INT8_C( 107), INT8_C( -60),
                           INT8_C(  41), INT8_C( -75), INT8_C( 112), INT8_C(-110),
                           INT8_C( 101), INT8_C( -13), INT8_C( 127), INT8_C( -84),
                           INT8_C( -84), INT8_C( -70), INT8_C( -49), INT8_C(  84),
                           INT8_C(  -1), INT8_C( -17), INT8_C(  -2), INT8_C(  84)),
      simde_mm256_set_epi8(INT8_C(-108), INT8_C(  63), INT8_C( 104), INT8_C(  -9),
                           INT8_C(  74), INT8_C( 114), INT8_C( -61), INT8_C(  39),
                           INT8_C(  15), INT8_C(   6), INT8_C(  41), INT8_C( 125),
                           INT8_C(  96), INT8_C( -39), INT8_C( -42), INT8_C(  75),
                           INT8_C( -60), INT8_C(  18), INT8_C(-100), INT8_C( -10),
                           INT8_C( -33), INT8_C( -88), INT8_C( -36), INT8_C(  79),
                           INT8_C(  49), INT8_C(  63), INT8_C( -18), INT8_C( -83),
                           INT8_C(  44), INT8_C(  75), INT8_C( -17), INT8_C(  -9)),
      simde_mm256_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C(  -1), INT8_C(  18),
                           INT8_C( -11), INT8_C( 127), INT8_C(-128), INT8_C(  70),
                           INT8_C(  -8), INT8_C( 100), INT8_C(  74), INT8_C( 127),
                           INT8_C( 127), INT8_C(  56), INT8_C(  65), INT8_C(  15),
                           INT8_C( -19), INT8_C( -57), INT8_C(  12), INT8_C(-120),
                           INT8_C(  68), INT8_C(-101), INT8_C(  91), INT8_C(  -5),
                           INT8_C( -35), INT8_C(  -7), INT8_C( -67), INT8_C(   1),
                           INT8_C(  43), INT8_C(  58), INT8_C( -19), INT8_C(  75)) },
    { simde_mm256_set_epi8(INT8_C( -41), INT8_C( -94), INT8_C(  69), INT8_C( 109),
                           INT8_C(-102), INT8_C(  64), INT8_C( 116), INT8_C(  22),
                           INT8_C( -63), INT8_C(  76), INT8_C( -59), INT8_C( -94),
                           INT8_C(  69), INT8_C(   1), INT8_C( -30), INT8_C( 101),
                           INT8_C( 121), INT8_C(  10), INT8_C( -82), INT8_C( -33),
                           INT8_C( -83), INT8_C( -50), INT8_C(-111), INT8_C( -72),
                           INT8_C(  42), INT8_C(-125), INT8_C(-128), INT8_C(  -8),
                           INT8_C(  27), INT8_C( -93), INT8_C(-126), INT8_C( -77)),
      simde_mm256_set_epi8(INT8_C(  -9), INT8_C( -48), INT8_C(  73), INT8_C(  31),
                           INT8_C( 127), INT8_C(  88), INT8_C(  20), INT8_C( -82),
                           INT8_C( -19), INT8_C(   3), INT8_C(  83), INT8_C( 114),
                           INT8_C(   1), INT8_C(  31), INT8_C(  44), INT8_C(   5),
                           INT8_C( -63), INT8_C( -96), INT8_C(   2), INT8_C(-126),
                           INT8_C(  96), INT8_C( -97), INT8_C( -87), INT8_C( -40),
                           INT8_C(-112), INT8_C(  92), INT8_C( -98), INT8_C( -50),
                           INT8_C(  63), INT8_C( -57), INT8_C(  24), INT8_C( -21)),
      simde_mm256_set_epi8(INT8_C( -50), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                           INT8_C(  25), INT8_C( 127), INT8_C( 127), INT8_C( -60),
                           INT8_C( -82), INT8_C(  79), INT8_C(  24), INT8_C(  20),
                           INT8_C(  70), INT8_C(  32), INT8_C(  14), INT8_C( 106),
                           INT8_C(  58), INT8_C( -86), INT8_C( -80), INT8_C(-128),
                           INT8_C(  13), INT8_C(-128), INT8_C(-128), INT8_C(-112),
                           INT8_C( -70), INT8_C( -33), INT8_C(-128), INT8_C( -58),
                           INT8_C(  90), INT8_C(-128), INT8_C(-102), INT8_C( -98)) },
    { simde_mm256_set_epi8(INT8_C(  59), INT8_C( -47), INT8_C(  -6), INT8_C( 114),
                           INT8_C( 104), INT8_C(  53), INT8_C(-112), INT8_C(  19),
                           INT8_C( 115), INT8_C(  22), INT8_C(  66), INT8_C(  27),
                           INT8_C( -25), INT8_C( -41), INT8_C(-111), INT8_C( 115),
                           INT8_C( -21), INT8_C( -32), INT8_C(  -5), INT8_C(  11),
                           INT8_C(  -6), INT8_C( 110), INT8_C( -89), INT8_C( -64),
                           INT8_C(-104), INT8_C(  74), INT8_C( -29), INT8_C(  87),
                           INT8_C(  -8), INT8_C(  96), INT8_C(   5), INT8_C( 122)),
      simde_mm256_set_epi8(INT8_C(  43), INT8_C(  18), INT8_C(  50), INT8_C(-115),
                           INT8_C(  38), INT8_C( -78), INT8_C( -51), INT8_C(  97),
                           INT8_C(  30), INT8_C(   7), INT8_C( -46), INT8_C( -16),
                           INT8_C( 109), INT8_C(-103), INT8_C( -61), INT8_C(  64),
                           INT8_C(  60), INT8_C( -63), INT8_C( -52), INT8_C( -15),
                           INT8_C( 104), INT8_C( 105), INT8_C(  61), INT8_C( 106),
                           INT8_C( -66), INT8_C( 111), INT8_C( -25), INT8_C(  24),
                           INT8_C(  28), INT8_C( 102), INT8_C(   8), INT8_C(  93)),
      simde_mm256_set_epi8(INT8_C( 102), INT8_C( -29), INT8_C(  44), INT8_C(  -1),
                           INT8_C( 127), INT8_C( -25), INT8_C(-128), INT8_C( 116),
                           INT8_C( 127), INT8_C(  29), INT8_C(  20), INT8_C(  11),
                           INT8_C(  84), INT8_C(-128), INT8_C(-128), INT8_C( 127),
                           INT8_C(  39), INT8_C( -95), INT8_C( -57), INT8_C(  -4),
                           INT8_C(  98), INT8_C( 127), INT8_C( -28), INT8_C(  42),
                           INT8_C(-128), INT8_C( 127), INT8_C( -54), INT8_C( 111),
                           INT8_C(  20), INT8_C( 127), INT8_C(  13), INT8_C( 127)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_adds_epi8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_adds_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(  -962), INT16_C( 12004), INT16_C( 15276), INT16_C( 19344),
                            INT16_C(  1682), INT16_C( 24393), INT16_C(-26791), INT16_C( 15115),
                            INT16_C( 26019), INT16_C(-25175), INT16_C(-17857), INT16_C( 29245),
                            INT16_C( -3095), INT16_C( -1043), INT16_C(  8684), INT16_C( -4510)),
      simde_mm256_set_epi16(INT16_C(  1387), INT16_C(  5995), INT16_C( 24092), INT16_C(-22020),
                            INT16_C( 17334), INT16_C(-21406), INT16_C(  5443), INT16_C( -4682),
                            INT16_C( 19873), INT16_C(-31638), INT16_C( 16556), INT16_C(-22745),
                            INT16_C(  6086), INT16_C(-26599), INT16_C(-32635), INT16_C( 18742)),
      simde_mm256_set_epi16(INT16_C(   425), INT16_C( 17999), INT16_C( 32767), INT16_C( -2676),
                            INT16_C( 19016), INT16_C(  2987), INT16_C(-21348), INT16_C( 10433),
                            INT16_C( 32767), INT16_C(-32768), INT16_C( -1301), INT16_C(  6500),
                            INT16_C(  2991), INT16_C(-27642), INT16_C(-23951), INT16_C( 14232)) },
    { simde_mm256_set_epi16(INT16_C(-15667), INT16_C( -4604), INT16_C( -1424), INT16_C( 14196),
                            INT16_C( -3271), INT16_C(-28350), INT16_C( 32228), INT16_C(-15812),
                            INT16_C( -3284), INT16_C(  -332), INT16_C(-21864), INT16_C(-23002),
                            INT16_C( 15429), INT16_C( 22829), INT16_C( -2222), INT16_C( 22367)),
      simde_mm256_set_epi16(INT16_C(  2292), INT16_C( 20266), INT16_C( 19204), INT16_C(-18548),
                            INT16_C(-10545), INT16_C( 20262), INT16_C( 18576), INT16_C( 18942),
                            INT16_C( 30300), INT16_C( -3505), INT16_C( 25496), INT16_C(-21517),
                            INT16_C( 23044), INT16_C( 26958), INT16_C( 16161), INT16_C( -8396)),
      simde_mm256_set_epi16(INT16_C(-13375), INT16_C( 15662), INT16_C( 17780), INT16_C( -4352),
                            INT16_C(-13816), INT16_C( -8088), INT16_C( 32767), INT16_C(  3130),
                            INT16_C( 27016), INT16_C( -3837), INT16_C(  3632), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 13939), INT16_C( 13971)) },
    { simde_mm256_set_epi16(INT16_C(-13807), INT16_C(-17934), INT16_C( 30355), INT16_C(  -764),
                            INT16_C( -4690), INT16_C( 11643), INT16_C( 12025), INT16_C( 13748),
                            INT16_C( 20416), INT16_C(-20138), INT16_C( -7279), INT16_C(-10075),
                            INT16_C( 30794), INT16_C( -8219), INT16_C(  7665), INT16_C( 31634)),
      simde_mm256_set_epi16(INT16_C( 19174), INT16_C( 30134), INT16_C(  -229), INT16_C( 19963),
                            INT16_C( -2304), INT16_C( 30679), INT16_C( 27179), INT16_C(-10745),
                            INT16_C(-27839), INT16_C( -4281), INT16_C( 23775), INT16_C( -9849),
                            INT16_C(-26461), INT16_C(-25537), INT16_C( -6859), INT16_C(-14106)),
      simde_mm256_set_epi16(INT16_C(  5367), INT16_C( 12200), INT16_C( 30126), INT16_C( 19199),
                            INT16_C( -6994), INT16_C( 32767), INT16_C( 32767), INT16_C(  3003),
                            INT16_C( -7423), INT16_C(-24419), INT16_C( 16496), INT16_C(-19924),
                            INT16_C(  4333), INT16_C(-32768), INT16_C(   806), INT16_C( 17528)) },
    { simde_mm256_set_epi16(INT16_C( -5643), INT16_C( 16549), INT16_C( 27397), INT16_C(-21486),
                            INT16_C( -4783), INT16_C( -4255), INT16_C( -9777), INT16_C( -2005),
                            INT16_C( -5487), INT16_C(  4410), INT16_C(  9721), INT16_C(-18951),
                            INT16_C( 27380), INT16_C( 27675), INT16_C(-18193), INT16_C( -8216)),
      simde_mm256_set_epi16(INT16_C(  3093), INT16_C(-27399), INT16_C( 25399), INT16_C( 21162),
                            INT16_C(-20342), INT16_C(-26357), INT16_C( 20961), INT16_C(-29046),
                            INT16_C(-12304), INT16_C(-23482), INT16_C( 31742), INT16_C( 26191),
                            INT16_C( 32539), INT16_C( 28035), INT16_C( -8379), INT16_C( 32320)),
      simde_mm256_set_epi16(INT16_C( -2550), INT16_C(-10850), INT16_C( 32767), INT16_C(  -324),
                            INT16_C(-25125), INT16_C(-30612), INT16_C( 11184), INT16_C(-31051),
                            INT16_C(-17791), INT16_C(-19072), INT16_C( 32767), INT16_C(  7240),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-26572), INT16_C( 24104)) },
    { simde_mm256_set_epi16(INT16_C(  8117), INT16_C(   765), INT16_C( -4891), INT16_C( -1773),
                            INT16_C( 21984), INT16_C( 23512), INT16_C(  8564), INT16_C(   676),
                            INT16_C(-22280), INT16_C(  5831), INT16_C(-15902), INT16_C( 13241),
                            INT16_C(  6903), INT16_C( -2164), INT16_C(-27428), INT16_C(-27012)),
      simde_mm256_set_epi16(INT16_C( 12023), INT16_C(-19040), INT16_C( 31146), INT16_C(-18380),
                            INT16_C(-24072), INT16_C( 14767), INT16_C( 22843), INT16_C( -2924),
                            INT16_C( -5044), INT16_C(-32368), INT16_C( 21585), INT16_C( -7796),
                            INT16_C( 32151), INT16_C(  8315), INT16_C( 19587), INT16_C(-17957)),
      simde_mm256_set_epi16(INT16_C( 20140), INT16_C(-18275), INT16_C( 26255), INT16_C(-20153),
                            INT16_C( -2088), INT16_C( 32767), INT16_C( 31407), INT16_C( -2248),
                            INT16_C(-27324), INT16_C(-26537), INT16_C(  5683), INT16_C(  5445),
                            INT16_C( 32767), INT16_C(  6151), INT16_C( -7841), INT16_C(-32768)) },
    { simde_mm256_set_epi16(INT16_C(-15432), INT16_C(-29555), INT16_C(-22086), INT16_C(-23352),
                            INT16_C(-24272), INT16_C( 28442), INT16_C( -6183), INT16_C( 20311),
                            INT16_C(-15448), INT16_C(-31565), INT16_C(-17613), INT16_C( -1655),
                            INT16_C( -3795), INT16_C( 27576), INT16_C(-23497), INT16_C( -9670)),
      simde_mm256_set_epi16(INT16_C(-16830), INT16_C(  -159), INT16_C(-22301), INT16_C(-32198),
                            INT16_C(  2834), INT16_C(   201), INT16_C(-29264), INT16_C( -1240),
                            INT16_C(  4796), INT16_C(-27180), INT16_C( 31617), INT16_C( -2210),
                            INT16_C( -9712), INT16_C( 10057), INT16_C( -1995), INT16_C( 27783)),
      simde_mm256_set_epi16(INT16_C(-32262), INT16_C(-29714), INT16_C(-32768), INT16_C(-32768),
                            INT16_C(-21438), INT16_C( 28643), INT16_C(-32768), INT16_C( 19071),
                            INT16_C(-10652), INT16_C(-32768), INT16_C( 14004), INT16_C( -3865),
                            INT16_C(-13507), INT16_C( 32767), INT16_C(-25492), INT16_C( 18113)) },
    { simde_mm256_set_epi16(INT16_C(-15154), INT16_C(  9051), INT16_C(  6258), INT16_C( -1499),
                            INT16_C(-27693), INT16_C( -8500), INT16_C(  -882), INT16_C(  9147),
                            INT16_C( 22175), INT16_C( 10790), INT16_C(-10382), INT16_C( 21347),
                            INT16_C( -8836), INT16_C(-17904), INT16_C(-31823), INT16_C(-28959)),
      simde_mm256_set_epi16(INT16_C( 31738), INT16_C( 18200), INT16_C(-12479), INT16_C(   972),
                            INT16_C(-30550), INT16_C(-19770), INT16_C( 32501), INT16_C( -4294),
                            INT16_C(-22747), INT16_C(-27490), INT16_C(-20241), INT16_C(-31535),
                            INT16_C(-31518), INT16_C(-18415), INT16_C( 12039), INT16_C(-31202)),
      simde_mm256_set_epi16(INT16_C( 16584), INT16_C( 27251), INT16_C( -6221), INT16_C(  -527),
                            INT16_C(-32768), INT16_C(-28270), INT16_C( 31619), INT16_C(  4853),
                            INT16_C(  -572), INT16_C(-16700), INT16_C(-30623), INT16_C(-10188),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-19784), INT16_C(-32768)) },
    { simde_mm256_set_epi16(INT16_C(-23218), INT16_C(-16504), INT16_C(-28974), INT16_C(-20854),
                            INT16_C( 27922), INT16_C(   717), INT16_C(-10156), INT16_C( 13317),
                            INT16_C( 12071), INT16_C(-29036), INT16_C( 16491), INT16_C(  6863),
                            INT16_C(  3054), INT16_C( -8502), INT16_C(-19858), INT16_C(-20664)),
      simde_mm256_set_epi16(INT16_C(-22864), INT16_C(-13701), INT16_C( -9243), INT16_C( -4457),
                            INT16_C( 16105), INT16_C(  3518), INT16_C(-11607), INT16_C(-19946),
                            INT16_C(  5641), INT16_C(  8617), INT16_C(-22446), INT16_C(  2151),
                            INT16_C( 16533), INT16_C(-18326), INT16_C( 16839), INT16_C( 26518)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-30205), INT16_C(-32768), INT16_C(-25311),
                            INT16_C( 32767), INT16_C(  4235), INT16_C(-21763), INT16_C( -6629),
                            INT16_C( 17712), INT16_C(-20419), INT16_C( -5955), INT16_C(  9014),
                            INT16_C( 19587), INT16_C(-26828), INT16_C( -3019), INT16_C(  5854)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_adds_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_adds_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu8(UINT8_C(253), UINT8_C(134), UINT8_C(240), UINT8_C(121),
                             UINT8_C(194), UINT8_C(  6), UINT8_C( 90), UINT8_C(185),
                             UINT8_C( 52), UINT8_C(188), UINT8_C(255), UINT8_C(213),
                             UINT8_C( 70), UINT8_C(140), UINT8_C( 59), UINT8_C(206),
                             UINT8_C( 91), UINT8_C( 56), UINT8_C(139), UINT8_C( 19),
                             UINT8_C( 62), UINT8_C( 91), UINT8_C( 24), UINT8_C( 86),
                             UINT8_C(156), UINT8_C( 89), UINT8_C( 98), UINT8_C(113),
                             UINT8_C(237), UINT8_C(  2), UINT8_C(237), UINT8_C(177)),
      simde_x_mm256_set_epu8(UINT8_C(213), UINT8_C( 63), UINT8_C( 15), UINT8_C(166),
                             UINT8_C( 63), UINT8_C(196), UINT8_C(141), UINT8_C(108),
                             UINT8_C( 47), UINT8_C(216), UINT8_C( 17), UINT8_C(218),
                             UINT8_C(111), UINT8_C(130), UINT8_C(  1), UINT8_C(159),
                             UINT8_C( 50), UINT8_C(145), UINT8_C(171), UINT8_C( 70),
                             UINT8_C( 84), UINT8_C(160), UINT8_C(222), UINT8_C(215),
                             UINT8_C( 44), UINT8_C(139), UINT8_C( 68), UINT8_C( 49),
                             UINT8_C(218), UINT8_C( 62), UINT8_C(151), UINT8_C(225)),
      simde_x_mm256_set_epu8(UINT8_C(255), UINT8_C(197), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(202), UINT8_C(231), UINT8_C(255),
                             UINT8_C( 99), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(181), UINT8_C(255), UINT8_C( 60), UINT8_C(255),
                             UINT8_C(141), UINT8_C(201), UINT8_C(255), UINT8_C( 89),
                             UINT8_C(146), UINT8_C(251), UINT8_C(246), UINT8_C(255),
                             UINT8_C(200), UINT8_C(228), UINT8_C(166), UINT8_C(162),
                             UINT8_C(255), UINT8_C( 64), UINT8_C(255), UINT8_C(255)) },
    { simde_x_mm256_set_epu8(UINT8_C( 46), UINT8_C( 60), UINT8_C( 83), UINT8_C( 69),
                             UINT8_C( 75), UINT8_C( 52), UINT8_C(182), UINT8_C( 56),
                             UINT8_C( 10), UINT8_C(180), UINT8_C(245), UINT8_C(208),
                             UINT8_C(165), UINT8_C( 77), UINT8_C(214), UINT8_C(172),
                             UINT8_C( 90), UINT8_C( 10), UINT8_C(190), UINT8_C(204),
                             UINT8_C(174), UINT8_C(200), UINT8_C( 75), UINT8_C(188),
                             UINT8_C(215), UINT8_C( 51), UINT8_C(188), UINT8_C( 47),
                             UINT8_C( 17), UINT8_C(116), UINT8_C(116), UINT8_C(103)),
      simde_x_mm256_set_epu8(UINT8_C( 76), UINT8_C( 23), UINT8_C( 40), UINT8_C(  7),
                             UINT8_C( 32), UINT8_C(238), UINT8_C(187), UINT8_C( 34),
                             UINT8_C(130), UINT8_C(185), UINT8_C(135), UINT8_C( 64),
                             UINT8_C(167), UINT8_C(215), UINT8_C(226), UINT8_C(221),
                             UINT8_C( 22), UINT8_C( 21), UINT8_C( 86), UINT8_C(166),
                             UINT8_C( 38), UINT8_C( 88), UINT8_C(194), UINT8_C( 97),
                             UINT8_C( 79), UINT8_C(177), UINT8_C(251), UINT8_C(218),
                             UINT8_C( 76), UINT8_C(121), UINT8_C(164), UINT8_C( 80)),
      simde_x_mm256_set_epu8(UINT8_C(122), UINT8_C( 83), UINT8_C(123), UINT8_C( 76),
                             UINT8_C(107), UINT8_C(255), UINT8_C(255), UINT8_C( 90),
                             UINT8_C(140), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(112), UINT8_C( 31), UINT8_C(255), UINT8_C(255),
                             UINT8_C(212), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(228), UINT8_C(255), UINT8_C(255),
                             UINT8_C( 93), UINT8_C(237), UINT8_C(255), UINT8_C(183)) },
    { simde_x_mm256_set_epu8(UINT8_C( 62), UINT8_C(125), UINT8_C(167), UINT8_C( 69),
                             UINT8_C(185), UINT8_C(211), UINT8_C(194), UINT8_C( 87),
                             UINT8_C( 42), UINT8_C( 67), UINT8_C(196), UINT8_C(131),
                             UINT8_C( 56), UINT8_C(103), UINT8_C( 93), UINT8_C(201),
                             UINT8_C(200), UINT8_C( 67), UINT8_C(  5), UINT8_C(251),
                             UINT8_C(171), UINT8_C(  0), UINT8_C(242), UINT8_C(219),
                             UINT8_C( 52), UINT8_C( 56), UINT8_C(153), UINT8_C(193),
                             UINT8_C(206), UINT8_C(184), UINT8_C(179), UINT8_C(112)),
      simde_x_mm256_set_epu8(UINT8_C(  3), UINT8_C( 94), UINT8_C(  2), UINT8_C( 80),
                             UINT8_C( 75), UINT8_C( 54), UINT8_C(116), UINT8_C(175),
                             UINT8_C(133), UINT8_C(183), UINT8_C(131), UINT8_C(  7),
                             UINT8_C( 65), UINT8_C( 80), UINT8_C(192), UINT8_C( 97),
                             UINT8_C(148), UINT8_C(232), UINT8_C(135), UINT8_C(251),
                             UINT8_C(194), UINT8_C( 84), UINT8_C(121), UINT8_C( 35),
                             UINT8_C( 55), UINT8_C(172), UINT8_C( 74), UINT8_C( 18),
                             UINT8_C( 14), UINT8_C( 11), UINT8_C( 86), UINT8_C(161)),
      simde_x_mm256_set_epu8(UINT8_C( 65), UINT8_C(219), UINT8_C(169), UINT8_C(149),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(175), UINT8_C(250), UINT8_C(255), UINT8_C(138),
                             UINT8_C(121), UINT8_C(183), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(140), UINT8_C(255),
                             UINT8_C(255), UINT8_C( 84), UINT8_C(255), UINT8_C(254),
                             UINT8_C(107), UINT8_C(228), UINT8_C(227), UINT8_C(211),
                             UINT8_C(220), UINT8_C(195), UINT8_C(255), UINT8_C(255)) },
    { simde_x_mm256_set_epu8(UINT8_C(180), UINT8_C(149), UINT8_C(241), UINT8_C( 31),
                             UINT8_C( 45), UINT8_C(238), UINT8_C(127), UINT8_C( 41),
                             UINT8_C( 61), UINT8_C( 40), UINT8_C(253), UINT8_C(133),
                             UINT8_C(247), UINT8_C(164), UINT8_C(139), UINT8_C(228),
                             UINT8_C( 62), UINT8_C(209), UINT8_C(132), UINT8_C( 80),
                             UINT8_C(102), UINT8_C(192), UINT8_C(185), UINT8_C(191),
                             UINT8_C(100), UINT8_C(154), UINT8_C( 33), UINT8_C( 61),
                             UINT8_C( 93), UINT8_C(153), UINT8_C(220), UINT8_C(160)),
      simde_x_mm256_set_epu8(UINT8_C(230), UINT8_C( 84), UINT8_C( 78), UINT8_C(157),
                             UINT8_C(192), UINT8_C( 80), UINT8_C(173), UINT8_C(216),
                             UINT8_C( 47), UINT8_C(218), UINT8_C( 93), UINT8_C(143),
                             UINT8_C( 45), UINT8_C( 59), UINT8_C(151), UINT8_C(134),
                             UINT8_C( 70), UINT8_C( 90), UINT8_C( 58), UINT8_C(114),
                             UINT8_C(223), UINT8_C(242), UINT8_C( 15), UINT8_C(131),
                             UINT8_C( 82), UINT8_C(204), UINT8_C(173), UINT8_C(  5),
                             UINT8_C( 58), UINT8_C(182), UINT8_C(252), UINT8_C(237)),
      simde_x_mm256_set_epu8(UINT8_C(255), UINT8_C(233), UINT8_C(255), UINT8_C(188),
                             UINT8_C(237), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(108), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(223), UINT8_C(255), UINT8_C(255),
                             UINT8_C(132), UINT8_C(255), UINT8_C(190), UINT8_C(194),
                             UINT8_C(255), UINT8_C(255), UINT8_C(200), UINT8_C(255),
                             UINT8_C(182), UINT8_C(255), UINT8_C(206), UINT8_C( 66),
                             UINT8_C(151), UINT8_C(255), UINT8_C(255), UINT8_C(255)) },
    { simde_x_mm256_set_epu8(UINT8_C(201), UINT8_C(214), UINT8_C(179), UINT8_C(233),
                             UINT8_C(198), UINT8_C( 81), UINT8_C(243), UINT8_C(119),
                             UINT8_C(160), UINT8_C(147), UINT8_C( 34), UINT8_C(138),
                             UINT8_C(127), UINT8_C( 31), UINT8_C(145), UINT8_C( 40),
                             UINT8_C(228), UINT8_C( 45), UINT8_C(  3), UINT8_C(  4),
                             UINT8_C(249), UINT8_C(180), UINT8_C(210), UINT8_C(  7),
                             UINT8_C(175), UINT8_C( 81), UINT8_C( 15), UINT8_C(137),
                             UINT8_C( 20), UINT8_C( 57), UINT8_C(  3), UINT8_C(157)),
      simde_x_mm256_set_epu8(UINT8_C(209), UINT8_C( 16), UINT8_C(253), UINT8_C(199),
                             UINT8_C(185), UINT8_C(238), UINT8_C( 28), UINT8_C( 87),
                             UINT8_C( 41), UINT8_C(232), UINT8_C( 54), UINT8_C(100),
                             UINT8_C(160), UINT8_C( 87), UINT8_C(101), UINT8_C(193),
                             UINT8_C(173), UINT8_C(242), UINT8_C(182), UINT8_C(  0),
                             UINT8_C(222), UINT8_C(142), UINT8_C(217), UINT8_C(177),
                             UINT8_C(237), UINT8_C(196), UINT8_C(145), UINT8_C(208),
                             UINT8_C( 95), UINT8_C(248), UINT8_C( 86), UINT8_C( 20)),
      simde_x_mm256_set_epu8(UINT8_C(255), UINT8_C(230), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(206),
                             UINT8_C(201), UINT8_C(255), UINT8_C( 88), UINT8_C(238),
                             UINT8_C(255), UINT8_C(118), UINT8_C(246), UINT8_C(233),
                             UINT8_C(255), UINT8_C(255), UINT8_C(185), UINT8_C(  4),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(184),
                             UINT8_C(255), UINT8_C(255), UINT8_C(160), UINT8_C(255),
                             UINT8_C(115), UINT8_C(255), UINT8_C( 89), UINT8_C(177)) },
    { simde_x_mm256_set_epu8(UINT8_C(117), UINT8_C(251), UINT8_C(156), UINT8_C( 72),
                             UINT8_C(241), UINT8_C( 25), UINT8_C( 80), UINT8_C(195),
                             UINT8_C(213), UINT8_C( 26), UINT8_C( 44), UINT8_C(154),
                             UINT8_C( 40), UINT8_C(201), UINT8_C(142), UINT8_C(110),
                             UINT8_C( 17), UINT8_C(100), UINT8_C( 41), UINT8_C(223),
                             UINT8_C(255), UINT8_C(232), UINT8_C(253), UINT8_C(190),
                             UINT8_C(155), UINT8_C(178), UINT8_C(150), UINT8_C(248),
                             UINT8_C(166), UINT8_C(223), UINT8_C( 92), UINT8_C(146)),
      simde_x_mm256_set_epu8(UINT8_C(164), UINT8_C(171), UINT8_C(230), UINT8_C(232),
                             UINT8_C( 86), UINT8_C(220), UINT8_C(153), UINT8_C(194),
                             UINT8_C( 83), UINT8_C( 81), UINT8_C( 80), UINT8_C( 93),
                             UINT8_C( 96), UINT8_C( 68), UINT8_C( 55), UINT8_C(183),
                             UINT8_C(110), UINT8_C(151), UINT8_C( 66), UINT8_C(168),
                             UINT8_C(210), UINT8_C( 58), UINT8_C(169), UINT8_C(189),
                             UINT8_C( 61), UINT8_C( 15), UINT8_C(124), UINT8_C(190),
                             UINT8_C( 93), UINT8_C(242), UINT8_C( 80), UINT8_C(225)),
      simde_x_mm256_set_epu8(UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(245), UINT8_C(233), UINT8_C(255),
                             UINT8_C(255), UINT8_C(107), UINT8_C(124), UINT8_C(247),
                             UINT8_C(136), UINT8_C(255), UINT8_C(197), UINT8_C(255),
                             UINT8_C(127), UINT8_C(251), UINT8_C(107), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(216), UINT8_C(193), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(172), UINT8_C(255)) },
    { simde_x_mm256_set_epu8(UINT8_C(130), UINT8_C( 76), UINT8_C( 98), UINT8_C(  8),
                             UINT8_C(243), UINT8_C(125), UINT8_C(189), UINT8_C(162),
                             UINT8_C(107), UINT8_C( 98), UINT8_C(171), UINT8_C( 80),
                             UINT8_C(243), UINT8_C(225), UINT8_C(  6), UINT8_C( 11),
                             UINT8_C(250), UINT8_C(210), UINT8_C( 60), UINT8_C(230),
                             UINT8_C( 17), UINT8_C(222), UINT8_C( 70), UINT8_C(180),
                             UINT8_C( 28), UINT8_C( 96), UINT8_C(128), UINT8_C(195),
                             UINT8_C(240), UINT8_C(119), UINT8_C(199), UINT8_C( 65)),
      simde_x_mm256_set_epu8(UINT8_C( 47), UINT8_C(225), UINT8_C( 74), UINT8_C(106),
                             UINT8_C(149), UINT8_C(250), UINT8_C(221), UINT8_C( 24),
                             UINT8_C(161), UINT8_C( 69), UINT8_C(209), UINT8_C(120),
                             UINT8_C(116), UINT8_C(235), UINT8_C(  4), UINT8_C(161),
                             UINT8_C(156), UINT8_C(193), UINT8_C(253), UINT8_C(  2),
                             UINT8_C(168), UINT8_C( 52), UINT8_C(231), UINT8_C(201),
                             UINT8_C(115), UINT8_C(  0), UINT8_C(176), UINT8_C(224),
                             UINT8_C(  4), UINT8_C(219), UINT8_C(202), UINT8_C(177)),
      simde_x_mm256_set_epu8(UINT8_C(177), UINT8_C(255), UINT8_C(172), UINT8_C(114),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(186),
                             UINT8_C(255), UINT8_C(167), UINT8_C(255), UINT8_C(200),
                             UINT8_C(255), UINT8_C(255), UINT8_C( 10), UINT8_C(172),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(232),
                             UINT8_C(185), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(143), UINT8_C( 96), UINT8_C(255), UINT8_C(255),
                             UINT8_C(244), UINT8_C(255), UINT8_C(255), UINT8_C(242)) },
    { simde_x_mm256_set_epu8(UINT8_C(144), UINT8_C( 62), UINT8_C(142), UINT8_C(211),
                             UINT8_C( 41), UINT8_C(162), UINT8_C(184), UINT8_C(180),
                             UINT8_C(242), UINT8_C( 80), UINT8_C( 73), UINT8_C( 63),
                             UINT8_C(168), UINT8_C(176), UINT8_C( 12), UINT8_C( 85),
                             UINT8_C( 62), UINT8_C( 83), UINT8_C( 52), UINT8_C( 39),
                             UINT8_C( 56), UINT8_C(245), UINT8_C( 65), UINT8_C(213),
                             UINT8_C( 94), UINT8_C( 88), UINT8_C(157), UINT8_C(124),
                             UINT8_C(123), UINT8_C(196), UINT8_C( 79), UINT8_C( 49)),
      simde_x_mm256_set_epu8(UINT8_C(142), UINT8_C( 71), UINT8_C(103), UINT8_C(139),
                             UINT8_C(236), UINT8_C(100), UINT8_C(139), UINT8_C(154),
                             UINT8_C(203), UINT8_C(125), UINT8_C(237), UINT8_C(236),
                             UINT8_C( 45), UINT8_C(103), UINT8_C( 56), UINT8_C( 68),
                             UINT8_C( 26), UINT8_C( 73), UINT8_C(130), UINT8_C( 19),
                             UINT8_C(222), UINT8_C(206), UINT8_C(129), UINT8_C(101),
                             UINT8_C( 56), UINT8_C(145), UINT8_C(213), UINT8_C(249),
                             UINT8_C(145), UINT8_C( 67), UINT8_C(236), UINT8_C(136)),
      simde_x_mm256_set_epu8(UINT8_C(255), UINT8_C(133), UINT8_C(245), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(205), UINT8_C(255), UINT8_C(255),
                             UINT8_C(213), UINT8_C(255), UINT8_C( 68), UINT8_C(153),
                             UINT8_C( 88), UINT8_C(156), UINT8_C(182), UINT8_C( 58),
                             UINT8_C(255), UINT8_C(255), UINT8_C(194), UINT8_C(255),
                             UINT8_C(150), UINT8_C(233), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(185)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_adds_epu8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_adds_epu16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu16(UINT16_C( 46173), UINT16_C( 51502), UINT16_C( 53334), UINT16_C( 43223),
                              UINT16_C( 23928), UINT16_C( 20321), UINT16_C( 51743), UINT16_C( 37618),
                              UINT16_C( 65078), UINT16_C(  7253), UINT16_C( 52827), UINT16_C( 55189),
                              UINT16_C(  4987), UINT16_C( 17665), UINT16_C( 14350), UINT16_C( 38811)),
      simde_x_mm256_set_epu16(UINT16_C( 37136), UINT16_C( 48792), UINT16_C(  4303), UINT16_C(  5479),
                              UINT16_C( 25191), UINT16_C( 58551), UINT16_C( 48285), UINT16_C(  4997),
                              UINT16_C( 11428), UINT16_C( 55078), UINT16_C( 56047), UINT16_C( 52960),
                              UINT16_C( 65202), UINT16_C( 38018), UINT16_C(   624), UINT16_C(  7777)),
      simde_x_mm256_set_epu16(UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 57637), UINT16_C( 48702),
                              UINT16_C( 49119), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 42615),
                              UINT16_C( 65535), UINT16_C( 62331), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 55683), UINT16_C( 14974), UINT16_C( 46588)) },
    { simde_x_mm256_set_epu16(UINT16_C( 32326), UINT16_C( 24612), UINT16_C( 21846), UINT16_C( 55200),
                              UINT16_C( 41586), UINT16_C( 19758), UINT16_C( 51650), UINT16_C( 38909),
                              UINT16_C( 46063), UINT16_C( 28335), UINT16_C(  9134), UINT16_C( 41639),
                              UINT16_C( 55060), UINT16_C( 39890), UINT16_C(  8293), UINT16_C( 53471)),
      simde_x_mm256_set_epu16(UINT16_C( 41234), UINT16_C(  2183), UINT16_C( 63540), UINT16_C(  1722),
                              UINT16_C( 28608), UINT16_C( 25104), UINT16_C( 45712), UINT16_C( 60419),
                              UINT16_C( 19160), UINT16_C(  3341), UINT16_C( 25253), UINT16_C( 20430),
                              UINT16_C( 42207), UINT16_C( 24498), UINT16_C( 15365), UINT16_C( 19541)),
      simde_x_mm256_set_epu16(UINT16_C( 65535), UINT16_C( 26795), UINT16_C( 65535), UINT16_C( 56922),
                              UINT16_C( 65535), UINT16_C( 44862), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65223), UINT16_C( 31676), UINT16_C( 34387), UINT16_C( 62069),
                              UINT16_C( 65535), UINT16_C( 64388), UINT16_C( 23658), UINT16_C( 65535)) },
    { simde_x_mm256_set_epu16(UINT16_C( 26386), UINT16_C( 63006), UINT16_C( 58203), UINT16_C( 62377),
                              UINT16_C( 47602), UINT16_C( 30067), UINT16_C( 23114), UINT16_C( 33473),
                              UINT16_C( 53575), UINT16_C( 38408), UINT16_C( 17337), UINT16_C( 42126),
                              UINT16_C(  5249), UINT16_C( 35315), UINT16_C( 42156), UINT16_C( 18091)),
      simde_x_mm256_set_epu16(UINT16_C( 52612), UINT16_C( 30768), UINT16_C( 53242), UINT16_C( 17367),
                              UINT16_C( 55155), UINT16_C( 55208), UINT16_C( 40791), UINT16_C( 34106),
                              UINT16_C( 38398), UINT16_C(  3526), UINT16_C( 48471), UINT16_C( 61865),
                              UINT16_C( 26735), UINT16_C( 59797), UINT16_C( 61911), UINT16_C(  9267)),
      simde_x_mm256_set_epu16(UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 63905), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 41934), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 31984), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 27358)) },
    { simde_x_mm256_set_epu16(UINT16_C( 52292), UINT16_C( 60997), UINT16_C( 31428), UINT16_C(  9305),
                              UINT16_C( 57362), UINT16_C( 62823), UINT16_C(  6394), UINT16_C( 15984),
                              UINT16_C( 50964), UINT16_C( 21850), UINT16_C( 44748), UINT16_C( 36872),
                              UINT16_C( 14263), UINT16_C( 48235), UINT16_C(  9404), UINT16_C( 55410)),
      simde_x_mm256_set_epu16(UINT16_C(  5822), UINT16_C( 53891), UINT16_C( 53572), UINT16_C( 35299),
                              UINT16_C( 32303), UINT16_C( 47360), UINT16_C(  2824), UINT16_C( 19749),
                              UINT16_C(  6763), UINT16_C( 47400), UINT16_C( 29201), UINT16_C( 32332),
                              UINT16_C( 24570), UINT16_C( 50755), UINT16_C( 22545), UINT16_C(  9382)),
      simde_x_mm256_set_epu16(UINT16_C( 58114), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 44604),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C(  9218), UINT16_C( 35733),
                              UINT16_C( 57727), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 38833), UINT16_C( 65535), UINT16_C( 31949), UINT16_C( 64792)) },
    { simde_x_mm256_set_epu16(UINT16_C( 50076), UINT16_C(  9743), UINT16_C( 10727), UINT16_C( 51592),
                              UINT16_C( 28130), UINT16_C( 34226), UINT16_C( 44764), UINT16_C( 58424),
                              UINT16_C( 15168), UINT16_C(  4051), UINT16_C( 54044), UINT16_C(  7020),
                              UINT16_C( 31115), UINT16_C( 49299), UINT16_C( 40742), UINT16_C( 48855)),
      simde_x_mm256_set_epu16(UINT16_C( 50898), UINT16_C( 60971), UINT16_C(  2964), UINT16_C( 53140),
                              UINT16_C( 39951), UINT16_C( 57637), UINT16_C( 63735), UINT16_C( 40101),
                              UINT16_C( 37326), UINT16_C( 12531), UINT16_C( 29670), UINT16_C( 49503),
                              UINT16_C( 64935), UINT16_C( 44011), UINT16_C( 59422), UINT16_C( 45053)),
      simde_x_mm256_set_epu16(UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 13691), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 52494), UINT16_C( 16582), UINT16_C( 65535), UINT16_C( 56523),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535)) },
    { simde_x_mm256_set_epu16(UINT16_C( 12906), UINT16_C( 34342), UINT16_C( 49900), UINT16_C( 16373),
                              UINT16_C( 41099), UINT16_C( 42680), UINT16_C( 11034), UINT16_C(  4050),
                              UINT16_C( 41370), UINT16_C( 18241), UINT16_C( 60694), UINT16_C( 59842),
                              UINT16_C(  7044), UINT16_C( 38288), UINT16_C( 18779), UINT16_C( 33204)),
      simde_x_mm256_set_epu16(UINT16_C( 39975), UINT16_C( 26379), UINT16_C( 49406), UINT16_C( 11197),
                              UINT16_C( 25665), UINT16_C( 45876), UINT16_C(  8978), UINT16_C( 56112),
                              UINT16_C(  5541), UINT16_C( 41816), UINT16_C(  1773), UINT16_C( 25366),
                              UINT16_C( 51395), UINT16_C( 19553), UINT16_C( 37079), UINT16_C( 32528)),
      simde_x_mm256_set_epu16(UINT16_C( 52881), UINT16_C( 60721), UINT16_C( 65535), UINT16_C( 27570),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 20012), UINT16_C( 60162),
                              UINT16_C( 46911), UINT16_C( 60057), UINT16_C( 62467), UINT16_C( 65535),
                              UINT16_C( 58439), UINT16_C( 57841), UINT16_C( 55858), UINT16_C( 65535)) },
    { simde_x_mm256_set_epu16(UINT16_C( 54610), UINT16_C( 56071), UINT16_C( 54559), UINT16_C( 11899),
                              UINT16_C( 32307), UINT16_C(  4962), UINT16_C( 27630), UINT16_C( 33600),
                              UINT16_C( 31852), UINT16_C( 56358), UINT16_C( 35796), UINT16_C(  8024),
                              UINT16_C( 46221), UINT16_C(  4529), UINT16_C( 49147), UINT16_C(  8518)),
      simde_x_mm256_set_epu16(UINT16_C( 57675), UINT16_C(  2883), UINT16_C( 55066), UINT16_C( 41648),
                              UINT16_C( 12159), UINT16_C( 20265), UINT16_C( 47525), UINT16_C( 54059),
                              UINT16_C( 12623), UINT16_C( 11063), UINT16_C( 34242), UINT16_C( 57692),
                              UINT16_C(  8071), UINT16_C(  9806), UINT16_C( 30691), UINT16_C( 35776)),
      simde_x_mm256_set_epu16(UINT16_C( 65535), UINT16_C( 58954), UINT16_C( 65535), UINT16_C( 53547),
                              UINT16_C( 44466), UINT16_C( 25227), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 44475), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 54292), UINT16_C( 14335), UINT16_C( 65535), UINT16_C( 44294)) },
    { simde_x_mm256_set_epu16(UINT16_C( 56834), UINT16_C( 64885), UINT16_C(  9140), UINT16_C( 13056),
                              UINT16_C( 40842), UINT16_C( 10347), UINT16_C(  7339), UINT16_C( 17877),
                              UINT16_C( 14924), UINT16_C( 16868), UINT16_C( 50139), UINT16_C( 42854),
                              UINT16_C( 20413), UINT16_C( 64148), UINT16_C( 24871), UINT16_C( 35734)),
      simde_x_mm256_set_epu16(UINT16_C( 20840), UINT16_C( 44144), UINT16_C( 58177), UINT16_C( 28709),
                              UINT16_C( 14233), UINT16_C( 55224), UINT16_C( 50824), UINT16_C( 22009),
                              UINT16_C( 46863), UINT16_C( 40997), UINT16_C( 17728), UINT16_C( 21679),
                              UINT16_C(  9552), UINT16_C( 17236), UINT16_C(  5658), UINT16_C( 51223)),
      simde_x_mm256_set_epu16(UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 41765),
                              UINT16_C( 55075), UINT16_C( 65535), UINT16_C( 58163), UINT16_C( 39886),
                              UINT16_C( 61787), UINT16_C( 57865), UINT16_C( 65535), UINT16_C( 64533),
                              UINT16_C( 29965), UINT16_C( 65535), UINT16_C( 30529), UINT16_C( 65535)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_adds_epu16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_avg_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu8(UINT8_C(132), UINT8_C(185), UINT8_C( 15), UINT8_C(235),
                             UINT8_C(102), UINT8_C(  8), UINT8_C(239), UINT8_C(181),
                             UINT8_C( 81), UINT8_C(155), UINT8_C(236), UINT8_C(191),
                             UINT8_C(133), UINT8_C( 65), UINT8_C( 73), UINT8_C( 40),
                             UINT8_C(181), UINT8_C( 86), UINT8_C( 73), UINT8_C(213),
                             UINT8_C( 85), UINT8_C( 28), UINT8_C( 53), UINT8_C(225),
                             UINT8_C(217), UINT8_C(129), UINT8_C( 68), UINT8_C(183),
                             UINT8_C(232), UINT8_C( 91), UINT8_C(  4), UINT8_C(129)),
      simde_x_mm256_set_epu8(UINT8_C(199), UINT8_C(146), UINT8_C( 73), UINT8_C(129),
                             UINT8_C( 53), UINT8_C( 30), UINT8_C(178), UINT8_C(252),
                             UINT8_C(125), UINT8_C( 44), UINT8_C( 67), UINT8_C( 83),
                             UINT8_C( 52), UINT8_C( 79), UINT8_C(239), UINT8_C(118),
                             UINT8_C(100), UINT8_C( 25), UINT8_C( 74), UINT8_C( 78),
                             UINT8_C( 90), UINT8_C(145), UINT8_C(118), UINT8_C(211),
                             UINT8_C( 29), UINT8_C( 31), UINT8_C(128), UINT8_C( 53),
                             UINT8_C( 19), UINT8_C(239), UINT8_C(181), UINT8_C(108)),
      simde_x_mm256_set_epu8(UINT8_C(166), UINT8_C(166), UINT8_C( 44), UINT8_C(182),
                             UINT8_C( 78), UINT8_C( 19), UINT8_C(209), UINT8_C(217),
                             UINT8_C(103), UINT8_C(100), UINT8_C(152), UINT8_C(137),
                             UINT8_C( 93), UINT8_C( 72), UINT8_C(156), UINT8_C( 79),
                             UINT8_C(141), UINT8_C( 56), UINT8_C( 74), UINT8_C(146),
                             UINT8_C( 88), UINT8_C( 87), UINT8_C( 86), UINT8_C(218),
                             UINT8_C(123), UINT8_C( 80), UINT8_C( 98), UINT8_C(118),
                             UINT8_C(126), UINT8_C(165), UINT8_C( 93), UINT8_C(119)) },
    { simde_x_mm256_set_epu8(UINT8_C(209), UINT8_C(137), UINT8_C(159), UINT8_C(201),
                             UINT8_C(142), UINT8_C(123), UINT8_C(233), UINT8_C(210),
                             UINT8_C(180), UINT8_C(  1), UINT8_C( 19), UINT8_C(  5),
                             UINT8_C( 35), UINT8_C(203), UINT8_C(189), UINT8_C( 26),
                             UINT8_C(153), UINT8_C(140), UINT8_C( 62), UINT8_C(144),
                             UINT8_C( 20), UINT8_C(125), UINT8_C(185), UINT8_C(133),
                             UINT8_C( 90), UINT8_C(243), UINT8_C( 18), UINT8_C(117),
                             UINT8_C(102), UINT8_C(222), UINT8_C( 27), UINT8_C( 12)),
      simde_x_mm256_set_epu8(UINT8_C(201), UINT8_C(154), UINT8_C( 75), UINT8_C(146),
                             UINT8_C( 84), UINT8_C(138), UINT8_C(110), UINT8_C( 18),
                             UINT8_C(142), UINT8_C(147), UINT8_C(192), UINT8_C(131),
                             UINT8_C(151), UINT8_C(232), UINT8_C(176), UINT8_C(  8),
                             UINT8_C(133), UINT8_C( 27), UINT8_C( 52), UINT8_C( 35),
                             UINT8_C(114), UINT8_C(148), UINT8_C(237), UINT8_C(121),
                             UINT8_C(209), UINT8_C( 97), UINT8_C(242), UINT8_C( 75),
                             UINT8_C(194), UINT8_C( 62), UINT8_C(242), UINT8_C(214)),
      simde_x_mm256_set_epu8(UINT8_C(205), UINT8_C(146), UINT8_C(117), UINT8_C(174),
                             UINT8_C(113), UINT8_C(131), UINT8_C(172), UINT8_C(114),
                             UINT8_C(161), UINT8_C( 74), UINT8_C(106), UINT8_C( 68),
                             UINT8_C( 93), UINT8_C(218), UINT8_C(183), UINT8_C( 17),
                             UINT8_C(143), UINT8_C( 84), UINT8_C( 57), UINT8_C( 90),
                             UINT8_C( 67), UINT8_C(137), UINT8_C(211), UINT8_C(127),
                             UINT8_C(150), UINT8_C(170), UINT8_C(130), UINT8_C( 96),
                             UINT8_C(148), UINT8_C(142), UINT8_C(135), UINT8_C(113)) },
    { simde_x_mm256_set_epu8(UINT8_C(223), UINT8_C( 80), UINT8_C( 95), UINT8_C( 57),
                             UINT8_C(173), UINT8_C( 14), UINT8_C( 75), UINT8_C( 79),
                             UINT8_C(206), UINT8_C( 37), UINT8_C(125), UINT8_C(225),
                             UINT8_C(172), UINT8_C(121), UINT8_C( 43), UINT8_C(249),
                             UINT8_C(122), UINT8_C(  3), UINT8_C( 25), UINT8_C(204),
                             UINT8_C(153), UINT8_C( 32), UINT8_C(245), UINT8_C( 58),
                             UINT8_C(211), UINT8_C(116), UINT8_C( 87), UINT8_C(228),
                             UINT8_C(139), UINT8_C(203), UINT8_C(106), UINT8_C(216)),
      simde_x_mm256_set_epu8(UINT8_C( 18), UINT8_C(169), UINT8_C( 46), UINT8_C(100),
                             UINT8_C(246), UINT8_C(  6), UINT8_C(208), UINT8_C(224),
                             UINT8_C(134), UINT8_C(248), UINT8_C( 90), UINT8_C(243),
                             UINT8_C( 95), UINT8_C(216), UINT8_C(232), UINT8_C(245),
                             UINT8_C( 35), UINT8_C(118), UINT8_C( 23), UINT8_C(111),
                             UINT8_C(137), UINT8_C(  1), UINT8_C( 43), UINT8_C(212),
                             UINT8_C(143), UINT8_C(143), UINT8_C(106), UINT8_C(242),
                             UINT8_C(188), UINT8_C( 78), UINT8_C( 94), UINT8_C( 49)),
      simde_x_mm256_set_epu8(UINT8_C(121), UINT8_C(125), UINT8_C( 71), UINT8_C( 79),
                             UINT8_C(210), UINT8_C( 10), UINT8_C(142), UINT8_C(152),
                             UINT8_C(170), UINT8_C(143), UINT8_C(108), UINT8_C(234),
                             UINT8_C(134), UINT8_C(169), UINT8_C(138), UINT8_C(247),
                             UINT8_C( 79), UINT8_C( 61), UINT8_C( 24), UINT8_C(158),
                             UINT8_C(145), UINT8_C( 17), UINT8_C(144), UINT8_C(135),
                             UINT8_C(177), UINT8_C(130), UINT8_C( 97), UINT8_C(235),
                             UINT8_C(164), UINT8_C(141), UINT8_C(100), UINT8_C(133)) },
    { simde_x_mm256_set_epu8(UINT8_C(186), UINT8_C( 51), UINT8_C(166), UINT8_C(159),
                             UINT8_C( 61), UINT8_C(189), UINT8_C(148), UINT8_C(156),
                             UINT8_C(199), UINT8_C( 59), UINT8_C(214), UINT8_C( 21),
                             UINT8_C( 92), UINT8_C( 24), UINT8_C( 35), UINT8_C( 33),
                             UINT8_C( 27), UINT8_C(133), UINT8_C(  9), UINT8_C(114),
                             UINT8_C(170), UINT8_C( 78), UINT8_C(149), UINT8_C(203),
                             UINT8_C(212), UINT8_C(108), UINT8_C(116), UINT8_C(217),
                             UINT8_C(102), UINT8_C(192), UINT8_C(223), UINT8_C( 98)),
      simde_x_mm256_set_epu8(UINT8_C( 51), UINT8_C(193), UINT8_C(129), UINT8_C(222),
                             UINT8_C(147), UINT8_C( 49), UINT8_C(210), UINT8_C(198),
                             UINT8_C(192), UINT8_C(158), UINT8_C( 49), UINT8_C(217),
                             UINT8_C( 20), UINT8_C(183), UINT8_C(213), UINT8_C( 71),
                             UINT8_C(164), UINT8_C( 92), UINT8_C(118), UINT8_C( 17),
                             UINT8_C(236), UINT8_C( 27), UINT8_C(162), UINT8_C( 98),
                             UINT8_C(196), UINT8_C(135), UINT8_C(  7), UINT8_C(172),
                             UINT8_C(233), UINT8_C( 47), UINT8_C(151), UINT8_C(128)),
      simde_x_mm256_set_epu8(UINT8_C(119), UINT8_C(122), UINT8_C(148), UINT8_C(191),
                             UINT8_C(104), UINT8_C(119), UINT8_C(179), UINT8_C(177),
                             UINT8_C(196), UINT8_C(109), UINT8_C(132), UINT8_C(119),
                             UINT8_C( 56), UINT8_C(104), UINT8_C(124), UINT8_C( 52),
                             UINT8_C( 96), UINT8_C(113), UINT8_C( 64), UINT8_C( 66),
                             UINT8_C(203), UINT8_C( 53), UINT8_C(156), UINT8_C(151),
                             UINT8_C(204), UINT8_C(122), UINT8_C( 62), UINT8_C(195),
                             UINT8_C(168), UINT8_C(120), UINT8_C(187), UINT8_C(113)) },
    { simde_x_mm256_set_epu8(UINT8_C(182), UINT8_C(141), UINT8_C( 93), UINT8_C( 91),
                             UINT8_C(  0), UINT8_C(189), UINT8_C(215), UINT8_C(221),
                             UINT8_C(105), UINT8_C(231), UINT8_C( 61), UINT8_C(224),
                             UINT8_C( 68), UINT8_C( 84), UINT8_C(247), UINT8_C(215),
                             UINT8_C(125), UINT8_C(197), UINT8_C( 69), UINT8_C(102),
                             UINT8_C(218), UINT8_C(120), UINT8_C(113), UINT8_C(175),
                             UINT8_C(134), UINT8_C( 33), UINT8_C(106), UINT8_C(117),
                             UINT8_C(129), UINT8_C(249), UINT8_C(194), UINT8_C( 70)),
      simde_x_mm256_set_epu8(UINT8_C(137), UINT8_C( 27), UINT8_C( 17), UINT8_C( 94),
                             UINT8_C(244), UINT8_C(142), UINT8_C(142), UINT8_C( 48),
                             UINT8_C( 54), UINT8_C(217), UINT8_C(209), UINT8_C(122),
                             UINT8_C(  1), UINT8_C(190), UINT8_C( 59), UINT8_C(250),
                             UINT8_C(179), UINT8_C(176), UINT8_C(167), UINT8_C( 57),
                             UINT8_C( 90), UINT8_C( 15), UINT8_C( 31), UINT8_C(140),
                             UINT8_C(209), UINT8_C(104), UINT8_C(139), UINT8_C(154),
                             UINT8_C( 57), UINT8_C(248), UINT8_C(225), UINT8_C( 65)),
      simde_x_mm256_set_epu8(UINT8_C(160), UINT8_C( 84), UINT8_C( 55), UINT8_C( 93),
                             UINT8_C(122), UINT8_C(166), UINT8_C(179), UINT8_C(135),
                             UINT8_C( 80), UINT8_C(224), UINT8_C(135), UINT8_C(173),
                             UINT8_C( 35), UINT8_C(137), UINT8_C(153), UINT8_C(233),
                             UINT8_C(152), UINT8_C(187), UINT8_C(118), UINT8_C( 80),
                             UINT8_C(154), UINT8_C( 68), UINT8_C( 72), UINT8_C(158),
                             UINT8_C(172), UINT8_C( 69), UINT8_C(123), UINT8_C(136),
                             UINT8_C( 93), UINT8_C(249), UINT8_C(210), UINT8_C( 68)) },
    { simde_x_mm256_set_epu8(UINT8_C(125), UINT8_C(242), UINT8_C( 34), UINT8_C(120),
                             UINT8_C(106), UINT8_C(202), UINT8_C(100), UINT8_C( 61),
                             UINT8_C(105), UINT8_C(145), UINT8_C( 46), UINT8_C(129),
                             UINT8_C(208), UINT8_C( 57), UINT8_C( 82), UINT8_C( 21),
                             UINT8_C( 59), UINT8_C( 97), UINT8_C(206), UINT8_C(  4),
                             UINT8_C(182), UINT8_C( 81), UINT8_C(203), UINT8_C(252),
                             UINT8_C(111), UINT8_C( 28), UINT8_C(210), UINT8_C( 57),
                             UINT8_C(214), UINT8_C(124), UINT8_C(137), UINT8_C(114)),
      simde_x_mm256_set_epu8(UINT8_C(208), UINT8_C( 60), UINT8_C( 51), UINT8_C( 61),
                             UINT8_C(249), UINT8_C(203), UINT8_C( 69), UINT8_C(195),
                             UINT8_C( 16), UINT8_C( 67), UINT8_C(241), UINT8_C(244),
                             UINT8_C(217), UINT8_C(201), UINT8_C(104), UINT8_C( 80),
                             UINT8_C( 30), UINT8_C( 40), UINT8_C( 69), UINT8_C( 88),
                             UINT8_C( 83), UINT8_C(141), UINT8_C(221), UINT8_C(174),
                             UINT8_C(165), UINT8_C(114), UINT8_C(107), UINT8_C( 42),
                             UINT8_C( 83), UINT8_C(  1), UINT8_C( 95), UINT8_C( 89)),
      simde_x_mm256_set_epu8(UINT8_C(167), UINT8_C(151), UINT8_C( 43), UINT8_C( 91),
                             UINT8_C(178), UINT8_C(203), UINT8_C( 85), UINT8_C(128),
                             UINT8_C( 61), UINT8_C(106), UINT8_C(144), UINT8_C(187),
                             UINT8_C(213), UINT8_C(129), UINT8_C( 93), UINT8_C( 51),
                             UINT8_C( 45), UINT8_C( 69), UINT8_C(138), UINT8_C( 46),
                             UINT8_C(133), UINT8_C(111), UINT8_C(212), UINT8_C(213),
                             UINT8_C(138), UINT8_C( 71), UINT8_C(159), UINT8_C( 50),
                             UINT8_C(149), UINT8_C( 63), UINT8_C(116), UINT8_C(102)) },
    { simde_x_mm256_set_epu8(UINT8_C( 59), UINT8_C(202), UINT8_C( 28), UINT8_C( 65),
                             UINT8_C( 60), UINT8_C( 92), UINT8_C(112), UINT8_C(105),
                             UINT8_C(229), UINT8_C(116), UINT8_C(242), UINT8_C(217),
                             UINT8_C(203), UINT8_C( 71), UINT8_C( 15), UINT8_C(143),
                             UINT8_C( 58), UINT8_C(228), UINT8_C( 36), UINT8_C(154),
                             UINT8_C( 96), UINT8_C(  2), UINT8_C( 86), UINT8_C( 36),
                             UINT8_C( 93), UINT8_C( 29), UINT8_C( 70), UINT8_C( 20),
                             UINT8_C(130), UINT8_C(172), UINT8_C(152), UINT8_C(189)),
      simde_x_mm256_set_epu8(UINT8_C(100), UINT8_C( 42), UINT8_C( 77), UINT8_C( 21),
                             UINT8_C(144), UINT8_C(197), UINT8_C(242), UINT8_C(243),
                             UINT8_C(205), UINT8_C(204), UINT8_C( 75), UINT8_C(102),
                             UINT8_C( 21), UINT8_C(148), UINT8_C( 70), UINT8_C(128),
                             UINT8_C( 95), UINT8_C(147), UINT8_C( 39), UINT8_C(190),
                             UINT8_C( 20), UINT8_C(128), UINT8_C(196), UINT8_C(160),
                             UINT8_C(  8), UINT8_C(206), UINT8_C( 13), UINT8_C(197),
                             UINT8_C( 93), UINT8_C(253), UINT8_C( 16), UINT8_C( 27)),
      simde_x_mm256_set_epu8(UINT8_C( 80), UINT8_C(122), UINT8_C( 53), UINT8_C( 43),
                             UINT8_C(102), UINT8_C(145), UINT8_C(177), UINT8_C(174),
                             UINT8_C(217), UINT8_C(160), UINT8_C(159), UINT8_C(160),
                             UINT8_C(112), UINT8_C(110), UINT8_C( 43), UINT8_C(136),
                             UINT8_C( 77), UINT8_C(188), UINT8_C( 38), UINT8_C(172),
                             UINT8_C( 58), UINT8_C( 65), UINT8_C(141), UINT8_C( 98),
                             UINT8_C( 51), UINT8_C(118), UINT8_C( 42), UINT8_C(109),
                             UINT8_C(112), UINT8_C(213), UINT8_C( 84), UINT8_C(108)) },
    { simde_x_mm256_set_epu8(UINT8_C( 75), UINT8_C( 17), UINT8_C(162), UINT8_C( 64),
                             UINT8_C(129), UINT8_C(250), UINT8_C(112), UINT8_C(166),
                             UINT8_C( 98), UINT8_C(126), UINT8_C(129), UINT8_C(211),
                             UINT8_C( 27), UINT8_C( 12), UINT8_C(183), UINT8_C(140),
                             UINT8_C(106), UINT8_C(255), UINT8_C(252), UINT8_C(224),
                             UINT8_C(116), UINT8_C(208), UINT8_C( 69), UINT8_C(  4),
                             UINT8_C(193), UINT8_C( 46), UINT8_C(111), UINT8_C( 96),
                             UINT8_C(101), UINT8_C(183), UINT8_C( 99), UINT8_C( 60)),
      simde_x_mm256_set_epu8(UINT8_C( 48), UINT8_C( 27), UINT8_C(253), UINT8_C(118),
                             UINT8_C(225), UINT8_C(134), UINT8_C(250), UINT8_C(133),
                             UINT8_C( 52), UINT8_C( 47), UINT8_C( 27), UINT8_C(213),
                             UINT8_C( 28), UINT8_C(208), UINT8_C( 73), UINT8_C( 89),
                             UINT8_C( 76), UINT8_C(160), UINT8_C( 57), UINT8_C(191),
                             UINT8_C( 34), UINT8_C(121), UINT8_C(194), UINT8_C(205),
                             UINT8_C(102), UINT8_C(106), UINT8_C(175), UINT8_C(219),
                             UINT8_C(174), UINT8_C(128), UINT8_C(137), UINT8_C(235)),
      simde_x_mm256_set_epu8(UINT8_C( 62), UINT8_C( 22), UINT8_C(208), UINT8_C( 91),
                             UINT8_C(177), UINT8_C(192), UINT8_C(181), UINT8_C(150),
                             UINT8_C( 75), UINT8_C( 87), UINT8_C( 78), UINT8_C(212),
                             UINT8_C( 28), UINT8_C(110), UINT8_C(128), UINT8_C(115),
                             UINT8_C( 91), UINT8_C(208), UINT8_C(155), UINT8_C(208),
                             UINT8_C( 75), UINT8_C(165), UINT8_C(132), UINT8_C(105),
                             UINT8_C(148), UINT8_C( 76), UINT8_C(143), UINT8_C(158),
                             UINT8_C(138), UINT8_C(156), UINT8_C(118), UINT8_C(148)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_avg_epu8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_avg_epu16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu16(UINT16_C( 33977), UINT16_C(  4075), UINT16_C( 26120), UINT16_C( 61365),
                              UINT16_C( 20891), UINT16_C( 60607), UINT16_C( 34113), UINT16_C( 18728),
                              UINT16_C( 46422), UINT16_C( 18901), UINT16_C( 21788), UINT16_C( 13793),
                              UINT16_C( 55681), UINT16_C( 17591), UINT16_C( 59483), UINT16_C(  1153)),
      simde_x_mm256_set_epu16(UINT16_C( 51090), UINT16_C( 18817), UINT16_C( 13598), UINT16_C( 45820),
                              UINT16_C( 32044), UINT16_C( 17235), UINT16_C( 13391), UINT16_C( 61302),
                              UINT16_C( 25625), UINT16_C( 19022), UINT16_C( 23185), UINT16_C( 30419),
                              UINT16_C(  7455), UINT16_C( 32821), UINT16_C(  5103), UINT16_C( 46444)),
      simde_x_mm256_set_epu16(UINT16_C( 42534), UINT16_C( 11446), UINT16_C( 19859), UINT16_C( 53593),
                              UINT16_C( 26468), UINT16_C( 38921), UINT16_C( 23752), UINT16_C( 40015),
                              UINT16_C( 36024), UINT16_C( 18962), UINT16_C( 22487), UINT16_C( 22106),
                              UINT16_C( 31568), UINT16_C( 25206), UINT16_C( 32293), UINT16_C( 23799)) },
    { simde_x_mm256_set_epu16(UINT16_C( 53641), UINT16_C( 40905), UINT16_C( 36475), UINT16_C( 59858),
                              UINT16_C( 46081), UINT16_C(  4869), UINT16_C(  9163), UINT16_C( 48410),
                              UINT16_C( 39308), UINT16_C( 16016), UINT16_C(  5245), UINT16_C( 47493),
                              UINT16_C( 23283), UINT16_C(  4725), UINT16_C( 26334), UINT16_C(  6924)),
      simde_x_mm256_set_epu16(UINT16_C( 51610), UINT16_C( 19346), UINT16_C( 21642), UINT16_C( 28178),
                              UINT16_C( 36499), UINT16_C( 49283), UINT16_C( 38888), UINT16_C( 45064),
                              UINT16_C( 34075), UINT16_C( 13347), UINT16_C( 29332), UINT16_C( 60793),
                              UINT16_C( 53601), UINT16_C( 62027), UINT16_C( 49726), UINT16_C( 62166)),
      simde_x_mm256_set_epu16(UINT16_C( 52626), UINT16_C( 30126), UINT16_C( 29059), UINT16_C( 44018),
                              UINT16_C( 41290), UINT16_C( 27076), UINT16_C( 24026), UINT16_C( 46737),
                              UINT16_C( 36692), UINT16_C( 14682), UINT16_C( 17289), UINT16_C( 54143),
                              UINT16_C( 38442), UINT16_C( 33376), UINT16_C( 38030), UINT16_C( 34545)) },
    { simde_x_mm256_set_epu16(UINT16_C( 57168), UINT16_C( 24377), UINT16_C( 44302), UINT16_C( 19279),
                              UINT16_C( 52773), UINT16_C( 32225), UINT16_C( 44153), UINT16_C( 11257),
                              UINT16_C( 31235), UINT16_C(  6604), UINT16_C( 39200), UINT16_C( 62778),
                              UINT16_C( 54132), UINT16_C( 22500), UINT16_C( 35787), UINT16_C( 27352)),
      simde_x_mm256_set_epu16(UINT16_C(  4777), UINT16_C( 11876), UINT16_C( 62982), UINT16_C( 53472),
                              UINT16_C( 34552), UINT16_C( 23283), UINT16_C( 24536), UINT16_C( 59637),
                              UINT16_C(  9078), UINT16_C(  5999), UINT16_C( 35073), UINT16_C( 11220),
                              UINT16_C( 36751), UINT16_C( 27378), UINT16_C( 48206), UINT16_C( 24113)),
      simde_x_mm256_set_epu16(UINT16_C( 30973), UINT16_C( 18127), UINT16_C( 53642), UINT16_C( 36376),
                              UINT16_C( 43663), UINT16_C( 27754), UINT16_C( 34345), UINT16_C( 35447),
                              UINT16_C( 20157), UINT16_C(  6302), UINT16_C( 37137), UINT16_C( 36999),
                              UINT16_C( 45442), UINT16_C( 24939), UINT16_C( 41997), UINT16_C( 25733)) },
    { simde_x_mm256_set_epu16(UINT16_C( 47667), UINT16_C( 42655), UINT16_C( 15805), UINT16_C( 38044),
                              UINT16_C( 51003), UINT16_C( 54805), UINT16_C( 23576), UINT16_C(  8993),
                              UINT16_C(  7045), UINT16_C(  2418), UINT16_C( 43598), UINT16_C( 38347),
                              UINT16_C( 54380), UINT16_C( 29913), UINT16_C( 26304), UINT16_C( 57186)),
      simde_x_mm256_set_epu16(UINT16_C( 13249), UINT16_C( 33246), UINT16_C( 37681), UINT16_C( 53958),
                              UINT16_C( 49310), UINT16_C( 12761), UINT16_C(  5303), UINT16_C( 54599),
                              UINT16_C( 42076), UINT16_C( 30225), UINT16_C( 60443), UINT16_C( 41570),
                              UINT16_C( 50311), UINT16_C(  1964), UINT16_C( 59695), UINT16_C( 38784)),
      simde_x_mm256_set_epu16(UINT16_C( 30458), UINT16_C( 37951), UINT16_C( 26743), UINT16_C( 46001),
                              UINT16_C( 50157), UINT16_C( 33783), UINT16_C( 14440), UINT16_C( 31796),
                              UINT16_C( 24561), UINT16_C( 16322), UINT16_C( 52021), UINT16_C( 39959),
                              UINT16_C( 52346), UINT16_C( 15939), UINT16_C( 43000), UINT16_C( 47985)) },
    { simde_x_mm256_set_epu16(UINT16_C( 46733), UINT16_C( 23899), UINT16_C(   189), UINT16_C( 55261),
                              UINT16_C( 27111), UINT16_C( 15840), UINT16_C( 17492), UINT16_C( 63447),
                              UINT16_C( 32197), UINT16_C( 17766), UINT16_C( 55928), UINT16_C( 29103),
                              UINT16_C( 34337), UINT16_C( 27253), UINT16_C( 33273), UINT16_C( 49734)),
      simde_x_mm256_set_epu16(UINT16_C( 35099), UINT16_C(  4446), UINT16_C( 62606), UINT16_C( 36400),
                              UINT16_C( 14041), UINT16_C( 53626), UINT16_C(   446), UINT16_C( 15354),
                              UINT16_C( 46000), UINT16_C( 42809), UINT16_C( 23055), UINT16_C(  8076),
                              UINT16_C( 53608), UINT16_C( 35738), UINT16_C( 14840), UINT16_C( 57665)),
      simde_x_mm256_set_epu16(UINT16_C( 40916), UINT16_C( 14173), UINT16_C( 31398), UINT16_C( 45831),
                              UINT16_C( 20576), UINT16_C( 34733), UINT16_C(  8969), UINT16_C( 39401),
                              UINT16_C( 39099), UINT16_C( 30288), UINT16_C( 39492), UINT16_C( 18590),
                              UINT16_C( 43973), UINT16_C( 31496), UINT16_C( 24057), UINT16_C( 53700)) },
    { simde_x_mm256_set_epu16(UINT16_C( 32242), UINT16_C(  8824), UINT16_C( 27338), UINT16_C( 25661),
                              UINT16_C( 27025), UINT16_C( 11905), UINT16_C( 53305), UINT16_C( 21013),
                              UINT16_C( 15201), UINT16_C( 52740), UINT16_C( 46673), UINT16_C( 52220),
                              UINT16_C( 28444), UINT16_C( 53817), UINT16_C( 54908), UINT16_C( 35186)),
      simde_x_mm256_set_epu16(UINT16_C( 53308), UINT16_C( 13117), UINT16_C( 63947), UINT16_C( 17859),
                              UINT16_C(  4163), UINT16_C( 61940), UINT16_C( 55753), UINT16_C( 26704),
                              UINT16_C(  7720), UINT16_C( 17752), UINT16_C( 21389), UINT16_C( 56750),
                              UINT16_C( 42354), UINT16_C( 27434), UINT16_C( 21249), UINT16_C( 24409)),
      simde_x_mm256_set_epu16(UINT16_C( 42775), UINT16_C( 10971), UINT16_C( 45643), UINT16_C( 21760),
                              UINT16_C( 15594), UINT16_C( 36923), UINT16_C( 54529), UINT16_C( 23859),
                              UINT16_C( 11461), UINT16_C( 35246), UINT16_C( 34031), UINT16_C( 54485),
                              UINT16_C( 35399), UINT16_C( 40626), UINT16_C( 38079), UINT16_C( 29798)) },
    { simde_x_mm256_set_epu16(UINT16_C( 15306), UINT16_C(  7233), UINT16_C( 15452), UINT16_C( 28777),
                              UINT16_C( 58740), UINT16_C( 62169), UINT16_C( 52039), UINT16_C(  3983),
                              UINT16_C( 15076), UINT16_C(  9370), UINT16_C( 24578), UINT16_C( 22052),
                              UINT16_C( 23837), UINT16_C( 17940), UINT16_C( 33452), UINT16_C( 39101)),
      simde_x_mm256_set_epu16(UINT16_C( 25642), UINT16_C( 19733), UINT16_C( 37061), UINT16_C( 62195),
                              UINT16_C( 52684), UINT16_C( 19302), UINT16_C(  5524), UINT16_C( 18048),
                              UINT16_C( 24467), UINT16_C( 10174), UINT16_C(  5248), UINT16_C( 50336),
                              UINT16_C(  2254), UINT16_C(  3525), UINT16_C( 24061), UINT16_C(  4123)),
      simde_x_mm256_set_epu16(UINT16_C( 20474), UINT16_C( 13483), UINT16_C( 26257), UINT16_C( 45486),
                              UINT16_C( 55712), UINT16_C( 40736), UINT16_C( 28782), UINT16_C( 11016),
                              UINT16_C( 19772), UINT16_C(  9772), UINT16_C( 14913), UINT16_C( 36194),
                              UINT16_C( 13046), UINT16_C( 10733), UINT16_C( 28757), UINT16_C( 21612)) },
    { simde_x_mm256_set_epu16(UINT16_C( 19217), UINT16_C( 41536), UINT16_C( 33274), UINT16_C( 28838),
                              UINT16_C( 25214), UINT16_C( 33235), UINT16_C(  6924), UINT16_C( 46988),
                              UINT16_C( 27391), UINT16_C( 64736), UINT16_C( 29904), UINT16_C( 17668),
                              UINT16_C( 49454), UINT16_C( 28512), UINT16_C( 26039), UINT16_C( 25404)),
      simde_x_mm256_set_epu16(UINT16_C( 12315), UINT16_C( 64886), UINT16_C( 57734), UINT16_C( 64133),
                              UINT16_C( 13359), UINT16_C(  7125), UINT16_C(  7376), UINT16_C( 18777),
                              UINT16_C( 19616), UINT16_C( 14783), UINT16_C(  8825), UINT16_C( 49869),
                              UINT16_C( 26218), UINT16_C( 45019), UINT16_C( 44672), UINT16_C( 35307)),
      simde_x_mm256_set_epu16(UINT16_C( 15766), UINT16_C( 53211), UINT16_C( 45504), UINT16_C( 46486),
                              UINT16_C( 19287), UINT16_C( 20180), UINT16_C(  7150), UINT16_C( 32883),
                              UINT16_C( 23504), UINT16_C( 39760), UINT16_C( 19365), UINT16_C( 33769),
                              UINT16_C( 37836), UINT16_C( 36766), UINT16_C( 35356), UINT16_C( 30356)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_avg_epu16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_blend_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C( -9012), INT16_C( 17188), INT16_C( 20170), INT16_C( -6948),
                            INT16_C(  9138), INT16_C( 24690), INT16_C( -6761), INT16_C( -2618),
                            INT16_C( 30583), INT16_C(  3343), INT16_C( -2458), INT16_C( 32235),
                            INT16_C(-14188), INT16_C( 15906), INT16_C(-17823), INT16_C(  7319)),
      simde_mm256_set_epi16(INT16_C( -5752), INT16_C(-23668), INT16_C(-25159), INT16_C(-19782),
                            INT16_C( 28185), INT16_C(-14343), INT16_C(-18599), INT16_C(   827),
                            INT16_C( -4902), INT16_C(  2482), INT16_C( 14836), INT16_C(-27028),
                            INT16_C( 23821), INT16_C( -1365), INT16_C( -4235), INT16_C(  -888)),
      simde_mm256_set_epi16(INT16_C( -9012), INT16_C( 17188), INT16_C( 20170), INT16_C( -6948),
                            INT16_C( 28185), INT16_C(-14343), INT16_C( -6761), INT16_C(   827),
                            INT16_C( 30583), INT16_C(  3343), INT16_C( -2458), INT16_C( 32235),
                            INT16_C( 23821), INT16_C( -1365), INT16_C(-17823), INT16_C(  -888)) },
    { simde_mm256_set_epi16(INT16_C(  2208), INT16_C( 24143), INT16_C( 20623), INT16_C( -5907),
                            INT16_C(  4359), INT16_C(  6016), INT16_C(  2606), INT16_C(-17968),
                            INT16_C( 24878), INT16_C(-20974), INT16_C( 11542), INT16_C( 18923),
                            INT16_C( 32276), INT16_C(-26730), INT16_C( 20467), INT16_C(-30404)),
      simde_mm256_set_epi16(INT16_C( 27588), INT16_C(-23388), INT16_C( 31848), INT16_C( 22463),
                            INT16_C(-31474), INT16_C(-14474), INT16_C(  2006), INT16_C(-32634),
                            INT16_C( 32036), INT16_C(-13145), INT16_C(-30339), INT16_C(-22528),
                            INT16_C( 17597), INT16_C( 17800), INT16_C( 16042), INT16_C( 23637)),
      simde_mm256_set_epi16(INT16_C(  2208), INT16_C( 24143), INT16_C( 20623), INT16_C( -5907),
                            INT16_C(-31474), INT16_C(-14474), INT16_C(  2606), INT16_C(-32634),
                            INT16_C( 24878), INT16_C(-20974), INT16_C( 11542), INT16_C( 18923),
                            INT16_C( 17597), INT16_C( 17800), INT16_C( 20467), INT16_C( 23637)) },
    { simde_mm256_set_epi16(INT16_C(-30302), INT16_C( -3491), INT16_C(-12187), INT16_C( -9390),
                            INT16_C( -5875), INT16_C(  4739), INT16_C( 19577), INT16_C(  7526),
                            INT16_C( 31347), INT16_C( 10086), INT16_C(-16595), INT16_C( 27410),
                            INT16_C(  1134), INT16_C(-11806), INT16_C(-16010), INT16_C(-25451)),
      simde_mm256_set_epi16(INT16_C(-29901), INT16_C( 18434), INT16_C(  -841), INT16_C( 28759),
                            INT16_C( -1918), INT16_C(  2817), INT16_C(-31249), INT16_C(  6853),
                            INT16_C( 24735), INT16_C(-25824), INT16_C( -1496), INT16_C( 12880),
                            INT16_C( 11586), INT16_C( 24977), INT16_C( 22341), INT16_C(-21470)),
      simde_mm256_set_epi16(INT16_C(-30302), INT16_C( -3491), INT16_C(-12187), INT16_C( -9390),
                            INT16_C( -1918), INT16_C(  2817), INT16_C( 19577), INT16_C(  6853),
                            INT16_C( 31347), INT16_C( 10086), INT16_C(-16595), INT16_C( 27410),
                            INT16_C( 11586), INT16_C( 24977), INT16_C(-16010), INT16_C(-21470)) },
    { simde_mm256_set_epi16(INT16_C( 17074), INT16_C(-20924), INT16_C( 13898), INT16_C( 20227),
                            INT16_C( 12334), INT16_C(-15702), INT16_C( 28564), INT16_C(-15082),
                            INT16_C(-19676), INT16_C(   796), INT16_C( 13442), INT16_C( -9023),
                            INT16_C( 10428), INT16_C( 21588), INT16_C(-25545), INT16_C( 22589)),
      simde_mm256_set_epi16(INT16_C( 13365), INT16_C(-16397), INT16_C(-14658), INT16_C(  8081),
                            INT16_C(  4626), INT16_C(-31038), INT16_C(-27498), INT16_C( -1797),
                            INT16_C(-14919), INT16_C( 31584), INT16_C( 32162), INT16_C( 21664),
                            INT16_C( 32327), INT16_C(  9046), INT16_C( 29457), INT16_C( 18165)),
      simde_mm256_set_epi16(INT16_C( 17074), INT16_C(-20924), INT16_C( 13898), INT16_C( 20227),
                            INT16_C(  4626), INT16_C(-31038), INT16_C( 28564), INT16_C( -1797),
                            INT16_C(-19676), INT16_C(   796), INT16_C( 13442), INT16_C( -9023),
                            INT16_C( 32327), INT16_C(  9046), INT16_C(-25545), INT16_C( 18165)) },
    { simde_mm256_set_epi16(INT16_C(-28976), INT16_C(-17452), INT16_C(-30835), INT16_C(-11288),
                            INT16_C( 23746), INT16_C(-12398), INT16_C( -9605), INT16_C(   914),
                            INT16_C( -6067), INT16_C(  4660), INT16_C( 15780), INT16_C( 30375),
                            INT16_C(-32484), INT16_C( 23271), INT16_C(-15980), INT16_C(  3969)),
      simde_mm256_set_epi16(INT16_C(-14502), INT16_C(-26489), INT16_C( -6738), INT16_C( -1193),
                            INT16_C( 15756), INT16_C(-12605), INT16_C(-12710), INT16_C( -8558),
                            INT16_C( 19027), INT16_C(-19772), INT16_C( 23814), INT16_C(-30071),
                            INT16_C(-29678), INT16_C( 31649), INT16_C(  4669), INT16_C( -4491)),
      simde_mm256_set_epi16(INT16_C(-28976), INT16_C(-17452), INT16_C(-30835), INT16_C(-11288),
                            INT16_C( 15756), INT16_C(-12605), INT16_C( -9605), INT16_C( -8558),
                            INT16_C( -6067), INT16_C(  4660), INT16_C( 15780), INT16_C( 30375),
                            INT16_C(-29678), INT16_C( 31649), INT16_C(-15980), INT16_C( -4491)) },
    { simde_mm256_set_epi16(INT16_C( 16416), INT16_C(-25375), INT16_C(-21092), INT16_C(-20302),
                            INT16_C(-10725), INT16_C(-20142), INT16_C( -4818), INT16_C(-14140),
                            INT16_C(-13625), INT16_C(-24584), INT16_C(  6087), INT16_C(-31850),
                            INT16_C(-29507), INT16_C(  7132), INT16_C( -6862), INT16_C( 26102)),
      simde_mm256_set_epi16(INT16_C(  3513), INT16_C(-30455), INT16_C(-14215), INT16_C(-31390),
                            INT16_C( 22371), INT16_C(-30450), INT16_C(-14197), INT16_C( -3991),
                            INT16_C( 25198), INT16_C( -1251), INT16_C( -4992), INT16_C(-16295),
                            INT16_C( 23622), INT16_C( 28506), INT16_C(-16087), INT16_C(-18392)),
      simde_mm256_set_epi16(INT16_C( 16416), INT16_C(-25375), INT16_C(-21092), INT16_C(-20302),
                            INT16_C( 22371), INT16_C(-30450), INT16_C( -4818), INT16_C( -3991),
                            INT16_C(-13625), INT16_C(-24584), INT16_C(  6087), INT16_C(-31850),
                            INT16_C( 23622), INT16_C( 28506), INT16_C( -6862), INT16_C(-18392)) },
    { simde_mm256_set_epi16(INT16_C( -2375), INT16_C(  3031), INT16_C( 26231), INT16_C(  5999),
                            INT16_C(-10519), INT16_C( 21791), INT16_C(  3889), INT16_C( 28062),
                            INT16_C(-23674), INT16_C(-25444), INT16_C( 16907), INT16_C( 20389),
                            INT16_C(-22712), INT16_C(   486), INT16_C( -2776), INT16_C(-21644)),
      simde_mm256_set_epi16(INT16_C(-29652), INT16_C(   489), INT16_C( -7346), INT16_C(-13391),
                            INT16_C( 21827), INT16_C(  9877), INT16_C(  7842), INT16_C(-13219),
                            INT16_C( 12847), INT16_C( 31187), INT16_C( -8174), INT16_C( -7953),
                            INT16_C(  8071), INT16_C(-19051), INT16_C( 30976), INT16_C( 20848)),
      simde_mm256_set_epi16(INT16_C( -2375), INT16_C(  3031), INT16_C( 26231), INT16_C(  5999),
                            INT16_C( 21827), INT16_C(  9877), INT16_C(  3889), INT16_C(-13219),
                            INT16_C(-23674), INT16_C(-25444), INT16_C( 16907), INT16_C( 20389),
                            INT16_C(  8071), INT16_C(-19051), INT16_C( -2776), INT16_C( 20848)) },
    { simde_mm256_set_epi16(INT16_C( 13214), INT16_C(-27703), INT16_C(  6386), INT16_C(  5153),
                            INT16_C( 26096), INT16_C(  8476), INT16_C( 10527), INT16_C(-23224),
                            INT16_C( 23690), INT16_C(  9355), INT16_C(  1283), INT16_C(-29402),
                            INT16_C( 22593), INT16_C(-12032), INT16_C( -8259), INT16_C( 13457)),
      simde_mm256_set_epi16(INT16_C(-25352), INT16_C( 21231), INT16_C(-11795), INT16_C( 17700),
                            INT16_C(-24048), INT16_C(-11558), INT16_C( -1645), INT16_C( 21362),
                            INT16_C( 18474), INT16_C( 30559), INT16_C(  -790), INT16_C( 30067),
                            INT16_C(  3488), INT16_C(  3834), INT16_C(  2645), INT16_C(-14787)),
      simde_mm256_set_epi16(INT16_C( 13214), INT16_C(-27703), INT16_C(  6386), INT16_C(  5153),
                            INT16_C(-24048), INT16_C(-11558), INT16_C( 10527), INT16_C( 21362),
                            INT16_C( 23690), INT16_C(  9355), INT16_C(  1283), INT16_C(-29402),
                            INT16_C(  3488), INT16_C(  3834), INT16_C( -8259), INT16_C(-14787)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_blend_epi16(test_vec[i].a, test_vec[i].b, 13);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_blend_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(   67571941), INT32_C(-1405773426), INT32_C( 1540271825), INT32_C( 2065572299),
                            INT32_C( -582398487), INT32_C( 1269568238), INT32_C( -277360429), INT32_C(  355946014)),
      simde_mm256_set_epi32(INT32_C(-1175528322), INT32_C( -128390122), INT32_C( 1870386786), INT32_C( 1369967555),
                            INT32_C(  417868105), INT32_C(  128490599), INT32_C( 1283738263), INT32_C( -937782732)),
      simde_mm256_set_epi32(INT32_C(   67571941), INT32_C(-1405773426), INT32_C( 1540271825), INT32_C( 2065572299),
                            INT32_C(  417868105), INT32_C(  128490599), INT32_C( -277360429), INT32_C( -937782732)) },
    { simde_mm256_set_epi32(INT32_C( 1471616457), INT32_C(  153196965), INT32_C(  177786947), INT32_C(-1953704859),
                            INT32_C(-2052334624), INT32_C(-1985445584), INT32_C(  389650479), INT32_C(-1304302924)),
      simde_mm256_set_epi32(INT32_C( -934897433), INT32_C( 1646862966), INT32_C(-1085276514), INT32_C(  112227015),
                            INT32_C(-1389537102), INT32_C(  687724210), INT32_C( 1265543631), INT32_C(  346850755)),
      simde_mm256_set_epi32(INT32_C( 1471616457), INT32_C(  153196965), INT32_C(  177786947), INT32_C(-1953704859),
                            INT32_C(-1389537102), INT32_C(  687724210), INT32_C(  389650479), INT32_C(  346850755)) },
    { simde_mm256_set_epi32(INT32_C(-1682060225), INT32_C(  867867583), INT32_C(  925546319), INT32_C( 1379938785),
                            INT32_C(  653018322), INT32_C( -687296073), INT32_C( -911101701), INT32_C( 1547072378)),
      simde_mm256_set_epi32(INT32_C( 1176167258), INT32_C( -301183666), INT32_C( -466020487), INT32_C(   52703344),
                            INT32_C( 1233020389), INT32_C( 1117532027), INT32_C( 1899739665), INT32_C(-2043295118)),
      simde_mm256_set_epi32(INT32_C(-1682060225), INT32_C(  867867583), INT32_C(  925546319), INT32_C( 1379938785),
                            INT32_C( 1233020389), INT32_C( 1117532027), INT32_C( -911101701), INT32_C(-2043295118)) },
    { simde_mm256_set_epi32(INT32_C(  359138398), INT32_C( -860526519), INT32_C( 1692947884), INT32_C(  772823662),
                            INT32_C( -270939677), INT32_C( 1412661540), INT32_C( 1070011153), INT32_C(  771375046)),
      simde_mm256_set_epi32(INT32_C( -974034130), INT32_C(   37087187), INT32_C( -871436522), INT32_C(   33095078),
                            INT32_C(  715849450), INT32_C(-1345812415), INT32_C(  -45115049), INT32_C( 1960320081)),
      simde_mm256_set_epi32(INT32_C(  359138398), INT32_C( -860526519), INT32_C( 1692947884), INT32_C(  772823662),
                            INT32_C(  715849450), INT32_C(-1345812415), INT32_C( 1070011153), INT32_C( 1960320081)) },
    { simde_mm256_set_epi32(INT32_C( -426383461), INT32_C( -768942960), INT32_C( -264677869), INT32_C( -822820045),
                            INT32_C( 1890345084), INT32_C(-2046745025), INT32_C( -207573670), INT32_C( 1399666591)),
      simde_mm256_set_epi32(INT32_C(  232105709), INT32_C(-1583898310), INT32_C( 1161298300), INT32_C(  169359829),
                            INT32_C(  621794425), INT32_C(  607256107), INT32_C( 1099667121), INT32_C( -184390486)),
      simde_mm256_set_epi32(INT32_C( -426383461), INT32_C( -768942960), INT32_C( -264677869), INT32_C( -822820045),
                            INT32_C(  621794425), INT32_C(  607256107), INT32_C( -207573670), INT32_C( -184390486)) },
    { simde_mm256_set_epi32(INT32_C(-1564290184), INT32_C( -240378472), INT32_C( 1142270593), INT32_C( 1000191111),
                            INT32_C(   20701140), INT32_C(   37555352), INT32_C( -694404400), INT32_C( 1055280730)),
      simde_mm256_set_epi32(INT32_C( 1835031057), INT32_C( 2079483638), INT32_C( 1962415366), INT32_C( -373228817),
                            INT32_C(  142245442), INT32_C(   51427720), INT32_C( 1717201652), INT32_C( 1177983710)),
      simde_mm256_set_epi32(INT32_C(-1564290184), INT32_C( -240378472), INT32_C( 1142270593), INT32_C( 1000191111),
                            INT32_C(  142245442), INT32_C(   51427720), INT32_C( -694404400), INT32_C( 1177983710)) },
    { simde_mm256_set_epi32(INT32_C(-1384452546), INT32_C( -108099055), INT32_C(   -3256672), INT32_C(-2139665218),
                            INT32_C( -280826539), INT32_C( -885573478), INT32_C( 2104257473), INT32_C( 1279376382)),
      simde_mm256_set_epi32(INT32_C( 1706246197), INT32_C(-1331652281), INT32_C( 1192842905), INT32_C( -885790109),
                            INT32_C(-1010846518), INT32_C( -536721191), INT32_C( 1967911533), INT32_C( 1933417937)),
      simde_mm256_set_epi32(INT32_C(-1384452546), INT32_C( -108099055), INT32_C(   -3256672), INT32_C(-2139665218),
                            INT32_C(-1010846518), INT32_C( -536721191), INT32_C( 2104257473), INT32_C( 1933417937)) },
    { simde_mm256_set_epi32(INT32_C( 1532802072), INT32_C(  125283422), INT32_C(-1578036874), INT32_C(  445027764),
                            INT32_C( -409254011), INT32_C( 1098938926), INT32_C(-1086732528), INT32_C( -812360922)),
      simde_mm256_set_epi32(INT32_C( 1974078859), INT32_C( 2037136311), INT32_C( 1463878416), INT32_C(  122656324),
                            INT32_C(-2126065903), INT32_C(-1726635542), INT32_C(-1755031182), INT32_C( 1725515904)),
      simde_mm256_set_epi32(INT32_C( 1532802072), INT32_C(  125283422), INT32_C(-1578036874), INT32_C(  445027764),
                            INT32_C(-2126065903), INT32_C(-1726635542), INT32_C(-1086732528), INT32_C( 1725515904)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_blend_epi32(test_vec[i].a, test_vec[i].b, 13);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_blendv_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i i;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C( 119), INT8_C(  14), INT8_C( 127), INT8_C(-117),
                           INT8_C( -92), INT8_C(  49), INT8_C( -15), INT8_C( -64),
                           INT8_C(  66), INT8_C(-125), INT8_C(  43), INT8_C(   6),
                           INT8_C(-117), INT8_C(  95), INT8_C(   4), INT8_C( 100),
                           INT8_C(  82), INT8_C( -66), INT8_C(  57), INT8_C( -98),
                           INT8_C( -94), INT8_C(   7), INT8_C(  58), INT8_C(-109),
                           INT8_C( -21), INT8_C( -88), INT8_C(  12), INT8_C(  65),
                           INT8_C(-108), INT8_C( -25), INT8_C( -96), INT8_C(  -4)),
      simde_mm256_set_epi8(INT8_C( -68), INT8_C(  -8), INT8_C(  57), INT8_C( 103),
                           INT8_C(  55), INT8_C( 120), INT8_C(   1), INT8_C(  68),
                           INT8_C(  85), INT8_C(-119), INT8_C(  -3), INT8_C(  60),
                           INT8_C(-115), INT8_C( -64), INT8_C( 126), INT8_C( -67),
                           INT8_C(  11), INT8_C(  73), INT8_C(  97), INT8_C( -47),
                           INT8_C(  73), INT8_C( -17), INT8_C(  -4), INT8_C(  19),
                           INT8_C( -66), INT8_C(  -1), INT8_C(  84), INT8_C( -63),
                           INT8_C(  58), INT8_C( 114), INT8_C( 107), INT8_C( -13)),
      simde_mm256_set_epi8(INT8_C( -94), INT8_C( -13), INT8_C( -63), INT8_C(  48),
                           INT8_C(  72), INT8_C( -33), INT8_C(-118), INT8_C(  86),
                           INT8_C(  17), INT8_C( 112), INT8_C(   9), INT8_C( 106),
                           INT8_C( 115), INT8_C( -15), INT8_C(  74), INT8_C(   5),
                           INT8_C(  17), INT8_C( -31), INT8_C(-120), INT8_C(  18),
                           INT8_C(  82), INT8_C( -67), INT8_C(-101), INT8_C(-109),
                           INT8_C(  81), INT8_C(-126), INT8_C(  -8), INT8_C(   4),
                           INT8_C( 123), INT8_C( 125), INT8_C(  70), INT8_C(  66)),
      simde_mm256_set_epi8(INT8_C( -68), INT8_C(  -8), INT8_C(  57), INT8_C(-117),
                           INT8_C( -92), INT8_C( 120), INT8_C(   1), INT8_C( -64),
                           INT8_C(  66), INT8_C(-125), INT8_C(  43), INT8_C(   6),
                           INT8_C(-117), INT8_C( -64), INT8_C(   4), INT8_C( 100),
                           INT8_C(  82), INT8_C(  73), INT8_C(  97), INT8_C( -98),
                           INT8_C( -94), INT8_C( -17), INT8_C(  -4), INT8_C(  19),
                           INT8_C( -21), INT8_C(  -1), INT8_C(  84), INT8_C(  65),
                           INT8_C(-108), INT8_C( -25), INT8_C( -96), INT8_C(  -4)) },
    { simde_mm256_set_epi8(INT8_C( -95), INT8_C(  48), INT8_C(-106), INT8_C(   2),
                           INT8_C( -98), INT8_C(  42), INT8_C( 111), INT8_C( -63),
                           INT8_C(  63), INT8_C(  60), INT8_C(-104), INT8_C(-108),
                           INT8_C( -18), INT8_C( -88), INT8_C(  55), INT8_C(  21),
                           INT8_C(-103), INT8_C( 114), INT8_C( -80), INT8_C(  89),
                           INT8_C(  60), INT8_C(  45), INT8_C( -20), INT8_C(  79),
                           INT8_C(-112), INT8_C( -14), INT8_C( -11), INT8_C( 117),
                           INT8_C(  14), INT8_C( -21), INT8_C(  11), INT8_C(-119)),
      simde_mm256_set_epi8(INT8_C(   7), INT8_C( -82), INT8_C( -49), INT8_C(   6),
                           INT8_C(-108), INT8_C(-115), INT8_C( -95), INT8_C( -53),
                           INT8_C(   6), INT8_C(  28), INT8_C(  47), INT8_C(  86),
                           INT8_C( -76), INT8_C(-114), INT8_C( 114), INT8_C(  94),
                           INT8_C(  90), INT8_C(  13), INT8_C( 121), INT8_C(-123),
                           INT8_C( -33), INT8_C( -67), INT8_C(  30), INT8_C( -90),
                           INT8_C(  -5), INT8_C(  42), INT8_C( 102), INT8_C(  53),
                           INT8_C(  95), INT8_C(  56), INT8_C(   1), INT8_C( -15)),
      simde_mm256_set_epi8(INT8_C(-117), INT8_C( -35), INT8_C(  72), INT8_C( -41),
                           INT8_C(-118), INT8_C( 110), INT8_C(  61), INT8_C(  45),
                           INT8_C(-112), INT8_C( 112), INT8_C(  46), INT8_C( -75),
                           INT8_C(  71), INT8_C( -85), INT8_C( -58), INT8_C( 104),
                           INT8_C(-126), INT8_C(-109), INT8_C(  86), INT8_C(  70),
                           INT8_C( -85), INT8_C( 113), INT8_C(  90), INT8_C(  35),
                           INT8_C( -48), INT8_C( -83), INT8_C(  16), INT8_C(  55),
                           INT8_C( -13), INT8_C(  42), INT8_C( -55), INT8_C( -15)),
      simde_mm256_set_epi8(INT8_C(   7), INT8_C( -82), INT8_C(-106), INT8_C(   6),
                           INT8_C(-108), INT8_C(  42), INT8_C( 111), INT8_C( -63),
                           INT8_C(   6), INT8_C(  60), INT8_C(-104), INT8_C(  86),
                           INT8_C( -18), INT8_C(-114), INT8_C( 114), INT8_C(  21),
                           INT8_C(  90), INT8_C(  13), INT8_C( -80), INT8_C(  89),
                           INT8_C( -33), INT8_C(  45), INT8_C( -20), INT8_C(  79),
                           INT8_C(  -5), INT8_C(  42), INT8_C( -11), INT8_C( 117),
                           INT8_C(  95), INT8_C( -21), INT8_C(   1), INT8_C( -15)) },
    { simde_mm256_set_epi8(INT8_C(  12), INT8_C( 118), INT8_C(-120), INT8_C(  17),
                           INT8_C(  61), INT8_C(  24), INT8_C(-114), INT8_C(-117),
                           INT8_C( -20), INT8_C(-118), INT8_C(  93), INT8_C( 106),
                           INT8_C( -99), INT8_C(  -8), INT8_C( 114), INT8_C(-115),
                           INT8_C( -65), INT8_C( -34), INT8_C(  66), INT8_C(  62),
                           INT8_C( 113), INT8_C( -44), INT8_C( 109), INT8_C(  20),
                           INT8_C( 122), INT8_C( -58), INT8_C(   7), INT8_C(-123),
                           INT8_C(  18), INT8_C(  89), INT8_C( -84), INT8_C(-114)),
      simde_mm256_set_epi8(INT8_C( -67), INT8_C( -15), INT8_C( -72), INT8_C( -39),
                           INT8_C( -90), INT8_C( 108), INT8_C(  93), INT8_C( -36),
                           INT8_C( -64), INT8_C(  36), INT8_C( -98), INT8_C(  76),
                           INT8_C( 123), INT8_C( -40), INT8_C( -48), INT8_C(-111),
                           INT8_C(-100), INT8_C(  76), INT8_C( 114), INT8_C(  51),
                           INT8_C(  72), INT8_C(  51), INT8_C( 101), INT8_C(-128),
                           INT8_C( -68), INT8_C(  35), INT8_C( -50), INT8_C(-100),
                           INT8_C(  85), INT8_C(-109), INT8_C( -86), INT8_C(  43)),
      simde_mm256_set_epi8(INT8_C(  21), INT8_C(  44), INT8_C(  93), INT8_C( -14),
                           INT8_C(  76), INT8_C(  78), INT8_C( 101), INT8_C(  40),
                           INT8_C( 126), INT8_C( -37), INT8_C( 115), INT8_C(  17),
                           INT8_C(  43), INT8_C( -71), INT8_C(-123), INT8_C( -85),
                           INT8_C( -19), INT8_C(  27), INT8_C(  72), INT8_C(  57),
                           INT8_C( 118), INT8_C( -19), INT8_C( -66), INT8_C( 124),
                           INT8_C( -52), INT8_C( -69), INT8_C(  32), INT8_C(  64),
                           INT8_C( -16), INT8_C(-103), INT8_C( -39), INT8_C( 121)),
      simde_mm256_set_epi8(INT8_C(  12), INT8_C( 118), INT8_C(-120), INT8_C( -39),
                           INT8_C(  61), INT8_C(  24), INT8_C(-114), INT8_C(-117),
                           INT8_C( -20), INT8_C(  36), INT8_C(  93), INT8_C( 106),
                           INT8_C( -99), INT8_C( -40), INT8_C( -48), INT8_C(-111),
                           INT8_C(-100), INT8_C( -34), INT8_C(  66), INT8_C(  62),
                           INT8_C( 113), INT8_C(  51), INT8_C( 101), INT8_C(  20),
                           INT8_C( -68), INT8_C(  35), INT8_C(   7), INT8_C(-123),
                           INT8_C(  85), INT8_C(-109), INT8_C( -86), INT8_C(-114)) },
    { simde_mm256_set_epi8(INT8_C( -23), INT8_C( 109), INT8_C(  45), INT8_C( -63),
                           INT8_C(  54), INT8_C( -58), INT8_C( -11), INT8_C(  18),
                           INT8_C(  27), INT8_C( -68), INT8_C(-102), INT8_C(  76),
                           INT8_C( -97), INT8_C(  26), INT8_C(  10), INT8_C( -41),
                           INT8_C(  50), INT8_C(  54), INT8_C(-128), INT8_C(  77),
                           INT8_C( -33), INT8_C( -46), INT8_C( 118), INT8_C(  76),
                           INT8_C( -72), INT8_C( 106), INT8_C( -54), INT8_C(  28),
                           INT8_C(  86), INT8_C(  37), INT8_C(  -1), INT8_C(  83)),
      simde_mm256_set_epi8(INT8_C( 123), INT8_C(  58), INT8_C(  98), INT8_C( 125),
                           INT8_C( 118), INT8_C(-119), INT8_C( -36), INT8_C(  78),
                           INT8_C(   0), INT8_C(  77), INT8_C(  67), INT8_C( 118),
                           INT8_C(  22), INT8_C( -94), INT8_C( -70), INT8_C(  21),
                           INT8_C(  65), INT8_C(  89), INT8_C(  34), INT8_C(  21),
                           INT8_C(  65), INT8_C( -53), INT8_C( -43), INT8_C( -55),
                           INT8_C( -12), INT8_C( -79), INT8_C(-123), INT8_C(  80),
                           INT8_C( -58), INT8_C( -76), INT8_C( -82), INT8_C(  16)),
      simde_mm256_set_epi8(INT8_C( -15), INT8_C( -31), INT8_C( 124), INT8_C(  18),
                           INT8_C(  63), INT8_C(-119), INT8_C( -84), INT8_C( -47),
                           INT8_C( 110), INT8_C(  99), INT8_C(  -7), INT8_C( -78),
                           INT8_C( 122), INT8_C(   9), INT8_C(  54), INT8_C(-112),
                           INT8_C(-100), INT8_C(  -9), INT8_C(-127), INT8_C(  67),
                           INT8_C( -60), INT8_C( -43), INT8_C(  76), INT8_C(  10),
                           INT8_C(  83), INT8_C( -83), INT8_C( -60), INT8_C( -94),
                           INT8_C( -63), INT8_C( 113), INT8_C(-114), INT8_C( 124)),
      simde_mm256_set_epi8(INT8_C( 123), INT8_C(  58), INT8_C(  45), INT8_C( -63),
                           INT8_C(  54), INT8_C(-119), INT8_C( -36), INT8_C(  78),
                           INT8_C(  27), INT8_C( -68), INT8_C(  67), INT8_C( 118),
                           INT8_C( -97), INT8_C(  26), INT8_C(  10), INT8_C(  21),
                           INT8_C(  65), INT8_C(  89), INT8_C(  34), INT8_C(  77),
                           INT8_C(  65), INT8_C( -53), INT8_C( 118), INT8_C(  76),
                           INT8_C( -72), INT8_C( -79), INT8_C(-123), INT8_C(  80),
                           INT8_C( -58), INT8_C(  37), INT8_C( -82), INT8_C(  83)) },
    { simde_mm256_set_epi8(INT8_C(  71), INT8_C(  12), INT8_C( 123), INT8_C(-112),
                           INT8_C(-126), INT8_C(  60), INT8_C( 108), INT8_C(-103),
                           INT8_C( 103), INT8_C( 120), INT8_C( -99), INT8_C( 114),
                           INT8_C(  34), INT8_C( -37), INT8_C( -98), INT8_C( -39),
                           INT8_C(-107), INT8_C( -14), INT8_C(  -8), INT8_C( -82),
                           INT8_C( 123), INT8_C(  78), INT8_C(  27), INT8_C(   7),
                           INT8_C(  90), INT8_C(-108), INT8_C( -54), INT8_C(  40),
                           INT8_C(  -9), INT8_C( -18), INT8_C(  12), INT8_C(   6)),
      simde_mm256_set_epi8(INT8_C(  20), INT8_C( -98), INT8_C(  96), INT8_C(   3),
                           INT8_C(  27), INT8_C(  45), INT8_C( -97), INT8_C( -19),
                           INT8_C( -40), INT8_C( -84), INT8_C( -70), INT8_C( -38),
                           INT8_C( -57), INT8_C(-122), INT8_C( -18), INT8_C( -59),
                           INT8_C( -66), INT8_C(  38), INT8_C(  28), INT8_C( -72),
                           INT8_C(  41), INT8_C( -85), INT8_C(-107), INT8_C( 124),
                           INT8_C(-120), INT8_C(   0), INT8_C(  52), INT8_C( -73),
                           INT8_C( -94), INT8_C(-103), INT8_C( 104), INT8_C(-108)),
      simde_mm256_set_epi8(INT8_C( 114), INT8_C(  67), INT8_C(  87), INT8_C(  60),
                           INT8_C( -47), INT8_C(-121), INT8_C( -54), INT8_C( 110),
                           INT8_C(-100), INT8_C( 105), INT8_C( -23), INT8_C(  81),
                           INT8_C(-121), INT8_C(  67), INT8_C(  97), INT8_C(-125),
                           INT8_C(  78), INT8_C( 118), INT8_C(  46), INT8_C( -85),
                           INT8_C(  78), INT8_C( -66), INT8_C(  11), INT8_C(  90),
                           INT8_C(  11), INT8_C( 120), INT8_C(  20), INT8_C(  83),
                           INT8_C( 103), INT8_C( -43), INT8_C(  72), INT8_C( -92)),
      simde_mm256_set_epi8(INT8_C(  71), INT8_C(  12), INT8_C( 123), INT8_C(-112),
                           INT8_C(  27), INT8_C(  45), INT8_C( -97), INT8_C(-103),
                           INT8_C( -40), INT8_C( 120), INT8_C( -70), INT8_C( 114),
                           INT8_C( -57), INT8_C( -37), INT8_C( -98), INT8_C( -59),
                           INT8_C(-107), INT8_C( -14), INT8_C(  -8), INT8_C( -72),
                           INT8_C( 123), INT8_C( -85), INT8_C(  27), INT8_C(   7),
                           INT8_C(  90), INT8_C(-108), INT8_C( -54), INT8_C(  40),
                           INT8_C(  -9), INT8_C(-103), INT8_C(  12), INT8_C(-108)) },
    { simde_mm256_set_epi8(INT8_C(-120), INT8_C( -17), INT8_C(  71), INT8_C( 108),
                           INT8_C(-124), INT8_C(   8), INT8_C(  38), INT8_C(-124),
                           INT8_C( -33), INT8_C( -80), INT8_C(  82), INT8_C( -52),
                           INT8_C( -85), INT8_C(  20), INT8_C( -43), INT8_C(  49),
                           INT8_C(-127), INT8_C(  82), INT8_C(-125), INT8_C(-107),
                           INT8_C(  99), INT8_C( -94), INT8_C(  98), INT8_C( 124),
                           INT8_C(  23), INT8_C(  -8), INT8_C( -30), INT8_C( 107),
                           INT8_C( -17), INT8_C( -36), INT8_C(  65), INT8_C(  35)),
      simde_mm256_set_epi8(INT8_C(  50), INT8_C( -42), INT8_C(  70), INT8_C(  74),
                           INT8_C( -25), INT8_C( 118), INT8_C(  53), INT8_C(  91),
                           INT8_C( -24), INT8_C(  44), INT8_C(-107), INT8_C(  38),
                           INT8_C(  62), INT8_C(  93), INT8_C(  84), INT8_C(  86),
                           INT8_C(  25), INT8_C(  25), INT8_C(  -1), INT8_C( -49),
                           INT8_C( -69), INT8_C( -81), INT8_C(  33), INT8_C( -90),
                           INT8_C( 118), INT8_C(  99), INT8_C( -68), INT8_C(  38),
                           INT8_C( -18), INT8_C( -20), INT8_C( -37), INT8_C( -57)),
      simde_mm256_set_epi8(INT8_C(  36), INT8_C(  48), INT8_C(-118), INT8_C( 125),
                           INT8_C(  12), INT8_C( 115), INT8_C( -38), INT8_C(   1),
                           INT8_C( -43), INT8_C(-120), INT8_C( 119), INT8_C(  88),
                           INT8_C(  19), INT8_C(   1), INT8_C(  40), INT8_C( -39),
                           INT8_C( -67), INT8_C(  66), INT8_C( -86), INT8_C(  38),
                           INT8_C(  40), INT8_C( -46), INT8_C( 104), INT8_C( -95),
                           INT8_C( 115), INT8_C( -49), INT8_C(  15), INT8_C(-101),
                           INT8_C( -79), INT8_C(  -5), INT8_C( 112), INT8_C( -45)),
      simde_mm256_set_epi8(INT8_C(-120), INT8_C( -17), INT8_C(  70), INT8_C( 108),
                           INT8_C(-124), INT8_C(   8), INT8_C(  53), INT8_C(-124),
                           INT8_C( -24), INT8_C(  44), INT8_C(  82), INT8_C( -52),
                           INT8_C( -85), INT8_C(  20), INT8_C( -43), INT8_C(  86),
                           INT8_C(  25), INT8_C(  82), INT8_C(  -1), INT8_C(-107),
                           INT8_C(  99), INT8_C( -81), INT8_C(  98), INT8_C( -90),
                           INT8_C(  23), INT8_C(  99), INT8_C( -30), INT8_C(  38),
                           INT8_C( -18), INT8_C( -20), INT8_C(  65), INT8_C( -57)) },
    { simde_mm256_set_epi8(INT8_C(  55), INT8_C(  33), INT8_C(  43), INT8_C(  65),
                           INT8_C( -59), INT8_C( -95), INT8_C(  10), INT8_C(  11),
                           INT8_C( 122), INT8_C(  35), INT8_C(  55), INT8_C( 107),
                           INT8_C( -46), INT8_C(-119), INT8_C(-106), INT8_C(  77),
                           INT8_C( -56), INT8_C( -78), INT8_C( -97), INT8_C(-126),
                           INT8_C( -36), INT8_C(  17), INT8_C(  13), INT8_C(  38),
                           INT8_C(  52), INT8_C( -31), INT8_C(  68), INT8_C(  51),
                           INT8_C( -16), INT8_C(  61), INT8_C( -72), INT8_C( -48)),
      simde_mm256_set_epi8(INT8_C( 123), INT8_C(-111), INT8_C( -83), INT8_C(  -8),
                           INT8_C(  66), INT8_C( -69), INT8_C( -27), INT8_C(-103),
                           INT8_C(-112), INT8_C( -80), INT8_C(  56), INT8_C(  60),
                           INT8_C(-106), INT8_C(  -4), INT8_C(-128), INT8_C(-122),
                           INT8_C(  74), INT8_C( -55), INT8_C( -48), INT8_C(-128),
                           INT8_C(  -2), INT8_C(  63), INT8_C( -37), INT8_C(  18),
                           INT8_C( -89), INT8_C( -76), INT8_C(  42), INT8_C(  62),
                           INT8_C(  74), INT8_C( -87), INT8_C(  30), INT8_C(-103)),
      simde_mm256_set_epi8(INT8_C(   4), INT8_C( -10), INT8_C(  -5), INT8_C(  -1),
                           INT8_C(  -4), INT8_C(  87), INT8_C( -59), INT8_C(  -9),
                           INT8_C( -53), INT8_C(  83), INT8_C( -63), INT8_C(  99),
                           INT8_C(  55), INT8_C( 119), INT8_C(  36), INT8_C( -96),
                           INT8_C( -98), INT8_C(  22), INT8_C(   4), INT8_C(-128),
                           INT8_C(  16), INT8_C( -72), INT8_C( 114), INT8_C(  35),
                           INT8_C( -15), INT8_C( -10), INT8_C(  97), INT8_C(-112),
                           INT8_C(-109), INT8_C( -68), INT8_C(   8), INT8_C( 101)),
      simde_mm256_set_epi8(INT8_C(  55), INT8_C(-111), INT8_C( -83), INT8_C(  -8),
                           INT8_C(  66), INT8_C( -95), INT8_C( -27), INT8_C(-103),
                           INT8_C(-112), INT8_C(  35), INT8_C(  56), INT8_C( 107),
                           INT8_C( -46), INT8_C(-119), INT8_C(-106), INT8_C(-122),
                           INT8_C(  74), INT8_C( -78), INT8_C( -97), INT8_C(-128),
                           INT8_C( -36), INT8_C(  63), INT8_C(  13), INT8_C(  38),
                           INT8_C( -89), INT8_C( -76), INT8_C(  68), INT8_C(  62),
                           INT8_C(  74), INT8_C( -87), INT8_C( -72), INT8_C( -48)) },
    { simde_mm256_set_epi8(INT8_C(  69), INT8_C( -38), INT8_C(-105), INT8_C( -77),
                           INT8_C(   3), INT8_C( -19), INT8_C( -91), INT8_C(  81),
                           INT8_C( -61), INT8_C(  88), INT8_C( -52), INT8_C( -76),
                           INT8_C(-100), INT8_C( -70), INT8_C( -64), INT8_C(  71),
                           INT8_C(  82), INT8_C(-122), INT8_C( -41), INT8_C( -86),
                           INT8_C(   9), INT8_C(  29), INT8_C( -31), INT8_C( -87),
                           INT8_C(-113), INT8_C( 100), INT8_C( 100), INT8_C( -55),
                           INT8_C( -11), INT8_C(  92), INT8_C( -87), INT8_C(  43)),
      simde_mm256_set_epi8(INT8_C(  96), INT8_C(-116), INT8_C(   5), INT8_C(  16),
                           INT8_C( -66), INT8_C(   4), INT8_C(   8), INT8_C(  34),
                           INT8_C(  93), INT8_C(  90), INT8_C( -27), INT8_C(  86),
                           INT8_C( 115), INT8_C(  27), INT8_C( -30), INT8_C(  15),
                           INT8_C(  94), INT8_C( -93), INT8_C( -55), INT8_C( -18),
                           INT8_C(-122), INT8_C(  51), INT8_C( -22), INT8_C( -72),
                           INT8_C( -80), INT8_C(  24), INT8_C(  38), INT8_C(  87),
                           INT8_C( -35), INT8_C( -67), INT8_C(  94), INT8_C( -17)),
      simde_mm256_set_epi8(INT8_C( -19), INT8_C( -11), INT8_C(  77), INT8_C( -75),
                           INT8_C(  98), INT8_C( -18), INT8_C( -28), INT8_C(   5),
                           INT8_C(  86), INT8_C(-117), INT8_C( 114), INT8_C( -52),
                           INT8_C(  63), INT8_C( -51), INT8_C( 110), INT8_C( -22),
                           INT8_C(  54), INT8_C(  36), INT8_C(  44), INT8_C(  72),
                           INT8_C( -14), INT8_C( -61), INT8_C( -76), INT8_C(  51),
                           INT8_C(  40), INT8_C( -28), INT8_C( -97), INT8_C( 114),
                           INT8_C(-128), INT8_C(   9), INT8_C(  94), INT8_C( -35)),
      simde_mm256_set_epi8(INT8_C(  96), INT8_C(-116), INT8_C(-105), INT8_C(  16),
                           INT8_C(   3), INT8_C(   4), INT8_C(   8), INT8_C(  81),
                           INT8_C( -61), INT8_C(  90), INT8_C( -52), INT8_C(  86),
                           INT8_C(-100), INT8_C(  27), INT8_C( -64), INT8_C(  15),
                           INT8_C(  82), INT8_C(-122), INT8_C( -41), INT8_C( -86),
                           INT8_C(-122), INT8_C(  51), INT8_C( -22), INT8_C( -87),
                           INT8_C(-113), INT8_C(  24), INT8_C(  38), INT8_C( -55),
                           INT8_C( -35), INT8_C(  92), INT8_C( -87), INT8_C( -17)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_blendv_epi8(test_vec[i].a, test_vec[i].b, test_vec[i].i);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cmpeq_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8( INT8_C( -34),  INT8_C(  65),  INT8_C( -18),  INT8_C( -94),
                            INT8_C(  20),  INT8_C(-105),  INT8_C(  79),  INT8_C( -26),
                            INT8_C(   7),  INT8_C( -36),  INT8_C(  33),  INT8_C( 123),
                            INT8_C( 115),  INT8_C( -52),  INT8_C( -98),  INT8_C(  62),
                            INT8_C(  64),  INT8_C( -91),  INT8_C(  45),  INT8_C( 102),
                            INT8_C(  21),  INT8_C(-111),  INT8_C( -41),  INT8_C( -14),
                            INT8_C(  91),  INT8_C(-106),  INT8_C(  23),  INT8_C(  86),
                            INT8_C(-108),  INT8_C(  34),  INT8_C(  -9),  INT8_C(  42)),
      simde_mm256_set_epi8( INT8_C( -34),  INT8_C(  65),  INT8_C(-119),  INT8_C( -94),
                            INT8_C( -15),  INT8_C(  95),  INT8_C(  72),  INT8_C(  66),
                            INT8_C(-113),  INT8_C( -36), ~INT8_C(   0),  INT8_C( 123),
                            INT8_C( -47),  INT8_C( -52),  INT8_C(  -7),  INT8_C(  62),
                            INT8_C(  31),  INT8_C(  79),  INT8_C(  20),  INT8_C( 102),
                            INT8_C(  21),  INT8_C(  17),  INT8_C(-125),  INT8_C( -14),
                            INT8_C(  22),  INT8_C(-111),  INT8_C(  23),  INT8_C( 100),
                            INT8_C(-122),  INT8_C(  34),  INT8_C( -26),  INT8_C(  42)),
      simde_mm256_set_epi8(~INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0)) },
    { simde_mm256_set_epi8( INT8_C(  61),  INT8_C(  96),  INT8_C( -54),  INT8_C(  88),
                            INT8_C(-115),  INT8_C(  14),  INT8_C( 100),  INT8_C( -28),
                            INT8_C(  78),  INT8_C( -79),  INT8_C(  80),  INT8_C( 114),
                            INT8_C(  29),  INT8_C( -69),  INT8_C(   6),  INT8_C(-127),
                            INT8_C( 116),  INT8_C(  35),  INT8_C(  49),  INT8_C( -24),
                            INT8_C(   4),  INT8_C(-123),  INT8_C( -95),  INT8_C( -23),
                            INT8_C(  44),  INT8_C(  68),  INT8_C( -84),  INT8_C(  -9),
                            INT8_C( -86),  INT8_C( 117),  INT8_C(  55),  INT8_C( 121)),
      simde_mm256_set_epi8( INT8_C(  61),  INT8_C(  81),  INT8_C(  33),  INT8_C(  88),
                            INT8_C( -34),  INT8_C(  14),  INT8_C( 111),  INT8_C( -28),
                            INT8_C( -70),  INT8_C(-102),  INT8_C( 116),  INT8_C(-102),
                            INT8_C(  29),  INT8_C( -10),  INT8_C(  13),  INT8_C( 109),
                            INT8_C( 116),  INT8_C( -83),  INT8_C(  49),  INT8_C(   1),
                            INT8_C(   4),  INT8_C(-123),  INT8_C( -18),  INT8_C( -23),
                            INT8_C(  44),  INT8_C(  68),  INT8_C(-109),  INT8_C(  54),
                            INT8_C( 123),  INT8_C(  62),  INT8_C(  71),  INT8_C(   3)),
      simde_mm256_set_epi8(~INT8_C(   0),  INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0)) },
    { simde_mm256_set_epi8( INT8_C( -58),  INT8_C( -55),  INT8_C( 126),  INT8_C( -17),
                            INT8_C( -94),  INT8_C( -45),  INT8_C( -42),  INT8_C(  39),
                            INT8_C( -81),  INT8_C( -73),  INT8_C( -20),  INT8_C( -73),
                            INT8_C(  79),  INT8_C(  25),  INT8_C( -86),  INT8_C( -52),
                            INT8_C( 106),  INT8_C(  58),  INT8_C( 113),  INT8_C(  61),
                            INT8_C( -71),  INT8_C(  -3),  INT8_C( 101),  INT8_C(-102),
                            INT8_C( -14),  INT8_C( -40),  INT8_C(  82),  INT8_C( -84),
                            INT8_C(  49),  INT8_C(  46),  INT8_C(  59),  INT8_C( -33)),
      simde_mm256_set_epi8( INT8_C( 110),  INT8_C( -67),  INT8_C( 126),  INT8_C( -17),
                            INT8_C( -94),  INT8_C( -46),  INT8_C( -74),  INT8_C( -37),
                            INT8_C( -81),  INT8_C( -93),  INT8_C(  97),  INT8_C( -73),
                            INT8_C( -66),  INT8_C(  25),  INT8_C(  37),  INT8_C( -52),
                            INT8_C( 106),  INT8_C( -80),  INT8_C( 113),  INT8_C(  61),
                            INT8_C( -50),  INT8_C(  68),  INT8_C( 101),  INT8_C(-102),
                            INT8_C( -14),  INT8_C( -35),  INT8_C(  54),  INT8_C(  -2),
                            INT8_C(  49),  INT8_C(  46),  INT8_C(   6),  INT8_C(   4)),
      simde_mm256_set_epi8( INT8_C(   0),  INT8_C(   0), ~INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0)) },
    { simde_mm256_set_epi8( INT8_C(  -9),  INT8_C(  28),  INT8_C(   1),  INT8_C(   0),
                            INT8_C(  57),  INT8_C( -72),  INT8_C( -61),  INT8_C(  74),
                            INT8_C(-105),  INT8_C(  58),  INT8_C(  84),  INT8_C(  60),
                            INT8_C(  21),  INT8_C(  38),  INT8_C(  16),  INT8_C(  25),
                            INT8_C(   2),  INT8_C( -63),  INT8_C(  28),  INT8_C(  93),
                            INT8_C( -44),  INT8_C( -78),  INT8_C( -50),  INT8_C( -21),
                            INT8_C(  47),  INT8_C(  30),  INT8_C(  97),  INT8_C(  18),
                            INT8_C( -36),  INT8_C(  72),  INT8_C( -66),  INT8_C( 124)),
      simde_mm256_set_epi8( INT8_C(-127),  INT8_C( -27),  INT8_C(  86),  INT8_C( -50),
                            INT8_C(  57),  INT8_C( -49),  INT8_C( -61),  INT8_C(  74),
                            INT8_C(-105),  INT8_C( -45),  INT8_C( 125),  INT8_C(  18),
                            INT8_C( -47),  INT8_C(  70),  INT8_C(  16),  INT8_C(  90),
                            INT8_C(   2),  INT8_C( -44),  INT8_C( 121),  INT8_C(  48),
                            INT8_C( -78),  INT8_C( 120),  INT8_C( -50),  INT8_C( -21),
                            INT8_C(  47),  INT8_C( -59),  INT8_C(  97),  INT8_C(  21),
                            INT8_C( -36),  INT8_C(  72),  INT8_C( -44),  INT8_C( 124)),
      simde_mm256_set_epi8( INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0)) },
    { simde_mm256_set_epi8( INT8_C(  75),  INT8_C(  -6),  INT8_C( -43),  INT8_C( 107),
                            INT8_C(  28),  INT8_C(  81),  INT8_C(  82),  INT8_C(  91),
                            INT8_C(  66),  INT8_C( 107),  INT8_C( -41),  INT8_C( -50),
                            INT8_C(  -5),  INT8_C(  66),  INT8_C(-100),  INT8_C( 112),
                            INT8_C(-109),  INT8_C(  92),  INT8_C( -50),  INT8_C(  68),
                            INT8_C( -54),  INT8_C( 111),  INT8_C(  25),  INT8_C(  36),
                            INT8_C(  30),  INT8_C( -33),  INT8_C(  28),  INT8_C(  68),
                            INT8_C( -69),  INT8_C( 124),  INT8_C(-124),  INT8_C(  48)),
      simde_mm256_set_epi8( INT8_C(  26),  INT8_C( 124),  INT8_C( -85),  INT8_C(  90),
                            INT8_C(  49),  INT8_C( -43),  INT8_C(  82),  INT8_C(-110),
                            INT8_C(  -4),  INT8_C(-103),  INT8_C(-116),  INT8_C( -91),
                            INT8_C( 101),  INT8_C( -60),  INT8_C(  28),  INT8_C( -62),
                            INT8_C(-109),  INT8_C(   2),  INT8_C(  71),  INT8_C(  68),
                            INT8_C(  92),  INT8_C( -96),  INT8_C( -47),  INT8_C(  89),
                            INT8_C(-111),  INT8_C( -89),  INT8_C(  38),  INT8_C( 109),
                            INT8_C( -69),  INT8_C(  57),  INT8_C(  87),  INT8_C(  94)),
      simde_mm256_set_epi8( INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0)) },
    { simde_mm256_set_epi8( INT8_C(  78),  INT8_C(-103),  INT8_C( -83),  INT8_C( -64),
                            INT8_C(  86),  INT8_C(  62),  INT8_C(  40),  INT8_C(-116),
                            INT8_C(  49),  INT8_C(  94),  INT8_C( -31),  INT8_C(  82),
                            INT8_C( -55),  INT8_C(-100),  INT8_C( -61),  INT8_C( -62),
                            INT8_C(-101),  INT8_C( 117),  INT8_C( -20),  INT8_C(  19),
                            INT8_C( -97),  INT8_C( -75),  INT8_C(-101),  INT8_C( -39),
                            INT8_C(  28),  INT8_C(  55),  INT8_C(  85),  INT8_C( 106),
                            INT8_C( -97),  INT8_C( -74),  INT8_C(  51),  INT8_C( -54)),
      simde_mm256_set_epi8( INT8_C(-116),  INT8_C( -91),  INT8_C( 105),  INT8_C( -64),
                            INT8_C(  -6),  INT8_C( -63),  INT8_C(  82),  INT8_C(   4),
                            INT8_C(  -7),  INT8_C( -70),  INT8_C( -31),  INT8_C( -83),
                            INT8_C(  22),  INT8_C(-100),  INT8_C(  96),  INT8_C( -62),
                            INT8_C(-120),  INT8_C( 120),  INT8_C( -20),  INT8_C( -80),
                            INT8_C( -97),  INT8_C( -75),  INT8_C( -29),  INT8_C(   4),
                            INT8_C(  28),  INT8_C(  60),  INT8_C(  79),  INT8_C(  73),
                            INT8_C(  17),  INT8_C(-120),  INT8_C( 110),  INT8_C( 100)),
      simde_mm256_set_epi8( INT8_C(   0),  INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0)) },
    { simde_mm256_set_epi8( INT8_C( 118),  INT8_C(  88),  INT8_C(  13),  INT8_C(  39),
                            INT8_C( -45),  INT8_C(  46),  INT8_C( -85),  INT8_C(  17),
                            INT8_C(  -7),  INT8_C(  75),  INT8_C(  -5),  INT8_C(-120),
                            INT8_C( -77),  INT8_C( -49),  INT8_C(  58),  INT8_C( -50),
                            INT8_C(   0),  INT8_C(  21),  INT8_C( -11),  INT8_C( -48),
                            INT8_C( -85),  INT8_C(  55),  INT8_C( 100), ~INT8_C(   0),
                            INT8_C(   7),  INT8_C(  72),  INT8_C(  25),  INT8_C(-126),
                            INT8_C( -81),  INT8_C( -15),  INT8_C( -25),  INT8_C( -55)),
      simde_mm256_set_epi8( INT8_C( 107),  INT8_C(  67),  INT8_C( -53),  INT8_C(  30),
                            INT8_C(  53),  INT8_C( -25),  INT8_C(  94),  INT8_C(  90),
                            INT8_C( -59),  INT8_C(  75),  INT8_C(  65),  INT8_C(-120),
                            INT8_C(  32),  INT8_C( -11),  INT8_C(  58),  INT8_C(  33),
                            INT8_C(   0),  INT8_C(  44),  INT8_C( -11),  INT8_C(  49),
                            INT8_C( -51),  INT8_C(  55),  INT8_C( 100),  INT8_C( -67),
                            INT8_C( -45),  INT8_C(  89),  INT8_C(  25),  INT8_C( 110),
                            INT8_C(  60),  INT8_C( -42),  INT8_C( -25),  INT8_C(   1)),
      simde_mm256_set_epi8( INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0)) },
    { simde_mm256_set_epi8( INT8_C(  47),  INT8_C( 126),  INT8_C( -95),  INT8_C(-121),
                            INT8_C(  84),  INT8_C(  11),  INT8_C(  -4),  INT8_C( -74),
                            INT8_C(-124),  INT8_C(   4),  INT8_C( -12),  INT8_C(  31),
                            INT8_C(  41),  INT8_C(  31),  INT8_C( -42),  INT8_C(  92),
                            INT8_C( -62),  INT8_C(  -8),  INT8_C( -73),  INT8_C(  14),
                            INT8_C(  22),  INT8_C(-109),  INT8_C( 103),  INT8_C(  90),
                            INT8_C(  98),  INT8_C( -28),  INT8_C( -11),  INT8_C(-120),
                            INT8_C( -81),  INT8_C(   0),  INT8_C(  34),  INT8_C(  36)),
      simde_mm256_set_epi8( INT8_C(  47),  INT8_C(-103),  INT8_C( -95),  INT8_C(-121),
                            INT8_C(  84),  INT8_C(  72),  INT8_C( 125),  INT8_C( -13),
                            INT8_C( -49),  INT8_C(  14),  INT8_C(  44),  INT8_C( 110),
                            INT8_C(  41),  INT8_C(  68),  INT8_C( -42),  INT8_C(  92),
                            INT8_C( 107),  INT8_C(  -8),  INT8_C( 127),  INT8_C(  14),
                            INT8_C(  22),  INT8_C(-109),  INT8_C( -52),  INT8_C(  -3),
                            INT8_C( -78),  INT8_C(  91),  INT8_C(   8),  INT8_C(  23),
                            INT8_C( 110),  INT8_C( -91),  INT8_C(  34),  INT8_C(  17)),
      simde_mm256_set_epi8(~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                           ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0), ~INT8_C(   0),
                            INT8_C(   0), ~INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),
                           ~INT8_C(   0), ~INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
                            INT8_C(   0),  INT8_C(   0), ~INT8_C(   0),  INT8_C(   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cmpeq_epi8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cmpeq_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16( INT16_C( -5413),  INT16_C( -8669),  INT16_C(-20019),  INT16_C( 13281),
                             INT16_C( 31513),  INT16_C( 29495),  INT16_C( 24515),  INT16_C( -4843),
                             INT16_C(-25942),  INT16_C(-22058),  INT16_C( 25862),  INT16_C( 17599),
                             INT16_C(  3410),  INT16_C(-25277),  INT16_C( -9899),  INT16_C( -1157)),
      simde_mm256_set_epi16( INT16_C(-24951),  INT16_C( -8669),  INT16_C(-20019),  INT16_C(-25059),
                             INT16_C(-16550),  INT16_C( 29495),  INT16_C( 14158),  INT16_C( 15756),
                             INT16_C(-12717),  INT16_C( 16217),  INT16_C( 23007),  INT16_C(-29661),
                             INT16_C( -4389),  INT16_C( -6794),  INT16_C( 24406),  INT16_C(  3979)),
      simde_mm256_set_epi16( INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0),  INT16_C(     0),
                             INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),  INT16_C(     0),
                             INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
                             INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0)) },
    { simde_mm256_set_epi16( INT16_C(-24592),  INT16_C(-14625),  INT16_C(-22785),  INT16_C(  -539),
                             INT16_C( 27023),  INT16_C( 22410),  INT16_C( 24512),  INT16_C( 28286),
                             INT16_C(  -371),  INT16_C(-19534),  INT16_C( 20760),  INT16_C( -2933),
                             INT16_C( 19420),  INT16_C( 28265),  INT16_C(  -411),  INT16_C(-25192)),
      simde_mm256_set_epi16( INT16_C(-13031),  INT16_C(-29481),  INT16_C(-21398),  INT16_C(  5762),
                             INT16_C(-11151),  INT16_C( 22410),  INT16_C(-17819),  INT16_C(-18343),
                             INT16_C(  -371),  INT16_C(-18178),  INT16_C( 32217),  INT16_C(-13082),
                             INT16_C( 19420),  INT16_C(-14308),  INT16_C(  -411),  INT16_C(-25192)),
      simde_mm256_set_epi16( INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
                             INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),  INT16_C(     0),
                             ~INT16_C(    0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
                             ~INT16_C(    0),  INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0)) },
    { simde_mm256_set_epi16( INT16_C( -1787),  INT16_C( 14472),  INT16_C(  9645),  INT16_C(  6467),
                             INT16_C(-28251),  INT16_C( -4608),  INT16_C(-20138),  INT16_C(-12375),
                             INT16_C( -3139),  INT16_C(-11799),  INT16_C(-25694),  INT16_C( -9505),
                             INT16_C(  2016),  INT16_C(-12994),  INT16_C( 12032),  INT16_C( -1625)),
      simde_mm256_set_epi16( INT16_C( 14082),  INT16_C( 14472),  INT16_C(  9645),  INT16_C(-10341),
                             INT16_C(-21377),  INT16_C(-28844),  INT16_C(-27109),  INT16_C(-21859),
                             INT16_C( -3139),  INT16_C(-20139),  INT16_C(-25694),  INT16_C( 28917),
                             INT16_C(  5339),  INT16_C( 19240),  INT16_C( 12032),  INT16_C(  2212)),
      simde_mm256_set_epi16( INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0),  INT16_C(     0),
                             INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
                             ~INT16_C(    0),  INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),
                             INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0)) },
    { simde_mm256_set_epi16( INT16_C( 29890),  INT16_C( 31032),  INT16_C(-29157),  INT16_C(-14947),
                             INT16_C(-10307),  INT16_C(-20105),  INT16_C( -2072),  INT16_C( 29196),
                             INT16_C(  1484),  INT16_C(-11865),  INT16_C(-30233),  INT16_C( 30603),
                             INT16_C( 20632),  INT16_C(-27772),  INT16_C(-32273),  INT16_C(  4867)),
      simde_mm256_set_epi16( INT16_C(-21410),  INT16_C(  9749),  INT16_C(-29157),  INT16_C(-14947),
                             INT16_C( 13547),  INT16_C( 28436),  INT16_C( -2072),  INT16_C( 29196),
                             INT16_C(  1484),  INT16_C(-14930),  INT16_C(  6002),  INT16_C( 30603),
                             INT16_C( 27473),  INT16_C(  2844),  INT16_C(-19044),  INT16_C(  4867)),
      simde_mm256_set_epi16( INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0),
                             INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0),
                             ~INT16_C(    0),  INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),
                             INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0)) },
    { simde_mm256_set_epi16( INT16_C( -2810),  INT16_C(-12778),  INT16_C( 16624),  INT16_C(  2706),
                             INT16_C( 14585),  INT16_C( -6866),  INT16_C(  -582),  INT16_C( 29880),
                             INT16_C( 20309),  INT16_C( 25315),  INT16_C( -5634),  INT16_C(-21292),
                             INT16_C( 11215),  INT16_C( 13817),  INT16_C( 26751),  INT16_C( -8288)),
      simde_mm256_set_epi16( INT16_C(-11737),  INT16_C(-19770),  INT16_C(  8400),  INT16_C(  2706),
                             INT16_C( 14599),  INT16_C( -6866),  INT16_C(  -582),  INT16_C( 29880),
                             INT16_C( 20309),  INT16_C( 32436),  INT16_C( -5634),  INT16_C(-21292),
                             INT16_C(-11784),  INT16_C( 13817),  INT16_C( 26751),  INT16_C(-12877)),
      simde_mm256_set_epi16( INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),
                             INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0),  ~INT16_C(    0),
                             ~INT16_C(    0),  INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0),
                             INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0),  INT16_C(     0)) },
    { simde_mm256_set_epi16( INT16_C( 11738),  INT16_C(-30358),  INT16_C(-23042),  INT16_C( -2979),
                             INT16_C( 32390),  INT16_C( 22571),  INT16_C( -5410),  INT16_C( 14750),
                             INT16_C(-16092),  INT16_C( 27869),  INT16_C( 28085),  INT16_C( 15025),
                             INT16_C(-11732),  INT16_C( 28458),  INT16_C( -3526),  INT16_C(-21900)),
      simde_mm256_set_epi16( INT16_C( 11738),  INT16_C(-30358),  INT16_C(-12411),  INT16_C( -2979),
                             INT16_C(-27990),  INT16_C( 25385),  INT16_C( -5410),  INT16_C( 13011),
                             INT16_C(-30216),  INT16_C(-15754),  INT16_C( 28085),  INT16_C(   142),
                             INT16_C(-11865),  INT16_C( 28458),  INT16_C(  2605),  INT16_C( 31867)),
      simde_mm256_set_epi16( ~INT16_C(    0),  ~INT16_C(    0),  INT16_C(     0),  ~INT16_C(    0),
                             INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),
                             INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),
                             INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),  INT16_C(     0)) },
    { simde_mm256_set_epi16( INT16_C(-15649),  INT16_C( 19434),  INT16_C( 30307),  INT16_C( -2580),
                             INT16_C(-17669),  INT16_C(-12636),  INT16_C(-26128),  INT16_C(  1047),
                             INT16_C( 31189),  INT16_C( -4800),  INT16_C(-21775),  INT16_C(  9584),
                             INT16_C(-10037),  INT16_C(-12969),  INT16_C( -2203),  INT16_C(  7107)),
      simde_mm256_set_epi16( INT16_C(-14042),  INT16_C( 19434),  INT16_C( 12761),  INT16_C( -2580),
                             INT16_C(-14129),  INT16_C(-21327),  INT16_C(  8254),  INT16_C( 17988),
                             INT16_C( 31189),  INT16_C(-12604),  INT16_C(-31847),  INT16_C(  9584),
                             INT16_C(  2606),  INT16_C(-15409),  INT16_C( -2203),  INT16_C(  7107)),
      simde_mm256_set_epi16( INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),  ~INT16_C(    0),
                             INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
                             ~INT16_C(    0),  INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),
                             INT16_C(     0),  INT16_C(     0),  ~INT16_C(    0),  ~INT16_C(    0)) },
    { simde_mm256_set_epi16( INT16_C(  2093),  INT16_C( 26175),  INT16_C(  5850),  INT16_C(-17892),
                             INT16_C(-28618),  INT16_C(-20409),  INT16_C(   201),  INT16_C(-27373),
                             INT16_C(-25181),  INT16_C( -3904),  INT16_C( 26317),  INT16_C( 18884),
                             INT16_C( 15503),  INT16_C( 21511),  INT16_C(-24060),  INT16_C( -1242)),
      simde_mm256_set_epi16( INT16_C(-27449),  INT16_C( 26175),  INT16_C( 26521),  INT16_C(-16887),
                             INT16_C(-13967),  INT16_C(-16902),  INT16_C(-25148),  INT16_C( 11331),
                             INT16_C(-22706),  INT16_C( -3904),  INT16_C( -3346),  INT16_C( 18884),
                             INT16_C(  2207),  INT16_C( 11164),  INT16_C( 22502),  INT16_C(-11226)),
      simde_mm256_set_epi16( INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),  INT16_C(     0),
                             INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
                             INT16_C(     0),  ~INT16_C(    0),  INT16_C(     0),  ~INT16_C(    0),
                             INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cmpeq_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cmpeq_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32( INT32_C( -566104414),  INT32_C(  345460710),  INT32_C(  131867003),  INT32_C( 1942789694),
                             INT32_C( 1084566886),  INT32_C(  361879538),  INT32_C( 1536563030),  INT32_C(-1809647830)),
      simde_mm256_set_epi32( INT32_C( 1749780765),  INT32_C( -245413822),  INT32_C(  131867003),  INT32_C( -777848334),
                             INT32_C(  525276311),  INT32_C(  361879538),  INT32_C(  378618724),  INT32_C(-1809647830)),
      simde_mm256_set_epi32( INT32_C(          0),  INT32_C(          0),  ~INT32_C(         0),  INT32_C(          0),
                             INT32_C(          0),  ~INT32_C(         0),  INT32_C(          0),  ~INT32_C(         0)) },
    { simde_mm256_set_epi32( INT32_C( 1746696722),  INT32_C(  940817566),  INT32_C( -345075038),  INT32_C( 1008650721),
                             INT32_C( 1982631981),  INT32_C( 1481863730),  INT32_C( 1367059109),  INT32_C( -627319339)),
      simde_mm256_set_epi32( INT32_C( 1746696722),  INT32_C(  940817566),  INT32_C(-2066241582),  INT32_C( 1008650721),
                             INT32_C(  174011254),  INT32_C(-1995628897),  INT32_C( -710324691),  INT32_C( -119123371)),
      simde_mm256_set_epi32( ~INT32_C(         0),  ~INT32_C(         0),  INT32_C(          0),  ~INT32_C(         0),
                             INT32_C(          0),  INT32_C(          0),  INT32_C(          0),  INT32_C(          0)) },
    { simde_mm256_set_epi32( INT32_C( 1029753432),  INT32_C(-1928436508),  INT32_C( 1320243314),  INT32_C(  498796161),
                             INT32_C( 1948463592),  INT32_C(   75866601),  INT32_C(  742698231),  INT32_C(-1435158663)),
      simde_mm256_set_epi32( INT32_C( 1029753432),  INT32_C(-1928436508),  INT32_C(-1164282726),  INT32_C(  267783533),
                             INT32_C( -156415231),  INT32_C(-1409290651),  INT32_C(  -98856138),  INT32_C( 2067678979)),
      simde_mm256_set_epi32( ~INT32_C(         0),  ~INT32_C(         0),  INT32_C(          0),  INT32_C(          0),
                             INT32_C(          0),  INT32_C(          0),  INT32_C(          0),  INT32_C(          0)) },
    { simde_mm256_set_epi32( INT32_C( -346859314),  INT32_C( 1692797387),  INT32_C(   66311928),  INT32_C(  460824773),
                             INT32_C( -548661058),  INT32_C( -254589283),  INT32_C(  833835845),  INT32_C(-1932331579)),
      simde_mm256_set_epi32( INT32_C( -346859314),  INT32_C(   60539810),  INT32_C( 1011930823),  INT32_C(  460824773),
                             INT32_C(  190841895),  INT32_C( -254589283),  INT32_C(  316499678),  INT32_C(-1932331579)),
      simde_mm256_set_epi32( ~INT32_C(         0),  INT32_C(          0),  INT32_C(          0),  ~INT32_C(         0),
                             INT32_C(          0),  ~INT32_C(         0),  INT32_C(          0),  ~INT32_C(         0)) },
    { simde_mm256_set_epi32( INT32_C( -959873297),  INT32_C(-1563175385),  INT32_C(-1346900809),  INT32_C( 1327082188),
                             INT32_C( 1782214973),  INT32_C(-1174575718),  INT32_C( -220704084),  INT32_C(  825113567)),
      simde_mm256_set_epi32( INT32_C( -959873297),  INT32_C(  735229659),  INT32_C( -660381298),  INT32_C(-1091295976),
                             INT32_C( 1782214973),  INT32_C(-1174575718),  INT32_C( 2094872318),  INT32_C( 1028851204)),
      simde_mm256_set_epi32( ~INT32_C(         0),  INT32_C(          0),  INT32_C(          0),  INT32_C(          0),
                             ~INT32_C(         0),  ~INT32_C(         0),  INT32_C(          0),  INT32_C(          0)) },
    { simde_mm256_set_epi32( INT32_C( 1661232006),  INT32_C( 1627888390),  INT32_C(  757018179),  INT32_C( -750573130),
                             INT32_C(-1367421353),  INT32_C(  198410180),  INT32_C(-1240440841),  INT32_C(-1417499113)),
      simde_mm256_set_epi32( INT32_C( -466269763),  INT32_C(   49857616),  INT32_C(  757018179),  INT32_C( -750573130),
                             INT32_C(-1367421353),  INT32_C(-1831110733),  INT32_C(-1233216938),  INT32_C(   84780453)),
      simde_mm256_set_epi32( INT32_C(          0),  INT32_C(          0),  ~INT32_C(         0),  ~INT32_C(         0),
                             ~INT32_C(         0),  INT32_C(          0),  INT32_C(          0),  INT32_C(          0)) },
    { simde_mm256_set_epi32( INT32_C( -149159680),  INT32_C(  968409930),  INT32_C(-1757785028),  INT32_C(  354816025),
                             INT32_C(   46210141),  INT32_C( -726479125),  INT32_C(  790520082),  INT32_C( -599212420)),
      simde_mm256_set_epi32( INT32_C( -149159680),  INT32_C( 1775235902),  INT32_C(-1757785028),  INT32_C( -783938470),
                             INT32_C(   46210141),  INT32_C( -726479125),  INT32_C( -272289003),  INT32_C( -599212420)),
      simde_mm256_set_epi32( ~INT32_C(         0),  INT32_C(          0),  ~INT32_C(         0),  INT32_C(          0),
                             ~INT32_C(         0),  ~INT32_C(         0),  INT32_C(          0),  ~INT32_C(         0)) },
    { simde_mm256_set_epi32( INT32_C(  431380596),  INT32_C( -800905622),  INT32_C(-1090716510),  INT32_C(  598832812),
                             INT32_C(  183783146),  INT32_C( 1473748330),  INT32_C( 1405235121),  INT32_C( 1193787762)),
      simde_mm256_set_epi32( INT32_C( 1093676520),  INT32_C( -410887875),  INT32_C( 1548812312),  INT32_C( -913535121),
                             INT32_C(  183783146),  INT32_C(-1507745238),  INT32_C( 1405235121),  INT32_C( 1193787762)),
      simde_mm256_set_epi32( INT32_C(          0),  INT32_C(          0),  INT32_C(          0),  INT32_C(          0),
                             ~INT32_C(         0),  INT32_C(          0),  ~INT32_C(         0),  ~INT32_C(         0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cmpeq_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cmpeq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi64x( INT64_C( 5666986029204224795),  INT64_C( 4139191150084672711),
                              INT64_C(-2995542033801296871),  INT64_C( 2916699395471658679)),
      simde_mm256_set_epi64x( INT64_C( 5666986029204224795),  INT64_C(-4286656252876457389),
                              INT64_C( 1630571993074201933),  INT64_C(-4091102519963379575)),
      simde_mm256_set_epi64x( INT64_C(                  -1),  INT64_C(                   0),
                              INT64_C(                   0),  INT64_C(                   0)) },
    { simde_mm256_set_epi64x( INT64_C( 3487161978562406497),  INT64_C( 3319820858778075593),
                              INT64_C(-2061262812497087366),  INT64_C( 3422398331177828808)),
      simde_mm256_set_epi64x( INT64_C( 3487161978562406497),  INT64_C( 3319820858778075593),
                              INT64_C( 4615290220825808616),  INT64_C(-1811913409974062507)),
      simde_mm256_set_epi64x( INT64_C(                  -1),  INT64_C(                  -1),
                              INT64_C(                   0),  INT64_C(                   0)) },
    { simde_mm256_set_epi64x( INT64_C( 6876132350949566240),  INT64_C(  286650737897020179),
                              INT64_C( 8647803947797074216),  INT64_C( 8045239586356137615)),
      simde_mm256_set_epi64x( INT64_C(-2007896190058863769),  INT64_C( 1398080065885028992),
                              INT64_C( 8575195519759543000),  INT64_C( 3842062270653749921)),
      simde_mm256_set_epi64x( INT64_C(                   0),  INT64_C(                   0),
                              INT64_C(                   0),  INT64_C(                   0)) },
    { simde_mm256_set_epi64x( INT64_C( -813114899828683326),  INT64_C( 8993356532590363052),
                              INT64_C( 1099980308418045920),  INT64_C( -959713571456510913)),
      simde_mm256_set_epi64x( INT64_C( -813114899828683326),  INT64_C( 8993356532590363052),
                              INT64_C( 2705033811286181232),  INT64_C(-9000650858577263758)),
      simde_mm256_set_epi64x( INT64_C(                  -1),  INT64_C(                  -1),
                              INT64_C(                   0),  INT64_C(                   0)) },
    { simde_mm256_set_epi64x( INT64_C(-2995469803746067901),  INT64_C( 8439110829491825822),
                              INT64_C(  622618816590200788),  INT64_C(-4486881289174213355)),
      simde_mm256_set_epi64x( INT64_C(-2995469803746067901),  INT64_C(-3762957269640528607),
                              INT64_C(  622618816590200788),  INT64_C(-3609138253339109026)),
      simde_mm256_set_epi64x( ~INT64_C(                  0),  INT64_C(                   0),
                              ~INT64_C(                  0),  INT64_C(                   0)) },
    { simde_mm256_set_epi64x( INT64_C(-1016066772373646945),  INT64_C( 3813602464259710788),
                              INT64_C( 7563424360708952600),  INT64_C(-5726115040178877821)),
      simde_mm256_set_epi64x( INT64_C(-1016066772373646945),  INT64_C( 3813602464259710788),
                              INT64_C(-7344230325683392237),  INT64_C(-5726115040178877821)),
      simde_mm256_set_epi64x( ~INT64_C(                  0),  ~INT64_C(                  0),
                              INT64_C(                   0),  ~INT64_C(                  0)) },
    { simde_mm256_set_epi64x( INT64_C(-5638985149235693304),  INT64_C( 3850441284517009347),
                              INT64_C(-7932931861129304736),  INT64_C(-7323455422359325640)),
      simde_mm256_set_epi64x( INT64_C(-6947747787688303029),  INT64_C( 7867789037175719368),
                              INT64_C(-7932931861129304736),  INT64_C(  356641616463870387)),
      simde_mm256_set_epi64x( INT64_C(                   0),  INT64_C(                   0),
                              ~INT64_C(                  0),  INT64_C(                   0)) },
    { simde_mm256_set_epi64x( INT64_C(-3620459218438416224),  INT64_C(-7960741818410807131),
                              INT64_C( 2210317486113607969),  INT64_C( 2056660070405601362)),
      simde_mm256_set_epi64x( INT64_C(  186345282622433582),  INT64_C(-7960741818410807131),
                              INT64_C( 2210317486113607969),  INT64_C( 8434183884659739058)),
      simde_mm256_set_epi64x( INT64_C(                   0),  ~INT64_C(                  0),
                              ~INT64_C(                  0),  INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cmpeq_epi64(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cmpgt_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
   { simde_mm256_set_epi8(INT8_C(  97), INT8_C(  38), INT8_C(  50), INT8_C( -69),
                           INT8_C(-120), INT8_C( 113), INT8_C( -33), INT8_C( -11),
                           INT8_C(  68), INT8_C(  95), INT8_C(  30), INT8_C(  12),
                           INT8_C( -73), INT8_C( -78), INT8_C( -31), INT8_C(  -6),
                           INT8_C( -85), INT8_C(   8), INT8_C( -57), INT8_C(  86),
                           INT8_C(-111), INT8_C(  96), INT8_C(  27), INT8_C( -37),
                           INT8_C(  36), INT8_C(-109), INT8_C( -29), INT8_C(  11),
                           INT8_C( 121), INT8_C(  68), INT8_C(  -8), INT8_C(-117)),
      simde_mm256_set_epi8(INT8_C( -11), INT8_C( -47), INT8_C(  44), INT8_C(  92),
                           INT8_C(-103), INT8_C( 119), INT8_C(  74), INT8_C( -33),
                           INT8_C(  46), INT8_C( -92), INT8_C( -68), INT8_C(  34),
                           INT8_C(  87), INT8_C( -27), INT8_C( -38), INT8_C(   8),
                           INT8_C( -43), INT8_C(  93), INT8_C(   8), INT8_C(-116),
                           INT8_C(-110), INT8_C(  42), INT8_C( -23), INT8_C(  50),
                           INT8_C(  67), INT8_C( -92), INT8_C( -32), INT8_C(  41),
                           INT8_C( -11), INT8_C( 105), INT8_C( -20), INT8_C(  57)),
      simde_mm256_set_epi8(INT8_C(  -1), INT8_C(  -1), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(  -1), INT8_C(   0), INT8_C(  -1), INT8_C(   0)) },
    { simde_mm256_set_epi8(INT8_C( -79), INT8_C( 114), INT8_C(  20), INT8_C(   6),
                           INT8_C( -80), INT8_C(  63), INT8_C( 115), INT8_C( -39),
                           INT8_C( 103), INT8_C(-111), INT8_C(  -9), INT8_C(  41),
                           INT8_C(-111), INT8_C(  96), INT8_C(-117), INT8_C(  -5),
                           INT8_C( -32), INT8_C( 112), INT8_C(  94), INT8_C( -30),
                           INT8_C(   1), INT8_C(  15), INT8_C(  71), INT8_C( 106),
                           INT8_C(  89), INT8_C(  14), INT8_C( -92), INT8_C(  -5),
                           INT8_C(  34), INT8_C(  41), INT8_C(   6), INT8_C(  46)),
      simde_mm256_set_epi8(INT8_C(  69), INT8_C(  68), INT8_C( -50), INT8_C( -53),
                           INT8_C(  53), INT8_C(-115), INT8_C(  -3), INT8_C( -63),
                           INT8_C(  44), INT8_C( -26), INT8_C( -60), INT8_C(  77),
                           INT8_C( -39), INT8_C(  69), INT8_C(  -4), INT8_C(  67),
                           INT8_C(   2), INT8_C( 127), INT8_C(-108), INT8_C( -32),
                           INT8_C( -90), INT8_C( -18), INT8_C(  30), INT8_C( 126),
                           INT8_C(  71), INT8_C(  87), INT8_C(  81), INT8_C(   8),
                           INT8_C(  97), INT8_C(  48), INT8_C( 122), INT8_C(  19)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(  -1), INT8_C(   0),
                           INT8_C(  -1), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  -1)) },
    { simde_mm256_set_epi8(INT8_C(   4), INT8_C( -93), INT8_C(  78), INT8_C( -99),
                           INT8_C( -33), INT8_C(  93), INT8_C(-119), INT8_C( -26),
                           INT8_C(-115), INT8_C( -63), INT8_C( -15), INT8_C( -15),
                           INT8_C(  51), INT8_C(  56), INT8_C(-113), INT8_C( -56),
                           INT8_C(  88), INT8_C(  97), INT8_C( 124), INT8_C( 124),
                           INT8_C( -61), INT8_C( 109), INT8_C(  81), INT8_C(  37),
                           INT8_C(  67), INT8_C( -23), INT8_C(  20), INT8_C( 101),
                           INT8_C(-111), INT8_C(  20), INT8_C( -17), INT8_C(  88)),
      simde_mm256_set_epi8(INT8_C( -45), INT8_C( -79), INT8_C(  95), INT8_C(  42),
                           INT8_C(  -8), INT8_C(  12), INT8_C(  84), INT8_C( -28),
                           INT8_C(  -9), INT8_C(  73), INT8_C( 108), INT8_C(  19),
                           INT8_C(  -1), INT8_C( -28), INT8_C(  40), INT8_C( -89),
                           INT8_C(  87), INT8_C(-118), INT8_C(  52), INT8_C( -73),
                           INT8_C(  83), INT8_C( -87), INT8_C( -94), INT8_C( -66),
                           INT8_C(  22), INT8_C( -68), INT8_C(-100), INT8_C( -90),
                           INT8_C(  59), INT8_C(-103), INT8_C( -68), INT8_C(  90)),
      simde_mm256_set_epi8(INT8_C(  -1), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(   0)) },
    { simde_mm256_set_epi8(INT8_C(  78), INT8_C( -98), INT8_C(-121), INT8_C(  92),
                           INT8_C(  -8), INT8_C(  96), INT8_C(-118), INT8_C( -55),
                           INT8_C( -74), INT8_C( 105), INT8_C( -18), INT8_C(  33),
                           INT8_C(  85), INT8_C( -52), INT8_C(  68), INT8_C(   9),
                           INT8_C( -37), INT8_C(  70), INT8_C(  98), INT8_C( -80),
                           INT8_C( -54), INT8_C(-128), INT8_C( -94), INT8_C(-119),
                           INT8_C(-113), INT8_C(-115), INT8_C(  71), INT8_C( -46),
                           INT8_C(  19), INT8_C( 125), INT8_C(  40), INT8_C(   3)),
      simde_mm256_set_epi8(INT8_C( -28), INT8_C(-110), INT8_C(-103), INT8_C(   3),
                           INT8_C(-127), INT8_C(  24), INT8_C( 119), INT8_C(  81),
                           INT8_C(  -7), INT8_C( -15), INT8_C( 106), INT8_C(   4),
                           INT8_C( -43), INT8_C(  98), INT8_C( -93), INT8_C(  36),
                           INT8_C(  -1), INT8_C(  44), INT8_C(  82), INT8_C(  14),
                           INT8_C(   0), INT8_C( 125), INT8_C(  43), INT8_C( -25),
                           INT8_C(  68), INT8_C(  74), INT8_C(   8), INT8_C(  53),
                           INT8_C(  -2), INT8_C(  27), INT8_C( -33), INT8_C(  -9)),
      simde_mm256_set_epi8(INT8_C(  -1), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1)) },
    { simde_mm256_set_epi8(INT8_C(  95), INT8_C( -80), INT8_C( -95), INT8_C(  93),
                           INT8_C(   6), INT8_C(-125), INT8_C( -26), INT8_C( 102),
                           INT8_C(  60), INT8_C( -54), INT8_C( 123), INT8_C( -73),
                           INT8_C( -99), INT8_C(  28), INT8_C(  26), INT8_C(-127),
                           INT8_C(  12), INT8_C( -51), INT8_C(  90), INT8_C( -32),
                           INT8_C( -10), INT8_C(  41), INT8_C( -30), INT8_C( 107),
                           INT8_C(  23), INT8_C( 114), INT8_C( 116), INT8_C( -31),
                           INT8_C( -33), INT8_C( -82), INT8_C(  21), INT8_C(  83)),
      simde_mm256_set_epi8(INT8_C(  49), INT8_C( -50), INT8_C( -70), INT8_C( -68),
                           INT8_C(  71), INT8_C( -80), INT8_C( -71), INT8_C(  39),
                           INT8_C(  88), INT8_C(-120), INT8_C(  70), INT8_C(-124),
                           INT8_C(  40), INT8_C(-103), INT8_C(  95), INT8_C(-100),
                           INT8_C(  -5), INT8_C(  64), INT8_C( -11), INT8_C(  -4),
                           INT8_C( -28), INT8_C( -66), INT8_C(  41), INT8_C(  31),
                           INT8_C(  82), INT8_C(  26), INT8_C( -64), INT8_C(  57),
                           INT8_C(  27), INT8_C(  15), INT8_C(  57), INT8_C(-125)),
      simde_mm256_set_epi8(INT8_C(  -1), INT8_C(   0), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(   0), INT8_C(  -1), INT8_C(   0), INT8_C(   0),
                           INT8_C(  -1), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  -1)) },
    { simde_mm256_set_epi8(INT8_C(  57), INT8_C(-121), INT8_C(  -6), INT8_C( -74),
                           INT8_C(  22), INT8_C( -63), INT8_C(  17), INT8_C( 101),
                           INT8_C(  43), INT8_C( -53), INT8_C(-112), INT8_C( -25),
                           INT8_C(  85), INT8_C(  55), INT8_C(  77), INT8_C( -58),
                           INT8_C(  73), INT8_C(  35), INT8_C( -70), INT8_C(  30),
                           INT8_C( -78), INT8_C(   4), INT8_C( -97), INT8_C(   5),
                           INT8_C( -36), INT8_C(  28), INT8_C(  95), INT8_C(  75),
                           INT8_C( -88), INT8_C(  11), INT8_C(  45), INT8_C( -19)),
      simde_mm256_set_epi8(INT8_C(  85), INT8_C( 121), INT8_C(   8), INT8_C( 127),
                           INT8_C(-122), INT8_C( -88), INT8_C( 107), INT8_C(  76),
                           INT8_C(-128), INT8_C(-125), INT8_C(  90), INT8_C(  68),
                           INT8_C( -36), INT8_C(  75), INT8_C( -58), INT8_C(  50),
                           INT8_C(  69), INT8_C(-119), INT8_C(-102), INT8_C( -69),
                           INT8_C(  16), INT8_C(  98), INT8_C(  34), INT8_C(-127),
                           INT8_C(  10), INT8_C(  69), INT8_C( 121), INT8_C(-126),
                           INT8_C(  88), INT8_C( -14), INT8_C(-110), INT8_C( -45)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(   0), INT8_C(   0),
                           INT8_C(  -1), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1)) },
    { simde_mm256_set_epi8(INT8_C( -41), INT8_C( -49), INT8_C(  85), INT8_C( 123),
                           INT8_C( -68), INT8_C( -82), INT8_C( -28), INT8_C( -31),
                           INT8_C(  16), INT8_C(  27), INT8_C(   9), INT8_C(-102),
                           INT8_C(  78), INT8_C(  37), INT8_C( -66), INT8_C(  41),
                           INT8_C( -11), INT8_C(-117), INT8_C(  83), INT8_C(  41),
                           INT8_C(-105), INT8_C(  71), INT8_C( -12), INT8_C( 118),
                           INT8_C( -49), INT8_C( -73), INT8_C( -68), INT8_C( -18),
                           INT8_C( 118), INT8_C( 110), INT8_C(  85), INT8_C(  87)),
      simde_mm256_set_epi8(INT8_C( 117), INT8_C( -99), INT8_C( 113), INT8_C(  26),
                           INT8_C( -41), INT8_C(  64), INT8_C(  93), INT8_C(  98),
                           INT8_C( 100), INT8_C(  92), INT8_C( -14), INT8_C(  86),
                           INT8_C(  83), INT8_C(  11), INT8_C( -24), INT8_C(  12),
                           INT8_C(  30), INT8_C(  43), INT8_C( 100), INT8_C(  94),
                           INT8_C(  19), INT8_C(  46), INT8_C( -17), INT8_C(   0),
                           INT8_C(   9), INT8_C( 104), INT8_C(  26), INT8_C(  39),
                           INT8_C( 102), INT8_C( -79), INT8_C( 114), INT8_C(   9)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(   0), INT8_C(  -1)) },
    { simde_mm256_set_epi8(INT8_C( 102), INT8_C(  -3), INT8_C( -18), INT8_C(-123),
                           INT8_C( -72), INT8_C(  10), INT8_C(  86), INT8_C(  30),
                           INT8_C( -77), INT8_C(  95), INT8_C( -77), INT8_C(   0),
                           INT8_C( -69), INT8_C( -21), INT8_C(  52), INT8_C( -84),
                           INT8_C(  45), INT8_C( -12), INT8_C( -37), INT8_C(  90),
                           INT8_C(  28), INT8_C(-104), INT8_C( 106), INT8_C( 118),
                           INT8_C( -89), INT8_C(   3), INT8_C(  10), INT8_C( 107),
                           INT8_C( 119), INT8_C(  18), INT8_C( -95), INT8_C( 109)),
      simde_mm256_set_epi8(INT8_C( -86), INT8_C(  18), INT8_C( -58), INT8_C(   3),
                           INT8_C( -64), INT8_C( -78), INT8_C( -50), INT8_C( 105),
                           INT8_C(  15), INT8_C(-117), INT8_C( -59), INT8_C( 105),
                           INT8_C( -20), INT8_C( -76), INT8_C( -88), INT8_C(-122),
                           INT8_C( -16), INT8_C( -87), INT8_C( 108), INT8_C( -44),
                           INT8_C( -34), INT8_C( -65), INT8_C( 125), INT8_C( -17),
                           INT8_C(-119), INT8_C(  55), INT8_C( -39), INT8_C( -68),
                           INT8_C(-116), INT8_C(  45), INT8_C(-101), INT8_C(  61)),
      simde_mm256_set_epi8(INT8_C(  -1), INT8_C(   0), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -1), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(  -1), INT8_C(   0), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(   0), INT8_C(   0), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(   0), INT8_C(  -1), INT8_C(  -1),
                           INT8_C(  -1), INT8_C(   0), INT8_C(  -1), INT8_C(  -1)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cmpgt_epi8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cmpgt_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C( 22495), INT16_C( -4666), INT16_C( 11364), INT16_C(  3408),
                            INT16_C( -7753), INT16_C( 22355), INT16_C( 25225), INT16_C(-16816),
                            INT16_C( 30843), INT16_C( 15445), INT16_C(-14188), INT16_C( 18672),
                            INT16_C(-17933), INT16_C( 19821), INT16_C( -7211), INT16_C( 10864)),
      simde_mm256_set_epi16(INT16_C(  -752), INT16_C(-15840), INT16_C(-30268), INT16_C( 11179),
                            INT16_C( 16785), INT16_C(-16319), INT16_C(  8147), INT16_C(-29538),
                            INT16_C( 19069), INT16_C( 27101), INT16_C( 26817), INT16_C(-21010),
                            INT16_C(-18350), INT16_C(  1385), INT16_C(-28379), INT16_C(-19254)),
      simde_mm256_set_epi16(INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(     0),
                            INT16_C(     0), INT16_C(    -1), INT16_C(    -1), INT16_C(    -1),
                            INT16_C(    -1), INT16_C(     0), INT16_C(     0), INT16_C(    -1),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(    -1)) },
    { simde_mm256_set_epi16(INT16_C( 22399), INT16_C(   -10), INT16_C( 26109), INT16_C( 19317),
                            INT16_C( 26547), INT16_C( 21309), INT16_C(-21770), INT16_C(-29851),
                            INT16_C(  8254), INT16_C( -6152), INT16_C( 14575), INT16_C(-26009),
                            INT16_C( -9485), INT16_C(  5665), INT16_C( 19505), INT16_C( -7563)),
      simde_mm256_set_epi16(INT16_C( 14612), INT16_C(-31260), INT16_C(-18743), INT16_C( -4117),
                            INT16_C( 18997), INT16_C(-26816), INT16_C( 13752), INT16_C( 24083),
                            INT16_C(-17683), INT16_C(-12944), INT16_C(-20564), INT16_C( -4251),
                            INT16_C( -8899), INT16_C( -5996), INT16_C( 20472), INT16_C( -9514)),
      simde_mm256_set_epi16(INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(    -1),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(     0), INT16_C(     0),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(     0),
                            INT16_C(     0), INT16_C(    -1), INT16_C(     0), INT16_C(    -1)) },
    { simde_mm256_set_epi16(INT16_C( -1570), INT16_C(-21545), INT16_C(-26360), INT16_C( -8434),
                            INT16_C( -3275), INT16_C(-16762), INT16_C( -5746), INT16_C( 30695),
                            INT16_C(  2714), INT16_C( 10463), INT16_C(  9000), INT16_C(  9451),
                            INT16_C( 23785), INT16_C( 12389), INT16_C(-26622), INT16_C(-15647)),
      simde_mm256_set_epi16(INT16_C(  1010), INT16_C(-25985), INT16_C( 16762), INT16_C(  1632),
                            INT16_C(-18641), INT16_C( 15349), INT16_C(-16838), INT16_C(-20536),
                            INT16_C(-25875), INT16_C( 26970), INT16_C(-23524), INT16_C( 11391),
                            INT16_C(  3100), INT16_C(-31773), INT16_C( 26059), INT16_C( 11041)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(    -1), INT16_C(     0), INT16_C(     0),
                            INT16_C(    -1), INT16_C(     0), INT16_C(    -1), INT16_C(    -1),
                            INT16_C(    -1), INT16_C(     0), INT16_C(    -1), INT16_C(     0),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(     0), INT16_C(     0)) },
    { simde_mm256_set_epi16(INT16_C( 23915), INT16_C(   959), INT16_C(-22514), INT16_C( 13747),
                            INT16_C(-26649), INT16_C( 22646), INT16_C( 13432), INT16_C( 27482),
                            INT16_C( 19401), INT16_C( 30181), INT16_C( 31161), INT16_C(-17967),
                            INT16_C(-19058), INT16_C( 31935), INT16_C( 11181), INT16_C( 23239)),
      simde_mm256_set_epi16(INT16_C(-28186), INT16_C(  2084), INT16_C( 31597), INT16_C(  7566),
                            INT16_C( 21799), INT16_C(-27106), INT16_C(  4736), INT16_C( 18793),
                            INT16_C(-13965), INT16_C(-17482), INT16_C( 12657), INT16_C(-20379),
                            INT16_C( 24896), INT16_C( 28918), INT16_C( 22577), INT16_C( 29712)),
      simde_mm256_set_epi16(INT16_C(    -1), INT16_C(     0), INT16_C(     0), INT16_C(    -1),
                            INT16_C(     0), INT16_C(    -1), INT16_C(    -1), INT16_C(    -1),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(    -1),
                            INT16_C(     0), INT16_C(    -1), INT16_C(     0), INT16_C(     0)) },
    { simde_mm256_set_epi16(INT16_C( 11952), INT16_C( 18081), INT16_C(  7197), INT16_C( -9524),
                            INT16_C( 18521), INT16_C(  8735), INT16_C( 31233), INT16_C(-18919),
                            INT16_C(  8906), INT16_C( 21111), INT16_C( 27720), INT16_C(  5158),
                            INT16_C(  2673), INT16_C( 31824), INT16_C(-18067), INT16_C(  -891)),
      simde_mm256_set_epi16(INT16_C(-18602), INT16_C( 15849), INT16_C(-28095), INT16_C( -3999),
                            INT16_C( 13518), INT16_C(-13535), INT16_C(-30865), INT16_C(   675),
                            INT16_C(-26284), INT16_C( -1429), INT16_C(-19333), INT16_C(-26108),
                            INT16_C( 10384), INT16_C(-21020), INT16_C(-15076), INT16_C( 30415)),
      simde_mm256_set_epi16(INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(     0),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(     0),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(    -1),
                            INT16_C(     0), INT16_C(    -1), INT16_C(     0), INT16_C(     0)) },
    { simde_mm256_set_epi16(INT16_C(-31253), INT16_C( -5290), INT16_C(-20180), INT16_C(  8300),
                            INT16_C( 29534), INT16_C( 10169), INT16_C( 27568), INT16_C(-21872),
                            INT16_C( 20546), INT16_C(  3171), INT16_C( -5551), INT16_C( 13838),
                            INT16_C( 10100), INT16_C(-27385), INT16_C( 19172), INT16_C(-26206)),
      simde_mm256_set_epi16(INT16_C(-12916), INT16_C( 16100), INT16_C(-14860), INT16_C(  6730),
                            INT16_C(  2608), INT16_C(-17352), INT16_C( 13860), INT16_C(  6700),
                            INT16_C( -7357), INT16_C( -1733), INT16_C(  7387), INT16_C(  5378),
                            INT16_C(-19225), INT16_C( -5805), INT16_C(   154), INT16_C(-20617)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(    -1),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(     0),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(     0), INT16_C(    -1),
                            INT16_C(    -1), INT16_C(     0), INT16_C(    -1), INT16_C(     0)) },
    { simde_mm256_set_epi16(INT16_C( 13973), INT16_C( 21863), INT16_C( -4187), INT16_C( 31713),
                            INT16_C( -2161), INT16_C(-12788), INT16_C( 13548), INT16_C( 19072),
                            INT16_C(-23365), INT16_C(  2809), INT16_C( 16101), INT16_C( -9274),
                            INT16_C( 12377), INT16_C( 26381), INT16_C( 29677), INT16_C(  4006)),
      simde_mm256_set_epi16(INT16_C( -9250), INT16_C(-11965), INT16_C(-16314), INT16_C( -8035),
                            INT16_C(-12852), INT16_C( -3996), INT16_C(  8694), INT16_C( -8455),
                            INT16_C( 12571), INT16_C(-15529), INT16_C( -8320), INT16_C( 14428),
                            INT16_C( -5397), INT16_C( 17843), INT16_C(  7761), INT16_C( -1266)),
      simde_mm256_set_epi16(INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(    -1),
                            INT16_C(    -1), INT16_C(     0), INT16_C(    -1), INT16_C(    -1),
                            INT16_C(     0), INT16_C(    -1), INT16_C(    -1), INT16_C(     0),
                            INT16_C(    -1), INT16_C(    -1), INT16_C(    -1), INT16_C(    -1)) },
    { simde_mm256_set_epi16(INT16_C(  7404), INT16_C( 10935), INT16_C(-26115), INT16_C(  8618),
                            INT16_C(-27599), INT16_C( 31224), INT16_C(  4654), INT16_C(-18707),
                            INT16_C( 12785), INT16_C( 22785), INT16_C( 25148), INT16_C(-31622),
                            INT16_C( 17070), INT16_C(-22617), INT16_C(  6953), INT16_C(-14046)),
      simde_mm256_set_epi16(INT16_C(-11447), INT16_C( 18342), INT16_C(   563), INT16_C(-13432),
                            INT16_C( 21117), INT16_C( 29427), INT16_C( 31704), INT16_C( 14377),
                            INT16_C( 14770), INT16_C( -3712), INT16_C(  7229), INT16_C(-22634),
                            INT16_C( 28493), INT16_C(-28170), INT16_C(  5912), INT16_C(  7039)),
      simde_mm256_set_epi16(INT16_C(    -1), INT16_C(     0), INT16_C(     0), INT16_C(    -1),
                            INT16_C(     0), INT16_C(    -1), INT16_C(     0), INT16_C(     0),
                            INT16_C(     0), INT16_C(    -1), INT16_C(    -1), INT16_C(     0),
                            INT16_C(     0), INT16_C(    -1), INT16_C(    -1), INT16_C(     0)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cmpgt_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cmpgt_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(-1910963751), INT32_C(-1037527336), INT32_C( -581253082), INT32_C( -805157505),
                            INT32_C( 1446966287), INT32_C(   -8913681), INT32_C( -494526366), INT32_C(-1857474161)),
      simde_mm256_set_epi32(INT32_C(-1143050049), INT32_C(-1545949366), INT32_C( 1503277288), INT32_C(-1357138171),
                            INT32_C( 1058844939), INT32_C(  309480335), INT32_C( 1825640960), INT32_C(-2083253752)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(         -1), INT32_C(          0), INT32_C(         -1),
                            INT32_C(         -1), INT32_C(          0), INT32_C(          0), INT32_C(         -1)) },
    { simde_mm256_set_epi32(INT32_C(-1125895018), INT32_C( -151199733), INT32_C(-1156572232), INT32_C(-1719423594),
                            INT32_C(-1350415044), INT32_C( 1941048360), INT32_C( 1462693644), INT32_C(  157276070)),
      simde_mm256_set_epi32(INT32_C( 1685470989), INT32_C(  -23549807), INT32_C( 2101982138), INT32_C(-1846214249),
                            INT32_C(-1855916776), INT32_C( -540831398), INT32_C(  233301808), INT32_C( 1667068415)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(         -1),
                            INT32_C(         -1), INT32_C(         -1), INT32_C(         -1), INT32_C(          0)) },
    { simde_mm256_set_epi32(INT32_C(-2064202369), INT32_C( -937796718), INT32_C( 1860922821), INT32_C( -943515104),
                            INT32_C( 1932702217), INT32_C(-1909235607), INT32_C( -701927701), INT32_C( 2122537608)),
      simde_mm256_set_epi32(INT32_C( 1359427872), INT32_C(  430067405), INT32_C(-2017266204), INT32_C( -112174075),
                            INT32_C(-2036257158), INT32_C(  794990098), INT32_C( 1595368835), INT32_C(-2025748789)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(         -1), INT32_C(          0),
                            INT32_C(         -1), INT32_C(          0), INT32_C(          0), INT32_C(         -1)) },
    { simde_mm256_set_epi32(INT32_C(  101914014), INT32_C( 1252307439), INT32_C( -251229086), INT32_C( 1747655874),
                            INT32_C( 1975156439), INT32_C(  244588822), INT32_C(-2030488384), INT32_C(  -41784348)),
      simde_mm256_set_epi32(INT32_C(  126704158), INT32_C(-1817043660), INT32_C(   67229317), INT32_C( 1709840887),
                            INT32_C( -401701180), INT32_C(-1816869894), INT32_C( -356935412), INT32_C( 1029497038)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(         -1), INT32_C(          0), INT32_C(         -1),
                            INT32_C(         -1), INT32_C(         -1), INT32_C(          0), INT32_C(          0)) },
    { simde_mm256_set_epi32(INT32_C(-1821031073), INT32_C(-1888821389), INT32_C( 1302274763), INT32_C( -889069785),
                            INT32_C(  849224332), INT32_C( 1448513971), INT32_C(-1444618137), INT32_C( -857531383)),
      simde_mm256_set_epi32(INT32_C( -294182987), INT32_C(-1911426210), INT32_C( 1273887477), INT32_C( -804272895),
                            INT32_C(-1138168885), INT32_C(-1273049807), INT32_C( -920424822), INT32_C(   75748881)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(         -1), INT32_C(         -1), INT32_C(          0),
                            INT32_C(         -1), INT32_C(         -1), INT32_C(          0), INT32_C(          0)) },
    { simde_mm256_set_epi32(INT32_C( -145989095), INT32_C( 1224886052), INT32_C(-1270092347), INT32_C(-2004839127),
                            INT32_C( -805168375), INT32_C(-2053011497), INT32_C(  926069792), INT32_C( -523797780)),
      simde_mm256_set_epi32(INT32_C( 1782736183), INT32_C( 1196895214), INT32_C( 2023829967), INT32_C(  431966763),
                            INT32_C(-1279096565), INT32_C(-1568278654), INT32_C( -225755322), INT32_C(  800135833)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(         -1), INT32_C(          0), INT32_C(          0),
                            INT32_C(         -1), INT32_C(          0), INT32_C(         -1), INT32_C(          0)) },
    { simde_mm256_set_epi32(INT32_C(   45181175), INT32_C( 2067174734), INT32_C(-1046962322), INT32_C( -770870625),
                            INT32_C(  166196701), INT32_C(-1211694319), INT32_C( 1891438592), INT32_C(-1986783509)),
      simde_mm256_set_epi32(INT32_C( 1357811899), INT32_C( -666006860), INT32_C(  -78093915), INT32_C(-1784875915),
                            INT32_C(-1592390514), INT32_C( -295345562), INT32_C(-1194315003), INT32_C( -378378269)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(         -1), INT32_C(          0), INT32_C(         -1),
                            INT32_C(         -1), INT32_C(          0), INT32_C(         -1), INT32_C(          0)) },
    { simde_mm256_set_epi32(INT32_C( 2029570724), INT32_C(  593067919), INT32_C( 1378001411), INT32_C(-1490077172),
                            INT32_C( -985008502), INT32_C( -780769236), INT32_C(  983706005), INT32_C( -145048806)),
      simde_mm256_set_epi32(INT32_C( -371614648), INT32_C(  790968401), INT32_C(  604322541), INT32_C( 1275463353),
                            INT32_C(  507685025), INT32_C(-1785065497), INT32_C( -324987069), INT32_C( 1403024939)),
      simde_mm256_set_epi32(INT32_C(         -1), INT32_C(          0), INT32_C(         -1), INT32_C(          0),
                            INT32_C(          0), INT32_C(         -1), INT32_C(         -1), INT32_C(          0)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cmpgt_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cmpgt_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
      { simde_mm256_set_epi64x(INT64_C( 2118945800826688975), INT64_C(-2048024407550915063),
                             INT64_C( 5214505670652994271), INT64_C( -257856994503089701)),
      simde_mm256_set_epi64x(INT64_C( 5695011750507465952), INT64_C( -909596801509157384),
                             INT64_C( 8030198869340496548), INT64_C(  154404577117411578)),
      simde_mm256_set_epi64x(INT64_C(                   0), INT64_C(                   0),
                             INT64_C(                   0), INT64_C(                   0)) },
    { simde_mm256_set_epi64x(INT64_C( 5911350820320548021), INT64_C( 1593992970237440641),
                             INT64_C( 1816057404225443518), INT64_C(-5807847363009183486)),
      simde_mm256_set_epi64x(INT64_C( 6716528373762493386), INT64_C(-8043125522610464194),
                             INT64_C( 8780083716644576089), INT64_C( 1141738695323567528)),
      simde_mm256_set_epi64x(INT64_C(                   0), INT64_C(                  -1),
                             INT64_C(                   0), INT64_C(                   0)) },
    { simde_mm256_set_epi64x(INT64_C( 2128200156105206458), INT64_C( 4268557007045133354),
                             INT64_C(  922795411729807384), INT64_C( -370673782849908924)),
      simde_mm256_set_epi64x(INT64_C(-4606711496612992363), INT64_C( 5770563142009272389),
                             INT64_C( 5439573582391425757), INT64_C( 3818549561656095124)),
      simde_mm256_set_epi64x(INT64_C(                  -1), INT64_C(                   0),
                             INT64_C(                   0), INT64_C(                   0)) },
    { simde_mm256_set_epi64x(INT64_C(-7763872078520682010), INT64_C( 8214461135943818795),
                             INT64_C( 6913361685886042209), INT64_C( 4675504907089870645)),
      simde_mm256_set_epi64x(INT64_C( 4559571845325885325), INT64_C(-5258431881764629271),
                             INT64_C( 5559569102824249097), INT64_C( 5799667037503013333)),
      simde_mm256_set_epi64x(INT64_C(                   0), INT64_C(                  -1),
                             INT64_C(                  -1), INT64_C(                   0)) },
    { simde_mm256_set_epi64x(INT64_C( 4800525853411880003), INT64_C( -184846840160593290),
                             INT64_C(-4953203027402144401), INT64_C( 2572609663077992835)),
      simde_mm256_set_epi64x(INT64_C(-5431416973650143300), INT64_C( 5030179310017915920),
                             INT64_C(-6125011459628645754), INT64_C(-8488302794525901027)),
      simde_mm256_set_epi64x(INT64_C(                  -1), INT64_C(                   0),
                             INT64_C(                  -1), INT64_C(                  -1)) },
    { simde_mm256_set_epi64x(INT64_C(-3860235829589315850), INT64_C( 8837948345888719516),
                             INT64_C( 3875197176959192831), INT64_C(-5180468743599698974)),
      simde_mm256_set_epi64x(INT64_C(-1290331525121336992), INT64_C(-2797732422128290053),
                             INT64_C(-1679378228033738615), INT64_C(-7981001780127731934)),
      simde_mm256_set_epi64x(INT64_C(                   0), INT64_C(                  -1),
                             INT64_C(                  -1), INT64_C(                  -1)) },
    { simde_mm256_set_epi64x(INT64_C(-1174099241428997420), INT64_C( -973237140110744253),
                             INT64_C(-2478103055519574033), INT64_C(-4160968278707001293)),
      simde_mm256_set_epi64x(INT64_C(-5836788839917359935), INT64_C( 9029070361682465515),
                             INT64_C( 7441474531728878768), INT64_C(-8910660672377811998)),
      simde_mm256_set_epi64x(INT64_C(                  -1), INT64_C(                   0),
                             INT64_C(                   0), INT64_C(                  -1)) },
    { simde_mm256_set_epi64x(INT64_C(  660412182468938166), INT64_C( 2628509021710534078),
                             INT64_C(-4480617326594733579), INT64_C( 6124790654743076244)),
      simde_mm256_set_epi64x(INT64_C(-2572783283758925366), INT64_C( 8355777023791813662),
                             INT64_C( 7632362282544129149), INT64_C(-5998036421176707691)),
      simde_mm256_set_epi64x(INT64_C(                  -1), INT64_C(                   0),
                             INT64_C(                   0), INT64_C(                  -1)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cmpgt_epi64(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm_broadcastb_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( -20), INT8_C( 103), INT8_C( -20), INT8_C( 116),
                        INT8_C(  -9), INT8_C(  73), INT8_C(  44), INT8_C(  79),
                        INT8_C( -20), INT8_C( -81), INT8_C(-114), INT8_C( -81),
                        INT8_C(  10), INT8_C(  63), INT8_C( -41), INT8_C(-117)),
      simde_mm_set_epi8(INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                        INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                        INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                        INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117)) },
    { simde_mm_set_epi8(INT8_C( -22), INT8_C(-124), INT8_C( -97), INT8_C(  -1),
                        INT8_C(  -4), INT8_C( -87), INT8_C( -49), INT8_C(-124),
                        INT8_C( -37), INT8_C( -17), INT8_C( -57), INT8_C(   0),
                        INT8_C(-121), INT8_C(  57), INT8_C(  49), INT8_C(-112)),
      simde_mm_set_epi8(INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                        INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                        INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                        INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112)) },
    { simde_mm_set_epi8(INT8_C( -37), INT8_C( -43), INT8_C( -30), INT8_C(  45),
                        INT8_C(  86), INT8_C(-100), INT8_C( -53), INT8_C(  -4),
                        INT8_C(  54), INT8_C(-107), INT8_C(  40), INT8_C( -68),
                        INT8_C(  49), INT8_C(  59), INT8_C( -37), INT8_C(  60)),
      simde_mm_set_epi8(INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                        INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                        INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                        INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60)) },
    { simde_mm_set_epi8(INT8_C(  35), INT8_C( -98), INT8_C( -73), INT8_C(  22),
                        INT8_C(-127), INT8_C(  78), INT8_C( 125), INT8_C(  84),
                        INT8_C(-115), INT8_C(-124), INT8_C(-103), INT8_C(  59),
                        INT8_C(  15), INT8_C( -58), INT8_C(  81), INT8_C(   4)),
      simde_mm_set_epi8(INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                        INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                        INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                        INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4)) },
    { simde_mm_set_epi8(INT8_C( -56), INT8_C(  88), INT8_C(   1), INT8_C( -59),
                        INT8_C( -14), INT8_C( -27), INT8_C( -63), INT8_C( -89),
                        INT8_C( -31), INT8_C( -81), INT8_C( -92), INT8_C( 115),
                        INT8_C( 106), INT8_C( -49), INT8_C( -19), INT8_C(  92)),
      simde_mm_set_epi8(INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                        INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                        INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                        INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92)) },
    { simde_mm_set_epi8(INT8_C(   7), INT8_C(  47), INT8_C(  55), INT8_C(  77),
                        INT8_C(  16), INT8_C( -71), INT8_C(-122), INT8_C( -81),
                        INT8_C(-120), INT8_C( -64), INT8_C( -77), INT8_C(  13),
                        INT8_C(  80), INT8_C(-114), INT8_C(-121), INT8_C(  92)),
      simde_mm_set_epi8(INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                        INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                        INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                        INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92)) },
    { simde_mm_set_epi8(INT8_C(  14), INT8_C( -85), INT8_C(  -8), INT8_C( -80),
                        INT8_C( -58), INT8_C(  84), INT8_C(  52), INT8_C(-106),
                        INT8_C( -46), INT8_C( 118), INT8_C( -96), INT8_C(  88),
                        INT8_C(  35), INT8_C(  50), INT8_C(-112), INT8_C(-112)),
      simde_mm_set_epi8(INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                        INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                        INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                        INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112)) },
    { simde_mm_set_epi8(INT8_C(  17), INT8_C(  88), INT8_C( -82), INT8_C(-109),
                        INT8_C(  56), INT8_C(  18), INT8_C(  73), INT8_C( 115),
                        INT8_C( -99), INT8_C( -71), INT8_C( -92), INT8_C( 121),
                        INT8_C(  93), INT8_C(  51), INT8_C( -65), INT8_C(  55)),
      simde_mm_set_epi8(INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                        INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                        INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                        INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm_broadcastb_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_broadcastb_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( -20), INT8_C( 103), INT8_C( -20), INT8_C( 116),
                        INT8_C(  -9), INT8_C(  73), INT8_C(  44), INT8_C(  79),
                        INT8_C( -20), INT8_C( -81), INT8_C(-114), INT8_C( -81),
                        INT8_C(  10), INT8_C(  63), INT8_C( -41), INT8_C(-117)),
      simde_mm256_set_epi8(INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117)) },
    { simde_mm_set_epi8(INT8_C( -22), INT8_C(-124), INT8_C( -97), INT8_C(  -1),
                        INT8_C(  -4), INT8_C( -87), INT8_C( -49), INT8_C(-124),
                        INT8_C( -37), INT8_C( -17), INT8_C( -57), INT8_C(   0),
                        INT8_C(-121), INT8_C(  57), INT8_C(  49), INT8_C(-112)),
      simde_mm256_set_epi8(INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112)) },
    { simde_mm_set_epi8(INT8_C( -37), INT8_C( -43), INT8_C( -30), INT8_C(  45),
                        INT8_C(  86), INT8_C(-100), INT8_C( -53), INT8_C(  -4),
                        INT8_C(  54), INT8_C(-107), INT8_C(  40), INT8_C( -68),
                        INT8_C(  49), INT8_C(  59), INT8_C( -37), INT8_C(  60)),
      simde_mm256_set_epi8(INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                           INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                           INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                           INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                           INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                           INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                           INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60),
                           INT8_C(  60), INT8_C(  60), INT8_C(  60), INT8_C(  60)) },
    { simde_mm_set_epi8(INT8_C(  35), INT8_C( -98), INT8_C( -73), INT8_C(  22),
                        INT8_C(-127), INT8_C(  78), INT8_C( 125), INT8_C(  84),
                        INT8_C(-115), INT8_C(-124), INT8_C(-103), INT8_C(  59),
                        INT8_C(  15), INT8_C( -58), INT8_C(  81), INT8_C(   4)),
      simde_mm256_set_epi8(INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                           INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                           INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                           INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                           INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                           INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                           INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4),
                           INT8_C(   4), INT8_C(   4), INT8_C(   4), INT8_C(   4)) },
    { simde_mm_set_epi8(INT8_C( -56), INT8_C(  88), INT8_C(   1), INT8_C( -59),
                        INT8_C( -14), INT8_C( -27), INT8_C( -63), INT8_C( -89),
                        INT8_C( -31), INT8_C( -81), INT8_C( -92), INT8_C( 115),
                        INT8_C( 106), INT8_C( -49), INT8_C( -19), INT8_C(  92)),
      simde_mm256_set_epi8(INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92)) },
    { simde_mm_set_epi8(INT8_C(   7), INT8_C(  47), INT8_C(  55), INT8_C(  77),
                        INT8_C(  16), INT8_C( -71), INT8_C(-122), INT8_C( -81),
                        INT8_C(-120), INT8_C( -64), INT8_C( -77), INT8_C(  13),
                        INT8_C(  80), INT8_C(-114), INT8_C(-121), INT8_C(  92)),
      simde_mm256_set_epi8(INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92),
                           INT8_C(  92), INT8_C(  92), INT8_C(  92), INT8_C(  92)) },
    { simde_mm_set_epi8(INT8_C(  14), INT8_C( -85), INT8_C(  -8), INT8_C( -80),
                        INT8_C( -58), INT8_C(  84), INT8_C(  52), INT8_C(-106),
                        INT8_C( -46), INT8_C( 118), INT8_C( -96), INT8_C(  88),
                        INT8_C(  35), INT8_C(  50), INT8_C(-112), INT8_C(-112)),
      simde_mm256_set_epi8(INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112),
                           INT8_C(-112), INT8_C(-112), INT8_C(-112), INT8_C(-112)) },
    { simde_mm_set_epi8(INT8_C(  17), INT8_C(  88), INT8_C( -82), INT8_C(-109),
                        INT8_C(  56), INT8_C(  18), INT8_C(  73), INT8_C( 115),
                        INT8_C( -99), INT8_C( -71), INT8_C( -92), INT8_C( 121),
                        INT8_C(  93), INT8_C(  51), INT8_C( -65), INT8_C(  55)),
      simde_mm256_set_epi8(INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_broadcastb_epi8(test_vec[i].a);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm_broadcastw_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi16(INT16_C(-14724), INT16_C(-11263), INT16_C(-14102), INT16_C(  6431),
                         INT16_C( 11838), INT16_C( -2695), INT16_C(-11290), INT16_C( 22147)),
      simde_mm_set_epi16(INT16_C( 22147), INT16_C( 22147), INT16_C( 22147), INT16_C( 22147),
                         INT16_C( 22147), INT16_C( 22147), INT16_C( 22147), INT16_C( 22147)) },
    { simde_mm_set_epi16(INT16_C(-16582), INT16_C(-25927), INT16_C(-23837), INT16_C( 29287),
                         INT16_C(-19772), INT16_C( 13452), INT16_C( -3861), INT16_C( -4316)),
      simde_mm_set_epi16(INT16_C( -4316), INT16_C( -4316), INT16_C( -4316), INT16_C( -4316),
                         INT16_C( -4316), INT16_C( -4316), INT16_C( -4316), INT16_C( -4316)) },
    { simde_mm_set_epi16(INT16_C(-22817), INT16_C(-13842), INT16_C(-28521), INT16_C( 14250),
                         INT16_C( 16773), INT16_C(-17571), INT16_C( 16721), INT16_C(-24996)),
      simde_mm_set_epi16(INT16_C(-24996), INT16_C(-24996), INT16_C(-24996), INT16_C(-24996),
                         INT16_C(-24996), INT16_C(-24996), INT16_C(-24996), INT16_C(-24996)) },
    { simde_mm_set_epi16(INT16_C( -7252), INT16_C( -8727), INT16_C(  7341), INT16_C( 25148),
                         INT16_C(-18544), INT16_C( 21940), INT16_C(  7393), INT16_C( -5844)),
      simde_mm_set_epi16(INT16_C( -5844), INT16_C( -5844), INT16_C( -5844), INT16_C( -5844),
                         INT16_C( -5844), INT16_C( -5844), INT16_C( -5844), INT16_C( -5844)) },
    { simde_mm_set_epi16(INT16_C(-20117), INT16_C(-17614), INT16_C( 27322), INT16_C(-16415),
                         INT16_C( 21044), INT16_C( 32548), INT16_C( 27672), INT16_C( 26452)),
      simde_mm_set_epi16(INT16_C( 26452), INT16_C( 26452), INT16_C( 26452), INT16_C( 26452),
                         INT16_C( 26452), INT16_C( 26452), INT16_C( 26452), INT16_C( 26452)) },
    { simde_mm_set_epi16(INT16_C( 15651), INT16_C( -9541), INT16_C( -1245), INT16_C( 19639),
                         INT16_C( -7877), INT16_C(-31496), INT16_C( -9293), INT16_C( 24923)),
      simde_mm_set_epi16(INT16_C( 24923), INT16_C( 24923), INT16_C( 24923), INT16_C( 24923),
                         INT16_C( 24923), INT16_C( 24923), INT16_C( 24923), INT16_C( 24923)) },
    { simde_mm_set_epi16(INT16_C( -1558), INT16_C( 24882), INT16_C(-27020), INT16_C( -1908),
                         INT16_C(  5712), INT16_C( 21207), INT16_C(  4307), INT16_C(  9371)),
      simde_mm_set_epi16(INT16_C(  9371), INT16_C(  9371), INT16_C(  9371), INT16_C(  9371),
                         INT16_C(  9371), INT16_C(  9371), INT16_C(  9371), INT16_C(  9371)) },
    { simde_mm_set_epi16(INT16_C( 28347), INT16_C(-26396), INT16_C(-17550), INT16_C(-16266),
                         INT16_C(-10219), INT16_C(  6683), INT16_C(-28102), INT16_C( 11110)),
      simde_mm_set_epi16(INT16_C( 11110), INT16_C( 11110), INT16_C( 11110), INT16_C( 11110),
                         INT16_C( 11110), INT16_C( 11110), INT16_C( 11110), INT16_C( 11110)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm_broadcastw_epi16(test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_broadcastw_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi16(INT16_C(-14724), INT16_C(-11263), INT16_C(-14102), INT16_C(  6431),
                         INT16_C( 11838), INT16_C( -2695), INT16_C(-11290), INT16_C( 22147)),
      simde_mm256_set_epi16(INT16_C( 22147), INT16_C( 22147), INT16_C( 22147), INT16_C( 22147),
                            INT16_C( 22147), INT16_C( 22147), INT16_C( 22147), INT16_C( 22147),
                            INT16_C( 22147), INT16_C( 22147), INT16_C( 22147), INT16_C( 22147),
                            INT16_C( 22147), INT16_C( 22147), INT16_C( 22147), INT16_C( 22147)) },
    { simde_mm_set_epi16(INT16_C(-16582), INT16_C(-25927), INT16_C(-23837), INT16_C( 29287),
                         INT16_C(-19772), INT16_C( 13452), INT16_C( -3861), INT16_C( -4316)),
      simde_mm256_set_epi16(INT16_C( -4316), INT16_C( -4316), INT16_C( -4316), INT16_C( -4316),
                            INT16_C( -4316), INT16_C( -4316), INT16_C( -4316), INT16_C( -4316),
                            INT16_C( -4316), INT16_C( -4316), INT16_C( -4316), INT16_C( -4316),
                            INT16_C( -4316), INT16_C( -4316), INT16_C( -4316), INT16_C( -4316)) },
    { simde_mm_set_epi16(INT16_C(-22817), INT16_C(-13842), INT16_C(-28521), INT16_C( 14250),
                         INT16_C( 16773), INT16_C(-17571), INT16_C( 16721), INT16_C(-24996)),
      simde_mm256_set_epi16(INT16_C(-24996), INT16_C(-24996), INT16_C(-24996), INT16_C(-24996),
                            INT16_C(-24996), INT16_C(-24996), INT16_C(-24996), INT16_C(-24996),
                            INT16_C(-24996), INT16_C(-24996), INT16_C(-24996), INT16_C(-24996),
                            INT16_C(-24996), INT16_C(-24996), INT16_C(-24996), INT16_C(-24996)) },
    { simde_mm_set_epi16(INT16_C( -7252), INT16_C( -8727), INT16_C(  7341), INT16_C( 25148),
                         INT16_C(-18544), INT16_C( 21940), INT16_C(  7393), INT16_C( -5844)),
      simde_mm256_set_epi16(INT16_C( -5844), INT16_C( -5844), INT16_C( -5844), INT16_C( -5844),
                            INT16_C( -5844), INT16_C( -5844), INT16_C( -5844), INT16_C( -5844),
                            INT16_C( -5844), INT16_C( -5844), INT16_C( -5844), INT16_C( -5844),
                            INT16_C( -5844), INT16_C( -5844), INT16_C( -5844), INT16_C( -5844)) },
    { simde_mm_set_epi16(INT16_C(-20117), INT16_C(-17614), INT16_C( 27322), INT16_C(-16415),
                         INT16_C( 21044), INT16_C( 32548), INT16_C( 27672), INT16_C( 26452)),
      simde_mm256_set_epi16(INT16_C( 26452), INT16_C( 26452), INT16_C( 26452), INT16_C( 26452),
                            INT16_C( 26452), INT16_C( 26452), INT16_C( 26452), INT16_C( 26452),
                            INT16_C( 26452), INT16_C( 26452), INT16_C( 26452), INT16_C( 26452),
                            INT16_C( 26452), INT16_C( 26452), INT16_C( 26452), INT16_C( 26452)) },
    { simde_mm_set_epi16(INT16_C( 15651), INT16_C( -9541), INT16_C( -1245), INT16_C( 19639),
                         INT16_C( -7877), INT16_C(-31496), INT16_C( -9293), INT16_C( 24923)),
      simde_mm256_set_epi16(INT16_C( 24923), INT16_C( 24923), INT16_C( 24923), INT16_C( 24923),
                            INT16_C( 24923), INT16_C( 24923), INT16_C( 24923), INT16_C( 24923),
                            INT16_C( 24923), INT16_C( 24923), INT16_C( 24923), INT16_C( 24923),
                            INT16_C( 24923), INT16_C( 24923), INT16_C( 24923), INT16_C( 24923)) },
    { simde_mm_set_epi16(INT16_C( -1558), INT16_C( 24882), INT16_C(-27020), INT16_C( -1908),
                         INT16_C(  5712), INT16_C( 21207), INT16_C(  4307), INT16_C(  9371)),
      simde_mm256_set_epi16(INT16_C(  9371), INT16_C(  9371), INT16_C(  9371), INT16_C(  9371),
                            INT16_C(  9371), INT16_C(  9371), INT16_C(  9371), INT16_C(  9371),
                            INT16_C(  9371), INT16_C(  9371), INT16_C(  9371), INT16_C(  9371),
                            INT16_C(  9371), INT16_C(  9371), INT16_C(  9371), INT16_C(  9371)) },
    { simde_mm_set_epi16(INT16_C( 28347), INT16_C(-26396), INT16_C(-17550), INT16_C(-16266),
                         INT16_C(-10219), INT16_C(  6683), INT16_C(-28102), INT16_C( 11110)),
      simde_mm256_set_epi16(INT16_C( 11110), INT16_C( 11110), INT16_C( 11110), INT16_C( 11110),
                            INT16_C( 11110), INT16_C( 11110), INT16_C( 11110), INT16_C( 11110),
                            INT16_C( 11110), INT16_C( 11110), INT16_C( 11110), INT16_C( 11110),
                            INT16_C( 11110), INT16_C( 11110), INT16_C( 11110), INT16_C( 11110)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_broadcastw_epi16(test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm_broadcastd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C( -964897791), INT32_C( -924182241), INT32_C(  775878009), INT32_C( -739879293)),
      simde_mm_set_epi32(INT32_C( -739879293), INT32_C( -739879293), INT32_C( -739879293), INT32_C( -739879293)) },
    { simde_mm_set_epi32(INT32_C(-1086678343), INT32_C(-1562152345), INT32_C(-1295764340), INT32_C( -252973276)),
      simde_mm_set_epi32(INT32_C( -252973276), INT32_C( -252973276), INT32_C( -252973276), INT32_C( -252973276)) },
    { simde_mm_set_epi32(INT32_C(-1495283218), INT32_C(-1869138006), INT32_C( 1099283293), INT32_C( 1095867996)),
      simde_mm_set_epi32(INT32_C( 1095867996), INT32_C( 1095867996), INT32_C( 1095867996), INT32_C( 1095867996)) },
    { simde_mm_set_epi32(INT32_C( -475210263), INT32_C(  481124924), INT32_C(-1215277644), INT32_C(  484567340)),
      simde_mm_set_epi32(INT32_C(  484567340), INT32_C(  484567340), INT32_C(  484567340), INT32_C(  484567340)) },
    { simde_mm_set_epi32(INT32_C(-1318339790), INT32_C( 1790623713), INT32_C( 1379172132), INT32_C( 1813538644)),
      simde_mm_set_epi32(INT32_C( 1813538644), INT32_C( 1813538644), INT32_C( 1813538644), INT32_C( 1813538644)) },
    { simde_mm_set_epi32(INT32_C( 1025759931), INT32_C(  -81572681), INT32_C( -516193032), INT32_C( -609001125)),
      simde_mm_set_epi32(INT32_C( -609001125), INT32_C( -609001125), INT32_C( -609001125), INT32_C( -609001125)) },
    { simde_mm_set_epi32(INT32_C( -102080206), INT32_C(-1770719092), INT32_C(  374362839), INT32_C(  282272923)),
      simde_mm_set_epi32(INT32_C(  282272923), INT32_C(  282272923), INT32_C(  282272923), INT32_C(  282272923)) },
    { simde_mm_set_epi32(INT32_C( 1857788132), INT32_C(-1150107530), INT32_C( -669705701), INT32_C(-1841681562)),
      simde_mm_set_epi32(INT32_C(-1841681562), INT32_C(-1841681562), INT32_C(-1841681562), INT32_C(-1841681562)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm_broadcastd_epi32(test_vec[i].a);
    simde_assert_m128i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_broadcastd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C( -964897791), INT32_C( -924182241), INT32_C(  775878009), INT32_C( -739879293)),
      simde_mm256_set_epi32(INT32_C( -739879293), INT32_C( -739879293), INT32_C( -739879293), INT32_C( -739879293),
                            INT32_C( -739879293), INT32_C( -739879293), INT32_C( -739879293), INT32_C( -739879293)) },
    { simde_mm_set_epi32(INT32_C(-1086678343), INT32_C(-1562152345), INT32_C(-1295764340), INT32_C( -252973276)),
      simde_mm256_set_epi32(INT32_C( -252973276), INT32_C( -252973276), INT32_C( -252973276), INT32_C( -252973276),
                            INT32_C( -252973276), INT32_C( -252973276), INT32_C( -252973276), INT32_C( -252973276)) },
    { simde_mm_set_epi32(INT32_C(-1495283218), INT32_C(-1869138006), INT32_C( 1099283293), INT32_C( 1095867996)),
      simde_mm256_set_epi32(INT32_C( 1095867996), INT32_C( 1095867996), INT32_C( 1095867996), INT32_C( 1095867996),
                            INT32_C( 1095867996), INT32_C( 1095867996), INT32_C( 1095867996), INT32_C( 1095867996)) },
    { simde_mm_set_epi32(INT32_C( -475210263), INT32_C(  481124924), INT32_C(-1215277644), INT32_C(  484567340)),
      simde_mm256_set_epi32(INT32_C(  484567340), INT32_C(  484567340), INT32_C(  484567340), INT32_C(  484567340),
                            INT32_C(  484567340), INT32_C(  484567340), INT32_C(  484567340), INT32_C(  484567340)) },
    { simde_mm_set_epi32(INT32_C(-1318339790), INT32_C( 1790623713), INT32_C( 1379172132), INT32_C( 1813538644)),
      simde_mm256_set_epi32(INT32_C( 1813538644), INT32_C( 1813538644), INT32_C( 1813538644), INT32_C( 1813538644),
                            INT32_C( 1813538644), INT32_C( 1813538644), INT32_C( 1813538644), INT32_C( 1813538644)) },
    { simde_mm_set_epi32(INT32_C( 1025759931), INT32_C(  -81572681), INT32_C( -516193032), INT32_C( -609001125)),
      simde_mm256_set_epi32(INT32_C( -609001125), INT32_C( -609001125), INT32_C( -609001125), INT32_C( -609001125),
                            INT32_C( -609001125), INT32_C( -609001125), INT32_C( -609001125), INT32_C( -609001125)) },
    { simde_mm_set_epi32(INT32_C( -102080206), INT32_C(-1770719092), INT32_C(  374362839), INT32_C(  282272923)),
      simde_mm256_set_epi32(INT32_C(  282272923), INT32_C(  282272923), INT32_C(  282272923), INT32_C(  282272923),
                            INT32_C(  282272923), INT32_C(  282272923), INT32_C(  282272923), INT32_C(  282272923)) },
    { simde_mm_set_epi32(INT32_C( 1857788132), INT32_C(-1150107530), INT32_C( -669705701), INT32_C(-1841681562)),
      simde_mm256_set_epi32(INT32_C(-1841681562), INT32_C(-1841681562), INT32_C(-1841681562), INT32_C(-1841681562),
                            INT32_C(-1841681562), INT32_C(-1841681562), INT32_C(-1841681562), INT32_C(-1841681562)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_broadcastd_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm_broadcastq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi64x(INT64_C(-4144204452956858081), INT64_C( 3332370677895681667)),
      simde_mm_set_epi64x(INT64_C( 3332370677895681667), INT64_C( 3332370677895681667)) },
    { simde_mm_set_epi64x(INT64_C(-4667247941723655577), INT64_C(-5565265459581030620)),
      simde_mm_set_epi64x(INT64_C(-5565265459581030620), INT64_C(-5565265459581030620)) },
    { simde_mm_set_epi64x(INT64_C(-6422192517141809238), INT64_C( 4721385793570053724)),
      simde_mm_set_epi64x(INT64_C( 4721385793570053724), INT64_C( 4721385793570053724)) },
    { simde_mm_set_epi64x(INT64_C(-2041012537827433924), INT64_C(-5219577736055363284)),
      simde_mm_set_epi64x(INT64_C(-5219577736055363284), INT64_C(-5219577736055363284)) },
    { simde_mm_set_epi64x(INT64_C(-5662226281274884127), INT64_C( 5923499204308133716)),
      simde_mm_set_epi64x(INT64_C( 5923499204308133716), INT64_C( 5923499204308133716)) },
    { simde_mm_set_epi64x(INT64_C( 4405605361405611191), INT64_C(-2217032187177115301)),
      simde_mm_set_epi64x(INT64_C(-2217032187177115301), INT64_C(-2217032187177115301)) },
    { simde_mm_set_epi64x(INT64_C( -438431143814694772), INT64_C( 1607876150624986267)),
      simde_mm_set_epi64x(INT64_C( 1607876150624986267), INT64_C( 1607876150624986267)) },
    { simde_mm_set_epi64x(INT64_C( 7979139272981790838), INT64_C(-2876364081286468762)),
      simde_mm_set_epi64x(INT64_C(-2876364081286468762), INT64_C(-2876364081286468762)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm_broadcastq_epi64(test_vec[i].a);
    simde_assert_m128i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_broadcastq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi64x(INT64_C(-4144204452956858081), INT64_C( 3332370677895681667)),
      simde_mm256_set_epi64x(INT64_C( 3332370677895681667), INT64_C( 3332370677895681667),
                             INT64_C( 3332370677895681667), INT64_C( 3332370677895681667)) },
    { simde_mm_set_epi64x(INT64_C(-4667247941723655577), INT64_C(-5565265459581030620)),
      simde_mm256_set_epi64x(INT64_C(-5565265459581030620), INT64_C(-5565265459581030620),
                             INT64_C(-5565265459581030620), INT64_C(-5565265459581030620)) },
    { simde_mm_set_epi64x(INT64_C(-6422192517141809238), INT64_C( 4721385793570053724)),
      simde_mm256_set_epi64x(INT64_C( 4721385793570053724), INT64_C( 4721385793570053724),
                             INT64_C( 4721385793570053724), INT64_C( 4721385793570053724)) },
    { simde_mm_set_epi64x(INT64_C(-2041012537827433924), INT64_C(-5219577736055363284)),
      simde_mm256_set_epi64x(INT64_C(-5219577736055363284), INT64_C(-5219577736055363284),
                             INT64_C(-5219577736055363284), INT64_C(-5219577736055363284)) },
    { simde_mm_set_epi64x(INT64_C(-5662226281274884127), INT64_C( 5923499204308133716)),
      simde_mm256_set_epi64x(INT64_C( 5923499204308133716), INT64_C( 5923499204308133716),
                             INT64_C( 5923499204308133716), INT64_C( 5923499204308133716)) },
    { simde_mm_set_epi64x(INT64_C( 4405605361405611191), INT64_C(-2217032187177115301)),
      simde_mm256_set_epi64x(INT64_C(-2217032187177115301), INT64_C(-2217032187177115301),
                             INT64_C(-2217032187177115301), INT64_C(-2217032187177115301)) },
    { simde_mm_set_epi64x(INT64_C( -438431143814694772), INT64_C( 1607876150624986267)),
      simde_mm256_set_epi64x(INT64_C( 1607876150624986267), INT64_C( 1607876150624986267),
                             INT64_C( 1607876150624986267), INT64_C( 1607876150624986267)) },
    { simde_mm_set_epi64x(INT64_C( 7979139272981790838), INT64_C(-2876364081286468762)),
      simde_mm256_set_epi64x(INT64_C(-2876364081286468762), INT64_C(-2876364081286468762),
                             INT64_C(-2876364081286468762), INT64_C(-2876364081286468762)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_broadcastq_epi64(test_vec[i].a);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm_broadcastss_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128 a;
    simde__m128 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   550.68), SIMDE_FLOAT32_C(   569.64), SIMDE_FLOAT32_C(  -638.70), SIMDE_FLOAT32_C(   655.47)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   655.47), SIMDE_FLOAT32_C(   655.47), SIMDE_FLOAT32_C(   655.47), SIMDE_FLOAT32_C(   655.47)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   493.98), SIMDE_FLOAT32_C(   272.57), SIMDE_FLOAT32_C(   396.61), SIMDE_FLOAT32_C(   882.20)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   882.20), SIMDE_FLOAT32_C(   882.20), SIMDE_FLOAT32_C(   882.20), SIMDE_FLOAT32_C(   882.20)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   303.70), SIMDE_FLOAT32_C(   129.61), SIMDE_FLOAT32_C(  -488.11), SIMDE_FLOAT32_C(  -489.70)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -489.70), SIMDE_FLOAT32_C(  -489.70), SIMDE_FLOAT32_C(  -489.70), SIMDE_FLOAT32_C(  -489.70)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   778.71), SIMDE_FLOAT32_C(  -775.96), SIMDE_FLOAT32_C(   434.09), SIMDE_FLOAT32_C(  -774.36)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -774.36), SIMDE_FLOAT32_C(  -774.36), SIMDE_FLOAT32_C(  -774.36), SIMDE_FLOAT32_C(  -774.36)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   386.10), SIMDE_FLOAT32_C(  -166.18), SIMDE_FLOAT32_C(  -357.77), SIMDE_FLOAT32_C(  -155.51)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -155.51), SIMDE_FLOAT32_C(  -155.51), SIMDE_FLOAT32_C(  -155.51), SIMDE_FLOAT32_C(  -155.51)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -522.34), SIMDE_FLOAT32_C(   962.01), SIMDE_FLOAT32_C(   759.63), SIMDE_FLOAT32_C(   716.41)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   716.41), SIMDE_FLOAT32_C(   716.41), SIMDE_FLOAT32_C(   716.41), SIMDE_FLOAT32_C(   716.41)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   952.47), SIMDE_FLOAT32_C(   175.44), SIMDE_FLOAT32_C(  -825.67), SIMDE_FLOAT32_C(  -868.56)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -868.56), SIMDE_FLOAT32_C(  -868.56), SIMDE_FLOAT32_C(  -868.56), SIMDE_FLOAT32_C(  -868.56)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -134.90), SIMDE_FLOAT32_C(   464.44), SIMDE_FLOAT32_C(   688.14), SIMDE_FLOAT32_C(   142.40)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   142.40), SIMDE_FLOAT32_C(   142.40), SIMDE_FLOAT32_C(   142.40), SIMDE_FLOAT32_C(   142.40)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128 r = simde_mm_broadcastss_ps(test_vec[i].a);
    simde_assert_m128_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm256_broadcastss_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128 a;
    simde__m256 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   550.68), SIMDE_FLOAT32_C(   569.64), SIMDE_FLOAT32_C(  -638.70), SIMDE_FLOAT32_C(   655.47)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   655.47), SIMDE_FLOAT32_C(   655.47),
                         SIMDE_FLOAT32_C(   655.47), SIMDE_FLOAT32_C(   655.47),
                         SIMDE_FLOAT32_C(   655.47), SIMDE_FLOAT32_C(   655.47),
                         SIMDE_FLOAT32_C(   655.47), SIMDE_FLOAT32_C(   655.47)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   303.70), SIMDE_FLOAT32_C(   129.61), SIMDE_FLOAT32_C(  -488.11), SIMDE_FLOAT32_C(  -489.70)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -489.70), SIMDE_FLOAT32_C(  -489.70),
                         SIMDE_FLOAT32_C(  -489.70), SIMDE_FLOAT32_C(  -489.70),
                         SIMDE_FLOAT32_C(  -489.70), SIMDE_FLOAT32_C(  -489.70),
                         SIMDE_FLOAT32_C(  -489.70), SIMDE_FLOAT32_C(  -489.70)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   386.10), SIMDE_FLOAT32_C(  -166.18), SIMDE_FLOAT32_C(  -357.77), SIMDE_FLOAT32_C(  -155.51)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -155.51), SIMDE_FLOAT32_C(  -155.51),
                         SIMDE_FLOAT32_C(  -155.51), SIMDE_FLOAT32_C(  -155.51),
                         SIMDE_FLOAT32_C(  -155.51), SIMDE_FLOAT32_C(  -155.51),
                         SIMDE_FLOAT32_C(  -155.51), SIMDE_FLOAT32_C(  -155.51)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   952.47), SIMDE_FLOAT32_C(   175.44), SIMDE_FLOAT32_C(  -825.67), SIMDE_FLOAT32_C(  -868.56)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -868.56), SIMDE_FLOAT32_C(  -868.56),
                         SIMDE_FLOAT32_C(  -868.56), SIMDE_FLOAT32_C(  -868.56),
                         SIMDE_FLOAT32_C(  -868.56), SIMDE_FLOAT32_C(  -868.56),
                         SIMDE_FLOAT32_C(  -868.56), SIMDE_FLOAT32_C(  -868.56)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -766.17), SIMDE_FLOAT32_C(   812.79), SIMDE_FLOAT32_C(  -842.83), SIMDE_FLOAT32_C(   -99.19)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   -99.19), SIMDE_FLOAT32_C(   -99.19),
                         SIMDE_FLOAT32_C(   -99.19), SIMDE_FLOAT32_C(   -99.19),
                         SIMDE_FLOAT32_C(   -99.19), SIMDE_FLOAT32_C(   -99.19),
                         SIMDE_FLOAT32_C(   -99.19), SIMDE_FLOAT32_C(   -99.19)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -434.84), SIMDE_FLOAT32_C(  -337.86), SIMDE_FLOAT32_C(   127.40), SIMDE_FLOAT32_C(   235.90)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   235.90), SIMDE_FLOAT32_C(   235.90),
                         SIMDE_FLOAT32_C(   235.90), SIMDE_FLOAT32_C(   235.90),
                         SIMDE_FLOAT32_C(   235.90), SIMDE_FLOAT32_C(   235.90),
                         SIMDE_FLOAT32_C(   235.90), SIMDE_FLOAT32_C(   235.90)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -349.94), SIMDE_FLOAT32_C(   403.87), SIMDE_FLOAT32_C(   -69.00), SIMDE_FLOAT32_C(   494.83)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   494.83), SIMDE_FLOAT32_C(   494.83),
                         SIMDE_FLOAT32_C(   494.83), SIMDE_FLOAT32_C(   494.83),
                         SIMDE_FLOAT32_C(   494.83), SIMDE_FLOAT32_C(   494.83),
                         SIMDE_FLOAT32_C(   494.83), SIMDE_FLOAT32_C(   494.83)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(    86.40), SIMDE_FLOAT32_C(   641.98), SIMDE_FLOAT32_C(  -789.02), SIMDE_FLOAT32_C(  -272.67)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -272.67), SIMDE_FLOAT32_C(  -272.67),
                         SIMDE_FLOAT32_C(  -272.67), SIMDE_FLOAT32_C(  -272.67),
                         SIMDE_FLOAT32_C(  -272.67), SIMDE_FLOAT32_C(  -272.67),
                         SIMDE_FLOAT32_C(  -272.67), SIMDE_FLOAT32_C(  -272.67)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256 r = simde_mm256_broadcastss_ps(test_vec[i].a);
    simde_assert_m256_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm_broadcastsd_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128d a;
    simde__m128d r;
  } test_vec[8] = {
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -638.70), SIMDE_FLOAT64_C(  655.47)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  655.47), SIMDE_FLOAT64_C(  655.47)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  550.68), SIMDE_FLOAT64_C(  569.64)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  569.64), SIMDE_FLOAT64_C(  569.64)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  396.61), SIMDE_FLOAT64_C(  882.20)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  882.20), SIMDE_FLOAT64_C(  882.20)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  493.98), SIMDE_FLOAT64_C(  272.57)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  272.57), SIMDE_FLOAT64_C(  272.57)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -488.11), SIMDE_FLOAT64_C( -489.70)),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -489.70), SIMDE_FLOAT64_C( -489.70)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  303.70), SIMDE_FLOAT64_C(  129.61)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  129.61), SIMDE_FLOAT64_C(  129.61)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  434.09), SIMDE_FLOAT64_C( -774.36)),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -774.36), SIMDE_FLOAT64_C( -774.36)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  778.71), SIMDE_FLOAT64_C( -775.96)),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -775.96), SIMDE_FLOAT64_C( -775.96)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    #if !defined(SIMDE_NATIVE_ALIASES_TESTING)
      simde__m128d r = simde_mm_broadcastsd_pd(test_vec[i].a);
    #else
      simde__m128d r = simde_mm_movedup_pd(test_vec[i].a);
    #endif
    simde_assert_m128d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm256_broadcastsd_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128d a;
    simde__m256d r;
  } test_vec[8] = {
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -638.70), SIMDE_FLOAT64_C(  655.47)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  655.47), SIMDE_FLOAT64_C(  655.47),
                         SIMDE_FLOAT64_C(  655.47), SIMDE_FLOAT64_C(  655.47)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  396.61), SIMDE_FLOAT64_C(  882.20)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  882.20), SIMDE_FLOAT64_C(  882.20),
                         SIMDE_FLOAT64_C(  882.20), SIMDE_FLOAT64_C(  882.20)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -488.11), SIMDE_FLOAT64_C( -489.70)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -489.70), SIMDE_FLOAT64_C( -489.70),
                         SIMDE_FLOAT64_C( -489.70), SIMDE_FLOAT64_C( -489.70)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  434.09), SIMDE_FLOAT64_C( -774.36)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -774.36), SIMDE_FLOAT64_C( -774.36),
                         SIMDE_FLOAT64_C( -774.36), SIMDE_FLOAT64_C( -774.36)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -357.77), SIMDE_FLOAT64_C( -155.51)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -155.51), SIMDE_FLOAT64_C( -155.51),
                         SIMDE_FLOAT64_C( -155.51), SIMDE_FLOAT64_C( -155.51)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  759.63), SIMDE_FLOAT64_C(  716.41)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  716.41), SIMDE_FLOAT64_C(  716.41),
                         SIMDE_FLOAT64_C(  716.41), SIMDE_FLOAT64_C(  716.41)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -825.67), SIMDE_FLOAT64_C( -868.56)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -868.56), SIMDE_FLOAT64_C( -868.56),
                         SIMDE_FLOAT64_C( -868.56), SIMDE_FLOAT64_C( -868.56)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  688.14), SIMDE_FLOAT64_C(  142.40)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  142.40), SIMDE_FLOAT64_C(  142.40),
                         SIMDE_FLOAT64_C(  142.40), SIMDE_FLOAT64_C(  142.40)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256d r = simde_mm256_broadcastsd_pd(test_vec[i].a);
    simde_assert_m256d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm256_broadcastsi128_si256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi64x   (INT64_C( 3783485884510153131), INT64_C( 5322848723863790661)),
      simde_mm256_set_epi64x(INT64_C( 3783485884510153131), INT64_C( 5322848723863790661),
                             INT64_C( 3783485884510153131), INT64_C( 5322848723863790661)) },
    { simde_mm_set_epi64x   (INT64_C( 3709494745882156155), INT64_C(-6688158465955285293)),
      simde_mm256_set_epi64x(INT64_C( 3709494745882156155), INT64_C(-6688158465955285293),
                             INT64_C( 3709494745882156155), INT64_C(-6688158465955285293)) },
    { simde_mm_set_epi64x   (INT64_C(-8419237038829073489), INT64_C(-1707044583358729761)),
      simde_mm256_set_epi64x(INT64_C(-8419237038829073489), INT64_C(-1707044583358729761),
                             INT64_C(-8419237038829073489), INT64_C(-1707044583358729761)) },
    { simde_mm_set_epi64x   (INT64_C( -168447772491176834), INT64_C(-4784151950425519944)),
      simde_mm256_set_epi64x(INT64_C( -168447772491176834), INT64_C(-4784151950425519944),
                             INT64_C( -168447772491176834), INT64_C(-4784151950425519944)) },
    { simde_mm_set_epi64x   (INT64_C(  992172666234330248), INT64_C( 7115358650695109080)),
      simde_mm256_set_epi64x(INT64_C(  992172666234330248), INT64_C( 7115358650695109080),
                             INT64_C(  992172666234330248), INT64_C( 7115358650695109080)) },
    { simde_mm_set_epi64x   (INT64_C( 6019682307676879898), INT64_C( -745177675559433697)),
      simde_mm256_set_epi64x(INT64_C( 6019682307676879898), INT64_C( -745177675559433697),
                             INT64_C( 6019682307676879898), INT64_C( -745177675559433697)) },
    { simde_mm_set_epi64x   (INT64_C(-8364368011425569946), INT64_C( 1685080214233190853)),
      simde_mm256_set_epi64x(INT64_C(-8364368011425569946), INT64_C( 1685080214233190853),
                             INT64_C(-8364368011425569946), INT64_C( 1685080214233190853)) },
    { simde_mm_set_epi64x   (INT64_C( 6712859087758853138), INT64_C( 2170550621944299043)),
      simde_mm256_set_epi64x(INT64_C( 6712859087758853138), INT64_C( 2170550621944299043),
                             INT64_C( 6712859087758853138), INT64_C( 2170550621944299043)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_broadcastsi128_si256(test_vec[i].a);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_bslli_epi128(SIMDE_MUNIT_TEST_ARGS) {
  simde__m256i r;

  const int8_t a1[32] = { -INT8_C( 101), -INT8_C(  22),  INT8_C( 109),  INT8_C(  62), -INT8_C(  16),  INT8_C(   2), -INT8_C(  71),  INT8_C( 107),
        -INT8_C( 122),  INT8_C(  94), -INT8_C( 113), -INT8_C(  23), -INT8_C(  99), -INT8_C( 110), -INT8_C(  42),  INT8_C( 105),
         INT8_C(   5), -INT8_C(  23), -INT8_C(  24),  INT8_C(  36), -INT8_C(  20),  INT8_C(   8), -INT8_C(  24),  INT8_C(  71),
        -INT8_C( 117),  INT8_C(  77), -INT8_C(  78), -INT8_C( 120), -INT8_C(  92),  INT8_C(  20),  INT8_C(  10),  INT8_C(  64) };

  const int8_t e1[32] = { -INT8_C( 101), -INT8_C(  22),  INT8_C( 109),  INT8_C(  62), -INT8_C(  16),  INT8_C(   2), -INT8_C(  71),  INT8_C( 107),
        -INT8_C( 122),  INT8_C(  94), -INT8_C( 113), -INT8_C(  23), -INT8_C(  99), -INT8_C( 110), -INT8_C(  42),  INT8_C( 105),
         INT8_C(   5), -INT8_C(  23), -INT8_C(  24),  INT8_C(  36), -INT8_C(  20),  INT8_C(   8), -INT8_C(  24),  INT8_C(  71),
        -INT8_C( 117),  INT8_C(  77), -INT8_C(  78), -INT8_C( 120), -INT8_C(  92),  INT8_C(  20),  INT8_C(  10),  INT8_C(  64) };
  r = simde_mm256_bslli_epi128(simde_x_mm256_loadu_epi8(a1), 0);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e1));

  const int8_t a2[32] = {  INT8_C(  62),  INT8_C(  94),  INT8_C(  58), -INT8_C(  15),  INT8_C(  92),  INT8_C( 124),  INT8_C(  43), -INT8_C(  58),
        -INT8_C(   7),  INT8_C(  63),  INT8_C(  54),  INT8_C(  76), -INT8_C(  88), -INT8_C(  90), -INT8_C(   2), -INT8_C(  57),
         INT8_C(  73),  INT8_C( 106),  INT8_C(  57),  INT8_C(  44),  INT8_C( 105), -INT8_C(  54), -INT8_C(  66),  INT8_C(  62),
        -INT8_C(  71), -INT8_C(  67), -INT8_C( 106), -INT8_C(  54),  INT8_C(  54), -INT8_C( 122), -INT8_C(  65),  INT8_C( 116) };

  const int8_t e2[32] = {  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(  62),  INT8_C(  94),  INT8_C(  58), -INT8_C(  15),  INT8_C(  92),
         INT8_C( 124),  INT8_C(  43), -INT8_C(  58), -INT8_C(   7),  INT8_C(  63),  INT8_C(  54),  INT8_C(  76), -INT8_C(  88),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(  73),  INT8_C( 106),  INT8_C(  57),  INT8_C(  44),  INT8_C( 105),
        -INT8_C(  54), -INT8_C(  66),  INT8_C(  62), -INT8_C(  71), -INT8_C(  67), -INT8_C( 106), -INT8_C(  54),  INT8_C(  54) };
  r = simde_mm256_bslli_epi128(simde_x_mm256_loadu_epi8(a2), 3);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e2));

  const int8_t a3[32] = { -INT8_C( 109),  INT8_C(  56),  INT8_C( 107), -INT8_C(  88),  INT8_C( 124),  INT8_C( 123), -INT8_C( 108), -INT8_C(  79),
         INT8_C(  10), -INT8_C( 123), -INT8_C(  57), -INT8_C(  23), -INT8_C(  32), -INT8_C(  10), -INT8_C(  85), -INT8_C(  22),
         INT8_C(  91), -INT8_C(  91),  INT8_C(   0), -INT8_C(  47),  INT8_C(  30), -INT8_C(   7), -INT8_C(  73),  INT8_C(  17),
         INT8_C(  74),  INT8_C(   6), -INT8_C(  73), -INT8_C(  13),  INT8_C(  59),  INT8_C(  44), -INT8_C(  34), -INT8_C(  49) };

  const int8_t e3[32] = {  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0), -INT8_C( 109),  INT8_C(  56),  INT8_C( 107),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(  91), -INT8_C(  91),  INT8_C(   0) };
  r = simde_mm256_bslli_epi128(simde_x_mm256_loadu_epi8(a3), 13);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e3));

  const int8_t a4[32] = {  INT8_C(  74), -INT8_C( 109), -INT8_C(  57),  INT8_C(  52),  INT8_C(  64), -INT8_C( 101),  INT8_C(  54), -INT8_C( 122),
        -INT8_C(  85),      INT8_MAX, -INT8_C( 106),  INT8_C(  80), -INT8_C(  15), -INT8_C(  65),  INT8_C(  88),  INT8_C(  92),
        -INT8_C(  12),  INT8_C( 103), -INT8_C(   4), -INT8_C(  36), -INT8_C(  67), -INT8_C( 101), -INT8_C(  91),      INT8_MAX,
        -INT8_C(  39), -INT8_C(  89), -INT8_C(  86), -INT8_C(  94), -INT8_C(  33), -INT8_C(  28),  INT8_C( 102),  INT8_C(  41) };

  const int8_t e4[32] = {  INT8_C(   0),  INT8_C(   0),  INT8_C(  74), -INT8_C( 109), -INT8_C(  57),  INT8_C(  52),  INT8_C(  64), -INT8_C( 101),
         INT8_C(  54), -INT8_C( 122), -INT8_C(  85),      INT8_MAX, -INT8_C( 106),  INT8_C(  80), -INT8_C(  15), -INT8_C(  65),
         INT8_C(   0),  INT8_C(   0), -INT8_C(  12),  INT8_C( 103), -INT8_C(   4), -INT8_C(  36), -INT8_C(  67), -INT8_C( 101),
        -INT8_C(  91),      INT8_MAX, -INT8_C(  39), -INT8_C(  89), -INT8_C(  86), -INT8_C(  94), -INT8_C(  33), -INT8_C(  28) };
  r = simde_mm256_bslli_epi128(simde_x_mm256_loadu_epi8(a4), 2);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e4));

  const int8_t a5[32] = {  INT8_C( 102), -INT8_C(   9),  INT8_C(  57), -INT8_C(  65), -INT8_C(  51),  INT8_C(  43), -INT8_C(  30),  INT8_C(  53),
         INT8_C(  68),  INT8_C(  76), -INT8_C(  34), -INT8_C(  69),  INT8_C(  55),  INT8_C(  91), -INT8_C( 101), -INT8_C(  32),
         INT8_C( 107),  INT8_C(  28),  INT8_C(  11), -INT8_C(  64),  INT8_C( 104), -INT8_C(  26),  INT8_C(  67),  INT8_C( 120),
         INT8_C(  27),  INT8_C( 100),  INT8_C(  46), -INT8_C(  10), -INT8_C(  68),  INT8_C(  86),  INT8_C(  29),  INT8_C(  34) };

  const int8_t e5[32] = {  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) };
  r = simde_mm256_bslli_epi128(simde_x_mm256_loadu_epi8(a5), 19);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e5));

  return 0;
}

static int
test_simde_mm256_bsrli_epi128(SIMDE_MUNIT_TEST_ARGS) {
  simde__m256i r;

  const int8_t a1[32] = { -INT8_C( 101), -INT8_C(  22),  INT8_C( 109),  INT8_C(  62), -INT8_C(  16),  INT8_C(   2), -INT8_C(  71),  INT8_C( 107),
        -INT8_C( 122),  INT8_C(  94), -INT8_C( 113), -INT8_C(  23), -INT8_C(  99), -INT8_C( 110), -INT8_C(  42),  INT8_C( 105),
         INT8_C(   5), -INT8_C(  23), -INT8_C(  24),  INT8_C(  36), -INT8_C(  20),  INT8_C(   8), -INT8_C(  24),  INT8_C(  71),
        -INT8_C( 117),  INT8_C(  77), -INT8_C(  78), -INT8_C( 120), -INT8_C(  92),  INT8_C(  20),  INT8_C(  10),  INT8_C(  64) };

  const int8_t e1[32] = { -INT8_C( 101), -INT8_C(  22),  INT8_C( 109),  INT8_C(  62), -INT8_C(  16),  INT8_C(   2), -INT8_C(  71),  INT8_C( 107),
        -INT8_C( 122),  INT8_C(  94), -INT8_C( 113), -INT8_C(  23), -INT8_C(  99), -INT8_C( 110), -INT8_C(  42),  INT8_C( 105),
         INT8_C(   5), -INT8_C(  23), -INT8_C(  24),  INT8_C(  36), -INT8_C(  20),  INT8_C(   8), -INT8_C(  24),  INT8_C(  71),
        -INT8_C( 117),  INT8_C(  77), -INT8_C(  78), -INT8_C( 120), -INT8_C(  92),  INT8_C(  20),  INT8_C(  10),  INT8_C(  64) };
  r = simde_mm256_bsrli_epi128(simde_x_mm256_loadu_epi8(a1), 0);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e1));

  const int8_t a2[32] = {  INT8_C(  16), -INT8_C(  10), -INT8_C( 116),  INT8_C(  62), -INT8_C(  37), -INT8_C(  30), -INT8_C( 105),  INT8_C(  80),
         INT8_C(   0), -INT8_C(  81),  INT8_C(  44),  INT8_C(  72), -INT8_C(  47), -INT8_C(  51),  INT8_C(  72), -INT8_C( 102),
         INT8_C(  90), -INT8_C(  37), -INT8_C( 126),  INT8_C( 111),  INT8_C(  75), -INT8_C(   3), -INT8_C(  50), -INT8_C(  67),
        -INT8_C( 112), -INT8_C(  59),  INT8_C(  80),  INT8_C(   2), -INT8_C(  10),  INT8_C(   8), -INT8_C(  72),  INT8_C(   7) };

  const int8_t e2[32] = {  INT8_C(  62), -INT8_C(  37), -INT8_C(  30), -INT8_C( 105),  INT8_C(  80),  INT8_C(   0), -INT8_C(  81),  INT8_C(  44),
         INT8_C(  72), -INT8_C(  47), -INT8_C(  51),  INT8_C(  72), -INT8_C( 102),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C( 111),  INT8_C(  75), -INT8_C(   3), -INT8_C(  50), -INT8_C(  67), -INT8_C( 112), -INT8_C(  59),  INT8_C(  80),
         INT8_C(   2), -INT8_C(  10),  INT8_C(   8), -INT8_C(  72),  INT8_C(   7),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) };
  r = simde_mm256_bsrli_epi128(simde_x_mm256_loadu_epi8(a2), 3);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e2));

  const int8_t a3[32] = { -INT8_C(  90),  INT8_C(  37),  INT8_C(   2),  INT8_C(  43),  INT8_C(  12),  INT8_C(   1), -INT8_C(  14), -INT8_C( 108),
        -INT8_C( 108), -INT8_C(  85),  INT8_C(  63),  INT8_C( 117), -INT8_C(  64),  INT8_C( 115), -INT8_C(  42), -INT8_C(  20),
         INT8_C(   9), -INT8_C(  75),  INT8_C(  98),  INT8_C( 100),  INT8_C(  13), -INT8_C( 115),  INT8_C( 124),  INT8_C(  78),
         INT8_C(  48), -INT8_C(  23),  INT8_C(  73),  INT8_C( 111), -INT8_C(  10), -INT8_C(  64),  INT8_C(  14), -INT8_C( 100) };

  const int8_t e3[32] = {  INT8_C( 115), -INT8_C(  42), -INT8_C(  20),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
        -INT8_C(  64),  INT8_C(  14), -INT8_C( 100),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) };
  r = simde_mm256_bsrli_epi128(simde_x_mm256_loadu_epi8(a3), 13);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e3));

  const int8_t a4[32] = { -INT8_C( 103), -INT8_C(  46), -INT8_C(  10),  INT8_C(  82), -INT8_C(  74), -INT8_C(  94), -INT8_C( 122), -INT8_C(   9),
         INT8_C( 105), -INT8_C(  36), -INT8_C(  46), -INT8_C(  40), -INT8_C( 102), -INT8_C(  76),  INT8_C(  23), -INT8_C(  45),
         INT8_C(  73), -INT8_C(  66),  INT8_C( 102),  INT8_C( 124), -INT8_C( 122),  INT8_C(   8),  INT8_C(  49), -INT8_C(  57),
        -INT8_C(  70), -INT8_C(  95), -INT8_C(  12), -INT8_C( 115), -INT8_C(  68),  INT8_C(  77),  INT8_C(  94),  INT8_C(  85) };

  const int8_t e4[32] = { -INT8_C(  10),  INT8_C(  82), -INT8_C(  74), -INT8_C(  94), -INT8_C( 122), -INT8_C(   9),  INT8_C( 105), -INT8_C(  36),
        -INT8_C(  46), -INT8_C(  40), -INT8_C( 102), -INT8_C(  76),  INT8_C(  23), -INT8_C(  45),  INT8_C(   0),  INT8_C(   0),
         INT8_C( 102),  INT8_C( 124), -INT8_C( 122),  INT8_C(   8),  INT8_C(  49), -INT8_C(  57), -INT8_C(  70), -INT8_C(  95),
        -INT8_C(  12), -INT8_C( 115), -INT8_C(  68),  INT8_C(  77),  INT8_C(  94),  INT8_C(  85),  INT8_C(   0),  INT8_C(   0) };
  r = simde_mm256_bsrli_epi128(simde_x_mm256_loadu_epi8(a4), 2);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e4));

  const int8_t a5[32] = {  INT8_C(   3), -INT8_C(  11),  INT8_C(  13), -INT8_C(  50),  INT8_C( 113), -INT8_C(  14), -INT8_C(  91), -INT8_C(  97),
        -INT8_C(  82), -INT8_C(  41), -INT8_C(  48),  INT8_C(  51),  INT8_C(   0),  INT8_C(  74),  INT8_C(  95), -INT8_C(  86),
        -INT8_C( 107),  INT8_C( 112), -INT8_C( 126),  INT8_C( 116),  INT8_C(  46),  INT8_C(  35),  INT8_C(  12), -INT8_C( 126),
         INT8_C(  51),  INT8_C(  63),  INT8_C(  58),  INT8_C( 109), -INT8_C( 105), -INT8_C(  22), -INT8_C(  72), -INT8_C( 102) };

  const int8_t e5[32] = {  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) };
  r = simde_mm256_bsrli_epi128(simde_x_mm256_loadu_epi8(a5), 19);
  simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(e5));

  return 0;
}

static int
test_simde_mm256_cvtepi8_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( -64), INT8_C( -39), INT8_C(  -1), INT8_C( 123),
                        INT8_C( -41), INT8_C(  42), INT8_C( -42), INT8_C(  48),
                        INT8_C(-101), INT8_C( -11), INT8_C(  78), INT8_C( -33),
                        INT8_C(  -4), INT8_C( -46), INT8_C(-128), INT8_C( 115)),
      simde_mm256_set_epi16(INT16_C(   -64), INT16_C(   -39), INT16_C(    -1), INT16_C(   123),
                            INT16_C(   -41), INT16_C(    42), INT16_C(   -42), INT16_C(    48),
                            INT16_C(  -101), INT16_C(   -11), INT16_C(    78), INT16_C(   -33),
                            INT16_C(    -4), INT16_C(   -46), INT16_C(  -128), INT16_C(   115)) },
    { simde_mm_set_epi8(INT8_C( -82), INT8_C(  92), INT8_C(  55), INT8_C( -41),
                        INT8_C(  61), INT8_C(-113), INT8_C( 108), INT8_C(  36),
                        INT8_C( 115), INT8_C(  53), INT8_C( -46), INT8_C( -96),
                        INT8_C( -71), INT8_C( -38), INT8_C(  25), INT8_C(  61)),
      simde_mm256_set_epi16(INT16_C(   -82), INT16_C(    92), INT16_C(    55), INT16_C(   -41),
                            INT16_C(    61), INT16_C(  -113), INT16_C(   108), INT16_C(    36),
                            INT16_C(   115), INT16_C(    53), INT16_C(   -46), INT16_C(   -96),
                            INT16_C(   -71), INT16_C(   -38), INT16_C(    25), INT16_C(    61)) },
    { simde_mm_set_epi8(INT8_C( -73), INT8_C(   5), INT8_C(  10), INT8_C(-111),
                        INT8_C(-127), INT8_C(  16), INT8_C( -16), INT8_C(-115),
                        INT8_C(  94), INT8_C(  88), INT8_C( -20), INT8_C( -24),
                        INT8_C( -27), INT8_C( -17), INT8_C( -31), INT8_C(  67)),
      simde_mm256_set_epi16(INT16_C(   -73), INT16_C(     5), INT16_C(    10), INT16_C(  -111),
                            INT16_C(  -127), INT16_C(    16), INT16_C(   -16), INT16_C(  -115),
                            INT16_C(    94), INT16_C(    88), INT16_C(   -20), INT16_C(   -24),
                            INT16_C(   -27), INT16_C(   -17), INT16_C(   -31), INT16_C(    67)) },
    { simde_mm_set_epi8(INT8_C( -17), INT8_C(   8), INT8_C(  67), INT8_C( -40),
                        INT8_C(  99), INT8_C(  89), INT8_C( -60), INT8_C( -24),
                        INT8_C( 120), INT8_C( -29), INT8_C(-127), INT8_C(  15),
                        INT8_C( -40), INT8_C(-106), INT8_C(  13), INT8_C( -27)),
      simde_mm256_set_epi16(INT16_C(   -17), INT16_C(     8), INT16_C(    67), INT16_C(   -40),
                            INT16_C(    99), INT16_C(    89), INT16_C(   -60), INT16_C(   -24),
                            INT16_C(   120), INT16_C(   -29), INT16_C(  -127), INT16_C(    15),
                            INT16_C(   -40), INT16_C(  -106), INT16_C(    13), INT16_C(   -27)) },
    { simde_mm_set_epi8(INT8_C(  -5), INT8_C(  52), INT8_C( 112), INT8_C( -86),
                        INT8_C(   7), INT8_C(   2), INT8_C(  89), INT8_C(  40),
                        INT8_C(  34), INT8_C( -14), INT8_C(-119), INT8_C(-115),
                        INT8_C(-103), INT8_C( 103), INT8_C( -15), INT8_C( -68)),
      simde_mm256_set_epi16(INT16_C(    -5), INT16_C(    52), INT16_C(   112), INT16_C(   -86),
                            INT16_C(     7), INT16_C(     2), INT16_C(    89), INT16_C(    40),
                            INT16_C(    34), INT16_C(   -14), INT16_C(  -119), INT16_C(  -115),
                            INT16_C(  -103), INT16_C(   103), INT16_C(   -15), INT16_C(   -68)) },
    { simde_mm_set_epi8(INT8_C(  34), INT8_C( 104), INT8_C(-116), INT8_C(-106),
                        INT8_C(-122), INT8_C(  51), INT8_C( -86), INT8_C(  26),
                        INT8_C(  57), INT8_C(  23), INT8_C( 125), INT8_C(  20),
                        INT8_C(  40), INT8_C( -87), INT8_C( -60), INT8_C( -93)),
      simde_mm256_set_epi16(INT16_C(    34), INT16_C(   104), INT16_C(  -116), INT16_C(  -106),
                            INT16_C(  -122), INT16_C(    51), INT16_C(   -86), INT16_C(    26),
                            INT16_C(    57), INT16_C(    23), INT16_C(   125), INT16_C(    20),
                            INT16_C(    40), INT16_C(   -87), INT16_C(   -60), INT16_C(   -93)) },
    { simde_mm_set_epi8(INT8_C(  35), INT8_C( 103), INT8_C(  83), INT8_C(  11),
                        INT8_C(   5), INT8_C( -26), INT8_C( -34), INT8_C( -28),
                        INT8_C( -15), INT8_C(  -2), INT8_C(  10), INT8_C( -97),
                        INT8_C(  35), INT8_C(-108), INT8_C(  38), INT8_C(-122)),
      simde_mm256_set_epi16(INT16_C(    35), INT16_C(   103), INT16_C(    83), INT16_C(    11),
                            INT16_C(     5), INT16_C(   -26), INT16_C(   -34), INT16_C(   -28),
                            INT16_C(   -15), INT16_C(    -2), INT16_C(    10), INT16_C(   -97),
                            INT16_C(    35), INT16_C(  -108), INT16_C(    38), INT16_C(  -122)) },
    { simde_mm_set_epi8(INT8_C(  44), INT8_C(-101), INT8_C(-122), INT8_C(  91),
                        INT8_C(  60), INT8_C(  60), INT8_C(  22), INT8_C(  81),
                        INT8_C(  13), INT8_C(-111), INT8_C(-125), INT8_C( -40),
                        INT8_C( -72), INT8_C(-102), INT8_C( -87), INT8_C( 108)),
      simde_mm256_set_epi16(INT16_C(    44), INT16_C(  -101), INT16_C(  -122), INT16_C(    91),
                            INT16_C(    60), INT16_C(    60), INT16_C(    22), INT16_C(    81),
                            INT16_C(    13), INT16_C(  -111), INT16_C(  -125), INT16_C(   -40),
                            INT16_C(   -72), INT16_C(  -102), INT16_C(   -87), INT16_C(   108)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepi8_epi16(test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepi8_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( 123), INT8_C(  -2), INT8_C( 102), INT8_C( -48),
                        INT8_C(  12), INT8_C(-119), INT8_C( -32), INT8_C(  68),
                        INT8_C( -88), INT8_C(  80), INT8_C(  32), INT8_C(-108),
                        INT8_C( -66), INT8_C(  35), INT8_C( -12), INT8_C(-103)),
      simde_mm256_set_epi32(INT32_C(        -88), INT32_C(         80), INT32_C(         32), INT32_C(       -108),
                            INT32_C(        -66), INT32_C(         35), INT32_C(        -12), INT32_C(       -103)) },
    { simde_mm_set_epi8(INT8_C( -12), INT8_C( -88), INT8_C( -20), INT8_C( -26),
                        INT8_C( -69), INT8_C( -40), INT8_C( 122), INT8_C(-121),
                        INT8_C(  47), INT8_C( 123), INT8_C( -15), INT8_C(-104),
                        INT8_C( -91), INT8_C( -13), INT8_C( -61), INT8_C(-104)),
      simde_mm256_set_epi32(INT32_C(         47), INT32_C(        123), INT32_C(        -15), INT32_C(       -104),
                            INT32_C(        -91), INT32_C(        -13), INT32_C(        -61), INT32_C(       -104)) },
    { simde_mm_set_epi8(INT8_C(-105), INT8_C(  35), INT8_C(-118), INT8_C( -44),
                        INT8_C( -91), INT8_C(  43), INT8_C(  18), INT8_C( -70),
                        INT8_C(  44), INT8_C(  98), INT8_C( -82), INT8_C(-126),
                        INT8_C( -80), INT8_C(-109), INT8_C(  69), INT8_C(-116)),
      simde_mm256_set_epi32(INT32_C(         44), INT32_C(         98), INT32_C(        -82), INT32_C(       -126),
                            INT32_C(        -80), INT32_C(       -109), INT32_C(         69), INT32_C(       -116)) },
    { simde_mm_set_epi8(INT8_C(  43), INT8_C(  92), INT8_C( -66), INT8_C( -32),
                        INT8_C(  71), INT8_C( -63), INT8_C(  51), INT8_C(  89),
                        INT8_C( 102), INT8_C(  -4), INT8_C( -28), INT8_C(  -5),
                        INT8_C(-107), INT8_C( -25), INT8_C(-107), INT8_C( -15)),
      simde_mm256_set_epi32(INT32_C(        102), INT32_C(         -4), INT32_C(        -28), INT32_C(         -5),
                            INT32_C(       -107), INT32_C(        -25), INT32_C(       -107), INT32_C(        -15)) },
    { simde_mm_set_epi8(INT8_C(  -7), INT8_C( -22), INT8_C(  -1), INT8_C(  29),
                        INT8_C(  -3), INT8_C( -97), INT8_C(  61), INT8_C( -36),
                        INT8_C( -81), INT8_C( -83), INT8_C(  54), INT8_C(  37),
                        INT8_C( -70), INT8_C( -70), INT8_C(  12), INT8_C( -61)),
      simde_mm256_set_epi32(INT32_C(        -81), INT32_C(        -83), INT32_C(         54), INT32_C(         37),
                            INT32_C(        -70), INT32_C(        -70), INT32_C(         12), INT32_C(        -61)) },
    { simde_mm_set_epi8(INT8_C(  88), INT8_C( -30), INT8_C( 125), INT8_C( -22),
                        INT8_C( -87), INT8_C(-109), INT8_C( -90), INT8_C( -86),
                        INT8_C(-124), INT8_C( -10), INT8_C(  57), INT8_C( -62),
                        INT8_C( -55), INT8_C(  40), INT8_C( -11), INT8_C(-100)),
      simde_mm256_set_epi32(INT32_C(       -124), INT32_C(        -10), INT32_C(         57), INT32_C(        -62),
                            INT32_C(        -55), INT32_C(         40), INT32_C(        -11), INT32_C(       -100)) },
    { simde_mm_set_epi8(INT8_C(  58), INT8_C(-110), INT8_C(  -6), INT8_C( -58),
                        INT8_C(-110), INT8_C( 125), INT8_C(  39), INT8_C( -82),
                        INT8_C( -88), INT8_C( 107), INT8_C(  76), INT8_C(  58),
                        INT8_C(  -2), INT8_C(-119), INT8_C( -27), INT8_C(  40)),
      simde_mm256_set_epi32(INT32_C(        -88), INT32_C(        107), INT32_C(         76), INT32_C(         58),
                            INT32_C(         -2), INT32_C(       -119), INT32_C(        -27), INT32_C(         40)) },
    { simde_mm_set_epi8(INT8_C(-110), INT8_C( -24), INT8_C( -33), INT8_C(-117),
                        INT8_C(  92), INT8_C( -35), INT8_C( 113), INT8_C(  13),
                        INT8_C(  46), INT8_C(-107), INT8_C(   4), INT8_C(   8),
                        INT8_C(  56), INT8_C(   9), INT8_C( 123), INT8_C(  36)),
      simde_mm256_set_epi32(INT32_C(         46), INT32_C(       -107), INT32_C(          4), INT32_C(          8),
                            INT32_C(         56), INT32_C(          9), INT32_C(        123), INT32_C(         36)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepi8_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepi8_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(   9), INT8_C( -74), INT8_C( -52), INT8_C( -68),
                        INT8_C(  95), INT8_C( -10), INT8_C( -99), INT8_C( 121),
                        INT8_C(   1), INT8_C(  69), INT8_C( -36), INT8_C(-102),
                        INT8_C(-124), INT8_C( -93), INT8_C(-101), INT8_C(-118)),
      simde_mm256_set_epi64x(INT64_C(                -124), INT64_C(                 -93),
                             INT64_C(                -101), INT64_C(                -118)) },
    { simde_mm_set_epi8(INT8_C(-107), INT8_C( -34), INT8_C( -75), INT8_C( 119),
                        INT8_C( -79), INT8_C(-103), INT8_C(  -9), INT8_C(-121),
                        INT8_C( -51), INT8_C(   4), INT8_C(  89), INT8_C( 110),
                        INT8_C( 112), INT8_C(  67), INT8_C(-101), INT8_C(  87)),
      simde_mm256_set_epi64x(INT64_C(                 112), INT64_C(                  67),
                             INT64_C(                -101), INT64_C(                  87)) },
    { simde_mm_set_epi8(INT8_C( -85), INT8_C( 121), INT8_C(-116), INT8_C( 121),
                        INT8_C( -43), INT8_C( -80), INT8_C(   1), INT8_C(-102),
                        INT8_C(  39), INT8_C(-105), INT8_C( -26), INT8_C(  88),
                        INT8_C( -52), INT8_C(  61), INT8_C(-127), INT8_C(  21)),
      simde_mm256_set_epi64x(INT64_C(                 -52), INT64_C(                  61),
                             INT64_C(                -127), INT64_C(                  21)) },
    { simde_mm_set_epi8(INT8_C(  85), INT8_C(   1), INT8_C(  94), INT8_C(  60),
                        INT8_C( 102), INT8_C(   9), INT8_C( -56), INT8_C(  64),
                        INT8_C(-103), INT8_C( -86), INT8_C(  51), INT8_C(-126),
                        INT8_C(  -9), INT8_C(  88), INT8_C( -52), INT8_C(  13)),
      simde_mm256_set_epi64x(INT64_C(                  -9), INT64_C(                  88),
                             INT64_C(                 -52), INT64_C(                  13)) },
    { simde_mm_set_epi8(INT8_C(   0), INT8_C( -40), INT8_C(  42), INT8_C(  42),
                        INT8_C( 109), INT8_C(  -5), INT8_C( -28), INT8_C( -80),
                        INT8_C(  70), INT8_C(   6), INT8_C( -51), INT8_C(  17),
                        INT8_C(-105), INT8_C(-108), INT8_C(  14), INT8_C( -91)),
      simde_mm256_set_epi64x(INT64_C(                -105), INT64_C(                -108),
                             INT64_C(                  14), INT64_C(                 -91)) },
    { simde_mm_set_epi8(INT8_C( 125), INT8_C(-114), INT8_C( -39), INT8_C(-118),
                        INT8_C(  29), INT8_C( -81), INT8_C(-121), INT8_C( 115),
                        INT8_C(-128), INT8_C(  99), INT8_C(  64), INT8_C(   7),
                        INT8_C( 114), INT8_C(-101), INT8_C(-127), INT8_C(  30)),
      simde_mm256_set_epi64x(INT64_C(                 114), INT64_C(                -101),
                             INT64_C(                -127), INT64_C(                  30)) },
    { simde_mm_set_epi8(INT8_C(-126), INT8_C( 122), INT8_C(  50), INT8_C( 115),
                        INT8_C(  72), INT8_C( -56), INT8_C(  52), INT8_C( 125),
                        INT8_C(-104), INT8_C( -76), INT8_C(  65), INT8_C( 103),
                        INT8_C(-122), INT8_C( -36), INT8_C( -93), INT8_C( 110)),
      simde_mm256_set_epi64x(INT64_C(                -122), INT64_C(                 -36),
                             INT64_C(                 -93), INT64_C(                 110)) },
    { simde_mm_set_epi8(INT8_C( -80), INT8_C(  98), INT8_C(-119), INT8_C(  51),
                        INT8_C( -37), INT8_C( -17), INT8_C( -94), INT8_C(-122),
                        INT8_C(-114), INT8_C(  13), INT8_C( -75), INT8_C( 111),
                        INT8_C( -76), INT8_C( -15), INT8_C(  -1), INT8_C( -12)),
      simde_mm256_set_epi64x(INT64_C(                 -76), INT64_C(                 -15),
                             INT64_C(                  -1), INT64_C(                 -12)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepi8_epi64(test_vec[i].a);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}


static int
test_simde_mm256_cvtepi16_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi16(INT16_C( 29201), INT16_C( 17763), INT16_C( 13480), INT16_C( 29487),
                         INT16_C( -6581), INT16_C( 13446), INT16_C( 26538), INT16_C( -3936)),
      simde_mm256_set_epi32(INT32_C(      29201), INT32_C(      17763), INT32_C(      13480), INT32_C(      29487),
                            INT32_C(      -6581), INT32_C(      13446), INT32_C(      26538), INT32_C(      -3936)) },
    { simde_mm_set_epi16(INT16_C( -3829), INT16_C( 12503), INT16_C( 10513), INT16_C( 11407),
                         INT16_C( 28611), INT16_C(  5727), INT16_C( 32199), INT16_C( 11434)),
      simde_mm256_set_epi32(INT32_C(      -3829), INT32_C(      12503), INT32_C(      10513), INT32_C(      11407),
                            INT32_C(      28611), INT32_C(       5727), INT32_C(      32199), INT32_C(      11434)) },
    { simde_mm_set_epi16(INT16_C( -4989), INT16_C(-23450), INT16_C(-15441), INT16_C( 22895),
                         INT16_C(  2927), INT16_C(-20709), INT16_C( -7582), INT16_C(-31347)),
      simde_mm256_set_epi32(INT32_C(      -4989), INT32_C(     -23450), INT32_C(     -15441), INT32_C(      22895),
                            INT32_C(       2927), INT32_C(     -20709), INT32_C(      -7582), INT32_C(     -31347)) },
    { simde_mm_set_epi16(INT16_C( 24635), INT16_C( 21162), INT16_C( 29781), INT16_C(  -488),
                         INT16_C( -6580), INT16_C(-27732), INT16_C(   607), INT16_C(-20782)),
      simde_mm256_set_epi32(INT32_C(      24635), INT32_C(      21162), INT32_C(      29781), INT32_C(       -488),
                            INT32_C(      -6580), INT32_C(     -27732), INT32_C(        607), INT32_C(     -20782)) },
    { simde_mm_set_epi16(INT16_C(  -401), INT16_C( -5605), INT16_C(-28681), INT16_C(  4577),
                         INT16_C( -9094), INT16_C(  6804), INT16_C( -9223), INT16_C( 10661)),
      simde_mm256_set_epi32(INT32_C(       -401), INT32_C(      -5605), INT32_C(     -28681), INT32_C(       4577),
                            INT32_C(      -9094), INT32_C(       6804), INT32_C(      -9223), INT32_C(      10661)) },
    { simde_mm_set_epi16(INT16_C(  2053), INT16_C(-19489), INT16_C(  7360), INT16_C( 16515),
                         INT16_C( 32080), INT16_C(-16555), INT16_C(-32752), INT16_C( 19001)),
      simde_mm256_set_epi32(INT32_C(       2053), INT32_C(     -19489), INT32_C(       7360), INT32_C(      16515),
                            INT32_C(      32080), INT32_C(     -16555), INT32_C(     -32752), INT32_C(      19001)) },
    { simde_mm_set_epi16(INT16_C(-31610), INT16_C(-22910), INT16_C( 22850), INT16_C(-27168),
                         INT16_C( 13092), INT16_C(-21449), INT16_C( 16024), INT16_C(  8597)),
      simde_mm256_set_epi32(INT32_C(     -31610), INT32_C(     -22910), INT32_C(      22850), INT32_C(     -27168),
                            INT32_C(      13092), INT32_C(     -21449), INT32_C(      16024), INT32_C(       8597)) },
    { simde_mm_set_epi16(INT16_C( 28081), INT16_C( -7416), INT16_C( 18632), INT16_C(-28896),
                         INT16_C(-23156), INT16_C(-15424), INT16_C(-13082), INT16_C( 23555)),
      simde_mm256_set_epi32(INT32_C(      28081), INT32_C(      -7416), INT32_C(      18632), INT32_C(     -28896),
                            INT32_C(     -23156), INT32_C(     -15424), INT32_C(     -13082), INT32_C(      23555)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepi16_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepi16_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi16(INT16_C(-31485), INT16_C(  6767), INT16_C( -1054), INT16_C(-15390),
                         INT16_C(-10897), INT16_C(-31336), INT16_C( -9551), INT16_C(  8617)),
      simde_mm256_set_epi64x(INT64_C(              -10897), INT64_C(              -31336),
                             INT64_C(               -9551), INT64_C(                8617)) },
    { simde_mm_set_epi16(INT16_C( 30988), INT16_C(  5181), INT16_C( 18938), INT16_C(-32273),
                         INT16_C(  5456), INT16_C( 19282), INT16_C(-21851), INT16_C( 27127)),
      simde_mm256_set_epi64x(INT64_C(                5456), INT64_C(               19282),
                             INT64_C(              -21851), INT64_C(               27127)) },
    { simde_mm_set_epi16(INT16_C( -7527), INT16_C(-20831), INT16_C( 23786), INT16_C(  3131),
                         INT16_C(-19886), INT16_C( -2045), INT16_C(-25062), INT16_C( -5779)),
      simde_mm256_set_epi64x(INT64_C(              -19886), INT64_C(               -2045),
                             INT64_C(              -25062), INT64_C(               -5779)) },
    { simde_mm_set_epi16(INT16_C( 31086), INT16_C( 22946), INT16_C( -1551), INT16_C(  9842),
                         INT16_C(-22586), INT16_C(  5015), INT16_C( 24465), INT16_C(-15199)),
      simde_mm256_set_epi64x(INT64_C(              -22586), INT64_C(                5015),
                             INT64_C(               24465), INT64_C(              -15199)) },
    { simde_mm_set_epi16(INT16_C( 20641), INT16_C(-16277), INT16_C( 32328), INT16_C( 21501),
                         INT16_C( 17537), INT16_C(-11745), INT16_C(-20371), INT16_C( 18200)),
      simde_mm256_set_epi64x(INT64_C(               17537), INT64_C(              -11745),
                             INT64_C(              -20371), INT64_C(               18200)) },
    { simde_mm_set_epi16(INT16_C(-26363), INT16_C(-18240), INT16_C( 21370), INT16_C(-23762),
                         INT16_C(  2610), INT16_C(  7301), INT16_C(-13295), INT16_C( 15532)),
      simde_mm256_set_epi64x(INT64_C(                2610), INT64_C(                7301),
                             INT64_C(              -13295), INT64_C(               15532)) },
    { simde_mm_set_epi16(INT16_C( -6112), INT16_C(-22921), INT16_C( -1544), INT16_C(-16225),
                         INT16_C( 14287), INT16_C(-23793), INT16_C(  5660), INT16_C(-22213)),
      simde_mm256_set_epi64x(INT64_C(               14287), INT64_C(              -23793),
                             INT64_C(                5660), INT64_C(              -22213)) },
    { simde_mm_set_epi16(INT16_C(  -130), INT16_C( -9648), INT16_C(-32446), INT16_C( 22661),
                         INT16_C( 18414), INT16_C(-28168), INT16_C( 10429), INT16_C(  9914)),
      simde_mm256_set_epi64x(INT64_C(               18414), INT64_C(              -28168),
                             INT64_C(               10429), INT64_C(                9914)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepi16_epi64(test_vec[i].a);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepi32_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C( 2035310840), INT32_C(  817509407), INT32_C( 1530478738), INT32_C( 1934275633)),
      simde_mm256_set_epi64x(INT64_C(          2035310840), INT64_C(           817509407),
                             INT64_C(          1530478738), INT64_C(          1934275633)) },
    { simde_mm_set_epi32(INT32_C(-1872979628), INT32_C( 1510684699), INT32_C( -362741060), INT32_C(   56318596)),
      simde_mm256_set_epi64x(INT64_C(         -1872979628), INT64_C(          1510684699),
                             INT64_C(          -362741060), INT64_C(            56318596)) },
    { simde_mm_set_epi32(INT32_C(  -15196779), INT32_C(    3808236), INT32_C( -178294426), INT32_C(-1675394154)),
      simde_mm256_set_epi64x(INT64_C(           -15196779), INT64_C(             3808236),
                             INT64_C(          -178294426), INT64_C(         -1675394154)) },
    { simde_mm_set_epi32(INT32_C( 1742018933), INT32_C( -523666920), INT32_C(-1989565458), INT32_C( -138158906)),
      simde_mm256_set_epi64x(INT64_C(          1742018933), INT64_C(          -523666920),
                             INT64_C(         -1989565458), INT64_C(          -138158906)) },
    { simde_mm_set_epi32(INT32_C( -851329384), INT32_C(-2094859646), INT32_C(-1382699819), INT32_C( 2104722305)),
      simde_mm256_set_epi64x(INT64_C(          -851329384), INT64_C(         -2094859646),
                             INT64_C(         -1382699819), INT64_C(          2104722305)) },
    { simde_mm_set_epi32(INT32_C( 1596198624), INT32_C(-1923442761), INT32_C( 1335983203), INT32_C(-1180029731)),
      simde_mm256_set_epi64x(INT64_C(          1596198624), INT64_C(         -1923442761),
                             INT64_C(          1335983203), INT64_C(         -1180029731)) },
    { simde_mm_set_epi32(INT32_C(  599337487), INT32_C( 1023189876), INT32_C( 1555456525), INT32_C( -191457824)),
      simde_mm256_set_epi64x(INT64_C(           599337487), INT64_C(          1023189876),
                             INT64_C(          1555456525), INT64_C(          -191457824)) },
    { simde_mm_set_epi32(INT32_C( 2100212902), INT32_C( -853030753), INT32_C( -606897046), INT32_C(-1882381199)),
      simde_mm256_set_epi64x(INT64_C(          2100212902), INT64_C(          -853030753),
                             INT64_C(          -606897046), INT64_C(         -1882381199)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepi32_epi64(test_vec[i].a);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepu8_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm_set_epu8(UINT8_C(240), UINT8_C( 50), UINT8_C(144), UINT8_C(  4),
                          UINT8_C(  7), UINT8_C(233), UINT8_C(157), UINT8_C( 74),
                          UINT8_C(217), UINT8_C( 90), UINT8_C(141), UINT8_C(254),
                          UINT8_C(115), UINT8_C( 96), UINT8_C(126), UINT8_C( 23)),
      simde_mm256_set_epi16(INT16_C(   240), INT16_C(    50), INT16_C(   144), INT16_C(     4),
                            INT16_C(     7), INT16_C(   233), INT16_C(   157), INT16_C(    74),
                            INT16_C(   217), INT16_C(    90), INT16_C(   141), INT16_C(   254),
                            INT16_C(   115), INT16_C(    96), INT16_C(   126), INT16_C(    23)) },
    { simde_x_mm_set_epu8(UINT8_C( 79), UINT8_C(126), UINT8_C(254), UINT8_C( 77),
                          UINT8_C(225), UINT8_C( 76), UINT8_C( 61), UINT8_C(223),
                          UINT8_C( 95), UINT8_C(244), UINT8_C( 66), UINT8_C(  4),
                          UINT8_C(241), UINT8_C(112), UINT8_C(120), UINT8_C(166)),
      simde_mm256_set_epi16(INT16_C(    79), INT16_C(   126), INT16_C(   254), INT16_C(    77),
                            INT16_C(   225), INT16_C(    76), INT16_C(    61), INT16_C(   223),
                            INT16_C(    95), INT16_C(   244), INT16_C(    66), INT16_C(     4),
                            INT16_C(   241), INT16_C(   112), INT16_C(   120), INT16_C(   166)) },
    { simde_x_mm_set_epu8(UINT8_C( 56), UINT8_C(107), UINT8_C(114), UINT8_C( 86),
                          UINT8_C( 35), UINT8_C( 47), UINT8_C( 25), UINT8_C(187),
                          UINT8_C(176), UINT8_C( 49), UINT8_C(164), UINT8_C(236),
                          UINT8_C(110), UINT8_C( 61), UINT8_C( 64), UINT8_C( 42)),
      simde_mm256_set_epi16(INT16_C(    56), INT16_C(   107), INT16_C(   114), INT16_C(    86),
                            INT16_C(    35), INT16_C(    47), INT16_C(    25), INT16_C(   187),
                            INT16_C(   176), INT16_C(    49), INT16_C(   164), INT16_C(   236),
                            INT16_C(   110), INT16_C(    61), INT16_C(    64), INT16_C(    42)) },
    { simde_x_mm_set_epu8(UINT8_C( 54), UINT8_C(130), UINT8_C(171), UINT8_C( 76),
                          UINT8_C( 75), UINT8_C(192), UINT8_C(152), UINT8_C(247),
                          UINT8_C(165), UINT8_C(252), UINT8_C(115), UINT8_C( 73),
                          UINT8_C( 86), UINT8_C( 52), UINT8_C( 29), UINT8_C(227)),
      simde_mm256_set_epi16(INT16_C(    54), INT16_C(   130), INT16_C(   171), INT16_C(    76),
                            INT16_C(    75), INT16_C(   192), INT16_C(   152), INT16_C(   247),
                            INT16_C(   165), INT16_C(   252), INT16_C(   115), INT16_C(    73),
                            INT16_C(    86), INT16_C(    52), INT16_C(    29), INT16_C(   227)) },
    { simde_x_mm_set_epu8(UINT8_C(175), UINT8_C(214), UINT8_C(138), UINT8_C(132),
                          UINT8_C( 59), UINT8_C( 53), UINT8_C(175), UINT8_C( 98),
                          UINT8_C(195), UINT8_C(230), UINT8_C(207), UINT8_C(189),
                          UINT8_C(100), UINT8_C(255), UINT8_C( 84), UINT8_C(129)),
      simde_mm256_set_epi16(INT16_C(   175), INT16_C(   214), INT16_C(   138), INT16_C(   132),
                            INT16_C(    59), INT16_C(    53), INT16_C(   175), INT16_C(    98),
                            INT16_C(   195), INT16_C(   230), INT16_C(   207), INT16_C(   189),
                            INT16_C(   100), INT16_C(   255), INT16_C(    84), INT16_C(   129)) },
    { simde_x_mm_set_epu8(UINT8_C( 40), UINT8_C( 10), UINT8_C(243), UINT8_C( 28),
                          UINT8_C( 48), UINT8_C(231), UINT8_C(240), UINT8_C(  2),
                          UINT8_C(  1), UINT8_C( 20), UINT8_C(184), UINT8_C(244),
                          UINT8_C(174), UINT8_C(138), UINT8_C( 47), UINT8_C(122)),
      simde_mm256_set_epi16(INT16_C(    40), INT16_C(    10), INT16_C(   243), INT16_C(    28),
                            INT16_C(    48), INT16_C(   231), INT16_C(   240), INT16_C(     2),
                            INT16_C(     1), INT16_C(    20), INT16_C(   184), INT16_C(   244),
                            INT16_C(   174), INT16_C(   138), INT16_C(    47), INT16_C(   122)) },
    { simde_x_mm_set_epu8(UINT8_C( 90), UINT8_C(144), UINT8_C(116), UINT8_C( 64),
                          UINT8_C(250), UINT8_C(233), UINT8_C(185), UINT8_C(193),
                          UINT8_C(172), UINT8_C(128), UINT8_C( 92), UINT8_C(230),
                          UINT8_C(153), UINT8_C( 17), UINT8_C(  8), UINT8_C(121)),
      simde_mm256_set_epi16(INT16_C(    90), INT16_C(   144), INT16_C(   116), INT16_C(    64),
                            INT16_C(   250), INT16_C(   233), INT16_C(   185), INT16_C(   193),
                            INT16_C(   172), INT16_C(   128), INT16_C(    92), INT16_C(   230),
                            INT16_C(   153), INT16_C(    17), INT16_C(     8), INT16_C(   121)) },
    { simde_x_mm_set_epu8(UINT8_C(135), UINT8_C(132), UINT8_C(184), UINT8_C(244),
                          UINT8_C(164), UINT8_C( 94), UINT8_C(216), UINT8_C(238),
                          UINT8_C(112), UINT8_C(252), UINT8_C( 78), UINT8_C(150),
                          UINT8_C( 72), UINT8_C(215), UINT8_C(214), UINT8_C(215)),
      simde_mm256_set_epi16(INT16_C(   135), INT16_C(   132), INT16_C(   184), INT16_C(   244),
                            INT16_C(   164), INT16_C(    94), INT16_C(   216), INT16_C(   238),
                            INT16_C(   112), INT16_C(   252), INT16_C(    78), INT16_C(   150),
                            INT16_C(    72), INT16_C(   215), INT16_C(   214), INT16_C(   215)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepu8_epi16(test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepu8_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm_set_epu8(UINT8_C(240), UINT8_C( 50), UINT8_C(144), UINT8_C(  4),
                          UINT8_C(  7), UINT8_C(233), UINT8_C(157), UINT8_C( 74),
                          UINT8_C(217), UINT8_C( 90), UINT8_C(141), UINT8_C(254),
                          UINT8_C(115), UINT8_C( 96), UINT8_C(126), UINT8_C( 23)),
      simde_mm256_set_epi32(INT32_C(        217), INT32_C(         90), INT32_C(        141), INT32_C(        254),
                            INT32_C(        115), INT32_C(         96), INT32_C(        126), INT32_C(         23)) },
    { simde_x_mm_set_epu8(UINT8_C( 79), UINT8_C(126), UINT8_C(254), UINT8_C( 77),
                          UINT8_C(225), UINT8_C( 76), UINT8_C( 61), UINT8_C(223),
                          UINT8_C( 95), UINT8_C(244), UINT8_C( 66), UINT8_C(  4),
                          UINT8_C(241), UINT8_C(112), UINT8_C(120), UINT8_C(166)),
      simde_mm256_set_epi32(INT32_C(         95), INT32_C(        244), INT32_C(         66), INT32_C(          4),
                            INT32_C(        241), INT32_C(        112), INT32_C(        120), INT32_C(        166)) },
    { simde_x_mm_set_epu8(UINT8_C( 56), UINT8_C(107), UINT8_C(114), UINT8_C( 86),
                          UINT8_C( 35), UINT8_C( 47), UINT8_C( 25), UINT8_C(187),
                          UINT8_C(176), UINT8_C( 49), UINT8_C(164), UINT8_C(236),
                          UINT8_C(110), UINT8_C( 61), UINT8_C( 64), UINT8_C( 42)),
      simde_mm256_set_epi32(INT32_C(        176), INT32_C(         49), INT32_C(        164), INT32_C(        236),
                            INT32_C(        110), INT32_C(         61), INT32_C(         64), INT32_C(         42)) },
    { simde_x_mm_set_epu8(UINT8_C( 54), UINT8_C(130), UINT8_C(171), UINT8_C( 76),
                          UINT8_C( 75), UINT8_C(192), UINT8_C(152), UINT8_C(247),
                          UINT8_C(165), UINT8_C(252), UINT8_C(115), UINT8_C( 73),
                          UINT8_C( 86), UINT8_C( 52), UINT8_C( 29), UINT8_C(227)),
      simde_mm256_set_epi32(INT32_C(        165), INT32_C(        252), INT32_C(        115), INT32_C(         73),
                            INT32_C(         86), INT32_C(         52), INT32_C(         29), INT32_C(        227)) },
    { simde_x_mm_set_epu8(UINT8_C(175), UINT8_C(214), UINT8_C(138), UINT8_C(132),
                          UINT8_C( 59), UINT8_C( 53), UINT8_C(175), UINT8_C( 98),
                          UINT8_C(195), UINT8_C(230), UINT8_C(207), UINT8_C(189),
                          UINT8_C(100), UINT8_C(255), UINT8_C( 84), UINT8_C(129)),
      simde_mm256_set_epi32(INT32_C(        195), INT32_C(        230), INT32_C(        207), INT32_C(        189),
                            INT32_C(        100), INT32_C(        255), INT32_C(         84), INT32_C(        129)) },
    { simde_x_mm_set_epu8(UINT8_C( 40), UINT8_C( 10), UINT8_C(243), UINT8_C( 28),
                          UINT8_C( 48), UINT8_C(231), UINT8_C(240), UINT8_C(  2),
                          UINT8_C(  1), UINT8_C( 20), UINT8_C(184), UINT8_C(244),
                          UINT8_C(174), UINT8_C(138), UINT8_C( 47), UINT8_C(122)),
      simde_mm256_set_epi32(INT32_C(          1), INT32_C(         20), INT32_C(        184), INT32_C(        244),
                            INT32_C(        174), INT32_C(        138), INT32_C(         47), INT32_C(        122)) },
    { simde_x_mm_set_epu8(UINT8_C( 90), UINT8_C(144), UINT8_C(116), UINT8_C( 64),
                          UINT8_C(250), UINT8_C(233), UINT8_C(185), UINT8_C(193),
                          UINT8_C(172), UINT8_C(128), UINT8_C( 92), UINT8_C(230),
                          UINT8_C(153), UINT8_C( 17), UINT8_C(  8), UINT8_C(121)),
      simde_mm256_set_epi32(INT32_C(        172), INT32_C(        128), INT32_C(         92), INT32_C(        230),
                            INT32_C(        153), INT32_C(         17), INT32_C(          8), INT32_C(        121)) },
    { simde_x_mm_set_epu8(UINT8_C(135), UINT8_C(132), UINT8_C(184), UINT8_C(244),
                          UINT8_C(164), UINT8_C( 94), UINT8_C(216), UINT8_C(238),
                          UINT8_C(112), UINT8_C(252), UINT8_C( 78), UINT8_C(150),
                          UINT8_C( 72), UINT8_C(215), UINT8_C(214), UINT8_C(215)),
      simde_mm256_set_epi32(INT32_C(        112), INT32_C(        252), INT32_C(         78), INT32_C(        150),
                            INT32_C(         72), INT32_C(        215), INT32_C(        214), INT32_C(        215)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepu8_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepu8_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm_set_epu8(UINT8_C(  9), UINT8_C(182), UINT8_C(204), UINT8_C(188),
                          UINT8_C( 95), UINT8_C(246), UINT8_C(157), UINT8_C(121),
                          UINT8_C(  1), UINT8_C( 69), UINT8_C(220), UINT8_C(154),
                          UINT8_C(132), UINT8_C(163), UINT8_C(155), UINT8_C(138)),
      simde_mm256_set_epi64x(INT64_C(                 132), INT64_C(                 163),
                             INT64_C(                 155), INT64_C(                 138)) },
    { simde_x_mm_set_epu8(UINT8_C(149), UINT8_C(222), UINT8_C(181), UINT8_C(119),
                          UINT8_C(177), UINT8_C(153), UINT8_C(247), UINT8_C(135),
                          UINT8_C(205), UINT8_C(  4), UINT8_C( 89), UINT8_C(110),
                          UINT8_C(112), UINT8_C( 67), UINT8_C(155), UINT8_C( 87)),
      simde_mm256_set_epi64x(INT64_C(                 112), INT64_C(                  67),
                             INT64_C(                 155), INT64_C(                  87)) },
    { simde_x_mm_set_epu8(UINT8_C(171), UINT8_C(121), UINT8_C(140), UINT8_C(121),
                          UINT8_C(213), UINT8_C(176), UINT8_C(  1), UINT8_C(154),
                          UINT8_C( 39), UINT8_C(151), UINT8_C(230), UINT8_C( 88),
                          UINT8_C(204), UINT8_C( 61), UINT8_C(129), UINT8_C( 21)),
      simde_mm256_set_epi64x(INT64_C(                 204), INT64_C(                  61),
                             INT64_C(                 129), INT64_C(                  21)) },
    { simde_x_mm_set_epu8(UINT8_C( 85), UINT8_C(  1), UINT8_C( 94), UINT8_C( 60),
                          UINT8_C(102), UINT8_C(  9), UINT8_C(200), UINT8_C( 64),
                          UINT8_C(153), UINT8_C(170), UINT8_C( 51), UINT8_C(130),
                          UINT8_C(247), UINT8_C( 88), UINT8_C(204), UINT8_C( 13)),
      simde_mm256_set_epi64x(INT64_C(                 247), INT64_C(                  88),
                             INT64_C(                 204), INT64_C(                  13)) },
    { simde_x_mm_set_epu8(UINT8_C(  0), UINT8_C(216), UINT8_C( 42), UINT8_C( 42),
                          UINT8_C(109), UINT8_C(251), UINT8_C(228), UINT8_C(176),
                          UINT8_C( 70), UINT8_C(  6), UINT8_C(205), UINT8_C( 17),
                          UINT8_C(151), UINT8_C(148), UINT8_C( 14), UINT8_C(165)),
      simde_mm256_set_epi64x(INT64_C(                 151), INT64_C(                 148),
                             INT64_C(                  14), INT64_C(                 165)) },
    { simde_x_mm_set_epu8(UINT8_C(125), UINT8_C(142), UINT8_C(217), UINT8_C(138),
                          UINT8_C( 29), UINT8_C(175), UINT8_C(135), UINT8_C(115),
                          UINT8_C(128), UINT8_C( 99), UINT8_C( 64), UINT8_C(  7),
                          UINT8_C(114), UINT8_C(155), UINT8_C(129), UINT8_C( 30)),
      simde_mm256_set_epi64x(INT64_C(                 114), INT64_C(                 155),
                             INT64_C(                 129), INT64_C(                  30)) },
    { simde_x_mm_set_epu8(UINT8_C(130), UINT8_C(122), UINT8_C( 50), UINT8_C(115),
                          UINT8_C( 72), UINT8_C(200), UINT8_C( 52), UINT8_C(125),
                          UINT8_C(152), UINT8_C(180), UINT8_C( 65), UINT8_C(103),
                          UINT8_C(134), UINT8_C(220), UINT8_C(163), UINT8_C(110)),
      simde_mm256_set_epi64x(INT64_C(                 134), INT64_C(                 220),
                             INT64_C(                 163), INT64_C(                 110)) },
    { simde_x_mm_set_epu8(UINT8_C(176), UINT8_C( 98), UINT8_C(137), UINT8_C( 51),
                          UINT8_C(219), UINT8_C(239), UINT8_C(162), UINT8_C(134),
                          UINT8_C(142), UINT8_C( 13), UINT8_C(181), UINT8_C(111),
                          UINT8_C(180), UINT8_C(241), UINT8_C(255), UINT8_C(244)),
      simde_mm256_set_epi64x(INT64_C(                 180), INT64_C(                 241),
                             INT64_C(                 255), INT64_C(                 244)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepu8_epi64(test_vec[i].a);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepu16_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm_set_epu16(UINT16_C(61490), UINT16_C(36868), UINT16_C( 2025), UINT16_C(40266),
                           UINT16_C(55642), UINT16_C(36350), UINT16_C(29536), UINT16_C(32279)),
      simde_mm256_set_epi32(INT32_C(      61490), INT32_C(      36868), INT32_C(       2025), INT32_C(      40266),
                            INT32_C(      55642), INT32_C(      36350), INT32_C(      29536), INT32_C(      32279)) },
    { simde_x_mm_set_epu16(UINT16_C(20350), UINT16_C(65101), UINT16_C(57676), UINT16_C(15839),
                           UINT16_C(24564), UINT16_C(16900), UINT16_C(61808), UINT16_C(30886)),
      simde_mm256_set_epi32(INT32_C(      20350), INT32_C(      65101), INT32_C(      57676), INT32_C(      15839),
                            INT32_C(      24564), INT32_C(      16900), INT32_C(      61808), INT32_C(      30886)) },
    { simde_x_mm_set_epu16(UINT16_C(14443), UINT16_C(29270), UINT16_C( 9007), UINT16_C( 6587),
                           UINT16_C(45105), UINT16_C(42220), UINT16_C(28221), UINT16_C(16426)),
      simde_mm256_set_epi32(INT32_C(      14443), INT32_C(      29270), INT32_C(       9007), INT32_C(       6587),
                            INT32_C(      45105), INT32_C(      42220), INT32_C(      28221), INT32_C(      16426)) },
    { simde_x_mm_set_epu16(UINT16_C(13954), UINT16_C(43852), UINT16_C(19392), UINT16_C(39159),
                           UINT16_C(42492), UINT16_C(29513), UINT16_C(22068), UINT16_C( 7651)),
      simde_mm256_set_epi32(INT32_C(      13954), INT32_C(      43852), INT32_C(      19392), INT32_C(      39159),
                            INT32_C(      42492), INT32_C(      29513), INT32_C(      22068), INT32_C(       7651)) },
    { simde_x_mm_set_epu16(UINT16_C(45014), UINT16_C(35460), UINT16_C(15157), UINT16_C(44898),
                           UINT16_C(50150), UINT16_C(53181), UINT16_C(25855), UINT16_C(21633)),
      simde_mm256_set_epi32(INT32_C(      45014), INT32_C(      35460), INT32_C(      15157), INT32_C(      44898),
                            INT32_C(      50150), INT32_C(      53181), INT32_C(      25855), INT32_C(      21633)) },
    { simde_x_mm_set_epu16(UINT16_C(10250), UINT16_C(62236), UINT16_C(12519), UINT16_C(61442),
                           UINT16_C(  276), UINT16_C(47348), UINT16_C(44682), UINT16_C(12154)),
      simde_mm256_set_epi32(INT32_C(      10250), INT32_C(      62236), INT32_C(      12519), INT32_C(      61442),
                            INT32_C(        276), INT32_C(      47348), INT32_C(      44682), INT32_C(      12154)) },
    { simde_x_mm_set_epu16(UINT16_C(23184), UINT16_C(29760), UINT16_C(64233), UINT16_C(47553),
                           UINT16_C(44160), UINT16_C(23782), UINT16_C(39185), UINT16_C( 2169)),
      simde_mm256_set_epi32(INT32_C(      23184), INT32_C(      29760), INT32_C(      64233), INT32_C(      47553),
                            INT32_C(      44160), INT32_C(      23782), INT32_C(      39185), INT32_C(       2169)) },
    { simde_x_mm_set_epu16(UINT16_C(34692), UINT16_C(47348), UINT16_C(42078), UINT16_C(55534),
                           UINT16_C(28924), UINT16_C(20118), UINT16_C(18647), UINT16_C(54999)),
      simde_mm256_set_epi32(INT32_C(      34692), INT32_C(      47348), INT32_C(      42078), INT32_C(      55534),
                            INT32_C(      28924), INT32_C(      20118), INT32_C(      18647), INT32_C(      54999)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepu16_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepu16_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm_set_epu16(UINT16_C(61490), UINT16_C(36868), UINT16_C( 2025), UINT16_C(40266),
                           UINT16_C(55642), UINT16_C(36350), UINT16_C(29536), UINT16_C(32279)),
      simde_mm256_set_epi64x(INT64_C(               55642), INT64_C(               36350),
                             INT64_C(               29536), INT64_C(               32279)) },
    { simde_x_mm_set_epu16(UINT16_C(20350), UINT16_C(65101), UINT16_C(57676), UINT16_C(15839),
                           UINT16_C(24564), UINT16_C(16900), UINT16_C(61808), UINT16_C(30886)),
      simde_mm256_set_epi64x(INT64_C(               24564), INT64_C(               16900),
                             INT64_C(               61808), INT64_C(               30886)) },
    { simde_x_mm_set_epu16(UINT16_C(14443), UINT16_C(29270), UINT16_C( 9007), UINT16_C( 6587),
                           UINT16_C(45105), UINT16_C(42220), UINT16_C(28221), UINT16_C(16426)),
      simde_mm256_set_epi64x(INT64_C(               45105), INT64_C(               42220),
                             INT64_C(               28221), INT64_C(               16426)) },
    { simde_x_mm_set_epu16(UINT16_C(13954), UINT16_C(43852), UINT16_C(19392), UINT16_C(39159),
                           UINT16_C(42492), UINT16_C(29513), UINT16_C(22068), UINT16_C( 7651)),
      simde_mm256_set_epi64x(INT64_C(               42492), INT64_C(               29513),
                             INT64_C(               22068), INT64_C(                7651)) },
    { simde_x_mm_set_epu16(UINT16_C(45014), UINT16_C(35460), UINT16_C(15157), UINT16_C(44898),
                           UINT16_C(50150), UINT16_C(53181), UINT16_C(25855), UINT16_C(21633)),
      simde_mm256_set_epi64x(INT64_C(               50150), INT64_C(               53181),
                             INT64_C(               25855), INT64_C(               21633)) },
    { simde_x_mm_set_epu16(UINT16_C(10250), UINT16_C(62236), UINT16_C(12519), UINT16_C(61442),
                           UINT16_C(  276), UINT16_C(47348), UINT16_C(44682), UINT16_C(12154)),
      simde_mm256_set_epi64x(INT64_C(                 276), INT64_C(               47348),
                             INT64_C(               44682), INT64_C(               12154)) },
    { simde_x_mm_set_epu16(UINT16_C(23184), UINT16_C(29760), UINT16_C(64233), UINT16_C(47553),
                           UINT16_C(44160), UINT16_C(23782), UINT16_C(39185), UINT16_C( 2169)),
      simde_mm256_set_epi64x(INT64_C(               44160), INT64_C(               23782),
                             INT64_C(               39185), INT64_C(                2169)) },
    { simde_x_mm_set_epu16(UINT16_C(34692), UINT16_C(47348), UINT16_C(42078), UINT16_C(55534),
                           UINT16_C(28924), UINT16_C(20118), UINT16_C(18647), UINT16_C(54999)),
      simde_mm256_set_epi64x(INT64_C(               28924), INT64_C(               20118),
                             INT64_C(               18647), INT64_C(               54999)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepu16_epi64(test_vec[i].a);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_cvtepu32_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm_set_epu32(UINT32_C(2027668512), UINT32_C(4262540660), UINT32_C(2279720356), UINT32_C( 579063940)),
      simde_mm256_set_epi64x(INT64_C(          2027668512), INT64_C(          4262540660),
                             INT64_C(          2279720356), INT64_C(           579063940)) },
    { simde_x_mm_set_epu32(UINT32_C( 214101781), UINT32_C(2515026933), UINT32_C(  10021235), UINT32_C(4284419101)),
      simde_mm256_set_epi64x(INT64_C(           214101781), INT64_C(          2515026933),
                             INT64_C(            10021235), INT64_C(          4284419101)) },
    { simde_x_mm_set_epu32(UINT32_C(2492492584), UINT32_C(1475674737), UINT32_C( 145233694), UINT32_C(2838555915)),
      simde_mm256_set_epi64x(INT64_C(          2492492584), INT64_C(          1475674737),
                             INT64_C(           145233694), INT64_C(          2838555915)) },
    { simde_x_mm_set_epu32(UINT32_C(3488754722), UINT32_C(2142666247), UINT32_C(4044693026), UINT32_C( 975481583)),
      simde_mm256_set_epi64x(INT64_C(          3488754722), INT64_C(          2142666247),
                             INT64_C(          4044693026), INT64_C(           975481583)) },
    { simde_x_mm_set_epu32(UINT32_C(3942926803), UINT32_C(  10053147), UINT32_C(3324554936), UINT32_C( 275092283)),
      simde_mm256_set_epi64x(INT64_C(          3942926803), INT64_C(            10053147),
                             INT64_C(          3324554936), INT64_C(           275092283)) },
    { simde_x_mm_set_epu32(UINT32_C( 609927901), UINT32_C(4032952140), UINT32_C(2163741382), UINT32_C(1197307836)),
      simde_mm256_set_epi64x(INT64_C(           609927901), INT64_C(          4032952140),
                             INT64_C(          2163741382), INT64_C(          1197307836)) },
    { simde_x_mm_set_epu32(UINT32_C( 667494753), UINT32_C(  25338810), UINT32_C( 229628292), UINT32_C(1413771580)),
      simde_mm256_set_epi64x(INT64_C(           667494753), INT64_C(            25338810),
                             INT64_C(           229628292), INT64_C(          1413771580)) },
    { simde_x_mm_set_epu32(UINT32_C(3461166204), UINT32_C(2882591041), UINT32_C(4038947223), UINT32_C(3672325978)),
      simde_mm256_set_epi64x(INT64_C(          3461166204), INT64_C(          2882591041),
                             INT64_C(          4038947223), INT64_C(          3672325978)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_cvtepu32_epi64(test_vec[i].a);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_extract_epi8(SIMDE_MUNIT_TEST_ARGS) {
  simde__m256i a;

  a = simde_mm256_set_epi8(INT8_C( 109), INT8_C( -15), INT8_C(  -7), INT8_C(  79),
                           INT8_C(  63), INT8_C(  13), INT8_C(  55), INT8_C(  74),
                           INT8_C( -46), INT8_C( 126), INT8_C(-104), INT8_C(  83),
                           INT8_C( -94), INT8_C(  46), INT8_C( -10), INT8_C( -51),
                           INT8_C( -63), INT8_C(   8), INT8_C(  47), INT8_C( 108),
                           INT8_C(  38), INT8_C( -24), INT8_C( -54), INT8_C(  83),
                           INT8_C(  41), INT8_C(  48), INT8_C(-118), INT8_C(  66),
                           INT8_C( -73), INT8_C( 110), INT8_C(  47), INT8_C(  78));
  simde_assert_equal_i8(HEDLEY_STATIC_CAST(int8_t, simde_mm256_extract_epi8(a,  1)), INT8_C(  47));

  a = simde_mm256_set_epi8(INT8_C(   9), INT8_C(  60), INT8_C( -76), INT8_C(  57),
                           INT8_C( 106), INT8_C( -35), INT8_C( -51), INT8_C(  33),
                           INT8_C(-119), INT8_C(  27), INT8_C(-103), INT8_C( -96),
                           INT8_C( 114), INT8_C(   4), INT8_C( -76), INT8_C( -65),
                           INT8_C( 105), INT8_C( -42), INT8_C(  71), INT8_C(  12),
                           INT8_C(-113), INT8_C(  32), INT8_C(-107), INT8_C( -55),
                           INT8_C(  87), INT8_C(  74), INT8_C(  33), INT8_C(  32),
                           INT8_C( -83), INT8_C( -97), INT8_C(  40), INT8_C(-118));
  simde_assert_equal_i8(HEDLEY_STATIC_CAST(int8_t, simde_mm256_extract_epi8(a, 12)), INT8_C(  12));

  a = simde_mm256_set_epi8(INT8_C(  53), INT8_C( -56), INT8_C( -74), INT8_C(  14),
                           INT8_C(  84), INT8_C(  78), INT8_C(  58), INT8_C(  56),
                           INT8_C(-126), INT8_C(  20), INT8_C(  56), INT8_C(  20),
                           INT8_C(   6), INT8_C( 127), INT8_C( 111), INT8_C(-117),
                           INT8_C( -66), INT8_C(-101), INT8_C(  45), INT8_C(   7),
                           INT8_C(-108), INT8_C( 110), INT8_C( -80), INT8_C( -37),
                           INT8_C( -87), INT8_C( -76), INT8_C( -93), INT8_C( -39),
                           INT8_C(   5), INT8_C(  40), INT8_C( 106), INT8_C( -44));
  simde_assert_equal_i8(HEDLEY_STATIC_CAST(int8_t, simde_mm256_extract_epi8(a,  2)), INT8_C(  40));

  a = simde_mm256_set_epi8(INT8_C( -54), INT8_C( -60), INT8_C(   8), INT8_C(-114),
                           INT8_C( -71), INT8_C(  78), INT8_C( -79), INT8_C(  48),
                           INT8_C( 112), INT8_C( 109), INT8_C(  51), INT8_C( -46),
                           INT8_C( -65), INT8_C( -24), INT8_C(  81), INT8_C(  65),
                           INT8_C(  88), INT8_C(  30), INT8_C( -69), INT8_C( -54),
                           INT8_C(  87), INT8_C( 123), INT8_C( 124), INT8_C( -89),
                           INT8_C(  14), INT8_C( -48), INT8_C(  73), INT8_C(  34),
                           INT8_C( -22), INT8_C( -74), INT8_C(   4), INT8_C( -22));
  simde_assert_equal_i8(HEDLEY_STATIC_CAST(int8_t, simde_mm256_extract_epi8(a,  1)), INT8_C(   4));

  a = simde_mm256_set_epi8(INT8_C( -34), INT8_C(  20), INT8_C(  68), INT8_C( -53),
                           INT8_C(  24), INT8_C( -70), INT8_C( -82), INT8_C(  20),
                           INT8_C(-104), INT8_C( -97), INT8_C( 126), INT8_C(-128),
                           INT8_C( 102), INT8_C( -37), INT8_C( -20), INT8_C(  -7),
                           INT8_C( -78), INT8_C( 110), INT8_C( -59), INT8_C(  89),
                           INT8_C( -18), INT8_C( -26), INT8_C( -89), INT8_C(  39),
                           INT8_C( -79), INT8_C( 100), INT8_C( -39), INT8_C(  76),
                           INT8_C( -51), INT8_C( -31), INT8_C(  26), INT8_C( -70));
  simde_assert_equal_i8(HEDLEY_STATIC_CAST(int8_t, simde_mm256_extract_epi8(a, 27)), INT8_C(  24));

  a = simde_mm256_set_epi8(INT8_C( -85), INT8_C( -44), INT8_C( 101), INT8_C( 109),
                           INT8_C(  58), INT8_C(  71), INT8_C(  75), INT8_C(  93),
                           INT8_C( -37), INT8_C(  91), INT8_C(   6), INT8_C(  95),
                           INT8_C( -47), INT8_C( 107), INT8_C( 114), INT8_C( -12),
                           INT8_C(  86), INT8_C(  23), INT8_C( -82), INT8_C(  84),
                           INT8_C( -80), INT8_C( -54), INT8_C(-107), INT8_C( -58),
                           INT8_C( -42), INT8_C( -79), INT8_C(  59), INT8_C( -50),
                           INT8_C(  63), INT8_C(-125), INT8_C( -96), INT8_C( -58));
  simde_assert_equal_i8(HEDLEY_STATIC_CAST(int8_t, simde_mm256_extract_epi8(a, 11)), INT8_C( -80));

  a = simde_mm256_set_epi8(INT8_C(  71), INT8_C(  18), INT8_C(  82), INT8_C( -74),
                           INT8_C(   9), INT8_C(  -6), INT8_C( -94), INT8_C( -46),
                           INT8_C(  17), INT8_C(  25), INT8_C( -11), INT8_C(-128),
                           INT8_C(-116), INT8_C(  77), INT8_C(  76), INT8_C(   8),
                           INT8_C(  80), INT8_C( -50), INT8_C(  11), INT8_C(  10),
                           INT8_C( 107), INT8_C( -99), INT8_C(  37), INT8_C(   5),
                           INT8_C( 107), INT8_C(-118), INT8_C( 119), INT8_C( -11),
                           INT8_C(  84), INT8_C(   8), INT8_C(  15), INT8_C(-102));
  simde_assert_equal_i8(HEDLEY_STATIC_CAST(int8_t, simde_mm256_extract_epi8(a,  9)), INT8_C(  37));

  a = simde_mm256_set_epi8(INT8_C(  84), INT8_C(-100), INT8_C( 116), INT8_C( -53),
                           INT8_C(-104), INT8_C(  52), INT8_C(  51), INT8_C(  60),
                           INT8_C(  91), INT8_C( 114), INT8_C(-106), INT8_C( -11),
                           INT8_C(  83), INT8_C(  57), INT8_C(   4), INT8_C(-120),
                           INT8_C(  77), INT8_C(  71), INT8_C(   1), INT8_C(  95),
                           INT8_C(  23), INT8_C(  89), INT8_C( 112), INT8_C(  -7),
                           INT8_C( -66), INT8_C(  78), INT8_C(  88), INT8_C(   5),
                           INT8_C(  75), INT8_C(  72), INT8_C( -87), INT8_C(  47));
  simde_assert_equal_i8(HEDLEY_STATIC_CAST(int8_t, simde_mm256_extract_epi8(a, 27)), INT8_C(-104));

  return 0;
}

static int
test_simde_mm256_extract_epi16(SIMDE_MUNIT_TEST_ARGS) {
  simde__m256i a;

  a = simde_mm256_set_epi16(INT16_C( -9152), INT16_C(-17321), INT16_C( -3541), INT16_C( 31629),
                            INT16_C(  4310), INT16_C(-20495), INT16_C(-28807), INT16_C(-17056),
                            INT16_C(-19221), INT16_C( 32236), INT16_C(-26695), INT16_C( -3004),
                            INT16_C( -4570), INT16_C(-14787), INT16_C(  7635), INT16_C(-23471));
  simde_assert_equal_i16(HEDLEY_STATIC_CAST(int16_t, simde_mm256_extract_epi16(a,  4)), INT16_C( -3004));

  a = simde_mm256_set_epi16(INT16_C( 10010), INT16_C(  6255), INT16_C(-20985), INT16_C( 25937),
                            INT16_C(   187), INT16_C(-14746), INT16_C(  4260), INT16_C( -5314),
                            INT16_C( 21499), INT16_C(-10726), INT16_C(-28550), INT16_C(-26957),
                            INT16_C( 25383), INT16_C(-32368), INT16_C(-28215), INT16_C(-29614));
  simde_assert_equal_i16(HEDLEY_STATIC_CAST(int16_t, simde_mm256_extract_epi16(a,  3)), INT16_C( 25383));

  a = simde_mm256_set_epi16(INT16_C( -9332), INT16_C( -3776), INT16_C(  -659), INT16_C(-11998),
                            INT16_C(-22530), INT16_C( 30025), INT16_C( -7620), INT16_C( -4652),
                            INT16_C( 12144), INT16_C(  2985), INT16_C(-29351), INT16_C(  7652),
                            INT16_C( 29358), INT16_C( -2482), INT16_C(  2031), INT16_C(-18027));
  simde_assert_equal_i16(HEDLEY_STATIC_CAST(int16_t, simde_mm256_extract_epi16(a, 12)), INT16_C(-11998));

  a = simde_mm256_set_epi16(INT16_C(  5566), INT16_C( 26716), INT16_C(-28005), INT16_C(-20328),
                            INT16_C(-22447), INT16_C(-29429), INT16_C( 18844), INT16_C(-28207),
                            INT16_C( 13304), INT16_C( 29506), INT16_C( 26640), INT16_C(-30988),
                            INT16_C(  7094), INT16_C( 15837), INT16_C( 17850), INT16_C( -4007));
  simde_assert_equal_i16(HEDLEY_STATIC_CAST(int16_t, simde_mm256_extract_epi16(a,  8)), INT16_C(-28207));

  a = simde_mm256_set_epi16(INT16_C(-22807), INT16_C( 19457), INT16_C(-27368), INT16_C(-30791),
                            INT16_C( -9280), INT16_C(-12893), INT16_C(-14741), INT16_C( -5696),
                            INT16_C(  4897), INT16_C( 24376), INT16_C(-27876), INT16_C( 29206),
                            INT16_C(-29469), INT16_C(-22113), INT16_C( 23067), INT16_C(  6063));
  simde_assert_equal_i16(HEDLEY_STATIC_CAST(int16_t, simde_mm256_extract_epi16(a,  3)), INT16_C(-29469));

  a = simde_mm256_set_epi16(INT16_C( -6124), INT16_C(  1398), INT16_C(   383), INT16_C(  1785),
                            INT16_C(-30725), INT16_C(-20290), INT16_C( 11961), INT16_C(-25919),
                            INT16_C(-25742), INT16_C(-31321), INT16_C( -6067), INT16_C( 17059),
                            INT16_C( 16943), INT16_C( 31917), INT16_C( 18056), INT16_C(-23274));
  simde_assert_equal_i16(HEDLEY_STATIC_CAST(int16_t, simde_mm256_extract_epi16(a,  7)), INT16_C(-25742));

  a = simde_mm256_set_epi16(INT16_C( 29739), INT16_C(-15442), INT16_C(-18760), INT16_C(  3107),
                            INT16_C(-25745), INT16_C(    -8), INT16_C( -5543), INT16_C(  3310),
                            INT16_C( 13765), INT16_C(-29431), INT16_C( 31912), INT16_C( 20910),
                            INT16_C(-32282), INT16_C(  8881), INT16_C(-19453), INT16_C(-25360));
  simde_assert_equal_i16(HEDLEY_STATIC_CAST(int16_t, simde_mm256_extract_epi16(a,  0)), INT16_C(-25360));

  a = simde_mm256_set_epi16(INT16_C( 19041), INT16_C( 28526), INT16_C(   632), INT16_C( 12890),
                            INT16_C( -9054), INT16_C(  9044), INT16_C(-24624), INT16_C(-18390),
                            INT16_C( -7692), INT16_C( 20879), INT16_C(  2760), INT16_C( 29306),
                            INT16_C(  5480), INT16_C( 22577), INT16_C(-15668), INT16_C(-16497));
  simde_assert_equal_i16(HEDLEY_STATIC_CAST(int16_t, simde_mm256_extract_epi16(a,  0)), INT16_C(-16497));

  return 0;
}

static int
test_simde_mm256_extracti128_si256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m128i ra;
    simde__m128i rb;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C( 1229394801), INT32_C(  992221618), INT32_C(-1388107406), INT32_C(  780445625),
                            INT32_C( 1795700153), INT32_C( -297324271), INT32_C( 1549329146), INT32_C( -534963225)),
      simde_mm256_set_epi32(INT32_C( -867719772), INT32_C(-1804212438), INT32_C( 1849818353), INT32_C(  405560893),
                            INT32_C( 1351338386), INT32_C( -886724662), INT32_C( 1197680760), INT32_C( 1003042592)),
      simde_mm_set_epi32(INT32_C( 1795700153), INT32_C( -297324271), INT32_C( 1549329146), INT32_C( -534963225)),
      simde_mm_set_epi32(INT32_C( 1229394801), INT32_C(  992221618), INT32_C(-1388107406), INT32_C(  780445625)) },
    { simde_mm256_set_epi32(INT32_C( 1839397279), INT32_C( -585358334), INT32_C(  779178160), INT32_C( -362976984),
                            INT32_C(-1015866216), INT32_C( -556342867), INT32_C( -766379029), INT32_C( -130781639)),
      simde_mm256_set_epi32(INT32_C( 2021215895), INT32_C( -422044488), INT32_C(-1385706777), INT32_C(   22702781),
                            INT32_C( 1076807274), INT32_C(-1923875401), INT32_C( -554846936), INT32_C(-1008226174)),
      simde_mm_set_epi32(INT32_C(-1015866216), INT32_C( -556342867), INT32_C( -766379029), INT32_C( -130781639)),
      simde_mm_set_epi32(INT32_C( 1839397279), INT32_C( -585358334), INT32_C(  779178160), INT32_C( -362976984)) },
    { simde_mm256_set_epi32(INT32_C(   23865749), INT32_C( -908972624), INT32_C(-1642418179), INT32_C(  111689864),
                            INT32_C( -835154412), INT32_C( -431540196), INT32_C( -161564683), INT32_C(  204589457)),
      simde_mm256_set_epi32(INT32_C( 2088662618), INT32_C(-1671363325), INT32_C( -997695043), INT32_C( -809764814),
                            INT32_C( 2092581708), INT32_C(-1073689737), INT32_C(-1556963227), INT32_C( -641330488)),
      simde_mm_set_epi32(INT32_C( -835154412), INT32_C( -431540196), INT32_C( -161564683), INT32_C(  204589457)),
      simde_mm_set_epi32(INT32_C(   23865749), INT32_C( -908972624), INT32_C(-1642418179), INT32_C(  111689864)) },
    { simde_mm256_set_epi32(INT32_C( 1727711569), INT32_C(-1915329589), INT32_C( -979233658), INT32_C( -409203179),
                            INT32_C( 1343207861), INT32_C(-1541174422), INT32_C(-2097250480), INT32_C(-1382492089)),
      simde_mm256_set_epi32(INT32_C(  -42175512), INT32_C(-2146588690), INT32_C(-1902868938), INT32_C( 1919945739),
                            INT32_C(  410749235), INT32_C(-1828962645), INT32_C(  525862553), INT32_C( -282512400)),
      simde_mm_set_epi32(INT32_C( 1343207861), INT32_C(-1541174422), INT32_C(-2097250480), INT32_C(-1382492089)),
      simde_mm_set_epi32(INT32_C( 1727711569), INT32_C(-1915329589), INT32_C( -979233658), INT32_C( -409203179)) },
    { simde_mm256_set_epi32(INT32_C(  377369527), INT32_C( 1159197718), INT32_C(  288677560), INT32_C(  828517622),
                            INT32_C( 1815109517), INT32_C( 1103735854), INT32_C( 1342116414), INT32_C( 1750949195)),
      simde_mm256_set_epi32(INT32_C( 1481955155), INT32_C( -119794855), INT32_C(-2109995042), INT32_C(  582656481),
                            INT32_C( 1178951500), INT32_C(  762286037), INT32_C(  628377158), INT32_C( -188026020)),
      simde_mm_set_epi32(INT32_C( 1815109517), INT32_C( 1103735854), INT32_C( 1342116414), INT32_C( 1750949195)),
      simde_mm_set_epi32(INT32_C(  377369527), INT32_C( 1159197718), INT32_C(  288677560), INT32_C(  828517622)) },
    { simde_mm256_set_epi32(INT32_C(-1996051424), INT32_C( -314294760), INT32_C( -770521150), INT32_C(  508113145),
                            INT32_C( -677093043), INT32_C( -527636644), INT32_C( 1238565466), INT32_C(-1592387355)),
      simde_mm256_set_epi32(INT32_C(-1510707643), INT32_C( 1988531398), INT32_C(-1182276921), INT32_C(  363503044),
                            INT32_C( 2086268932), INT32_C( -428647595), INT32_C( 1685321543), INT32_C( 1979089365)),
      simde_mm_set_epi32(INT32_C( -677093043), INT32_C( -527636644), INT32_C( 1238565466), INT32_C(-1592387355)),
      simde_mm_set_epi32(INT32_C(-1996051424), INT32_C( -314294760), INT32_C( -770521150), INT32_C(  508113145)) },
    { simde_mm256_set_epi32(INT32_C(    4593159), INT32_C( 1779671737), INT32_C( -569674634), INT32_C( -184254965),
                            INT32_C( -665786654), INT32_C(  663766301), INT32_C(-1237697897), INT32_C( -260948936)),
      simde_mm256_set_epi32(INT32_C( -575114102), INT32_C( -399786699), INT32_C(-1468780124), INT32_C( 2032090700),
                            INT32_C(  723386747), INT32_C(-1766232746), INT32_C(   73837413), INT32_C(  496540408)),
      simde_mm_set_epi32(INT32_C( -665786654), INT32_C(  663766301), INT32_C(-1237697897), INT32_C( -260948936)),
      simde_mm_set_epi32(INT32_C(    4593159), INT32_C( 1779671737), INT32_C( -569674634), INT32_C( -184254965)) },
    { simde_mm256_set_epi32(INT32_C( -328197013), INT32_C( 1036318270), INT32_C(-1930293157), INT32_C( 1948339432),
                            INT32_C( 1903716614), INT32_C(-1951673698), INT32_C(-1858071379), INT32_C( 2070124471)),
      simde_mm256_set_epi32(INT32_C(-1815372819), INT32_C( -102535612), INT32_C(  115383384), INT32_C( 1004544095),
                            INT32_C( 1506420054), INT32_C(-1014523798), INT32_C(-1776388104), INT32_C( 1550371104)),
      simde_mm_set_epi32(INT32_C( 1903716614), INT32_C(-1951673698), INT32_C(-1858071379), INT32_C( 2070124471)),
      simde_mm_set_epi32(INT32_C( -328197013), INT32_C( 1036318270), INT32_C(-1930293157), INT32_C( 1948339432)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i ra = simde_mm256_extracti128_si256(test_vec[i].a, 0);
    simde__m128i rb = simde_mm256_extracti128_si256(test_vec[i].a, 1);
    simde_assert_m128i_i32(ra, ==, test_vec[i].ra);
    simde_assert_m128i_i32(rb, ==, test_vec[i].rb);
  }

  return 0;
}

static int
test_simde_mm256_hadd_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(-18444), INT16_C(-18028), INT16_C( 29026), INT16_C(  2084),
                            INT16_C(  4112), INT16_C(-30013), INT16_C( 26536), INT16_C(-22613),
                            INT16_C( 28256), INT16_C(-13992), INT16_C(  1054), INT16_C(-17387),
                            INT16_C( 28361), INT16_C(-13892), INT16_C(-23621), INT16_C( -6864)),
      simde_mm256_set_epi16(INT16_C( -2077), INT16_C( 32093), INT16_C(-32096), INT16_C( -1919),
                            INT16_C(-29576), INT16_C(  1752), INT16_C( -9460), INT16_C( 18637),
                            INT16_C( 12808), INT16_C(  5796), INT16_C(  8151), INT16_C( -2897),
                            INT16_C(-24515), INT16_C(-31464), INT16_C(-13532), INT16_C( -9881)),
      simde_mm256_set_epi16(INT16_C( 30016), INT16_C( 31521), INT16_C(-27824), INT16_C(  9177),
                            INT16_C( 29064), INT16_C( 31110), INT16_C(-25901), INT16_C(  3923),
                            INT16_C( 18604), INT16_C(  5254), INT16_C(  9557), INT16_C(-23413),
                            INT16_C( 14264), INT16_C(-16333), INT16_C( 14469), INT16_C(-30485)) },
    { simde_mm256_set_epi16(INT16_C(  9100), INT16_C( -1983), INT16_C( 28981), INT16_C( 31131),
                            INT16_C( -8267), INT16_C(  9829), INT16_C(-19843), INT16_C( 28705),
                            INT16_C( 15872), INT16_C( 21932), INT16_C( -9103), INT16_C( 20935),
                            INT16_C(-24996), INT16_C( 13650), INT16_C( 30282), INT16_C(-21973)),
      simde_mm256_set_epi16(INT16_C( -9238), INT16_C(-15000), INT16_C( 28358), INT16_C(-31774),
                            INT16_C( 20723), INT16_C( 27208), INT16_C(-20512), INT16_C( 10808),
                            INT16_C( 17124), INT16_C( 20983), INT16_C(   113), INT16_C( 22835),
                            INT16_C(  -190), INT16_C(-28607), INT16_C( 22616), INT16_C(  4805)),
      simde_mm256_set_epi16(INT16_C(-24238), INT16_C( -3416), INT16_C(-17605), INT16_C( -9704),
                            INT16_C(  7117), INT16_C( -5424), INT16_C(  1562), INT16_C(  8862),
                            INT16_C(-27429), INT16_C( 22948), INT16_C(-28797), INT16_C( 27421),
                            INT16_C(-27732), INT16_C( 11832), INT16_C(-11346), INT16_C(  8309)) },
    { simde_mm256_set_epi16(INT16_C( 26837), INT16_C( 17774), INT16_C(-32310), INT16_C( -9579),
                            INT16_C(  3637), INT16_C(-14106), INT16_C( 30046), INT16_C(-13930),
                            INT16_C( -1887), INT16_C( 13772), INT16_C( 19874), INT16_C(-18102),
                            INT16_C(-11204), INT16_C( -4897), INT16_C(  3260), INT16_C(-17962)),
      simde_mm256_set_epi16(INT16_C( 22389), INT16_C(-14730), INT16_C( 31871), INT16_C( 17642),
                            INT16_C(-26185), INT16_C( -2982), INT16_C(-14158), INT16_C(  4590),
                            INT16_C( 20601), INT16_C( 21976), INT16_C( 25432), INT16_C( 31010),
                            INT16_C(-23314), INT16_C( -9816), INT16_C(  8140), INT16_C( -7858)),
      simde_mm256_set_epi16(INT16_C(  7659), INT16_C(-16023), INT16_C(-29167), INT16_C( -9568),
                            INT16_C(-20925), INT16_C( 23647), INT16_C(-10469), INT16_C( 16116),
                            INT16_C(-22959), INT16_C( -9094), INT16_C( 32406), INT16_C(   282),
                            INT16_C( 11885), INT16_C(  1772), INT16_C(-16101), INT16_C(-14702)) },
    { simde_mm256_set_epi16(INT16_C( 12057), INT16_C(-24517), INT16_C( -9967), INT16_C( -7239),
                            INT16_C( -2143), INT16_C(-23349), INT16_C(-25096), INT16_C(-21587),
                            INT16_C(  -802), INT16_C( 25377), INT16_C(-22001), INT16_C( 28281),
                            INT16_C(-28768), INT16_C(-19834), INT16_C(  8025), INT16_C(-23943)),
      simde_mm256_set_epi16(INT16_C(  6731), INT16_C(-13423), INT16_C( -6728), INT16_C(-19678),
                            INT16_C(-29476), INT16_C( -9124), INT16_C(-21742), INT16_C( 14418),
                            INT16_C( 27306), INT16_C( 25972), INT16_C(-12648), INT16_C(-16450),
                            INT16_C(-25118), INT16_C(  4239), INT16_C(-17383), INT16_C(-20693)),
      simde_mm256_set_epi16(INT16_C( -6692), INT16_C(-26406), INT16_C( 26936), INT16_C( -7324),
                            INT16_C(-12460), INT16_C(-17206), INT16_C(-25492), INT16_C( 18853),
                            INT16_C(-12258), INT16_C(-29098), INT16_C(-20879), INT16_C( 27460),
                            INT16_C( 24575), INT16_C(  6280), INT16_C( 16934), INT16_C(-15918)) },
    { simde_mm256_set_epi16(INT16_C(-15868), INT16_C(-27060), INT16_C(   528), INT16_C(-29935),
                            INT16_C(-12298), INT16_C( 18504), INT16_C( -5289), INT16_C(   430),
                            INT16_C(-29328), INT16_C(  3228), INT16_C( 18568), INT16_C(-13568),
                            INT16_C( -2471), INT16_C( -1530), INT16_C(  5334), INT16_C( 31888)),
      simde_mm256_set_epi16(INT16_C( -2903), INT16_C(-32300), INT16_C( 29546), INT16_C( 12096),
                            INT16_C(-31860), INT16_C(-30618), INT16_C(  1236), INT16_C( -9131),
                            INT16_C(  -497), INT16_C(-22865), INT16_C( 30177), INT16_C( 29201),
                            INT16_C(-26632), INT16_C(-25915), INT16_C( 14957), INT16_C(  6108)),
      simde_mm256_set_epi16(INT16_C( 30333), INT16_C(-23894), INT16_C(  3058), INT16_C( -7895),
                            INT16_C( 22608), INT16_C(-29407), INT16_C(  6206), INT16_C( -4859),
                            INT16_C(-23362), INT16_C( -6158), INT16_C( 12989), INT16_C( 21065),
                            INT16_C(-26100), INT16_C(  5000), INT16_C( -4001), INT16_C(-28314)) },
    { simde_mm256_set_epi16(INT16_C(  4205), INT16_C( 23268), INT16_C(-32329), INT16_C(-20305),
                            INT16_C( -9331), INT16_C( -4387), INT16_C(-30285), INT16_C( 13621),
                            INT16_C( 15818), INT16_C( 16370), INT16_C(-24200), INT16_C( 10901),
                            INT16_C( 17049), INT16_C( 14552), INT16_C( -5992), INT16_C(-17203)),
      simde_mm256_set_epi16(INT16_C( 15660), INT16_C( 25069), INT16_C(-11965), INT16_C( -9994),
                            INT16_C( -7271), INT16_C( 17410), INT16_C( 14381), INT16_C(-30587),
                            INT16_C( 11396), INT16_C(-22996), INT16_C( 26099), INT16_C(-11680),
                            INT16_C(-18604), INT16_C( 20100), INT16_C(  5959), INT16_C(-27899)),
      simde_mm256_set_epi16(INT16_C(-24807), INT16_C(-21959), INT16_C( 10139), INT16_C(-16206),
                            INT16_C( 27473), INT16_C( 12902), INT16_C(-13718), INT16_C(-16664),
                            INT16_C(-11600), INT16_C( 14419), INT16_C(  1496), INT16_C(-21940),
                            INT16_C( 32188), INT16_C(-13299), INT16_C( 31601), INT16_C(-23195)) },
    { simde_mm256_set_epi16(INT16_C( 27483), INT16_C(  -502), INT16_C(   138), INT16_C( -7690),
                            INT16_C( 19176), INT16_C(-26261), INT16_C(-11443), INT16_C(-23576),
                            INT16_C( 22206), INT16_C(-25802), INT16_C( 16334), INT16_C(-18863),
                            INT16_C( 31357), INT16_C(-31063), INT16_C( 28401), INT16_C(-12622)),
      simde_mm256_set_epi16(INT16_C( 13717), INT16_C( 31993), INT16_C( 16367), INT16_C(-21175),
                            INT16_C( 23569), INT16_C( -7701), INT16_C( -1073), INT16_C( 24510),
                            INT16_C( 22852), INT16_C( 29964), INT16_C( -1882), INT16_C( -8354),
                            INT16_C(-16212), INT16_C( 25899), INT16_C( 11699), INT16_C(-20150)),
      simde_mm256_set_epi16(INT16_C(-19826), INT16_C( -4808), INT16_C( 15868), INT16_C( 23437),
                            INT16_C( 26981), INT16_C( -7552), INT16_C( -7085), INT16_C( 30517),
                            INT16_C(-12720), INT16_C(-10236), INT16_C(  9687), INT16_C( -8451),
                            INT16_C( -3596), INT16_C( -2529), INT16_C(   294), INT16_C( 15779)) },
    { simde_mm256_set_epi16(INT16_C( -5381), INT16_C(-10578), INT16_C(-31523), INT16_C(-23458),
                            INT16_C( 22155), INT16_C( 10052), INT16_C( 22251), INT16_C(-22857),
                            INT16_C(  -468), INT16_C(  4672), INT16_C(-23974), INT16_C(-31691),
                            INT16_C( 11898), INT16_C(  4678), INT16_C(  5316), INT16_C(-17657)),
      simde_mm256_set_epi16(INT16_C( 15026), INT16_C(-17248), INT16_C(-13886), INT16_C( 26220),
                            INT16_C( -1198), INT16_C( -7005), INT16_C(-21616), INT16_C(-30390),
                            INT16_C(-30119), INT16_C(-22301), INT16_C(-29984), INT16_C(  6696),
                            INT16_C( 12300), INT16_C( 16913), INT16_C( 12770), INT16_C(  3850)),
      simde_mm256_set_epi16(INT16_C( -2222), INT16_C( 12334), INT16_C( -8203), INT16_C( 13530),
                            INT16_C(-15959), INT16_C( 10555), INT16_C( 32207), INT16_C(  -606),
                            INT16_C( 13116), INT16_C(-23288), INT16_C( 29213), INT16_C( 16620),
                            INT16_C(  4204), INT16_C(  9871), INT16_C( 16576), INT16_C(-12341)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_hadd_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_hadd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(-1208698476), INT32_C( 1902250020), INT32_C(  269519555), INT32_C( 1739106219),
                            INT32_C( 1851836760), INT32_C(   69123093), INT32_C( 1858718140), INT32_C(-1547967184)),
      simde_mm256_set_epi32(INT32_C( -136086179), INT32_C(-2103379839), INT32_C(-1938290984), INT32_C( -619951923),
                            INT32_C(  839390884), INT32_C(  534246575), INT32_C(-1606580968), INT32_C( -886777497)),
      simde_mm256_set_epi32(INT32_C( 2055501278), INT32_C( 1736724389), INT32_C(  693551544), INT32_C( 2008625774),
                            INT32_C( 1373637459), INT32_C( 1801608831), INT32_C( 1920959853), INT32_C(  310750956)) },
    { simde_mm256_set_epi32(INT32_C(  596441153), INT32_C( 1899329947), INT32_C( -541776283), INT32_C(-1300402143),
                            INT32_C( 1040209324), INT32_C( -596553273), INT32_C(-1638124206), INT32_C( 1984604715)),
      simde_mm256_set_epi32(INT32_C( -605371032), INT32_C( 1858503650), INT32_C( 1358129736), INT32_C(-1344263624),
                            INT32_C( 1122259447), INT32_C(    7428403), INT32_C(  -12414911), INT32_C( 1482166981)),
      simde_mm256_set_epi32(INT32_C( 1253132618), INT32_C(   13866112), INT32_C(-1799196196), INT32_C(-1842178426),
                            INT32_C( 1129687850), INT32_C( 1469752070), INT32_C(  443656051), INT32_C(  346480509)) },
    { simde_mm256_set_epi32(INT32_C( 1758807406), INT32_C(-2117412203), INT32_C(  238405862), INT32_C( 1969146262),
                            INT32_C( -123652660), INT32_C( 1302509898), INT32_C( -734204705), INT32_C(  213694934)),
      simde_mm256_set_epi32(INT32_C( 1467336310), INT32_C( 2088715498), INT32_C(-1715997606), INT32_C( -927854098),
                            INT32_C( 1350129112), INT32_C( 1666742562), INT32_C(-1527850584), INT32_C(  533520718)),
      simde_mm256_set_epi32(INT32_C( -738915488), INT32_C( 1651115592), INT32_C( -358604797), INT32_C(-2087415172),
                            INT32_C(-1278095622), INT32_C( -994329866), INT32_C( 1178857238), INT32_C( -520509771)) },
    { simde_mm256_set_epi32(INT32_C(  790208571), INT32_C( -653139015), INT32_C( -140401461), INT32_C(-1644647507),
                            INT32_C(  -52534495), INT32_C(-1441829255), INT32_C(-1885293946), INT32_C(  525967993)),
      simde_mm256_set_epi32(INT32_C(  441174929), INT32_C( -440880350), INT32_C(-1931682724), INT32_C(-1424869294),
                            INT32_C( 1789551988), INT32_C( -828850242), INT32_C(-1646129009), INT32_C(-1139167445)),
      simde_mm256_set_epi32(INT32_C(     294579), INT32_C(  938415278), INT32_C(  137069556), INT32_C(-1785048968),
                            INT32_C(  960701746), INT32_C( 1509670842), INT32_C(-1494363750), INT32_C(-1359325953)) },
    { simde_mm256_set_epi32(INT32_C(-1039886772), INT32_C(   34638609), INT32_C( -805943224), INT32_C( -346619474),
                            INT32_C(-1922036580), INT32_C( 1216924416), INT32_C( -161875450), INT32_C(  349600912)),
      simde_mm256_set_epi32(INT32_C( -190217772), INT32_C( 1936338752), INT32_C(-2087942042), INT32_C(   81058901),
                            INT32_C(  -32528721), INT32_C( 1977709073), INT32_C(-1745315131), INT32_C(  980228060)),
      simde_mm256_set_epi32(INT32_C( 1746120980), INT32_C(-2006883141), INT32_C(-1005248163), INT32_C(-1152562698),
                            INT32_C( 1945180352), INT32_C( -765087071), INT32_C( -705112164), INT32_C(  187725462)) },
    { simde_mm256_set_epi32(INT32_C(  275602148), INT32_C(-2118668113), INT32_C( -611455267), INT32_C(-1984744139),
                            INT32_C( 1036664818), INT32_C(-1585960299), INT32_C( 1117337816), INT32_C( -392643379)),
      simde_mm256_set_epi32(INT32_C( 1026318829), INT32_C( -784082698), INT32_C( -476494846), INT32_C(  942508165),
                            INT32_C(  746890796), INT32_C( 1710477920), INT32_C(-1219211644), INT32_C(  390566661)),
      simde_mm256_set_epi32(INT32_C(  242236131), INT32_C(  466013319), INT32_C(-1843065965), INT32_C( 1698767890),
                            INT32_C(-1837598580), INT32_C( -828644983), INT32_C( -549295481), INT32_C(  724694437)) },
    { simde_mm256_set_epi32(INT32_C( 1801190922), INT32_C(    9101814), INT32_C( 1256757611), INT32_C( -749886488),
                            INT32_C( 1455332150), INT32_C( 1070511697), INT32_C( 2055046825), INT32_C( 1861340850)),
      simde_mm256_set_epi32(INT32_C(  898989305), INT32_C( 1072672073), INT32_C( 1544675819), INT32_C(  -70295618),
                            INT32_C( 1497658636), INT32_C( -123281570), INT32_C(-1062443733), INT32_C(  766751050)),
      simde_mm256_set_epi32(INT32_C( 1971661378), INT32_C( 1474380201), INT32_C( 1810292736), INT32_C(  506871123),
                            INT32_C( 1374377066), INT32_C( -295692683), INT32_C(-1769123449), INT32_C( -378579621)) },
    { simde_mm256_set_epi32(INT32_C( -352594258), INT32_C(-2065849250), INT32_C( 1451960132), INT32_C( 1458284215),
                            INT32_C(  -30666176), INT32_C(-1571126219), INT32_C(  779752006), INT32_C(  348437255)),
      simde_mm256_set_epi32(INT32_C(  984792224), INT32_C( -910006676), INT32_C(  -78453597), INT32_C(-1416591030),
                            INT32_C(-1973835549), INT32_C(-1965024728), INT32_C(  806109713), INT32_C(  836898570)),
      simde_mm256_set_epi32(INT32_C(   74785548), INT32_C(-1495044627), INT32_C( 1876523788), INT32_C(-1384722949),
                            INT32_C(  356107019), INT32_C( 1643008283), INT32_C(-1601792395), INT32_C( 1128189261)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_hadd_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_hadds_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[16];
    const int16_t b[16];
    const int16_t r[16];
  } test_vec[] = {
    { { -INT16_C( 10867),  INT16_C(  3622), -INT16_C( 28779), -INT16_C( 15004),  INT16_C( 21958),  INT16_C( 28225), -INT16_C(  9443), -INT16_C( 17328),
         INT16_C(  9723),  INT16_C( 22832),  INT16_C( 14776),  INT16_C( 12956),  INT16_C( 11112), -INT16_C( 20463),  INT16_C( 32154),  INT16_C( 10109) },
      { -INT16_C( 23726), -INT16_C(  6346), -INT16_C( 26062), -INT16_C(  1876), -INT16_C(  4369),  INT16_C(  3175), -INT16_C( 18487), -INT16_C( 14904),
        -INT16_C(  1572), -INT16_C( 27362), -INT16_C( 17614), -INT16_C( 25913), -INT16_C( 10010), -INT16_C( 32693), -INT16_C( 14251), -INT16_C( 22360) },
      { -INT16_C(  7245),       INT16_MIN,       INT16_MAX, -INT16_C( 26771), -INT16_C( 30072), -INT16_C( 27938), -INT16_C(  1194),       INT16_MIN,
         INT16_C( 32555),  INT16_C( 27732), -INT16_C(  9351),       INT16_MAX, -INT16_C( 28934),       INT16_MIN,       INT16_MIN,       INT16_MIN } },
    { { -INT16_C(  8596), -INT16_C( 24945),  INT16_C( 15480),  INT16_C( 26519), -INT16_C(   470), -INT16_C(  3213),  INT16_C( 15285), -INT16_C( 28232),
        -INT16_C( 10444),  INT16_C( 26406), -INT16_C(  4718),  INT16_C( 30721),  INT16_C( 19654),  INT16_C(  7160), -INT16_C( 24555), -INT16_C( 32317) },
      {  INT16_C( 21374), -INT16_C(  2529), -INT16_C( 18801), -INT16_C( 18083), -INT16_C( 12108),  INT16_C( 27052),  INT16_C( 25868),  INT16_C( 16635),
         INT16_C(  8508), -INT16_C( 12633), -INT16_C( 22257), -INT16_C( 10938),  INT16_C( 16117),  INT16_C(  2800), -INT16_C( 19233),  INT16_C( 23947) },
      {       INT16_MIN,       INT16_MAX, -INT16_C(  3683), -INT16_C( 12947),  INT16_C( 18845),       INT16_MIN,  INT16_C( 14944),       INT16_MAX,
         INT16_C( 15962),  INT16_C( 26003),  INT16_C( 26814),       INT16_MIN, -INT16_C(  4125),       INT16_MIN,  INT16_C( 18917),  INT16_C(  4714) } },
    { { -INT16_C( 21753), -INT16_C( 27052), -INT16_C( 20127),  INT16_C(  5711), -INT16_C(  1150), -INT16_C( 29057),  INT16_C( 31328), -INT16_C( 25394),
         INT16_C( 30364), -INT16_C( 21654), -INT16_C( 20449),  INT16_C(  5248),  INT16_C( 28911), -INT16_C( 12769), -INT16_C( 21980),  INT16_C( 11051) },
      {  INT16_C( 32597), -INT16_C( 18495),  INT16_C(  4145), -INT16_C( 19507),  INT16_C( 19468),  INT16_C( 27713),  INT16_C(  4039),  INT16_C( 25353),
         INT16_C( 29573), -INT16_C( 23538), -INT16_C( 29148),  INT16_C(  5049), -INT16_C(  9986),  INT16_C(  9185),  INT16_C(  3202), -INT16_C( 10162) },
      {       INT16_MIN, -INT16_C( 14416), -INT16_C( 30207),  INT16_C(  5934),  INT16_C( 14102), -INT16_C( 15362),       INT16_MAX,  INT16_C( 29392),
         INT16_C(  8710), -INT16_C( 15201),  INT16_C( 16142), -INT16_C( 10929),  INT16_C(  6035), -INT16_C( 24099), -INT16_C(   801), -INT16_C(  6960) } },
    { {  INT16_C(  4236), -INT16_C( 17009),  INT16_C( 23584),  INT16_C( 11376), -INT16_C( 20056),  INT16_C( 28569), -INT16_C( 23872),  INT16_C( 18130),
        -INT16_C(  8171),  INT16_C( 14826), -INT16_C( 23698),  INT16_C( 27980),  INT16_C( 11643), -INT16_C(   368), -INT16_C(  8646), -INT16_C( 14634) },
      {  INT16_C( 26094),  INT16_C(  3971), -INT16_C(  3135),  INT16_C( 26939), -INT16_C( 11100),  INT16_C( 25817), -INT16_C( 21642), -INT16_C( 29526),
        -INT16_C( 27252), -INT16_C(  1339),  INT16_C(  4664), -INT16_C( 19353), -INT16_C(  2241),  INT16_C( 31154), -INT16_C( 30506), -INT16_C( 15297) },
      { -INT16_C( 12773),       INT16_MAX,  INT16_C(  8513), -INT16_C(  5742),  INT16_C( 30065),  INT16_C( 23804),  INT16_C( 14717),       INT16_MIN,
         INT16_C(  6655),  INT16_C(  4282),  INT16_C( 11275), -INT16_C( 23280), -INT16_C( 28591), -INT16_C( 14689),  INT16_C( 28913),       INT16_MIN } },
    { { -INT16_C( 15635), -INT16_C( 20781),  INT16_C(  4021),  INT16_C( 22807), -INT16_C(  3869),  INT16_C( 23230),  INT16_C( 26780),  INT16_C( 10470),
        -INT16_C( 21507),  INT16_C( 13858), -INT16_C( 30019), -INT16_C(   534), -INT16_C( 25471),  INT16_C( 22390), -INT16_C( 18908),  INT16_C(  4380) },
      { -INT16_C(  4232),  INT16_C( 11967), -INT16_C( 10498), -INT16_C(  7545),  INT16_C( 17863),  INT16_C( 25404),  INT16_C(  8878), -INT16_C( 21621),
        -INT16_C( 21043), -INT16_C( 29727), -INT16_C( 13513), -INT16_C( 18040), -INT16_C(   409), -INT16_C( 29936),  INT16_C( 11444),  INT16_C( 11676) },
      {       INT16_MIN,  INT16_C( 26828),  INT16_C( 19361),       INT16_MAX,  INT16_C(  7735), -INT16_C( 18043),       INT16_MAX, -INT16_C( 12743),
        -INT16_C(  7649), -INT16_C( 30553), -INT16_C(  3081), -INT16_C( 14528),       INT16_MIN, -INT16_C( 31553), -INT16_C( 30345),  INT16_C( 23120) } },
    { {  INT16_C( 23324),  INT16_C(  6747), -INT16_C(  7630), -INT16_C(  1540),  INT16_C( 14376), -INT16_C( 10660), -INT16_C(  6310),  INT16_C( 10369),
         INT16_C( 25492), -INT16_C( 13133),  INT16_C( 15150), -INT16_C( 27003), -INT16_C( 27335), -INT16_C(  4575), -INT16_C( 16702), -INT16_C(  8677) },
      {  INT16_C( 30233),  INT16_C( 19448), -INT16_C(  2728), -INT16_C( 32700), -INT16_C( 24531), -INT16_C( 30634), -INT16_C( 10105),  INT16_C(  7344),
         INT16_C( 25403),  INT16_C( 27112),  INT16_C( 28062), -INT16_C( 10241),  INT16_C(  8450), -INT16_C( 15163), -INT16_C(  7969), -INT16_C(  1886) },
      {  INT16_C( 30071), -INT16_C(  9170),  INT16_C(  3716),  INT16_C(  4059),       INT16_MAX,       INT16_MIN,       INT16_MIN, -INT16_C(  2761),
         INT16_C( 12359), -INT16_C( 11853), -INT16_C( 31910), -INT16_C( 25379),       INT16_MAX,  INT16_C( 17821), -INT16_C(  6713), -INT16_C(  9855) } },
    { { -INT16_C( 25770), -INT16_C( 20668), -INT16_C( 30576), -INT16_C( 17105), -INT16_C( 31191), -INT16_C( 20411), -INT16_C(  2722), -INT16_C( 26164),
        -INT16_C( 19368), -INT16_C(  2558),  INT16_C(   545),  INT16_C(  9422), -INT16_C( 27869),  INT16_C(   744), -INT16_C( 29836), -INT16_C( 13574) },
      {  INT16_C( 15910), -INT16_C( 18823), -INT16_C( 22073), -INT16_C(  3981), -INT16_C( 18129), -INT16_C( 29280),  INT16_C( 28078),  INT16_C(  1830),
         INT16_C( 10273),  INT16_C( 17405), -INT16_C( 13526),  INT16_C( 19815),  INT16_C( 20319), -INT16_C( 11441),  INT16_C( 19162),  INT16_C(   157) },
      {       INT16_MIN,       INT16_MIN,       INT16_MIN, -INT16_C( 28886), -INT16_C(  2913), -INT16_C( 26054),       INT16_MIN,  INT16_C( 29908),
        -INT16_C( 21926),  INT16_C(  9967), -INT16_C( 27125),       INT16_MIN,  INT16_C( 27678),  INT16_C(  6289),  INT16_C(  8878),  INT16_C( 19319) } },
    { {  INT16_C(  6024),  INT16_C( 20406),  INT16_C( 10944), -INT16_C(  4289), -INT16_C(  7965), -INT16_C( 28292), -INT16_C( 23987),  INT16_C( 28312),
        -INT16_C( 26934), -INT16_C(  2639),  INT16_C(  6241), -INT16_C( 16318), -INT16_C( 28056),  INT16_C( 17043),  INT16_C( 12764),  INT16_C( 25667) },
      { -INT16_C(  1720),  INT16_C(  2228), -INT16_C(  3293),  INT16_C(  1783),  INT16_C( 29651),  INT16_C(  8344),  INT16_C( 12309), -INT16_C(  8305),
         INT16_C( 16582),  INT16_C( 10452),  INT16_C(  5977), -INT16_C( 15896),  INT16_C( 31913), -INT16_C( 31485),  INT16_C( 18093), -INT16_C(  2583) },
      {  INT16_C( 26430),  INT16_C(  6655),       INT16_MIN,  INT16_C(  4325),  INT16_C(   508), -INT16_C(  1510),       INT16_MAX,  INT16_C(  4004),
        -INT16_C( 29573), -INT16_C( 10077), -INT16_C( 11013),       INT16_MAX,  INT16_C( 27034), -INT16_C(  9919),  INT16_C(   428),  INT16_C( 15510) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_hadds_epi16(a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_hsub_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(-18444), INT16_C(-18028), INT16_C( 29026), INT16_C(  2084),
                            INT16_C(  4112), INT16_C(-30013), INT16_C( 26536), INT16_C(-22613),
                            INT16_C( 28256), INT16_C(-13992), INT16_C(  1054), INT16_C(-17387),
                            INT16_C( 28361), INT16_C(-13892), INT16_C(-23621), INT16_C( -6864)),
      simde_mm256_set_epi16(INT16_C( -2077), INT16_C( 32093), INT16_C(-32096), INT16_C( -1919),
                            INT16_C(-29576), INT16_C(  1752), INT16_C( -9460), INT16_C( 18637),
                            INT16_C( 12808), INT16_C(  5796), INT16_C(  8151), INT16_C( -2897),
                            INT16_C(-24515), INT16_C(-31464), INT16_C(-13532), INT16_C( -9881)),
      simde_mm256_set_epi16(INT16_C(-31366), INT16_C( 30177), INT16_C( 31328), INT16_C( 28097),
                            INT16_C(   416), INT16_C(-26942), INT16_C( 31411), INT16_C( 16387),
                            INT16_C( -7012), INT16_C(-11048), INT16_C( -6949), INT16_C(  3651),
                            INT16_C( 23288), INT16_C(-18441), INT16_C( 23283), INT16_C( 16757)) },
    { simde_mm256_set_epi16(INT16_C(  9100), INT16_C( -1983), INT16_C( 28981), INT16_C( 31131),
                            INT16_C( -8267), INT16_C(  9829), INT16_C(-19843), INT16_C( 28705),
                            INT16_C( 15872), INT16_C( 21932), INT16_C( -9103), INT16_C( 20935),
                            INT16_C(-24996), INT16_C( 13650), INT16_C( 30282), INT16_C(-21973)),
      simde_mm256_set_epi16(INT16_C( -9238), INT16_C(-15000), INT16_C( 28358), INT16_C(-31774),
                            INT16_C( 20723), INT16_C( 27208), INT16_C(-20512), INT16_C( 10808),
                            INT16_C( 17124), INT16_C( 20983), INT16_C(   113), INT16_C( 22835),
                            INT16_C(  -190), INT16_C(-28607), INT16_C( 22616), INT16_C(  4805)),
      simde_mm256_set_epi16(INT16_C( -5762), INT16_C(  5404), INT16_C(  6485), INT16_C( 31320),
                            INT16_C(-11083), INT16_C(  2150), INT16_C( 18096), INT16_C(-16988),
                            INT16_C(  3859), INT16_C( 22722), INT16_C(-28417), INT16_C(-17811),
                            INT16_C(  6060), INT16_C( 30038), INT16_C(-26890), INT16_C( 13281)) },
    { simde_mm256_set_epi16(INT16_C( 26837), INT16_C( 17774), INT16_C(-32310), INT16_C( -9579),
                            INT16_C(  3637), INT16_C(-14106), INT16_C( 30046), INT16_C(-13930),
                            INT16_C( -1887), INT16_C( 13772), INT16_C( 19874), INT16_C(-18102),
                            INT16_C(-11204), INT16_C( -4897), INT16_C(  3260), INT16_C(-17962)),
      simde_mm256_set_epi16(INT16_C( 22389), INT16_C(-14730), INT16_C( 31871), INT16_C( 17642),
                            INT16_C(-26185), INT16_C( -2982), INT16_C(-14158), INT16_C(  4590),
                            INT16_C( 20601), INT16_C( 21976), INT16_C( 25432), INT16_C( 31010),
                            INT16_C(-23314), INT16_C( -9816), INT16_C(  8140), INT16_C( -7858)),
      simde_mm256_set_epi16(INT16_C( 28417), INT16_C(-14229), INT16_C( 23203), INT16_C( 18748),
                            INT16_C( -9063), INT16_C( 22731), INT16_C(-17743), INT16_C( 21560),
                            INT16_C(  1375), INT16_C(  5578), INT16_C( 13498), INT16_C(-15998),
                            INT16_C( 15659), INT16_C( 27560), INT16_C(  6307), INT16_C(-21222)) },
    { simde_mm256_set_epi16(INT16_C( 12057), INT16_C(-24517), INT16_C( -9967), INT16_C( -7239),
                            INT16_C( -2143), INT16_C(-23349), INT16_C(-25096), INT16_C(-21587),
                            INT16_C(  -802), INT16_C( 25377), INT16_C(-22001), INT16_C( 28281),
                            INT16_C(-28768), INT16_C(-19834), INT16_C(  8025), INT16_C(-23943)),
      simde_mm256_set_epi16(INT16_C(  6731), INT16_C(-13423), INT16_C( -6728), INT16_C(-19678),
                            INT16_C(-29476), INT16_C( -9124), INT16_C(-21742), INT16_C( 14418),
                            INT16_C( 27306), INT16_C( 25972), INT16_C(-12648), INT16_C(-16450),
                            INT16_C(-25118), INT16_C(  4239), INT16_C(-17383), INT16_C(-20693)),
      simde_mm256_set_epi16(INT16_C(-20154), INT16_C(-12950), INT16_C( 20352), INT16_C(-29376),
                            INT16_C( 28962), INT16_C(  2728), INT16_C(-21206), INT16_C(  3509),
                            INT16_C( -1334), INT16_C( -3802), INT16_C( 29357), INT16_C( -3310),
                            INT16_C( 26179), INT16_C(-15254), INT16_C(  8934), INT16_C(-31968)) },
    { simde_mm256_set_epi16(INT16_C(-15868), INT16_C(-27060), INT16_C(   528), INT16_C(-29935),
                            INT16_C(-12298), INT16_C( 18504), INT16_C( -5289), INT16_C(   430),
                            INT16_C(-29328), INT16_C(  3228), INT16_C( 18568), INT16_C(-13568),
                            INT16_C( -2471), INT16_C( -1530), INT16_C(  5334), INT16_C( 31888)),
      simde_mm256_set_epi16(INT16_C( -2903), INT16_C(-32300), INT16_C( 29546), INT16_C( 12096),
                            INT16_C(-31860), INT16_C(-30618), INT16_C(  1236), INT16_C( -9131),
                            INT16_C(  -497), INT16_C(-22865), INT16_C( 30177), INT16_C( 29201),
                            INT16_C(-26632), INT16_C(-25915), INT16_C( 14957), INT16_C(  6108)),
      simde_mm256_set_epi16(INT16_C(-29397), INT16_C(-17450), INT16_C(  1242), INT16_C(-10367),
                            INT16_C(-11192), INT16_C(-30463), INT16_C( 30802), INT16_C(  5719),
                            INT16_C(-22368), INT16_C(  -976), INT16_C(   717), INT16_C( -8849),
                            INT16_C( 32556), INT16_C(-32136), INT16_C(   941), INT16_C( 26554)) },
    { simde_mm256_set_epi16(INT16_C(  4205), INT16_C( 23268), INT16_C(-32329), INT16_C(-20305),
                            INT16_C( -9331), INT16_C( -4387), INT16_C(-30285), INT16_C( 13621),
                            INT16_C( 15818), INT16_C( 16370), INT16_C(-24200), INT16_C( 10901),
                            INT16_C( 17049), INT16_C( 14552), INT16_C( -5992), INT16_C(-17203)),
      simde_mm256_set_epi16(INT16_C( 15660), INT16_C( 25069), INT16_C(-11965), INT16_C( -9994),
                            INT16_C( -7271), INT16_C( 17410), INT16_C( 14381), INT16_C(-30587),
                            INT16_C( 11396), INT16_C(-22996), INT16_C( 26099), INT16_C(-11680),
                            INT16_C(-18604), INT16_C( 20100), INT16_C(  5959), INT16_C(-27899)),
      simde_mm256_set_epi16(INT16_C(  9409), INT16_C(  1971), INT16_C( 24681), INT16_C( 20568),
                            INT16_C( 19063), INT16_C( 12024), INT16_C(  4944), INT16_C(-21630),
                            INT16_C( 31144), INT16_C( 27757), INT16_C(-26832), INT16_C( 31678),
                            INT16_C(   552), INT16_C(-30435), INT16_C( -2497), INT16_C(-11211)) },
    { simde_mm256_set_epi16(INT16_C( 27483), INT16_C(  -502), INT16_C(   138), INT16_C( -7690),
                            INT16_C( 19176), INT16_C(-26261), INT16_C(-11443), INT16_C(-23576),
                            INT16_C( 22206), INT16_C(-25802), INT16_C( 16334), INT16_C(-18863),
                            INT16_C( 31357), INT16_C(-31063), INT16_C( 28401), INT16_C(-12622)),
      simde_mm256_set_epi16(INT16_C( 13717), INT16_C( 31993), INT16_C( 16367), INT16_C(-21175),
                            INT16_C( 23569), INT16_C( -7701), INT16_C( -1073), INT16_C( 24510),
                            INT16_C( 22852), INT16_C( 29964), INT16_C( -1882), INT16_C( -8354),
                            INT16_C(-16212), INT16_C( 25899), INT16_C( 11699), INT16_C(-20150)),
      simde_mm256_set_epi16(INT16_C( 18276), INT16_C( 27994), INT16_C(-31270), INT16_C( 25583),
                            INT16_C(-27985), INT16_C( -7828), INT16_C( 20099), INT16_C(-12133),
                            INT16_C(  7112), INT16_C( -6472), INT16_C(-23425), INT16_C(-31849),
                            INT16_C( 17528), INT16_C( 30339), INT16_C(  3116), INT16_C( 24513)) },
    { simde_mm256_set_epi16(INT16_C( -5381), INT16_C(-10578), INT16_C(-31523), INT16_C(-23458),
                            INT16_C( 22155), INT16_C( 10052), INT16_C( 22251), INT16_C(-22857),
                            INT16_C(  -468), INT16_C(  4672), INT16_C(-23974), INT16_C(-31691),
                            INT16_C( 11898), INT16_C(  4678), INT16_C(  5316), INT16_C(-17657)),
      simde_mm256_set_epi16(INT16_C( 15026), INT16_C(-17248), INT16_C(-13886), INT16_C( 26220),
                            INT16_C( -1198), INT16_C( -7005), INT16_C(-21616), INT16_C(-30390),
                            INT16_C(-30119), INT16_C(-22301), INT16_C(-29984), INT16_C(  6696),
                            INT16_C( 12300), INT16_C( 16913), INT16_C( 12770), INT16_C(  3850)),
      simde_mm256_set_epi16(INT16_C(-32274), INT16_C(-25430), INT16_C( -5807), INT16_C( -8774),
                            INT16_C( -5197), INT16_C(  8065), INT16_C(-12103), INT16_C( 20428),
                            INT16_C(  7818), INT16_C(-28856), INT16_C(  4613), INT16_C( -8920),
                            INT16_C(  5140), INT16_C( -7717), INT16_C( -7220), INT16_C(-22973)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_hsub_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_hsub_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(-1208698476), INT32_C( 1902250020), INT32_C(  269519555), INT32_C( 1739106219),
                            INT32_C( 1851836760), INT32_C(   69123093), INT32_C( 1858718140), INT32_C(-1547967184)),
      simde_mm256_set_epi32(INT32_C( -136086179), INT32_C(-2103379839), INT32_C(-1938290984), INT32_C( -619951923),
                            INT32_C(  839390884), INT32_C(  534246575), INT32_C(-1606580968), INT32_C( -886777497)),
      simde_mm256_set_epi32(INT32_C(-1967293660), INT32_C( 1318339061), INT32_C(-1184018800), INT32_C( 1469586664),
                            INT32_C( -305144309), INT32_C(  719803471), INT32_C(-1782713667), INT32_C(  888281972)) },
    { simde_mm256_set_epi32(INT32_C(  596441153), INT32_C( 1899329947), INT32_C( -541776283), INT32_C(-1300402143),
                            INT32_C( 1040209324), INT32_C( -596553273), INT32_C(-1638124206), INT32_C( 1984604715)),
      simde_mm256_set_epi32(INT32_C( -605371032), INT32_C( 1858503650), INT32_C( 1358129736), INT32_C(-1344263624),
                            INT32_C( 1122259447), INT32_C(    7428403), INT32_C(  -12414911), INT32_C( 1482166981)),
      simde_mm256_set_epi32(INT32_C(-1831092614), INT32_C( 1592573936), INT32_C( 1302888794), INT32_C( -758625860),
                            INT32_C(-1114831044), INT32_C( 1494581892), INT32_C(-1636762597), INT32_C( -672238375)) },
    { simde_mm256_set_epi32(INT32_C( 1758807406), INT32_C(-2117412203), INT32_C(  238405862), INT32_C( 1969146262),
                            INT32_C( -123652660), INT32_C( 1302509898), INT32_C( -734204705), INT32_C(  213694934)),
      simde_mm256_set_epi32(INT32_C( 1467336310), INT32_C( 2088715498), INT32_C(-1715997606), INT32_C( -927854098),
                            INT32_C( 1350129112), INT32_C( 1666742562), INT32_C(-1527850584), INT32_C(  533520718)),
      simde_mm256_set_epi32(INT32_C(  621379188), INT32_C(  788143508), INT32_C(  418747687), INT32_C( 1730740400),
                            INT32_C(  316613450), INT32_C( 2061371302), INT32_C( 1426162558), INT32_C(  947899639)) },
    { simde_mm256_set_epi32(INT32_C(  790208571), INT32_C( -653139015), INT32_C( -140401461), INT32_C(-1644647507),
                            INT32_C(  -52534495), INT32_C(-1441829255), INT32_C(-1885293946), INT32_C(  525967993)),
      simde_mm256_set_epi32(INT32_C(  441174929), INT32_C( -440880350), INT32_C(-1931682724), INT32_C(-1424869294),
                            INT32_C( 1789551988), INT32_C( -828850242), INT32_C(-1646129009), INT32_C(-1139167445)),
      simde_mm256_set_epi32(INT32_C( -882055279), INT32_C(  506813430), INT32_C(-1443347586), INT32_C(-1504246046),
                            INT32_C( 1676565066), INT32_C(  506961564), INT32_C(-1389294760), INT32_C(-1883705357)) },
    { simde_mm256_set_epi32(INT32_C(-1039886772), INT32_C(   34638609), INT32_C( -805943224), INT32_C( -346619474),
                            INT32_C(-1922036580), INT32_C( 1216924416), INT32_C( -161875450), INT32_C(  349600912)),
      simde_mm256_set_epi32(INT32_C( -190217772), INT32_C( 1936338752), INT32_C(-2087942042), INT32_C(   81058901),
                            INT32_C(  -32528721), INT32_C( 1977709073), INT32_C(-1745315131), INT32_C(  980228060)),
      simde_mm256_set_epi32(INT32_C( 2126556524), INT32_C(-2125966353), INT32_C( 1074525381), INT32_C(  459323750),
                            INT32_C( 2010237794), INT32_C(-1569424105), INT32_C(-1156006300), INT32_C(  511476362)) },
    { simde_mm256_set_epi32(INT32_C(  275602148), INT32_C(-2118668113), INT32_C( -611455267), INT32_C(-1984744139),
                            INT32_C( 1036664818), INT32_C(-1585960299), INT32_C( 1117337816), INT32_C( -392643379)),
      simde_mm256_set_epi32(INT32_C( 1026318829), INT32_C( -784082698), INT32_C( -476494846), INT32_C(  942508165),
                            INT32_C(  746890796), INT32_C( 1710477920), INT32_C(-1219211644), INT32_C(  390566661)),
      simde_mm256_set_epi32(INT32_C(-1810401527), INT32_C( 1419003011), INT32_C( 1900697035), INT32_C(-1373288872),
                            INT32_C(  963587124), INT32_C( 1609778305), INT32_C( 1672342179), INT32_C(-1509981195)) },
    { simde_mm256_set_epi32(INT32_C( 1801190922), INT32_C(    9101814), INT32_C( 1256757611), INT32_C( -749886488),
                            INT32_C( 1455332150), INT32_C( 1070511697), INT32_C( 2055046825), INT32_C( 1861340850)),
      simde_mm256_set_epi32(INT32_C(  898989305), INT32_C( 1072672073), INT32_C( 1544675819), INT32_C(  -70295618),
                            INT32_C( 1497658636), INT32_C( -123281570), INT32_C(-1062443733), INT32_C(  766751050)),
      simde_mm256_set_epi32(INT32_C(  173682768), INT32_C(-1614971437), INT32_C(-1792089108), INT32_C(-2006644099),
                            INT32_C(-1620940206), INT32_C( 1829194783), INT32_C( -384820453), INT32_C( -193705975)) },
    { simde_mm256_set_epi32(INT32_C( -352594258), INT32_C(-2065849250), INT32_C( 1451960132), INT32_C( 1458284215),
                            INT32_C(  -30666176), INT32_C(-1571126219), INT32_C(  779752006), INT32_C(  348437255)),
      simde_mm256_set_epi32(INT32_C(  984792224), INT32_C( -910006676), INT32_C(  -78453597), INT32_C(-1416591030),
                            INT32_C(-1973835549), INT32_C(-1965024728), INT32_C(  806109713), INT32_C(  836898570)),
      simde_mm256_set_epi32(INT32_C(-1894798900), INT32_C(-1338137433), INT32_C(-1713254992), INT32_C(    6324083),
                            INT32_C(    8810821), INT32_C(   30788857), INT32_C(-1540460043), INT32_C( -431314751)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_hsub_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_hsubs_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[16];
    const int16_t b[16];
    const int16_t r[16];
  } test_vec[] = {
    { { -INT16_C( 26245), -INT16_C( 12255), -INT16_C( 13259), -INT16_C( 32044),  INT16_C( 31061),  INT16_C( 23401), -INT16_C(  9352), -INT16_C( 31983),
        -INT16_C(  3563), -INT16_C(  3232), -INT16_C(  4207),  INT16_C(  2011), -INT16_C( 23858),  INT16_C( 31980),  INT16_C(  8339),  INT16_C(  3620) },
      {  INT16_C( 17849), -INT16_C(  4386), -INT16_C( 19950),  INT16_C( 26481), -INT16_C(  9685), -INT16_C( 23357), -INT16_C( 11082), -INT16_C( 13529),
        -INT16_C( 30778),  INT16_C( 22463), -INT16_C( 25994),  INT16_C( 17502),  INT16_C( 19261), -INT16_C( 12096), -INT16_C(  6805),  INT16_C(  9438) },
      { -INT16_C( 13990),  INT16_C( 18785),  INT16_C(  7660),  INT16_C( 22631),  INT16_C( 22235),       INT16_MIN,  INT16_C( 13672),  INT16_C(  2447),
        -INT16_C(   331), -INT16_C(  6218),       INT16_MIN,  INT16_C(  4719),       INT16_MIN,       INT16_MIN,  INT16_C( 31357), -INT16_C( 16243) } },
    { { -INT16_C( 17366),  INT16_C( 15379), -INT16_C( 31634), -INT16_C( 26204),  INT16_C( 26462),  INT16_C(  5181),  INT16_C( 25659),  INT16_C(   480),
        -INT16_C( 24597),  INT16_C( 24921), -INT16_C( 18631),  INT16_C( 30373),  INT16_C( 26114),  INT16_C( 27974),  INT16_C(  9291),  INT16_C( 30098) },
      { -INT16_C( 23072),  INT16_C( 20146),  INT16_C( 22057), -INT16_C( 30744),  INT16_C(  9661), -INT16_C(  1892),  INT16_C( 31882),  INT16_C( 30201),
         INT16_C( 21019),  INT16_C( 21719),  INT16_C( 31754),  INT16_C(  3275),  INT16_C(  4578),  INT16_C( 11642),  INT16_C(  3126),  INT16_C(  5795) },
      { -INT16_C( 32745), -INT16_C(  5430),  INT16_C( 21281),  INT16_C( 25179),       INT16_MIN,       INT16_MAX,  INT16_C( 11553),  INT16_C(  1681),
              INT16_MIN,       INT16_MIN, -INT16_C(  1860), -INT16_C( 20807), -INT16_C(   700),  INT16_C( 28479), -INT16_C(  7064), -INT16_C(  2669) } },
    { {  INT16_C( 21937), -INT16_C(  9627),  INT16_C( 19883),  INT16_C( 26721), -INT16_C(   654), -INT16_C(   928),  INT16_C( 22905), -INT16_C( 27534),
         INT16_C( 18860), -INT16_C( 18711), -INT16_C( 19259), -INT16_C( 22334),  INT16_C( 15557), -INT16_C(  1067),  INT16_C( 30792), -INT16_C(  1774) },
      {  INT16_C( 30669),  INT16_C( 30931),  INT16_C( 13764),  INT16_C( 14048),  INT16_C( 16434), -INT16_C( 21453), -INT16_C( 23142),  INT16_C( 17984),
         INT16_C( 10734), -INT16_C( 19460), -INT16_C( 16675), -INT16_C( 23717),  INT16_C( 12795),  INT16_C( 17310), -INT16_C( 20311),  INT16_C( 30525) },
      {  INT16_C( 31564), -INT16_C(  6838),  INT16_C(   274),       INT16_MAX, -INT16_C(   262), -INT16_C(   284),       INT16_MAX,       INT16_MIN,
              INT16_MAX,  INT16_C(  3075),  INT16_C( 16624),  INT16_C( 32566),  INT16_C( 30194),  INT16_C(  7042), -INT16_C(  4515),       INT16_MIN } },
    { {  INT16_C(  4135), -INT16_C(  5137), -INT16_C( 12219),  INT16_C( 30754),  INT16_C( 21776), -INT16_C( 21980),  INT16_C( 25850), -INT16_C(  5904),
        -INT16_C(  4978),  INT16_C( 27547), -INT16_C(  2133), -INT16_C( 23026), -INT16_C( 21208), -INT16_C( 11799),  INT16_C(  9821), -INT16_C( 31416) },
      {  INT16_C( 14391),  INT16_C( 31856), -INT16_C( 28152),  INT16_C(  6388),  INT16_C(  6375), -INT16_C(  7741), -INT16_C( 19587),  INT16_C(  3017),
         INT16_C( 26016),  INT16_C( 19318), -INT16_C( 31396), -INT16_C( 31503), -INT16_C(  9678), -INT16_C( 28843), -INT16_C( 25087),  INT16_C( 14356) },
      {  INT16_C(  9272),       INT16_MIN,       INT16_MAX,  INT16_C( 31754), -INT16_C( 17465),       INT16_MIN,  INT16_C( 14116), -INT16_C( 22604),
        -INT16_C( 32525),  INT16_C( 20893), -INT16_C(  9409),       INT16_MAX,  INT16_C(  6698),  INT16_C(   107),  INT16_C( 19165),       INT16_MIN } },
    { { -INT16_C( 31274), -INT16_C(  8524), -INT16_C( 22249), -INT16_C(    10), -INT16_C( 17983),  INT16_C( 16096), -INT16_C( 21907),  INT16_C(  3401),
        -INT16_C( 16369),  INT16_C( 27480),  INT16_C( 18757),  INT16_C( 30703),  INT16_C( 17443),  INT16_C(  9222),  INT16_C(  7138), -INT16_C( 18340) },
      {  INT16_C(  4512), -INT16_C( 18538), -INT16_C( 29254),  INT16_C( 31670), -INT16_C( 26810), -INT16_C( 19526),  INT16_C(   833),  INT16_C( 20672),
         INT16_C(  6339),  INT16_C(  2235), -INT16_C( 21919), -INT16_C( 31361), -INT16_C( 30994), -INT16_C( 11863),  INT16_C(  1697),  INT16_C( 16777) },
      { -INT16_C( 22750), -INT16_C( 22239),       INT16_MIN, -INT16_C( 25308),  INT16_C( 23050),       INT16_MIN, -INT16_C(  7284), -INT16_C( 19839),
              INT16_MIN, -INT16_C( 11946),  INT16_C(  8221),  INT16_C( 25478),  INT16_C(  4104),  INT16_C(  9442), -INT16_C( 19131), -INT16_C( 15080) } },
    { {  INT16_C(  8215), -INT16_C( 11784), -INT16_C( 20563), -INT16_C(  3252),  INT16_C(  1606), -INT16_C( 30809),  INT16_C( 26378), -INT16_C( 12841),
        -INT16_C( 28032), -INT16_C(  7722),  INT16_C( 21820),  INT16_C( 10854),  INT16_C(  4315),  INT16_C( 31995), -INT16_C( 31466),  INT16_C( 11709) },
      { -INT16_C( 18779),  INT16_C( 21246),  INT16_C( 19045), -INT16_C( 21691), -INT16_C(  5039),  INT16_C( 23346),  INT16_C(  2388), -INT16_C( 11224),
        -INT16_C(   357), -INT16_C( 10315),  INT16_C(  7252),  INT16_C( 12033), -INT16_C(   724),  INT16_C( 17068),  INT16_C( 27010),  INT16_C( 10095) },
      {  INT16_C( 19999), -INT16_C( 17311),  INT16_C( 32415),       INT16_MAX,       INT16_MIN,       INT16_MAX, -INT16_C( 28385),  INT16_C( 13612),
        -INT16_C( 20310),  INT16_C( 10966), -INT16_C( 27680),       INT16_MIN,  INT16_C(  9958), -INT16_C(  4781), -INT16_C( 17792),  INT16_C( 16915) } },
    { {  INT16_C( 27935), -INT16_C( 31623), -INT16_C( 16713),  INT16_C(  2095),  INT16_C( 25003), -INT16_C(   157), -INT16_C( 29590),  INT16_C(  1491),
        -INT16_C( 30582), -INT16_C(  8484), -INT16_C(  8540), -INT16_C( 12274), -INT16_C( 17701),  INT16_C( 23826), -INT16_C( 32477),  INT16_C( 17284) },
      { -INT16_C(   530), -INT16_C( 22841), -INT16_C(  2117),  INT16_C( 26286),  INT16_C(  4696), -INT16_C( 15515),  INT16_C( 14494),  INT16_C( 10440),
        -INT16_C( 23103),  INT16_C( 25863),  INT16_C(  5507),  INT16_C( 24118),  INT16_C( 18639), -INT16_C(  3397),  INT16_C( 16330), -INT16_C( 18379) },
      {       INT16_MAX, -INT16_C( 18808),  INT16_C( 25160), -INT16_C( 31081),  INT16_C( 22311), -INT16_C( 28403),  INT16_C( 20211),  INT16_C(  4054),
        -INT16_C( 22098),  INT16_C(  3734),       INT16_MIN,       INT16_MIN,       INT16_MIN, -INT16_C( 18611),  INT16_C( 22036),       INT16_MAX } },
    { { -INT16_C(   708), -INT16_C(  2210),  INT16_C(  3572),  INT16_C( 19550), -INT16_C( 15585), -INT16_C( 17137), -INT16_C(  9988), -INT16_C( 16923),
        -INT16_C(  4995),  INT16_C(    34),  INT16_C( 22529), -INT16_C( 12194),  INT16_C(  6561),  INT16_C( 27587), -INT16_C(  1960), -INT16_C( 27613) },
      { -INT16_C( 32011), -INT16_C(  5749), -INT16_C(  5745), -INT16_C( 20938),  INT16_C( 17837), -INT16_C( 22165),  INT16_C( 20509), -INT16_C( 26010),
        -INT16_C( 30659),  INT16_C( 16026), -INT16_C(  1823), -INT16_C( 32241), -INT16_C( 11759),  INT16_C( 27117),  INT16_C(  4298), -INT16_C( 16131) },
      {  INT16_C(  1502), -INT16_C( 15978),  INT16_C(  1552),  INT16_C(  6935), -INT16_C( 26262),  INT16_C( 15193),       INT16_MAX,       INT16_MAX,
        -INT16_C(  5029),       INT16_MAX, -INT16_C( 21026),  INT16_C( 25653),       INT16_MIN,  INT16_C( 30418),       INT16_MIN,  INT16_C( 20429) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_hsubs_epi16(a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int32_t i32gather_buffer[4096];

static int
test_simde_mm_i32gather_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t vindex[4];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(         141),  INT32_C(          78),  INT32_C(         249),  INT32_C(          88) },
      {  INT32_C(         141),  INT32_C(          78),  INT32_C(         249),  INT32_C(          88) } },
    { {  INT32_C(         189),  INT32_C(         204),  INT32_C(          14),  INT32_C(         231) },
      {  INT32_C(         189),  INT32_C(         204),  INT32_C(          14),  INT32_C(         231) } },
    { {  INT32_C(         199),  INT32_C(          52),  INT32_C(         133),  INT32_C(         101) },
      {  INT32_C(         199),  INT32_C(          52),  INT32_C(         133),  INT32_C(         101) } },
    { {  INT32_C(         239),  INT32_C(          12),  INT32_C(         121),  INT32_C(         226) },
      {  INT32_C(         239),  INT32_C(          12),  INT32_C(         121),  INT32_C(         226) } },
    { {  INT32_C(         197),  INT32_C(         167),  INT32_C(         235),  INT32_C(          15) },
      {  INT32_C(         197),  INT32_C(         167),  INT32_C(         235),  INT32_C(          15) } },
    { {  INT32_C(         239),  INT32_C(         157),  INT32_C(         219),  INT32_C(          83) },
      {  INT32_C(         239),  INT32_C(         157),  INT32_C(         219),  INT32_C(          83) } },
    { {  INT32_C(         230),  INT32_C(          67),  INT32_C(         195),  INT32_C(          27) },
      {  INT32_C(         230),  INT32_C(          67),  INT32_C(         195),  INT32_C(          27) } },
    { {  INT32_C(         203),  INT32_C(         150),  INT32_C(         133),  INT32_C(          68) },
      {  INT32_C(         203),  INT32_C(         150),  INT32_C(         133),  INT32_C(          68) } }
  };
  for (size_t i = 0 ; i < (sizeof(i32gather_buffer) / sizeof(i32gather_buffer[0])) ; i++) { i32gather_buffer[i] = HEDLEY_STATIC_CAST(int32_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m128i r = simde_mm_i32gather_epi32(i32gather_buffer, vindex, 4);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_mask_i32gather_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t src[4];
    const int32_t vindex[4];
    const int32_t mask[4];
    const int32_t r[4];
  } test_vec[] = {
    { { -INT32_C(   685361840), -INT32_C(  1057547638), -INT32_C(  1601608401), -INT32_C(  2110383967) },
      {  INT32_C(          85),  INT32_C(          81),  INT32_C(         250),  INT32_C(         146) },
      {            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN },
      {  INT32_C(          85),  INT32_C(          81),  INT32_C(         250),  INT32_C(         146) } },
    { { -INT32_C(   612533238), -INT32_C(  2144538583), -INT32_C(  1518687133),  INT32_C(  1765241328) },
      {  INT32_C(         198),  INT32_C(         202),  INT32_C(         225),  INT32_C(         124) },
      {  INT32_C(           0),  INT32_C(           0),            INT32_MIN,            INT32_MIN },
      { -INT32_C(   612533238), -INT32_C(  2144538583),  INT32_C(         225),  INT32_C(         124) } },
    { {  INT32_C(  1518663255), -INT32_C(  1557876442),  INT32_C(  1485068261),  INT32_C(   248810868) },
      {  INT32_C(         138),  INT32_C(         226),  INT32_C(           2),  INT32_C(         239) },
      {            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN },
      {  INT32_C(         138),  INT32_C(         226),  INT32_C(           2),  INT32_C(         239) } },
    { { -INT32_C(   436463052),  INT32_C(    46620719), -INT32_C(   637220286), -INT32_C(   624310953) },
      {  INT32_C(         255),  INT32_C(          71),  INT32_C(           2),  INT32_C(          72) },
      {  INT32_C(           0),            INT32_MIN,  INT32_C(           0),  INT32_C(           0) },
      { -INT32_C(   436463052),  INT32_C(          71), -INT32_C(   637220286), -INT32_C(   624310953) } },
    { {  INT32_C(  1290237130), -INT32_C(   376192698), -INT32_C(   487909938), -INT32_C(  1020567585) },
      {  INT32_C(         161),  INT32_C(          11),  INT32_C(         227),  INT32_C(          34) },
      {            INT32_MIN,            INT32_MIN,  INT32_C(           0),            INT32_MIN },
      {  INT32_C(         161),  INT32_C(          11), -INT32_C(   487909938),  INT32_C(          34) } },
    { { -INT32_C(   193884505),  INT32_C(  2130650489),  INT32_C(   526459079), -INT32_C(  1220465615) },
      {  INT32_C(          81),  INT32_C(         199),  INT32_C(         213),  INT32_C(         171) },
      {            INT32_MIN,            INT32_MIN,            INT32_MIN,  INT32_C(           0) },
      {  INT32_C(          81),  INT32_C(         199),  INT32_C(         213), -INT32_C(  1220465615) } },
    { { -INT32_C(  1028183535),  INT32_C(   931744097),  INT32_C(    34424318), -INT32_C(  1095912774) },
      {  INT32_C(           3),  INT32_C(         170),  INT32_C(          26),  INT32_C(         155) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),            INT32_MIN },
      { -INT32_C(  1028183535),  INT32_C(   931744097),  INT32_C(    34424318),  INT32_C(         155) } },
    { {  INT32_C(  1720704891), -INT32_C(  1492085483),  INT32_C(  1187097276),  INT32_C(  1491223020) },
      {  INT32_C(         248),  INT32_C(          82),  INT32_C(         255),  INT32_C(         161) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) },
      {  INT32_C(  1720704891), -INT32_C(  1492085483),  INT32_C(  1187097276),  INT32_C(  1491223020) } }
  };

  for (size_t i = 0 ; i < (sizeof(i32gather_buffer) / sizeof(i32gather_buffer[0])) ; i++) { i32gather_buffer[i] = HEDLEY_STATIC_CAST(int32_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi32(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m128i mask = simde_x_mm_loadu_epi32(test_vec[i].mask);
    simde__m128i r = simde_mm_mask_i32gather_epi32(src, i32gather_buffer, vindex, mask, 4);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;

}

static int
test_simde_mm256_i32gather_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t vindex[8];
    const int32_t r[8];
  } test_vec[] = {
    { {  INT32_C(          27),  INT32_C(         135),  INT32_C(         222),  INT32_C(          60),  INT32_C(         120),  INT32_C(         248),  INT32_C(         252),  INT32_C(          33) },
      {  INT32_C(          27),  INT32_C(         135),  INT32_C(         222),  INT32_C(          60),  INT32_C(         120),  INT32_C(         248),  INT32_C(         252),  INT32_C(          33) } },
    { {  INT32_C(          94),  INT32_C(         185),  INT32_C(         221),  INT32_C(         160),  INT32_C(          28),  INT32_C(         199),  INT32_C(         216),  INT32_C(          91) },
      {  INT32_C(          94),  INT32_C(         185),  INT32_C(         221),  INT32_C(         160),  INT32_C(          28),  INT32_C(         199),  INT32_C(         216),  INT32_C(          91) } },
    { {  INT32_C(         198),  INT32_C(          71),  INT32_C(         176),  INT32_C(          54),  INT32_C(         235),  INT32_C(         249),  INT32_C(           5),  INT32_C(         236) },
      {  INT32_C(         198),  INT32_C(          71),  INT32_C(         176),  INT32_C(          54),  INT32_C(         235),  INT32_C(         249),  INT32_C(           5),  INT32_C(         236) } },
    { {  INT32_C(         152),  INT32_C(         161),  INT32_C(         168),  INT32_C(         209),  INT32_C(         201),  INT32_C(         153),  INT32_C(           8),  INT32_C(          97) },
      {  INT32_C(         152),  INT32_C(         161),  INT32_C(         168),  INT32_C(         209),  INT32_C(         201),  INT32_C(         153),  INT32_C(           8),  INT32_C(          97) } },
    { {  INT32_C(         210),  INT32_C(          35),  INT32_C(          29),  INT32_C(         112),  INT32_C(         115),  INT32_C(           2),  INT32_C(         240),  INT32_C(         195) },
      {  INT32_C(         210),  INT32_C(          35),  INT32_C(          29),  INT32_C(         112),  INT32_C(         115),  INT32_C(           2),  INT32_C(         240),  INT32_C(         195) } },
    { {  INT32_C(         126),  INT32_C(           0),  INT32_C(          49),  INT32_C(         241),  INT32_C(         211),  INT32_C(         104),  INT32_C(         198),  INT32_C(         131) },
      {  INT32_C(         126),  INT32_C(           0),  INT32_C(          49),  INT32_C(         241),  INT32_C(         211),  INT32_C(         104),  INT32_C(         198),  INT32_C(         131) } },
    { {  INT32_C(          25),  INT32_C(         242),  INT32_C(          37),  INT32_C(         251),  INT32_C(         120),  INT32_C(          10),  INT32_C(          98),  INT32_C(         217) },
      {  INT32_C(          25),  INT32_C(         242),  INT32_C(          37),  INT32_C(         251),  INT32_C(         120),  INT32_C(          10),  INT32_C(          98),  INT32_C(         217) } },
    { {  INT32_C(         159),  INT32_C(         152),  INT32_C(         136),  INT32_C(          27),  INT32_C(          62),  INT32_C(         120),  INT32_C(         145),  INT32_C(         235) },
      {  INT32_C(         159),  INT32_C(         152),  INT32_C(         136),  INT32_C(          27),  INT32_C(          62),  INT32_C(         120),  INT32_C(         145),  INT32_C(         235) } }
  };
  for (size_t i = 0 ; i < (sizeof(i32gather_buffer) / sizeof(i32gather_buffer[0])) ; i++) { i32gather_buffer[i] = HEDLEY_STATIC_CAST(int32_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i vindex = simde_x_mm256_loadu_epi32(test_vec[i].vindex);
    simde__m256i r = simde_mm256_i32gather_epi32(i32gather_buffer, vindex, 4);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_i32gather_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t src[8];
    const int32_t vindex[8];
    const int32_t mask[8];
    const int32_t r[8];
  } test_vec[] = {
    { { -INT32_C(  1086860273), -INT32_C(   849127954), -INT32_C(   348023026), -INT32_C(  1484367608),  INT32_C(   287657063), -INT32_C(   380909789),  INT32_C(  1004087424), -INT32_C(  1260891740) },
      {  INT32_C(          19),  INT32_C(         114),  INT32_C(         118),  INT32_C(         226),  INT32_C(          83),  INT32_C(         138),  INT32_C(          11),  INT32_C(         140) },
      {  INT32_C(           0),            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN,  INT32_C(           0),  INT32_C(           0),            INT32_MIN },
      { -INT32_C(  1086860273),  INT32_C(         114),  INT32_C(         118),  INT32_C(         226),  INT32_C(          83), -INT32_C(   380909789),  INT32_C(  1004087424),  INT32_C(         140) } },
    { { -INT32_C(  1577111400),  INT32_C(  1410892465),  INT32_C(   678561379),  INT32_C(  1578988305), -INT32_C(  1665833368), -INT32_C(  2026400021),  INT32_C(  2139872828), -INT32_C(   596763836) },
      {  INT32_C(         252),  INT32_C(          84),  INT32_C(         173),  INT32_C(         158),  INT32_C(         235),  INT32_C(         148),  INT32_C(         148),  INT32_C(         100) },
      {  INT32_C(           0),            INT32_MIN,  INT32_C(           0),            INT32_MIN,            INT32_MIN,  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) },
      { -INT32_C(  1577111400),  INT32_C(          84),  INT32_C(   678561379),  INT32_C(         158),  INT32_C(         235), -INT32_C(  2026400021),  INT32_C(  2139872828), -INT32_C(   596763836) } },
    { { -INT32_C(  1074690759), -INT32_C(  1478416843),  INT32_C(   592490712),  INT32_C(   695179148),  INT32_C(   220112409), -INT32_C(   128800944),  INT32_C(  1706634309), -INT32_C(  1334452618) },
      {  INT32_C(         152),  INT32_C(         101),  INT32_C(         140),  INT32_C(          63),  INT32_C(         238),  INT32_C(         192),  INT32_C(          66),  INT32_C(         224) },
      {            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN,  INT32_C(           0),            INT32_MIN },
      {  INT32_C(         152),  INT32_C(         101),  INT32_C(         140),  INT32_C(          63),  INT32_C(         238),  INT32_C(         192),  INT32_C(  1706634309),  INT32_C(         224) } },
    { { -INT32_C(  1420665862),  INT32_C(  1515996691),  INT32_C(   969382596),  INT32_C(  1431139470),  INT32_C(   144133742), -INT32_C(  1133668042), -INT32_C(  2118118208), -INT32_C(  1488673875) },
      {  INT32_C(          18),  INT32_C(         161),  INT32_C(         215),  INT32_C(          22),  INT32_C(         172),  INT32_C(          28),  INT32_C(          76),  INT32_C(         203) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),            INT32_MIN,            INT32_MIN,            INT32_MIN,  INT32_C(           0) },
      { -INT32_C(  1420665862),  INT32_C(  1515996691),  INT32_C(   969382596),  INT32_C(  1431139470),  INT32_C(         172),  INT32_C(          28),  INT32_C(          76), -INT32_C(  1488673875) } },
    { { -INT32_C(    20439286), -INT32_C(   449069208), -INT32_C(  2012655728), -INT32_C(  1829288817), -INT32_C(  2081353314),  INT32_C(  1074228372),  INT32_C(  2142754948), -INT32_C(  1477248355) },
      {  INT32_C(         130),  INT32_C(           9),  INT32_C(         190),  INT32_C(          83),  INT32_C(          47),  INT32_C(          49),  INT32_C(          83),  INT32_C(          58) },
      {            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN,  INT32_C(           0),  INT32_C(           0) },
      {  INT32_C(         130),  INT32_C(           9),  INT32_C(         190),  INT32_C(          83),  INT32_C(          47),  INT32_C(          49),  INT32_C(  2142754948), -INT32_C(  1477248355) } },
    { {  INT32_C(  1885499633),  INT32_C(   656294547), -INT32_C(  1351644492),  INT32_C(  1128532806), -INT32_C(   873657988), -INT32_C(  1764702148), -INT32_C(  2096027564),  INT32_C(  1796130170) },
      {  INT32_C(         201),  INT32_C(         102),  INT32_C(         201),  INT32_C(         231),  INT32_C(         159),  INT32_C(          22),  INT32_C(          38),  INT32_C(          66) },
      {  INT32_C(           0),            INT32_MIN,  INT32_C(           0),            INT32_MIN,  INT32_C(           0),            INT32_MIN,  INT32_C(           0),  INT32_C(           0) },
      {  INT32_C(  1885499633),  INT32_C(         102), -INT32_C(  1351644492),  INT32_C(         231), -INT32_C(   873657988),  INT32_C(          22), -INT32_C(  2096027564),  INT32_C(  1796130170) } },
    { {  INT32_C(   515341239), -INT32_C(   737547912), -INT32_C(   778257104),  INT32_C(   936725373), -INT32_C(  1833731923),  INT32_C(  1914979922),  INT32_C(  1184881778), -INT32_C(   583921882) },
      {  INT32_C(           2),  INT32_C(           3),  INT32_C(          63),  INT32_C(         107),  INT32_C(         115),  INT32_C(         226),  INT32_C(          35),  INT32_C(          31) },
      {            INT32_MIN,  INT32_C(           0),            INT32_MIN,  INT32_C(           0),            INT32_MIN,            INT32_MIN,  INT32_C(           0),            INT32_MIN },
      {  INT32_C(           2), -INT32_C(   737547912),  INT32_C(          63),  INT32_C(   936725373),  INT32_C(         115),  INT32_C(         226),  INT32_C(  1184881778),  INT32_C(          31) } },
    { { -INT32_C(   724013614), -INT32_C(   778505370),  INT32_C(  2063310050), -INT32_C(  1154183402), -INT32_C(   707698399),  INT32_C(   210159988),  INT32_C(  1698257641),  INT32_C(    36013360) },
      {  INT32_C(          89),  INT32_C(         212),  INT32_C(          34),  INT32_C(         214),  INT32_C(          87),  INT32_C(          97),  INT32_C(         188),  INT32_C(          22) },
      {            INT32_MIN,            INT32_MIN,            INT32_MIN,  INT32_C(           0),  INT32_C(           0),            INT32_MIN,  INT32_C(           0),  INT32_C(           0) },
      {  INT32_C(          89),  INT32_C(         212),  INT32_C(          34), -INT32_C(  1154183402), -INT32_C(   707698399),  INT32_C(          97),  INT32_C(  1698257641),  INT32_C(    36013360) } }
  };
  for (size_t i = 0 ; i < (sizeof(i32gather_buffer) / sizeof(i32gather_buffer[0])) ; i++) { i32gather_buffer[i] = HEDLEY_STATIC_CAST(int32_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi32(test_vec[i].src);
    simde__m256i vindex = simde_x_mm256_loadu_epi32(test_vec[i].vindex);
    simde__m256i mask = simde_x_mm256_loadu_epi32(test_vec[i].mask);
    simde__m256i r = simde_mm256_mask_i32gather_epi32(src, i32gather_buffer, vindex, mask, 4);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_i64gather_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t vindex[2];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT64_C(                 136),  INT64_C(                  22) },
      {  INT32_C(         136),  INT32_C(          22),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT64_C(                 173),  INT64_C(                  86) },
      {  INT32_C(         173),  INT32_C(          86),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT64_C(                 157),  INT64_C(                 106) },
      {  INT32_C(         157),  INT32_C(         106),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT64_C(                  81),  INT64_C(                 112) },
      {  INT32_C(          81),  INT32_C(         112),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT64_C(                  42),  INT64_C(                  54) },
      {  INT32_C(          42),  INT32_C(          54),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT64_C(                  75),  INT64_C(                 158) },
      {  INT32_C(          75),  INT32_C(         158),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT64_C(                   9),  INT64_C(                  95) },
      {  INT32_C(           9),  INT32_C(          95),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT64_C(                 192),  INT64_C(                 148) },
      {  INT32_C(         192),  INT32_C(         148),  INT32_C(           0),  INT32_C(           0) } }
  };
  for (size_t i = 0 ; i < (sizeof(i32gather_buffer) / sizeof(i32gather_buffer[0])) ; i++) { i32gather_buffer[i] = HEDLEY_STATIC_CAST(int32_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi64(test_vec[i].vindex);
    simde__m128i r = simde_mm_i64gather_epi32(i32gather_buffer, vindex, 4);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_mask_i64gather_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t src[4];
    const int64_t vindex[2];
    const int32_t mask[4];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(  1478898212),  INT32_C(   916774907), -INT32_C(  1556893248),  INT32_C(  1777183058) },
      {  INT64_C(                 141),  INT64_C(                 139) },
      {            INT32_MIN,            INT32_MIN,  INT32_C(           0),  INT32_C(           0) },
      {  INT32_C(         141),  INT32_C(         139),  INT32_C(           0),  INT32_C(           0) } },
    { { -INT32_C(   632300097), -INT32_C(   462958966),  INT32_C(  1851006215),  INT32_C(   721091466) },
      {  INT64_C(                 157),  INT64_C(                 177) },
      {            INT32_MIN,            INT32_MIN,            INT32_MIN,            INT32_MIN },
      {  INT32_C(         157),  INT32_C(         177),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT32_C(  1526041333), -INT32_C(  1124607967), -INT32_C(  1106894900),  INT32_C(   879726651) },
      {  INT64_C(                   0),  INT64_C(                 233) },
      {            INT32_MIN,            INT32_MIN,  INT32_C(           0),            INT32_MIN },
      {  INT32_C(           0),  INT32_C(         233),  INT32_C(           0),  INT32_C(           0) } },
    { { -INT32_C(  1009155372),  INT32_C(  2126747810),  INT32_C(  1779523445), -INT32_C(  1420614464) },
      {  INT64_C(                  44),  INT64_C(                 205) },
      {            INT32_MIN,  INT32_C(           0),            INT32_MIN,  INT32_C(           0) },
      {  INT32_C(          44),  INT32_C(  2126747810),  INT32_C(           0),  INT32_C(           0) } },
    { { -INT32_C(  2036541516), -INT32_C(  1464708264),  INT32_C(  1817736563),  INT32_C(   289001730) },
      {  INT64_C(                 210),  INT64_C(                   6) },
      {  INT32_C(           0),            INT32_MIN,  INT32_C(           0),  INT32_C(           0) },
      { -INT32_C(  2036541516),  INT32_C(           6),  INT32_C(           0),  INT32_C(           0) } },
    { {  INT32_C(   529894144), -INT32_C(  1242496641), -INT32_C(  1991166154), -INT32_C(   661684580) },
      {  INT64_C(                  15),  INT64_C(                  39) },
      {            INT32_MIN,  INT32_C(           0),            INT32_MIN,            INT32_MIN },
      {  INT32_C(          15), -INT32_C(  1242496641),  INT32_C(           0),  INT32_C(           0) } },
    { { -INT32_C(  2000072659),  INT32_C(   932691705), -INT32_C(   673489744),  INT32_C(    16648425) },
      {  INT64_C(                 234),  INT64_C(                  70) },
      {  INT32_C(           0),            INT32_MIN,  INT32_C(           0),            INT32_MIN },
      { -INT32_C(  2000072659),  INT32_C(          70),  INT32_C(           0),  INT32_C(           0) } },
    { { -INT32_C(   913030322), -INT32_C(  1531700955),  INT32_C(   960408096), -INT32_C(  1367393148) },
      {  INT64_C(                 118),  INT64_C(                  32) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),            INT32_MIN },
      { -INT32_C(   913030322), -INT32_C(  1531700955),  INT32_C(           0),  INT32_C(           0) } }
  };

  for (size_t i = 0 ; i < (sizeof(i32gather_buffer) / sizeof(i32gather_buffer[0])) ; i++) { i32gather_buffer[i] = HEDLEY_STATIC_CAST(int32_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi32(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi64(test_vec[i].vindex);
    simde__m128i mask = simde_x_mm_loadu_epi32(test_vec[i].mask);
    simde__m128i r = simde_mm_mask_i64gather_epi32(src, i32gather_buffer, vindex, mask, 4);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_i64gather_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t vindex[4];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT64_C(                 164),  INT64_C(                 255),  INT64_C(                  34),  INT64_C(                 127) },
      {  INT32_C(         164),  INT32_C(         255),  INT32_C(          34),  INT32_C(         127) } },
    { {  INT64_C(                  52),  INT64_C(                  61),  INT64_C(                  40),  INT64_C(                  26) },
      {  INT32_C(          52),  INT32_C(          61),  INT32_C(          40),  INT32_C(          26) } },
    { {  INT64_C(                 166),  INT64_C(                 126),  INT64_C(                 130),  INT64_C(                  24) },
      {  INT32_C(         166),  INT32_C(         126),  INT32_C(         130),  INT32_C(          24) } },
    { {  INT64_C(                 246),  INT64_C(                  51),  INT64_C(                  78),  INT64_C(                 212) },
      {  INT32_C(         246),  INT32_C(          51),  INT32_C(          78),  INT32_C(         212) } },
    { {  INT64_C(                 104),  INT64_C(                 184),  INT64_C(                  15),  INT64_C(                 222) },
      {  INT32_C(         104),  INT32_C(         184),  INT32_C(          15),  INT32_C(         222) } },
    { {  INT64_C(                 136),  INT64_C(                  54),  INT64_C(                 141),  INT64_C(                  30) },
      {  INT32_C(         136),  INT32_C(          54),  INT32_C(         141),  INT32_C(          30) } },
    { {  INT64_C(                 103),  INT64_C(                 148),  INT64_C(                 191),  INT64_C(                 239) },
      {  INT32_C(         103),  INT32_C(         148),  INT32_C(         191),  INT32_C(         239) } },
    { {  INT64_C(                 123),  INT64_C(                 179),  INT64_C(                  92),  INT64_C(                 156) },
      {  INT32_C(         123),  INT32_C(         179),  INT32_C(          92),  INT32_C(         156) } }
  };
  for (size_t i = 0 ; i < (sizeof(i32gather_buffer) / sizeof(i32gather_buffer[0])) ; i++) { i32gather_buffer[i] = HEDLEY_STATIC_CAST(int32_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i vindex = simde_x_mm256_loadu_epi64(test_vec[i].vindex);
    simde__m128i r = simde_mm256_i64gather_epi32(i32gather_buffer, vindex, 4);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_i64gather_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t src[4];
    const int64_t vindex[4];
    const int32_t mask[4];
    const int32_t r[4];
  } test_vec[] = {
    { { -INT32_C(  1914974857), -INT32_C(   268305992),  INT32_C(  1600826892),  INT32_C(    91051765) },
      {  INT64_C(                  21),  INT64_C(                 250),  INT64_C(                  89),  INT64_C(                  48) },
      {  INT32_C(           0),  INT32_C(           0),            INT32_MIN,  INT32_C(           0) },
      { -INT32_C(  1914974857), -INT32_C(   268305992),  INT32_C(          89),  INT32_C(    91051765) } },
    { { -INT32_C(   807128191), -INT32_C(   215463748),  INT32_C(   975167766),  INT32_C(   627724550) },
      {  INT64_C(                  89),  INT64_C(                  70),  INT64_C(                 162),  INT64_C(                 179) },
      {  INT32_C(           0),  INT32_C(           0),            INT32_MIN,            INT32_MIN },
      { -INT32_C(   807128191), -INT32_C(   215463748),  INT32_C(         162),  INT32_C(         179) } },
    { { -INT32_C(   569972142),  INT32_C(  1199611944),  INT32_C(  1668045913), -INT32_C(   770263134) },
      {  INT64_C(                  96),  INT64_C(                  18),  INT64_C(                 116),  INT64_C(                   3) },
      {            INT32_MIN,  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) },
      {  INT32_C(          96),  INT32_C(  1199611944),  INT32_C(  1668045913), -INT32_C(   770263134) } },
    { {  INT32_C(   209437937),  INT32_C(    25204532),  INT32_C(  1584355103), -INT32_C(  1738428347) },
      {  INT64_C(                 249),  INT64_C(                 196),  INT64_C(                 215),  INT64_C(                 197) },
      {  INT32_C(           0),  INT32_C(           0),            INT32_MIN,            INT32_MIN },
      {  INT32_C(   209437937),  INT32_C(    25204532),  INT32_C(         215),  INT32_C(         197) } },
    { {  INT32_C(  1431392925),  INT32_C(  1278007459),  INT32_C(  1966760398),  INT32_C(   926662903) },
      {  INT64_C(                 228),  INT64_C(                 100),  INT64_C(                 202),  INT64_C(                  39) },
      {  INT32_C(           0),            INT32_MIN,            INT32_MIN,            INT32_MIN },
      {  INT32_C(  1431392925),  INT32_C(         100),  INT32_C(         202),  INT32_C(          39) } },
    { {  INT32_C(   286484245), -INT32_C(  1831110836),  INT32_C(  1942952725),  INT32_C(  2140816278) },
      {  INT64_C(                  25),  INT64_C(                 234),  INT64_C(                 181),  INT64_C(                 145) },
      {            INT32_MIN,  INT32_C(           0),            INT32_MIN,  INT32_C(           0) },
      {  INT32_C(          25), -INT32_C(  1831110836),  INT32_C(         181),  INT32_C(  2140816278) } },
    { {  INT32_C(   478193020),  INT32_C(  1842534011),  INT32_C(  1693907963), -INT32_C(   520749634) },
      {  INT64_C(                  40),  INT64_C(                  39),  INT64_C(                 239),  INT64_C(                 122) },
      {  INT32_C(           0),            INT32_MIN,            INT32_MIN,            INT32_MIN },
      {  INT32_C(   478193020),  INT32_C(          39),  INT32_C(         239),  INT32_C(         122) } },
    { { -INT32_C(   683278108),  INT32_C(   667313686), -INT32_C(  1862854276), -INT32_C(   552950175) },
      {  INT64_C(                 248),  INT64_C(                 188),  INT64_C(                  99),  INT64_C(                  19) },
      {  INT32_C(           0),            INT32_MIN,  INT32_C(           0),            INT32_MIN },
      { -INT32_C(   683278108),  INT32_C(         188), -INT32_C(  1862854276),  INT32_C(          19) } }
  };

  for (size_t i = 0 ; i < (sizeof(i32gather_buffer) / sizeof(i32gather_buffer[0])) ; i++) { i32gather_buffer[i] = HEDLEY_STATIC_CAST(int32_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi32(test_vec[i].src);
    simde__m256i vindex = simde_x_mm256_loadu_epi64(test_vec[i].vindex);
    simde__m128i mask = simde_x_mm_loadu_epi32(test_vec[i].mask);
    simde__m128i r = simde_mm256_mask_i64gather_epi32(src, i32gather_buffer, vindex, mask, 4);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int64_t i64gather_buffer[4096];

static int
test_simde_mm_i32gather_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t vindex[4];
    const int64_t r[2];
  } test_vec[] = {
    { {  INT32_C(         213),  INT32_C(         182),  INT32_C(         137),  INT32_C(          61) },
      {  INT64_C(                 213),  INT64_C(                 182) } },
    { {  INT32_C(         194),  INT32_C(         187),  INT32_C(          51),  INT32_C(         119) },
      {  INT64_C(                 194),  INT64_C(                 187) } },
    { {  INT32_C(         228),  INT32_C(         167),  INT32_C(         116),  INT32_C(         164) },
      {  INT64_C(                 228),  INT64_C(                 167) } },
    { {  INT32_C(         165),  INT32_C(         240),  INT32_C(         155),  INT32_C(          11) },
      {  INT64_C(                 165),  INT64_C(                 240) } },
    { {  INT32_C(          24),  INT32_C(          15),  INT32_C(          38),  INT32_C(         204) },
      {  INT64_C(                  24),  INT64_C(                  15) } },
    { {  INT32_C(         213),  INT32_C(         248),  INT32_C(          99),  INT32_C(         197) },
      {  INT64_C(                 213),  INT64_C(                 248) } },
    { {  INT32_C(         246),  INT32_C(         206),  INT32_C(         198),  INT32_C(         189) },
      {  INT64_C(                 246),  INT64_C(                 206) } },
    { {  INT32_C(         177),  INT32_C(         184),  INT32_C(         118),  INT32_C(         235) },
      {  INT64_C(                 177),  INT64_C(                 184) } }
  };
  for (size_t i = 0 ; i < (sizeof(i64gather_buffer) / sizeof(i64gather_buffer[0])) ; i++) { i64gather_buffer[i] = HEDLEY_STATIC_CAST(int64_t, i); }
  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m128i r = simde_mm_i32gather_epi64(i64gather_buffer, vindex, 8);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_mask_i32gather_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t src[2];
    const int32_t vindex[4];
    const int64_t mask[2];
    const int64_t r[2];
  } test_vec[] = {
    { { -INT64_C( 8019067203857977457), -INT64_C( 8061722764401183075) },
      {  INT32_C(         183),  INT32_C(         168),  INT32_C(         220),  INT32_C(         145) },
      {  INT64_C(                   0),  INT64_C(                   0) },
      { -INT64_C( 8019067203857977457), -INT64_C( 8061722764401183075) } },
    { { -INT64_C(  919569984677581009), -INT64_C( 1648047741553663091) },
      {  INT32_C(         235),  INT32_C(         124),  INT32_C(         216),  INT32_C(         241) },
      {  INT64_C(                   0),                     INT64_MIN },
      { -INT64_C(  919569984677581009),  INT64_C(                 124) } },
    { { -INT64_C( 8634359355635040034), -INT64_C( 3764535099121455859) },
      {  INT32_C(         178),  INT32_C(         242),  INT32_C(          53),  INT32_C(           4) },
      {  INT64_C(                   0),  INT64_C(                   0) },
      { -INT64_C( 8634359355635040034), -INT64_C( 3764535099121455859) } },
    { {  INT64_C( 2549016659115993441), -INT64_C( 7458823204181079982) },
      {  INT32_C(          77),  INT32_C(          31),  INT32_C(         104),  INT32_C(         168) },
      {  INT64_C(                   0),  INT64_C(                   0) },
      {  INT64_C( 2549016659115993441), -INT64_C( 7458823204181079982) } },
    { {  INT64_C( 5881278022537621357), -INT64_C( 7234300930576215892) },
      {  INT32_C(         151),  INT32_C(          32),  INT32_C(          36),  INT32_C(          71) },
      {                     INT64_MIN,  INT64_C(                   0) },
      {  INT64_C(                 151), -INT64_C( 7234300930576215892) } },
    { {  INT64_C(  349012936767990930),  INT64_C( 1707613122462270608) },
      {  INT32_C(         126),  INT32_C(          80),  INT32_C(         233),  INT32_C(         146) },
      {  INT64_C(                   0),                     INT64_MIN },
      {  INT64_C(  349012936767990930),  INT64_C(                  80) } },
    { {  INT64_C( 4375785989412903820), -INT64_C( 4856981328699777274) },
      {  INT32_C(         103),  INT32_C(         227),  INT32_C(         125),  INT32_C(          99) },
      {                     INT64_MIN,  INT64_C(                   0) },
      {  INT64_C(                 103), -INT64_C( 4856981328699777274) } },
    { { -INT64_C( 7386706167218158950), -INT64_C( 7989056672551773465) },
      {  INT32_C(         107),  INT32_C(          21),  INT32_C(         236),  INT32_C(         252) },
      {  INT64_C(                   0),  INT64_C(                   0) },
      { -INT64_C( 7386706167218158950), -INT64_C( 7989056672551773465) } }
  };
  for (size_t i = 0 ; i < (sizeof(i64gather_buffer) / sizeof(i64gather_buffer[0])) ; i++) { i64gather_buffer[i] = HEDLEY_STATIC_CAST(int64_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi64(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m128i mask = simde_x_mm_loadu_epi64(test_vec[i].mask);
    simde__m128i r = simde_mm_mask_i32gather_epi64(src, i64gather_buffer, vindex, mask, 8);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_i32gather_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t vindex[4];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT32_C(         222),  INT32_C(          35),  INT32_C(         153),  INT32_C(         134) },
      {  INT64_C(                 222),  INT64_C(                  35),  INT64_C(                 153),  INT64_C(                 134) } },
    { {  INT32_C(          57),  INT32_C(         242),  INT32_C(         165),  INT32_C(         182) },
      {  INT64_C(                  57),  INT64_C(                 242),  INT64_C(                 165),  INT64_C(                 182) } },
    { {  INT32_C(         137),  INT32_C(         208),  INT32_C(         222),  INT32_C(          21) },
      {  INT64_C(                 137),  INT64_C(                 208),  INT64_C(                 222),  INT64_C(                  21) } },
    { {  INT32_C(          45),  INT32_C(          10),  INT32_C(         128),  INT32_C(          27) },
      {  INT64_C(                  45),  INT64_C(                  10),  INT64_C(                 128),  INT64_C(                  27) } },
    { {  INT32_C(          23),  INT32_C(           3),  INT32_C(         105),  INT32_C(         216) },
      {  INT64_C(                  23),  INT64_C(                   3),  INT64_C(                 105),  INT64_C(                 216) } },
    { {  INT32_C(          28),  INT32_C(         117),  INT32_C(         171),  INT32_C(         117) },
      {  INT64_C(                  28),  INT64_C(                 117),  INT64_C(                 171),  INT64_C(                 117) } },
    { {  INT32_C(         238),  INT32_C(          69),  INT32_C(         187),  INT32_C(         167) },
      {  INT64_C(                 238),  INT64_C(                  69),  INT64_C(                 187),  INT64_C(                 167) } },
    { {  INT32_C(         243),  INT32_C(          37),  INT32_C(         145),  INT32_C(         129) },
      {  INT64_C(                 243),  INT64_C(                  37),  INT64_C(                 145),  INT64_C(                 129) } }
  };
  for (size_t i = 0 ; i < (sizeof(i64gather_buffer) / sizeof(i64gather_buffer[0])) ; i++) { i64gather_buffer[i] = HEDLEY_STATIC_CAST(int64_t, i); }
  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m256i r = simde_mm256_i32gather_epi64(i64gather_buffer, vindex, 8);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_i32gather_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t src[4];
    const int32_t vindex[4];
    const int64_t mask[4];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT64_C( 3369142246547613560),  INT64_C( 7744810680774779185), -INT64_C( 7552795409020785613),  INT64_C( 1387622825387924421) },
      {  INT32_C(          64),  INT32_C(          70),  INT32_C(          80),  INT32_C(         245) },
      {  INT64_C(                   0),                     INT64_MIN,  INT64_C(                   0),  INT64_C(                   0) },
      {  INT64_C( 3369142246547613560),  INT64_C(                  70), -INT64_C( 7552795409020785613),  INT64_C( 1387622825387924421) } },
    { {  INT64_C( 3103440719973267570), -INT64_C( 1919373598316996544), -INT64_C( 6201068952727107105), -INT64_C( 2614573325330952241) },
      {  INT32_C(          72),  INT32_C(          23),  INT32_C(         132),  INT32_C(          99) },
      {                     INT64_MIN,  INT64_C(                   0),                     INT64_MIN,                     INT64_MIN },
      {  INT64_C(                  72), -INT64_C( 1919373598316996544),  INT64_C(                 132),  INT64_C(                  99) } },
    { { -INT64_C( 1752065872900610483),  INT64_C( 8405587345429565333),  INT64_C( 1653618346573801900), -INT64_C( 7161726922272969511) },
      {  INT32_C(         116),  INT32_C(          71),  INT32_C(         153),  INT32_C(         134) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),                     INT64_MIN },
      { -INT64_C( 1752065872900610483),  INT64_C( 8405587345429565333),  INT64_C( 1653618346573801900),  INT64_C(                 134) } },
    { {  INT64_C( 1257101528912083106), -INT64_C(   77386101739500697), -INT64_C( 6860348042974974061),  INT64_C( 6311937495677122959) },
      {  INT32_C(         104),  INT32_C(         134),  INT32_C(         114),  INT32_C(          96) },
      {  INT64_C(                   0),  INT64_C(                   0),                     INT64_MIN,  INT64_C(                   0) },
      {  INT64_C( 1257101528912083106), -INT64_C(   77386101739500697),  INT64_C(                 114),  INT64_C( 6311937495677122959) } },
    { { -INT64_C( 2861233286667249227),  INT64_C( 2994139127905595224),  INT64_C( 1902462180205604141),  INT64_C( 6568991147572350640) },
      {  INT32_C(          20),  INT32_C(           9),  INT32_C(         185),  INT32_C(          83) },
      {                     INT64_MIN,                     INT64_MIN,  INT64_C(                   0),  INT64_C(                   0) },
      {  INT64_C(                  20),  INT64_C(                   9),  INT64_C( 1902462180205604141),  INT64_C( 6568991147572350640) } },
    { { -INT64_C( 4429501574753673542), -INT64_C( 8290183239203434124),  INT64_C( 9208322051927348199), -INT64_C( 3990225492664517654) },
      {  INT32_C(         199),  INT32_C(          26),  INT32_C(         173),  INT32_C(          45) },
      {  INT64_C(                   0),                     INT64_MIN,                     INT64_MIN,                     INT64_MIN },
      { -INT64_C( 4429501574753673542),  INT64_C(                  26),  INT64_C(                 173),  INT64_C(                  45) } },
    { {  INT64_C( 5138321306576926676),  INT64_C( 3583659973203966119), -INT64_C( 5991628195680088859), -INT64_C( 7026477670733466586) },
      {  INT32_C(         168),  INT32_C(         117),  INT32_C(         131),  INT32_C(          38) },
      {                     INT64_MIN,  INT64_C(                   0),  INT64_C(                   0),                     INT64_MIN },
      {  INT64_C(                 168),  INT64_C( 3583659973203966119), -INT64_C( 5991628195680088859),  INT64_C(                  38) } },
    { { -INT64_C( 5654470186572994433), -INT64_C(    2076098425164798),  INT64_C( 4968725845142138655), -INT64_C( 4249484596989279597) },
      {  INT32_C(         199),  INT32_C(         152),  INT32_C(         117),  INT32_C(          16) },
      {                     INT64_MIN,                     INT64_MIN,  INT64_C(                   0),  INT64_C(                   0) },
      {  INT64_C(                 199),  INT64_C(                 152),  INT64_C( 4968725845142138655), -INT64_C( 4249484596989279597) } }
  };
  for (size_t i = 0 ; i < (sizeof(i64gather_buffer) / sizeof(i64gather_buffer[0])) ; i++) { i64gather_buffer[i] = HEDLEY_STATIC_CAST(int64_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi64(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m256i mask = simde_x_mm256_loadu_epi64(test_vec[i].mask);
    simde__m256i r = simde_mm256_mask_i32gather_epi64(src, i64gather_buffer, vindex, mask, 8);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_i64gather_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t vindex[2];
    const int64_t r[2];
  } test_vec[] = {
    { {  INT64_C(                   0),  INT64_C(                 234) },
      {  INT64_C(                   0),  INT64_C(                 234) } },
    { {  INT64_C(                  22),  INT64_C(                 245) },
      {  INT64_C(                  22),  INT64_C(                 245) } },
    { {  INT64_C(                 151),  INT64_C(                 230) },
      {  INT64_C(                 151),  INT64_C(                 230) } },
    { {  INT64_C(                 184),  INT64_C(                 194) },
      {  INT64_C(                 184),  INT64_C(                 194) } },
    { {  INT64_C(                 213),  INT64_C(                 241) },
      {  INT64_C(                 213),  INT64_C(                 241) } },
    { {  INT64_C(                 191),  INT64_C(                 195) },
      {  INT64_C(                 191),  INT64_C(                 195) } },
    { {  INT64_C(                 174),  INT64_C(                 128) },
      {  INT64_C(                 174),  INT64_C(                 128) } },
    { {  INT64_C(                 186),  INT64_C(                  50) },
      {  INT64_C(                 186),  INT64_C(                  50) } }
  };
  for (size_t i = 0 ; i < (sizeof(i64gather_buffer) / sizeof(i64gather_buffer[0])) ; i++) { i64gather_buffer[i] = HEDLEY_STATIC_CAST(int64_t, i); }
  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi64(test_vec[i].vindex);
    simde__m128i r = simde_mm_i64gather_epi64(i64gather_buffer, vindex, 8);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_mask_i64gather_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t src[2];
    const int64_t vindex[2];
    const int64_t mask[2];
    const int64_t r[2];
  } test_vec[] = {
    { { -INT64_C( 2403370840913539277),  INT64_C( 5164194594589355633) },
      {  INT64_C(                  73),  INT64_C(                 179) },
      {  INT64_C(                   0),  INT64_C(                   0) },
      { -INT64_C( 2403370840913539277),  INT64_C( 5164194594589355633) } },
    { {  INT64_C(  935513772061340819),  INT64_C( 7894743952153131099) },
      {  INT64_C(                  36),  INT64_C(                  84) },
      {  INT64_C(                   0),                     INT64_MIN },
      {  INT64_C(  935513772061340819),  INT64_C(                  84) } },
    { {  INT64_C( 6537362916212700834), -INT64_C( 8981235666207658170) },
      {  INT64_C(                 157),  INT64_C(                 192) },
      {  INT64_C(                   0),                     INT64_MIN },
      {  INT64_C( 6537362916212700834),  INT64_C(                 192) } },
    { { -INT64_C( 1290141850782853098),  INT64_C( 6360822988240042599) },
      {  INT64_C(                 102),  INT64_C(                  84) },
      {  INT64_C(                   0),                     INT64_MIN },
      { -INT64_C( 1290141850782853098),  INT64_C(                  84) } },
    { {  INT64_C( 6060159807846979595), -INT64_C( 2729284862310875763) },
      {  INT64_C(                 118),  INT64_C(                 230) },
      {  INT64_C(                   0),  INT64_C(                   0) },
      {  INT64_C( 6060159807846979595), -INT64_C( 2729284862310875763) } },
    { { -INT64_C( 4478319151860018035), -INT64_C( 5765320646130876869) },
      {  INT64_C(                 255),  INT64_C(                  33) },
      {                     INT64_MIN,                     INT64_MIN },
      {  INT64_C(                 255),  INT64_C(                  33) } },
    { {  INT64_C( 6993169290329548479), -INT64_C( 5736351597774397777) },
      {  INT64_C(                  95),  INT64_C(                 101) },
      {                     INT64_MIN,                     INT64_MIN },
      {  INT64_C(                  95),  INT64_C(                 101) } },
    { {  INT64_C( 1384741590647677331), -INT64_C( 5107012464534872811) },
      {  INT64_C(                  42),  INT64_C(                 182) },
      {  INT64_C(                   0),                     INT64_MIN },
      {  INT64_C( 1384741590647677331),  INT64_C(                 182) } }
  };

  for (size_t i = 0 ; i < (sizeof(i64gather_buffer) / sizeof(i64gather_buffer[0])) ; i++) { i64gather_buffer[i] = HEDLEY_STATIC_CAST(int64_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi64(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi64(test_vec[i].vindex);
    simde__m128i mask = simde_x_mm_loadu_epi64(test_vec[i].mask);
    simde__m128i r = simde_mm_mask_i64gather_epi64(src, i64gather_buffer, vindex, mask, 8);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_i64gather_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t vindex[4];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT64_C(                 226),  INT64_C(                 163),  INT64_C(                 130),  INT64_C(                 153) },
      {  INT64_C(                 226),  INT64_C(                 163),  INT64_C(                 130),  INT64_C(                 153) } },
    { {  INT64_C(                 200),  INT64_C(                  98),  INT64_C(                   2),  INT64_C(                  50) },
      {  INT64_C(                 200),  INT64_C(                  98),  INT64_C(                   2),  INT64_C(                  50) } },
    { {  INT64_C(                  16),  INT64_C(                  32),  INT64_C(                 233),  INT64_C(                 157) },
      {  INT64_C(                  16),  INT64_C(                  32),  INT64_C(                 233),  INT64_C(                 157) } },
    { {  INT64_C(                 136),  INT64_C(                 125),  INT64_C(                 114),  INT64_C(                  31) },
      {  INT64_C(                 136),  INT64_C(                 125),  INT64_C(                 114),  INT64_C(                  31) } },
    { {  INT64_C(                 170),  INT64_C(                 197),  INT64_C(                  24),  INT64_C(                  71) },
      {  INT64_C(                 170),  INT64_C(                 197),  INT64_C(                  24),  INT64_C(                  71) } },
    { {  INT64_C(                 171),  INT64_C(                 221),  INT64_C(                 156),  INT64_C(                 172) },
      {  INT64_C(                 171),  INT64_C(                 221),  INT64_C(                 156),  INT64_C(                 172) } },
    { {  INT64_C(                 165),  INT64_C(                 159),  INT64_C(                 250),  INT64_C(                 110) },
      {  INT64_C(                 165),  INT64_C(                 159),  INT64_C(                 250),  INT64_C(                 110) } },
    { {  INT64_C(                 135),  INT64_C(                 146),  INT64_C(                 141),  INT64_C(                 229) },
      {  INT64_C(                 135),  INT64_C(                 146),  INT64_C(                 141),  INT64_C(                 229) } }
  };
  for (size_t i = 0 ; i < (sizeof(i64gather_buffer) / sizeof(i64gather_buffer[0])) ; i++) { i64gather_buffer[i] = HEDLEY_STATIC_CAST(int64_t, i); }
  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i vindex = simde_x_mm256_loadu_epi64(test_vec[i].vindex);
    simde__m256i r = simde_mm256_i64gather_epi64(i64gather_buffer, vindex, 8);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_i64gather_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t src[4];
    const int64_t vindex[4];
    const int64_t mask[4];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT64_C( 8348925912988061266),  INT64_C( 4593288946901735853),  INT64_C( 4308902935962320046),  INT64_C( 5084668380847658629) },
      {  INT64_C(                  74),  INT64_C(                 225),  INT64_C(                  80),  INT64_C(                 129) },
      {  INT64_C(                   0),  INT64_C(                   0),                     INT64_MIN,                     INT64_MIN },
      {  INT64_C( 8348925912988061266),  INT64_C( 4593288946901735853),  INT64_C(                  80),  INT64_C(                 129) } },
    { {  INT64_C( 1819345364246751166),  INT64_C( 6003474620557878227),  INT64_C( 4745430889957619642), -INT64_C(  181287204719907732) },
      {  INT64_C(                 153),  INT64_C(                  47),  INT64_C(                 169),  INT64_C(                  18) },
      {                     INT64_MIN,  INT64_C(                   0),                     INT64_MIN,  INT64_C(                   0) },
      {  INT64_C(                 153),  INT64_C( 6003474620557878227),  INT64_C(                 169), -INT64_C(  181287204719907732) } },
    { { -INT64_C( 4982644635249127578),  INT64_C( 1248644929523017696), -INT64_C( 5881855532059445236), -INT64_C( 3838907038599371630) },
      {  INT64_C(                 169),  INT64_C(                 136),  INT64_C(                 124),  INT64_C(                 218) },
      {  INT64_C(                   0),                     INT64_MIN,                     INT64_MIN,                     INT64_MIN },
      { -INT64_C( 4982644635249127578),  INT64_C(                 136),  INT64_C(                 124),  INT64_C(                 218) } },
    { {  INT64_C( 3695928590331946469), -INT64_C( 3354854445674728064), -INT64_C( 7502590046551983353),  INT64_C( 2128958219520435651) },
      {  INT64_C(                 145),  INT64_C(                   0),  INT64_C(                 131),  INT64_C(                 115) },
      {  INT64_C(                   0),                     INT64_MIN,                     INT64_MIN,                     INT64_MIN },
      {  INT64_C( 3695928590331946469),  INT64_C(                   0),  INT64_C(                 131),  INT64_C(                 115) } },
    { { -INT64_C( 1365028202850547012),  INT64_C( 7416463151597603534),  INT64_C( 5974682522185630920),  INT64_C( 4041606267062842147) },
      {  INT64_C(                  11),  INT64_C(                 207),  INT64_C(                 105),  INT64_C(                  70) },
      {  INT64_C(                   0),                     INT64_MIN,  INT64_C(                   0),  INT64_C(                   0) },
      { -INT64_C( 1365028202850547012),  INT64_C(                 207),  INT64_C( 5974682522185630920),  INT64_C( 4041606267062842147) } },
    { { -INT64_C( 9156116817736860426),  INT64_C( 4444164503705539924),  INT64_C(  449387376615079680), -INT64_C( 2018570388200701601) },
      {  INT64_C(                  16),  INT64_C(                 116),  INT64_C(                 131),  INT64_C(                 179) },
      {                     INT64_MIN,  INT64_C(                   0),  INT64_C(                   0),                     INT64_MIN },
      {  INT64_C(                  16),  INT64_C( 4444164503705539924),  INT64_C(  449387376615079680),  INT64_C(                 179) } },
    { { -INT64_C( 4232490538612136547), -INT64_C( 7797947821915074691), -INT64_C(   42789732815080928),  INT64_C( 6176971354028858165) },
      {  INT64_C(                  52),  INT64_C(                 108),  INT64_C(                 215),  INT64_C(                 104) },
      {                     INT64_MIN,                     INT64_MIN,                     INT64_MIN,  INT64_C(                   0) },
      {  INT64_C(                  52),  INT64_C(                 108),  INT64_C(                 215),  INT64_C( 6176971354028858165) } },
    { {  INT64_C( 1917495916396991492), -INT64_C( 3737546505400132308), -INT64_C( 3617519179428978074),  INT64_C( 5470992047431380744) },
      {  INT64_C(                 131),  INT64_C(                  67),  INT64_C(                 187),  INT64_C(                  54) },
      {  INT64_C(                   0),  INT64_C(                   0),                     INT64_MIN,                     INT64_MIN },
      {  INT64_C( 1917495916396991492), -INT64_C( 3737546505400132308),  INT64_C(                 187),  INT64_C(                  54) } }
  };
  for (size_t i = 0 ; i < (sizeof(i64gather_buffer) / sizeof(i64gather_buffer[0])) ; i++) { i64gather_buffer[i] = HEDLEY_STATIC_CAST(int64_t, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi64(test_vec[i].src);
    simde__m256i vindex = simde_x_mm256_loadu_epi64(test_vec[i].vindex);
    simde__m256i mask = simde_x_mm256_loadu_epi64(test_vec[i].mask);
    simde__m256i r = simde_mm256_mask_i64gather_epi64(src, i64gather_buffer, vindex, mask, 8);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static simde_float32 f32gather_buffer[4096];

static int
test_simde_mm_i32gather_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t vindex[4];
    const simde_float32 r[4];
  } test_vec[] = {
    { {  INT32_C(         161),  INT32_C(          17),  INT32_C(          47),  INT32_C(         177) },
      { SIMDE_FLOAT32_C(   161.00), SIMDE_FLOAT32_C(    17.00), SIMDE_FLOAT32_C(    47.00), SIMDE_FLOAT32_C(   177.00) } },
    { {  INT32_C(         192),  INT32_C(         196),  INT32_C(          32),  INT32_C(          56) },
      { SIMDE_FLOAT32_C(   192.00), SIMDE_FLOAT32_C(   196.00), SIMDE_FLOAT32_C(    32.00), SIMDE_FLOAT32_C(    56.00) } },
    { {  INT32_C(         243),  INT32_C(         253),  INT32_C(         198),  INT32_C(         186) },
      { SIMDE_FLOAT32_C(   243.00), SIMDE_FLOAT32_C(   253.00), SIMDE_FLOAT32_C(   198.00), SIMDE_FLOAT32_C(   186.00) } },
    { {  INT32_C(         201),  INT32_C(          25),  INT32_C(          83),  INT32_C(          64) },
      { SIMDE_FLOAT32_C(   201.00), SIMDE_FLOAT32_C(    25.00), SIMDE_FLOAT32_C(    83.00), SIMDE_FLOAT32_C(    64.00) } },
    { {  INT32_C(         173),  INT32_C(         203),  INT32_C(          45),  INT32_C(          53) },
      { SIMDE_FLOAT32_C(   173.00), SIMDE_FLOAT32_C(   203.00), SIMDE_FLOAT32_C(    45.00), SIMDE_FLOAT32_C(    53.00) } },
    { {  INT32_C(         232),  INT32_C(         218),  INT32_C(          33),  INT32_C(         228) },
      { SIMDE_FLOAT32_C(   232.00), SIMDE_FLOAT32_C(   218.00), SIMDE_FLOAT32_C(    33.00), SIMDE_FLOAT32_C(   228.00) } },
    { {  INT32_C(          26),  INT32_C(          51),  INT32_C(         182),  INT32_C(         102) },
      { SIMDE_FLOAT32_C(    26.00), SIMDE_FLOAT32_C(    51.00), SIMDE_FLOAT32_C(   182.00), SIMDE_FLOAT32_C(   102.00) } },
    { {  INT32_C(           4),  INT32_C(         101),  INT32_C(         200),  INT32_C(         229) },
      { SIMDE_FLOAT32_C(     4.00), SIMDE_FLOAT32_C(   101.00), SIMDE_FLOAT32_C(   200.00), SIMDE_FLOAT32_C(   229.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f32gather_buffer) / sizeof(f32gather_buffer[0])) ; i++) { f32gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float32, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m128 r = simde_mm_i32gather_ps(f32gather_buffer, vindex, 4);
    simde_test_x86_assert_equal_f32x4(r, simde_mm_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm_mask_i32gather_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 src[4];
    const int32_t vindex[4];
    const simde_float32 mask[4];
    const simde_float32 r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   349.70), SIMDE_FLOAT32_C(  -696.38), SIMDE_FLOAT32_C(   625.38), SIMDE_FLOAT32_C(    53.26) },
      {  INT32_C(          66),  INT32_C(         235),  INT32_C(         243),  INT32_C(          24) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   349.70), SIMDE_FLOAT32_C(   235.00), SIMDE_FLOAT32_C(   625.38), SIMDE_FLOAT32_C(    24.00) } },
    { { SIMDE_FLOAT32_C(  -959.90), SIMDE_FLOAT32_C(  -440.10), SIMDE_FLOAT32_C(   558.05), SIMDE_FLOAT32_C(   -71.77) },
      {  INT32_C(          40),  INT32_C(          16),  INT32_C(         251),  INT32_C(          54) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -959.90), SIMDE_FLOAT32_C(    16.00), SIMDE_FLOAT32_C(   558.05), SIMDE_FLOAT32_C(    54.00) } },
    { { SIMDE_FLOAT32_C(   152.90), SIMDE_FLOAT32_C(   196.28), SIMDE_FLOAT32_C(  -304.14), SIMDE_FLOAT32_C(    91.85) },
      {  INT32_C(          37),  INT32_C(          26),  INT32_C(         139),  INT32_C(         218) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   152.90), SIMDE_FLOAT32_C(    26.00), SIMDE_FLOAT32_C(  -304.14), SIMDE_FLOAT32_C(   218.00) } },
    { { SIMDE_FLOAT32_C(    91.05), SIMDE_FLOAT32_C(   444.96), SIMDE_FLOAT32_C(   775.00), SIMDE_FLOAT32_C(  -616.08) },
      {  INT32_C(          74),  INT32_C(         136),  INT32_C(         157),  INT32_C(          68) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(    91.05), SIMDE_FLOAT32_C(   136.00), SIMDE_FLOAT32_C(   775.00), SIMDE_FLOAT32_C(    68.00) } },
    { { SIMDE_FLOAT32_C(  -111.35), SIMDE_FLOAT32_C(   886.72), SIMDE_FLOAT32_C(   255.61), SIMDE_FLOAT32_C(  -399.36) },
      {  INT32_C(         126),  INT32_C(         154),  INT32_C(         177),  INT32_C(          65) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -111.35), SIMDE_FLOAT32_C(   154.00), SIMDE_FLOAT32_C(   255.61), SIMDE_FLOAT32_C(    65.00) } },
    { { SIMDE_FLOAT32_C(    48.78), SIMDE_FLOAT32_C(   652.36), SIMDE_FLOAT32_C(  -139.19), SIMDE_FLOAT32_C(  -584.66) },
      {  INT32_C(         215),  INT32_C(          81),  INT32_C(         153),  INT32_C(          75) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(    48.78), SIMDE_FLOAT32_C(    81.00), SIMDE_FLOAT32_C(  -139.19), SIMDE_FLOAT32_C(    75.00) } },
    { { SIMDE_FLOAT32_C(    39.45), SIMDE_FLOAT32_C(   995.37), SIMDE_FLOAT32_C(   468.08), SIMDE_FLOAT32_C(   -82.06) },
      {  INT32_C(         220),  INT32_C(         183),  INT32_C(         125),  INT32_C(          44) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(    39.45), SIMDE_FLOAT32_C(   183.00), SIMDE_FLOAT32_C(   468.08), SIMDE_FLOAT32_C(    44.00) } },
    { { SIMDE_FLOAT32_C(   883.35), SIMDE_FLOAT32_C(  -889.08), SIMDE_FLOAT32_C(   355.42), SIMDE_FLOAT32_C(   899.85) },
      {  INT32_C(         173),  INT32_C(         192),  INT32_C(         223),  INT32_C(         168) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   883.35), SIMDE_FLOAT32_C(   192.00), SIMDE_FLOAT32_C(   355.42), SIMDE_FLOAT32_C(   168.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(f32gather_buffer) / sizeof(f32gather_buffer[0])) ; i++) { f32gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float32, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 src = simde_mm_loadu_ps(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m128 mask = simde_mm_loadu_ps(test_vec[i].mask);
    simde__m128 r = simde_mm_mask_i32gather_ps(src, f32gather_buffer, vindex, mask, 4);
    simde_test_x86_assert_equal_f32x4(r, simde_mm_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_i32gather_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t vindex[8];
    const simde_float32 r[8];
  } test_vec[] = {
    { {  INT32_C(         156),  INT32_C(         110),  INT32_C(         202),  INT32_C(          30),  INT32_C(          85),  INT32_C(          63),  INT32_C(         139),  INT32_C(          34) },
      { SIMDE_FLOAT32_C(   156.00), SIMDE_FLOAT32_C(   110.00), SIMDE_FLOAT32_C(   202.00), SIMDE_FLOAT32_C(    30.00),
        SIMDE_FLOAT32_C(    85.00), SIMDE_FLOAT32_C(    63.00), SIMDE_FLOAT32_C(   139.00), SIMDE_FLOAT32_C(    34.00) } },
    { {  INT32_C(          20),  INT32_C(         209),  INT32_C(          84),  INT32_C(         116),  INT32_C(         159),  INT32_C(           9),  INT32_C(          45),  INT32_C(         240) },
      { SIMDE_FLOAT32_C(    20.00), SIMDE_FLOAT32_C(   209.00), SIMDE_FLOAT32_C(    84.00), SIMDE_FLOAT32_C(   116.00),
        SIMDE_FLOAT32_C(   159.00), SIMDE_FLOAT32_C(     9.00), SIMDE_FLOAT32_C(    45.00), SIMDE_FLOAT32_C(   240.00) } },
    { {  INT32_C(         229),  INT32_C(          18),  INT32_C(         146),  INT32_C(         118),  INT32_C(          19),  INT32_C(         136),  INT32_C(         196),  INT32_C(         132) },
      { SIMDE_FLOAT32_C(   229.00), SIMDE_FLOAT32_C(    18.00), SIMDE_FLOAT32_C(   146.00), SIMDE_FLOAT32_C(   118.00),
        SIMDE_FLOAT32_C(    19.00), SIMDE_FLOAT32_C(   136.00), SIMDE_FLOAT32_C(   196.00), SIMDE_FLOAT32_C(   132.00) } },
    { {  INT32_C(          95),  INT32_C(         188),  INT32_C(          60),  INT32_C(         152),  INT32_C(          20),  INT32_C(          16),  INT32_C(         188),  INT32_C(         106) },
      { SIMDE_FLOAT32_C(    95.00), SIMDE_FLOAT32_C(   188.00), SIMDE_FLOAT32_C(    60.00), SIMDE_FLOAT32_C(   152.00),
        SIMDE_FLOAT32_C(    20.00), SIMDE_FLOAT32_C(    16.00), SIMDE_FLOAT32_C(   188.00), SIMDE_FLOAT32_C(   106.00) } },
    { {  INT32_C(         126),  INT32_C(          56),  INT32_C(         197),  INT32_C(           2),  INT32_C(         172),  INT32_C(          70),  INT32_C(          92),  INT32_C(          90) },
      { SIMDE_FLOAT32_C(   126.00), SIMDE_FLOAT32_C(    56.00), SIMDE_FLOAT32_C(   197.00), SIMDE_FLOAT32_C(     2.00),
        SIMDE_FLOAT32_C(   172.00), SIMDE_FLOAT32_C(    70.00), SIMDE_FLOAT32_C(    92.00), SIMDE_FLOAT32_C(    90.00) } },
    { {  INT32_C(         179),  INT32_C(         221),  INT32_C(         122),  INT32_C(         117),  INT32_C(         214),  INT32_C(          77),  INT32_C(         138),  INT32_C(          77) },
      { SIMDE_FLOAT32_C(   179.00), SIMDE_FLOAT32_C(   221.00), SIMDE_FLOAT32_C(   122.00), SIMDE_FLOAT32_C(   117.00),
        SIMDE_FLOAT32_C(   214.00), SIMDE_FLOAT32_C(    77.00), SIMDE_FLOAT32_C(   138.00), SIMDE_FLOAT32_C(    77.00) } },
    { {  INT32_C(         201),  INT32_C(         109),  INT32_C(         209),  INT32_C(          78),  INT32_C(          64),  INT32_C(         182),  INT32_C(         193),  INT32_C(         166) },
      { SIMDE_FLOAT32_C(   201.00), SIMDE_FLOAT32_C(   109.00), SIMDE_FLOAT32_C(   209.00), SIMDE_FLOAT32_C(    78.00),
        SIMDE_FLOAT32_C(    64.00), SIMDE_FLOAT32_C(   182.00), SIMDE_FLOAT32_C(   193.00), SIMDE_FLOAT32_C(   166.00) } },
    { {  INT32_C(         179),  INT32_C(          70),  INT32_C(         166),  INT32_C(         103),  INT32_C(          13),  INT32_C(         186),  INT32_C(          19),  INT32_C(          11) },
      { SIMDE_FLOAT32_C(   179.00), SIMDE_FLOAT32_C(    70.00), SIMDE_FLOAT32_C(   166.00), SIMDE_FLOAT32_C(   103.00),
        SIMDE_FLOAT32_C(    13.00), SIMDE_FLOAT32_C(   186.00), SIMDE_FLOAT32_C(    19.00), SIMDE_FLOAT32_C(    11.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f32gather_buffer) / sizeof(f32gather_buffer[0])) ; i++) { f32gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float32, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i vindex = simde_x_mm256_loadu_epi32(test_vec[i].vindex);
    simde__m256 r = simde_mm256_i32gather_ps(f32gather_buffer, vindex, 4);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_mask_i32gather_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 src[8];
    const int32_t vindex[8];
    const simde_float32 mask[8];
    const simde_float32 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   265.75), SIMDE_FLOAT32_C(   153.07), SIMDE_FLOAT32_C(   737.27), SIMDE_FLOAT32_C(   671.90),
        SIMDE_FLOAT32_C(   596.53), SIMDE_FLOAT32_C(   857.73), SIMDE_FLOAT32_C(   355.81), SIMDE_FLOAT32_C(   117.96) },
      {  INT32_C(         116),  INT32_C(          49),  INT32_C(           1),  INT32_C(         154),  INT32_C(         238),  INT32_C(         237),  INT32_C(          63),  INT32_C(          95) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   265.75), SIMDE_FLOAT32_C(    49.00), SIMDE_FLOAT32_C(   737.27), SIMDE_FLOAT32_C(   154.00),
        SIMDE_FLOAT32_C(   596.53), SIMDE_FLOAT32_C(   237.00), SIMDE_FLOAT32_C(   355.81), SIMDE_FLOAT32_C(    95.00) } },
    { { SIMDE_FLOAT32_C(   355.76), SIMDE_FLOAT32_C(  -146.74), SIMDE_FLOAT32_C(   908.75), SIMDE_FLOAT32_C(   492.00),
        SIMDE_FLOAT32_C(  -494.15), SIMDE_FLOAT32_C(  -429.18), SIMDE_FLOAT32_C(   884.15), SIMDE_FLOAT32_C(   245.56) },
      {  INT32_C(         197),  INT32_C(         188),  INT32_C(         203),  INT32_C(           9),  INT32_C(         115),  INT32_C(         247),  INT32_C(          86),  INT32_C(         118) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   355.76), SIMDE_FLOAT32_C(   188.00), SIMDE_FLOAT32_C(   908.75), SIMDE_FLOAT32_C(     9.00),
        SIMDE_FLOAT32_C(  -494.15), SIMDE_FLOAT32_C(   247.00), SIMDE_FLOAT32_C(   884.15), SIMDE_FLOAT32_C(   118.00) } },
    { { SIMDE_FLOAT32_C(  -488.70), SIMDE_FLOAT32_C(   468.34), SIMDE_FLOAT32_C(  -802.96), SIMDE_FLOAT32_C(   324.69),
        SIMDE_FLOAT32_C(  -471.47), SIMDE_FLOAT32_C(  -786.76), SIMDE_FLOAT32_C(  -838.26), SIMDE_FLOAT32_C(   905.98) },
      {  INT32_C(         142),  INT32_C(          89),  INT32_C(         225),  INT32_C(          62),  INT32_C(         206),  INT32_C(         139),  INT32_C(          54),  INT32_C(         104) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -488.70), SIMDE_FLOAT32_C(    89.00), SIMDE_FLOAT32_C(  -802.96), SIMDE_FLOAT32_C(    62.00),
        SIMDE_FLOAT32_C(  -471.47), SIMDE_FLOAT32_C(   139.00), SIMDE_FLOAT32_C(  -838.26), SIMDE_FLOAT32_C(   104.00) } },
    { { SIMDE_FLOAT32_C(  -743.20), SIMDE_FLOAT32_C(   452.48), SIMDE_FLOAT32_C(   844.69), SIMDE_FLOAT32_C(   377.91),
        SIMDE_FLOAT32_C(  -629.91), SIMDE_FLOAT32_C(  -501.68), SIMDE_FLOAT32_C(   485.08), SIMDE_FLOAT32_C(   859.54) },
      {  INT32_C(         110),  INT32_C(          70),  INT32_C(           8),  INT32_C(         120),  INT32_C(         248),  INT32_C(          78),  INT32_C(         213),  INT32_C(          81) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -743.20), SIMDE_FLOAT32_C(    70.00), SIMDE_FLOAT32_C(   844.69), SIMDE_FLOAT32_C(   120.00),
        SIMDE_FLOAT32_C(  -629.91), SIMDE_FLOAT32_C(    78.00), SIMDE_FLOAT32_C(   485.08), SIMDE_FLOAT32_C(    81.00) } },
    { { SIMDE_FLOAT32_C(  -292.29), SIMDE_FLOAT32_C(  -955.02), SIMDE_FLOAT32_C(  -774.96), SIMDE_FLOAT32_C(   472.21),
        SIMDE_FLOAT32_C(   826.00), SIMDE_FLOAT32_C(  -866.19), SIMDE_FLOAT32_C(   101.43), SIMDE_FLOAT32_C(  -113.16) },
      {  INT32_C(         154),  INT32_C(          23),  INT32_C(         119),  INT32_C(          42),  INT32_C(         169),  INT32_C(         248),  INT32_C(          64),  INT32_C(         209) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -292.29), SIMDE_FLOAT32_C(    23.00), SIMDE_FLOAT32_C(  -774.96), SIMDE_FLOAT32_C(    42.00),
        SIMDE_FLOAT32_C(   826.00), SIMDE_FLOAT32_C(   248.00), SIMDE_FLOAT32_C(   101.43), SIMDE_FLOAT32_C(   209.00) } },
    { { SIMDE_FLOAT32_C(    24.55), SIMDE_FLOAT32_C(   652.58), SIMDE_FLOAT32_C(   245.18), SIMDE_FLOAT32_C(  -784.38),
        SIMDE_FLOAT32_C(  -205.61), SIMDE_FLOAT32_C(  -224.89), SIMDE_FLOAT32_C(  -769.96), SIMDE_FLOAT32_C(   -33.65) },
      {  INT32_C(          77),  INT32_C(         199),  INT32_C(         107),  INT32_C(          90),  INT32_C(          74),  INT32_C(          93),  INT32_C(          41),  INT32_C(          69) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(    24.55), SIMDE_FLOAT32_C(   199.00), SIMDE_FLOAT32_C(   245.18), SIMDE_FLOAT32_C(    90.00),
        SIMDE_FLOAT32_C(  -205.61), SIMDE_FLOAT32_C(    93.00), SIMDE_FLOAT32_C(  -769.96), SIMDE_FLOAT32_C(    69.00) } },
    { { SIMDE_FLOAT32_C(   709.81), SIMDE_FLOAT32_C(  -130.13), SIMDE_FLOAT32_C(  -541.91), SIMDE_FLOAT32_C(  -394.05),
        SIMDE_FLOAT32_C(   142.57), SIMDE_FLOAT32_C(  -331.04), SIMDE_FLOAT32_C(   771.89), SIMDE_FLOAT32_C(  -815.62) },
      {  INT32_C(          45),  INT32_C(         168),  INT32_C(         242),  INT32_C(          53),  INT32_C(         156),  INT32_C(          67),  INT32_C(         131),  INT32_C(         254) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   709.81), SIMDE_FLOAT32_C(   168.00), SIMDE_FLOAT32_C(  -541.91), SIMDE_FLOAT32_C(    53.00),
        SIMDE_FLOAT32_C(   142.57), SIMDE_FLOAT32_C(    67.00), SIMDE_FLOAT32_C(   771.89), SIMDE_FLOAT32_C(   254.00) } },
    { { SIMDE_FLOAT32_C(   913.20), SIMDE_FLOAT32_C(  -738.56), SIMDE_FLOAT32_C(   273.37), SIMDE_FLOAT32_C(  -405.28),
        SIMDE_FLOAT32_C(   609.46), SIMDE_FLOAT32_C(  -442.74), SIMDE_FLOAT32_C(   800.15), SIMDE_FLOAT32_C(   259.23) },
      {  INT32_C(          86),  INT32_C(         169),  INT32_C(         125),  INT32_C(          69),  INT32_C(          23),  INT32_C(         149),  INT32_C(         111),  INT32_C(         202) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   913.20), SIMDE_FLOAT32_C(   169.00), SIMDE_FLOAT32_C(   273.37), SIMDE_FLOAT32_C(    69.00),
        SIMDE_FLOAT32_C(   609.46), SIMDE_FLOAT32_C(   149.00), SIMDE_FLOAT32_C(   800.15), SIMDE_FLOAT32_C(   202.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(f32gather_buffer) / sizeof(f32gather_buffer[0])) ; i++) { f32gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float32, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256 src = simde_mm256_loadu_ps(test_vec[i].src);
    simde__m256i vindex = simde_x_mm256_loadu_epi32(test_vec[i].vindex);
    simde__m256 mask = simde_mm256_loadu_ps(test_vec[i].mask);
    simde__m256 r = simde_mm256_mask_i32gather_ps(src, f32gather_buffer, vindex, mask, 4);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm_i64gather_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t vindex[2];
    const simde_float32 r[4];
  } test_vec[] = {
    { {  INT64_C(                 255),  INT64_C(                  74) },
      { SIMDE_FLOAT32_C(   255.00), SIMDE_FLOAT32_C(    74.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { {  INT64_C(                 169),  INT64_C(                 122) },
      { SIMDE_FLOAT32_C(   169.00), SIMDE_FLOAT32_C(   122.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { {  INT64_C(                 224),  INT64_C(                 199) },
      { SIMDE_FLOAT32_C(   224.00), SIMDE_FLOAT32_C(   199.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { {  INT64_C(                  59),  INT64_C(                 158) },
      { SIMDE_FLOAT32_C(    59.00), SIMDE_FLOAT32_C(   158.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { {  INT64_C(                  50),  INT64_C(                   7) },
      { SIMDE_FLOAT32_C(    50.00), SIMDE_FLOAT32_C(     7.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { {  INT64_C(                 139),  INT64_C(                  78) },
      { SIMDE_FLOAT32_C(   139.00), SIMDE_FLOAT32_C(    78.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { {  INT64_C(                 100),  INT64_C(                 188) },
      { SIMDE_FLOAT32_C(   100.00), SIMDE_FLOAT32_C(   188.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { {  INT64_C(                 215),  INT64_C(                  83) },
      { SIMDE_FLOAT32_C(   215.00), SIMDE_FLOAT32_C(    83.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f32gather_buffer) / sizeof(f32gather_buffer[0])) ; i++) { f32gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float32, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi64(test_vec[i].vindex);
    simde__m128 r = simde_mm_i64gather_ps(f32gather_buffer, vindex, 4);
    simde_test_x86_assert_equal_f32x4(r, simde_mm_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm_mask_i64gather_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 src[4];
    const int64_t vindex[2];
    const simde_float32 mask[4];
    const simde_float32 r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -910.45), SIMDE_FLOAT32_C(   471.73), SIMDE_FLOAT32_C(  -109.25), SIMDE_FLOAT32_C(  -741.72) },
      {  INT64_C(                 149),  INT64_C(                 105) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -910.45), SIMDE_FLOAT32_C(   105.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { { SIMDE_FLOAT32_C(   501.51), SIMDE_FLOAT32_C(  -569.16), SIMDE_FLOAT32_C(  -498.09), SIMDE_FLOAT32_C(   591.93) },
      {  INT64_C(                 253),  INT64_C(                  55) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   501.51), SIMDE_FLOAT32_C(    55.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { { SIMDE_FLOAT32_C(  -500.50), SIMDE_FLOAT32_C(  -787.96), SIMDE_FLOAT32_C(    -8.71), SIMDE_FLOAT32_C(  -327.02) },
      {  INT64_C(                  86),  INT64_C(                   3) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -500.50), SIMDE_FLOAT32_C(     3.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { { SIMDE_FLOAT32_C(   268.16), SIMDE_FLOAT32_C(  -168.09), SIMDE_FLOAT32_C(   771.83), SIMDE_FLOAT32_C(   974.65) },
      {  INT64_C(                 183),  INT64_C(                 149) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   268.16), SIMDE_FLOAT32_C(   149.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { { SIMDE_FLOAT32_C(  -167.43), SIMDE_FLOAT32_C(   -25.72), SIMDE_FLOAT32_C(   800.73), SIMDE_FLOAT32_C(   362.38) },
      {  INT64_C(                  37),  INT64_C(                  58) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -167.43), SIMDE_FLOAT32_C(    58.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { { SIMDE_FLOAT32_C(   879.51), SIMDE_FLOAT32_C(  -467.85), SIMDE_FLOAT32_C(  -818.26), SIMDE_FLOAT32_C(   221.05) },
      {  INT64_C(                 234),  INT64_C(                 186) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   879.51), SIMDE_FLOAT32_C(   186.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { { SIMDE_FLOAT32_C(  -462.62), SIMDE_FLOAT32_C(  -246.74), SIMDE_FLOAT32_C(  -719.57), SIMDE_FLOAT32_C(   388.25) },
      {  INT64_C(                  12),  INT64_C(                 199) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -462.62), SIMDE_FLOAT32_C(   199.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { { SIMDE_FLOAT32_C(  -212.03), SIMDE_FLOAT32_C(  -784.09), SIMDE_FLOAT32_C(   477.31), SIMDE_FLOAT32_C(   490.83) },
      {  INT64_C(                  10),  INT64_C(                 106) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -212.03), SIMDE_FLOAT32_C(   106.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(f32gather_buffer) / sizeof(f32gather_buffer[0])) ; i++) { f32gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float32, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 src = simde_mm_loadu_ps(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi64(test_vec[i].vindex);
    simde__m128 mask = simde_mm_loadu_ps(test_vec[i].mask);
    simde__m128 r = simde_mm_mask_i64gather_ps(src, f32gather_buffer, vindex, mask, 4);
    simde_test_x86_assert_equal_f32x4(r, simde_mm_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_i64gather_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t vindex[4];
    const simde_float32 r[4];
  } test_vec[] = {
    { {  INT64_C(                 102),  INT64_C(                 248),  INT64_C(                 114),  INT64_C(                 223) },
      { SIMDE_FLOAT32_C(   102.00), SIMDE_FLOAT32_C(   248.00), SIMDE_FLOAT32_C(   114.00), SIMDE_FLOAT32_C(   223.00) } },
    { {  INT64_C(                 250),  INT64_C(                 132),  INT64_C(                 220),  INT64_C(                 173) },
      { SIMDE_FLOAT32_C(   250.00), SIMDE_FLOAT32_C(   132.00), SIMDE_FLOAT32_C(   220.00), SIMDE_FLOAT32_C(   173.00) } },
    { {  INT64_C(                  19),  INT64_C(                 230),  INT64_C(                  86),  INT64_C(                   7) },
      { SIMDE_FLOAT32_C(    19.00), SIMDE_FLOAT32_C(   230.00), SIMDE_FLOAT32_C(    86.00), SIMDE_FLOAT32_C(     7.00) } },
    { {  INT64_C(                 137),  INT64_C(                 241),  INT64_C(                 195),  INT64_C(                 198) },
      { SIMDE_FLOAT32_C(   137.00), SIMDE_FLOAT32_C(   241.00), SIMDE_FLOAT32_C(   195.00), SIMDE_FLOAT32_C(   198.00) } },
    { {  INT64_C(                 252),  INT64_C(                 215),  INT64_C(                  67),  INT64_C(                 180) },
      { SIMDE_FLOAT32_C(   252.00), SIMDE_FLOAT32_C(   215.00), SIMDE_FLOAT32_C(    67.00), SIMDE_FLOAT32_C(   180.00) } },
    { {  INT64_C(                 100),  INT64_C(                 137),  INT64_C(                 243),  INT64_C(                 185) },
      { SIMDE_FLOAT32_C(   100.00), SIMDE_FLOAT32_C(   137.00), SIMDE_FLOAT32_C(   243.00), SIMDE_FLOAT32_C(   185.00) } },
    { {  INT64_C(                 175),  INT64_C(                  29),  INT64_C(                 122),  INT64_C(                 116) },
      { SIMDE_FLOAT32_C(   175.00), SIMDE_FLOAT32_C(    29.00), SIMDE_FLOAT32_C(   122.00), SIMDE_FLOAT32_C(   116.00) } },
    { {  INT64_C(                  48),  INT64_C(                 182),  INT64_C(                  53),  INT64_C(                 192) },
      { SIMDE_FLOAT32_C(    48.00), SIMDE_FLOAT32_C(   182.00), SIMDE_FLOAT32_C(    53.00), SIMDE_FLOAT32_C(   192.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f32gather_buffer) / sizeof(f32gather_buffer[0])) ; i++) { f32gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float32, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i vindex = simde_x_mm256_loadu_epi64(test_vec[i].vindex);
    simde__m128 r = simde_mm256_i64gather_ps(f32gather_buffer, vindex, 4);
    simde_test_x86_assert_equal_f32x4(r, simde_mm_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_mask_i64gather_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 src[4];
    const int64_t vindex[4];
    const simde_float32 mask[4];
    const simde_float32 r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -557.85), SIMDE_FLOAT32_C(  -785.29), SIMDE_FLOAT32_C(  -569.41), SIMDE_FLOAT32_C(  -404.80) },
      {  INT64_C(                 204),  INT64_C(                 120),  INT64_C(                  37),  INT64_C(                 130) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -557.85), SIMDE_FLOAT32_C(   120.00), SIMDE_FLOAT32_C(  -569.41), SIMDE_FLOAT32_C(   130.00) } },
    { { SIMDE_FLOAT32_C(   831.45), SIMDE_FLOAT32_C(  -114.89), SIMDE_FLOAT32_C(  -779.68), SIMDE_FLOAT32_C(     6.19) },
      {  INT64_C(                 168),  INT64_C(                 241),  INT64_C(                 231),  INT64_C(                  98) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   831.45), SIMDE_FLOAT32_C(   241.00), SIMDE_FLOAT32_C(  -779.68), SIMDE_FLOAT32_C(    98.00) } },
    { { SIMDE_FLOAT32_C(  -926.84), SIMDE_FLOAT32_C(  -613.11), SIMDE_FLOAT32_C(   474.57), SIMDE_FLOAT32_C(  -308.37) },
      {  INT64_C(                 188),  INT64_C(                 117),  INT64_C(                 230),  INT64_C(                 107) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -926.84), SIMDE_FLOAT32_C(   117.00), SIMDE_FLOAT32_C(   474.57), SIMDE_FLOAT32_C(   107.00) } },
    { { SIMDE_FLOAT32_C(  -210.98), SIMDE_FLOAT32_C(  -337.89), SIMDE_FLOAT32_C(   -71.79), SIMDE_FLOAT32_C(  -805.12) },
      {  INT64_C(                 188),  INT64_C(                 100),  INT64_C(                 159),  INT64_C(                  80) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -210.98), SIMDE_FLOAT32_C(   100.00), SIMDE_FLOAT32_C(   -71.79), SIMDE_FLOAT32_C(    80.00) } },
    { { SIMDE_FLOAT32_C(  -900.97), SIMDE_FLOAT32_C(   631.07), SIMDE_FLOAT32_C(   698.37), SIMDE_FLOAT32_C(   195.85) },
      {  INT64_C(                  49),  INT64_C(                  72),  INT64_C(                 255),  INT64_C(                 112) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -900.97), SIMDE_FLOAT32_C(    72.00), SIMDE_FLOAT32_C(   698.37), SIMDE_FLOAT32_C(   112.00) } },
    { { SIMDE_FLOAT32_C(  -812.07), SIMDE_FLOAT32_C(    60.40), SIMDE_FLOAT32_C(  -955.01), SIMDE_FLOAT32_C(  -625.48) },
      {  INT64_C(                   7),  INT64_C(                 209),  INT64_C(                  12),  INT64_C(                  97) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -812.07), SIMDE_FLOAT32_C(   209.00), SIMDE_FLOAT32_C(  -955.01), SIMDE_FLOAT32_C(    97.00) } },
    { { SIMDE_FLOAT32_C(  -948.25), SIMDE_FLOAT32_C(    53.13), SIMDE_FLOAT32_C(  -171.41), SIMDE_FLOAT32_C(   245.75) },
      {  INT64_C(                 170),  INT64_C(                 104),  INT64_C(                  82),  INT64_C(                 123) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(  -948.25), SIMDE_FLOAT32_C(   104.00), SIMDE_FLOAT32_C(  -171.41), SIMDE_FLOAT32_C(   123.00) } },
    { { SIMDE_FLOAT32_C(   -63.25), SIMDE_FLOAT32_C(   125.71), SIMDE_FLOAT32_C(    69.11), SIMDE_FLOAT32_C(   -25.31) },
      {  INT64_C(                 145),  INT64_C(                  84),  INT64_C(                 244),  INT64_C(                  22) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-2147483648.00) },
      { SIMDE_FLOAT32_C(   -63.25), SIMDE_FLOAT32_C(    84.00), SIMDE_FLOAT32_C(    69.11), SIMDE_FLOAT32_C(    22.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(f32gather_buffer) / sizeof(f32gather_buffer[0])) ; i++) { f32gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float32, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 src = simde_mm_loadu_ps(test_vec[i].src);
    simde__m256i vindex = simde_x_mm256_loadu_epi64(test_vec[i].vindex);
    simde__m128 mask = simde_mm_loadu_ps(test_vec[i].mask);
    simde__m128 r = simde_mm256_mask_i64gather_ps(src, f32gather_buffer, vindex, mask, 4);
    simde_test_x86_assert_equal_f32x4(r, simde_mm_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static simde_float64 f64gather_buffer[4096];

static int
test_simde_mm_i32gather_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t vindex[4];
    const simde_float64 r[2];
  } test_vec[] = {
    { {  INT32_C(         151),  INT32_C(         228),  INT32_C(          22),  INT32_C(         128) },
      { SIMDE_FLOAT64_C(   151.00), SIMDE_FLOAT64_C(   228.00) } },
    { {  INT32_C(         162),  INT32_C(          84),  INT32_C(           9),  INT32_C(          70) },
      { SIMDE_FLOAT64_C(   162.00), SIMDE_FLOAT64_C(    84.00) } },
    { {  INT32_C(         218),  INT32_C(         181),  INT32_C(         119),  INT32_C(         133) },
      { SIMDE_FLOAT64_C(   218.00), SIMDE_FLOAT64_C(   181.00) } },
    { {  INT32_C(          22),  INT32_C(         114),  INT32_C(         193),  INT32_C(         112) },
      { SIMDE_FLOAT64_C(    22.00), SIMDE_FLOAT64_C(   114.00) } },
    { {  INT32_C(          79),  INT32_C(         111),  INT32_C(         117),  INT32_C(          77) },
      { SIMDE_FLOAT64_C(    79.00), SIMDE_FLOAT64_C(   111.00) } },
    { {  INT32_C(         123),  INT32_C(         187),  INT32_C(          81),  INT32_C(          26) },
      { SIMDE_FLOAT64_C(   123.00), SIMDE_FLOAT64_C(   187.00) } },
    { {  INT32_C(         171),  INT32_C(          12),  INT32_C(          85),  INT32_C(          56) },
      { SIMDE_FLOAT64_C(   171.00), SIMDE_FLOAT64_C(    12.00) } },
    { {  INT32_C(          64),  INT32_C(         180),  INT32_C(          87),  INT32_C(         104) },
      { SIMDE_FLOAT64_C(    64.00), SIMDE_FLOAT64_C(   180.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f64gather_buffer) / sizeof(f64gather_buffer[0])) ; i++) { f64gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float64, i); }

  #if defined(SIMDE_DIAGNOSTIC_DISABLE_UNINITIALIZED_) && HEDLEY_GCC_VERSION_CHECK(12,0,0)
    HEDLEY_DIAGNOSTIC_PUSH
    SIMDE_DIAGNOSTIC_DISABLE_UNINITIALIZED_
  #endif
  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m128d r = simde_mm_i32gather_pd(f64gather_buffer, vindex, 8);
    simde_test_x86_assert_equal_f64x2(r, simde_mm_loadu_pd(test_vec[i].r), 1);
  }
  #if defined(SIMDE_DIAGNOSTIC_DISABLE_UNINITIALIZED_) && HEDLEY_GCC_VERSION_CHECK(12,0,0)
    HEDLEY_DIAGNOSTIC_POP
  #endif

  return 0;
}

static int
test_simde_mm_mask_i32gather_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 src[2];
    const int32_t vindex[4];
    const simde_float64 mask[2];
    const simde_float64 r[2];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -480.33), SIMDE_FLOAT64_C(  -315.66) },
      {  INT32_C(         197),  INT32_C(           5),  INT32_C(          11),  INT32_C(         103) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -480.33), SIMDE_FLOAT64_C(     5.00) } },
    { { SIMDE_FLOAT64_C(  -766.75), SIMDE_FLOAT64_C(  -247.86) },
      {  INT32_C(          60),  INT32_C(         245),  INT32_C(          66),  INT32_C(         155) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -766.75), SIMDE_FLOAT64_C(   245.00) } },
    { { SIMDE_FLOAT64_C(  -243.19), SIMDE_FLOAT64_C(   794.07) },
      {  INT32_C(          29),  INT32_C(          87),  INT32_C(          99),  INT32_C(          99) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -243.19), SIMDE_FLOAT64_C(    87.00) } },
    { { SIMDE_FLOAT64_C(   424.26), SIMDE_FLOAT64_C(  -504.72) },
      {  INT32_C(          71),  INT32_C(          70),  INT32_C(         235),  INT32_C(         194) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   424.26), SIMDE_FLOAT64_C(    70.00) } },
    { { SIMDE_FLOAT64_C(   881.31), SIMDE_FLOAT64_C(  -768.55) },
      {  INT32_C(          65),  INT32_C(         188),  INT32_C(         232),  INT32_C(          73) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   881.31), SIMDE_FLOAT64_C(   188.00) } },
    { { SIMDE_FLOAT64_C(  -774.67), SIMDE_FLOAT64_C(   356.86) },
      {  INT32_C(         174),  INT32_C(         126),  INT32_C(         251),  INT32_C(          40) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -774.67), SIMDE_FLOAT64_C(   126.00) } },
    { { SIMDE_FLOAT64_C(  -351.27), SIMDE_FLOAT64_C(  -343.71) },
      {  INT32_C(         190),  INT32_C(         201),  INT32_C(          87),  INT32_C(          59) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -351.27), SIMDE_FLOAT64_C(   201.00) } },
    { { SIMDE_FLOAT64_C(  -898.19), SIMDE_FLOAT64_C(  -375.56) },
      {  INT32_C(         234),  INT32_C(         148),  INT32_C(         170),  INT32_C(          24) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -898.19), SIMDE_FLOAT64_C(   148.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f64gather_buffer) / sizeof(f64gather_buffer[0])) ; i++) { f64gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float64, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128d src = simde_mm_loadu_pd(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m128d mask = simde_mm_loadu_pd(test_vec[i].mask);
    simde__m128d r = simde_mm_mask_i32gather_pd(src, f64gather_buffer, vindex, mask, 8);
    simde_test_x86_assert_equal_f64x2(r, simde_mm_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_i32gather_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t vindex[4];
    const simde_float64 r[4];
  } test_vec[] = {
    { {  INT32_C(         177),  INT32_C(         141),  INT32_C(         100),  INT32_C(          45) },
      { SIMDE_FLOAT64_C(   177.00), SIMDE_FLOAT64_C(   141.00), SIMDE_FLOAT64_C(   100.00), SIMDE_FLOAT64_C(    45.00) } },
    { {  INT32_C(         150),  INT32_C(         114),  INT32_C(         229),  INT32_C(         101) },
      { SIMDE_FLOAT64_C(   150.00), SIMDE_FLOAT64_C(   114.00), SIMDE_FLOAT64_C(   229.00), SIMDE_FLOAT64_C(   101.00) } },
    { {  INT32_C(         226),  INT32_C(         173),  INT32_C(          89),  INT32_C(           9) },
      { SIMDE_FLOAT64_C(   226.00), SIMDE_FLOAT64_C(   173.00), SIMDE_FLOAT64_C(    89.00), SIMDE_FLOAT64_C(     9.00) } },
    { {  INT32_C(          29),  INT32_C(          54),  INT32_C(         230),  INT32_C(         153) },
      { SIMDE_FLOAT64_C(    29.00), SIMDE_FLOAT64_C(    54.00), SIMDE_FLOAT64_C(   230.00), SIMDE_FLOAT64_C(   153.00) } },
    { {  INT32_C(         130),  INT32_C(         112),  INT32_C(         170),  INT32_C(          75) },
      { SIMDE_FLOAT64_C(   130.00), SIMDE_FLOAT64_C(   112.00), SIMDE_FLOAT64_C(   170.00), SIMDE_FLOAT64_C(    75.00) } },
    { {  INT32_C(          81),  INT32_C(          83),  INT32_C(         244),  INT32_C(          81) },
      { SIMDE_FLOAT64_C(    81.00), SIMDE_FLOAT64_C(    83.00), SIMDE_FLOAT64_C(   244.00), SIMDE_FLOAT64_C(    81.00) } },
    { {  INT32_C(          29),  INT32_C(          15),  INT32_C(         115),  INT32_C(          79) },
      { SIMDE_FLOAT64_C(    29.00), SIMDE_FLOAT64_C(    15.00), SIMDE_FLOAT64_C(   115.00), SIMDE_FLOAT64_C(    79.00) } },
    { {  INT32_C(         131),  INT32_C(         207),  INT32_C(          76),  INT32_C(         123) },
      { SIMDE_FLOAT64_C(   131.00), SIMDE_FLOAT64_C(   207.00), SIMDE_FLOAT64_C(    76.00), SIMDE_FLOAT64_C(   123.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f64gather_buffer) / sizeof(f64gather_buffer[0])) ; i++) { f64gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float64, i); }

  #if defined(SIMDE_DIAGNOSTIC_DISABLE_UNINITIALIZED_) && HEDLEY_GCC_VERSION_CHECK(12,0,0)
    HEDLEY_DIAGNOSTIC_PUSH
    SIMDE_DIAGNOSTIC_DISABLE_UNINITIALIZED_
  #endif
  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m256d r = simde_mm256_i32gather_pd(f64gather_buffer, vindex, 8);
    simde_test_x86_assert_equal_f64x4(r, simde_mm256_loadu_pd(test_vec[i].r), 1);
  }
  #if defined(SIMDE_DIAGNOSTIC_DISABLE_UNINITIALIZED_) && HEDLEY_GCC_VERSION_CHECK(12,0,0)
    HEDLEY_DIAGNOSTIC_POP
  #endif

  return 0;
}

static int
test_simde_mm256_mask_i32gather_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 src[4];
    const int32_t vindex[4];
    const simde_float64 mask[4];
    const simde_float64 r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   267.18), SIMDE_FLOAT64_C(   752.26), SIMDE_FLOAT64_C(   216.84), SIMDE_FLOAT64_C(   987.12) },
      {  INT32_C(         123),  INT32_C(         197),  INT32_C(         179),  INT32_C(           0) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   267.18), SIMDE_FLOAT64_C(   197.00), SIMDE_FLOAT64_C(   216.84), SIMDE_FLOAT64_C(     0.00) } },
    { { SIMDE_FLOAT64_C(   608.28), SIMDE_FLOAT64_C(   322.28), SIMDE_FLOAT64_C(   399.60), SIMDE_FLOAT64_C(  -753.21) },
      {  INT32_C(         249),  INT32_C(          84),  INT32_C(         194),  INT32_C(          44) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   608.28), SIMDE_FLOAT64_C(    84.00), SIMDE_FLOAT64_C(   399.60), SIMDE_FLOAT64_C(    44.00) } },
    { { SIMDE_FLOAT64_C(   765.40), SIMDE_FLOAT64_C(   -73.02), SIMDE_FLOAT64_C(   385.76), SIMDE_FLOAT64_C(   103.67) },
      {  INT32_C(          55),  INT32_C(          66),  INT32_C(          86),  INT32_C(         189) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   765.40), SIMDE_FLOAT64_C(    66.00), SIMDE_FLOAT64_C(   385.76), SIMDE_FLOAT64_C(   189.00) } },
    { { SIMDE_FLOAT64_C(   170.06), SIMDE_FLOAT64_C(   333.86), SIMDE_FLOAT64_C(  -731.28), SIMDE_FLOAT64_C(  -526.76) },
      {  INT32_C(          11),  INT32_C(         174),  INT32_C(         199),  INT32_C(         205) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   170.06), SIMDE_FLOAT64_C(   174.00), SIMDE_FLOAT64_C(  -731.28), SIMDE_FLOAT64_C(   205.00) } },
    { { SIMDE_FLOAT64_C(  -655.08), SIMDE_FLOAT64_C(    43.94), SIMDE_FLOAT64_C(   458.52), SIMDE_FLOAT64_C(    50.01) },
      {  INT32_C(          17),  INT32_C(         204),  INT32_C(          61),  INT32_C(          41) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -655.08), SIMDE_FLOAT64_C(   204.00), SIMDE_FLOAT64_C(   458.52), SIMDE_FLOAT64_C(    41.00) } },
    { { SIMDE_FLOAT64_C(    91.50), SIMDE_FLOAT64_C(   334.64), SIMDE_FLOAT64_C(  -285.42), SIMDE_FLOAT64_C(   872.18) },
      {  INT32_C(         206),  INT32_C(          59),  INT32_C(           5),  INT32_C(         108) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(    91.50), SIMDE_FLOAT64_C(    59.00), SIMDE_FLOAT64_C(  -285.42), SIMDE_FLOAT64_C(   108.00) } },
    { { SIMDE_FLOAT64_C(  -329.65), SIMDE_FLOAT64_C(   179.37), SIMDE_FLOAT64_C(  -815.96), SIMDE_FLOAT64_C(   824.89) },
      {  INT32_C(         228),  INT32_C(         234),  INT32_C(          46),  INT32_C(          94) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -329.65), SIMDE_FLOAT64_C(   234.00), SIMDE_FLOAT64_C(  -815.96), SIMDE_FLOAT64_C(    94.00) } },
    { { SIMDE_FLOAT64_C(  -490.85), SIMDE_FLOAT64_C(   503.59), SIMDE_FLOAT64_C(  -623.45), SIMDE_FLOAT64_C(   519.67) },
      {  INT32_C(         127),  INT32_C(         223),  INT32_C(         120),  INT32_C(          66) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -490.85), SIMDE_FLOAT64_C(   223.00), SIMDE_FLOAT64_C(  -623.45), SIMDE_FLOAT64_C(    66.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f64gather_buffer) / sizeof(f64gather_buffer[0])) ; i++) { f64gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float64, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256d src = simde_mm256_loadu_pd(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi32(test_vec[i].vindex);
    simde__m256d mask = simde_mm256_loadu_pd(test_vec[i].mask);
    simde__m256d r = simde_mm256_mask_i32gather_pd(src, f64gather_buffer, vindex, mask, 8);
    simde_test_x86_assert_equal_f64x4(r, simde_mm256_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm_i64gather_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t vindex[2];
    const simde_float64 r[2];
  } test_vec[] = {
    { {  INT64_C(                  44),  INT64_C(                  35) },
      { SIMDE_FLOAT64_C(    44.00), SIMDE_FLOAT64_C(    35.00) } },
    { {  INT64_C(                  67),  INT64_C(                 111) },
      { SIMDE_FLOAT64_C(    67.00), SIMDE_FLOAT64_C(   111.00) } },
    { {  INT64_C(                 136),  INT64_C(                 227) },
      { SIMDE_FLOAT64_C(   136.00), SIMDE_FLOAT64_C(   227.00) } },
    { {  INT64_C(                 149),  INT64_C(                  42) },
      { SIMDE_FLOAT64_C(   149.00), SIMDE_FLOAT64_C(    42.00) } },
    { {  INT64_C(                 162),  INT64_C(                  73) },
      { SIMDE_FLOAT64_C(   162.00), SIMDE_FLOAT64_C(    73.00) } },
    { {  INT64_C(                  34),  INT64_C(                  86) },
      { SIMDE_FLOAT64_C(    34.00), SIMDE_FLOAT64_C(    86.00) } },
    { {  INT64_C(                   0),  INT64_C(                  81) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(    81.00) } },
    { {  INT64_C(                 121),  INT64_C(                  62) },
      { SIMDE_FLOAT64_C(   121.00), SIMDE_FLOAT64_C(    62.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f64gather_buffer) / sizeof(f64gather_buffer[0])) ; i++) { f64gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float64, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i vindex = simde_x_mm_loadu_epi64(test_vec[i].vindex);
    simde__m128d r = simde_mm_i64gather_pd(f64gather_buffer, vindex, 8);
    simde_test_x86_assert_equal_f64x2(r, simde_mm_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm_mask_i64gather_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 src[2];
    const int64_t vindex[2];
    const simde_float64 mask[2];
    const simde_float64 r[2];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   228.75), SIMDE_FLOAT64_C(   -39.16) },
      {  INT64_C(                 151),  INT64_C(                  61) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   228.75), SIMDE_FLOAT64_C(    61.00) } },
    { { SIMDE_FLOAT64_C(  -882.35), SIMDE_FLOAT64_C(   591.91) },
      {  INT64_C(                 204),  INT64_C(                 199) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -882.35), SIMDE_FLOAT64_C(   199.00) } },
    { { SIMDE_FLOAT64_C(   141.24), SIMDE_FLOAT64_C(   212.29) },
      {  INT64_C(                 117),  INT64_C(                 141) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   141.24), SIMDE_FLOAT64_C(   141.00) } },
    { { SIMDE_FLOAT64_C(  -950.10), SIMDE_FLOAT64_C(   891.20) },
      {  INT64_C(                  83),  INT64_C(                 248) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -950.10), SIMDE_FLOAT64_C(   248.00) } },
    { { SIMDE_FLOAT64_C(    89.46), SIMDE_FLOAT64_C(  -130.83) },
      {  INT64_C(                  70),  INT64_C(                 253) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(    89.46), SIMDE_FLOAT64_C(   253.00) } },
    { { SIMDE_FLOAT64_C(  -460.75), SIMDE_FLOAT64_C(  -728.93) },
      {  INT64_C(                   2),  INT64_C(                 214) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -460.75), SIMDE_FLOAT64_C(   214.00) } },
    { { SIMDE_FLOAT64_C(   878.60), SIMDE_FLOAT64_C(   965.00) },
      {  INT64_C(                  72),  INT64_C(                 241) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   878.60), SIMDE_FLOAT64_C(   241.00) } },
    { { SIMDE_FLOAT64_C(   564.46), SIMDE_FLOAT64_C(   962.41) },
      {  INT64_C(                 206),  INT64_C(                 147) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   564.46), SIMDE_FLOAT64_C(   147.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f64gather_buffer) / sizeof(f64gather_buffer[0])) ; i++) { f64gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float64, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128d src = simde_mm_loadu_pd(test_vec[i].src);
    simde__m128i vindex = simde_x_mm_loadu_epi64(test_vec[i].vindex);
    simde__m128d mask = simde_mm_loadu_pd(test_vec[i].mask);
    simde__m128d r = simde_mm_mask_i64gather_pd(src, f64gather_buffer, vindex, mask, 8);
    simde_test_x86_assert_equal_f64x2(r, simde_mm_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_i64gather_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t vindex[4];
    const simde_float64 r[4];
  } test_vec[] = {
    { {  INT64_C(                 155),  INT64_C(                 134),  INT64_C(                 102),  INT64_C(                  36) },
      { SIMDE_FLOAT64_C(   155.00), SIMDE_FLOAT64_C(   134.00), SIMDE_FLOAT64_C(   102.00), SIMDE_FLOAT64_C(    36.00) } },
    { {  INT64_C(                 188),  INT64_C(                  82),  INT64_C(                 119),  INT64_C(                 247) },
      { SIMDE_FLOAT64_C(   188.00), SIMDE_FLOAT64_C(    82.00), SIMDE_FLOAT64_C(   119.00), SIMDE_FLOAT64_C(   247.00) } },
    { {  INT64_C(                  60),  INT64_C(                   4),  INT64_C(                 193),  INT64_C(                 157) },
      { SIMDE_FLOAT64_C(    60.00), SIMDE_FLOAT64_C(     4.00), SIMDE_FLOAT64_C(   193.00), SIMDE_FLOAT64_C(   157.00) } },
    { {  INT64_C(                 191),  INT64_C(                 201),  INT64_C(                 171),  INT64_C(                 123) },
      { SIMDE_FLOAT64_C(   191.00), SIMDE_FLOAT64_C(   201.00), SIMDE_FLOAT64_C(   171.00), SIMDE_FLOAT64_C(   123.00) } },
    { {  INT64_C(                  25),  INT64_C(                  82),  INT64_C(                 149),  INT64_C(                  61) },
      { SIMDE_FLOAT64_C(    25.00), SIMDE_FLOAT64_C(    82.00), SIMDE_FLOAT64_C(   149.00), SIMDE_FLOAT64_C(    61.00) } },
    { {  INT64_C(                 213),  INT64_C(                  24),  INT64_C(                 205),  INT64_C(                  61) },
      { SIMDE_FLOAT64_C(   213.00), SIMDE_FLOAT64_C(    24.00), SIMDE_FLOAT64_C(   205.00), SIMDE_FLOAT64_C(    61.00) } },
    { {  INT64_C(                 212),  INT64_C(                  90),  INT64_C(                 134),  INT64_C(                  96) },
      { SIMDE_FLOAT64_C(   212.00), SIMDE_FLOAT64_C(    90.00), SIMDE_FLOAT64_C(   134.00), SIMDE_FLOAT64_C(    96.00) } },
    { {  INT64_C(                  73),  INT64_C(                  18),  INT64_C(                  98),  INT64_C(                 201) },
      { SIMDE_FLOAT64_C(    73.00), SIMDE_FLOAT64_C(    18.00), SIMDE_FLOAT64_C(    98.00), SIMDE_FLOAT64_C(   201.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f64gather_buffer) / sizeof(f64gather_buffer[0])) ; i++) { f64gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float64, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i vindex = simde_x_mm256_loadu_epi64(test_vec[i].vindex);
    simde__m256d r = simde_mm256_i64gather_pd(f64gather_buffer, vindex, 8);
    simde_test_x86_assert_equal_f64x4(r, simde_mm256_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_mask_i64gather_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 src[4];
    const int64_t vindex[4];
    const simde_float64 mask[4];
    const simde_float64 r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -225.70), SIMDE_FLOAT64_C(    -7.36), SIMDE_FLOAT64_C(   407.57), SIMDE_FLOAT64_C(   907.28) },
      {  INT64_C(                  44),  INT64_C(                 253),  INT64_C(                 186),  INT64_C(                 235) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -225.70), SIMDE_FLOAT64_C(   253.00), SIMDE_FLOAT64_C(   407.57), SIMDE_FLOAT64_C(   235.00) } },
    { { SIMDE_FLOAT64_C(  -714.19), SIMDE_FLOAT64_C(   993.64), SIMDE_FLOAT64_C(  -956.79), SIMDE_FLOAT64_C(   142.66) },
      {  INT64_C(                 117),  INT64_C(                   6),  INT64_C(                 107),  INT64_C(                 114) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -714.19), SIMDE_FLOAT64_C(     6.00), SIMDE_FLOAT64_C(  -956.79), SIMDE_FLOAT64_C(   114.00) } },
    { { SIMDE_FLOAT64_C(   927.00), SIMDE_FLOAT64_C(   732.99), SIMDE_FLOAT64_C(  -422.33), SIMDE_FLOAT64_C(   720.35) },
      {  INT64_C(                  41),  INT64_C(                   9),  INT64_C(                 229),  INT64_C(                  58) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   927.00), SIMDE_FLOAT64_C(     9.00), SIMDE_FLOAT64_C(  -422.33), SIMDE_FLOAT64_C(    58.00) } },
    { { SIMDE_FLOAT64_C(  -504.21), SIMDE_FLOAT64_C(  -460.73), SIMDE_FLOAT64_C(   337.37), SIMDE_FLOAT64_C(  -249.52) },
      {  INT64_C(                  86),  INT64_C(                  35),  INT64_C(                  77),  INT64_C(                   8) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -504.21), SIMDE_FLOAT64_C(    35.00), SIMDE_FLOAT64_C(   337.37), SIMDE_FLOAT64_C(     8.00) } },
    { { SIMDE_FLOAT64_C(  -359.51), SIMDE_FLOAT64_C(   323.22), SIMDE_FLOAT64_C(  -224.22), SIMDE_FLOAT64_C(   888.99) },
      {  INT64_C(                  88),  INT64_C(                 225),  INT64_C(                  40),  INT64_C(                  71) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -359.51), SIMDE_FLOAT64_C(   225.00), SIMDE_FLOAT64_C(  -224.22), SIMDE_FLOAT64_C(    71.00) } },
    { { SIMDE_FLOAT64_C(  -595.68), SIMDE_FLOAT64_C(     6.34), SIMDE_FLOAT64_C(    51.10), SIMDE_FLOAT64_C(    42.13) },
      {  INT64_C(                  61),  INT64_C(                  53),  INT64_C(                 127),  INT64_C(                 227) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -595.68), SIMDE_FLOAT64_C(    53.00), SIMDE_FLOAT64_C(    51.10), SIMDE_FLOAT64_C(   227.00) } },
    { { SIMDE_FLOAT64_C(   356.87), SIMDE_FLOAT64_C(   122.53), SIMDE_FLOAT64_C(  -844.94), SIMDE_FLOAT64_C(    51.26) },
      {  INT64_C(                  71),  INT64_C(                  43),  INT64_C(                 173),  INT64_C(                 232) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(   356.87), SIMDE_FLOAT64_C(    43.00), SIMDE_FLOAT64_C(  -844.94), SIMDE_FLOAT64_C(   232.00) } },
    { { SIMDE_FLOAT64_C(  -545.64), SIMDE_FLOAT64_C(   253.09), SIMDE_FLOAT64_C(  -327.77), SIMDE_FLOAT64_C(   486.99) },
      {  INT64_C(                  97),  INT64_C(                 231),  INT64_C(                 140),  INT64_C(                 243) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(-9223372036854775808.00) },
      { SIMDE_FLOAT64_C(  -545.64), SIMDE_FLOAT64_C(   231.00), SIMDE_FLOAT64_C(  -327.77), SIMDE_FLOAT64_C(   243.00) } }
  };
  for (size_t i = 0 ; i < (sizeof(f64gather_buffer) / sizeof(f64gather_buffer[0])) ; i++) { f64gather_buffer[i] = HEDLEY_STATIC_CAST(simde_float64, i); }

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256d src = simde_mm256_loadu_pd(test_vec[i].src);
    simde__m256i vindex = simde_x_mm256_loadu_epi64(test_vec[i].vindex);
    simde__m256d mask = simde_mm256_loadu_pd(test_vec[i].mask);
    simde__m256d r = simde_mm256_mask_i64gather_pd(src, f64gather_buffer, vindex, mask, 8);
    simde_test_x86_assert_equal_f64x4(r, simde_mm256_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_inserti128_si256 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[8];
    const int32_t b[4];
    const int32_t ra[8];
    const int32_t rb[8];
  } test_vec[] = {
    { { -INT32_C(    21047385), -INT32_C(   230819693), -INT32_C(  2121983276),  INT32_C(   325772553), -INT32_C(   910275464),  INT32_C(  1357952964),  INT32_C(  2017751920),  INT32_C(     9626969) },
      {  INT32_C(  1358909628),  INT32_C(   474102856),  INT32_C(  1604241494),  INT32_C(   628230317) },
      {  INT32_C(  1358909628),  INT32_C(   474102856),  INT32_C(  1604241494),  INT32_C(   628230317), -INT32_C(   910275464),  INT32_C(  1357952964),  INT32_C(  2017751920),  INT32_C(     9626969) },
      { -INT32_C(    21047385), -INT32_C(   230819693), -INT32_C(  2121983276),  INT32_C(   325772553),  INT32_C(  1358909628),  INT32_C(   474102856),  INT32_C(  1604241494),  INT32_C(   628230317) } },
    { {  INT32_C(   435106133),  INT32_C(  1483333608), -INT32_C(  1412321710),  INT32_C(  1353474963), -INT32_C(    73356365),  INT32_C(  1041752807),  INT32_C(  1486730666),  INT32_C(   326963390) },
      {  INT32_C(   690777153), -INT32_C(  1652451765), -INT32_C(   683060668),  INT32_C(  1747449269) },
      {  INT32_C(   690777153), -INT32_C(  1652451765), -INT32_C(   683060668),  INT32_C(  1747449269), -INT32_C(    73356365),  INT32_C(  1041752807),  INT32_C(  1486730666),  INT32_C(   326963390) },
      {  INT32_C(   435106133),  INT32_C(  1483333608), -INT32_C(  1412321710),  INT32_C(  1353474963),  INT32_C(   690777153), -INT32_C(  1652451765), -INT32_C(   683060668),  INT32_C(  1747449269) } },
    { { -INT32_C(  2023503968),  INT32_C(  1422228394), -INT32_C(   290692304), -INT32_C(  1274992013), -INT32_C(   488821098),  INT32_C(   142565060),  INT32_C(  1725941937),  INT32_C(  1573849021) },
      {  INT32_C(  2028286670), -INT32_C(   556946771), -INT32_C(  2134083315),  INT32_C(   976539299) },
      {  INT32_C(  2028286670), -INT32_C(   556946771), -INT32_C(  2134083315),  INT32_C(   976539299), -INT32_C(   488821098),  INT32_C(   142565060),  INT32_C(  1725941937),  INT32_C(  1573849021) },
      { -INT32_C(  2023503968),  INT32_C(  1422228394), -INT32_C(   290692304), -INT32_C(  1274992013),  INT32_C(  2028286670), -INT32_C(   556946771), -INT32_C(  2134083315),  INT32_C(   976539299) } },
    { { -INT32_C(  1071902212),  INT32_C(   566795120),  INT32_C(   562538340),  INT32_C(  2105497262),  INT32_C(   922051721),  INT32_C(   471122446), -INT32_C(   543366852), -INT32_C(  1424371281) },
      {  INT32_C(  1382757858),  INT32_C(   896742353), -INT32_C(  1990788134), -INT32_C(   637086383) },
      {  INT32_C(  1382757858),  INT32_C(   896742353), -INT32_C(  1990788134), -INT32_C(   637086383),  INT32_C(   922051721),  INT32_C(   471122446), -INT32_C(   543366852), -INT32_C(  1424371281) },
      { -INT32_C(  1071902212),  INT32_C(   566795120),  INT32_C(   562538340),  INT32_C(  2105497262),  INT32_C(  1382757858),  INT32_C(   896742353), -INT32_C(  1990788134), -INT32_C(   637086383) } },
    { {  INT32_C(  1209137977), -INT32_C(    94100034), -INT32_C(  1244069882), -INT32_C(  1268714543), -INT32_C(   116995288), -INT32_C(   651265282), -INT32_C(   966621835), -INT32_C(  1801361318) },
      {  INT32_C(   568111715), -INT32_C(   568639273),  INT32_C(   311686464),  INT32_C(   281474280) },
      {  INT32_C(   568111715), -INT32_C(   568639273),  INT32_C(   311686464),  INT32_C(   281474280), -INT32_C(   116995288), -INT32_C(   651265282), -INT32_C(   966621835), -INT32_C(  1801361318) },
      {  INT32_C(  1209137977), -INT32_C(    94100034), -INT32_C(  1244069882), -INT32_C(  1268714543),  INT32_C(   568111715), -INT32_C(   568639273),  INT32_C(   311686464),  INT32_C(   281474280) } },
    { { -INT32_C(  1106588481), -INT32_C(  1147717562),  INT32_C(   411236797), -INT32_C(   995351711), -INT32_C(  1394177835),  INT32_C(   143262152), -INT32_C(   568713482), -INT32_C(   772808686) },
      { -INT32_C(   208668243), -INT32_C(   273734095), -INT32_C(  2147012321),  INT32_C(   692433748) },
      { -INT32_C(   208668243), -INT32_C(   273734095), -INT32_C(  2147012321),  INT32_C(   692433748), -INT32_C(  1394177835),  INT32_C(   143262152), -INT32_C(   568713482), -INT32_C(   772808686) },
      { -INT32_C(  1106588481), -INT32_C(  1147717562),  INT32_C(   411236797), -INT32_C(   995351711), -INT32_C(   208668243), -INT32_C(   273734095), -INT32_C(  2147012321),  INT32_C(   692433748) } },
    { {  INT32_C(    64301883),  INT32_C(   587948076), -INT32_C(  1878972802), -INT32_C(  1285427194),  INT32_C(   463991273),  INT32_C(   923424279), -INT32_C(   608759417), -INT32_C(    16450364) },
      {  INT32_C(  1409472807), -INT32_C(  1216934599),  INT32_C(   977762355),  INT32_C(  1391307113) },
      {  INT32_C(  1409472807), -INT32_C(  1216934599),  INT32_C(   977762355),  INT32_C(  1391307113),  INT32_C(   463991273),  INT32_C(   923424279), -INT32_C(   608759417), -INT32_C(    16450364) },
      {  INT32_C(    64301883),  INT32_C(   587948076), -INT32_C(  1878972802), -INT32_C(  1285427194),  INT32_C(  1409472807), -INT32_C(  1216934599),  INT32_C(   977762355),  INT32_C(  1391307113) } },
    { { -INT32_C(  1318218598),  INT32_C(  1911060458),  INT32_C(  1280090248), -INT32_C(  1001697124),  INT32_C(  1662537002), -INT32_C(  1910796453),  INT32_C(  1892180487), -INT32_C(  1513900533) },
      {  INT32_C(   894906442),  INT32_C(   816201640),  INT32_C(  2088629215),  INT32_C(  1832962115) },
      {  INT32_C(   894906442),  INT32_C(   816201640),  INT32_C(  2088629215),  INT32_C(  1832962115),  INT32_C(  1662537002), -INT32_C(  1910796453),  INT32_C(  1892180487), -INT32_C(  1513900533) },
      { -INT32_C(  1318218598),  INT32_C(  1911060458),  INT32_C(  1280090248), -INT32_C(  1001697124),  INT32_C(   894906442),  INT32_C(   816201640),  INT32_C(  2088629215),  INT32_C(  1832962115) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m256i ra = simde_mm256_inserti128_si256(a, b, 0);
    simde__m256i rb = simde_mm256_inserti128_si256(a, b, 1);
    simde_test_x86_assert_equal_i32x8(ra, simde_x_mm256_loadu_epi32(test_vec[i].ra));
    simde_test_x86_assert_equal_i32x8(rb, simde_x_mm256_loadu_epi32(test_vec[i].rb));
  }

  return 0;
}

static int
test_simde_mm256_madd_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(-22074), INT16_C( 27892), INT16_C(-27402), INT16_C( -5185),
                            INT16_C(-13617), INT16_C(  6733), INT16_C( 31914), INT16_C( 16627),
                            INT16_C( 14296), INT16_C(   527), INT16_C(-18797), INT16_C( 25549),
                            INT16_C( 30759), INT16_C(-12360), INT16_C(  6414), INT16_C( 21507)),
      simde_mm256_set_epi16(INT16_C( 13087), INT16_C(   552), INT16_C(-32388), INT16_C( 21398),
                            INT16_C( -2724), INT16_C( 12527), INT16_C(  4881), INT16_C( -1411),
                            INT16_C(  4971), INT16_C(  -716), INT16_C(-15438), INT16_C( 22737),
                            INT16_C( 22649), INT16_C( 14157), INT16_C( 16131), INT16_C( 25742)),
      simde_mm256_set_epi32(INT32_C( -273486054), INT32_C(  776547346), INT32_C(  121436999), INT32_C(  132311537),
                            INT32_C(   70688084), INT32_C(  871095699), INT32_C(  521680071), INT32_C(  657097428)) },
    { simde_mm256_set_epi16(INT16_C( 31591), INT16_C( 27463), INT16_C(-11448), INT16_C( 12754),
                            INT16_C( 23360), INT16_C( 24732), INT16_C( 21501), INT16_C(  9477),
                            INT16_C(-29628), INT16_C( 15818), INT16_C( 22405), INT16_C(-17463),
                            INT16_C( -4252), INT16_C(  1744), INT16_C( 29107), INT16_C(  -236)),
      simde_mm256_set_epi16(INT16_C( 17376), INT16_C(-10713), INT16_C(  2486), INT16_C(  1967),
                            INT16_C(  3397), INT16_C( 25143), INT16_C( 11210), INT16_C( -8470),
                            INT16_C( -7431), INT16_C(-24781), INT16_C( 28646), INT16_C(-15461),
                            INT16_C( -7439), INT16_C(-17272), INT16_C(-12699), INT16_C( 15557)),
      simde_mm256_set_epi32(INT32_C(  254714097), INT32_C(   -3372610), INT32_C(  701190596), INT32_C(  160756020),
                            INT32_C( -171820190), INT32_C(  911809073), INT32_C(    1508260), INT32_C( -373301245)) },
    { simde_mm256_set_epi16(INT16_C(-23018), INT16_C( 27443), INT16_C( -9704), INT16_C(  6815),
                            INT16_C(-10742), INT16_C(-15199), INT16_C(-25926), INT16_C( 11907),
                            INT16_C(-11173), INT16_C( -1296), INT16_C(-31401), INT16_C(-11984),
                            INT16_C( 29153), INT16_C( 26221), INT16_C( 23204), INT16_C( 24693)),
      simde_mm256_set_epi16(INT16_C( 22883), INT16_C( -1430), INT16_C(-22444), INT16_C( -4173),
                            INT16_C( -9271), INT16_C(-22344), INT16_C(-19214), INT16_C(-15237),
                            INT16_C(  8913), INT16_C(-31636), INT16_C(-32308), INT16_C( 15943),
                            INT16_C(  4837), INT16_C( -6600), INT16_C(-14527), INT16_C(-29993)),
      simde_mm256_set_epi32(INT32_C( -565964384), INT32_C(  189357581), INT32_C(  439195538), INT32_C(  316715205),
                            INT32_C(  -58584693), INT32_C(  823442596), INT32_C(  -32045539), INT32_C(-1077701657)) },
    { simde_mm256_set_epi16(INT16_C( -2513), INT16_C( -1446), INT16_C( 20507), INT16_C(-25668),
                            INT16_C( 32595), INT16_C(  1090), INT16_C(  8204), INT16_C(  9120),
                            INT16_C( -8133), INT16_C( 31849), INT16_C(-18457), INT16_C(-12347),
                            INT16_C( 18795), INT16_C( -8246), INT16_C( 23278), INT16_C(-14987)),
      simde_mm256_set_epi16(INT16_C( 12529), INT16_C(-11077), INT16_C(  5410), INT16_C(-23993),
                            INT16_C(-26377), INT16_C( -6112), INT16_C(-21857), INT16_C(  3969),
                            INT16_C( -2477), INT16_C(-13689), INT16_C( 21824), INT16_C(  -644),
                            INT16_C( 10631), INT16_C(-12974), INT16_C(-28564), INT16_C( 32352)),
      simde_mm256_set_epi32(INT32_C(  -15468035), INT32_C(  726795194), INT32_C( -866420395), INT32_C( -143117548),
                            INT32_C( -415835520), INT32_C( -394854100), INT32_C(  306793249), INT32_C(-1149772216)) },
    { simde_mm256_set_epi16(INT16_C(  7571), INT16_C( 21562), INT16_C( 24839), INT16_C( 27056),
                            INT16_C(-18448), INT16_C( 29209), INT16_C(  1880), INT16_C(-21767),
                            INT16_C( 26198), INT16_C(-31641), INT16_C(  3244), INT16_C(-13098),
                            INT16_C( -4443), INT16_C(  -521), INT16_C(-27791), INT16_C(-13063)),
      simde_mm256_set_epi16(INT16_C(-17648), INT16_C(  8337), INT16_C( -3551), INT16_C(-28013),
                            INT16_C(-17930), INT16_C(   577), INT16_C( 32382), INT16_C( 20122),
                            INT16_C(-14435), INT16_C(-21581), INT16_C(-32759), INT16_C(  2792),
                            INT16_C(-24268), INT16_C( 11663), INT16_C( -1946), INT16_C( 19139)),
      simde_mm256_set_epi32(INT32_C(   46149386), INT32_C( -846123017), INT32_C(  347626233), INT32_C( -377117414),
                            INT32_C(  304676291), INT32_C( -142839812), INT32_C(  101746301), INT32_C( -195931471)) },
    { simde_mm256_set_epi16(INT16_C(-11993), INT16_C( -3203), INT16_C( 15681), INT16_C(-18383),
                            INT16_C( 16847), INT16_C(  2437), INT16_C( -8441), INT16_C( 14338),
                            INT16_C(  7300), INT16_C(-21082), INT16_C(-17580), INT16_C(  1429),
                            INT16_C(-32388), INT16_C(-10418), INT16_C(-19218), INT16_C(-31595)),
      simde_mm256_set_epi16(INT16_C(-24077), INT16_C(-32143), INT16_C( 18864), INT16_C( 11719),
                            INT16_C( -5555), INT16_C( -6220), INT16_C(-30370), INT16_C( 23594),
                            INT16_C( 29026), INT16_C( 10950), INT16_C( 31566), INT16_C(-28277),
                            INT16_C( 28777), INT16_C( 17653), INT16_C(-21542), INT16_C(  7263)),
      simde_mm256_set_epi32(INT32_C(  391709490), INT32_C(   80376007), INT32_C( -108743225), INT32_C(  594643942),
                            INT32_C(  -18958100), INT32_C( -595338113), INT32_C(-1115938430), INT32_C(  184519671)) },
    { simde_mm256_set_epi16(INT16_C(-31708), INT16_C( -6736), INT16_C(-12543), INT16_C(-19831),
                            INT16_C( -5660), INT16_C(-20189), INT16_C( 31844), INT16_C( 20882),
                            INT16_C(-22159), INT16_C( -7550), INT16_C(  8622), INT16_C(-20427),
                            INT16_C(-19784), INT16_C(  4739), INT16_C( -9632), INT16_C(  3416)),
      simde_mm256_set_epi16(INT16_C( 18684), INT16_C(  4347), INT16_C( 18074), INT16_C(-24539),
                            INT16_C( 14613), INT16_C(-28009), INT16_C( 18528), INT16_C(-16867),
                            INT16_C(  1794), INT16_C(-22885), INT16_C( 27546), INT16_C( 25964),
                            INT16_C( -3488), INT16_C(-31053), INT16_C(  6969), INT16_C(-12364)),
      simde_mm256_set_epi32(INT32_C( -621713664), INT32_C(  259930727), INT32_C(  482764121), INT32_C(  237788938),
                            INT32_C(  133028504), INT32_C( -292865016), INT32_C(  -78153575), INT32_C( -109360832)) },
    { simde_mm256_set_epi16(INT16_C( 32611), INT16_C(-26810), INT16_C( 32585), INT16_C(  6944),
                            INT16_C( 26033), INT16_C(  2406), INT16_C(-32105), INT16_C(-11091),
                            INT16_C(-32475), INT16_C(-29532), INT16_C( 13591), INT16_C(-15433),
                            INT16_C( 29089), INT16_C(-24035), INT16_C( 15832), INT16_C(-10008)),
      simde_mm256_set_epi16(INT16_C( 16370), INT16_C( -9733), INT16_C( -7456), INT16_C( -2346),
                            INT16_C( -1285), INT16_C( -5955), INT16_C(-24955), INT16_C( -4486),
                            INT16_C(  4170), INT16_C(-17666), INT16_C(  7986), INT16_C( 19848),
                            INT16_C(-13940), INT16_C( 24511), INT16_C( 18142), INT16_C(-26201)),
      simde_mm256_set_epi32(INT32_C(  794783800), INT32_C( -259244384), INT32_C(  -47780135), INT32_C(  850934501),
                            INT32_C(  386291562), INT32_C( -197776458), INT32_C( -994622545), INT32_C(  549443752)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_madd_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_maddubs_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint8_t a[32];
    const int8_t b[32];
    const int16_t r[16];
  } test_vec[] = {
    { { UINT8_C(109), UINT8_C(252), UINT8_C( 25), UINT8_C(124), UINT8_C(145), UINT8_C(132), UINT8_C( 41), UINT8_C(134),
        UINT8_C( 24), UINT8_C( 84), UINT8_C(233), UINT8_C(164), UINT8_C(189), UINT8_C( 12), UINT8_C(171), UINT8_C( 39),
        UINT8_C(233), UINT8_C(115), UINT8_C(157), UINT8_C(228), UINT8_C(143), UINT8_C(175), UINT8_C(252), UINT8_C( 18),
        UINT8_C(160), UINT8_C(  8), UINT8_C(174), UINT8_C(181), UINT8_C(235), UINT8_C( 97), UINT8_C(108), UINT8_C( 88) },
      {  INT8_C(  94), -INT8_C( 123), -INT8_C(  44), -INT8_C(  17),  INT8_C(  10), -INT8_C(   3),  INT8_C( 117),  INT8_C(  34),
         INT8_C(  81),  INT8_C(  94), -INT8_C(  58),  INT8_C(  15),  INT8_C( 107),  INT8_C( 113),  INT8_C(  54),  INT8_C(  84),
        -INT8_C(  27), -INT8_C(  44),  INT8_C(  56),  INT8_C( 116), -INT8_C( 125),  INT8_C(  52), -INT8_C( 122),  INT8_C(  36),
         INT8_C(  60),  INT8_C(  53), -INT8_C(  39),  INT8_C(  39), -INT8_C( 106),  INT8_C(  70),      INT8_MAX, -INT8_C(  12) },
      { -INT16_C( 20750), -INT16_C(  3208),  INT16_C(  1054),  INT16_C(  9353),  INT16_C(  9840), -INT16_C( 11054),  INT16_C( 21579),  INT16_C( 12510),
        -INT16_C( 11351),       INT16_MAX, -INT16_C(  8775), -INT16_C( 30096),  INT16_C( 10024),  INT16_C(   273), -INT16_C( 18120),  INT16_C( 12660) } },
    { { UINT8_C(203), UINT8_C( 83), UINT8_C(228), UINT8_C(213), UINT8_C( 80), UINT8_C( 89), UINT8_C(247), UINT8_C(162),
        UINT8_C(184), UINT8_C(190), UINT8_C(177), UINT8_C( 35), UINT8_C( 47), UINT8_C(231), UINT8_C(119), UINT8_C( 20),
        UINT8_C(187), UINT8_C(176), UINT8_C(137), UINT8_C( 63), UINT8_C(228), UINT8_C( 15), UINT8_C( 99), UINT8_C( 33),
        UINT8_C( 68), UINT8_C( 60), UINT8_C( 72), UINT8_C(219), UINT8_C(130), UINT8_C(200), UINT8_C(207), UINT8_C( 78) },
      {  INT8_C(  27), -INT8_C(  77),  INT8_C(  35),  INT8_C( 108),  INT8_C(  13),  INT8_C(  27),  INT8_C(  14), -INT8_C(  59),
        -INT8_C(  39), -INT8_C(  65), -INT8_C(  24),  INT8_C(   8), -INT8_C(  90),  INT8_C(  95),  INT8_C(  29),  INT8_C(  98),
         INT8_C(  15), -INT8_C(  90), -INT8_C(  95), -INT8_C(  12), -INT8_C(  75),  INT8_C(   4),  INT8_C(  21), -INT8_C(   6),
         INT8_C(  64),  INT8_C(  93), -INT8_C(  43), -INT8_C(  61),  INT8_C(  37), -INT8_C(  92),  INT8_C(  17),  INT8_C(  65) },
      { -INT16_C(   910),  INT16_C( 30984),  INT16_C(  3443), -INT16_C(  6100), -INT16_C( 19526), -INT16_C(  3968),  INT16_C( 17715),  INT16_C(  5411),
        -INT16_C( 13035), -INT16_C( 13771), -INT16_C( 17040),  INT16_C(  1881),  INT16_C(  9932), -INT16_C( 16455), -INT16_C( 13590),  INT16_C(  8589) } },
    { { UINT8_C( 88), UINT8_C( 52), UINT8_C(173), UINT8_C(101), UINT8_C( 79), UINT8_C(187), UINT8_C( 42), UINT8_C( 40),
        UINT8_C(122), UINT8_C( 18), UINT8_C( 49), UINT8_C( 32), UINT8_C(113), UINT8_C( 78), UINT8_C(130), UINT8_C(129),
        UINT8_C(244), UINT8_C( 35), UINT8_C(117), UINT8_C(169), UINT8_C( 39), UINT8_C(138), UINT8_C(163), UINT8_C(104),
        UINT8_C(231), UINT8_C(120), UINT8_C( 43), UINT8_C( 13), UINT8_C( 29), UINT8_C( 60), UINT8_C( 78), UINT8_C(117) },
      {  INT8_C( 112), -INT8_C(   5), -INT8_C(  38), -INT8_C(  64), -INT8_C(  74),  INT8_C(   4), -INT8_C(  24),  INT8_C(  48),
         INT8_C(  22),  INT8_C(  25),  INT8_C(  80), -INT8_C( 121),  INT8_C( 103), -INT8_C(  45),  INT8_C(   8),  INT8_C(  91),
        -INT8_C(  10),  INT8_C( 125),  INT8_C(   5),  INT8_C(  30),  INT8_C(   7), -INT8_C(  88), -INT8_C( 122), -INT8_C(  17),
         INT8_C(  33), -INT8_C(  79), -INT8_C(   4),  INT8_C(  62), -INT8_C(  19),  INT8_C(  74), -INT8_C(  77),  INT8_C(  93) },
      {  INT16_C(  9596), -INT16_C( 13038), -INT16_C(  5098),  INT16_C(   912),  INT16_C(  3134),  INT16_C(    48),  INT16_C(  8129),  INT16_C( 12779),
         INT16_C(  1935),  INT16_C(  5655), -INT16_C( 11871), -INT16_C( 21654), -INT16_C(  1857),  INT16_C(   634),  INT16_C(  3889),  INT16_C(  4875) } },
    { { UINT8_C( 69), UINT8_C(141), UINT8_C( 29), UINT8_C(251), UINT8_C(145), UINT8_C(  6), UINT8_C( 43), UINT8_C(167),
        UINT8_C( 31), UINT8_C(123), UINT8_C( 46), UINT8_C(135), UINT8_C( 78), UINT8_C( 55), UINT8_C(226), UINT8_C( 69),
        UINT8_C(180), UINT8_C(231), UINT8_C( 99), UINT8_C(188), UINT8_C(144), UINT8_C(233), UINT8_C(171), UINT8_C(177),
        UINT8_C(154), UINT8_C(167), UINT8_C(239), UINT8_C(135), UINT8_C(241), UINT8_C(162), UINT8_C(228), UINT8_C( 54) },
      {  INT8_C(  47),  INT8_C(   2),  INT8_C(  49), -INT8_C(  64),  INT8_C(   8),  INT8_C(  92),  INT8_C( 103),  INT8_C(  39),
        -INT8_C(  41), -INT8_C( 107), -INT8_C(  82),  INT8_C(  38), -INT8_C(  52), -INT8_C( 111),  INT8_C( 107), -INT8_C( 127),
         INT8_C( 120), -INT8_C(  50),  INT8_C(  61),  INT8_C(   8), -INT8_C(  73), -INT8_C(  24), -INT8_C(  71),  INT8_C(  81),
        -INT8_C( 113), -INT8_C(  88), -INT8_C(  40),      INT8_MIN,  INT8_C(  74), -INT8_C(  68), -INT8_C(  74),  INT8_C( 121) },
      {  INT16_C(  3525), -INT16_C( 14643),  INT16_C(  1712),  INT16_C( 10942), -INT16_C( 14432),  INT16_C(  1358), -INT16_C( 10161),  INT16_C( 15419),
         INT16_C( 10050),  INT16_C(  7543), -INT16_C( 16104),  INT16_C(  2196), -INT16_C( 32098), -INT16_C( 26840),  INT16_C(  6818), -INT16_C( 10338) } },
    { { UINT8_C(190), UINT8_C(231), UINT8_C( 57), UINT8_C(198), UINT8_C( 67), UINT8_C(160), UINT8_C(238), UINT8_C( 26),
        UINT8_C( 54), UINT8_C(156), UINT8_C( 64), UINT8_C(  2), UINT8_C( 45), UINT8_C(171), UINT8_C(131), UINT8_C(166),
        UINT8_C(121), UINT8_C(192), UINT8_C(174), UINT8_C( 48), UINT8_C(168), UINT8_C(104), UINT8_C(129), UINT8_C( 55),
        UINT8_C( 16), UINT8_C( 89), UINT8_C(183), UINT8_C( 91), UINT8_C( 22), UINT8_C(109), UINT8_C(212), UINT8_C(212) },
      {  INT8_C(  84),  INT8_C(  14), -INT8_C( 101), -INT8_C( 105), -INT8_C(  82), -INT8_C( 119), -INT8_C(  78), -INT8_C(  28),
         INT8_C(  37), -INT8_C(  14), -INT8_C(  25),  INT8_C(  83), -INT8_C(  98),  INT8_C( 106), -INT8_C(   7),  INT8_C(  23),
         INT8_C(  43), -INT8_C(  89),  INT8_C(  72), -INT8_C(  45),  INT8_C(  15), -INT8_C(  55),  INT8_C(  11),  INT8_C(  32),
         INT8_C(  35), -INT8_C(  62),  INT8_C( 123),  INT8_C(  57),  INT8_C(  48),  INT8_C(  79),  INT8_C(  13), -INT8_C( 124) },
      {  INT16_C( 19194), -INT16_C( 26547), -INT16_C( 24534), -INT16_C( 19292), -INT16_C(   186), -INT16_C(  1434),  INT16_C( 13716),  INT16_C(  2901),
        -INT16_C( 11885),  INT16_C( 10368), -INT16_C(  3200),  INT16_C(  3179), -INT16_C(  4958),  INT16_C( 27696),  INT16_C(  9667), -INT16_C( 23532) } },
    { { UINT8_C( 93), UINT8_C(168), UINT8_C( 28), UINT8_C( 12), UINT8_C( 49), UINT8_C(206), UINT8_C(240), UINT8_C( 87),
        UINT8_C(192), UINT8_C(215), UINT8_C(170), UINT8_C( 94), UINT8_C( 66), UINT8_C(163), UINT8_C(118), UINT8_C(109),
        UINT8_C( 74), UINT8_C(190), UINT8_C( 64), UINT8_C( 90), UINT8_C(135), UINT8_C( 75), UINT8_C(122), UINT8_C(170),
        UINT8_C( 14), UINT8_C(245), UINT8_C(227), UINT8_C( 62), UINT8_C( 68), UINT8_C(241), UINT8_C(194), UINT8_C(162) },
      { -INT8_C( 103), -INT8_C(  34), -INT8_C(  82), -INT8_C(  53), -INT8_C(  84), -INT8_C(  98),  INT8_C(  34),  INT8_C( 109),
         INT8_C( 118), -INT8_C(  52), -INT8_C(  53), -INT8_C(  72),  INT8_C( 111),  INT8_C(  65),  INT8_C(  37), -INT8_C(  71),
        -INT8_C(   1),  INT8_C( 101),  INT8_C(  19), -INT8_C( 121), -INT8_C(  79), -INT8_C( 115),  INT8_C(  49), -INT8_C(  65),
        -INT8_C( 126),  INT8_C(  21), -INT8_C(   3), -INT8_C(  57),  INT8_C(   6), -INT8_C(  65),  INT8_C( 105), -INT8_C(  97) },
      { -INT16_C( 15291), -INT16_C(  2932), -INT16_C( 24304),  INT16_C( 17643),  INT16_C( 11476), -INT16_C( 15778),  INT16_C( 17921), -INT16_C(  3373),
         INT16_C( 19116), -INT16_C(  9674), -INT16_C( 19290), -INT16_C(  5072),  INT16_C(  3381), -INT16_C(  4215), -INT16_C( 15257),  INT16_C(  4656) } },
    { { UINT8_C(158), UINT8_C( 23), UINT8_C(106), UINT8_C( 74), UINT8_C(181), UINT8_C(140), UINT8_C(183), UINT8_C( 43),
        UINT8_C( 88), UINT8_C(131), UINT8_C(227), UINT8_C(199), UINT8_C(196), UINT8_C(  8), UINT8_C(129), UINT8_C(196),
        UINT8_C(110), UINT8_C(148), UINT8_C( 75), UINT8_C( 31), UINT8_C( 34), UINT8_C(124), UINT8_C(222), UINT8_C(164),
        UINT8_C(145), UINT8_C(219), UINT8_C(107), UINT8_C(151), UINT8_C(154), UINT8_C(212), UINT8_C( 55), UINT8_C( 56) },
      { -INT8_C(  21), -INT8_C(  95), -INT8_C( 125), -INT8_C(  95),  INT8_C(  46),  INT8_C(  58), -INT8_C(  52), -INT8_C( 122),
        -INT8_C(  67), -INT8_C(  80),  INT8_C(  78), -INT8_C( 126), -INT8_C(  72), -INT8_C(  49),  INT8_C(  70),  INT8_C(  38),
         INT8_C(  99), -INT8_C( 111),  INT8_C(  69), -INT8_C( 123),  INT8_C(  13),  INT8_C(  35),  INT8_C(  42), -INT8_C(  97),
        -INT8_C(   2), -INT8_C( 107),  INT8_C(  54), -INT8_C( 103),  INT8_C( 106),  INT8_C( 109), -INT8_C(  47),  INT8_C(  85) },
      { -INT16_C(  5503), -INT16_C( 20280),  INT16_C( 16446), -INT16_C( 14762), -INT16_C( 16376), -INT16_C(  7368), -INT16_C( 14504),  INT16_C( 16478),
        -INT16_C(  5538),  INT16_C(  1362),  INT16_C(  4782), -INT16_C(  6584), -INT16_C( 23723), -INT16_C(  9775),       INT16_MAX,  INT16_C(  2175) } },
    { { UINT8_C( 15), UINT8_C( 84), UINT8_C(246), UINT8_C( 61), UINT8_C(143), UINT8_C(195), UINT8_C(195), UINT8_C( 76),
        UINT8_C(115), UINT8_C( 17), UINT8_C(206), UINT8_C( 43), UINT8_C(224), UINT8_C( 20), UINT8_C( 82), UINT8_C( 68),
        UINT8_C(165), UINT8_C(151), UINT8_C(201), UINT8_C(179), UINT8_C(187), UINT8_C(243), UINT8_C( 82), UINT8_C(185),
        UINT8_C(137), UINT8_C(136), UINT8_C( 82), UINT8_C(243), UINT8_C(246), UINT8_C( 36), UINT8_C( 72), UINT8_C(  5) },
      {  INT8_C( 120),  INT8_C(  63),  INT8_C(  66),  INT8_C(   7),  INT8_C(   2),  INT8_C(   5),  INT8_C(  84),  INT8_C( 117),
         INT8_C(  23),  INT8_C(  34), -INT8_C(  96), -INT8_C(   9),  INT8_C(  55), -INT8_C(  14),  INT8_C(  59), -INT8_C(  36),
        -INT8_C( 118),  INT8_C(   5), -INT8_C( 113),  INT8_C(  69), -INT8_C(   8), -INT8_C(  31), -INT8_C(   2), -INT8_C( 127),
         INT8_C( 106),  INT8_C(  81),  INT8_C( 116),  INT8_C(  96),  INT8_C( 117), -INT8_C(  67),  INT8_C( 101), -INT8_C(  19) },
      {  INT16_C(  7092),  INT16_C( 16663),  INT16_C(  1261),  INT16_C( 25272),  INT16_C(  3223), -INT16_C( 20163),  INT16_C( 12040),  INT16_C(  2390),
        -INT16_C( 18715), -INT16_C( 10362), -INT16_C(  9029), -INT16_C( 23659),  INT16_C( 25538),       INT16_MAX,  INT16_C( 26370),  INT16_C(  7177) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_maddubs_epi16(a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_maskload_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[4];
    const int32_t b[4];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(   534968926),  INT32_C(    77452639), -INT32_C(  2020739113), -INT32_C(   154957017) },
      {  INT32_C(   730249678), -INT32_C(  1555222833),  INT32_C(  1430166726),  INT32_C(   515979712) },
      {  INT32_C(           0),  INT32_C(    77452639),  INT32_C(           0),  INT32_C(           0) } },
    { { -INT32_C(  1824611532),  INT32_C(  1335417720), -INT32_C(   103406126),  INT32_C(  2129697456) },
      {  INT32_C(   581531219),  INT32_C(  1808201381),  INT32_C(  1405158547),  INT32_C(  1920107070) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { { -INT32_C(  1643794395),  INT32_C(  1575853451),  INT32_C(  1918289090), -INT32_C(  1309653410) },
      {  INT32_C(  1658034877),  INT32_C(   600676752), -INT32_C(   596144482),  INT32_C(   911141136) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(  1918289090),  INT32_C(           0) } },
    { {  INT32_C(   617894809), -INT32_C(  1283341839), -INT32_C(   483993723), -INT32_C(   611052002) },
      {  INT32_C(  1094543536), -INT32_C(  1620833791), -INT32_C(  1451500647),  INT32_C(  1574947268) },
      {  INT32_C(           0), -INT32_C(  1283341839), -INT32_C(   483993723),  INT32_C(           0) } },
    { {  INT32_C(   243381021), -INT32_C(    88014219), -INT32_C(   119609382), -INT32_C(  1361874178) },
      { -INT32_C(   588312358), -INT32_C(  1283763174), -INT32_C(   195168465), -INT32_C(   581878592) },
      {  INT32_C(   243381021), -INT32_C(    88014219), -INT32_C(   119609382), -INT32_C(  1361874178) } },
    { {  INT32_C(  1709954032), -INT32_C(  1352684075), -INT32_C(  1834533484), -INT32_C(  1975420240) },
      { -INT32_C(  1520029558), -INT32_C(  1269243260), -INT32_C(  1716996647), -INT32_C(   495453710) },
      {  INT32_C(  1709954032), -INT32_C(  1352684075), -INT32_C(  1834533484), -INT32_C(  1975420240) } },
    { { -INT32_C(  1572379956), -INT32_C(  1538152945), -INT32_C(  1808336412), -INT32_C(    31557773) },
      {  INT32_C(   748914088),  INT32_C(  1088486247), -INT32_C(  1546024783),  INT32_C(  1317359746) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(  1808336412),  INT32_C(           0) } },
    { { -INT32_C(  1024406349),  INT32_C(  1449542258), -INT32_C(  1360356037), -INT32_C(  1129576172) },
      { -INT32_C(   169259122), -INT32_C(    63583925), -INT32_C(   727773614),  INT32_C(   304227423) },
      { -INT32_C(  1024406349),  INT32_C(  1449542258), -INT32_C(  1360356037),  INT32_C(           0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m128i r = simde_mm_maskload_epi32(test_vec[i].a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

#if !defined(HEDLEY_MSVC_VERSION)
static int
test_simde_mm_maskload_epi32_no_illegal_memory_access (SIMDE_MUNIT_TEST_ARGS) {
  // ref: https://github.com/simd-everywhere/simde/issues/998
  // make sure maskload never accesses memory for masked out regions
  // will segfault in case memory is accessed
  #if defined(_GNU_SOURCE)
    int32_t *ptr = HEDLEY_STATIC_CAST(int32_t *, mmap(NULL, 4 * sizeof(int32_t), PROT_NONE , MAP_PRIVATE | MAP_ANONYMOUS, -1, 0));
  #else
    int32_t *ptr = HEDLEY_STATIC_CAST(int32_t *, mmap(NULL, 4 * sizeof(int32_t), PROT_NONE , MAP_PRIVATE, -1, 0));
  #endif
  const simde__m128i mask = simde_mm_set_epi32(0, 0, 0, 0);
  simde__m128i test = simde_mm_maskload_epi32(ptr, mask);
  int32_t r[4] = { INT32_C(0), INT32_C(0), INT32_C(0), INT32_C(0) };
  simde_test_x86_assert_equal_i32x4(test, simde_mm_loadu_epi32(r));
  return 0;
}
#endif

static int
test_simde_mm256_maskload_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { { -INT32_C(   678062514),  INT32_C(   351869071),  INT32_C(  2136335632), -INT32_C(   212796243),  INT32_C(  1333699519), -INT32_C(  1581125468),  INT32_C(   304063629),  INT32_C(   630727383) },
      {  INT32_C(  1207709368),  INT32_C(  1448932678), -INT32_C(  1999261221),  INT32_C(  1836852910),  INT32_C(  1841167049),  INT32_C(  1997438698), -INT32_C(   108450526),  INT32_C(   119415375) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(  2136335632),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   304063629),  INT32_C(           0) } },
    { { -INT32_C(  1773266096), -INT32_C(   336811504),  INT32_C(   175423836), -INT32_C(  1334316825), -INT32_C(   719506198), -INT32_C(   716428366), -INT32_C(  1479616936),  INT32_C(  1219423736) },
      {  INT32_C(   417266952),  INT32_C(    50645671),  INT32_C(  1930262667),  INT32_C(  1378059624),  INT32_C(  1814512057), -INT32_C(   985566100),  INT32_C(  1114378058),  INT32_C(    76159996) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   716428366),  INT32_C(           0),  INT32_C(           0) } },
    { { -INT32_C(  1088591848), -INT32_C(  1111285454),  INT32_C(    19976345),  INT32_C(   257119317),  INT32_C(    24869781),  INT32_C(   969325807), -INT32_C(   931449909),  INT32_C(  1724646734) },
      { -INT32_C(  1624905363), -INT32_C(  1554192374),  INT32_C(   245665209),  INT32_C(  1981675745),  INT32_C(  1652004979),  INT32_C(   547044948), -INT32_C(  1075308943), -INT32_C(  2010794981) },
      { -INT32_C(  1088591848), -INT32_C(  1111285454),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   931449909),  INT32_C(  1724646734) } },
    { { -INT32_C(  1473820002), -INT32_C(   330529997), -INT32_C(   235212784),  INT32_C(  1533483240),  INT32_C(    96329648), -INT32_C(  1910155235), -INT32_C(  1991438994),  INT32_C(  1594979009) },
      { -INT32_C(   284673860), -INT32_C(   858041156),  INT32_C(   767481412), -INT32_C(  1635244562),  INT32_C(   564348420),  INT32_C(   229623966), -INT32_C(  1751712555),  INT32_C(   737585262) },
      { -INT32_C(  1473820002), -INT32_C(   330529997),  INT32_C(           0),  INT32_C(  1533483240),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1991438994),  INT32_C(           0) } },
    { { -INT32_C(  1675952416), -INT32_C(  1754663342), -INT32_C(  1161549876),  INT32_C(  1364741196),  INT32_C(   829619346), -INT32_C(  1707203900), -INT32_C(  1926114274),  INT32_C(  1572349820) },
      {  INT32_C(  2029638182), -INT32_C(  1810931000), -INT32_C(   699477111), -INT32_C(  1306024160),  INT32_C(  1742969251), -INT32_C(   637460037),  INT32_C(  1919365878), -INT32_C(  2133909670) },
      {  INT32_C(           0), -INT32_C(  1754663342), -INT32_C(  1161549876),  INT32_C(  1364741196),  INT32_C(           0), -INT32_C(  1707203900),  INT32_C(           0),  INT32_C(  1572349820) } },
    { { -INT32_C(  1158100495), -INT32_C(  1253177301), -INT32_C(    74736165), -INT32_C(   407981500),  INT32_C(   122589515), -INT32_C(  1444851533), -INT32_C(   602191742),  INT32_C(  1482484583) },
      { -INT32_C(   552446540),  INT32_C(   949248349),  INT32_C(  1110712318),  INT32_C(   489284305),  INT32_C(   639924083),  INT32_C(  1255081415), -INT32_C(  1272517811), -INT32_C(  1978891306) },
      { -INT32_C(  1158100495),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   602191742),  INT32_C(  1482484583) } },
    { {  INT32_C(   896081880),  INT32_C(  2121137792), -INT32_C(   272588515), -INT32_C(   150148733),  INT32_C(   673001568), -INT32_C(  2106397387), -INT32_C(  1372153640), -INT32_C(   214416869) },
      { -INT32_C(   517430943), -INT32_C(  1117808993), -INT32_C(  1146347721),  INT32_C(  1773320200),  INT32_C(   496095208), -INT32_C(  1801518148), -INT32_C(  1220356709),  INT32_C(  2041215511) },
      {  INT32_C(   896081880),  INT32_C(  2121137792), -INT32_C(   272588515),  INT32_C(           0),  INT32_C(           0), -INT32_C(  2106397387), -INT32_C(  1372153640),  INT32_C(           0) } },
    { { -INT32_C(  1151675620), -INT32_C(  1602700695), -INT32_C(   497343271), -INT32_C(  1001714212), -INT32_C(  1713251107),  INT32_C(  2049867999),  INT32_C(  1831956565),  INT32_C(   132570347) },
      {  INT32_C(   415383727), -INT32_C(   726123526),  INT32_C(  1018565727), -INT32_C(    33554143), -INT32_C(  1130831139), -INT32_C(  1204369822),  INT32_C(   556099638), -INT32_C(   215479484) },
      {  INT32_C(           0), -INT32_C(  1602700695),  INT32_C(           0), -INT32_C(  1001714212), -INT32_C(  1713251107),  INT32_C(  2049867999),  INT32_C(           0),  INT32_C(   132570347) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_maskload_epi32(test_vec[i].a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

#if !defined(HEDLEY_MSVC_VERSION)
static int
test_simde_mm256_maskload_epi32_no_illegal_memory_access (SIMDE_MUNIT_TEST_ARGS) {
  // ref: https://github.com/simd-everywhere/simde/issues/998
  // make sure maskload never accesses memory for masked out regions
  // will segfault in case memory is accessed
  #if defined(_GNU_SOURCE)
    int32_t *ptr = HEDLEY_STATIC_CAST(int32_t *, mmap(NULL, 8 * sizeof(int32_t), PROT_NONE , MAP_PRIVATE | MAP_ANONYMOUS, -1, 0));
  #else
    int32_t *ptr = HEDLEY_STATIC_CAST(int32_t *, mmap(NULL, 8 * sizeof(int32_t), PROT_NONE , MAP_PRIVATE, -1, 0));
  #endif
  const simde__m256i mask = simde_mm256_set_epi32(0, 0, 0, 0, 0, 0, 0, 0);
  simde__m256i test = simde_mm256_maskload_epi32(ptr, mask);
  int32_t r[8] = { INT32_C(0), INT32_C(0), INT32_C(0), INT32_C(0),
                   INT32_C(0), INT32_C(0), INT32_C(0), INT32_C(0)};
  simde_test_x86_assert_equal_i32x8(test, simde_mm256_loadu_epi32(r));
  return 0;
}
#endif

static int
test_simde_mm_maskload_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t a[2];
    const int64_t b[2];
    const int64_t r[2];
  } test_vec[] = {
    { {  INT64_C( 7349151601351223182),  INT64_C( 2152075836815026901) },
      { -INT64_C( 3451509024372858266),  INT64_C( 7639172437864636040) },
      {  INT64_C( 7349151601351223182),  INT64_C(                   0) } },
    { {  INT64_C( 5694817644399627057),  INT64_C( 3631998945759467448) },
      { -INT64_C( 1550766173522912299),  INT64_C( 6212325517381229296) },
      {  INT64_C( 5694817644399627057),  INT64_C(                   0) } },
    { {  INT64_C( 1040332979076417448),  INT64_C( 5698954012533458456) },
      {  INT64_C( 1415945725891154011), -INT64_C( 8842793825335295795) },
      {  INT64_C(                   0),  INT64_C( 5698954012533458456) } },
    { {  INT64_C( 6538970759181058396),  INT64_C( 3740051304798361630) },
      {  INT64_C( 2385384236005712100),  INT64_C( 7654249160401746390) },
      {  INT64_C(                   0),  INT64_C(                   0) } },
    { {  INT64_C( 1601271529454952604), -INT64_C( 5765005774754515213) },
      {  INT64_C( 6551277872407309686), -INT64_C(  846316808078164535) },
      {  INT64_C(                   0), -INT64_C( 5765005774754515213) } },
    { {  INT64_C( 3041629830225484840), -INT64_C( 1154342652176112512) },
      {  INT64_C( 5313864382184796385),  INT64_C( 4390553429439728801) },
      {  INT64_C(                   0),  INT64_C(                   0) } },
    { {  INT64_C( 5413206681815138493), -INT64_C( 1594397375012137610) },
      { -INT64_C( 2591526659806029760),  INT64_C( 5199903345750714841) },
      {  INT64_C( 5413206681815138493),  INT64_C(                   0) } },
    { { -INT64_C( 8242893437788125820),  INT64_C(  172082101606953329) },
      { -INT64_C( 7562726032021409293),  INT64_C( 3082904871272505846) },
      { -INT64_C( 8242893437788125820),  INT64_C(                   0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m128i r;
    #if defined(SIMDE_X86_AVX2_NATIVE) && defined(SIMDE_NATIVE_ALIASES_TESTING)
      r = simde_mm_maskload_epi64(HEDLEY_REINTERPRET_CAST(const long long *, test_vec[i].a), b);
    #else
      r = simde_mm_maskload_epi64(test_vec[i].a, b);
    #endif
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

#if !defined(HEDLEY_MSVC_VERSION)
static int
test_simde_mm_maskload_epi64_no_illegal_memory_access (SIMDE_MUNIT_TEST_ARGS) {
  // ref: https://github.com/simd-everywhere/simde/issues/998
  // make sure maskload never accesses memory for masked out regions
  // will segfault in case memory is accessed
  #if defined(_GNU_SOURCE)
    int64_t *ptr = HEDLEY_STATIC_CAST(int64_t *, mmap(NULL, 2 * sizeof(int64_t), PROT_NONE , MAP_PRIVATE | MAP_ANONYMOUS, -1, 0));
  #else
    int64_t *ptr = HEDLEY_STATIC_CAST(int64_t *, mmap(NULL, 2 * sizeof(int64_t), PROT_NONE , MAP_PRIVATE, -1, 0));
  #endif
  const simde__m128i mask = simde_mm_set_epi64x(INT64_C(0), INT64_C(0));
  simde__m128i test;
  #if defined(SIMDE_X86_AVX2_NATIVE) && defined(SIMDE_NATIVE_ALIASES_TESTING)
    test = simde_mm_maskload_epi64(HEDLEY_REINTERPRET_CAST(const long long *, ptr), mask);
  #else
    test = simde_mm_maskload_epi64(ptr, mask);
  #endif
  int64_t r[2] = { INT64_C(0), INT64_C(0) };
  simde_test_x86_assert_equal_i64x2(test, simde_x_mm_loadu_epi64(r));
  return 0;
}
#endif

static int
test_simde_mm256_maskload_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t a[4];
    const int64_t b[4];
    const int64_t r[4];
  } test_vec[] = {
    { { -INT64_C( 1070553612364617574), -INT64_C(  114323686149473380), -INT64_C( 3053698850669963787), -INT64_C( 5793007821577797203) },
      { -INT64_C( 2874087956492281342), -INT64_C( 8133550842580450581), -INT64_C( 5976205236281087193), -INT64_C(  520589115781817598) },
      { -INT64_C( 1070553612364617574), -INT64_C(  114323686149473380), -INT64_C( 3053698850669963787), -INT64_C( 5793007821577797203) } },
    { { -INT64_C( 1284580311155027432), -INT64_C( 8201847224291781372),  INT64_C( 1425908320736694043),  INT64_C( 4873440337809026227) },
      {  INT64_C(  771455870039475650),  INT64_C( 1030716232118617176), -INT64_C( 6788626756700334683), -INT64_C( 6553603994163678305) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 1425908320736694043),  INT64_C( 4873440337809026227) } },
    { { -INT64_C( 4701442009973677136), -INT64_C( 1880874889532052765), -INT64_C( 6814117747969290272),  INT64_C( 6260135033578371671) },
      {  INT64_C( 5566394621280209690),  INT64_C( 7448400211646594360), -INT64_C( 5351783383607936745), -INT64_C( 5306699842386385569) },
      {  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 6814117747969290272),  INT64_C( 6260135033578371671) } },
    { {  INT64_C( 7985576165328250651),  INT64_C( 1523138059357910324),  INT64_C( 6652636875035990078),  INT64_C( 3270100493289996814) },
      {  INT64_C( 4884017522370844707),  INT64_C( 3642606102698241657), -INT64_C( 5416668556751356806),  INT64_C( 1486680395769328994) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 6652636875035990078),  INT64_C(                   0) } },
    { {  INT64_C( 5884855437847515992),  INT64_C( 5046377720733805283),  INT64_C( 6515463978070497041), -INT64_C( 2175959308119914040) },
      {  INT64_C( 5975418466285927107),  INT64_C( 1995148555708839069), -INT64_C( 3865991726324617720),  INT64_C( 6589137445521139719) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 6515463978070497041),  INT64_C(                   0) } },
    { {  INT64_C( 5353903293904370059), -INT64_C( 5855694748648180329), -INT64_C( 3198110109063606318), -INT64_C(  134311449431170769) },
      { -INT64_C( 7888822226924332658),  INT64_C( 8074655740636379578), -INT64_C( 1071888862176560257), -INT64_C( 1912662773025177085) },
      {  INT64_C( 5353903293904370059),  INT64_C(                   0), -INT64_C( 3198110109063606318), -INT64_C(  134311449431170769) } },
    { {  INT64_C( 7746936492670348532),  INT64_C( 2660740784655676895),  INT64_C( 1605972051073446037), -INT64_C( 3437089993967009519) },
      { -INT64_C(  970171594833891988), -INT64_C( 4341088310040570377), -INT64_C( 3263873051487686261), -INT64_C( 6047621540034756507) },
      {  INT64_C( 7746936492670348532),  INT64_C( 2660740784655676895),  INT64_C( 1605972051073446037), -INT64_C( 3437089993967009519) } },
    { {  INT64_C(  935152317656388341),  INT64_C( 5736295944050796969),  INT64_C( 9164439426345842332), -INT64_C( 1815279467665200671) },
      { -INT64_C( 7653891838718273370),  INT64_C( 8531516276872147026), -INT64_C( 3001928471202802754), -INT64_C( 7835481678918706838) },
      {  INT64_C(  935152317656388341),  INT64_C(                   0),  INT64_C( 9164439426345842332), -INT64_C( 1815279467665200671) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i b = simde_x_mm256_loadu_epi64(test_vec[i].b);
    simde__m256i r;
    #if defined(SIMDE_X86_AVX2_NATIVE) && defined(SIMDE_NATIVE_ALIASES_TESTING)
      r = simde_mm256_maskload_epi64(HEDLEY_REINTERPRET_CAST(const long long *, test_vec[i].a), b);
    #else
      r = simde_mm256_maskload_epi64(test_vec[i].a, b);
    #endif
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

#if !defined(HEDLEY_MSVC_VERSION)
static int
test_simde_mm256_maskload_epi64_no_illegal_memory_access (SIMDE_MUNIT_TEST_ARGS) {
  // ref: https://github.com/simd-everywhere/simde/issues/998
  // make sure maskload never accesses memory for masked out regions
  // will segfault in case memory is accessed
  #if defined(_GNU_SOURCE)
    int64_t *ptr = HEDLEY_STATIC_CAST(int64_t *, mmap(NULL, 4 * sizeof(int64_t), PROT_NONE , MAP_PRIVATE | MAP_ANONYMOUS, -1, 0));
  #else
    int64_t *ptr = HEDLEY_STATIC_CAST(int64_t *, mmap(NULL, 4 * sizeof(int64_t), PROT_NONE , MAP_PRIVATE, -1, 0));
  #endif
  const simde__m256i mask = simde_mm256_set_epi64x(INT64_C(0), INT64_C(0), INT64_C(0), INT64_C(0));
  simde__m256i test;
  #if defined(SIMDE_X86_AVX2_NATIVE) && defined(SIMDE_NATIVE_ALIASES_TESTING)
    test = simde_mm256_maskload_epi64(HEDLEY_REINTERPRET_CAST(const long long *, ptr), mask);
  #else
    test = simde_mm256_maskload_epi64(ptr, mask);
  #endif
  int64_t r[4] = { INT64_C(0), INT64_C(0), INT64_C(0), INT64_C(0) };
  simde_test_x86_assert_equal_i64x4(test, simde_x_mm256_loadu_epi64(r));
  return 0;
}
#endif

static int
test_simde_mm_maskstore_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t mask[4];
    const int32_t a[4];
    const int32_t ri[4];
    const int32_t ro[4];
  } test_vec[] = {
    { { -INT32_C(  1072310678), -INT32_C(  1753702976), -INT32_C(  1000721038),  INT32_C(  1139936175) },
      { -INT32_C(   921988596), -INT32_C(  1384169384),  INT32_C(  1969759351),  INT32_C(   323044521) },
      { -INT32_C(   921988596), -INT32_C(  1384169384),  INT32_C(  1969759351),  INT32_C(       22034) },
      { -INT32_C(   921988596), -INT32_C(  1384169384),  INT32_C(  1969759351),  INT32_C(       22034) } },
    { { -INT32_C(   623618278),  INT32_C(  1718766836),  INT32_C(   824888449), -INT32_C(   462152488) },
      {  INT32_C(   112033710),  INT32_C(   917712319), -INT32_C(    72606895),  INT32_C(  2047798624) },
      {  INT32_C(   112033710), -INT32_C(  1384169384),  INT32_C(  1969759351),  INT32_C(  2047798624) },
      {  INT32_C(   112033710), -INT32_C(  1384169384),  INT32_C(  1969759351),  INT32_C(  2047798624) } },
    { {  INT32_C(   945152580), -INT32_C(  1331771601),  INT32_C(  1809959315), -INT32_C(  1806674203) },
      { -INT32_C(  1785004587),  INT32_C(  2093698602), -INT32_C(   914917527), -INT32_C(  1455127195) },
      {  INT32_C(   112033710),  INT32_C(  2093698602),  INT32_C(  1969759351), -INT32_C(  1455127195) },
      {  INT32_C(   112033710),  INT32_C(  2093698602),  INT32_C(  1969759351), -INT32_C(  1455127195) } },
    { { -INT32_C(  1746757272), -INT32_C(   213417888),  INT32_C(   794765641),  INT32_C(  1422110591) },
      { -INT32_C(   672571988),  INT32_C(   357807531), -INT32_C(  1847670228), -INT32_C(  1220861361) },
      { -INT32_C(   672571988),  INT32_C(   357807531),  INT32_C(  1969759351), -INT32_C(  1455127195) },
      { -INT32_C(   672571988),  INT32_C(   357807531),  INT32_C(  1969759351), -INT32_C(  1455127195) } },
    { {  INT32_C(   458104251), -INT32_C(   418408803),  INT32_C(  1041657535), -INT32_C(   913123043) },
      { -INT32_C(   492798922),  INT32_C(  1576530737),  INT32_C(   233821629), -INT32_C(  1278989576) },
      { -INT32_C(   672571988),  INT32_C(  1576530737),  INT32_C(  1969759351), -INT32_C(  1278989576) },
      { -INT32_C(   672571988),  INT32_C(  1576530737),  INT32_C(  1969759351), -INT32_C(  1278989576) } },
    { { -INT32_C(   456191161),  INT32_C(  1758191273),  INT32_C(  1772544332), -INT32_C(   248366918) },
      { -INT32_C(   439102540), -INT32_C(  2075931962), -INT32_C(  1752092001), -INT32_C(  1555344036) },
      { -INT32_C(   439102540),  INT32_C(  1576530737),  INT32_C(  1969759351), -INT32_C(  1555344036) },
      { -INT32_C(   439102540),  INT32_C(  1576530737),  INT32_C(  1969759351), -INT32_C(  1555344036) } },
    { {  INT32_C(   294066792),  INT32_C(  1148802040), -INT32_C(   273866956),  INT32_C(   216063832) },
      {  INT32_C(  2045948850),  INT32_C(   486356349), -INT32_C(  1011577241),  INT32_C(  1281818595) },
      { -INT32_C(   439102540),  INT32_C(  1576530737), -INT32_C(  1011577241), -INT32_C(  1555344036) },
      { -INT32_C(   439102540),  INT32_C(  1576530737), -INT32_C(  1011577241), -INT32_C(  1555344036) } },
    { {  INT32_C(   291368217),  INT32_C(  1968559936),  INT32_C(  1315177206), -INT32_C(  1805957919) },
      {  INT32_C(  1947028983), -INT32_C(   376436094),  INT32_C(  2074887320),  INT32_C(  1556550211) },
      { -INT32_C(   439102540),  INT32_C(  1576530737), -INT32_C(  1011577241),  INT32_C(  1556550211) },
      { -INT32_C(   439102540),  INT32_C(  1576530737), -INT32_C(  1011577241),  INT32_C(  1556550211) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i mask = simde_x_mm_loadu_epi32(test_vec[i].mask);
    simde__m128i a = simde_x_mm_loadu_epi32(test_vec[i].a);
    int32_t r[4];
    simde_memcpy(r, test_vec[i].ri, sizeof(r));
    simde_mm_maskstore_epi32(r, mask, a);
    simde_assert_equal_vi32(sizeof(r) / sizeof(r[0]), r, test_vec[i].ro);
  }

  return 0;
}

static int
test_simde_mm256_maskstore_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t mask[8];
    const int32_t a[8];
    const int32_t ri[8];
    const int32_t ro[8];
  } test_vec[] = {
    { { -INT32_C(  2076816164),  INT32_C(   922486962),  INT32_C(   556407151),  INT32_C(  1733355511),  INT32_C(   401058348), -INT32_C(    81912571), -INT32_C(  1174954754),  INT32_C(   302965558) },
      { -INT32_C(   493403088), -INT32_C(  1139240115), -INT32_C(  1545780564),  INT32_C(  1426795817), -INT32_C(   580062504),  INT32_C(   232294927),  INT32_C(  1573376039), -INT32_C(   479210061) },
      { -INT32_C(   493403088),  INT32_C(           0),  INT32_C(  1022469250),  INT32_C(       32547),  INT32_C(  1022406520),  INT32_C(   232294927),  INT32_C(  1573376039),  INT32_C(       32547) },
      { -INT32_C(   493403088),  INT32_C(           0),  INT32_C(  1022469250),  INT32_C(       32547),  INT32_C(  1022406520),  INT32_C(   232294927),  INT32_C(  1573376039),  INT32_C(       32547) } },
    { {  INT32_C(  1724188185),  INT32_C(  1176690073),  INT32_C(  1240071968),  INT32_C(    94368813), -INT32_C(   152957977), -INT32_C(  1123763306),  INT32_C(  1041943435), -INT32_C(  1188918880) },
      {  INT32_C(   690022288), -INT32_C(   445693243),  INT32_C(  1865308482),  INT32_C(   880069965),  INT32_C(  1865111257), -INT32_C(  1674825967), -INT32_C(  1696905478),  INT32_C(  1599339983) },
      { -INT32_C(   493403088),  INT32_C(           0),  INT32_C(  1022469250),  INT32_C(       32547),  INT32_C(  1865111257), -INT32_C(  1674825967),  INT32_C(  1573376039),  INT32_C(  1599339983) },
      { -INT32_C(   493403088),  INT32_C(           0),  INT32_C(  1022469250),  INT32_C(       32547),  INT32_C(  1865111257), -INT32_C(  1674825967),  INT32_C(  1573376039),  INT32_C(  1599339983) } },
    { { -INT32_C(  1450609692), -INT32_C(   124847946), -INT32_C(  1620591279),  INT32_C(  1674828682),  INT32_C(  1137901105),  INT32_C(   668991021),  INT32_C(   348240452), -INT32_C(  1670179401) },
      {  INT32_C(  1044774024),  INT32_C(  1177998581),  INT32_C(   468032913), -INT32_C(  1434470024), -INT32_C(   437431881), -INT32_C(  1811100592),  INT32_C(  1051250311),  INT32_C(  1809456355) },
      {  INT32_C(  1044774024),  INT32_C(  1177998581),  INT32_C(   468032913),  INT32_C(       32547),  INT32_C(  1865111257), -INT32_C(  1674825967),  INT32_C(  1573376039),  INT32_C(  1809456355) },
      {  INT32_C(  1044774024),  INT32_C(  1177998581),  INT32_C(   468032913),  INT32_C(       32547),  INT32_C(  1865111257), -INT32_C(  1674825967),  INT32_C(  1573376039),  INT32_C(  1809456355) } },
    { {  INT32_C(   229253144), -INT32_C(  2058034956), -INT32_C(   157271682), -INT32_C(  1432346638), -INT32_C(  1047556751), -INT32_C(   514483366),  INT32_C(  1277165161),  INT32_C(   850983450) },
      {  INT32_C(   239100442), -INT32_C(  1064070078), -INT32_C(  1061735475), -INT32_C(   999663789),  INT32_C(  1065744869), -INT32_C(    31401324), -INT32_C(   230015016),  INT32_C(  1394934329) },
      {  INT32_C(  1044774024), -INT32_C(  1064070078), -INT32_C(  1061735475), -INT32_C(   999663789),  INT32_C(  1065744869), -INT32_C(    31401324),  INT32_C(  1573376039),  INT32_C(  1809456355) },
      {  INT32_C(  1044774024), -INT32_C(  1064070078), -INT32_C(  1061735475), -INT32_C(   999663789),  INT32_C(  1065744869), -INT32_C(    31401324),  INT32_C(  1573376039),  INT32_C(  1809456355) } },
    { { -INT32_C(  1486789276), -INT32_C(   966265607),  INT32_C(  2072387112),  INT32_C(  1530916982),  INT32_C(  2124072169),  INT32_C(  2004662942),  INT32_C(   862570233),  INT32_C(   763793097) },
      { -INT32_C(   321591053),  INT32_C(    78855388), -INT32_C(   796968614),  INT32_C(   321633834),  INT32_C(   563201410),  INT32_C(  2040008063), -INT32_C(  1649671724), -INT32_C(  2083900784) },
      { -INT32_C(   321591053),  INT32_C(    78855388), -INT32_C(  1061735475), -INT32_C(   999663789),  INT32_C(  1065744869), -INT32_C(    31401324),  INT32_C(  1573376039),  INT32_C(  1809456355) },
      { -INT32_C(   321591053),  INT32_C(    78855388), -INT32_C(  1061735475), -INT32_C(   999663789),  INT32_C(  1065744869), -INT32_C(    31401324),  INT32_C(  1573376039),  INT32_C(  1809456355) } },
    { { -INT32_C(   143614182),  INT32_C(   905651163), -INT32_C(  2046395556), -INT32_C(  1130745543),  INT32_C(  1994206199),  INT32_C(   233796921),  INT32_C(   111844214), -INT32_C(   393579314) },
      { -INT32_C(   287311341),  INT32_C(  2032458525), -INT32_C(  1895814570),  INT32_C(  1380686427), -INT32_C(    20371259),  INT32_C(   336312477),  INT32_C(   572175700),  INT32_C(  1024107562) },
      { -INT32_C(   287311341),  INT32_C(    78855388), -INT32_C(  1895814570),  INT32_C(  1380686427),  INT32_C(  1065744869), -INT32_C(    31401324),  INT32_C(  1573376039),  INT32_C(  1024107562) },
      { -INT32_C(   287311341),  INT32_C(    78855388), -INT32_C(  1895814570),  INT32_C(  1380686427),  INT32_C(  1065744869), -INT32_C(    31401324),  INT32_C(  1573376039),  INT32_C(  1024107562) } },
    { { -INT32_C(  1154684258),  INT32_C(   456478917), -INT32_C(   710265478), -INT32_C(  1809254705), -INT32_C(  1131155170), -INT32_C(    36659543),  INT32_C(  2116020820),  INT32_C(   767240847) },
      { -INT32_C(   638982380), -INT32_C(  1309401545),  INT32_C(   579313235), -INT32_C(  1279873132),  INT32_C(  1232030112),  INT32_C(  1011302376), -INT32_C(  1195743703), -INT32_C(  1528400496) },
      { -INT32_C(   638982380),  INT32_C(    78855388),  INT32_C(   579313235), -INT32_C(  1279873132),  INT32_C(  1232030112),  INT32_C(  1011302376),  INT32_C(  1573376039),  INT32_C(  1024107562) },
      { -INT32_C(   638982380),  INT32_C(    78855388),  INT32_C(   579313235), -INT32_C(  1279873132),  INT32_C(  1232030112),  INT32_C(  1011302376),  INT32_C(  1573376039),  INT32_C(  1024107562) } },
    { { -INT32_C(  1803694243),  INT32_C(  1078358509), -INT32_C(  1537028848),  INT32_C(   475469948),  INT32_C(  1248183906),  INT32_C(   797355013), -INT32_C(  1545125869),  INT32_C(   306761141) },
      { -INT32_C(  1985493604),  INT32_C(  1204415799),  INT32_C(   921381818), -INT32_C(  1504558268),  INT32_C(   267433737),  INT32_C(  2000582244),  INT32_C(  1796875702), -INT32_C(  1887542541) },
      { -INT32_C(  1985493604),  INT32_C(    78855388),  INT32_C(   921381818), -INT32_C(  1279873132),  INT32_C(  1232030112),  INT32_C(  1011302376),  INT32_C(  1796875702),  INT32_C(  1024107562) },
      { -INT32_C(  1985493604),  INT32_C(    78855388),  INT32_C(   921381818), -INT32_C(  1279873132),  INT32_C(  1232030112),  INT32_C(  1011302376),  INT32_C(  1796875702),  INT32_C(  1024107562) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i mask = simde_x_mm256_loadu_epi32(test_vec[i].mask);
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    int32_t r[8];
    simde_memcpy(r, test_vec[i].ri, sizeof(r));
    simde_mm256_maskstore_epi32(r, mask, a);
    simde_assert_equal_vi32(sizeof(r) / sizeof(r[0]), r, test_vec[i].ro);
  }

  return 0;
}

static int
test_simde_mm_maskstore_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t mask[2];
    const int64_t a[2];
    const int64_t ri[2];
    const int64_t ro[2];
  } test_vec[] = {
    { {  INT64_C( 3149967905596390459), -INT64_C( 4778628840774069374) },
      {  INT64_C( 5664027239959376000),  INT64_C( 8937778571530162224) },
      {  INT64_C(     140732988756112),  INT64_C( 8937778571530162224) },
      {  INT64_C(     140732988756112),  INT64_C( 8937778571530162224) } },
    { {  INT64_C( 8062610935831789210), -INT64_C( 8946025712957579848) },
      { -INT64_C( 7091179927189929083), -INT64_C( 5251394950330663071) },
      {  INT64_C(     140732988756112), -INT64_C( 5251394950330663071) },
      {  INT64_C(     140732988756112), -INT64_C( 5251394950330663071) } },
    { {  INT64_C( 7684922938565365066),  INT64_C( 2879875638071020913) },
      { -INT64_C( 2293448770573013726),  INT64_C(  332036821611605703) },
      {  INT64_C(     140732988756112), -INT64_C( 5251394950330663071) },
      {  INT64_C(     140732988756112), -INT64_C( 5251394950330663071) } },
    { { -INT64_C( 3437124613673645265),  INT64_C( 6050651184505868846) },
      {  INT64_C( 3965153871791286441),  INT64_C( 6501991883275471484) },
      {  INT64_C( 3965153871791286441), -INT64_C( 5251394950330663071) },
      {  INT64_C( 3965153871791286441), -INT64_C( 5251394950330663071) } },
    { { -INT64_C( 8650702022112413758),  INT64_C( 2008129837236434778) },
      {  INT64_C( 5359365805870137528), -INT64_C(  767092764058747948) },
      {  INT64_C( 5359365805870137528), -INT64_C( 5251394950330663071) },
      {  INT64_C( 5359365805870137528), -INT64_C( 5251394950330663071) } },
    { {  INT64_C( 8652269335912879668),  INT64_C( 8272797757509692224) },
      {  INT64_C( 3511395233916719521),  INT64_C( 3215161084095773558) },
      {  INT64_C( 5359365805870137528), -INT64_C( 5251394950330663071) },
      {  INT64_C( 5359365805870137528), -INT64_C( 5251394950330663071) } },
    { { -INT64_C(   20041080821402335),  INT64_C( 4567917337930546231) },
      {  INT64_C( 8151067802542735250), -INT64_C(  303508250809446492) },
      {  INT64_C( 8151067802542735250), -INT64_C( 5251394950330663071) },
      {  INT64_C( 8151067802542735250), -INT64_C( 5251394950330663071) } },
    { { -INT64_C( 8778935907035413907), -INT64_C( 1185330308615447376) },
      {  INT64_C( 5136350581142404060), -INT64_C( 8781828949867732662) },
      {  INT64_C( 5136350581142404060), -INT64_C( 8781828949867732662) },
      {  INT64_C( 5136350581142404060), -INT64_C( 8781828949867732662) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i mask = simde_x_mm_loadu_epi64(test_vec[i].mask);
    simde__m128i a = simde_x_mm_loadu_epi64(test_vec[i].a);
    int64_t r[2];
    simde_memcpy(r, test_vec[i].ri, sizeof(r));
    #if defined(SIMDE_X86_AVX2_NATIVE) && defined(SIMDE_NATIVE_ALIASES_TESTING)
      simde_mm_maskstore_epi64((long long *)r, mask, a);
    #else
      simde_mm_maskstore_epi64(r, mask, a);
    #endif
    simde_assert_equal_vi64(sizeof(r) / sizeof(r[0]), r, test_vec[i].ro);
  }

  return 0;
}

static int
test_simde_mm256_maskstore_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t mask[4];
    const int64_t a[4];
    const int64_t ri[4];
    const int64_t ro[4];
  } test_vec[] = {
    { {  INT64_C( 5242526618242621661),  INT64_C( 6166759832868221493), -INT64_C( 3838943471189419310), -INT64_C(  300229214128524470) },
      {  INT64_C( 2195119886963881727),  INT64_C( 4994165925608667027), -INT64_C( 2162954226736872975),  INT64_C(  464305096456775511) },
      {  INT64_C(                2050),  INT64_C(             1313770), -INT64_C( 2162954226736872975),  INT64_C(  464305096456775511) },
      {  INT64_C(                2050),  INT64_C(             1313770), -INT64_C( 2162954226736872975),  INT64_C(  464305096456775511) } },
    { { -INT64_C( 8770573412453887679),  INT64_C( 5987762628439542624), -INT64_C(  105974608678354161),  INT64_C(  663406593554201895) },
      {  INT64_C( 8191901560846116662),  INT64_C(  917429192545808134), -INT64_C( 1800249354620813217), -INT64_C( 7905707010328004489) },
      {  INT64_C( 8191901560846116662),  INT64_C(             1313770), -INT64_C( 1800249354620813217),  INT64_C(  464305096456775511) },
      {  INT64_C( 8191901560846116662),  INT64_C(             1313770), -INT64_C( 1800249354620813217),  INT64_C(  464305096456775511) } },
    { { -INT64_C(  129641248100483919),  INT64_C( 3792913607644641434),  INT64_C(  579345394647586402),  INT64_C( 1920382171601182647) },
      {  INT64_C( 7351017860721565965), -INT64_C( 1112211479437939833), -INT64_C( 4570841319541126409),  INT64_C( 8628750009532471612) },
      {  INT64_C( 7351017860721565965),  INT64_C(             1313770), -INT64_C( 1800249354620813217),  INT64_C(  464305096456775511) },
      {  INT64_C( 7351017860721565965),  INT64_C(             1313770), -INT64_C( 1800249354620813217),  INT64_C(  464305096456775511) } },
    { {  INT64_C( 3606962921541573595), -INT64_C( 6599710048517223188), -INT64_C( 3120222027706958616), -INT64_C( 2560234815484229993) },
      {  INT64_C( 7111344849244452307),  INT64_C( 4967546651701690014),  INT64_C( 8227770697573080513), -INT64_C( 5454725699460683665) },
      {  INT64_C( 7351017860721565965),  INT64_C( 4967546651701690014),  INT64_C( 8227770697573080513), -INT64_C( 5454725699460683665) },
      {  INT64_C( 7351017860721565965),  INT64_C( 4967546651701690014),  INT64_C( 8227770697573080513), -INT64_C( 5454725699460683665) } },
    { {  INT64_C(  124660851703730401), -INT64_C( 7079591700845521376),  INT64_C( 5739316420021843801), -INT64_C( 1845900778233980467) },
      { -INT64_C( 7620099957265165093),  INT64_C( 2379821211310001037), -INT64_C( 1082557009605755447), -INT64_C( 6989304304425322474) },
      {  INT64_C( 7351017860721565965),  INT64_C( 2379821211310001037),  INT64_C( 8227770697573080513), -INT64_C( 6989304304425322474) },
      {  INT64_C( 7351017860721565965),  INT64_C( 2379821211310001037),  INT64_C( 8227770697573080513), -INT64_C( 6989304304425322474) } },
    { { -INT64_C( 4350446221603815788),  INT64_C( 5661361750980473512), -INT64_C(  413503352329476305),  INT64_C( 8362083761278804137) },
      { -INT64_C( 5886734649484449196), -INT64_C( 2548280180651164976),  INT64_C( 3707993608283921047), -INT64_C(  746280734714459568) },
      { -INT64_C( 5886734649484449196),  INT64_C( 2379821211310001037),  INT64_C( 3707993608283921047), -INT64_C( 6989304304425322474) },
      { -INT64_C( 5886734649484449196),  INT64_C( 2379821211310001037),  INT64_C( 3707993608283921047), -INT64_C( 6989304304425322474) } },
    { {  INT64_C( 1174539280240812695), -INT64_C( 5561656252458243512), -INT64_C( 4512708599429136223),  INT64_C( 4954969688283086893) },
      {  INT64_C( 6343651101063070392),  INT64_C( 1685505655689801368),  INT64_C( 7084341664285954590), -INT64_C( 8028524129605893725) },
      { -INT64_C( 5886734649484449196),  INT64_C( 1685505655689801368),  INT64_C( 7084341664285954590), -INT64_C( 6989304304425322474) },
      { -INT64_C( 5886734649484449196),  INT64_C( 1685505655689801368),  INT64_C( 7084341664285954590), -INT64_C( 6989304304425322474) } },
    { {  INT64_C( 4296032324518769974),  INT64_C( 2922363408830294785),  INT64_C( 1764011878569726692), -INT64_C( 3048659187356095416) },
      { -INT64_C( 3724554084945282703),  INT64_C( 8347090473923591588), -INT64_C( 2580272778841387612),  INT64_C(  443794320655161779) },
      { -INT64_C( 5886734649484449196),  INT64_C( 1685505655689801368),  INT64_C( 7084341664285954590),  INT64_C(  443794320655161779) },
      { -INT64_C( 5886734649484449196),  INT64_C( 1685505655689801368),  INT64_C( 7084341664285954590),  INT64_C(  443794320655161779) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i mask = simde_x_mm256_loadu_epi64(test_vec[i].mask);
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    int64_t r[4];
    simde_memcpy(r, test_vec[i].ri, sizeof(r));
    #if defined(SIMDE_X86_AVX2_NATIVE) && defined(SIMDE_NATIVE_ALIASES_TESTING)
      simde_mm256_maskstore_epi64((long long *)r, mask, a);
    #else
      simde_mm256_maskstore_epi64(r, mask, a);
    #endif
    simde_assert_equal_vi64(sizeof(r) / sizeof(r[0]), r, test_vec[i].ro);
  }

  return 0;
}

static int
test_simde_mm256_max_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C(  82), INT8_C(  80), INT8_C( 100), INT8_C(-114),
                           INT8_C(-121), INT8_C(-115), INT8_C( -33), INT8_C( -36),
                           INT8_C(-105), INT8_C( -52), INT8_C(  94), INT8_C(  97),
                           INT8_C( -84), INT8_C( 116), INT8_C( 107), INT8_C( -76),
                           INT8_C(   5), INT8_C(  53), INT8_C( 122), INT8_C(-101),
                           INT8_C(  32), INT8_C(-122), INT8_C( -66), INT8_C( -61),
                           INT8_C( -25), INT8_C(  -2), INT8_C(  14), INT8_C( -94),
                           INT8_C(  18), INT8_C( -35), INT8_C(-102), INT8_C(  57)),
      simde_mm256_set_epi8(INT8_C(  46), INT8_C( -89), INT8_C(  83), INT8_C( 115),
                           INT8_C( 114), INT8_C(  48), INT8_C(   9), INT8_C(   1),
                           INT8_C(  54), INT8_C( -93), INT8_C( 118), INT8_C( -84),
                           INT8_C(  50), INT8_C( -43), INT8_C(-125), INT8_C( -87),
                           INT8_C(  48), INT8_C(  67), INT8_C(  65), INT8_C( 103),
                           INT8_C( 105), INT8_C( -97), INT8_C( -54), INT8_C(  17),
                           INT8_C( -90), INT8_C( -86), INT8_C( 116), INT8_C( 127),
                           INT8_C(  77), INT8_C(  -5), INT8_C(  31), INT8_C(  24)),
      simde_mm256_set_epi8(INT8_C(  82), INT8_C(  80), INT8_C( 100), INT8_C( 115),
                           INT8_C( 114), INT8_C(  48), INT8_C(   9), INT8_C(   1),
                           INT8_C(  54), INT8_C( -52), INT8_C( 118), INT8_C(  97),
                           INT8_C(  50), INT8_C( 116), INT8_C( 107), INT8_C( -76),
                           INT8_C(  48), INT8_C(  67), INT8_C( 122), INT8_C( 103),
                           INT8_C( 105), INT8_C( -97), INT8_C( -54), INT8_C(  17),
                           INT8_C( -25), INT8_C(  -2), INT8_C( 116), INT8_C( 127),
                           INT8_C(  77), INT8_C(  -5), INT8_C(  31), INT8_C(  57)) },
    { simde_mm256_set_epi8(INT8_C( 104), INT8_C( 106), INT8_C(-125), INT8_C(  -4),
                           INT8_C( -59), INT8_C(  95), INT8_C( -56), INT8_C(  31),
                           INT8_C( 108), INT8_C(   5), INT8_C( 101), INT8_C(  27),
                           INT8_C( -64), INT8_C(  57), INT8_C(-101), INT8_C( -55),
                           INT8_C(  77), INT8_C( 118), INT8_C(  99), INT8_C(  21),
                           INT8_C(  95), INT8_C(  17), INT8_C(  54), INT8_C( -21),
                           INT8_C(  92), INT8_C( -54), INT8_C(  78), INT8_C(  40),
                           INT8_C( 102), INT8_C(-101), INT8_C(   0), INT8_C( 127)),
      simde_mm256_set_epi8(INT8_C( -59), INT8_C( 102), INT8_C(  95), INT8_C( -82),
                           INT8_C( -77), INT8_C( -39), INT8_C( -88), INT8_C( -22),
                           INT8_C(  54), INT8_C(  84), INT8_C( -33), INT8_C(  49),
                           INT8_C(  63), INT8_C(  99), INT8_C(  64), INT8_C(-122),
                           INT8_C( 102), INT8_C(  28), INT8_C(  90), INT8_C(  72),
                           INT8_C(  24), INT8_C(  50), INT8_C(  78), INT8_C( -15),
                           INT8_C( -20), INT8_C( -56), INT8_C(  73), INT8_C(  48),
                           INT8_C( -27), INT8_C(   7), INT8_C( -15), INT8_C(-109)),
      simde_mm256_set_epi8(INT8_C( 104), INT8_C( 106), INT8_C(  95), INT8_C(  -4),
                           INT8_C( -59), INT8_C(  95), INT8_C( -56), INT8_C(  31),
                           INT8_C( 108), INT8_C(  84), INT8_C( 101), INT8_C(  49),
                           INT8_C(  63), INT8_C(  99), INT8_C(  64), INT8_C( -55),
                           INT8_C( 102), INT8_C( 118), INT8_C(  99), INT8_C(  72),
                           INT8_C(  95), INT8_C(  50), INT8_C(  78), INT8_C( -15),
                           INT8_C(  92), INT8_C( -54), INT8_C(  78), INT8_C(  48),
                           INT8_C( 102), INT8_C(   7), INT8_C(   0), INT8_C( 127)) },
    { simde_mm256_set_epi8(INT8_C(  65), INT8_C(  23), INT8_C(  64), INT8_C( -32),
                           INT8_C( -17), INT8_C(  98), INT8_C( -25), INT8_C( 100),
                           INT8_C(-100), INT8_C( -55), INT8_C( -13), INT8_C( 105),
                           INT8_C(   8), INT8_C(  62), INT8_C(-102), INT8_C(  30),
                           INT8_C(  -9), INT8_C(  71), INT8_C( -37), INT8_C( -11),
                           INT8_C( -97), INT8_C(  54), INT8_C( -23), INT8_C( 103),
                           INT8_C(  11), INT8_C( -76), INT8_C(  47), INT8_C(  45),
                           INT8_C( -29), INT8_C( -83), INT8_C( -52), INT8_C(  82)),
      simde_mm256_set_epi8(INT8_C(  -2), INT8_C( -84), INT8_C( -69), INT8_C( -58),
                           INT8_C(-128), INT8_C(  76), INT8_C( 110), INT8_C( -99),
                           INT8_C(  46), INT8_C(   8), INT8_C( -50), INT8_C(  -2),
                           INT8_C( 114), INT8_C(  31), INT8_C( -27), INT8_C( -16),
                           INT8_C(  63), INT8_C( -83), INT8_C(-114), INT8_C( 116),
                           INT8_C(  14), INT8_C( -31), INT8_C(   3), INT8_C(-105),
                           INT8_C( -84), INT8_C( -19), INT8_C(  81), INT8_C(  57),
                           INT8_C( -55), INT8_C( -67), INT8_C( -89), INT8_C(  74)),
      simde_mm256_set_epi8(INT8_C(  65), INT8_C(  23), INT8_C(  64), INT8_C( -32),
                           INT8_C( -17), INT8_C(  98), INT8_C( 110), INT8_C( 100),
                           INT8_C(  46), INT8_C(   8), INT8_C( -13), INT8_C( 105),
                           INT8_C( 114), INT8_C(  62), INT8_C( -27), INT8_C(  30),
                           INT8_C(  63), INT8_C(  71), INT8_C( -37), INT8_C( 116),
                           INT8_C(  14), INT8_C(  54), INT8_C(   3), INT8_C( 103),
                           INT8_C(  11), INT8_C( -19), INT8_C(  81), INT8_C(  57),
                           INT8_C( -29), INT8_C( -67), INT8_C( -52), INT8_C(  82)) },
    { simde_mm256_set_epi8(INT8_C(   3), INT8_C( -48), INT8_C( -19), INT8_C( 101),
                           INT8_C(  -3), INT8_C( -85), INT8_C(-102), INT8_C(  26),
                           INT8_C(  89), INT8_C( -95), INT8_C(  51), INT8_C(  -5),
                           INT8_C(  13), INT8_C(  30), INT8_C( -92), INT8_C( -13),
                           INT8_C(  68), INT8_C(  71), INT8_C(-127), INT8_C( -65),
                           INT8_C(-102), INT8_C(  -1), INT8_C(  48), INT8_C(  30),
                           INT8_C(  43), INT8_C(  71), INT8_C(  53), INT8_C(  78),
                           INT8_C(  81), INT8_C( 119), INT8_C( -32), INT8_C(  18)),
      simde_mm256_set_epi8(INT8_C( -81), INT8_C( -54), INT8_C( -70), INT8_C( 102),
                           INT8_C(  86), INT8_C( 106), INT8_C(  99), INT8_C(  -7),
                           INT8_C( -74), INT8_C( -52), INT8_C(-121), INT8_C(-127),
                           INT8_C(  49), INT8_C(-119), INT8_C(  52), INT8_C( 119),
                           INT8_C( 123), INT8_C(-104), INT8_C(  44), INT8_C( -14),
                           INT8_C(  56), INT8_C( -83), INT8_C(  21), INT8_C( 108),
                           INT8_C( -43), INT8_C(  75), INT8_C(  75), INT8_C(  51),
                           INT8_C(  54), INT8_C( -18), INT8_C( 100), INT8_C(  49)),
      simde_mm256_set_epi8(INT8_C(   3), INT8_C( -48), INT8_C( -19), INT8_C( 102),
                           INT8_C(  86), INT8_C( 106), INT8_C(  99), INT8_C(  26),
                           INT8_C(  89), INT8_C( -52), INT8_C(  51), INT8_C(  -5),
                           INT8_C(  49), INT8_C(  30), INT8_C(  52), INT8_C( 119),
                           INT8_C( 123), INT8_C(  71), INT8_C(  44), INT8_C( -14),
                           INT8_C(  56), INT8_C(  -1), INT8_C(  48), INT8_C( 108),
                           INT8_C(  43), INT8_C(  75), INT8_C(  75), INT8_C(  78),
                           INT8_C(  81), INT8_C( 119), INT8_C( 100), INT8_C(  49)) },
    { simde_mm256_set_epi8(INT8_C( -92), INT8_C(  89), INT8_C(  61), INT8_C( 117),
                           INT8_C( -42), INT8_C(  29), INT8_C( -36), INT8_C(  67),
                           INT8_C(-107), INT8_C(  95), INT8_C(-103), INT8_C( -42),
                           INT8_C( -63), INT8_C(  54), INT8_C(  53), INT8_C( -40),
                           INT8_C( -66), INT8_C( -31), INT8_C( -91), INT8_C(  82),
                           INT8_C( 110), INT8_C( -33), INT8_C(  21), INT8_C(  38),
                           INT8_C( -69), INT8_C(  34), INT8_C( -74), INT8_C(  24),
                           INT8_C( -26), INT8_C(-124), INT8_C( -81), INT8_C(  36)),
      simde_mm256_set_epi8(INT8_C(-105), INT8_C(  55), INT8_C(-117), INT8_C(  72),
                           INT8_C( -82), INT8_C( -42), INT8_C( 100), INT8_C( -21),
                           INT8_C(  19), INT8_C( 104), INT8_C(  11), INT8_C(  52),
                           INT8_C(  30), INT8_C(   8), INT8_C(  70), INT8_C( -51),
                           INT8_C( -61), INT8_C( -91), INT8_C( 126), INT8_C( -28),
                           INT8_C( -34), INT8_C( 102), INT8_C(  69), INT8_C(  99),
                           INT8_C(  81), INT8_C(  11), INT8_C( -51), INT8_C(   2),
                           INT8_C(  60), INT8_C( -70), INT8_C(  92), INT8_C( -37)),
      simde_mm256_set_epi8(INT8_C( -92), INT8_C(  89), INT8_C(  61), INT8_C( 117),
                           INT8_C( -42), INT8_C(  29), INT8_C( 100), INT8_C(  67),
                           INT8_C(  19), INT8_C( 104), INT8_C(  11), INT8_C(  52),
                           INT8_C(  30), INT8_C(  54), INT8_C(  70), INT8_C( -40),
                           INT8_C( -61), INT8_C( -31), INT8_C( 126), INT8_C(  82),
                           INT8_C( 110), INT8_C( 102), INT8_C(  69), INT8_C(  99),
                           INT8_C(  81), INT8_C(  34), INT8_C( -51), INT8_C(  24),
                           INT8_C(  60), INT8_C( -70), INT8_C(  92), INT8_C(  36)) },
    { simde_mm256_set_epi8(INT8_C(-123), INT8_C( 126), INT8_C(  24), INT8_C(  94),
                           INT8_C( -42), INT8_C( 118), INT8_C(   5), INT8_C( -41),
                           INT8_C( -65), INT8_C( -51), INT8_C( -68), INT8_C( 107),
                           INT8_C( -74), INT8_C( 112), INT8_C(  53), INT8_C(  87),
                           INT8_C(  26), INT8_C( -33), INT8_C(  99), INT8_C(-107),
                           INT8_C(   6), INT8_C( 119), INT8_C( -42), INT8_C(  32),
                           INT8_C(  54), INT8_C( -97), INT8_C(-117), INT8_C(  -9),
                           INT8_C(-107), INT8_C(  44), INT8_C( -40), INT8_C(  33)),
      simde_mm256_set_epi8(INT8_C(-115), INT8_C( -12), INT8_C( -81), INT8_C(-120),
                           INT8_C(  82), INT8_C( 114), INT8_C( -55), INT8_C(-106),
                           INT8_C( -54), INT8_C( 126), INT8_C(  70), INT8_C(  83),
                           INT8_C(  64), INT8_C( -11), INT8_C(  10), INT8_C(  72),
                           INT8_C(  19), INT8_C( -35), INT8_C( 123), INT8_C( -50),
                           INT8_C(  95), INT8_C(  -2), INT8_C(  77), INT8_C( -12),
                           INT8_C(-127), INT8_C( -73), INT8_C( 117), INT8_C(  14),
                           INT8_C(  40), INT8_C( -92), INT8_C(  74), INT8_C( 114)),
      simde_mm256_set_epi8(INT8_C(-115), INT8_C( 126), INT8_C(  24), INT8_C(  94),
                           INT8_C(  82), INT8_C( 118), INT8_C(   5), INT8_C( -41),
                           INT8_C( -54), INT8_C( 126), INT8_C(  70), INT8_C( 107),
                           INT8_C(  64), INT8_C( 112), INT8_C(  53), INT8_C(  87),
                           INT8_C(  26), INT8_C( -33), INT8_C( 123), INT8_C( -50),
                           INT8_C(  95), INT8_C( 119), INT8_C(  77), INT8_C(  32),
                           INT8_C(  54), INT8_C( -73), INT8_C( 117), INT8_C(  14),
                           INT8_C(  40), INT8_C(  44), INT8_C(  74), INT8_C( 114)) },
    { simde_mm256_set_epi8(INT8_C(  29), INT8_C(  41), INT8_C( 107), INT8_C(  24),
                           INT8_C(-113), INT8_C( 113), INT8_C( -53), INT8_C(   6),
                           INT8_C( -82), INT8_C(  34), INT8_C(   0), INT8_C(  35),
                           INT8_C( -43), INT8_C(-128), INT8_C( -61), INT8_C( -69),
                           INT8_C(   2), INT8_C(  64), INT8_C(  89), INT8_C( -88),
                           INT8_C( 111), INT8_C( -95), INT8_C(  42), INT8_C( -15),
                           INT8_C(  21), INT8_C(  73), INT8_C(  70), INT8_C( 127),
                           INT8_C(  32), INT8_C( 116), INT8_C(  36), INT8_C( -14)),
      simde_mm256_set_epi8(INT8_C(  52), INT8_C( -53), INT8_C( 101), INT8_C(  79),
                           INT8_C(  -8), INT8_C( -70), INT8_C( -81), INT8_C(-117),
                           INT8_C(  27), INT8_C( -60), INT8_C(  78), INT8_C(   6),
                           INT8_C(  79), INT8_C(  72), INT8_C(   1), INT8_C( -62),
                           INT8_C( -36), INT8_C( -42), INT8_C(   0), INT8_C( -10),
                           INT8_C( -19), INT8_C(  -1), INT8_C(  25), INT8_C( -88),
                           INT8_C(  -5), INT8_C( -10), INT8_C( 127), INT8_C( 114),
                           INT8_C( 110), INT8_C(  -1), INT8_C( -27), INT8_C(  51)),
      simde_mm256_set_epi8(INT8_C(  52), INT8_C(  41), INT8_C( 107), INT8_C(  79),
                           INT8_C(  -8), INT8_C( 113), INT8_C( -53), INT8_C(   6),
                           INT8_C(  27), INT8_C(  34), INT8_C(  78), INT8_C(  35),
                           INT8_C(  79), INT8_C(  72), INT8_C(   1), INT8_C( -62),
                           INT8_C(   2), INT8_C(  64), INT8_C(  89), INT8_C( -10),
                           INT8_C( 111), INT8_C(  -1), INT8_C(  42), INT8_C( -15),
                           INT8_C(  21), INT8_C(  73), INT8_C( 127), INT8_C( 127),
                           INT8_C( 110), INT8_C( 116), INT8_C(  36), INT8_C(  51)) },
    { simde_mm256_set_epi8(INT8_C(  94), INT8_C( -83), INT8_C( -81), INT8_C( 109),
                           INT8_C(  -8), INT8_C( 100), INT8_C(  40), INT8_C(-125),
                           INT8_C( 114), INT8_C( -99), INT8_C(  42), INT8_C(  35),
                           INT8_C(  59), INT8_C(  67), INT8_C(  26), INT8_C( -39),
                           INT8_C(  79), INT8_C( 116), INT8_C(   3), INT8_C( -47),
                           INT8_C(  73), INT8_C(  13), INT8_C( -83), INT8_C( -95),
                           INT8_C(  75), INT8_C( -48), INT8_C( -36), INT8_C( 127),
                           INT8_C( 117), INT8_C(  -9), INT8_C(  94), INT8_C( -87)),
      simde_mm256_set_epi8(INT8_C( -44), INT8_C( -67), INT8_C(-100), INT8_C(  73),
                           INT8_C(  63), INT8_C(  69), INT8_C( -48), INT8_C( -87),
                           INT8_C(  -1), INT8_C( -19), INT8_C(  -9), INT8_C(  66),
                           INT8_C(  27), INT8_C(  17), INT8_C(-109), INT8_C(  22),
                           INT8_C(  98), INT8_C( 106), INT8_C( -50), INT8_C( -90),
                           INT8_C(  92), INT8_C( 106), INT8_C( -59), INT8_C(  31),
                           INT8_C( -18), INT8_C( -17), INT8_C( -80), INT8_C( -46),
                           INT8_C(  67), INT8_C( -29), INT8_C(  44), INT8_C(  76)),
      simde_mm256_set_epi8(INT8_C(  94), INT8_C( -67), INT8_C( -81), INT8_C( 109),
                           INT8_C(  63), INT8_C( 100), INT8_C(  40), INT8_C( -87),
                           INT8_C( 114), INT8_C( -19), INT8_C(  42), INT8_C(  66),
                           INT8_C(  59), INT8_C(  67), INT8_C(  26), INT8_C(  22),
                           INT8_C(  98), INT8_C( 116), INT8_C(   3), INT8_C( -47),
                           INT8_C(  92), INT8_C( 106), INT8_C( -59), INT8_C(  31),
                           INT8_C(  75), INT8_C( -17), INT8_C( -36), INT8_C( 127),
                           INT8_C( 117), INT8_C(  -9), INT8_C(  94), INT8_C(  76)) },
  };


  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_max_epi8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_max_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
   { simde_x_mm256_set_epu8(UINT8_C(180), UINT8_C(215), UINT8_C( 58), UINT8_C(173),
                             UINT8_C(238), UINT8_C(242), UINT8_C( 74), UINT8_C(116),
                             UINT8_C(237), UINT8_C( 59), UINT8_C(170), UINT8_C(217),
                             UINT8_C(180), UINT8_C(149), UINT8_C(157), UINT8_C(250),
                             UINT8_C(224), UINT8_C(201), UINT8_C(109), UINT8_C(247),
                             UINT8_C( 25), UINT8_C(111), UINT8_C( 62), UINT8_C(129),
                             UINT8_C(115), UINT8_C(244), UINT8_C( 23), UINT8_C(181),
                             UINT8_C(205), UINT8_C(130), UINT8_C( 15), UINT8_C(  6)),
      simde_x_mm256_set_epu8(UINT8_C(129), UINT8_C(102), UINT8_C(249), UINT8_C(232),
                             UINT8_C(171), UINT8_C(250), UINT8_C(155), UINT8_C( 41),
                             UINT8_C(184), UINT8_C(159), UINT8_C(213), UINT8_C( 99),
                             UINT8_C(220), UINT8_C(155), UINT8_C(220), UINT8_C(163),
                             UINT8_C(219), UINT8_C(184), UINT8_C( 97), UINT8_C(230),
                             UINT8_C( 89), UINT8_C( 94), UINT8_C( 73), UINT8_C(152),
                             UINT8_C( 89), UINT8_C( 86), UINT8_C(115), UINT8_C(206),
                             UINT8_C(132), UINT8_C(230), UINT8_C(207), UINT8_C(  0)),
      simde_x_mm256_set_epu8(UINT8_C(180), UINT8_C(215), UINT8_C(249), UINT8_C(232),
                             UINT8_C(238), UINT8_C(250), UINT8_C(155), UINT8_C(116),
                             UINT8_C(237), UINT8_C(159), UINT8_C(213), UINT8_C(217),
                             UINT8_C(220), UINT8_C(155), UINT8_C(220), UINT8_C(250),
                             UINT8_C(224), UINT8_C(201), UINT8_C(109), UINT8_C(247),
                             UINT8_C( 89), UINT8_C(111), UINT8_C( 73), UINT8_C(152),
                             UINT8_C(115), UINT8_C(244), UINT8_C(115), UINT8_C(206),
                             UINT8_C(205), UINT8_C(230), UINT8_C(207), UINT8_C(  6)) },
    { simde_x_mm256_set_epu8(UINT8_C(218), UINT8_C(157), UINT8_C(237), UINT8_C(159),
                             UINT8_C( 73), UINT8_C(122), UINT8_C( 14), UINT8_C(243),
                             UINT8_C(232), UINT8_C(133), UINT8_C( 42), UINT8_C( 15),
                             UINT8_C( 24), UINT8_C(199), UINT8_C(216), UINT8_C( 25),
                             UINT8_C(242), UINT8_C(254), UINT8_C(156), UINT8_C(113),
                             UINT8_C(  5), UINT8_C( 95), UINT8_C( 73), UINT8_C(123),
                             UINT8_C(224), UINT8_C(233), UINT8_C(137), UINT8_C(219),
                             UINT8_C(122), UINT8_C(103), UINT8_C( 32), UINT8_C( 32)),
      simde_x_mm256_set_epu8(UINT8_C(255), UINT8_C(178), UINT8_C( 89), UINT8_C( 15),
                             UINT8_C( 21), UINT8_C(211), UINT8_C(214), UINT8_C(175),
                             UINT8_C( 24), UINT8_C(208), UINT8_C(242), UINT8_C( 16),
                             UINT8_C( 86), UINT8_C(207), UINT8_C(159), UINT8_C( 23),
                             UINT8_C( 74), UINT8_C( 72), UINT8_C(123), UINT8_C( 85),
                             UINT8_C( 99), UINT8_C(  6), UINT8_C( 15), UINT8_C(240),
                             UINT8_C(112), UINT8_C( 36), UINT8_C(217), UINT8_C( 82),
                             UINT8_C(251), UINT8_C( 67), UINT8_C(226), UINT8_C(252)),
      simde_x_mm256_set_epu8(UINT8_C(255), UINT8_C(178), UINT8_C(237), UINT8_C(159),
                             UINT8_C( 73), UINT8_C(211), UINT8_C(214), UINT8_C(243),
                             UINT8_C(232), UINT8_C(208), UINT8_C(242), UINT8_C( 16),
                             UINT8_C( 86), UINT8_C(207), UINT8_C(216), UINT8_C( 25),
                             UINT8_C(242), UINT8_C(254), UINT8_C(156), UINT8_C(113),
                             UINT8_C( 99), UINT8_C( 95), UINT8_C( 73), UINT8_C(240),
                             UINT8_C(224), UINT8_C(233), UINT8_C(217), UINT8_C(219),
                             UINT8_C(251), UINT8_C(103), UINT8_C(226), UINT8_C(252)) },
    { simde_x_mm256_set_epu8(UINT8_C( 99), UINT8_C(186), UINT8_C(163), UINT8_C(  4),
                             UINT8_C( 50), UINT8_C( 29), UINT8_C(  2), UINT8_C( 45),
                             UINT8_C( 57), UINT8_C(  0), UINT8_C(206), UINT8_C( 78),
                             UINT8_C(164), UINT8_C( 87), UINT8_C( 32), UINT8_C(133),
                             UINT8_C(239), UINT8_C(167), UINT8_C( 26), UINT8_C(218),
                             UINT8_C(142), UINT8_C( 23), UINT8_C( 41), UINT8_C( 63),
                             UINT8_C(230), UINT8_C(150), UINT8_C( 27), UINT8_C(237),
                             UINT8_C(105), UINT8_C(166), UINT8_C( 82), UINT8_C( 50)),
      simde_x_mm256_set_epu8(UINT8_C(129), UINT8_C(111), UINT8_C( 84), UINT8_C(213),
                             UINT8_C( 28), UINT8_C( 33), UINT8_C(141), UINT8_C(175),
                             UINT8_C( 38), UINT8_C(248), UINT8_C(  9), UINT8_C(  4),
                             UINT8_C(199), UINT8_C( 40), UINT8_C(185), UINT8_C(144),
                             UINT8_C( 46), UINT8_C( 14), UINT8_C( 90), UINT8_C(214),
                             UINT8_C( 91), UINT8_C( 56), UINT8_C( 35), UINT8_C(120),
                             UINT8_C(228), UINT8_C( 66), UINT8_C( 89), UINT8_C(196),
                             UINT8_C(150), UINT8_C(233), UINT8_C(114), UINT8_C(153)),
      simde_x_mm256_set_epu8(UINT8_C(129), UINT8_C(186), UINT8_C(163), UINT8_C(213),
                             UINT8_C( 50), UINT8_C( 33), UINT8_C(141), UINT8_C(175),
                             UINT8_C( 57), UINT8_C(248), UINT8_C(206), UINT8_C( 78),
                             UINT8_C(199), UINT8_C( 87), UINT8_C(185), UINT8_C(144),
                             UINT8_C(239), UINT8_C(167), UINT8_C( 90), UINT8_C(218),
                             UINT8_C(142), UINT8_C( 56), UINT8_C( 41), UINT8_C(120),
                             UINT8_C(230), UINT8_C(150), UINT8_C( 89), UINT8_C(237),
                             UINT8_C(150), UINT8_C(233), UINT8_C(114), UINT8_C(153)) },
    { simde_x_mm256_set_epu8(UINT8_C( 80), UINT8_C(  4), UINT8_C(125), UINT8_C(  7),
                             UINT8_C( 48), UINT8_C(207), UINT8_C( 44), UINT8_C(221),
                             UINT8_C(215), UINT8_C( 45), UINT8_C(223), UINT8_C(194),
                             UINT8_C(186), UINT8_C( 12), UINT8_C(145), UINT8_C(171),
                             UINT8_C(238), UINT8_C(191), UINT8_C( 72), UINT8_C(118),
                             UINT8_C( 66), UINT8_C(123), UINT8_C(140), UINT8_C(190),
                             UINT8_C(  2), UINT8_C(202), UINT8_C( 43), UINT8_C( 29),
                             UINT8_C( 54), UINT8_C( 64), UINT8_C(115), UINT8_C( 16)),
      simde_x_mm256_set_epu8(UINT8_C( 80), UINT8_C(158), UINT8_C(246), UINT8_C( 30),
                             UINT8_C(178), UINT8_C(129), UINT8_C(199), UINT8_C(213),
                             UINT8_C( 85), UINT8_C(149), UINT8_C(108), UINT8_C( 59),
                             UINT8_C(205), UINT8_C(149), UINT8_C(119), UINT8_C(194),
                             UINT8_C(246), UINT8_C( 94), UINT8_C(221), UINT8_C(162),
                             UINT8_C( 94), UINT8_C(125), UINT8_C(138), UINT8_C(231),
                             UINT8_C(191), UINT8_C( 11), UINT8_C( 15), UINT8_C( 66),
                             UINT8_C(133), UINT8_C(100), UINT8_C( 76), UINT8_C( 29)),
      simde_x_mm256_set_epu8(UINT8_C( 80), UINT8_C(158), UINT8_C(246), UINT8_C( 30),
                             UINT8_C(178), UINT8_C(207), UINT8_C(199), UINT8_C(221),
                             UINT8_C(215), UINT8_C(149), UINT8_C(223), UINT8_C(194),
                             UINT8_C(205), UINT8_C(149), UINT8_C(145), UINT8_C(194),
                             UINT8_C(246), UINT8_C(191), UINT8_C(221), UINT8_C(162),
                             UINT8_C( 94), UINT8_C(125), UINT8_C(140), UINT8_C(231),
                             UINT8_C(191), UINT8_C(202), UINT8_C( 43), UINT8_C( 66),
                             UINT8_C(133), UINT8_C(100), UINT8_C(115), UINT8_C( 29)) },
    { simde_x_mm256_set_epu8(UINT8_C(197), UINT8_C( 85), UINT8_C( 13), UINT8_C(197),
                             UINT8_C( 97), UINT8_C(213), UINT8_C( 19), UINT8_C(106),
                             UINT8_C(135), UINT8_C(  2), UINT8_C(117), UINT8_C(164),
                             UINT8_C(206), UINT8_C(103), UINT8_C( 74), UINT8_C( 88),
                             UINT8_C(183), UINT8_C( 33), UINT8_C(103), UINT8_C(216),
                             UINT8_C(136), UINT8_C(200), UINT8_C( 59), UINT8_C(124),
                             UINT8_C(188), UINT8_C(115), UINT8_C(181), UINT8_C(132),
                             UINT8_C(156), UINT8_C( 81), UINT8_C(255), UINT8_C(214)),
      simde_x_mm256_set_epu8(UINT8_C( 95), UINT8_C( 13), UINT8_C(157), UINT8_C(137),
                             UINT8_C( 41), UINT8_C(108), UINT8_C(  8), UINT8_C( 29),
                             UINT8_C( 52), UINT8_C(238), UINT8_C( 31), UINT8_C( 23),
                             UINT8_C(  9), UINT8_C( 86), UINT8_C(  2), UINT8_C( 88),
                             UINT8_C( 88), UINT8_C(185), UINT8_C(173), UINT8_C(108),
                             UINT8_C(234), UINT8_C(252), UINT8_C(231), UINT8_C( 15),
                             UINT8_C( 68), UINT8_C(138), UINT8_C( 60), UINT8_C(162),
                             UINT8_C(131), UINT8_C(215), UINT8_C( 87), UINT8_C( 44)),
      simde_x_mm256_set_epu8(UINT8_C(197), UINT8_C( 85), UINT8_C(157), UINT8_C(197),
                             UINT8_C( 97), UINT8_C(213), UINT8_C( 19), UINT8_C(106),
                             UINT8_C(135), UINT8_C(238), UINT8_C(117), UINT8_C(164),
                             UINT8_C(206), UINT8_C(103), UINT8_C( 74), UINT8_C( 88),
                             UINT8_C(183), UINT8_C(185), UINT8_C(173), UINT8_C(216),
                             UINT8_C(234), UINT8_C(252), UINT8_C(231), UINT8_C(124),
                             UINT8_C(188), UINT8_C(138), UINT8_C(181), UINT8_C(162),
                             UINT8_C(156), UINT8_C(215), UINT8_C(255), UINT8_C(214)) },
    { simde_x_mm256_set_epu8(UINT8_C(231), UINT8_C(112), UINT8_C(155), UINT8_C( 33),
                             UINT8_C( 64), UINT8_C(148), UINT8_C(180), UINT8_C( 63),
                             UINT8_C( 75), UINT8_C(171), UINT8_C(170), UINT8_C(114),
                             UINT8_C(142), UINT8_C(212), UINT8_C(162), UINT8_C(149),
                             UINT8_C( 47), UINT8_C(201), UINT8_C( 71), UINT8_C( 17),
                             UINT8_C(237), UINT8_C(172), UINT8_C(186), UINT8_C( 26),
                             UINT8_C(227), UINT8_C(106), UINT8_C(215), UINT8_C(216),
                             UINT8_C(247), UINT8_C(225), UINT8_C(  9), UINT8_C( 60)),
      simde_x_mm256_set_epu8(UINT8_C(136), UINT8_C( 20), UINT8_C(248), UINT8_C(149),
                             UINT8_C(  4), UINT8_C(138), UINT8_C(203), UINT8_C( 63),
                             UINT8_C(168), UINT8_C(148), UINT8_C(215), UINT8_C( 68),
                             UINT8_C(209), UINT8_C(109), UINT8_C(191), UINT8_C(165),
                             UINT8_C( 20), UINT8_C( 98), UINT8_C(113), UINT8_C(151),
                             UINT8_C(142), UINT8_C(111), UINT8_C(162), UINT8_C(190),
                             UINT8_C(230), UINT8_C(122), UINT8_C(213), UINT8_C(213),
                             UINT8_C(118), UINT8_C(189), UINT8_C( 22), UINT8_C(229)),
      simde_x_mm256_set_epu8(UINT8_C(231), UINT8_C(112), UINT8_C(248), UINT8_C(149),
                             UINT8_C( 64), UINT8_C(148), UINT8_C(203), UINT8_C( 63),
                             UINT8_C(168), UINT8_C(171), UINT8_C(215), UINT8_C(114),
                             UINT8_C(209), UINT8_C(212), UINT8_C(191), UINT8_C(165),
                             UINT8_C( 47), UINT8_C(201), UINT8_C(113), UINT8_C(151),
                             UINT8_C(237), UINT8_C(172), UINT8_C(186), UINT8_C(190),
                             UINT8_C(230), UINT8_C(122), UINT8_C(215), UINT8_C(216),
                             UINT8_C(247), UINT8_C(225), UINT8_C( 22), UINT8_C(229)) },
    { simde_x_mm256_set_epu8(UINT8_C(183), UINT8_C(  9), UINT8_C( 46), UINT8_C( 70),
                             UINT8_C( 48), UINT8_C(117), UINT8_C(202), UINT8_C(154),
                             UINT8_C(250), UINT8_C(204), UINT8_C(191), UINT8_C( 51),
                             UINT8_C( 37), UINT8_C(  5), UINT8_C(178), UINT8_C( 19),
                             UINT8_C(105), UINT8_C( 57), UINT8_C( 19), UINT8_C( 60),
                             UINT8_C( 26), UINT8_C( 52), UINT8_C(197), UINT8_C( 41),
                             UINT8_C(112), UINT8_C(146), UINT8_C(171), UINT8_C( 51),
                             UINT8_C( 94), UINT8_C(195), UINT8_C(226), UINT8_C(203)),
      simde_x_mm256_set_epu8(UINT8_C( 96), UINT8_C( 68), UINT8_C( 60), UINT8_C( 83),
                             UINT8_C(130), UINT8_C( 56), UINT8_C(227), UINT8_C(106),
                             UINT8_C(254), UINT8_C(175), UINT8_C(176), UINT8_C(122),
                             UINT8_C(126), UINT8_C(122), UINT8_C(171), UINT8_C(205),
                             UINT8_C( 85), UINT8_C(250), UINT8_C( 22), UINT8_C(153),
                             UINT8_C(140), UINT8_C(119), UINT8_C( 56), UINT8_C(225),
                             UINT8_C( 78), UINT8_C(180), UINT8_C(234), UINT8_C(136),
                             UINT8_C( 47), UINT8_C(214), UINT8_C( 40), UINT8_C(206)),
      simde_x_mm256_set_epu8(UINT8_C(183), UINT8_C( 68), UINT8_C( 60), UINT8_C( 83),
                             UINT8_C(130), UINT8_C(117), UINT8_C(227), UINT8_C(154),
                             UINT8_C(254), UINT8_C(204), UINT8_C(191), UINT8_C(122),
                             UINT8_C(126), UINT8_C(122), UINT8_C(178), UINT8_C(205),
                             UINT8_C(105), UINT8_C(250), UINT8_C( 22), UINT8_C(153),
                             UINT8_C(140), UINT8_C(119), UINT8_C(197), UINT8_C(225),
                             UINT8_C(112), UINT8_C(180), UINT8_C(234), UINT8_C(136),
                             UINT8_C( 94), UINT8_C(214), UINT8_C(226), UINT8_C(206)) },
    { simde_x_mm256_set_epu8(UINT8_C( 93), UINT8_C(234), UINT8_C(139), UINT8_C(  2),
                             UINT8_C(  9), UINT8_C(232), UINT8_C( 35), UINT8_C( 78),
                             UINT8_C(197), UINT8_C( 13), UINT8_C(224), UINT8_C( 83),
                             UINT8_C( 37), UINT8_C(182), UINT8_C( 94), UINT8_C( 69),
                             UINT8_C(143), UINT8_C( 54), UINT8_C(219), UINT8_C(171),
                             UINT8_C( 22), UINT8_C(117), UINT8_C( 46), UINT8_C(238),
                             UINT8_C(  4), UINT8_C(125), UINT8_C(  7), UINT8_C(106),
                             UINT8_C(127), UINT8_C( 87), UINT8_C( 56), UINT8_C( 20)),
      simde_x_mm256_set_epu8(UINT8_C( 73), UINT8_C(103), UINT8_C( 49), UINT8_C(198),
                             UINT8_C(212), UINT8_C(255), UINT8_C(227), UINT8_C( 52),
                             UINT8_C(122), UINT8_C( 22), UINT8_C(213), UINT8_C(204),
                             UINT8_C(103), UINT8_C( 93), UINT8_C( 45), UINT8_C( 45),
                             UINT8_C(221), UINT8_C(118), UINT8_C( 73), UINT8_C( 16),
                             UINT8_C(194), UINT8_C( 60), UINT8_C(246), UINT8_C(126),
                             UINT8_C( 84), UINT8_C(155), UINT8_C(128), UINT8_C( 83),
                             UINT8_C(  5), UINT8_C( 37), UINT8_C(157), UINT8_C(132)),
      simde_x_mm256_set_epu8(UINT8_C( 93), UINT8_C(234), UINT8_C(139), UINT8_C(198),
                             UINT8_C(212), UINT8_C(255), UINT8_C(227), UINT8_C( 78),
                             UINT8_C(197), UINT8_C( 22), UINT8_C(224), UINT8_C(204),
                             UINT8_C(103), UINT8_C(182), UINT8_C( 94), UINT8_C( 69),
                             UINT8_C(221), UINT8_C(118), UINT8_C(219), UINT8_C(171),
                             UINT8_C(194), UINT8_C(117), UINT8_C(246), UINT8_C(238),
                             UINT8_C( 84), UINT8_C(155), UINT8_C(128), UINT8_C(106),
                             UINT8_C(127), UINT8_C( 87), UINT8_C(157), UINT8_C(132)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_max_epu8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_max_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
   { simde_mm256_set_epi16(INT16_C( 13945), INT16_C(-32397), INT16_C( -8500), INT16_C(-16936),
                            INT16_C(  -615), INT16_C( 28879), INT16_C( 19952), INT16_C(-20844),
                            INT16_C(-18762), INT16_C( 23311), INT16_C(-22090), INT16_C( 16355),
                            INT16_C( 18939), INT16_C( -9393), INT16_C( 19655), INT16_C( 25483)),
      simde_mm256_set_epi16(INT16_C(-30076), INT16_C(-27428), INT16_C(-24061), INT16_C( -9620),
                            INT16_C(-17974), INT16_C(  3150), INT16_C(  6986), INT16_C(-23891),
                            INT16_C( 12942), INT16_C( 12645), INT16_C(-13904), INT16_C(  -868),
                            INT16_C(-14591), INT16_C( 24935), INT16_C(  4654), INT16_C(  4591)),
      simde_mm256_set_epi16(INT16_C( 13945), INT16_C(-27428), INT16_C( -8500), INT16_C( -9620),
                            INT16_C(  -615), INT16_C( 28879), INT16_C( 19952), INT16_C(-20844),
                            INT16_C( 12942), INT16_C( 23311), INT16_C(-13904), INT16_C( 16355),
                            INT16_C( 18939), INT16_C( 24935), INT16_C( 19655), INT16_C( 25483)) },
    { simde_mm256_set_epi16(INT16_C( 15746), INT16_C( 16476), INT16_C(-26892), INT16_C( 15551),
                            INT16_C(  4802), INT16_C( 22020), INT16_C(  7684), INT16_C( 20433),
                            INT16_C(-15213), INT16_C( -9067), INT16_C( 11028), INT16_C(  2768),
                            INT16_C( 19036), INT16_C(-29021), INT16_C( 27796), INT16_C(-20181)),
      simde_mm256_set_epi16(INT16_C(-15017), INT16_C(-10120), INT16_C(-23789), INT16_C(-21730),
                            INT16_C( 16078), INT16_C(  3607), INT16_C(-18668), INT16_C(-12209),
                            INT16_C(  -562), INT16_C( 30247), INT16_C( 20324), INT16_C( -8924),
                            INT16_C( 24524), INT16_C( -5426), INT16_C( 25503), INT16_C(  2215)),
      simde_mm256_set_epi16(INT16_C( 15746), INT16_C( 16476), INT16_C(-23789), INT16_C( 15551),
                            INT16_C( 16078), INT16_C( 22020), INT16_C(  7684), INT16_C( 20433),
                            INT16_C(  -562), INT16_C( 30247), INT16_C( 20324), INT16_C(  2768),
                            INT16_C( 24524), INT16_C( -5426), INT16_C( 27796), INT16_C(  2215)) },
    { simde_mm256_set_epi16(INT16_C(  9558), INT16_C( 24602), INT16_C(-11854), INT16_C( 13316),
                            INT16_C( -7111), INT16_C(  5322), INT16_C( 10474), INT16_C( 14309),
                            INT16_C( 10175), INT16_C(-23285), INT16_C( -7105), INT16_C(-18408),
                            INT16_C( -9456), INT16_C(-31797), INT16_C(  6677), INT16_C( 24246)),
      simde_mm256_set_epi16(INT16_C(  8353), INT16_C(-12225), INT16_C( 10989), INT16_C( 20160),
                            INT16_C(-24928), INT16_C( 11285), INT16_C(-21080), INT16_C(-21637),
                            INT16_C(  -557), INT16_C(-15431), INT16_C(-14247), INT16_C(  1813),
                            INT16_C( -7571), INT16_C( 22502), INT16_C( 12550), INT16_C(  3083)),
      simde_mm256_set_epi16(INT16_C(  9558), INT16_C( 24602), INT16_C( 10989), INT16_C( 20160),
                            INT16_C( -7111), INT16_C( 11285), INT16_C( 10474), INT16_C( 14309),
                            INT16_C( 10175), INT16_C(-15431), INT16_C( -7105), INT16_C(  1813),
                            INT16_C( -7571), INT16_C( 22502), INT16_C( 12550), INT16_C( 24246)) },
    { simde_mm256_set_epi16(INT16_C( 25542), INT16_C(-21715), INT16_C(-12723), INT16_C(  3641),
                            INT16_C( 10626), INT16_C(  1975), INT16_C( 32038), INT16_C( 13257),
                            INT16_C( 26030), INT16_C(-29459), INT16_C( 27784), INT16_C(-29348),
                            INT16_C( 24986), INT16_C( 19201), INT16_C(-21133), INT16_C(  -544)),
      simde_mm256_set_epi16(INT16_C(-15360), INT16_C( -9091), INT16_C(  9964), INT16_C(-26119),
                            INT16_C(-13193), INT16_C( -4473), INT16_C(-16910), INT16_C( -8355),
                            INT16_C( 29088), INT16_C( 26903), INT16_C( 19970), INT16_C( 20929),
                            INT16_C( 21866), INT16_C(  -554), INT16_C(-31252), INT16_C(-27281)),
      simde_mm256_set_epi16(INT16_C( 25542), INT16_C( -9091), INT16_C(  9964), INT16_C(  3641),
                            INT16_C( 10626), INT16_C(  1975), INT16_C( 32038), INT16_C( 13257),
                            INT16_C( 29088), INT16_C( 26903), INT16_C( 27784), INT16_C( 20929),
                            INT16_C( 24986), INT16_C( 19201), INT16_C(-21133), INT16_C(  -544)) },
    { simde_mm256_set_epi16(INT16_C(-28957), INT16_C(  7788), INT16_C(-11350), INT16_C( 29385),
                            INT16_C( -7207), INT16_C(-21363), INT16_C(-10963), INT16_C(-23177),
                            INT16_C( 22883), INT16_C(-14765), INT16_C( 17155), INT16_C(-19202),
                            INT16_C(-32343), INT16_C(  8167), INT16_C( 17107), INT16_C(-17194)),
      simde_mm256_set_epi16(INT16_C(-21587), INT16_C( 21036), INT16_C( 16036), INT16_C(-20844),
                            INT16_C(  2600), INT16_C(-15232), INT16_C(  4125), INT16_C( 22708),
                            INT16_C(-29000), INT16_C(-22285), INT16_C(  8338), INT16_C( 25191),
                            INT16_C(  5401), INT16_C(-28508), INT16_C(-16518), INT16_C(-23781)),
      simde_mm256_set_epi16(INT16_C(-21587), INT16_C( 21036), INT16_C( 16036), INT16_C( 29385),
                            INT16_C(  2600), INT16_C(-15232), INT16_C(  4125), INT16_C( 22708),
                            INT16_C( 22883), INT16_C(-14765), INT16_C( 17155), INT16_C( 25191),
                            INT16_C(  5401), INT16_C(  8167), INT16_C( 17107), INT16_C(-17194)) },
    { simde_mm256_set_epi16(INT16_C( 26823), INT16_C( 30422), INT16_C(-10399), INT16_C(  6469),
                            INT16_C(-22608), INT16_C(  3602), INT16_C( 24356), INT16_C( -5312),
                            INT16_C( 22553), INT16_C(  4057), INT16_C( 16984), INT16_C( 14642),
                            INT16_C( -4725), INT16_C(-13132), INT16_C(-32486), INT16_C( -9795)),
      simde_mm256_set_epi16(INT16_C(-16670), INT16_C( 11037), INT16_C(-12867), INT16_C( -9733),
                            INT16_C(-19057), INT16_C( 28626), INT16_C( 27349), INT16_C( 21991),
                            INT16_C(-17860), INT16_C( -7267), INT16_C( 21708), INT16_C( 27219),
                            INT16_C(  3344), INT16_C(-13000), INT16_C( -5702), INT16_C(  8873)),
      simde_mm256_set_epi16(INT16_C( 26823), INT16_C( 30422), INT16_C(-10399), INT16_C(  6469),
                            INT16_C(-19057), INT16_C( 28626), INT16_C( 27349), INT16_C( 21991),
                            INT16_C( 22553), INT16_C(  4057), INT16_C( 21708), INT16_C( 27219),
                            INT16_C(  3344), INT16_C(-13000), INT16_C( -5702), INT16_C(  8873)) },
    { simde_mm256_set_epi16(INT16_C( 28327), INT16_C( 28059), INT16_C(-12455), INT16_C(-22892),
                            INT16_C(  9516), INT16_C(-23091), INT16_C( 28067), INT16_C( 29028),
                            INT16_C(-24610), INT16_C(-13303), INT16_C( -3871), INT16_C( 25753),
                            INT16_C(-19764), INT16_C(-10115), INT16_C( -9313), INT16_C(-10807)),
      simde_mm256_set_epi16(INT16_C(  1267), INT16_C( -7929), INT16_C( 29505), INT16_C( 22745),
                            INT16_C(  1801), INT16_C(  -829), INT16_C(  2659), INT16_C( 24204),
                            INT16_C( 27266), INT16_C(-10805), INT16_C(  7071), INT16_C( 12404),
                            INT16_C( 27748), INT16_C( 11490), INT16_C(-10130), INT16_C(  7320)),
      simde_mm256_set_epi16(INT16_C( 28327), INT16_C( 28059), INT16_C( 29505), INT16_C( 22745),
                            INT16_C(  9516), INT16_C(  -829), INT16_C( 28067), INT16_C( 29028),
                            INT16_C( 27266), INT16_C(-10805), INT16_C(  7071), INT16_C( 25753),
                            INT16_C( 27748), INT16_C( 11490), INT16_C( -9313), INT16_C(  7320)) },
    { simde_mm256_set_epi16(INT16_C(-28396), INT16_C(  3489), INT16_C( 18687), INT16_C( 31879),
                            INT16_C( 15536), INT16_C(-17843), INT16_C( 25087), INT16_C(-27693),
                            INT16_C(-17170), INT16_C( 22369), INT16_C(-27360), INT16_C( -5829),
                            INT16_C(-28550), INT16_C(   650), INT16_C(-11446), INT16_C(  3172)),
      simde_mm256_set_epi16(INT16_C(-17057), INT16_C(-18950), INT16_C( 15210), INT16_C(-25469),
                            INT16_C(-29475), INT16_C( 24211), INT16_C(-14409), INT16_C( -9847),
                            INT16_C( 14530), INT16_C(-24184), INT16_C(-29404), INT16_C(  9879),
                            INT16_C(  2544), INT16_C(-17943), INT16_C(  7392), INT16_C( 21259)),
      simde_mm256_set_epi16(INT16_C(-17057), INT16_C(  3489), INT16_C( 18687), INT16_C( 31879),
                            INT16_C( 15536), INT16_C( 24211), INT16_C( 25087), INT16_C( -9847),
                            INT16_C( 14530), INT16_C( 22369), INT16_C(-27360), INT16_C(  9879),
                            INT16_C(  2544), INT16_C(   650), INT16_C(  7392), INT16_C( 21259)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_max_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_max_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(-1578701412), INT32_C(-1861943275), INT32_C( 1717826073), INT32_C( -689858277),
                            INT32_C(-2120069619), INT32_C( -269745295), INT32_C(  993893699), INT32_C( 1747535129)),
      simde_mm256_set_epi32(INT32_C(-1662415513), INT32_C(  528745592), INT32_C( -219858588), INT32_C(  622357704),
                            INT32_C(-2013314779), INT32_C(-1188086430), INT32_C( 1171120200), INT32_C(  259652605)),
      simde_mm256_set_epi32(INT32_C(-1578701412), INT32_C(  528745592), INT32_C( 1717826073), INT32_C(  622357704),
                            INT32_C(-2013314779), INT32_C( -269745295), INT32_C( 1171120200), INT32_C( 1747535129)) },
    { simde_mm256_set_epi32(INT32_C( 1892880717), INT32_C(  489135272), INT32_C( 1334433155), INT32_C(  820950025),
                            INT32_C(  875780372), INT32_C( 1165542940), INT32_C(-1922521413), INT32_C( 1603602866)),
      simde_mm256_set_epi32(INT32_C( 1774078122), INT32_C(  717307143), INT32_C( 1385706638), INT32_C( 1722931608),
                            INT32_C(  722989282), INT32_C(  346685962), INT32_C(  888351034), INT32_C(-1040558861)),
      simde_mm256_set_epi32(INT32_C( 1892880717), INT32_C(  717307143), INT32_C( 1385706638), INT32_C( 1722931608),
                            INT32_C(  875780372), INT32_C( 1165542940), INT32_C(  888351034), INT32_C( 1603602866)) },
    { simde_mm256_set_epi32(INT32_C(  542527818), INT32_C(  704072326), INT32_C(-1102544845), INT32_C( 1444592706),
                            INT32_C(-1925092178), INT32_C( -204731801), INT32_C(  775742710), INT32_C( 1177505754)),
      simde_mm256_set_epi32(INT32_C( -551142491), INT32_C(  444795418), INT32_C( -825745617), INT32_C(-1787304548),
                            INT32_C(-1241777147), INT32_C(  265612525), INT32_C( 2134936507), INT32_C(-1780599144)),
      simde_mm256_set_epi32(INT32_C(  542527818), INT32_C(  704072326), INT32_C( -825745617), INT32_C( 1444592706),
                            INT32_C(-1241777147), INT32_C(  265612525), INT32_C( 2134936507), INT32_C( 1177505754)) },
    { simde_mm256_set_epi32(INT32_C(  788610578), INT32_C( 1831830497), INT32_C( -755374494), INT32_C( 2130364415),
                            INT32_C( -342185910), INT32_C(  854700402), INT32_C( 1343385181), INT32_C(  891289886)),
      simde_mm256_set_epi32(INT32_C( 1917738489), INT32_C(  690751883), INT32_C(-1037858966), INT32_C(-1567909551),
                            INT32_C(  -26783282), INT32_C( -715472333), INT32_C( -713074037), INT32_C(  -17697982)),
      simde_mm256_set_epi32(INT32_C( 1917738489), INT32_C( 1831830497), INT32_C( -755374494), INT32_C( 2130364415),
                            INT32_C(  -26783282), INT32_C(  854700402), INT32_C( 1343385181), INT32_C(  891289886)) },
    { simde_mm256_set_epi32(INT32_C( 1796568981), INT32_C(-1846046069), INT32_C(-1495880353), INT32_C(-1105562137),
                            INT32_C(-2000119429), INT32_C(  450352139), INT32_C( -734796291), INT32_C(-1851159287)),
      simde_mm256_set_epi32(INT32_C( 1076027923), INT32_C( 1999677975), INT32_C( -498539521), INT32_C(-1166856281),
                            INT32_C(-1611419248), INT32_C(-1268341170), INT32_C(-1115547457), INT32_C( 1554907000)),
      simde_mm256_set_epi32(INT32_C( 1796568981), INT32_C( 1999677975), INT32_C( -498539521), INT32_C(-1105562137),
                            INT32_C(-1611419248), INT32_C(  450352139), INT32_C( -734796291), INT32_C( 1554907000)) },
    { simde_mm256_set_epi32(INT32_C( -263364521), INT32_C( 1666932430), INT32_C(  378039954), INT32_C( 1866502452),
                            INT32_C( -756222443), INT32_C( -752660448), INT32_C( 1087715357), INT32_C( 1808069656)),
      simde_mm256_set_epi32(INT32_C(   10375777), INT32_C(  958545984), INT32_C(-1916055393), INT32_C(-1450264731),
                            INT32_C( -550755823), INT32_C( 2131394316), INT32_C(  243861812), INT32_C( 1031114919)),
      simde_mm256_set_epi32(INT32_C(   10375777), INT32_C( 1666932430), INT32_C(  378039954), INT32_C( 1866502452),
                            INT32_C( -550755823), INT32_C( 2131394316), INT32_C( 1087715357), INT32_C( 1808069656)) },
    { simde_mm256_set_epi32(INT32_C(-1075159077), INT32_C( -789508054), INT32_C(-2102436600), INT32_C(  177892995),
                            INT32_C(-1910430929), INT32_C( -135620958), INT32_C( 1899951190), INT32_C(-1286116105)),
      simde_mm256_set_epi32(INT32_C(-1117240644), INT32_C(-1011771686), INT32_C(-1764247251), INT32_C( -953836385),
                            INT32_C(-1633093106), INT32_C( 1815106343), INT32_C( 1418749534), INT32_C( 1718021188)),
      simde_mm256_set_epi32(INT32_C(-1075159077), INT32_C( -789508054), INT32_C(-1764247251), INT32_C(  177892995),
                            INT32_C(-1633093106), INT32_C( 1815106343), INT32_C( 1899951190), INT32_C( 1718021188)) },
    { simde_mm256_set_epi32(INT32_C( -585726505), INT32_C( -735532451), INT32_C( 1572773329), INT32_C(-1610167093),
                            INT32_C(  934479765), INT32_C( 1726304740), INT32_C(-1433078949), INT32_C(-1056217637)),
      simde_mm256_set_epi32(INT32_C(  201330788), INT32_C( 1825214883), INT32_C(  489777084), INT32_C( -824102072),
                            INT32_C(-1255028012), INT32_C(-1300324544), INT32_C(-1269112569), INT32_C( -124131174)),
      simde_mm256_set_epi32(INT32_C(  201330788), INT32_C( 1825214883), INT32_C( 1572773329), INT32_C( -824102072),
                            INT32_C(  934479765), INT32_C( 1726304740), INT32_C(-1269112569), INT32_C( -124131174)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_max_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_min_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu8(UINT8_C( 83), UINT8_C( 77), UINT8_C(142), UINT8_C(252),
                             UINT8_C( 19), UINT8_C( 26), UINT8_C(193), UINT8_C( 92),
                             UINT8_C(253), UINT8_C(183), UINT8_C(225), UINT8_C(205),
                             UINT8_C(165), UINT8_C( 19), UINT8_C(154), UINT8_C(158),
                             UINT8_C(126), UINT8_C(221), UINT8_C(206), UINT8_C( 22),
                             UINT8_C(217), UINT8_C(166), UINT8_C(237), UINT8_C(161),
                             UINT8_C(207), UINT8_C( 26), UINT8_C( 36), UINT8_C( 78),
                             UINT8_C( 55), UINT8_C(167), UINT8_C( 36), UINT8_C(198)),
      simde_x_mm256_set_epu8(UINT8_C(203), UINT8_C( 44), UINT8_C( 47), UINT8_C( 51),
                             UINT8_C(185), UINT8_C( 39), UINT8_C(221), UINT8_C( 75),
                             UINT8_C(182), UINT8_C(100), UINT8_C(238), UINT8_C(172),
                             UINT8_C(248), UINT8_C(164), UINT8_C( 91), UINT8_C(138),
                             UINT8_C(128), UINT8_C( 29), UINT8_C(136), UINT8_C(164),
                             UINT8_C( 43), UINT8_C( 85), UINT8_C(168), UINT8_C(105),
                             UINT8_C(240), UINT8_C(205), UINT8_C(221), UINT8_C(217),
                             UINT8_C(244), UINT8_C(220), UINT8_C(102), UINT8_C( 37)),
      simde_x_mm256_set_epu8(UINT8_C( 83), UINT8_C( 44), UINT8_C( 47), UINT8_C( 51),
                             UINT8_C( 19), UINT8_C( 26), UINT8_C(193), UINT8_C( 75),
                             UINT8_C(182), UINT8_C(100), UINT8_C(225), UINT8_C(172),
                             UINT8_C(165), UINT8_C( 19), UINT8_C( 91), UINT8_C(138),
                             UINT8_C(126), UINT8_C( 29), UINT8_C(136), UINT8_C( 22),
                             UINT8_C( 43), UINT8_C( 85), UINT8_C(168), UINT8_C(105),
                             UINT8_C(207), UINT8_C( 26), UINT8_C( 36), UINT8_C( 78),
                             UINT8_C( 55), UINT8_C(167), UINT8_C( 36), UINT8_C( 37)) },
    { simde_x_mm256_set_epu8(UINT8_C(177), UINT8_C(210), UINT8_C( 28), UINT8_C(116),
                             UINT8_C(174), UINT8_C(162), UINT8_C(241), UINT8_C( 21),
                             UINT8_C(126), UINT8_C(202), UINT8_C(250), UINT8_C(169),
                             UINT8_C( 43), UINT8_C(239), UINT8_C(224), UINT8_C(202),
                             UINT8_C(208), UINT8_C( 48), UINT8_C(132), UINT8_C( 78),
                             UINT8_C( 59), UINT8_C(  2), UINT8_C(213), UINT8_C(228),
                             UINT8_C( 42), UINT8_C( 45), UINT8_C(244), UINT8_C( 41),
                             UINT8_C( 49), UINT8_C( 48), UINT8_C(111), UINT8_C(211)),
      simde_x_mm256_set_epu8(UINT8_C(114), UINT8_C( 33), UINT8_C(210), UINT8_C(232),
                             UINT8_C(128), UINT8_C(122), UINT8_C(216), UINT8_C(228),
                             UINT8_C( 91), UINT8_C( 37), UINT8_C(119), UINT8_C(203),
                             UINT8_C(236), UINT8_C( 24), UINT8_C(170), UINT8_C(120),
                             UINT8_C( 95), UINT8_C(231), UINT8_C(152), UINT8_C(143),
                             UINT8_C( 42), UINT8_C( 38), UINT8_C(240), UINT8_C(125),
                             UINT8_C(124), UINT8_C(251), UINT8_C(118), UINT8_C( 44),
                             UINT8_C( 68), UINT8_C( 42), UINT8_C( 54), UINT8_C(185)),
      simde_x_mm256_set_epu8(UINT8_C(114), UINT8_C( 33), UINT8_C( 28), UINT8_C(116),
                             UINT8_C(128), UINT8_C(122), UINT8_C(216), UINT8_C( 21),
                             UINT8_C( 91), UINT8_C( 37), UINT8_C(119), UINT8_C(169),
                             UINT8_C( 43), UINT8_C( 24), UINT8_C(170), UINT8_C(120),
                             UINT8_C( 95), UINT8_C( 48), UINT8_C(132), UINT8_C( 78),
                             UINT8_C( 42), UINT8_C(  2), UINT8_C(213), UINT8_C(125),
                             UINT8_C( 42), UINT8_C( 45), UINT8_C(118), UINT8_C( 41),
                             UINT8_C( 49), UINT8_C( 42), UINT8_C( 54), UINT8_C(185)) },
    { simde_x_mm256_set_epu8(UINT8_C( 78), UINT8_C( 16), UINT8_C( 42), UINT8_C(217),
                             UINT8_C(  5), UINT8_C( 96), UINT8_C(  7), UINT8_C( 83),
                             UINT8_C( 16), UINT8_C(208), UINT8_C(  1), UINT8_C(207),
                             UINT8_C( 75), UINT8_C(207), UINT8_C(252), UINT8_C( 92),
                             UINT8_C(122), UINT8_C( 24), UINT8_C( 32), UINT8_C(104),
                             UINT8_C( 88), UINT8_C( 59), UINT8_C(218), UINT8_C( 28),
                             UINT8_C( 51), UINT8_C( 64), UINT8_C( 65), UINT8_C( 92),
                             UINT8_C( 24), UINT8_C( 83), UINT8_C(248), UINT8_C(243)),
      simde_x_mm256_set_epu8(UINT8_C(167), UINT8_C( 87), UINT8_C( 75), UINT8_C( 10),
                             UINT8_C( 44), UINT8_C(173), UINT8_C(210), UINT8_C(150),
                             UINT8_C(187), UINT8_C(229), UINT8_C(246), UINT8_C( 10),
                             UINT8_C( 80), UINT8_C( 74), UINT8_C(174), UINT8_C(128),
                             UINT8_C( 64), UINT8_C(160), UINT8_C(126), UINT8_C(231),
                             UINT8_C(215), UINT8_C( 80), UINT8_C(115), UINT8_C( 66),
                             UINT8_C(204), UINT8_C( 18), UINT8_C(158), UINT8_C(133),
                             UINT8_C(170), UINT8_C(119), UINT8_C(216), UINT8_C(196)),
      simde_x_mm256_set_epu8(UINT8_C( 78), UINT8_C( 16), UINT8_C( 42), UINT8_C( 10),
                             UINT8_C(  5), UINT8_C( 96), UINT8_C(  7), UINT8_C( 83),
                             UINT8_C( 16), UINT8_C(208), UINT8_C(  1), UINT8_C( 10),
                             UINT8_C( 75), UINT8_C( 74), UINT8_C(174), UINT8_C( 92),
                             UINT8_C( 64), UINT8_C( 24), UINT8_C( 32), UINT8_C(104),
                             UINT8_C( 88), UINT8_C( 59), UINT8_C(115), UINT8_C( 28),
                             UINT8_C( 51), UINT8_C( 18), UINT8_C( 65), UINT8_C( 92),
                             UINT8_C( 24), UINT8_C( 83), UINT8_C(216), UINT8_C(196)) },
    { simde_x_mm256_set_epu8(UINT8_C(  4), UINT8_C(145), UINT8_C( 75), UINT8_C( 42),
                             UINT8_C( 60), UINT8_C(146), UINT8_C(158), UINT8_C(  3),
                             UINT8_C( 57), UINT8_C(210), UINT8_C( 14), UINT8_C(106),
                             UINT8_C( 96), UINT8_C(136), UINT8_C( 82), UINT8_C( 54),
                             UINT8_C(227), UINT8_C(202), UINT8_C( 35), UINT8_C(184),
                             UINT8_C( 41), UINT8_C(205), UINT8_C(162), UINT8_C(206),
                             UINT8_C(247), UINT8_C( 69), UINT8_C(139), UINT8_C(186),
                             UINT8_C( 13), UINT8_C(119), UINT8_C( 79), UINT8_C( 53)),
      simde_x_mm256_set_epu8(UINT8_C(239), UINT8_C(239), UINT8_C( 37), UINT8_C(187),
                             UINT8_C(237), UINT8_C(203), UINT8_C( 68), UINT8_C( 35),
                             UINT8_C(147), UINT8_C(109), UINT8_C(244), UINT8_C(194),
                             UINT8_C(131), UINT8_C(189), UINT8_C( 51), UINT8_C( 84),
                             UINT8_C( 48), UINT8_C( 26), UINT8_C(203), UINT8_C( 35),
                             UINT8_C(128), UINT8_C( 76), UINT8_C( 16), UINT8_C( 52),
                             UINT8_C(103), UINT8_C(192), UINT8_C(105), UINT8_C( 65),
                             UINT8_C(117), UINT8_C(100), UINT8_C( 71), UINT8_C(124)),
      simde_x_mm256_set_epu8(UINT8_C(  4), UINT8_C(145), UINT8_C( 37), UINT8_C( 42),
                             UINT8_C( 60), UINT8_C(146), UINT8_C( 68), UINT8_C(  3),
                             UINT8_C( 57), UINT8_C(109), UINT8_C( 14), UINT8_C(106),
                             UINT8_C( 96), UINT8_C(136), UINT8_C( 51), UINT8_C( 54),
                             UINT8_C( 48), UINT8_C( 26), UINT8_C( 35), UINT8_C( 35),
                             UINT8_C( 41), UINT8_C( 76), UINT8_C( 16), UINT8_C( 52),
                             UINT8_C(103), UINT8_C( 69), UINT8_C(105), UINT8_C( 65),
                             UINT8_C( 13), UINT8_C(100), UINT8_C( 71), UINT8_C( 53)) },
    { simde_x_mm256_set_epu8(UINT8_C(144), UINT8_C(140), UINT8_C(121), UINT8_C(161),
                             UINT8_C(  2), UINT8_C( 56), UINT8_C(102), UINT8_C(220),
                             UINT8_C(246), UINT8_C( 67), UINT8_C( 19), UINT8_C( 67),
                             UINT8_C( 18), UINT8_C(117), UINT8_C(155), UINT8_C( 84),
                             UINT8_C(  3), UINT8_C(204), UINT8_C( 72), UINT8_C( 34),
                             UINT8_C(218), UINT8_C(208), UINT8_C( 99), UINT8_C( 27),
                             UINT8_C(224), UINT8_C( 99), UINT8_C(119), UINT8_C(221),
                             UINT8_C( 98), UINT8_C(202), UINT8_C(146), UINT8_C( 18)),
      simde_x_mm256_set_epu8(UINT8_C(127), UINT8_C(108), UINT8_C( 76), UINT8_C(146),
                             UINT8_C(206), UINT8_C(221), UINT8_C(206), UINT8_C(168),
                             UINT8_C( 11), UINT8_C(166), UINT8_C(228), UINT8_C(131),
                             UINT8_C( 46), UINT8_C( 42), UINT8_C(192), UINT8_C(117),
                             UINT8_C(166), UINT8_C(151), UINT8_C(189), UINT8_C(138),
                             UINT8_C(121), UINT8_C( 19), UINT8_C( 89), UINT8_C(188),
                             UINT8_C(255), UINT8_C(104), UINT8_C(124), UINT8_C(206),
                             UINT8_C( 89), UINT8_C(160), UINT8_C(162), UINT8_C( 10)),
      simde_x_mm256_set_epu8(UINT8_C(127), UINT8_C(108), UINT8_C( 76), UINT8_C(146),
                             UINT8_C(  2), UINT8_C( 56), UINT8_C(102), UINT8_C(168),
                             UINT8_C( 11), UINT8_C( 67), UINT8_C( 19), UINT8_C( 67),
                             UINT8_C( 18), UINT8_C( 42), UINT8_C(155), UINT8_C( 84),
                             UINT8_C(  3), UINT8_C(151), UINT8_C( 72), UINT8_C( 34),
                             UINT8_C(121), UINT8_C( 19), UINT8_C( 89), UINT8_C( 27),
                             UINT8_C(224), UINT8_C( 99), UINT8_C(119), UINT8_C(206),
                             UINT8_C( 89), UINT8_C(160), UINT8_C(146), UINT8_C( 10)) },
    { simde_x_mm256_set_epu8(UINT8_C( 74), UINT8_C(143), UINT8_C( 19), UINT8_C(  8),
                             UINT8_C( 11), UINT8_C(124), UINT8_C( 76), UINT8_C(  6),
                             UINT8_C(148), UINT8_C( 67), UINT8_C(224), UINT8_C(163),
                             UINT8_C(113), UINT8_C(245), UINT8_C( 59), UINT8_C( 27),
                             UINT8_C(131), UINT8_C(161), UINT8_C(251), UINT8_C(125),
                             UINT8_C(201), UINT8_C(252), UINT8_C(140), UINT8_C( 24),
                             UINT8_C(254), UINT8_C(183), UINT8_C(205), UINT8_C(238),
                             UINT8_C(180), UINT8_C( 21), UINT8_C( 59), UINT8_C(174)),
      simde_x_mm256_set_epu8(UINT8_C( 98), UINT8_C(152), UINT8_C(212), UINT8_C(148),
                             UINT8_C(183), UINT8_C( 37), UINT8_C(170), UINT8_C( 93),
                             UINT8_C( 52), UINT8_C(182), UINT8_C(181), UINT8_C(242),
                             UINT8_C(229), UINT8_C(182), UINT8_C(143), UINT8_C( 43),
                             UINT8_C(177), UINT8_C(115), UINT8_C(177), UINT8_C(170),
                             UINT8_C( 71), UINT8_C(222), UINT8_C(162), UINT8_C(198),
                             UINT8_C(195), UINT8_C(222), UINT8_C( 10), UINT8_C( 86),
                             UINT8_C( 48), UINT8_C(215), UINT8_C( 16), UINT8_C( 69)),
      simde_x_mm256_set_epu8(UINT8_C( 74), UINT8_C(143), UINT8_C( 19), UINT8_C(  8),
                             UINT8_C( 11), UINT8_C( 37), UINT8_C( 76), UINT8_C(  6),
                             UINT8_C( 52), UINT8_C( 67), UINT8_C(181), UINT8_C(163),
                             UINT8_C(113), UINT8_C(182), UINT8_C( 59), UINT8_C( 27),
                             UINT8_C(131), UINT8_C(115), UINT8_C(177), UINT8_C(125),
                             UINT8_C( 71), UINT8_C(222), UINT8_C(140), UINT8_C( 24),
                             UINT8_C(195), UINT8_C(183), UINT8_C( 10), UINT8_C( 86),
                             UINT8_C( 48), UINT8_C( 21), UINT8_C( 16), UINT8_C( 69)) },
    { simde_x_mm256_set_epu8(UINT8_C(119), UINT8_C(117), UINT8_C( 13), UINT8_C(207),
                             UINT8_C( 70), UINT8_C(197), UINT8_C(152), UINT8_C( 89),
                             UINT8_C(135), UINT8_C(159), UINT8_C( 50), UINT8_C(184),
                             UINT8_C(155), UINT8_C( 62), UINT8_C(253), UINT8_C(248),
                             UINT8_C(240), UINT8_C(207), UINT8_C( 53), UINT8_C(202),
                             UINT8_C(203), UINT8_C(241), UINT8_C(218), UINT8_C(118),
                             UINT8_C(121), UINT8_C(140), UINT8_C(125), UINT8_C( 65),
                             UINT8_C( 70), UINT8_C(249), UINT8_C(217), UINT8_C(237)),
      simde_x_mm256_set_epu8(UINT8_C(125), UINT8_C( 67), UINT8_C( 52), UINT8_C(  8),
                             UINT8_C(207), UINT8_C(205), UINT8_C( 56), UINT8_C(196),
                             UINT8_C( 42), UINT8_C(240), UINT8_C(116), UINT8_C( 44),
                             UINT8_C(109), UINT8_C(201), UINT8_C(220), UINT8_C(182),
                             UINT8_C(224), UINT8_C( 99), UINT8_C(151), UINT8_C(222),
                             UINT8_C(220), UINT8_C(252), UINT8_C(  6), UINT8_C(245),
                             UINT8_C(215), UINT8_C(221), UINT8_C(177), UINT8_C(240),
                             UINT8_C(118), UINT8_C(155), UINT8_C(143), UINT8_C(240)),
      simde_x_mm256_set_epu8(UINT8_C(119), UINT8_C( 67), UINT8_C( 13), UINT8_C(  8),
                             UINT8_C( 70), UINT8_C(197), UINT8_C( 56), UINT8_C( 89),
                             UINT8_C( 42), UINT8_C(159), UINT8_C( 50), UINT8_C( 44),
                             UINT8_C(109), UINT8_C( 62), UINT8_C(220), UINT8_C(182),
                             UINT8_C(224), UINT8_C( 99), UINT8_C( 53), UINT8_C(202),
                             UINT8_C(203), UINT8_C(241), UINT8_C(  6), UINT8_C(118),
                             UINT8_C(121), UINT8_C(140), UINT8_C(125), UINT8_C( 65),
                             UINT8_C( 70), UINT8_C(155), UINT8_C(143), UINT8_C(237)) },
    { simde_x_mm256_set_epu8(UINT8_C(201), UINT8_C(240), UINT8_C( 36), UINT8_C( 35),
                             UINT8_C(236), UINT8_C( 73), UINT8_C( 29), UINT8_C(244),
                             UINT8_C(140), UINT8_C( 36), UINT8_C(200), UINT8_C(155),
                             UINT8_C(  8), UINT8_C(245), UINT8_C( 10), UINT8_C( 93),
                             UINT8_C(124), UINT8_C(125), UINT8_C( 25), UINT8_C(192),
                             UINT8_C( 32), UINT8_C(119), UINT8_C(142), UINT8_C(147),
                             UINT8_C( 56), UINT8_C( 66), UINT8_C(172), UINT8_C(214),
                             UINT8_C(227), UINT8_C(203), UINT8_C( 62), UINT8_C( 37)),
      simde_x_mm256_set_epu8(UINT8_C(  5), UINT8_C( 64), UINT8_C(205), UINT8_C(118),
                             UINT8_C( 53), UINT8_C(  5), UINT8_C( 28), UINT8_C( 17),
                             UINT8_C( 93), UINT8_C(223), UINT8_C( 18), UINT8_C( 93),
                             UINT8_C(210), UINT8_C(158), UINT8_C( 37), UINT8_C( 66),
                             UINT8_C(184), UINT8_C(142), UINT8_C(246), UINT8_C( 15),
                             UINT8_C(153), UINT8_C(  9), UINT8_C(121), UINT8_C(211),
                             UINT8_C(  7), UINT8_C( 12), UINT8_C( 41), UINT8_C( 14),
                             UINT8_C(127), UINT8_C(208), UINT8_C(  7), UINT8_C( 93)),
      simde_x_mm256_set_epu8(UINT8_C(  5), UINT8_C( 64), UINT8_C( 36), UINT8_C( 35),
                             UINT8_C( 53), UINT8_C(  5), UINT8_C( 28), UINT8_C( 17),
                             UINT8_C( 93), UINT8_C( 36), UINT8_C( 18), UINT8_C( 93),
                             UINT8_C(  8), UINT8_C(158), UINT8_C( 10), UINT8_C( 66),
                             UINT8_C(124), UINT8_C(125), UINT8_C( 25), UINT8_C( 15),
                             UINT8_C( 32), UINT8_C(  9), UINT8_C(121), UINT8_C(147),
                             UINT8_C(  7), UINT8_C( 12), UINT8_C( 41), UINT8_C( 14),
                             UINT8_C(127), UINT8_C(203), UINT8_C(  7), UINT8_C( 37)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_min_epu8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_min_epu16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu16(UINT16_C( 21985), UINT16_C( 37408), UINT16_C( 64559), UINT16_C( 50506),
                              UINT16_C( 21348), UINT16_C( 50917), UINT16_C( 43331), UINT16_C(  4092),
                              UINT16_C( 58372), UINT16_C(  9106), UINT16_C( 30380), UINT16_C( 36427),
                              UINT16_C( 37095), UINT16_C( 38478), UINT16_C( 20272), UINT16_C( 48447)),
      simde_x_mm256_set_epu16(UINT16_C(  1564), UINT16_C( 15505), UINT16_C( 60508), UINT16_C( 59764),
                              UINT16_C(  4697), UINT16_C( 47817), UINT16_C(  4510), UINT16_C( 63199),
                              UINT16_C( 46692), UINT16_C( 14440), UINT16_C( 53716), UINT16_C(  9747),
                              UINT16_C( 59579), UINT16_C( 43113), UINT16_C( 42711), UINT16_C( 11767)),
      simde_x_mm256_set_epu16(UINT16_C(  1564), UINT16_C( 15505), UINT16_C( 60508), UINT16_C( 50506),
                              UINT16_C(  4697), UINT16_C( 47817), UINT16_C(  4510), UINT16_C(  4092),
                              UINT16_C( 46692), UINT16_C(  9106), UINT16_C( 30380), UINT16_C(  9747),
                              UINT16_C( 37095), UINT16_C( 38478), UINT16_C( 20272), UINT16_C( 11767)) },
    { simde_x_mm256_set_epu16(UINT16_C( 19955), UINT16_C(   258), UINT16_C(  7499), UINT16_C( 10868),
                              UINT16_C(  6000), UINT16_C( 24962), UINT16_C( 19695), UINT16_C( 32604),
                              UINT16_C( 25690), UINT16_C(  5375), UINT16_C(  7487), UINT16_C( 63875),
                              UINT16_C( 58580), UINT16_C( 15823), UINT16_C( 50945), UINT16_C( 25251)),
      simde_x_mm256_set_epu16(UINT16_C( 40574), UINT16_C( 18810), UINT16_C( 13888), UINT16_C( 49326),
                              UINT16_C( 64223), UINT16_C( 65297), UINT16_C( 61868), UINT16_C( 28086),
                              UINT16_C( 41020), UINT16_C( 35514), UINT16_C( 58890), UINT16_C(  3107),
                              UINT16_C(  7813), UINT16_C( 34771), UINT16_C( 34306), UINT16_C( 41125)),
      simde_x_mm256_set_epu16(UINT16_C( 19955), UINT16_C(   258), UINT16_C(  7499), UINT16_C( 10868),
                              UINT16_C(  6000), UINT16_C( 24962), UINT16_C( 19695), UINT16_C( 28086),
                              UINT16_C( 25690), UINT16_C(  5375), UINT16_C(  7487), UINT16_C(  3107),
                              UINT16_C(  7813), UINT16_C( 15823), UINT16_C( 34306), UINT16_C( 25251)) },
    { simde_x_mm256_set_epu16(UINT16_C( 54635), UINT16_C( 31649), UINT16_C( 37203), UINT16_C( 44269),
                              UINT16_C( 35808), UINT16_C( 48759), UINT16_C( 30288), UINT16_C( 48436),
                              UINT16_C( 40625), UINT16_C( 61846), UINT16_C( 14756), UINT16_C( 44804),
                              UINT16_C(  3949), UINT16_C( 42131), UINT16_C( 39125), UINT16_C( 37028)),
      simde_x_mm256_set_epu16(UINT16_C( 57949), UINT16_C( 27621), UINT16_C( 31226), UINT16_C( 53362),
                              UINT16_C( 27488), UINT16_C( 47169), UINT16_C( 31030), UINT16_C( 24207),
                              UINT16_C(  4423), UINT16_C( 14910), UINT16_C( 30727), UINT16_C( 21908),
                              UINT16_C( 50230), UINT16_C( 46262), UINT16_C( 25828), UINT16_C( 42233)),
      simde_x_mm256_set_epu16(UINT16_C( 54635), UINT16_C( 27621), UINT16_C( 31226), UINT16_C( 44269),
                              UINT16_C( 27488), UINT16_C( 47169), UINT16_C( 30288), UINT16_C( 24207),
                              UINT16_C(  4423), UINT16_C( 14910), UINT16_C( 14756), UINT16_C( 21908),
                              UINT16_C(  3949), UINT16_C( 42131), UINT16_C( 25828), UINT16_C( 37028)) },
    { simde_x_mm256_set_epu16(UINT16_C( 20152), UINT16_C( 27686), UINT16_C( 12466), UINT16_C( 27372),
                              UINT16_C( 44680), UINT16_C(  5041), UINT16_C( 65098), UINT16_C( 61815),
                              UINT16_C( 46798), UINT16_C( 41939), UINT16_C( 44487), UINT16_C( 43543),
                              UINT16_C( 48144), UINT16_C( 56243), UINT16_C( 61060), UINT16_C( 26152)),
      simde_x_mm256_set_epu16(UINT16_C(  5079), UINT16_C( 44741), UINT16_C( 26080), UINT16_C( 22525),
                              UINT16_C( 28605), UINT16_C(  1946), UINT16_C( 59845), UINT16_C( 20686),
                              UINT16_C( 33395), UINT16_C( 20058), UINT16_C( 31413), UINT16_C( 18280),
                              UINT16_C( 56644), UINT16_C( 26935), UINT16_C( 15289), UINT16_C( 20043)),
      simde_x_mm256_set_epu16(UINT16_C(  5079), UINT16_C( 27686), UINT16_C( 12466), UINT16_C( 22525),
                              UINT16_C( 28605), UINT16_C(  1946), UINT16_C( 59845), UINT16_C( 20686),
                              UINT16_C( 33395), UINT16_C( 20058), UINT16_C( 31413), UINT16_C( 18280),
                              UINT16_C( 48144), UINT16_C( 26935), UINT16_C( 15289), UINT16_C( 20043)) },
    { simde_x_mm256_set_epu16(UINT16_C( 50822), UINT16_C( 22037), UINT16_C( 40380), UINT16_C( 27848),
                              UINT16_C( 22453), UINT16_C( 65299), UINT16_C(   416), UINT16_C( 55389),
                              UINT16_C(  3972), UINT16_C( 20597), UINT16_C(  3476), UINT16_C( 46123),
                              UINT16_C( 29301), UINT16_C( 51108), UINT16_C( 35981), UINT16_C(  8905)),
      simde_x_mm256_set_epu16(UINT16_C( 25885), UINT16_C( 59271), UINT16_C( 16174), UINT16_C( 33039),
                              UINT16_C( 55355), UINT16_C( 49380), UINT16_C( 36750), UINT16_C( 16476),
                              UINT16_C( 22855), UINT16_C( 12806), UINT16_C(  3167), UINT16_C( 41606),
                              UINT16_C( 57407), UINT16_C( 61681), UINT16_C( 28474), UINT16_C( 18472)),
      simde_x_mm256_set_epu16(UINT16_C( 25885), UINT16_C( 22037), UINT16_C( 16174), UINT16_C( 27848),
                              UINT16_C( 22453), UINT16_C( 49380), UINT16_C(   416), UINT16_C( 16476),
                              UINT16_C(  3972), UINT16_C( 12806), UINT16_C(  3167), UINT16_C( 41606),
                              UINT16_C( 29301), UINT16_C( 51108), UINT16_C( 28474), UINT16_C(  8905)) },
    { simde_x_mm256_set_epu16(UINT16_C( 38411), UINT16_C( 38062), UINT16_C( 63184), UINT16_C(  5327),
                              UINT16_C(  2662), UINT16_C( 37879), UINT16_C( 38436), UINT16_C( 35926),
                              UINT16_C( 37505), UINT16_C( 16059), UINT16_C( 30023), UINT16_C( 13435),
                              UINT16_C( 55897), UINT16_C( 41272), UINT16_C( 35066), UINT16_C( 45732)),
      simde_x_mm256_set_epu16(UINT16_C(  4290), UINT16_C(  8244), UINT16_C( 17610), UINT16_C( 27365),
                              UINT16_C( 41082), UINT16_C( 32808), UINT16_C( 11681), UINT16_C( 25455),
                              UINT16_C(  5969), UINT16_C( 52658), UINT16_C( 58404), UINT16_C( 43100),
                              UINT16_C( 23433), UINT16_C( 56801), UINT16_C( 40070), UINT16_C( 43765)),
      simde_x_mm256_set_epu16(UINT16_C(  4290), UINT16_C(  8244), UINT16_C( 17610), UINT16_C(  5327),
                              UINT16_C(  2662), UINT16_C( 32808), UINT16_C( 11681), UINT16_C( 25455),
                              UINT16_C(  5969), UINT16_C( 16059), UINT16_C( 30023), UINT16_C( 13435),
                              UINT16_C( 23433), UINT16_C( 41272), UINT16_C( 35066), UINT16_C( 43765)) },
    { simde_x_mm256_set_epu16(UINT16_C(  3920), UINT16_C(  2672), UINT16_C(   125), UINT16_C( 16695),
                              UINT16_C(  9026), UINT16_C(  9313), UINT16_C( 36585), UINT16_C( 38441),
                              UINT16_C( 50905), UINT16_C( 11221), UINT16_C( 28122), UINT16_C( 18875),
                              UINT16_C( 30095), UINT16_C( 22178), UINT16_C( 51689), UINT16_C( 19916)),
      simde_x_mm256_set_epu16(UINT16_C( 58371), UINT16_C( 46494), UINT16_C( 62748), UINT16_C( 27388),
                              UINT16_C( 64543), UINT16_C( 23688), UINT16_C( 42332), UINT16_C( 43841),
                              UINT16_C( 27632), UINT16_C( 34295), UINT16_C( 21862), UINT16_C( 32789),
                              UINT16_C( 42560), UINT16_C( 49564), UINT16_C( 10833), UINT16_C( 33089)),
      simde_x_mm256_set_epu16(UINT16_C(  3920), UINT16_C(  2672), UINT16_C(   125), UINT16_C( 16695),
                              UINT16_C(  9026), UINT16_C(  9313), UINT16_C( 36585), UINT16_C( 38441),
                              UINT16_C( 27632), UINT16_C( 11221), UINT16_C( 21862), UINT16_C( 18875),
                              UINT16_C( 30095), UINT16_C( 22178), UINT16_C( 10833), UINT16_C( 19916)) },
    { simde_x_mm256_set_epu16(UINT16_C( 32155), UINT16_C( 62138), UINT16_C( 45089), UINT16_C( 64592),
                              UINT16_C( 57638), UINT16_C( 36363), UINT16_C( 16857), UINT16_C( 65526),
                              UINT16_C(  4908), UINT16_C( 33576), UINT16_C( 61433), UINT16_C( 64967),
                              UINT16_C( 30812), UINT16_C( 35007), UINT16_C(  3114), UINT16_C( 45540)),
      simde_x_mm256_set_epu16(UINT16_C( 46521), UINT16_C(  5351), UINT16_C( 22645), UINT16_C( 25668),
                              UINT16_C( 14295), UINT16_C( 52574), UINT16_C(  9596), UINT16_C( 37163),
                              UINT16_C( 27912), UINT16_C( 36566), UINT16_C( 34315), UINT16_C(  2157),
                              UINT16_C( 51210), UINT16_C( 14248), UINT16_C( 30537), UINT16_C( 57705)),
      simde_x_mm256_set_epu16(UINT16_C( 32155), UINT16_C(  5351), UINT16_C( 22645), UINT16_C( 25668),
                              UINT16_C( 14295), UINT16_C( 36363), UINT16_C(  9596), UINT16_C( 37163),
                              UINT16_C(  4908), UINT16_C( 33576), UINT16_C( 34315), UINT16_C(  2157),
                              UINT16_C( 30812), UINT16_C( 14248), UINT16_C(  3114), UINT16_C( 45540)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_min_epu16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_min_epu32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu32(UINT32_C(1967336836), UINT32_C(4229175532), UINT32_C(3383228965), UINT32_C( 979444700),
                              UINT32_C( 138906106), UINT32_C( 146006389), UINT32_C(2494756173), UINT32_C(2105904126)),
      simde_x_mm256_set_epu32(UINT32_C(2893387611), UINT32_C(2460478173), UINT32_C(3299027518), UINT32_C( 896184310),
                              UINT32_C(1699227452), UINT32_C(1922730015), UINT32_C(1220777705), UINT32_C( 483108197)),
      simde_x_mm256_set_epu32(UINT32_C(1967336836), UINT32_C(2460478173), UINT32_C(3299027518), UINT32_C( 896184310),
                              UINT32_C( 138906106), UINT32_C( 146006389), UINT32_C(1220777705), UINT32_C( 483108197)) },
    { simde_x_mm256_set_epu32(UINT32_C( 596030925), UINT32_C(1623933296), UINT32_C(4116080130), UINT32_C(2140384796),
                              UINT32_C(1701047302), UINT32_C( 386212074), UINT32_C( 328829151), UINT32_C(3670208831)),
      simde_x_mm256_set_epu32(UINT32_C(3740465827), UINT32_C(1627250042), UINT32_C(2355376593), UINT32_C(1538421544),
                              UINT32_C( 533181547), UINT32_C(1474624420), UINT32_C( 464742120), UINT32_C(1469792713)),
      simde_x_mm256_set_epu32(UINT32_C( 596030925), UINT32_C(1623933296), UINT32_C(2355376593), UINT32_C(1538421544),
                              UINT32_C( 533181547), UINT32_C( 386212074), UINT32_C( 328829151), UINT32_C(1469792713)) },
    { simde_x_mm256_set_epu32(UINT32_C(3969449230), UINT32_C(1656808571), UINT32_C(3512723294), UINT32_C(1894827634),
                              UINT32_C( 143242580), UINT32_C(1411474427), UINT32_C(1474375050), UINT32_C(2576321811)),
      simde_x_mm256_set_epu32(UINT32_C(3817774721), UINT32_C(1995677222), UINT32_C(2801037071), UINT32_C(1160265207),
                              UINT32_C( 536799050), UINT32_C(3040213718), UINT32_C(2960203135), UINT32_C( 387289056)),
      simde_x_mm256_set_epu32(UINT32_C(3817774721), UINT32_C(1656808571), UINT32_C(2801037071), UINT32_C(1160265207),
                              UINT32_C( 143242580), UINT32_C(1411474427), UINT32_C(1474375050), UINT32_C( 387289056)) },
    { simde_x_mm256_set_epu32(UINT32_C(4117731720), UINT32_C(2896203570), UINT32_C(2130869721), UINT32_C( 624464130),
                              UINT32_C(3920434556), UINT32_C(3882005287), UINT32_C(2912704980), UINT32_C(3353740323)),
      simde_x_mm256_set_epu32(UINT32_C(1011878374), UINT32_C( 876977997), UINT32_C( 633144937), UINT32_C(2591224872),
                              UINT32_C(1689064732), UINT32_C(1345088039), UINT32_C(3928457299), UINT32_C(1317789172)),
      simde_x_mm256_set_epu32(UINT32_C(1011878374), UINT32_C( 876977997), UINT32_C( 633144937), UINT32_C( 624464130),
                              UINT32_C(1689064732), UINT32_C(1345088039), UINT32_C(2912704980), UINT32_C(1317789172)) },
    { simde_x_mm256_set_epu32(UINT32_C(1278890315), UINT32_C(3068059236), UINT32_C( 937423722), UINT32_C( 545836753),
                              UINT32_C(3944086739), UINT32_C(2155957693), UINT32_C( 750306742), UINT32_C( 995938818)),
      simde_x_mm256_set_epu32(UINT32_C(2300290567), UINT32_C(3884383026), UINT32_C( 682756216), UINT32_C( 434660596),
                              UINT32_C(3895444851), UINT32_C(3229085704), UINT32_C(3028266335), UINT32_C(2490258842)),
      simde_x_mm256_set_epu32(UINT32_C(1278890315), UINT32_C(3068059236), UINT32_C( 682756216), UINT32_C( 434660596),
                              UINT32_C(3895444851), UINT32_C(2155957693), UINT32_C( 750306742), UINT32_C( 995938818)) },
    { simde_x_mm256_set_epu32(UINT32_C( 953347239), UINT32_C(3233201384), UINT32_C( 883460426), UINT32_C( 630153716),
                              UINT32_C(2626834474), UINT32_C(4260188706), UINT32_C(4276291548), UINT32_C(1697478493)),
      simde_x_mm256_set_epu32(UINT32_C(3841797977), UINT32_C( 519016629), UINT32_C(  54139722), UINT32_C(4160085404),
                              UINT32_C(2354740665), UINT32_C( 224670449), UINT32_C(2606748626), UINT32_C(2604287898)),
      simde_x_mm256_set_epu32(UINT32_C( 953347239), UINT32_C( 519016629), UINT32_C(  54139722), UINT32_C( 630153716),
                              UINT32_C(2354740665), UINT32_C( 224670449), UINT32_C(2606748626), UINT32_C(1697478493)) },
    { simde_x_mm256_set_epu32(UINT32_C(4276795094), UINT32_C(3517498069), UINT32_C(3286132221), UINT32_C(1640896057),
                              UINT32_C(1497672480), UINT32_C(  40644986), UINT32_C(1824934232), UINT32_C(1194285849)),
      simde_x_mm256_set_epu32(UINT32_C(1850977199), UINT32_C(1916865152), UINT32_C(2772610612), UINT32_C(2574813520),
                              UINT32_C(1514631464), UINT32_C(2960447777), UINT32_C(2276426609), UINT32_C(1419491712)),
      simde_x_mm256_set_epu32(UINT32_C(1850977199), UINT32_C(1916865152), UINT32_C(2772610612), UINT32_C(1640896057),
                              UINT32_C(1497672480), UINT32_C(  40644986), UINT32_C(1824934232), UINT32_C(1194285849)) },
    { simde_x_mm256_set_epu32(UINT32_C( 990577222), UINT32_C(1025245358), UINT32_C(2406551265), UINT32_C(4071927667),
                              UINT32_C(3189139328), UINT32_C(1739830541), UINT32_C(1739044254), UINT32_C( 584922997)),
      simde_x_mm256_set_epu32(UINT32_C(  72076484), UINT32_C(2950382985), UINT32_C( 915753842), UINT32_C( 355749909),
                              UINT32_C(3000181234), UINT32_C(3918623971), UINT32_C(3711879869), UINT32_C(3103913192)),
      simde_x_mm256_set_epu32(UINT32_C(  72076484), UINT32_C(1025245358), UINT32_C( 915753842), UINT32_C( 355749909),
                              UINT32_C(3000181234), UINT32_C(1739830541), UINT32_C(1739044254), UINT32_C( 584922997)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_min_epu32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_movemask_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    int32_t r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C( -96), INT8_C(-118), INT8_C(  98), INT8_C(  62),
                           INT8_C( -51), INT8_C(  97), INT8_C( -33), INT8_C( 125),
                           INT8_C(-127), INT8_C( -72), INT8_C(  -2), INT8_C(  75),
                           INT8_C(   7), INT8_C(  73), INT8_C( -28), INT8_C(  60),
                           INT8_C(   8), INT8_C( -37), INT8_C(-119), INT8_C(  83),
                           INT8_C( -63), INT8_C( 105), INT8_C(-120), INT8_C( -81),
                           INT8_C(  66), INT8_C( -93), INT8_C(  75), INT8_C( -69),
                           INT8_C(  47), INT8_C(  11), INT8_C(  51), INT8_C(  35)),
      -891131056 },
    { simde_mm256_set_epi8(INT8_C(  15), INT8_C( 104), INT8_C( -65), INT8_C(-125),
                           INT8_C(  29), INT8_C( 110), INT8_C( -50), INT8_C(  21),
                           INT8_C( -48), INT8_C( 105), INT8_C(  56), INT8_C( 122),
                           INT8_C( -60), INT8_C( 127), INT8_C(  65), INT8_C(-126),
                           INT8_C(  -5), INT8_C( -40), INT8_C( -84), INT8_C( -80),
                           INT8_C(  27), INT8_C(  14), INT8_C(  89), INT8_C(  45),
                           INT8_C(-125), INT8_C( -33), INT8_C( 119), INT8_C(  -9),
                           INT8_C(  20), INT8_C(-117), INT8_C( -34), INT8_C( -66)),
      847900887 },
    { simde_mm256_set_epi8(INT8_C( 106), INT8_C(  46), INT8_C( 114), INT8_C( -45),
                           INT8_C(  75), INT8_C(  29), INT8_C( -66), INT8_C(-117),
                           INT8_C(  47), INT8_C(  53), INT8_C(  50), INT8_C(  31),
                           INT8_C(-111), INT8_C(  36), INT8_C( -73), INT8_C(  38),
                           INT8_C( -23), INT8_C( 112), INT8_C( -88), INT8_C(  42),
                           INT8_C( -89), INT8_C( 120), INT8_C(  50), INT8_C(  27),
                           INT8_C(   6), INT8_C(   1), INT8_C( 127), INT8_C( 127),
                           INT8_C(  38), INT8_C(  57), INT8_C(  13), INT8_C( -14)),
      319465473 },
    { simde_mm256_set_epi8(INT8_C( -21), INT8_C( 113), INT8_C( 127), INT8_C( -53),
                           INT8_C( 111), INT8_C( 121), INT8_C( -27), INT8_C(  17),
                           INT8_C(-104), INT8_C(  11), INT8_C( -41), INT8_C( -39),
                           INT8_C(  51), INT8_C(  41), INT8_C(  91), INT8_C( -62),
                           INT8_C(-116), INT8_C(  34), INT8_C(  15), INT8_C( -55),
                           INT8_C( -90), INT8_C( -31), INT8_C( -66), INT8_C( -64),
                           INT8_C( 115), INT8_C( -38), INT8_C( -54), INT8_C(  24),
                           INT8_C( -59), INT8_C( -48), INT8_C(  15), INT8_C(  60)),
      -1833853076 },
    { simde_mm256_set_epi8(INT8_C( -84), INT8_C(  66), INT8_C( 126), INT8_C( -52),
                           INT8_C(  88), INT8_C(  79), INT8_C(  71), INT8_C( -11),
                           INT8_C( -43), INT8_C( -40), INT8_C(-120), INT8_C(  75),
                           INT8_C(  12), INT8_C( -40), INT8_C(  86), INT8_C( 111),
                           INT8_C( 107), INT8_C( -40), INT8_C( -47), INT8_C(  90),
                           INT8_C(  21), INT8_C( 126), INT8_C( -72), INT8_C( -52),
                           INT8_C( -36), INT8_C( -82), INT8_C( -69), INT8_C(  97),
                           INT8_C(-122), INT8_C( -39), INT8_C(  59), INT8_C(  25)),
      -1847303188 },
    { simde_mm256_set_epi8(INT8_C(  67), INT8_C(  64), INT8_C(  17), INT8_C(  -4),
                           INT8_C( -84), INT8_C(  57), INT8_C(  94), INT8_C(  94),
                           INT8_C(-112), INT8_C(  59), INT8_C( -47), INT8_C( -43),
                           INT8_C( -74), INT8_C(  39), INT8_C(  45), INT8_C( -64),
                           INT8_C( -47), INT8_C( 114), INT8_C( -10), INT8_C(  33),
                           INT8_C(  47), INT8_C( -82), INT8_C( -45), INT8_C(  28),
                           INT8_C(  16), INT8_C(  34), INT8_C(  94), INT8_C(  53),
                           INT8_C(  64), INT8_C(-113), INT8_C( -53), INT8_C(  74)),
      414819846 },
    { simde_mm256_set_epi8(INT8_C(  27), INT8_C(  -3), INT8_C(  33), INT8_C( -42),
                           INT8_C( 113), INT8_C( -79), INT8_C( 119), INT8_C(  38),
                           INT8_C(  96), INT8_C( 109), INT8_C( 125), INT8_C(  82),
                           INT8_C(   8), INT8_C( -29), INT8_C(  10), INT8_C( -22),
                           INT8_C( -49), INT8_C( 123), INT8_C( 109), INT8_C( -49),
                           INT8_C(   6), INT8_C( -16), INT8_C( -14), INT8_C( 102),
                           INT8_C(  -5), INT8_C(  88), INT8_C(  66), INT8_C( -63),
                           INT8_C(  82), INT8_C(  34), INT8_C(  44), INT8_C(  56)),
      1409652368 },
    { simde_mm256_set_epi8(INT8_C( -69), INT8_C( -65), INT8_C(  16), INT8_C( 111),
                           INT8_C( 123), INT8_C(  89), INT8_C(  77), INT8_C(   3),
                           INT8_C(  37), INT8_C( -13), INT8_C(  28), INT8_C(  56),
                           INT8_C( -40), INT8_C( -18), INT8_C( -12), INT8_C(  32),
                           INT8_C( -91), INT8_C( -40), INT8_C( 109), INT8_C(  79),
                           INT8_C(  14), INT8_C(  52), INT8_C(  95), INT8_C(  73),
                           INT8_C(  62), INT8_C( -36), INT8_C( -31), INT8_C(  24),
                           INT8_C(  60), INT8_C( -72), INT8_C(   1), INT8_C( -18)),
      -1068580763 },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    int32_t r = simde_mm256_movemask_epi8(test_vec[i].a);
    simde_assert_equal_i32(r, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_mpsadbw_epu8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint8_t a[32];
    const uint8_t b[32];
    const uint16_t r[16];
  } test_vec[] = {
    { { UINT8_C(116), UINT8_C(168), UINT8_C(162), UINT8_C(173), UINT8_C(122), UINT8_C(179), UINT8_C(242), UINT8_C( 70),
        UINT8_C(251), UINT8_C(139), UINT8_C(136), UINT8_C(248), UINT8_C(103), UINT8_C(122), UINT8_C( 83), UINT8_C(207),
        UINT8_C(164), UINT8_C( 64), UINT8_C(139), UINT8_C(142),    UINT8_MAX, UINT8_C( 80), UINT8_C(177), UINT8_C(166),
        UINT8_C( 81), UINT8_C(193), UINT8_C(178), UINT8_C(140), UINT8_C( 66), UINT8_C(185), UINT8_C( 86), UINT8_C(182) },
      { UINT8_C( 98), UINT8_C(248), UINT8_C( 99), UINT8_C(220), UINT8_C(171), UINT8_C( 85), UINT8_C( 34), UINT8_C(166),
        UINT8_C(225), UINT8_C(170), UINT8_C(158), UINT8_C( 72), UINT8_C( 36), UINT8_C(242), UINT8_C( 23), UINT8_C(201),
        UINT8_C( 50), UINT8_C(162), UINT8_C( 87), UINT8_C( 49), UINT8_C(242), UINT8_C(  8), UINT8_C(215), UINT8_C( 68),
        UINT8_C(201), UINT8_C(137), UINT8_C(208), UINT8_C( 12), UINT8_C( 67), UINT8_C( 38), UINT8_C(194), UINT8_C(165) },
      { UINT16_C(  499), UINT16_C(  240), UINT16_C(  668), UINT16_C(  224), UINT16_C(  478), UINT16_C(  532), UINT16_C(  265), UINT16_C(  568),
        UINT16_C(  357), UINT16_C(  298), UINT16_C(  308), UINT16_C(  320), UINT16_C(  494), UINT16_C(  156), UINT16_C(  281), UINT16_C(  432) } },
    { { UINT8_C( 30), UINT8_C( 38), UINT8_C(129), UINT8_C(202), UINT8_C(123), UINT8_C(163), UINT8_C(112), UINT8_C( 92),
        UINT8_C( 77), UINT8_C( 15), UINT8_C(164), UINT8_C(114), UINT8_C(  1), UINT8_C(187), UINT8_C( 59), UINT8_C( 51),
        UINT8_C( 94), UINT8_C(146), UINT8_C(100), UINT8_C( 80), UINT8_C(155), UINT8_C( 60), UINT8_C(148), UINT8_C(100),
        UINT8_C(197), UINT8_C(100), UINT8_C(112), UINT8_C(  8), UINT8_C(138), UINT8_C( 51), UINT8_C(173), UINT8_C(169) },
      { UINT8_C( 89), UINT8_C( 46), UINT8_C(115), UINT8_C(212), UINT8_C(209), UINT8_C(227), UINT8_C( 49), UINT8_C( 31),
        UINT8_C(242), UINT8_C(213), UINT8_C(145), UINT8_C(243), UINT8_C(145), UINT8_C(204), UINT8_C( 38), UINT8_C(239),
        UINT8_C( 94), UINT8_C(139), UINT8_C( 63), UINT8_C(249), UINT8_C(199), UINT8_C(212), UINT8_C( 94), UINT8_C(140),
        UINT8_C( 56), UINT8_C(206), UINT8_C(149), UINT8_C(195), UINT8_C(  1), UINT8_C( 66), UINT8_C(108), UINT8_C( 90) },
      { UINT16_C(  284), UINT16_C(  326), UINT16_C(  408), UINT16_C(  278), UINT16_C(  508), UINT16_C(  484), UINT16_C(  198), UINT16_C(  563),
        UINT16_C(  213), UINT16_C(  202), UINT16_C(  346), UINT16_C(  134), UINT16_C(  374), UINT16_C(  132), UINT16_C(  376), UINT16_C(  238) } },
    { { UINT8_C(113), UINT8_C(223), UINT8_C( 47), UINT8_C( 66), UINT8_C(194), UINT8_C( 96), UINT8_C( 97), UINT8_C(181),
        UINT8_C( 53), UINT8_C(242), UINT8_C(168), UINT8_C(198), UINT8_C(190), UINT8_C(207), UINT8_C(181), UINT8_C( 29),
        UINT8_C( 90), UINT8_C(245), UINT8_C( 22), UINT8_C( 33), UINT8_C(201), UINT8_C(116), UINT8_C(173), UINT8_C(  1),
        UINT8_C( 67), UINT8_C( 66), UINT8_C(196), UINT8_C( 68), UINT8_C(133), UINT8_C( 48), UINT8_C(159), UINT8_C(246) },
      { UINT8_C( 15), UINT8_C(206), UINT8_C( 56), UINT8_C(210), UINT8_C( 46), UINT8_C(154), UINT8_C(135), UINT8_C( 99),
        UINT8_C(140), UINT8_C( 47), UINT8_C( 42), UINT8_C( 75), UINT8_C(254), UINT8_C(223), UINT8_C(104), UINT8_C( 88),
        UINT8_C(212), UINT8_C(126), UINT8_C(121), UINT8_C(157), UINT8_C(243), UINT8_C( 39), UINT8_C(159), UINT8_C( 54),
        UINT8_C(105), UINT8_C( 99), UINT8_C(122), UINT8_C(238), UINT8_C(148), UINT8_C( 25), UINT8_C(228), UINT8_C(163) },
      { UINT16_C(  287), UINT16_C(  396), UINT16_C(  404), UINT16_C(  461), UINT16_C(  394), UINT16_C(  263), UINT16_C(  316), UINT16_C(  285),
        UINT16_C(  464), UINT16_C(  269), UINT16_C(  404), UINT16_C(  275), UINT16_C(  229), UINT16_C(  353), UINT16_C(  309), UINT16_C(  364) } },
    { { UINT8_C(231), UINT8_C( 29), UINT8_C(117), UINT8_C( 21), UINT8_C(183), UINT8_C(252), UINT8_C(121), UINT8_C( 67),
        UINT8_C( 44), UINT8_C(163), UINT8_C(142), UINT8_C( 42), UINT8_C(130), UINT8_C(246), UINT8_C(131), UINT8_C( 87),
        UINT8_C(117), UINT8_C(252), UINT8_C(244), UINT8_C(104), UINT8_C( 35), UINT8_C(147), UINT8_C(158), UINT8_C(141),
        UINT8_C(247), UINT8_C( 24), UINT8_C(123), UINT8_C(139), UINT8_C( 50), UINT8_C( 96), UINT8_C( 46), UINT8_C( 25) },
      { UINT8_C(125), UINT8_C(164), UINT8_C( 47), UINT8_C( 52), UINT8_C(160), UINT8_C(168), UINT8_C(119), UINT8_C(204),
        UINT8_C( 75), UINT8_C(  6), UINT8_C(247), UINT8_C(205), UINT8_C(252), UINT8_C(122), UINT8_C( 36), UINT8_C(113),
        UINT8_C(118), UINT8_C( 25), UINT8_C(217), UINT8_C(154), UINT8_C(172), UINT8_C(119), UINT8_C( 39), UINT8_C(163),
        UINT8_C(144), UINT8_C(162), UINT8_C( 46), UINT8_C(194), UINT8_C(  2), UINT8_C( 93), UINT8_C(219), UINT8_C(127) },
      { UINT16_C(  330), UINT16_C(  101), UINT16_C(  244), UINT16_C(  419), UINT16_C(  426), UINT16_C(  132), UINT16_C(  417), UINT16_C(  446),
        UINT16_C(  305), UINT16_C(  585), UINT16_C(  394), UINT16_C(   98), UINT16_C(  277), UINT16_C(  331), UINT16_C(  316), UINT16_C(  469) } },
    { { UINT8_C(  1), UINT8_C( 10), UINT8_C(179), UINT8_C(161), UINT8_C(178), UINT8_C( 43), UINT8_C(110), UINT8_C(253),
        UINT8_C( 49), UINT8_C(101), UINT8_C(203), UINT8_C( 45), UINT8_C(223), UINT8_C(239), UINT8_C(159), UINT8_C( 85),
        UINT8_C(  8), UINT8_C(120), UINT8_C(239), UINT8_C(181), UINT8_C(240), UINT8_C( 22), UINT8_C( 88), UINT8_C(128),
        UINT8_C(185), UINT8_C(135), UINT8_C( 66), UINT8_C(187), UINT8_C(228), UINT8_C( 29), UINT8_C( 59), UINT8_C(229) },
      { UINT8_C( 40), UINT8_C(238), UINT8_C(134), UINT8_C(218), UINT8_C( 25), UINT8_C(244), UINT8_C(216), UINT8_C( 74),
        UINT8_C( 89), UINT8_C(163), UINT8_C(120), UINT8_C( 56), UINT8_C(146), UINT8_C( 23), UINT8_C(142), UINT8_C(155),
        UINT8_C(143), UINT8_C(125), UINT8_C( 80), UINT8_C(127), UINT8_C(148), UINT8_C(168),    UINT8_MAX, UINT8_C( 77),
        UINT8_C( 47), UINT8_C( 65), UINT8_C(  8), UINT8_C( 19), UINT8_C( 95), UINT8_C( 67), UINT8_C(248), UINT8_C(135) },
      { UINT16_C(  182), UINT16_C(  407), UINT16_C(  413), UINT16_C(  222), UINT16_C(  346), UINT16_C(  390), UINT16_C(  244), UINT16_C(  402),
        UINT16_C(  353), UINT16_C(  351), UINT16_C(  417), UINT16_C(  250), UINT16_C(  209), UINT16_C(  264), UINT16_C(  171), UINT16_C(  191) } },
    { { UINT8_C( 50), UINT8_C(127), UINT8_C( 97), UINT8_C( 75), UINT8_C(115), UINT8_C( 57), UINT8_C(150), UINT8_C(205),
        UINT8_C(220), UINT8_C( 14), UINT8_C(  5), UINT8_C(111), UINT8_C( 37), UINT8_C(147), UINT8_C( 10), UINT8_C(180),
        UINT8_C( 17), UINT8_C( 90), UINT8_C( 52), UINT8_C(165), UINT8_C(  2), UINT8_C( 51), UINT8_C(242), UINT8_C( 50),
        UINT8_C(117), UINT8_C(250), UINT8_C( 69), UINT8_C(212), UINT8_C( 62), UINT8_C( 62), UINT8_C( 91), UINT8_C(112) },
      { UINT8_C(189), UINT8_C(188), UINT8_C(187), UINT8_C( 48), UINT8_C(246), UINT8_C( 81), UINT8_C(253), UINT8_C(210),
        UINT8_C( 95), UINT8_C(  3), UINT8_C( 65), UINT8_C(132), UINT8_C(150), UINT8_C( 75), UINT8_C( 57), UINT8_C(167),
        UINT8_C(165), UINT8_C(109), UINT8_C( 76), UINT8_C(168), UINT8_C(160), UINT8_C( 62), UINT8_C(218), UINT8_C( 21),
        UINT8_C( 57), UINT8_C( 31), UINT8_C(233), UINT8_C(119), UINT8_C( 93), UINT8_C( 68), UINT8_C(231), UINT8_C( 26) },
      { UINT16_C(  184), UINT16_C(  369), UINT16_C(  446), UINT16_C(  405), UINT16_C(  239), UINT16_C(  390), UINT16_C(  221), UINT16_C(  324),
        UINT16_C(  194), UINT16_C(  387), UINT16_C(  360), UINT16_C(  206), UINT16_C(  505), UINT16_C(  324), UINT16_C(  259), UINT16_C(  396) } },
    { { UINT8_C(  1), UINT8_C(162), UINT8_C( 75), UINT8_C(247), UINT8_C(244), UINT8_C( 72), UINT8_C(201), UINT8_C( 83),
        UINT8_C( 75), UINT8_C( 11), UINT8_C(216), UINT8_C(226), UINT8_C( 86), UINT8_C( 17), UINT8_C(137), UINT8_C(252),
        UINT8_C(126), UINT8_C(214), UINT8_C(164), UINT8_C( 30), UINT8_C( 20), UINT8_C(126), UINT8_C( 52), UINT8_C( 77),
        UINT8_C(157), UINT8_C( 29), UINT8_C(196), UINT8_C(251), UINT8_C( 98), UINT8_C(171), UINT8_C( 21), UINT8_C( 99) },
      { UINT8_C( 78), UINT8_C( 96), UINT8_C( 90), UINT8_C( 66), UINT8_C(169), UINT8_C( 35), UINT8_C(149), UINT8_C(244),
        UINT8_C( 46), UINT8_C(109), UINT8_C(214), UINT8_C(133), UINT8_C(126), UINT8_C( 96), UINT8_C(129), UINT8_C(252),
        UINT8_C( 54), UINT8_C( 37), UINT8_C( 27), UINT8_C( 74), UINT8_C(163), UINT8_C( 79), UINT8_C(152), UINT8_C( 64),
        UINT8_C(108), UINT8_C( 92), UINT8_C( 59), UINT8_C(206), UINT8_C(  8), UINT8_C( 81), UINT8_C( 49), UINT8_C( 86) },
      { UINT16_C(  383), UINT16_C(  382), UINT16_C(  383), UINT16_C(  218), UINT16_C(  249), UINT16_C(  498), UINT16_C(  498), UINT16_C(  337),
        UINT16_C(  430), UINT16_C(  344), UINT16_C(  176), UINT16_C(  162), UINT16_C(  151), UINT16_C(  220), UINT16_C(  217), UINT16_C(  267) } },
    { { UINT8_C(177), UINT8_C(139), UINT8_C(152), UINT8_C( 90), UINT8_C(175), UINT8_C( 45), UINT8_C( 79), UINT8_C(221),
        UINT8_C(155), UINT8_C( 37), UINT8_C( 98), UINT8_C( 25), UINT8_C(133), UINT8_C(227), UINT8_C( 22), UINT8_C(187),
        UINT8_C(  8), UINT8_C( 49), UINT8_C(  6), UINT8_C(171), UINT8_C(128), UINT8_C(158), UINT8_C(236), UINT8_C(236),
        UINT8_C(250), UINT8_C( 39), UINT8_C(187), UINT8_C(  2), UINT8_C(120), UINT8_C(236), UINT8_C( 88), UINT8_C( 42) },
      { UINT8_C(120), UINT8_C(240), UINT8_C(132), UINT8_C( 39), UINT8_C( 30), UINT8_C(211), UINT8_C(  4), UINT8_C(185),
        UINT8_C(249), UINT8_C(103), UINT8_C(210), UINT8_C(126), UINT8_C( 74), UINT8_C(232), UINT8_C( 58), UINT8_C( 83),
        UINT8_C( 25), UINT8_C( 64), UINT8_C(254), UINT8_C(153), UINT8_C(222), UINT8_C(234), UINT8_C(134), UINT8_C(216),
        UINT8_C( 18), UINT8_C( 65), UINT8_C(219), UINT8_C(138), UINT8_C( 45), UINT8_C( 51), UINT8_C(180), UINT8_C(165) },
      { UINT16_C(  447), UINT16_C(  417), UINT16_C(  159), UINT16_C(  260), UINT16_C(  374), UINT16_C(  254), UINT16_C(  450), UINT16_C(  378),
        UINT16_C(  298), UINT16_C(  190), UINT16_C(  257), UINT16_C(  389), UINT16_C(  298), UINT16_C(  420), UINT16_C(  501), UINT16_C(  646) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi8(test_vec[i].b);
    simde__m256i r = simde_mm256_mpsadbw_epu8(a, b, 7);
    simde_test_x86_assert_equal_u16x16(r, simde_x_mm256_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mul_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[8];
    const int32_t b[8];
    const int64_t r[4];
  } test_vec[] = {
    { { -INT32_C(  1249839109),  INT32_C(   383478711), -INT32_C(   552642143),  INT32_C(   566134082), -INT32_C(  1345465419), -INT32_C(  1119937342),  INT32_C(  1351952468),  INT32_C(    75355400) },
      { -INT32_C(  2068185395),  INT32_C(   177902953),  INT32_C(   820620014), -INT32_C(   464410577),  INT32_C(  1033117563), -INT32_C(  1879387333),  INT32_C(    81825788),  INT32_C(   839408997) },
      {  INT64_C( 2584898991333613055), -INT64_C(  453509203125650002), -INT64_C( 1390023954778053897),  INT64_C(  110624576032644784) } },
    { { -INT32_C(   994655653),  INT32_C(  1171214679),  INT32_C(   712358139), -INT32_C(   602945952),  INT32_C(   555328485),  INT32_C(  1924142198),  INT32_C(   142053539),  INT32_C(  1228570606) },
      { -INT32_C(  1743851199),  INT32_C(  1037950274), -INT32_C(   161000811), -INT32_C(    19761639), -INT32_C(  1876956391), -INT32_C(  1560096513),  INT32_C(  1319860576),  INT32_C(   983099129) },
      {  INT64_C( 1734531453076177947), -INT64_C(  114690238101450729), -INT64_C( 1042327349025097635),  INT64_C(  187490865807378464) } },
    { {  INT32_C(   433301207),  INT32_C(   408334467),  INT32_C(   470727939),  INT32_C(  1293606964), -INT32_C(   874693940),  INT32_C(  1785651210),  INT32_C(  1387862617), -INT32_C(   678604544) },
      {  INT32_C(  2062573815),  INT32_C(   328353296),  INT32_C(   925868291),  INT32_C(  1300580993), -INT32_C(  1910939004), -INT32_C(  1661434046), -INT32_C(  1578192479), -INT32_C(   109544702) },
      {  INT64_C(  893715723566094705),  INT64_C(  435832072407882249),  INT64_C( 1671486766508435760), -INT64_C( 2190314344034657543) } },
    { { -INT32_C(   327980837), -INT32_C(  1291909458),  INT32_C(   686370727), -INT32_C(    42570119),  INT32_C(   327913425), -INT32_C(  1196457194),  INT32_C(   928620085), -INT32_C(   181284327) },
      { -INT32_C(   371088326),  INT32_C(  1369170090), -INT32_C(  2005236721), -INT32_C(   997854989), -INT32_C(  1781067649), -INT32_C(   917665900),  INT32_C(  1040230181), -INT32_C(  1288490631) },
      {  INT64_C(  121709859762408862), -INT64_C( 1376335785999866167), -INT64_C(  584035992940287825),  INT64_C(  965978639099785385) } },
    { { -INT32_C(  2137254698),  INT32_C(    80885749), -INT32_C(  1349694276), -INT32_C(  1150086596), -INT32_C(  1219474909), -INT32_C(   159342895), -INT32_C(  1103790011), -INT32_C(  2005833550) },
      {  INT32_C(  1913130621),  INT32_C(    24566341),  INT32_C(  1655702310),  INT32_C(   958210838),  INT32_C(  1072721518),  INT32_C(  1362522124), -INT32_C(  1576047632),  INT32_C(  1344962771) },
      { -INT64_C( 4088847407619907458), -INT64_C( 2234691930566977560), -INT64_C( 1308156975545391862),  INT64_C( 1739625633061803952) } },
    { { -INT32_C(   725470322),  INT32_C(   886389005),  INT32_C(  1385596476),  INT32_C(   395031721),  INT32_C(   777485090), -INT32_C(   595620373), -INT32_C(   880898312), -INT32_C(  1659066098) },
      { -INT32_C(   378413348),  INT32_C(  1411204631),  INT32_C(  1990636748), -INT32_C(  1970458008), -INT32_C(  1715936083),  INT32_C(  1769289585), -INT32_C(   751438907),  INT32_C(  2020626844) },
      {  INT64_C(  274527653422658056),  INT64_C( 2758219263024900048), -INT64_C( 1334114719925502470),  INT64_C(  661941264747424984) } },
    { {  INT32_C(  1197662511), -INT32_C(   191135960), -INT32_C(  1687535309),  INT32_C(   556136563),  INT32_C(  1320869340), -INT32_C(   642306284), -INT32_C(  1079120862),  INT32_C(  1832328509) },
      {  INT32_C(   666147327),  INT32_C(  1276858137),  INT32_C(    82347664),  INT32_C(  1512377726), -INT32_C(     5709845),  INT32_C(   819552270), -INT32_C(  1964014004), -INT32_C(  1560860765) },
      {  INT64_C(  797819680350758097), -INT64_C(  138964590613668176), -INT64_C(    7541959196652300),  INT64_C( 2119408484976551448) } },
    { { -INT32_C(   641094720), -INT32_C(  1977162246), -INT32_C(   376566165),  INT32_C(   105100059), -INT32_C(  1610158958), -INT32_C(  1731076276),  INT32_C(   136495205), -INT32_C(  1465181721) },
      { -INT32_C(  1098812220), -INT32_C(  1001805991), -INT32_C(   777136203),  INT32_C(   500691339),  INT32_C(   700374493),  INT32_C(   566333372),  INT32_C(   925557839), -INT32_C(  1025518083) },
      {  INT64_C(  704442712513478400),  INT64_C(  292643199646371495), -INT64_C( 1127714263858658294),  INT64_C(  126334206973661995) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_mul_epi32(a, b);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mul_epu32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint32_t a[8];
    const uint32_t b[8];
    const uint64_t r[4];
  } test_vec[] = {
    { { UINT32_C(1983947362), UINT32_C(3053479846), UINT32_C(1792953067), UINT32_C(3215038466), UINT32_C( 607004743), UINT32_C(2578666130), UINT32_C(1073743766), UINT32_C(1510921976) },
      { UINT32_C( 533745272), UINT32_C(2715144374), UINT32_C( 487371546), UINT32_C(2480713036), UINT32_C(1756826326), UINT32_C(4060244572), UINT32_C(1764885105), UINT32_C(1371816409) },
      { UINT64_C( 1058922524364372464), UINT64_C(  873834308169231582), UINT64_C( 1066401912509264218), UINT64_C( 1895034379200005430) } },
    { { UINT32_C(1165005967), UINT32_C(2145863013), UINT32_C(1167913977), UINT32_C(1993898144), UINT32_C(3739193218), UINT32_C(1792074233), UINT32_C(3167945699), UINT32_C(3540948804) },
      { UINT32_C(2434367020), UINT32_C(3171942596), UINT32_C(2466426355), UINT32_C(2819283493), UINT32_C(1653008745), UINT32_C(2915850186), UINT32_C(2657787738), UINT32_C(1668380727) },
      { UINT64_C( 2836052104168008340), UINT64_C( 2880573813245663835), UINT64_C( 6180919088598691410), UINT64_C( 8419727233452038862) } },
    { { UINT32_C(3136588534), UINT32_C(2104951946), UINT32_C(3608246705), UINT32_C(3162446675), UINT32_C(3458073860), UINT32_C(3061574236), UINT32_C(3243566474), UINT32_C(1411696221) },
      { UINT32_C(3675134032), UINT32_C(3461907996), UINT32_C(1403349503), UINT32_C(2282693764), UINT32_C(2253794857), UINT32_C(2721894936), UINT32_C( 358846903), UINT32_C(2808710999) },
      { UINT64_C(11527383265684389088), UINT64_C( 5063631220163137615), UINT64_C( 7793789080794138020), UINT64_C( 1163943783869530022) } },
    { { UINT32_C(3162666911), UINT32_C(4253735933), UINT32_C(3377475396), UINT32_C(2085707603), UINT32_C(2785192077), UINT32_C( 826818426), UINT32_C( 658943184), UINT32_C(3553537843) },
      { UINT32_C( 613372199), UINT32_C(1881217324), UINT32_C(2604233032), UINT32_C(1578601425), UINT32_C(2902727219), UINT32_C( 702434649), UINT32_C( 743450105), UINT32_C(4227800788) },
      { UINT64_C( 1939891957904607289), UINT64_C( 8795732991030480672), UINT64_C( 8084652852051043863), UINT64_C(  489891379333834320) } },
    { { UINT32_C(2602602095), UINT32_C(4010557863), UINT32_C(2223654323), UINT32_C(  65184464), UINT32_C( 363915196), UINT32_C( 759074612), UINT32_C(2287570612), UINT32_C( 478435756) },
      { UINT32_C(2411177191), UINT32_C(2558444517), UINT32_C(3642493193), UINT32_C(1742602155), UINT32_C( 444370406), UINT32_C(3494361628), UINT32_C(4099514440), UINT32_C(3775979001) },
      { UINT64_C( 6275334808712815145), UINT64_C( 8099645735112523339), UINT64_C(  161713143396089576), UINT64_C( 9377928756413637280) } },
    { { UINT32_C(1718667393), UINT32_C(2499800715), UINT32_C(2725125111), UINT32_C(    609050), UINT32_C(4112156120), UINT32_C(2277859647), UINT32_C(4219215362), UINT32_C(2094828795) },
      { UINT32_C(3772992596), UINT32_C( 846520891), UINT32_C( 416670461), UINT32_C( 102293037), UINT32_C(2751148900), UINT32_C(2519449748), UINT32_C(3667044319), UINT32_C(2287431219) },
      { UINT64_C( 6484519348775622228), UINT64_C( 1135479136283046171), UINT64_C(11313153786166268000), UINT64_C(15472049723859628478) } },
    { { UINT32_C(4134025915), UINT32_C( 422108188), UINT32_C(3962699199), UINT32_C(1089620700), UINT32_C( 300150141), UINT32_C(2376601262), UINT32_C(3915856565), UINT32_C(1668398760) },
      { UINT32_C( 341432824), UINT32_C(1949205173), UINT32_C(1533108095), UINT32_C( 647713705), UINT32_C(4013457217), UINT32_C(1132257421), UINT32_C(3257721626), UINT32_C(2586221986) },
      { UINT64_C( 1411492142647633960), UINT64_C( 6075246220036915905), UINT64_C( 1204639749580017597), UINT64_C(12756770616114574690) } },
    { { UINT32_C( 732921718), UINT32_C(2174803201), UINT32_C(3873177916), UINT32_C(2500622420), UINT32_C(2223260919), UINT32_C(1053229092), UINT32_C(2248274916), UINT32_C( 102770576) },
      { UINT32_C(2821902246), UINT32_C(3911832236), UINT32_C( 667878867), UINT32_C(1958599549), UINT32_C(1157185824), UINT32_C( 646168642), UINT32_C(1152156852), UINT32_C(1363922091) },
      { UINT64_C( 2068233442166378628), UINT64_C( 2586813678227501172), UINT64_C( 2572726018520012256), UINT64_C( 2590365349649124432) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_mul_epu32(a, b);
    simde_test_x86_assert_equal_u64x4(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mulhi_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[16];
    const int16_t b[16];
    const int16_t r[16];
  } test_vec[] = {
    { { -INT16_C(  3387),  INT16_C(  9328), -INT16_C(   608), -INT16_C(  8041), -INT16_C( 24120), -INT16_C( 28930),  INT16_C( 24168), -INT16_C( 16718),
        -INT16_C( 28202), -INT16_C( 22224), -INT16_C( 31685), -INT16_C( 30796), -INT16_C( 32360), -INT16_C( 29908),  INT16_C(  7198), -INT16_C(  7385) },
      { -INT16_C( 26610), -INT16_C( 20985), -INT16_C( 24939),  INT16_C( 23950), -INT16_C( 29633), -INT16_C( 22548), -INT16_C( 24853), -INT16_C( 16027),
        -INT16_C( 27089),  INT16_C( 27498),  INT16_C(  7962), -INT16_C( 19726),  INT16_C(  7840), -INT16_C( 16835),  INT16_C( 25914),  INT16_C( 18593) },
      {  INT16_C(  1375), -INT16_C(  2987),  INT16_C(   231), -INT16_C(  2939),  INT16_C( 10906),  INT16_C(  9953), -INT16_C(  9166),  INT16_C(  4088),
         INT16_C( 11657), -INT16_C(  9325), -INT16_C(  3850),  INT16_C(  9269), -INT16_C(  3872),  INT16_C(  7682),  INT16_C(  2846), -INT16_C(  2096) } },
    { { -INT16_C( 22275), -INT16_C( 27913), -INT16_C( 31418), -INT16_C( 31248), -INT16_C(  9198), -INT16_C(   724), -INT16_C( 28038), -INT16_C( 21826),
         INT16_C( 10536),  INT16_C( 16917),  INT16_C(  1864), -INT16_C(  5900),  INT16_C( 12582),  INT16_C( 24743),  INT16_C( 18582), -INT16_C( 27735) },
      { -INT16_C( 24335),  INT16_C( 14118),  INT16_C(  5669),  INT16_C( 14269), -INT16_C(  5646),  INT16_C( 27700), -INT16_C(  3205), -INT16_C( 23786),
         INT16_C( 11036),  INT16_C( 25829), -INT16_C(  9933),  INT16_C( 22860), -INT16_C(  3317), -INT16_C( 24135),  INT16_C( 25148),  INT16_C( 11573) },
      {  INT16_C(  8271), -INT16_C(  6014), -INT16_C(  2718), -INT16_C(  6804),  INT16_C(   792), -INT16_C(   307),  INT16_C(  1371),  INT16_C(  7921),
         INT16_C(  1774),  INT16_C(  6667), -INT16_C(   283), -INT16_C(  2059), -INT16_C(   637), -INT16_C(  9113),  INT16_C(  7130), -INT16_C(  4898) } },
    { {  INT16_C( 23298),  INT16_C( 10340),  INT16_C(  8561),  INT16_C( 25439), -INT16_C( 27637), -INT16_C( 31025), -INT16_C(  6521), -INT16_C( 23766),
         INT16_C(  3857),  INT16_C( 17415),  INT16_C( 21481), -INT16_C(  2915),  INT16_C( 22343), -INT16_C( 31851), -INT16_C( 13639), -INT16_C( 17232) },
      {  INT16_C(  5157), -INT16_C( 26908),  INT16_C( 17206),  INT16_C( 16889), -INT16_C( 13865),  INT16_C( 24263), -INT16_C(  3665), -INT16_C( 16383),
         INT16_C(  2049), -INT16_C(  5627), -INT16_C( 23972), -INT16_C( 23586),  INT16_C( 29689), -INT16_C( 19674), -INT16_C( 10690),  INT16_C( 25455) },
      {  INT16_C(  1833), -INT16_C(  4246),  INT16_C(  2247),  INT16_C(  6555),  INT16_C(  5846), -INT16_C( 11487),  INT16_C(   364),  INT16_C(  5941),
         INT16_C(   120), -INT16_C(  1496), -INT16_C(  7858),  INT16_C(  1049),  INT16_C( 10121),  INT16_C(  9561),  INT16_C(  2224), -INT16_C(  6694) } },
    { {  INT16_C( 21482),  INT16_C(  8442), -INT16_C(  3178),  INT16_C( 28257),  INT16_C( 10684),  INT16_C( 27596), -INT16_C( 12774),  INT16_C(  6956),
         INT16_C( 12758),  INT16_C( 12805), -INT16_C(  7213), -INT16_C( 12843), -INT16_C(  1193), -INT16_C( 27264), -INT16_C(  4143), -INT16_C( 17160) },
      { -INT16_C(  3518), -INT16_C( 10020),  INT16_C( 16102), -INT16_C( 23994),  INT16_C(  4967), -INT16_C( 32498),  INT16_C( 15073), -INT16_C( 18531),
        -INT16_C( 23957),  INT16_C( 16106), -INT16_C( 16506), -INT16_C(  8949), -INT16_C( 29765), -INT16_C( 29582),  INT16_C( 27258), -INT16_C( 17336) },
      { -INT16_C(  1154), -INT16_C(  1291), -INT16_C(   781), -INT16_C( 10346),  INT16_C(   809), -INT16_C( 13685), -INT16_C(  2938), -INT16_C(  1967),
        -INT16_C(  4664),  INT16_C(  3146),  INT16_C(  1816),  INT16_C(  1753),  INT16_C(   541),  INT16_C( 12306), -INT16_C(  1724),  INT16_C(  4539) } },
    { {  INT16_C(  9565),  INT16_C( 17301), -INT16_C(  9373), -INT16_C( 13595), -INT16_C(  3090), -INT16_C( 12469), -INT16_C(  6099), -INT16_C( 26489),
         INT16_C( 29067),  INT16_C(  4567), -INT16_C(  7632), -INT16_C(  5138),  INT16_C( 24686), -INT16_C(  6024), -INT16_C( 16182),  INT16_C( 10149) },
      {  INT16_C( 15077),  INT16_C( 18538),  INT16_C( 20501),  INT16_C(  1042),  INT16_C( 24131),  INT16_C( 29139),  INT16_C( 23110), -INT16_C( 12023),
        -INT16_C(  7989), -INT16_C(   798), -INT16_C( 12093),  INT16_C( 12775),  INT16_C( 24368), -INT16_C(  1255), -INT16_C( 16864),  INT16_C(  1314) },
      {  INT16_C(  2200),  INT16_C(  4893), -INT16_C(  2933), -INT16_C(   217), -INT16_C(  1138), -INT16_C(  5545), -INT16_C(  2151),  INT16_C(  4859),
        -INT16_C(  3544), -INT16_C(    56),  INT16_C(  1408), -INT16_C(  1002),  INT16_C(  9178),  INT16_C(   115),  INT16_C(  4164),  INT16_C(   203) } },
    { { -INT16_C( 29192),  INT16_C(  3662),  INT16_C( 24797),  INT16_C(  8210), -INT16_C(  6722),  INT16_C(  1425), -INT16_C( 25792),  INT16_C(  3030),
        -INT16_C( 18053),  INT16_C( 15879), -INT16_C(  4215), -INT16_C( 17809), -INT16_C( 30386),  INT16_C( 28341), -INT16_C( 10425),  INT16_C( 16500) },
      { -INT16_C( 15772),  INT16_C( 16718),  INT16_C( 24610), -INT16_C(  7838), -INT16_C(  3259), -INT16_C( 31258), -INT16_C( 17266),  INT16_C(  2705),
        -INT16_C( 26507), -INT16_C(   184), -INT16_C( 18297), -INT16_C( 10567),  INT16_C( 28225), -INT16_C( 30652), -INT16_C( 18363), -INT16_C( 21816) },
      {  INT16_C(  7025),  INT16_C(   934),  INT16_C(  9311), -INT16_C(   982),  INT16_C(   334), -INT16_C(   680),  INT16_C(  6795),  INT16_C(   125),
         INT16_C(  7301), -INT16_C(    45),  INT16_C(  1176),  INT16_C(  2871), -INT16_C( 13087), -INT16_C( 13256),  INT16_C(  2921), -INT16_C(  5493) } },
    { {  INT16_C(  5754), -INT16_C( 25109),  INT16_C( 19830), -INT16_C( 17282),  INT16_C( 25665), -INT16_C( 12479), -INT16_C( 11744), -INT16_C( 26919),
         INT16_C(  8811), -INT16_C(  3435),  INT16_C( 20186),  INT16_C(  7112),  INT16_C(  3516),  INT16_C(   419),  INT16_C( 27845),  INT16_C( 16555) },
      { -INT16_C( 26750), -INT16_C(  1571),  INT16_C( 23524),  INT16_C(  9653), -INT16_C(  2369), -INT16_C(  8203), -INT16_C( 12599),  INT16_C( 13429),
         INT16_C(  2800), -INT16_C( 13786), -INT16_C(  4264),  INT16_C(  5349), -INT16_C( 30212), -INT16_C( 16106), -INT16_C( 15883),  INT16_C( 30465) },
      { -INT16_C(  2349),  INT16_C(   601),  INT16_C(  7117), -INT16_C(  2546), -INT16_C(   928),  INT16_C(  1561),  INT16_C(  2257), -INT16_C(  5516),
         INT16_C(   376),  INT16_C(   722), -INT16_C(  1314),  INT16_C(   580), -INT16_C(  1621), -INT16_C(   103), -INT16_C(  6749),  INT16_C(  7695) } },
    { { -INT16_C(  8616),  INT16_C( 15728),  INT16_C(  9529), -INT16_C(  1950),  INT16_C( 22300), -INT16_C(  6696),  INT16_C( 19750),  INT16_C(  5657),
         INT16_C( 16216), -INT16_C( 20255), -INT16_C( 14802),  INT16_C( 10949), -INT16_C(  9393),  INT16_C( 17644), -INT16_C(  4708), -INT16_C(  2628) },
      {  INT16_C( 11468),  INT16_C(  1330), -INT16_C( 27566),  INT16_C( 28414), -INT16_C( 10516),  INT16_C(  4691),  INT16_C( 27683),  INT16_C( 31528),
         INT16_C(  2475), -INT16_C(  9684), -INT16_C(  3632),  INT16_C(  7940), -INT16_C(  3892),  INT16_C( 26724),  INT16_C(  8414), -INT16_C( 21923) },
      { -INT16_C(  1508),  INT16_C(   319), -INT16_C(  4009), -INT16_C(   846), -INT16_C(  3579), -INT16_C(   480),  INT16_C(  8342),  INT16_C(  2721),
         INT16_C(   612),  INT16_C(  2993),  INT16_C(   820),  INT16_C(  1326),  INT16_C(   557),  INT16_C(  7194), -INT16_C(   605),  INT16_C(   879) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_mulhi_epi16(a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mulhi_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint16_t a[16];
    const uint16_t b[16];
    const uint16_t r[16];
  } test_vec[] = {
    { { UINT16_C(10332), UINT16_C(42272), UINT16_C(44135), UINT16_C( 8711), UINT16_C(23908), UINT16_C(41433), UINT16_C(47685), UINT16_C( 9375),
        UINT16_C( 1656), UINT16_C(30122), UINT16_C( 5686), UINT16_C(62995), UINT16_C(28193), UINT16_C(48077), UINT16_C(50425), UINT16_C(21926) },
      { UINT16_C(50924), UINT16_C(21499), UINT16_C(  626), UINT16_C(54901), UINT16_C(20063), UINT16_C(42103), UINT16_C( 5641), UINT16_C(33224),
        UINT16_C(29468), UINT16_C(21238), UINT16_C( 2441), UINT16_C(43592), UINT16_C( 5751), UINT16_C(29029), UINT16_C( 3034), UINT16_C(50886) },
      { UINT16_C( 8028), UINT16_C(13867), UINT16_C(  421), UINT16_C( 7297), UINT16_C( 7319), UINT16_C(26618), UINT16_C( 4104), UINT16_C( 4752),
        UINT16_C(  744), UINT16_C( 9761), UINT16_C(  211), UINT16_C(41901), UINT16_C( 2474), UINT16_C(21295), UINT16_C( 2334), UINT16_C(17024) } },
    { { UINT16_C(49617), UINT16_C(17177), UINT16_C(36803), UINT16_C( 8729), UINT16_C(37341), UINT16_C(59079), UINT16_C(36775), UINT16_C(50280),
        UINT16_C(24066), UINT16_C(35606), UINT16_C(24424), UINT16_C(57142), UINT16_C(39797), UINT16_C(20304), UINT16_C( 6055), UINT16_C(30741) },
      { UINT16_C(11992), UINT16_C(40124), UINT16_C(54717), UINT16_C(39870), UINT16_C(34150), UINT16_C( 3713), UINT16_C(59669), UINT16_C( 6098),
        UINT16_C(59464), UINT16_C(45219), UINT16_C(55623), UINT16_C(48271), UINT16_C(57460), UINT16_C( 6923), UINT16_C( 8439), UINT16_C(53140) },
      { UINT16_C( 9079), UINT16_C(10516), UINT16_C(30727), UINT16_C( 5310), UINT16_C(19457), UINT16_C( 3347), UINT16_C(33482), UINT16_C( 4678),
        UINT16_C(21836), UINT16_C(24567), UINT16_C(20729), UINT16_C(42088), UINT16_C(34892), UINT16_C( 2144), UINT16_C(  779), UINT16_C(24926) } },
    { { UINT16_C(20559), UINT16_C( 3179), UINT16_C(10789), UINT16_C(36007), UINT16_C(10671), UINT16_C(50330), UINT16_C(27666), UINT16_C(23260),
        UINT16_C(32596), UINT16_C(39946), UINT16_C(39512), UINT16_C(52312), UINT16_C(25722), UINT16_C(29160), UINT16_C(31876), UINT16_C(54080) },
      { UINT16_C(44236), UINT16_C(61920), UINT16_C(34774), UINT16_C(34173), UINT16_C( 6064), UINT16_C(49994), UINT16_C( 9859), UINT16_C(55325),
        UINT16_C(10405), UINT16_C(64884), UINT16_C(52418), UINT16_C(15561), UINT16_C(45360), UINT16_C(46509), UINT16_C(60717), UINT16_C(63880) },
      { UINT16_C(13877), UINT16_C( 3003), UINT16_C( 5724), UINT16_C(18775), UINT16_C(  987), UINT16_C(38394), UINT16_C( 4161), UINT16_C(19635),
        UINT16_C( 5175), UINT16_C(39548), UINT16_C(31603), UINT16_C(12421), UINT16_C(17803), UINT16_C(20694), UINT16_C(29532), UINT16_C(52713) } },
    { { UINT16_C(26777), UINT16_C(28651), UINT16_C(26864), UINT16_C(41205), UINT16_C(16256), UINT16_C(  867), UINT16_C(33125), UINT16_C( 2779),
        UINT16_C(20393), UINT16_C(27399), UINT16_C(53276), UINT16_C(19623), UINT16_C(21634), UINT16_C(44801), UINT16_C(35393), UINT16_C(56233) },
      { UINT16_C(38130), UINT16_C(57930), UINT16_C(16380), UINT16_C(31875), UINT16_C(59006), UINT16_C(58240), UINT16_C(23399), UINT16_C( 4333),
        UINT16_C(62635), UINT16_C(51067), UINT16_C( 8901), UINT16_C(18195), UINT16_C( 5494), UINT16_C(47350), UINT16_C(40863), UINT16_C(37267) },
      { UINT16_C(15579), UINT16_C(25325), UINT16_C( 6714), UINT16_C(20041), UINT16_C(14636), UINT16_C(  770), UINT16_C(11826), UINT16_C(  183),
        UINT16_C(19490), UINT16_C(21349), UINT16_C( 7235), UINT16_C( 5448), UINT16_C( 1813), UINT16_C(32368), UINT16_C(22068), UINT16_C(31976) } },
    { { UINT16_C(56627), UINT16_C(12404), UINT16_C(63261), UINT16_C(39852), UINT16_C(11485), UINT16_C(17791), UINT16_C(27784), UINT16_C(13141),
        UINT16_C(53601), UINT16_C( 9978), UINT16_C( 3571), UINT16_C(27245), UINT16_C(25378), UINT16_C(49442), UINT16_C(46339), UINT16_C(13907) },
      { UINT16_C(51090), UINT16_C(44902), UINT16_C( 5054), UINT16_C(39755), UINT16_C(51775), UINT16_C(51168), UINT16_C(13878), UINT16_C(38906),
        UINT16_C(62471), UINT16_C(64189), UINT16_C(10754), UINT16_C( 9316), UINT16_C(34446), UINT16_C(37350), UINT16_C(14651), UINT16_C(52935) },
      { UINT16_C(44144), UINT16_C( 8498), UINT16_C( 4878), UINT16_C(24174), UINT16_C( 9073), UINT16_C(13890), UINT16_C( 5883), UINT16_C( 7801),
        UINT16_C(51094), UINT16_C( 9772), UINT16_C(  585), UINT16_C( 3872), UINT16_C(13338), UINT16_C(28177), UINT16_C(10359), UINT16_C(11233) } },
    { { UINT16_C(11776), UINT16_C(48765), UINT16_C(51265), UINT16_C(32857), UINT16_C(14994), UINT16_C(51528), UINT16_C(17008), UINT16_C(30560),
        UINT16_C( 7735), UINT16_C(14705), UINT16_C(54856), UINT16_C(54877), UINT16_C(17244), UINT16_C(39015), UINT16_C(12156), UINT16_C(31846) },
      { UINT16_C(58205), UINT16_C(40506), UINT16_C(38060), UINT16_C(15902), UINT16_C(26318), UINT16_C(15879), UINT16_C(26793), UINT16_C(57525),
        UINT16_C( 9862), UINT16_C(52761), UINT16_C(30460), UINT16_C(22949), UINT16_C( 3258), UINT16_C(14065), UINT16_C(22331), UINT16_C(39091) },
      { UINT16_C(10458), UINT16_C(30140), UINT16_C(29772), UINT16_C( 7972), UINT16_C( 6021), UINT16_C(12484), UINT16_C( 6953), UINT16_C(26824),
        UINT16_C( 1163), UINT16_C(11838), UINT16_C(25496), UINT16_C(19216), UINT16_C(  857), UINT16_C( 8373), UINT16_C( 4142), UINT16_C(18995) } },
    { { UINT16_C(60730), UINT16_C(58934), UINT16_C(21889), UINT16_C(20261), UINT16_C(11451), UINT16_C(25741), UINT16_C(17044), UINT16_C( 6724),
        UINT16_C(23913), UINT16_C(26089), UINT16_C(36564), UINT16_C(36542), UINT16_C(44954), UINT16_C(54980), UINT16_C(30470), UINT16_C(16750) },
      { UINT16_C(42341), UINT16_C(58919), UINT16_C(19706), UINT16_C(46390), UINT16_C(50041), UINT16_C( 3354), UINT16_C(24070), UINT16_C(28456),
        UINT16_C( 4540), UINT16_C(37076), UINT16_C(37791), UINT16_C(14622), UINT16_C(57922), UINT16_C(18703), UINT16_C(32346), UINT16_C(49034) },
      { UINT16_C(39235), UINT16_C(52983), UINT16_C( 6581), UINT16_C(14341), UINT16_C( 8743), UINT16_C( 1317), UINT16_C( 6259), UINT16_C( 2919),
        UINT16_C( 1656), UINT16_C(14759), UINT16_C(21084), UINT16_C( 8153), UINT16_C(39731), UINT16_C(15690), UINT16_C(15038), UINT16_C(12532) } },
    { { UINT16_C(45347), UINT16_C( 7589), UINT16_C(56318), UINT16_C(30674), UINT16_C(60575), UINT16_C(42372), UINT16_C(44107), UINT16_C( 1812),
        UINT16_C(59581), UINT16_C(23703), UINT16_C(46459), UINT16_C(48790), UINT16_C(42391), UINT16_C(61703), UINT16_C(37155), UINT16_C(18096) },
      { UINT16_C(22082), UINT16_C(16483), UINT16_C(13873), UINT16_C(53431), UINT16_C(15394), UINT16_C(28021), UINT16_C(35304), UINT16_C(42612),
        UINT16_C( 2930), UINT16_C(60674), UINT16_C(39104), UINT16_C(22699), UINT16_C(45630), UINT16_C(24905), UINT16_C(64067), UINT16_C(34472) },
      { UINT16_C(15279), UINT16_C( 1908), UINT16_C(11921), UINT16_C(25008), UINT16_C(14228), UINT16_C(18116), UINT16_C(23760), UINT16_C( 1178),
        UINT16_C( 2663), UINT16_C(21944), UINT16_C(27721), UINT16_C(16898), UINT16_C(29515), UINT16_C(23448), UINT16_C(36322), UINT16_C( 9518) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_mulhi_epu16(a, b);
    simde_test_x86_assert_equal_u16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mulhrs_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[16];
    const int16_t b[16];
    const int16_t r[16];
  } test_vec[] = {
    { { -INT16_C(  7424), -INT16_C( 12265),  INT16_C( 30100),  INT16_C( 15580),  INT16_C( 26159), -INT16_C( 29591), -INT16_C( 20062), -INT16_C( 16404),
         INT16_C( 25875), -INT16_C( 16332),  INT16_C(  6642),  INT16_C( 19995), -INT16_C( 16871),  INT16_C(  7464), -INT16_C( 21479),  INT16_C(  6866) },
      { -INT16_C(  5744),  INT16_C(  9450), -INT16_C( 14498), -INT16_C( 29088), -INT16_C( 13779), -INT16_C( 12518),  INT16_C(  1915), -INT16_C( 28786),
        -INT16_C( 15764),  INT16_C( 24143),  INT16_C( 27355), -INT16_C(  2900), -INT16_C( 10967),  INT16_C( 16914), -INT16_C(  7039),  INT16_C(  4444) },
      {  INT16_C(  1301), -INT16_C(  3537), -INT16_C( 13318), -INT16_C( 13830), -INT16_C( 11000),  INT16_C( 11304), -INT16_C(  1172),  INT16_C( 14411),
        -INT16_C( 12448), -INT16_C( 12033),  INT16_C(  5545), -INT16_C(  1770),  INT16_C(  5646),  INT16_C(  3853),  INT16_C(  4614),  INT16_C(   931) } },
    { {  INT16_C( 18381),  INT16_C( 11062), -INT16_C( 27122),  INT16_C( 15289), -INT16_C( 11168), -INT16_C(  9206), -INT16_C( 26405),  INT16_C( 18283),
        -INT16_C( 17829),  INT16_C( 13989),  INT16_C( 20773),  INT16_C( 20011),  INT16_C( 15654), -INT16_C( 22384), -INT16_C(  4831), -INT16_C(  4423) },
      { -INT16_C(  4300),  INT16_C( 16921), -INT16_C( 11386), -INT16_C(  6531), -INT16_C( 30809), -INT16_C( 32062),  INT16_C( 11551),  INT16_C( 31433),
         INT16_C( 28392),  INT16_C(  3505), -INT16_C(  9025), -INT16_C(  6565), -INT16_C(  5351),  INT16_C( 14990),  INT16_C( 18392),  INT16_C(  3112) },
      { -INT16_C(  2412),  INT16_C(  5712),  INT16_C(  9424), -INT16_C(  3047),  INT16_C( 10500),  INT16_C(  9008), -INT16_C(  9308),  INT16_C( 17538),
        -INT16_C( 15448),  INT16_C(  1496), -INT16_C(  5721), -INT16_C(  4009), -INT16_C(  2556), -INT16_C( 10240), -INT16_C(  2712), -INT16_C(   420) } },
    { {  INT16_C( 16695), -INT16_C( 17074), -INT16_C( 13548), -INT16_C( 17501),  INT16_C( 26194),  INT16_C( 29245),  INT16_C(  1683),  INT16_C( 31724),
        -INT16_C( 25228),  INT16_C( 13448), -INT16_C(  7303), -INT16_C( 28134), -INT16_C( 22321), -INT16_C( 22580), -INT16_C(  2833),  INT16_C(  9908) },
      {  INT16_C(   566),  INT16_C( 19171), -INT16_C( 30770),  INT16_C(  8198),  INT16_C( 17389), -INT16_C( 32622),  INT16_C( 32586), -INT16_C( 16644),
        -INT16_C( 31716), -INT16_C( 26894),  INT16_C(  3176),  INT16_C( 14120), -INT16_C(  2636), -INT16_C( 23330), -INT16_C( 27927),  INT16_C(  8138) },
      {  INT16_C(   288), -INT16_C(  9989),  INT16_C( 12722), -INT16_C(  4378),  INT16_C( 13900), -INT16_C( 29115),  INT16_C(  1674), -INT16_C( 16114),
         INT16_C( 24418), -INT16_C( 11037), -INT16_C(   708), -INT16_C( 12123),  INT16_C(  1796),  INT16_C( 16076),  INT16_C(  2414),  INT16_C(  2461) } },
    { { -INT16_C( 20843),  INT16_C( 25450),  INT16_C( 28725),  INT16_C(  8835),  INT16_C(  5811), -INT16_C(   606), -INT16_C( 24939), -INT16_C( 20036),
        -INT16_C( 20957), -INT16_C( 29881),  INT16_C( 28859),  INT16_C( 28610), -INT16_C( 24475),  INT16_C( 19987), -INT16_C(  8653), -INT16_C( 14226) },
      { -INT16_C( 10100), -INT16_C( 16085), -INT16_C( 20920), -INT16_C(  1053), -INT16_C( 31292),  INT16_C( 23033), -INT16_C( 19164),  INT16_C( 18187),
         INT16_C( 21091),  INT16_C(  7890), -INT16_C( 27454),  INT16_C( 10126), -INT16_C( 24268),  INT16_C( 26486), -INT16_C(  7041),  INT16_C(  2863) },
      {  INT16_C(  6424), -INT16_C( 12493), -INT16_C( 18339), -INT16_C(   284), -INT16_C(  5549), -INT16_C(   426),  INT16_C( 14585), -INT16_C( 11120),
        -INT16_C( 13489), -INT16_C(  7195), -INT16_C( 24179),  INT16_C(  8841),  INT16_C( 18126),  INT16_C( 16155),  INT16_C(  1859), -INT16_C(  1243) } },
    { {  INT16_C( 23228),  INT16_C(  1228), -INT16_C( 20727), -INT16_C( 12801), -INT16_C(  1995),  INT16_C( 22823),  INT16_C( 12973),  INT16_C(  4512),
         INT16_C( 29316),  INT16_C( 18223), -INT16_C( 17146),  INT16_C( 14958), -INT16_C(  7073), -INT16_C(  8542), -INT16_C( 11832), -INT16_C( 31510) },
      { -INT16_C( 18900),  INT16_C( 13704), -INT16_C( 30618), -INT16_C( 25854),  INT16_C( 10624),  INT16_C( 12020), -INT16_C( 27557), -INT16_C(  8129),
         INT16_C( 28166),  INT16_C(  3111), -INT16_C( 27348), -INT16_C( 29882), -INT16_C(  6022),  INT16_C( 17001),  INT16_C( 21434), -INT16_C(  6457) },
      { -INT16_C( 13397),  INT16_C(   514),  INT16_C( 19367),  INT16_C( 10100), -INT16_C(   647),  INT16_C(  8372), -INT16_C( 10910), -INT16_C(  1119),
         INT16_C( 25199),  INT16_C(  1730),  INT16_C( 14310), -INT16_C( 13641),  INT16_C(  1300), -INT16_C(  4432), -INT16_C(  7739),  INT16_C(  6209) } },
    { {  INT16_C( 20234),  INT16_C( 28699),  INT16_C(  7639),  INT16_C( 22539), -INT16_C(   185), -INT16_C( 23930), -INT16_C( 14957), -INT16_C( 26238),
        -INT16_C( 22221),  INT16_C( 24485), -INT16_C(  5313), -INT16_C( 17942),  INT16_C( 21716), -INT16_C( 28933), -INT16_C( 15705), -INT16_C( 20108) },
      { -INT16_C( 28910), -INT16_C(  5855),  INT16_C( 11436), -INT16_C(  3263), -INT16_C( 14549), -INT16_C( 16746),  INT16_C(  6284), -INT16_C( 16297),
        -INT16_C(   830),  INT16_C(   287),  INT16_C(  2792), -INT16_C( 17222), -INT16_C( 19106),  INT16_C(  1354), -INT16_C( 16776), -INT16_C( 30025) },
      { -INT16_C( 17852), -INT16_C(  5128),  INT16_C(  2666), -INT16_C(  2244),  INT16_C(    82),  INT16_C( 12229), -INT16_C(  2868),  INT16_C( 13049),
         INT16_C(   563),  INT16_C(   214), -INT16_C(   453),  INT16_C(  9430), -INT16_C( 12662), -INT16_C(  1196),  INT16_C(  8040),  INT16_C( 18425) } },
    { { -INT16_C( 10163), -INT16_C(  1677), -INT16_C( 19195),  INT16_C( 12525), -INT16_C( 31876),  INT16_C(  2543),  INT16_C( 18075),  INT16_C( 24009),
        -INT16_C(  6077),  INT16_C( 11102),  INT16_C(  6386),  INT16_C( 20711),  INT16_C( 12750),  INT16_C( 18006),  INT16_C(  3567),  INT16_C( 15568) },
      {  INT16_C( 17381), -INT16_C(  5579),  INT16_C(  8952),  INT16_C( 29979),  INT16_C(  2725),  INT16_C( 16766),  INT16_C( 18256), -INT16_C( 27746),
        -INT16_C(   721),  INT16_C(  8894), -INT16_C( 23275), -INT16_C(  7310), -INT16_C( 14122), -INT16_C( 15063), -INT16_C(  1579), -INT16_C( 17663) },
      { -INT16_C(  5391),  INT16_C(   286), -INT16_C(  5244),  INT16_C( 11459), -INT16_C(  2651),  INT16_C(  1301),  INT16_C( 10070), -INT16_C( 20329),
         INT16_C(   134),  INT16_C(  3013), -INT16_C(  4536), -INT16_C(  4620), -INT16_C(  5495), -INT16_C(  8277), -INT16_C(   172), -INT16_C(  8392) } },
    { {  INT16_C( 14141),  INT16_C( 13733), -INT16_C( 16295), -INT16_C(    86),  INT16_C( 10442),  INT16_C(  6976), -INT16_C(  8593), -INT16_C( 24658),
         INT16_C( 28123), -INT16_C(  3647),  INT16_C( 13074), -INT16_C(  5676), -INT16_C(   260), -INT16_C( 11858), -INT16_C( 20233),  INT16_C( 13452) },
      {  INT16_C( 13031),  INT16_C( 16490),  INT16_C(  5362), -INT16_C( 17089),  INT16_C( 32573), -INT16_C( 21288), -INT16_C( 31138),  INT16_C( 14667),
         INT16_C(  3315),  INT16_C(  1578), -INT16_C(   192),  INT16_C( 15599), -INT16_C( 25091), -INT16_C(  3059), -INT16_C( 26035),  INT16_C( 13353) },
      {  INT16_C(  5624),  INT16_C(  6911), -INT16_C(  2666),  INT16_C(    45),  INT16_C( 10380), -INT16_C(  4532),  INT16_C(  8166), -INT16_C( 11037),
         INT16_C(  2845), -INT16_C(   176), -INT16_C(    77), -INT16_C(  2702),  INT16_C(   199),  INT16_C(  1107),  INT16_C( 16076),  INT16_C(  5482) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_mulhrs_epi16(a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mullo_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C( 26958), INT16_C(  5839), INT16_C( 10773), INT16_C(-17217),
                            INT16_C( 20782), INT16_C(-24278), INT16_C( 14053), INT16_C(  4872),
                            INT16_C(-31512), INT16_C( -5844), INT16_C(  1857), INT16_C(  9311),
                            INT16_C( 26459), INT16_C( 31943), INT16_C(-26611), INT16_C( 26444)),
      simde_mm256_set_epi16(INT16_C( 25062), INT16_C(-26342), INT16_C(  2282), INT16_C( -1929),
                            INT16_C( -4943), INT16_C(-11597), INT16_C(-15401), INT16_C(-15540),
                            INT16_C( -1566), INT16_C( -8249), INT16_C(  3344), INT16_C(-23800),
                            INT16_C( 15220), INT16_C( -3764), INT16_C( 25181), INT16_C( 11266)),
      simde_mm256_set_epi16(INT16_C( 10772), INT16_C(  2054), INT16_C(  7986), INT16_C(-15159),
                            INT16_C(-30514), INT16_C(  9310), INT16_C(-30381), INT16_C(-16800),
                            INT16_C(  -816), INT16_C(-27340), INT16_C(-16112), INT16_C(-24584),
                            INT16_C(-12740), INT16_C( 25108), INT16_C( 14009), INT16_C( -8552)) },
    { simde_mm256_set_epi16(INT16_C( -5684), INT16_C( 15383), INT16_C(-12220), INT16_C(  1990),
                            INT16_C( -3866), INT16_C(-10346), INT16_C( 21545), INT16_C( 18065),
                            INT16_C(-21267), INT16_C( -3518), INT16_C( 27216), INT16_C( 24702),
                            INT16_C( -8721), INT16_C(-16371), INT16_C( 21885), INT16_C( 22906)),
      simde_mm256_set_epi16(INT16_C( -9241), INT16_C( -3030), INT16_C(-27821), INT16_C( 13222),
                            INT16_C( -7134), INT16_C(  9766), INT16_C(  1304), INT16_C( 21664),
                            INT16_C( 21613), INT16_C( 29626), INT16_C(-23683), INT16_C( 15587),
                            INT16_C( 14936), INT16_C( 31442), INT16_C(  1986), INT16_C(-23873)),
      simde_mm256_set_epi16(INT16_C( 31508), INT16_C(-14394), INT16_C(-28148), INT16_C( 31844),
                            INT16_C(-10612), INT16_C( 17476), INT16_C(-20264), INT16_C(-20832),
                            INT16_C( 25833), INT16_C(-22028), INT16_C( -9968), INT16_C(  6074),
                            INT16_C( 28712), INT16_C(-17238), INT16_C( 13242), INT16_C( -2554)) },
    { simde_mm256_set_epi16(INT16_C(-22642), INT16_C(   406), INT16_C( -2741), INT16_C( 24854),
                            INT16_C(-32159), INT16_C(  5357), INT16_C( 22365), INT16_C(-19783),
                            INT16_C(  6458), INT16_C( -2382), INT16_C( 27277), INT16_C( 20167),
                            INT16_C(   308), INT16_C( 11773), INT16_C(-18240), INT16_C(-22526)),
      simde_mm256_set_epi16(INT16_C(  3067), INT16_C( -5016), INT16_C( 13492), INT16_C(-13562),
                            INT16_C(-32027), INT16_C(-13606), INT16_C(-15731), INT16_C( 23689),
                            INT16_C( -1980), INT16_C(-21001), INT16_C( 20300), INT16_C( -4296),
                            INT16_C(  -123), INT16_C( 14799), INT16_C(-32140), INT16_C( 31677)),
      simde_mm256_set_epi16(INT16_C( 25146), INT16_C( -4880), INT16_C(-19268), INT16_C(-18300),
                            INT16_C( -7483), INT16_C(-11310), INT16_C(-26567), INT16_C(  8449),
                            INT16_C( -7320), INT16_C( 20414), INT16_C(  9436), INT16_C(  1160),
                            INT16_C( 27652), INT16_C(-31597), INT16_C( 14080), INT16_C(  -134)) },
    { simde_mm256_set_epi16(INT16_C(-19186), INT16_C(  1301), INT16_C( 31295), INT16_C(-15933),
                            INT16_C( 15507), INT16_C( -5145), INT16_C( 22638), INT16_C( -9549),
                            INT16_C(  5226), INT16_C( -8321), INT16_C( 11534), INT16_C( -7469),
                            INT16_C( 21265), INT16_C( -8572), INT16_C(  3867), INT16_C( 30789)),
      simde_mm256_set_epi16(INT16_C(-11613), INT16_C( 15229), INT16_C( -2018), INT16_C( 27597),
                            INT16_C(-11741), INT16_C(-30691), INT16_C( 13581), INT16_C(-20193),
                            INT16_C( 19002), INT16_C(-20438), INT16_C( 23792), INT16_C( 10741),
                            INT16_C( 24708), INT16_C(-12234), INT16_C(-16212), INT16_C(-32518)),
      simde_mm256_set_epi16(INT16_C(-15382), INT16_C( 21057), INT16_C( 23394), INT16_C(-21977),
                            INT16_C( -8679), INT16_C( 28971), INT16_C( 17302), INT16_C( 16045),
                            INT16_C( 17412), INT16_C( -1322), INT16_C( 17696), INT16_C( -8465),
                            INT16_C( 13508), INT16_C( 12248), INT16_C( 26148), INT16_C( -3230)) },
    { simde_mm256_set_epi16(INT16_C( 31461), INT16_C( 28893), INT16_C(-27940), INT16_C(-14179),
                            INT16_C( -7147), INT16_C(  8716), INT16_C( -5522), INT16_C( -7988),
                            INT16_C( 11144), INT16_C( 18257), INT16_C(  -233), INT16_C( 22445),
                            INT16_C(-19828), INT16_C( 15498), INT16_C( 17919), INT16_C( -2256)),
      simde_mm256_set_epi16(INT16_C(-10903), INT16_C(  5816), INT16_C( 24259), INT16_C(-27791),
                            INT16_C(-16404), INT16_C(  3280), INT16_C(-17628), INT16_C( -2370),
                            INT16_C( 12909), INT16_C( 25550), INT16_C(  7994), INT16_C(   693),
                            INT16_C(  -412), INT16_C( 18409), INT16_C(-18714), INT16_C(    -7)),
      simde_mm256_set_epi16(INT16_C( -3859), INT16_C(  7384), INT16_C(-23148), INT16_C(-19379),
                            INT16_C( -4516), INT16_C( 14784), INT16_C( 20856), INT16_C( -8344),
                            INT16_C(  6376), INT16_C(-18898), INT16_C(-27594), INT16_C( 22353),
                            INT16_C(-22864), INT16_C( 24474), INT16_C( 11546), INT16_C( 15792)) },
    { simde_mm256_set_epi16(INT16_C(-23599), INT16_C( -6775), INT16_C(-22042), INT16_C( 23599),
                            INT16_C( 29883), INT16_C(-16427), INT16_C( 18767), INT16_C( 13204),
                            INT16_C(-18900), INT16_C(-23120), INT16_C( 17609), INT16_C(-28983),
                            INT16_C( 10186), INT16_C(-12557), INT16_C(  9866), INT16_C( 22138)),
      simde_mm256_set_epi16(INT16_C(-21041), INT16_C(-13045), INT16_C( 15958), INT16_C( 31878),
                            INT16_C( 18632), INT16_C( 10843), INT16_C(-29976), INT16_C(-16463),
                            INT16_C(-15244), INT16_C( -3717), INT16_C( 16396), INT16_C(-24330),
                            INT16_C( -6946), INT16_C( 27306), INT16_C( -9977), INT16_C(-30515)),
      simde_mm256_set_epi16(INT16_C(-19713), INT16_C(-28189), INT16_C(-14524), INT16_C(  1178),
                            INT16_C(-13800), INT16_C(  8887), INT16_C(  1432), INT16_C(  5460),
                            INT16_C( 15344), INT16_C( 19344), INT16_C( 31084), INT16_C(-10970),
                            INT16_C( 26924), INT16_C(  2910), INT16_C(  1990), INT16_C(  4018)) },
    { simde_mm256_set_epi16(INT16_C(  8019), INT16_C(-18318), INT16_C(-27174), INT16_C(-24268),
                            INT16_C(-11542), INT16_C(  3812), INT16_C( 30491), INT16_C(  9957),
                            INT16_C( 24267), INT16_C( 14634), INT16_C(   742), INT16_C(  6819),
                            INT16_C( 18671), INT16_C(-25958), INT16_C( 11320), INT16_C( 22969)),
      simde_mm256_set_epi16(INT16_C(-30081), INT16_C(-15383), INT16_C(-30862), INT16_C( 19583),
                            INT16_C(  4817), INT16_C( 15541), INT16_C( 22774), INT16_C( -2106),
                            INT16_C(-10048), INT16_C(-27305), INT16_C(-18367), INT16_C(-29706),
                            INT16_C(-10409), INT16_C( 15691), INT16_C(  9193), INT16_C( 19182)),
      simde_mm256_set_epi16(INT16_C( 18477), INT16_C(-19006), INT16_C(-20204), INT16_C( 26828),
                            INT16_C(-23286), INT16_C( -2252), INT16_C(-17422), INT16_C(  2078),
                            INT16_C( 24640), INT16_C( -8378), INT16_C(  3174), INT16_C(  6562),
                            INT16_C(-32199), INT16_C(  -738), INT16_C( -6408), INT16_C( -7170)) },
    { simde_mm256_set_epi16(INT16_C(  1498), INT16_C(-31368), INT16_C( -3455), INT16_C(-19849),
                            INT16_C(-16083), INT16_C(-23087), INT16_C( 26835), INT16_C( 10141),
                            INT16_C( 24239), INT16_C(-15471), INT16_C(   117), INT16_C(-26512),
                            INT16_C( -4941), INT16_C( 10703), INT16_C( 12266), INT16_C(-25802)),
      simde_mm256_set_epi16(INT16_C(-20053), INT16_C(-31092), INT16_C( 14330), INT16_C( 14498),
                            INT16_C( 16232), INT16_C(-21016), INT16_C( 26384), INT16_C(-26667),
                            INT16_C( 26626), INT16_C(  -346), INT16_C( 22688), INT16_C(-31544),
                            INT16_C( 24113), INT16_C(  2549), INT16_C( 12831), INT16_C(-19325)),
      simde_mm256_set_epi16(INT16_C(-23906), INT16_C(-12896), INT16_C(-30470), INT16_C( -2226),
                            INT16_C(-29368), INT16_C(-32152), INT16_C( 29232), INT16_C(-28511),
                            INT16_C(-10914), INT16_C(-20986), INT16_C(-32480), INT16_C(-10368),
                            INT16_C(  2115), INT16_C( 18971), INT16_C(-32426), INT16_C( 25762)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_mullo_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_mullo_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(-1352403780), INT32_C( -343467662), INT32_C(  517643457), INT32_C( -488960691),
                            INT32_C(-1266352672), INT32_C( 1517008609), INT32_C(  990411931), INT32_C(-1870795966)),
      simde_mm256_set_epi32(INT32_C(-1381411484), INT32_C(-1688230631), INT32_C( 2122460393), INT32_C( 1331099088),
                            INT32_C(-1172838687), INT32_C(-1435873650), INT32_C(  806691655), INT32_C( 1528853489)),
      simde_mm256_set_epi32(INT32_C( 1332348272), INT32_C(-1964640734), INT32_C( 1072420777), INT32_C(  814972816),
                            INT32_C(-1273339424), INT32_C(  480378574), INT32_C( -786776067), INT32_C( 1970872098)) },
    { simde_mm256_set_epi32(INT32_C( 1337827173), INT32_C( -259025503), INT32_C( -584925635), INT32_C(  195250370),
                            INT32_C(-1297361156), INT32_C( -136339105), INT32_C(  279578676), INT32_C(-1541670246)),
      simde_mm256_set_epi32(INT32_C(  163529464), INT32_C( 1163098936), INT32_C(-2107715246), INT32_C(  687192711),
                            INT32_C(  869285726), INT32_C(-1567378281), INT32_C(-1698583124), INT32_C( 1922904107)),
      simde_mm256_set_epi32(INT32_C(-1675267112), INT32_C(-2071980488), INT32_C(  114736266), INT32_C(-1214098866),
                            INT32_C( -977210232), INT32_C(  971766537), INT32_C(-1862490384), INT32_C(  337770462)) },
    { simde_mm256_set_epi32(INT32_C(-1141523047), INT32_C(   75738515), INT32_C( 2021274638), INT32_C(-1605111533),
                            INT32_C(  702401071), INT32_C(  991513903), INT32_C( 1097525967), INT32_C(   12869194)),
      simde_mm256_set_epi32(INT32_C( -236040676), INT32_C(-1937249843), INT32_C( -759698458), INT32_C( 1440000042),
                            INT32_C(  500464056), INT32_C( 1039800065), INT32_C( 1696902588), INT32_C( 1988285066)),
      simde_mm256_set_epi32(INT32_C(-1142728004), INT32_C( 1660012983), INT32_C(  496344724), INT32_C( 1454769438),
                            INT32_C(-1377466168), INT32_C(  334399023), INT32_C(  -78753020), INT32_C( 2093451236)) },
    { simde_mm256_set_epi32(INT32_C(  135806382), INT32_C( 1225419686), INT32_C(-1943331695), INT32_C( -184770167),
                            INT32_C( -868496558), INT32_C(-1570632013), INT32_C(-1946534455), INT32_C(  456616503)),
      simde_mm256_set_epi32(INT32_C( 1379577055), INT32_C( -512074604), INT32_C(-1891332670), INT32_C( 1111532874),
                            INT32_C(-1428646370), INT32_C(  907679144), INT32_C(  -10646910), INT32_C( 1881159279)),
      simde_mm256_set_epi32(INT32_C(  200974994), INT32_C(  735175672), INT32_C(-1930722590), INT32_C( 1499725466),
                            INT32_C( -111943780), INT32_C( -608414600), INT32_C(-1553548782), INT32_C( -733042727)) },
    { simde_mm256_set_epi32(INT32_C( 1387302426), INT32_C( -589733281), INT32_C(-1148378464), INT32_C(-1369430370),
                            INT32_C(   64719355), INT32_C( 1048033330), INT32_C( 1019366599), INT32_C(-1908464696)),
      simde_mm256_set_epi32(INT32_C( -277676972), INT32_C( -370073323), INT32_C( -963247981), INT32_C(  258227968),
                            INT32_C( 1397531888), INT32_C(   91286530), INT32_C(  441718636), INT32_C(  180722050)),
      simde_mm256_set_epi32(INT32_C( 2070165640), INT32_C(  905072843), INT32_C(-1144331808), INT32_C(   20505088),
                            INT32_C(-1715853488), INT32_C( 1788708964), INT32_C( 2116699380), INT32_C( 1364898704)) },
    { simde_mm256_set_epi32(INT32_C( 1766434696), INT32_C( -696400655), INT32_C( -511755431), INT32_C(  254323910),
                            INT32_C(-1407716551), INT32_C( -866109177), INT32_C( 1454483112), INT32_C(  830274169)),
      simde_mm256_set_epi32(INT32_C( -369468938), INT32_C(  827481876), INT32_C(  733336376), INT32_C( -709198563),
                            INT32_C(-1510456310), INT32_C( 2091700298), INT32_C(-1848267445), INT32_C( -214051693)),
      simde_mm256_set_epi32(INT32_C(  718619824), INT32_C(-1541746220), INT32_C(  447501944), INT32_C( -640401298),
                            INT32_C( 1908581434), INT32_C( -153474042), INT32_C( 1165187896), INT32_C( 1946486651)) },
    { simde_mm256_set_epi32(INT32_C(  551854829), INT32_C(-1524250015), INT32_C(  248754089), INT32_C( 2058702947),
                            INT32_C(  514801021), INT32_C(-2062150747), INT32_C( 1811376814), INT32_C( 1655721768)),
      simde_mm256_set_epi32(INT32_C(  -68197299), INT32_C( -108251896), INT32_C(  732768373), INT32_C(-1458146720),
                            INT32_C(-1648648199), INT32_C(  765297486), INT32_C( 1888788167), INT32_C( 1580342871)),
      simde_mm256_set_epi32(INT32_C(-1364421303), INT32_C(   35678216), INT32_C( -228151235), INT32_C(   88881952),
                            INT32_C( 1123185045), INT32_C( -436704442), INT32_C( 1841516866), INT32_C( 1493276312)) },
    { simde_mm256_set_epi32(INT32_C(-1004493292), INT32_C(   86312384), INT32_C(-1589794735), INT32_C( 1105371360),
                            INT32_C(  893767357), INT32_C( -562966901), INT32_C(  980757301), INT32_C( 2070176970)),
      simde_mm256_set_epi32(INT32_C( 1498513889), INT32_C( 1078743553), INT32_C(  911009242), INT32_C(  -31879959),
                            INT32_C(  673790886), INT32_C(-1100901508), INT32_C(  837704078), INT32_C( -540129822)),
      simde_mm256_set_epi32(INT32_C( 1421216660), INT32_C(-1778612800), INT32_C(-2078422534), INT32_C( -585176096),
                            INT32_C( 1177460110), INT32_C(-2056987564), INT32_C( 2034852966), INT32_C(  240610388)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_mullo_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_x_mm256_mullo_epu32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu32(UINT32_C(2258322750), UINT32_C(2861166599), UINT32_C(3174424968), UINT32_C(2016553993),
                              UINT32_C(2997181236), UINT32_C( 363976099), UINT32_C(1103728177), UINT32_C(2198010875)),
      simde_x_mm256_set_epu32(UINT32_C(1159500967), UINT32_C(2653222606), UINT32_C(3215542902), UINT32_C(2811870533),
                              UINT32_C(3872912803), UINT32_C(1788506759), UINT32_C(1042504603), UINT32_C(1249290459)),
      simde_x_mm256_set_epu32(UINT32_C(3692692338), UINT32_C(3668041634), UINT32_C( 617563312), UINT32_C(2691269485),
                              UINT32_C(1346229788), UINT32_C(2750955253), UINT32_C(3376906923), UINT32_C(3164358585)) },
    { simde_x_mm256_set_epu32(UINT32_C(2548089923), UINT32_C(1352192202), UINT32_C(3738306426), UINT32_C( 536787617),
                              UINT32_C( 525476003), UINT32_C(3633025093), UINT32_C(1784195446), UINT32_C(1453616288)),
      simde_x_mm256_set_epu32(UINT32_C(1186306915), UINT32_C( 779102467), UINT32_C(1127274760), UINT32_C(2603214034),
                              UINT32_C( 521716141), UINT32_C(3878310832), UINT32_C(2866604767), UINT32_C(3646073150)),
      simde_x_mm256_set_epu32(UINT32_C(3894376169), UINT32_C( 261803102), UINT32_C(1405527504), UINT32_C(2079685650),
                              UINT32_C(3781357863), UINT32_C( 235638384), UINT32_C(2877913546), UINT32_C(4073189056)) },
    { simde_x_mm256_set_epu32(UINT32_C(1846808532), UINT32_C(4103184512), UINT32_C(1968463192), UINT32_C(1782167042),
                              UINT32_C(2474531325), UINT32_C( 998377243), UINT32_C(  44320792), UINT32_C( 386122774)),
      simde_x_mm256_set_epu32(UINT32_C(3246150743), UINT32_C(1171028203), UINT32_C(3269727308), UINT32_C(1281786774),
                              UINT32_C(4190872936), UINT32_C( 118349934), UINT32_C(3463597682), UINT32_C(  47529222)),
      simde_x_mm256_set_epu32(UINT32_C(2092101900), UINT32_C( 720000384), UINT32_C(1052153376), UINT32_C(3804212012),
                              UINT32_C(3421741000), UINT32_C(1903537562), UINT32_C(2781833904), UINT32_C(3256735364)) },
    { simde_x_mm256_set_epu32(UINT32_C(4276815888), UINT32_C(1336815165), UINT32_C( 961795267), UINT32_C( 314541168),
                              UINT32_C(2077700186), UINT32_C(2373614491), UINT32_C(2478361141), UINT32_C(3586675075)),
      simde_x_mm256_set_epu32(UINT32_C(1354372236), UINT32_C(3218632640), UINT32_C(3308884037), UINT32_C( 276466624),
                              UINT32_C(1857596357), UINT32_C(2760506734), UINT32_C( 503740282), UINT32_C(1124719285)),
      simde_x_mm256_set_epu32(UINT32_C(2697570496), UINT32_C(3475448000), UINT32_C( 162597007), UINT32_C( 703341568),
                              UINT32_C(1989432130), UINT32_C(1220299674), UINT32_C(1062010946), UINT32_C(3802541983)) },
    { simde_x_mm256_set_epu32(UINT32_C(1408536206), UINT32_C( 904931218), UINT32_C(3779421746), UINT32_C( 832565776),
                              UINT32_C(  37616047), UINT32_C(1134646772), UINT32_C(3784150216), UINT32_C(1429477789)),
      simde_x_mm256_set_epu32(UINT32_C( 961414585), UINT32_C(1502350843), UINT32_C(3315915686), UINT32_C(2253735990),
                              UINT32_C(2519475515), UINT32_C( 980432319), UINT32_C( 560915899), UINT32_C(2488230114)),
      simde_x_mm256_set_epu32(UINT32_C(2143799966), UINT32_C(2052142630), UINT32_C(1668614764), UINT32_C( 857837408),
                              UINT32_C(3983726165), UINT32_C(3203048716), UINT32_C(2806300184), UINT32_C(1829534874)) },
    { simde_x_mm256_set_epu32(UINT32_C( 574695235), UINT32_C( 304227009), UINT32_C(1668479769), UINT32_C(3452443080),
                              UINT32_C(3762070562), UINT32_C(2046023294), UINT32_C( 475815618), UINT32_C( 179358113)),
      simde_x_mm256_set_epu32(UINT32_C( 260558052), UINT32_C(3663947713), UINT32_C(4165399884), UINT32_C(1745062207),
                              UINT32_C(2120455131), UINT32_C(4011446154), UINT32_C(1023802013), UINT32_C( 827938078)),
      simde_x_mm256_set_epu32(UINT32_C( 245717932), UINT32_C( 844814977), UINT32_C(1900509292), UINT32_C(1197629496),
                              UINT32_C(3678793494), UINT32_C(2083488236), UINT32_C(2902036730), UINT32_C(3718452702)) },
    { simde_x_mm256_set_epu32(UINT32_C(1617715092), UINT32_C(2193069624), UINT32_C( 650454244), UINT32_C(2538964293),
                              UINT32_C(1532176753), UINT32_C(3080365125), UINT32_C(2879482590), UINT32_C(3982850403)),
      simde_x_mm256_set_epu32(UINT32_C(1884881854), UINT32_C(3265025687), UINT32_C(1041888717), UINT32_C(1140822020),
                              UINT32_C(4094158163), UINT32_C( 967388894), UINT32_C(1291557283), UINT32_C(2562715165)),
      simde_x_mm256_set_epu32(UINT32_C(3489770456), UINT32_C(3235932936), UINT32_C(2322644628), UINT32_C(1191699732),
                              UINT32_C(3592389795), UINT32_C(3492534742), UINT32_C(1884272986), UINT32_C(2486553143)) },
    { simde_x_mm256_set_epu32(UINT32_C(3590735797), UINT32_C(1425583171), UINT32_C( 334293210), UINT32_C( 791405491),
                              UINT32_C(1931435573), UINT32_C( 591893589), UINT32_C(1994069612), UINT32_C( 354132544)),
      simde_x_mm256_set_epu32(UINT32_C(2748385695), UINT32_C( 731899882), UINT32_C(1077607699), UINT32_C(1910108083),
                              UINT32_C(1267460605), UINT32_C(2091709757), UINT32_C(2754232525), UINT32_C(2186524445)),
      simde_x_mm256_set_epu32(UINT32_C(4029736555), UINT32_C(1297803838), UINT32_C(3726162478), UINT32_C(4211036969),
                              UINT32_C(   8738657), UINT32_C(3189748033), UINT32_C(3502503036), UINT32_C(1071456576)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_x_mm256_mullo_epu32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_or_si256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi64x(INT64_C( -801044498564576659), INT64_C(-2909946603020252481),
                             INT64_C( 5958340648204315976), INT64_C( 8713768337389103061)),
      simde_mm256_set_epi64x(INT64_C( 9176724763357309327), INT64_C( 4054644920102546891),
                             INT64_C( 4782244109117166481), INT64_C( 3831721763102663031)),
      simde_mm256_set_epi64x(INT64_C(   -1584611843413009), INT64_C(   -9571266241499137),
                             INT64_C( 5980208482798747609), INT64_C( 9074056875634305015)) },
    { simde_mm256_set_epi64x(INT64_C( 1317517346722662736), INT64_C( 8192878697228400830),
                             INT64_C(-3537455209908178968), INT64_C(-2713816682012121382)),
      simde_mm256_set_epi64x(INT64_C( 7957277620212510994), INT64_C( -737217900053345188),
                             INT64_C( 8353910688937076237), INT64_C(  240232259721200655)),
      simde_mm256_set_epi64x(INT64_C( 9110201366055221586), INT64_C( -723122418076615426),
                             INT64_C(   -4662226728259603), INT64_C(-2641361182872049953)) },
    { simde_mm256_set_epi64x(INT64_C(-1030545204507091849), INT64_C(-1542600680052722313),
                             INT64_C(-7648307982573512602), INT64_C( 5973019580240685616)),
      simde_mm256_set_epi64x(INT64_C(-6290470397500953523), INT64_C( 8109710997204180941),
                             INT64_C( 5917924879433877736), INT64_C( 8502004464391034004)),
      simde_mm256_set_epi64x(INT64_C( -453800777020900225), INT64_C( -387309568701530113),
                             INT64_C(-2883445622328010514), INT64_C( 8646205037023002292)) },
    { simde_mm256_set_epi64x(INT64_C(-2990334454120409171), INT64_C(-3220201474370514905),
                             INT64_C( 8548083516217107397), INT64_C( 1251663319653874101)),
      simde_mm256_set_epi64x(INT64_C(-2621282330722334206), INT64_C( 5235652619773460077),
                             INT64_C(-8007055325654862889), INT64_C( 4775726838041815408)),
      simde_mm256_set_epi64x(INT64_C(-2333014432620503121), INT64_C(-2598629952962888081),
                             INT64_C( -656965087403232297), INT64_C( 6007467006593006069)) },
    { simde_mm256_set_epi64x(INT64_C( 1315645066342648861), INT64_C( 3754004658427516786),
                             INT64_C(-7880307939890805097), INT64_C(-5701204371115270443)),
      simde_mm256_set_epi64x(INT64_C( 4172903126396830914), INT64_C( -493154668521044871),
                             INT64_C(-2309759438976524777), INT64_C(-1689539225349388212)),
      simde_mm256_set_epi64x(INT64_C( 4317612287190453471), INT64_C( -198167755223214213),
                             INT64_C(-2309335405310321001), INT64_C( -509524860864094499)) },
    { simde_mm256_set_epi64x(INT64_C(-7748112100043814155), INT64_C( 5814291251258484552),
                             INT64_C( 8569511450246080549), INT64_C(-3900190118960098388)),
      simde_mm256_set_epi64x(INT64_C( 9015646334468450927), INT64_C(  883710405382046595),
                             INT64_C( 2743428167896968049), INT64_C(-6564603084509542605)),
      simde_mm256_set_epi64x(INT64_C( -180373114503566593), INT64_C( 6697859792108349387),
                             INT64_C( 8574745263641455477), INT64_C(-1297039137678485569)) },
    { simde_mm256_set_epi64x(INT64_C( 5980675563351081308), INT64_C( 7108230643859206772),
                             INT64_C(-7185068082285956895), INT64_C(-5748801677096031915)),
      simde_mm256_set_epi64x(INT64_C(-1209090942768865396), INT64_C(-7402713372895048445),
                             INT64_C(  -24471728257632960), INT64_C( 3473093230644658861)),
      simde_mm256_set_epi64x(INT64_C(     -14577395916836), INT64_C( -295610862468677769),
                             INT64_C(   -6315904187370015), INT64_C(-5748026933373309955)) },
    { simde_mm256_set_epi64x(INT64_C( 3669045510431781214), INT64_C(-8656850301840548621),
                             INT64_C(-5639311717074453893), INT64_C(-8609899897096571068)),
      simde_mm256_set_epi64x(INT64_C(-5772405160554679118), INT64_C( 8581290868842963452),
                             INT64_C(-7553387725647900846), INT64_C( 1768046205102779153)),
      simde_mm256_set_epi64x(INT64_C(-4616368323934308866), INT64_C( -585750987759698433),
                             INT64_C(-5206728520596308101), INT64_C(-7454726563416572075)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_or_si256(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_packs_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[16];
    const int16_t b[16];
    const int8_t r[32];
  } test_vec[] = {
    { {  INT16_C(   217),  INT16_C(   133),  INT16_C(    35),  INT16_C(   164),  INT16_C(    69),  INT16_C(   101),  INT16_C(   149),  INT16_C(   103),
         INT16_C(    41),  INT16_C(    56),  INT16_C(    76),  INT16_C(   183),  INT16_C(    83),  INT16_C(   188),  INT16_C(   132),  INT16_C(   172) },
      {  INT16_C(   172),  INT16_C(   254),  INT16_C(   196),  INT16_C(    19),  INT16_C(   250),  INT16_C(   196),  INT16_C(     4),  INT16_C(   186),
         INT16_C(   183),  INT16_C(    52),  INT16_C(   215),  INT16_C(    95),  INT16_C(   224),  INT16_C(    86),  INT16_C(   229),  INT16_C(   194) },
      {      INT8_MAX,      INT8_MAX,  INT8_C(  35),      INT8_MAX,  INT8_C(  69),  INT8_C( 101),      INT8_MAX,  INT8_C( 103),
             INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C(  19),      INT8_MAX,      INT8_MAX,  INT8_C(   4),      INT8_MAX,
         INT8_C(  41),  INT8_C(  56),  INT8_C(  76),      INT8_MAX,  INT8_C(  83),      INT8_MAX,      INT8_MAX,      INT8_MAX,
             INT8_MAX,  INT8_C(  52),      INT8_MAX,  INT8_C(  95),      INT8_MAX,  INT8_C(  86),      INT8_MAX,      INT8_MAX } },
    { {  INT16_C(    51),  INT16_C(    97),  INT16_C(    99),  INT16_C(     1),  INT16_C(   236),  INT16_C(   236),  INT16_C(   241),  INT16_C(    30),
         INT16_C(   153),  INT16_C(   173),  INT16_C(    62),  INT16_C(   155),  INT16_C(    40),  INT16_C(   131),  INT16_C(   243),  INT16_C(   160) },
      {  INT16_C(     5),  INT16_C(    30),  INT16_C(   118),  INT16_C(   197),  INT16_C(   229),  INT16_C(    83),  INT16_C(    88),  INT16_C(   128),
         INT16_C(   195),  INT16_C(    97),  INT16_C(    57),  INT16_C(    32),  INT16_C(   238),  INT16_C(   112),  INT16_C(   232),  INT16_C(     8) },
      {  INT8_C(  51),  INT8_C(  97),  INT8_C(  99),  INT8_C(   1),      INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C(  30),
         INT8_C(   5),  INT8_C(  30),  INT8_C( 118),      INT8_MAX,      INT8_MAX,  INT8_C(  83),  INT8_C(  88),      INT8_MAX,
             INT8_MAX,      INT8_MAX,  INT8_C(  62),      INT8_MAX,  INT8_C(  40),      INT8_MAX,      INT8_MAX,      INT8_MAX,
             INT8_MAX,  INT8_C(  97),  INT8_C(  57),  INT8_C(  32),      INT8_MAX,  INT8_C( 112),      INT8_MAX,  INT8_C(   8) } },
    { {  INT16_C(    17),  INT16_C(    86),  INT16_C(    70),  INT16_C(   234),  INT16_C(   205),  INT16_C(     2),  INT16_C(   174),  INT16_C(    22),
         INT16_C(   175),  INT16_C(   115),  INT16_C(   116),  INT16_C(    75),  INT16_C(    54),  INT16_C(    69),  INT16_C(   203),  INT16_C(    11) },
      {  INT16_C(   116),  INT16_C(   100),  INT16_C(   125),  INT16_C(   229),  INT16_C(   139),  INT16_C(   111),  INT16_C(   106),  INT16_C(   170),
         INT16_C(   252),  INT16_C(     3),  INT16_C(   176),  INT16_C(   212),  INT16_C(     9),  INT16_C(     4),  INT16_C(   102),  INT16_C(   176) },
      {  INT8_C(  17),  INT8_C(  86),  INT8_C(  70),      INT8_MAX,      INT8_MAX,  INT8_C(   2),      INT8_MAX,  INT8_C(  22),
         INT8_C( 116),  INT8_C( 100),  INT8_C( 125),      INT8_MAX,      INT8_MAX,  INT8_C( 111),  INT8_C( 106),      INT8_MAX,
             INT8_MAX,  INT8_C( 115),  INT8_C( 116),  INT8_C(  75),  INT8_C(  54),  INT8_C(  69),      INT8_MAX,  INT8_C(  11),
             INT8_MAX,  INT8_C(   3),      INT8_MAX,      INT8_MAX,  INT8_C(   9),  INT8_C(   4),  INT8_C( 102),      INT8_MAX } },
    { {  INT16_C(   113),  INT16_C(   148),  INT16_C(    98),  INT16_C(    56),  INT16_C(    98),  INT16_C(    38),  INT16_C(    44),  INT16_C(   230),
         INT16_C(   238),  INT16_C(   154),  INT16_C(    55),  INT16_C(   133),  INT16_C(   135),  INT16_C(    20),  INT16_C(   154),  INT16_C(   200) },
      {  INT16_C(   217),  INT16_C(   250),  INT16_C(   214),  INT16_C(    41),  INT16_C(   218),  INT16_C(     5),  INT16_C(    33),  INT16_C(    47),
         INT16_C(   213),  INT16_C(   174),  INT16_C(    55),  INT16_C(    77),  INT16_C(   190),  INT16_C(   172),  INT16_C(    38),  INT16_C(    99) },
      {  INT8_C( 113),      INT8_MAX,  INT8_C(  98),  INT8_C(  56),  INT8_C(  98),  INT8_C(  38),  INT8_C(  44),      INT8_MAX,
             INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C(  41),      INT8_MAX,  INT8_C(   5),  INT8_C(  33),  INT8_C(  47),
             INT8_MAX,      INT8_MAX,  INT8_C(  55),      INT8_MAX,      INT8_MAX,  INT8_C(  20),      INT8_MAX,      INT8_MAX,
             INT8_MAX,      INT8_MAX,  INT8_C(  55),  INT8_C(  77),      INT8_MAX,      INT8_MAX,  INT8_C(  38),  INT8_C(  99) } },
    { {  INT16_C(   208),  INT16_C(    60),  INT16_C(   144),  INT16_C(   223),  INT16_C(   181),  INT16_C(   112),  INT16_C(   208),  INT16_C(   230),
         INT16_C(   105),  INT16_C(   177),  INT16_C(   200),  INT16_C(    95),  INT16_C(    96),  INT16_C(   222),  INT16_C(   127),  INT16_C(   134) },
      {  INT16_C(   159),  INT16_C(   247),  INT16_C(    40),  INT16_C(   153),  INT16_C(   187),  INT16_C(   180),  INT16_C(   170),  INT16_C(    48),
         INT16_C(    46),  INT16_C(   180),  INT16_C(   224),  INT16_C(   125),  INT16_C(    31),  INT16_C(   198),  INT16_C(   158),  INT16_C(   239) },
      {      INT8_MAX,  INT8_C(  60),      INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C( 112),      INT8_MAX,      INT8_MAX,
             INT8_MAX,      INT8_MAX,  INT8_C(  40),      INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C(  48),
         INT8_C( 105),      INT8_MAX,      INT8_MAX,  INT8_C(  95),  INT8_C(  96),      INT8_MAX,      INT8_MAX,      INT8_MAX,
         INT8_C(  46),      INT8_MAX,      INT8_MAX,  INT8_C( 125),  INT8_C(  31),      INT8_MAX,      INT8_MAX,      INT8_MAX } },
    { {  INT16_C(    15),  INT16_C(   109),  INT16_C(   188),  INT16_C(    21),  INT16_C(    16),  INT16_C(     2),  INT16_C(    99),  INT16_C(   206),
         INT16_C(    20),  INT16_C(   136),  INT16_C(   150),  INT16_C(    52),  INT16_C(    98),  INT16_C(    85),  INT16_C(    71),  INT16_C(    61) },
      {  INT16_C(    42),  INT16_C(   142),  INT16_C(   177),  INT16_C(    94),  INT16_C(   108),  INT16_C(   124),  INT16_C(   147),  INT16_C(    97),
         INT16_C(   204),  INT16_C(   155),  INT16_C(   239),  INT16_C(    25),  INT16_C(   201),  INT16_C(    81),  INT16_C(   178),  INT16_C(   104) },
      {  INT8_C(  15),  INT8_C( 109),      INT8_MAX,  INT8_C(  21),  INT8_C(  16),  INT8_C(   2),  INT8_C(  99),      INT8_MAX,
         INT8_C(  42),      INT8_MAX,      INT8_MAX,  INT8_C(  94),  INT8_C( 108),  INT8_C( 124),      INT8_MAX,  INT8_C(  97),
         INT8_C(  20),      INT8_MAX,      INT8_MAX,  INT8_C(  52),  INT8_C(  98),  INT8_C(  85),  INT8_C(  71),  INT8_C(  61),
             INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C(  25),      INT8_MAX,  INT8_C(  81),      INT8_MAX,  INT8_C( 104) } },
    { {  INT16_C(    57),  INT16_C(   194),  INT16_C(   154),  INT16_C(   172),  INT16_C(   129),  INT16_C(   214),  INT16_C(   114),  INT16_C(   187),
         INT16_C(    32),  INT16_C(   161),  INT16_C(    37),  INT16_C(    97),  INT16_C(    40),  INT16_C(   255),  INT16_C(    66),  INT16_C(   182) },
      {  INT16_C(    94),  INT16_C(   102),  INT16_C(   153),  INT16_C(   254),  INT16_C(    58),  INT16_C(    46),  INT16_C(    11),  INT16_C(   234),
         INT16_C(    63),  INT16_C(    60),  INT16_C(    70),  INT16_C(    83),  INT16_C(    80),  INT16_C(    73),  INT16_C(   186),  INT16_C(    14) },
      {  INT8_C(  57),      INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C( 114),      INT8_MAX,
         INT8_C(  94),  INT8_C( 102),      INT8_MAX,      INT8_MAX,  INT8_C(  58),  INT8_C(  46),  INT8_C(  11),      INT8_MAX,
         INT8_C(  32),      INT8_MAX,  INT8_C(  37),  INT8_C(  97),  INT8_C(  40),      INT8_MAX,  INT8_C(  66),      INT8_MAX,
         INT8_C(  63),  INT8_C(  60),  INT8_C(  70),  INT8_C(  83),  INT8_C(  80),  INT8_C(  73),      INT8_MAX,  INT8_C(  14) } },
    { {  INT16_C(   121),  INT16_C(    16),  INT16_C(   134),  INT16_C(    45),  INT16_C(   227),  INT16_C(   108),  INT16_C(    69),  INT16_C(    26),
         INT16_C(   226),  INT16_C(   233),  INT16_C(   244),  INT16_C(   152),  INT16_C(   142),  INT16_C(   215),  INT16_C(   225),  INT16_C(    96) },
      {  INT16_C(    89),  INT16_C(   109),  INT16_C(   127),  INT16_C(   159),  INT16_C(   245),  INT16_C(    80),  INT16_C(    97),  INT16_C(   191),
         INT16_C(   193),  INT16_C(   109),  INT16_C(   228),  INT16_C(   250),  INT16_C(   230),  INT16_C(   187),  INT16_C(   182),  INT16_C(    34) },
      {  INT8_C( 121),  INT8_C(  16),      INT8_MAX,  INT8_C(  45),      INT8_MAX,  INT8_C( 108),  INT8_C(  69),  INT8_C(  26),
         INT8_C(  89),  INT8_C( 109),      INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C(  80),  INT8_C(  97),      INT8_MAX,
             INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C(  96),
             INT8_MAX,  INT8_C( 109),      INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,      INT8_MAX,  INT8_C(  34) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_packs_epi16(a, b);
    simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_packs_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[8];
    const int32_t b[8];
    const int16_t r[16];
  } test_vec[] = {
    { {  INT32_C(       26989),  INT32_C(       24063),  INT32_C(       64705),  INT32_C(       36658),  INT32_C(       50142),  INT32_C(       39242),  INT32_C(        8518),  INT32_C(        3656) },
      {  INT32_C(       34423),  INT32_C(        1251),  INT32_C(       52992),  INT32_C(       41566),  INT32_C(       36965),  INT32_C(        1065),  INT32_C(        2853),  INT32_C(       14105) },
      {  INT16_C( 26989),  INT16_C( 24063),       INT16_MAX,       INT16_MAX,       INT16_MAX,  INT16_C(  1251),       INT16_MAX,       INT16_MAX,
              INT16_MAX,       INT16_MAX,  INT16_C(  8518),  INT16_C(  3656),       INT16_MAX,  INT16_C(  1065),  INT16_C(  2853),  INT16_C( 14105) } },
    { {  INT32_C(       23229),  INT32_C(       38238),  INT32_C(       64100),  INT32_C(       13468),  INT32_C(       43716),  INT32_C(       10670),  INT32_C(        7220),  INT32_C(       25683) },
      {  INT32_C(       59070),  INT32_C(       63100),  INT32_C(        3056),  INT32_C(       41535),  INT32_C(       15948),  INT32_C(       20583),  INT32_C(        3692),  INT32_C(       51058) },
      {  INT16_C( 23229),       INT16_MAX,       INT16_MAX,  INT16_C( 13468),       INT16_MAX,       INT16_MAX,  INT16_C(  3056),       INT16_MAX,
              INT16_MAX,  INT16_C( 10670),  INT16_C(  7220),  INT16_C( 25683),  INT16_C( 15948),  INT16_C( 20583),  INT16_C(  3692),       INT16_MAX } },
    { {  INT32_C(       33198),  INT32_C(       51063),  INT32_C(       44498),  INT32_C(       33103),  INT32_C(        2239),  INT32_C(       25432),  INT32_C(       43889),  INT32_C(       21875) },
      {  INT32_C(       25046),  INT32_C(       21800),  INT32_C(       43267),  INT32_C(        8746),  INT32_C(       33578),  INT32_C(       54246),  INT32_C(       52094),  INT32_C(       20512) },
      {       INT16_MAX,       INT16_MAX,       INT16_MAX,       INT16_MAX,  INT16_C( 25046),  INT16_C( 21800),       INT16_MAX,  INT16_C(  8746),
         INT16_C(  2239),  INT16_C( 25432),       INT16_MAX,  INT16_C( 21875),       INT16_MAX,       INT16_MAX,       INT16_MAX,  INT16_C( 20512) } },
    { {  INT32_C(       23985),  INT32_C(       64179),  INT32_C(       57507),  INT32_C(       62979),  INT32_C(       51066),  INT32_C(       63386),  INT32_C(       62658),  INT32_C(        7493) },
      {  INT32_C(        8058),  INT32_C(       42009),  INT32_C(       60548),  INT32_C(       16866),  INT32_C(       25865),  INT32_C(       30044),  INT32_C(       51050),  INT32_C(       56292) },
      {  INT16_C( 23985),       INT16_MAX,       INT16_MAX,       INT16_MAX,  INT16_C(  8058),       INT16_MAX,       INT16_MAX,  INT16_C( 16866),
              INT16_MAX,       INT16_MAX,       INT16_MAX,  INT16_C(  7493),  INT16_C( 25865),  INT16_C( 30044),       INT16_MAX,       INT16_MAX } },
    { {  INT32_C(       30202),  INT32_C(       28441),  INT32_C(       23387),  INT32_C(       55965),  INT32_C(       22335),  INT32_C(        1741),  INT32_C(       48077),  INT32_C(       35735) },
      {  INT32_C(       39681),  INT32_C(       30219),  INT32_C(       57042),  INT32_C(       16312),  INT32_C(       24214),  INT32_C(       19812),  INT32_C(       32777),  INT32_C(       61964) },
      {  INT16_C( 30202),  INT16_C( 28441),  INT16_C( 23387),       INT16_MAX,       INT16_MAX,  INT16_C( 30219),       INT16_MAX,  INT16_C( 16312),
         INT16_C( 22335),  INT16_C(  1741),       INT16_MAX,       INT16_MAX,  INT16_C( 24214),  INT16_C( 19812),       INT16_MAX,       INT16_MAX } },
    { {  INT32_C(       55181),  INT32_C(       57421),  INT32_C(       41918),  INT32_C(       41954),  INT32_C(         514),  INT32_C(       30543),  INT32_C(       31479),  INT32_C(       10860) },
      {  INT32_C(       14337),  INT32_C(       37144),  INT32_C(       64565),  INT32_C(       48032),  INT32_C(       28093),  INT32_C(       41188),  INT32_C(       23835),  INT32_C(       61319) },
      {       INT16_MAX,       INT16_MAX,       INT16_MAX,       INT16_MAX,  INT16_C( 14337),       INT16_MAX,       INT16_MAX,       INT16_MAX,
         INT16_C(   514),  INT16_C( 30543),  INT16_C( 31479),  INT16_C( 10860),  INT16_C( 28093),       INT16_MAX,  INT16_C( 23835),       INT16_MAX } },
    { {  INT32_C(        5159),  INT32_C(       17829),  INT32_C(       25153),  INT32_C(       33309),  INT32_C(       36079),  INT32_C(       19244),  INT32_C(       36521),  INT32_C(       20605) },
      {  INT32_C(       36964),  INT32_C(       64213),  INT32_C(       54876),  INT32_C(       31577),  INT32_C(       14599),  INT32_C(       52101),  INT32_C(       19033),  INT32_C(        6043) },
      {  INT16_C(  5159),  INT16_C( 17829),  INT16_C( 25153),       INT16_MAX,       INT16_MAX,       INT16_MAX,       INT16_MAX,  INT16_C( 31577),
              INT16_MAX,  INT16_C( 19244),       INT16_MAX,  INT16_C( 20605),  INT16_C( 14599),       INT16_MAX,  INT16_C( 19033),  INT16_C(  6043) } },
    { {  INT32_C(       24743),  INT32_C(       61018),  INT32_C(       35780),  INT32_C(       33286),  INT32_C(       33467),  INT32_C(       48461),  INT32_C(       52488),  INT32_C(       63716) },
      {  INT32_C(       44120),  INT32_C(       39834),  INT32_C(       38694),  INT32_C(       57881),  INT32_C(       32100),  INT32_C(       33594),  INT32_C(       54096),  INT32_C(       35019) },
      {  INT16_C( 24743),       INT16_MAX,       INT16_MAX,       INT16_MAX,       INT16_MAX,       INT16_MAX,       INT16_MAX,       INT16_MAX,
              INT16_MAX,       INT16_MAX,       INT16_MAX,       INT16_MAX,  INT16_C( 32100),       INT16_MAX,       INT16_MAX,       INT16_MAX } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_packs_epi32(a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_packus_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[16];
    const int16_t b[16];
    const uint8_t r[32];
  } test_vec[] = {
    { {  INT16_C(  1278),  INT16_C( 16487),  INT16_C( 23908), -INT16_C( 22641), -INT16_C( 20243),  INT16_C( 19138),  INT16_C(  5855), -INT16_C( 25236),
         INT16_C( 27106), -INT16_C( 20692), -INT16_C( 28351), -INT16_C( 14963), -INT16_C( 25251), -INT16_C(  8084), -INT16_C(  9791), -INT16_C( 16413) },
      {  INT16_C(    75),  INT16_C(    65),  INT16_C(   142),  INT16_C(   150),  INT16_C(   171),  INT16_C(    30),  INT16_C(    77),  INT16_C(   163),
         INT16_C(   231),  INT16_C(   247),  INT16_C(   224),  INT16_C(   213),  INT16_C(    41),  INT16_C(    62),  INT16_C(   153),  INT16_C(   223) },
      {    UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0),
        UINT8_C( 75), UINT8_C( 65), UINT8_C(142), UINT8_C(150), UINT8_C(171), UINT8_C( 30), UINT8_C( 77), UINT8_C(163),
           UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(231), UINT8_C(247), UINT8_C(224), UINT8_C(213), UINT8_C( 41), UINT8_C( 62), UINT8_C(153), UINT8_C(223) } },
    { { -INT16_C(   796), -INT16_C( 29664),  INT16_C(  2443), -INT16_C( 14046),  INT16_C(   948),  INT16_C( 30183), -INT16_C( 23984),  INT16_C(  1561),
         INT16_C( 27529),  INT16_C(   765), -INT16_C( 17845), -INT16_C( 13865), -INT16_C( 29213), -INT16_C(  6905),  INT16_C(  1318),  INT16_C(  2756) },
      {  INT16_C(   228),  INT16_C(   140),  INT16_C(   185),  INT16_C(   161),  INT16_C(    61),  INT16_C(    12),  INT16_C(    48),  INT16_C(   105),
         INT16_C(    15),  INT16_C(   231),  INT16_C(    67),  INT16_C(   172),  INT16_C(   183),  INT16_C(   246),  INT16_C(    85),  INT16_C(   190) },
      { UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,
        UINT8_C(228), UINT8_C(140), UINT8_C(185), UINT8_C(161), UINT8_C( 61), UINT8_C( 12), UINT8_C( 48), UINT8_C(105),
           UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,
        UINT8_C( 15), UINT8_C(231), UINT8_C( 67), UINT8_C(172), UINT8_C(183), UINT8_C(246), UINT8_C( 85), UINT8_C(190) } },
    { { -INT16_C( 27078),  INT16_C( 10058), -INT16_C( 24497),  INT16_C(  3017), -INT16_C(  7970), -INT16_C( 16873),  INT16_C( 10512), -INT16_C( 21721),
        -INT16_C( 27847),  INT16_C(   658),  INT16_C( 17110), -INT16_C( 22865),  INT16_C( 16634), -INT16_C( 18788), -INT16_C( 25450), -INT16_C( 12172) },
      {  INT16_C(   191),  INT16_C(   130),  INT16_C(   192),  INT16_C(    61),  INT16_C(   165),  INT16_C(   176),  INT16_C(    35),  INT16_C(     7),
         INT16_C(   238),  INT16_C(   140),  INT16_C(   185),  INT16_C(    43),  INT16_C(   206),  INT16_C(   143),  INT16_C(    86),  INT16_C(   156) },
      { UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),
        UINT8_C(191), UINT8_C(130), UINT8_C(192), UINT8_C( 61), UINT8_C(165), UINT8_C(176), UINT8_C( 35), UINT8_C(  7),
        UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(238), UINT8_C(140), UINT8_C(185), UINT8_C( 43), UINT8_C(206), UINT8_C(143), UINT8_C( 86), UINT8_C(156) } },
    { {  INT16_C( 22293),  INT16_C( 29726), -INT16_C( 21481), -INT16_C( 18254), -INT16_C( 21167),  INT16_C(  8040), -INT16_C( 15152), -INT16_C( 31193),
         INT16_C( 12723), -INT16_C(  7150),  INT16_C( 17642), -INT16_C(  7409), -INT16_C(  4078),  INT16_C( 31859), -INT16_C( 11706),  INT16_C( 23321) },
      {  INT16_C(    55),  INT16_C(    65),  INT16_C(   130),  INT16_C(    52),  INT16_C(    97),  INT16_C(     0),  INT16_C(   123),  INT16_C(   217),
         INT16_C(   153),  INT16_C(   150),  INT16_C(   204),  INT16_C(   240),  INT16_C(   236),  INT16_C(     3),  INT16_C(   133),  INT16_C(   232) },
      {    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),
        UINT8_C( 55), UINT8_C( 65), UINT8_C(130), UINT8_C( 52), UINT8_C( 97), UINT8_C(  0), UINT8_C(123), UINT8_C(217),
           UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,
        UINT8_C(153), UINT8_C(150), UINT8_C(204), UINT8_C(240), UINT8_C(236), UINT8_C(  3), UINT8_C(133), UINT8_C(232) } },
    { {  INT16_C( 11965), -INT16_C( 24535),  INT16_C(  8880), -INT16_C(  7979),  INT16_C( 10628), -INT16_C( 21792),  INT16_C( 26276),  INT16_C( 20611),
         INT16_C( 16639), -INT16_C(  8730),  INT16_C( 24332), -INT16_C( 14131),  INT16_C( 14668),  INT16_C(  3019),  INT16_C( 10943),  INT16_C( 31987) },
      {  INT16_C(    29),  INT16_C(     9),  INT16_C(   241),  INT16_C(   195),  INT16_C(   201),  INT16_C(   190),  INT16_C(   240),  INT16_C(    47),
         INT16_C(   244),  INT16_C(    60),  INT16_C(   217),  INT16_C(   160),  INT16_C(   208),  INT16_C(   209),  INT16_C(   158),  INT16_C(    83) },
      {    UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,
        UINT8_C( 29), UINT8_C(  9), UINT8_C(241), UINT8_C(195), UINT8_C(201), UINT8_C(190), UINT8_C(240), UINT8_C( 47),
           UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX,    UINT8_MAX,
        UINT8_C(244), UINT8_C( 60), UINT8_C(217), UINT8_C(160), UINT8_C(208), UINT8_C(209), UINT8_C(158), UINT8_C( 83) } },
    { {  INT16_C( 27323), -INT16_C(  1188),  INT16_C( 17755),  INT16_C( 30398),  INT16_C( 11278),  INT16_C( 15668),  INT16_C( 17180),  INT16_C( 19820),
         INT16_C( 30775), -INT16_C( 29815), -INT16_C( 29103),  INT16_C( 25643), -INT16_C( 10657),  INT16_C( 22837), -INT16_C( 31883),  INT16_C( 12460) },
      {  INT16_C(     8),  INT16_C(    72),  INT16_C(   234),  INT16_C(    91),  INT16_C(   243),  INT16_C(    50),  INT16_C(     5),  INT16_C(   109),
         INT16_C(     9),  INT16_C(   207),  INT16_C(    36),  INT16_C(   246),  INT16_C(   105),  INT16_C(   112),  INT16_C(   252),  INT16_C(   217) },
      {    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX,    UINT8_MAX,    UINT8_MAX,    UINT8_MAX,
        UINT8_C(  8), UINT8_C( 72), UINT8_C(234), UINT8_C( 91), UINT8_C(243), UINT8_C( 50), UINT8_C(  5), UINT8_C(109),
           UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,
        UINT8_C(  9), UINT8_C(207), UINT8_C( 36), UINT8_C(246), UINT8_C(105), UINT8_C(112), UINT8_C(252), UINT8_C(217) } },
    { { -INT16_C( 13307),  INT16_C( 21025), -INT16_C(  8010), -INT16_C( 13138),  INT16_C( 18387),  INT16_C(  2558),  INT16_C( 32332), -INT16_C( 13706),
         INT16_C( 28551),  INT16_C(  7834), -INT16_C( 12908), -INT16_C( 28907),  INT16_C( 25910),  INT16_C(  8959), -INT16_C( 24735),  INT16_C( 26363) },
      {  INT16_C(    29),  INT16_C(    33),  INT16_C(   103),  INT16_C(   208),  INT16_C(   236),  INT16_C(   250),  INT16_C(    79),  INT16_C(   241),
         INT16_C(    95),  INT16_C(    83),  INT16_C(    36),  INT16_C(    99),  INT16_C(   225),  INT16_C(   235),  INT16_C(   129),  INT16_C(   236) },
      { UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(  0),
        UINT8_C( 29), UINT8_C( 33), UINT8_C(103), UINT8_C(208), UINT8_C(236), UINT8_C(250), UINT8_C( 79), UINT8_C(241),
           UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,
        UINT8_C( 95), UINT8_C( 83), UINT8_C( 36), UINT8_C( 99), UINT8_C(225), UINT8_C(235), UINT8_C(129), UINT8_C(236) } },
    { {  INT16_C(  2718), -INT16_C( 25843), -INT16_C(  1167),  INT16_C(  8043),  INT16_C( 17639),  INT16_C( 20762), -INT16_C(  8301),  INT16_C( 21058),
         INT16_C( 20798),  INT16_C( 27301), -INT16_C( 30858), -INT16_C(    51),  INT16_C( 21352), -INT16_C(  5654),  INT16_C( 15572),  INT16_C( 29397) },
      {  INT16_C(   226),  INT16_C(   184),  INT16_C(   120),  INT16_C(   196),  INT16_C(   241),  INT16_C(    79),  INT16_C(    87),  INT16_C(    14),
         INT16_C(    71),  INT16_C(    31),  INT16_C(    70),  INT16_C(    55),  INT16_C(     9),  INT16_C(   109),  INT16_C(   245),  INT16_C(   139) },
      {    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,
        UINT8_C(226), UINT8_C(184), UINT8_C(120), UINT8_C(196), UINT8_C(241), UINT8_C( 79), UINT8_C( 87), UINT8_C( 14),
           UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,
        UINT8_C( 71), UINT8_C( 31), UINT8_C( 70), UINT8_C( 55), UINT8_C(  9), UINT8_C(109), UINT8_C(245), UINT8_C(139) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_packus_epi16(a, b);
    simde_test_x86_assert_equal_u8x32(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_packus_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[8];
    const int32_t b[8];
    const uint16_t r[16];
  } test_vec[] = {
    { {  INT32_C(       56303),  INT32_C(       60373),  INT32_C(       50246),  INT32_C(        4724),  INT32_C(       15585),  INT32_C(       12721),  INT32_C(       31788),  INT32_C(       19403) },
      { -INT32_C(  1490634197),  INT32_C(  1234369705),  INT32_C(   638441621),  INT32_C(  2100920576),  INT32_C(    12196591),  INT32_C(  1060203433),  INT32_C(   196828782),  INT32_C(  1750501180) },
      { UINT16_C(56303), UINT16_C(60373), UINT16_C(50246), UINT16_C( 4724), UINT16_C(    0),      UINT16_MAX,      UINT16_MAX,      UINT16_MAX,
        UINT16_C(15585), UINT16_C(12721), UINT16_C(31788), UINT16_C(19403),      UINT16_MAX,      UINT16_MAX,      UINT16_MAX,      UINT16_MAX } },
    { {  INT32_C(       59919),  INT32_C(        3636),  INT32_C(       31284),  INT32_C(       45815),  INT32_C(       12722),  INT32_C(       35697),  INT32_C(       32406),  INT32_C(       62950) },
      { -INT32_C(   471861910),  INT32_C(   301011863),  INT32_C(   411772245),  INT32_C(   466256531),  INT32_C(  1363967284), -INT32_C(  1545814687), -INT32_C(  1658752278), -INT32_C(   913176481) },
      { UINT16_C(59919), UINT16_C( 3636), UINT16_C(31284), UINT16_C(45815), UINT16_C(    0),      UINT16_MAX,      UINT16_MAX,      UINT16_MAX,
        UINT16_C(12722), UINT16_C(35697), UINT16_C(32406), UINT16_C(62950),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { {  INT32_C(       38060),  INT32_C(       55973),  INT32_C(       22258),  INT32_C(       59249),  INT32_C(       39736),  INT32_C(       25918),  INT32_C(       58882),  INT32_C(       26031) },
      { -INT32_C(  1929815033), -INT32_C(  1134059527), -INT32_C(  2112726577),  INT32_C(  1349092118), -INT32_C(  1125408447),  INT32_C(  1008806325), -INT32_C(   266198135), -INT32_C(  1084894792) },
      { UINT16_C(38060), UINT16_C(55973), UINT16_C(22258), UINT16_C(59249), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX,
        UINT16_C(39736), UINT16_C(25918), UINT16_C(58882), UINT16_C(26031), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0) } },
    { {  INT32_C(       10060),  INT32_C(       48611),  INT32_C(        9024),  INT32_C(       47731),  INT32_C(         118),  INT32_C(        4412),  INT32_C(       29442),  INT32_C(       23859) },
      { -INT32_C(  1803255898),  INT32_C(  1045588018), -INT32_C(   681471394), -INT32_C(  2037197509), -INT32_C(  1132001228),  INT32_C(  1540277152),  INT32_C(  1389285410), -INT32_C(   827391705) },
      { UINT16_C(10060), UINT16_C(48611), UINT16_C( 9024), UINT16_C(47731), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),
        UINT16_C(  118), UINT16_C( 4412), UINT16_C(29442), UINT16_C(23859), UINT16_C(    0),      UINT16_MAX,      UINT16_MAX, UINT16_C(    0) } },
    { {  INT32_C(       45666),  INT32_C(       64241),  INT32_C(       33489),  INT32_C(       23304),  INT32_C(        3096),  INT32_C(       30055),  INT32_C(       56775),  INT32_C(       47019) },
      {  INT32_C(  1181355690),  INT32_C(   155212738), -INT32_C(   712306003), -INT32_C(   516910219),  INT32_C(  1995261987), -INT32_C(   454339538), -INT32_C(  1044204918), -INT32_C(   747082456) },
      { UINT16_C(45666), UINT16_C(64241), UINT16_C(33489), UINT16_C(23304),      UINT16_MAX,      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),
        UINT16_C( 3096), UINT16_C(30055), UINT16_C(56775), UINT16_C(47019),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { {  INT32_C(       15897),  INT32_C(       60231),  INT32_C(       57792),  INT32_C(       35011),  INT32_C(       26622),  INT32_C(       36684),  INT32_C(       50256),  INT32_C(       63383) },
      { -INT32_C(   399134549),  INT32_C(  1993571338), -INT32_C(  1286040754), -INT32_C(  1120134268), -INT32_C(   802932021), -INT32_C(  1084264412), -INT32_C(    92033154),  INT32_C(   603003512) },
      { UINT16_C(15897), UINT16_C(60231), UINT16_C(57792), UINT16_C(35011), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),
        UINT16_C(26622), UINT16_C(36684), UINT16_C(50256), UINT16_C(63383), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX } },
    { {  INT32_C(       54539),  INT32_C(       61516),  INT32_C(       63139),  INT32_C(       35507),  INT32_C(       15963),  INT32_C(       51197),  INT32_C(       58049),  INT32_C(       26117) },
      {  INT32_C(  2050756824),  INT32_C(  1651149039), -INT32_C(   346550996),  INT32_C(   125111533),  INT32_C(   742772964), -INT32_C(   168607093),  INT32_C(  1607972035),  INT32_C(  1053154406) },
      { UINT16_C(54539), UINT16_C(61516), UINT16_C(63139), UINT16_C(35507),      UINT16_MAX,      UINT16_MAX, UINT16_C(    0),      UINT16_MAX,
        UINT16_C(15963), UINT16_C(51197), UINT16_C(58049), UINT16_C(26117),      UINT16_MAX, UINT16_C(    0),      UINT16_MAX,      UINT16_MAX } },
    { {  INT32_C(       56504),  INT32_C(       46398),  INT32_C(        7584),  INT32_C(       34340),  INT32_C(       29107),  INT32_C(       28518),  INT32_C(       49614),  INT32_C(        1536) },
      {  INT32_C(   518174869),  INT32_C(   198451419),  INT32_C(  1495823542),  INT32_C(  1910459530),  INT32_C(  1625461429), -INT32_C(  1798354631), -INT32_C(  1605001850), -INT32_C(   962177743) },
      { UINT16_C(56504), UINT16_C(46398), UINT16_C( 7584), UINT16_C(34340),      UINT16_MAX,      UINT16_MAX,      UINT16_MAX,      UINT16_MAX,
        UINT16_C(29107), UINT16_C(28518), UINT16_C(49614), UINT16_C( 1536),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_packus_epi32(a, b);
    simde_test_x86_assert_equal_u16x16(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_permute4x64_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi64x(INT64_C(-4031273950158647677), INT64_C(-7434948453373431243),
                             INT64_C(  966046851086666502), INT64_C(-3558090175042735721)),
      simde_mm256_set_epi64x(INT64_C(-3558090175042735721), INT64_C(-3558090175042735721),
                             INT64_C(-4031273950158647677), INT64_C(  966046851086666502)) },
    { simde_mm256_set_epi64x(INT64_C(-5846597928812893198), INT64_C(-4353963629209663352),
                             INT64_C(-7799994890686903985), INT64_C( 1444957477620918324)),
      simde_mm256_set_epi64x(INT64_C( 1444957477620918324), INT64_C( 1444957477620918324),
                             INT64_C(-5846597928812893198), INT64_C(-7799994890686903985)) },
    { simde_mm256_set_epi64x(INT64_C(-1184806487964558659), INT64_C( 7043949117721512702),
                             INT64_C(  -92438279376413162), INT64_C( 2263934164871463775)),
      simde_mm256_set_epi64x(INT64_C( 2263934164871463775), INT64_C( 2263934164871463775),
                             INT64_C(-1184806487964558659), INT64_C(  -92438279376413162)) },
    { simde_mm256_set_epi64x(INT64_C( 9090919205935740251), INT64_C( 3797255434791406626),
                             INT64_C(-3974983398240952043), INT64_C(-2667637164037811982)),
      simde_mm256_set_epi64x(INT64_C(-2667637164037811982), INT64_C(-2667637164037811982),
                             INT64_C( 9090919205935740251), INT64_C(-3974983398240952043)) },
    { simde_mm256_set_epi64x(INT64_C( 7885365925671452944), INT64_C( 8557735835567037410),
                             INT64_C( 1805700887716213163), INT64_C(-5945530108016559723)),
      simde_mm256_set_epi64x(INT64_C(-5945530108016559723), INT64_C(-5945530108016559723),
                             INT64_C( 7885365925671452944), INT64_C( 1805700887716213163)) },
    { simde_mm256_set_epi64x(INT64_C( -666700084400918528), INT64_C( 2293046882897477780),
                             INT64_C(-4361422993016110212), INT64_C( 5540865589910111090)),
      simde_mm256_set_epi64x(INT64_C( 5540865589910111090), INT64_C( 5540865589910111090),
                             INT64_C( -666700084400918528), INT64_C(-4361422993016110212)) },
    { simde_mm256_set_epi64x(INT64_C(-4089126903474854143), INT64_C(-3405442608942374627),
                             INT64_C(-5965708747641475330), INT64_C( 3779098457061206514)),
      simde_mm256_set_epi64x(INT64_C( 3779098457061206514), INT64_C( 3779098457061206514),
                             INT64_C(-4089126903474854143), INT64_C(-5965708747641475330)) },
    { simde_mm256_set_epi64x(INT64_C(-7195404196599220190), INT64_C( 4846123797420351534),
                             INT64_C( 1973553066803872882), INT64_C(-2603358823346386940)),
      simde_mm256_set_epi64x(INT64_C(-2603358823346386940), INT64_C(-2603358823346386940),
                             INT64_C(-7195404196599220190), INT64_C( 1973553066803872882)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_permute4x64_epi64(test_vec[i].a, 13);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_permute4x64_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256d a;
    simde__m256d r;
  } test_vec[8] = {
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -641.76), SIMDE_FLOAT64_C(  477.18),
                         SIMDE_FLOAT64_C(  278.49), SIMDE_FLOAT64_C(  569.18)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  569.18), SIMDE_FLOAT64_C(  569.18),
                         SIMDE_FLOAT64_C( -641.76), SIMDE_FLOAT64_C(  278.49)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -489.82), SIMDE_FLOAT64_C( -641.14),
                         SIMDE_FLOAT64_C( -951.91), SIMDE_FLOAT64_C(  935.01)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  935.01), SIMDE_FLOAT64_C(  935.01),
                         SIMDE_FLOAT64_C( -489.82), SIMDE_FLOAT64_C( -951.91)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -602.75), SIMDE_FLOAT64_C(  339.47),
                         SIMDE_FLOAT64_C( -820.66), SIMDE_FLOAT64_C( -740.61)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -740.61), SIMDE_FLOAT64_C( -740.61),
                         SIMDE_FLOAT64_C( -602.75), SIMDE_FLOAT64_C( -820.66)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -303.98), SIMDE_FLOAT64_C( -350.58),
                         SIMDE_FLOAT64_C( -574.59), SIMDE_FLOAT64_C(  850.86)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  850.86), SIMDE_FLOAT64_C(  850.86),
                         SIMDE_FLOAT64_C( -303.98), SIMDE_FLOAT64_C( -574.59)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -623.12), SIMDE_FLOAT64_C( -143.91),
                         SIMDE_FLOAT64_C( -840.93), SIMDE_FLOAT64_C(  411.53)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  411.53), SIMDE_FLOAT64_C(  411.53),
                         SIMDE_FLOAT64_C( -623.12), SIMDE_FLOAT64_C( -840.93)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -167.73), SIMDE_FLOAT64_C( -215.71),
                         SIMDE_FLOAT64_C( -717.06), SIMDE_FLOAT64_C( -432.13)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -432.13), SIMDE_FLOAT64_C( -432.13),
                         SIMDE_FLOAT64_C( -167.73), SIMDE_FLOAT64_C( -717.06)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  572.27), SIMDE_FLOAT64_C( -942.37),
                         SIMDE_FLOAT64_C(  905.44), SIMDE_FLOAT64_C( -810.45)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -810.45), SIMDE_FLOAT64_C( -810.45),
                         SIMDE_FLOAT64_C(  572.27), SIMDE_FLOAT64_C(  905.44)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -340.56), SIMDE_FLOAT64_C(  537.41),
                         SIMDE_FLOAT64_C( -960.96), SIMDE_FLOAT64_C( -230.88)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -230.88), SIMDE_FLOAT64_C( -230.88),
                         SIMDE_FLOAT64_C( -340.56), SIMDE_FLOAT64_C( -960.96)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256d r = simde_mm256_permute4x64_pd(test_vec[i].a, 13);
    simde_assert_m256d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm256_permute2x128_si256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi64x(INT64_C( 9096692030846176105), INT64_C(  644260392039444522),
                             INT64_C(-4583540275174352405), INT64_C(-6816753880857675259)),
      simde_mm256_set_epi64x(INT64_C(-7886827988827131690), INT64_C(-2107575233125845054),
                             INT64_C(-8398644678734943287), INT64_C( 7172114359254607016)),
      simde_mm256_set_epi64x(INT64_C( 9096692030846176105), INT64_C(  644260392039444522),
                             INT64_C(-7886827988827131690), INT64_C(-2107575233125845054)) },
    { simde_mm256_set_epi64x(INT64_C(-6314946612387904819), INT64_C(-1883921203594067636),
                             INT64_C(-5030251871897832411), INT64_C( 8348493077761215789)),
      simde_mm256_set_epi64x(INT64_C( 2582913386835954622), INT64_C(  153655168481379701),
                             INT64_C(-1087064137492042031), INT64_C( 1783808851867973139)),
      simde_mm256_set_epi64x(INT64_C(-6314946612387904819), INT64_C(-1883921203594067636),
                             INT64_C( 2582913386835954622), INT64_C(  153655168481379701)) },
    { simde_mm256_set_epi64x(INT64_C(-3208478614025680333), INT64_C(-3409066786741744502),
                             INT64_C(-6957643043766269700), INT64_C( 3219428559958296960)),
      simde_mm256_set_epi64x(INT64_C(-1736924574103250292), INT64_C(-2810347538827243748),
                             INT64_C( -297965813118371045), INT64_C(-7618358042182251122)),
      simde_mm256_set_epi64x(INT64_C(-3208478614025680333), INT64_C(-3409066786741744502),
                             INT64_C(-1736924574103250292), INT64_C(-2810347538827243748)) },
    { simde_mm256_set_epi64x(INT64_C(  796366024780064289), INT64_C(-1489690745108457074),
                             INT64_C(-7990282097237082056), INT64_C(-7545130296515735090)),
      simde_mm256_set_epi64x(INT64_C(  951803776889232332), INT64_C(-6640461449591045668),
                             INT64_C( 5271740244822761531), INT64_C( 3149915688837762175)),
      simde_mm256_set_epi64x(INT64_C(  796366024780064289), INT64_C(-1489690745108457074),
                             INT64_C(  951803776889232332), INT64_C(-6640461449591045668)) },
    { simde_mm256_set_epi64x(INT64_C(  -95120238103258498), INT64_C(-1762353908339260045),
                             INT64_C( 6992845328844002662), INT64_C(-5939283762406250642)),
      simde_mm256_set_epi64x(INT64_C(-5885001620821736092), INT64_C(-6745062192544323367),
                             INT64_C( 7803931770148523943), INT64_C(-8993062880293478576)),
      simde_mm256_set_epi64x(INT64_C(  -95120238103258498), INT64_C(-1762353908339260045),
                             INT64_C(-5885001620821736092), INT64_C(-6745062192544323367)) },
    { simde_mm256_set_epi64x(INT64_C(-3708437875152674849), INT64_C( 8243162546537572005),
                             INT64_C(-1103721052327437925), INT64_C(-2925489198757650175)),
      simde_mm256_set_epi64x(INT64_C( 1440085788748654982), INT64_C( 1725906984156202179),
                             INT64_C( 5845599904819452784), INT64_C( 7162548421658470679)),
      simde_mm256_set_epi64x(INT64_C(-3708437875152674849), INT64_C( 8243162546537572005),
                             INT64_C( 1440085788748654982), INT64_C( 1725906984156202179)) },
    { simde_mm256_set_epi64x(INT64_C( 6834943649491098623), INT64_C( 2759372331225584008),
                             INT64_C(-1659900994892419246), INT64_C(-2119655686628377164)),
      simde_mm256_set_epi64x(INT64_C( 1954610004667753515), INT64_C( 5688482191974230934),
                             INT64_C(-3937849964004809456), INT64_C(-8694088207381845200)),
      simde_mm256_set_epi64x(INT64_C( 6834943649491098623), INT64_C( 2759372331225584008),
                             INT64_C( 1954610004667753515), INT64_C( 5688482191974230934)) },
    { simde_mm256_set_epi64x(INT64_C( 5051547726856501651), INT64_C( 7333908238294102632),
                             INT64_C( 7118133466490521985), INT64_C( 6243950982549416292)),
      simde_mm256_set_epi64x(INT64_C(-6805527145604381785), INT64_C(-1282569833996306134),
                             INT64_C(-1497859500202369050), INT64_C( 1581543684384159070)),
      simde_mm256_set_epi64x(INT64_C( 5051547726856501651), INT64_C( 7333908238294102632),
                             INT64_C(-6805527145604381785), INT64_C(-1282569833996306134)) },
  };

  //printf("\n");
  //for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
  //  simde__m256i_private a, b, r;

  //  munit_rand_memory(sizeof(a), (uint8_t*) &a);
  //  munit_rand_memory(sizeof(b), (uint8_t*) &b);

  //  r = simde__m256i_to_private(simde_mm256_permute2x128_si256(simde__m256i_from_private(a), simde__m256i_from_private(b), 23));

  //  printf("    { simde_mm256_set_epi64x(INT64_C(%20" PRId64 "), INT64_C(%20" PRId64 "),\n"
  //         "                             INT64_C(%20" PRId64 "), INT64_C(%20" PRId64 ")),\n",
  //         a.i64[3], a.i64[2], a.i64[1], a.i64[0]);
  //  printf("      simde_mm256_set_epi64x(INT64_C(%20" PRId64 "), INT64_C(%20" PRId64 "),\n"
  //         "                             INT64_C(%20" PRId64 "), INT64_C(%20" PRId64 ")),\n",
  //         b.i64[3], b.i64[2], b.i64[1], b.i64[0]);
  //  printf("      simde_mm256_set_epi64x(INT64_C(%20" PRId64 "), INT64_C(%20" PRId64 "),\n"
  //         "                             INT64_C(%20" PRId64 "), INT64_C(%20" PRId64 ")) },\n",
  //         r.i64[3], r.i64[2], r.i64[1], r.i64[0]);
  //}
  //return MUNIT_FAIL;

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_permute2x128_si256(test_vec[i].a, test_vec[i].b, 23);
    simde_assert_m256i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_permutevar8x32_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { {  INT32_C(  1408063718),  INT32_C(  1053738302), -INT32_C(  1926009139),  INT32_C(   971995611), -INT32_C(    79435850),  INT32_C(  1766687132), -INT32_C(  1602616261), -INT32_C(   891684124) },
      { -INT32_C(  1877031342),  INT32_C(  1489955979),  INT32_C(   920978011),  INT32_C(   829478011),  INT32_C(  1462612923),  INT32_C(  1874885172),  INT32_C(  1645165438), -INT32_C(  2027034571) },
      { -INT32_C(  1926009139),  INT32_C(   971995611),  INT32_C(   971995611),  INT32_C(   971995611),  INT32_C(   971995611), -INT32_C(    79435850), -INT32_C(  1602616261),  INT32_C(  1766687132) } },
    { {  INT32_C(   957828014), -INT32_C(  1835932105),  INT32_C(  1674147816),  INT32_C(   127220043),  INT32_C(   543081196), -INT32_C(  1165025476), -INT32_C(  1893949862),  INT32_C(   873875846) },
      { -INT32_C(   865260140), -INT32_C(    77660653), -INT32_C(  1050728587),  INT32_C(  1288238176), -INT32_C(   227793226), -INT32_C(  1616053435),  INT32_C(   523159705), -INT32_C(  1504492526) },
      {  INT32_C(   543081196),  INT32_C(   127220043), -INT32_C(  1165025476),  INT32_C(   957828014), -INT32_C(  1893949862), -INT32_C(  1165025476), -INT32_C(  1835932105),  INT32_C(  1674147816) } },
    { { -INT32_C(  2056077198),  INT32_C(   880923071),  INT32_C(  1509286136), -INT32_C(  1968849452),  INT32_C(   695997156), -INT32_C(  1479989235),  INT32_C(    46594032), -INT32_C(  1364649412) },
      { -INT32_C(  1724703782), -INT32_C(   439438100),  INT32_C(  1748943764),  INT32_C(  1710416769),  INT32_C(    59666165), -INT32_C(  2018879594), -INT32_C(  1953927089),  INT32_C(  1698247306) },
      {  INT32_C(  1509286136),  INT32_C(   695997156),  INT32_C(   695997156),  INT32_C(   880923071), -INT32_C(  1479989235),  INT32_C(    46594032), -INT32_C(  1364649412),  INT32_C(  1509286136) } },
    { {  INT32_C(   989752398), -INT32_C(  1256207327),  INT32_C(   287202704),  INT32_C(   913707073),  INT32_C(   356058239), -INT32_C(  1415781540), -INT32_C(   566876588), -INT32_C(  1505530024) },
      { -INT32_C(    52346149), -INT32_C(  1632501746), -INT32_C(  1615867810),  INT32_C(  1607804384), -INT32_C(  2039148758),  INT32_C(  1177620978), -INT32_C(  1876596937), -INT32_C(  1321834282) },
      {  INT32_C(   913707073), -INT32_C(   566876588), -INT32_C(   566876588),  INT32_C(   989752398),  INT32_C(   287202704),  INT32_C(   287202704), -INT32_C(  1505530024), -INT32_C(   566876588) } },
    { { -INT32_C(  1179773014),  INT32_C(  1985437720),  INT32_C(   269813552),  INT32_C(  1450240556), -INT32_C(   321067527),  INT32_C(   775032310),  INT32_C(  1253988212),  INT32_C(  1794962624) },
      {  INT32_C(   606317068),  INT32_C(   983202570), -INT32_C(  1370837118), -INT32_C(  1828341095), -INT32_C(  1770004065),  INT32_C(  1673834991), -INT32_C(   911310327), -INT32_C(  2110543242) },
      { -INT32_C(   321067527),  INT32_C(   269813552),  INT32_C(   269813552),  INT32_C(  1985437720),  INT32_C(  1794962624),  INT32_C(  1794962624),  INT32_C(  1985437720),  INT32_C(  1253988212) } },
    { {  INT32_C(  1587959636),  INT32_C(  1419264210), -INT32_C(  1996299537),  INT32_C(  1008469917), -INT32_C(   657286167),  INT32_C(  1429968460), -INT32_C(  1893799656), -INT32_C(   418295149) },
      {  INT32_C(  2068166825), -INT32_C(   389030408),  INT32_C(  1567740352), -INT32_C(  1030124071),  INT32_C(  1956277288),  INT32_C(   466277634),  INT32_C(  1386932415), -INT32_C(   482690246) },
      {  INT32_C(  1419264210),  INT32_C(  1587959636),  INT32_C(  1587959636),  INT32_C(  1419264210),  INT32_C(  1587959636), -INT32_C(  1996299537), -INT32_C(   418295149), -INT32_C(  1996299537) } },
    { {  INT32_C(  1818132339),  INT32_C(   492055901), -INT32_C(   663042561),  INT32_C(  2056917842), -INT32_C(  2098318209), -INT32_C(   929187831), -INT32_C(   618969183),  INT32_C(  1992250626) },
      {  INT32_C(   836902356),  INT32_C(  1246639691),  INT32_C(  1294125307),  INT32_C(  1539816668), -INT32_C(   102910480),  INT32_C(   264403566), -INT32_C(   991240767),  INT32_C(   104507698) },
      { -INT32_C(  2098318209),  INT32_C(  2056917842),  INT32_C(  2056917842), -INT32_C(  2098318209),  INT32_C(  1818132339), -INT32_C(   618969183),  INT32_C(   492055901), -INT32_C(   663042561) } },
    { {  INT32_C(   305667271),  INT32_C(  1297909330),  INT32_C(   731545167),  INT32_C(   713449786), -INT32_C(  2078055402), -INT32_C(  1600920098), -INT32_C(   194740542), -INT32_C(   285499865) },
      {  INT32_C(   201339834),  INT32_C(   140074169),  INT32_C(   338949082),  INT32_C(  1782495828), -INT32_C(    51420642),  INT32_C(   178029127),  INT32_C(   687734784),  INT32_C(  1477900702) },
      {  INT32_C(   731545167),  INT32_C(  1297909330),  INT32_C(   731545167), -INT32_C(  2078055402), -INT32_C(   194740542), -INT32_C(   285499865),  INT32_C(   305667271), -INT32_C(   194740542) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_permutevar8x32_epi32(a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_permutevar8x32_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[8];
    const int32_t b[8];
    const simde_float32 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   809.43), SIMDE_FLOAT32_C(   302.44), SIMDE_FLOAT32_C(  -655.90), SIMDE_FLOAT32_C(   628.58),
        SIMDE_FLOAT32_C(   242.76), SIMDE_FLOAT32_C(  -149.67), SIMDE_FLOAT32_C(   759.40), SIMDE_FLOAT32_C(   143.90) },
      { -INT32_C(   716838906),  INT32_C(   339742123), -INT32_C(  1286153911), -INT32_C(   443567707), -INT32_C(  1899934705),  INT32_C(  2112822676), -INT32_C(  1385550335),  INT32_C(    16765946) },
      { SIMDE_FLOAT32_C(   759.40), SIMDE_FLOAT32_C(   628.58), SIMDE_FLOAT32_C(   302.44), SIMDE_FLOAT32_C(  -149.67),
        SIMDE_FLOAT32_C(   143.90), SIMDE_FLOAT32_C(   242.76), SIMDE_FLOAT32_C(   302.44), SIMDE_FLOAT32_C(  -655.90) } },
    { { SIMDE_FLOAT32_C(   110.93), SIMDE_FLOAT32_C(   350.59), SIMDE_FLOAT32_C(  -543.10), SIMDE_FLOAT32_C(  -725.49),
        SIMDE_FLOAT32_C(    18.95), SIMDE_FLOAT32_C(  -444.18), SIMDE_FLOAT32_C(  -148.59), SIMDE_FLOAT32_C(   920.56) },
      { -INT32_C(  1806839313), -INT32_C(  1854284414), -INT32_C(  1122026711),  INT32_C(  1513754200),  INT32_C(   923247932),  INT32_C(   859244152), -INT32_C(  1667625654),  INT32_C(   322311203) },
      { SIMDE_FLOAT32_C(   920.56), SIMDE_FLOAT32_C(  -543.10), SIMDE_FLOAT32_C(   350.59), SIMDE_FLOAT32_C(   110.93),
        SIMDE_FLOAT32_C(    18.95), SIMDE_FLOAT32_C(   110.93), SIMDE_FLOAT32_C(  -543.10), SIMDE_FLOAT32_C(  -725.49) } },
    { { SIMDE_FLOAT32_C(   -73.44), SIMDE_FLOAT32_C(   -32.92), SIMDE_FLOAT32_C(   765.45), SIMDE_FLOAT32_C(   761.19),
        SIMDE_FLOAT32_C(   685.64), SIMDE_FLOAT32_C(   282.32), SIMDE_FLOAT32_C(   215.31), SIMDE_FLOAT32_C(  -532.59) },
      { -INT32_C(  1270343332),  INT32_C(  1678672423), -INT32_C(  1617226457),  INT32_C(  1725157915),  INT32_C(    50490848),  INT32_C(  1729509505),  INT32_C(   500154044),  INT32_C(  1000917215) },
      { SIMDE_FLOAT32_C(   685.64), SIMDE_FLOAT32_C(  -532.59), SIMDE_FLOAT32_C(  -532.59), SIMDE_FLOAT32_C(   761.19),
        SIMDE_FLOAT32_C(   -73.44), SIMDE_FLOAT32_C(   -32.92), SIMDE_FLOAT32_C(   685.64), SIMDE_FLOAT32_C(  -532.59) } },
    { { SIMDE_FLOAT32_C(  -786.97), SIMDE_FLOAT32_C(  -512.61), SIMDE_FLOAT32_C(   516.41), SIMDE_FLOAT32_C(  -558.86),
        SIMDE_FLOAT32_C(  -760.16), SIMDE_FLOAT32_C(   955.28), SIMDE_FLOAT32_C(  -651.17), SIMDE_FLOAT32_C(   933.60) },
      {  INT32_C(   792266771), -INT32_C(  1164636966), -INT32_C(    71395463), -INT32_C(  1939680049),  INT32_C(  1906913682), -INT32_C(   609463815), -INT32_C(  1260086207), -INT32_C(  1387376230) },
      { SIMDE_FLOAT32_C(  -558.86), SIMDE_FLOAT32_C(   516.41), SIMDE_FLOAT32_C(  -512.61), SIMDE_FLOAT32_C(   933.60),
        SIMDE_FLOAT32_C(   516.41), SIMDE_FLOAT32_C(  -512.61), SIMDE_FLOAT32_C(  -512.61), SIMDE_FLOAT32_C(   516.41) } },
    { { SIMDE_FLOAT32_C(   -59.26), SIMDE_FLOAT32_C(   439.31), SIMDE_FLOAT32_C(   841.23), SIMDE_FLOAT32_C(  -266.81),
        SIMDE_FLOAT32_C(   199.59), SIMDE_FLOAT32_C(   -17.60), SIMDE_FLOAT32_C(   866.57), SIMDE_FLOAT32_C(  -521.77) },
      { -INT32_C(   670520312),  INT32_C(   325347968), -INT32_C(  1786507877), -INT32_C(  1603260065),  INT32_C(  1733579981), -INT32_C(    32202075), -INT32_C(  1104023254),  INT32_C(  1808473954) },
      { SIMDE_FLOAT32_C(   -59.26), SIMDE_FLOAT32_C(   -59.26), SIMDE_FLOAT32_C(  -266.81), SIMDE_FLOAT32_C(  -521.77),
        SIMDE_FLOAT32_C(   -17.60), SIMDE_FLOAT32_C(   -17.60), SIMDE_FLOAT32_C(   841.23), SIMDE_FLOAT32_C(   841.23) } },
    { { SIMDE_FLOAT32_C(  -400.14), SIMDE_FLOAT32_C(    83.44), SIMDE_FLOAT32_C(  -751.08), SIMDE_FLOAT32_C(   794.29),
        SIMDE_FLOAT32_C(   639.38), SIMDE_FLOAT32_C(  -222.75), SIMDE_FLOAT32_C(  -375.60), SIMDE_FLOAT32_C(   682.13) },
      {  INT32_C(   326099892), -INT32_C(   508240108), -INT32_C(   666367949), -INT32_C(   707371605), -INT32_C(  1332541618), -INT32_C(   233087449),  INT32_C(  1899912754), -INT32_C(  1169449723) },
      { SIMDE_FLOAT32_C(   639.38), SIMDE_FLOAT32_C(   639.38), SIMDE_FLOAT32_C(   794.29), SIMDE_FLOAT32_C(   794.29),
        SIMDE_FLOAT32_C(  -375.60), SIMDE_FLOAT32_C(   682.13), SIMDE_FLOAT32_C(  -751.08), SIMDE_FLOAT32_C(  -222.75) } },
    { { SIMDE_FLOAT32_C(  -742.55), SIMDE_FLOAT32_C(  -101.38), SIMDE_FLOAT32_C(  -337.44), SIMDE_FLOAT32_C(   380.26),
        SIMDE_FLOAT32_C(   642.80), SIMDE_FLOAT32_C(   959.92), SIMDE_FLOAT32_C(  -937.04), SIMDE_FLOAT32_C(  -939.55) },
      {  INT32_C(   899989386),  INT32_C(  1779071516), -INT32_C(  1474585215),  INT32_C(   781924091), -INT32_C(  1700800107), -INT32_C(   128652681),  INT32_C(  1032659620),  INT32_C(   755565731) },
      { SIMDE_FLOAT32_C(  -337.44), SIMDE_FLOAT32_C(   642.80), SIMDE_FLOAT32_C(  -101.38), SIMDE_FLOAT32_C(   380.26),
        SIMDE_FLOAT32_C(   959.92), SIMDE_FLOAT32_C(  -939.55), SIMDE_FLOAT32_C(   642.80), SIMDE_FLOAT32_C(   380.26) } },
    { { SIMDE_FLOAT32_C(    -5.16), SIMDE_FLOAT32_C(   584.21), SIMDE_FLOAT32_C(   220.31), SIMDE_FLOAT32_C(  -944.84),
        SIMDE_FLOAT32_C(  -786.38), SIMDE_FLOAT32_C(  -527.06), SIMDE_FLOAT32_C(  -779.98), SIMDE_FLOAT32_C(  -331.72) },
      {  INT32_C(    89220617),  INT32_C(   825486492),  INT32_C(  1020056261),  INT32_C(  1614028988), -INT32_C(   425868734), -INT32_C(  1978423610),  INT32_C(  2053862995), -INT32_C(   333269278) },
      { SIMDE_FLOAT32_C(   584.21), SIMDE_FLOAT32_C(  -786.38), SIMDE_FLOAT32_C(  -527.06), SIMDE_FLOAT32_C(  -786.38),
        SIMDE_FLOAT32_C(   220.31), SIMDE_FLOAT32_C(  -779.98), SIMDE_FLOAT32_C(  -944.84), SIMDE_FLOAT32_C(   220.31) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256 a = simde_mm256_loadu_ps(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256 r = simde_mm256_permutevar8x32_ps(a, b);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_sad_epu8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint8_t a[32];
    const uint8_t b[32];
    const uint64_t r[4];
  } test_vec[] = {
    { { UINT8_C(193), UINT8_C(212), UINT8_C(  9), UINT8_C(139), UINT8_C(179), UINT8_C(138), UINT8_C(126), UINT8_C(149),
        UINT8_C(188), UINT8_C(191), UINT8_C(174), UINT8_C( 53), UINT8_C(220), UINT8_C(132), UINT8_C(137), UINT8_C(189),
        UINT8_C( 77), UINT8_C( 99), UINT8_C(211), UINT8_C(102), UINT8_C(  2), UINT8_C(  4), UINT8_C(118), UINT8_C(127),
        UINT8_C(149), UINT8_C( 77), UINT8_C( 65), UINT8_C(195), UINT8_C(184), UINT8_C(249), UINT8_C(233), UINT8_C(122) },
      { UINT8_C(205), UINT8_C(242), UINT8_C(  5), UINT8_C(128), UINT8_C(125), UINT8_C(131), UINT8_C( 22), UINT8_C( 57),
        UINT8_C( 66), UINT8_C(196), UINT8_C(111), UINT8_C( 30), UINT8_C( 72), UINT8_C(248), UINT8_C(220), UINT8_C(149),
        UINT8_C( 91), UINT8_C(175), UINT8_C(251), UINT8_C( 93), UINT8_C(179), UINT8_C(113), UINT8_C(220), UINT8_C( 72),
        UINT8_C(190), UINT8_C( 29), UINT8_C( 11), UINT8_C(119), UINT8_C( 23), UINT8_C(244), UINT8_C(241), UINT8_C(228) },
      { UINT64_C(                 314), UINT64_C(                 600), UINT64_C(                 582), UINT64_C(                 499) } },
    { { UINT8_C(231), UINT8_C(246), UINT8_C(101), UINT8_C(100), UINT8_C(122), UINT8_C(123), UINT8_C(157), UINT8_C(188),
        UINT8_C( 63), UINT8_C( 12), UINT8_C(219), UINT8_C(135), UINT8_C(  4), UINT8_C(183), UINT8_C( 28), UINT8_C( 95),
        UINT8_C(102), UINT8_C( 24), UINT8_C(189), UINT8_C( 26), UINT8_C(137), UINT8_C(153), UINT8_C( 98), UINT8_C( 72),
        UINT8_C(183), UINT8_C(110), UINT8_C(191), UINT8_C(206), UINT8_C( 98), UINT8_C(176), UINT8_C(178), UINT8_C( 73) },
      { UINT8_C(166), UINT8_C( 23), UINT8_C(173), UINT8_C( 32), UINT8_C(146), UINT8_C( 75), UINT8_C(221), UINT8_C(209),
        UINT8_C( 87), UINT8_C(184), UINT8_C( 88), UINT8_C( 92), UINT8_C(111), UINT8_C(117), UINT8_C(187), UINT8_C(213),
        UINT8_C(141), UINT8_C(120), UINT8_C(239), UINT8_C( 22), UINT8_C( 18), UINT8_C( 82), UINT8_C( 94), UINT8_C(201),
        UINT8_C(192), UINT8_C( 29), UINT8_C(151), UINT8_C( 34), UINT8_C(205), UINT8_C( 73), UINT8_C(108), UINT8_C(116) },
      { UINT64_C(                 585), UINT64_C(                 820), UINT64_C(                 512), UINT64_C(                 625) } },
    { { UINT8_C( 97), UINT8_C( 25), UINT8_C(148), UINT8_C(243), UINT8_C(100), UINT8_C(113), UINT8_C(197), UINT8_C(188),
        UINT8_C( 41), UINT8_C( 29), UINT8_C( 24), UINT8_C(152), UINT8_C(146), UINT8_C(211), UINT8_C(110), UINT8_C( 31),
        UINT8_C( 76), UINT8_C( 93), UINT8_C( 54), UINT8_C( 94), UINT8_C(175), UINT8_C(148), UINT8_C( 39), UINT8_C(111),
        UINT8_C(178), UINT8_C(190), UINT8_C(146), UINT8_C(127), UINT8_C(  7), UINT8_C(254), UINT8_C(243), UINT8_C(104) },
      { UINT8_C( 23), UINT8_C(136), UINT8_C( 92), UINT8_C(124), UINT8_C(249), UINT8_C( 33), UINT8_C( 56), UINT8_C( 35),
        UINT8_C( 62), UINT8_C( 80), UINT8_C(187), UINT8_C(209), UINT8_C( 35), UINT8_C( 41), UINT8_C(240), UINT8_C(111),
        UINT8_C(135), UINT8_C( 38), UINT8_C(205), UINT8_C( 54), UINT8_C(187), UINT8_C(244), UINT8_C(166), UINT8_C(109),
        UINT8_C(178), UINT8_C( 56), UINT8_C(236), UINT8_C(186), UINT8_C( 54), UINT8_C(224), UINT8_C( 34), UINT8_C( 77) },
      { UINT64_C(                 883), UINT64_C(                 783), UINT64_C(                 542), UINT64_C(                 596) } },
    { { UINT8_C(104), UINT8_C(126), UINT8_C(201), UINT8_C( 97), UINT8_C(159), UINT8_C(  1), UINT8_C(132), UINT8_C(222),
        UINT8_C( 81), UINT8_C( 64), UINT8_C(175), UINT8_C(117), UINT8_C(105), UINT8_C(159), UINT8_C(228), UINT8_C(240),
        UINT8_C(198), UINT8_C(178), UINT8_C( 39), UINT8_C(129), UINT8_C(166), UINT8_C(205), UINT8_C(238), UINT8_C( 89),
        UINT8_C(  5), UINT8_C(218), UINT8_C( 19), UINT8_C( 59), UINT8_C(186), UINT8_C( 53), UINT8_C(136), UINT8_C( 34) },
      { UINT8_C(180), UINT8_C( 82), UINT8_C(132), UINT8_C( 83), UINT8_C( 83), UINT8_C(  8), UINT8_C( 49), UINT8_C(165),
        UINT8_C( 72), UINT8_C(224), UINT8_C( 26), UINT8_C(178), UINT8_C(128), UINT8_C(254), UINT8_C(162), UINT8_C( 70),
        UINT8_C(176), UINT8_C(201), UINT8_C(199), UINT8_C( 87), UINT8_C(150), UINT8_C(181), UINT8_C(176), UINT8_C(155),
        UINT8_C(143), UINT8_C(195), UINT8_C(214), UINT8_C( 74), UINT8_C(248), UINT8_C( 95), UINT8_C(108), UINT8_C(172) },
      { UINT64_C(                 426), UINT64_C(                 733), UINT64_C(                 415), UINT64_C(                 641) } },
    { { UINT8_C(177), UINT8_C(240), UINT8_C(  0), UINT8_C(  4), UINT8_C(249), UINT8_C( 49), UINT8_C(169), UINT8_C( 65),
        UINT8_C( 18), UINT8_C(195), UINT8_C(243), UINT8_C(146), UINT8_C(194), UINT8_C(150), UINT8_C(216), UINT8_C(114),
        UINT8_C( 95), UINT8_C(159), UINT8_C(201), UINT8_C(246), UINT8_C( 84), UINT8_C(121), UINT8_C(145), UINT8_C(227),
        UINT8_C( 60), UINT8_C(104), UINT8_C( 45), UINT8_C( 53), UINT8_C(199), UINT8_C(154), UINT8_C(225), UINT8_C(120) },
      { UINT8_C(138), UINT8_C(225), UINT8_C(124), UINT8_C(131), UINT8_C( 19), UINT8_C( 38), UINT8_C(197), UINT8_C( 37),
        UINT8_C(233), UINT8_C(184), UINT8_C(183), UINT8_C(171), UINT8_C( 78), UINT8_C(143), UINT8_C( 30), UINT8_C(174),
        UINT8_C( 46), UINT8_C(231), UINT8_C(164), UINT8_C(130), UINT8_C( 97), UINT8_C( 53), UINT8_C(101), UINT8_C(157),
        UINT8_C(157), UINT8_C(147), UINT8_C(210), UINT8_C(100), UINT8_C( 45), UINT8_C(180), UINT8_C(220), UINT8_C(183) },
      { UINT64_C(                 602), UINT64_C(                 680), UINT64_C(                 469), UINT64_C(                 600) } },
    { { UINT8_C(149), UINT8_C( 89), UINT8_C( 59), UINT8_C(168), UINT8_C(127), UINT8_C(  0), UINT8_C(205), UINT8_C(104),
        UINT8_C(184), UINT8_C(132), UINT8_C( 20), UINT8_C(  7), UINT8_C( 19), UINT8_C( 50), UINT8_C(181), UINT8_C( 65),
        UINT8_C( 25), UINT8_C( 89), UINT8_C(195), UINT8_C(122), UINT8_C(142), UINT8_C( 41), UINT8_C( 24), UINT8_C( 44),
        UINT8_C(188), UINT8_C(234), UINT8_C(144), UINT8_C(233), UINT8_C(158), UINT8_C(109), UINT8_C(160), UINT8_C( 52) },
      { UINT8_C(198), UINT8_C(219), UINT8_C(220), UINT8_C( 69), UINT8_C(219), UINT8_C(170), UINT8_C(173), UINT8_C(148),
        UINT8_C( 46), UINT8_C(193), UINT8_C(155), UINT8_C( 66), UINT8_C(243), UINT8_C( 80), UINT8_C(131), UINT8_C( 13),
        UINT8_C(169), UINT8_C( 71), UINT8_C(135), UINT8_C( 55), UINT8_C(112), UINT8_C(159), UINT8_C( 99), UINT8_C( 44),
        UINT8_C(138), UINT8_C(244), UINT8_C( 21), UINT8_C( 40), UINT8_C( 97), UINT8_C(181), UINT8_C( 92), UINT8_C( 39) },
      { UINT64_C(                 777), UINT64_C(                 749), UINT64_C(                 512), UINT64_C(                 590) } },
    { { UINT8_C(145), UINT8_C( 57), UINT8_C(108), UINT8_C(108), UINT8_C(227), UINT8_C( 25), UINT8_C(  0), UINT8_C( 17),
        UINT8_C(219), UINT8_C(155), UINT8_C( 83), UINT8_C(206), UINT8_C(235), UINT8_C(215), UINT8_C(219), UINT8_C(148),
        UINT8_C( 30), UINT8_C( 99), UINT8_C(204), UINT8_C(142), UINT8_C(  2), UINT8_C( 47), UINT8_C(186), UINT8_C(140),
        UINT8_C( 35), UINT8_C(207), UINT8_C(181), UINT8_C(132), UINT8_C(132), UINT8_C( 17), UINT8_C(171), UINT8_C( 21) },
      { UINT8_C( 74), UINT8_C( 23), UINT8_C(130), UINT8_C( 45), UINT8_C( 49), UINT8_C(130), UINT8_C( 63), UINT8_C( 12),
        UINT8_C( 30), UINT8_C(146), UINT8_C(218), UINT8_C(  9), UINT8_C(105), UINT8_C(182), UINT8_C(158), UINT8_C(135),
        UINT8_C( 25), UINT8_C(106), UINT8_C( 21), UINT8_C( 27), UINT8_C(153), UINT8_C(207), UINT8_C(168), UINT8_C(189),
        UINT8_C(158), UINT8_C( 93), UINT8_C( 65), UINT8_C( 35), UINT8_C(110), UINT8_C(237), UINT8_C( 56), UINT8_C(185) },
      { UINT64_C(                 541), UINT64_C(                 767), UINT64_C(                 688), UINT64_C(                 971) } },
    { { UINT8_C(  4), UINT8_C(186), UINT8_C(230), UINT8_C( 53), UINT8_C( 61), UINT8_C( 37), UINT8_C( 65), UINT8_C( 91),
        UINT8_C(184), UINT8_C( 28), UINT8_C(100), UINT8_C( 33), UINT8_C(210), UINT8_C(  2), UINT8_C(169), UINT8_C(235),
        UINT8_C(108), UINT8_C(190), UINT8_C(  6), UINT8_C(  6), UINT8_C(142), UINT8_C(174), UINT8_C(195), UINT8_C( 44),
        UINT8_C( 11), UINT8_C(  4), UINT8_C( 79), UINT8_C(122), UINT8_C(241), UINT8_C(136), UINT8_C( 51), UINT8_C(246) },
      { UINT8_C( 66), UINT8_C( 25), UINT8_C( 43), UINT8_C(127), UINT8_C( 63), UINT8_C(109), UINT8_C(218), UINT8_C(247),
        UINT8_C(137), UINT8_C( 63), UINT8_C( 24), UINT8_C( 91), UINT8_C( 65), UINT8_C(193), UINT8_C( 70), UINT8_C(174),
        UINT8_C(128), UINT8_C( 76), UINT8_C(180), UINT8_C( 14), UINT8_C(251), UINT8_C(119), UINT8_C( 58), UINT8_C(  6),
        UINT8_C(123), UINT8_C(138), UINT8_C(128), UINT8_C(109), UINT8_C( 18), UINT8_C(179), UINT8_C( 99), UINT8_C( 84) },
      { UINT64_C(                 867), UINT64_C(                 712), UINT64_C(                 655), UINT64_C(                 784) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi8(test_vec[i].b);
    simde__m256i r = simde_mm256_sad_epu8(a, b);
    simde_test_x86_assert_equal_u64x4(r, simde_x_mm256_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_shuffle_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_x_mm256_set_epu8(UINT8_C(132), UINT8_C(100), UINT8_C(115), UINT8_C( 94),
                             UINT8_C( 73), UINT8_C(247), UINT8_C(104), UINT8_C(220),
                             UINT8_C(117), UINT8_C( 74), UINT8_C( 39), UINT8_C(125),
                             UINT8_C( 51), UINT8_C( 55), UINT8_C(148), UINT8_C(183),
                             UINT8_C(  5), UINT8_C(228), UINT8_C( 33), UINT8_C( 72),
                             UINT8_C( 84), UINT8_C(  4), UINT8_C(172), UINT8_C(134),
                             UINT8_C( 26), UINT8_C(247), UINT8_C(250), UINT8_C(  1),
                             UINT8_C(220), UINT8_C( 43), UINT8_C( 83), UINT8_C(118)),
      simde_x_mm256_set_epu8(UINT8_C( 24), UINT8_C(160), UINT8_C( 38), UINT8_C(201),
                             UINT8_C(108), UINT8_C( 32), UINT8_C( 93), UINT8_C( 16),
                             UINT8_C(104), UINT8_C(146), UINT8_C(128), UINT8_C(127),
                             UINT8_C(240), UINT8_C(187), UINT8_C(163), UINT8_C( 93),
                             UINT8_C(161), UINT8_C(148), UINT8_C(236), UINT8_C(208),
                             UINT8_C(110), UINT8_C( 19), UINT8_C(  3), UINT8_C(142),
                             UINT8_C( 56), UINT8_C( 74), UINT8_C(  2), UINT8_C(168),
                             UINT8_C(100), UINT8_C(209), UINT8_C( 27), UINT8_C( 46)),
      simde_x_mm256_set_epu8(UINT8_C(220), UINT8_C(  0), UINT8_C( 74), UINT8_C(  0),
                             UINT8_C( 94), UINT8_C(183), UINT8_C(115), UINT8_C(183),
                             UINT8_C(220), UINT8_C(  0), UINT8_C(  0), UINT8_C(132),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(115),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(228), UINT8_C(220), UINT8_C(220), UINT8_C(  0),
                             UINT8_C(134), UINT8_C(  4), UINT8_C( 43), UINT8_C(  0),
                             UINT8_C(  1), UINT8_C(  0), UINT8_C( 84), UINT8_C(228)) },
    { simde_x_mm256_set_epu8(UINT8_C( 53), UINT8_C(227), UINT8_C(193), UINT8_C(203),
                             UINT8_C(188), UINT8_C( 31), UINT8_C(134), UINT8_C(151),
                             UINT8_C( 90), UINT8_C(227), UINT8_C(228), UINT8_C( 42),
                             UINT8_C(223), UINT8_C( 71), UINT8_C( 37), UINT8_C(166),
                             UINT8_C(115), UINT8_C( 13), UINT8_C(136), UINT8_C( 67),
                             UINT8_C(142), UINT8_C( 55), UINT8_C(218), UINT8_C(135),
                             UINT8_C(160), UINT8_C(121), UINT8_C(105), UINT8_C(225),
                             UINT8_C( 67), UINT8_C(106), UINT8_C( 80), UINT8_C(187)),
      simde_x_mm256_set_epu8(UINT8_C(139), UINT8_C(207), UINT8_C(188), UINT8_C(170),
                             UINT8_C(232), UINT8_C(172), UINT8_C( 51), UINT8_C( 14),
                             UINT8_C( 50), UINT8_C(186), UINT8_C( 76), UINT8_C(187),
                             UINT8_C( 22), UINT8_C( 90), UINT8_C(150), UINT8_C(148),
                             UINT8_C(123), UINT8_C( 36), UINT8_C(145), UINT8_C( 72),
                             UINT8_C(252), UINT8_C(171), UINT8_C(109), UINT8_C( 77),
                             UINT8_C(145), UINT8_C( 77), UINT8_C(121), UINT8_C(107),
                             UINT8_C(138), UINT8_C(154), UINT8_C(236), UINT8_C( 13)),
      simde_x_mm256_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(223), UINT8_C(227),
                             UINT8_C( 71), UINT8_C(  0), UINT8_C(203), UINT8_C(  0),
                             UINT8_C(227), UINT8_C( 31), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(142), UINT8_C(225), UINT8_C(  0), UINT8_C(135),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(136), UINT8_C(136),
                             UINT8_C(  0), UINT8_C(136), UINT8_C(218), UINT8_C(142),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(136)) },
    { simde_x_mm256_set_epu8(UINT8_C(147), UINT8_C(175), UINT8_C(103), UINT8_C(105),
                             UINT8_C(115), UINT8_C( 25), UINT8_C( 88), UINT8_C( 95),
                             UINT8_C(108), UINT8_C(195), UINT8_C( 30), UINT8_C( 32),
                             UINT8_C(183), UINT8_C(198), UINT8_C(177), UINT8_C( 61),
                             UINT8_C(213), UINT8_C(  7), UINT8_C(150), UINT8_C(186),
                             UINT8_C( 52), UINT8_C(  9), UINT8_C( 76), UINT8_C(240),
                             UINT8_C(240), UINT8_C(126), UINT8_C(242), UINT8_C(203),
                             UINT8_C(162), UINT8_C( 72), UINT8_C(186), UINT8_C( 35)),
      simde_x_mm256_set_epu8(UINT8_C( 26), UINT8_C(  1), UINT8_C(253), UINT8_C(220),
                             UINT8_C(156), UINT8_C(126), UINT8_C(237), UINT8_C(228),
                             UINT8_C(210), UINT8_C( 82), UINT8_C(  8), UINT8_C(  8),
                             UINT8_C(104), UINT8_C( 35), UINT8_C( 98), UINT8_C( 24),
                             UINT8_C(230), UINT8_C(227), UINT8_C( 73), UINT8_C(233),
                             UINT8_C(132), UINT8_C( 49), UINT8_C( 63), UINT8_C(172),
                             UINT8_C(137), UINT8_C( 87), UINT8_C( 85), UINT8_C(128),
                             UINT8_C(136), UINT8_C(108), UINT8_C( 70), UINT8_C(218)),
      simde_x_mm256_set_epu8(UINT8_C( 25), UINT8_C(177), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(175), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(198), UINT8_C( 95), UINT8_C( 95),
                             UINT8_C( 95), UINT8_C(183), UINT8_C(198), UINT8_C( 95),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C( 76), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(186), UINT8_C(213), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(240), UINT8_C(242), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(186), UINT8_C(126), UINT8_C(  0)) },
    { simde_x_mm256_set_epu8(UINT8_C(132), UINT8_C( 22), UINT8_C(115), UINT8_C(245),
                             UINT8_C(139), UINT8_C(127), UINT8_C( 30), UINT8_C( 96),
                             UINT8_C(123), UINT8_C(228), UINT8_C( 73), UINT8_C(244),
                             UINT8_C( 12), UINT8_C( 46), UINT8_C( 26), UINT8_C( 36),
                             UINT8_C(186), UINT8_C( 18), UINT8_C(169), UINT8_C(233),
                             UINT8_C(  8), UINT8_C(185), UINT8_C(123), UINT8_C( 50),
                             UINT8_C(171), UINT8_C(240), UINT8_C(159), UINT8_C(176),
                             UINT8_C( 29), UINT8_C(146), UINT8_C( 85), UINT8_C( 34)),
      simde_x_mm256_set_epu8(UINT8_C(197), UINT8_C( 93), UINT8_C( 52), UINT8_C(130),
                             UINT8_C(124), UINT8_C( 87), UINT8_C( 88), UINT8_C( 89),
                             UINT8_C( 56), UINT8_C(209), UINT8_C( 85), UINT8_C(235),
                             UINT8_C( 45), UINT8_C(236), UINT8_C(154), UINT8_C( 72),
                             UINT8_C(164), UINT8_C(253), UINT8_C(196), UINT8_C( 41),
                             UINT8_C( 28), UINT8_C(181), UINT8_C(180), UINT8_C(237),
                             UINT8_C( 65), UINT8_C(122), UINT8_C(110), UINT8_C( 94),
                             UINT8_C(125), UINT8_C( 60), UINT8_C( 80), UINT8_C(209)),
      simde_x_mm256_set_epu8(UINT8_C(  0), UINT8_C(115), UINT8_C(244), UINT8_C(  0),
                             UINT8_C(245), UINT8_C(123), UINT8_C( 96), UINT8_C( 30),
                             UINT8_C( 96), UINT8_C(  0), UINT8_C( 73), UINT8_C(  0),
                             UINT8_C(115), UINT8_C(  0), UINT8_C(  0), UINT8_C( 96),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(123),
                             UINT8_C(233), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C( 85), UINT8_C(185), UINT8_C( 18), UINT8_C( 18),
                             UINT8_C(169), UINT8_C(233), UINT8_C( 34), UINT8_C(  0)) },
    { simde_x_mm256_set_epu8(UINT8_C(233), UINT8_C(108), UINT8_C(215), UINT8_C(181),
                             UINT8_C(100), UINT8_C( 72), UINT8_C(220), UINT8_C( 52),
                             UINT8_C( 24), UINT8_C(  0), UINT8_C(242), UINT8_C(125),
                             UINT8_C(116), UINT8_C(102), UINT8_C( 94), UINT8_C( 64),
                             UINT8_C(158), UINT8_C(177), UINT8_C(  3), UINT8_C(246),
                             UINT8_C(164), UINT8_C(171), UINT8_C(138), UINT8_C( 15),
                             UINT8_C( 30), UINT8_C( 68), UINT8_C(246), UINT8_C(147),
                             UINT8_C(162), UINT8_C( 13), UINT8_C( 93), UINT8_C( 58)),
      simde_x_mm256_set_epu8(UINT8_C(159), UINT8_C(224), UINT8_C(140), UINT8_C(117),
                             UINT8_C( 95), UINT8_C( 45), UINT8_C(144), UINT8_C( 69),
                             UINT8_C(174), UINT8_C( 34), UINT8_C(237), UINT8_C( 42),
                             UINT8_C( 78), UINT8_C(  7), UINT8_C(200), UINT8_C(173),
                             UINT8_C(137), UINT8_C( 75), UINT8_C(170), UINT8_C(110),
                             UINT8_C(182), UINT8_C(180), UINT8_C(173), UINT8_C(172),
                             UINT8_C(227), UINT8_C(222), UINT8_C(132), UINT8_C(  4),
                             UINT8_C(190), UINT8_C(  7), UINT8_C( 12), UINT8_C( 21)),
      simde_x_mm256_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(242),
                             UINT8_C(233), UINT8_C(215), UINT8_C(  0), UINT8_C(242),
                             UINT8_C(  0), UINT8_C(102), UINT8_C(  0), UINT8_C( 72),
                             UINT8_C(108), UINT8_C( 24), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(164), UINT8_C(  0), UINT8_C(177),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(147),
                             UINT8_C(  0), UINT8_C( 30), UINT8_C(246), UINT8_C(246)) },
    { simde_x_mm256_set_epu8(UINT8_C(239), UINT8_C(190), UINT8_C( 67), UINT8_C(183),
                             UINT8_C(162), UINT8_C( 57), UINT8_C(249), UINT8_C(218),
                             UINT8_C(176), UINT8_C(139), UINT8_C(  3), UINT8_C(248),
                             UINT8_C(214), UINT8_C( 36), UINT8_C(105), UINT8_C( 96),
                             UINT8_C( 80), UINT8_C(108), UINT8_C(142), UINT8_C(  2),
                             UINT8_C(196), UINT8_C(162), UINT8_C( 38), UINT8_C(  4),
                             UINT8_C(175), UINT8_C(178), UINT8_C( 88), UINT8_C(165),
                             UINT8_C(168), UINT8_C( 71), UINT8_C( 76), UINT8_C(232)),
      simde_x_mm256_set_epu8(UINT8_C(199), UINT8_C( 95), UINT8_C( 43), UINT8_C(216),
                             UINT8_C(128), UINT8_C(137), UINT8_C(103), UINT8_C( 51),
                             UINT8_C(121), UINT8_C( 84), UINT8_C(196), UINT8_C(101),
                             UINT8_C(251), UINT8_C(  4), UINT8_C( 89), UINT8_C( 54),
                             UINT8_C(216), UINT8_C(149), UINT8_C(145), UINT8_C( 73),
                             UINT8_C(162), UINT8_C( 73), UINT8_C(124), UINT8_C(146),
                             UINT8_C( 70), UINT8_C(151), UINT8_C(137), UINT8_C(218),
                             UINT8_C(230), UINT8_C(112), UINT8_C( 45), UINT8_C(226)),
      simde_x_mm256_set_epu8(UINT8_C(  0), UINT8_C(239), UINT8_C(162), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(176), UINT8_C(214),
                             UINT8_C(249), UINT8_C(248), UINT8_C(  0), UINT8_C(  3),
                             UINT8_C(  0), UINT8_C(248), UINT8_C(249), UINT8_C(139),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 38),
                             UINT8_C(  0), UINT8_C( 38), UINT8_C(  2), UINT8_C(  0),
                             UINT8_C(178), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(232), UINT8_C(142), UINT8_C(  0)) },
    { simde_x_mm256_set_epu8(UINT8_C( 81), UINT8_C(102), UINT8_C( 49), UINT8_C(133),
                             UINT8_C(250), UINT8_C( 13), UINT8_C(233), UINT8_C( 67),
                             UINT8_C(205), UINT8_C(254), UINT8_C( 41), UINT8_C(  6),
                             UINT8_C(117), UINT8_C(148), UINT8_C(222), UINT8_C(171),
                             UINT8_C(  8), UINT8_C( 99), UINT8_C( 93), UINT8_C( 39),
                             UINT8_C(103), UINT8_C(215), UINT8_C(216), UINT8_C( 75),
                             UINT8_C( 94), UINT8_C(171), UINT8_C( 41), UINT8_C(173),
                             UINT8_C(217), UINT8_C(138), UINT8_C(180), UINT8_C(160)),
      simde_x_mm256_set_epu8(UINT8_C(200), UINT8_C(107), UINT8_C(119), UINT8_C(216),
                             UINT8_C( 73), UINT8_C(187), UINT8_C(212), UINT8_C( 20),
                             UINT8_C( 22), UINT8_C( 11), UINT8_C(155), UINT8_C(106),
                             UINT8_C( 35), UINT8_C(108), UINT8_C( 43), UINT8_C( 58),
                             UINT8_C(102), UINT8_C(228), UINT8_C(  5), UINT8_C(227),
                             UINT8_C(106), UINT8_C(119), UINT8_C(184), UINT8_C(199),
                             UINT8_C( 57), UINT8_C(120), UINT8_C( 14), UINT8_C( 56),
                             UINT8_C( 82), UINT8_C( 19), UINT8_C( 33), UINT8_C( 82)),
      simde_x_mm256_set_epu8(UINT8_C(  0), UINT8_C(250), UINT8_C(205), UINT8_C(  0),
                             UINT8_C(233), UINT8_C(  0), UINT8_C(  0), UINT8_C(  6),
                             UINT8_C(254), UINT8_C(250), UINT8_C(  0), UINT8_C( 13),
                             UINT8_C(117), UINT8_C(133), UINT8_C(250), UINT8_C( 13),
                             UINT8_C(171), UINT8_C(  0), UINT8_C( 41), UINT8_C(  0),
                             UINT8_C(215), UINT8_C( 94), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(216), UINT8_C( 75), UINT8_C( 99), UINT8_C( 75),
                             UINT8_C(138), UINT8_C(217), UINT8_C(180), UINT8_C(138)) },
    { simde_x_mm256_set_epu8(UINT8_C(173), UINT8_C(218), UINT8_C( 20), UINT8_C( 45),
                             UINT8_C(110), UINT8_C( 90), UINT8_C(183), UINT8_C( 54),
                             UINT8_C(102), UINT8_C(126), UINT8_C(167), UINT8_C(135),
                             UINT8_C( 42), UINT8_C( 74), UINT8_C( 22), UINT8_C(137),
                             UINT8_C( 56), UINT8_C(  5), UINT8_C( 18), UINT8_C(105),
                             UINT8_C( 83), UINT8_C(146), UINT8_C(229), UINT8_C( 54),
                             UINT8_C(169), UINT8_C( 55), UINT8_C(228), UINT8_C(168),
                             UINT8_C(  1), UINT8_C(126), UINT8_C(169), UINT8_C(235)),
      simde_x_mm256_set_epu8(UINT8_C(254), UINT8_C( 56), UINT8_C(  4), UINT8_C(163),
                             UINT8_C( 35), UINT8_C( 52), UINT8_C(182), UINT8_C( 52),
                             UINT8_C(  1), UINT8_C( 45), UINT8_C(119), UINT8_C(169),
                             UINT8_C(175), UINT8_C(254), UINT8_C( 71), UINT8_C( 63),
                             UINT8_C( 24), UINT8_C(116), UINT8_C(194), UINT8_C(184),
                             UINT8_C(150), UINT8_C(167), UINT8_C(101), UINT8_C(133),
                             UINT8_C( 95), UINT8_C( 84), UINT8_C(199), UINT8_C(134),
                             UINT8_C(251), UINT8_C(101), UINT8_C( 38), UINT8_C( 43)),
      simde_x_mm256_set_epu8(UINT8_C(  0), UINT8_C( 54), UINT8_C(135), UINT8_C(  0),
                             UINT8_C( 42), UINT8_C(135), UINT8_C(  0), UINT8_C(135),
                             UINT8_C( 22), UINT8_C( 20), UINT8_C(102), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(102), UINT8_C(173),
                             UINT8_C( 54), UINT8_C(168), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(228), UINT8_C(  0),
                             UINT8_C( 56), UINT8_C(168), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(228), UINT8_C( 55), UINT8_C( 83)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_shuffle_epi8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_shuffle_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(    7953530), INT32_C( 1626445648), INT32_C(  338474584), INT32_C( 1314132322),
                            INT32_C( -165526664), INT32_C( 1862544223), INT32_C( 1736492928), INT32_C( -812721862)),
      simde_mm256_set_epi32(INT32_C( 1626445648), INT32_C( 1626445648), INT32_C( 1626445648), INT32_C( 1626445648),
                            INT32_C( 1862544223), INT32_C( 1862544223), INT32_C( 1862544223), INT32_C( 1862544223)) },
    { simde_mm256_set_epi32(INT32_C( 1571455878), INT32_C( 1654912639), INT32_C( 1961566624), INT32_C( 1369731271),
                            INT32_C( -802314883), INT32_C( 1895104772), INT32_C(-1294636783), INT32_C(   -2090880)),
      simde_mm256_set_epi32(INT32_C( 1654912639), INT32_C( 1654912639), INT32_C( 1654912639), INT32_C( 1654912639),
                            INT32_C( 1895104772), INT32_C( 1895104772), INT32_C( 1895104772), INT32_C( 1895104772)) },
    { simde_mm256_set_epi32(INT32_C(  716787034), INT32_C(  279681787), INT32_C(  -21556639), INT32_C(-1981624037),
                            INT32_C(  330738474), INT32_C(-1021797214), INT32_C(  117555788), INT32_C( 1295745410)),
      simde_mm256_set_epi32(INT32_C(  279681787), INT32_C(  279681787), INT32_C(  279681787), INT32_C(  279681787),
                            INT32_C(-1021797214), INT32_C(-1021797214), INT32_C(-1021797214), INT32_C(-1021797214)) },
    { simde_mm256_set_epi32(INT32_C(-1161252350), INT32_C( 1333069761), INT32_C( -746913676), INT32_C(-1297385199),
                            INT32_C( 1762552906), INT32_C( 1819201471), INT32_C(-1000519846), INT32_C( 2096577459)),
      simde_mm256_set_epi32(INT32_C( 1333069761), INT32_C( 1333069761), INT32_C( 1333069761), INT32_C( 1333069761),
                            INT32_C( 1819201471), INT32_C( 1819201471), INT32_C( 1819201471), INT32_C( 1819201471)) },
    { simde_mm256_set_epi32(INT32_C( 1772174643), INT32_C( 2012282759), INT32_C(  407185018), INT32_C(-1907491648),
                            INT32_C(-1950535235), INT32_C(   59473466), INT32_C( -212502371), INT32_C( -171378108)),
      simde_mm256_set_epi32(INT32_C( 2012282759), INT32_C( 2012282759), INT32_C( 2012282759), INT32_C( 2012282759),
                            INT32_C(   59473466), INT32_C(   59473466), INT32_C(   59473466), INT32_C(   59473466)) },
    { simde_mm256_set_epi32(INT32_C(-1786867928), INT32_C(  183968379), INT32_C(-1710767833), INT32_C(  419209661),
                            INT32_C( 1470446764), INT32_C( -838411183), INT32_C( 1711518303), INT32_C(-1929674962)),
      simde_mm256_set_epi32(INT32_C(  183968379), INT32_C(  183968379), INT32_C(  183968379), INT32_C(  183968379),
                            INT32_C( -838411183), INT32_C( -838411183), INT32_C( -838411183), INT32_C( -838411183)) },
    { simde_mm256_set_epi32(INT32_C(-1296827172), INT32_C(-1426782704), INT32_C(  461497475), INT32_C(  857689164),
                            INT32_C( -127267738), INT32_C( -225620875), INT32_C( -476268334), INT32_C(     275470)),
      simde_mm256_set_epi32(INT32_C(-1426782704), INT32_C(-1426782704), INT32_C(-1426782704), INT32_C(-1426782704),
                            INT32_C( -225620875), INT32_C( -225620875), INT32_C( -225620875), INT32_C( -225620875)) },
    { simde_mm256_set_epi32(INT32_C(  492995740), INT32_C(-1095702956), INT32_C( 1746718954), INT32_C( 2048506117),
                            INT32_C(  180431580), INT32_C(-1463759846), INT32_C( 1212242273), INT32_C(-1640731308)),
      simde_mm256_set_epi32(INT32_C(-1095702956), INT32_C(-1095702956), INT32_C(-1095702956), INT32_C(-1095702956),
                            INT32_C(-1463759846), INT32_C(-1463759846), INT32_C(-1463759846), INT32_C(-1463759846)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_shuffle_epi32(test_vec[i].a, 0xaa);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_shufflehi_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[16];
    const int16_t r[16];
  } test_vec[] = {
    { { -INT16_C(  2516), -INT16_C( 21378),  INT16_C(  2569),  INT16_C( 28926),  INT16_C( 23995),  INT16_C( 19519),  INT16_C(   305),  INT16_C(  3639),
         INT16_C( 27436),  INT16_C( 23441), -INT16_C( 10560),  INT16_C(  8981), -INT16_C( 30716), -INT16_C(  1213), -INT16_C( 20665),  INT16_C( 29550) },
      { -INT16_C(  2516), -INT16_C( 21378),  INT16_C(  2569),  INT16_C( 28926),  INT16_C( 19519),  INT16_C( 19519),  INT16_C( 19519),  INT16_C(   305),
         INT16_C( 27436),  INT16_C( 23441), -INT16_C( 10560),  INT16_C(  8981), -INT16_C(  1213), -INT16_C(  1213), -INT16_C(  1213), -INT16_C( 20665) } },
    { { -INT16_C(  4955), -INT16_C( 20961),  INT16_C(  7670), -INT16_C( 19938),  INT16_C( 24187), -INT16_C( 21250),  INT16_C( 13663), -INT16_C( 29766),
         INT16_C( 19360),  INT16_C( 24806), -INT16_C(  1247),  INT16_C(  9603), -INT16_C( 14717), -INT16_C( 13792), -INT16_C( 29067),  INT16_C(  6973) },
      { -INT16_C(  4955), -INT16_C( 20961),  INT16_C(  7670), -INT16_C( 19938), -INT16_C( 21250), -INT16_C( 21250), -INT16_C( 21250),  INT16_C( 13663),
         INT16_C( 19360),  INT16_C( 24806), -INT16_C(  1247),  INT16_C(  9603), -INT16_C( 13792), -INT16_C( 13792), -INT16_C( 13792), -INT16_C( 29067) } },
    { {  INT16_C( 23675),  INT16_C( 29129), -INT16_C(  6022), -INT16_C(  2781),  INT16_C(  8518), -INT16_C( 23135),  INT16_C( 23638), -INT16_C(  2512),
         INT16_C(  5799), -INT16_C( 13993), -INT16_C(  9710), -INT16_C( 27154),  INT16_C(  4001),  INT16_C(  5728), -INT16_C( 25187),  INT16_C(  6193) },
      {  INT16_C( 23675),  INT16_C( 29129), -INT16_C(  6022), -INT16_C(  2781), -INT16_C( 23135), -INT16_C( 23135), -INT16_C( 23135),  INT16_C( 23638),
         INT16_C(  5799), -INT16_C( 13993), -INT16_C(  9710), -INT16_C( 27154),  INT16_C(  5728),  INT16_C(  5728),  INT16_C(  5728), -INT16_C( 25187) } },
    { { -INT16_C(  1030),  INT16_C( 29834), -INT16_C( 21021),  INT16_C( 10601),  INT16_C(  2767),  INT16_C(  9678), -INT16_C(   410),  INT16_C(  3612),
         INT16_C( 29460),  INT16_C(  9943), -INT16_C( 15027), -INT16_C(  4420),  INT16_C(  7380),  INT16_C( 29189),  INT16_C( 14009), -INT16_C( 19574) },
      { -INT16_C(  1030),  INT16_C( 29834), -INT16_C( 21021),  INT16_C( 10601),  INT16_C(  9678),  INT16_C(  9678),  INT16_C(  9678), -INT16_C(   410),
         INT16_C( 29460),  INT16_C(  9943), -INT16_C( 15027), -INT16_C(  4420),  INT16_C( 29189),  INT16_C( 29189),  INT16_C( 29189),  INT16_C( 14009) } },
    { {  INT16_C(  5169),  INT16_C(  5159), -INT16_C( 28478), -INT16_C( 28355),  INT16_C(  2971),  INT16_C(   438), -INT16_C( 11767),  INT16_C(  7695),
        -INT16_C(  6587), -INT16_C( 27836),  INT16_C(   172), -INT16_C( 32639), -INT16_C( 31204), -INT16_C( 10510),  INT16_C( 32189), -INT16_C(  4471) },
      {  INT16_C(  5169),  INT16_C(  5159), -INT16_C( 28478), -INT16_C( 28355),  INT16_C(   438),  INT16_C(   438),  INT16_C(   438), -INT16_C( 11767),
        -INT16_C(  6587), -INT16_C( 27836),  INT16_C(   172), -INT16_C( 32639), -INT16_C( 10510), -INT16_C( 10510), -INT16_C( 10510),  INT16_C( 32189) } },
    { { -INT16_C( 20079),  INT16_C( 21251),  INT16_C( 16449), -INT16_C(  8988), -INT16_C( 25780),  INT16_C( 21982), -INT16_C(  4755), -INT16_C( 19597),
        -INT16_C( 18220), -INT16_C( 32698), -INT16_C( 14408), -INT16_C( 11008), -INT16_C(  3250),  INT16_C(  2987),  INT16_C( 13424),  INT16_C(   505) },
      { -INT16_C( 20079),  INT16_C( 21251),  INT16_C( 16449), -INT16_C(  8988),  INT16_C( 21982),  INT16_C( 21982),  INT16_C( 21982), -INT16_C(  4755),
        -INT16_C( 18220), -INT16_C( 32698), -INT16_C( 14408), -INT16_C( 11008),  INT16_C(  2987),  INT16_C(  2987),  INT16_C(  2987),  INT16_C( 13424) } },
    { { -INT16_C(   795),  INT16_C( 10069),  INT16_C( 14653), -INT16_C( 30461), -INT16_C(  7724),  INT16_C( 17118),  INT16_C( 21199), -INT16_C( 23563),
         INT16_C( 15114), -INT16_C( 15837),  INT16_C(  8962),  INT16_C( 20631),  INT16_C( 16918), -INT16_C( 31141),  INT16_C( 21879),  INT16_C( 23688) },
      { -INT16_C(   795),  INT16_C( 10069),  INT16_C( 14653), -INT16_C( 30461),  INT16_C( 17118),  INT16_C( 17118),  INT16_C( 17118),  INT16_C( 21199),
         INT16_C( 15114), -INT16_C( 15837),  INT16_C(  8962),  INT16_C( 20631), -INT16_C( 31141), -INT16_C( 31141), -INT16_C( 31141),  INT16_C( 21879) } },
    { { -INT16_C(  8879), -INT16_C( 29053), -INT16_C( 30954), -INT16_C(  5353), -INT16_C(  2456),  INT16_C( 14125),  INT16_C(  8776),  INT16_C( 21210),
        -INT16_C(   675),  INT16_C( 24340), -INT16_C( 21471),  INT16_C( 14256),  INT16_C(  3054),  INT16_C( 26046),  INT16_C( 18016), -INT16_C( 19774) },
      { -INT16_C(  8879), -INT16_C( 29053), -INT16_C( 30954), -INT16_C(  5353),  INT16_C( 14125),  INT16_C( 14125),  INT16_C( 14125),  INT16_C(  8776),
        -INT16_C(   675),  INT16_C( 24340), -INT16_C( 21471),  INT16_C( 14256),  INT16_C( 26046),  INT16_C( 26046),  INT16_C( 26046),  INT16_C( 18016) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i r = simde_mm256_shufflehi_epi16(a, 149);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_shufflelo_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C( -2179), INT16_C( 30608), INT16_C( 19808), INT16_C(-28307),
                            INT16_C( 25873), INT16_C( -9642), INT16_C(    56), INT16_C(   338),
                            INT16_C(-29256), INT16_C(  9693), INT16_C( 18909), INT16_C(  3915),
                            INT16_C(  2653), INT16_C( 26676), INT16_C(-23808), INT16_C( 25025)),
      simde_mm256_set_epi16(INT16_C( -2179), INT16_C( 30608), INT16_C( 19808), INT16_C(-28307),
                            INT16_C( -9642), INT16_C(    56), INT16_C(    56), INT16_C(    56),
                            INT16_C(-29256), INT16_C(  9693), INT16_C( 18909), INT16_C(  3915),
                            INT16_C( 26676), INT16_C(-23808), INT16_C(-23808), INT16_C(-23808)) },
    { simde_mm256_set_epi16(INT16_C( 18348), INT16_C(-29440), INT16_C( -6978), INT16_C( 22811),
                            INT16_C(-31995), INT16_C( 32157), INT16_C( -1931), INT16_C( 15650),
                            INT16_C( 25681), INT16_C( 27258), INT16_C( -9033), INT16_C( 28640),
                            INT16_C(-16045), INT16_C(  2509), INT16_C(-23756), INT16_C(  4399)),
      simde_mm256_set_epi16(INT16_C( 18348), INT16_C(-29440), INT16_C( -6978), INT16_C( 22811),
                            INT16_C( 32157), INT16_C( -1931), INT16_C( -1931), INT16_C( -1931),
                            INT16_C( 25681), INT16_C( 27258), INT16_C( -9033), INT16_C( 28640),
                            INT16_C(  2509), INT16_C(-23756), INT16_C(-23756), INT16_C(-23756)) },
    { simde_mm256_set_epi16(INT16_C( 32389), INT16_C(-20100), INT16_C( 26342), INT16_C(-12082),
                            INT16_C( 27809), INT16_C( 10468), INT16_C( 29312), INT16_C( -5875),
                            INT16_C( 10801), INT16_C(-11039), INT16_C(-32348), INT16_C(-19420),
                            INT16_C(-32754), INT16_C(-20940), INT16_C( 17235), INT16_C(-29506)),
      simde_mm256_set_epi16(INT16_C( 32389), INT16_C(-20100), INT16_C( 26342), INT16_C(-12082),
                            INT16_C( 10468), INT16_C( 29312), INT16_C( 29312), INT16_C( 29312),
                            INT16_C( 10801), INT16_C(-11039), INT16_C(-32348), INT16_C(-19420),
                            INT16_C(-20940), INT16_C( 17235), INT16_C( 17235), INT16_C( 17235)) },
    { simde_mm256_set_epi16(INT16_C(-30489), INT16_C( 27619), INT16_C(-16535), INT16_C(-14927),
                            INT16_C( 10253), INT16_C( 31108), INT16_C(    46), INT16_C( 10088),
                            INT16_C(-25344), INT16_C( -5295), INT16_C( 22387), INT16_C( 27540),
                            INT16_C(-30287), INT16_C(  3057), INT16_C( -2994), INT16_C( 31366)),
      simde_mm256_set_epi16(INT16_C(-30489), INT16_C( 27619), INT16_C(-16535), INT16_C(-14927),
                            INT16_C( 31108), INT16_C(    46), INT16_C(    46), INT16_C(    46),
                            INT16_C(-25344), INT16_C( -5295), INT16_C( 22387), INT16_C( 27540),
                            INT16_C(  3057), INT16_C( -2994), INT16_C( -2994), INT16_C( -2994)) },
    { simde_mm256_set_epi16(INT16_C( -7255), INT16_C(-20910), INT16_C( 13809), INT16_C(  9595),
                            INT16_C( -1612), INT16_C( 23053), INT16_C(  7111), INT16_C(-11581),
                            INT16_C(-12829), INT16_C( 20686), INT16_C(-16865), INT16_C(  5649),
                            INT16_C(-25623), INT16_C( -2444), INT16_C( 11296), INT16_C(  7832)),
      simde_mm256_set_epi16(INT16_C( -7255), INT16_C(-20910), INT16_C( 13809), INT16_C(  9595),
                            INT16_C( 23053), INT16_C(  7111), INT16_C(  7111), INT16_C(  7111),
                            INT16_C(-12829), INT16_C( 20686), INT16_C(-16865), INT16_C(  5649),
                            INT16_C( -2444), INT16_C( 11296), INT16_C( 11296), INT16_C( 11296)) },
    { simde_mm256_set_epi16(INT16_C(-26606), INT16_C(  4305), INT16_C( 13601), INT16_C(  6245),
                            INT16_C( -6095), INT16_C(  5518), INT16_C(-13456), INT16_C(-16529),
                            INT16_C(-18890), INT16_C( 32306), INT16_C( 18285), INT16_C(  9413),
                            INT16_C( -4831), INT16_C( 31394), INT16_C(  7403), INT16_C(  8863)),
      simde_mm256_set_epi16(INT16_C(-26606), INT16_C(  4305), INT16_C( 13601), INT16_C(  6245),
                            INT16_C(  5518), INT16_C(-13456), INT16_C(-13456), INT16_C(-13456),
                            INT16_C(-18890), INT16_C( 32306), INT16_C( 18285), INT16_C(  9413),
                            INT16_C( 31394), INT16_C(  7403), INT16_C(  7403), INT16_C(  7403)) },
    { simde_mm256_set_epi16(INT16_C(-22090), INT16_C(  3292), INT16_C(-24353), INT16_C(-13630),
                            INT16_C(  1106), INT16_C( 20149), INT16_C( 12111), INT16_C( -9481),
                            INT16_C(  8059), INT16_C( 23722), INT16_C(-24846), INT16_C(-19082),
                            INT16_C( 18649), INT16_C( 23374), INT16_C(-15098), INT16_C(  5703)),
      simde_mm256_set_epi16(INT16_C(-22090), INT16_C(  3292), INT16_C(-24353), INT16_C(-13630),
                            INT16_C( 20149), INT16_C( 12111), INT16_C( 12111), INT16_C( 12111),
                            INT16_C(  8059), INT16_C( 23722), INT16_C(-24846), INT16_C(-19082),
                            INT16_C( 23374), INT16_C(-15098), INT16_C(-15098), INT16_C(-15098)) },
    { simde_mm256_set_epi16(INT16_C( 31351), INT16_C(-28147), INT16_C(-17219), INT16_C( 13266),
                            INT16_C( 26309), INT16_C(  5438), INT16_C( -9093), INT16_C(  -541),
                            INT16_C( -7590), INT16_C(-28792), INT16_C(  9449), INT16_C(-27866),
                            INT16_C(  7986), INT16_C(-24032), INT16_C( 23568), INT16_C( 18541)),
      simde_mm256_set_epi16(INT16_C( 31351), INT16_C(-28147), INT16_C(-17219), INT16_C( 13266),
                            INT16_C(  5438), INT16_C( -9093), INT16_C( -9093), INT16_C( -9093),
                            INT16_C( -7590), INT16_C(-28792), INT16_C(  9449), INT16_C(-27866),
                            INT16_C(-24032), INT16_C( 23568), INT16_C( 23568), INT16_C( 23568)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_shufflelo_epi16(test_vec[i].a, 149);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_sign_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C(-124), INT8_C( -48), INT8_C(  10), INT8_C(  63),
                           INT8_C( -84), INT8_C( 103), INT8_C( -14), INT8_C(  63),
                           INT8_C(  82), INT8_C( -68), INT8_C( -17), INT8_C( 117),
                           INT8_C( -80), INT8_C(-104), INT8_C(  37), INT8_C(  59),
                           INT8_C( 121), INT8_C(  11), INT8_C( -20), INT8_C( -38),
                           INT8_C(  36), INT8_C( -63), INT8_C( -89), INT8_C( -39),
                           INT8_C( 115), INT8_C(  43), INT8_C(-116), INT8_C(   8),
                           INT8_C( -68), INT8_C( -15), INT8_C(  35), INT8_C( -73)),
      simde_mm256_set_epi8(INT8_C(  64), INT8_C( -46), INT8_C( 104), INT8_C(  85),
                           INT8_C( -66), INT8_C( 103), INT8_C( -19), INT8_C( -91),
                           INT8_C(-116), INT8_C(  86), INT8_C(  54), INT8_C(  90),
                           INT8_C(-128), INT8_C( 120), INT8_C( 127), INT8_C( -41),
                           INT8_C(  56), INT8_C(  45), INT8_C( -60), INT8_C(  14),
                           INT8_C(  -4), INT8_C( -42), INT8_C( -79), INT8_C(  83),
                           INT8_C(  72), INT8_C(  65), INT8_C( -99), INT8_C( 111),
                           INT8_C( -84), INT8_C( 103), INT8_C( -29), INT8_C( -65)),
      simde_mm256_set_epi8(INT8_C(-124), INT8_C(  48), INT8_C(  10), INT8_C(  63),
                           INT8_C(  84), INT8_C( 103), INT8_C(  14), INT8_C( -63),
                           INT8_C( -82), INT8_C( -68), INT8_C( -17), INT8_C( 117),
                           INT8_C(  80), INT8_C(-104), INT8_C(  37), INT8_C( -59),
                           INT8_C( 121), INT8_C(  11), INT8_C(  20), INT8_C( -38),
                           INT8_C( -36), INT8_C(  63), INT8_C(  89), INT8_C( -39),
                           INT8_C( 115), INT8_C(  43), INT8_C( 116), INT8_C(   8),
                           INT8_C(  68), INT8_C( -15), INT8_C( -35), INT8_C(  73)) },
    { simde_mm256_set_epi8(INT8_C( 122), INT8_C( 101), INT8_C(  57), INT8_C(-117),
                           INT8_C( -22), INT8_C( -56), INT8_C(  -5), INT8_C(  60),
                           INT8_C( 110), INT8_C( -69), INT8_C( -30), INT8_C( -83),
                           INT8_C(  50), INT8_C(  67), INT8_C( -85), INT8_C(  24),
                           INT8_C(  64), INT8_C( 117), INT8_C(  23), INT8_C( -97),
                           INT8_C(  88), INT8_C(  95), INT8_C(  73), INT8_C(  84),
                           INT8_C(  44), INT8_C( 108), INT8_C( -73), INT8_C(-105),
                           INT8_C(  58), INT8_C( 125), INT8_C(  21), INT8_C(  -8)),
      simde_mm256_set_epi8(INT8_C( -54), INT8_C( -52), INT8_C( 108), INT8_C(  17),
                           INT8_C( -61), INT8_C(  77), INT8_C( -68), INT8_C(-127),
                           INT8_C(  55), INT8_C( -25), INT8_C( -44), INT8_C(-111),
                           INT8_C(  12), INT8_C( -35), INT8_C( 116), INT8_C(  64),
                           INT8_C( -32), INT8_C( 127), INT8_C( -91), INT8_C( -72),
                           INT8_C( -92), INT8_C(  13), INT8_C(  97), INT8_C( -10),
                           INT8_C(-120), INT8_C( -40), INT8_C( -37), INT8_C( -49),
                           INT8_C( 123), INT8_C( -15), INT8_C( 100), INT8_C( -27)),
      simde_mm256_set_epi8(INT8_C(-122), INT8_C(-101), INT8_C(  57), INT8_C(-117),
                           INT8_C(  22), INT8_C( -56), INT8_C(   5), INT8_C( -60),
                           INT8_C( 110), INT8_C(  69), INT8_C(  30), INT8_C(  83),
                           INT8_C(  50), INT8_C( -67), INT8_C( -85), INT8_C(  24),
                           INT8_C( -64), INT8_C( 117), INT8_C( -23), INT8_C(  97),
                           INT8_C( -88), INT8_C(  95), INT8_C(  73), INT8_C( -84),
                           INT8_C( -44), INT8_C(-108), INT8_C(  73), INT8_C( 105),
                           INT8_C(  58), INT8_C(-125), INT8_C(  21), INT8_C(   8)) },
    { simde_mm256_set_epi8(INT8_C(  80), INT8_C(-104), INT8_C(  33), INT8_C( -96),
                           INT8_C( 105), INT8_C( -57), INT8_C(-123), INT8_C(  89),
                           INT8_C(  62), INT8_C( -47), INT8_C( -92), INT8_C( -33),
                           INT8_C( 113), INT8_C(  44), INT8_C( 125), INT8_C( -43),
                           INT8_C(  28), INT8_C( -43), INT8_C( 122), INT8_C(-115),
                           INT8_C(  45), INT8_C(  72), INT8_C(  66), INT8_C(  86),
                           INT8_C( -11), INT8_C(  17), INT8_C(  64), INT8_C( -36),
                           INT8_C( -86), INT8_C(  -6), INT8_C( -64), INT8_C(  31)),
      simde_mm256_set_epi8(INT8_C(  65), INT8_C( -61), INT8_C(  83), INT8_C( -37),
                           INT8_C(-100), INT8_C(  96), INT8_C(  90), INT8_C( -24),
                           INT8_C(  86), INT8_C( -18), INT8_C( -62), INT8_C(-110),
                           INT8_C(-107), INT8_C(  -6), INT8_C(   9), INT8_C(  -6),
                           INT8_C(  80), INT8_C( -71), INT8_C( -81), INT8_C( 127),
                           INT8_C( -21), INT8_C( -14), INT8_C(  27), INT8_C(  82),
                           INT8_C(-126), INT8_C(  70), INT8_C(  18), INT8_C( -75),
                           INT8_C( 126), INT8_C(-103), INT8_C(-109), INT8_C( 116)),
      simde_mm256_set_epi8(INT8_C(  80), INT8_C( 104), INT8_C(  33), INT8_C(  96),
                           INT8_C(-105), INT8_C( -57), INT8_C(-123), INT8_C( -89),
                           INT8_C(  62), INT8_C(  47), INT8_C(  92), INT8_C(  33),
                           INT8_C(-113), INT8_C( -44), INT8_C( 125), INT8_C(  43),
                           INT8_C(  28), INT8_C(  43), INT8_C(-122), INT8_C(-115),
                           INT8_C( -45), INT8_C( -72), INT8_C(  66), INT8_C(  86),
                           INT8_C(  11), INT8_C(  17), INT8_C(  64), INT8_C(  36),
                           INT8_C( -86), INT8_C(   6), INT8_C(  64), INT8_C(  31)) },
    { simde_mm256_set_epi8(INT8_C(  88), INT8_C(  57), INT8_C( -36), INT8_C(  10),
                           INT8_C(  93), INT8_C( -67), INT8_C(  35), INT8_C(   4),
                           INT8_C(  82), INT8_C( -81), INT8_C(  18), INT8_C(  74),
                           INT8_C(-119), INT8_C( -48), INT8_C(  50), INT8_C( -37),
                           INT8_C(  85), INT8_C(  99), INT8_C( -94), INT8_C(-127),
                           INT8_C( 126), INT8_C(  36), INT8_C(  87), INT8_C( -95),
                           INT8_C( 111), INT8_C(  86), INT8_C( -49), INT8_C(  -3),
                           INT8_C( -39), INT8_C(  19), INT8_C(  -3), INT8_C( -93)),
      simde_mm256_set_epi8(INT8_C( -50), INT8_C(  47), INT8_C( 110), INT8_C(  89),
                           INT8_C( -98), INT8_C( -62), INT8_C( 100), INT8_C( -53),
                           INT8_C( -92), INT8_C(  -2), INT8_C(  91), INT8_C( -12),
                           INT8_C( -28), INT8_C( -91), INT8_C( -27), INT8_C(  32),
                           INT8_C(-110), INT8_C( -42), INT8_C(  73), INT8_C(-102),
                           INT8_C(-122), INT8_C(  48), INT8_C(  -4), INT8_C( -50),
                           INT8_C(   9), INT8_C( -36), INT8_C( 102), INT8_C(-126),
                           INT8_C(  34), INT8_C(  96), INT8_C( -57), INT8_C(   7)),
      simde_mm256_set_epi8(INT8_C( -88), INT8_C(  57), INT8_C( -36), INT8_C(  10),
                           INT8_C( -93), INT8_C(  67), INT8_C(  35), INT8_C(  -4),
                           INT8_C( -82), INT8_C(  81), INT8_C(  18), INT8_C( -74),
                           INT8_C( 119), INT8_C(  48), INT8_C( -50), INT8_C( -37),
                           INT8_C( -85), INT8_C( -99), INT8_C( -94), INT8_C( 127),
                           INT8_C(-126), INT8_C(  36), INT8_C( -87), INT8_C(  95),
                           INT8_C( 111), INT8_C( -86), INT8_C( -49), INT8_C(   3),
                           INT8_C( -39), INT8_C(  19), INT8_C(   3), INT8_C( -93)) },
    { simde_mm256_set_epi8(INT8_C(  77), INT8_C( -76), INT8_C( 117), INT8_C( -63),
                           INT8_C( -98), INT8_C( -23), INT8_C(  99), INT8_C(  28),
                           INT8_C( 114), INT8_C( -99), INT8_C( 117), INT8_C( -16),
                           INT8_C( -35), INT8_C(  79), INT8_C( -35), INT8_C( -40),
                           INT8_C( 116), INT8_C(  97), INT8_C(  16), INT8_C(  32),
                           INT8_C(-122), INT8_C(  -2), INT8_C( 116), INT8_C( -53),
                           INT8_C(-103), INT8_C( -52), INT8_C(  50), INT8_C( -27),
                           INT8_C(-112), INT8_C( 115), INT8_C( -42), INT8_C( -87)),
      simde_mm256_set_epi8(INT8_C(  49), INT8_C( 101), INT8_C( -91), INT8_C( -11),
                           INT8_C(  38), INT8_C( 110), INT8_C(  63), INT8_C( 118),
                           INT8_C(  65), INT8_C( -80), INT8_C( 118), INT8_C( -75),
                           INT8_C( 103), INT8_C( -76), INT8_C(  10), INT8_C( -66),
                           INT8_C( 107), INT8_C(-120), INT8_C(  11), INT8_C( -38),
                           INT8_C( -59), INT8_C( -40), INT8_C(  -9), INT8_C( 123),
                           INT8_C(  -3), INT8_C( -44), INT8_C( -64), INT8_C(  26),
                           INT8_C( -30), INT8_C(  14), INT8_C(-100), INT8_C(  22)),
      simde_mm256_set_epi8(INT8_C(  77), INT8_C( -76), INT8_C(-117), INT8_C(  63),
                           INT8_C( -98), INT8_C( -23), INT8_C(  99), INT8_C(  28),
                           INT8_C( 114), INT8_C(  99), INT8_C( 117), INT8_C(  16),
                           INT8_C( -35), INT8_C( -79), INT8_C( -35), INT8_C(  40),
                           INT8_C( 116), INT8_C( -97), INT8_C(  16), INT8_C( -32),
                           INT8_C( 122), INT8_C(   2), INT8_C(-116), INT8_C( -53),
                           INT8_C( 103), INT8_C(  52), INT8_C( -50), INT8_C( -27),
                           INT8_C( 112), INT8_C( 115), INT8_C(  42), INT8_C( -87)) },
    { simde_mm256_set_epi8(INT8_C( -77), INT8_C(  55), INT8_C(  69), INT8_C(   5),
                           INT8_C( -56), INT8_C(  30), INT8_C( 123), INT8_C(  84),
                           INT8_C(  88), INT8_C(-108), INT8_C(  47), INT8_C( -23),
                           INT8_C( -58), INT8_C( -90), INT8_C(  18), INT8_C(-120),
                           INT8_C( -15), INT8_C(  43), INT8_C( 102), INT8_C(   1),
                           INT8_C( -38), INT8_C(  45), INT8_C( 120), INT8_C( 124),
                           INT8_C(  39), INT8_C(  42), INT8_C(-106), INT8_C(  42),
                           INT8_C(  50), INT8_C( 123), INT8_C(  83), INT8_C( -75)),
      simde_mm256_set_epi8(INT8_C( -99), INT8_C(   1), INT8_C( 103), INT8_C( -24),
                           INT8_C(  77), INT8_C(  77), INT8_C( -55), INT8_C(  64),
                           INT8_C( -51), INT8_C( 124), INT8_C( -74), INT8_C(  85),
                           INT8_C(  26), INT8_C( -10), INT8_C( -19), INT8_C( -23),
                           INT8_C( -11), INT8_C(  18), INT8_C(-116), INT8_C(  62),
                           INT8_C( -42), INT8_C( -20), INT8_C( 104), INT8_C(-123),
                           INT8_C(-113), INT8_C(   1), INT8_C( -68), INT8_C( -90),
                           INT8_C(   1), INT8_C( -32), INT8_C(   1), INT8_C(-108)),
      simde_mm256_set_epi8(INT8_C(  77), INT8_C(  55), INT8_C(  69), INT8_C(  -5),
                           INT8_C( -56), INT8_C(  30), INT8_C(-123), INT8_C(  84),
                           INT8_C( -88), INT8_C(-108), INT8_C( -47), INT8_C( -23),
                           INT8_C( -58), INT8_C(  90), INT8_C( -18), INT8_C( 120),
                           INT8_C(  15), INT8_C(  43), INT8_C(-102), INT8_C(   1),
                           INT8_C(  38), INT8_C( -45), INT8_C( 120), INT8_C(-124),
                           INT8_C( -39), INT8_C(  42), INT8_C( 106), INT8_C( -42),
                           INT8_C(  50), INT8_C(-123), INT8_C(  83), INT8_C(  75)) },
    { simde_mm256_set_epi8(INT8_C( 124), INT8_C(  74), INT8_C( -32), INT8_C(  30),
                           INT8_C(-109), INT8_C(  52), INT8_C(  31), INT8_C(  94),
                           INT8_C( -78), INT8_C( -59), INT8_C(  -8), INT8_C(  92),
                           INT8_C(   9), INT8_C( -91), INT8_C( -63), INT8_C( -72),
                           INT8_C( -85), INT8_C(   1), INT8_C(  60), INT8_C(  84),
                           INT8_C(  43), INT8_C( -36), INT8_C(-112), INT8_C(   6),
                           INT8_C( -19), INT8_C(-123), INT8_C(  80), INT8_C( -53),
                           INT8_C( -54), INT8_C(  94), INT8_C(   3), INT8_C( 112)),
      simde_mm256_set_epi8(INT8_C( -31), INT8_C( -40), INT8_C( -74), INT8_C(  49),
                           INT8_C(-107), INT8_C(-106), INT8_C( 102), INT8_C(   9),
                           INT8_C( 117), INT8_C(  90), INT8_C( 114), INT8_C( -61),
                           INT8_C(  32), INT8_C(  -8), INT8_C(  -4), INT8_C( -57),
                           INT8_C( 123), INT8_C(-125), INT8_C(   9), INT8_C(-128),
                           INT8_C(-101), INT8_C(   1), INT8_C(  32), INT8_C( -63),
                           INT8_C(  99), INT8_C(  41), INT8_C(  38), INT8_C(-106),
                           INT8_C( -65), INT8_C(  58), INT8_C( -23), INT8_C( 120)),
      simde_mm256_set_epi8(INT8_C(-124), INT8_C( -74), INT8_C(  32), INT8_C(  30),
                           INT8_C( 109), INT8_C( -52), INT8_C(  31), INT8_C(  94),
                           INT8_C( -78), INT8_C( -59), INT8_C(  -8), INT8_C( -92),
                           INT8_C(   9), INT8_C(  91), INT8_C(  63), INT8_C(  72),
                           INT8_C( -85), INT8_C(  -1), INT8_C(  60), INT8_C( -84),
                           INT8_C( -43), INT8_C( -36), INT8_C(-112), INT8_C(  -6),
                           INT8_C( -19), INT8_C(-123), INT8_C(  80), INT8_C(  53),
                           INT8_C(  54), INT8_C(  94), INT8_C(  -3), INT8_C( 112)) },
    { simde_mm256_set_epi8(INT8_C( -49), INT8_C(-106), INT8_C(  23), INT8_C(  37),
                           INT8_C( -87), INT8_C( 105), INT8_C(  66), INT8_C( -47),
                           INT8_C(-100), INT8_C( -77), INT8_C(-123), INT8_C( -73),
                           INT8_C(  -4), INT8_C( -48), INT8_C(  97), INT8_C( -78),
                           INT8_C(  21), INT8_C( 125), INT8_C(-128), INT8_C(  30),
                           INT8_C( -87), INT8_C(  46), INT8_C( -21), INT8_C( -69),
                           INT8_C(  -4), INT8_C(  15), INT8_C( -17), INT8_C( -95),
                           INT8_C(   7), INT8_C(  32), INT8_C(  93), INT8_C( -95)),
      simde_mm256_set_epi8(INT8_C(  27), INT8_C(  61), INT8_C(  26), INT8_C(  17),
                           INT8_C( 110), INT8_C(  35), INT8_C(  70), INT8_C( -63),
                           INT8_C(  51), INT8_C( -79), INT8_C( -47), INT8_C(  74),
                           INT8_C(  44), INT8_C(   7), INT8_C( 125), INT8_C( -59),
                           INT8_C( -26), INT8_C( -74), INT8_C( -55), INT8_C( -22),
                           INT8_C(  18), INT8_C( -40), INT8_C(  33), INT8_C( 126),
                           INT8_C(-127), INT8_C(  -2), INT8_C(-125), INT8_C(  79),
                           INT8_C( 113), INT8_C(  14), INT8_C( 113), INT8_C(  21)),
      simde_mm256_set_epi8(INT8_C( -49), INT8_C(-106), INT8_C(  23), INT8_C(  37),
                           INT8_C( -87), INT8_C( 105), INT8_C(  66), INT8_C(  47),
                           INT8_C(-100), INT8_C(  77), INT8_C( 123), INT8_C( -73),
                           INT8_C(  -4), INT8_C( -48), INT8_C(  97), INT8_C(  78),
                           INT8_C( -21), INT8_C(-125), INT8_C(-128), INT8_C( -30),
                           INT8_C( -87), INT8_C( -46), INT8_C( -21), INT8_C( -69),
                           INT8_C(   4), INT8_C( -15), INT8_C(  17), INT8_C( -95),
                           INT8_C(   7), INT8_C(  32), INT8_C(  93), INT8_C( -95)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_sign_epi8(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_sign_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(-31536), INT16_C(  2623), INT16_C(-21401), INT16_C( -3521),
                            INT16_C( 21180), INT16_C( -4235), INT16_C(-20328), INT16_C(  9531),
                            INT16_C( 30987), INT16_C( -4902), INT16_C(  9409), INT16_C(-22567),
                            INT16_C( 29483), INT16_C(-29688), INT16_C(-17167), INT16_C(  9143)),
      simde_mm256_set_epi16(INT16_C( 16594), INT16_C( 26709), INT16_C(-16793), INT16_C( -4699),
                            INT16_C(-29610), INT16_C( 13914), INT16_C(-32648), INT16_C( 32727),
                            INT16_C( 14381), INT16_C(-15346), INT16_C(  -810), INT16_C(-20141),
                            INT16_C( 18497), INT16_C(-25233), INT16_C(-21401), INT16_C( -7233)),
      simde_mm256_set_epi16(INT16_C(-31536), INT16_C(  2623), INT16_C( 21401), INT16_C(  3521),
                            INT16_C(-21180), INT16_C( -4235), INT16_C( 20328), INT16_C(  9531),
                            INT16_C( 30987), INT16_C(  4902), INT16_C( -9409), INT16_C( 22567),
                            INT16_C( 29483), INT16_C( 29688), INT16_C( 17167), INT16_C( -9143)) },
    { simde_mm256_set_epi16(INT16_C( 31333), INT16_C( 14731), INT16_C( -5432), INT16_C( -1220),
                            INT16_C( 28347), INT16_C( -7507), INT16_C( 12867), INT16_C(-21736),
                            INT16_C( 16501), INT16_C(  6047), INT16_C( 22623), INT16_C( 18772),
                            INT16_C( 11372), INT16_C(-18537), INT16_C( 14973), INT16_C(  5624)),
      simde_mm256_set_epi16(INT16_C(-13620), INT16_C( 27665), INT16_C(-15539), INT16_C(-17279),
                            INT16_C( 14311), INT16_C(-11119), INT16_C(  3293), INT16_C( 29760),
                            INT16_C( -8065), INT16_C(-23112), INT16_C(-23539), INT16_C( 25078),
                            INT16_C(-30504), INT16_C( -9265), INT16_C( 31729), INT16_C( 25829)),
      simde_mm256_set_epi16(INT16_C(-31333), INT16_C( 14731), INT16_C(  5432), INT16_C(  1220),
                            INT16_C( 28347), INT16_C(  7507), INT16_C( 12867), INT16_C(-21736),
                            INT16_C(-16501), INT16_C( -6047), INT16_C(-22623), INT16_C( 18772),
                            INT16_C(-11372), INT16_C( 18537), INT16_C( 14973), INT16_C(  5624)) },
    { simde_mm256_set_epi16(INT16_C( 20632), INT16_C(  8608), INT16_C( 27079), INT16_C(-31399),
                            INT16_C( 16081), INT16_C(-23329), INT16_C( 28972), INT16_C( 32213),
                            INT16_C(  7381), INT16_C( 31373), INT16_C( 11592), INT16_C( 16982),
                            INT16_C( -2799), INT16_C( 16604), INT16_C(-21766), INT16_C(-16353)),
      simde_mm256_set_epi16(INT16_C( 16835), INT16_C( 21467), INT16_C(-25504), INT16_C( 23272),
                            INT16_C( 22254), INT16_C(-15726), INT16_C(-27142), INT16_C(  2554),
                            INT16_C( 20665), INT16_C(-20609), INT16_C( -5134), INT16_C(  6994),
                            INT16_C(-32186), INT16_C(  4789), INT16_C( 32409), INT16_C(-27788)),
      simde_mm256_set_epi16(INT16_C( 20632), INT16_C(  8608), INT16_C(-27079), INT16_C(-31399),
                            INT16_C( 16081), INT16_C( 23329), INT16_C(-28972), INT16_C( 32213),
                            INT16_C(  7381), INT16_C(-31373), INT16_C(-11592), INT16_C( 16982),
                            INT16_C(  2799), INT16_C( 16604), INT16_C(-21766), INT16_C( 16353)) },
    { simde_mm256_set_epi16(INT16_C( 22585), INT16_C( -9206), INT16_C( 23997), INT16_C(  8964),
                            INT16_C( 21167), INT16_C(  4682), INT16_C(-30256), INT16_C( 13019),
                            INT16_C( 21859), INT16_C(-23935), INT16_C( 32292), INT16_C( 22433),
                            INT16_C( 28502), INT16_C(-12291), INT16_C( -9965), INT16_C(  -605)),
      simde_mm256_set_epi16(INT16_C(-12753), INT16_C( 28249), INT16_C(-24894), INT16_C( 25803),
                            INT16_C(-23298), INT16_C( 23540), INT16_C( -7003), INT16_C( -6880),
                            INT16_C(-27946), INT16_C( 18842), INT16_C(-31184), INT16_C(  -818),
                            INT16_C(  2524), INT16_C( 26242), INT16_C(  8800), INT16_C(-14585)),
      simde_mm256_set_epi16(INT16_C(-22585), INT16_C( -9206), INT16_C(-23997), INT16_C(  8964),
                            INT16_C(-21167), INT16_C(  4682), INT16_C( 30256), INT16_C(-13019),
                            INT16_C(-21859), INT16_C(-23935), INT16_C(-32292), INT16_C(-22433),
                            INT16_C( 28502), INT16_C(-12291), INT16_C( -9965), INT16_C(   605)) },
    { simde_mm256_set_epi16(INT16_C( 19892), INT16_C( 30145), INT16_C(-24855), INT16_C( 25372),
                            INT16_C( 29341), INT16_C( 30192), INT16_C( -8881), INT16_C( -8744),
                            INT16_C( 29793), INT16_C(  4128), INT16_C(-30978), INT16_C( 29899),
                            INT16_C(-26164), INT16_C( 13029), INT16_C(-28557), INT16_C(-10583)),
      simde_mm256_set_epi16(INT16_C( 12645), INT16_C(-23051), INT16_C(  9838), INT16_C( 16246),
                            INT16_C( 16816), INT16_C( 30389), INT16_C( 26548), INT16_C(  2750),
                            INT16_C( 27528), INT16_C(  3034), INT16_C(-14888), INT16_C( -2181),
                            INT16_C(  -556), INT16_C(-16358), INT16_C( -7666), INT16_C(-25578)),
      simde_mm256_set_epi16(INT16_C( 19892), INT16_C(-30145), INT16_C(-24855), INT16_C( 25372),
                            INT16_C( 29341), INT16_C( 30192), INT16_C( -8881), INT16_C( -8744),
                            INT16_C( 29793), INT16_C(  4128), INT16_C( 30978), INT16_C(-29899),
                            INT16_C( 26164), INT16_C(-13029), INT16_C( 28557), INT16_C( 10583)) },
    { simde_mm256_set_epi16(INT16_C(-19657), INT16_C( 17669), INT16_C(-14306), INT16_C( 31572),
                            INT16_C( 22676), INT16_C( 12265), INT16_C(-14682), INT16_C(  4744),
                            INT16_C( -3797), INT16_C( 26113), INT16_C( -9683), INT16_C( 30844),
                            INT16_C( 10026), INT16_C(-27094), INT16_C( 12923), INT16_C( 21429)),
      simde_mm256_set_epi16(INT16_C(-25343), INT16_C( 26600), INT16_C( 19789), INT16_C(-14016),
                            INT16_C(-12932), INT16_C(-18859), INT16_C(  6902), INT16_C( -4631),
                            INT16_C( -2798), INT16_C(-29634), INT16_C(-10516), INT16_C( 26757),
                            INT16_C(-28927), INT16_C(-17242), INT16_C(   480), INT16_C(   404)),
      simde_mm256_set_epi16(INT16_C( 19657), INT16_C( 17669), INT16_C(-14306), INT16_C(-31572),
                            INT16_C(-22676), INT16_C(-12265), INT16_C(-14682), INT16_C( -4744),
                            INT16_C(  3797), INT16_C(-26113), INT16_C(  9683), INT16_C( 30844),
                            INT16_C(-10026), INT16_C( 27094), INT16_C( 12923), INT16_C( 21429)) },
    { simde_mm256_set_epi16(INT16_C( 31818), INT16_C( -8162), INT16_C(-27852), INT16_C(  8030),
                            INT16_C(-19771), INT16_C( -1956), INT16_C(  2469), INT16_C(-15944),
                            INT16_C(-21759), INT16_C( 15444), INT16_C( 11228), INT16_C(-28666),
                            INT16_C( -4731), INT16_C( 20683), INT16_C(-13730), INT16_C(   880)),
      simde_mm256_set_epi16(INT16_C( -7720), INT16_C(-18895), INT16_C(-27242), INT16_C( 26121),
                            INT16_C( 30042), INT16_C( 29379), INT16_C(  8440), INT16_C(  -825),
                            INT16_C( 31619), INT16_C(  2432), INT16_C(-25855), INT16_C(  8385),
                            INT16_C( 25385), INT16_C(  9878), INT16_C(-16582), INT16_C( -5768)),
      simde_mm256_set_epi16(INT16_C(-31818), INT16_C(  8162), INT16_C( 27852), INT16_C(  8030),
                            INT16_C(-19771), INT16_C( -1956), INT16_C(  2469), INT16_C( 15944),
                            INT16_C(-21759), INT16_C( 15444), INT16_C(-11228), INT16_C(-28666),
                            INT16_C( -4731), INT16_C( 20683), INT16_C( 13730), INT16_C(  -880)) },
    { simde_mm256_set_epi16(INT16_C(-12394), INT16_C(  5925), INT16_C(-22167), INT16_C( 17105),
                            INT16_C(-25421), INT16_C(-31305), INT16_C(  -816), INT16_C( 25010),
                            INT16_C(  5501), INT16_C(-32738), INT16_C(-22226), INT16_C( -5189),
                            INT16_C( -1009), INT16_C( -4191), INT16_C(  1824), INT16_C( 23969)),
      simde_mm256_set_epi16(INT16_C(  6973), INT16_C(  6673), INT16_C( 28195), INT16_C( 18113),
                            INT16_C( 13233), INT16_C(-11958), INT16_C( 11271), INT16_C( 32197),
                            INT16_C( -6474), INT16_C(-13846), INT16_C(  4824), INT16_C(  8574),
                            INT16_C(-32258), INT16_C(-31921), INT16_C( 28942), INT16_C( 28949)),
      simde_mm256_set_epi16(INT16_C(-12394), INT16_C(  5925), INT16_C(-22167), INT16_C( 17105),
                            INT16_C(-25421), INT16_C( 31305), INT16_C(  -816), INT16_C( 25010),
                            INT16_C( -5501), INT16_C( 32738), INT16_C(-22226), INT16_C( -5189),
                            INT16_C(  1009), INT16_C(  4191), INT16_C(  1824), INT16_C( 23969)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_sign_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_sign_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(-2066740673), INT32_C(-1402473921), INT32_C( 1388113781), INT32_C(-1332206277),
                            INT32_C( 2030824666), INT32_C(  616671193), INT32_C( 1932233736), INT32_C(-1125047369)),
      simde_mm256_set_epi32(INT32_C( 1087531093), INT32_C(-1100485211), INT32_C(-1940507046), INT32_C(-2139586601),
                            INT32_C(  942523406), INT32_C(  -53038765), INT32_C( 1212259695), INT32_C(-1402477633)),
      simde_mm256_set_epi32(INT32_C(-2066740673), INT32_C( 1402473921), INT32_C(-1388113781), INT32_C( 1332206277),
                            INT32_C( 2030824666), INT32_C( -616671193), INT32_C( 1932233736), INT32_C( 1125047369)) },
    { simde_mm256_set_epi32(INT32_C( 2053454219), INT32_C( -355927236), INT32_C( 1857807021), INT32_C(  843295512),
                            INT32_C( 1081415583), INT32_C( 1482639700), INT32_C(  745322391), INT32_C(  981276152)),
      simde_mm256_set_epi32(INT32_C( -892572655), INT32_C(-1018315647), INT32_C(  937940113), INT32_C(  215839808),
                            INT32_C( -528505416), INT32_C(-1542626826), INT32_C(-1999053873), INT32_C( 2079417573)),
      simde_mm256_set_epi32(INT32_C(-2053454219), INT32_C(  355927236), INT32_C( 1857807021), INT32_C(  843295512),
                            INT32_C(-1081415583), INT32_C(-1482639700), INT32_C( -745322391), INT32_C(  981276152)) },
    { simde_mm256_set_epi32(INT32_C( 1352147360), INT32_C( 1774683481), INT32_C( 1053926623), INT32_C( 1898741205),
                            INT32_C(  483752589), INT32_C(  759710294), INT32_C( -183418660), INT32_C(-1426407393)),
      simde_mm256_set_epi32(INT32_C( 1103320027), INT32_C(-1671406872), INT32_C( 1458487954), INT32_C(-1778775558),
                            INT32_C( 1354346367), INT32_C( -336454830), INT32_C(-2109336907), INT32_C( 2123993972)),
      simde_mm256_set_epi32(INT32_C( 1352147360), INT32_C(-1774683481), INT32_C( 1053926623), INT32_C(-1898741205),
                            INT32_C(  483752589), INT32_C( -759710294), INT32_C(  183418660), INT32_C(-1426407393)) },
    { simde_mm256_set_epi32(INT32_C( 1480186890), INT32_C( 1572676356), INT32_C( 1387205194), INT32_C(-1982844197),
                            INT32_C( 1432593025), INT32_C( 2116310945), INT32_C( 1867960317), INT32_C( -653001309)),
      simde_mm256_set_epi32(INT32_C( -835752359), INT32_C(-1631427381), INT32_C(-1526834188), INT32_C( -458889952),
                            INT32_C(-1831450214), INT32_C(-2043609906), INT32_C(  165439106), INT32_C(  576767751)),
      simde_mm256_set_epi32(INT32_C(-1480186890), INT32_C(-1572676356), INT32_C(-1387205194), INT32_C( 1982844197),
                            INT32_C(-1432593025), INT32_C(-2116310945), INT32_C( 1867960317), INT32_C( -653001309)) },
    { simde_mm256_set_epi32(INT32_C( 1303672257), INT32_C(-1628871908), INT32_C( 1922921968), INT32_C( -581968424),
                            INT32_C( 1952518176), INT32_C(-2030144309), INT32_C(-1714670875), INT32_C(-1871456599)),
      simde_mm256_set_epi32(INT32_C(  828745205), INT32_C(  644759414), INT32_C( 1102083765), INT32_C( 1739852478),
                            INT32_C( 1804078042), INT32_C( -975636613), INT32_C(  -36388838), INT32_C( -502359018)),
      simde_mm256_set_epi32(INT32_C( 1303672257), INT32_C(-1628871908), INT32_C( 1922921968), INT32_C( -581968424),
                            INT32_C( 1952518176), INT32_C( 2030144309), INT32_C( 1714670875), INT32_C( 1871456599)) },
    { simde_mm256_set_epi32(INT32_C(-1288223483), INT32_C( -937526444), INT32_C( 1486106601), INT32_C( -962194808),
                            INT32_C( -248814079), INT32_C( -634554244), INT32_C(  657102378), INT32_C(  846943157)),
      simde_mm256_set_epi32(INT32_C(-1660852248), INT32_C( 1296943424), INT32_C( -847464875), INT32_C(  452390377),
                            INT32_C( -183333826), INT32_C( -689149819), INT32_C(-1895711578), INT32_C(   31457684)),
      simde_mm256_set_epi32(INT32_C( 1288223483), INT32_C( -937526444), INT32_C(-1486106601), INT32_C( -962194808),
                            INT32_C(  248814079), INT32_C(  634554244), INT32_C( -657102378), INT32_C(  846943157)) },
    { simde_mm256_set_epi32(INT32_C( 2085281822), INT32_C(-1825300642), INT32_C(-1295648676), INT32_C(  161857976),
                            INT32_C(-1425982380), INT32_C(  735875078), INT32_C( -310030133), INT32_C( -899808400)),
      simde_mm256_set_epi32(INT32_C( -505891279), INT32_C(-1785305591), INT32_C( 1968861891), INT32_C(  553188551),
                            INT32_C( 2072185216), INT32_C(-1694424895), INT32_C( 1663641238), INT32_C(-1086658184)),
      simde_mm256_set_epi32(INT32_C(-2085281822), INT32_C( 1825300642), INT32_C(-1295648676), INT32_C(  161857976),
                            INT32_C(-1425982380), INT32_C( -735875078), INT32_C( -310030133), INT32_C(  899808400)) },
    { simde_mm256_set_epi32(INT32_C( -812247259), INT32_C(-1452719407), INT32_C(-1665956425), INT32_C(  -53452366),
                            INT32_C(  360546334), INT32_C(-1456542789), INT32_C(  -66064479), INT32_C(  119561633)),
      simde_mm256_set_epi32(INT32_C(  456989201), INT32_C( 1847805633), INT32_C(  867291466), INT32_C(  738688453),
                            INT32_C( -424228374), INT32_C(  316154238), INT32_C(-2114026673), INT32_C( 1896771861)),
      simde_mm256_set_epi32(INT32_C( -812247259), INT32_C(-1452719407), INT32_C(-1665956425), INT32_C(  -53452366),
                            INT32_C( -360546334), INT32_C(-1456542789), INT32_C(   66064479), INT32_C(  119561633)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_sign_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_sll_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[16];
    const int64_t count[2];
    const int16_t r[16];
  } test_vec[] = {
    { { -INT16_C( 30967), -INT16_C( 15643),  INT16_C(  9802),  INT16_C( 10151),  INT16_C( 21287),  INT16_C( 10197),  INT16_C( 23843), -INT16_C( 11577),
        -INT16_C( 30272),  INT16_C( 28806),  INT16_C(  2696),  INT16_C( 16693),  INT16_C( 19713), -INT16_C(  3783), -INT16_C( 24678), -INT16_C( 23756) },
      {  INT64_C(                   6),  INT64_C(                   8) },
      { -INT16_C( 15808), -INT16_C( 18112), -INT16_C( 28032), -INT16_C(  5696), -INT16_C( 13888), -INT16_C(  2752),  INT16_C( 18624), -INT16_C( 20032),
         INT16_C( 28672),  INT16_C(  8576), -INT16_C( 24064),  INT16_C( 19776),  INT16_C( 16448),  INT16_C( 20032), -INT16_C(  6528), -INT16_C( 13056) } },
    { { -INT16_C(  8994),  INT16_C( 26363),  INT16_C( 12774), -INT16_C(  6233), -INT16_C(  7810),  INT16_C(  6361),  INT16_C(  3456), -INT16_C( 22596),
         INT16_C(  8486),  INT16_C( 25880), -INT16_C( 20434), -INT16_C( 28724),  INT16_C( 22813), -INT16_C(  6381),  INT16_C( 27054), -INT16_C( 29582) },
      {  INT64_C(                   1),  INT64_C(                   1) },
      { -INT16_C( 17988), -INT16_C( 12810),  INT16_C( 25548), -INT16_C( 12466), -INT16_C( 15620),  INT16_C( 12722),  INT16_C(  6912),  INT16_C( 20344),
         INT16_C( 16972), -INT16_C( 13776),  INT16_C( 24668),  INT16_C(  8088), -INT16_C( 19910), -INT16_C( 12762), -INT16_C( 11428),  INT16_C(  6372) } },
    { { -INT16_C( 17901),  INT16_C( 16772),  INT16_C( 20586), -INT16_C( 30768), -INT16_C(  7255),  INT16_C( 22382), -INT16_C(  7860), -INT16_C( 28189),
        -INT16_C( 10929), -INT16_C(  4420), -INT16_C( 12433), -INT16_C(  5877),  INT16_C( 16570), -INT16_C( 19484), -INT16_C( 31182),  INT16_C( 17873) },
      {  INT64_C(                   4),  INT64_C(                  13) },
      { -INT16_C( 24272),  INT16_C(  6208),  INT16_C(  1696),  INT16_C( 32000),  INT16_C( 14992),  INT16_C( 30432),  INT16_C(  5312),  INT16_C(  7728),
         INT16_C( 21744), -INT16_C(  5184), -INT16_C(  2320), -INT16_C( 28496),  INT16_C(  2976),  INT16_C( 15936),  INT16_C( 25376),  INT16_C( 23824) } },
    { { -INT16_C( 10913), -INT16_C( 12610), -INT16_C( 13916),  INT16_C( 24248), -INT16_C( 25591),  INT16_C( 15121), -INT16_C(  7389),  INT16_C( 25472),
         INT16_C(  1848), -INT16_C(  8690),  INT16_C( 16222), -INT16_C( 26579), -INT16_C( 11297),  INT16_C( 24607),  INT16_C( 14429), -INT16_C( 17104) },
      {  INT64_C(                  12),  INT64_C(                   3) },
      { -INT16_C(  4096), -INT16_C(  8192),  INT16_C( 16384),       INT16_MIN, -INT16_C( 28672),  INT16_C(  4096),  INT16_C( 12288),  INT16_C(     0),
              INT16_MIN, -INT16_C(  8192), -INT16_C(  8192), -INT16_C( 12288), -INT16_C(  4096), -INT16_C(  4096), -INT16_C( 12288),  INT16_C(     0) } },
    { {  INT16_C( 29828), -INT16_C(  7654),  INT16_C( 18356), -INT16_C( 27782), -INT16_C( 26085),  INT16_C( 30964),  INT16_C(  9426), -INT16_C(  8395),
        -INT16_C( 16109), -INT16_C( 13680), -INT16_C( 24828), -INT16_C(  7029), -INT16_C( 30528), -INT16_C( 15129),  INT16_C( 19973), -INT16_C( 30463) },
      {  INT64_C(                   7),  INT64_C(                   3) },
      {  INT16_C( 16896),  INT16_C(  3328), -INT16_C(  9728), -INT16_C( 17152),  INT16_C(  3456),  INT16_C( 31232),  INT16_C( 26880), -INT16_C( 25984),
        -INT16_C( 30336),  INT16_C( 18432), -INT16_C( 32256),  INT16_C( 17792),  INT16_C( 24576),  INT16_C( 29568),  INT16_C(   640), -INT16_C( 32640) } },
    { { -INT16_C( 16147), -INT16_C(  3840), -INT16_C( 29856),  INT16_C(  8406), -INT16_C( 17133),  INT16_C(  6373), -INT16_C(  6645), -INT16_C( 12639),
         INT16_C(  3073),  INT16_C( 25668),  INT16_C( 20210),  INT16_C( 29154), -INT16_C(  9908),  INT16_C( 28611), -INT16_C(  3323), -INT16_C(  3420) },
      {  INT64_C(                   4),  INT64_C(                   0) },
      {  INT16_C(  3792),  INT16_C(  4096), -INT16_C( 18944),  INT16_C(  3424), -INT16_C( 11984), -INT16_C( 29104),  INT16_C( 24752), -INT16_C(  5616),
        -INT16_C( 16368),  INT16_C( 17472), -INT16_C(  4320),  INT16_C(  7712), -INT16_C( 27456), -INT16_C(   976),  INT16_C( 12368),  INT16_C( 10816) } },
    { { -INT16_C( 27382), -INT16_C(   923),  INT16_C( 18659),  INT16_C( 12397),  INT16_C( 12321),  INT16_C(  9887),  INT16_C( 17188), -INT16_C( 10216),
        -INT16_C(  1048),  INT16_C(  6380),  INT16_C(  8373),  INT16_C( 11355), -INT16_C( 18630),  INT16_C( 14766), -INT16_C(    75), -INT16_C( 16582) },
      {  INT64_C(                   0),  INT64_C(                   7) },
      { -INT16_C( 27382), -INT16_C(   923),  INT16_C( 18659),  INT16_C( 12397),  INT16_C( 12321),  INT16_C(  9887),  INT16_C( 17188), -INT16_C( 10216),
        -INT16_C(  1048),  INT16_C(  6380),  INT16_C(  8373),  INT16_C( 11355), -INT16_C( 18630),  INT16_C( 14766), -INT16_C(    75), -INT16_C( 16582) } },
    { {  INT16_C( 16706), -INT16_C(  2166), -INT16_C(  6815), -INT16_C( 25821), -INT16_C( 11619),  INT16_C( 21205),  INT16_C(  4049),  INT16_C( 25873),
        -INT16_C( 13137), -INT16_C( 26660), -INT16_C( 31500),  INT16_C( 19872), -INT16_C( 12342),  INT16_C( 21706),  INT16_C(  7958),  INT16_C( 22982) },
      {  INT64_C(                  13),  INT64_C(                  15) },
      {  INT16_C( 16384),  INT16_C( 16384),  INT16_C(  8192),  INT16_C( 24576), -INT16_C( 24576), -INT16_C( 24576),  INT16_C(  8192),  INT16_C(  8192),
        -INT16_C(  8192),       INT16_MIN,       INT16_MIN,  INT16_C(     0),  INT16_C( 16384),  INT16_C( 16384), -INT16_C( 16384), -INT16_C( 16384) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m128i count = simde_x_mm_loadu_epi64(test_vec[i].count);
    simde__m256i r = simde_mm256_sll_epi16(a, count);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_sll_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[8];
    const int64_t count[2];
    const int32_t r[8];
  } test_vec[] = {
    { { -INT32_C(    42405030),  INT32_C(  2070762355), -INT32_C(   707236265),  INT32_C(   839284725),  INT32_C(   514773574),  INT32_C(   579134346),  INT32_C(   416515106),  INT32_C(  1106560486) },
      {  INT64_C(                   2),  INT64_C(                   7) },
      { -INT32_C(   169620120), -INT32_C(   306885172),  INT32_C(  1466022236), -INT32_C(   937828396),  INT32_C(  2059094296), -INT32_C(  1978429912),  INT32_C(  1666060424),  INT32_C(   131274648) } },
    { {  INT32_C(  1331425221), -INT32_C(   177086510),  INT32_C(  1242383459), -INT32_C(  1031077367),  INT32_C(   787335534), -INT32_C(  1924819852),  INT32_C(   194785556), -INT32_C(   381100764) },
      {  INT64_C(                  28),  INT64_C(                   2) },
      {  INT32_C(  1342177280),  INT32_C(   536870912),  INT32_C(   805306368), -INT32_C(  1879048192), -INT32_C(   536870912),  INT32_C(  1073741824),  INT32_C(  1073741824),  INT32_C(  1073741824) } },
    { { -INT32_C(   113203067),  INT32_C(  1350993212), -INT32_C(   614784330), -INT32_C(   859527937), -INT32_C(   882115256), -INT32_C(  1749876824),  INT32_C(  1150280608),  INT32_C(   609634975) },
      {  INT64_C(                  26),  INT64_C(                  24) },
      {  INT32_C(   335544320), -INT32_C(   268435456), -INT32_C(   671088640), -INT32_C(    67108864),  INT32_C(   536870912), -INT32_C(  1610612736),            INT32_MIN,  INT32_C(  2080374784) } },
    { {  INT32_C(   378731886), -INT32_C(  1666366212), -INT32_C(   941605592),  INT32_C(  2045523591), -INT32_C(   408483380),  INT32_C(  1958291117),  INT32_C(  2134533377),  INT32_C(  1162267862) },
      {  INT64_C(                   8),  INT64_C(                   8) },
      { -INT32_C(  1828884992), -INT32_C(  1387987968), -INT32_C(   532862976), -INT32_C(   331970816), -INT32_C(  1492530176), -INT32_C(  1188647680),  INT32_C(   979697920),  INT32_C(  1187829248) } },
    { { -INT32_C(  1356010239),  INT32_C(   354673940),  INT32_C(   563371339), -INT32_C(   228140503), -INT32_C(   810040654),  INT32_C(   286464203), -INT32_C(   740417702),  INT32_C(   421012503) },
      {  INT64_C(                  14),  INT64_C(                  11) },
      {  INT32_C(   994066432), -INT32_C(   112918528),  INT32_C(   391299072), -INT32_C(  1232453632), -INT32_C(   257130496), -INT32_C(   969752576), -INT32_C(  2015985664),  INT32_C(   151371776) } },
    { { -INT32_C(  1976886593), -INT32_C(  1919205838),  INT32_C(  1969256798), -INT32_C(  1366394723), -INT32_C(   422291780), -INT32_C(  1990521279), -INT32_C(  2013596963), -INT32_C(  1360830993) },
      {  INT64_C(                  21),  INT64_C(                  25) },
      {  INT32_C(  1474297856), -INT32_C(   968884224),  INT32_C(   734003200),  INT32_C(   329252864), -INT32_C(   679477248), -INT32_C(   937426944),  INT32_C(  1537212416), -INT32_C(  1109393408) } },
    { {  INT32_C(  1186097925),  INT32_C(  1473252730), -INT32_C(   287388929), -INT32_C(  1617116888),  INT32_C(   474600655), -INT32_C(   205094745),  INT32_C(   652011031), -INT32_C(  1695386475) },
      {  INT64_C(                  22),  INT64_C(                   4) },
      { -INT32_C(  1052770304),  INT32_C(  1585446912), -INT32_C(  1077936128),  INT32_C(  1241513984),  INT32_C(   868220928),  INT32_C(   700448768), -INT32_C(  2051014656),  INT32_C(   624951296) } },
    { { -INT32_C(  1203991536),  INT32_C(   615187212),  INT32_C(  2118813929), -INT32_C(   619168516), -INT32_C(  1841956384),  INT32_C(   608429737), -INT32_C(   959978891),  INT32_C(   803670558) },
      {  INT64_C(                   2),  INT64_C(                   5) },
      { -INT32_C(   520998848), -INT32_C(  1834218448), -INT32_C(   114678876),  INT32_C(  1818293232),  INT32_C(  1222109056), -INT32_C(  1861248348),  INT32_C(   455051732), -INT32_C(  1080285064) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m128i count = simde_x_mm_loadu_epi64(test_vec[i].count);
    simde__m256i r = simde_mm256_sll_epi32(a, count);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_sll_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t a[4];
    const int64_t count[2];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT64_C( 6353669260759125320), -INT64_C( 3799575433940554765), -INT64_C( 3221953462652208759), -INT64_C( 5994917397228281305) },
      {  INT64_C(                  24),  INT64_C(                  54) },
      { -INT64_C( 8679599983080505344),  INT64_C( 5051539236809867264),  INT64_C( 7311530745351110656),  INT64_C( 7187857302152544256) } },
    { { -INT64_C( 2273019595137522117),  INT64_C( 3805099269717809854), -INT64_C(  582719818775460585),  INT64_C( 7138324329339315939) },
      {  INT64_C(                  47),  INT64_C(                  36) },
      { -INT64_C( 5540131229107486720),  INT64_C( 7881017872921657344), -INT64_C( 9039991089527783424),  INT64_C( 3706884705790984192) } },
    { {  INT64_C( 2023522753362425524),  INT64_C( 2373503989335581029), -INT64_C( 4790959826946689851), -INT64_C( 4167002754303171549) },
      {  INT64_C(                  41),  INT64_C(                   6) },
      {  INT64_C( 3237358056358543360), -INT64_C( 6646527998696620032),  INT64_C( 6478861271553802240),  INT64_C( 6829785800721301504) } },
    { {  INT64_C( 3260993203488027417),  INT64_C( 1269439804849385668),  INT64_C( 5917508330730193206), -INT64_C( 7956805592445070552) },
      {  INT64_C(                  37),  INT64_C(                  55) },
      { -INT64_C( 7224227763165593600), -INT64_C(  486924771674554368),  INT64_C( 1355626093914095616),  INT64_C( 8001318738651512832) } },
    { {  INT64_C( 1465206910307710346),  INT64_C( 4671115228599013065), -INT64_C( 8577132701579089518), -INT64_C(  639029091708279909) },
      {  INT64_C(                  45),  INT64_C(                  43) },
      { -INT64_C( 2976527509971009536), -INT64_C(  443287903947194368), -INT64_C( 4813714682961461248), -INT64_C( 3714519714534260736) } },
    { {  INT64_C( 7745356627688155572),  INT64_C( 3877828844525065068),  INT64_C( 4246041348744727368),  INT64_C( 7187132229681079538) },
      {  INT64_C(                  16),  INT64_C(                  47) },
      {  INT64_C(  635275905231749120), -INT64_C( 3401948701828317184), -INT64_C(  568520574133338112), -INT64_C( 3265373720462360576) } },
    { {  INT64_C( 6705542549428466616), -INT64_C( 8947198048595020349), -INT64_C( 1652251508452938248), -INT64_C( 3179399383378874708) },
      {  INT64_C(                  29),  INT64_C(                  19) },
      {  INT64_C( 8835620976327655424), -INT64_C( 1275188604754198528),  INT64_C( 4170867338423304192),  INT64_C( 7229233794509701120) } },
    { { -INT64_C( 5199636620313426708), -INT64_C( 1708347553783123364),  INT64_C( 5888461629188206990),  INT64_C( 5606339840074555950) },
      {  INT64_C(                  28),  INT64_C(                  39) },
      { -INT64_C( 5849700588702400512),  INT64_C( 3998518232565678080),  INT64_C( 8364450152747892736),  INT64_C( 1737327477719564288) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    simde__m128i count = simde_x_mm_loadu_epi64(test_vec[i].count);
    simde__m256i r = simde_mm256_sll_epi64(a, count);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_slli_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i r;
  } test_vec[] = {
    { simde_mm256_set_epi16(INT16_C( -7189), INT16_C( -4038), INT16_C(  8832), INT16_C(-31599),
                            INT16_C(  9071), INT16_C(-26166), INT16_C(  4984), INT16_C(-29916),
                            INT16_C( 26692), INT16_C(  7557), INT16_C(-30970), INT16_C(-31903),
                            INT16_C( 32721), INT16_C( 14732), INT16_C( -9072), INT16_C(-28469)),
      simde_mm256_set_epi16(INT16_C( -7189), INT16_C( -4038), INT16_C(  8832), INT16_C(-31599),
                            INT16_C(  9071), INT16_C(-26166), INT16_C(  4984), INT16_C(-29916),
                            INT16_C( 26692), INT16_C(  7557), INT16_C(-30970), INT16_C(-31903),
                            INT16_C( 32721), INT16_C( 14732), INT16_C( -9072), INT16_C(-28469)) },
    { simde_mm256_set_epi16(INT16_C( 27658), INT16_C(-18253), INT16_C( 19135), INT16_C(-22021),
                            INT16_C(-31750), INT16_C( 26403), INT16_C(-14265), INT16_C( -1890),
                            INT16_C( 16323), INT16_C( -5176), INT16_C(-29730), INT16_C( -7646),
                            INT16_C(-24362), INT16_C( 28979), INT16_C( 13248), INT16_C(  6628)),
      simde_mm256_set_epi16(INT16_C(-10220), INT16_C( 29030), INT16_C(-27266), INT16_C( 21494),
                            INT16_C(  2036), INT16_C(-12730), INT16_C(-28530), INT16_C( -3780),
                            INT16_C( 32646), INT16_C(-10352), INT16_C(  6076), INT16_C(-15292),
                            INT16_C( 16812), INT16_C( -7578), INT16_C( 26496), INT16_C( 13256)) },
    { simde_mm256_set_epi16(INT16_C(-18549), INT16_C( -1433), INT16_C( 15072), INT16_C( 28665),
                            INT16_C(-22672), INT16_C( 30426), INT16_C(-11873), INT16_C(-28559),
                            INT16_C( 26366), INT16_C( 20007), INT16_C( 14676), INT16_C( -4391),
                            INT16_C(-13223), INT16_C(-20349), INT16_C(-14071), INT16_C(  6326)),
      simde_mm256_set_epi16(INT16_C( -8660), INT16_C( -5732), INT16_C( -5248), INT16_C(-16412),
                            INT16_C(-25152), INT16_C( -9368), INT16_C( 18044), INT16_C( 16836),
                            INT16_C(-25608), INT16_C( 14492), INT16_C( -6832), INT16_C(-17564),
                            INT16_C( 12644), INT16_C(-15860), INT16_C(  9252), INT16_C( 25304)) },
    { simde_mm256_set_epi16(INT16_C(-15802), INT16_C(  1115), INT16_C(-15134), INT16_C(-24432),
                            INT16_C( 29893), INT16_C( 28473), INT16_C( 18873), INT16_C( 20173),
                            INT16_C(   191), INT16_C( 32281), INT16_C( 13007), INT16_C( 21879),
                            INT16_C( 31137), INT16_C(-25345), INT16_C(  8960), INT16_C(  9998)),
      simde_mm256_set_epi16(INT16_C(  4656), INT16_C(  8920), INT16_C( 10000), INT16_C(  1152),
                            INT16_C(-23000), INT16_C( 31176), INT16_C( 19912), INT16_C( 30312),
                            INT16_C(  1528), INT16_C( -3896), INT16_C(-27016), INT16_C(-21576),
                            INT16_C(-13048), INT16_C( -6152), INT16_C(  6144), INT16_C( 14448)) },
    { simde_mm256_set_epi16(INT16_C(-11235), INT16_C( 12162), INT16_C(-25050), INT16_C(-16691),
                            INT16_C(-21645), INT16_C(-11525), INT16_C( 24581), INT16_C( 20473),
                            INT16_C( 17987), INT16_C( 13992), INT16_C( 21082), INT16_C(  9896),
                            INT16_C( 20946), INT16_C(  -338), INT16_C(  -890), INT16_C(  3047)),
      simde_mm256_set_epi16(INT16_C( 16848), INT16_C( -2016), INT16_C( -7584), INT16_C( -4912),
                            INT16_C(-18640), INT16_C( 12208), INT16_C(    80), INT16_C(  -112),
                            INT16_C( 25648), INT16_C( 27264), INT16_C(  9632), INT16_C( 27264),
                            INT16_C(  7456), INT16_C( -5408), INT16_C(-14240), INT16_C(-16784)) },
    { simde_mm256_set_epi16(INT16_C(  4232), INT16_C(  9528), INT16_C(  5625), INT16_C( -2056),
                            INT16_C( -5147), INT16_C(-18162), INT16_C(-12026), INT16_C(-23947),
                            INT16_C( 19702), INT16_C( 18533), INT16_C( 17268), INT16_C( -7285),
                            INT16_C(-25540), INT16_C( 15541), INT16_C(-16480), INT16_C(-30543)),
      simde_mm256_set_epi16(INT16_C(  4352), INT16_C(-22784), INT16_C(-16608), INT16_C(  -256),
                            INT16_C( 31904), INT16_C(  8640), INT16_C(  8384), INT16_C( 20128),
                            INT16_C(-24896), INT16_C(  3232), INT16_C( 28288), INT16_C( 29024),
                            INT16_C(-30848), INT16_C(-26976), INT16_C( -3072), INT16_C(  5664)) },
    { simde_mm256_set_epi16(INT16_C( 21401), INT16_C(-26197), INT16_C(-15084), INT16_C(  9940),
                            INT16_C(-15195), INT16_C( 26408), INT16_C( 31953), INT16_C(  9654),
                            INT16_C(-16569), INT16_C( 29048), INT16_C(-29616), INT16_C(  6990),
                            INT16_C(  2847), INT16_C(-28587), INT16_C( 20626), INT16_C(  4687)),
      simde_mm256_set_epi16(INT16_C( -6592), INT16_C( 27328), INT16_C( 17664), INT16_C(-19200),
                            INT16_C( 10560), INT16_C(-13824), INT16_C( 13376), INT16_C( 28032),
                            INT16_C(-11840), INT16_C( 24064), INT16_C(  5120), INT16_C(-11392),
                            INT16_C(-14400), INT16_C(  5440), INT16_C(  9344), INT16_C(-27712)) },
    { simde_mm256_set_epi16(INT16_C( -7756), INT16_C( -3283), INT16_C(-22803), INT16_C(-16163),
                            INT16_C(  5629), INT16_C( 29150), INT16_C( -9667), INT16_C( 12335),
                            INT16_C(-18839), INT16_C(-10112), INT16_C(-18195), INT16_C(-15969),
                            INT16_C(-29495), INT16_C( 29894), INT16_C(-13459), INT16_C( 16744)),
      simde_mm256_set_epi16(INT16_C( -9728), INT16_C(-27008), INT16_C( 30336), INT16_C( 28288),
                            INT16_C(  -384), INT16_C( -4352), INT16_C(  7808), INT16_C(  6016),
                            INT16_C( 13440), INT16_C( 16384), INT16_C( 30336), INT16_C(-12416),
                            INT16_C( 25728), INT16_C( 25344), INT16_C(-18816), INT16_C(-19456)) },
    { simde_mm256_set_epi16(INT16_C(-28557), INT16_C(-10724), INT16_C(  2722), INT16_C(-10967),
                            INT16_C(  5210), INT16_C( 29322), INT16_C( 15248), INT16_C( -8080),
                            INT16_C(-19889), INT16_C( 30125), INT16_C( 14116), INT16_C(-29183),
                            INT16_C( 11578), INT16_C(-15372), INT16_C(-24444), INT16_C( 10332)),
      simde_mm256_set_epi16(INT16_C( 29440), INT16_C(  7168), INT16_C(-24064), INT16_C( 10496),
                            INT16_C( 23040), INT16_C(-30208), INT16_C(-28672), INT16_C( 28672),
                            INT16_C( 20224), INT16_C(-21248), INT16_C(  9216), INT16_C(   256),
                            INT16_C( 14848), INT16_C( -3072), INT16_C(-31744), INT16_C( 23552)) },
    { simde_mm256_set_epi16(INT16_C(-21274), INT16_C(-31982), INT16_C(-20961), INT16_C(  5088),
                            INT16_C(-17462), INT16_C( -9409), INT16_C( 15757), INT16_C(  1966),
                            INT16_C(-22236), INT16_C(-28953), INT16_C( -1919), INT16_C(-19201),
                            INT16_C( 20002), INT16_C( 13277), INT16_C(-29203), INT16_C(-31968)),
      simde_mm256_set_epi16(INT16_C(-13312), INT16_C(  9216), INT16_C( 15872), INT16_C(-16384),
                            INT16_C(-27648), INT16_C( 32256), INT16_C(  6656), INT16_C( 23552),
                            INT16_C( 18432), INT16_C(-12800), INT16_C(   512), INT16_C(  -512),
                            INT16_C( 17408), INT16_C(-17920), INT16_C( -9728), INT16_C( 16384)) },
    { simde_mm256_set_epi16(INT16_C( 18957), INT16_C( -9706), INT16_C( -6614), INT16_C( 10095),
                            INT16_C(  5161), INT16_C(  -605), INT16_C( 13764), INT16_C( 27464),
                            INT16_C( -3974), INT16_C( -5977), INT16_C(-10905), INT16_C(   357),
                            INT16_C(-20180), INT16_C(  8910), INT16_C( -3748), INT16_C(  6077)),
      simde_mm256_set_epi16(INT16_C( 13312), INT16_C( 22528), INT16_C(-22528), INT16_C(-17408),
                            INT16_C(-23552), INT16_C(-29696), INT16_C(  4096), INT16_C(  8192),
                            INT16_C( -6144), INT16_C(-25600), INT16_C(-25600), INT16_C(-27648),
                            INT16_C(-20480), INT16_C( 14336), INT16_C( 28672), INT16_C( -3072)) },
    { simde_mm256_set_epi16(INT16_C(-31972), INT16_C( 18963), INT16_C( 30683), INT16_C(-21818),
                            INT16_C(-14936), INT16_C(-28324), INT16_C( -9468), INT16_C(-30588),
                            INT16_C(-23376), INT16_C( 11957), INT16_C(-26836), INT16_C(  1095),
                            INT16_C(  3892), INT16_C( 16953), INT16_C( 18603), INT16_C(-10451)),
      simde_mm256_set_epi16(INT16_C( -8192), INT16_C(-26624), INT16_C(-10240), INT16_C( 12288),
                            INT16_C( 16384), INT16_C( -8192), INT16_C(  8192), INT16_C(  8192),
                            INT16_C(-32768), INT16_C(-22528), INT16_C( 24576), INT16_C( 14336),
                            INT16_C(-24576), INT16_C(-14336), INT16_C( 22528), INT16_C( 26624)) },
    { simde_mm256_set_epi16(INT16_C( -5389), INT16_C(-19407), INT16_C( 25290), INT16_C( -4640),
                            INT16_C( 29672), INT16_C( 23128), INT16_C(-13360), INT16_C(-25779),
                            INT16_C(-15993), INT16_C( -8711), INT16_C(-27319), INT16_C(  5516),
                            INT16_C(-14272), INT16_C( 31123), INT16_C(-10448), INT16_C( 22014)),
      simde_mm256_set_epi16(INT16_C( 12288), INT16_C(  4096), INT16_C(-24576), INT16_C(     0),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(     0), INT16_C(-12288),
                            INT16_C( 28672), INT16_C(-28672), INT16_C(-28672), INT16_C(-16384),
                            INT16_C(     0), INT16_C( 12288), INT16_C(     0), INT16_C( -8192)) },
    { simde_mm256_set_epi16(INT16_C( -8792), INT16_C( -1044), INT16_C(-11356), INT16_C( -9495),
                            INT16_C(  4809), INT16_C(-13629), INT16_C( 15424), INT16_C(-18049),
                            INT16_C(  4498), INT16_C(-15799), INT16_C(-26777), INT16_C(  7570),
                            INT16_C( 20105), INT16_C( -5032), INT16_C( -5734), INT16_C(-24145)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(-32768), INT16_C(-32768), INT16_C(  8192),
                            INT16_C(  8192), INT16_C( 24576), INT16_C(     0), INT16_C( -8192),
                            INT16_C( 16384), INT16_C(  8192), INT16_C( -8192), INT16_C( 16384),
                            INT16_C(  8192), INT16_C(     0), INT16_C( 16384), INT16_C( -8192)) },
    { simde_mm256_set_epi16(INT16_C(-28948), INT16_C(-14792), INT16_C( 23966), INT16_C( 21707),
                            INT16_C(  7390), INT16_C(  8499), INT16_C( 25220), INT16_C( 26210),
                            INT16_C(  3798), INT16_C( 20760), INT16_C(-20165), INT16_C(-15694),
                            INT16_C(   538), INT16_C(    97), INT16_C( -3558), INT16_C(-31643)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(-32768), INT16_C(-16384),
                            INT16_C(-32768), INT16_C(-16384), INT16_C(     0), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(     0), INT16_C(-16384), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 16384), INT16_C(-32768), INT16_C( 16384)) },
    { simde_mm256_set_epi16(INT16_C(-17181), INT16_C(  9599), INT16_C(-10139), INT16_C(-21574),
                            INT16_C( -1148), INT16_C(-14339), INT16_C( -9621), INT16_C(  4987),
                            INT16_C(-25046), INT16_C( 11688), INT16_C( -9678), INT16_C( -6619),
                            INT16_C( 24962), INT16_C(-11214), INT16_C( 26930), INT16_C(-12095)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C(     0),
                            INT16_C(     0), INT16_C(-32768), INT16_C(-32768), INT16_C(-32768),
                            INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(-32768),
                            INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(-32768)) }
  };

  simde__m256i r;

  r = simde_mm256_slli_epi16(test_vec[0].a, 0);
  simde_assert_m256i_i16(r, ==, test_vec[0].r);

  r = simde_mm256_slli_epi16(test_vec[1].a, 1);
  simde_assert_m256i_i16(r, ==, test_vec[1].r);

  r = simde_mm256_slli_epi16(test_vec[2].a, 2);
  simde_assert_m256i_i16(r, ==, test_vec[2].r);

  r = simde_mm256_slli_epi16(test_vec[3].a, 3);
  simde_assert_m256i_i16(r, ==, test_vec[3].r);

  r = simde_mm256_slli_epi16(test_vec[4].a, 4);
  simde_assert_m256i_i16(r, ==, test_vec[4].r);

  r = simde_mm256_slli_epi16(test_vec[5].a, 5);
  simde_assert_m256i_i16(r, ==, test_vec[5].r);

  r = simde_mm256_slli_epi16(test_vec[6].a, 6);
  simde_assert_m256i_i16(r, ==, test_vec[6].r);

  r = simde_mm256_slli_epi16(test_vec[7].a, 7);
  simde_assert_m256i_i16(r, ==, test_vec[7].r);

  r = simde_mm256_slli_epi16(test_vec[8].a, 8);
  simde_assert_m256i_i16(r, ==, test_vec[8].r);

  r = simde_mm256_slli_epi16(test_vec[9].a, 9);
  simde_assert_m256i_i16(r, ==, test_vec[9].r);

  r = simde_mm256_slli_epi16(test_vec[10].a, 10);
  simde_assert_m256i_i16(r, ==, test_vec[10].r);

  r = simde_mm256_slli_epi16(test_vec[11].a, 11);
  simde_assert_m256i_i16(r, ==, test_vec[11].r);

  r = simde_mm256_slli_epi16(test_vec[12].a, 12);
  simde_assert_m256i_i16(r, ==, test_vec[12].r);

  r = simde_mm256_slli_epi16(test_vec[13].a, 13);
  simde_assert_m256i_i16(r, ==, test_vec[13].r);

  r = simde_mm256_slli_epi16(test_vec[14].a, 14);
  simde_assert_m256i_i16(r, ==, test_vec[14].r);

  r = simde_mm256_slli_epi16(test_vec[15].a, 15);
  simde_assert_m256i_i16(r, ==, test_vec[15].r);

  return 0;
}

static int
test_simde_mm256_slli_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C( 1857969468), INT32_C( 1569141389), INT32_C(-1894985594), INT32_C( 1398609693),
                            INT32_C( 1177229575), INT32_C( 1655079421), INT32_C(-1753400065), INT32_C( -607538910)),
      simde_mm256_set_epi32(INT32_C(-1434991502), INT32_C(-1643531728), INT32_C( 1435031148), INT32_C( -241637048),
                            INT32_C(  325610914), INT32_C(  482634268), INT32_C(-1954026803), INT32_C( -489641480)),
      simde_mm256_set_epi32(INT32_C( -674519168), INT32_C(-1327083104), INT32_C( -509996864), INT32_C( 1805837216),
                            INT32_C( -983359264), INT32_C( 1422933920), INT32_C( -274227232), INT32_C( 2033591360)) },
    { simde_mm256_set_epi32(INT32_C(-1508060177), INT32_C( 2047506344), INT32_C( 1926956558), INT32_C( -567187171),
                            INT32_C( -718834191), INT32_C(  316061425), INT32_C(-1354037966), INT32_C(-2043187872)),
      simde_mm256_set_epi32(INT32_C( -787812104), INT32_C(-1305007947), INT32_C( 1142922615), INT32_C( 1342194494),
                            INT32_C(-1552966924), INT32_C(  272965007), INT32_C(-2116417663), INT32_C( 1446174254)),
      simde_mm256_set_epi32(INT32_C(-1013285408), INT32_C( 1095693568), INT32_C( 1533067712), INT32_C( -970120288),
                            INT32_C(-1527857632), INT32_C( 1524031008), INT32_C( -379541952), INT32_C( -957502464)) },
    { simde_mm256_set_epi32(INT32_C( 1315357752), INT32_C(-1727529785), INT32_C(-1290894210), INT32_C(-1585668235),
                            INT32_C( 1227224502), INT32_C( -793752248), INT32_C( -629408802), INT32_C( 1394059551)),
      simde_mm256_set_epi32(INT32_C(  743833146), INT32_C( -347973949), INT32_C(  521095178), INT32_C( -742841452),
                            INT32_C( -734810084), INT32_C(-1542766714), INT32_C(  575957526), INT32_C(-1816399246)),
      simde_mm256_set_epi32(INT32_C( -858224896), INT32_C(  553621728), INT32_C( 1641058240), INT32_C(  798224032),
                            INT32_C(  616478400), INT32_C(  369731840), INT32_C( 1333754816), INT32_C( 1660232672)) },
    { simde_mm256_set_epi32(INT32_C(  150628023), INT32_C(-2069974717), INT32_C( 1177413093), INT32_C(-2053792688),
                            INT32_C(-1561038597), INT32_C( -690144185), INT32_C( -416879322), INT32_C( 1986359122)),
      simde_mm256_set_epi32(INT32_C(  306184069), INT32_C( 2091432312), INT32_C( 1455569082), INT32_C( 1460019701),
                            INT32_C(-1968965681), INT32_C(-1344732712), INT32_C( 1352620988), INT32_C( 1100122997)),
      simde_mm256_set_epi32(INT32_C(  525129440), INT32_C(-1814681504), INT32_C( -977486688), INT32_C(-1296856576),
                            INT32_C( 1586372448), INT32_C( -609777440), INT32_C( -455236416), INT32_C( -861017536)) },
    { simde_mm256_set_epi32(INT32_C(-1165370523), INT32_C(  190390907), INT32_C( -318174800), INT32_C(-1752249068),
                            INT32_C( 1366077326), INT32_C( 1019085647), INT32_C( 1171551091), INT32_C(-1805589438)),
      simde_mm256_set_epi32(INT32_C(-1007170348), INT32_C( 2038045091), INT32_C(  483759032), INT32_C(-1769389731),
                            INT32_C( 1901958985), INT32_C(-1845366512), INT32_C(  234958311), INT32_C( -992749206)),
      simde_mm256_set_epi32(INT32_C( 1362848928), INT32_C( 1797541728), INT32_C(-1591659008), INT32_C( -237395328),
                            INT32_C(  764801472), INT32_C(-1748997664), INT32_C(-1165070752), INT32_C(-1944287168)) },
    { simde_mm256_set_epi32(INT32_C( -719890300), INT32_C(-1894949227), INT32_C( 1354912660), INT32_C(  262146704),
                            INT32_C(-2066437384), INT32_C( -277466785), INT32_C( 1307284092), INT32_C( -145490102)),
      simde_mm256_set_epi32(INT32_C( 1880856976), INT32_C( -113670921), INT32_C(-1146592891), INT32_C( -833902193),
                            INT32_C( -226567429), INT32_C(  -58637801), INT32_C( -547686157), INT32_C(  870297549)),
      simde_mm256_set_epi32(INT32_C(-1561653120), INT32_C( -508833120), INT32_C(  407532160), INT32_C( -201240064),
                            INT32_C(-1701486848), INT32_C( -289002528), INT32_C(-1116582016), INT32_C( -360715968)) },
    { simde_mm256_set_epi32(INT32_C( 1194726573), INT32_C( -714629781), INT32_C(  721069762), INT32_C(-1027424294),
                            INT32_C(-1451303753), INT32_C( -307072244), INT32_C(  760761319), INT32_C(-1826453170)),
      simde_mm256_set_epi32(INT32_C(-2121454006), INT32_C(  354423919), INT32_C( -610410351), INT32_C(-1822148121),
                            INT32_C( 2107148479), INT32_C(-1935868808), INT32_C( -573099828), INT32_C(  986406309)),
      simde_mm256_set_epi32(INT32_C( -423455328), INT32_C(-1393316512), INT32_C( 1599395904), INT32_C( 1482160960),
                            INT32_C(  802920160), INT32_C(-1236377216), INT32_C(-1425441568), INT32_C( 1683040704)) },
    { simde_mm256_set_epi32(INT32_C( -461263801), INT32_C( 1863945688), INT32_C(  508376318), INT32_C(-1101321410),
                            INT32_C( 1682501002), INT32_C(-1564690744), INT32_C(  469304116), INT32_C(  162493937)),
      simde_mm256_set_epi32(INT32_C( 1027297411), INT32_C(  436621801), INT32_C( 2048624604), INT32_C(  613668083),
                            INT32_C( 2064314140), INT32_C( -717287635), INT32_C( 1286678409), INT32_C(-1591587787)),
      simde_mm256_set_epi32(INT32_C(-1875539744), INT32_C( -483280128), INT32_C( -911827008), INT32_C( -882546752),
                            INT32_C(-1994542784), INT32_C( 1469503744), INT32_C( 2132829824), INT32_C(  904838688)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm256_slli_epi32(test_vec[i].a, 5);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm256_slli_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m256i b;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi64x(INT64_C(-1180812000753094585), INT64_C(-5675573813228216402),
                             INT64_C(-2208027268370360000), INT64_C( 6465428915389083026)),
      simde_mm256_set_epi64x(INT64_C( 8452611756027991400), INT64_C(-5020149848375082542),
                             INT64_C( 7925308354439905072), INT64_C(  953522845551677435)),
      simde_mm256_set_epi64x(INT64_C( -892495876679923488), INT64_C( 2849078713792591296),
                             INT64_C( 3130103706986686464), INT64_C( 3979540481645589056)) },
    { simde_mm256_set_epi64x(INT64_C(-4047432104007102232), INT64_C(-7778060476316871326),
                             INT64_C( 2270807563927334286), INT64_C(-5320870512275475361)),
      simde_mm256_set_epi64x(INT64_C( 5866182958150697676), INT64_C(-5938517638175025937),
                             INT64_C( 6284264256050873550), INT64_C( 1402097755121835442)),
      simde_mm256_set_epi64x(INT64_C( -390618812260410112), INT64_C(-9090262283915711424),
                             INT64_C(-1121134249163509312), INT64_C(-4247159729429247008)) },
    { simde_mm256_set_epi64x(INT64_C(-5299935544215164629), INT64_C(-5358218550815045801),
                             INT64_C(-2743808865442122704), INT64_C(-5577727586229156383)),
      simde_mm256_set_epi64x(INT64_C(-1628942167927235670), INT64_C( 2616318076895434913),
                             INT64_C(-3093772338434873461), INT64_C( 5967250390824240978)),
      simde_mm256_set_epi64x(INT64_C(-3577240751499303584), INT64_C(-5442296962695501088),
                             INT64_C( 4431836674399831552), INT64_C( 5980157977762511904)) },
    { simde_mm256_set_epi64x(INT64_C( 5567340862163337259), INT64_C(-5967855323816927748),
                             INT64_C(-3043785185238641430), INT64_C( 8117316208405658103)),
      simde_mm256_set_epi64x(INT64_C( 8644099717862406948), INT64_C(-2019559086403734381),
                             INT64_C(-4658546452517300382), INT64_C(-8872814693937714953)),
      simde_mm256_set_epi64x(INT64_C(-6312533147868723872), INT64_C(-6503929625046171776),
                             INT64_C(-5167405559088767680), INT64_C( 1499701637047336672)) },
    { simde_mm256_set_epi64x(INT64_C(-8259641032964555923), INT64_C( 6678157623055419482),
                             INT64_C( -532130775756370858), INT64_C(-6271412084163076656)),
      simde_mm256_set_epi64x(INT64_C( 6145157523311970224), INT64_C(-5190146230173537442),
                             INT64_C(-7773875053091172024), INT64_C(-4385818839583220370)),
      simde_mm256_set_epi64x(INT64_C(-6054096022932066912), INT64_C(-7659884946741195968),
                             INT64_C( 1418559249505684160), INT64_C( 2228998117586614784)) },
    { simde_mm256_set_epi64x(INT64_C(-3893461849415744807), INT64_C(-9064502035867553709),
                             INT64_C( 3098807305629640641), INT64_C( 6650302822439097584)),
      simde_mm256_set_epi64x(INT64_C( 2307018380829868937), INT64_C( 8592643653599303501),
                             INT64_C( 8944692494744620746), INT64_C(-7922755342769649115)),
      simde_mm256_set_epi64x(INT64_C( 4536429334663027488), INT64_C( 5083840031591107168),
                             INT64_C( 6928113411600742432), INT64_C(-8551238566463496704)) },
    { simde_mm256_set_epi64x(INT64_C(-1984148929379503272), INT64_C(-5205661836978001324),
                             INT64_C( 6359749525414039769), INT64_C(-1541764205226378294)),
      simde_mm256_set_epi64x(INT64_C(-1490660722702882908), INT64_C( -633122769786086240),
                             INT64_C(-8686691198507413769), INT64_C(-9123917909533884922)),
      simde_mm256_set_epi64x(INT64_C(-8152533519015449856), INT64_C( -560482119910077824),
                             INT64_C(  597800002444204832), INT64_C( 6003777653884549440)) },
    { simde_mm256_set_epi64x(INT64_C(-5232449947985336963), INT64_C( -518473474085028208),
                             INT64_C( 1841137168562663956), INT64_C( 5952445751598523173)),
      simde_mm256_set_epi64x(INT64_C(-2687140035203018503), INT64_C( -119809082612743153),
                             INT64_C( 6113467954286988131), INT64_C(-1537862337526048352)),
      simde_mm256_set_epi64x(INT64_C(-1417701672144818272), INT64_C( 1855592902988648960),
                             INT64_C( 3576157172876591744), INT64_C( 6010823314057225376)) }
  