/* Copyright (c) 2018, 2019 Evan Nemerson <evan@nemerson.com>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#define SIMDE_TESTS_CURRENT_ISAX avx512f
#include <simde/x86/avx512f.h>
#include <test/x86/test-avx512.h>

#if defined(SIMDE_X86_AVX512F_NATIVE) || defined(SIMDE_NO_NATIVE) || defined(SIMDE_ALWAYS_BUILD_NATIVE_TESTS)

static MunitResult
test_simde_mm512_abs_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-1095158286), INT32_C( -133595553), INT32_C( -941949577), INT32_C(-1117722052),
                            INT32_C(-1053667317), INT32_C( -662420643), INT32_C( 2095193825), INT32_C( -799061081),
                            INT32_C(  347912513), INT32_C( -439299809), INT32_C( 2053030698), INT32_C( -277514113),
                            INT32_C( 1476262970), INT32_C( 1955038119), INT32_C(  -77085072), INT32_C( 1014069144)),
      simde_mm512_set_epi32(INT32_C( 1095158286), INT32_C(  133595553), INT32_C(  941949577), INT32_C( 1117722052),
                            INT32_C( 1053667317), INT32_C(  662420643), INT32_C( 2095193825), INT32_C(  799061081),
                            INT32_C(  347912513), INT32_C(  439299809), INT32_C( 2053030698), INT32_C(  277514113),
                            INT32_C( 1476262970), INT32_C( 1955038119), INT32_C(   77085072), INT32_C( 1014069144)) },
    { simde_mm512_set_epi32(INT32_C( 1865049709), INT32_C( -408997463), INT32_C( 1771073477), INT32_C( 1463780468),
                            INT32_C(  268019741), INT32_C(-1725054429), INT32_C( 1408597864), INT32_C( 1576985133),
                            INT32_C(  170783936), INT32_C(  836522882), INT32_C( 1364040350), INT32_C(  563663058),
                            INT32_C(-1491438903), INT32_C( -873504608), INT32_C( 1431273511), INT32_C( -164765086)),
      simde_mm512_set_epi32(INT32_C( 1865049709), INT32_C(  408997463), INT32_C( 1771073477), INT32_C( 1463780468),
                            INT32_C(  268019741), INT32_C( 1725054429), INT32_C( 1408597864), INT32_C( 1576985133),
                            INT32_C(  170783936), INT32_C(  836522882), INT32_C( 1364040350), INT32_C(  563663058),
                            INT32_C( 1491438903), INT32_C(  873504608), INT32_C( 1431273511), INT32_C(  164765086)) },
    { simde_mm512_set_epi32(INT32_C( 1505063340), INT32_C(  -79208486), INT32_C( -115790145), INT32_C( 1137793635),
                            INT32_C( -719063760), INT32_C( -465633360), INT32_C( 1417132608), INT32_C( 1715322300),
                            INT32_C( 1194443989), INT32_C( 1598244723), INT32_C( -360509626), INT32_C( -844528776),
                            INT32_C( -291907566), INT32_C( -980752736), INT32_C(  701363552), INT32_C( 1148036152)),
      simde_mm512_set_epi32(INT32_C( 1505063340), INT32_C(   79208486), INT32_C(  115790145), INT32_C( 1137793635),
                            INT32_C(  719063760), INT32_C(  465633360), INT32_C( 1417132608), INT32_C( 1715322300),
                            INT32_C( 1194443989), INT32_C( 1598244723), INT32_C(  360509626), INT32_C(  844528776),
                            INT32_C(  291907566), INT32_C(  980752736), INT32_C(  701363552), INT32_C( 1148036152)) },
    { simde_mm512_set_epi32(INT32_C(-1538804784), INT32_C(  -43683957), INT32_C(  -70380459), INT32_C(  259050545),
                            INT32_C(-1140217223), INT32_C(  -24242506), INT32_C(-1281378925), INT32_C( -426768587),
                            INT32_C(-1825251144), INT32_C( -975195895), INT32_C(  758020113), INT32_C(   -3401471),
                            INT32_C(  154668063), INT32_C( -827616009), INT32_C(  793625070), INT32_C( -735990247)),
      simde_mm512_set_epi32(INT32_C( 1538804784), INT32_C(   43683957), INT32_C(   70380459), INT32_C(  259050545),
                            INT32_C( 1140217223), INT32_C(   24242506), INT32_C( 1281378925), INT32_C(  426768587),
                            INT32_C( 1825251144), INT32_C(  975195895), INT32_C(  758020113), INT32_C(    3401471),
                            INT32_C(  154668063), INT32_C(  827616009), INT32_C(  793625070), INT32_C(  735990247)) },
    { simde_mm512_set_epi32(INT32_C( -919197120), INT32_C( 1902742720), INT32_C(  576001152), INT32_C(  772608991),
                            INT32_C( 1373611304), INT32_C(  156079462), INT32_C(  392030686), INT32_C( 1159450969),
                            INT32_C( 1376625025), INT32_C( -701917672), INT32_C( 1911493359), INT32_C( -115817480),
                            INT32_C( -875216623), INT32_C( 1333681477), INT32_C(-1067533891), INT32_C( 1671330781)),
      simde_mm512_set_epi32(INT32_C(  919197120), INT32_C( 1902742720), INT32_C(  576001152), INT32_C(  772608991),
                            INT32_C( 1373611304), INT32_C(  156079462), INT32_C(  392030686), INT32_C( 1159450969),
                            INT32_C( 1376625025), INT32_C(  701917672), INT32_C( 1911493359), INT32_C(  115817480),
                            INT32_C(  875216623), INT32_C( 1333681477), INT32_C( 1067533891), INT32_C( 1671330781)) },
    { simde_mm512_set_epi32(INT32_C(-1168385947), INT32_C(-1671882855), INT32_C(-1182456995), INT32_C(-1803534861),
                            INT32_C(  443878759), INT32_C(  702169153), INT32_C(-1879742181), INT32_C( 1627978919),
                            INT32_C(  583873330), INT32_C( -857098109), INT32_C(  710347808), INT32_C( 1707849385),
                            INT32_C( 1863512780), INT32_C( -371421167), INT32_C( 1902179408), INT32_C(-1189025654)),
      simde_mm512_set_epi32(INT32_C( 1168385947), INT32_C( 1671882855), INT32_C( 1182456995), INT32_C( 1803534861),
                            INT32_C(  443878759), INT32_C(  702169153), INT32_C( 1879742181), INT32_C( 1627978919),
                            INT32_C(  583873330), INT32_C(  857098109), INT32_C(  710347808), INT32_C( 1707849385),
                            INT32_C( 1863512780), INT32_C(  371421167), INT32_C( 1902179408), INT32_C( 1189025654)) },
    { simde_mm512_set_epi32(INT32_C(    7990856), INT32_C(-1991291137), INT32_C( 1404443548), INT32_C(-1023849862),
                            INT32_C( 2054941409), INT32_C(-1604088325), INT32_C(  721271909), INT32_C(-1622295089),
                            INT32_C( 1869222605), INT32_C(-1583998423), INT32_C( -801626928), INT32_C( -940395766),
                            INT32_C( 1108931720), INT32_C( -471669445), INT32_C( 1204289475), INT32_C( -752679106)),
      simde_mm512_set_epi32(INT32_C(    7990856), INT32_C( 1991291137), INT32_C( 1404443548), INT32_C( 1023849862),
                            INT32_C( 2054941409), INT32_C( 1604088325), INT32_C(  721271909), INT32_C( 1622295089),
                            INT32_C( 1869222605), INT32_C( 1583998423), INT32_C(  801626928), INT32_C(  940395766),
                            INT32_C( 1108931720), INT32_C(  471669445), INT32_C( 1204289475), INT32_C(  752679106)) },
    { simde_mm512_set_epi32(INT32_C( 1399806844), INT32_C( 1131841699), INT32_C( -346937782), INT32_C(  567816154),
                            INT32_C(-1589012616), INT32_C(-2005496894), INT32_C( 1401681986), INT32_C(  423760716),
                            INT32_C(  431684101), INT32_C(  852583616), INT32_C(-1369299290), INT32_C( -663899319),
                            INT32_C( 1580470265), INT32_C(  298083241), INT32_C( -630373638), INT32_C(-1937828661)),
      simde_mm512_set_epi32(INT32_C( 1399806844), INT32_C( 1131841699), INT32_C(  346937782), INT32_C(  567816154),
                            INT32_C( 1589012616), INT32_C( 2005496894), INT32_C( 1401681986), INT32_C(  423760716),
                            INT32_C(  431684101), INT32_C(  852583616), INT32_C( 1369299290), INT32_C(  663899319),
                            INT32_C( 1580470265), INT32_C(  298083241), INT32_C(  630373638), INT32_C( 1937828661)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_abs_epi32(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_abs_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(  114710097), INT32_C( 1837246098), INT32_C(-1399577225), INT32_C(-1388127606),
                            INT32_C( 1116027725), INT32_C( -871797325), INT32_C(-1979326643), INT32_C( 1477004857),
                            INT32_C( 1670723749), INT32_C(-1006052339), INT32_C( 1863789116), INT32_C( -690396684),
                            INT32_C( -629773535), INT32_C(  667046523), INT32_C( 1317445565), INT32_C( -732937024)),
      UINT16_C(28888),
      simde_mm512_set_epi32(INT32_C(-1877548571), INT32_C( -399920351), INT32_C(   15525797), INT32_C(   18165921),
                            INT32_C( 2085930596), INT32_C( 1662282658), INT32_C(-1842752263), INT32_C(  499820912),
                            INT32_C( 1419797765), INT32_C(  -44818966), INT32_C( 1761152620), INT32_C(-1288657930),
                            INT32_C(  894078020), INT32_C(-1369431563), INT32_C( -103362440), INT32_C(-1559726025)),
      simde_mm512_set_epi32(INT32_C(  114710097), INT32_C(  399920351), INT32_C(   15525797), INT32_C(   18165921),
                            INT32_C( 1116027725), INT32_C( -871797325), INT32_C(-1979326643), INT32_C( 1477004857),
                            INT32_C( 1419797765), INT32_C(   44818966), INT32_C( 1863789116), INT32_C( 1288657930),
                            INT32_C(  894078020), INT32_C(  667046523), INT32_C( 1317445565), INT32_C( -732937024)) },
    { simde_mm512_set_epi32(INT32_C( 1505436737), INT32_C(  342253548), INT32_C( 1435179252), INT32_C( 1326766533),
                            INT32_C(  156769011), INT32_C(  343760696), INT32_C(  611303965), INT32_C( 1457341409),
                            INT32_C(  165452421), INT32_C(-1824090116), INT32_C( -184738383), INT32_C( -191086464),
                            INT32_C( -117301127), INT32_C(-1526262537), INT32_C( -208138847), INT32_C(  807348777)),
      UINT16_C(61134),
      simde_mm512_set_epi32(INT32_C( 1427056174), INT32_C( 2097896620), INT32_C( 1813263538), INT32_C( 1909821993),
                            INT32_C( 1439822042), INT32_C(-1049213292), INT32_C( 1557133349), INT32_C( 1168931268),
                            INT32_C( -810546774), INT32_C(-1283013132), INT32_C(  654302587), INT32_C(  314275905),
                            INT32_C(-1091094079), INT32_C( -114174508), INT32_C(  407580338), INT32_C( 1906809805)),
      simde_mm512_set_epi32(INT32_C( 1427056174), INT32_C( 2097896620), INT32_C( 1813263538), INT32_C( 1326766533),
                            INT32_C( 1439822042), INT32_C( 1049213292), INT32_C( 1557133349), INT32_C( 1457341409),
                            INT32_C(  810546774), INT32_C( 1283013132), INT32_C( -184738383), INT32_C( -191086464),
                            INT32_C( 1091094079), INT32_C(  114174508), INT32_C(  407580338), INT32_C(  807348777)) },
    { simde_mm512_set_epi32(INT32_C(  905172649), INT32_C(-1044778809), INT32_C(-1938215986), INT32_C(-1138753169),
                            INT32_C(-1689961651), INT32_C(  890456168), INT32_C( 1382435241), INT32_C( -803845344),
                            INT32_C(  430838507), INT32_C( 1075259040), INT32_C(-1956785379), INT32_C(-1586468297),
                            INT32_C(  622055688), INT32_C(-1127740382), INT32_C(  466514910), INT32_C(-1745879628)),
      UINT16_C(30570),
      simde_mm512_set_epi32(INT32_C( -310045086), INT32_C(  560822999), INT32_C( -680371476), INT32_C( 1838395052),
                            INT32_C(-1152635838), INT32_C( -481448106), INT32_C(  871399876), INT32_C( -939960538),
                            INT32_C( -898000986), INT32_C( -641497176), INT32_C(  657638908), INT32_C(-1796735419),
                            INT32_C(-1032150818), INT32_C(  151713087), INT32_C( 1554707006), INT32_C( -318690470)),
      simde_mm512_set_epi32(INT32_C(  905172649), INT32_C(  560822999), INT32_C(  680371476), INT32_C( 1838395052),
                            INT32_C(-1689961651), INT32_C(  481448106), INT32_C(  871399876), INT32_C(  939960538),
                            INT32_C(  430838507), INT32_C(  641497176), INT32_C(  657638908), INT32_C(-1586468297),
                            INT32_C( 1032150818), INT32_C(-1127740382), INT32_C( 1554707006), INT32_C(-1745879628)) },
    { simde_mm512_set_epi32(INT32_C(-1675700291), INT32_C(  -85412591), INT32_C(-1865493216), INT32_C(-1122257925),
                            INT32_C(  955620837), INT32_C( -725693586), INT32_C( 1056307491), INT32_C( 1924019839),
                            INT32_C(-2012466116), INT32_C(-1808881746), INT32_C( -887453452), INT32_C(  160221724),
                            INT32_C( -886018282), INT32_C( 1222780200), INT32_C( 1877396684), INT32_C(  283360472)),
      UINT16_C(28339),
      simde_mm512_set_epi32(INT32_C(-1238615237), INT32_C(  583893938), INT32_C( -594441984), INT32_C( 1561597956),
                            INT32_C(  174377227), INT32_C(  319460903), INT32_C(-1295208114), INT32_C(  659707887),
                            INT32_C( 1117898731), INT32_C( -209622907), INT32_C(-1431480123), INT32_C(-2058827609),
                            INT32_C(-1519596795), INT32_C(   24332922), INT32_C( -338106630), INT32_C(-1565374776)),
      simde_mm512_set_epi32(INT32_C(-1675700291), INT32_C(  583893938), INT32_C(  594441984), INT32_C(-1122257925),
                            INT32_C(  174377227), INT32_C(  319460903), INT32_C( 1295208114), INT32_C( 1924019839),
                            INT32_C( 1117898731), INT32_C(-1808881746), INT32_C( 1431480123), INT32_C( 2058827609),
                            INT32_C( -886018282), INT32_C( 1222780200), INT32_C(  338106630), INT32_C( 1565374776)) },
    { simde_mm512_set_epi32(INT32_C(  178377352), INT32_C( -324510384), INT32_C(  446946466), INT32_C(-1323398690),
                            INT32_C( -720979875), INT32_C( -512216094), INT32_C( 1145272930), INT32_C( -706074883),
                            INT32_C(-1863795060), INT32_C( -525595897), INT32_C( 1357119557), INT32_C(  837734387),
                            INT32_C( -607392699), INT32_C( -498581669), INT32_C(-2108693629), INT32_C( -476969927)),
      UINT16_C(42507),
      simde_mm512_set_epi32(INT32_C(   -5472621), INT32_C( -263868960), INT32_C(-1867831731), INT32_C(  955254216),
                            INT32_C( 1990179011), INT32_C(-1729740457), INT32_C( 1711933869), INT32_C(-1566075058),
                            INT32_C( -550106516), INT32_C(-1087591249), INT32_C(  919917002), INT32_C(-1410389997),
                            INT32_C( -188117230), INT32_C( 1025569327), INT32_C(-1456210246), INT32_C( -254945819)),
      simde_mm512_set_epi32(INT32_C(    5472621), INT32_C( -324510384), INT32_C( 1867831731), INT32_C(-1323398690),
                            INT32_C( -720979875), INT32_C( 1729740457), INT32_C( 1711933869), INT32_C( -706074883),
                            INT32_C(-1863795060), INT32_C( -525595897), INT32_C( 1357119557), INT32_C(  837734387),
                            INT32_C(  188117230), INT32_C( -498581669), INT32_C( 1456210246), INT32_C(  254945819)) },
    { simde_mm512_set_epi32(INT32_C(-1007934437), INT32_C(  201253136), INT32_C( 2123754123), INT32_C( 1034305262),
                            INT32_C( 2139323878), INT32_C( -545410429), INT32_C(-1549231865), INT32_C( 1779895500),
                            INT32_C( 1932853973), INT32_C( 2135732954), INT32_C( 1232725518), INT32_C(  339564914),
                            INT32_C( -113030707), INT32_C(-1715459937), INT32_C( -492435091), INT32_C(-1720946495)),
      UINT16_C(49758),
      simde_mm512_set_epi32(INT32_C(  348473993), INT32_C(-1624874318), INT32_C(  361690252), INT32_C(  165927413),
                            INT32_C(-1864332117), INT32_C( -524477604), INT32_C(  481484649), INT32_C(-1499715490),
                            INT32_C(-1683117466), INT32_C(-2055457330), INT32_C( -850617531), INT32_C(-2081246973),
                            INT32_C( 1276057415), INT32_C( 1619064589), INT32_C(-1536816688), INT32_C( 2060578085)),
      simde_mm512_set_epi32(INT32_C(  348473993), INT32_C( 1624874318), INT32_C( 2123754123), INT32_C( 1034305262),
                            INT32_C( 2139323878), INT32_C( -545410429), INT32_C(  481484649), INT32_C( 1779895500),
                            INT32_C( 1932853973), INT32_C( 2055457330), INT32_C( 1232725518), INT32_C( 2081246973),
                            INT32_C( 1276057415), INT32_C( 1619064589), INT32_C( 1536816688), INT32_C(-1720946495)) },
    { simde_mm512_set_epi32(INT32_C(  860828042), INT32_C( 1459856596), INT32_C(-1901530659), INT32_C( 1296141157),
                            INT32_C(  778663095), INT32_C(-1872048536), INT32_C(-1115787645), INT32_C(-1142406643),
                            INT32_C( 1518955242), INT32_C( -174688543), INT32_C( 1537062129), INT32_C( -974095643),
                            INT32_C(  125816377), INT32_C(-1032428044), INT32_C( -374455538), INT32_C( -648832583)),
      UINT16_C(41340),
      simde_mm512_set_epi32(INT32_C( 1553986008), INT32_C( -808715903), INT32_C(-2114331727), INT32_C(  878797396),
                            INT32_C( 1547560130), INT32_C( -931453209), INT32_C(  639671594), INT32_C(  734358771),
                            INT32_C(-1802430748), INT32_C(   38083245), INT32_C(  636500349), INT32_C( 2020438947),
                            INT32_C(   89083218), INT32_C( 2041918986), INT32_C(-2068453500), INT32_C( 1772569863)),
      simde_mm512_set_epi32(INT32_C( 1553986008), INT32_C( 1459856596), INT32_C( 2114331727), INT32_C( 1296141157),
                            INT32_C(  778663095), INT32_C(-1872048536), INT32_C(-1115787645), INT32_C(  734358771),
                            INT32_C( 1518955242), INT32_C(   38083245), INT32_C(  636500349), INT32_C( 2020438947),
                            INT32_C(   89083218), INT32_C( 2041918986), INT32_C( -374455538), INT32_C( -648832583)) },
    { simde_mm512_set_epi32(INT32_C(-1208548961), INT32_C( 1705109710), INT32_C( -159097588), INT32_C( -879037423),
                            INT32_C( 2121552533), INT32_C(  595529007), INT32_C( -405863552), INT32_C( 1431630584),
                            INT32_C( -616000216), INT32_C(  444327364), INT32_C(  613413664), INT32_C(-2128463203),
                            INT32_C(  939927077), INT32_C(-1255659348), INT32_C(-1631544337), INT32_C(-1727626838)),
      UINT16_C(49163),
      simde_mm512_set_epi32(INT32_C(  895846723), INT32_C(  449272422), INT32_C( 1127330699), INT32_C(-1084895433),
                            INT32_C( -399265722), INT32_C(  697840482), INT32_C( -598276089), INT32_C(  -50403840),
                            INT32_C( 1970006978), INT32_C( 1602141812), INT32_C(-1773480652), INT32_C(  740913018),
                            INT32_C( 1668822994), INT32_C(  698152405), INT32_C( 1772335922), INT32_C(  847772835)),
      simde_mm512_set_epi32(INT32_C(  895846723), INT32_C(  449272422), INT32_C( -159097588), INT32_C( -879037423),
                            INT32_C( 2121552533), INT32_C(  595529007), INT32_C( -405863552), INT32_C( 1431630584),
                            INT32_C( -616000216), INT32_C(  444327364), INT32_C(  613413664), INT32_C(-2128463203),
                            INT32_C( 1668822994), INT32_C(-1255659348), INT32_C( 1772335922), INT32_C(  847772835)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_abs_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_abs_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(17600),
      simde_mm512_set_epi32(INT32_C(  393115914), INT32_C(   -9604904), INT32_C(  114710097), INT32_C( 1837246098),
                            INT32_C(-1399577225), INT32_C(-1388127606), INT32_C( 1116027725), INT32_C( -871797325),
                            INT32_C(-1979326643), INT32_C( 1477004857), INT32_C( 1670723749), INT32_C(-1006052339),
                            INT32_C( 1863789116), INT32_C( -690396684), INT32_C( -629773535), INT32_C(  667046523)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(    9604904), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1388127606), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1979326643), INT32_C( 1477004857), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(30775),
      simde_mm512_set_epi32(INT32_C( -208138847), INT32_C(  807348777), INT32_C(-1877548571), INT32_C( -399920351),
                            INT32_C(   15525797), INT32_C(   18165921), INT32_C( 2085930596), INT32_C( 1662282658),
                            INT32_C(-1842752263), INT32_C(  499820912), INT32_C( 1419797765), INT32_C(  -44818966),
                            INT32_C( 1761152620), INT32_C(-1288657930), INT32_C(  894078020), INT32_C(-1369431563)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  807348777), INT32_C( 1877548571), INT32_C(  399920351),
                            INT32_C(   15525797), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1419797765), INT32_C(   44818966),
                            INT32_C(          0), INT32_C( 1288657930), INT32_C(  894078020), INT32_C( 1369431563)) },
    { UINT16_C( 5367),
      simde_mm512_set_epi32(INT32_C(  407580338), INT32_C( 1906809805), INT32_C( -849801752), INT32_C(-1965822258),
                            INT32_C( 1505436737), INT32_C(  342253548), INT32_C( 1435179252), INT32_C( 1326766533),
                            INT32_C(  156769011), INT32_C(  343760696), INT32_C(  611303965), INT32_C( 1457341409),
                            INT32_C(  165452421), INT32_C(-1824090116), INT32_C( -184738383), INT32_C( -191086464)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C( 1965822258),
                            INT32_C(          0), INT32_C(  342253548), INT32_C(          0), INT32_C(          0),
                            INT32_C(  156769011), INT32_C(  343760696), INT32_C(  611303965), INT32_C( 1457341409),
                            INT32_C(          0), INT32_C( 1824090116), INT32_C(  184738383), INT32_C(  191086464)) },
    { UINT16_C(54740),
      simde_mm512_set_epi32(INT32_C(  622055688), INT32_C(-1127740382), INT32_C(  466514910), INT32_C(-1745879628),
                            INT32_C( 1427056174), INT32_C( 2097896620), INT32_C( 1813263538), INT32_C( 1909821993),
                            INT32_C( 1439822042), INT32_C(-1049213292), INT32_C( 1557133349), INT32_C( 1168931268),
                            INT32_C( -810546774), INT32_C(-1283013132), INT32_C(  654302587), INT32_C(  314275905)),
      simde_mm512_set_epi32(INT32_C(  622055688), INT32_C( 1127740382), INT32_C(          0), INT32_C( 1745879628),
                            INT32_C(          0), INT32_C( 2097896620), INT32_C(          0), INT32_C( 1909821993),
                            INT32_C( 1439822042), INT32_C( 1049213292), INT32_C(          0), INT32_C( 1168931268),
                            INT32_C(          0), INT32_C( 1283013132), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(27191),
      simde_mm512_set_epi32(INT32_C(-1032150818), INT32_C(  151713087), INT32_C( 1554707006), INT32_C( -318690470),
                            INT32_C(  788893537), INT32_C( -230394006), INT32_C(  905172649), INT32_C(-1044778809),
                            INT32_C(-1938215986), INT32_C(-1138753169), INT32_C(-1689961651), INT32_C(  890456168),
                            INT32_C( 1382435241), INT32_C( -803845344), INT32_C(  430838507), INT32_C( 1075259040)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  151713087), INT32_C( 1554707006), INT32_C(          0),
                            INT32_C(  788893537), INT32_C(          0), INT32_C(  905172649), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1689961651), INT32_C(  890456168),
                            INT32_C(          0), INT32_C(  803845344), INT32_C(  430838507), INT32_C( 1075259040)) },
    { UINT16_C(65093),
      simde_mm512_set_epi32(INT32_C( -887453452), INT32_C(  160221724), INT32_C( -886018282), INT32_C( 1222780200),
                            INT32_C( 1877396684), INT32_C(  283360472), INT32_C( -310045086), INT32_C(  560822999),
                            INT32_C( -680371476), INT32_C( 1838395052), INT32_C(-1152635838), INT32_C( -481448106),
                            INT32_C(  871399876), INT32_C( -939960538), INT32_C( -898000986), INT32_C( -641497176)),
      simde_mm512_set_epi32(INT32_C(  887453452), INT32_C(  160221724), INT32_C(  886018282), INT32_C( 1222780200),
                            INT32_C( 1877396684), INT32_C(  283360472), INT32_C(  310045086), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1838395052), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(  939960538), INT32_C(          0), INT32_C(  641497176)) },
    { UINT16_C(42926),
      simde_mm512_set_epi32(INT32_C(-1431480123), INT32_C(-2058827609), INT32_C(-1519596795), INT32_C(   24332922),
                            INT32_C( -338106630), INT32_C(-1565374776), INT32_C(-1426452996), INT32_C( -680300877),
                            INT32_C(-1675700291), INT32_C(  -85412591), INT32_C(-1865493216), INT32_C(-1122257925),
                            INT32_C(  955620837), INT32_C( -725693586), INT32_C( 1056307491), INT32_C( 1924019839)),
      simde_mm512_set_epi32(INT32_C( 1431480123), INT32_C(          0), INT32_C( 1519596795), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1565374776), INT32_C( 1426452996), INT32_C(  680300877),
                            INT32_C( 1675700291), INT32_C(          0), INT32_C( 1865493216), INT32_C(          0),
                            INT32_C(  955620837), INT32_C(  725693586), INT32_C( 1056307491), INT32_C(          0)) },
    { UINT16_C(26757),
      simde_mm512_set_epi32(INT32_C(-1863795060), INT32_C( -525595897), INT32_C( 1357119557), INT32_C(  837734387),
                            INT32_C( -607392699), INT32_C( -498581669), INT32_C(-2108693629), INT32_C( -476969927),
                            INT32_C(-1238615237), INT32_C(  583893938), INT32_C( -594441984), INT32_C( 1561597956),
                            INT32_C(  174377227), INT32_C(  319460903), INT32_C(-1295208114), INT32_C(  659707887)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  525595897), INT32_C( 1357119557), INT32_C(          0),
                            INT32_C(  607392699), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1238615237), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(  319460903), INT32_C(          0), INT32_C(  659707887)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_abs_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_abs_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-4703669018152042913), INT64_C(-4045642624518788548),
                            INT64_C(-4525466663746518179), INT64_C( 8998788960652053415),
                            INT64_C( 1494272869059842335), INT64_C( 8817699709611505791),
                            INT64_C( 6340501178400867239), INT64_C( -331077862235736168)),
      simde_mm512_set_epi64(INT64_C( 4703669018152042913), INT64_C( 4045642624518788548),
                            INT64_C( 4525466663746518179), INT64_C( 8998788960652053415),
                            INT64_C( 1494272869059842335), INT64_C( 8817699709611505791),
                            INT64_C( 6340501178400867239), INT64_C(  331077862235736168)) },
    { simde_mm512_set_epi64(INT64_C( 8010327509455286697), INT64_C( 7606702663991788660),
                            INT64_C( 1151136024847303203), INT64_C( 6049881760672440877),
                            INT64_C(  733511420638679938), INT64_C( 5858508694238056658),
                            INT64_C(-6405681308945653600), INT64_C( 6147272925506298466)),
      simde_mm512_set_epi64(INT64_C( 8010327509455286697), INT64_C( 7606702663991788660),
                            INT64_C( 1151136024847303203), INT64_C( 6049881760672440877),
                            INT64_C(  733511420638679938), INT64_C( 5858508694238056658),
                            INT64_C( 6405681308945653600), INT64_C( 6147272925506298466)) },
    { simde_mm512_set_epi64(INT64_C( 6464197827924287450), INT64_C( -497314884836304285),
                            INT64_C(-3088355329109459024), INT64_C( 6086538207170510268),
                            INT64_C( 5130097871257028467), INT64_C(-1548377050112752776),
                            INT64_C(-1253733446110746976), INT64_C( 3012333519594431544)),
      simde_mm512_set_epi64(INT64_C( 6464197827924287450), INT64_C(  497314884836304285),
                            INT64_C( 3088355329109459024), INT64_C( 6086538207170510268),
                            INT64_C( 5130097871257028467), INT64_C( 1548377050112752776),
                            INT64_C( 1253733446110746976), INT64_C( 3012333519594431544)) },
    { simde_mm512_set_epi64(INT64_C(-6609116217957060725), INT64_C( -302281769423418319),
                            INT64_C(-4897195678850214218), INT64_C(-5503480572790438091),
                            INT64_C(-7839393967146815223), INT64_C( 3255671599336790273),
                            INT64_C(  664294275788018935), INT64_C( 3408593724494687769)),
      simde_mm512_set_epi64(INT64_C( 6609116217957060725), INT64_C(  302281769423418319),
                            INT64_C( 4897195678850214218), INT64_C( 5503480572790438091),
                            INT64_C( 7839393967146815223), INT64_C( 3255671599336790273),
                            INT64_C(  664294275788018935), INT64_C( 3408593724494687769)) },
    { simde_mm512_set_epi64(INT64_C(-3947921567074644800), INT64_C( 2473906111070933983),
                            INT64_C( 5899615628251993446), INT64_C( 1683758976557896025),
                            INT64_C( 5912559464823232024), INT64_C( 8209801467605337080),
                            INT64_C(-3759026771366879931), INT64_C(-4585023147545297955)),
      simde_mm512_set_epi64(INT64_C( 3947921567074644800), INT64_C( 2473906111070933983),
                            INT64_C( 5899615628251993446), INT64_C( 1683758976557896025),
                            INT64_C( 5912559464823232024), INT64_C( 8209801467605337080),
                            INT64_C( 3759026771366879931), INT64_C( 4585023147545297955)) },
    { simde_mm512_set_epi64(INT64_C(-5018179428847904871), INT64_C(-5078614119960003085),
                            INT64_C( 1906444753996234817), INT64_C(-8073431190678733657),
                            INT64_C( 2507716860794484867), INT64_C( 3050920605853136553),
                            INT64_C( 8003726449701589009), INT64_C( 8169798351590582410)),
      simde_mm512_set_epi64(INT64_C( 5018179428847904871), INT64_C( 5078614119960003085),
                            INT64_C( 1906444753996234817), INT64_C( 8073431190678733657),
                            INT64_C( 2507716860794484867), INT64_C( 3050920605853136553),
                            INT64_C( 8003726449701589009), INT64_C( 8169798351590582410)) },
    { simde_mm512_set_epi64(INT64_C(   34320467490721535), INT64_C( 6032039111009323642),
                            INT64_C( 8825906149542039035), INT64_C( 3097839263351160271),
                            INT64_C( 8028249960129894953), INT64_C(-3442961435998375158),
                            INT64_C( 4762825474720326971), INT64_C( 5172383913584297790)),
      simde_mm512_set_epi64(INT64_C(   34320467490721535), INT64_C( 6032039111009323642),
                            INT64_C( 8825906149542039035), INT64_C( 3097839263351160271),
                            INT64_C( 8028249960129894953), INT64_C( 3442961435998375158),
                            INT64_C( 4762825474720326971), INT64_C( 5172383913584297790)) },
    { simde_mm512_set_epi64(INT64_C( 6012124616828815523), INT64_C(-1490086426868961318),
                            INT64_C(-6824757216361935934), INT64_C( 6020178289686090572),
                            INT64_C( 1854069096850744512), INT64_C(-5881095665354951863),
                            INT64_C( 6788068100773536681), INT64_C(-2707434157113404213)),
      simde_mm512_set_epi64(INT64_C( 6012124616828815523), INT64_C( 1490086426868961318),
                            INT64_C( 6824757216361935934), INT64_C( 6020178289686090572),
                            INT64_C( 1854069096850744512), INT64_C( 5881095665354951863),
                            INT64_C( 6788068100773536681), INT64_C( 2707434157113404213)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_abs_epi64(test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_abs_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(  492676116973233810), INT64_C(-6011138406694593910),
                            INT64_C( 4793302583727451571), INT64_C(-8501143198309462471),
                            INT64_C( 7175703865894427661), INT64_C( 8004913303465320948),
                            INT64_C(-2704856736044264837), INT64_C( 5658385619497272512)),
      UINT8_C(216),
      simde_mm512_set_epi64(INT64_C(-1717644828534315099), INT64_C(   78022038682650212),
                            INT64_C( 7139449655270167801), INT64_C( 2146714472316691717),
                            INT64_C( -192495991449383316), INT64_C(-5534743664186979260),
                            INT64_C(-5881663773003558792), INT64_C(-6698972267701962486)),
      simde_mm512_set_epi64(INT64_C( 1717644828534315099), INT64_C(   78022038682650212),
                            INT64_C( 4793302583727451571), INT64_C( 2146714472316691717),
                            INT64_C(  192495991449383316), INT64_C( 8004913303465320948),
                            INT64_C(-2704856736044264837), INT64_C( 5658385619497272512)) },
    { simde_mm512_set_epi64(INT64_C( 1469967797035145460), INT64_C( 5698418868819073779),
                            INT64_C( 1476440947581501981), INT64_C( 6259233690927012485),
                            INT64_C(-7834407389066617423), INT64_C( -820710109410615175),
                            INT64_C(-6555247677438161503), INT64_C( 3467536596098015717)),
      UINT8_C( 65),
      simde_mm512_set_epi64(INT64_C( 7787907596649075241), INT64_C( 6183988585695692436),
                            INT64_C( 6687836810634885572), INT64_C(-3481271883196348940),
                            INT64_C( 2810208213167470657), INT64_C(-4686213381983447596),
                            INT64_C( 1750544224109435853), INT64_C(-3649870730594357554)),
      simde_mm512_set_epi64(INT64_C( 1469967797035145460), INT64_C( 6183988585695692436),
                            INT64_C( 1476440947581501981), INT64_C( 6259233690927012485),
                            INT64_C(-7834407389066617423), INT64_C( -820710109410615175),
                            INT64_C(-6555247677438161503), INT64_C( 3649870730594357554)) },
    { simde_mm512_set_epi64(INT64_C(-8324574269298179729), INT64_C(-7258330021648709528),
                            INT64_C( 5937514152424000288), INT64_C( 1850437298497726112),
                            INT64_C(-8404329205387466185), INT64_C( 2671708839418006562),
                            INT64_C( 2003666284095471028), INT64_C( 6129159598982782124)),
      UINT8_C(199),
      simde_mm512_set_epi64(INT64_C( 7895846628610550850), INT64_C(-2067803869119741500),
                            INT64_C(-4037099766843598938), INT64_C(-2755209390738717188),
                            INT64_C(-7716919860907040546), INT64_C(  651602748594909758),
                            INT64_C(-1368765145407975583), INT64_C( -989534720059255127)),
      simde_mm512_set_epi64(INT64_C( 7895846628610550850), INT64_C( 2067803869119741500),
                            INT64_C( 5937514152424000288), INT64_C( 1850437298497726112),
                            INT64_C(-8404329205387466185), INT64_C(  651602748594909758),
                            INT64_C( 1368765145407975583), INT64_C(  989534720059255127)) },
    { simde_mm512_set_epi64(INT64_C(-4820061084596199963), INT64_C(-3116830217730655965),
                            INT64_C( 8263602287642686524), INT64_C(-7769087937993864972),
                            INT64_C(  688147068097687318), INT64_C( 5251800971073735884),
                            INT64_C( 1217023964204045922), INT64_C( 2408716443164236524)),
      UINT8_C( 32),
      simde_mm512_set_epi64(INT64_C(  748944487451629095), INT64_C(-5562876490484131857),
                            INT64_C( 4801338493970245765), INT64_C(-6148160310922917721),
                            INT64_C(-6526618537607083398), INT64_C(-1452156915681179960),
                            INT64_C(-6126568963486552397), INT64_C(-7197077943533128431)),
      simde_mm512_set_epi64(INT64_C(-4820061084596199963), INT64_C(-3116830217730655965),
                            INT64_C( 4801338493970245765), INT64_C(-7769087937993864972),
                            INT64_C(  688147068097687318), INT64_C( 5251800971073735884),
                            INT64_C( 1217023964204045922), INT64_C( 2408716443164236524)) },
    { simde_mm512_set_epi64(INT64_C(-3096584980416416798), INT64_C( 4918909782932989693),
                            INT64_C(-8004938825376986361), INT64_C( 5828784114914742259),
                            INT64_C(-2608731774237786277), INT64_C(-9056770170020559815),
                            INT64_C(-5319811934658395214), INT64_C(-2553108879087757308)),
      UINT8_C(222),
      simde_mm512_set_epi64(INT64_C(-7429178691671160403), INT64_C(-6726241153446442388),
                            INT64_C(-4671168844950875702), INT64_C(-6057578907613688046),
                            INT64_C( 4404786722084486842), INT64_C(-1094983952222664046),
                            INT64_C( 5287971478839612040), INT64_C(-1393761486045455198)),
      simde_mm512_set_epi64(INT64_C( 7429178691671160403), INT64_C( 6726241153446442388),
                            INT64_C(-8004938825376986361), INT64_C( 6057578907613688046),
                            INT64_C( 4404786722084486842), INT64_C( 1094983952222664046),
                            INT64_C( 5287971478839612040), INT64_C(-2553108879087757308)) },
    { simde_mm512_set_epi64(INT64_C(-2342519952706594553), INT64_C( 7644592964730421973),
                            INT64_C( 9172903191652197902), INT64_C( 1458420204680989133),
                            INT64_C(-7367844323210688147), INT64_C(-7391408909901332845),
                            INT64_C(-1133308551202396595), INT64_C( 4102785619076298947)),
      UINT8_C(230),
      simde_mm512_set_epi64(INT64_C( 2067960823776290910), INT64_C(-7228934469556881970),
                            INT64_C(-3653374474835545853), INT64_C( 5480624866862364429),
                            INT64_C(-6600577412846457563), INT64_C( 2529415530022027870),
                            INT64_C(-4329045443225919216), INT64_C( 9121454504064466670)),
      simde_mm512_set_epi64(INT64_C( 2067960823776290910), INT64_C( 7228934469556881970),
                            INT64_C( 3653374474835545853), INT64_C( 1458420204680989133),
                            INT64_C(-7367844323210688147), INT64_C( 2529415530022027870),
                            INT64_C( 4329045443225919216), INT64_C( 4102785619076298947)) },
    { simde_mm512_set_epi64(INT64_C(-4792271441403297267), INT64_C( 6523863092598044385),
                            INT64_C( 6601631579296004837), INT64_C(  540377227778745844),
                            INT64_C(-1608274285869950535), INT64_C( 1496684406111625906),
                            INT64_C( 1553447803787926005), INT64_C(-8007245467626955940)),
      UINT8_C(104),
      simde_mm512_set_epi64(INT64_C( 3154046907468289764), INT64_C(  163566292437055869),
                            INT64_C( 8677719201018760530), INT64_C( 8769975268177995652),
                            INT64_C( 7613129594859420923), INT64_C(-1998576254813523574),
                            INT64_C( 6270036339063321053), INT64_C( 5566883881093264567)),
      simde_mm512_set_epi64(INT64_C(-4792271441403297267), INT64_C(  163566292437055869),
                            INT64_C( 8677719201018760530), INT64_C(  540377227778745844),
                            INT64_C( 7613129594859420923), INT64_C( 1496684406111625906),
                            INT64_C( 1553447803787926005), INT64_C(-8007245467626955940)) },
    { simde_mm512_set_epi64(INT64_C( 6148806541912347944), INT64_C( 1908371497711301408),
                            INT64_C(-9141679846684482011), INT64_C(-5393015831913260049),
                            INT64_C(-7420100767347904040), INT64_C(-3473408352959472719),
                            INT64_C( 3774406077177521346), INT64_C(-4000561069769581270)),
      UINT8_C(128),
      simde_mm512_set_epi64(INT64_C( 8461115545003933300), INT64_C(-7617041399687843974),
                            INT64_C( 7167540182740956629), INT64_C( 7612124823363779747),
                            INT64_C(-1891905030773424117), INT64_C(-5190678261404669746),
                            INT64_C( -683318933916552175), INT64_C( 9111998746576489775)),
      simde_mm512_set_epi64(INT64_C( 8461115545003933300), INT64_C( 1908371497711301408),
                            INT64_C(-9141679846684482011), INT64_C(-5393015831913260049),
                            INT64_C(-7420100767347904040), INT64_C(-3473408352959472719),
                            INT64_C( 3774406077177521346), INT64_C(-4000561069769581270)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_abs_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_abs_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C(192),
      simde_mm512_set_epi64(INT64_C(  -41252748446509487), INT64_C( 7890911908509001079),
                            INT64_C(-5961962669328745651), INT64_C(-3744340997299642547),
                            INT64_C( 6343687558518880421), INT64_C(-4320961892205516228),
                            INT64_C(-2965231175381652703), INT64_C( 2864943002512957373)),
      simde_mm512_set_epi64(INT64_C(   41252748446509487), INT64_C( 7890911908509001079),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C( 10),
      simde_mm512_set_epi64(INT64_C(-8064009705201487071), INT64_C(   66682790377500833),
                            INT64_C( 8959003693208071074), INT64_C(-7914560703715169936),
                            INT64_C( 6097984971859041770), INT64_C( 7564092909171024886),
                            INT64_C( 3840035858897969653), INT64_C( -443938296699520969)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 6097984971859041770), INT64_C(                   0),
                            INT64_C( 3840035858897969653), INT64_C(                   0)) },
    { UINT8_C( 41),
      simde_mm512_set_epi64(INT64_C(-8443142306353437631), INT64_C( 1469967797035145460),
                            INT64_C( 5698418868819073779), INT64_C( 1476440947581501981),
                            INT64_C( 6259233690927012485), INT64_C(-7834407389066617423),
                            INT64_C( -820710109410615175), INT64_C(-6555247677438161503)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 5698418868819073779), INT64_C(                   0),
                            INT64_C( 6259233690927012485), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 6555247677438161503)) },
    { UINT8_C(232),
      simde_mm512_set_epi64(INT64_C( 6129159598982782124), INT64_C( 7787907596649075241),
                            INT64_C( 6183988585695692436), INT64_C( 6687836810634885572),
                            INT64_C(-3481271883196348940), INT64_C( 2810208213167470657),
                            INT64_C(-4686213381983447596), INT64_C( 1750544224109435853)),
      simde_mm512_set_epi64(INT64_C( 6129159598982782124), INT64_C( 7787907596649075241),
                            INT64_C( 6183988585695692436), INT64_C(                   0),
                            INT64_C( 3481271883196348940), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(180),
      simde_mm512_set_epi64(INT64_C( -989534720059255127), INT64_C(-4487290813852079154),
                            INT64_C(-4890907616466355379), INT64_C( 3824480121463916969),
                            INT64_C(-3452489463091031317), INT64_C( 4618202413866537757),
                            INT64_C(-6813829451133759224), INT64_C(-4843608058602032162)),
      simde_mm512_set_epi64(INT64_C(  989534720059255127), INT64_C(                   0),
                            INT64_C( 4890907616466355379), INT64_C( 3824480121463916969),
                            INT64_C(                   0), INT64_C( 4618202413866537757),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C( 97),
      simde_mm512_set_epi64(INT64_C(-1331633504094684457), INT64_C(-2922173236712853844),
                            INT64_C(-4950533224594034858), INT64_C( 3742633972513462054),
                            INT64_C(-3856884862992283736), INT64_C( 2824537604935384645),
                            INT64_C(-4433054007697935041), INT64_C( 6677415749608352602)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 2922173236712853844),
                            INT64_C( 4950533224594034858), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 6677415749608352602)) },
    { UINT8_C(216),
      simde_mm512_set_epi64(INT64_C(-2921870015535851587), INT64_C( -366844282582149856),
                            INT64_C(-4820061084596199963), INT64_C(-3116830217730655965),
                            INT64_C( 8263602287642686524), INT64_C(-7769087937993864972),
                            INT64_C(  688147068097687318), INT64_C( 5251800971073735884)),
      simde_mm512_set_epi64(INT64_C( 2921870015535851587), INT64_C(  366844282582149856),
                            INT64_C(                   0), INT64_C( 3116830217730655965),
                            INT64_C( 8263602287642686524), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(252),
      simde_mm512_set_epi64(INT64_C(-5319811934658395214), INT64_C(-2553108879087757308),
                            INT64_C(  748944487451629095), INT64_C(-5562876490484131857),
                            INT64_C( 4801338493970245765), INT64_C(-6148160310922917721),
                            INT64_C(-6526618537607083398), INT64_C(-1452156915681179960)),
      simde_mm512_set_epi64(INT64_C( 5319811934658395214), INT64_C( 2553108879087757308),
                            INT64_C(  748944487451629095), INT64_C( 5562876490484131857),
                            INT64_C( 4801338493970245765), INT64_C( 6148160310922917721),
                            INT64_C(                   0), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_abs_epi64(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_abs_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   747.74), SIMDE_FLOAT32_C(  -874.37), SIMDE_FLOAT32_C(   751.90), SIMDE_FLOAT32_C(  -592.77),
                         SIMDE_FLOAT32_C(  -708.81), SIMDE_FLOAT32_C(   252.42), SIMDE_FLOAT32_C(  -787.46), SIMDE_FLOAT32_C(  -882.47),
                         SIMDE_FLOAT32_C(  -140.56), SIMDE_FLOAT32_C(  -558.99), SIMDE_FLOAT32_C(   240.08), SIMDE_FLOAT32_C(  -481.72),
                         SIMDE_FLOAT32_C(   489.35), SIMDE_FLOAT32_C(   686.76), SIMDE_FLOAT32_C(  -206.54), SIMDE_FLOAT32_C(   728.61)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   747.74), SIMDE_FLOAT32_C(   874.37), SIMDE_FLOAT32_C(   751.90), SIMDE_FLOAT32_C(   592.77),
                         SIMDE_FLOAT32_C(   708.81), SIMDE_FLOAT32_C(   252.42), SIMDE_FLOAT32_C(   787.46), SIMDE_FLOAT32_C(   882.47),
                         SIMDE_FLOAT32_C(   140.56), SIMDE_FLOAT32_C(   558.99), SIMDE_FLOAT32_C(   240.08), SIMDE_FLOAT32_C(   481.72),
                         SIMDE_FLOAT32_C(   489.35), SIMDE_FLOAT32_C(   686.76), SIMDE_FLOAT32_C(   206.54), SIMDE_FLOAT32_C(   728.61)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    76.28), SIMDE_FLOAT32_C(  -319.23), SIMDE_FLOAT32_C(   655.09), SIMDE_FLOAT32_C(   773.21),
                         SIMDE_FLOAT32_C(  -928.32), SIMDE_FLOAT32_C(   -25.13), SIMDE_FLOAT32_C(  -847.53), SIMDE_FLOAT32_C(   859.40),
                         SIMDE_FLOAT32_C(   388.54), SIMDE_FLOAT32_C(  -184.67), SIMDE_FLOAT32_C(   102.38), SIMDE_FLOAT32_C(   833.56),
                         SIMDE_FLOAT32_C(  -722.29), SIMDE_FLOAT32_C(  -441.84), SIMDE_FLOAT32_C(  -821.42), SIMDE_FLOAT32_C(  -761.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    76.28), SIMDE_FLOAT32_C(   319.23), SIMDE_FLOAT32_C(   655.09), SIMDE_FLOAT32_C(   773.21),
                         SIMDE_FLOAT32_C(   928.32), SIMDE_FLOAT32_C(    25.13), SIMDE_FLOAT32_C(   847.53), SIMDE_FLOAT32_C(   859.40),
                         SIMDE_FLOAT32_C(   388.54), SIMDE_FLOAT32_C(   184.67), SIMDE_FLOAT32_C(   102.38), SIMDE_FLOAT32_C(   833.56),
                         SIMDE_FLOAT32_C(   722.29), SIMDE_FLOAT32_C(   441.84), SIMDE_FLOAT32_C(   821.42), SIMDE_FLOAT32_C(   761.98)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -645.61), SIMDE_FLOAT32_C(   594.77), SIMDE_FLOAT32_C(  -171.69), SIMDE_FLOAT32_C(   108.08),
                         SIMDE_FLOAT32_C(    -7.24), SIMDE_FLOAT32_C(   885.82), SIMDE_FLOAT32_C(   296.84), SIMDE_FLOAT32_C(  -408.70),
                         SIMDE_FLOAT32_C(   -40.31), SIMDE_FLOAT32_C(   866.84), SIMDE_FLOAT32_C(  -660.11), SIMDE_FLOAT32_C(   121.17),
                         SIMDE_FLOAT32_C(   988.31), SIMDE_FLOAT32_C(  -622.26), SIMDE_FLOAT32_C(   206.00), SIMDE_FLOAT32_C(   520.48)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   645.61), SIMDE_FLOAT32_C(   594.77), SIMDE_FLOAT32_C(   171.69), SIMDE_FLOAT32_C(   108.08),
                         SIMDE_FLOAT32_C(     7.24), SIMDE_FLOAT32_C(   885.82), SIMDE_FLOAT32_C(   296.84), SIMDE_FLOAT32_C(   408.70),
                         SIMDE_FLOAT32_C(    40.31), SIMDE_FLOAT32_C(   866.84), SIMDE_FLOAT32_C(   660.11), SIMDE_FLOAT32_C(   121.17),
                         SIMDE_FLOAT32_C(   988.31), SIMDE_FLOAT32_C(   622.26), SIMDE_FLOAT32_C(   206.00), SIMDE_FLOAT32_C(   520.48)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   530.77), SIMDE_FLOAT32_C(   966.97), SIMDE_FLOAT32_C(   -63.51), SIMDE_FLOAT32_C(   360.07),
                         SIMDE_FLOAT32_C(  -846.61), SIMDE_FLOAT32_C(  -749.79), SIMDE_FLOAT32_C(   510.77), SIMDE_FLOAT32_C(  -104.12),
                         SIMDE_FLOAT32_C(  -838.06), SIMDE_FLOAT32_C(  -901.25), SIMDE_FLOAT32_C(   -89.58), SIMDE_FLOAT32_C(   539.88),
                         SIMDE_FLOAT32_C(    88.35), SIMDE_FLOAT32_C(   773.77), SIMDE_FLOAT32_C(  -729.20), SIMDE_FLOAT32_C(  -254.72)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   530.77), SIMDE_FLOAT32_C(   966.97), SIMDE_FLOAT32_C(    63.51), SIMDE_FLOAT32_C(   360.07),
                         SIMDE_FLOAT32_C(   846.61), SIMDE_FLOAT32_C(   749.79), SIMDE_FLOAT32_C(   510.77), SIMDE_FLOAT32_C(   104.12),
                         SIMDE_FLOAT32_C(   838.06), SIMDE_FLOAT32_C(   901.25), SIMDE_FLOAT32_C(    89.58), SIMDE_FLOAT32_C(   539.88),
                         SIMDE_FLOAT32_C(    88.35), SIMDE_FLOAT32_C(   773.77), SIMDE_FLOAT32_C(   729.20), SIMDE_FLOAT32_C(   254.72)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -814.45), SIMDE_FLOAT32_C(  -377.80), SIMDE_FLOAT32_C(   640.68), SIMDE_FLOAT32_C(   778.00),
                         SIMDE_FLOAT32_C(   377.67), SIMDE_FLOAT32_C(  -489.06), SIMDE_FLOAT32_C(   933.74), SIMDE_FLOAT32_C(  -749.41),
                         SIMDE_FLOAT32_C(   193.12), SIMDE_FLOAT32_C(  -423.37), SIMDE_FLOAT32_C(  -194.06), SIMDE_FLOAT32_C(  -118.88),
                         SIMDE_FLOAT32_C(   -77.74), SIMDE_FLOAT32_C(  -506.16), SIMDE_FLOAT32_C(  -617.33), SIMDE_FLOAT32_C(  -947.60)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   814.45), SIMDE_FLOAT32_C(   377.80), SIMDE_FLOAT32_C(   640.68), SIMDE_FLOAT32_C(   778.00),
                         SIMDE_FLOAT32_C(   377.67), SIMDE_FLOAT32_C(   489.06), SIMDE_FLOAT32_C(   933.74), SIMDE_FLOAT32_C(   749.41),
                         SIMDE_FLOAT32_C(   193.12), SIMDE_FLOAT32_C(   423.37), SIMDE_FLOAT32_C(   194.06), SIMDE_FLOAT32_C(   118.88),
                         SIMDE_FLOAT32_C(    77.74), SIMDE_FLOAT32_C(   506.16), SIMDE_FLOAT32_C(   617.33), SIMDE_FLOAT32_C(   947.60)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   525.11), SIMDE_FLOAT32_C(   299.00), SIMDE_FLOAT32_C(   814.48), SIMDE_FLOAT32_C(   676.51),
                         SIMDE_FLOAT32_C(  -481.76), SIMDE_FLOAT32_C(   528.75), SIMDE_FLOAT32_C(  -375.20), SIMDE_FLOAT32_C(   146.55),
                         SIMDE_FLOAT32_C(   199.14), SIMDE_FLOAT32_C(  -505.05), SIMDE_FLOAT32_C(   833.96), SIMDE_FLOAT32_C(  -388.48),
                         SIMDE_FLOAT32_C(  -212.57), SIMDE_FLOAT32_C(   943.89), SIMDE_FLOAT32_C(   651.63), SIMDE_FLOAT32_C(   695.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   525.11), SIMDE_FLOAT32_C(   299.00), SIMDE_FLOAT32_C(   814.48), SIMDE_FLOAT32_C(   676.51),
                         SIMDE_FLOAT32_C(   481.76), SIMDE_FLOAT32_C(   528.75), SIMDE_FLOAT32_C(   375.20), SIMDE_FLOAT32_C(   146.55),
                         SIMDE_FLOAT32_C(   199.14), SIMDE_FLOAT32_C(   505.05), SIMDE_FLOAT32_C(   833.96), SIMDE_FLOAT32_C(   388.48),
                         SIMDE_FLOAT32_C(   212.57), SIMDE_FLOAT32_C(   943.89), SIMDE_FLOAT32_C(   651.63), SIMDE_FLOAT32_C(   695.54)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -13.73), SIMDE_FLOAT32_C(  -546.29), SIMDE_FLOAT32_C(  -787.44), SIMDE_FLOAT32_C(  -104.88),
                         SIMDE_FLOAT32_C(   979.47), SIMDE_FLOAT32_C(  -744.23), SIMDE_FLOAT32_C(   836.15), SIMDE_FLOAT32_C(   495.73),
                         SIMDE_FLOAT32_C(  -301.39), SIMDE_FLOAT32_C(   262.00), SIMDE_FLOAT32_C(   466.22), SIMDE_FLOAT32_C(   536.10),
                         SIMDE_FLOAT32_C(  -613.16), SIMDE_FLOAT32_C(  -393.36), SIMDE_FLOAT32_C(   -56.94), SIMDE_FLOAT32_C(   670.22)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    13.73), SIMDE_FLOAT32_C(   546.29), SIMDE_FLOAT32_C(   787.44), SIMDE_FLOAT32_C(   104.88),
                         SIMDE_FLOAT32_C(   979.47), SIMDE_FLOAT32_C(   744.23), SIMDE_FLOAT32_C(   836.15), SIMDE_FLOAT32_C(   495.73),
                         SIMDE_FLOAT32_C(   301.39), SIMDE_FLOAT32_C(   262.00), SIMDE_FLOAT32_C(   466.22), SIMDE_FLOAT32_C(   536.10),
                         SIMDE_FLOAT32_C(   613.16), SIMDE_FLOAT32_C(   393.36), SIMDE_FLOAT32_C(    56.94), SIMDE_FLOAT32_C(   670.22)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   493.79), SIMDE_FLOAT32_C(   -29.44), SIMDE_FLOAT32_C(  -941.83), SIMDE_FLOAT32_C(  -567.95),
                         SIMDE_FLOAT32_C(   535.05), SIMDE_FLOAT32_C(    43.85), SIMDE_FLOAT32_C(  -963.94), SIMDE_FLOAT32_C(   235.87),
                         SIMDE_FLOAT32_C(   143.93), SIMDE_FLOAT32_C(  -236.80), SIMDE_FLOAT32_C(   550.36), SIMDE_FLOAT32_C(    -8.58),
                         SIMDE_FLOAT32_C(   374.16), SIMDE_FLOAT32_C(   714.91), SIMDE_FLOAT32_C(  -355.51), SIMDE_FLOAT32_C(  -520.52)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   493.79), SIMDE_FLOAT32_C(    29.44), SIMDE_FLOAT32_C(   941.83), SIMDE_FLOAT32_C(   567.95),
                         SIMDE_FLOAT32_C(   535.05), SIMDE_FLOAT32_C(    43.85), SIMDE_FLOAT32_C(   963.94), SIMDE_FLOAT32_C(   235.87),
                         SIMDE_FLOAT32_C(   143.93), SIMDE_FLOAT32_C(   236.80), SIMDE_FLOAT32_C(   550.36), SIMDE_FLOAT32_C(     8.58),
                         SIMDE_FLOAT32_C(   374.16), SIMDE_FLOAT32_C(   714.91), SIMDE_FLOAT32_C(   355.51), SIMDE_FLOAT32_C(   520.52)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_abs_ps(test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_abs_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -319.23), SIMDE_FLOAT32_C(   773.21), SIMDE_FLOAT32_C(   -25.13), SIMDE_FLOAT32_C(   859.40),
                         SIMDE_FLOAT32_C(  -184.67), SIMDE_FLOAT32_C(   833.56), SIMDE_FLOAT32_C(  -441.84), SIMDE_FLOAT32_C(  -761.98),
                         SIMDE_FLOAT32_C(  -874.37), SIMDE_FLOAT32_C(  -592.77), SIMDE_FLOAT32_C(   252.42), SIMDE_FLOAT32_C(  -882.47),
                         SIMDE_FLOAT32_C(  -558.99), SIMDE_FLOAT32_C(  -481.72), SIMDE_FLOAT32_C(   686.76), SIMDE_FLOAT32_C(   728.61)),
      UINT16_C(15540),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    76.28), SIMDE_FLOAT32_C(   655.09), SIMDE_FLOAT32_C(  -928.32), SIMDE_FLOAT32_C(  -847.53),
                         SIMDE_FLOAT32_C(   388.54), SIMDE_FLOAT32_C(   102.38), SIMDE_FLOAT32_C(  -722.29), SIMDE_FLOAT32_C(  -821.42),
                         SIMDE_FLOAT32_C(   747.74), SIMDE_FLOAT32_C(   751.90), SIMDE_FLOAT32_C(  -708.81), SIMDE_FLOAT32_C(  -787.46),
                         SIMDE_FLOAT32_C(  -140.56), SIMDE_FLOAT32_C(   240.08), SIMDE_FLOAT32_C(   489.35), SIMDE_FLOAT32_C(  -206.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -319.23), SIMDE_FLOAT32_C(   773.21), SIMDE_FLOAT32_C(   928.32), SIMDE_FLOAT32_C(   847.53),
                         SIMDE_FLOAT32_C(   388.54), SIMDE_FLOAT32_C(   102.38), SIMDE_FLOAT32_C(  -441.84), SIMDE_FLOAT32_C(  -761.98),
                         SIMDE_FLOAT32_C(   747.74), SIMDE_FLOAT32_C(  -592.77), SIMDE_FLOAT32_C(   708.81), SIMDE_FLOAT32_C(   787.46),
                         SIMDE_FLOAT32_C(  -558.99), SIMDE_FLOAT32_C(   240.08), SIMDE_FLOAT32_C(   686.76), SIMDE_FLOAT32_C(   728.61)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -947.60), SIMDE_FLOAT32_C(   966.97), SIMDE_FLOAT32_C(   360.07), SIMDE_FLOAT32_C(  -749.79),
                         SIMDE_FLOAT32_C(  -104.12), SIMDE_FLOAT32_C(  -901.25), SIMDE_FLOAT32_C(   539.88), SIMDE_FLOAT32_C(   773.77),
                         SIMDE_FLOAT32_C(  -254.72), SIMDE_FLOAT32_C(   594.77), SIMDE_FLOAT32_C(   108.08), SIMDE_FLOAT32_C(   885.82),
                         SIMDE_FLOAT32_C(  -408.70), SIMDE_FLOAT32_C(   866.84), SIMDE_FLOAT32_C(   121.17), SIMDE_FLOAT32_C(  -622.26)),
      UINT16_C( 6415),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -617.33), SIMDE_FLOAT32_C(   530.77), SIMDE_FLOAT32_C(   -63.51), SIMDE_FLOAT32_C(  -846.61),
                         SIMDE_FLOAT32_C(   510.77), SIMDE_FLOAT32_C(  -838.06), SIMDE_FLOAT32_C(   -89.58), SIMDE_FLOAT32_C(    88.35),
                         SIMDE_FLOAT32_C(  -729.20), SIMDE_FLOAT32_C(  -645.61), SIMDE_FLOAT32_C(  -171.69), SIMDE_FLOAT32_C(    -7.24),
                         SIMDE_FLOAT32_C(   296.84), SIMDE_FLOAT32_C(   -40.31), SIMDE_FLOAT32_C(  -660.11), SIMDE_FLOAT32_C(   988.31)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -947.60), SIMDE_FLOAT32_C(   966.97), SIMDE_FLOAT32_C(   360.07), SIMDE_FLOAT32_C(   846.61),
                         SIMDE_FLOAT32_C(   510.77), SIMDE_FLOAT32_C(  -901.25), SIMDE_FLOAT32_C(   539.88), SIMDE_FLOAT32_C(    88.35),
                         SIMDE_FLOAT32_C(  -254.72), SIMDE_FLOAT32_C(   594.77), SIMDE_FLOAT32_C(   108.08), SIMDE_FLOAT32_C(   885.82),
                         SIMDE_FLOAT32_C(   296.84), SIMDE_FLOAT32_C(    40.31), SIMDE_FLOAT32_C(   660.11), SIMDE_FLOAT32_C(   988.31)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -393.36), SIMDE_FLOAT32_C(   670.22), SIMDE_FLOAT32_C(   299.00), SIMDE_FLOAT32_C(   676.51),
                         SIMDE_FLOAT32_C(   528.75), SIMDE_FLOAT32_C(   146.55), SIMDE_FLOAT32_C(  -505.05), SIMDE_FLOAT32_C(  -388.48),
                         SIMDE_FLOAT32_C(   943.89), SIMDE_FLOAT32_C(   695.54), SIMDE_FLOAT32_C(  -377.80), SIMDE_FLOAT32_C(   778.00),
                         SIMDE_FLOAT32_C(  -489.06), SIMDE_FLOAT32_C(  -749.41), SIMDE_FLOAT32_C(  -423.37), SIMDE_FLOAT32_C(  -118.88)),
      UINT16_C( 1525),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -613.16), SIMDE_FLOAT32_C(   -56.94), SIMDE_FLOAT32_C(   525.11), SIMDE_FLOAT32_C(   814.48),
                         SIMDE_FLOAT32_C(  -481.76), SIMDE_FLOAT32_C(  -375.20), SIMDE_FLOAT32_C(   199.14), SIMDE_FLOAT32_C(   833.96),
                         SIMDE_FLOAT32_C(  -212.57), SIMDE_FLOAT32_C(   651.63), SIMDE_FLOAT32_C(  -814.45), SIMDE_FLOAT32_C(   640.68),
                         SIMDE_FLOAT32_C(   377.67), SIMDE_FLOAT32_C(   933.74), SIMDE_FLOAT32_C(   193.12), SIMDE_FLOAT32_C(  -194.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -393.36), SIMDE_FLOAT32_C(   670.22), SIMDE_FLOAT32_C(   299.00), SIMDE_FLOAT32_C(   676.51),
                         SIMDE_FLOAT32_C(   528.75), SIMDE_FLOAT32_C(   375.20), SIMDE_FLOAT32_C(  -505.05), SIMDE_FLOAT32_C(   833.96),
                         SIMDE_FLOAT32_C(   212.57), SIMDE_FLOAT32_C(   651.63), SIMDE_FLOAT32_C(   814.45), SIMDE_FLOAT32_C(   640.68),
                         SIMDE_FLOAT32_C(  -489.06), SIMDE_FLOAT32_C(   933.74), SIMDE_FLOAT32_C(  -423.37), SIMDE_FLOAT32_C(   194.06)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    61.51), SIMDE_FLOAT32_C(  -643.69), SIMDE_FLOAT32_C(   -16.59), SIMDE_FLOAT32_C(   -29.44),
                         SIMDE_FLOAT32_C(  -567.95), SIMDE_FLOAT32_C(    43.85), SIMDE_FLOAT32_C(   235.87), SIMDE_FLOAT32_C(  -236.80),
                         SIMDE_FLOAT32_C(    -8.58), SIMDE_FLOAT32_C(   714.91), SIMDE_FLOAT32_C(  -520.52), SIMDE_FLOAT32_C(  -546.29),
                         SIMDE_FLOAT32_C(  -104.88), SIMDE_FLOAT32_C(  -744.23), SIMDE_FLOAT32_C(   495.73), SIMDE_FLOAT32_C(   262.00)),
      UINT16_C(29879),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   515.30), SIMDE_FLOAT32_C(   896.28), SIMDE_FLOAT32_C(   660.35), SIMDE_FLOAT32_C(   493.79),
                         SIMDE_FLOAT32_C(  -941.83), SIMDE_FLOAT32_C(   535.05), SIMDE_FLOAT32_C(  -963.94), SIMDE_FLOAT32_C(   143.93),
                         SIMDE_FLOAT32_C(   550.36), SIMDE_FLOAT32_C(   374.16), SIMDE_FLOAT32_C(  -355.51), SIMDE_FLOAT32_C(   -13.73),
                         SIMDE_FLOAT32_C(  -787.44), SIMDE_FLOAT32_C(   979.47), SIMDE_FLOAT32_C(   836.15), SIMDE_FLOAT32_C(  -301.39)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    61.51), SIMDE_FLOAT32_C(   896.28), SIMDE_FLOAT32_C(   660.35), SIMDE_FLOAT32_C(   493.79),
                         SIMDE_FLOAT32_C(  -567.95), SIMDE_FLOAT32_C(   535.05), SIMDE_FLOAT32_C(   235.87), SIMDE_FLOAT32_C(  -236.80),
                         SIMDE_FLOAT32_C(   550.36), SIMDE_FLOAT32_C(   714.91), SIMDE_FLOAT32_C(   355.51), SIMDE_FLOAT32_C(    13.73),
                         SIMDE_FLOAT32_C(  -104.88), SIMDE_FLOAT32_C(   979.47), SIMDE_FLOAT32_C(   836.15), SIMDE_FLOAT32_C(   301.39)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   148.99), SIMDE_FLOAT32_C(  -963.65), SIMDE_FLOAT32_C(   149.45), SIMDE_FLOAT32_C(  -850.34),
                         SIMDE_FLOAT32_C(  -524.37), SIMDE_FLOAT32_C(  -513.69), SIMDE_FLOAT32_C(    22.08), SIMDE_FLOAT32_C(   488.53),
                         SIMDE_FLOAT32_C(   770.65), SIMDE_FLOAT32_C(   491.66), SIMDE_FLOAT32_C(    89.59), SIMDE_FLOAT32_C(   924.64),
                         SIMDE_FLOAT32_C(  -763.40), SIMDE_FLOAT32_C(  -404.62), SIMDE_FLOAT32_C(  -957.75), SIMDE_FLOAT32_C(   281.78)),
      UINT16_C(44157),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -979.51), SIMDE_FLOAT32_C(  -129.70), SIMDE_FLOAT32_C(  -587.42), SIMDE_FLOAT32_C(    94.97),
                         SIMDE_FLOAT32_C(  -887.16), SIMDE_FLOAT32_C(  -189.75), SIMDE_FLOAT32_C(   881.78), SIMDE_FLOAT32_C(  -152.81),
                         SIMDE_FLOAT32_C(   943.19), SIMDE_FLOAT32_C(  -229.02), SIMDE_FLOAT32_C(  -577.41), SIMDE_FLOAT32_C(  -719.96),
                         SIMDE_FLOAT32_C(   770.58), SIMDE_FLOAT32_C(  -153.52), SIMDE_FLOAT32_C(  -991.64), SIMDE_FLOAT32_C(   -53.48)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   979.51), SIMDE_FLOAT32_C(  -963.65), SIMDE_FLOAT32_C(   587.42), SIMDE_FLOAT32_C(  -850.34),
                         SIMDE_FLOAT32_C(   887.16), SIMDE_FLOAT32_C(   189.75), SIMDE_FLOAT32_C(    22.08), SIMDE_FLOAT32_C(   488.53),
                         SIMDE_FLOAT32_C(   770.65), SIMDE_FLOAT32_C(   229.02), SIMDE_FLOAT32_C(   577.41), SIMDE_FLOAT32_C(   719.96),
                         SIMDE_FLOAT32_C(   770.58), SIMDE_FLOAT32_C(   153.52), SIMDE_FLOAT32_C(  -957.75), SIMDE_FLOAT32_C(    53.48)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   631.94), SIMDE_FLOAT32_C(  -409.79), SIMDE_FLOAT32_C(   668.07), SIMDE_FLOAT32_C(   542.88),
                         SIMDE_FLOAT32_C(  -896.06), SIMDE_FLOAT32_C(   248.80), SIMDE_FLOAT32_C(   200.01), SIMDE_FLOAT32_C(   669.33),
                         SIMDE_FLOAT32_C(  -642.07), SIMDE_FLOAT32_C(  -212.55), SIMDE_FLOAT32_C(  -356.51), SIMDE_FLOAT32_C(  -440.95),
                         SIMDE_FLOAT32_C(  -982.52), SIMDE_FLOAT32_C(  -842.67), SIMDE_FLOAT32_C(  -420.59), SIMDE_FLOAT32_C(  -949.02)),
      UINT16_C(15240),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   966.19), SIMDE_FLOAT32_C(   529.24), SIMDE_FLOAT32_C(  -544.06), SIMDE_FLOAT32_C(  -881.83),
                         SIMDE_FLOAT32_C(  -242.38), SIMDE_FLOAT32_C(  -380.44), SIMDE_FLOAT32_C(  -752.70), SIMDE_FLOAT32_C(  -160.45),
                         SIMDE_FLOAT32_C(   773.41), SIMDE_FLOAT32_C(  -474.98), SIMDE_FLOAT32_C(   573.78), SIMDE_FLOAT32_C(  -190.69),
                         SIMDE_FLOAT32_C(  -743.99), SIMDE_FLOAT32_C(  -698.61), SIMDE_FLOAT32_C(  -633.81), SIMDE_FLOAT32_C(   938.50)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   631.94), SIMDE_FLOAT32_C(  -409.79), SIMDE_FLOAT32_C(   544.06), SIMDE_FLOAT32_C(   881.83),
                         SIMDE_FLOAT32_C(   242.38), SIMDE_FLOAT32_C(   248.80), SIMDE_FLOAT32_C(   752.70), SIMDE_FLOAT32_C(   160.45),
                         SIMDE_FLOAT32_C(   773.41), SIMDE_FLOAT32_C(  -212.55), SIMDE_FLOAT32_C(  -356.51), SIMDE_FLOAT32_C(  -440.95),
                         SIMDE_FLOAT32_C(   743.99), SIMDE_FLOAT32_C(  -842.67), SIMDE_FLOAT32_C(  -420.59), SIMDE_FLOAT32_C(  -949.02)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   156.92), SIMDE_FLOAT32_C(  -736.34), SIMDE_FLOAT32_C(   166.92), SIMDE_FLOAT32_C(   300.41),
                         SIMDE_FLOAT32_C(  -295.98), SIMDE_FLOAT32_C(  -702.22), SIMDE_FLOAT32_C(  -740.49), SIMDE_FLOAT32_C(   -80.99),
                         SIMDE_FLOAT32_C(  -785.06), SIMDE_FLOAT32_C(    87.65), SIMDE_FLOAT32_C(  -482.52), SIMDE_FLOAT32_C(  -681.02),
                         SIMDE_FLOAT32_C(   764.25), SIMDE_FLOAT32_C(   305.46), SIMDE_FLOAT32_C(   526.44), SIMDE_FLOAT32_C(   369.20)),
      UINT16_C(49024),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   809.64), SIMDE_FLOAT32_C(  -790.72), SIMDE_FLOAT32_C(   295.53), SIMDE_FLOAT32_C(  -856.33),
                         SIMDE_FLOAT32_C(   237.04), SIMDE_FLOAT32_C(  -607.75), SIMDE_FLOAT32_C(  -732.96), SIMDE_FLOAT32_C(  -497.56),
                         SIMDE_FLOAT32_C(  -918.03), SIMDE_FLOAT32_C(   488.66), SIMDE_FLOAT32_C(  -523.80), SIMDE_FLOAT32_C(  -224.58),
                         SIMDE_FLOAT32_C(   298.04), SIMDE_FLOAT32_C(   606.61), SIMDE_FLOAT32_C(  -852.36), SIMDE_FLOAT32_C(  -314.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   809.64), SIMDE_FLOAT32_C(  -736.34), SIMDE_FLOAT32_C(   295.53), SIMDE_FLOAT32_C(   856.33),
                         SIMDE_FLOAT32_C(   237.04), SIMDE_FLOAT32_C(   607.75), SIMDE_FLOAT32_C(   732.96), SIMDE_FLOAT32_C(   497.56),
                         SIMDE_FLOAT32_C(   918.03), SIMDE_FLOAT32_C(    87.65), SIMDE_FLOAT32_C(  -482.52), SIMDE_FLOAT32_C(  -681.02),
                         SIMDE_FLOAT32_C(   764.25), SIMDE_FLOAT32_C(   305.46), SIMDE_FLOAT32_C(   526.44), SIMDE_FLOAT32_C(   369.20)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -897.44), SIMDE_FLOAT32_C(   113.95), SIMDE_FLOAT32_C(   654.01), SIMDE_FLOAT32_C(   620.23),
                         SIMDE_FLOAT32_C(   623.09), SIMDE_FLOAT32_C(  -407.46), SIMDE_FLOAT32_C(  -763.16), SIMDE_FLOAT32_C(  -768.89),
                         SIMDE_FLOAT32_C(   966.30), SIMDE_FLOAT32_C(   863.50), SIMDE_FLOAT32_C(   709.25), SIMDE_FLOAT32_C(   348.50),
                         SIMDE_FLOAT32_C(  -816.66), SIMDE_FLOAT32_C(  -662.92), SIMDE_FLOAT32_C(   913.50), SIMDE_FLOAT32_C(   301.72)),
      UINT16_C(64661),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -54.30), SIMDE_FLOAT32_C(  -771.33), SIMDE_FLOAT32_C(   -34.80), SIMDE_FLOAT32_C(   -55.97),
                         SIMDE_FLOAT32_C(  -654.29), SIMDE_FLOAT32_C(   768.64), SIMDE_FLOAT32_C(  -409.48), SIMDE_FLOAT32_C(   859.32),
                         SIMDE_FLOAT32_C(  -160.39), SIMDE_FLOAT32_C(  -988.34), SIMDE_FLOAT32_C(  -518.87), SIMDE_FLOAT32_C(  -778.28),
                         SIMDE_FLOAT32_C(   357.12), SIMDE_FLOAT32_C(   449.29), SIMDE_FLOAT32_C(   -46.50), SIMDE_FLOAT32_C(    93.99)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    54.30), SIMDE_FLOAT32_C(   771.33), SIMDE_FLOAT32_C(    34.80), SIMDE_FLOAT32_C(    55.97),
                         SIMDE_FLOAT32_C(   654.29), SIMDE_FLOAT32_C(   768.64), SIMDE_FLOAT32_C(  -763.16), SIMDE_FLOAT32_C(  -768.89),
                         SIMDE_FLOAT32_C(   160.39), SIMDE_FLOAT32_C(   863.50), SIMDE_FLOAT32_C(   709.25), SIMDE_FLOAT32_C(   778.28),
                         SIMDE_FLOAT32_C(  -816.66), SIMDE_FLOAT32_C(   449.29), SIMDE_FLOAT32_C(   913.50), SIMDE_FLOAT32_C(    93.99)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_abs_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_abs_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -140.56), SIMDE_FLOAT64_C( -558.99),
                         SIMDE_FLOAT64_C(  240.08), SIMDE_FLOAT64_C( -481.72),
                         SIMDE_FLOAT64_C(  489.35), SIMDE_FLOAT64_C(  686.76),
                         SIMDE_FLOAT64_C( -206.54), SIMDE_FLOAT64_C(  728.61)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  140.56), SIMDE_FLOAT64_C(  558.99),
                         SIMDE_FLOAT64_C(  240.08), SIMDE_FLOAT64_C(  481.72),
                         SIMDE_FLOAT64_C(  489.35), SIMDE_FLOAT64_C(  686.76),
                         SIMDE_FLOAT64_C(  206.54), SIMDE_FLOAT64_C(  728.61)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  747.74), SIMDE_FLOAT64_C( -874.37),
                         SIMDE_FLOAT64_C(  751.90), SIMDE_FLOAT64_C( -592.77),
                         SIMDE_FLOAT64_C( -708.81), SIMDE_FLOAT64_C(  252.42),
                         SIMDE_FLOAT64_C( -787.46), SIMDE_FLOAT64_C( -882.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  747.74), SIMDE_FLOAT64_C(  874.37),
                         SIMDE_FLOAT64_C(  751.90), SIMDE_FLOAT64_C(  592.77),
                         SIMDE_FLOAT64_C(  708.81), SIMDE_FLOAT64_C(  252.42),
                         SIMDE_FLOAT64_C(  787.46), SIMDE_FLOAT64_C(  882.47)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  388.54), SIMDE_FLOAT64_C( -184.67),
                         SIMDE_FLOAT64_C(  102.38), SIMDE_FLOAT64_C(  833.56),
                         SIMDE_FLOAT64_C( -722.29), SIMDE_FLOAT64_C( -441.84),
                         SIMDE_FLOAT64_C( -821.42), SIMDE_FLOAT64_C( -761.98)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  388.54), SIMDE_FLOAT64_C(  184.67),
                         SIMDE_FLOAT64_C(  102.38), SIMDE_FLOAT64_C(  833.56),
                         SIMDE_FLOAT64_C(  722.29), SIMDE_FLOAT64_C(  441.84),
                         SIMDE_FLOAT64_C(  821.42), SIMDE_FLOAT64_C(  761.98)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   76.28), SIMDE_FLOAT64_C( -319.23),
                         SIMDE_FLOAT64_C(  655.09), SIMDE_FLOAT64_C(  773.21),
                         SIMDE_FLOAT64_C( -928.32), SIMDE_FLOAT64_C(  -25.13),
                         SIMDE_FLOAT64_C( -847.53), SIMDE_FLOAT64_C(  859.40)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   76.28), SIMDE_FLOAT64_C(  319.23),
                         SIMDE_FLOAT64_C(  655.09), SIMDE_FLOAT64_C(  773.21),
                         SIMDE_FLOAT64_C(  928.32), SIMDE_FLOAT64_C(   25.13),
                         SIMDE_FLOAT64_C(  847.53), SIMDE_FLOAT64_C(  859.40)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  -40.31), SIMDE_FLOAT64_C(  866.84),
                         SIMDE_FLOAT64_C( -660.11), SIMDE_FLOAT64_C(  121.17),
                         SIMDE_FLOAT64_C(  988.31), SIMDE_FLOAT64_C( -622.26),
                         SIMDE_FLOAT64_C(  206.00), SIMDE_FLOAT64_C(  520.48)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   40.31), SIMDE_FLOAT64_C(  866.84),
                         SIMDE_FLOAT64_C(  660.11), SIMDE_FLOAT64_C(  121.17),
                         SIMDE_FLOAT64_C(  988.31), SIMDE_FLOAT64_C(  622.26),
                         SIMDE_FLOAT64_C(  206.00), SIMDE_FLOAT64_C(  520.48)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -645.61), SIMDE_FLOAT64_C(  594.77),
                         SIMDE_FLOAT64_C( -171.69), SIMDE_FLOAT64_C(  108.08),
                         SIMDE_FLOAT64_C(   -7.24), SIMDE_FLOAT64_C(  885.82),
                         SIMDE_FLOAT64_C(  296.84), SIMDE_FLOAT64_C( -408.70)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  645.61), SIMDE_FLOAT64_C(  594.77),
                         SIMDE_FLOAT64_C(  171.69), SIMDE_FLOAT64_C(  108.08),
                         SIMDE_FLOAT64_C(    7.24), SIMDE_FLOAT64_C(  885.82),
                         SIMDE_FLOAT64_C(  296.84), SIMDE_FLOAT64_C(  408.70)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -838.06), SIMDE_FLOAT64_C( -901.25),
                         SIMDE_FLOAT64_C(  -89.58), SIMDE_FLOAT64_C(  539.88),
                         SIMDE_FLOAT64_C(   88.35), SIMDE_FLOAT64_C(  773.77),
                         SIMDE_FLOAT64_C( -729.20), SIMDE_FLOAT64_C( -254.72)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  838.06), SIMDE_FLOAT64_C(  901.25),
                         SIMDE_FLOAT64_C(   89.58), SIMDE_FLOAT64_C(  539.88),
                         SIMDE_FLOAT64_C(   88.35), SIMDE_FLOAT64_C(  773.77),
                         SIMDE_FLOAT64_C(  729.20), SIMDE_FLOAT64_C(  254.72)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  530.77), SIMDE_FLOAT64_C(  966.97),
                         SIMDE_FLOAT64_C(  -63.51), SIMDE_FLOAT64_C(  360.07),
                         SIMDE_FLOAT64_C( -846.61), SIMDE_FLOAT64_C( -749.79),
                         SIMDE_FLOAT64_C(  510.77), SIMDE_FLOAT64_C( -104.12)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  530.77), SIMDE_FLOAT64_C(  966.97),
                         SIMDE_FLOAT64_C(   63.51), SIMDE_FLOAT64_C(  360.07),
                         SIMDE_FLOAT64_C(  846.61), SIMDE_FLOAT64_C(  749.79),
                         SIMDE_FLOAT64_C(  510.77), SIMDE_FLOAT64_C(  104.12)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_abs_pd(test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_abs_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -874.37), SIMDE_FLOAT64_C( -592.77),
                         SIMDE_FLOAT64_C(  252.42), SIMDE_FLOAT64_C( -882.47),
                         SIMDE_FLOAT64_C( -558.99), SIMDE_FLOAT64_C( -481.72),
                         SIMDE_FLOAT64_C(  686.76), SIMDE_FLOAT64_C(  728.61)),
      UINT8_C( 67),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  747.74), SIMDE_FLOAT64_C(  751.90),
                         SIMDE_FLOAT64_C( -708.81), SIMDE_FLOAT64_C( -787.46),
                         SIMDE_FLOAT64_C( -140.56), SIMDE_FLOAT64_C(  240.08),
                         SIMDE_FLOAT64_C(  489.35), SIMDE_FLOAT64_C( -206.54)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -874.37), SIMDE_FLOAT64_C(  751.90),
                         SIMDE_FLOAT64_C(  252.42), SIMDE_FLOAT64_C( -882.47),
                         SIMDE_FLOAT64_C( -558.99), SIMDE_FLOAT64_C( -481.72),
                         SIMDE_FLOAT64_C(  489.35), SIMDE_FLOAT64_C(  206.54)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   76.28), SIMDE_FLOAT64_C(  655.09),
                         SIMDE_FLOAT64_C( -928.32), SIMDE_FLOAT64_C( -847.53),
                         SIMDE_FLOAT64_C(  388.54), SIMDE_FLOAT64_C(  102.38),
                         SIMDE_FLOAT64_C( -722.29), SIMDE_FLOAT64_C( -821.42)),
      UINT8_C(153),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  520.48), SIMDE_FLOAT64_C( -319.23),
                         SIMDE_FLOAT64_C(  773.21), SIMDE_FLOAT64_C(  -25.13),
                         SIMDE_FLOAT64_C(  859.40), SIMDE_FLOAT64_C( -184.67),
                         SIMDE_FLOAT64_C(  833.56), SIMDE_FLOAT64_C( -441.84)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  520.48), SIMDE_FLOAT64_C(  655.09),
                         SIMDE_FLOAT64_C( -928.32), SIMDE_FLOAT64_C(   25.13),
                         SIMDE_FLOAT64_C(  859.40), SIMDE_FLOAT64_C(  102.38),
                         SIMDE_FLOAT64_C( -722.29), SIMDE_FLOAT64_C(  441.84)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -254.72), SIMDE_FLOAT64_C(  594.77),
                         SIMDE_FLOAT64_C(  108.08), SIMDE_FLOAT64_C(  885.82),
                         SIMDE_FLOAT64_C( -408.70), SIMDE_FLOAT64_C(  866.84),
                         SIMDE_FLOAT64_C(  121.17), SIMDE_FLOAT64_C( -622.26)),
      UINT8_C( 41),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -729.20), SIMDE_FLOAT64_C( -645.61),
                         SIMDE_FLOAT64_C( -171.69), SIMDE_FLOAT64_C(   -7.24),
                         SIMDE_FLOAT64_C(  296.84), SIMDE_FLOAT64_C(  -40.31),
                         SIMDE_FLOAT64_C( -660.11), SIMDE_FLOAT64_C(  988.31)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -254.72), SIMDE_FLOAT64_C(  594.77),
                         SIMDE_FLOAT64_C(  171.69), SIMDE_FLOAT64_C(  885.82),
                         SIMDE_FLOAT64_C(  296.84), SIMDE_FLOAT64_C(  866.84),
                         SIMDE_FLOAT64_C(  121.17), SIMDE_FLOAT64_C(  988.31)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -617.33), SIMDE_FLOAT64_C(  530.77),
                         SIMDE_FLOAT64_C(  -63.51), SIMDE_FLOAT64_C( -846.61),
                         SIMDE_FLOAT64_C(  510.77), SIMDE_FLOAT64_C( -838.06),
                         SIMDE_FLOAT64_C(  -89.58), SIMDE_FLOAT64_C(   88.35)),
      UINT8_C(208),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -506.16), SIMDE_FLOAT64_C( -947.60),
                         SIMDE_FLOAT64_C(  966.97), SIMDE_FLOAT64_C(  360.07),
                         SIMDE_FLOAT64_C( -749.79), SIMDE_FLOAT64_C( -104.12),
                         SIMDE_FLOAT64_C( -901.25), SIMDE_FLOAT64_C(  539.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  506.16), SIMDE_FLOAT64_C(  947.60),
                         SIMDE_FLOAT64_C(  -63.51), SIMDE_FLOAT64_C(  360.07),
                         SIMDE_FLOAT64_C(  510.77), SIMDE_FLOAT64_C( -838.06),
                         SIMDE_FLOAT64_C(  -89.58), SIMDE_FLOAT64_C(   88.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  943.89), SIMDE_FLOAT64_C(  695.54),
                         SIMDE_FLOAT64_C( -377.80), SIMDE_FLOAT64_C(  778.00),
                         SIMDE_FLOAT64_C( -489.06), SIMDE_FLOAT64_C( -749.41),
                         SIMDE_FLOAT64_C( -423.37), SIMDE_FLOAT64_C( -118.88)),
      UINT8_C( 52),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -212.57), SIMDE_FLOAT64_C(  651.63),
                         SIMDE_FLOAT64_C( -814.45), SIMDE_FLOAT64_C(  640.68),
                         SIMDE_FLOAT64_C(  377.67), SIMDE_FLOAT64_C(  933.74),
                         SIMDE_FLOAT64_C(  193.12), SIMDE_FLOAT64_C( -194.06)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  943.89), SIMDE_FLOAT64_C(  695.54),
                         SIMDE_FLOAT64_C(  814.45), SIMDE_FLOAT64_C(  640.68),
                         SIMDE_FLOAT64_C( -489.06), SIMDE_FLOAT64_C(  933.74),
                         SIMDE_FLOAT64_C( -423.37), SIMDE_FLOAT64_C( -118.88)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -613.16), SIMDE_FLOAT64_C(  -56.94),
                         SIMDE_FLOAT64_C(  525.11), SIMDE_FLOAT64_C(  814.48),
                         SIMDE_FLOAT64_C( -481.76), SIMDE_FLOAT64_C( -375.20),
                         SIMDE_FLOAT64_C(  199.14), SIMDE_FLOAT64_C(  833.96)),
      UINT8_C(108),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  536.10), SIMDE_FLOAT64_C( -393.36),
                         SIMDE_FLOAT64_C(  670.22), SIMDE_FLOAT64_C(  299.00),
                         SIMDE_FLOAT64_C(  676.51), SIMDE_FLOAT64_C(  528.75),
                         SIMDE_FLOAT64_C(  146.55), SIMDE_FLOAT64_C( -505.05)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -613.16), SIMDE_FLOAT64_C(  393.36),
                         SIMDE_FLOAT64_C(  670.22), SIMDE_FLOAT64_C(  814.48),
                         SIMDE_FLOAT64_C(  676.51), SIMDE_FLOAT64_C(  528.75),
                         SIMDE_FLOAT64_C(  199.14), SIMDE_FLOAT64_C(  833.96)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   -8.58), SIMDE_FLOAT64_C(  714.91),
                         SIMDE_FLOAT64_C( -520.52), SIMDE_FLOAT64_C( -546.29),
                         SIMDE_FLOAT64_C( -104.88), SIMDE_FLOAT64_C( -744.23),
                         SIMDE_FLOAT64_C(  495.73), SIMDE_FLOAT64_C(  262.00)),
      UINT8_C(147),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  550.36), SIMDE_FLOAT64_C(  374.16),
                         SIMDE_FLOAT64_C( -355.51), SIMDE_FLOAT64_C(  -13.73),
                         SIMDE_FLOAT64_C( -787.44), SIMDE_FLOAT64_C(  979.47),
                         SIMDE_FLOAT64_C(  836.15), SIMDE_FLOAT64_C( -301.39)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  550.36), SIMDE_FLOAT64_C(  714.91),
                         SIMDE_FLOAT64_C( -520.52), SIMDE_FLOAT64_C(   13.73),
                         SIMDE_FLOAT64_C( -104.88), SIMDE_FLOAT64_C( -744.23),
                         SIMDE_FLOAT64_C(  836.15), SIMDE_FLOAT64_C(  301.39)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  515.30), SIMDE_FLOAT64_C(  896.28),
                         SIMDE_FLOAT64_C(  660.35), SIMDE_FLOAT64_C(  493.79),
                         SIMDE_FLOAT64_C( -941.83), SIMDE_FLOAT64_C(  535.05),
                         SIMDE_FLOAT64_C( -963.94), SIMDE_FLOAT64_C(  143.93)),
      UINT8_C( 75),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  520.16), SIMDE_FLOAT64_C(   61.51),
                         SIMDE_FLOAT64_C( -643.69), SIMDE_FLOAT64_C(  -16.59),
                         SIMDE_FLOAT64_C(  -29.44), SIMDE_FLOAT64_C( -567.95),
                         SIMDE_FLOAT64_C(   43.85), SIMDE_FLOAT64_C(  235.87)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  515.30), SIMDE_FLOAT64_C(   61.51),
                         SIMDE_FLOAT64_C(  660.35), SIMDE_FLOAT64_C(  493.79),
                         SIMDE_FLOAT64_C(   29.44), SIMDE_FLOAT64_C(  535.05),
                         SIMDE_FLOAT64_C(   43.85), SIMDE_FLOAT64_C(  235.87)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_abs_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_add_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -620437579), INT32_C( -723756998), INT32_C( 2089862132), INT32_C(  928615324),
                            INT32_C( 1352582940), INT32_C( 1412449807), INT32_C( -180438223), INT32_C( 1987031180),
                            INT32_C( -515955035), INT32_C( 1724776466), INT32_C(  534751449), INT32_C(  -23608579),
                            INT32_C(-1857730421), INT32_C(-2108373984), INT32_C(-2047745621), INT32_C(-1616522706)),
      simde_mm512_set_epi32(INT32_C(-1828719907), INT32_C(-1665857743), INT32_C(  633675792), INT32_C(  -87619707),
                            INT32_C(  206282352), INT32_C(-1301173823), INT32_C(-1377624366), INT32_C(-1722858856),
                            INT32_C(-1747195587), INT32_C(-1421499089), INT32_C(-1466275595), INT32_C( 1005942821),
                            INT32_C(  653597122), INT32_C( 1511265780), INT32_C(-1057049813), INT32_C(  988582315)),
      simde_mm512_set_epi32(INT32_C( 1845809810), INT32_C( 1905352555), INT32_C(-1571429372), INT32_C(  840995617),
                            INT32_C( 1558865292), INT32_C(  111275984), INT32_C(-1558062589), INT32_C(  264172324),
                            INT32_C( 2031816674), INT32_C(  303277377), INT32_C( -931524146), INT32_C(  982334242),
                            INT32_C(-1204133299), INT32_C( -597108204), INT32_C( 1190171862), INT32_C( -627940391)) },
    { simde_mm512_set_epi32(INT32_C(-1937590036), INT32_C( -332801366), INT32_C( -471964647), INT32_C(-1821490876),
                            INT32_C( -730720612), INT32_C( -212155491), INT32_C(  557730573), INT32_C( 1419303203),
                            INT32_C( -649726768), INT32_C(-1343949182), INT32_C(  543388357), INT32_C( 2016683828),
                            INT32_C(-1787609785), INT32_C( 1170714264), INT32_C( 1838698095), INT32_C( -610530733)),
      simde_mm512_set_epi32(INT32_C(  -78760490), INT32_C(-1762700896), INT32_C(  743846170), INT32_C( 2011763261),
                            INT32_C( -319606822), INT32_C( -887822109), INT32_C(  843030788), INT32_C( 1100063012),
                            INT32_C( 1306201477), INT32_C( -328092402), INT32_C(  760791953), INT32_C( 1607075963),
                            INT32_C(-2029269339), INT32_C(-1246855319), INT32_C(  373846974), INT32_C(-2039859469)),
      simde_mm512_set_epi32(INT32_C(-2016350526), INT32_C(-2095502262), INT32_C(  271881523), INT32_C(  190272385),
                            INT32_C(-1050327434), INT32_C(-1099977600), INT32_C( 1400761361), INT32_C(-1775601081),
                            INT32_C(  656474709), INT32_C(-1672041584), INT32_C( 1304180310), INT32_C( -671207505),
                            INT32_C(  478088172), INT32_C(  -76141055), INT32_C(-2082422227), INT32_C( 1644577094)) },
    { simde_mm512_set_epi32(INT32_C(   57776624), INT32_C( 1788183002), INT32_C(-1974433832), INT32_C(-1794133389),
                            INT32_C(-1069162986), INT32_C(-1427272558), INT32_C(-1253805688), INT32_C( -660872028),
                            INT32_C(-1129884149), INT32_C(  173413635), INT32_C( 2028757327), INT32_C( 1565906228),
                            INT32_C(-2137992833), INT32_C(-1203890748), INT32_C(  706474135), INT32_C(-1778193688)),
      simde_mm512_set_epi32(INT32_C( -515592024), INT32_C( -146353312), INT32_C(  242268084), INT32_C( -988968038),
                            INT32_C(-2038559805), INT32_C(   54811004), INT32_C( 1173402348), INT32_C(  422621509),
                            INT32_C(  566804477), INT32_C( 1847170418), INT32_C( -922088165), INT32_C( -324583669),
                            INT32_C( 2069465832), INT32_C(-1437889074), INT32_C(  238343196), INT32_C( 1745183762)),
      simde_mm512_set_epi32(INT32_C( -457815400), INT32_C( 1641829690), INT32_C(-1732165748), INT32_C( 1511865869),
                            INT32_C( 1187244505), INT32_C(-1372461554), INT32_C(  -80403340), INT32_C( -238250519),
                            INT32_C( -563079672), INT32_C( 2020584053), INT32_C( 1106669162), INT32_C( 1241322559),
                            INT32_C(  -68527001), INT32_C( 1653187474), INT32_C(  944817331), INT32_C(  -33009926)) },
    { simde_mm512_set_epi32(INT32_C( 1004166983), INT32_C(-2037237484), INT32_C(  975969300), INT32_C(  221374965),
                            INT32_C( -878025387), INT32_C( 1166912453), INT32_C( 1981901973), INT32_C(  976355918),
                            INT32_C( 1384872269), INT32_C(  -15473064), INT32_C( -289026084), INT32_C( 1652811113),
                            INT32_C( 2056881111), INT32_C( 1621101202), INT32_C( 1517253606), INT32_C( 1625264348)),
      simde_mm512_set_epi32(INT32_C(-1184683174), INT32_C(  619504240), INT32_C(-1150416788), INT32_C(-1085597388),
                            INT32_C( 1766693029), INT32_C( 1152891465), INT32_C(-2035468310), INT32_C( 1319983622),
                            INT32_C(-1615386348), INT32_C(  261012444), INT32_C(  223560144), INT32_C(  331591481),
                            INT32_C(  -33325842), INT32_C( -398184636), INT32_C( 1301281245), INT32_C(-1280622867)),
      simde_mm512_set_epi32(INT32_C( -180516191), INT32_C(-1417733244), INT32_C( -174447488), INT32_C( -864222423),
                            INT32_C(  888667642), INT32_C(-1975163378), INT32_C(  -53566337), INT32_C(-1998627756),
                            INT32_C( -230514079), INT32_C(  245539380), INT32_C(  -65465940), INT32_C( 1984402594),
                            INT32_C( 2023555269), INT32_C( 1222916566), INT32_C(-1476432445), INT32_C(  344641481)) },
    { simde_mm512_set_epi32(INT32_C(  -99917228), INT32_C(  209251690), INT32_C(  108789393), INT32_C(-1056574639),
                            INT32_C(  178125384), INT32_C( 1688380363), INT32_C( -910324723), INT32_C( 1731780669),
                            INT32_C( 1989188149), INT32_C(  999202624), INT32_C(  198474190), INT32_C( 1270934715),
                            INT32_C( -715363116), INT32_C( 1839116730), INT32_C(  -53141198), INT32_C( -590771199)),
      simde_mm512_set_epi32(INT32_C(  -77041932), INT32_C(  927171423), INT32_C( -178390373), INT32_C(-1958045464),
                            INT32_C( 1172980373), INT32_C(-1805733723), INT32_C( -859792119), INT32_C( 1334306095),
                            INT32_C(-1717931855), INT32_C( -244574814), INT32_C( 1638507066), INT32_C( -551865858),
                            INT32_C( -152534880), INT32_C(-1198583052), INT32_C( -869344314), INT32_C(-1920653371)),
      simde_mm512_set_epi32(INT32_C( -176959160), INT32_C( 1136423113), INT32_C(  -69600980), INT32_C( 1280347193),
                            INT32_C( 1351105757), INT32_C( -117353360), INT32_C(-1770116842), INT32_C(-1228880532),
                            INT32_C(  271256294), INT32_C(  754627810), INT32_C( 1836981256), INT32_C(  719068857),
                            INT32_C( -867897996), INT32_C(  640533678), INT32_C( -922485512), INT32_C( 1783542726)) },
    { simde_mm512_set_epi32(INT32_C(-1801907952), INT32_C(  126986794), INT32_C( 2099256067), INT32_C(  915807861),
                            INT32_C( -313053214), INT32_C( 1205934161), INT32_C(   25151882), INT32_C( -823105542),
                            INT32_C(-1390117486), INT32_C(  385956806), INT32_C( 1986964939), INT32_C(   56415632),
                            INT32_C(-1186749471), INT32_C( 1048153466), INT32_C( 1387660497), INT32_C( 2108528480)),
      simde_mm512_set_epi32(INT32_C(-1572695910), INT32_C(  592114445), INT32_C(  -85836842), INT32_C(-1721616170),
                            INT32_C(  222979620), INT32_C( -376092888), INT32_C(  405576062), INT32_C(  902804532),
                            INT32_C( 2135373993), INT32_C( 1677980182), INT32_C(-1127993042), INT32_C( -771911734),
                            INT32_C(  -41719457), INT32_C(-1772059837), INT32_C(-1722167164), INT32_C(  812561930)),
      simde_mm512_set_epi32(INT32_C(  920363434), INT32_C(  719101239), INT32_C( 2013419225), INT32_C( -805808309),
                            INT32_C(  -90073594), INT32_C(  829841273), INT32_C(  430727944), INT32_C(   79698990),
                            INT32_C(  745256507), INT32_C( 2063936988), INT32_C(  858971897), INT32_C( -715496102),
                            INT32_C(-1228468928), INT32_C( -723906371), INT32_C( -334506667), INT32_C(-1373876886)) },
    { simde_mm512_set_epi32(INT32_C(  -87552733), INT32_C( -497247153), INT32_C( 1468487386), INT32_C( 1003068863),
                            INT32_C( 1747861143), INT32_C( -646367560), INT32_C( -755743025), INT32_C(  683078293),
                            INT32_C( 1090643160), INT32_C( 1972147188), INT32_C(-2077095609), INT32_C(  731411049),
                            INT32_C( -484220816), INT32_C( -672982839), INT32_C(  -64061145), INT32_C(-1731914804)),
      simde_mm512_set_epi32(INT32_C( 1287212577), INT32_C(  799931806), INT32_C(  620076827), INT32_C(  806976960),
                            INT32_C(-1787684636), INT32_C(  -68422538), INT32_C(  591651588), INT32_C( 2136172897),
                            INT32_C(  538232126), INT32_C( 1910632119), INT32_C(  280273419), INT32_C( 1179224181),
                            INT32_C( -178809858), INT32_C(-1741617588), INT32_C(-2045147428), INT32_C( 1639562384)),
      simde_mm512_set_epi32(INT32_C( 1199659844), INT32_C(  302684653), INT32_C( 2088564213), INT32_C( 1810045823),
                            INT32_C(  -39823493), INT32_C( -714790098), INT32_C( -164091437), INT32_C(-1475716106),
                            INT32_C( 1628875286), INT32_C( -412187989), INT32_C(-1796822190), INT32_C( 1910635230),
                            INT32_C( -663030674), INT32_C( 1880366869), INT32_C(-2109208573), INT32_C(  -92352420)) },
    { simde_mm512_set_epi32(INT32_C(  446376492), INT32_C( 1910481415), INT32_C(  625332216), INT32_C(  541212187),
                            INT32_C(  298536203), INT32_C(-1502376568), INT32_C( -699613431), INT32_C( -125340207),
                            INT32_C( -849753035), INT32_C( -538074567), INT32_C( 1356067945), INT32_C( 1088886620),
                            INT32_C(-1643178888), INT32_C(  855336737), INT32_C( 1638140255), INT32_C(-2120465324)),
      simde_mm512_set_epi32(INT32_C( 1091209248), INT32_C(-1950373665), INT32_C(-2108427702), INT32_C(  969460626),
                            INT32_C(  438255242), INT32_C( -807569468), INT32_C( -746566484), INT32_C( 1685713375),
                            INT32_C(   93210471), INT32_C( 1249365881), INT32_C(  945726578), INT32_C( 1471824111),
                            INT32_C( 1944720672), INT32_C(-1780061567), INT32_C( 1132193802), INT32_C(  151873348)),
      simde_mm512_set_epi32(INT32_C( 1537585740), INT32_C(  -39892250), INT32_C(-1483095486), INT32_C( 1510672813),
                            INT32_C(  736791445), INT32_C( 1985021260), INT32_C(-1446179915), INT32_C( 1560373168),
                            INT32_C( -756542564), INT32_C(  711291314), INT32_C(-1993172773), INT32_C(-1734256565),
                            INT32_C(  301541784), INT32_C( -924724830), INT32_C(-1524633239), INT32_C(-1968591976)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_add_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_add_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-1624161104), INT32_C( 1349327659), INT32_C( -773590663), INT32_C(  202594565),
                            INT32_C( -776802767), INT32_C( -399771521), INT32_C(-1587123399), INT32_C( -957492424),
                            INT32_C(-1115495751), INT32_C(  873816413), INT32_C(-1422024337), INT32_C( 1239250518),
                            INT32_C( 1265405916), INT32_C( 1763960778), INT32_C(-2025442731), INT32_C( 1782061668)),
      UINT16_C(58332),
      simde_mm512_set_epi32(INT32_C( 1896250898), INT32_C(-1975121973), INT32_C(-1813333619), INT32_C(-1686659774),
                            INT32_C( 1969336918), INT32_C( 1790471064), INT32_C( 1431740603), INT32_C( -283912469),
                            INT32_C( 2131478183), INT32_C( -840826493), INT32_C(  380092159), INT32_C(-1927739478),
                            INT32_C( -487729180), INT32_C( 1461889227), INT32_C( -379836768), INT32_C( 1202522762)),
      simde_mm512_set_epi32(INT32_C( -753874778), INT32_C(  918866519), INT32_C(-1585600432), INT32_C(  879905937),
                            INT32_C( -949239113), INT32_C(  506766584), INT32_C( -440591062), INT32_C(-1279590464),
                            INT32_C( 1610700476), INT32_C(   23515362), INT32_C(  -68571660), INT32_C(-1351358395),
                            INT32_C(-1929546098), INT32_C( 1178622777), INT32_C(-1978880777), INT32_C( -838531659)),
      simde_mm512_set_epi32(INT32_C( 1142376120), INT32_C(-1056255454), INT32_C(  896033245), INT32_C(  202594565),
                            INT32_C( -776802767), INT32_C( -399771521), INT32_C(  991149541), INT32_C(-1563502933),
                            INT32_C( -552788637), INT32_C( -817311131), INT32_C(-1422024337), INT32_C( 1015869423),
                            INT32_C( 1877692018), INT32_C(-1654455292), INT32_C(-2025442731), INT32_C( 1782061668)) },
    { simde_mm512_set_epi32(INT32_C( 1302762176), INT32_C( -882614115), INT32_C( 1404982069), INT32_C( 2114303183),
                            INT32_C(-1232865397), INT32_C(  471502333), INT32_C( 1510887307), INT32_C( 1591970418),
                            INT32_C(-1919536055), INT32_C( 1916415088), INT32_C( -597668621), INT32_C(-1146237202),
                            INT32_C(  397648016), INT32_C(  390728591), INT32_C( -391091736), INT32_C(-2096352492)),
      UINT16_C(24508),
      simde_mm512_set_epi32(INT32_C( -329978949), INT32_C(  917197994), INT32_C(  183082258), INT32_C(  406981385),
                            INT32_C( 1262580279), INT32_C(-2107001396), INT32_C(-1782137064), INT32_C( 1969679701),
                            INT32_C(   72991140), INT32_C(-1531863718), INT32_C(-1294933909), INT32_C(-1793493099),
                            INT32_C(-1290212500), INT32_C(   38707735), INT32_C(  670120881), INT32_C(  729166183)),
      simde_mm512_set_epi32(INT32_C( 1326302935), INT32_C( 1436338657), INT32_C(-1721025649), INT32_C( 1135570890),
                            INT32_C(  322795023), INT32_C(  585942662), INT32_C( 1660039591), INT32_C(-1880921663),
                            INT32_C(  917919791), INT32_C(-1697677003), INT32_C(  355955540), INT32_C(  -89719066),
                            INT32_C(-1599109389), INT32_C(  500376477), INT32_C(  894768365), INT32_C(  678417142)),
      simde_mm512_set_epi32(INT32_C( 1302762176), INT32_C(-1941430645), INT32_C( 1404982069), INT32_C( 1542552275),
                            INT32_C( 1585375302), INT32_C(-1521058734), INT32_C( -122097473), INT32_C(   88758038),
                            INT32_C(  990910931), INT32_C( 1916415088), INT32_C( -938978369), INT32_C(-1883212165),
                            INT32_C( 1405645407), INT32_C(  539084212), INT32_C( -391091736), INT32_C(-2096352492)) },
    { simde_mm512_set_epi32(INT32_C( 1174860998), INT32_C( -880427450), INT32_C(-1858758913), INT32_C(  -96862036),
                            INT32_C( -796551598), INT32_C( 1745947202), INT32_C(-1440860851), INT32_C( -316371577),
                            INT32_C( 2119959954), INT32_C(-1234010527), INT32_C( -607821866), INT32_C( -207403706),
                            INT32_C( -488974092), INT32_C( -530330398), INT32_C(-1280521827), INT32_C(  -30002805)),
      UINT16_C(64556),
      simde_mm512_set_epi32(INT32_C( 1300630223), INT32_C( -416275571), INT32_C(  274879402), INT32_C(-1285235693),
                            INT32_C( 1642409948), INT32_C(-1885968606), INT32_C( 1050217372), INT32_C(-1544064335),
                            INT32_C( -924582063), INT32_C(-1959255243), INT32_C(-1209083477), INT32_C(  479380588),
                            INT32_C( -559284942), INT32_C(-1289663853), INT32_C( 1463163718), INT32_C(-1596878601)),
      simde_mm512_set_epi32(INT32_C(-1343344437), INT32_C( -273565731), INT32_C(-1263791162), INT32_C(  562202801),
                            INT32_C( 1392714893), INT32_C(-1388744472), INT32_C(  190920597), INT32_C( 1163820219),
                            INT32_C(  347111490), INT32_C(-1175484604), INT32_C(-1452068318), INT32_C( 1960327870),
                            INT32_C( 1869872049), INT32_C(-1282001950), INT32_C( 1598992534), INT32_C(-1956727465)),
      simde_mm512_set_epi32(INT32_C(  -42714214), INT32_C( -689841302), INT32_C( -988911760), INT32_C( -723032892),
                            INT32_C(-1259842455), INT32_C( 1020254218), INT32_C(-1440860851), INT32_C( -316371577),
                            INT32_C( 2119959954), INT32_C(-1234010527), INT32_C( 1633815501), INT32_C( -207403706),
                            INT32_C( 1310587107), INT32_C( 1723301493), INT32_C(-1280521827), INT32_C(  -30002805)) },
    { simde_mm512_set_epi32(INT32_C( -715378322), INT32_C(   38558021), INT32_C( -821295633), INT32_C( 1165546518),
                            INT32_C( -691638364), INT32_C( -465098284), INT32_C( -323391111), INT32_C(-1739139035),
                            INT32_C(-1697445904), INT32_C( 1151412813), INT32_C( 1052265884), INT32_C( -111443496),
                            INT32_C(  574474718), INT32_C(  502491567), INT32_C(-1280394401), INT32_C(-1982568444)),
      UINT16_C(17528),
      simde_mm512_set_epi32(INT32_C(  272471880), INT32_C( -172561500), INT32_C(-1081837229), INT32_C( -100041742),
                            INT32_C(-1115160509), INT32_C(-1125554136), INT32_C( 1667472997), INT32_C( 1125907170),
                            INT32_C(  695609232), INT32_C( 1254240200), INT32_C(-1668916282), INT32_C( -177946147),
                            INT32_C(-1812879792), INT32_C(  167253498), INT32_C( -546762102), INT32_C(-1487511960)),
      simde_mm512_set_epi32(INT32_C( -706618476), INT32_C(  243209514), INT32_C(-1723690853), INT32_C(-2137764059),
                            INT32_C(-1694142803), INT32_C(  517013494), INT32_C(  111610753), INT32_C( 1124784718),
                            INT32_C(  477672094), INT32_C( 2093152650), INT32_C(-2068271518), INT32_C(-1818643484),
                            INT32_C( -581921883), INT32_C(-1541399578), INT32_C(-1753866030), INT32_C(  318536534)),
      simde_mm512_set_epi32(INT32_C( -715378322), INT32_C(   70648014), INT32_C( -821295633), INT32_C( 1165546518),
                            INT32_C( -691638364), INT32_C( -608540642), INT32_C( -323391111), INT32_C(-1739139035),
                            INT32_C(-1697445904), INT32_C( -947574446), INT32_C(  557779496), INT32_C(-1996589631),
                            INT32_C( 1900165621), INT32_C(  502491567), INT32_C(-1280394401), INT32_C(-1982568444)) },
    { simde_mm512_set_epi32(INT32_C( 1092108526), INT32_C( 1226819840), INT32_C(-2023739275), INT32_C( -272774278),
                            INT32_C( -821948118), INT32_C(  395521731), INT32_C( 1806233641), INT32_C( -835262616),
                            INT32_C(   15700951), INT32_C( 2044963604), INT32_C( -509954286), INT32_C( 1986904207),
                            INT32_C( -207392737), INT32_C( 1684880947), INT32_C( 1593254599), INT32_C( 1691267542)),
      UINT16_C( 9866),
      simde_mm512_set_epi32(INT32_C( 1989266232), INT32_C(-1451052935), INT32_C(-1775580706), INT32_C(-1342830971),
                            INT32_C(  533482859), INT32_C( 1701385587), INT32_C(-1013134859), INT32_C( -295709603),
                            INT32_C(  765098610), INT32_C(-1985135009), INT32_C( -398166926), INT32_C( 1936468336),
                            INT32_C( -926051977), INT32_C( 1694570192), INT32_C(  592057582), INT32_C( 1192408406)),
      simde_mm512_set_epi32(INT32_C(-1471768201), INT32_C( -365084156), INT32_C(  749827642), INT32_C(  456081081),
                            INT32_C( 1611322321), INT32_C( 1918615230), INT32_C( 1260889506), INT32_C(  220186540),
                            INT32_C( -704932367), INT32_C( 2036303616), INT32_C( 1130346108), INT32_C(   85133926),
                            INT32_C( -930062623), INT32_C( -341537186), INT32_C(-1350238769), INT32_C(  258510629)),
      simde_mm512_set_epi32(INT32_C( 1092108526), INT32_C( 1226819840), INT32_C(-1025753064), INT32_C( -272774278),
                            INT32_C( -821948118), INT32_C( -674966479), INT32_C(  247754647), INT32_C( -835262616),
                            INT32_C(   60166243), INT32_C( 2044963604), INT32_C( -509954286), INT32_C( 1986904207),
                            INT32_C(-1856114600), INT32_C( 1684880947), INT32_C( -758181187), INT32_C( 1691267542)) },
    { simde_mm512_set_epi32(INT32_C( 1067430684), INT32_C( -902480752), INT32_C( -725414654), INT32_C( -682173663),
                            INT32_C( -824999108), INT32_C(  610503628), INT32_C(   27839428), INT32_C( -509900833),
                            INT32_C(  296614283), INT32_C( 1441936352), INT32_C(  391930371), INT32_C(-1474743751),
                            INT32_C(-1390424455), INT32_C( -831282815), INT32_C(-1747591713), INT32_C(-1966661099)),
      UINT16_C(33042),
      simde_mm512_set_epi32(INT32_C( -454711583), INT32_C(-2145538894), INT32_C( -698606149), INT32_C( 1860960361),
                            INT32_C(  209054104), INT32_C(-1198084158), INT32_C(  693203255), INT32_C(-1710867352),
                            INT32_C( 1800761221), INT32_C(  -67128604), INT32_C( 1518807460), INT32_C(  642346447),
                            INT32_C(  960921211), INT32_C( -966186708), INT32_C( 1502320358), INT32_C( 1019101619)),
      simde_mm512_set_epi32(INT32_C(-1260881299), INT32_C(-1489638506), INT32_C(-1909268290), INT32_C( -420567952),
                            INT32_C(-1777903705), INT32_C(  -92439215), INT32_C(   30169805), INT32_C( 1287984262),
                            INT32_C(-1867997855), INT32_C( 2029047155), INT32_C( 1974126108), INT32_C(   81477772),
                            INT32_C( -284879646), INT32_C(  423259845), INT32_C( 1482936149), INT32_C(  775124224)),
      simde_mm512_set_epi32(INT32_C(-1715592882), INT32_C( -902480752), INT32_C( -725414654), INT32_C( -682173663),
                            INT32_C( -824999108), INT32_C(  610503628), INT32_C(   27839428), INT32_C( -422883090),
                            INT32_C(  296614283), INT32_C( 1441936352), INT32_C(  391930371), INT32_C(  723824219),
                            INT32_C(-1390424455), INT32_C( -831282815), INT32_C(-1309710789), INT32_C(-1966661099)) },
    { simde_mm512_set_epi32(INT32_C(  536551420), INT32_C(-1663334705), INT32_C( -680475343), INT32_C( 1911508818),
                            INT32_C(  973743946), INT32_C(-1445524834), INT32_C( 1776429422), INT32_C(-1921552773),
                            INT32_C(-1475144099), INT32_C(  333730656), INT32_C( 1861208887), INT32_C(-1550831547),
                            INT32_C(  913565672), INT32_C(-1482150044), INT32_C( -134808620), INT32_C( 1260966396)),
      UINT16_C(37765),
      simde_mm512_set_epi32(INT32_C( -311057844), INT32_C( 1759678212), INT32_C(-1836488430), INT32_C( -995678270),
                            INT32_C( 1712282900), INT32_C(-1865239182), INT32_C( -356482670), INT32_C( 1582561214),
                            INT32_C( -629668938), INT32_C( 1097364274), INT32_C( -806555820), INT32_C( 1803199967),
                            INT32_C(  927979811), INT32_C( -175347025), INT32_C( 1236121226), INT32_C(  844026133)),
      simde_mm512_set_epi32(INT32_C(-2053315551), INT32_C(-1349948208), INT32_C(-1018238026), INT32_C(  452561622),
                            INT32_C(-1065347327), INT32_C( 1611585490), INT32_C(  564714625), INT32_C(  270704493),
                            INT32_C( 1746822574), INT32_C( 2111586550), INT32_C(-1423101985), INT32_C( 1093125919),
                            INT32_C(  744070160), INT32_C(-1075582637), INT32_C( -862803202), INT32_C(-1332981287)),
      simde_mm512_set_epi32(INT32_C( 1930593901), INT32_C(-1663334705), INT32_C( -680475343), INT32_C( -543116648),
                            INT32_C(  973743946), INT32_C(-1445524834), INT32_C(  208231955), INT32_C( 1853265707),
                            INT32_C( 1117153636), INT32_C(  333730656), INT32_C( 1861208887), INT32_C(-1550831547),
                            INT32_C(  913565672), INT32_C(-1250929662), INT32_C( -134808620), INT32_C( -488955154)) },
    { simde_mm512_set_epi32(INT32_C(  232476539), INT32_C(-1618256937), INT32_C( -921374205), INT32_C(-1054970097),
                            INT32_C(-1076121842), INT32_C(-1798454285), INT32_C(  999650858), INT32_C( -684849425),
                            INT32_C( -175946824), INT32_C( 1263680378), INT32_C( 1967511219), INT32_C( 1203309886),
                            INT32_C( -614984044), INT32_C( -768724171), INT32_C(  394289516), INT32_C( 1184804742)),
      UINT16_C(58136),
      simde_mm512_set_epi32(INT32_C( 1205787189), INT32_C(-1766786942), INT32_C( -192550726), INT32_C( 1009558740),
                            INT32_C( -269296946), INT32_C( -208448423), INT32_C(-1169491369), INT32_C( 1646367794),
                            INT32_C( -477436308), INT32_C( -901513289), INT32_C(-1611904316), INT32_C(  273215493),
                            INT32_C(   97905469), INT32_C(-1643161292), INT32_C( 1410612456), INT32_C(-2109125155)),
      simde_mm512_set_epi32(INT32_C(-1731772964), INT32_C(  875298769), INT32_C(-2111727649), INT32_C( 1106420076),
                            INT32_C( 1162135413), INT32_C( -665441062), INT32_C( -657896738), INT32_C(-1274599975),
                            INT32_C(-1886818210), INT32_C(-1269608085), INT32_C(-1717288194), INT32_C( 1039118950),
                            INT32_C(-1191741760), INT32_C( -378408410), INT32_C( -975126773), INT32_C(  215399677)),
      simde_mm512_set_epi32(INT32_C( -525985775), INT32_C( -891488173), INT32_C( 1990688921), INT32_C(-1054970097),
                            INT32_C(-1076121842), INT32_C(-1798454285), INT32_C(-1827388107), INT32_C(  371767819),
                            INT32_C( -175946824), INT32_C( 1263680378), INT32_C( 1967511219), INT32_C( 1312334443),
                            INT32_C(-1093836291), INT32_C( -768724171), INT32_C(  394289516), INT32_C( 1184804742)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_add_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_add_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(31559),
      simde_mm512_set_epi32(INT32_C( 1185842290), INT32_C( -832929072), INT32_C( -860387864), INT32_C(  211908911),
                            INT32_C(  497306141), INT32_C( 1740446190), INT32_C(  -30796973), INT32_C(  546522980),
                            INT32_C( 2019321788), INT32_C( -409706126), INT32_C(-2121872276), INT32_C( 1503967564),
                            INT32_C(  352022279), INT32_C(  625811896), INT32_C(  383029660), INT32_C( 2022592628)),
      simde_mm512_set_epi32(INT32_C(  794837942), INT32_C( 1623360790), INT32_C(-1283684153), INT32_C(-1617211019),
                            INT32_C( 1777497196), INT32_C(  405830398), INT32_C( 1331428450), INT32_C(  996590280),
                            INT32_C(-1247732201), INT32_C(-2033940216), INT32_C(-1407220826), INT32_C( -609231039),
                            INT32_C( -694118526), INT32_C( 2023877555), INT32_C(-2004555241), INT32_C( -665463129)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  790431718), INT32_C(-2144072017), INT32_C(-1405302108),
                            INT32_C(-2020163959), INT32_C(          0), INT32_C( 1300631477), INT32_C( 1543113260),
                            INT32_C(          0), INT32_C( 1851320954), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(-1645277845), INT32_C(-1621525581), INT32_C( 1357129499)) },
    { UINT16_C(44160),
      simde_mm512_set_epi32(INT32_C( 1586234702), INT32_C(  585597803), INT32_C( 1485431902), INT32_C(  -53574823),
                            INT32_C( -698862547), INT32_C( -297088246), INT32_C(  982100566), INT32_C( 1538055332),
                            INT32_C(  539848686), INT32_C( -746214295), INT32_C(   28052902), INT32_C(  -31662421),
                            INT32_C( 1607918893), INT32_C(-1432315312), INT32_C( -915684520), INT32_C( 2098723781)),
      simde_mm512_set_epi32(INT32_C(-1374188372), INT32_C( 2100708960), INT32_C(-1082275485), INT32_C(-1902240023),
                            INT32_C( 1872385874), INT32_C( 1610351959), INT32_C(  842733793), INT32_C(-1706419611),
                            INT32_C(  140510196), INT32_C(-1337034882), INT32_C( 1716721732), INT32_C(  690310192),
                            INT32_C( -701752067), INT32_C( 1836384400), INT32_C(  235539400), INT32_C(  -98860507)),
      simde_mm512_set_epi32(INT32_C(  212046330), INT32_C(          0), INT32_C(  403156417), INT32_C(          0),
                            INT32_C( 1173523327), INT32_C( 1313263713), INT32_C(          0), INT32_C(          0),
                            INT32_C(  680358882), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(22757),
      simde_mm512_set_epi32(INT32_C(   67146738), INT32_C( -628278177), INT32_C(  -42163692), INT32_C(-1872759151),
                            INT32_C( 1331593475), INT32_C(-1180636493), INT32_C(-1110698854), INT32_C( -322539115),
                            INT32_C(  344430931), INT32_C( -939252682), INT32_C(-1963540195), INT32_C( -679954819),
                            INT32_C( 1553884107), INT32_C( 1959000856), INT32_C(-1734889858), INT32_C(-1593392163)),
      simde_mm512_set_epi32(INT32_C(-1507288305), INT32_C( -440507593), INT32_C( -858818938), INT32_C(  133369119),
                            INT32_C( 2036463331), INT32_C( -646281729), INT32_C(-1218095899), INT32_C(   97686364),
                            INT32_C(  417101812), INT32_C( 1286204826), INT32_C(-1135015073), INT32_C( 1717602707),
                            INT32_C( 2013040731), INT32_C( 1732944555), INT32_C(-2039774627), INT32_C(-1868781950)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(-1068785770), INT32_C(          0), INT32_C(-1739390032),
                            INT32_C( -926910490), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  761532743), INT32_C(  346952144), INT32_C( 1196412028), INT32_C(          0),
                            INT32_C(          0), INT32_C( -603021885), INT32_C(          0), INT32_C(  832793183)) },
    { UINT16_C(19740),
      simde_mm512_set_epi32(INT32_C( -375332858), INT32_C(-1485347139), INT32_C(-1071997300), INT32_C(  -65357797),
                            INT32_C(  305161244), INT32_C(-1870131295), INT32_C(-2118931809), INT32_C( -770474176),
                            INT32_C(   78466466), INT32_C(  393403545), INT32_C( -452742526), INT32_C( -318562673),
                            INT32_C( 1241851960), INT32_C(  924270974), INT32_C( 1167797288), INT32_C( -485103161)),
      simde_mm512_set_epi32(INT32_C(-1728116332), INT32_C(-1299033022), INT32_C( -344200142), INT32_C(-1772981551),
                            INT32_C( 1734152045), INT32_C( -752401577), INT32_C(-1199473353), INT32_C( 1143381213),
                            INT32_C( -227043934), INT32_C( 2118579705), INT32_C(  -26291514), INT32_C(-1975764192),
                            INT32_C(  582333880), INT32_C( -278978899), INT32_C(-1529910569), INT32_C(-1979488988)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C( 1510587135), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2039313289), INT32_C( 1672434424), INT32_C(          0), INT32_C(  372907037),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C( 2000640431),
                            INT32_C( 1824185840), INT32_C(  645292075), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(31494),
      simde_mm512_set_epi32(INT32_C( 1998117926), INT32_C( 1427724712), INT32_C(-1703182853), INT32_C(-1976476203),
                            INT32_C(-1319662573), INT32_C( 1468825012), INT32_C( 1826804756), INT32_C(-1011363710),
                            INT32_C(-2011277433), INT32_C(  465439692), INT32_C(   74440550), INT32_C(  351968587),
                            INT32_C( -155284179), INT32_C( -431396309), INT32_C( -157966824), INT32_C(-1713680503)),
      simde_mm512_set_epi32(INT32_C( 1516284790), INT32_C( -721940049), INT32_C( 1127762182), INT32_C( -387369919),
                            INT32_C(  360990929), INT32_C(-1838442093), INT32_C(-1771580501), INT32_C( 1125000832),
                            INT32_C(  495619106), INT32_C( 1436729145), INT32_C(  511390630), INT32_C( 1472165566),
                            INT32_C( -620438060), INT32_C(-1444151087), INT32_C(-1984204291), INT32_C(-1976349845)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  705784663), INT32_C( -575420671), INT32_C( 1931121174),
                            INT32_C( -958671644), INT32_C(          0), INT32_C(   55224255), INT32_C(  113637122),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(-1875547396), INT32_C(-2142171115), INT32_C(          0)) },
    { UINT16_C(58419),
      simde_mm512_set_epi32(INT32_C(  254367127), INT32_C( 1911099668), INT32_C( -550954273), INT32_C( 2134719329),
                            INT32_C( -847023820), INT32_C(-1733047632), INT32_C(-1873966548), INT32_C(-1089938633),
                            INT32_C(-1764228153), INT32_C(  482610511), INT32_C(  337038002), INT32_C(  186657461),
                            INT32_C(  365728719), INT32_C( -679184202), INT32_C( 1489378251), INT32_C(  181555471)),
      simde_mm512_set_epi32(INT32_C( 1855123617), INT32_C( 1186906852), INT32_C(-1133243324), INT32_C( -884765881),
                            INT32_C(  912493085), INT32_C( -893201655), INT32_C(  -41414365), INT32_C( 2013786074),
                            INT32_C(  946895980), INT32_C( -177197648), INT32_C(  721481447), INT32_C(  271822408),
                            INT32_C( -326846317), INT32_C(-1949999474), INT32_C( -465448304), INT32_C( -911725433)),
      simde_mm512_set_epi32(INT32_C( 2109490744), INT32_C(-1196960776), INT32_C(-1684197597), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1668718009), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1058519449), INT32_C(  458479869),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1023929947), INT32_C( -730169962)) },
    { UINT16_C(25559),
      simde_mm512_set_epi32(INT32_C( -351403119), INT32_C( -607298021), INT32_C(-1581014205), INT32_C( 1340112481),
                            INT32_C( 1053342855), INT32_C(-1544863459), INT32_C(-1474576593), INT32_C( -230964593),
                            INT32_C( -880359551), INT32_C( 1793425528), INT32_C(  316992754), INT32_C( -590214510),
                            INT32_C(  264342180), INT32_C(-1949175563), INT32_C(   11634047), INT32_C( 1192791254)),
      simde_mm512_set_epi32(INT32_C(  388347029), INT32_C(  563251779), INT32_C( 1435179197), INT32_C(  891578710),
                            INT32_C(-2025463861), INT32_C(   19279584), INT32_C(-1517405529), INT32_C(-1959826797),
                            INT32_C( -946203575), INT32_C(  729043417), INT32_C(  982696116), INT32_C( 2016323891),
                            INT32_C(  586062709), INT32_C(  123130224), INT32_C(-2056328126), INT32_C(-1935851300)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  -44046242), INT32_C( -145835008), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1302985174), INT32_C( 2104175906),
                            INT32_C(-1826563126), INT32_C(-1772498351), INT32_C(          0), INT32_C( 1426109381),
                            INT32_C(          0), INT32_C(-1826045339), INT32_C(-2044694079), INT32_C( -743060046)) },
    { UINT16_C(42497),
      simde_mm512_set_epi32(INT32_C( -545657112), INT32_C( -884694296), INT32_C( 1285102216), INT32_C(-1138689690),
                            INT32_C( 1226043147), INT32_C( 1042523904), INT32_C( 1937073517), INT32_C(-1704745834),
                            INT32_C(-2071532873), INT32_C(-1497944839), INT32_C( 1332853179), INT32_C(-1700412779),
                            INT32_C(  882896510), INT32_C(-1885680775), INT32_C(  445396531), INT32_C(-1074352769)),
      simde_mm512_set_epi32(INT32_C( 1926459862), INT32_C( -591082714), INT32_C(   78727481), INT32_C( 2027108491),
                            INT32_C( 1943717077), INT32_C(  800922523), INT32_C( -575208739), INT32_C( -241086265),
                            INT32_C( 1028915143), INT32_C( 1790964719), INT32_C(-1601980964), INT32_C(-2069882910),
                            INT32_C(  848966946), INT32_C( -398170962), INT32_C( 2133797006), INT32_C( -557366203)),
      simde_mm512_set_epi32(INT32_C( 1380802750), INT32_C(          0), INT32_C( 1363829697), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1843446427), INT32_C( 1361864778), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-1631718972)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_add_epi32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_add_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-1142113623186190173), INT64_C( 2669401519388559587),
                            INT64_C(-4041526363448340541), INT64_C( 8890984525115992605),
                            INT64_C( 3710059072002351182), INT64_C(   -8844419669430530),
                            INT64_C( -583435793687448962), INT64_C(-3776976191330287246)),
      simde_mm512_set_epi64(INT64_C(-2572258444501888207), INT64_C( 8010943607820280661),
                            INT64_C(-7896534424393030159), INT64_C( 8282192647960665352),
                            INT64_C(  811312670062340262), INT64_C(-8174776120225966748),
                            INT64_C( 4815827375621462801), INT64_C(-4343109153599957409)),
      simde_mm512_set_epi64(INT64_C(-3714372067688078380), INT64_C(-7766398946500711368),
                            INT64_C( 6508683285868180916), INT64_C(-1273566900632893659),
                            INT64_C( 4521371742064691444), INT64_C(-8183620539895397278),
                            INT64_C( 4232391581934013839), INT64_C(-8120085344930244655)) },
    { simde_mm512_set_epi64(INT64_C(-8907763853043432709), INT64_C(-7664489506315516041),
                            INT64_C( 7881121837601630619), INT64_C(-1123258072002783793),
                            INT64_C( 6852591379280896525), INT64_C(-3366432558441729450),
                            INT64_C(-5990500579016281828), INT64_C( 7624785959951223888)),
      simde_mm512_set_epi64(INT64_C(  530774369540308193), INT64_C( 4966843317235809520),
                            INT64_C(-4575853019860390176), INT64_C(-5468596229394507748),
                            INT64_C( 5016860099104730248), INT64_C(-4030848910458752721),
                            INT64_C(-5726415349997053878), INT64_C(-2689606844288066445)),
      simde_mm512_set_epi64(INT64_C(-8376989483503124516), INT64_C(-2697646189079706521),
                            INT64_C( 3305268817741240443), INT64_C(-6591854301397291541),
                            INT64_C(-6577292595323924843), INT64_C(-7397281468900482171),
                            INT64_C( 6729828144696215910), INT64_C( 4935179115663157443)) },
    { simde_mm512_set_epi64(INT64_C(-1147603065405303411), INT64_C(-7360179338648948250),
                            INT64_C( 8522846282553983063), INT64_C(-6411739365277986684),
                            INT64_C( 2565410720902708591), INT64_C(-6999797153964411090),
                            INT64_C(-4638909294820957529), INT64_C(-6689250119856274996)),
      simde_mm512_set_epi64(INT64_C( 5073596640780038992), INT64_C( 9103358253828460511),
                            INT64_C( -705008066916281166), INT64_C(-6536611133780086738),
                            INT64_C( 8265445360104219638), INT64_C( 3801153343625786085),
                            INT64_C( 2669262289718667251), INT64_C( 5656097673759919545)),
      simde_mm512_set_epi64(INT64_C( 3925993575374735581), INT64_C( 1743178915179512261),
                            INT64_C( 7817838215637701897), INT64_C( 5498393574651478194),
                            INT64_C(-7615887992702623387), INT64_C(-3198643810338625005),
                            INT64_C(-1969647005102290278), INT64_C(-1033152446096355451)) },
    { simde_mm512_set_epi64(INT64_C( 4726294533983517199), INT64_C(-3194102718522779616),
                            INT64_C( 7968557550946485055), INT64_C( 3231249301417879275),
                            INT64_C( 2730855362291069772), INT64_C(-7006805907158396614),
                            INT64_C( 7013656719434070104), INT64_C(-5111614041831206337)),
      simde_mm512_set_epi64(INT64_C( 8077331611956864824), INT64_C( 9219505211765165401),
                            INT64_C( -526170803618465025), INT64_C(-6117963664293386621),
                            INT64_C( 8138058063794249913), INT64_C(-6512904449490940829),
                            INT64_C(  474025353181411678), INT64_C(-7774133174508991700)),
      simde_mm512_set_epi64(INT64_C(-5643117927769169593), INT64_C( 6025402493242385785),
                            INT64_C( 7442386747328020030), INT64_C(-2886714362875507346),
                            INT64_C(-7577830647624231931), INT64_C( 4927033717060214173),
                            INT64_C( 7487682072615481782), INT64_C( 5560996857369353579)) },
    { simde_mm512_set_epi64(INT64_C(  193050968130854053), INT64_C( 8259941741302086974),
                            INT64_C( -453186687001595332), INT64_C( 7712011730228920808),
                            INT64_C(-6104336988767299426), INT64_C( 5308601606176372502),
                            INT64_C(-2287583704165123047), INT64_C( 5680091922249770976)),
      simde_mm512_set_epi64(INT64_C(-4961468284661849310), INT64_C( 9060485449513062801),
                            INT64_C( 6632655590889112567), INT64_C( 7501604346789135910),
                            INT64_C(-7298384925887684809), INT64_C( 2881824411969821346),
                            INT64_C(-8132400455203655002), INT64_C( 4133962836264448240)),
      simde_mm512_set_epi64(INT64_C(-4768417316530995257), INT64_C(-1126316882894401841),
                            INT64_C( 6179468903887517235), INT64_C(-3233127996691494898),
                            INT64_C( 5044022159054567381), INT64_C( 8190426018146193848),
                            INT64_C( 8026759914340773567), INT64_C(-8632689315195332400)) },
    { simde_mm512_set_epi64(INT64_C( 8956245117105388401), INT64_C(-1279259458875765791),
                            INT64_C(-4844132309560126483), INT64_C( 8020518294074851688),
                            INT64_C(-5641381585453289469), INT64_C(-7336114943591686134),
                            INT64_C( 7396026256462185783), INT64_C(-7159923149096431079)),
      simde_mm512_set_epi64(INT64_C( 7681308578881334632), INT64_C( 6762384852768739876),
                            INT64_C(-6630226433339556438), INT64_C(  895229301334345958),
                            INT64_C( 8748757877138934431), INT64_C(-3406024454166778034),
                            INT64_C(-1971110785327035786), INT64_C( 5257409320594533793)),
      simde_mm512_set_epi64(INT64_C(-1809190377722828583), INT64_C( 5483125393892974085),
                            INT64_C( 6972385330809868695), INT64_C( 8915747595409197646),
                            INT64_C( 3107376291685644962), INT64_C( 7704604675951087448),
                            INT64_C( 5424915471135149997), INT64_C(-1902513828501897286)) },
    { simde_mm512_set_epi64(INT64_C(-6379556098233125245), INT64_C(-7036815414983308437),
                            INT64_C( 6449458293699998062), INT64_C(-8662596582560742605),
                            INT64_C( 8559396047271930933), INT64_C( 8319158628050257118),
                            INT64_C( 2671429253434382084), INT64_C(-3771778240982532322)),
      simde_mm512_set_epi64(INT64_C( 4847839000068024729), INT64_C( 7410858097550413452),
                            INT64_C( -481636656067004168), INT64_C(  273087245529831397),
                            INT64_C( 2014413537357415871), INT64_C(-3897516487309028112),
                            INT64_C( 4133432427690444225), INT64_C(-7762159775807857014)),
      simde_mm512_set_epi64(INT64_C(-1531717098165100516), INT64_C(  374042682567105015),
                            INT64_C( 5967821637632993894), INT64_C(-8389509337030911208),
                            INT64_C(-7872934489080204812), INT64_C( 4421642140741229006),
                            INT64_C( 6804861681124826309), INT64_C( 6912806056919162280)) },
    { simde_mm512_set_epi64(INT64_C( 7044617451253925485), INT64_C( 8554512071902279421),
                            INT64_C(-5774887429267945902), INT64_C( 8744694126758591835),
                            INT64_C( 6726346195930174353), INT64_C(-3624175965481610004),
                            INT64_C(-2980185457068069894), INT64_C(-1599338563378690807)),
      simde_mm512_set_epi64(INT64_C( 7457111512897216818), INT64_C(-2685664891179263207),
                            INT64_C( 3286698240604454682), INT64_C( 1197616399803304076),
                            INT64_C(-5755426011606503746), INT64_C(-7010024580994263662),
                            INT64_C(-1798652028950367168), INT64_C( 1422740686196608462)),
      simde_mm512_set_epi64(INT64_C(-3945015109558409313), INT64_C( 5868847180723016214),
                            INT64_C(-2488189188663491220), INT64_C(-8504433547147655705),
                            INT64_C(  970920184323670607), INT64_C( 7812543527233677950),
                            INT64_C(-4778837486018437062), INT64_C( -176597877182082345)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_add_epi64(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_add_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 5160606124915516278), INT64_C( 2686703283816926761),
                            INT64_C(-8226509414835337154), INT64_C( 5690216433899884438),
                            INT64_C(  219055368047565404), INT64_C(-6881247440148553068),
                            INT64_C(-2952194433022468951), INT64_C( 7998993867733664281)),
      UINT8_C(125),
      simde_mm512_set_epi64(INT64_C( 4359991822743182609), INT64_C( 6124953194992826221),
                            INT64_C(-5425342806155035646), INT64_C(  342961080598008522),
                            INT64_C( 1630649218949829874), INT64_C( 2808761313448984793),
                            INT64_C( 7567698855771953354), INT64_C( 7868549762334135240)),
      simde_mm512_set_epi64(INT64_C( 1716328492358350901), INT64_C( 7741977536996877481),
                            INT64_C(-6018347814321137283), INT64_C(-4359739956485737054),
                            INT64_C( 4652815018988362828), INT64_C(-6617209762203042095),
                            INT64_C(-7906031360269238320), INT64_C(-7708708033457348466)),
      simde_mm512_set_epi64(INT64_C( 5160606124915516278), INT64_C(-4579813341719847914),
                            INT64_C( 7003053453233378687), INT64_C(-4016778875887728532),
                            INT64_C( 6283464237938192702), INT64_C(-3808448448754057302),
                            INT64_C(-2952194433022468951), INT64_C(  159841728876786774)) },
    { simde_mm512_set_epi64(INT64_C(-4212727990389005953), INT64_C( 5489701730401906438),
                            INT64_C( 4330548005927972038), INT64_C(-3367425702478845714),
                            INT64_C( -251207382631141631), INT64_C(-6209666457612301578),
                            INT64_C( 2837216589874600576), INT64_C(-3808817973521776956)),
      UINT8_C( 28),
      simde_mm512_set_epi64(INT64_C( 3507085476272642910), INT64_C( 6803029509347915096),
                            INT64_C(-6877330114460900045), INT64_C(-2428113608477746314),
                            INT64_C(-3365973898038585786), INT64_C( 8113909718639972209),
                            INT64_C( 4208716190152504995), INT64_C(-5402592352873076933)),
      simde_mm512_set_epi64(INT64_C(-2196363492277266517), INT64_C(-6663287497478328654),
                            INT64_C( -815907504733491749), INT64_C(  -23068696008893187),
                            INT64_C(-2958298126746040750), INT64_C( 2693538183984094692),
                            INT64_C( 7757226695218661902), INT64_C(  617919997580221406)),
      simde_mm512_set_epi64(INT64_C(-4212727990389005953), INT64_C( 5489701730401906438),
                            INT64_C( 4330548005927972038), INT64_C(-2451182304486639501),
                            INT64_C(-6324272024784626536), INT64_C(-7639296171085484715),
                            INT64_C( 2837216589874600576), INT64_C(-3808817973521776956)) },
    { simde_mm512_set_epi64(INT64_C(-2658186320062013475), INT64_C( 6472396198048668259),
                            INT64_C(-5364106486051370763), INT64_C( 2934978626049102270),
                            INT64_C( 5139389583907077143), INT64_C(-1809434287129424605),
                            INT64_C(-2838108343154748008), INT64_C( 3320008369946313548)),
      UINT8_C( 79),
      simde_mm512_set_epi64(INT64_C( 5496809340468304504), INT64_C( 3368405333208462924),
                            INT64_C(-3519008522301999995), INT64_C(-7481564464207937019),
                            INT64_C(-4704132991357348649), INT64_C( 9130219773469471208),
                            INT64_C(-6923581550719270919), INT64_C( 4881655826255274884)),
      simde_mm512_set_epi64(INT64_C(-3861097258215961654), INT64_C(-2296374770448026399),
                            INT64_C( 7577168671573663803), INT64_C(-5990872388562171119),
                            INT64_C( 7263214171707552987), INT64_C( 6808296108057623076),
                            INT64_C(-5871336145405392778), INT64_C( 7331789029441415338)),
      simde_mm512_set_epi64(INT64_C(-2658186320062013475), INT64_C( 1072030562760436525),
                            INT64_C(-5364106486051370763), INT64_C( 2934978626049102270),
                            INT64_C( 2559081180350204338), INT64_C(-2508228192182457332),
                            INT64_C( 5651826377584887919), INT64_C(-6233299218012861394)) },
    { simde_mm512_set_epi64(INT64_C( 5289725073500410972), INT64_C( 6717956330087056566),
                            INT64_C( -983047186113812262), INT64_C(-1997572120626723278),
                            INT64_C( 2827816390590156831), INT64_C( 4212031401732227465),
                            INT64_C(-7136969504671814778), INT64_C( 7259023190572205496)),
      UINT8_C(154),
      simde_mm512_set_epi64(INT64_C(  610245780431113079), INT64_C(-2461930398177597570),
                            INT64_C( 5237068016333533006), INT64_C( 4084499040607648130),
                            INT64_C(-5607163251111733967), INT64_C(-3295018021625017525),
                            INT64_C( 4015916744866468685), INT64_C(-7233812242085237251)),
      simde_mm512_set_epi64(INT64_C( 1275989633124597649), INT64_C(-2235043592779723835),
                            INT64_C( 3584069642166863788), INT64_C( 1603795481570313714),
                            INT64_C( 4544979301003664648), INT64_C( 1257449042927607643),
                            INT64_C(  981820998427179335), INT64_C( 3724733603231200149)),
      simde_mm512_set_epi64(INT64_C( 1886235413555710728), INT64_C( 6717956330087056566),
                            INT64_C( -983047186113812262), INT64_C( 5688294522177961844),
                            INT64_C(-1062183950108069319), INT64_C( 4212031401732227465),
                            INT64_C( 4997737743293648020), INT64_C( 7259023190572205496)) },
    { simde_mm512_set_epi64(INT64_C( -344322387915524053), INT64_C(-4868072091941636376),
                            INT64_C(-4549031926386074228), INT64_C(  566826057630382860),
                            INT64_C( -657390120933763660), INT64_C(-4251701049171047602),
                            INT64_C(-5614545355007378848), INT64_C( 5955783881071001067)),
      UINT8_C(134),
      simde_mm512_set_epi64(INT64_C(  524181298629282005), INT64_C(-8548936336365113208),
                            INT64_C( 3598784896330239568), INT64_C( 5295942590094852910),
                            INT64_C(-9043448539137453184), INT64_C(-8387918846648895588),
                            INT64_C(-7982644523660563361), INT64_C(-8202542553254534213)),
      simde_mm512_set_epi64(INT64_C(-4387280847879026019), INT64_C(-3303620315308853860),
                            INT64_C(  953555559191704496), INT64_C(-6394540538650954983),
                            INT64_C(-7662782970452208157), INT64_C( 4905118310935268319),
                            INT64_C(-4967018180570261969), INT64_C( 1368543426140171683)),
      simde_mm512_set_epi64(INT64_C(-3863099549249744014), INT64_C(-4868072091941636376),
                            INT64_C(-4549031926386074228), INT64_C(  566826057630382860),
                            INT64_C( -657390120933763660), INT64_C(-3482800535713627269),
                            INT64_C( 5497081369478726286), INT64_C( 5955783881071001067)) },
    { simde_mm512_set_epi64(INT64_C(-5040599758757488360), INT64_C(-8127631176289564902),
                            INT64_C( -229664378178268662), INT64_C(-5479802575354552579),
                            INT64_C( 4832979330378877040), INT64_C( 7046908214192494697),
                            INT64_C(  432279363446219305), INT64_C( 2399549731827167415)),
      UINT8_C(226),
      simde_mm512_set_epi64(INT64_C( 5961166598398813754), INT64_C( 7362715620039542918),
                            INT64_C( -198040368546109595), INT64_C( 8508063665637766245),
                            INT64_C(-3093028728322401296), INT64_C( 4334047704750276202),
                            INT64_C( 7297510088042055655), INT64_C(-5958925472978890262)),
      simde_mm512_set_epi64(INT64_C(-8157078932371790609), INT64_C( 2478738248267923915),
                            INT64_C( 4599441206936119875), INT64_C( -481782687609933528),
                            INT64_C(-1903047019345848051), INT64_C(-9194280126517020760),
                            INT64_C( 3561889952318847548), INT64_C( 8732950799460131342)),
      simde_mm512_set_epi64(INT64_C(-2195912333972976855), INT64_C(-8605290205402084783),
                            INT64_C( 4401400838390010280), INT64_C(-5479802575354552579),
                            INT64_C( 4832979330378877040), INT64_C( 7046908214192494697),
                            INT64_C(-7587344033348648413), INT64_C( 2399549731827167415)) },
    { simde_mm512_set_epi64(INT64_C(-8104078709393797509), INT64_C( -701085671815626380),
                            INT64_C(-2418640555709185426), INT64_C( 1772911152318416135),
                            INT64_C( 4752194605515385270), INT64_C( 8081927143547208670),
                            INT64_C( 2029190253246208730), INT64_C(-2955091050006511678)),
      UINT8_C( 31),
      simde_mm512_set_epi64(INT64_C(-2130079524565547542), INT64_C( 4308781613304087141),
                            INT64_C(-6512539784195852568), INT64_C(-5667810152138428565),
                            INT64_C( 5047940836630437561), INT64_C( 6025421452869547263),
                            INT64_C( 4091802541986151804), INT64_C( 3800854672489510112)),
      simde_mm512_set_epi64(INT64_C(-8689276013500918074), INT64_C( -281059922984814169),
                            INT64_C( 2615008611710254316), INT64_C(-1642316481488703866),
                            INT64_C( 2756455466191421737), INT64_C( 4849950500349584730),
                            INT64_C( 8998921784155780820), INT64_C( 3478741257218521835)),
      simde_mm512_set_epi64(INT64_C(-8104078709393797509), INT64_C( -701085671815626380),
                            INT64_C(-2418640555709185426), INT64_C(-7310126633627132431),
                            INT64_C( 7804396302821859298), INT64_C(-7571372120490419623),
                            INT64_C(-5356019747567618992), INT64_C( 7279595929708031947)) },
    { simde_mm512_set_epi64(INT64_C( 4214463940983249865), INT64_C(-3985446734163962000),
                            INT64_C( 8929692039888006294), INT64_C(-6197389418010506128),
                            INT64_C( 3172660499364021696), INT64_C( 4790686542915338404),
                            INT64_C( 3895272345102635530), INT64_C( 7002879752154026617)),
      UINT8_C(191),
      simde_mm512_set_epi64(INT64_C(-6572676402753401153), INT64_C(-5358711999471905537),
                            INT64_C(-8788555168038094419), INT64_C( -254982934203170038),
                            INT64_C(-1335218364203681854), INT64_C( 4928945664557712110),
                            INT64_C(-8999202966683896655), INT64_C( 2909251939064430104)),
      simde_mm512_set_epi64(INT64_C(-6624500623925256726), INT64_C(-1530365027491741231),
                            INT64_C(-7760352547768795779), INT64_C( 5915109941470916806),
                            INT64_C(-3660568171856511019), INT64_C(-2085912392048614822),
                            INT64_C(-8710444771289969701), INT64_C( 5184359963834841280)),
      simde_mm512_set_epi64(INT64_C( 5249567047030893737), INT64_C(-3985446734163962000),
                            INT64_C( 1897836357902661418), INT64_C( 5660127007267746768),
                            INT64_C(-4995786536060192873), INT64_C( 2843033272509097288),
                            INT64_C(  737096335735685260), INT64_C( 8093611902899271384)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_add_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_add_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 71),
      simde_mm512_set_epi64(INT64_C(-3577403120693049880), INT64_C(  910141842973280797),
                            INT64_C( 7475159470761972563), INT64_C( 2347298327631783868),
                            INT64_C(-1759674409967760276), INT64_C( 6459491501976809223),
                            INT64_C( 2687841627150782876), INT64_C( 8686969193579912345)),
      simde_mm512_set_epi64(INT64_C( 6972281505670006983), INT64_C(-6945868435558337428),
                            INT64_C( 1743028288464092258), INT64_C( 4280322663158717975),
                            INT64_C(-8735706706851429466), INT64_C(-2616627384612251774),
                            INT64_C( 8692487912123853335), INT64_C(-2858142374562986894)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-6035726592585056631),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 3842864117364557449),
                            INT64_C(-7066414534434915405), INT64_C( 5828826819016925451)) },
    { UINT8_C(182),
      simde_mm512_set_epi64(INT64_C( 6379881443766469465), INT64_C(-3001591779766383862),
                            INT64_C( 4218089813891144868), INT64_C( 2318632454707326057),
                            INT64_C(  120486300911197867), INT64_C( 6905959062918175312),
                            INT64_C(-3932835064754734139), INT64_C(-4910443416603153280)),
      simde_mm512_set_epi64(INT64_C(-4648337810944811287), INT64_C( 8041836095932728663),
                            INT64_C( 3619514082757581413), INT64_C(  603486699532482430),
                            INT64_C( 7373263695962786864), INT64_C(-3014002175829016432),
                            INT64_C( 1011634024115569189), INT64_C( 6812826169455903595)),
      simde_mm512_set_epi64(INT64_C( 1731543632821658178), INT64_C(                   0),
                            INT64_C( 7837603896648726281), INT64_C( 2922119154239808487),
                            INT64_C(                   0), INT64_C( 3891956887089158880),
                            INT64_C(-2921201040639164950), INT64_C(                   0)) },
    { UINT8_C( 96),
      simde_mm512_set_epi64(INT64_C(-8043439305498132221), INT64_C(-5070795122714864486),
                            INT64_C(-1385294950261352109), INT64_C(-4034059549538860771),
                            INT64_C(-2920383708808715317), INT64_C( 8413844611916082814),
                            INT64_C(-6843567226026140647), INT64_C( 5077061898421636268)),
      simde_mm512_set_epi64(INT64_C(  572816006437795555), INT64_C(-2775758886980463387),
                            INT64_C(  419559739062253556), INT64_C( 5524207666787322719),
                            INT64_C( 7377047456099111003), INT64_C( 7442940191761465949),
                            INT64_C(-8026357358537960462), INT64_C(-2698434218752695788)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-7846554009695327873),
                            INT64_C( -965735211199098553), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(134),
      simde_mm512_set_epi64(INT64_C( 1310657565411512225), INT64_C(-9100742818584625344),
                            INT64_C(  337010905696099481), INT64_C(-1944514338702025073),
                            INT64_C( 5333713555597771134), INT64_C( 5015651164127357383),
                            INT64_C( 4609583739357777180), INT64_C(-6473753971763813577)),
      simde_mm512_set_epi64(INT64_C( 7448126323109086039), INT64_C(-5151698822415082275),
                            INT64_C( -975146269166602759), INT64_C( -112921190473123040),
                            INT64_C( 2501104973968776877), INT64_C(-6570915857344273116),
                            INT64_C(-1612042347414591811), INT64_C(-4604193340670691301)),
      simde_mm512_set_epi64(INT64_C( 8758783888520598264), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-1555264693216915733),
                            INT64_C( 2997541391943185369), INT64_C(                   0)) },
    { UINT8_C(209),
      simde_mm512_set_epi64(INT64_C( 6308555391913612308), INT64_C(-4343774056527538297),
                            INT64_C( 1999048255474753382), INT64_C( 1511693574524013869),
                            INT64_C(-1852833034633109992), INT64_C(-7360201712415530197),
                            INT64_C(  247833247772248468), INT64_C(-5579304341963281358)),
      simde_mm512_set_epi64(INT64_C(-7896048662501403733), INT64_C( 4831841781908409378),
                            INT64_C( 6170704691496432550), INT64_C( 6322902963941858772),
                            INT64_C(-6202581686837087747), INT64_C(-8488357947731551194),
                            INT64_C( 6132030948322803195), INT64_C(-8488900650231952365)),
      simde_mm512_set_epi64(INT64_C(-1587493270587791425), INT64_C(  488067725380871081),
                            INT64_C(                   0), INT64_C( 7834596538465872641),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 4378539081514317893)) },
    { UINT8_C(209),
      simde_mm512_set_epi64(INT64_C(-8048625034252985545), INT64_C(-7577302219334873777),
                            INT64_C( 1447567196285840053), INT64_C( 1570792890928756918),
                            INT64_C( 6396830879600234767), INT64_C( 5436678673226720307),
                            INT64_C( 6512393588045255087), INT64_C( 4843701693263197249)),
      simde_mm512_set_epi64(INT64_C( -177873341245820966), INT64_C( 4066887270931639728),
                            INT64_C( 3098739219807579720), INT64_C(-1403794239988081010),
                            INT64_C(-1999085240275424121), INT64_C( 1092498493553578260),
                            INT64_C(-2366330581991736479), INT64_C(-3637939603271071056)),
      simde_mm512_set_epi64(INT64_C(-8226498375498806511), INT64_C(-3510414948403234049),
                            INT64_C(                   0), INT64_C(  166998650940675908),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 1205762089992126193)) },
    { UINT8_C(  9),
      simde_mm512_set_epi64(INT64_C( -991985370054342783), INT64_C( 7702703990888525042),
                            INT64_C(-2534952017810322780), INT64_C(-8371645297235753601),
                            INT64_C( 5122999428276907021), INT64_C( 7323807195958800545),
                            INT64_C( 5097726115900036164), INT64_C(-3800040522599134691)),
      simde_mm512_set_epi64(INT64_C(-8417391995590667191), INT64_C( 3131217634361786548),
                            INT64_C( 8660045170574531445), INT64_C(  528840287467793474),
                            INT64_C(-8314418019475520623), INT64_C(-2608325136406568125),
                            INT64_C( 5755739279909764231), INT64_C(-6635138030370046161)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-3191418591198613602), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 8011565520740370764)) },
    { UINT8_C(167),
      simde_mm512_set_epi64(INT64_C(-8897165939326898951), INT64_C( 5724560816769188501),
                            INT64_C( 3792011638611823481), INT64_C( 1912963537617464703),
                            INT64_C(-3154656186080844287), INT64_C( 1667937789617015363),
                            INT64_C( 6164047715906120022), INT64_C(-8699301042205610272)),
      simde_mm512_set_epi64(INT64_C( 4419156891335128047), INT64_C(-6880455846969468958),
                            INT64_C( 3646285272351794350), INT64_C( 9164588360810316869),
                            INT64_C(-2343579447459536152), INT64_C( 5519471992893405542),
                            INT64_C( 5265815220892444416), INT64_C( 8319667408052921494)),
      simde_mm512_set_epi64(INT64_C(-4478009047991770904), INT64_C(                   0),
                            INT64_C( 7438296910963617831), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 7187409782510420905),
                            INT64_C(-7016881136910987178), INT64_C( -379633634152688778)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_add_epi64(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_add_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -546.47), SIMDE_FLOAT32_C(  -401.21), SIMDE_FLOAT32_C(  -240.00), SIMDE_FLOAT32_C(  -513.10),
                         SIMDE_FLOAT32_C(  -121.28), SIMDE_FLOAT32_C(   728.43), SIMDE_FLOAT32_C(  -822.51), SIMDE_FLOAT32_C(     7.45),
                         SIMDE_FLOAT32_C(   960.90), SIMDE_FLOAT32_C(   805.67), SIMDE_FLOAT32_C(  -690.58), SIMDE_FLOAT32_C(   568.90),
                         SIMDE_FLOAT32_C(  -289.09), SIMDE_FLOAT32_C(  -506.27), SIMDE_FLOAT32_C(  -651.61), SIMDE_FLOAT32_C(   163.00)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   732.60), SIMDE_FLOAT32_C(   -88.88), SIMDE_FLOAT32_C(   508.96), SIMDE_FLOAT32_C(   -54.22),
                         SIMDE_FLOAT32_C(    -9.60), SIMDE_FLOAT32_C(   -95.16), SIMDE_FLOAT32_C(  -147.10), SIMDE_FLOAT32_C(   920.71),
                         SIMDE_FLOAT32_C(  -406.34), SIMDE_FLOAT32_C(   822.42), SIMDE_FLOAT32_C(    -9.72), SIMDE_FLOAT32_C(  -576.02),
                         SIMDE_FLOAT32_C(  -438.73), SIMDE_FLOAT32_C(  -733.21), SIMDE_FLOAT32_C(  -238.85), SIMDE_FLOAT32_C(   207.13)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   186.13), SIMDE_FLOAT32_C(  -490.09), SIMDE_FLOAT32_C(   268.96), SIMDE_FLOAT32_C(  -567.32),
                         SIMDE_FLOAT32_C(  -130.88), SIMDE_FLOAT32_C(   633.27), SIMDE_FLOAT32_C(  -969.61), SIMDE_FLOAT32_C(   928.16),
                         SIMDE_FLOAT32_C(   554.56), SIMDE_FLOAT32_C(  1628.09), SIMDE_FLOAT32_C(  -700.30), SIMDE_FLOAT32_C(    -7.12),
                         SIMDE_FLOAT32_C(  -727.82), SIMDE_FLOAT32_C( -1239.48), SIMDE_FLOAT32_C(  -890.46), SIMDE_FLOAT32_C(   370.13)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -926.20), SIMDE_FLOAT32_C(   375.14), SIMDE_FLOAT32_C(   940.73), SIMDE_FLOAT32_C(  -720.51),
                         SIMDE_FLOAT32_C(  -257.05), SIMDE_FLOAT32_C(   669.48), SIMDE_FLOAT32_C(    29.10), SIMDE_FLOAT32_C(   763.38),
                         SIMDE_FLOAT32_C(  -968.83), SIMDE_FLOAT32_C(   846.48), SIMDE_FLOAT32_C(  -399.89), SIMDE_FLOAT32_C(   648.70),
                         SIMDE_FLOAT32_C(   190.06), SIMDE_FLOAT32_C(  -421.71), SIMDE_FLOAT32_C(  -916.40), SIMDE_FLOAT32_C(   249.69)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -941.68), SIMDE_FLOAT32_C(  -700.28), SIMDE_FLOAT32_C(    37.43), SIMDE_FLOAT32_C(  -320.67),
                         SIMDE_FLOAT32_C(  -363.05), SIMDE_FLOAT32_C(   846.07), SIMDE_FLOAT32_C(   806.96), SIMDE_FLOAT32_C(   -60.84),
                         SIMDE_FLOAT32_C(   730.53), SIMDE_FLOAT32_C(   987.80), SIMDE_FLOAT32_C(   378.07), SIMDE_FLOAT32_C(  -394.92),
                         SIMDE_FLOAT32_C(   567.33), SIMDE_FLOAT32_C(   -54.37), SIMDE_FLOAT32_C(  -977.08), SIMDE_FLOAT32_C(  -714.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C( -1867.88), SIMDE_FLOAT32_C(  -325.14), SIMDE_FLOAT32_C(   978.16), SIMDE_FLOAT32_C( -1041.18),
                         SIMDE_FLOAT32_C(  -620.10), SIMDE_FLOAT32_C(  1515.55), SIMDE_FLOAT32_C(   836.06), SIMDE_FLOAT32_C(   702.54),
                         SIMDE_FLOAT32_C(  -238.30), SIMDE_FLOAT32_C(  1834.28), SIMDE_FLOAT32_C(   -21.82), SIMDE_FLOAT32_C(   253.78),
                         SIMDE_FLOAT32_C(   757.39), SIMDE_FLOAT32_C(  -476.08), SIMDE_FLOAT32_C( -1893.48), SIMDE_FLOAT32_C(  -464.73)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -240.64), SIMDE_FLOAT32_C(  -901.13), SIMDE_FLOAT32_C(   248.79), SIMDE_FLOAT32_C(  -761.42),
                         SIMDE_FLOAT32_C(   -34.99), SIMDE_FLOAT32_C(   819.20), SIMDE_FLOAT32_C(  -715.24), SIMDE_FLOAT32_C(  -958.36),
                         SIMDE_FLOAT32_C(  -876.64), SIMDE_FLOAT32_C(   224.96), SIMDE_FLOAT32_C(  -241.32), SIMDE_FLOAT32_C(  -833.06),
                         SIMDE_FLOAT32_C(  -928.34), SIMDE_FLOAT32_C(   822.01), SIMDE_FLOAT32_C(   -16.05), SIMDE_FLOAT32_C(  -241.91)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -998.98), SIMDE_FLOAT32_C(   951.38), SIMDE_FLOAT32_C(  -528.70), SIMDE_FLOAT32_C(  -961.34),
                         SIMDE_FLOAT32_C(  -781.68), SIMDE_FLOAT32_C(   919.38), SIMDE_FLOAT32_C(   660.71), SIMDE_FLOAT32_C(   -29.22),
                         SIMDE_FLOAT32_C(  -977.96), SIMDE_FLOAT32_C(  -923.68), SIMDE_FLOAT32_C(    -3.15), SIMDE_FLOAT32_C(  -732.59),
                         SIMDE_FLOAT32_C(  -504.08), SIMDE_FLOAT32_C(   111.05), SIMDE_FLOAT32_C(   700.38), SIMDE_FLOAT32_C(  -844.60)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C( -1239.62), SIMDE_FLOAT32_C(    50.25), SIMDE_FLOAT32_C(  -279.91), SIMDE_FLOAT32_C( -1722.76),
                         SIMDE_FLOAT32_C(  -816.67), SIMDE_FLOAT32_C(  1738.58), SIMDE_FLOAT32_C(   -54.53), SIMDE_FLOAT32_C(  -987.58),
                         SIMDE_FLOAT32_C( -1854.60), SIMDE_FLOAT32_C(  -698.72), SIMDE_FLOAT32_C(  -244.47), SIMDE_FLOAT32_C( -1565.65),
                         SIMDE_FLOAT32_C( -1432.42), SIMDE_FLOAT32_C(   933.06), SIMDE_FLOAT32_C(   684.33), SIMDE_FLOAT32_C( -1086.51)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   662.71), SIMDE_FLOAT32_C(   564.94), SIMDE_FLOAT32_C(  -317.08), SIMDE_FLOAT32_C(  -219.40),
                         SIMDE_FLOAT32_C(  -599.10), SIMDE_FLOAT32_C(  -944.25), SIMDE_FLOAT32_C(   691.43), SIMDE_FLOAT32_C(   -94.27),
                         SIMDE_FLOAT32_C(   505.61), SIMDE_FLOAT32_C(   476.10), SIMDE_FLOAT32_C(  -623.97), SIMDE_FLOAT32_C(  -863.44),
                         SIMDE_FLOAT32_C(   522.48), SIMDE_FLOAT32_C(   -79.88), SIMDE_FLOAT32_C(  -355.36), SIMDE_FLOAT32_C(  -248.57)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -575.57), SIMDE_FLOAT32_C(   909.53), SIMDE_FLOAT32_C(   384.99), SIMDE_FLOAT32_C(  -171.88),
                         SIMDE_FLOAT32_C(   -66.16), SIMDE_FLOAT32_C(  -405.98), SIMDE_FLOAT32_C(  -149.45), SIMDE_FLOAT32_C(  -972.01),
                         SIMDE_FLOAT32_C(    82.36), SIMDE_FLOAT32_C(  -907.67), SIMDE_FLOAT32_C(   832.44), SIMDE_FLOAT32_C(   -67.87),
                         SIMDE_FLOAT32_C(  -429.02), SIMDE_FLOAT32_C(  -820.79), SIMDE_FLOAT32_C(    81.64), SIMDE_FLOAT32_C(    69.75)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    87.14), SIMDE_FLOAT32_C(  1474.47), SIMDE_FLOAT32_C(    67.91), SIMDE_FLOAT32_C(  -391.28),
                         SIMDE_FLOAT32_C(  -665.26), SIMDE_FLOAT32_C( -1350.23), SIMDE_FLOAT32_C(   541.98), SIMDE_FLOAT32_C( -1066.28),
                         SIMDE_FLOAT32_C(   587.97), SIMDE_FLOAT32_C(  -431.57), SIMDE_FLOAT32_C(   208.47), SIMDE_FLOAT32_C(  -931.31),
                         SIMDE_FLOAT32_C(    93.46), SIMDE_FLOAT32_C(  -900.67), SIMDE_FLOAT32_C(  -273.72), SIMDE_FLOAT32_C(  -178.82)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -260.02), SIMDE_FLOAT32_C(  -334.14), SIMDE_FLOAT32_C(   105.88), SIMDE_FLOAT32_C(  -393.27),
                         SIMDE_FLOAT32_C(   549.30), SIMDE_FLOAT32_C(   752.10), SIMDE_FLOAT32_C(   339.34), SIMDE_FLOAT32_C(  -761.70),
                         SIMDE_FLOAT32_C(   564.82), SIMDE_FLOAT32_C(  -275.31), SIMDE_FLOAT32_C(  -952.67), SIMDE_FLOAT32_C(  -216.82),
                         SIMDE_FLOAT32_C(  -270.02), SIMDE_FLOAT32_C(   133.54), SIMDE_FLOAT32_C(   513.36), SIMDE_FLOAT32_C(   -92.17)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -669.70), SIMDE_FLOAT32_C(   -59.27), SIMDE_FLOAT32_C(   771.68), SIMDE_FLOAT32_C(   474.66),
                         SIMDE_FLOAT32_C(    41.08), SIMDE_FLOAT32_C(  -386.46), SIMDE_FLOAT32_C(   740.49), SIMDE_FLOAT32_C(   -90.54),
                         SIMDE_FLOAT32_C(   947.02), SIMDE_FLOAT32_C(  -119.81), SIMDE_FLOAT32_C(   259.70), SIMDE_FLOAT32_C(   -54.65),
                         SIMDE_FLOAT32_C(  -882.33), SIMDE_FLOAT32_C(  -869.82), SIMDE_FLOAT32_C(   646.89), SIMDE_FLOAT32_C(  -460.05)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -929.72), SIMDE_FLOAT32_C(  -393.41), SIMDE_FLOAT32_C(   877.56), SIMDE_FLOAT32_C(    81.39),
                         SIMDE_FLOAT32_C(   590.38), SIMDE_FLOAT32_C(   365.64), SIMDE_FLOAT32_C(  1079.83), SIMDE_FLOAT32_C(  -852.24),
                         SIMDE_FLOAT32_C(  1511.84), SIMDE_FLOAT32_C(  -395.12), SIMDE_FLOAT32_C(  -692.97), SIMDE_FLOAT32_C(  -271.47),
                         SIMDE_FLOAT32_C( -1152.35), SIMDE_FLOAT32_C(  -736.28), SIMDE_FLOAT32_C(  1160.25), SIMDE_FLOAT32_C(  -552.22)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   756.25), SIMDE_FLOAT32_C(   511.04), SIMDE_FLOAT32_C(   859.11), SIMDE_FLOAT32_C(  -867.90),
                         SIMDE_FLOAT32_C(  -479.94), SIMDE_FLOAT32_C(  -217.41), SIMDE_FLOAT32_C(  -447.63), SIMDE_FLOAT32_C(   911.19),
                         SIMDE_FLOAT32_C(  -709.58), SIMDE_FLOAT32_C(  -654.53), SIMDE_FLOAT32_C(  -416.78), SIMDE_FLOAT32_C(   728.12),
                         SIMDE_FLOAT32_C(  -682.52), SIMDE_FLOAT32_C(   389.98), SIMDE_FLOAT32_C(   929.37), SIMDE_FLOAT32_C(  -933.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   921.09), SIMDE_FLOAT32_C(    94.70), SIMDE_FLOAT32_C(  -824.20), SIMDE_FLOAT32_C(   359.66),
                         SIMDE_FLOAT32_C(  -948.58), SIMDE_FLOAT32_C(   259.61), SIMDE_FLOAT32_C(  -263.97), SIMDE_FLOAT32_C(   203.12),
                         SIMDE_FLOAT32_C(   565.73), SIMDE_FLOAT32_C(   268.27), SIMDE_FLOAT32_C(   574.31), SIMDE_FLOAT32_C(  -841.55),
                         SIMDE_FLOAT32_C(   681.29), SIMDE_FLOAT32_C(   574.95), SIMDE_FLOAT32_C(   623.04), SIMDE_FLOAT32_C(    56.66)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  1677.34), SIMDE_FLOAT32_C(   605.74), SIMDE_FLOAT32_C(    34.91), SIMDE_FLOAT32_C(  -508.24),
                         SIMDE_FLOAT32_C( -1428.52), SIMDE_FLOAT32_C(    42.20), SIMDE_FLOAT32_C(  -711.60), SIMDE_FLOAT32_C(  1114.31),
                         SIMDE_FLOAT32_C(  -143.85), SIMDE_FLOAT32_C(  -386.26), SIMDE_FLOAT32_C(   157.53), SIMDE_FLOAT32_C(  -113.43),
                         SIMDE_FLOAT32_C(    -1.23), SIMDE_FLOAT32_C(   964.93), SIMDE_FLOAT32_C(  1552.41), SIMDE_FLOAT32_C(  -876.61)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   437.52), SIMDE_FLOAT32_C(  -206.27), SIMDE_FLOAT32_C(   -27.72), SIMDE_FLOAT32_C(  -407.95),
                         SIMDE_FLOAT32_C(  -100.99), SIMDE_FLOAT32_C(   -27.12), SIMDE_FLOAT32_C(   232.25), SIMDE_FLOAT32_C(  -679.34),
                         SIMDE_FLOAT32_C(   493.07), SIMDE_FLOAT32_C(   235.84), SIMDE_FLOAT32_C(  -247.65), SIMDE_FLOAT32_C(   639.70),
                         SIMDE_FLOAT32_C(   408.65), SIMDE_FLOAT32_C(   173.51), SIMDE_FLOAT32_C(    33.96), SIMDE_FLOAT32_C(  -654.24)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   995.52), SIMDE_FLOAT32_C(   101.75), SIMDE_FLOAT32_C(  -157.89), SIMDE_FLOAT32_C(   762.54),
                         SIMDE_FLOAT32_C(  -359.84), SIMDE_FLOAT32_C(   895.91), SIMDE_FLOAT32_C(  -723.29), SIMDE_FLOAT32_C(  -396.31),
                         SIMDE_FLOAT32_C(  -209.52), SIMDE_FLOAT32_C(   465.97), SIMDE_FLOAT32_C(   723.67), SIMDE_FLOAT32_C(   -27.91),
                         SIMDE_FLOAT32_C(   178.75), SIMDE_FLOAT32_C(   111.72), SIMDE_FLOAT32_C(    49.88), SIMDE_FLOAT32_C(   -16.33)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  1433.04), SIMDE_FLOAT32_C(  -104.52), SIMDE_FLOAT32_C(  -185.61), SIMDE_FLOAT32_C(   354.59),
                         SIMDE_FLOAT32_C(  -460.83), SIMDE_FLOAT32_C(   868.79), SIMDE_FLOAT32_C(  -491.04), SIMDE_FLOAT32_C( -1075.65),
                         SIMDE_FLOAT32_C(   283.55), SIMDE_FLOAT32_C(   701.81), SIMDE_FLOAT32_C(   476.02), SIMDE_FLOAT32_C(   611.79),
                         SIMDE_FLOAT32_C(   587.40), SIMDE_FLOAT32_C(   285.23), SIMDE_FLOAT32_C(    83.84), SIMDE_FLOAT32_C(  -670.57)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -375.93), SIMDE_FLOAT32_C(   812.42), SIMDE_FLOAT32_C(   139.50), SIMDE_FLOAT32_C(  -189.29),
                         SIMDE_FLOAT32_C(   145.73), SIMDE_FLOAT32_C(  -679.16), SIMDE_FLOAT32_C(  -470.57), SIMDE_FLOAT32_C(   204.76),
                         SIMDE_FLOAT32_C(  -513.19), SIMDE_FLOAT32_C(   720.36), SIMDE_FLOAT32_C(   372.49), SIMDE_FLOAT32_C(   878.61),
                         SIMDE_FLOAT32_C(  -970.72), SIMDE_FLOAT32_C(   793.74), SIMDE_FLOAT32_C(   338.04), SIMDE_FLOAT32_C(  -645.89)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   929.83), SIMDE_FLOAT32_C(   319.52), SIMDE_FLOAT32_C(   731.40), SIMDE_FLOAT32_C(   721.97),
                         SIMDE_FLOAT32_C(  -865.47), SIMDE_FLOAT32_C(   180.74), SIMDE_FLOAT32_C(  -262.02), SIMDE_FLOAT32_C(   877.25),
                         SIMDE_FLOAT32_C(  -600.16), SIMDE_FLOAT32_C(  -927.22), SIMDE_FLOAT32_C(  -354.63), SIMDE_FLOAT32_C(  -776.27),
                         SIMDE_FLOAT32_C(  -950.90), SIMDE_FLOAT32_C(   958.35), SIMDE_FLOAT32_C(   483.36), SIMDE_FLOAT32_C(   612.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   553.90), SIMDE_FLOAT32_C(  1131.94), SIMDE_FLOAT32_C(   870.90), SIMDE_FLOAT32_C(   532.68),
                         SIMDE_FLOAT32_C(  -719.74), SIMDE_FLOAT32_C(  -498.42), SIMDE_FLOAT32_C(  -732.59), SIMDE_FLOAT32_C(  1082.01),
                         SIMDE_FLOAT32_C( -1113.35), SIMDE_FLOAT32_C(  -206.86), SIMDE_FLOAT32_C(    17.86), SIMDE_FLOAT32_C(   102.34),
                         SIMDE_FLOAT32_C( -1921.62), SIMDE_FLOAT32_C(  1752.09), SIMDE_FLOAT32_C(   821.40), SIMDE_FLOAT32_C(   -32.91)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_add_ps(test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_add_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   557.98), SIMDE_FLOAT32_C(   404.14), SIMDE_FLOAT32_C(   968.07), SIMDE_FLOAT32_C(  -451.16),
                         SIMDE_FLOAT32_C(  -116.99), SIMDE_FLOAT32_C(   214.59), SIMDE_FLOAT32_C(  -333.29), SIMDE_FLOAT32_C(   608.46),
                         SIMDE_FLOAT32_C(   772.88), SIMDE_FLOAT32_C(  -440.03), SIMDE_FLOAT32_C(   243.69), SIMDE_FLOAT32_C(  -905.66),
                         SIMDE_FLOAT32_C(   260.94), SIMDE_FLOAT32_C(  -593.10), SIMDE_FLOAT32_C(  -410.75), SIMDE_FLOAT32_C(  -170.16)),
      UINT16_C(51799),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -590.26), SIMDE_FLOAT32_C(   794.83), SIMDE_FLOAT32_C(  -989.05), SIMDE_FLOAT32_C(   101.49),
                         SIMDE_FLOAT32_C(   609.53), SIMDE_FLOAT32_C(   155.60), SIMDE_FLOAT32_C(  -166.25), SIMDE_FLOAT32_C(    -7.45),
                         SIMDE_FLOAT32_C(   102.33), SIMDE_FLOAT32_C(   823.12), SIMDE_FLOAT32_C(   163.08), SIMDE_FLOAT32_C(   639.77),
                         SIMDE_FLOAT32_C(   813.84), SIMDE_FLOAT32_C(   480.56), SIMDE_FLOAT32_C(  -422.93), SIMDE_FLOAT32_C(    56.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   261.65), SIMDE_FLOAT32_C(  -764.02), SIMDE_FLOAT32_C(  -249.96), SIMDE_FLOAT32_C(   370.72),
                         SIMDE_FLOAT32_C(    78.51), SIMDE_FLOAT32_C(    80.26), SIMDE_FLOAT32_C(   -82.96), SIMDE_FLOAT32_C(   867.79),
                         SIMDE_FLOAT32_C(  -823.01), SIMDE_FLOAT32_C(  -319.25), SIMDE_FLOAT32_C(   448.69), SIMDE_FLOAT32_C(  -371.67),
                         SIMDE_FLOAT32_C(   638.27), SIMDE_FLOAT32_C(   554.13), SIMDE_FLOAT32_C(   337.82), SIMDE_FLOAT32_C(  -178.59)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -328.61), SIMDE_FLOAT32_C(    30.81), SIMDE_FLOAT32_C(   968.07), SIMDE_FLOAT32_C(  -451.16),
                         SIMDE_FLOAT32_C(   688.04), SIMDE_FLOAT32_C(   214.59), SIMDE_FLOAT32_C(  -249.21), SIMDE_FLOAT32_C(   608.46),
                         SIMDE_FLOAT32_C(   772.88), SIMDE_FLOAT32_C(   503.87), SIMDE_FLOAT32_C(   243.69), SIMDE_FLOAT32_C(   268.10),
                         SIMDE_FLOAT32_C(   260.94), SIMDE_FLOAT32_C(  1034.69), SIMDE_FLOAT32_C(   -85.11), SIMDE_FLOAT32_C(  -121.76)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -849.69), SIMDE_FLOAT32_C(   124.13), SIMDE_FLOAT32_C(  -834.25), SIMDE_FLOAT32_C(  -766.99),
                         SIMDE_FLOAT32_C(   846.34), SIMDE_FLOAT32_C(  -810.48), SIMDE_FLOAT32_C(   170.13), SIMDE_FLOAT32_C(   286.67),
                         SIMDE_FLOAT32_C(   399.20), SIMDE_FLOAT32_C(  -660.46), SIMDE_FLOAT32_C(  -393.35), SIMDE_FLOAT32_C(   -15.45),
                         SIMDE_FLOAT32_C(  -296.44), SIMDE_FLOAT32_C(  -107.60), SIMDE_FLOAT32_C(  -814.83), SIMDE_FLOAT32_C(    23.81)),
      UINT16_C(51681),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -471.21), SIMDE_FLOAT32_C(  -226.98), SIMDE_FLOAT32_C(   209.46), SIMDE_FLOAT32_C(   255.36),
                         SIMDE_FLOAT32_C(  -684.09), SIMDE_FLOAT32_C(  -914.75), SIMDE_FLOAT32_C(    18.85), SIMDE_FLOAT32_C(  -966.01),
                         SIMDE_FLOAT32_C(   164.84), SIMDE_FLOAT32_C(  -687.95), SIMDE_FLOAT32_C(  -425.28), SIMDE_FLOAT32_C(  -345.75),
                         SIMDE_FLOAT32_C(  -780.44), SIMDE_FLOAT32_C(   106.15), SIMDE_FLOAT32_C(   466.24), SIMDE_FLOAT32_C(   817.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   198.58), SIMDE_FLOAT32_C(  -727.15), SIMDE_FLOAT32_C(  -572.56), SIMDE_FLOAT32_C(   958.22),
                         SIMDE_FLOAT32_C(  -583.34), SIMDE_FLOAT32_C(  -572.90), SIMDE_FLOAT32_C(  -412.07), SIMDE_FLOAT32_C(   -82.80),
                         SIMDE_FLOAT32_C(   397.00), SIMDE_FLOAT32_C(  -981.98), SIMDE_FLOAT32_C(  -278.55), SIMDE_FLOAT32_C(   589.00),
                         SIMDE_FLOAT32_C(   425.90), SIMDE_FLOAT32_C(  -258.68), SIMDE_FLOAT32_C(   721.69), SIMDE_FLOAT32_C(  -818.05)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -272.63), SIMDE_FLOAT32_C(  -954.13), SIMDE_FLOAT32_C(  -834.25), SIMDE_FLOAT32_C(  -766.99),
                         SIMDE_FLOAT32_C( -1267.43), SIMDE_FLOAT32_C(  -810.48), SIMDE_FLOAT32_C(   170.13), SIMDE_FLOAT32_C( -1048.81),
                         SIMDE_FLOAT32_C(   561.84), SIMDE_FLOAT32_C( -1669.93), SIMDE_FLOAT32_C(  -703.83), SIMDE_FLOAT32_C(   -15.45),
                         SIMDE_FLOAT32_C(  -296.44), SIMDE_FLOAT32_C(  -107.60), SIMDE_FLOAT32_C(  -814.83), SIMDE_FLOAT32_C(    -0.17)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -351.47), SIMDE_FLOAT32_C(  -458.05), SIMDE_FLOAT32_C(   323.83), SIMDE_FLOAT32_C(   403.02),
                         SIMDE_FLOAT32_C(  -394.35), SIMDE_FLOAT32_C(   401.52), SIMDE_FLOAT32_C(  -510.95), SIMDE_FLOAT32_C(    87.65),
                         SIMDE_FLOAT32_C(   739.56), SIMDE_FLOAT32_C(   256.40), SIMDE_FLOAT32_C(  -452.91), SIMDE_FLOAT32_C(   954.90),
                         SIMDE_FLOAT32_C(   329.05), SIMDE_FLOAT32_C(   425.37), SIMDE_FLOAT32_C(   772.30), SIMDE_FLOAT32_C(   986.03)),
      UINT16_C(47069),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -738.20), SIMDE_FLOAT32_C(  -911.10), SIMDE_FLOAT32_C(   452.62), SIMDE_FLOAT32_C(  -129.27),
                         SIMDE_FLOAT32_C(    88.83), SIMDE_FLOAT32_C(  -872.00), SIMDE_FLOAT32_C(   121.78), SIMDE_FLOAT32_C(   569.46),
                         SIMDE_FLOAT32_C(  -776.77), SIMDE_FLOAT32_C(  -318.66), SIMDE_FLOAT32_C(  -780.09), SIMDE_FLOAT32_C(   134.45),
                         SIMDE_FLOAT32_C(  -186.98), SIMDE_FLOAT32_C(   -12.82), SIMDE_FLOAT32_C(   903.42), SIMDE_FLOAT32_C(   403.71)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   411.50), SIMDE_FLOAT32_C(   353.32), SIMDE_FLOAT32_C(  -838.36), SIMDE_FLOAT32_C(   -87.15),
                         SIMDE_FLOAT32_C(  -255.41), SIMDE_FLOAT32_C(   806.16), SIMDE_FLOAT32_C(  -235.19), SIMDE_FLOAT32_C(   280.99),
                         SIMDE_FLOAT32_C(   436.98), SIMDE_FLOAT32_C(   399.45), SIMDE_FLOAT32_C(  -470.66), SIMDE_FLOAT32_C(   590.02),
                         SIMDE_FLOAT32_C(   629.08), SIMDE_FLOAT32_C(   852.68), SIMDE_FLOAT32_C(   716.96), SIMDE_FLOAT32_C(   753.05)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -326.70), SIMDE_FLOAT32_C(  -458.05), SIMDE_FLOAT32_C(  -385.74), SIMDE_FLOAT32_C(  -216.42),
                         SIMDE_FLOAT32_C(  -394.35), SIMDE_FLOAT32_C(   -65.84), SIMDE_FLOAT32_C(  -113.41), SIMDE_FLOAT32_C(   850.45),
                         SIMDE_FLOAT32_C(  -339.79), SIMDE_FLOAT32_C(    80.79), SIMDE_FLOAT32_C(  -452.91), SIMDE_FLOAT32_C(   724.47),
                         SIMDE_FLOAT32_C(   442.10), SIMDE_FLOAT32_C(   839.86), SIMDE_FLOAT32_C(   772.30), SIMDE_FLOAT32_C(  1156.76)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   211.10), SIMDE_FLOAT32_C(  -476.23), SIMDE_FLOAT32_C(    36.89), SIMDE_FLOAT32_C(   282.23),
                         SIMDE_FLOAT32_C(  -873.12), SIMDE_FLOAT32_C(   953.41), SIMDE_FLOAT32_C(  -223.52), SIMDE_FLOAT32_C(  -415.95),
                         SIMDE_FLOAT32_C(   155.81), SIMDE_FLOAT32_C(   307.32), SIMDE_FLOAT32_C(   666.88), SIMDE_FLOAT32_C(  -457.25),
                         SIMDE_FLOAT32_C(   849.41), SIMDE_FLOAT32_C(  -463.83), SIMDE_FLOAT32_C(  -732.49), SIMDE_FLOAT32_C(    76.79)),
      UINT16_C( 5418),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     4.53), SIMDE_FLOAT32_C(  -948.03), SIMDE_FLOAT32_C(   -25.30), SIMDE_FLOAT32_C(   729.02),
                         SIMDE_FLOAT32_C(  -851.67), SIMDE_FLOAT32_C(   496.23), SIMDE_FLOAT32_C(   475.87), SIMDE_FLOAT32_C(  -676.08),
                         SIMDE_FLOAT32_C(   917.14), SIMDE_FLOAT32_C(   745.39), SIMDE_FLOAT32_C(   262.89), SIMDE_FLOAT32_C(   617.55),
                         SIMDE_FLOAT32_C(   783.42), SIMDE_FLOAT32_C(   209.57), SIMDE_FLOAT32_C(   948.11), SIMDE_FLOAT32_C(   403.77)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   197.34), SIMDE_FLOAT32_C(  -759.25), SIMDE_FLOAT32_C(  -777.57), SIMDE_FLOAT32_C(   153.13),
                         SIMDE_FLOAT32_C(   183.29), SIMDE_FLOAT32_C(   919.64), SIMDE_FLOAT32_C(   480.71), SIMDE_FLOAT32_C(  -475.71),
                         SIMDE_FLOAT32_C(   222.85), SIMDE_FLOAT32_C(  -922.12), SIMDE_FLOAT32_C(  -597.55), SIMDE_FLOAT32_C(  -982.05),
                         SIMDE_FLOAT32_C(   677.93), SIMDE_FLOAT32_C(   190.15), SIMDE_FLOAT32_C(  -510.00), SIMDE_FLOAT32_C(  -766.01)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   211.10), SIMDE_FLOAT32_C(  -476.23), SIMDE_FLOAT32_C(    36.89), SIMDE_FLOAT32_C(   882.15),
                         SIMDE_FLOAT32_C(  -873.12), SIMDE_FLOAT32_C(  1415.87), SIMDE_FLOAT32_C(  -223.52), SIMDE_FLOAT32_C( -1151.79),
                         SIMDE_FLOAT32_C(   155.81), SIMDE_FLOAT32_C(   307.32), SIMDE_FLOAT32_C(  -334.66), SIMDE_FLOAT32_C(  -457.25),
                         SIMDE_FLOAT32_C(  1461.35), SIMDE_FLOAT32_C(  -463.83), SIMDE_FLOAT32_C(   438.11), SIMDE_FLOAT32_C(    76.79)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -249.67), SIMDE_FLOAT32_C(  -897.47), SIMDE_FLOAT32_C(  -473.64), SIMDE_FLOAT32_C(   840.96),
                         SIMDE_FLOAT32_C(   -73.68), SIMDE_FLOAT32_C(   374.70), SIMDE_FLOAT32_C(   528.22), SIMDE_FLOAT32_C(    75.60),
                         SIMDE_FLOAT32_C(   568.77), SIMDE_FLOAT32_C(  -444.74), SIMDE_FLOAT32_C(  -491.45), SIMDE_FLOAT32_C(   872.98),
                         SIMDE_FLOAT32_C(  -158.91), SIMDE_FLOAT32_C(   -47.74), SIMDE_FLOAT32_C(   903.43), SIMDE_FLOAT32_C(  -212.44)),
      UINT16_C(16900),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -787.62), SIMDE_FLOAT32_C(  -412.85), SIMDE_FLOAT32_C(   -51.77), SIMDE_FLOAT32_C(   566.91),
                         SIMDE_FLOAT32_C(  -879.62), SIMDE_FLOAT32_C(   173.18), SIMDE_FLOAT32_C(  -207.73), SIMDE_FLOAT32_C(  -643.72),
                         SIMDE_FLOAT32_C(   -98.26), SIMDE_FLOAT32_C(  -724.30), SIMDE_FLOAT32_C(   568.36), SIMDE_FLOAT32_C(    57.62),
                         SIMDE_FLOAT32_C(  -815.82), SIMDE_FLOAT32_C(  -992.69), SIMDE_FLOAT32_C(   -74.78), SIMDE_FLOAT32_C(  -258.08)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -650.83), SIMDE_FLOAT32_C(  -106.58), SIMDE_FLOAT32_C(   671.74), SIMDE_FLOAT32_C(  -960.36),
                         SIMDE_FLOAT32_C(   371.25), SIMDE_FLOAT32_C(   324.30), SIMDE_FLOAT32_C(  -751.58), SIMDE_FLOAT32_C(   862.30),
                         SIMDE_FLOAT32_C(   814.59), SIMDE_FLOAT32_C(  -210.90), SIMDE_FLOAT32_C(  -609.97), SIMDE_FLOAT32_C(  -428.72),
                         SIMDE_FLOAT32_C(   617.25), SIMDE_FLOAT32_C(   611.05), SIMDE_FLOAT32_C(   762.53), SIMDE_FLOAT32_C(  -215.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -249.67), SIMDE_FLOAT32_C(  -519.43), SIMDE_FLOAT32_C(  -473.64), SIMDE_FLOAT32_C(   840.96),
                         SIMDE_FLOAT32_C(   -73.68), SIMDE_FLOAT32_C(   374.70), SIMDE_FLOAT32_C(  -959.31), SIMDE_FLOAT32_C(    75.60),
                         SIMDE_FLOAT32_C(   568.77), SIMDE_FLOAT32_C(  -444.74), SIMDE_FLOAT32_C(  -491.45), SIMDE_FLOAT32_C(   872.98),
                         SIMDE_FLOAT32_C(  -158.91), SIMDE_FLOAT32_C(  -381.64), SIMDE_FLOAT32_C(   903.43), SIMDE_FLOAT32_C(  -212.44)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   172.10), SIMDE_FLOAT32_C(  -400.24), SIMDE_FLOAT32_C(   -80.73), SIMDE_FLOAT32_C(  -802.90),
                         SIMDE_FLOAT32_C(   788.26), SIMDE_FLOAT32_C(  -133.42), SIMDE_FLOAT32_C(  -677.20), SIMDE_FLOAT32_C(   968.74),
                         SIMDE_FLOAT32_C(  -552.54), SIMDE_FLOAT32_C(  -525.44), SIMDE_FLOAT32_C(  -502.94), SIMDE_FLOAT32_C(   682.34),
                         SIMDE_FLOAT32_C(  -987.04), SIMDE_FLOAT32_C(  -328.55), SIMDE_FLOAT32_C(   352.53), SIMDE_FLOAT32_C(    84.20)),
      UINT16_C(60310),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   804.16), SIMDE_FLOAT32_C(  -985.95), SIMDE_FLOAT32_C(   -55.15), SIMDE_FLOAT32_C(   867.34),
                         SIMDE_FLOAT32_C(  -639.05), SIMDE_FLOAT32_C(   674.69), SIMDE_FLOAT32_C(   442.10), SIMDE_FLOAT32_C(  -161.46),
                         SIMDE_FLOAT32_C(  -700.88), SIMDE_FLOAT32_C(  -300.43), SIMDE_FLOAT32_C(   402.11), SIMDE_FLOAT32_C(   662.20),
                         SIMDE_FLOAT32_C(  -715.71), SIMDE_FLOAT32_C(  -861.88), SIMDE_FLOAT32_C(   313.27), SIMDE_FLOAT32_C(   186.21)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   110.93), SIMDE_FLOAT32_C(   956.95), SIMDE_FLOAT32_C(   130.15), SIMDE_FLOAT32_C(  -962.06),
                         SIMDE_FLOAT32_C(  -309.45), SIMDE_FLOAT32_C(     0.91), SIMDE_FLOAT32_C(  -902.65), SIMDE_FLOAT32_C(   203.32),
                         SIMDE_FLOAT32_C(  -292.75), SIMDE_FLOAT32_C(   550.08), SIMDE_FLOAT32_C(   285.46), SIMDE_FLOAT32_C(   579.75),
                         SIMDE_FLOAT32_C(   615.83), SIMDE_FLOAT32_C(   762.56), SIMDE_FLOAT32_C(  -817.49), SIMDE_FLOAT32_C(   612.90)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   915.09), SIMDE_FLOAT32_C(   -29.00), SIMDE_FLOAT32_C(    75.00), SIMDE_FLOAT32_C(  -802.90),
                         SIMDE_FLOAT32_C(  -948.50), SIMDE_FLOAT32_C(  -133.42), SIMDE_FLOAT32_C(  -460.55), SIMDE_FLOAT32_C(    41.86),
                         SIMDE_FLOAT32_C(  -993.63), SIMDE_FLOAT32_C(  -525.44), SIMDE_FLOAT32_C(  -502.94), SIMDE_FLOAT32_C(  1241.95),
                         SIMDE_FLOAT32_C(  -987.04), SIMDE_FLOAT32_C(   -99.32), SIMDE_FLOAT32_C(  -504.22), SIMDE_FLOAT32_C(    84.20)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   503.91), SIMDE_FLOAT32_C(  -873.94), SIMDE_FLOAT32_C(   337.32), SIMDE_FLOAT32_C(   499.14),
                         SIMDE_FLOAT32_C(   855.15), SIMDE_FLOAT32_C(   536.35), SIMDE_FLOAT32_C(   834.00), SIMDE_FLOAT32_C(  -489.00),
                         SIMDE_FLOAT32_C(  -567.88), SIMDE_FLOAT32_C(  -606.97), SIMDE_FLOAT32_C(  -750.15), SIMDE_FLOAT32_C(  -109.88),
                         SIMDE_FLOAT32_C(  -172.79), SIMDE_FLOAT32_C(  -844.59), SIMDE_FLOAT32_C(  -574.59), SIMDE_FLOAT32_C(  -412.82)),
      UINT16_C(27856),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -789.26), SIMDE_FLOAT32_C(  -737.03), SIMDE_FLOAT32_C(   -16.72), SIMDE_FLOAT32_C(  -653.52),
                         SIMDE_FLOAT32_C(   379.28), SIMDE_FLOAT32_C(   144.82), SIMDE_FLOAT32_C(   131.43), SIMDE_FLOAT32_C(   706.79),
                         SIMDE_FLOAT32_C(  -160.32), SIMDE_FLOAT32_C(  -424.39), SIMDE_FLOAT32_C(   860.70), SIMDE_FLOAT32_C(   683.13),
                         SIMDE_FLOAT32_C(   326.88), SIMDE_FLOAT32_C(   313.08), SIMDE_FLOAT32_C(   277.84), SIMDE_FLOAT32_C(   937.22)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   525.85), SIMDE_FLOAT32_C(  -249.55), SIMDE_FLOAT32_C(  -186.57), SIMDE_FLOAT32_C(  -490.97),
                         SIMDE_FLOAT32_C(   598.23), SIMDE_FLOAT32_C(  -180.59), SIMDE_FLOAT32_C(  -202.66), SIMDE_FLOAT32_C(  -263.06),
                         SIMDE_FLOAT32_C(   624.42), SIMDE_FLOAT32_C(   918.35), SIMDE_FLOAT32_C(  -564.93), SIMDE_FLOAT32_C(   225.45),
                         SIMDE_FLOAT32_C(  -546.57), SIMDE_FLOAT32_C(   105.21), SIMDE_FLOAT32_C(  -133.31), SIMDE_FLOAT32_C(   309.82)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   503.91), SIMDE_FLOAT32_C(  -986.58), SIMDE_FLOAT32_C(  -203.29), SIMDE_FLOAT32_C(   499.14),
                         SIMDE_FLOAT32_C(   977.51), SIMDE_FLOAT32_C(   -35.77), SIMDE_FLOAT32_C(   834.00), SIMDE_FLOAT32_C(  -489.00),
                         SIMDE_FLOAT32_C(   464.10), SIMDE_FLOAT32_C(   493.96), SIMDE_FLOAT32_C(  -750.15), SIMDE_FLOAT32_C(   908.58),
                         SIMDE_FLOAT32_C(  -172.79), SIMDE_FLOAT32_C(  -844.59), SIMDE_FLOAT32_C(  -574.59), SIMDE_FLOAT32_C(  -412.82)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -458.84), SIMDE_FLOAT32_C(   406.47), SIMDE_FLOAT32_C(   200.33), SIMDE_FLOAT32_C(   823.79),
                         SIMDE_FLOAT32_C(  -438.51), SIMDE_FLOAT32_C(  -529.89), SIMDE_FLOAT32_C(   455.41), SIMDE_FLOAT32_C(   580.20),
                         SIMDE_FLOAT32_C(  -954.41), SIMDE_FLOAT32_C(    17.86), SIMDE_FLOAT32_C(  -891.74), SIMDE_FLOAT32_C(   508.74),
                         SIMDE_FLOAT32_C(  -534.50), SIMDE_FLOAT32_C(  -411.55), SIMDE_FLOAT32_C(   713.63), SIMDE_FLOAT32_C(  -448.28)),
      UINT16_C(65489),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -484.78), SIMDE_FLOAT32_C(   693.64), SIMDE_FLOAT32_C(   408.79), SIMDE_FLOAT32_C(   445.05),
                         SIMDE_FLOAT32_C(  -899.70), SIMDE_FLOAT32_C(   910.34), SIMDE_FLOAT32_C(   902.93), SIMDE_FLOAT32_C(   777.68),
                         SIMDE_FLOAT32_C(  -872.77), SIMDE_FLOAT32_C(  -343.13), SIMDE_FLOAT32_C(   660.18), SIMDE_FLOAT32_C(   570.95),
                         SIMDE_FLOAT32_C(   162.53), SIMDE_FLOAT32_C(   918.07), SIMDE_FLOAT32_C(  -439.67), SIMDE_FLOAT32_C(  -816.39)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    16.65), SIMDE_FLOAT32_C(   690.13), SIMDE_FLOAT32_C(   121.38), SIMDE_FLOAT32_C(  -516.12),
                         SIMDE_FLOAT32_C(   545.92), SIMDE_FLOAT32_C(   177.28), SIMDE_FLOAT32_C(   874.60), SIMDE_FLOAT32_C(  -233.35),
                         SIMDE_FLOAT32_C(   249.40), SIMDE_FLOAT32_C(   234.84), SIMDE_FLOAT32_C(  -806.64), SIMDE_FLOAT32_C(   246.44),
                         SIMDE_FLOAT32_C(   498.89), SIMDE_FLOAT32_C(   681.09), SIMDE_FLOAT32_C(   -83.81), SIMDE_FLOAT32_C(   642.03)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -468.13), SIMDE_FLOAT32_C(  1383.77), SIMDE_FLOAT32_C(   530.17), SIMDE_FLOAT32_C(   -71.07),
                         SIMDE_FLOAT32_C(  -353.78), SIMDE_FLOAT32_C(  1087.62), SIMDE_FLOAT32_C(  1777.53), SIMDE_FLOAT32_C(   544.33),
                         SIMDE_FLOAT32_C(  -623.37), SIMDE_FLOAT32_C(  -108.29), SIMDE_FLOAT32_C(  -891.74), SIMDE_FLOAT32_C(   817.39),
                         SIMDE_FLOAT32_C(  -534.50), SIMDE_FLOAT32_C(  -411.55), SIMDE_FLOAT32_C(   713.63), SIMDE_FLOAT32_C(  -174.36)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_add_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_add_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(34070),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   246.93), SIMDE_FLOAT32_C(  -811.02), SIMDE_FLOAT32_C(  -535.93), SIMDE_FLOAT32_C(    52.87),
                         SIMDE_FLOAT32_C(   716.30), SIMDE_FLOAT32_C(   -57.56), SIMDE_FLOAT32_C(   690.12), SIMDE_FLOAT32_C(   612.14),
                         SIMDE_FLOAT32_C(  -901.32), SIMDE_FLOAT32_C(  -189.54), SIMDE_FLOAT32_C(  -745.51), SIMDE_FLOAT32_C(   809.22),
                         SIMDE_FLOAT32_C(  -299.66), SIMDE_FLOAT32_C(  -708.58), SIMDE_FLOAT32_C(   -58.16), SIMDE_FLOAT32_C(  -769.76)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   402.24), SIMDE_FLOAT32_C(  -172.29), SIMDE_FLOAT32_C(  -380.01), SIMDE_FLOAT32_C(   418.98),
                         SIMDE_FLOAT32_C(   344.71), SIMDE_FLOAT32_C(   676.78), SIMDE_FLOAT32_C(    66.56), SIMDE_FLOAT32_C(  -447.80),
                         SIMDE_FLOAT32_C(   599.35), SIMDE_FLOAT32_C(  -768.42), SIMDE_FLOAT32_C(   985.66), SIMDE_FLOAT32_C(   -59.68),
                         SIMDE_FLOAT32_C(    11.93), SIMDE_FLOAT32_C(  -836.08), SIMDE_FLOAT32_C(  -821.64), SIMDE_FLOAT32_C(   485.10)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   649.17), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   619.22), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   164.34),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   749.54),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1544.66), SIMDE_FLOAT32_C(  -879.80), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(18016),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   114.20), SIMDE_FLOAT32_C(  -250.12), SIMDE_FLOAT32_C(   205.39), SIMDE_FLOAT32_C(   377.39),
                         SIMDE_FLOAT32_C(  -678.55), SIMDE_FLOAT32_C(  -144.87), SIMDE_FLOAT32_C(   953.96), SIMDE_FLOAT32_C(  -727.31),
                         SIMDE_FLOAT32_C(   975.05), SIMDE_FLOAT32_C(   861.66), SIMDE_FLOAT32_C(  -283.79), SIMDE_FLOAT32_C(   652.52),
                         SIMDE_FLOAT32_C(   985.26), SIMDE_FLOAT32_C(   333.03), SIMDE_FLOAT32_C(   -22.71), SIMDE_FLOAT32_C(   372.76)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   496.03), SIMDE_FLOAT32_C(  -128.10), SIMDE_FLOAT32_C(  -607.57), SIMDE_FLOAT32_C(  -934.57),
                         SIMDE_FLOAT32_C(  -200.59), SIMDE_FLOAT32_C(   673.22), SIMDE_FLOAT32_C(  -890.32), SIMDE_FLOAT32_C(  -261.35),
                         SIMDE_FLOAT32_C(  -308.29), SIMDE_FLOAT32_C(   674.57), SIMDE_FLOAT32_C(  -542.67), SIMDE_FLOAT32_C(  -748.61),
                         SIMDE_FLOAT32_C(  -986.94), SIMDE_FLOAT32_C(  -251.25), SIMDE_FLOAT32_C(   573.60), SIMDE_FLOAT32_C(   467.61)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -378.22), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.35), SIMDE_FLOAT32_C(    63.64), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1536.23), SIMDE_FLOAT32_C(  -826.46), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(25399),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -937.90), SIMDE_FLOAT32_C(   699.05), SIMDE_FLOAT32_C(  -954.51), SIMDE_FLOAT32_C(  -401.06),
                         SIMDE_FLOAT32_C(  -200.18), SIMDE_FLOAT32_C(  -193.03), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   707.44),
                         SIMDE_FLOAT32_C(   127.93), SIMDE_FLOAT32_C(   450.22), SIMDE_FLOAT32_C(   849.81), SIMDE_FLOAT32_C(   562.63),
                         SIMDE_FLOAT32_C(   683.37), SIMDE_FLOAT32_C(   -87.77), SIMDE_FLOAT32_C(   258.02), SIMDE_FLOAT32_C(  -449.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   600.08), SIMDE_FLOAT32_C(   -51.70), SIMDE_FLOAT32_C(   432.78), SIMDE_FLOAT32_C(  -805.77),
                         SIMDE_FLOAT32_C(   471.47), SIMDE_FLOAT32_C(   -62.60), SIMDE_FLOAT32_C(    50.16), SIMDE_FLOAT32_C(  -968.73),
                         SIMDE_FLOAT32_C(   980.37), SIMDE_FLOAT32_C(  -379.93), SIMDE_FLOAT32_C(   482.79), SIMDE_FLOAT32_C(  -839.61),
                         SIMDE_FLOAT32_C(    85.66), SIMDE_FLOAT32_C(  -276.42), SIMDE_FLOAT32_C(   192.13), SIMDE_FLOAT32_C(   751.61)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   647.35), SIMDE_FLOAT32_C(  -521.73), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   179.94), SIMDE_FLOAT32_C(  -261.29),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1332.60), SIMDE_FLOAT32_C(  -276.98),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -364.19), SIMDE_FLOAT32_C(   450.15), SIMDE_FLOAT32_C(   302.07)) },
    { UINT16_C(21570),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   174.39), SIMDE_FLOAT32_C(   649.64), SIMDE_FLOAT32_C(  -467.57), SIMDE_FLOAT32_C(   -13.46),
                         SIMDE_FLOAT32_C(    79.96), SIMDE_FLOAT32_C(   870.09), SIMDE_FLOAT32_C(    78.23), SIMDE_FLOAT32_C(   308.33),
                         SIMDE_FLOAT32_C(   969.57), SIMDE_FLOAT32_C(   129.15), SIMDE_FLOAT32_C(   641.22), SIMDE_FLOAT32_C(  -816.81),
                         SIMDE_FLOAT32_C(   851.66), SIMDE_FLOAT32_C(  -569.60), SIMDE_FLOAT32_C(   774.11), SIMDE_FLOAT32_C(  -977.84)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   839.72), SIMDE_FLOAT32_C(  -192.47), SIMDE_FLOAT32_C(   441.45), SIMDE_FLOAT32_C(   894.27),
                         SIMDE_FLOAT32_C(   987.76), SIMDE_FLOAT32_C(  -728.83), SIMDE_FLOAT32_C(   287.58), SIMDE_FLOAT32_C(   825.22),
                         SIMDE_FLOAT32_C(   500.81), SIMDE_FLOAT32_C(  -857.90), SIMDE_FLOAT32_C(    13.30), SIMDE_FLOAT32_C(  -963.46),
                         SIMDE_FLOAT32_C(   789.18), SIMDE_FLOAT32_C(  -421.72), SIMDE_FLOAT32_C(  -456.20), SIMDE_FLOAT32_C(  -500.23)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   457.17), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   880.81),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   141.26), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -728.75), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   317.91), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C( 4527),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   819.62), SIMDE_FLOAT32_C(   143.91), SIMDE_FLOAT32_C(  -476.13), SIMDE_FLOAT32_C(  -330.97),
                         SIMDE_FLOAT32_C(  -314.47), SIMDE_FLOAT32_C(   327.51), SIMDE_FLOAT32_C(    79.69), SIMDE_FLOAT32_C(  -335.16),
                         SIMDE_FLOAT32_C(    79.63), SIMDE_FLOAT32_C(  -316.03), SIMDE_FLOAT32_C(   529.05), SIMDE_FLOAT32_C(  -783.26),
                         SIMDE_FLOAT32_C(  -836.10), SIMDE_FLOAT32_C(   799.12), SIMDE_FLOAT32_C(   202.01), SIMDE_FLOAT32_C(  -973.13)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -474.84), SIMDE_FLOAT32_C(  -831.90), SIMDE_FLOAT32_C(   175.04), SIMDE_FLOAT32_C(  -769.21),
                         SIMDE_FLOAT32_C(  -761.87), SIMDE_FLOAT32_C(   711.09), SIMDE_FLOAT32_C(    76.03), SIMDE_FLOAT32_C(   -69.55),
                         SIMDE_FLOAT32_C(   206.89), SIMDE_FLOAT32_C(   385.48), SIMDE_FLOAT32_C(  -149.33), SIMDE_FLOAT32_C(    63.43),
                         SIMDE_FLOAT32_C(  -965.34), SIMDE_FLOAT32_C(   927.69), SIMDE_FLOAT32_C(   926.44), SIMDE_FLOAT32_C(   751.96)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1100.18),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -404.71),
                         SIMDE_FLOAT32_C(   286.52), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   379.72), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C( -1801.44), SIMDE_FLOAT32_C(  1726.81), SIMDE_FLOAT32_C(  1128.45), SIMDE_FLOAT32_C(  -221.17)) },
    { UINT16_C(49892),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   588.00), SIMDE_FLOAT32_C(   584.07), SIMDE_FLOAT32_C(   -62.26), SIMDE_FLOAT32_C(   917.49),
                         SIMDE_FLOAT32_C(  -873.42), SIMDE_FLOAT32_C(    91.96), SIMDE_FLOAT32_C(   575.44), SIMDE_FLOAT32_C(  -110.07),
                         SIMDE_FLOAT32_C(    -5.94), SIMDE_FLOAT32_C(   192.99), SIMDE_FLOAT32_C(   492.46), SIMDE_FLOAT32_C(  -775.27),
                         SIMDE_FLOAT32_C(  -913.08), SIMDE_FLOAT32_C(   683.73), SIMDE_FLOAT32_C(  -915.46), SIMDE_FLOAT32_C(   653.29)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   472.29), SIMDE_FLOAT32_C(  -575.09), SIMDE_FLOAT32_C(   980.71), SIMDE_FLOAT32_C(  -559.07),
                         SIMDE_FLOAT32_C(  -664.03), SIMDE_FLOAT32_C(   847.80), SIMDE_FLOAT32_C(   783.26), SIMDE_FLOAT32_C(  -881.55),
                         SIMDE_FLOAT32_C(   743.44), SIMDE_FLOAT32_C(   605.57), SIMDE_FLOAT32_C(   127.37), SIMDE_FLOAT32_C(   178.47),
                         SIMDE_FLOAT32_C(  -843.05), SIMDE_FLOAT32_C(  -829.69), SIMDE_FLOAT32_C(  -306.45), SIMDE_FLOAT32_C(  -410.55)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  1060.29), SIMDE_FLOAT32_C(     8.98), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1358.70), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   737.50), SIMDE_FLOAT32_C(   798.56), SIMDE_FLOAT32_C(   619.83), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -145.96), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(35395),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -584.83), SIMDE_FLOAT32_C(  -991.02), SIMDE_FLOAT32_C(    87.38), SIMDE_FLOAT32_C(  -660.51),
                         SIMDE_FLOAT32_C(   -61.08), SIMDE_FLOAT32_C(  -942.66), SIMDE_FLOAT32_C(    98.55), SIMDE_FLOAT32_C(   717.20),
                         SIMDE_FLOAT32_C(  -375.96), SIMDE_FLOAT32_C(   280.62), SIMDE_FLOAT32_C(   892.45), SIMDE_FLOAT32_C(  -164.87),
                         SIMDE_FLOAT32_C(   725.16), SIMDE_FLOAT32_C(    92.34), SIMDE_FLOAT32_C(  -444.56), SIMDE_FLOAT32_C(  -205.95)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -331.69), SIMDE_FLOAT32_C(    56.82), SIMDE_FLOAT32_C(   293.40), SIMDE_FLOAT32_C(   559.39),
                         SIMDE_FLOAT32_C(  -542.40), SIMDE_FLOAT32_C(  -727.09), SIMDE_FLOAT32_C(    42.45), SIMDE_FLOAT32_C(   836.37),
                         SIMDE_FLOAT32_C(   263.78), SIMDE_FLOAT32_C(  -509.50), SIMDE_FLOAT32_C(   313.35), SIMDE_FLOAT32_C(   590.05),
                         SIMDE_FLOAT32_C(  -852.39), SIMDE_FLOAT32_C(  -876.91), SIMDE_FLOAT32_C(  -994.58), SIMDE_FLOAT32_C(  -351.76)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -916.52), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -603.48), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   141.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -228.88), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1439.14), SIMDE_FLOAT32_C(  -557.71)) },
    { UINT16_C(52006),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -56.05), SIMDE_FLOAT32_C(  -627.04), SIMDE_FLOAT32_C(   887.74), SIMDE_FLOAT32_C(  -166.02),
                         SIMDE_FLOAT32_C(    36.14), SIMDE_FLOAT32_C(   814.59), SIMDE_FLOAT32_C(   740.46), SIMDE_FLOAT32_C(   588.03),
                         SIMDE_FLOAT32_C(   469.76), SIMDE_FLOAT32_C(  -514.54), SIMDE_FLOAT32_C(   206.17), SIMDE_FLOAT32_C(   302.47),
                         SIMDE_FLOAT32_C(   208.18), SIMDE_FLOAT32_C(   121.91), SIMDE_FLOAT32_C(   499.72), SIMDE_FLOAT32_C(   376.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -963.34), SIMDE_FLOAT32_C(   -94.89), SIMDE_FLOAT32_C(   732.15), SIMDE_FLOAT32_C(  -520.87),
                         SIMDE_FLOAT32_C(   254.02), SIMDE_FLOAT32_C(  -604.67), SIMDE_FLOAT32_C(    -6.37), SIMDE_FLOAT32_C(   745.91),
                         SIMDE_FLOAT32_C(  -401.58), SIMDE_FLOAT32_C(  -429.08), SIMDE_FLOAT32_C(   -97.98), SIMDE_FLOAT32_C(    35.37),
                         SIMDE_FLOAT32_C(  -379.34), SIMDE_FLOAT32_C(  -588.87), SIMDE_FLOAT32_C(  -792.60), SIMDE_FLOAT32_C(   657.97)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C( -1019.39), SIMDE_FLOAT32_C(  -721.93), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   290.16), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   734.09), SIMDE_FLOAT32_C(  1333.94),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   108.19), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -466.96), SIMDE_FLOAT32_C(  -292.88), SIMDE_FLOAT32_C(     0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_add_ps(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_add_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -236.16), SIMDE_FLOAT64_C(   20.07),
                         SIMDE_FLOAT64_C(  257.66), SIMDE_FLOAT64_C(  863.96),
                         SIMDE_FLOAT64_C(  593.27), SIMDE_FLOAT64_C( -503.87),
                         SIMDE_FLOAT64_C( -362.00), SIMDE_FLOAT64_C(  273.73)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  196.72), SIMDE_FLOAT64_C(   63.78),
                         SIMDE_FLOAT64_C(  794.12), SIMDE_FLOAT64_C(  -29.20),
                         SIMDE_FLOAT64_C( -795.86), SIMDE_FLOAT64_C( -299.89),
                         SIMDE_FLOAT64_C(  346.27), SIMDE_FLOAT64_C(  203.60)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -39.44), SIMDE_FLOAT64_C(   83.85),
                         SIMDE_FLOAT64_C( 1051.78), SIMDE_FLOAT64_C(  834.76),
                         SIMDE_FLOAT64_C( -202.59), SIMDE_FLOAT64_C( -803.76),
                         SIMDE_FLOAT64_C(  -15.73), SIMDE_FLOAT64_C(  477.33)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  770.84), SIMDE_FLOAT64_C( -754.36),
                         SIMDE_FLOAT64_C(  737.96), SIMDE_FLOAT64_C( -529.83),
                         SIMDE_FLOAT64_C( -355.96), SIMDE_FLOAT64_C( -623.94),
                         SIMDE_FLOAT64_C(  550.26), SIMDE_FLOAT64_C(  883.81)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -87.42), SIMDE_FLOAT64_C( -548.47),
                         SIMDE_FLOAT64_C( -633.24), SIMDE_FLOAT64_C(  450.21),
                         SIMDE_FLOAT64_C(  645.87), SIMDE_FLOAT64_C( -366.47),
                         SIMDE_FLOAT64_C(  695.56), SIMDE_FLOAT64_C(  617.08)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  683.42), SIMDE_FLOAT64_C(-1302.83),
                         SIMDE_FLOAT64_C(  104.72), SIMDE_FLOAT64_C(  -79.62),
                         SIMDE_FLOAT64_C(  289.91), SIMDE_FLOAT64_C( -990.41),
                         SIMDE_FLOAT64_C( 1245.82), SIMDE_FLOAT64_C( 1500.89)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -504.50), SIMDE_FLOAT64_C( -227.85),
                         SIMDE_FLOAT64_C( -966.51), SIMDE_FLOAT64_C( -410.18),
                         SIMDE_FLOAT64_C(  -80.72), SIMDE_FLOAT64_C( -882.81),
                         SIMDE_FLOAT64_C( -139.97), SIMDE_FLOAT64_C(  617.51)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -598.19), SIMDE_FLOAT64_C(  737.33),
                         SIMDE_FLOAT64_C(  488.15), SIMDE_FLOAT64_C(  218.70),
                         SIMDE_FLOAT64_C(  134.86), SIMDE_FLOAT64_C( -311.92),
                         SIMDE_FLOAT64_C(  725.05), SIMDE_FLOAT64_C(  687.14)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-1102.69), SIMDE_FLOAT64_C(  509.48),
                         SIMDE_FLOAT64_C( -478.36), SIMDE_FLOAT64_C( -191.48),
                         SIMDE_FLOAT64_C(   54.14), SIMDE_FLOAT64_C(-1194.73),
                         SIMDE_FLOAT64_C(  585.08), SIMDE_FLOAT64_C( 1304.65)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  472.12), SIMDE_FLOAT64_C( -365.98),
                         SIMDE_FLOAT64_C(  659.31), SIMDE_FLOAT64_C( -290.87),
                         SIMDE_FLOAT64_C( -373.51), SIMDE_FLOAT64_C( -669.26),
                         SIMDE_FLOAT64_C( -520.33), SIMDE_FLOAT64_C(  675.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  635.51), SIMDE_FLOAT64_C( -125.87),
                         SIMDE_FLOAT64_C( -802.87), SIMDE_FLOAT64_C(  655.14),
                         SIMDE_FLOAT64_C(  196.92), SIMDE_FLOAT64_C(  789.13),
                         SIMDE_FLOAT64_C(  116.39), SIMDE_FLOAT64_C( -455.20)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( 1107.63), SIMDE_FLOAT64_C( -491.85),
                         SIMDE_FLOAT64_C( -143.56), SIMDE_FLOAT64_C(  364.27),
                         SIMDE_FLOAT64_C( -176.59), SIMDE_FLOAT64_C(  119.87),
                         SIMDE_FLOAT64_C( -403.94), SIMDE_FLOAT64_C(  220.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  788.90), SIMDE_FLOAT64_C( -954.97),
                         SIMDE_FLOAT64_C(  125.34), SIMDE_FLOAT64_C(  169.40),
                         SIMDE_FLOAT64_C(  224.26), SIMDE_FLOAT64_C( -169.19),
                         SIMDE_FLOAT64_C(  154.36), SIMDE_FLOAT64_C( -652.64)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -435.33), SIMDE_FLOAT64_C( -537.34),
                         SIMDE_FLOAT64_C( -563.73), SIMDE_FLOAT64_C(  279.31),
                         SIMDE_FLOAT64_C( -443.50), SIMDE_FLOAT64_C(   -9.90),
                         SIMDE_FLOAT64_C(  404.97), SIMDE_FLOAT64_C(  991.70)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  353.57), SIMDE_FLOAT64_C(-1492.31),
                         SIMDE_FLOAT64_C( -438.39), SIMDE_FLOAT64_C(  448.71),
                         SIMDE_FLOAT64_C( -219.24), SIMDE_FLOAT64_C( -179.09),
                         SIMDE_FLOAT64_C(  559.33), SIMDE_FLOAT64_C(  339.06)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -409.86), SIMDE_FLOAT64_C( -564.70),
                         SIMDE_FLOAT64_C(  978.74), SIMDE_FLOAT64_C( -670.81),
                         SIMDE_FLOAT64_C( -437.36), SIMDE_FLOAT64_C( -920.10),
                         SIMDE_FLOAT64_C(   26.94), SIMDE_FLOAT64_C(  254.30)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   88.55), SIMDE_FLOAT64_C( -559.38),
                         SIMDE_FLOAT64_C( -420.16), SIMDE_FLOAT64_C(  439.30),
                         SIMDE_FLOAT64_C( -160.71), SIMDE_FLOAT64_C(  -56.82),
                         SIMDE_FLOAT64_C(   12.24), SIMDE_FLOAT64_C(  276.77)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -321.31), SIMDE_FLOAT64_C(-1124.08),
                         SIMDE_FLOAT64_C(  558.58), SIMDE_FLOAT64_C( -231.51),
                         SIMDE_FLOAT64_C( -598.07), SIMDE_FLOAT64_C( -976.92),
                         SIMDE_FLOAT64_C(   39.18), SIMDE_FLOAT64_C(  531.07)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -252.56), SIMDE_FLOAT64_C( -715.99),
                         SIMDE_FLOAT64_C( -694.53), SIMDE_FLOAT64_C(  556.76),
                         SIMDE_FLOAT64_C(  839.65), SIMDE_FLOAT64_C( -743.60),
                         SIMDE_FLOAT64_C( -343.44), SIMDE_FLOAT64_C( -212.93)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -450.77), SIMDE_FLOAT64_C(  107.13),
                         SIMDE_FLOAT64_C(  937.09), SIMDE_FLOAT64_C(  938.48),
                         SIMDE_FLOAT64_C( -943.33), SIMDE_FLOAT64_C( -298.38),
                         SIMDE_FLOAT64_C(  752.15), SIMDE_FLOAT64_C( -313.15)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -703.33), SIMDE_FLOAT64_C( -608.86),
                         SIMDE_FLOAT64_C(  242.56), SIMDE_FLOAT64_C( 1495.24),
                         SIMDE_FLOAT64_C( -103.68), SIMDE_FLOAT64_C(-1041.98),
                         SIMDE_FLOAT64_C(  408.71), SIMDE_FLOAT64_C( -526.08)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  940.55), SIMDE_FLOAT64_C(  673.86),
                         SIMDE_FLOAT64_C( -765.42), SIMDE_FLOAT64_C( -188.42),
                         SIMDE_FLOAT64_C( -980.28), SIMDE_FLOAT64_C( -223.74),
                         SIMDE_FLOAT64_C(  -19.52), SIMDE_FLOAT64_C( -256.77)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  128.08), SIMDE_FLOAT64_C(  862.66),
                         SIMDE_FLOAT64_C(  151.47), SIMDE_FLOAT64_C(  825.63),
                         SIMDE_FLOAT64_C( -245.57), SIMDE_FLOAT64_C( -364.98),
                         SIMDE_FLOAT64_C(  -60.78), SIMDE_FLOAT64_C(  224.87)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( 1068.63), SIMDE_FLOAT64_C( 1536.52),
                         SIMDE_FLOAT64_C( -613.95), SIMDE_FLOAT64_C(  637.21),
                         SIMDE_FLOAT64_C(-1225.85), SIMDE_FLOAT64_C( -588.72),
                         SIMDE_FLOAT64_C(  -80.30), SIMDE_FLOAT64_C(  -31.90)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_add_pd(test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_add_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  772.88), SIMDE_FLOAT64_C( -440.03),
                         SIMDE_FLOAT64_C(  243.69), SIMDE_FLOAT64_C( -905.66),
                         SIMDE_FLOAT64_C(  260.94), SIMDE_FLOAT64_C( -593.10),
                         SIMDE_FLOAT64_C( -410.75), SIMDE_FLOAT64_C( -170.16)),
      UINT8_C(131),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  102.33), SIMDE_FLOAT64_C(  823.12),
                         SIMDE_FLOAT64_C(  163.08), SIMDE_FLOAT64_C(  639.77),
                         SIMDE_FLOAT64_C(  813.84), SIMDE_FLOAT64_C(  480.56),
                         SIMDE_FLOAT64_C( -422.93), SIMDE_FLOAT64_C(   56.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -823.01), SIMDE_FLOAT64_C( -319.25),
                         SIMDE_FLOAT64_C(  448.69), SIMDE_FLOAT64_C( -371.67),
                         SIMDE_FLOAT64_C(  638.27), SIMDE_FLOAT64_C(  554.13),
                         SIMDE_FLOAT64_C(  337.82), SIMDE_FLOAT64_C( -178.59)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -720.68), SIMDE_FLOAT64_C( -440.03),
                         SIMDE_FLOAT64_C(  243.69), SIMDE_FLOAT64_C( -905.66),
                         SIMDE_FLOAT64_C(  260.94), SIMDE_FLOAT64_C( -593.10),
                         SIMDE_FLOAT64_C(  -85.11), SIMDE_FLOAT64_C( -121.76)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -590.26), SIMDE_FLOAT64_C(  794.83),
                         SIMDE_FLOAT64_C( -989.05), SIMDE_FLOAT64_C(  101.49),
                         SIMDE_FLOAT64_C(  609.53), SIMDE_FLOAT64_C(  155.60),
                         SIMDE_FLOAT64_C( -166.25), SIMDE_FLOAT64_C(   -7.45)),
      UINT8_C(166),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  261.65), SIMDE_FLOAT64_C( -764.02),
                         SIMDE_FLOAT64_C( -249.96), SIMDE_FLOAT64_C(  370.72),
                         SIMDE_FLOAT64_C(   78.51), SIMDE_FLOAT64_C(   80.26),
                         SIMDE_FLOAT64_C(  -82.96), SIMDE_FLOAT64_C(  867.79)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -572.12), SIMDE_FLOAT64_C(  557.98),
                         SIMDE_FLOAT64_C(  404.14), SIMDE_FLOAT64_C(  968.07),
                         SIMDE_FLOAT64_C( -451.16), SIMDE_FLOAT64_C( -116.99),
                         SIMDE_FLOAT64_C(  214.59), SIMDE_FLOAT64_C( -333.29)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -310.47), SIMDE_FLOAT64_C(  794.83),
                         SIMDE_FLOAT64_C(  154.18), SIMDE_FLOAT64_C(  101.49),
                         SIMDE_FLOAT64_C(  609.53), SIMDE_FLOAT64_C(  -36.73),
                         SIMDE_FLOAT64_C(  131.63), SIMDE_FLOAT64_C(   -7.45)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  399.20), SIMDE_FLOAT64_C( -660.46),
                         SIMDE_FLOAT64_C( -393.35), SIMDE_FLOAT64_C(  -15.45),
                         SIMDE_FLOAT64_C( -296.44), SIMDE_FLOAT64_C( -107.60),
                         SIMDE_FLOAT64_C( -814.83), SIMDE_FLOAT64_C(   23.81)),
      UINT8_C( 90),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  164.84), SIMDE_FLOAT64_C( -687.95),
                         SIMDE_FLOAT64_C( -425.28), SIMDE_FLOAT64_C( -345.75),
                         SIMDE_FLOAT64_C( -780.44), SIMDE_FLOAT64_C(  106.15),
                         SIMDE_FLOAT64_C(  466.24), SIMDE_FLOAT64_C(  817.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  397.00), SIMDE_FLOAT64_C( -981.98),
                         SIMDE_FLOAT64_C( -278.55), SIMDE_FLOAT64_C(  589.00),
                         SIMDE_FLOAT64_C(  425.90), SIMDE_FLOAT64_C( -258.68),
                         SIMDE_FLOAT64_C(  721.69), SIMDE_FLOAT64_C( -818.05)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  399.20), SIMDE_FLOAT64_C(-1669.93),
                         SIMDE_FLOAT64_C( -393.35), SIMDE_FLOAT64_C(  243.25),
                         SIMDE_FLOAT64_C( -354.54), SIMDE_FLOAT64_C( -107.60),
                         SIMDE_FLOAT64_C( 1187.93), SIMDE_FLOAT64_C(   23.81)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -471.21), SIMDE_FLOAT64_C( -226.98),
                         SIMDE_FLOAT64_C(  209.46), SIMDE_FLOAT64_C(  255.36),
                         SIMDE_FLOAT64_C( -684.09), SIMDE_FLOAT64_C( -914.75),
                         SIMDE_FLOAT64_C(   18.85), SIMDE_FLOAT64_C( -966.01)),
      UINT8_C(215),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  198.58), SIMDE_FLOAT64_C( -727.15),
                         SIMDE_FLOAT64_C( -572.56), SIMDE_FLOAT64_C(  958.22),
                         SIMDE_FLOAT64_C( -583.34), SIMDE_FLOAT64_C( -572.90),
                         SIMDE_FLOAT64_C( -412.07), SIMDE_FLOAT64_C(  -82.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -331.15), SIMDE_FLOAT64_C( -849.69),
                         SIMDE_FLOAT64_C(  124.13), SIMDE_FLOAT64_C( -834.25),
                         SIMDE_FLOAT64_C( -766.99), SIMDE_FLOAT64_C(  846.34),
                         SIMDE_FLOAT64_C( -810.48), SIMDE_FLOAT64_C(  170.13)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -132.57), SIMDE_FLOAT64_C(-1576.84),
                         SIMDE_FLOAT64_C(  209.46), SIMDE_FLOAT64_C(  123.97),
                         SIMDE_FLOAT64_C( -684.09), SIMDE_FLOAT64_C(  273.44),
                         SIMDE_FLOAT64_C(-1222.55), SIMDE_FLOAT64_C(   87.33)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  739.56), SIMDE_FLOAT64_C(  256.40),
                         SIMDE_FLOAT64_C( -452.91), SIMDE_FLOAT64_C(  954.90),
                         SIMDE_FLOAT64_C(  329.05), SIMDE_FLOAT64_C(  425.37),
                         SIMDE_FLOAT64_C(  772.30), SIMDE_FLOAT64_C(  986.03)),
      UINT8_C( 53),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -776.77), SIMDE_FLOAT64_C( -318.66),
                         SIMDE_FLOAT64_C( -780.09), SIMDE_FLOAT64_C(  134.45),
                         SIMDE_FLOAT64_C( -186.98), SIMDE_FLOAT64_C(  -12.82),
                         SIMDE_FLOAT64_C(  903.42), SIMDE_FLOAT64_C(  403.71)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  436.98), SIMDE_FLOAT64_C(  399.45),
                         SIMDE_FLOAT64_C( -470.66), SIMDE_FLOAT64_C(  590.02),
                         SIMDE_FLOAT64_C(  629.08), SIMDE_FLOAT64_C(  852.68),
                         SIMDE_FLOAT64_C(  716.96), SIMDE_FLOAT64_C(  753.05)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  739.56), SIMDE_FLOAT64_C(  256.40),
                         SIMDE_FLOAT64_C(-1250.75), SIMDE_FLOAT64_C(  724.47),
                         SIMDE_FLOAT64_C(  329.05), SIMDE_FLOAT64_C(  839.86),
                         SIMDE_FLOAT64_C(  772.30), SIMDE_FLOAT64_C( 1156.76)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -738.20), SIMDE_FLOAT64_C( -911.10),
                         SIMDE_FLOAT64_C(  452.62), SIMDE_FLOAT64_C( -129.27),
                         SIMDE_FLOAT64_C(   88.83), SIMDE_FLOAT64_C( -872.00),
                         SIMDE_FLOAT64_C(  121.78), SIMDE_FLOAT64_C(  569.46)),
      UINT8_C(203),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  411.50), SIMDE_FLOAT64_C(  353.32),
                         SIMDE_FLOAT64_C( -838.36), SIMDE_FLOAT64_C(  -87.15),
                         SIMDE_FLOAT64_C( -255.41), SIMDE_FLOAT64_C(  806.16),
                         SIMDE_FLOAT64_C( -235.19), SIMDE_FLOAT64_C(  280.99)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  872.61), SIMDE_FLOAT64_C( -351.47),
                         SIMDE_FLOAT64_C( -458.05), SIMDE_FLOAT64_C(  323.83),
                         SIMDE_FLOAT64_C(  403.02), SIMDE_FLOAT64_C( -394.35),
                         SIMDE_FLOAT64_C(  401.52), SIMDE_FLOAT64_C( -510.95)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( 1284.11), SIMDE_FLOAT64_C(    1.85),
                         SIMDE_FLOAT64_C(  452.62), SIMDE_FLOAT64_C( -129.27),
                         SIMDE_FLOAT64_C(  147.61), SIMDE_FLOAT64_C( -872.00),
                         SIMDE_FLOAT64_C(  166.33), SIMDE_FLOAT64_C( -229.96)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  155.81), SIMDE_FLOAT64_C(  307.32),
                         SIMDE_FLOAT64_C(  666.88), SIMDE_FLOAT64_C( -457.25),
                         SIMDE_FLOAT64_C(  849.41), SIMDE_FLOAT64_C( -463.83),
                         SIMDE_FLOAT64_C( -732.49), SIMDE_FLOAT64_C(   76.79)),
      UINT8_C(200),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  917.14), SIMDE_FLOAT64_C(  745.39),
                         SIMDE_FLOAT64_C(  262.89), SIMDE_FLOAT64_C(  617.55),
                         SIMDE_FLOAT64_C(  783.42), SIMDE_FLOAT64_C(  209.57),
                         SIMDE_FLOAT64_C(  948.11), SIMDE_FLOAT64_C(  403.77)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  222.85), SIMDE_FLOAT64_C( -922.12),
                         SIMDE_FLOAT64_C( -597.55), SIMDE_FLOAT64_C( -982.05),
                         SIMDE_FLOAT64_C(  677.93), SIMDE_FLOAT64_C(  190.15),
                         SIMDE_FLOAT64_C( -510.00), SIMDE_FLOAT64_C( -766.01)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( 1139.99), SIMDE_FLOAT64_C( -176.73),
                         SIMDE_FLOAT64_C(  666.88), SIMDE_FLOAT64_C( -457.25),
                         SIMDE_FLOAT64_C( 1461.35), SIMDE_FLOAT64_C( -463.83),
                         SIMDE_FLOAT64_C( -732.49), SIMDE_FLOAT64_C(   76.79)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(    4.53), SIMDE_FLOAT64_C( -948.03),
                         SIMDE_FLOAT64_C(  -25.30), SIMDE_FLOAT64_C(  729.02),
                         SIMDE_FLOAT64_C( -851.67), SIMDE_FLOAT64_C(  496.23),
                         SIMDE_FLOAT64_C(  475.87), SIMDE_FLOAT64_C( -676.08)),
      UINT8_C(148),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  197.34), SIMDE_FLOAT64_C( -759.25),
                         SIMDE_FLOAT64_C( -777.57), SIMDE_FLOAT64_C(  153.13),
                         SIMDE_FLOAT64_C(  183.29), SIMDE_FLOAT64_C(  919.64),
                         SIMDE_FLOAT64_C(  480.71), SIMDE_FLOAT64_C( -475.71)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -886.75), SIMDE_FLOAT64_C(  211.10),
                         SIMDE_FLOAT64_C( -476.23), SIMDE_FLOAT64_C(   36.89),
                         SIMDE_FLOAT64_C(  282.23), SIMDE_FLOAT64_C( -873.12),
                         SIMDE_FLOAT64_C(  953.41), SIMDE_FLOAT64_C( -223.52)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -689.41), SIMDE_FLOAT64_C( -948.03),
                         SIMDE_FLOAT64_C(  -25.30), SIMDE_FLOAT64_C(  190.02),
                         SIMDE_FLOAT64_C( -851.67), SIMDE_FLOAT64_C(   46.52),
                         SIMDE_FLOAT64_C(  475.87), SIMDE_FLOAT64_C( -676.08)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_add_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_add_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C(208),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -901.32), SIMDE_FLOAT64_C( -189.54),
                         SIMDE_FLOAT64_C( -745.51), SIMDE_FLOAT64_C(  809.22),
                         SIMDE_FLOAT64_C( -299.66), SIMDE_FLOAT64_C( -708.58),
                         SIMDE_FLOAT64_C(  -58.16), SIMDE_FLOAT64_C( -769.76)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  599.35), SIMDE_FLOAT64_C( -768.42),
                         SIMDE_FLOAT64_C(  985.66), SIMDE_FLOAT64_C(  -59.68),
                         SIMDE_FLOAT64_C(   11.93), SIMDE_FLOAT64_C( -836.08),
                         SIMDE_FLOAT64_C( -821.64), SIMDE_FLOAT64_C(  485.10)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -301.97), SIMDE_FLOAT64_C( -957.96),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  749.54),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(182),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  402.24), SIMDE_FLOAT64_C( -172.29),
                         SIMDE_FLOAT64_C( -380.01), SIMDE_FLOAT64_C(  418.98),
                         SIMDE_FLOAT64_C(  344.71), SIMDE_FLOAT64_C(  676.78),
                         SIMDE_FLOAT64_C(   66.56), SIMDE_FLOAT64_C( -447.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -244.06), SIMDE_FLOAT64_C(  246.93),
                         SIMDE_FLOAT64_C( -811.02), SIMDE_FLOAT64_C( -535.93),
                         SIMDE_FLOAT64_C(   52.87), SIMDE_FLOAT64_C(  716.30),
                         SIMDE_FLOAT64_C(  -57.56), SIMDE_FLOAT64_C(  690.12)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  158.18), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(-1191.03), SIMDE_FLOAT64_C( -116.95),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( 1393.08),
                         SIMDE_FLOAT64_C(    9.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(107),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  975.05), SIMDE_FLOAT64_C(  861.66),
                         SIMDE_FLOAT64_C( -283.79), SIMDE_FLOAT64_C(  652.52),
                         SIMDE_FLOAT64_C(  985.26), SIMDE_FLOAT64_C(  333.03),
                         SIMDE_FLOAT64_C(  -22.71), SIMDE_FLOAT64_C(  372.76)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -308.29), SIMDE_FLOAT64_C(  674.57),
                         SIMDE_FLOAT64_C( -542.67), SIMDE_FLOAT64_C( -748.61),
                         SIMDE_FLOAT64_C( -986.94), SIMDE_FLOAT64_C( -251.25),
                         SIMDE_FLOAT64_C(  573.60), SIMDE_FLOAT64_C(  467.61)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( 1536.23),
                         SIMDE_FLOAT64_C( -826.46), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   -1.68), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  550.89), SIMDE_FLOAT64_C(  840.37)) },
    { UINT8_C(172),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  496.03), SIMDE_FLOAT64_C( -128.10),
                         SIMDE_FLOAT64_C( -607.57), SIMDE_FLOAT64_C( -934.57),
                         SIMDE_FLOAT64_C( -200.59), SIMDE_FLOAT64_C(  673.22),
                         SIMDE_FLOAT64_C( -890.32), SIMDE_FLOAT64_C( -261.35)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -21.78), SIMDE_FLOAT64_C(  114.20),
                         SIMDE_FLOAT64_C( -250.12), SIMDE_FLOAT64_C(  205.39),
                         SIMDE_FLOAT64_C(  377.39), SIMDE_FLOAT64_C( -678.55),
                         SIMDE_FLOAT64_C( -144.87), SIMDE_FLOAT64_C(  953.96)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  474.25), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -857.69), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  176.80), SIMDE_FLOAT64_C(   -5.33),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C( 95),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  127.93), SIMDE_FLOAT64_C(  450.22),
                         SIMDE_FLOAT64_C(  849.81), SIMDE_FLOAT64_C(  562.63),
                         SIMDE_FLOAT64_C(  683.37), SIMDE_FLOAT64_C(  -87.77),
                         SIMDE_FLOAT64_C(  258.02), SIMDE_FLOAT64_C( -449.54)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  980.37), SIMDE_FLOAT64_C( -379.93),
                         SIMDE_FLOAT64_C(  482.79), SIMDE_FLOAT64_C( -839.61),
                         SIMDE_FLOAT64_C(   85.66), SIMDE_FLOAT64_C( -276.42),
                         SIMDE_FLOAT64_C(  192.13), SIMDE_FLOAT64_C(  751.61)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   70.29),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -276.98),
                         SIMDE_FLOAT64_C(  769.03), SIMDE_FLOAT64_C( -364.19),
                         SIMDE_FLOAT64_C(  450.15), SIMDE_FLOAT64_C(  302.07)) },
    { UINT8_C( 15),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  600.08), SIMDE_FLOAT64_C(  -51.70),
                         SIMDE_FLOAT64_C(  432.78), SIMDE_FLOAT64_C( -805.77),
                         SIMDE_FLOAT64_C(  471.47), SIMDE_FLOAT64_C(  -62.60),
                         SIMDE_FLOAT64_C(   50.16), SIMDE_FLOAT64_C( -968.73)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  794.87), SIMDE_FLOAT64_C( -937.90),
                         SIMDE_FLOAT64_C(  699.05), SIMDE_FLOAT64_C( -954.51),
                         SIMDE_FLOAT64_C( -401.06), SIMDE_FLOAT64_C( -200.18),
                         SIMDE_FLOAT64_C( -193.03), SIMDE_FLOAT64_C(  129.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   70.41), SIMDE_FLOAT64_C( -262.78),
                         SIMDE_FLOAT64_C( -142.87), SIMDE_FLOAT64_C( -838.95)) },
    { UINT8_C(189),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  969.57), SIMDE_FLOAT64_C(  129.15),
                         SIMDE_FLOAT64_C(  641.22), SIMDE_FLOAT64_C( -816.81),
                         SIMDE_FLOAT64_C(  851.66), SIMDE_FLOAT64_C( -569.60),
                         SIMDE_FLOAT64_C(  774.11), SIMDE_FLOAT64_C( -977.84)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  500.81), SIMDE_FLOAT64_C( -857.90),
                         SIMDE_FLOAT64_C(   13.30), SIMDE_FLOAT64_C( -963.46),
                         SIMDE_FLOAT64_C(  789.18), SIMDE_FLOAT64_C( -421.72),
                         SIMDE_FLOAT64_C( -456.20), SIMDE_FLOAT64_C( -500.23)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( 1470.38), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  654.52), SIMDE_FLOAT64_C(-1780.27),
                         SIMDE_FLOAT64_C( 1640.84), SIMDE_FLOAT64_C( -991.32),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(-1478.07)) },
    { UINT8_C(148),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  839.72), SIMDE_FLOAT64_C( -192.47),
                         SIMDE_FLOAT64_C(  441.45), SIMDE_FLOAT64_C(  894.27),
                         SIMDE_FLOAT64_C(  987.76), SIMDE_FLOAT64_C( -728.83),
                         SIMDE_FLOAT64_C(  287.58), SIMDE_FLOAT64_C(  825.22)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  395.09), SIMDE_FLOAT64_C(  174.39),
                         SIMDE_FLOAT64_C(  649.64), SIMDE_FLOAT64_C( -467.57),
                         SIMDE_FLOAT64_C(  -13.46), SIMDE_FLOAT64_C(   79.96),
                         SIMDE_FLOAT64_C(  870.09), SIMDE_FLOAT64_C(   78.23)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( 1234.81), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  426.70),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -648.87),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_add_pd(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_and_si512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1762603276), INT32_C(-1316946536), INT32_C( -409636803), INT32_C(-1096492450),
                            INT32_C( 1487241173), INT32_C(-1940071138), INT32_C( 1116126146), INT32_C( -916337722),
                            INT32_C(   52488417), INT32_C( 1044081507), INT32_C(-1035184013), INT32_C(-1384518181),
                            INT32_C(-1687535487), INT32_C(-1003450758), INT32_C(  131212491), INT32_C(-1389162000)),
      simde_mm512_set_epi32(INT32_C(-1226890085), INT32_C(  294090078), INT32_C( 1993383303), INT32_C( 2117895966),
                            INT32_C( -579088274), INT32_C( 1336927221), INT32_C(   43647960), INT32_C( 1458493074),
                            INT32_C( -558283956), INT32_C(  428402022), INT32_C(  974860288), INT32_C( -624769004),
                            INT32_C(-1594949458), INT32_C(  -55819000), INT32_C(  423209449), INT32_C( -449605661)),
      simde_mm512_set_epi32(INT32_C(  537862152), INT32_C(  293631256), INT32_C( 1720717317), INT32_C( 1042579998),
                            INT32_C( 1478578244), INT32_C(  202162452), INT32_C(   42075072), INT32_C( 1080083074),
                            INT32_C(   35668032), INT32_C(  403202402), INT32_C(   34082816), INT32_C(-2009020400),
                            INT32_C(-2140536704), INT32_C(-1004534776), INT32_C(   17834185), INT32_C(-1523383840)) },
    { simde_mm512_set_epi32(INT32_C(-1226448376), INT32_C( 1927095485), INT32_C( 1303264402), INT32_C(  926300607),
                            INT32_C( 2116494628), INT32_C(-1100758331), INT32_C(  853434891), INT32_C( 1856581548),
                            INT32_C( 1730450085), INT32_C(-1233336711), INT32_C(  410802607), INT32_C(-1458345357),
                            INT32_C(-1414257532), INT32_C(-1908909941), INT32_C(  216004947), INT32_C(-1153616267)),
      simde_mm512_set_epi32(INT32_C(-1170743204), INT32_C( 1817377482), INT32_C(  934231558), INT32_C( -128119768),
                            INT32_C(  674694491), INT32_C( 1902405145), INT32_C(-1438313883), INT32_C( -508947384),
                            INT32_C( -334819615), INT32_C(-1499616800), INT32_C( -300643115), INT32_C(-1687918613),
                            INT32_C(  204696129), INT32_C( -790552335), INT32_C( -837803722), INT32_C(-1569322126)),
      simde_mm512_set_epi32(INT32_C(-1306140664), INT32_C( 1615863944), INT32_C(   95304706), INT32_C(  806619176),
                            INT32_C(  673645824), INT32_C(  811614721), INT32_C(  574882305), INT32_C( 1621623816),
                            INT32_C( 1677722273), INT32_C(-1508081568), INT32_C(  135530629), INT32_C(-1996463517),
                            INT32_C(  137371648), INT32_C(-2145386367), INT32_C(  202383634), INT32_C(-1573649808)) },
    { simde_mm512_set_epi32(INT32_C( -967914791), INT32_C( 1028004547), INT32_C( 1106145634), INT32_C(-2126623640),
                            INT32_C(   -6485699), INT32_C( -256904631), INT32_C(-1220204919), INT32_C(  917835787),
                            INT32_C( 1623427491), INT32_C( -222464855), INT32_C( -472879958), INT32_C( -762982604),
                            INT32_C(-1085405824), INT32_C( 1812599478), INT32_C( -613988136), INT32_C(  621554720)),
      simde_mm512_set_epi32(INT32_C( 1377438428), INT32_C( 1201451322), INT32_C(  619734582), INT32_C(-1992526637),
                            INT32_C( -868585296), INT32_C( 1439924174), INT32_C( 1656419868), INT32_C(-1390170089),
                            INT32_C( 1058692615), INT32_C( -836785520), INT32_C(-1830636694), INT32_C(  554982917),
                            INT32_C(-1514534388), INT32_C( -747673601), INT32_C( -692811287), INT32_C(-1799571639)),
      simde_mm512_set_epi32(INT32_C( 1107954392), INT32_C(   84152322), INT32_C(   14704674), INT32_C(-2126756800),
                            INT32_C( -870842320), INT32_C( 1350791752), INT32_C(  570435080), INT32_C(  606146563),
                            INT32_C(  537004035), INT32_C(-1038278528), INT32_C(-2101338070), INT32_C(     282116),
                            INT32_C(-1526071296), INT32_C( 1074397878), INT32_C( -769390392), INT32_C(   67904512)) },
    { simde_mm512_set_epi32(INT32_C(-1436950998), INT32_C(-1742059387), INT32_C( 1377677769), INT32_C(-2097193192),
                            INT32_C( 1556973207), INT32_C(   58040738), INT32_C(-1875805492), INT32_C( -452882923),
                            INT32_C(-2070651162), INT32_C(-1417594324), INT32_C( -990171302), INT32_C(  444234765),
                            INT32_C( -651701039), INT32_C( -296257488), INT32_C( 1302666953), INT32_C( 1243668562)),
      simde_mm512_set_epi32(INT32_C( -228023402), INT32_C( 1737651280), INT32_C(  890037909), INT32_C(  822465192),
                            INT32_C( 1525557148), INT32_C( 1672658803), INT32_C( 1808682106), INT32_C( 1316739447),
                            INT32_C(  903813947), INT32_C(  221590740), INT32_C( 1668581990), INT32_C(-1092503304),
                            INT32_C( 1369460064), INT32_C( 1353181098), INT32_C(  652356799), INT32_C( -684439573)),
      simde_mm512_set_epi32(INT32_C(-1572306430), INT32_C(     147456), INT32_C(  269262977), INT32_C(     348680),
                            INT32_C( 1489766036), INT32_C(   53518626), INT32_C(      18504), INT32_C( 1140951061),
                            INT32_C(   76814882), INT32_C(  151072772), INT32_C( 1081084482), INT32_C(  442513416),
                            INT32_C( 1361069120), INT32_C( 1074225184), INT32_C(   77597833), INT32_C( 1109409858)) },
    { simde_mm512_set_epi32(INT32_C(-1043054173), INT32_C( -396216896), INT32_C(-1145802326), INT32_C( -804000246),
                            INT32_C( -145399860), INT32_C( -890427310), INT32_C( -401401997), INT32_C(  802016776),
                            INT32_C( 1929893502), INT32_C(   73827769), INT32_C(-1971097644), INT32_C(-1831682098),
                            INT32_C(  546355465), INT32_C( -199725455), INT32_C(  931867413), INT32_C(-1496909535)),
      simde_mm512_set_epi32(INT32_C(-1796636811), INT32_C(-1576316556), INT32_C( 1080356179), INT32_C(-1830141457),
                            INT32_C(-1444813077), INT32_C( 1282909316), INT32_C(  814589845), INT32_C(  563073613),
                            INT32_C( -161574330), INT32_C( 1115054069), INT32_C(-1922096352), INT32_C( 1283172543),
                            INT32_C( 1028016376), INT32_C( 1652445236), INT32_C( 1602581177), INT32_C(-1986713581)),
      simde_mm512_set_epi32(INT32_C(-2134884063), INT32_C(-1610477248), INT32_C(    2384130), INT32_C(-1878900726),
                            INT32_C(-1589557560), INT32_C( 1214586880), INT32_C(  536938257), INT32_C(  562938376),
                            INT32_C( 1913033286), INT32_C(    6686129), INT32_C(-2013060352), INT32_C(    5417102),
                            INT32_C(  536872968), INT32_C( 1612202032), INT32_C(  394338833), INT32_C(-2138822655)) },
    { simde_mm512_set_epi32(INT32_C(  213329535), INT32_C( -522060385), INT32_C( -710729699), INT32_C(  911515198),
                            INT32_C(-1475915599), INT32_C(-1846311235), INT32_C(-1624654725), INT32_C( -496488954),
                            INT32_C(-2105881976), INT32_C( -863113580), INT32_C( -870973395), INT32_C(-2135017149),
                            INT32_C( 1179500895), INT32_C(  102238134), INT32_C( 1890546920), INT32_C( 1651955955)),
      simde_mm512_set_epi32(INT32_C(-1460720620), INT32_C(-1283988079), INT32_C( 2139823103), INT32_C(-2058406982),
                            INT32_C( -677653135), INT32_C(  526832430), INT32_C(  918576849), INT32_C(-1987609349),
                            INT32_C( -819905099), INT32_C( 2043707434), INT32_C( 1005516756), INT32_C(  646673888),
                            INT32_C( -792085599), INT32_C(  923333390), INT32_C(  549762390), INT32_C( 1063027034)),
      simde_mm512_set_epi32(INT32_C(  145171476), INT32_C(-1604196975), INT32_C( 1434648605), INT32_C(   71569978),
                            INT32_C(-2147266511), INT32_C(  291668524), INT32_C(  369106001), INT32_C(-2147472382),
                            INT32_C(-2111829888), INT32_C( 1216380928), INT32_C(  134610948), INT32_C(    9056576),
                            INT32_C( 1078558977), INT32_C(  101188358), INT32_C(  545535040), INT32_C(  575963218)) },
    { simde_mm512_set_epi32(INT32_C(-1614227898), INT32_C(-1072924213), INT32_C(-2048516742), INT32_C(-1735505047),
                            INT32_C(  409846045), INT32_C( -501166301), INT32_C(  385735082), INT32_C(-1379445210),
                            INT32_C( 1301699864), INT32_C( -237316746), INT32_C( -173549926), INT32_C(-1638681430),
                            INT32_C( 1204990643), INT32_C( -623938106), INT32_C(  621663116), INT32_C(-2139715294)),
      simde_mm512_set_epi32(INT32_C( 1168648208), INT32_C(  679514223), INT32_C(-1255159953), INT32_C(-2016174737),
                            INT32_C( -817087094), INT32_C( 1605116212), INT32_C(  684814447), INT32_C( 1274003485),
                            INT32_C( 1881744290), INT32_C(  579021373), INT32_C( -658206082), INT32_C( 1152351107),
                            INT32_C( -539739024), INT32_C( 1438387923), INT32_C( -569943597), INT32_C(  -79238784)),
      simde_mm512_set_epi32(INT32_C(   92798976), INT32_C(       4171), INT32_C(-2061105814), INT32_C(-2138961559),
                            INT32_C(  139198728), INT32_C( 1109393696), INT32_C(   13717546), INT32_C(  164052996),
                            INT32_C( 1073742080), INT32_C(  545390644), INT32_C( -796618214), INT32_C(   67317890),
                            INT32_C( 1204826160), INT32_C( 1351353538), INT32_C(   67457408), INT32_C(-2142861056)) },
    { simde_mm512_set_epi32(INT32_C( -593800358), INT32_C( -124181915), INT32_C( 2110561848), INT32_C( 1255401496),
                            INT32_C( -282522813), INT32_C( -286538666), INT32_C(-2011412362), INT32_C(-1839527164),
                            INT32_C(-1330408299), INT32_C( 1769934774), INT32_C( -358481155), INT32_C( -123958768),
                            INT32_C( 1676106379), INT32_C(-1305862521), INT32_C( 1797940107), INT32_C(  653525737)),
      simde_mm512_set_epi32(INT32_C(-1432835313), INT32_C( 1661538833), INT32_C( 1372337273), INT32_C(-1604084834),
                            INT32_C( -921184393), INT32_C(-1395990480), INT32_C( 1258870002), INT32_C( -947895097),
                            INT32_C(-1351881935), INT32_C(   99634026), INT32_C( 2033361976), INT32_C( 1231716550),
                            INT32_C( -228173591), INT32_C(-1552770129), INT32_C( -338049103), INT32_C(-1393391283)),
      simde_mm512_set_epi32(INT32_C(-2003303670), INT32_C( 1611203073), INT32_C( 1372329016), INT32_C(    4425752),
                            INT32_C( -922680509), INT32_C(-1395998704), INT32_C(  134758514), INT32_C(-2113929212),
                            INT32_C(-1608286191), INT32_C(   24120098), INT32_C( 1747058744), INT32_C( 1208516608),
                            INT32_C( 1650874505), INT32_C(-1574823289), INT32_C( 1795703681), INT32_C(  619708489)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_and_si512(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_andnot_si512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -335330897), INT32_C( 1860840666), INT32_C( -837102383), INT32_C( 1544121603),
                            INT32_C(  -31451516), INT32_C(  294501250), INT32_C( 1844141610), INT32_C(  711066163),
                            INT32_C( 1032767823), INT32_C(  466876164), INT32_C( 1432923079), INT32_C( -137339965),
                            INT32_C(-1216294439), INT32_C( 1110388055), INT32_C(  754234048), INT32_C( -712839683)),
      simde_mm512_set_epi32(INT32_C(  645969883), INT32_C(   45407696), INT32_C(-1431929981), INT32_C(-1744968675),
                            INT32_C( 1491740298), INT32_C( -669732847), INT32_C(-1678703719), INT32_C(-1110558488),
                            INT32_C(-1993251114), INT32_C( 1910816421), INT32_C( 2013403088), INT32_C(  882585036),
                            INT32_C( 1733706468), INT32_C( 1763057771), INT32_C(  303070795), INT32_C( -805966849)),
      simde_mm512_set_epi32(INT32_C(   41989712), INT32_C(    1364224), INT32_C(  547629826), INT32_C(-2081128420),
                            INT32_C(   13246474), INT32_C( -938475503), INT32_C(-1844444783), INT32_C(-1785972536),
                            INT32_C(-2144262512), INT32_C( 1612974241), INT32_C(  671224848), INT32_C(     730124),
                            INT32_C( 1079378468), INT32_C(  688914472), INT32_C(  302009355), INT32_C(  175440386)) },
    { simde_mm512_set_epi32(INT32_C( 1267266514), INT32_C(-1810114077), INT32_C(-1631053656), INT32_C(-1291259659),
                            INT32_C(-1797405973), INT32_C(-2052491824), INT32_C(  218690610), INT32_C(  434694077),
                            INT32_C(  322569513), INT32_C( -492306370), INT32_C( 1714124310), INT32_C(  757183592),
                            INT32_C( 1904845371), INT32_C( 1921390915), INT32_C( 1219016836), INT32_C( -491589854)),
      simde_mm512_set_epi32(INT32_C( -843887215), INT32_C(-1144045392), INT32_C( 1523671305), INT32_C( -687015924),
                            INT32_C( -651771268), INT32_C(-1812069901), INT32_C(  132880464), INT32_C( 1912329512),
                            INT32_C( -208209918), INT32_C(-1079631083), INT32_C( -134611197), INT32_C(-1062410635),
                            INT32_C( -896925558), INT32_C( -559765979), INT32_C( 1912148196), INT32_C( -437846049)),
      simde_mm512_set_epi32(INT32_C(-2077029375), INT32_C(  734265360), INT32_C( 1074880769), INT32_C( 1141114888),
                            INT32_C( 1226965012), INT32_C(  307626019), INT32_C(   48433216), INT32_C( 1611862016),
                            INT32_C( -528156670), INT32_C(  486941441), INT32_C(-1848604415), INT32_C(-1064549867),
                            INT32_C(-1979580288), INT32_C(-1944017372), INT32_C(  827392096), INT32_C(   88346845)) },
    { simde_mm512_set_epi32(INT32_C(  451034606), INT32_C(  160382101), INT32_C(-1268862602), INT32_C(  782115678),
                            INT32_C(-1160318793), INT32_C( -575355195), INT32_C( 1432838242), INT32_C(-2114154695),
                            INT32_C(-1020410376), INT32_C( -714076046), INT32_C(-1407849113), INT32_C(  996241684),
                            INT32_C(  481606881), INT32_C(-1834956523), INT32_C(  493396975), INT32_C(-1084672800)),
      simde_mm512_set_epi32(INT32_C( 1458493934), INT32_C( 1051105030), INT32_C( -836083742), INT32_C( 1407748874),
                            INT32_C(-1387312486), INT32_C(  776481471), INT32_C(  275093143), INT32_C( -137438390),
                            INT32_C( 1860284960), INT32_C(  540502552), INT32_C( 1411461258), INT32_C( 1517918194),
                            INT32_C( -266161178), INT32_C( 1269265702), INT32_C(  809771495), INT32_C(-1968711037)),
      simde_mm512_set_epi32(INT32_C( 1141686272), INT32_C(  908100354), INT32_C( 1243631232), INT32_C( 1365280768),
                            INT32_C(   84478472), INT32_C(  575154234), INT32_C(      38549), INT32_C( 1979863106),
                            INT32_C(  750793216), INT32_C(  537356808), INT32_C( 1344278664), INT32_C( 1075349218),
                            INT32_C( -536739066), INT32_C( 1225208866), INT32_C(  537133056), INT32_C(   10930691)) },
    { simde_mm512_set_epi32(INT32_C(-1562592645), INT32_C(  -32255724), INT32_C( -923416118), INT32_C(-2134713284),
                            INT32_C(-1313323965), INT32_C(-1729518909), INT32_C( 1286411285), INT32_C( -376910154),
                            INT32_C(-1786193108), INT32_C(-2035089818), INT32_C( 1552020826), INT32_C(  726998554),
                            INT32_C( 1864619074), INT32_C( 1828024315), INT32_C( -824341738), INT32_C(-1420030579)),
      simde_mm512_set_epi32(INT32_C( 1087836695), INT32_C(-2094233976), INT32_C( 1148487684), INT32_C(-1514127182),
                            INT32_C( -524459384), INT32_C(  725104708), INT32_C( 1787286694), INT32_C(-1533684832),
                            INT32_C(   46575098), INT32_C( 2086853653), INT32_C(  815292575), INT32_C(-1270435744),
                            INT32_C( 2014177347), INT32_C( 1099600134), INT32_C( -622983952), INT32_C(  822011154)),
      simde_mm512_set_epi32(INT32_C( 1073938436), INT32_C(   19662472), INT32_C(   67108868), INT32_C(  620757122),
                            INT32_C( 1074078344), INT32_C(  588257284), INT32_C(  570605730), INT32_C(   68489472),
                            INT32_C(   38151378), INT32_C( 2017460241), INT32_C(  538443909), INT32_C(-1811767200),
                            INT32_C(  269232129), INT32_C(   17469444), INT32_C(  268567776), INT32_C(  279109650)) },
    { simde_mm512_set_epi32(INT32_C(-1657115762), INT32_C( 1585840022), INT32_C(-1070898703), INT32_C( 1022031619),
                            INT32_C(-1380717315), INT32_C( 1086658406), INT32_C( -124039065), INT32_C(-1974944947),
                            INT32_C( 2044249149), INT32_C( 1638783653), INT32_C( 1466240446), INT32_C(-1803146403),
                            INT32_C( 1060682707), INT32_C(-1592428518), INT32_C(  156586666), INT32_C( -266957088)),
      simde_mm512_set_epi32(INT32_C( -703454581), INT32_C(  797686885), INT32_C( 1723425278), INT32_C( -158454369),
                            INT32_C(-1043830066), INT32_C(  709622512), INT32_C(-2136296570), INT32_C( -863350926),
                            INT32_C( 1844461284), INT32_C(  -21472306), INT32_C(-1932483198), INT32_C(-1320584016),
                            INT32_C( -370591173), INT32_C( -330170023), INT32_C( -975385097), INT32_C( -654562432)),
      simde_mm512_set_epi32(INT32_C( 1107296257), INT32_C(  554303585), INT32_C(  646971406), INT32_C(-1039923044),
                            INT32_C( 1078460930), INT32_C(  705357968), INT32_C(    2139008), INT32_C( 1149387826),
                            INT32_C(   69221056), INT32_C(-1643118262), INT32_C(-2003787776), INT32_C(  558453920),
                            INT32_C(-1061093336), INT32_C( 1279394113), INT32_C( -997683883), INT32_C(  149430528)) },
    { simde_mm512_set_epi32(INT32_C(  962558787), INT32_C(-1212292378), INT32_C(-1698562444), INT32_C(-1456708578),
                            INT32_C( 1605522258), INT32_C(-1389853810), INT32_C(  605095260), INT32_C(  449573803),
                            INT32_C(-1932095036), INT32_C( 1214045264), INT32_C(-1966228541), INT32_C(  484352026),
                            INT32_C(-1251622562), INT32_C(   97048183), INT32_C( 1801957969), INT32_C(   39148591)),
      simde_mm512_set_epi32(INT32_C( 1144673524), INT32_C(-1837539909), INT32_C(-1995926176), INT32_C( -775830454),
                            INT32_C( 1197039500), INT32_C(  605086417), INT32_C(-1681915928), INT32_C(-1694227594),
                            INT32_C(  250277648), INT32_C( 1517650405), INT32_C( -529860796), INT32_C(  319331129),
                            INT32_C( 1337610221), INT32_C( -515158609), INT32_C(-1958759875), INT32_C(  480005412)),
      simde_mm512_set_epi32(INT32_C( 1142949044), INT32_C(    4201753), INT32_C(   17301760), INT32_C( 1354858560),
                            INT32_C(    4787340), INT32_C(    1073233), INT32_C(-1683031392), INT32_C(-2130444204),
                            INT32_C(   36204048), INT32_C(  304152997), INT32_C( 1612858372), INT32_C(   50338593),
                            INT32_C( 1251610273), INT32_C( -536671864), INT32_C(-2145910740), INT32_C(  478675200)) },
    { simde_mm512_set_epi32(INT32_C(  477799556), INT32_C(  718106947), INT32_C( -702434720), INT32_C(  911156446),
                            INT32_C(  692922531), INT32_C( -634559193), INT32_C( -541024501), INT32_C(    6957260),
                            INT32_C(  891904501), INT32_C( 1674261328), INT32_C(  463285837), INT32_C(  465636281),
                            INT32_C( -567453998), INT32_C( -675807734), INT32_C( 1242869264), INT32_C(-2003535835)),
      simde_mm512_set_epi32(INT32_C( -440269466), INT32_C( 1069561863), INT32_C( -850138274), INT32_C( 1324108467),
                            INT32_C(  996083706), INT32_C(-1741332408), INT32_C(-1720688024), INT32_C( -195389802),
                            INT32_C( -122163269), INT32_C(-1678986062), INT32_C( -261742027), INT32_C(  147621305),
                            INT32_C( 1928957095), INT32_C(  647911914), INT32_C(-1231783784), INT32_C(-1597793099)),
      simde_mm512_set_epi32(INT32_C( -511704734), INT32_C(  352323588), INT32_C(  156387614), INT32_C( 1218464289),
                            INT32_C(  303171416), INT32_C(    1048648), INT32_C(    3170400), INT32_C( -200239598),
                            INT32_C( -929657334), INT32_C(-1742437214), INT32_C( -530448336), INT32_C(     820224),
                            INT32_C(  550537253), INT32_C(  537395680), INT32_C(-1266659192), INT32_C(  541295760)) },
    { simde_mm512_set_epi32(INT32_C(-1322452749), INT32_C(-1191485380), INT32_C(   61071601), INT32_C( -255981709),
                            INT32_C( 1745472557), INT32_C( 1521357726), INT32_C(-1111842070), INT32_C( 1783291089),
                            INT32_C(  718609371), INT32_C( -553071779), INT32_C(-1373014967), INT32_C(  751334079),
                            INT32_C( -828271800), INT32_C(-1578484948), INT32_C(-1597074675), INT32_C(  393018558)),
      simde_mm512_set_epi32(INT32_C(-1722624236), INT32_C( -955857282), INT32_C( 1790216473), INT32_C( -762838785),
                            INT32_C( -108799681), INT32_C( -975838651), INT32_C( 1961237228), INT32_C(   52752901),
                            INT32_C(-1440122977), INT32_C(-1167835972), INT32_C( 1345250484), INT32_C( 2101674065),
                            INT32_C( -149671798), INT32_C(  738167968), INT32_C( -764040824), INT32_C( -514982245)),
      simde_mm512_set_epi32(INT32_C(  139593476), INT32_C( 1191478850), INT32_C( 1746143496), INT32_C(   33554572),
                            INT32_C(-1853746926), INT32_C(-2058231743), INT32_C( 1078204420), INT32_C(   19145220),
                            INT32_C(-2144836604), INT32_C(  543437984), INT32_C( 1342603444), INT32_C( 1359282240),
                            INT32_C(  823402626), INT32_C(  169182336), INT32_C( 1378953344), INT32_C( -536804863)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_andnot_si512(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_andnot_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1786505147), INT32_C(  366806262), INT32_C(-1595474360), INT32_C( -741125130),
                            INT32_C(  623580589), INT32_C( 1819639708), INT32_C(-1998267151), INT32_C(   54696203),
                            INT32_C( 1230356730), INT32_C( -528215990), INT32_C(-1085976265), INT32_C(  -88891472),
                            INT32_C(  263402075), INT32_C( 2072408294), INT32_C( 1041613232), INT32_C(-1299496488)),
      simde_mm512_set_epi32(INT32_C(  634085978), INT32_C( 1880231468), INT32_C( 1457044755), INT32_C( -852481645),
                            INT32_C( 1274177455), INT32_C( -223885439), INT32_C(  582318482), INT32_C( 1515067636),
                            INT32_C(-1348943060), INT32_C(   88850487), INT32_C(-1067534176), INT32_C( 1770437803),
                            INT32_C(-1608407464), INT32_C(-1335624696), INT32_C( 1637100454), INT32_C(-1783030263)),
      simde_mm512_set_epi32(INT32_C(   92274752), INT32_C( 1610746376), INT32_C( 1444413715), INT32_C(  203433985),
                            INT32_C( 1255170562), INT32_C(-1836941311), INT32_C(  571547906), INT32_C( 1477181684),
                            INT32_C(-1501035772), INT32_C(   88850485), INT32_C( 1075488896), INT32_C(   17047563),
                            INT32_C(-1610578944), INT32_C(-2141090808), INT32_C( 1098918406), INT32_C(   87039489)) },
    { simde_mm512_set_epi32(INT32_C(  923576423), INT32_C(-1078925154), INT32_C( -430557576), INT32_C(-1684773193),
                            INT32_C(-1179101215), INT32_C(-1985153431), INT32_C(  584718967), INT32_C( -112765469),
                            INT32_C( 1515864234), INT32_C(-1118210252), INT32_C(  931438007), INT32_C( -352031421),
                            INT32_C( 1134370188), INT32_C( 1556623900), INT32_C(   57329867), INT32_C(  254759017)),
      simde_mm512_set_epi32(INT32_C( -300442460), INT32_C( -893266841), INT32_C(-1015236925), INT32_C(  262163323),
                            INT32_C( 2095940386), INT32_C( 1009617335), INT32_C(  458760718), INT32_C(-1732442867),
                            INT32_C( 1273945161), INT32_C(  946706622), INT32_C( 1469023509), INT32_C(-2064451876),
                            INT32_C( -256697390), INT32_C(-1934774398), INT32_C(  433298181), INT32_C( -530351918)),
      simde_mm512_set_epi32(INT32_C( -938273664), INT32_C( 1078006881), INT32_C(   19431555), INT32_C(   69208392),
                            INT32_C( 1145405442), INT32_C(  872483222), INT32_C(  419438600), INT32_C(   12058636),
                            INT32_C(   27565121), INT32_C(    2394250), INT32_C( 1074462720), INT32_C(   83006108),
                            INT32_C(-1339948462), INT32_C(-2145022590), INT32_C(  412160772), INT32_C( -532666222)) },
    { simde_mm512_set_epi32(INT32_C(  835311518), INT32_C(  593132209), INT32_C(-1205845883), INT32_C(-2103435972),
                            INT32_C(  331121937), INT32_C(-1122763027), INT32_C(  -11044623), INT32_C( 1217358106),
                            INT32_C(  899389553), INT32_C(   61750829), INT32_C(-1644418892), INT32_C( 1179256254),
                            INT32_C( -236468269), INT32_C( -666751062), INT32_C( -733547571), INT32_C( 2125570021)),
      simde_mm512_set_epi32(INT32_C(-1443754597), INT32_C( 1972174992), INT32_C(-2074962423), INT32_C( -531291976),
                            INT32_C( 1382830722), INT32_C( -282269267), INT32_C( 1453780297), INT32_C(  363272438),
                            INT32_C( 1819778130), INT32_C(-1488646809), INT32_C( 1000774887), INT32_C( 2075973242),
                            INT32_C(  251762527), INT32_C(  254090322), INT32_C( -106442053), INT32_C(-1147166459)),
      simde_mm512_set_epi32(INT32_C(-2009987071), INT32_C( 1418002432), INT32_C(   72521224), INT32_C( 1616191616),
                            INT32_C( 1077957250), INT32_C( 1110180096), INT32_C(   10519816), INT32_C(  354423012),
                            INT32_C( 1214514178), INT32_C(-1539243710), INT32_C(  570594371), INT32_C(  968151616),
                            INT32_C(  234885132), INT32_C(  119869520), INT32_C(  698417202), INT32_C(-2129780736)) },
    { simde_mm512_set_epi32(INT32_C( 1259282838), INT32_C( -167567006), INT32_C( 1470440257), INT32_C(-1702928569),
                            INT32_C(-1493129242), INT32_C( -361616020), INT32_C( 1148861436), INT32_C(-2140586026),
                            INT32_C(-1901343726), INT32_C( 1258604211), INT32_C( 1382183555), INT32_C(  464481172),
                            INT32_C(   87817013), INT32_C(  -25672201), INT32_C(-1647580547), INT32_C( -833959607)),
      simde_mm512_set_epi32(INT32_C( -711482206), INT32_C(-1110405208), INT32_C(  -55795162), INT32_C(-1789106875),
                            INT32_C(-1077987504), INT32_C( 2002242576), INT32_C(  879044440), INT32_C(  728498187),
                            INT32_C( -580810324), INT32_C(-1054241155), INT32_C(  416673383), INT32_C( 1924176623),
                            INT32_C( 1323235160), INT32_C(  659292758), INT32_C(-2101310960), INT32_C( 1303315999)),
      simde_mm512_set_epi32(INT32_C(-1802468320), INT32_C(  164662920), INT32_C(-1475837914), INT32_C(   83894272),
                            INT32_C(  415174672), INT32_C(  352698384), INT32_C(  805635072), INT32_C(  721682441),
                            INT32_C( 1363149228), INT32_C(-2144794548), INT32_C(  144018532), INT32_C( 1611698283),
                            INT32_C( 1254360136), INT32_C(   17039872), INT32_C(   33554432), INT32_C(   27592214)) },
    { simde_mm512_set_epi32(INT32_C( 1317706320), INT32_C( 1095937634), INT32_C(-2042379654), INT32_C( -425062813),
                            INT32_C(-1422676870), INT32_C(-1972727484), INT32_C( 1448617643), INT32_C( 1446030445),
                            INT32_C(-1203372071), INT32_C( 1257548767), INT32_C(   95515950), INT32_C(  288075556),
                            INT32_C( -562902724), INT32_C( 1866018725), INT32_C( -140491543), INT32_C( -853598261)),
      simde_mm512_set_epi32(INT32_C(-1862602245), INT32_C( 1299263323), INT32_C(-1100697239), INT32_C(-1165132701),
                            INT32_C(-1312528679), INT32_C(-2057483334), INT32_C(-2116201571), INT32_C(-1004874347),
                            INT32_C( -792865239), INT32_C(  167838662), INT32_C(-1158285246), INT32_C(  788705850),
                            INT32_C(-1470598876), INT32_C( -300747724), INT32_C( -732019428), INT32_C(-1060860437)),
      simde_mm512_set_epi32(INT32_C(-1871683157), INT32_C(  203489561), INT32_C(  941895937), INT32_C(  403009536),
                            INT32_C(  281297537), INT32_C(   85281466), INT32_C(-2121969388), INT32_C(-2146807408),
                            INT32_C( 1085800480), INT32_C(      65536), INT32_C(-1169913792), INT32_C(  771756058),
                            INT32_C(  537407488), INT32_C(-2147299312), INT32_C(    6160660), INT32_C(   12615712)) },
    { simde_mm512_set_epi32(INT32_C(  782435122), INT32_C( 1862046610), INT32_C( 2063073020), INT32_C(-2039040635),
                            INT32_C( 1210624813), INT32_C( 1482889596), INT32_C(-1693737823), INT32_C( -742414353),
                            INT32_C(  769657412), INT32_C(-1049696640), INT32_C(  237587070), INT32_C( 1546361918),
                            INT32_C( -364413489), INT32_C(-1858108224), INT32_C(-1524047519), INT32_C( -892082969)),
      simde_mm512_set_epi32(INT32_C( 1276319466), INT32_C( -348382036), INT32_C(  -54124638), INT32_C(-1613416797),
                            INT32_C( -277896350), INT32_C(-1555914365), INT32_C( 1602672291), INT32_C(  612591504),
                            INT32_C(-1670560036), INT32_C( 2118020891), INT32_C(-1204159467), INT32_C(  299945581),
                            INT32_C( 1470077526), INT32_C(-1901456818), INT32_C( 1982811443), INT32_C(  366998615)),
      simde_mm512_set_epi32(INT32_C( 1074795720), INT32_C(-2130703316), INT32_C(-2080374526), INT32_C(  427885090),
                            INT32_C(-1488519102), INT32_C(-1560239997), INT32_C( 1149518338), INT32_C(  603996176),
                            INT32_C(-1878178664), INT32_C( 1041238299), INT32_C(-1341082623), INT32_C(   29377089),
                            INT32_C(  362316304), INT32_C(  243274254), INT32_C( 1376193554), INT32_C(  353112080)) },
    { simde_mm512_set_epi32(INT32_C( -664438730), INT32_C( 1158162569), INT32_C(-1048438639), INT32_C(  819552403),
                            INT32_C(  486427093), INT32_C(-1267830843), INT32_C( 1178270581), INT32_C(-1348447676),
                            INT32_C( -981472284), INT32_C( 1962298807), INT32_C( -393093452), INT32_C(-1754911100),
                            INT32_C(-1506604227), INT32_C( -220324223), INT32_C(  856278899), INT32_C(   15706156)),
      simde_mm512_set_epi32(INT32_C( -689282393), INT32_C( -261985647), INT32_C(-1390325708), INT32_C(-1552766747),
                            INT32_C(-1576064212), INT32_C( -185898645), INT32_C(-1798232738), INT32_C( -401409831),
                            INT32_C( 1975803231), INT32_C( 1826250001), INT32_C(-1038398890), INT32_C( -306355124),
                            INT32_C(-1154269982), INT32_C( -209110535), INT32_C(-2033491342), INT32_C( -971905248)),
      simde_mm512_set_epi32(INT32_C(  109707905), INT32_C(-1335737840), INT32_C(  740376612), INT32_C(-2094888860),
                            INT32_C(-1576984024), INT32_C( 1082196010), INT32_C(-1866398710), INT32_C( 1074964633),
                            INT32_C(  813700123), INT32_C(  134746112), INT32_C(   34209858), INT32_C( 1754873928),
                            INT32_C(  419443906), INT32_C(   16851320), INT32_C(-2067652608), INT32_C( -972011776)) },
    { simde_mm512_set_epi32(INT32_C(-1519344071), INT32_C( 1556822852), INT32_C(-1382496853), INT32_C( -624683333),
                            INT32_C( 1477411394), INT32_C( -704833096), INT32_C(-1957423151), INT32_C( -471773069),
                            INT32_C( 1263493389), INT32_C( 2117955521), INT32_C(-1143959230), INT32_C( -832581030),
                            INT32_C(-1273834890), INT32_C( -392148704), INT32_C( 1764655366), INT32_C( -721713055)),
      simde_mm512_set_epi32(INT32_C(-1396008954), INT32_C( -651865449), INT32_C(  452267102), INT32_C( -741136221),
                            INT32_C( 1539744858), INT32_C(-2014766256), INT32_C(-1095604449), INT32_C(-1527666044),
                            INT32_C( -826073132), INT32_C(   -8340331), INT32_C( 1447376741), INT32_C( 1608478316),
                            INT32_C( 1253487795), INT32_C( 2056029052), INT32_C( -880457902), INT32_C( -691872315)),
      simde_mm512_set_epi32(INT32_C(  143267846), INT32_C(-2128330605), INT32_C(  308611156), INT32_C(   18032640),
                            INT32_C(   62923800), INT32_C(   33555008), INT32_C(  883056654), INT32_C(   68201092),
                            INT32_C(-2071978288), INT32_C(-2122280940), INT32_C( 1141188133), INT32_C(  293612580),
                            INT32_C( 1252271233), INT32_C(  302809692), INT32_C(-2105457072), INT32_C(   33576324)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_andnot_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_andnot_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-1056724565), INT32_C( 1525326722), INT32_C( -860629095), INT32_C( 1674345138),
                            INT32_C( -780517906), INT32_C(-1953060088), INT32_C(-1307294727), INT32_C(-1463687440),
                            INT32_C( -675695615), INT32_C( 1308561010), INT32_C(  639253006), INT32_C( -651243687),
                            INT32_C( -612977662), INT32_C(  549809264), INT32_C(  644920842), INT32_C( 1882016760)),
      UINT16_C(49595),
      simde_mm512_set_epi32(INT32_C( 1189938329), INT32_C( 1797835672), INT32_C(  297527233), INT32_C( 1889709809),
                            INT32_C( -409509393), INT32_C(  994472936), INT32_C( -666015338), INT32_C( -260985600),
                            INT32_C(  750956055), INT32_C( 1716384261), INT32_C(-1960820967), INT32_C(  548004165),
                            INT32_C( 1158678479), INT32_C( 1692264694), INT32_C(  789910754), INT32_C(-1468927401)),
      simde_mm512_set_epi32(INT32_C( -428021677), INT32_C( 2072345032), INT32_C(-1760567882), INT32_C( -446864592),
                            INT32_C( 1299362117), INT32_C( 1402678741), INT32_C( -667918085), INT32_C( 1296019907),
                            INT32_C(-1260791809), INT32_C( 1231406525), INT32_C( -414651973), INT32_C( 1577314987),
                            INT32_C( 1453608195), INT32_C( 1199168765), INT32_C( 1764761558), INT32_C(-1053144882)),
      simde_mm512_set_epi32(INT32_C(-1609504702), INT32_C(  277164096), INT32_C( -860629095), INT32_C( 1674345138),
                            INT32_C( -780517906), INT32_C(-1953060088), INT32_C(-1307294727), INT32_C(  219025603),
                            INT32_C(-1877388824), INT32_C( 1308561010), INT32_C( 1682489506), INT32_C( 1577192106),
                            INT32_C(  312756224), INT32_C(  549809264), INT32_C( 1075841812), INT32_C( 1091176584)) },
    { simde_mm512_set_epi32(INT32_C( -922226792), INT32_C( 1177794317), INT32_C(-1987384202), INT32_C(  817250921),
                            INT32_C( 1296662639), INT32_C(   64131133), INT32_C(-1048693918), INT32_C( 1748498328),
                            INT32_C( -392119279), INT32_C(-1074948281), INT32_C( 1219088991), INT32_C(  346956559),
                            INT32_C( -778487174), INT32_C( 2030262893), INT32_C( -325938509), INT32_C( 2088865417)),
      UINT16_C(43842),
      simde_mm512_set_epi32(INT32_C( -911190750), INT32_C( -150954698), INT32_C(-2108244068), INT32_C( -219871492),
                            INT32_C(  954142226), INT32_C( -657696450), INT32_C(  -40171606), INT32_C(  523799369),
                            INT32_C(-1984820679), INT32_C( -352318109), INT32_C( 1527484465), INT32_C( 1078897849),
                            INT32_C( -979432773), INT32_C( -222789591), INT32_C( -127333602), INT32_C( 1547833861)),
      simde_mm512_set_epi32(INT32_C( 1706771302), INT32_C(-1876132949), INT32_C( -300867745), INT32_C(-1574226708),
                            INT32_C(  909541228), INT32_C(-1473521559), INT32_C(-2035272090), INT32_C( -843632177),
                            INT32_C(-1617888467), INT32_C( -960934829), INT32_C( -805571508), INT32_C( -811280081),
                            INT32_C(-1033748670), INT32_C(-1374688928), INT32_C( -924697051), INT32_C( -396703151)),
      simde_mm512_set_epi32(INT32_C(  604701252), INT32_C( 1177794317), INT32_C( 1812004931), INT32_C(  817250921),
                            INT32_C(  102789484), INT32_C(   64131133), INT32_C(   35663940), INT32_C(-1064882042),
                            INT32_C( -392119279), INT32_C(   79249424), INT32_C( 1219088991), INT32_C(  346956559),
                            INT32_C( -778487174), INT32_C( 2030262893), INT32_C(    8533025), INT32_C( 2088865417)) },
    { simde_mm512_set_epi32(INT32_C(-1406718947), INT32_C(  276558393), INT32_C(  154803470), INT32_C( 1010355861),
                            INT32_C( -906943422), INT32_C(-1458735792), INT32_C( -135902673), INT32_C( 2125322250),
                            INT32_C(  668612521), INT32_C( 2134097324), INT32_C( 1431164540), INT32_C(-1097880462),
                            INT32_C( 1895279922), INT32_C( -455917584), INT32_C(-1635623774), INT32_C( 1646110584)),
      UINT16_C(61721),
      simde_mm512_set_epi32(INT32_C(-1147100012), INT32_C( -529153170), INT32_C(-1710107397), INT32_C( 1085126684),
                            INT32_C( -365628842), INT32_C( 1126939173), INT32_C(-1962930746), INT32_C(-2032518388),
                            INT32_C( -893793955), INT32_C(-1793978656), INT32_C(  353794556), INT32_C(  484459160),
                            INT32_C( 1795576890), INT32_C(-1800969495), INT32_C(  570832120), INT32_C( -805110645)),
      simde_mm512_set_epi32(INT32_C(-1152323073), INT32_C(-1880366011), INT32_C( 1623795528), INT32_C(  779718762),
                            INT32_C( -950308445), INT32_C(  601329882), INT32_C( 1983067756), INT32_C( 1014514692),
                            INT32_C(  192697146), INT32_C( 1393627685), INT32_C( -618845734), INT32_C(-1526656596),
                            INT32_C( -668243521), INT32_C(  858775967), INT32_C( -874197170), INT32_C(-1013451033)),
      simde_mm512_set_epi32(INT32_C(    5263723), INT32_C(  260711425), INT32_C( 1623728896), INT32_C(  776994914),
                            INT32_C( -906943422), INT32_C(-1458735792), INT32_C( -135902673), INT32_C(  941638656),
                            INT32_C(  668612521), INT32_C( 2134097324), INT32_C( 1431164540), INT32_C(-1593765596),
                            INT32_C(-1876350587), INT32_C( -455917584), INT32_C(-1635623774), INT32_C(   60097124)) },
    { simde_mm512_set_epi32(INT32_C(   73765979), INT32_C( 1196192749), INT32_C( -212227718), INT32_C(-1980699203),
                            INT32_C(  -37222007), INT32_C(-1986328859), INT32_C( 1483201456), INT32_C(  129080387),
                            INT32_C( -259597220), INT32_C(-1814466623), INT32_C( 1536667113), INT32_C( 1702406736),
                            INT32_C( 1032855403), INT32_C( -907220805), INT32_C( -744099936), INT32_C( -484286001)),
      UINT16_C(60398),
      simde_mm512_set_epi32(INT32_C( 2131878120), INT32_C( -709717494), INT32_C(  677603870), INT32_C( 1110837767),
                            INT32_C(  137332416), INT32_C( 1049147481), INT32_C( -429123521), INT32_C(  562109282),
                            INT32_C( -475857832), INT32_C(-1750530864), INT32_C(-1098694184), INT32_C(-1278646805),
                            INT32_C(  274075622), INT32_C(  310096866), INT32_C( 1944249360), INT32_C(-1457965117)),
      simde_mm512_set_epi32(INT32_C(-1770120574), INT32_C(-1267999916), INT32_C(  920660290), INT32_C( 1218524275),
                            INT32_C( -813719782), INT32_C(   17574100), INT32_C( 1228269274), INT32_C( -540460196),
                            INT32_C( -544630186), INT32_C( -973323962), INT32_C( -900762472), INT32_C( 1800691074),
                            INT32_C( -934840396), INT32_C(-2024059127), INT32_C( 2050139755), INT32_C(-1648520849)),
      simde_mm512_set_epi32(INT32_C(-2140268030), INT32_C(  541673812), INT32_C(  377487680), INT32_C(-1980699203),
                            INT32_C( -951052006), INT32_C(-1986328859), INT32_C(  152168128), INT32_C( -565698532),
                            INT32_C(  470352390), INT32_C( 1079263494), INT32_C( 1078735872), INT32_C( 1702406736),
                            INT32_C( -939429872), INT32_C(-2063578103), INT32_C(  135266923), INT32_C( -484286001)) },
    { simde_mm512_set_epi32(INT32_C(  359551557), INT32_C(  851518101), INT32_C( 1700885885), INT32_C( 1144006274),
                            INT32_C(  718077661), INT32_C( 1054313754), INT32_C(   65647391), INT32_C(-1867262731),
                            INT32_C(  208941224), INT32_C(  989467762), INT32_C(-1763663368), INT32_C(  732190820),
                            INT32_C( -780985117), INT32_C(-1786203682), INT32_C( -893464048), INT32_C(-1930046056)),
      UINT16_C( 5280),
      simde_mm512_set_epi32(INT32_C( 2082802710), INT32_C(  398405458), INT32_C( -610997258), INT32_C(  830342728),
                            INT32_C( -327286830), INT32_C( 1285368273), INT32_C(-1636339073), INT32_C( 1467021210),
                            INT32_C( -637556884), INT32_C( 1464578281), INT32_C(  -78771124), INT32_C(-1194071193),
                            INT32_C(-1454776494), INT32_C(  224158188), INT32_C( 1578376173), INT32_C( 2022699384)),
      simde_mm512_set_epi32(INT32_C(-1580866758), INT32_C( 1705729088), INT32_C(-1204463345), INT32_C(  806420788),
                            INT32_C(-1410408996), INT32_C(  863225653), INT32_C(-2071560363), INT32_C( 1819484417),
                            INT32_C( -246595685), INT32_C(  243263522), INT32_C( 2052176477), INT32_C(  253176681),
                            INT32_C( 1676258794), INT32_C(-1129907739), INT32_C(  395133900), INT32_C(  -86934818)),
      simde_mm512_set_epi32(INT32_C(  359551557), INT32_C(  851518101), INT32_C( 1700885885), INT32_C(      65844),
                            INT32_C(  718077661), INT32_C(  862111268), INT32_C(   65647391), INT32_C(-1867262731),
                            INT32_C(  536877203), INT32_C(  989467762), INT32_C(    1159697), INT32_C(  732190820),
                            INT32_C( -780985117), INT32_C(-1786203682), INT32_C( -893464048), INT32_C(-1930046056)) },
    { simde_mm512_set_epi32(INT32_C( -763717484), INT32_C(-1454287993), INT32_C( -815713015), INT32_C( -381645662),
                            INT32_C( 1143121149), INT32_C(-2120634980), INT32_C( -259357121), INT32_C( -593579957),
                            INT32_C(-1529041977), INT32_C(-2065541499), INT32_C( 1009471119), INT32_C(  674532491),
                            INT32_C( -605291509), INT32_C( -802607554), INT32_C( -850350011), INT32_C(  732847081)),
      UINT16_C(41568),
      simde_mm512_set_epi32(INT32_C( 1295870302), INT32_C(  336570348), INT32_C(-1662536141), INT32_C(-1054381248),
                            INT32_C( 1593114303), INT32_C(-1017054773), INT32_C(-1409414000), INT32_C(  227338784),
                            INT32_C( 1117509139), INT32_C( 1937140770), INT32_C( 1843080524), INT32_C(  775622876),
                            INT32_C(  903821795), INT32_C(-1108923393), INT32_C( -348808591), INT32_C(  691553406)),
      simde_mm512_set_epi32(INT32_C( -957741997), INT32_C( -389978329), INT32_C(-1992364300), INT32_C(-1194120095),
                            INT32_C( 1460280679), INT32_C( -461012902), INT32_C(  191451119), INT32_C(  395863574),
                            INT32_C( 2007897293), INT32_C(  647995187), INT32_C( 1812181798), INT32_C(-1288356108),
                            INT32_C(-1946740515), INT32_C(-1688294491), INT32_C( -146679692), INT32_C( -960173252)),
      simde_mm512_set_epi32(INT32_C(-2101214207), INT32_C(-1454287993), INT32_C(   18368708), INT32_C( -381645662),
                            INT32_C( 1143121149), INT32_C(-2120634980), INT32_C(      82799), INT32_C( -593579957),
                            INT32_C(-1529041977), INT32_C(   76124945), INT32_C(      37410), INT32_C(  674532491),
                            INT32_C( -605291509), INT32_C( -802607554), INT32_C( -850350011), INT32_C(  732847081)) },
    { simde_mm512_set_epi32(INT32_C(-1543080560), INT32_C(  326946931), INT32_C(  691349892), INT32_C( 1226829378),
                            INT32_C( 1127061143), INT32_C( 1548237043), INT32_C(-1885371906), INT32_C(  673215002),
                            INT32_C(   -2545554), INT32_C(-1367277302), INT32_C( -227991301), INT32_C(  746457208),
                            INT32_C(-1737407854), INT32_C( 1988034150), INT32_C( -605858038), INT32_C( -752579769)),
      UINT16_C(24718),
      simde_mm512_set_epi32(INT32_C( 1517976828), INT32_C(  453076709), INT32_C( 1155311084), INT32_C(-1730593997),
                            INT32_C( 2009897302), INT32_C( -813354987), INT32_C( 1160389453), INT32_C(-1543844644),
                            INT32_C( -908777016), INT32_C(  107061968), INT32_C(-1889800585), INT32_C(-1309816398),
                            INT32_C( 1760607631), INT32_C(-1373730647), INT32_C( 1475928392), INT32_C(-1415204909)),
      simde_mm512_set_epi32(INT32_C(  901302066), INT32_C(  236605933), INT32_C( 1144123725), INT32_C(  765559000),
                            INT32_C( -272466037), INT32_C(  489940181), INT32_C( 1285546635), INT32_C(  894611583),
                            INT32_C(-1280504231), INT32_C( -511809158), INT32_C(  517714821), INT32_C( -458114298),
                            INT32_C(-1583011646), INT32_C( 2050708057), INT32_C(-1873361568), INT32_C( 1295393304)),
      simde_mm512_set_epi32(INT32_C(-1543080560), INT32_C(   68817160), INT32_C(    2183169), INT32_C( 1226829378),
                            INT32_C( 1127061143), INT32_C( 1548237043), INT32_C(-1885371906), INT32_C(  673215002),
                            INT32_C(  841483793), INT32_C(-1367277302), INT32_C( -227991301), INT32_C(  746457208),
                            INT32_C(-2130378688), INT32_C( 1344361040), INT32_C(-2147089376), INT32_C( -752579769)) },
    { simde_mm512_set_epi32(INT32_C( -203532895), INT32_C(-1671983312), INT32_C( -485765980), INT32_C(-1920770849),
                            INT32_C(  -87193791), INT32_C( 1659979037), INT32_C(-1337410362), INT32_C( 1209029675),
                            INT32_C(  587197109), INT32_C( -530755740), INT32_C(  281664792), INT32_C(  -47077792),
                            INT32_C( -945013045), INT32_C( -166692659), INT32_C( 1790118115), INT32_C(  689330771)),
      UINT16_C( 7519),
      simde_mm512_set_epi32(INT32_C( -384323470), INT32_C(  473195364), INT32_C(  206146438), INT32_C(-1217279332),
                            INT32_C(-1088463893), INT32_C(  970520784), INT32_C( -929499045), INT32_C(-1086034653),
                            INT32_C(-1051759609), INT32_C(-1753508816), INT32_C( 1464082608), INT32_C(  492133710),
                            INT32_C( 1610388137), INT32_C(-2026322187), INT32_C(-1721391979), INT32_C(  466414066)),
      simde_mm512_set_epi32(INT32_C( 1039275088), INT32_C( -195464931), INT32_C(-1467895249), INT32_C( 1829711637),
                            INT32_C( 2006708634), INT32_C(  837542220), INT32_C( -759309790), INT32_C( -498075629),
                            INT32_C(  922280800), INT32_C(  925077084), INT32_C( 1941328295), INT32_C(   27280850),
                            INT32_C( -499921640), INT32_C(  738410205), INT32_C(  972641353), INT32_C( 1011602801)),
      simde_mm512_set_epi32(INT32_C( -203532895), INT32_C(-1671983312), INT32_C( -485765980), INT32_C( 1208886529),
                            INT32_C( 1082171408), INT32_C(    2294028), INT32_C(-1337410362), INT32_C( 1074499600),
                            INT32_C(  587197109), INT32_C(  536873548), INT32_C(  281664792), INT32_C(   10485904),
                            INT32_C(-1610396400), INT32_C(  671299592), INT32_C(  546852936), INT32_C(  604181505)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_andnot_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_andnot_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(56303),
      simde_mm512_set_epi32(INT32_C(  684353163), INT32_C( -624296854), INT32_C(-1626870831), INT32_C( 1693659819),
                            INT32_C( 1814966119), INT32_C( 1428960968), INT32_C( 1709146671), INT32_C(-1269736679),
                            INT32_C( -399245267), INT32_C(  128121852), INT32_C(  623395494), INT32_C( 1817163956),
                            INT32_C( 1991308671), INT32_C( -978886098), INT32_C( 1436967950), INT32_C(  227176170)),
      simde_mm512_set_epi32(INT32_C( -155316348), INT32_C( 1821995326), INT32_C(-1956349521), INT32_C( 2078645861),
                            INT32_C(-2002962850), INT32_C( 1961273418), INT32_C( 1026886280), INT32_C( 1852456749),
                            INT32_C( 1549356853), INT32_C(  905982506), INT32_C( -562722910), INT32_C( 1231420121),
                            INT32_C(  786944005), INT32_C(-1682464667), INT32_C(   12357782), INT32_C(  913777965)),
      simde_mm512_set_epi32(INT32_C( -701232892), INT32_C(  605028628), INT32_C(          0), INT32_C(  453282884),
                            INT32_C(-2137976808), INT32_C(          0), INT32_C(  404752512), INT32_C( 1244275748),
                            INT32_C(  340348688), INT32_C(  805306370), INT32_C( -631929600), INT32_C(          0),
                            INT32_C(  138870784), INT32_C(  437289025), INT32_C(    1609872), INT32_C(  846528773)) },
    { UINT16_C(56200),
      simde_mm512_set_epi32(INT32_C( -452164103), INT32_C( 1890508390), INT32_C( 1258638805), INT32_C( -750109723),
                            INT32_C( -513503890), INT32_C( -379667747), INT32_C(-1651966538), INT32_C(  418163645),
                            INT32_C(-1484633406), INT32_C(  128570401), INT32_C(-1432905388), INT32_C(-1460529893),
                            INT32_C( -808466332), INT32_C(-1300168003), INT32_C(  153276923), INT32_C( -912847520)),
      simde_mm512_set_epi32(INT32_C( 1849401350), INT32_C(-2046167065), INT32_C(-1772087293), INT32_C(  763578781),
                            INT32_C(  -59556630), INT32_C( -574235850), INT32_C(-1931079616), INT32_C(  856557360),
                            INT32_C( 1798494574), INT32_C( -255236934), INT32_C( -498039931), INT32_C( 1916101155),
                            INT32_C( 1291737736), INT32_C(-1818740725), INT32_C( 1042711156), INT32_C(  770521823)),
      simde_mm512_set_epi32(INT32_C(  171122694), INT32_C(-2046746239), INT32_C(          0), INT32_C(  746668056),
                            INT32_C(  471019648), INT32_C(          0), INT32_C(    6684736), INT32_C(  587334656),
                            INT32_C( 1211142444), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(    3150472), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(29534),
      simde_mm512_set_epi32(INT32_C(-1569526022), INT32_C( -566961257), INT32_C( -454262297), INT32_C(-2011970966),
                            INT32_C( 1729229439), INT32_C(  515441803), INT32_C( 1629075756), INT32_C( -633945234),
                            INT32_C(-1517000454), INT32_C(-2129179491), INT32_C(-1082415130), INT32_C( -643068488),
                            INT32_C(-1177678851), INT32_C(  811665360), INT32_C(-1120986687), INT32_C( 1945770944)),
      simde_mm512_set_epi32(INT32_C( 1206445472), INT32_C( 1685117563), INT32_C( -105634979), INT32_C(  300875900),
                            INT32_C( 1292473590), INT32_C( -154568093), INT32_C( -725481309), INT32_C( 1537059805),
                            INT32_C(-1299234249), INT32_C( 1342055246), INT32_C( 1121196977), INT32_C( -936323200),
                            INT32_C(  284920534), INT32_C( -501374627), INT32_C(  523356394), INT32_C( 2082914622)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  541065320), INT32_C(  420487704), INT32_C(  300679188),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1799225213), INT32_C(   25766033),
                            INT32_C(          0), INT32_C( 1323836226), INT32_C(          0), INT32_C(    1069568),
                            INT32_C(    3244034), INT32_C(-1038311411), INT32_C(   34652202), INT32_C(          0)) },
    { UINT16_C(18467),
      simde_mm512_set_epi32(INT32_C(-1126901666), INT32_C(-1540993522), INT32_C( -310394649), INT32_C( 1569425965),
                            INT32_C( 1860055197), INT32_C( 1022884520), INT32_C(  886587779), INT32_C(   -7751100),
                            INT32_C(  725782952), INT32_C( 1524528742), INT32_C(-1901622691), INT32_C( -205155472),
                            INT32_C( 1297212229), INT32_C(-1562315637), INT32_C(-1561800150), INT32_C( 1969817622)),
      simde_mm512_set_epi32(INT32_C( 1691822441), INT32_C( -747576101), INT32_C(  526461787), INT32_C(-1551035253),
                            INT32_C( -494445545), INT32_C(  601243904), INT32_C( 1621282220), INT32_C(   87983768),
                            INT32_C( 1749180883), INT32_C(  653596692), INT32_C( 1933605299), INT32_C( 2110990238),
                            INT32_C( 1287872496), INT32_C( -947101027), INT32_C(-1469323630), INT32_C( -103698146)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C( 1397792977), INT32_C(          0), INT32_C(          0),
                            INT32_C(-2147398654), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1900048802), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(  134423696), INT32_C(-2003783416)) },
    { UINT16_C(10499),
      simde_mm512_set_epi32(INT32_C(-1800578563), INT32_C( 1189147870), INT32_C( -918534753), INT32_C(-2046784432),
                            INT32_C( 2146267513), INT32_C( 1185116678), INT32_C(  743422455), INT32_C( -958735431),
                            INT32_C(-1272492795), INT32_C(-1993475811), INT32_C( -901911405), INT32_C( -444376352),
                            INT32_C( 1645484254), INT32_C( 1890851846), INT32_C(  632187417), INT32_C( 2142729898)),
      simde_mm512_set_epi32(INT32_C( -752859034), INT32_C( -661272677), INT32_C( 1736074301), INT32_C( 1246429845),
                            INT32_C(-1327059157), INT32_C(-1760626525), INT32_C(  693999571), INT32_C(  179503183),
                            INT32_C(-1261277577), INT32_C( 2014601419), INT32_C(   45385261), INT32_C( 1333239387),
                            INT32_C( 1950214560), INT32_C( 2050540474), INT32_C(  -73887902), INT32_C(-1586317941)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(  641343520), INT32_C(          0),
                            INT32_C(-2147316222), INT32_C(          0), INT32_C(          0), INT32_C(  136380486),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( -636451486), INT32_C(-2143256319)) },
    { UINT16_C(17481),
      simde_mm512_set_epi32(INT32_C( -441498123), INT32_C(  324738064), INT32_C(  -27713047), INT32_C(  322022433),
                            INT32_C( -623687192), INT32_C(  441486000), INT32_C(-1091397610), INT32_C(  486920838),
                            INT32_C(  727930899), INT32_C(  134578624), INT32_C( -229821250), INT32_C(-1459771681),
                            INT32_C(  786852212), INT32_C(-1562273484), INT32_C(  592450244), INT32_C( -391708168)),
      simde_mm512_set_epi32(INT32_C(  792156312), INT32_C(  407601311), INT32_C(-1255558455), INT32_C( 1648353396),
                            INT32_C(-1874603621), INT32_C(-1962724996), INT32_C(-1379808132), INT32_C(-1917277067),
                            INT32_C( -327375348), INT32_C( -266290190), INT32_C( -446684576), INT32_C( -218289365),
                            INT32_C( 1659849163), INT32_C(  313080914), INT32_C(  914897986), INT32_C( -690088867)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  134250639), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(-2130497204), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( -266323406), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1074331787), INT32_C(          0), INT32_C(          0), INT32_C(  374871045)) },
    { UINT16_C(25655),
      simde_mm512_set_epi32(INT32_C(    7734189), INT32_C(-1107618186), INT32_C( 1291997837), INT32_C( -657618671),
                            INT32_C( -523204184), INT32_C(  197247571), INT32_C(-1924672781), INT32_C( 1367953812),
                            INT32_C( 1671605226), INT32_C( -667696065), INT32_C(  734579404), INT32_C(  -25998720),
                            INT32_C( -791898275), INT32_C(-1848361166), INT32_C(  302446873), INT32_C(-1290034089)),
      simde_mm512_set_epi32(INT32_C(-2140777278), INT32_C( 1356458144), INT32_C(  990615850), INT32_C(  122581591),
                            INT32_C( 1842174798), INT32_C( 1633161914), INT32_C( 1487544794), INT32_C( 1680890315),
                            INT32_C(-1051319145), INT32_C( 1671869354), INT32_C( -657093416), INT32_C(   76483879),
                            INT32_C(  897241075), INT32_C(-1385812547), INT32_C(  518745683), INT32_C( 1278998383)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C( 1073801344), INT32_C(  839485730), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1612189864), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( -803929072), INT32_C(    9176359),
                            INT32_C(          0), INT32_C(  740425869), INT32_C(  216598082), INT32_C( 1277186856)) },
    { UINT16_C( 9319),
      simde_mm512_set_epi32(INT32_C(  359510622), INT32_C( 1667719225), INT32_C(  630674948), INT32_C(  610105763),
                            INT32_C(   20744378), INT32_C(-1334671422), INT32_C( 1934181344), INT32_C( -207473635),
                            INT32_C(  -12247390), INT32_C(  935971775), INT32_C( -814870615), INT32_C(  272416728),
                            INT32_C(-2094904434), INT32_C(  118285194), INT32_C( 1770668331), INT32_C(-1463910375)),
      simde_mm512_set_epi32(INT32_C(  399098366), INT32_C(-1713281213), INT32_C( 2124618772), INT32_C(-1052563089),
                            INT32_C( 1851869047), INT32_C( 2020277970), INT32_C(-1035589842), INT32_C(-1789987668),
                            INT32_C(  733487930), INT32_C( -497440680), INT32_C(-1951336884), INT32_C(-1752937795),
                            INT32_C(-1263292061), INT32_C(     242422), INT32_C( 1531342059), INT32_C( -447099781)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C( 1512048656), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1208514576), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(-1072684992), INT32_C(    9498692), INT32_C(          0),
                            INT32_C(          0), INT32_C(     200820), INT32_C(  306457792), INT32_C( 1161907298)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_andnot_epi32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_andnot_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(  207721957124820559), INT64_C( 7800065217939756514),
                            INT64_C(-3924116943760495845), INT64_C(-4670511705337769443),
                            INT64_C( 8681164262815197674), INT64_C(-1748050366477277388),
                            INT64_C( 6521714148432251392), INT64_C( -441034964778727222)),
      simde_mm512_set_epi64(INT64_C( 1906622887772594248), INT64_C(-6188571553716009650),
                            INT64_C(  264280323541139152), INT64_C( 1584607577489802492),
                            INT64_C( 1109502791419861047), INT64_C( 7178327557348084990),
                            INT64_C( 1395553581907359924), INT64_C(-6533533183118051768)),
      simde_mm512_set_epi64(INT64_C( 1735017709269196800), INT64_C(-9079113909020786676),
                            INT64_C(  153194412888495296), INT64_C(   58731659148920032),
                            INT64_C(  505544490090532885), INT64_C(     565705191721162),
                            INT64_C(   98516792007995572), INT64_C(  293888628881438208)) },
    { simde_mm512_set_epi64(INT64_C(-6724768156717290819), INT64_C(-5250906792133082841),
                            INT64_C( 7101763469273509464), INT64_C(-6606445878350250265),
                            INT64_C( -992513133092468415), INT64_C( 6991549638647222936),
                            INT64_C(-6702609966967541799), INT64_C( 6463119549714578187)),
      simde_mm512_set_epi64(INT64_C( -887590357697526439), INT64_C( 7877951253899372304),
                            INT64_C(-2913340636735054567), INT64_C(-9221955371178601401),
                            INT64_C(-8332586207462320569), INT64_C(-6065729331557277752),
                            INT64_C(-5495188752553836498), INT64_C(-4545091831756868823)),
      simde_mm512_set_epi64(INT64_C( 5837268749247317312), INT64_C( 5211811456626140688),
                            INT64_C(-7705320311288082175), INT64_C(    1134704830971904),
                            INT64_C(  883853889565267462), INT64_C(-8444247080808460992),
                            INT64_C( 1226140460016631846), INT64_C(-9201976626802327520)) },
    { simde_mm512_set_epi64(INT64_C(-8081018174907924542), INT64_C( 6936249846129023242),
                            INT64_C(-1059210610078769383), INT64_C( 1593162574725548027),
                            INT64_C( 2194029932784271057), INT64_C( 2297742112014824027),
                            INT64_C( 6872936620014531062), INT64_C(-4458741002964204726)),
      simde_mm512_set_epi64(INT64_C( 7389599045220123111), INT64_C(-4734617337151831127),
                            INT64_C(-3688698012661984630), INT64_C( 2942411497108224949),
                            INT64_C( 3088165388972230068), INT64_C(-8598989874996476457),
                            INT64_C(-2925060949778391940), INT64_C( 1600057734617632940)),
      simde_mm512_set_epi64(INT64_C( 6918937665425915941), INT64_C(-7059040014775614815),
                            INT64_C(  901582149085413506), INT64_C( 2936772469119858692),
                            INT64_C( 2344435893274706212), INT64_C(-9221049868269222524),
                            INT64_C(-9221110889849910264), INT64_C( 1450299817904312484)) },
    { simde_mm512_set_epi64(INT64_C(-2851531746227363368), INT64_C( 2067892326136395565),
                            INT64_C( 5955544350840259834), INT64_C(-9215158447496033102),
                            INT64_C(-6496129397571023850), INT64_C( 6580537045822776099),
                            INT64_C(-1881492268188536860), INT64_C( 6477581622128112348)),
      simde_mm512_set_epi64(INT64_C( 4736931688263401886), INT64_C( -422510099501192510),
                            INT64_C( 3904035851984069712), INT64_C(-1269778779692298262),
                            INT64_C( 7103388094266435672), INT64_C( 8538164081108009860),
                            INT64_C( 7657481289221491954), INT64_C(-6346831563088898420)),
      simde_mm512_set_epi64(INT64_C(  112770318310899718), INT64_C(-2161639582911543102),
                            INT64_C( 2596786860877701120), INT64_C( 7953586499903062856),
                            INT64_C( 4756998866794012744), INT64_C( 2606779805598826628),
                            INT64_C(  721778983603339282), INT64_C(-6482086895067069440)) },
    { simde_mm512_set_epi64(INT64_C( -821005629772787069), INT64_C(-4647973389902912809),
                            INT64_C( 6459900742609080709), INT64_C(   -1266809698382208),
                            INT64_C(  701020828809534395), INT64_C(-8547290149729742964),
                            INT64_C( -440779604644636577), INT64_C(-3509307452635316669)),
      simde_mm512_set_epi64(INT64_C( 8999318376500703433), INT64_C( 1719097867730734351),
                            INT64_C(  360091487853740826), INT64_C(-6254537314592943558),
                            INT64_C( -632347399973673450), INT64_C( 2614451855333869078),
                            INT64_C( 6887846494654494209), INT64_C( 6275950466702179569)),
      simde_mm512_set_epi64(INT64_C(  604608525006544968), INT64_C(   36136703980768520),
                            INT64_C(  313352018360009242), INT64_C(        137573240890),
                            INT64_C( -720169941136284668), INT64_C( 2596327487390613522),
                            INT64_C(  438118704866436608), INT64_C( 1157566394459521200)) },
    { simde_mm512_set_epi64(INT64_C(-5483950330033170066), INT64_C(-4153699507396814554),
                            INT64_C( 1686943364333831141), INT64_C(-6155572369391990976),
                            INT64_C(-2338197867102969548), INT64_C( 4970317907692585902),
                            INT64_C( -659027381808082615), INT64_C(-8301976371410819309)),
      simde_mm512_set_epi64(INT64_C(-5922203424268985599), INT64_C( 1802271341012641429),
                            INT64_C(-7199161640250473305), INT64_C( 4184910176757162424),
                            INT64_C(-5885970898589897236), INT64_C( 5320604596895707800),
                            INT64_C(-7049806138053003152), INT64_C( 7856069210784274088)),
      simde_mm512_set_epi64(INT64_C(  869198318683570689), INT64_C( 1801690747234690705),
                            INT64_C(-8640647776843037694), INT64_C( 1153141544681808056),
                            INT64_C( 2328590264702274760), INT64_C(  649785191505621008),
                            INT64_C(  585473076492838960), INT64_C( 6991285376398659752)) },
    { simde_mm512_set_epi64(INT64_C(  772369500911491951), INT64_C(-3487181344595680581),
                            INT64_C(-6776954808191866646), INT64_C( 1437133779275187040),
                            INT64_C(-3742444221385296201), INT64_C( 3619551202282748987),
                            INT64_C(-5676058734881350704), INT64_C( 3034639668798379519)),
      simde_mm512_set_epi64(INT64_C( 7799576852730631653), INT64_C(-4611614721990756478),
                            INT64_C( 4179897201710999091), INT64_C(-6554042946408561565),
                            INT64_C( 7858455943023474684), INT64_C(-4868663260305658784),
                            INT64_C(-6563387696243649675), INT64_C( -252761203575600938)),
      simde_mm512_set_epi64(INT64_C( 7207351508714783872), INT64_C(      71283551638784),
                            INT64_C( 1873656161226589713), INT64_C(-6626408997484215293),
                            INT64_C( 2382069952524845384), INT64_C(-8339540561327800256),
                            INT64_C(  342318686555209765), INT64_C(-3142946274104309760)) },
    { simde_mm512_set_epi64(INT64_C(-6272776462503295319), INT64_C(-8894851852280934479),
                            INT64_C( 6828037840473322695), INT64_C( -784763491569829334),
                            INT64_C(-6956613286547242208), INT64_C(-7641604144835014945),
                            INT64_C( 4137535773895137731), INT64_C( 3122415965305276610)),
      simde_mm512_set_epi64(INT64_C( 5967240469174938071), INT64_C( 2271146860082105533),
                            INT64_C( 2488999494207974941), INT64_C(-7245269557183082373),
                            INT64_C(-6094983942162054282), INT64_C( 5272800144124782830),
                            INT64_C(-1112016268759137335), INT64_C( 3873297534982922048)),
      simde_mm512_set_epi64(INT64_C( 5912488079989451094), INT64_C( 1945726568221376524),
                            INT64_C( 2308274862648494616), INT64_C(  747597780979417169),
                            INT64_C( 2308728562190385238), INT64_C( 5191575370143047712),
                            INT64_C(-4571152522202316280), INT64_C( 1477377559112455936)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_andnot_epi64(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_andnot_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-8706055201876274534), INT64_C(-2974526497282267924),
                            INT64_C(-5064099105424399850), INT64_C( 4173762680971677425),
                            INT64_C( 5058953897646810163), INT64_C( 3129329827313761969),
                            INT64_C(-7680990319456213473), INT64_C( 3095613893972693568)),
      UINT8_C(148),
      simde_mm512_set_epi64(INT64_C( -438459145642420823), INT64_C( 2788318060387771818),
                            INT64_C(-6405634033298828022), INT64_C( 5697280571633296693),
                            INT64_C(-4038706177987584167), INT64_C( 7050984609072161968),
                            INT64_C( 2749018709708772273), INT64_C(-2265592192997989021)),
      simde_mm512_set_epi64(INT64_C(-1049270424665539045), INT64_C(-5614406584732574076),
                            INT64_C(  631202638299991092), INT64_C(-8590255914187036925),
                            INT64_C(-6307315262773811693), INT64_C(-1209843912248425712),
                            INT64_C(-3872834841544228683), INT64_C(-3085083838104197908)),
      simde_mm512_set_epi64(INT64_C(    4565176708278802), INT64_C(-2974526497282267924),
                            INT64_C(-5064099105424399850), INT64_C(-9166793916060532222),
                            INT64_C( 5058953897646810163), INT64_C(-8203935250787499264),
                            INT64_C(-7680990319456213473), INT64_C( 3095613893972693568)) },
    { simde_mm512_set_epi64(INT64_C(-4842938149095873389), INT64_C( -846085209911123390),
                            INT64_C(  902030110892207375), INT64_C(-8179884512098486778),
                            INT64_C( 7136180633023633249), INT64_C(-7202514001649392691),
                            INT64_C(-4512985345247872566), INT64_C( 6280820093975482096)),
      UINT8_C( 88),
      simde_mm512_set_epi64(INT64_C(-5899542268894168412), INT64_C( 5687678929880926481),
                            INT64_C(  754471637334648472), INT64_C( 1530269878614188173),
                            INT64_C(-3481843836368626596), INT64_C( 7214537798473258692),
                            INT64_C( 3186147264512503626), INT64_C(-2220217993706522327)),
      simde_mm512_set_epi64(INT64_C( 4741426381855247639), INT64_C(-6093431436741802321),
                            INT64_C(-7277776184535270866), INT64_C(-5890238516652006119),
                            INT64_C(-8031043717190201593), INT64_C( 7604814614465185239),
                            INT64_C( 6712821644684838579), INT64_C(-1700024539209227072)),
      simde_mm512_set_epi64(INT64_C(-4842938149095873389), INT64_C(-6845110636993050962),
                            INT64_C(  902030110892207375), INT64_C(-6178618704551475952),
                            INT64_C( 1152921504676217603), INT64_C(-7202514001649392691),
                            INT64_C(-4512985345247872566), INT64_C( 6280820093975482096)) },
    { simde_mm512_set_epi64(INT64_C( -647905387169688868), INT64_C(-8461625299591442725),
                            INT64_C(-4959110866452894415), INT64_C(-6046186632754619075),
                            INT64_C(-1792277330244185216), INT64_C( 7899374623587606112),
                            INT64_C(-2530906147097710338), INT64_C(-3452464982464189359)),
      UINT8_C(234),
      simde_mm512_set_epi64(INT64_C( 1092825191169264761), INT64_C(  518154175979275913),
                            INT64_C(-2540128939765803497), INT64_C( 7206989642204137224),
                            INT64_C( 5053971549089664110), INT64_C(  275130895293265200),
                            INT64_C( 5870095287105445532), INT64_C( 3766077764635497461)),
      simde_mm512_set_epi64(INT64_C( 4726923138274336458), INT64_C( 3036293318033390010),
                            INT64_C( 3265833753663381966), INT64_C(-5548402770380826836),
                            INT64_C(-1910939043053590920), INT64_C(-2803972634053834044),
                            INT64_C( 8571307896088376800), INT64_C(-2906367800591944553)),
      simde_mm512_set_epi64(INT64_C( 4652501007819903618), INT64_C( 2883153157893175602),
                            INT64_C( 2395932937578488264), INT64_C(-6046186632754619075),
                            INT64_C(-6820513618777071088), INT64_C( 7899374623587606112),
                            INT64_C( 2774537390188929376), INT64_C(-3452464982464189359)) },
    { simde_mm512_set_epi64(INT64_C( 1235103765186305905), INT64_C( 8251648155281492223),
                            INT64_C( 6607793927948629202), INT64_C(-4956133557414585628),
                            INT64_C( -962568210701922461), INT64_C( 7520783669412628517),
                            INT64_C( 4493695514722238610), INT64_C( 6191552237626999876)),
      UINT8_C(175),
      simde_mm512_set_epi64(INT64_C(-1999731829913464848), INT64_C( 7072204574593617968),
                            INT64_C( -329416891633690006), INT64_C( 4219653511875682573),
                            INT64_C(-5631405021388401918), INT64_C( -157450572284011331),
                            INT64_C(-6448890677231800514), INT64_C(-7780641104162742337)),
      simde_mm512_set_epi64(INT64_C(  261057906798578959), INT64_C(-4964336716206621793),
                            INT64_C(-2469501117696455323), INT64_C( 2339328587648411167),
                            INT64_C( 8220620103791574591), INT64_C(  273538927111600315),
                            INT64_C(-3298288074488883789), INT64_C(-8357787233131660724)),
      simde_mm512_set_epi64(INT64_C(  252325274594050063), INT64_C( 8251648155281492223),
                            INT64_C(  329344140649481477), INT64_C(-4956133557414585628),
                            INT64_C( 4757067868831771709), INT64_C(  147282005282398210),
                            INT64_C( 5780933484690985089), INT64_C(  577059746971148352)) },
    { simde_mm512_set_epi64(INT64_C(-4285851555602414983), INT64_C(-8492982904341423564),
                            INT64_C(-2837093742585682248), INT64_C(  267283033869441308),
                            INT64_C( 4311088349833897908), INT64_C( -647706517356585524),
                            INT64_C(-3770716194274572842), INT64_C(-8566807519504738391)),
      UINT8_C( 75),
      simde_mm512_set_epi64(INT64_C(-6282230583383062251), INT64_C(-7841791912404359359),
                            INT64_C(-7579575622870303941), INT64_C(-2922061146712111361),
                            INT64_C( 4606944383693507801), INT64_C(-6882069134795290712),
                            INT64_C(-4540648442557822523), INT64_C( 8626282944079879495)),
      simde_mm512_set_epi64(INT64_C(-1823698107073259294), INT64_C( 8029233569224881686),
                            INT64_C(   46900467487790247), INT64_C( 8663098726891022114),
                            INT64_C( 2596646339415618602), INT64_C( 7059567741718714192),
                            INT64_C( 7446336952031093968), INT64_C(   16931348739669095)),
      simde_mm512_set_epi64(INT64_C(-4285851555602414983), INT64_C( 7800656914580246550),
                            INT64_C(-2837093742585682248), INT64_C(  267283033869441308),
                            INT64_C(         88250757154), INT64_C( -647706517356585524),
                            INT64_C( 2810971851134903312), INT64_C(    2252181026775072)) },
    { simde_mm512_set_epi64(INT64_C( 2037127205197222183), INT64_C( 3451898891201360501),
                            INT64_C( 1455211247092394628), INT64_C( 2206658725580708086),
                            INT64_C( 5349364315141837270), INT64_C( 7849256443344717184),
                            INT64_C( 4856719246957022704), INT64_C(-4923001172558722698)),
      UINT8_C(149),
      simde_mm512_set_epi64(INT64_C( 6411014556179012579), INT64_C(-8290562023531042118),
                            INT64_C( 3513406971994598159), INT64_C(  170515694744852127),
                            INT64_C( 7762613428125762288), INT64_C( 4486051683696872920),
                            INT64_C(-3347799382542858009), INT64_C( 7877354972766519961)),
      simde_mm512_set_epi64(INT64_C( 2384233607786009160), INT64_C( 7136321197786935066),
                            INT64_C(-2775012291419678803), INT64_C( 1447324989515017380),
                            INT64_C(-5436087904826886612), INT64_C( 7888585058472078205),
                            INT64_C(-7864278168616859201), INT64_C( 8559884086409161720)),
      simde_mm512_set_epi64(INT64_C( 2379589521848270856), INT64_C( 3451898891201360501),
                            INT64_C( 1455211247092394628), INT64_C( 1441191875528796192),
                            INT64_C( 5349364315141837270), INT64_C( 4699579053875929637),
                            INT64_C( 4856719246957022704), INT64_C( 1335881482333858144)) },
    { simde_mm512_set_epi64(INT64_C( -626073311570320561), INT64_C( 4678237318537021585),
                            INT64_C( 7326175960335696621), INT64_C( 2614088339478761539),
                            INT64_C(-3404519381245739218), INT64_C( 8481274767690754747),
                            INT64_C(-4945537623263429760), INT64_C( 5945167030889147721)),
      UINT8_C(209),
      simde_mm512_set_epi64(INT64_C( 1396956538408270925), INT64_C(  433531675836732237),
                            INT64_C(-2740776246441943234), INT64_C(  627773489989817177),
                            INT64_C( 2334235533617502306), INT64_C( 5200994462656867787),
                            INT64_C( 6058971438237170661), INT64_C(-1718043134590880356)),
      simde_mm512_set_epi64(INT64_C( 6582702301060698834), INT64_C(-6620728110496909408),
                            INT64_C(-2674893574601157335), INT64_C(-3191892667818640289),
                            INT64_C(-1755995440120031315), INT64_C(-9164966479234216120),
                            INT64_C( -811539623059483440), INT64_C(-8790398035654865383)),
      simde_mm512_set_epi64(INT64_C( 5194902496598033042), INT64_C(-6910140186789469024),
                            INT64_C( 7326175960335696621), INT64_C(-3242571914706752506),
                            INT64_C(-3404519381245739218), INT64_C( 8481274767690754747),
                            INT64_C(-4945537623263429760), INT64_C(  432964590381304321)) },
    { simde_mm512_set_epi64(INT64_C(-6743158443935274483), INT64_C( -109319504177728220),
                            INT64_C(-4028288193005214442), INT64_C(  132288430860812468),
                            INT64_C(  917336920958928215), INT64_C(-8592087087533075804),
                            INT64_C( -911564553413882344), INT64_C(-5778334739542351628)),
      UINT8_C(132),
      simde_mm512_set_epi64(INT64_C(-8373098054511418162), INT64_C( 7896680406183363835),
                            INT64_C( 4931162839211744539), INT64_C(-7345169465412510410),
                            INT64_C(-7349547769362151281), INT64_C(    1089692206936889),
                            INT64_C( 6524506004040415129), INT64_C( 6226593529101379713)),
      simde_mm512_set_epi64(INT64_C( 3458147115787789114), INT64_C( 7210094384770191006),
                            INT64_C( 7088560670460655534), INT64_C( -803268445524244375),
                            INT64_C( 4723424603414443741), INT64_C( 1370109689785890561),
                            INT64_C(-4376650697011830162), INT64_C( -620804834547376669)),
      simde_mm512_set_epi64(INT64_C( 2607874799996928816), INT64_C( -109319504177728220),
                            INT64_C(-4028288193005214442), INT64_C(  132288430860812468),
                            INT64_C(  917336920958928215), INT64_C( 1369094837600650240),
                            INT64_C( -911564553413882344), INT64_C(-5778334739542351628)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_andnot_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_andnot_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C(  7),
      simde_mm512_set_epi64(INT64_C(-2016264017930850215), INT64_C( 6207900603916400351),
                            INT64_C( 7392720324711365837), INT64_C( 8770333430120422633),
                            INT64_C(  490532205378570002), INT64_C(-6106476949393880649),
                            INT64_C(-1854090463849988422), INT64_C( 2161894352221900559)),
      simde_mm512_set_epi64(INT64_C( 2471053143203888378), INT64_C( 4307108638624930374),
                            INT64_C( 8813537095665060151), INT64_C( -722272124812023485),
                            INT64_C( -967288076808354317), INT64_C(-6013850093851417513),
                            INT64_C( 3331958923341291108), INT64_C( -281534168919433716)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(  327214808695383104),
                            INT64_C(  592518518720905284), INT64_C(-2299289876066988032)) },
    { UINT8_C( 76),
      simde_mm512_set_epi64(INT64_C(-7188491746248886702), INT64_C( 3795103503776882624),
                            INT64_C( 8025930014425820340), INT64_C(-7929605366413196523),
                            INT64_C( 5924420044782879602), INT64_C(-3302350069387149227),
                            INT64_C(-1821341009738891830), INT64_C(-6812922588519498817)),
      simde_mm512_set_epi64(INT64_C(-1266328346505933550), INT64_C( 1669938728598205410),
                            INT64_C(-7350359895777029108), INT64_C( 9139543262716722238),
                            INT64_C(-9200593584210926828), INT64_C(-3449434666635797941),
                            INT64_C( 4314658246940308870), INT64_C( -478133805478226079)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(  217316721059520546),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-9205313376526131196), INT64_C(       5910164412938),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(148),
      simde_mm512_set_epi64(INT64_C( 2173045647004856331), INT64_C(-9109531323294262314),
                            INT64_C(-2493109132018654878), INT64_C( 6270825741977490200),
                            INT64_C( 8719769943602297687), INT64_C(-4201021528893071940),
                            INT64_C( 9011627797455533120), INT64_C( 6620301637478416060)),
      simde_mm512_set_epi64(INT64_C(-4851330938418837166), INT64_C( 8567660546009495156),
                            INT64_C(-2946935282469126440), INT64_C(-3944680176869437518),
                            INT64_C(-3189291857021003507), INT64_C(  852944387991302704),
                            INT64_C( 5948575888921546761), INT64_C( 4930911444432807162)),
      simde_mm512_set_epi64(INT64_C(-6880224560885528240), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-8628461452576158558),
                            INT64_C(                   0), INT64_C(  739720922782507520),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C( 97),
      simde_mm512_set_epi64(INT64_C( -924406031683798297), INT64_C( 1082742291630099615),
                            INT64_C( 3950666752159487194), INT64_C( 8443851551588188807),
                            INT64_C( 5838662214875022266), INT64_C(-6073322957639126750),
                            INT64_C( 1174103819847041898), INT64_C(  693926700598930845)),
      simde_mm512_set_epi64(INT64_C(  917406711858321823), INT64_C(-2954398701286057389),
                            INT64_C(  580508427727522845), INT64_C(-4656281121400174897),
                            INT64_C(-3028496641912979897), INT64_C( 6357018899588818011),
                            INT64_C( 5102737467710367164), INT64_C( 1099306012957445482)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-3388587049163943360),
                            INT64_C(  579945122294155269), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(  450505469795237986)) },
    { UINT8_C(210),
      simde_mm512_set_epi64(INT64_C( 4586733821042914141), INT64_C(-6562128612845469564),
                            INT64_C(-5088971089241108691), INT64_C( 4584509013736167571),
                            INT64_C( 7541158438725419821), INT64_C(-6577447853347647248),
                            INT64_C( 8000393737083977627), INT64_C(-3838210298295657456)),
      simde_mm512_set_epi64(INT64_C(-3810154219907114893), INT64_C( 1390546034528663938),
                            INT64_C(-7278948997228835946), INT64_C(-6400015342302035742),
                            INT64_C(-5025729231272531675), INT64_C( -727304839347940122),
                            INT64_C( 5841837551579279726), INT64_C(-6256756974903097514)),
      simde_mm512_set_epi64(INT64_C(-4604784503990056926), INT64_C( 1369116277674653954),
                            INT64_C(                   0), INT64_C(-9214222637876019104),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 1157530966609723492), INT64_C(                   0)) },
    { UINT8_C(171),
      simde_mm512_set_epi64(INT64_C( 2614947921582018787), INT64_C(-4851561713766233132),
                            INT64_C(-9121795968209612126), INT64_C(  784084589312935430),
                            INT64_C( 3206750945776122646), INT64_C( 2956179786298753960),
                            INT64_C( 5449808455866424595), INT64_C(  314020808054955060)),
      simde_mm512_set_epi64(INT64_C(  420924716680581769), INT64_C(  634178498505834615),
                            INT64_C(-2861544115657502554), INT64_C(-7045300656768620560),
                            INT64_C( 3724569018417139461), INT64_C( 7684038547017787602),
                            INT64_C( 4661447160348399809), INT64_C( 8780209518656646828)),
      simde_mm512_set_epi64(INT64_C(  114072716522619400), INT64_C(                   0),
                            INT64_C( 6341349786890797060), INT64_C(                   0),
                            INT64_C( 1382694151414203393), INT64_C(                   0),
                            INT64_C(    4574054841401536), INT64_C( 8755017506026431112)) },
    { UINT8_C(225),
      simde_mm512_set_epi64(INT64_C(-3697729744057786539), INT64_C(-2459882991819182775),
                            INT64_C( 6065837030945349572), INT64_C( 8437722782224197038),
                            INT64_C( 1700648554253726454), INT64_C(-4293199790864835662),
                            INT64_C( 6581402203822969825), INT64_C(-6231169800047978744)),
      simde_mm512_set_epi64(INT64_C( 8326587265612039337), INT64_C( 6780517041864519531),
                            INT64_C(-7817226648374121699), INT64_C(-3500732471169369834),
                            INT64_C( 1796671772602068213), INT64_C( 1885612779837593615),
                            INT64_C(-6040660189943903948), INT64_C(  675381603587673544)),
      simde_mm512_set_epi64(INT64_C( 3675211075874242728), INT64_C(  144396663190979106),
                            INT64_C(-8970746854625959911), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(   25068767419582656)) },
    { UINT8_C(182),
      simde_mm512_set_epi64(INT64_C(-3172486234888138881), INT64_C( 3556874573334620913),
                            INT64_C(-7174772828994546158), INT64_C( -768272060832782008),
                            INT64_C(-7948383401788128664), INT64_C(-3962825949835743119),
                            INT64_C(-4177466042331622142), INT64_C(-4344904134560657490)),
      simde_mm512_set_epi64(INT64_C( 3131865100191000199), INT64_C( 3277342092864256055),
                            INT64_C( 2638156770812089616), INT64_C(-5499406567603861656),
                            INT64_C( 5836973950118592576), INT64_C(-4232123399129603430),
                            INT64_C( 8656431254350139121), INT64_C(-8853511068983619849)),
      simde_mm512_set_epi64(INT64_C( 2884148896870883456), INT64_C(                   0),
                            INT64_C( 2346526704673489152), INT64_C(  191440763308049952),
                            INT64_C(                   0), INT64_C(  307450048046744202),
                            INT64_C( 4044604204229069553), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_andnot_epi64(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_blend_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(12684),
      simde_mm512_set_epi32(INT32_C( 2139597246), INT32_C(-2035467821), INT32_C(-1381016544), INT32_C( -293624181),
                            INT32_C( 1610331725), INT32_C(  134146865), INT32_C(  837546022), INT32_C(-1561535917),
                            INT32_C(  522767958), INT32_C(  350987310), INT32_C( 1200416302), INT32_C( 1035712103),
                            INT32_C( -697441354), INT32_C(-1209277994), INT32_C( 1905768221), INT32_C(  941422574)),
      simde_mm512_set_epi32(INT32_C( -854245863), INT32_C(   77819890), INT32_C( -597706880), INT32_C(  -28687979),
                            INT32_C( 1384494246), INT32_C(-1885694903), INT32_C(  479641666), INT32_C(  436747778),
                            INT32_C( 1142933685), INT32_C(  -36150185), INT32_C(  234764144), INT32_C( -925516387),
                            INT32_C( 1528722995), INT32_C( 1957265461), INT32_C( -558613563), INT32_C(  535737103)),
      simde_mm512_set_epi32(INT32_C( 2139597246), INT32_C(-2035467821), INT32_C( -597706880), INT32_C(  -28687979),
                            INT32_C( 1610331725), INT32_C(  134146865), INT32_C(  837546022), INT32_C(  436747778),
                            INT32_C( 1142933685), INT32_C(  350987310), INT32_C( 1200416302), INT32_C( 1035712103),
                            INT32_C( 1528722995), INT32_C( 1957265461), INT32_C( 1905768221), INT32_C(  941422574)) },
    { UINT16_C(12889),
      simde_mm512_set_epi32(INT32_C(-1009854213), INT32_C(   19207470), INT32_C( 2053019824), INT32_C( 1679893080),
                            INT32_C( -761309092), INT32_C(-1797634461), INT32_C( 1499461014), INT32_C( -349931656),
                            INT32_C( 1308377490), INT32_C(-1862232386), INT32_C( -706282442), INT32_C( 1752887042),
                            INT32_C( 1045610342), INT32_C(   88096217), INT32_C(-1144289151), INT32_C( 1410502196)),
      simde_mm512_set_epi32(INT32_C( 1437842356), INT32_C(-1817562257), INT32_C(  808924311), INT32_C( 1765692072),
                            INT32_C(-1346910557), INT32_C(  -92284700), INT32_C( 1533217965), INT32_C( 1732689820),
                            INT32_C(-1401128233), INT32_C( -762168473), INT32_C(   97276971), INT32_C( 2145432631),
                            INT32_C(-1561525899), INT32_C(-2005427238), INT32_C( -455460474), INT32_C( -933959435)),
      simde_mm512_set_epi32(INT32_C(-1009854213), INT32_C(   19207470), INT32_C(  808924311), INT32_C( 1765692072),
                            INT32_C( -761309092), INT32_C(-1797634461), INT32_C( 1533217965), INT32_C( -349931656),
                            INT32_C( 1308377490), INT32_C( -762168473), INT32_C( -706282442), INT32_C( 2145432631),
                            INT32_C(-1561525899), INT32_C(   88096217), INT32_C(-1144289151), INT32_C( -933959435)) },
    { UINT16_C(18390),
      simde_mm512_set_epi32(INT32_C(  191788932), INT32_C(  410937469), INT32_C(  218604234), INT32_C( -632545043),
                            INT32_C(  246082482), INT32_C( 2029197195), INT32_C( 1188965621), INT32_C( -844747875),
                            INT32_C(  989502056), INT32_C( 2119540790), INT32_C(-1766179858), INT32_C(-1109416221),
                            INT32_C(-1963025204), INT32_C( -932958949), INT32_C(   47867627), INT32_C( -567270366)),
      simde_mm512_set_epi32(INT32_C(  558500028), INT32_C( 1400907983), INT32_C(-1581715774), INT32_C(  -29022872),
                            INT32_C( 1773849857), INT32_C( -311975417), INT32_C( 1183760637), INT32_C(-1160252785),
                            INT32_C( 2107838031), INT32_C( 1909470743), INT32_C(-2018375211), INT32_C(  267812095),
                            INT32_C( -225335539), INT32_C( -871226308), INT32_C( -872412082), INT32_C( 1435481672)),
      simde_mm512_set_epi32(INT32_C(  191788932), INT32_C( 1400907983), INT32_C(  218604234), INT32_C( -632545043),
                            INT32_C(  246082482), INT32_C( -311975417), INT32_C( 1183760637), INT32_C(-1160252785),
                            INT32_C( 2107838031), INT32_C( 1909470743), INT32_C(-1766179858), INT32_C(  267812095),
                            INT32_C(-1963025204), INT32_C( -871226308), INT32_C( -872412082), INT32_C( -567270366)) },
    { UINT16_C(46409),
      simde_mm512_set_epi32(INT32_C( -239336968), INT32_C( 1154172094), INT32_C( 1382102779), INT32_C(-1946237060),
                            INT32_C( -995869857), INT32_C( 1050338310), INT32_C(-2047829467), INT32_C( -881484106),
                            INT32_C(  -67227529), INT32_C( -510303256), INT32_C( -387177060), INT32_C( 2007995362),
                            INT32_C( 1596393504), INT32_C( 1716443052), INT32_C( 1056333857), INT32_C( -879795312)),
      simde_mm512_set_epi32(INT32_C(-1321156942), INT32_C(-1763902924), INT32_C( -145319736), INT32_C(  356975558),
                            INT32_C(  109934631), INT32_C( 1326272066), INT32_C(  723198088), INT32_C(-1812908400),
                            INT32_C( 1665386649), INT32_C( 1770577849), INT32_C(-2015166919), INT32_C(-1565649496),
                            INT32_C( 1045296779), INT32_C( 1401153164), INT32_C( -294475079), INT32_C(  378377774)),
      simde_mm512_set_epi32(INT32_C(-1321156942), INT32_C( 1154172094), INT32_C( -145319736), INT32_C(  356975558),
                            INT32_C( -995869857), INT32_C( 1326272066), INT32_C(-2047829467), INT32_C(-1812908400),
                            INT32_C(  -67227529), INT32_C( 1770577849), INT32_C( -387177060), INT32_C( 2007995362),
                            INT32_C( 1045296779), INT32_C( 1716443052), INT32_C( 1056333857), INT32_C(  378377774)) },
    { UINT16_C(35033),
      simde_mm512_set_epi32(INT32_C(  576121858), INT32_C(  -83274089), INT32_C( 1081604364), INT32_C( 1853977291),
                            INT32_C(-1408149319), INT32_C(-1793071292), INT32_C( -580417531), INT32_C( 1708989591),
                            INT32_C(-1803428364), INT32_C(-1884594628), INT32_C(-1049896819), INT32_C( 1351777033),
                            INT32_C( -543435799), INT32_C(   45073785), INT32_C(  310971883), INT32_C(  586295496)),
      simde_mm512_set_epi32(INT32_C(  274706498), INT32_C( 1339140311), INT32_C(  773365916), INT32_C( -407443831),
                            INT32_C(  -44487881), INT32_C( -363465063), INT32_C( -778555208), INT32_C( -640038352),
                            INT32_C( -428291654), INT32_C(-1080717955), INT32_C(  906661653), INT32_C( 1353353955),
                            INT32_C(-1311936279), INT32_C( 1168846380), INT32_C(  -71806717), INT32_C(  617275327)),
      simde_mm512_set_epi32(INT32_C(  274706498), INT32_C(  -83274089), INT32_C( 1081604364), INT32_C( 1853977291),
                            INT32_C(  -44487881), INT32_C(-1793071292), INT32_C( -580417531), INT32_C( 1708989591),
                            INT32_C( -428291654), INT32_C(-1080717955), INT32_C(-1049896819), INT32_C( 1353353955),
                            INT32_C(-1311936279), INT32_C(   45073785), INT32_C(  310971883), INT32_C(  617275327)) },
    { UINT16_C(62826),
      simde_mm512_set_epi32(INT32_C( -943712419), INT32_C( -981833223), INT32_C( 2020022414), INT32_C(  630972788),
                            INT32_C( 1615502534), INT32_C(  991949979), INT32_C(  601817641), INT32_C(-2063962607),
                            INT32_C(-1545145030), INT32_C( 1626575612), INT32_C(-1511315708), INT32_C( 1422623346),
                            INT32_C( 1496301111), INT32_C(-1751918881), INT32_C(  333195983), INT32_C( 1655699275)),
      simde_mm512_set_epi32(INT32_C(-1770653828), INT32_C( -674401292), INT32_C(-2023667251), INT32_C( 1038799540),
                            INT32_C(-1877506849), INT32_C(  791301479), INT32_C(-2115975814), INT32_C( 1430860109),
                            INT32_C(-2123570597), INT32_C(-2010985064), INT32_C( 1367050649), INT32_C( -268988786),
                            INT32_C( 1975120887), INT32_C(   83320183), INT32_C( 2120549505), INT32_C(  163195572)),
      simde_mm512_set_epi32(INT32_C(-1770653828), INT32_C( -674401292), INT32_C(-2023667251), INT32_C( 1038799540),
                            INT32_C( 1615502534), INT32_C(  791301479), INT32_C(  601817641), INT32_C( 1430860109),
                            INT32_C(-1545145030), INT32_C(-2010985064), INT32_C( 1367050649), INT32_C( 1422623346),
                            INT32_C( 1975120887), INT32_C(-1751918881), INT32_C( 2120549505), INT32_C( 1655699275)) },
    { UINT16_C( 2185),
      simde_mm512_set_epi32(INT32_C( 1990735603), INT32_C( -226564964), INT32_C( 2030923034), INT32_C( 1100474296),
                            INT32_C( 1370205128), INT32_C( 1609607637), INT32_C( -586788969), INT32_C( 1072075481),
                            INT32_C( -723316478), INT32_C(-1331909036), INT32_C(-1524508879), INT32_C( -832646284),
                            INT32_C(-1636241911), INT32_C(  215718784), INT32_C(  478814832), INT32_C( 1231303010)),
      simde_mm512_set_epi32(INT32_C(   96871414), INT32_C(  245175863), INT32_C( 1584772139), INT32_C( 1604795180),
                            INT32_C(-1972440261), INT32_C(-1471456371), INT32_C( 1108051273), INT32_C(   -4141330),
                            INT32_C(-1064630270), INT32_C( -911670021), INT32_C(-1808660435), INT32_C(-2035294308),
                            INT32_C(-2014140232), INT32_C(-1194657062), INT32_C(-1732550793), INT32_C(-1643523135)),
      simde_mm512_set_epi32(INT32_C( 1990735603), INT32_C( -226564964), INT32_C( 2030923034), INT32_C( 1100474296),
                            INT32_C(-1972440261), INT32_C( 1609607637), INT32_C( -586788969), INT32_C( 1072075481),
                            INT32_C(-1064630270), INT32_C(-1331909036), INT32_C(-1524508879), INT32_C( -832646284),
                            INT32_C(-2014140232), INT32_C(  215718784), INT32_C(  478814832), INT32_C(-1643523135)) },
    { UINT16_C(11244),
      simde_mm512_set_epi32(INT32_C( 1605800253), INT32_C( 1825375434), INT32_C( -280209407), INT32_C( 1616462798),
                            INT32_C( 1380939760), INT32_C( 1670822362), INT32_C(-1872387325), INT32_C(  451486273),
                            INT32_C( -978012107), INT32_C( -718128180), INT32_C(-1625787118), INT32_C( -879101117),
                            INT32_C(  173600397), INT32_C( 1426384314), INT32_C( -517748272), INT32_C(   76760759)),
      simde_mm512_set_epi32(INT32_C(  805228357), INT32_C(  980137697), INT32_C( -555168446), INT32_C(-2016549382),
                            INT32_C(  927860791), INT32_C(-1515288559), INT32_C( -918296563), INT32_C(-1858382028),
                            INT32_C( 1365882699), INT32_C(  616589376), INT32_C(-1573056329), INT32_C( 1014781400),
                            INT32_C( -260360112), INT32_C( -953114112), INT32_C( 1191757764), INT32_C( -706360509)),
      simde_mm512_set_epi32(INT32_C( 1605800253), INT32_C( 1825375434), INT32_C( -555168446), INT32_C( 1616462798),
                            INT32_C(  927860791), INT32_C( 1670822362), INT32_C( -918296563), INT32_C(-1858382028),
                            INT32_C( 1365882699), INT32_C(  616589376), INT32_C(-1573056329), INT32_C( -879101117),
                            INT32_C( -260360112), INT32_C( -953114112), INT32_C( -517748272), INT32_C(   76760759)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_blend_epi32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_blend_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C(140),
      simde_mm512_set_epi64(INT64_C(-8742267720341431264), INT64_C(-1261106253099452851),
                            INT64_C(  576156398873473062), INT64_C(-6706745694521602474),
                            INT64_C( 1507479018961430062), INT64_C( 4448349614053909430),
                            INT64_C(-5193809434096716003), INT64_C( 4043379170809950035)),
      simde_mm512_set_epi64(INT64_C(  334233886225577856), INT64_C( -123213930208840538),
                            INT64_C(-8098997938139250622), INT64_C( 1875817424253601973),
                            INT64_C( -155263862084585616), INT64_C(-3975062612548356557),
                            INT64_C( 8406391148321717189), INT64_C( 2300973338778380734)),
      simde_mm512_set_epi64(INT64_C(  334233886225577856), INT64_C(-1261106253099452851),
                            INT64_C(  576156398873473062), INT64_C(-6706745694521602474),
                            INT64_C( -155263862084585616), INT64_C(-3975062612548356557),
                            INT64_C(-5193809434096716003), INT64_C( 4043379170809950035)) },
    { UINT8_C( 25),
      simde_mm512_set_epi64(INT64_C( 8817653003799568984), INT64_C(-3269797649790122397),
                            INT64_C( 6440136020702033784), INT64_C( 5619438532805301950),
                            INT64_C(-3033459988376129790), INT64_C( 4490862223337471449),
                            INT64_C(-4914684479302103500), INT64_C(  357175739365339737)),
      simde_mm512_set_epi64(INT64_C( 3474303462450025128), INT64_C(-5784936788749461276),
                            INT64_C( 6585121019047362460), INT64_C(-6017799934704469145),
                            INT64_C(  417801411244373047), INT64_C(-6706702665772459046),
                            INT64_C(-1956187837089650443), INT64_C(-4337290818543610578)),
      simde_mm512_set_epi64(INT64_C( 8817653003799568984), INT64_C(-3269797649790122397),
                            INT64_C( 6440136020702033784), INT64_C(-6017799934704469145),
                            INT64_C(  417801411244373047), INT64_C( 4490862223337471449),
                            INT64_C(-4914684479302103500), INT64_C(-4337290818543610578)) },
    { UINT8_C(111),
      simde_mm512_set_epi64(INT64_C(-2716760272685831246), INT64_C( 8715335590848900341),
                            INT64_C(-3628164495500993944), INT64_C( 9103358378116791278),
                            INT64_C(-4764906384514966324), INT64_C(-4007028174417664277),
                            INT64_C(-2436407666547579589), INT64_C( 1334270375494925236)),
      simde_mm512_set_epi64(INT64_C( -124652284302144255), INT64_C(-1339924211987201795),
                            INT64_C(-4983247764560081329), INT64_C( 8201114396130413013),
                            INT64_C( 1150244193567876877), INT64_C(-3741888496852267954),
                            INT64_C( 6165346835439187844), INT64_C( 1764962990274618058)),
      simde_mm512_set_epi64(INT64_C(-2716760272685831246), INT64_C(-1339924211987201795),
                            INT64_C(-4983247764560081329), INT64_C( 9103358378116791278),
                            INT64_C( 1150244193567876877), INT64_C(-3741888496852267954),
                            INT64_C( 6165346835439187844), INT64_C( 1764962990274618058)) },
    { UINT8_C(194),
      simde_mm512_set_epi64(INT64_C(-4277228465836858362), INT64_C(-8795360585136628042),
                            INT64_C( -288740034661227544), INT64_C(-1662912808453434398),
                            INT64_C( 6856457892943288236), INT64_C( 4536919372887712656),
                            INT64_C(-1781208167188155063), INT64_C( 2398739356475992271)),
      simde_mm512_set_epi64(INT64_C(  472165646169099842), INT64_C( 3106112138971788944),
                            INT64_C( 7152781194420608953), INT64_C(-8655076010356763224),
                            INT64_C( 4489515481820292748), INT64_C(-1264760833413638610),
                            INT64_C(-1027944449129626434), INT64_C( 5936086237864445820)),
      simde_mm512_set_epi64(INT64_C(  472165646169099842), INT64_C( 3106112138971788944),
                            INT64_C( -288740034661227544), INT64_C(-1662912808453434398),
                            INT64_C( 6856457892943288236), INT64_C( 4536919372887712656),
                            INT64_C(-1027944449129626434), INT64_C( 2398739356475992271)) },
    { UINT8_C(198),
      simde_mm512_set_epi64(INT64_C(-7701182554821916667), INT64_C( 7340054405040954868),
                            INT64_C(-8094272290232215411), INT64_C( 5805838151970444265),
                            INT64_C(  193590432792907243), INT64_C( 2518119983696523684),
                            INT64_C(-7781788212556415310), INT64_C(-7575905367749125944)),
      simde_mm512_set_epi64(INT64_C(-1561070555307167560), INT64_C(-2748943786159060550),
                            INT64_C(-4641648272018338027), INT64_C( 5812610979620286697),
                            INT64_C( 5020156980371149059), INT64_C( 2651177342668827650),
                            INT64_C( -357659487777588980), INT64_C( 7962771835258493113)),
      simde_mm512_set_epi64(INT64_C(-1561070555307167560), INT64_C(-2748943786159060550),
                            INT64_C(-8094272290232215411), INT64_C( 5805838151970444265),
                            INT64_C(  193590432792907243), INT64_C( 2651177342668827650),
                            INT64_C( -357659487777588980), INT64_C(-7575905367749125944)) },
    { UINT8_C( 55),
      simde_mm512_set_epi64(INT64_C( 2584787088481873425), INT64_C(-6636347369800363268),
                            INT64_C(-6491051538368462222), INT64_C( 6426564339256514271),
                            INT64_C( 1431065851799271243), INT64_C(-5052971989011532438),
                            INT64_C( 1179855426247829719), INT64_C( 3321581320948606601)),
      simde_mm512_set_epi64(INT64_C(-9088046918826118835), INT64_C(-9120666262578213480),
                            INT64_C( 5871437833456553614), INT64_C( 8483079615394831735),
                            INT64_C( 9107690773687184052), INT64_C(-4053213973120914951),
                            INT64_C( 8675930205947945332), INT64_C( 6938530551127078043)),
      simde_mm512_set_epi64(INT64_C( 2584787088481873425), INT64_C(-6636347369800363268),
                            INT64_C( 5871437833456553614), INT64_C( 8483079615394831735),
                            INT64_C( 1431065851799271243), INT64_C(-4053213973120914951),
                            INT64_C( 8675930205947945332), INT64_C( 6938530551127078043)) },
    { UINT8_C(103),
      simde_mm512_set_epi64(INT64_C( 4604529133310120194), INT64_C(-5720505748096428239),
                            INT64_C(-3576188556257202679), INT64_C(  926505122891702896),
                            INT64_C( 5288406162053320871), INT64_C(-5908713324082235524),
                            INT64_C(-2896531491248846387), INT64_C( 4461610053817304287)),
      simde_mm512_set_epi64(INT64_C(  -17786873681606654), INT64_C(-3915592922452326355),
                            INT64_C(-8741522488314124104), INT64_C(-5131013008663027849),
                            INT64_C(-7058878113053657357), INT64_C( -973089108768494310),
                            INT64_C( 4726501112778828744), INT64_C( 6913212164015017879)),
      simde_mm512_set_epi64(INT64_C( 4604529133310120194), INT64_C(-3915592922452326355),
                            INT64_C(-8741522488314124104), INT64_C(  926505122891702896),
                            INT64_C( 5288406162053320871), INT64_C( -973089108768494310),
                            INT64_C( 4726501112778828744), INT64_C( 6913212164015017879)) },
    { UINT8_C( 73),
      simde_mm512_set_epi64(INT64_C(-4200530011080213556), INT64_C(-6982702498652226749),
                            INT64_C(  745608029114000826), INT64_C(-2223711895723751753),
                            INT64_C( 4918324162995104748), INT64_C(  416059555292452407),
                            INT64_C( 6806544510221761324), INT64_C(-8471566411485193331)),
      simde_mm512_set_epi64(INT64_C( 5866421522993801280), INT64_C(-6756225486806034984),
                            INT64_C(-1118238162881043968), INT64_C( 5118560624722692931),
                            INT64_C( 6896859572368901322), INT64_C(-1203490237480090674),
                            INT64_C( 5931091108616911322), INT64_C(-8041842325868436927)),
      simde_mm512_set_epi64(INT64_C(-4200530011080213556), INT64_C(-6756225486806034984),
                            INT64_C(  745608029114000826), INT64_C(-2223711895723751753),
                            INT64_C( 6896859572368901322), INT64_C(  416059555292452407),
                            INT64_C( 6806544510221761324), INT64_C(-8041842325868436927)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_blend_epi64(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_blend_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(28658),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   986.64), SIMDE_FLOAT32_C(   121.90), SIMDE_FLOAT32_C(  -796.62), SIMDE_FLOAT32_C(   983.17),
                         SIMDE_FLOAT32_C(   569.02), SIMDE_FLOAT32_C(   -88.58), SIMDE_FLOAT32_C(  -750.53), SIMDE_FLOAT32_C(    52.16),
                         SIMDE_FLOAT32_C(   863.27), SIMDE_FLOAT32_C(  -937.53), SIMDE_FLOAT32_C(   272.85), SIMDE_FLOAT32_C(  -836.56),
                         SIMDE_FLOAT32_C(  -517.71), SIMDE_FLOAT32_C(   436.89), SIMDE_FLOAT32_C(  -561.62), SIMDE_FLOAT32_C(  -796.29)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   721.67), SIMDE_FLOAT32_C(  -355.29), SIMDE_FLOAT32_C(  -776.65), SIMDE_FLOAT32_C(  -467.78),
                         SIMDE_FLOAT32_C(  -890.68), SIMDE_FLOAT32_C(  -288.13), SIMDE_FLOAT32_C(   739.88), SIMDE_FLOAT32_C(    -3.67),
                         SIMDE_FLOAT32_C(   356.91), SIMDE_FLOAT32_C(  -250.13), SIMDE_FLOAT32_C(  -609.99), SIMDE_FLOAT32_C(  -756.57),
                         SIMDE_FLOAT32_C(  -441.01), SIMDE_FLOAT32_C(   675.23), SIMDE_FLOAT32_C(  -112.56), SIMDE_FLOAT32_C(   752.66)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   986.64), SIMDE_FLOAT32_C(  -355.29), SIMDE_FLOAT32_C(  -776.65), SIMDE_FLOAT32_C(   983.17),
                         SIMDE_FLOAT32_C(  -890.68), SIMDE_FLOAT32_C(  -288.13), SIMDE_FLOAT32_C(   739.88), SIMDE_FLOAT32_C(    -3.67),
                         SIMDE_FLOAT32_C(   356.91), SIMDE_FLOAT32_C(  -250.13), SIMDE_FLOAT32_C(  -609.99), SIMDE_FLOAT32_C(  -756.57),
                         SIMDE_FLOAT32_C(  -517.71), SIMDE_FLOAT32_C(   436.89), SIMDE_FLOAT32_C(  -112.56), SIMDE_FLOAT32_C(  -796.29)) },
    { UINT16_C(13167),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -177.79), SIMDE_FLOAT32_C(   957.03), SIMDE_FLOAT32_C(  -193.15), SIMDE_FLOAT32_C(   645.09),
                         SIMDE_FLOAT32_C(    -0.96), SIMDE_FLOAT32_C(    66.15), SIMDE_FLOAT32_C(   565.09), SIMDE_FLOAT32_C(  -991.06),
                         SIMDE_FLOAT32_C(  -217.74), SIMDE_FLOAT32_C(   162.91), SIMDE_FLOAT32_C(   837.05), SIMDE_FLOAT32_C(   132.83),
                         SIMDE_FLOAT32_C(  -183.75), SIMDE_FLOAT32_C(  -958.98), SIMDE_FLOAT32_C(  -343.18), SIMDE_FLOAT32_C(  -412.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -623.32), SIMDE_FLOAT32_C(   372.80), SIMDE_FLOAT32_C(  -286.04), SIMDE_FLOAT32_C(   347.55),
                         SIMDE_FLOAT32_C(  -954.70), SIMDE_FLOAT32_C(   272.86), SIMDE_FLOAT32_C(   787.91), SIMDE_FLOAT32_C(   529.75),
                         SIMDE_FLOAT32_C(   -43.99), SIMDE_FLOAT32_C(   645.49), SIMDE_FLOAT32_C(  -301.76), SIMDE_FLOAT32_C(  -390.74),
                         SIMDE_FLOAT32_C(   671.11), SIMDE_FLOAT32_C(  -513.10), SIMDE_FLOAT32_C(   467.15), SIMDE_FLOAT32_C(  -961.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -177.79), SIMDE_FLOAT32_C(   957.03), SIMDE_FLOAT32_C(  -286.04), SIMDE_FLOAT32_C(   347.55),
                         SIMDE_FLOAT32_C(    -0.96), SIMDE_FLOAT32_C(    66.15), SIMDE_FLOAT32_C(   787.91), SIMDE_FLOAT32_C(   529.75),
                         SIMDE_FLOAT32_C(  -217.74), SIMDE_FLOAT32_C(   645.49), SIMDE_FLOAT32_C(  -301.76), SIMDE_FLOAT32_C(   132.83),
                         SIMDE_FLOAT32_C(   671.11), SIMDE_FLOAT32_C(  -513.10), SIMDE_FLOAT32_C(   467.15), SIMDE_FLOAT32_C(  -961.27)) },
    { UINT16_C(10447),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   986.49), SIMDE_FLOAT32_C(   854.73), SIMDE_FLOAT32_C(   459.72), SIMDE_FLOAT32_C(  -110.83),
                         SIMDE_FLOAT32_C(  -875.29), SIMDE_FLOAT32_C(   594.30), SIMDE_FLOAT32_C(  -331.55), SIMDE_FLOAT32_C(  -808.64),
                         SIMDE_FLOAT32_C(   705.45), SIMDE_FLOAT32_C(   -55.08), SIMDE_FLOAT32_C(   606.63), SIMDE_FLOAT32_C(   -13.01),
                         SIMDE_FLOAT32_C(   483.39), SIMDE_FLOAT32_C(   565.56), SIMDE_FLOAT32_C(   735.84), SIMDE_FLOAT32_C(  -855.34)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   263.46), SIMDE_FLOAT32_C(  -173.99), SIMDE_FLOAT32_C(  -448.77), SIMDE_FLOAT32_C(   -18.46),
                         SIMDE_FLOAT32_C(    60.12), SIMDE_FLOAT32_C(   895.07), SIMDE_FLOAT32_C(   593.75), SIMDE_FLOAT32_C(  -910.69),
                         SIMDE_FLOAT32_C(  -898.20), SIMDE_FLOAT32_C(  -885.41), SIMDE_FLOAT32_C(  -446.34), SIMDE_FLOAT32_C(  -539.23),
                         SIMDE_FLOAT32_C(   177.56), SIMDE_FLOAT32_C(    85.90), SIMDE_FLOAT32_C(  -977.71), SIMDE_FLOAT32_C(   589.01)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   986.49), SIMDE_FLOAT32_C(   854.73), SIMDE_FLOAT32_C(  -448.77), SIMDE_FLOAT32_C(  -110.83),
                         SIMDE_FLOAT32_C(    60.12), SIMDE_FLOAT32_C(   594.30), SIMDE_FLOAT32_C(  -331.55), SIMDE_FLOAT32_C(  -808.64),
                         SIMDE_FLOAT32_C(  -898.20), SIMDE_FLOAT32_C(  -885.41), SIMDE_FLOAT32_C(   606.63), SIMDE_FLOAT32_C(   -13.01),
                         SIMDE_FLOAT32_C(   177.56), SIMDE_FLOAT32_C(    85.90), SIMDE_FLOAT32_C(  -977.71), SIMDE_FLOAT32_C(   589.01)) },
    { UINT16_C(64052),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -833.77), SIMDE_FLOAT32_C(  -382.41), SIMDE_FLOAT32_C(   155.80), SIMDE_FLOAT32_C(  -175.51),
                         SIMDE_FLOAT32_C(   270.94), SIMDE_FLOAT32_C(  -347.54), SIMDE_FLOAT32_C(  -823.80), SIMDE_FLOAT32_C(  -462.55),
                         SIMDE_FLOAT32_C(    93.71), SIMDE_FLOAT32_C(  -510.90), SIMDE_FLOAT32_C(   589.53), SIMDE_FLOAT32_C(   762.37),
                         SIMDE_FLOAT32_C(   -64.95), SIMDE_FLOAT32_C(  -200.72), SIMDE_FLOAT32_C(   590.31), SIMDE_FLOAT32_C(   904.10)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   932.33), SIMDE_FLOAT32_C(  -948.81), SIMDE_FLOAT32_C(  -663.23), SIMDE_FLOAT32_C(  -224.49),
                         SIMDE_FLOAT32_C(    61.61), SIMDE_FLOAT32_C(  -513.25), SIMDE_FLOAT32_C(   862.87), SIMDE_FLOAT32_C(   888.55),
                         SIMDE_FLOAT32_C(  -356.41), SIMDE_FLOAT32_C(   536.26), SIMDE_FLOAT32_C(    46.41), SIMDE_FLOAT32_C(   968.69),
                         SIMDE_FLOAT32_C(   819.71), SIMDE_FLOAT32_C(  -256.62), SIMDE_FLOAT32_C(  -508.11), SIMDE_FLOAT32_C(   806.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   932.33), SIMDE_FLOAT32_C(  -948.81), SIMDE_FLOAT32_C(  -663.23), SIMDE_FLOAT32_C(  -224.49),
                         SIMDE_FLOAT32_C(    61.61), SIMDE_FLOAT32_C(  -347.54), SIMDE_FLOAT32_C(   862.87), SIMDE_FLOAT32_C(  -462.55),
                         SIMDE_FLOAT32_C(    93.71), SIMDE_FLOAT32_C(  -510.90), SIMDE_FLOAT32_C(    46.41), SIMDE_FLOAT32_C(   968.69),
                         SIMDE_FLOAT32_C(   -64.95), SIMDE_FLOAT32_C(  -256.62), SIMDE_FLOAT32_C(   590.31), SIMDE_FLOAT32_C(   904.10)) },
    { UINT16_C(43223),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.27), SIMDE_FLOAT32_C(   830.75), SIMDE_FLOAT32_C(   701.96), SIMDE_FLOAT32_C(   496.75),
                         SIMDE_FLOAT32_C(  -369.80), SIMDE_FLOAT32_C(  -455.71), SIMDE_FLOAT32_C(  -712.56), SIMDE_FLOAT32_C(   961.22),
                         SIMDE_FLOAT32_C(  -136.67), SIMDE_FLOAT32_C(   165.04), SIMDE_FLOAT32_C(  -204.19), SIMDE_FLOAT32_C(   122.42),
                         SIMDE_FLOAT32_C(  -370.53), SIMDE_FLOAT32_C(  -979.01), SIMDE_FLOAT32_C(  -726.98), SIMDE_FLOAT32_C(   156.30)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -639.87), SIMDE_FLOAT32_C(   979.28), SIMDE_FLOAT32_C(   637.46), SIMDE_FLOAT32_C(   800.56),
                         SIMDE_FLOAT32_C(  -577.80), SIMDE_FLOAT32_C(   389.08), SIMDE_FLOAT32_C(   966.56), SIMDE_FLOAT32_C(  -731.72),
                         SIMDE_FLOAT32_C(  -496.34), SIMDE_FLOAT32_C(   344.28), SIMDE_FLOAT32_C(   729.72), SIMDE_FLOAT32_C(   160.21),
                         SIMDE_FLOAT32_C(   511.10), SIMDE_FLOAT32_C(   746.94), SIMDE_FLOAT32_C(  -855.19), SIMDE_FLOAT32_C(   203.47)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -639.87), SIMDE_FLOAT32_C(   830.75), SIMDE_FLOAT32_C(   637.46), SIMDE_FLOAT32_C(   496.75),
                         SIMDE_FLOAT32_C(  -577.80), SIMDE_FLOAT32_C(  -455.71), SIMDE_FLOAT32_C(  -712.56), SIMDE_FLOAT32_C(   961.22),
                         SIMDE_FLOAT32_C(  -496.34), SIMDE_FLOAT32_C(   344.28), SIMDE_FLOAT32_C(  -204.19), SIMDE_FLOAT32_C(   160.21),
                         SIMDE_FLOAT32_C(  -370.53), SIMDE_FLOAT32_C(   746.94), SIMDE_FLOAT32_C(  -855.19), SIMDE_FLOAT32_C(   203.47)) },
    { UINT16_C(29684),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -516.27), SIMDE_FLOAT32_C(  -631.52), SIMDE_FLOAT32_C(  -333.70), SIMDE_FLOAT32_C(    63.56),
                         SIMDE_FLOAT32_C(   874.74), SIMDE_FLOAT32_C(  -961.20), SIMDE_FLOAT32_C(  -924.01), SIMDE_FLOAT32_C(   542.80),
                         SIMDE_FLOAT32_C(  -706.18), SIMDE_FLOAT32_C(  -538.09), SIMDE_FLOAT32_C(    38.89), SIMDE_FLOAT32_C(  -242.57),
                         SIMDE_FLOAT32_C(  -337.54), SIMDE_FLOAT32_C(   184.20), SIMDE_FLOAT32_C(  -229.00), SIMDE_FLOAT32_C(  -133.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    57.66), SIMDE_FLOAT32_C(   125.72), SIMDE_FLOAT32_C(    14.67), SIMDE_FLOAT32_C(    11.14),
                         SIMDE_FLOAT32_C(  -363.42), SIMDE_FLOAT32_C(   -80.26), SIMDE_FLOAT32_C(   -12.54), SIMDE_FLOAT32_C(   560.55),
                         SIMDE_FLOAT32_C(   -59.35), SIMDE_FLOAT32_C(  -247.72), SIMDE_FLOAT32_C(  -719.76), SIMDE_FLOAT32_C(   280.49),
                         SIMDE_FLOAT32_C(   296.24), SIMDE_FLOAT32_C(  -303.23), SIMDE_FLOAT32_C(  -844.84), SIMDE_FLOAT32_C(   452.16)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -516.27), SIMDE_FLOAT32_C(   125.72), SIMDE_FLOAT32_C(    14.67), SIMDE_FLOAT32_C(    11.14),
                         SIMDE_FLOAT32_C(   874.74), SIMDE_FLOAT32_C(  -961.20), SIMDE_FLOAT32_C(   -12.54), SIMDE_FLOAT32_C(   560.55),
                         SIMDE_FLOAT32_C(   -59.35), SIMDE_FLOAT32_C(  -247.72), SIMDE_FLOAT32_C(  -719.76), SIMDE_FLOAT32_C(   280.49),
                         SIMDE_FLOAT32_C(  -337.54), SIMDE_FLOAT32_C(  -303.23), SIMDE_FLOAT32_C(  -229.00), SIMDE_FLOAT32_C(  -133.06)) },
    { UINT16_C( 5687),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -252.71), SIMDE_FLOAT32_C(   314.80), SIMDE_FLOAT32_C(   998.07), SIMDE_FLOAT32_C(   575.47),
                         SIMDE_FLOAT32_C(    52.24), SIMDE_FLOAT32_C(   443.69), SIMDE_FLOAT32_C(   234.67), SIMDE_FLOAT32_C(   894.50),
                         SIMDE_FLOAT32_C(  -487.55), SIMDE_FLOAT32_C(  -250.47), SIMDE_FLOAT32_C(  -500.78), SIMDE_FLOAT32_C(   379.78),
                         SIMDE_FLOAT32_C(   612.27), SIMDE_FLOAT32_C(  -899.55), SIMDE_FLOAT32_C(  -426.63), SIMDE_FLOAT32_C(   359.38)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -262.03), SIMDE_FLOAT32_C(    81.51), SIMDE_FLOAT32_C(  -484.02), SIMDE_FLOAT32_C(   504.24),
                         SIMDE_FLOAT32_C(   157.78), SIMDE_FLOAT32_C(    62.09), SIMDE_FLOAT32_C(   193.22), SIMDE_FLOAT32_C(   -72.99),
                         SIMDE_FLOAT32_C(   -54.28), SIMDE_FLOAT32_C(  -361.95), SIMDE_FLOAT32_C(   726.76), SIMDE_FLOAT32_C(   663.18),
                         SIMDE_FLOAT32_C(   290.10), SIMDE_FLOAT32_C(   238.07), SIMDE_FLOAT32_C(  -777.03), SIMDE_FLOAT32_C(   227.93)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -252.71), SIMDE_FLOAT32_C(   314.80), SIMDE_FLOAT32_C(   998.07), SIMDE_FLOAT32_C(   504.24),
                         SIMDE_FLOAT32_C(    52.24), SIMDE_FLOAT32_C(    62.09), SIMDE_FLOAT32_C(   193.22), SIMDE_FLOAT32_C(   894.50),
                         SIMDE_FLOAT32_C(  -487.55), SIMDE_FLOAT32_C(  -250.47), SIMDE_FLOAT32_C(   726.76), SIMDE_FLOAT32_C(   663.18),
                         SIMDE_FLOAT32_C(   612.27), SIMDE_FLOAT32_C(   238.07), SIMDE_FLOAT32_C(  -777.03), SIMDE_FLOAT32_C(   227.93)) },
    { UINT16_C(46817),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    60.97), SIMDE_FLOAT32_C(   294.39), SIMDE_FLOAT32_C(   134.62), SIMDE_FLOAT32_C(  -712.88),
                         SIMDE_FLOAT32_C(  -527.46), SIMDE_FLOAT32_C(   556.17), SIMDE_FLOAT32_C(   671.08), SIMDE_FLOAT32_C(  -149.99),
                         SIMDE_FLOAT32_C(  -247.28), SIMDE_FLOAT32_C(  -221.96), SIMDE_FLOAT32_C(  -789.76), SIMDE_FLOAT32_C(   665.60),
                         SIMDE_FLOAT32_C(   590.64), SIMDE_FLOAT32_C(  -335.79), SIMDE_FLOAT32_C(  -964.26), SIMDE_FLOAT32_C(   137.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   741.48), SIMDE_FLOAT32_C(  -567.93), SIMDE_FLOAT32_C(   572.38), SIMDE_FLOAT32_C(  -363.96),
                         SIMDE_FLOAT32_C(   267.49), SIMDE_FLOAT32_C(   878.76), SIMDE_FLOAT32_C(  -445.04), SIMDE_FLOAT32_C(  -252.24),
                         SIMDE_FLOAT32_C(   869.52), SIMDE_FLOAT32_C(  -356.95), SIMDE_FLOAT32_C(   128.10), SIMDE_FLOAT32_C(   544.58),
                         SIMDE_FLOAT32_C(   242.93), SIMDE_FLOAT32_C(  -919.16), SIMDE_FLOAT32_C(   758.90), SIMDE_FLOAT32_C(  -466.75)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   741.48), SIMDE_FLOAT32_C(   294.39), SIMDE_FLOAT32_C(   572.38), SIMDE_FLOAT32_C(  -363.96),
                         SIMDE_FLOAT32_C(  -527.46), SIMDE_FLOAT32_C(   878.76), SIMDE_FLOAT32_C(  -445.04), SIMDE_FLOAT32_C(  -149.99),
                         SIMDE_FLOAT32_C(   869.52), SIMDE_FLOAT32_C(  -356.95), SIMDE_FLOAT32_C(   128.10), SIMDE_FLOAT32_C(   665.60),
                         SIMDE_FLOAT32_C(   590.64), SIMDE_FLOAT32_C(  -335.79), SIMDE_FLOAT32_C(  -964.26), SIMDE_FLOAT32_C(  -466.75)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_blend_ps(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_blend_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C(211),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  863.27), SIMDE_FLOAT64_C( -937.53),
                         SIMDE_FLOAT64_C(  272.85), SIMDE_FLOAT64_C( -836.56),
                         SIMDE_FLOAT64_C( -517.71), SIMDE_FLOAT64_C(  436.89),
                         SIMDE_FLOAT64_C( -561.62), SIMDE_FLOAT64_C( -796.29)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  356.91), SIMDE_FLOAT64_C( -250.13),
                         SIMDE_FLOAT64_C( -609.99), SIMDE_FLOAT64_C( -756.57),
                         SIMDE_FLOAT64_C( -441.01), SIMDE_FLOAT64_C(  675.23),
                         SIMDE_FLOAT64_C( -112.56), SIMDE_FLOAT64_C(  752.66)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  356.91), SIMDE_FLOAT64_C( -250.13),
                         SIMDE_FLOAT64_C(  272.85), SIMDE_FLOAT64_C( -756.57),
                         SIMDE_FLOAT64_C( -517.71), SIMDE_FLOAT64_C(  436.89),
                         SIMDE_FLOAT64_C( -112.56), SIMDE_FLOAT64_C(  752.66)) },
    { UINT8_C( 25),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  721.67), SIMDE_FLOAT64_C( -355.29),
                         SIMDE_FLOAT64_C( -776.65), SIMDE_FLOAT64_C( -467.78),
                         SIMDE_FLOAT64_C( -890.68), SIMDE_FLOAT64_C( -288.13),
                         SIMDE_FLOAT64_C(  739.88), SIMDE_FLOAT64_C(   -3.67)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -963.76), SIMDE_FLOAT64_C(  986.64),
                         SIMDE_FLOAT64_C(  121.90), SIMDE_FLOAT64_C( -796.62),
                         SIMDE_FLOAT64_C(  983.17), SIMDE_FLOAT64_C(  569.02),
                         SIMDE_FLOAT64_C(  -88.58), SIMDE_FLOAT64_C( -750.53)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  721.67), SIMDE_FLOAT64_C( -355.29),
                         SIMDE_FLOAT64_C( -776.65), SIMDE_FLOAT64_C( -796.62),
                         SIMDE_FLOAT64_C(  983.17), SIMDE_FLOAT64_C( -288.13),
                         SIMDE_FLOAT64_C(  739.88), SIMDE_FLOAT64_C( -750.53)) },
    { UINT8_C( 46),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -217.74), SIMDE_FLOAT64_C(  162.91),
                         SIMDE_FLOAT64_C(  837.05), SIMDE_FLOAT64_C(  132.83),
                         SIMDE_FLOAT64_C( -183.75), SIMDE_FLOAT64_C( -958.98),
                         SIMDE_FLOAT64_C( -343.18), SIMDE_FLOAT64_C( -412.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -43.99), SIMDE_FLOAT64_C(  645.49),
                         SIMDE_FLOAT64_C( -301.76), SIMDE_FLOAT64_C( -390.74),
                         SIMDE_FLOAT64_C(  671.11), SIMDE_FLOAT64_C( -513.10),
                         SIMDE_FLOAT64_C(  467.15), SIMDE_FLOAT64_C( -961.27)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -217.74), SIMDE_FLOAT64_C(  162.91),
                         SIMDE_FLOAT64_C( -301.76), SIMDE_FLOAT64_C(  132.83),
                         SIMDE_FLOAT64_C(  671.11), SIMDE_FLOAT64_C( -513.10),
                         SIMDE_FLOAT64_C(  467.15), SIMDE_FLOAT64_C( -412.04)) },
    { UINT8_C(180),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -623.32), SIMDE_FLOAT64_C(  372.80),
                         SIMDE_FLOAT64_C( -286.04), SIMDE_FLOAT64_C(  347.55),
                         SIMDE_FLOAT64_C( -954.70), SIMDE_FLOAT64_C(  272.86),
                         SIMDE_FLOAT64_C(  787.91), SIMDE_FLOAT64_C(  529.75)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  153.63), SIMDE_FLOAT64_C( -177.79),
                         SIMDE_FLOAT64_C(  957.03), SIMDE_FLOAT64_C( -193.15),
                         SIMDE_FLOAT64_C(  645.09), SIMDE_FLOAT64_C(   -0.96),
                         SIMDE_FLOAT64_C(   66.15), SIMDE_FLOAT64_C(  565.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  153.63), SIMDE_FLOAT64_C(  372.80),
                         SIMDE_FLOAT64_C(  957.03), SIMDE_FLOAT64_C( -193.15),
                         SIMDE_FLOAT64_C( -954.70), SIMDE_FLOAT64_C(   -0.96),
                         SIMDE_FLOAT64_C(  787.91), SIMDE_FLOAT64_C(  529.75)) },
    { UINT8_C(125),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  705.45), SIMDE_FLOAT64_C(  -55.08),
                         SIMDE_FLOAT64_C(  606.63), SIMDE_FLOAT64_C(  -13.01),
                         SIMDE_FLOAT64_C(  483.39), SIMDE_FLOAT64_C(  565.56),
                         SIMDE_FLOAT64_C(  735.84), SIMDE_FLOAT64_C( -855.34)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -898.20), SIMDE_FLOAT64_C( -885.41),
                         SIMDE_FLOAT64_C( -446.34), SIMDE_FLOAT64_C( -539.23),
                         SIMDE_FLOAT64_C(  177.56), SIMDE_FLOAT64_C(   85.90),
                         SIMDE_FLOAT64_C( -977.71), SIMDE_FLOAT64_C(  589.01)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  705.45), SIMDE_FLOAT64_C( -885.41),
                         SIMDE_FLOAT64_C( -446.34), SIMDE_FLOAT64_C( -539.23),
                         SIMDE_FLOAT64_C(  177.56), SIMDE_FLOAT64_C(   85.90),
                         SIMDE_FLOAT64_C(  735.84), SIMDE_FLOAT64_C(  589.01)) },
    { UINT8_C(188),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  263.46), SIMDE_FLOAT64_C( -173.99),
                         SIMDE_FLOAT64_C( -448.77), SIMDE_FLOAT64_C(  -18.46),
                         SIMDE_FLOAT64_C(   60.12), SIMDE_FLOAT64_C(  895.07),
                         SIMDE_FLOAT64_C(  593.75), SIMDE_FLOAT64_C( -910.69)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -347.65), SIMDE_FLOAT64_C(  986.49),
                         SIMDE_FLOAT64_C(  854.73), SIMDE_FLOAT64_C(  459.72),
                         SIMDE_FLOAT64_C( -110.83), SIMDE_FLOAT64_C( -875.29),
                         SIMDE_FLOAT64_C(  594.30), SIMDE_FLOAT64_C( -331.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -347.65), SIMDE_FLOAT64_C( -173.99),
                         SIMDE_FLOAT64_C(  854.73), SIMDE_FLOAT64_C(  459.72),
                         SIMDE_FLOAT64_C( -110.83), SIMDE_FLOAT64_C( -875.29),
                         SIMDE_FLOAT64_C(  593.75), SIMDE_FLOAT64_C( -910.69)) },
    { UINT8_C(190),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   93.71), SIMDE_FLOAT64_C( -510.90),
                         SIMDE_FLOAT64_C(  589.53), SIMDE_FLOAT64_C(  762.37),
                         SIMDE_FLOAT64_C(  -64.95), SIMDE_FLOAT64_C( -200.72),
                         SIMDE_FLOAT64_C(  590.31), SIMDE_FLOAT64_C(  904.10)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -356.41), SIMDE_FLOAT64_C(  536.26),
                         SIMDE_FLOAT64_C(   46.41), SIMDE_FLOAT64_C(  968.69),
                         SIMDE_FLOAT64_C(  819.71), SIMDE_FLOAT64_C( -256.62),
                         SIMDE_FLOAT64_C( -508.11), SIMDE_FLOAT64_C(  806.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -356.41), SIMDE_FLOAT64_C( -510.90),
                         SIMDE_FLOAT64_C(   46.41), SIMDE_FLOAT64_C(  968.69),
                         SIMDE_FLOAT64_C(  819.71), SIMDE_FLOAT64_C( -256.62),
                         SIMDE_FLOAT64_C( -508.11), SIMDE_FLOAT64_C(  904.10)) },
    { UINT8_C(178),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  932.33), SIMDE_FLOAT64_C( -948.81),
                         SIMDE_FLOAT64_C( -663.23), SIMDE_FLOAT64_C( -224.49),
                         SIMDE_FLOAT64_C(   61.61), SIMDE_FLOAT64_C( -513.25),
                         SIMDE_FLOAT64_C(  862.87), SIMDE_FLOAT64_C(  888.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  178.62), SIMDE_FLOAT64_C( -833.77),
                         SIMDE_FLOAT64_C( -382.41), SIMDE_FLOAT64_C(  155.80),
                         SIMDE_FLOAT64_C( -175.51), SIMDE_FLOAT64_C(  270.94),
                         SIMDE_FLOAT64_C( -347.54), SIMDE_FLOAT64_C( -823.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  178.62), SIMDE_FLOAT64_C( -948.81),
                         SIMDE_FLOAT64_C( -382.41), SIMDE_FLOAT64_C(  155.80),
                         SIMDE_FLOAT64_C(   61.61), SIMDE_FLOAT64_C( -513.25),
                         SIMDE_FLOAT64_C( -347.54), SIMDE_FLOAT64_C(  888.55)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_blend_pd(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_broadcast_i32x4(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853)),
      simde_mm512_set_epi32(INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853)) },
    { simde_mm_set_epi32(INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165)),
      simde_mm512_set_epi32(INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165)) },
    { simde_mm_set_epi32(INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792)),
      simde_mm512_set_epi32(INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792)) },
    { simde_mm_set_epi32(INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300)),
      simde_mm512_set_epi32(INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300)) },
    { simde_mm_set_epi32(INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562)),
      simde_mm512_set_epi32(INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562)) },
    { simde_mm_set_epi32(INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438)),
      simde_mm512_set_epi32(INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438)) },
    { simde_mm_set_epi32(INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347)),
      simde_mm512_set_epi32(INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347)) },
    { simde_mm_set_epi32(INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769)),
      simde_mm512_set_epi32(INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcast_i32x4(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_broadcastd_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C(-1051270324), INT32_C(-1977183446), INT32_C( -548195640), INT32_C(-1363461466)),
      simde_mm512_set_epi32(INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466)) },
    { simde_mm_set_epi32(INT32_C(  979094891), INT32_C(  416506319), INT32_C( 2123490297), INT32_C(  200388421)),
      simde_mm512_set_epi32(INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421)) },
    { simde_mm_set_epi32(INT32_C( 1927260635), INT32_C( 1201458882), INT32_C(-1448742498), INT32_C(-1111904220)),
      simde_mm512_set_epi32(INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220)) },
    { simde_mm_set_epi32(INT32_C( -976455818), INT32_C(  542613123), INT32_C(  -15911923), INT32_C( -562895064)),
      simde_mm512_set_epi32(INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064)) },
    { simde_mm_set_epi32(INT32_C(  836747087), INT32_C(-1431045412), INT32_C(-1356396683), INT32_C( 1489138473)),
      simde_mm512_set_epi32(INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473)) },
    { simde_mm_set_epi32(INT32_C(-1783426961), INT32_C( -263517415), INT32_C(-1697630001), INT32_C( 2025142863)),
      simde_mm512_set_epi32(INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863)) },
    { simde_mm_set_epi32(INT32_C(  300619496), INT32_C( -659754204), INT32_C(-1019736463), INT32_C( 1022872166)),
      simde_mm512_set_epi32(INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166)) },
    { simde_mm_set_epi32(INT32_C( -274893610), INT32_C(  171227717), INT32_C( 1187872667), INT32_C( -590903223)),
      simde_mm512_set_epi32(INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcastd_epi32(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_broadcastd_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1638944021), INT32_C( -385149059), INT32_C(  852916680), INT32_C(-1839015366),
                            INT32_C( 1146921463), INT32_C(  765234486), INT32_C( -388218844), INT32_C(-1402803832),
                            INT32_C( 1245942358), INT32_C( 2001202713), INT32_C(  868062804), INT32_C(-1988191751),
                            INT32_C(  807099340), INT32_C(  -38350755), INT32_C( -443928971), INT32_C( -432554813)),
      UINT16_C(24594),
      simde_mm_set_epi32(INT32_C( -255909174), INT32_C(-1302917278), INT32_C(  327520540), INT32_C(  176606543)),
      simde_mm512_set_epi32(INT32_C( 1638944021), INT32_C(  176606543), INT32_C(  176606543), INT32_C(-1839015366),
                            INT32_C( 1146921463), INT32_C(  765234486), INT32_C( -388218844), INT32_C(-1402803832),
                            INT32_C( 1245942358), INT32_C( 2001202713), INT32_C(  868062804), INT32_C(  176606543),
                            INT32_C(  807099340), INT32_C(  -38350755), INT32_C(  176606543), INT32_C( -432554813)) },
    { simde_mm512_set_epi32(INT32_C( -115460801), INT32_C( 1889676725), INT32_C(    2912775), INT32_C(-1289469215),
                            INT32_C( 1033489041), INT32_C(  147853139), INT32_C(  706073024), INT32_C( -130092746),
                            INT32_C( -799642653), INT32_C(-1439962375), INT32_C(-1798405841), INT32_C( 1190396108),
                            INT32_C(-1013986568), INT32_C(  994541610), INT32_C(-1127995400), INT32_C( 1108325476)),
      UINT16_C(40849),
      simde_mm_set_epi32(INT32_C(  250706831), INT32_C( -936079925), INT32_C(-1129184131), INT32_C(  803417186)),
      simde_mm512_set_epi32(INT32_C(  803417186), INT32_C( 1889676725), INT32_C(    2912775), INT32_C(  803417186),
                            INT32_C(  803417186), INT32_C(  803417186), INT32_C(  803417186), INT32_C(  803417186),
                            INT32_C(  803417186), INT32_C(-1439962375), INT32_C(-1798405841), INT32_C(  803417186),
                            INT32_C(-1013986568), INT32_C(  994541610), INT32_C(-1127995400), INT32_C(  803417186)) },
    { simde_mm512_set_epi32(INT32_C(  357625867), INT32_C( -157238200), INT32_C(  909767636), INT32_C( 1422277073),
                            INT32_C( 2123935701), INT32_C(-1040550911), INT32_C(  686758291), INT32_C(-2090356905),
                            INT32_C( -362358815), INT32_C( -482453842), INT32_C(  117787421), INT32_C( 1300554279),
                            INT32_C(-1085613264), INT32_C( -109297466), INT32_C(-1230203271), INT32_C(-1731521429)),
      UINT16_C(53728),
      simde_mm_set_epi32(INT32_C( -707786971), INT32_C( 1712040202), INT32_C(-2012675757), INT32_C(-1396559749)),
      simde_mm512_set_epi32(INT32_C(-1396559749), INT32_C(-1396559749), INT32_C(  909767636), INT32_C(-1396559749),
                            INT32_C( 2123935701), INT32_C(-1040550911), INT32_C(  686758291), INT32_C(-1396559749),
                            INT32_C(-1396559749), INT32_C(-1396559749), INT32_C(-1396559749), INT32_C( 1300554279),
                            INT32_C(-1085613264), INT32_C( -109297466), INT32_C(-1230203271), INT32_C(-1731521429)) },
    { simde_mm512_set_epi32(INT32_C( 2041534605), INT32_C( 1255681923), INT32_C( 1220121473), INT32_C( 1819952522),
                            INT32_C(-1737362693), INT32_C(  712438877), INT32_C(-1234448370), INT32_C(  217554028),
                            INT32_C(-1878093154), INT32_C( -741869417), INT32_C(  943666007), INT32_C(  622675686),
                            INT32_C( -269910912), INT32_C(  137195559), INT32_C(  469574756), INT32_C( 1490101689)),
      UINT16_C(50038),
      simde_mm_set_epi32(INT32_C( -272719467), INT32_C( -594597983), INT32_C( -820913821), INT32_C(  345700481)),
      simde_mm512_set_epi32(INT32_C(  345700481), INT32_C(  345700481), INT32_C( 1220121473), INT32_C( 1819952522),
                            INT32_C(-1737362693), INT32_C(  712438877), INT32_C(  345700481), INT32_C(  345700481),
                            INT32_C(-1878093154), INT32_C(  345700481), INT32_C(  345700481), INT32_C(  345700481),
                            INT32_C( -269910912), INT32_C(  345700481), INT32_C(  345700481), INT32_C( 1490101689)) },
    { simde_mm512_set_epi32(INT32_C(  605201121), INT32_C(    2188130), INT32_C( -956406632), INT32_C(-1144421408),
                            INT32_C(-2008693903), INT32_C( 1823632430), INT32_C( 2043624683), INT32_C(  457225971),
                            INT32_C( 1484257119), INT32_C(  719932227), INT32_C( 1722430058), INT32_C(  916001650),
                            INT32_C(  553469699), INT32_C(-2003831430), INT32_C(-1834906502), INT32_C(  225358926)),
      UINT16_C(22657),
      simde_mm_set_epi32(INT32_C(  290541765), INT32_C( -479926223), INT32_C( 2079119915), INT32_C( -331512500)),
      simde_mm512_set_epi32(INT32_C(  605201121), INT32_C( -331512500), INT32_C( -956406632), INT32_C( -331512500),
                            INT32_C( -331512500), INT32_C( 1823632430), INT32_C( 2043624683), INT32_C(  457225971),
                            INT32_C( -331512500), INT32_C(  719932227), INT32_C( 1722430058), INT32_C(  916001650),
                            INT32_C(  553469699), INT32_C(-2003831430), INT32_C(-1834906502), INT32_C( -331512500)) },
    { simde_mm512_set_epi32(INT32_C( -545987817), INT32_C(-1146550995), INT32_C(  963048631), INT32_C( -701605919),
                            INT32_C(  432096480), INT32_C(-2030393254), INT32_C(-1236899565), INT32_C(-1697034971),
                            INT32_C( -998012960), INT32_C(-1579141793), INT32_C( 1664269708), INT32_C( -667117157),
                            INT32_C( -708117814), INT32_C(   85211107), INT32_C(  909670673), INT32_C( 1616737139)),
      UINT16_C( 4531),
      simde_mm_set_epi32(INT32_C( -503580732), INT32_C(-1790221512), INT32_C(-1663970343), INT32_C( 1633501790)),
      simde_mm512_set_epi32(INT32_C( -545987817), INT32_C(-1146550995), INT32_C(  963048631), INT32_C( 1633501790),
                            INT32_C(  432096480), INT32_C(-2030393254), INT32_C(-1236899565), INT32_C( 1633501790),
                            INT32_C( 1633501790), INT32_C(-1579141793), INT32_C( 1633501790), INT32_C( 1633501790),
                            INT32_C( -708117814), INT32_C(   85211107), INT32_C( 1633501790), INT32_C( 1633501790)) },
    { simde_mm512_set_epi32(INT32_C(-1668661089), INT32_C( 1895031925), INT32_C( 2107029353), INT32_C(-1915428586),
                            INT32_C(  963718296), INT32_C( 1878898594), INT32_C( -403168746), INT32_C(  502390291),
                            INT32_C( 1855826407), INT32_C(-1442018177), INT32_C( -244961355), INT32_C( 1777042193),
                            INT32_C(  373997996), INT32_C( -684064874), INT32_C(  930695451), INT32_C(-1073438864)),
      UINT16_C(53861),
      simde_mm_set_epi32(INT32_C( 1599859635), INT32_C(  543659234), INT32_C(-1222091200), INT32_C(  817594139)),
      simde_mm512_set_epi32(INT32_C(  817594139), INT32_C(  817594139), INT32_C( 2107029353), INT32_C(  817594139),
                            INT32_C(  963718296), INT32_C( 1878898594), INT32_C(  817594139), INT32_C(  502390291),
                            INT32_C( 1855826407), INT32_C(  817594139), INT32_C(  817594139), INT32_C( 1777042193),
                            INT32_C(  373997996), INT32_C(  817594139), INT32_C(  930695451), INT32_C(  817594139)) },
    { simde_mm512_set_epi32(INT32_C( -831807470), INT32_C( -591553083), INT32_C( -492649784), INT32_C(-1394371521),
                            INT32_C(-1760655625), INT32_C( 2135736563), INT32_C(-2075134444), INT32_C( -933317766),
                            INT32_C( -731013025), INT32_C(-2091361347), INT32_C( 1562364760), INT32_C( -612070110),
                            INT32_C( 1365385309), INT32_C( -121237183), INT32_C( 1543044931), INT32_C(-1490381593)),
      UINT16_C(20921),
      simde_mm_set_epi32(INT32_C(-1466503600), INT32_C(  824864478), INT32_C(-1491396230), INT32_C(-1907140086)),
      simde_mm512_set_epi32(INT32_C( -831807470), INT32_C(-1907140086), INT32_C( -492649784), INT32_C(-1907140086),
                            INT32_C(-1760655625), INT32_C( 2135736563), INT32_C(-2075134444), INT32_C(-1907140086),
                            INT32_C(-1907140086), INT32_C(-2091361347), INT32_C(-1907140086), INT32_C(-1907140086),
                            INT32_C(-1907140086), INT32_C( -121237183), INT32_C( 1543044931), INT32_C(-1907140086)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcastd_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_broadcastd_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(21274),
      simde_mm_set_epi32(INT32_C( 1459257075), INT32_C(  587801532), INT32_C( 1631678564), INT32_C(  715337051)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  715337051), INT32_C(          0), INT32_C(  715337051),
                            INT32_C(          0), INT32_C(          0), INT32_C(  715337051), INT32_C(  715337051),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(  715337051),
                            INT32_C(  715337051), INT32_C(          0), INT32_C(  715337051), INT32_C(          0)) },
    { UINT16_C(59357),
      simde_mm_set_epi32(INT32_C(-2022546688), INT32_C( 2145084340), INT32_C(   29275255), INT32_C( -827125259)),
      simde_mm512_set_epi32(INT32_C( -827125259), INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0),
                            INT32_C(          0), INT32_C( -827125259), INT32_C( -827125259), INT32_C( -827125259),
                            INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0), INT32_C( -827125259),
                            INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0), INT32_C( -827125259)) },
    { UINT16_C(34446),
      simde_mm_set_epi32(INT32_C(  973425906), INT32_C( -935954345), INT32_C( 1285315081), INT32_C( 2142489532)),
      simde_mm512_set_epi32(INT32_C( 2142489532), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 2142489532), INT32_C( 2142489532), INT32_C(          0),
                            INT32_C( 2142489532), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2142489532), INT32_C( 2142489532), INT32_C( 2142489532), INT32_C(          0)) },
    { UINT16_C(33955),
      simde_mm_set_epi32(INT32_C(-1114656122), INT32_C( 1221674060), INT32_C( -740975665), INT32_C( 2132760332)),
      simde_mm512_set_epi32(INT32_C( 2132760332), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 2132760332), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2132760332), INT32_C(          0), INT32_C( 2132760332), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 2132760332), INT32_C( 2132760332)) },
    { UINT16_C(52572),
      simde_mm_set_epi32(INT32_C( -724774954), INT32_C( -166426332), INT32_C(-1571631693), INT32_C( -124417294)),
      simde_mm512_set_epi32(INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C(          0),
                            INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C( -124417294),
                            INT32_C(          0), INT32_C( -124417294), INT32_C(          0), INT32_C( -124417294),
                            INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(38931),
      simde_mm_set_epi32(INT32_C(-1992244525), INT32_C( -292982508), INT32_C( -691380397), INT32_C(-1292068161)),
      simde_mm512_set_epi32(INT32_C(-1292068161), INT32_C(          0), INT32_C(          0), INT32_C(-1292068161),
                            INT32_C(-1292068161), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-1292068161),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1292068161), INT32_C(-1292068161)) },
    { UINT16_C(32377),
      simde_mm_set_epi32(INT32_C( -766689829), INT32_C(-1724046912), INT32_C( 1799018744), INT32_C(  623047724)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724),
                            INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724), INT32_C(          0),
                            INT32_C(          0), INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724),
                            INT32_C(  623047724), INT32_C(          0), INT32_C(          0), INT32_C(  623047724)) },
    { UINT16_C(18782),
      simde_mm_set_epi32(INT32_C(-2020669200), INT32_C( -170583969), INT32_C( -628885190), INT32_C(  818636447)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  818636447), INT32_C(          0), INT32_C(          0),
                            INT32_C(  818636447), INT32_C(          0), INT32_C(          0), INT32_C(  818636447),
                            INT32_C(          0), INT32_C(  818636447), INT32_C(          0), INT32_C(  818636447),
                            INT32_C(  818636447), INT32_C(  818636447), INT32_C(  818636447), INT32_C(          0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcastd_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_broadcastq_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi64x(INT64_C(-4515171658517540054), INT64_C(-2354482342678283610)),
      simde_mm512_set_epi64(INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610)) },
    { simde_mm_set_epi64x(INT64_C( 4205180536942191055), INT64_C( 9120321379188715333)),
      simde_mm512_set_epi64(INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333)) },
    { simde_mm_set_epi64x(INT64_C( 8277521399394651842), INT64_C(-6222301646052282332)),
      simde_mm512_set_epi64(INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332)) },
    { simde_mm_set_epi64x(INT64_C(-4193845803756315005), INT64_C(  -68341185169397976)),
      simde_mm512_set_epi64(INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976)) },
    { simde_mm_set_epi64x(INT64_C( 3593801376552188636), INT64_C(-5825679392398740695)),
      simde_mm512_set_epi64(INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695)) },
    { simde_mm_set_epi64x(INT64_C(-7659760468268217575), INT64_C(-7291265332978304433)),
      simde_mm512_set_epi64(INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433)) },
    { simde_mm_set_epi64x(INT64_C( 1291150907495215908), INT64_C(-4379734758100841882)),
      simde_mm512_set_epi64(INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882)) },
    { simde_mm_set_epi64x(INT64_C(-1180659064658150843), INT64_C( 5101874260281362505)),
      simde_mm512_set_epi64(INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcastq_epi64(test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_broadcastq_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 7039210974079555453), INT64_C( 3663249249268849210),
                            INT64_C( 4925990175430708534), INT64_C(-1667387235778762360),
                            INT64_C( 5351281682312326681), INT64_C( 3728301356360833529),
                            INT64_C( 3466465274179801181), INT64_C(-1906660408329519933)),
      UINT8_C( 18),
      simde_mm_set_epi64x(INT64_C(-5595987098075819748), INT64_C(  758519329013942001)),
      simde_mm512_set_epi64(INT64_C( 7039210974079555453), INT64_C( 3663249249268849210),
                            INT64_C( 4925990175430708534), INT64_C(  758519329013942001),
                            INT64_C( 5351281682312326681), INT64_C( 3728301356360833529),
                            INT64_C(  758519329013942001), INT64_C(-1906660408329519933)) },
    { simde_mm512_set_epi64(INT64_C( 8116099733890298375), INT64_C(-5538228106590303599),
                            INT64_C(  635024397322015168), INT64_C( -558744086021510173),
                            INT64_C(-6184591305598926545), INT64_C( 5112712356426664696),
                            INT64_C( 4271523692628158456), INT64_C( 4760221676782691018)),
      UINT8_C( 63),
      simde_mm_set_epi64x(INT64_C(-4849808913003762590), INT64_C( 7901381612815228817)),
      simde_mm512_set_epi64(INT64_C( 8116099733890298375), INT64_C(-5538228106590303599),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817)) },
    { simde_mm512_set_epi64(INT64_C( 3907422245001509329), INT64_C( 9122234377856250881),
                            INT64_C( 2949604402306461527), INT64_C(-1556319256029800786),
                            INT64_C(  505893122375737895), INT64_C(-4662673460798144314),
                            INT64_C(-5283682813813779349), INT64_C( 1076777643387686347)),
      UINT8_C( 72),
      simde_mm_set_epi64x(INT64_C(-5998178448496319999), INT64_C( 7927410529462710283)),
      simde_mm512_set_epi64(INT64_C( 3907422245001509329), INT64_C( 7927410529462710283),
                            INT64_C( 2949604402306461527), INT64_C(-1556319256029800786),
                            INT64_C( 7927410529462710283), INT64_C(-4662673460798144314),
                            INT64_C(-5283682813813779349), INT64_C( 1076777643387686347)) },
    { simde_mm512_set_epi64(INT64_C( 7816636564820325115), INT64_C( 3059901680174485518),
                            INT64_C(  934387437789942430), INT64_C(-3186304882973920425),
                            INT64_C( 2674371711409421440), INT64_C(  589250439531013220),
                            INT64_C( 6399938025556543269), INT64_C( 7353156679309525331)),
      UINT8_C(129),
      simde_mm_set_epi64x(INT64_C(-3132262719190613130), INT64_C( 8768324363382960003)),
      simde_mm512_set_epi64(INT64_C( 8768324363382960003), INT64_C( 3059901680174485518),
                            INT64_C(  934387437789942430), INT64_C(-3186304882973920425),
                            INT64_C( 2674371711409421440), INT64_C(  589250439531013220),
                            INT64_C( 6399938025556543269), INT64_C( 8768324363382960003)) },
    { simde_mm512_set_epi64(INT64_C(-8627274619235963858), INT64_C( 8777301179240593139),
                            INT64_C( 6374835785680112451), INT64_C( 7397780769673384818),
                            INT64_C( 2377134258823099770), INT64_C(-7880863417082399666),
                            INT64_C(-1171321188047181919), INT64_C(-3525798013683697535)),
      UINT8_C(224),
      simde_mm_set_epi64x(INT64_C(-5077429793204296991), INT64_C(    9397950127957144)),
      simde_mm512_set_epi64(INT64_C(    9397950127957144), INT64_C(    9397950127957144),
                            INT64_C(    9397950127957144), INT64_C( 7397780769673384818),
                            INT64_C( 2377134258823099770), INT64_C(-7880863417082399666),
                            INT64_C(-1171321188047181919), INT64_C(-3525798013683697535)) },
    { simde_mm512_set_epi64(INT64_C(-8720472620890953453), INT64_C(-7288709697316354080),
                            INT64_C(-6782362355017532020), INT64_C(-2865246368328647990),
                            INT64_C(  365978918730627345), INT64_C( 6943833138524147909),
                            INT64_C(-2061267430198683093), INT64_C(-1423835345422209809)),
      UINT8_C(224),
      simde_mm_set_epi64x(INT64_C(-2344999814881016531), INT64_C( 4136262378195933153)),
      simde_mm512_set_epi64(INT64_C( 4136262378195933153), INT64_C( 4136262378195933153),
                            INT64_C( 4136262378195933153), INT64_C(-2865246368328647990),
                            INT64_C(  365978918730627345), INT64_C( 6943833138524147909),
                            INT64_C(-2061267430198683093), INT64_C(-1423835345422209809)) },
    { simde_mm512_set_epi64(INT64_C(-1731596578336940525), INT64_C( 7970713727971134591),
                            INT64_C(-1052101006731803887), INT64_C( 1606309165200441238),
                            INT64_C( 3997306527802498928), INT64_C(-2162862772330994888),
                            INT64_C(-7146698203065400738), INT64_C(-4784261768320577101)),
      UINT8_C(162),
      simde_mm_set_epi64x(INT64_C( 8139100144857954153), INT64_C(-8226703133729805160)),
      simde_mm512_set_epi64(INT64_C(-8226703133729805160), INT64_C( 7970713727971134591),
                            INT64_C(-8226703133729805160), INT64_C( 1606309165200441238),
                            INT64_C( 3997306527802498928), INT64_C(-2162862772330994888),
                            INT64_C(-8226703133729805160), INT64_C(-4784261768320577101)) },
    { simde_mm512_set_epi64(INT64_C(-4008569278181826465), INT64_C(-8982328587921142952),
                            INT64_C(-2628821103943737251), INT64_C( -520709734501122237),
                            INT64_C(-6401140198895522893), INT64_C( 2334998633271287360),
                            INT64_C( 3511540092651127844), INT64_C( 3766647997225123999)),
      UINT8_C( 20),
      simde_mm_set_epi64x(INT64_C(-2115914707760868289), INT64_C(-7561958326757703437)),
      simde_mm512_set_epi64(INT64_C(-4008569278181826465), INT64_C(-8982328587921142952),
                            INT64_C(-2628821103943737251), INT64_C(-7561958326757703437),
                            INT64_C(-6401140198895522893), INT64_C(-7561958326757703437),
                            INT64_C( 3511540092651127844), INT64_C( 3766647997225123999)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcastq_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_broadcastq_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 26),
      simde_mm_set_epi64x(INT64_C( 2524588358110376036), INT64_C( 3072349241054123220)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 3072349241054123220),
                            INT64_C( 3072349241054123220), INT64_C(                   0),
                            INT64_C( 3072349241054123220), INT64_C(                   0)) },
    { UINT8_C(243),
      simde_mm_set_epi64x(INT64_C(  125736266274902517), INT64_C( 4529119523676940253)),
      simde_mm512_set_epi64(INT64_C( 4529119523676940253), INT64_C( 4529119523676940253),
                            INT64_C( 4529119523676940253), INT64_C( 4529119523676940253),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 4529119523676940253), INT64_C( 4529119523676940253)) },
    { UINT8_C(180),
      simde_mm_set_epi64x(INT64_C( 9201922475629043961), INT64_C(-5256397243355602176)),
      simde_mm512_set_epi64(INT64_C(-5256397243355602176), INT64_C(                   0),
                            INT64_C(-5256397243355602176), INT64_C(-5256397243355602176),
                            INT64_C(                   0), INT64_C(-5256397243355602176),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(  9),
      simde_mm_set_epi64x(INT64_C( 4382010425855345827), INT64_C( 4180832434708183127)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 4180832434708183127), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 4180832434708183127)) },
    { UINT8_C( 12),
      simde_mm_set_epi64x(INT64_C(-1597707644585397626), INT64_C( 5247050137625533391)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 5247050137625533391), INT64_C( 5247050137625533391),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(150),
      simde_mm_set_epi64x(INT64_C(-3112884720261363420), INT64_C(-6750106718621562126)),
      simde_mm512_set_epi64(INT64_C(-6750106718621562126), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-6750106718621562126),
                            INT64_C(                   0), INT64_C(-6750106718621562126),
                            INT64_C(-6750106718621562126), INT64_C(                   0)) },
    { UINT8_C( 19),
      simde_mm_set_epi64x(INT64_C(-1258350286556471469), INT64_C(-5549390491787734701)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-5549390491787734701),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-5549390491787734701), INT64_C(-5549390491787734701)) },
    { UINT8_C(211),
      simde_mm_set_epi64x(INT64_C( 7726726670994043948), INT64_C( 5635717459582615161)),
      simde_mm512_set_epi64(INT64_C( 5635717459582615161), INT64_C( 5635717459582615161),
                            INT64_C(                   0), INT64_C( 5635717459582615161),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 5635717459582615161), INT64_C( 5635717459582615161)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcastq_epi64(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_broadcastss_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   410.97), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   631.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   668.18), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -627.99), SIMDE_FLOAT32_C(  -347.00)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(  -776.74), SIMDE_FLOAT32_C(   643.82), SIMDE_FLOAT32_C(  -941.79)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -401.52), SIMDE_FLOAT32_C(   338.53), SIMDE_FLOAT32_C(  -725.48), SIMDE_FLOAT32_C(   387.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   119.04), SIMDE_FLOAT32_C(   263.81), SIMDE_FLOAT32_C(   717.18), SIMDE_FLOAT32_C(  -996.30)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -278.68), SIMDE_FLOAT32_C(   120.15), SIMDE_FLOAT32_C(   751.98), SIMDE_FLOAT32_C(   536.33)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(  -448.76), SIMDE_FLOAT32_C(   234.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   -42.20), SIMDE_FLOAT32_C(  -923.83), SIMDE_FLOAT32_C(   357.03), SIMDE_FLOAT32_C(  -933.51)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_broadcastss_ps(test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_broadcastss_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -227.30), SIMDE_FLOAT32_C(   999.04), SIMDE_FLOAT32_C(   956.07), SIMDE_FLOAT32_C(  -270.40),
                         SIMDE_FLOAT32_C(   132.00), SIMDE_FLOAT32_C(   480.19), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -927.52), SIMDE_FLOAT32_C(   -67.87), SIMDE_FLOAT32_C(   891.86), SIMDE_FLOAT32_C(  -870.50),
                         SIMDE_FLOAT32_C(   932.69), SIMDE_FLOAT32_C(   244.86), SIMDE_FLOAT32_C(  -621.59), SIMDE_FLOAT32_C(    36.25)),
      UINT16_C(30253),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   134.75), SIMDE_FLOAT32_C(   871.12), SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   548.32)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -227.30), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32),
                         SIMDE_FLOAT32_C(   132.00), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -927.52), SIMDE_FLOAT32_C(   -67.87), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -870.50),
                         SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -621.59), SIMDE_FLOAT32_C(   548.32)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(   389.31), SIMDE_FLOAT32_C(   670.35), SIMDE_FLOAT32_C(   396.13),
                         SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   275.37), SIMDE_FLOAT32_C(   338.53),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(    29.64), SIMDE_FLOAT32_C(   199.34), SIMDE_FLOAT32_C(  -686.40),
                         SIMDE_FLOAT32_C(   717.18), SIMDE_FLOAT32_C(   416.06), SIMDE_FLOAT32_C(   645.78), SIMDE_FLOAT32_C(  -990.79)),
      UINT16_C(37933),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   442.28), SIMDE_FLOAT32_C(   811.14), SIMDE_FLOAT32_C(  -767.79), SIMDE_FLOAT32_C(   236.12)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   389.31), SIMDE_FLOAT32_C(   670.35), SIMDE_FLOAT32_C(   236.12),
                         SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   275.37), SIMDE_FLOAT32_C(   338.53),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(    29.64), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(  -686.40),
                         SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   645.78), SIMDE_FLOAT32_C(   236.12)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   101.39), SIMDE_FLOAT32_C(  -366.10),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   -55.56), SIMDE_FLOAT32_C(   896.89), SIMDE_FLOAT32_C(   697.60),
                         SIMDE_FLOAT32_C(   120.15), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(  -156.71), SIMDE_FLOAT32_C(  -331.13),
                         SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(  -448.76), SIMDE_FLOAT32_C(  -628.22), SIMDE_FLOAT32_C(   318.72)),
      UINT16_C(19701),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -248.84), SIMDE_FLOAT32_C(   566.99), SIMDE_FLOAT32_C(  -650.08), SIMDE_FLOAT32_C(  -460.40)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(   101.39), SIMDE_FLOAT32_C(  -366.10),
                         SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(   896.89), SIMDE_FLOAT32_C(   697.60),
                         SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40),
                         SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -628.22), SIMDE_FLOAT32_C(  -460.40)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   779.71), SIMDE_FLOAT32_C(   196.66), SIMDE_FLOAT32_C(    -0.50), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(   429.50), SIMDE_FLOAT32_C(   650.80), SIMDE_FLOAT32_C(   509.10), SIMDE_FLOAT32_C(   709.57),
                         SIMDE_FLOAT32_C(  -561.64), SIMDE_FLOAT32_C(  -923.83), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -304.13),
                         SIMDE_FLOAT32_C(   728.72), SIMDE_FLOAT32_C(  -511.49), SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(   848.91)),
      UINT16_C(27468),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   949.19), SIMDE_FLOAT32_C(  -102.63), SIMDE_FLOAT32_C(    87.04), SIMDE_FLOAT32_C(   914.16)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   779.71), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   650.80), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16),
                         SIMDE_FLOAT32_C(  -561.64), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -304.13),
                         SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(   848.91)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -960.76), SIMDE_FLOAT32_C(  -613.57), SIMDE_FLOAT32_C(   864.92), SIMDE_FLOAT32_C(   278.02),
                         SIMDE_FLOAT32_C(   573.37), SIMDE_FLOAT32_C(   393.40), SIMDE_FLOAT32_C(  -782.91), SIMDE_FLOAT32_C(  -933.90),
                         SIMDE_FLOAT32_C(  -291.87), SIMDE_FLOAT32_C(   382.75), SIMDE_FLOAT32_C(   -62.73), SIMDE_FLOAT32_C(   163.52),
                         SIMDE_FLOAT32_C(    87.09), SIMDE_FLOAT32_C(  -486.60), SIMDE_FLOAT32_C(  -157.79), SIMDE_FLOAT32_C(  -247.69)),
      UINT16_C(56353),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   -97.06), SIMDE_FLOAT32_C(    -2.41), SIMDE_FLOAT32_C(   418.81), SIMDE_FLOAT32_C(  -141.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(   864.92), SIMDE_FLOAT32_C(  -141.42),
                         SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -782.91), SIMDE_FLOAT32_C(  -933.90),
                         SIMDE_FLOAT32_C(  -291.87), SIMDE_FLOAT32_C(   382.75), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(   163.52),
                         SIMDE_FLOAT32_C(    87.09), SIMDE_FLOAT32_C(  -486.60), SIMDE_FLOAT32_C(  -157.79), SIMDE_FLOAT32_C(  -141.42)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -490.98), SIMDE_FLOAT32_C(  -718.54), SIMDE_FLOAT32_C(  -172.08), SIMDE_FLOAT32_C(   476.19),
                         SIMDE_FLOAT32_C(  -825.45), SIMDE_FLOAT32_C(  -528.02), SIMDE_FLOAT32_C(  -604.26), SIMDE_FLOAT32_C(  -201.78),
                         SIMDE_FLOAT32_C(  -105.47), SIMDE_FLOAT32_C(   619.70), SIMDE_FLOAT32_C(   603.28), SIMDE_FLOAT32_C(  -553.28),
                         SIMDE_FLOAT32_C(   787.83), SIMDE_FLOAT32_C(  -945.21), SIMDE_FLOAT32_C(  -786.09), SIMDE_FLOAT32_C(   628.77)),
      UINT16_C(51486),
      simde_mm_set_ps(SIMDE_FLOAT32_C(    54.48), SIMDE_FLOAT32_C(   679.92), SIMDE_FLOAT32_C(  -550.45), SIMDE_FLOAT32_C(  -482.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -172.08), SIMDE_FLOAT32_C(   476.19),
                         SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -528.02), SIMDE_FLOAT32_C(  -604.26), SIMDE_FLOAT32_C(  -482.87),
                         SIMDE_FLOAT32_C(  -105.47), SIMDE_FLOAT32_C(   619.70), SIMDE_FLOAT32_C(   603.28), SIMDE_FLOAT32_C(  -482.87),
                         SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(   628.77)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -124.63), SIMDE_FLOAT32_C(  -948.04), SIMDE_FLOAT32_C(  -344.11), SIMDE_FLOAT32_C(  -424.86),
                         SIMDE_FLOAT32_C(   640.76), SIMDE_FLOAT32_C(  -243.42), SIMDE_FLOAT32_C(   962.71), SIMDE_FLOAT32_C(   314.11),
                         SIMDE_FLOAT32_C(   599.88), SIMDE_FLOAT32_C(  -844.53), SIMDE_FLOAT32_C(  -530.48), SIMDE_FLOAT32_C(   563.54),
                         SIMDE_FLOAT32_C(   165.16), SIMDE_FLOAT32_C(   384.17), SIMDE_FLOAT32_C(   149.22), SIMDE_FLOAT32_C(   712.14)),
      UINT16_C(53759),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -284.59), SIMDE_FLOAT32_C(  -286.48), SIMDE_FLOAT32_C(  -340.65), SIMDE_FLOAT32_C(   563.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(  -344.11), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   640.76), SIMDE_FLOAT32_C(  -243.42), SIMDE_FLOAT32_C(   962.71), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -434.52), SIMDE_FLOAT32_C(   678.42), SIMDE_FLOAT32_C(   -65.20), SIMDE_FLOAT32_C(  -319.19),
                         SIMDE_FLOAT32_C(   664.97), SIMDE_FLOAT32_C(     9.01), SIMDE_FLOAT32_C(  -334.08), SIMDE_FLOAT32_C(  -870.44),
                         SIMDE_FLOAT32_C(   269.08), SIMDE_FLOAT32_C(  -345.75), SIMDE_FLOAT32_C(  -732.77), SIMDE_FLOAT32_C(   374.12),
                         SIMDE_FLOAT32_C(  -491.24), SIMDE_FLOAT32_C(   525.54), SIMDE_FLOAT32_C(  -178.26), SIMDE_FLOAT32_C(  -733.62)),
      UINT16_C(50870),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -300.63), SIMDE_FLOAT32_C(  -396.75), SIMDE_FLOAT32_C(   745.02), SIMDE_FLOAT32_C(   369.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   -65.20), SIMDE_FLOAT32_C(  -319.19),
                         SIMDE_FLOAT32_C(   664.97), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -870.44),
                         SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -345.75), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43),
                         SIMDE_FLOAT32_C(  -491.24), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -733.62)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_broadcastss_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_broadcastss_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(25371),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   410.97), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   631.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04)) },
    { UINT16_C(49342),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -559.08), SIMDE_FLOAT32_C(   480.19), SIMDE_FLOAT32_C(   668.18), SIMDE_FLOAT32_C(  -107.97)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97),
                         SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(24820),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   811.14), SIMDE_FLOAT32_C(  -333.00), SIMDE_FLOAT32_C(  -767.79), SIMDE_FLOAT32_C(   825.12)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(45881),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -610.29), SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   997.86), SIMDE_FLOAT32_C(   528.69)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69)) },
    { UINT16_C(28771),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -248.84), SIMDE_FLOAT32_C(   102.57), SIMDE_FLOAT32_C(   566.99), SIMDE_FLOAT32_C(   900.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54)) },
    { UINT16_C(61611),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   318.72), SIMDE_FLOAT32_C(  -366.10), SIMDE_FLOAT32_C(   625.17), SIMDE_FLOAT32_C(   855.96)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96)) },
    { UINT16_C(55548),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   324.92), SIMDE_FLOAT32_C(  -304.13), SIMDE_FLOAT32_C(   949.19), SIMDE_FLOAT32_C(   617.60)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   617.60),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(15841),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(    -0.50), SIMDE_FLOAT32_C(   848.91), SIMDE_FLOAT32_C(  -273.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_broadcastss_ps(test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_broadcastsd_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  842.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  -93.61), SIMDE_FLOAT64_C( -903.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -219.93), SIMDE_FLOAT64_C( -754.32)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  137.51), SIMDE_FLOAT64_C(  527.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  596.82), SIMDE_FLOAT64_C(  365.41)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -229.59), SIMDE_FLOAT64_C( -642.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  308.04), SIMDE_FLOAT64_C( -958.64)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -172.14)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_broadcastsd_pd(test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_broadcastsd_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -746.75), SIMDE_FLOAT64_C(  634.39),
                         SIMDE_FLOAT64_C( -651.68), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C(  689.73), SIMDE_FLOAT64_C(  178.89),
                         SIMDE_FLOAT64_C( -342.04), SIMDE_FLOAT64_C( -292.58)),
      UINT8_C(162),
      simde_mm_set_pd(SIMDE_FLOAT64_C(   63.63), SIMDE_FLOAT64_C(  912.41)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  634.39),
                         SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C(  689.73), SIMDE_FLOAT64_C(  178.89),
                         SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C( -292.58)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -287.03), SIMDE_FLOAT64_C(  851.62),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  137.51),
                         SIMDE_FLOAT64_C( -457.60), SIMDE_FLOAT64_C(  815.46),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  250.27)),
      UINT8_C( 66),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -149.06), SIMDE_FLOAT64_C( -899.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -287.03), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  137.51),
                         SIMDE_FLOAT64_C( -457.60), SIMDE_FLOAT64_C(  815.46),
                         SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C(  250.27)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C(  523.36),
                         SIMDE_FLOAT64_C( -361.34), SIMDE_FLOAT64_C( -153.87),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C(  573.19),
                         SIMDE_FLOAT64_C(  308.04), SIMDE_FLOAT64_C(  -38.88)),
      UINT8_C(115),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -758.12), SIMDE_FLOAT64_C(   12.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C(   12.83),
                         SIMDE_FLOAT64_C(   12.83), SIMDE_FLOAT64_C(   12.83),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C(  573.19),
                         SIMDE_FLOAT64_C(   12.83), SIMDE_FLOAT64_C(   12.83)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -221.94), SIMDE_FLOAT64_C(  499.58),
                         SIMDE_FLOAT64_C(   49.04), SIMDE_FLOAT64_C( -205.69),
                         SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -984.94),
                         SIMDE_FLOAT64_C(  224.44), SIMDE_FLOAT64_C(  644.01)),
      UINT8_C(  4),
      simde_mm_set_pd(SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C( -691.82)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -221.94), SIMDE_FLOAT64_C(  499.58),
                         SIMDE_FLOAT64_C(   49.04), SIMDE_FLOAT64_C( -205.69),
                         SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -691.82),
                         SIMDE_FLOAT64_C(  224.44), SIMDE_FLOAT64_C(  644.01)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -225.42), SIMDE_FLOAT64_C( -144.08),
                         SIMDE_FLOAT64_C( -549.59), SIMDE_FLOAT64_C(  465.78),
                         SIMDE_FLOAT64_C( -316.69), SIMDE_FLOAT64_C( -133.94),
                         SIMDE_FLOAT64_C( -646.50), SIMDE_FLOAT64_C(  160.17)),
      UINT8_C(172),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  841.59), SIMDE_FLOAT64_C(  843.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C( -144.08),
                         SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C(  465.78),
                         SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C(  843.47),
                         SIMDE_FLOAT64_C( -646.50), SIMDE_FLOAT64_C(  160.17)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -560.31), SIMDE_FLOAT64_C( -882.49),
                         SIMDE_FLOAT64_C(  -54.78), SIMDE_FLOAT64_C( -896.38),
                         SIMDE_FLOAT64_C(  607.65), SIMDE_FLOAT64_C( -296.43),
                         SIMDE_FLOAT64_C(  124.51), SIMDE_FLOAT64_C( -913.38)),
      UINT8_C(201),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  625.18), SIMDE_FLOAT64_C(   54.43)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   54.43), SIMDE_FLOAT64_C(   54.43),
                         SIMDE_FLOAT64_C(  -54.78), SIMDE_FLOAT64_C( -896.38),
                         SIMDE_FLOAT64_C(   54.43), SIMDE_FLOAT64_C( -296.43),
                         SIMDE_FLOAT64_C(  124.51), SIMDE_FLOAT64_C(   54.43)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  349.28), SIMDE_FLOAT64_C( -559.74),
                         SIMDE_FLOAT64_C( -116.49), SIMDE_FLOAT64_C(  342.49),
                         SIMDE_FLOAT64_C( -608.07), SIMDE_FLOAT64_C(  778.83),
                         SIMDE_FLOAT64_C( -284.17), SIMDE_FLOAT64_C( -113.81)),
      UINT8_C(234),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  312.83), SIMDE_FLOAT64_C(  -27.64)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  -27.64),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  342.49),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  778.83),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C( -113.81)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  364.84), SIMDE_FLOAT64_C(   86.19),
                         SIMDE_FLOAT64_C( -699.29), SIMDE_FLOAT64_C(  244.26),
                         SIMDE_FLOAT64_C( -206.27), SIMDE_FLOAT64_C( -921.17),
                         SIMDE_FLOAT64_C(  483.42), SIMDE_FLOAT64_C( -935.00)),
      UINT8_C( 12),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -328.98), SIMDE_FLOAT64_C(  803.91)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  364.84), SIMDE_FLOAT64_C(   86.19),
                         SIMDE_FLOAT64_C( -699.29), SIMDE_FLOAT64_C(  244.26),
                         SIMDE_FLOAT64_C(  803.91), SIMDE_FLOAT64_C(  803.91),
                         SIMDE_FLOAT64_C(  483.42), SIMDE_FLOAT64_C( -935.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_broadcastsd_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_broadcastsd_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C(128),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  842.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(  2),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -651.68), SIMDE_FLOAT64_C(  -93.61)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  -93.61), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(216),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  881.57), SIMDE_FLOAT64_C( -899.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(183),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -292.94), SIMDE_FLOAT64_C(  765.97)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  765.97),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  765.97),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  765.97)) },
    { UINT8_C(169),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -758.12), SIMDE_FLOAT64_C(  593.03)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  593.03)) },
    { UINT8_C(243),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  523.36), SIMDE_FLOAT64_C(  761.91)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91),
                         SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91)) },
    { UINT8_C(109),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -984.94), SIMDE_FLOAT64_C(   53.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   53.80),
                         SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C(   53.80),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   53.80)) },
    { UINT8_C(168),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  644.01), SIMDE_FLOAT64_C(  499.58)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_broadcastsd_pd(test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castpd512_pd128(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d a;
    simde__m128d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -503.58), SIMDE_FLOAT64_C(  409.99),
                         SIMDE_FLOAT64_C(  882.79), SIMDE_FLOAT64_C(  967.41),
                         SIMDE_FLOAT64_C(  644.16), SIMDE_FLOAT64_C(   22.93),
                         SIMDE_FLOAT64_C(  629.49), SIMDE_FLOAT64_C(  -19.16)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  629.49), SIMDE_FLOAT64_C(  -19.16)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -371.76), SIMDE_FLOAT64_C( -903.70),
                         SIMDE_FLOAT64_C( -183.89), SIMDE_FLOAT64_C(   -1.20),
                         SIMDE_FLOAT64_C(  -62.86), SIMDE_FLOAT64_C(  444.19),
                         SIMDE_FLOAT64_C(  672.79), SIMDE_FLOAT64_C(  471.18)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  672.79), SIMDE_FLOAT64_C(  471.18)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -771.58), SIMDE_FLOAT64_C(  494.03),
                         SIMDE_FLOAT64_C(  138.84), SIMDE_FLOAT64_C( -195.59),
                         SIMDE_FLOAT64_C( -760.58), SIMDE_FLOAT64_C( -773.64),
                         SIMDE_FLOAT64_C(  870.25), SIMDE_FLOAT64_C(  314.25)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  870.25), SIMDE_FLOAT64_C(  314.25)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   -2.05), SIMDE_FLOAT64_C( -418.62),
                         SIMDE_FLOAT64_C( -507.96), SIMDE_FLOAT64_C( -261.33),
                         SIMDE_FLOAT64_C( -683.39), SIMDE_FLOAT64_C(  -27.96),
                         SIMDE_FLOAT64_C(  534.55), SIMDE_FLOAT64_C(  185.44)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  534.55), SIMDE_FLOAT64_C(  185.44)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  596.68), SIMDE_FLOAT64_C( -902.04),
                         SIMDE_FLOAT64_C(  847.76), SIMDE_FLOAT64_C(  134.78),
                         SIMDE_FLOAT64_C(  116.62), SIMDE_FLOAT64_C(    7.30),
                         SIMDE_FLOAT64_C(  382.57), SIMDE_FLOAT64_C(  434.75)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  382.57), SIMDE_FLOAT64_C(  434.75)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  688.01), SIMDE_FLOAT64_C(  829.27),
                         SIMDE_FLOAT64_C( -970.03), SIMDE_FLOAT64_C(  528.04),
                         SIMDE_FLOAT64_C( -386.08), SIMDE_FLOAT64_C(  603.97),
                         SIMDE_FLOAT64_C( -132.68), SIMDE_FLOAT64_C(  463.66)),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -132.68), SIMDE_FLOAT64_C(  463.66)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  235.78), SIMDE_FLOAT64_C( -244.37),
                         SIMDE_FLOAT64_C(  846.69), SIMDE_FLOAT64_C(  794.12),
                         SIMDE_FLOAT64_C(  672.11), SIMDE_FLOAT64_C( -445.78),
                         SIMDE_FLOAT64_C(  396.01), SIMDE_FLOAT64_C( -284.10)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  396.01), SIMDE_FLOAT64_C( -284.10)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  603.46), SIMDE_FLOAT64_C( -988.15),
                         SIMDE_FLOAT64_C( -342.31), SIMDE_FLOAT64_C( -282.11),
                         SIMDE_FLOAT64_C(  379.72), SIMDE_FLOAT64_C( -680.63),
                         SIMDE_FLOAT64_C( -541.51), SIMDE_FLOAT64_C(  -40.90)),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -541.51), SIMDE_FLOAT64_C(  -40.90)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128d r = simde_mm512_castpd512_pd128(test_vec[i].a);
    simde_assert_m128d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castpd512_pd256(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d a;
    simde__m256d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  414.39), SIMDE_FLOAT64_C(  -15.63),
                         SIMDE_FLOAT64_C(  546.05), SIMDE_FLOAT64_C( -960.01),
                         SIMDE_FLOAT64_C( -752.11), SIMDE_FLOAT64_C( -702.83),
                         SIMDE_FLOAT64_C( -642.16), SIMDE_FLOAT64_C(  617.65)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -752.11), SIMDE_FLOAT64_C( -702.83),
                         SIMDE_FLOAT64_C( -642.16), SIMDE_FLOAT64_C(  617.65)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -964.21), SIMDE_FLOAT64_C( -388.73),
                         SIMDE_FLOAT64_C(  322.52), SIMDE_FLOAT64_C(  699.65),
                         SIMDE_FLOAT64_C(  390.02), SIMDE_FLOAT64_C( -490.64),
                         SIMDE_FLOAT64_C( -297.41), SIMDE_FLOAT64_C(  563.23)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  390.02), SIMDE_FLOAT64_C( -490.64),
                         SIMDE_FLOAT64_C( -297.41), SIMDE_FLOAT64_C(  563.23)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  920.55), SIMDE_FLOAT64_C(   39.56),
                         SIMDE_FLOAT64_C(  750.72), SIMDE_FLOAT64_C(  440.72),
                         SIMDE_FLOAT64_C( -246.45), SIMDE_FLOAT64_C(  981.88),
                         SIMDE_FLOAT64_C( -299.81), SIMDE_FLOAT64_C(  -98.82)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -246.45), SIMDE_FLOAT64_C(  981.88),
                         SIMDE_FLOAT64_C( -299.81), SIMDE_FLOAT64_C(  -98.82)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -471.70), SIMDE_FLOAT64_C( -359.51),
                         SIMDE_FLOAT64_C(  724.77), SIMDE_FLOAT64_C( -820.10),
                         SIMDE_FLOAT64_C(  777.44), SIMDE_FLOAT64_C( -220.31),
                         SIMDE_FLOAT64_C(  567.27), SIMDE_FLOAT64_C(  977.08)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  777.44), SIMDE_FLOAT64_C( -220.31),
                         SIMDE_FLOAT64_C(  567.27), SIMDE_FLOAT64_C(  977.08)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  438.55), SIMDE_FLOAT64_C( -230.11),
                         SIMDE_FLOAT64_C(  759.60), SIMDE_FLOAT64_C(  615.37),
                         SIMDE_FLOAT64_C( -356.18), SIMDE_FLOAT64_C(  777.70),
                         SIMDE_FLOAT64_C( -385.32), SIMDE_FLOAT64_C( -478.17)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -356.18), SIMDE_FLOAT64_C(  777.70),
                         SIMDE_FLOAT64_C( -385.32), SIMDE_FLOAT64_C( -478.17)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -337.84), SIMDE_FLOAT64_C( -274.76),
                         SIMDE_FLOAT64_C(  865.32), SIMDE_FLOAT64_C( -257.44),
                         SIMDE_FLOAT64_C(  325.76), SIMDE_FLOAT64_C(  -67.76),
                         SIMDE_FLOAT64_C( -238.87), SIMDE_FLOAT64_C( -384.04)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  325.76), SIMDE_FLOAT64_C(  -67.76),
                         SIMDE_FLOAT64_C( -238.87), SIMDE_FLOAT64_C( -384.04)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  120.04), SIMDE_FLOAT64_C(  416.14),
                         SIMDE_FLOAT64_C(  299.72), SIMDE_FLOAT64_C(  991.84),
                         SIMDE_FLOAT64_C(  325.55), SIMDE_FLOAT64_C(  685.94),
                         SIMDE_FLOAT64_C( -182.11), SIMDE_FLOAT64_C(  943.15)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  325.55), SIMDE_FLOAT64_C(  685.94),
                         SIMDE_FLOAT64_C( -182.11), SIMDE_FLOAT64_C(  943.15)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  556.40), SIMDE_FLOAT64_C( -775.19),
                         SIMDE_FLOAT64_C( -814.91), SIMDE_FLOAT64_C(   73.06),
                         SIMDE_FLOAT64_C(  317.95), SIMDE_FLOAT64_C( -552.86),
                         SIMDE_FLOAT64_C( -767.41), SIMDE_FLOAT64_C(  130.66)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  317.95), SIMDE_FLOAT64_C( -552.86),
                         SIMDE_FLOAT64_C( -767.41), SIMDE_FLOAT64_C(  130.66)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256d r = simde_mm512_castpd512_pd256(test_vec[i].a);
    simde_assert_m256d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castpd128_pd512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  605.48), SIMDE_FLOAT64_C(  349.95)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  605.48), SIMDE_FLOAT64_C(  349.95)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  125.01), SIMDE_FLOAT64_C(  -52.81)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  125.01), SIMDE_FLOAT64_C(  -52.81)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -137.83), SIMDE_FLOAT64_C( -367.46)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -137.83), SIMDE_FLOAT64_C( -367.46)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -706.70), SIMDE_FLOAT64_C(  -54.57)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -706.70), SIMDE_FLOAT64_C(  -54.57)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -477.44), SIMDE_FLOAT64_C( -678.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -477.44), SIMDE_FLOAT64_C( -678.80)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -750.01), SIMDE_FLOAT64_C(  943.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -750.01), SIMDE_FLOAT64_C(  943.83)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  329.32), SIMDE_FLOAT64_C(  600.02)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  329.32), SIMDE_FLOAT64_C(  600.02)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  495.92), SIMDE_FLOAT64_C(  855.97)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  495.92), SIMDE_FLOAT64_C(  855.97)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_castpd128_pd512(test_vec[i].a);
    r=simde_mm512_maskz_mov_pd(UINT8_C(3), r);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castpd256_pd512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m256d a;
    simde__m512d r;
  } test_vec[8] = {
        { simde_mm256_set_pd(SIMDE_FLOAT64_C( -610.76), SIMDE_FLOAT64_C( -445.99),
                         SIMDE_FLOAT64_C( -292.19), SIMDE_FLOAT64_C(  327.18)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -610.76), SIMDE_FLOAT64_C( -445.99),
                         SIMDE_FLOAT64_C( -292.19), SIMDE_FLOAT64_C(  327.18)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  498.95), SIMDE_FLOAT64_C(  765.72),
                         SIMDE_FLOAT64_C( -978.89), SIMDE_FLOAT64_C( -427.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  498.95), SIMDE_FLOAT64_C(  765.72),
                         SIMDE_FLOAT64_C( -978.89), SIMDE_FLOAT64_C( -427.55)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -991.71), SIMDE_FLOAT64_C(  -66.46),
                         SIMDE_FLOAT64_C(  -87.69), SIMDE_FLOAT64_C(  225.73)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -991.71), SIMDE_FLOAT64_C(  -66.46),
                         SIMDE_FLOAT64_C(  -87.69), SIMDE_FLOAT64_C(  225.73)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -983.02), SIMDE_FLOAT64_C(  768.15),
                         SIMDE_FLOAT64_C(  423.32), SIMDE_FLOAT64_C( -601.96)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -983.02), SIMDE_FLOAT64_C(  768.15),
                         SIMDE_FLOAT64_C(  423.32), SIMDE_FLOAT64_C( -601.96)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  269.59), SIMDE_FLOAT64_C(  -12.98),
                         SIMDE_FLOAT64_C(  833.22), SIMDE_FLOAT64_C( -167.34)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  269.59), SIMDE_FLOAT64_C(  -12.98),
                         SIMDE_FLOAT64_C(  833.22), SIMDE_FLOAT64_C( -167.34)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -976.56), SIMDE_FLOAT64_C( -631.12),
                         SIMDE_FLOAT64_C( -613.64), SIMDE_FLOAT64_C( -574.89)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -976.56), SIMDE_FLOAT64_C( -631.12),
                         SIMDE_FLOAT64_C( -613.64), SIMDE_FLOAT64_C( -574.89)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  900.29), SIMDE_FLOAT64_C(  197.75),
                         SIMDE_FLOAT64_C( -976.11), SIMDE_FLOAT64_C( -958.45)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  900.29), SIMDE_FLOAT64_C(  197.75),
                         SIMDE_FLOAT64_C( -976.11), SIMDE_FLOAT64_C( -958.45)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -467.55), SIMDE_FLOAT64_C( -302.23),
                         SIMDE_FLOAT64_C(  207.74), SIMDE_FLOAT64_C( -217.89)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -467.55), SIMDE_FLOAT64_C( -302.23),
                         SIMDE_FLOAT64_C(  207.74), SIMDE_FLOAT64_C( -217.89)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_castpd256_pd512(test_vec[i].a);
    r=simde_mm512_maskz_mov_pd(UINT8_C(15), r);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castps512_ps128(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 a;
    simde__m128 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   764.32), SIMDE_FLOAT32_C(  -951.31), SIMDE_FLOAT32_C(   179.06), SIMDE_FLOAT32_C(  -437.66),
                         SIMDE_FLOAT32_C(   402.64), SIMDE_FLOAT32_C(   734.29), SIMDE_FLOAT32_C(   267.34), SIMDE_FLOAT32_C(   208.00),
                         SIMDE_FLOAT32_C(   587.55), SIMDE_FLOAT32_C(  -635.96), SIMDE_FLOAT32_C(  -958.84), SIMDE_FLOAT32_C(  -271.31),
                         SIMDE_FLOAT32_C(   711.90), SIMDE_FLOAT32_C(   959.36), SIMDE_FLOAT32_C(   988.07), SIMDE_FLOAT32_C(  -648.23)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   711.90), SIMDE_FLOAT32_C(   959.36), SIMDE_FLOAT32_C(   988.07), SIMDE_FLOAT32_C(  -648.23)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   193.72), SIMDE_FLOAT32_C(  -347.47), SIMDE_FLOAT32_C(   437.71), SIMDE_FLOAT32_C(  -756.77),
                         SIMDE_FLOAT32_C(  -591.14), SIMDE_FLOAT32_C(  -431.84), SIMDE_FLOAT32_C(  -564.12), SIMDE_FLOAT32_C(   252.26),
                         SIMDE_FLOAT32_C(  -784.50), SIMDE_FLOAT32_C(   149.30), SIMDE_FLOAT32_C(  -246.02), SIMDE_FLOAT32_C(   -60.58),
                         SIMDE_FLOAT32_C(   384.90), SIMDE_FLOAT32_C(   156.04), SIMDE_FLOAT32_C(  -595.88), SIMDE_FLOAT32_C(  -574.45)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   384.90), SIMDE_FLOAT32_C(   156.04), SIMDE_FLOAT32_C(  -595.88), SIMDE_FLOAT32_C(  -574.45)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -425.80), SIMDE_FLOAT32_C(   886.22), SIMDE_FLOAT32_C(  -478.46), SIMDE_FLOAT32_C(  -993.15),
                         SIMDE_FLOAT32_C(  -912.29), SIMDE_FLOAT32_C(  -564.52), SIMDE_FLOAT32_C(   -25.07), SIMDE_FLOAT32_C(  -469.67),
                         SIMDE_FLOAT32_C(   828.84), SIMDE_FLOAT32_C(  -525.49), SIMDE_FLOAT32_C(  -521.07), SIMDE_FLOAT32_C(   132.36),
                         SIMDE_FLOAT32_C(   368.34), SIMDE_FLOAT32_C(   996.83), SIMDE_FLOAT32_C(   514.03), SIMDE_FLOAT32_C(   350.58)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   368.34), SIMDE_FLOAT32_C(   996.83), SIMDE_FLOAT32_C(   514.03), SIMDE_FLOAT32_C(   350.58)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -806.26), SIMDE_FLOAT32_C(  -958.10), SIMDE_FLOAT32_C(   -61.10), SIMDE_FLOAT32_C(   698.78),
                         SIMDE_FLOAT32_C(   893.49), SIMDE_FLOAT32_C(  -774.73), SIMDE_FLOAT32_C(  -469.54), SIMDE_FLOAT32_C(    88.88),
                         SIMDE_FLOAT32_C(  -797.01), SIMDE_FLOAT32_C(  -143.15), SIMDE_FLOAT32_C(  -201.38), SIMDE_FLOAT32_C(  -439.08),
                         SIMDE_FLOAT32_C(   122.21), SIMDE_FLOAT32_C(   251.76), SIMDE_FLOAT32_C(   853.77), SIMDE_FLOAT32_C(  -306.48)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   122.21), SIMDE_FLOAT32_C(   251.76), SIMDE_FLOAT32_C(   853.77), SIMDE_FLOAT32_C(  -306.48)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   995.27), SIMDE_FLOAT32_C(   762.38), SIMDE_FLOAT32_C(   849.07), SIMDE_FLOAT32_C(   533.84),
                         SIMDE_FLOAT32_C(   128.75), SIMDE_FLOAT32_C(   581.04), SIMDE_FLOAT32_C(   680.92), SIMDE_FLOAT32_C(    97.54),
                         SIMDE_FLOAT32_C(   183.54), SIMDE_FLOAT32_C(    89.55), SIMDE_FLOAT32_C(  -801.20), SIMDE_FLOAT32_C(  -458.50),
                         SIMDE_FLOAT32_C(  -944.26), SIMDE_FLOAT32_C(  -648.24), SIMDE_FLOAT32_C(  -216.16), SIMDE_FLOAT32_C(   273.51)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -944.26), SIMDE_FLOAT32_C(  -648.24), SIMDE_FLOAT32_C(  -216.16), SIMDE_FLOAT32_C(   273.51)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   563.30), SIMDE_FLOAT32_C(  -488.59), SIMDE_FLOAT32_C(   -41.61), SIMDE_FLOAT32_C(  -745.38),
                         SIMDE_FLOAT32_C(  -156.54), SIMDE_FLOAT32_C(  -651.09), SIMDE_FLOAT32_C(   393.55), SIMDE_FLOAT32_C(  -610.90),
                         SIMDE_FLOAT32_C(   656.44), SIMDE_FLOAT32_C(   254.16), SIMDE_FLOAT32_C(   -27.43), SIMDE_FLOAT32_C(  -862.48),
                         SIMDE_FLOAT32_C(   639.71), SIMDE_FLOAT32_C(  -674.10), SIMDE_FLOAT32_C(  -151.15), SIMDE_FLOAT32_C(  -736.28)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   639.71), SIMDE_FLOAT32_C(  -674.10), SIMDE_FLOAT32_C(  -151.15), SIMDE_FLOAT32_C(  -736.28)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -997.30), SIMDE_FLOAT32_C(   958.88), SIMDE_FLOAT32_C(   414.33), SIMDE_FLOAT32_C(  -486.71),
                         SIMDE_FLOAT32_C(   347.81), SIMDE_FLOAT32_C(   -73.64), SIMDE_FLOAT32_C(   301.45), SIMDE_FLOAT32_C(  -935.75),
                         SIMDE_FLOAT32_C(   554.43), SIMDE_FLOAT32_C(   239.85), SIMDE_FLOAT32_C(  -172.14), SIMDE_FLOAT32_C(  -200.20),
                         SIMDE_FLOAT32_C(   383.84), SIMDE_FLOAT32_C(    -4.79), SIMDE_FLOAT32_C(   659.24), SIMDE_FLOAT32_C(   178.95)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   383.84), SIMDE_FLOAT32_C(    -4.79), SIMDE_FLOAT32_C(   659.24), SIMDE_FLOAT32_C(   178.95)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   282.49), SIMDE_FLOAT32_C(  -287.66), SIMDE_FLOAT32_C(  -300.74), SIMDE_FLOAT32_C(  -282.12),
                         SIMDE_FLOAT32_C(   228.49), SIMDE_FLOAT32_C(   194.42), SIMDE_FLOAT32_C(   232.99), SIMDE_FLOAT32_C(   312.60),
                         SIMDE_FLOAT32_C(    59.18), SIMDE_FLOAT32_C(   591.94), SIMDE_FLOAT32_C(   490.41), SIMDE_FLOAT32_C(   768.05),
                         SIMDE_FLOAT32_C(   628.85), SIMDE_FLOAT32_C(   497.59), SIMDE_FLOAT32_C(   700.20), SIMDE_FLOAT32_C(  -694.51)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   628.85), SIMDE_FLOAT32_C(   497.59), SIMDE_FLOAT32_C(   700.20), SIMDE_FLOAT32_C(  -694.51)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128 r = simde_mm512_castps512_ps128(test_vec[i].a);
    simde_assert_m128_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castps512_ps256(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 a;
    simde__m256 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   516.61), SIMDE_FLOAT32_C(   494.30), SIMDE_FLOAT32_C(   266.21), SIMDE_FLOAT32_C(   450.63),
                         SIMDE_FLOAT32_C(  -862.95), SIMDE_FLOAT32_C(  -528.18), SIMDE_FLOAT32_C(   206.23), SIMDE_FLOAT32_C(  -212.40),
                         SIMDE_FLOAT32_C(   805.40), SIMDE_FLOAT32_C(  -902.72), SIMDE_FLOAT32_C(  -631.10), SIMDE_FLOAT32_C(  -480.24),
                         SIMDE_FLOAT32_C(   577.49), SIMDE_FLOAT32_C(   490.33), SIMDE_FLOAT32_C(  -227.96), SIMDE_FLOAT32_C(   938.93)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   805.40), SIMDE_FLOAT32_C(  -902.72),
                         SIMDE_FLOAT32_C(  -631.10), SIMDE_FLOAT32_C(  -480.24),
                         SIMDE_FLOAT32_C(   577.49), SIMDE_FLOAT32_C(   490.33),
                         SIMDE_FLOAT32_C(  -227.96), SIMDE_FLOAT32_C(   938.93)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   205.80), SIMDE_FLOAT32_C(  -896.03), SIMDE_FLOAT32_C(  -533.62), SIMDE_FLOAT32_C(  -495.13),
                         SIMDE_FLOAT32_C(   100.81), SIMDE_FLOAT32_C(   789.28), SIMDE_FLOAT32_C(   976.66), SIMDE_FLOAT32_C(  -300.54),
                         SIMDE_FLOAT32_C(  -430.09), SIMDE_FLOAT32_C(   725.00), SIMDE_FLOAT32_C(    80.84), SIMDE_FLOAT32_C(   -71.63),
                         SIMDE_FLOAT32_C(   804.31), SIMDE_FLOAT32_C(   798.70), SIMDE_FLOAT32_C(   251.20), SIMDE_FLOAT32_C(   -32.38)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -430.09), SIMDE_FLOAT32_C(   725.00),
                         SIMDE_FLOAT32_C(    80.84), SIMDE_FLOAT32_C(   -71.63),
                         SIMDE_FLOAT32_C(   804.31), SIMDE_FLOAT32_C(   798.70),
                         SIMDE_FLOAT32_C(   251.20), SIMDE_FLOAT32_C(   -32.38)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   635.87), SIMDE_FLOAT32_C(    20.88), SIMDE_FLOAT32_C(   462.09), SIMDE_FLOAT32_C(   342.91),
                         SIMDE_FLOAT32_C(  -645.79), SIMDE_FLOAT32_C(   103.74), SIMDE_FLOAT32_C(  -376.41), SIMDE_FLOAT32_C(   439.99),
                         SIMDE_FLOAT32_C(   895.02), SIMDE_FLOAT32_C(   204.80), SIMDE_FLOAT32_C(     1.77), SIMDE_FLOAT32_C(   909.62),
                         SIMDE_FLOAT32_C(  -446.42), SIMDE_FLOAT32_C(     7.95), SIMDE_FLOAT32_C(  -614.77), SIMDE_FLOAT32_C(   315.09)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   895.02), SIMDE_FLOAT32_C(   204.80),
                         SIMDE_FLOAT32_C(     1.77), SIMDE_FLOAT32_C(   909.62),
                         SIMDE_FLOAT32_C(  -446.42), SIMDE_FLOAT32_C(     7.95),
                         SIMDE_FLOAT32_C(  -614.77), SIMDE_FLOAT32_C(   315.09)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -372.65), SIMDE_FLOAT32_C(   316.64), SIMDE_FLOAT32_C(   -78.40), SIMDE_FLOAT32_C(   677.31),
                         SIMDE_FLOAT32_C(   299.82), SIMDE_FLOAT32_C(   452.58), SIMDE_FLOAT32_C(   876.09), SIMDE_FLOAT32_C(   746.57),
                         SIMDE_FLOAT32_C(   916.69), SIMDE_FLOAT32_C(  -811.99), SIMDE_FLOAT32_C(   637.74), SIMDE_FLOAT32_C(   -83.23),
                         SIMDE_FLOAT32_C(   289.79), SIMDE_FLOAT32_C(  -689.87), SIMDE_FLOAT32_C(   493.07), SIMDE_FLOAT32_C(   108.56)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   916.69), SIMDE_FLOAT32_C(  -811.99),
                         SIMDE_FLOAT32_C(   637.74), SIMDE_FLOAT32_C(   -83.23),
                         SIMDE_FLOAT32_C(   289.79), SIMDE_FLOAT32_C(  -689.87),
                         SIMDE_FLOAT32_C(   493.07), SIMDE_FLOAT32_C(   108.56)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -296.96), SIMDE_FLOAT32_C(   756.35), SIMDE_FLOAT32_C(  -395.61), SIMDE_FLOAT32_C(  -521.80),
                         SIMDE_FLOAT32_C(  -721.04), SIMDE_FLOAT32_C(   -68.70), SIMDE_FLOAT32_C(  -334.12), SIMDE_FLOAT32_C(   -56.72),
                         SIMDE_FLOAT32_C(    59.02), SIMDE_FLOAT32_C(   -66.01), SIMDE_FLOAT32_C(  -767.54), SIMDE_FLOAT32_C(   502.66),
                         SIMDE_FLOAT32_C(  -591.60), SIMDE_FLOAT32_C(  -826.33), SIMDE_FLOAT32_C(  -334.73), SIMDE_FLOAT32_C(   -13.18)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(    59.02), SIMDE_FLOAT32_C(   -66.01),
                         SIMDE_FLOAT32_C(  -767.54), SIMDE_FLOAT32_C(   502.66),
                         SIMDE_FLOAT32_C(  -591.60), SIMDE_FLOAT32_C(  -826.33),
                         SIMDE_FLOAT32_C(  -334.73), SIMDE_FLOAT32_C(   -13.18)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    -2.88), SIMDE_FLOAT32_C(  -966.98), SIMDE_FLOAT32_C(   854.50), SIMDE_FLOAT32_C(   668.99),
                         SIMDE_FLOAT32_C(   259.81), SIMDE_FLOAT32_C(   -77.24), SIMDE_FLOAT32_C(  -462.32), SIMDE_FLOAT32_C(   688.88),
                         SIMDE_FLOAT32_C(   754.80), SIMDE_FLOAT32_C(   986.78), SIMDE_FLOAT32_C(   961.39), SIMDE_FLOAT32_C(   850.00),
                         SIMDE_FLOAT32_C(   721.06), SIMDE_FLOAT32_C(   -70.04), SIMDE_FLOAT32_C(   136.98), SIMDE_FLOAT32_C(   949.65)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   754.80), SIMDE_FLOAT32_C(   986.78),
                         SIMDE_FLOAT32_C(   961.39), SIMDE_FLOAT32_C(   850.00),
                         SIMDE_FLOAT32_C(   721.06), SIMDE_FLOAT32_C(   -70.04),
                         SIMDE_FLOAT32_C(   136.98), SIMDE_FLOAT32_C(   949.65)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   841.78), SIMDE_FLOAT32_C(  -315.64), SIMDE_FLOAT32_C(   747.52), SIMDE_FLOAT32_C(   946.88),
                         SIMDE_FLOAT32_C(   871.78), SIMDE_FLOAT32_C(   775.22), SIMDE_FLOAT32_C(  -159.38), SIMDE_FLOAT32_C(   270.48),
                         SIMDE_FLOAT32_C(  -276.64), SIMDE_FLOAT32_C(   602.71), SIMDE_FLOAT32_C(  -825.17), SIMDE_FLOAT32_C(  -882.05),
                         SIMDE_FLOAT32_C(   -12.61), SIMDE_FLOAT32_C(  -809.78), SIMDE_FLOAT32_C(  -114.18), SIMDE_FLOAT32_C(   260.44)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -276.64), SIMDE_FLOAT32_C(   602.71),
                         SIMDE_FLOAT32_C(  -825.17), SIMDE_FLOAT32_C(  -882.05),
                         SIMDE_FLOAT32_C(   -12.61), SIMDE_FLOAT32_C(  -809.78),
                         SIMDE_FLOAT32_C(  -114.18), SIMDE_FLOAT32_C(   260.44)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   258.67), SIMDE_FLOAT32_C(   265.23), SIMDE_FLOAT32_C(   761.29), SIMDE_FLOAT32_C(  -932.57),
                         SIMDE_FLOAT32_C(   366.86), SIMDE_FLOAT32_C(  -726.22), SIMDE_FLOAT32_C(   179.78), SIMDE_FLOAT32_C(   965.96),
                         SIMDE_FLOAT32_C(  -420.83), SIMDE_FLOAT32_C(  -371.51), SIMDE_FLOAT32_C(   392.50), SIMDE_FLOAT32_C(   925.30),
                         SIMDE_FLOAT32_C(   990.85), SIMDE_FLOAT32_C(   -34.78), SIMDE_FLOAT32_C(   -24.58), SIMDE_FLOAT32_C(  -150.35)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -420.83), SIMDE_FLOAT32_C(  -371.51),
                         SIMDE_FLOAT32_C(   392.50), SIMDE_FLOAT32_C(   925.30),
                         SIMDE_FLOAT32_C(   990.85), SIMDE_FLOAT32_C(   -34.78),
                         SIMDE_FLOAT32_C(   -24.58), SIMDE_FLOAT32_C(  -150.35)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256 r = simde_mm512_castps512_ps256(test_vec[i].a);
    simde_assert_m256_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castps_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -475.09), SIMDE_FLOAT32_C(   736.84), SIMDE_FLOAT32_C(  -702.97), SIMDE_FLOAT32_C(  -433.94),
                         SIMDE_FLOAT32_C(   854.93), SIMDE_FLOAT32_C(  -157.11), SIMDE_FLOAT32_C(    43.61), SIMDE_FLOAT32_C(  -310.79),
                         SIMDE_FLOAT32_C(  -893.63), SIMDE_FLOAT32_C(    37.00), SIMDE_FLOAT32_C(   245.96), SIMDE_FLOAT32_C(   381.92),
                         SIMDE_FLOAT32_C(  -702.80), SIMDE_FLOAT32_C(   931.47), SIMDE_FLOAT32_C(   429.22), SIMDE_FLOAT32_C(  -639.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -475.09), SIMDE_FLOAT32_C(   736.84), SIMDE_FLOAT32_C(  -702.97), SIMDE_FLOAT32_C(  -433.94),
                         SIMDE_FLOAT32_C(   854.93), SIMDE_FLOAT32_C(  -157.11), SIMDE_FLOAT32_C(    43.61), SIMDE_FLOAT32_C(  -310.79),
                         SIMDE_FLOAT32_C(  -893.63), SIMDE_FLOAT32_C(    37.00), SIMDE_FLOAT32_C(   245.96), SIMDE_FLOAT32_C(   381.92),
                         SIMDE_FLOAT32_C(  -702.80), SIMDE_FLOAT32_C(   931.47), SIMDE_FLOAT32_C(   429.22), SIMDE_FLOAT32_C(  -639.06)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -147.89), SIMDE_FLOAT32_C(   347.19), SIMDE_FLOAT32_C(   203.54), SIMDE_FLOAT32_C(  -988.99),
                         SIMDE_FLOAT32_C(   306.65), SIMDE_FLOAT32_C(   641.36), SIMDE_FLOAT32_C(  -965.49), SIMDE_FLOAT32_C(  -587.11),
                         SIMDE_FLOAT32_C(   724.40), SIMDE_FLOAT32_C(  -636.19), SIMDE_FLOAT32_C(  -368.06), SIMDE_FLOAT32_C(   887.68),
                         SIMDE_FLOAT32_C(  -970.41), SIMDE_FLOAT32_C(  -566.45), SIMDE_FLOAT32_C(   791.35), SIMDE_FLOAT32_C(  -284.64)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -147.89), SIMDE_FLOAT32_C(   347.19), SIMDE_FLOAT32_C(   203.54), SIMDE_FLOAT32_C(  -988.99),
                         SIMDE_FLOAT32_C(   306.65), SIMDE_FLOAT32_C(   641.36), SIMDE_FLOAT32_C(  -965.49), SIMDE_FLOAT32_C(  -587.11),
                         SIMDE_FLOAT32_C(   724.40), SIMDE_FLOAT32_C(  -636.19), SIMDE_FLOAT32_C(  -368.06), SIMDE_FLOAT32_C(   887.68),
                         SIMDE_FLOAT32_C(  -970.41), SIMDE_FLOAT32_C(  -566.45), SIMDE_FLOAT32_C(   791.35), SIMDE_FLOAT32_C(  -284.64)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -73.77), SIMDE_FLOAT32_C(  -499.31), SIMDE_FLOAT32_C(   113.82), SIMDE_FLOAT32_C(  -847.95),
                         SIMDE_FLOAT32_C(  -593.73), SIMDE_FLOAT32_C(   978.06), SIMDE_FLOAT32_C(   354.00), SIMDE_FLOAT32_C(  -120.89),
                         SIMDE_FLOAT32_C(  -554.86), SIMDE_FLOAT32_C(  -105.83), SIMDE_FLOAT32_C(   247.41), SIMDE_FLOAT32_C(   728.79),
                         SIMDE_FLOAT32_C(  -359.78), SIMDE_FLOAT32_C(  -375.68), SIMDE_FLOAT32_C(  -456.84), SIMDE_FLOAT32_C(  -859.51)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -73.77), SIMDE_FLOAT32_C(  -499.31), SIMDE_FLOAT32_C(   113.82), SIMDE_FLOAT32_C(  -847.95),
                         SIMDE_FLOAT32_C(  -593.73), SIMDE_FLOAT32_C(   978.06), SIMDE_FLOAT32_C(   354.00), SIMDE_FLOAT32_C(  -120.89),
                         SIMDE_FLOAT32_C(  -554.86), SIMDE_FLOAT32_C(  -105.83), SIMDE_FLOAT32_C(   247.41), SIMDE_FLOAT32_C(   728.79),
                         SIMDE_FLOAT32_C(  -359.78), SIMDE_FLOAT32_C(  -375.68), SIMDE_FLOAT32_C(  -456.84), SIMDE_FLOAT32_C(  -859.51)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -892.63), SIMDE_FLOAT32_C(   554.75), SIMDE_FLOAT32_C(   786.31), SIMDE_FLOAT32_C(   695.92),
                         SIMDE_FLOAT32_C(  -629.78), SIMDE_FLOAT32_C(   376.00), SIMDE_FLOAT32_C(   490.47), SIMDE_FLOAT32_C(  -658.22),
                         SIMDE_FLOAT32_C(  -993.58), SIMDE_FLOAT32_C(   931.16), SIMDE_FLOAT32_C(   896.06), SIMDE_FLOAT32_C(  -621.22),
                         SIMDE_FLOAT32_C(   936.87), SIMDE_FLOAT32_C(  -330.96), SIMDE_FLOAT32_C(   683.22), SIMDE_FLOAT32_C(   273.63)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -892.63), SIMDE_FLOAT32_C(   554.75), SIMDE_FLOAT32_C(   786.31), SIMDE_FLOAT32_C(   695.92),
                         SIMDE_FLOAT32_C(  -629.78), SIMDE_FLOAT32_C(   376.00), SIMDE_FLOAT32_C(   490.47), SIMDE_FLOAT32_C(  -658.22),
                         SIMDE_FLOAT32_C(  -993.58), SIMDE_FLOAT32_C(   931.16), SIMDE_FLOAT32_C(   896.06), SIMDE_FLOAT32_C(  -621.22),
                         SIMDE_FLOAT32_C(   936.87), SIMDE_FLOAT32_C(  -330.96), SIMDE_FLOAT32_C(   683.22), SIMDE_FLOAT32_C(   273.63)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -464.48), SIMDE_FLOAT32_C(   711.99), SIMDE_FLOAT32_C(   146.80), SIMDE_FLOAT32_C(  -790.19),
                         SIMDE_FLOAT32_C(  -196.31), SIMDE_FLOAT32_C(   -19.08), SIMDE_FLOAT32_C(   390.50), SIMDE_FLOAT32_C(  -693.04),
                         SIMDE_FLOAT32_C(   121.41), SIMDE_FLOAT32_C(  -677.25), SIMDE_FLOAT32_C(   840.59), SIMDE_FLOAT32_C(   653.82),
                         SIMDE_FLOAT32_C(  -341.60), SIMDE_FLOAT32_C(  -152.67), SIMDE_FLOAT32_C(  -910.13), SIMDE_FLOAT32_C(   648.18)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -464.48), SIMDE_FLOAT32_C(   711.99), SIMDE_FLOAT32_C(   146.80), SIMDE_FLOAT32_C(  -790.19),
                         SIMDE_FLOAT32_C(  -196.31), SIMDE_FLOAT32_C(   -19.08), SIMDE_FLOAT32_C(   390.50), SIMDE_FLOAT32_C(  -693.04),
                         SIMDE_FLOAT32_C(   121.41), SIMDE_FLOAT32_C(  -677.25), SIMDE_FLOAT32_C(   840.59), SIMDE_FLOAT32_C(   653.82),
                         SIMDE_FLOAT32_C(  -341.60), SIMDE_FLOAT32_C(  -152.67), SIMDE_FLOAT32_C(  -910.13), SIMDE_FLOAT32_C(   648.18)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   610.99), SIMDE_FLOAT32_C(   354.13), SIMDE_FLOAT32_C(  -805.98), SIMDE_FLOAT32_C(   470.53),
                         SIMDE_FLOAT32_C(   848.62), SIMDE_FLOAT32_C(  -853.14), SIMDE_FLOAT32_C(   898.19), SIMDE_FLOAT32_C(  -186.52),
                         SIMDE_FLOAT32_C(  -954.10), SIMDE_FLOAT32_C(   887.52), SIMDE_FLOAT32_C(   607.34), SIMDE_FLOAT32_C(    42.34),
                         SIMDE_FLOAT32_C(  -842.20), SIMDE_FLOAT32_C(   627.26), SIMDE_FLOAT32_C(   417.38), SIMDE_FLOAT32_C(  -300.32)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   610.99), SIMDE_FLOAT32_C(   354.13), SIMDE_FLOAT32_C(  -805.98), SIMDE_FLOAT32_C(   470.53),
                         SIMDE_FLOAT32_C(   848.62), SIMDE_FLOAT32_C(  -853.14), SIMDE_FLOAT32_C(   898.19), SIMDE_FLOAT32_C(  -186.52),
                         SIMDE_FLOAT32_C(  -954.10), SIMDE_FLOAT32_C(   887.52), SIMDE_FLOAT32_C(   607.34), SIMDE_FLOAT32_C(    42.34),
                         SIMDE_FLOAT32_C(  -842.20), SIMDE_FLOAT32_C(   627.26), SIMDE_FLOAT32_C(   417.38), SIMDE_FLOAT32_C(  -300.32)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -91.92), SIMDE_FLOAT32_C(  -128.53), SIMDE_FLOAT32_C(  -157.39), SIMDE_FLOAT32_C(   922.31),
                         SIMDE_FLOAT32_C(    68.36), SIMDE_FLOAT32_C(  -842.81), SIMDE_FLOAT32_C(    60.56), SIMDE_FLOAT32_C(   689.02),
                         SIMDE_FLOAT32_C(   -70.29), SIMDE_FLOAT32_C(  -753.33), SIMDE_FLOAT32_C(  -999.31), SIMDE_FLOAT32_C(    72.69),
                         SIMDE_FLOAT32_C(    47.99), SIMDE_FLOAT32_C(   903.18), SIMDE_FLOAT32_C(    -5.05), SIMDE_FLOAT32_C(  -435.35)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -91.92), SIMDE_FLOAT32_C(  -128.53), SIMDE_FLOAT32_C(  -157.39), SIMDE_FLOAT32_C(   922.31),
                         SIMDE_FLOAT32_C(    68.36), SIMDE_FLOAT32_C(  -842.81), SIMDE_FLOAT32_C(    60.56), SIMDE_FLOAT32_C(   689.02),
                         SIMDE_FLOAT32_C(   -70.29), SIMDE_FLOAT32_C(  -753.33), SIMDE_FLOAT32_C(  -999.31), SIMDE_FLOAT32_C(    72.69),
                         SIMDE_FLOAT32_C(    47.99), SIMDE_FLOAT32_C(   903.18), SIMDE_FLOAT32_C(    -5.05), SIMDE_FLOAT32_C(  -435.35)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -687.13), SIMDE_FLOAT32_C(  -562.12), SIMDE_FLOAT32_C(  -564.90), SIMDE_FLOAT32_C(  -948.64),
                         SIMDE_FLOAT32_C(   287.72), SIMDE_FLOAT32_C(   -13.16), SIMDE_FLOAT32_C(   706.69), SIMDE_FLOAT32_C(    62.48),
                         SIMDE_FLOAT32_C(    15.33), SIMDE_FLOAT32_C(  -938.52), SIMDE_FLOAT32_C(  -548.66), SIMDE_FLOAT32_C(   893.31),
                         SIMDE_FLOAT32_C(  -166.31), SIMDE_FLOAT32_C(     3.21), SIMDE_FLOAT32_C(  -121.90), SIMDE_FLOAT32_C(  -237.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -687.13), SIMDE_FLOAT32_C(  -562.12), SIMDE_FLOAT32_C(  -564.90), SIMDE_FLOAT32_C(  -948.64),
                         SIMDE_FLOAT32_C(   287.72), SIMDE_FLOAT32_C(   -13.16), SIMDE_FLOAT32_C(   706.69), SIMDE_FLOAT32_C(    62.48),
                         SIMDE_FLOAT32_C(    15.33), SIMDE_FLOAT32_C(  -938.52), SIMDE_FLOAT32_C(  -548.66), SIMDE_FLOAT32_C(   893.31),
                         SIMDE_FLOAT32_C(  -166.31), SIMDE_FLOAT32_C(     3.21), SIMDE_FLOAT32_C(  -121.90), SIMDE_FLOAT32_C(  -237.87)) }
  };
  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
  simde__m512 r = simde_mm512_castpd_ps(simde_mm512_castps_pd(test_vec[i].a));
  simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castsi512_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1318886849), INT32_C( -963615992), INT32_C( 1168255165), INT32_C(-1910220907),
                            INT32_C( -362848940), INT32_C( 1228894571), INT32_C( -604141281), INT32_C( -310556576),
                            INT32_C( -195291453), INT32_C(-1762187610), INT32_C(  785462248), INT32_C(-1654799886),
                            INT32_C(  443157727), INT32_C(  291457152), INT32_C( -673220379), INT32_C( 1588648097)),
      simde_mm512_set_epi32(INT32_C( 1318886849), INT32_C( -963615992), INT32_C( 1168255165), INT32_C(-1910220907),
                            INT32_C( -362848940), INT32_C( 1228894571), INT32_C( -604141281), INT32_C( -310556576),
                            INT32_C( -195291453), INT32_C(-1762187610), INT32_C(  785462248), INT32_C(-1654799886),
                            INT32_C(  443157727), INT32_C(  291457152), INT32_C( -673220379), INT32_C( 1588648097)) },
    { simde_mm512_set_epi32(INT32_C(  388365668), INT32_C( 1922479678), INT32_C(-1025869072), INT32_C( 1053113602),
                            INT32_C( -948060413), INT32_C( 1300260897), INT32_C(  723694906), INT32_C( 1579982945),
                            INT32_C(  551168292), INT32_C( 2117629385), INT32_C( 1278333888), INT32_C(-1219831740),
                            INT32_C( 1012175805), INT32_C(-2142193359), INT32_C(-1234937614), INT32_C(  296489222)),
      simde_mm512_set_epi32(INT32_C(  388365668), INT32_C( 1922479678), INT32_C(-1025869072), INT32_C( 1053113602),
                            INT32_C( -948060413), INT32_C( 1300260897), INT32_C(  723694906), INT32_C( 1579982945),
                            INT32_C(  551168292), INT32_C( 2117629385), INT32_C( 1278333888), INT32_C(-1219831740),
                            INT32_C( 1012175805), INT32_C(-2142193359), INT32_C(-1234937614), INT32_C(  296489222)) },
    { simde_mm512_set_epi32(INT32_C( 1558252976), INT32_C( -896391163), INT32_C( -282653026), INT32_C( -261252112),
                            INT32_C(-1886684518), INT32_C( -792082029), INT32_C(-1821569388), INT32_C(  452082945),
                            INT32_C( 1996286767), INT32_C( -816391022), INT32_C(   52318481), INT32_C( -732357064),
                            INT32_C(-1730294171), INT32_C( 1249707888), INT32_C( 1882801706), INT32_C( -348126448)),
      simde_mm512_set_epi32(INT32_C( 1558252976), INT32_C( -896391163), INT32_C( -282653026), INT32_C( -261252112),
                            INT32_C(-1886684518), INT32_C( -792082029), INT32_C(-1821569388), INT32_C(  452082945),
                            INT32_C( 1996286767), INT32_C( -816391022), INT32_C(   52318481), INT32_C( -732357064),
                            INT32_C(-1730294171), INT32_C( 1249707888), INT32_C( 1882801706), INT32_C( -348126448)) },
    { simde_mm512_set_epi32(INT32_C(-1306007403), INT32_C(  -14823844), INT32_C(   68140915), INT32_C(-1279691583),
                            INT32_C( -366833324), INT32_C( 1029809788), INT32_C(  398169753), INT32_C( 1644631001),
                            INT32_C(-1495643320), INT32_C( -349662299), INT32_C(  337000079), INT32_C(-1599869755),
                            INT32_C( -326525511), INT32_C( 2145777358), INT32_C( -671246537), INT32_C(-1483766126)),
      simde_mm512_set_epi32(INT32_C(-1306007403), INT32_C(  -14823844), INT32_C(   68140915), INT32_C(-1279691583),
                            INT32_C( -366833324), INT32_C( 1029809788), INT32_C(  398169753), INT32_C( 1644631001),
                            INT32_C(-1495643320), INT32_C( -349662299), INT32_C(  337000079), INT32_C(-1599869755),
                            INT32_C( -326525511), INT32_C( 2145777358), INT32_C( -671246537), INT32_C(-1483766126)) },
    { simde_mm512_set_epi32(INT32_C( 1306718049), INT32_C( -481786167), INT32_C( 1036658403), INT32_C(-1275842496),
                            INT32_C(-1384128337), INT32_C( 1110346848), INT32_C( 1633808121), INT32_C(  818679772),
                            INT32_C(  528120539), INT32_C( 1454269875), INT32_C(-1309905851), INT32_C(  195834431),
                            INT32_C( 1710057355), INT32_C(-1286641516), INT32_C( -825070396), INT32_C(-1314661204)),
      simde_mm512_set_epi32(INT32_C( 1306718049), INT32_C( -481786167), INT32_C( 1036658403), INT32_C(-1275842496),
                            INT32_C(-1384128337), INT32_C( 1110346848), INT32_C( 1633808121), INT32_C(  818679772),
                            INT32_C(  528120539), INT32_C( 1454269875), INT32_C(-1309905851), INT32_C(  195834431),
                            INT32_C( 1710057355), INT32_C(-1286641516), INT32_C( -825070396), INT32_C(-1314661204)) },
    { simde_mm512_set_epi32(INT32_C(  897666230), INT32_C(-1247200387), INT32_C(-1594960612), INT32_C(  784358708),
                            INT32_C( -639715103), INT32_C( 1722588202), INT32_C( -957042730), INT32_C( 1748493888),
                            INT32_C(-1424605467), INT32_C( 1715503797), INT32_C(-1877902145), INT32_C( 1433291829),
                            INT32_C(-1672795293), INT32_C( -125523250), INT32_C(  817060741), INT32_C(-1940108057)),
      simde_mm512_set_epi32(INT32_C(  897666230), INT32_C(-1247200387), INT32_C(-1594960612), INT32_C(  784358708),
                            INT32_C( -639715103), INT32_C( 1722588202), INT32_C( -957042730), INT32_C( 1748493888),
                            INT32_C(-1424605467), INT32_C( 1715503797), INT32_C(-1877902145), INT32_C( 1433291829),
                            INT32_C(-1672795293), INT32_C( -125523250), INT32_C(  817060741), INT32_C(-1940108057)) },
    { simde_mm512_set_epi32(INT32_C( -771897847), INT32_C(   51088448), INT32_C(  731748986), INT32_C(-2010019631),
                            INT32_C(-1805256328), INT32_C( -982668321), INT32_C(-1986983933), INT32_C( -806537017),
                            INT32_C( 2103732941), INT32_C(-1950119891), INT32_C( -262758582), INT32_C( -704905824),
                            INT32_C( -298698020), INT32_C( -290664422), INT32_C( -169474404), INT32_C( -381923585)),
      simde_mm512_set_epi32(INT32_C( -771897847), INT32_C(   51088448), INT32_C(  731748986), INT32_C(-2010019631),
                            INT32_C(-1805256328), INT32_C( -982668321), INT32_C(-1986983933), INT32_C( -806537017),
                            INT32_C( 2103732941), INT32_C(-1950119891), INT32_C( -262758582), INT32_C( -704905824),
                            INT32_C( -298698020), INT32_C( -290664422), INT32_C( -169474404), INT32_C( -381923585)) },
    { simde_mm512_set_epi32(INT32_C(-1424890095), INT32_C( -662922774), INT32_C( 1172732731), INT32_C(  220940559),
                            INT32_C(  736508018), INT32_C(-1201335155), INT32_C( -747976663), INT32_C( 1864271935),
                            INT32_C( 1893933430), INT32_C(  214467364), INT32_C(-1230640603), INT32_C(-1529352277),
                            INT32_C(  301681975), INT32_C(  -86355089), INT32_C( 1945634979), INT32_C( 1310017249)),
      simde_mm512_set_epi32(INT32_C(-1424890095), INT32_C( -662922774), INT32_C( 1172732731), INT32_C(  220940559),
                            INT32_C(  736508018), INT32_C(-1201335155), INT32_C( -747976663), INT32_C( 1864271935),
                            INT32_C( 1893933430), INT32_C(  214467364), INT32_C(-1230640603), INT32_C(-1529352277),
                            INT32_C(  301681975), INT32_C(  -86355089), INT32_C( 1945634979), INT32_C( 1310017249)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_castsi512_ps(test_vec[i].a));
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castpd_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -628.24), SIMDE_FLOAT64_C( -135.47),
                         SIMDE_FLOAT64_C( -682.87), SIMDE_FLOAT64_C(  278.82),
                         SIMDE_FLOAT64_C(   71.00), SIMDE_FLOAT64_C(  284.08),
                         SIMDE_FLOAT64_C( -576.74), SIMDE_FLOAT64_C( -655.56)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -628.24), SIMDE_FLOAT64_C( -135.47),
                         SIMDE_FLOAT64_C( -682.87), SIMDE_FLOAT64_C(  278.82),
                         SIMDE_FLOAT64_C(   71.00), SIMDE_FLOAT64_C(  284.08),
                         SIMDE_FLOAT64_C( -576.74), SIMDE_FLOAT64_C( -655.56)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  730.25), SIMDE_FLOAT64_C(  427.27),
                         SIMDE_FLOAT64_C( -482.13), SIMDE_FLOAT64_C( -725.95),
                         SIMDE_FLOAT64_C(  102.42), SIMDE_FLOAT64_C( -673.41),
                         SIMDE_FLOAT64_C(  528.31), SIMDE_FLOAT64_C( -870.66)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  730.25), SIMDE_FLOAT64_C(  427.27),
                         SIMDE_FLOAT64_C( -482.13), SIMDE_FLOAT64_C( -725.95),
                         SIMDE_FLOAT64_C(  102.42), SIMDE_FLOAT64_C( -673.41),
                         SIMDE_FLOAT64_C(  528.31), SIMDE_FLOAT64_C( -870.66)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -250.54), SIMDE_FLOAT64_C( -740.40),
                         SIMDE_FLOAT64_C(  -94.04), SIMDE_FLOAT64_C( -104.23),
                         SIMDE_FLOAT64_C( -857.17), SIMDE_FLOAT64_C( -236.14),
                         SIMDE_FLOAT64_C( -926.88), SIMDE_FLOAT64_C(  849.79)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -250.54), SIMDE_FLOAT64_C( -740.40),
                         SIMDE_FLOAT64_C(  -94.04), SIMDE_FLOAT64_C( -104.23),
                         SIMDE_FLOAT64_C( -857.17), SIMDE_FLOAT64_C( -236.14),
                         SIMDE_FLOAT64_C( -926.88), SIMDE_FLOAT64_C(  849.79)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   72.78), SIMDE_FLOAT64_C(  -73.84),
                         SIMDE_FLOAT64_C(  -20.96), SIMDE_FLOAT64_C( -723.10),
                         SIMDE_FLOAT64_C( -895.17), SIMDE_FLOAT64_C(  -12.07),
                         SIMDE_FLOAT64_C( -741.57), SIMDE_FLOAT64_C(  210.75)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   72.78), SIMDE_FLOAT64_C(  -73.84),
                         SIMDE_FLOAT64_C(  -20.96), SIMDE_FLOAT64_C( -723.10),
                         SIMDE_FLOAT64_C( -895.17), SIMDE_FLOAT64_C(  -12.07),
                         SIMDE_FLOAT64_C( -741.57), SIMDE_FLOAT64_C(  210.75)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -460.57), SIMDE_FLOAT64_C( -774.17),
                         SIMDE_FLOAT64_C(  356.36), SIMDE_FLOAT64_C(  996.28),
                         SIMDE_FLOAT64_C(    2.33), SIMDE_FLOAT64_C( -321.17),
                         SIMDE_FLOAT64_C(  407.13), SIMDE_FLOAT64_C(  410.39)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -460.57), SIMDE_FLOAT64_C( -774.17),
                         SIMDE_FLOAT64_C(  356.36), SIMDE_FLOAT64_C(  996.28),
                         SIMDE_FLOAT64_C(    2.33), SIMDE_FLOAT64_C( -321.17),
                         SIMDE_FLOAT64_C(  407.13), SIMDE_FLOAT64_C(  410.39)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  913.16), SIMDE_FLOAT64_C( -861.99),
                         SIMDE_FLOAT64_C( -778.41), SIMDE_FLOAT64_C( -877.17),
                         SIMDE_FLOAT64_C( -312.46), SIMDE_FLOAT64_C(  942.54),
                         SIMDE_FLOAT64_C(  824.90), SIMDE_FLOAT64_C( -524.63)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  913.16), SIMDE_FLOAT64_C( -861.99),
                         SIMDE_FLOAT64_C( -778.41), SIMDE_FLOAT64_C( -877.17),
                         SIMDE_FLOAT64_C( -312.46), SIMDE_FLOAT64_C(  942.54),
                         SIMDE_FLOAT64_C(  824.90), SIMDE_FLOAT64_C( -524.63)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  993.37), SIMDE_FLOAT64_C( -877.52),
                         SIMDE_FLOAT64_C( -379.85), SIMDE_FLOAT64_C( -359.06),
                         SIMDE_FLOAT64_C(   56.95), SIMDE_FLOAT64_C( -974.33),
                         SIMDE_FLOAT64_C(  222.57), SIMDE_FLOAT64_C(  341.02)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  993.37), SIMDE_FLOAT64_C( -877.52),
                         SIMDE_FLOAT64_C( -379.85), SIMDE_FLOAT64_C( -359.06),
                         SIMDE_FLOAT64_C(   56.95), SIMDE_FLOAT64_C( -974.33),
                         SIMDE_FLOAT64_C(  222.57), SIMDE_FLOAT64_C(  341.02)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  935.31), SIMDE_FLOAT64_C(  -34.65),
                         SIMDE_FLOAT64_C( -660.60), SIMDE_FLOAT64_C(  275.30),
                         SIMDE_FLOAT64_C( -994.69), SIMDE_FLOAT64_C( -110.75),
                         SIMDE_FLOAT64_C( -596.29), SIMDE_FLOAT64_C(  897.05)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  935.31), SIMDE_FLOAT64_C(  -34.65),
                         SIMDE_FLOAT64_C( -660.60), SIMDE_FLOAT64_C(  275.30),
                         SIMDE_FLOAT64_C( -994.69), SIMDE_FLOAT64_C( -110.75),
                         SIMDE_FLOAT64_C( -596.29), SIMDE_FLOAT64_C(  897.05)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
  simde__m512d r = simde_mm512_castps_pd(simde_mm512_castpd_ps(test_vec[i].a));
  simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castpd_si512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-2932283473482861325), INT64_C(-7551244776617231168),
                            INT64_C(-7807931705287408960), INT64_C(-5709657210396534887),
                            INT64_C(-5733822187822983321), INT64_C(-6449026762131886574),
                            INT64_C(-4469461462291751367), INT64_C(-1966061413581863839)),
      simde_mm512_set_epi64(INT64_C(-2932283473482861325), INT64_C(-7551244776617231168),
                            INT64_C(-7807931705287408960), INT64_C(-5709657210396534887),
                            INT64_C(-5733822187822983321), INT64_C(-6449026762131886574),
                            INT64_C(-4469461462291751367), INT64_C(-1966061413581863839)) },
    { simde_mm512_set_epi64(INT64_C( 1279499818540525091), INT64_C( 7861921123166401201),
                            INT64_C(-7776868620274070112), INT64_C( 1282257894426558327),
                            INT64_C( 7845161632627040500), INT64_C(-5240804517978550546),
                            INT64_C( 5302531798719229176), INT64_C(-5843726612220411524)),
      simde_mm512_set_epi64(INT64_C( 1279499818540525091), INT64_C( 7861921123166401201),
                            INT64_C(-7776868620274070112), INT64_C( 1282257894426558327),
                            INT64_C( 7845161632627040500), INT64_C(-5240804517978550546),
                            INT64_C( 5302531798719229176), INT64_C(-5843726612220411524)) },
    { simde_mm512_set_epi64(INT64_C( 8264653255264008622), INT64_C(  162552689407469387),
                            INT64_C(-6727945509653055258), INT64_C(-8051656683653588284),
                            INT64_C(-3673194952315675544), INT64_C(-6715795272606770523),
                            INT64_C( 4713570617590173150), INT64_C(-1460297597223426823)),
      simde_mm512_set_epi64(INT64_C( 8264653255264008622), INT64_C(  162552689407469387),
                            INT64_C(-6727945509653055258), INT64_C(-8051656683653588284),
                            INT64_C(-3673194952315675544), INT64_C(-6715795272606770523),
                            INT64_C( 4713570617590173150), INT64_C(-1460297597223426823)) },
    { simde_mm512_set_epi64(INT64_C( 8052472118589501494), INT64_C(-9124867990213523455),
                            INT64_C( 8236047422487698023), INT64_C( 5945811080038499438),
                            INT64_C( 6518955141271267147), INT64_C( 2853045589014014838),
                            INT64_C( 2581043198697401036), INT64_C( 3427873189039658818)),
      simde_mm512_set_epi64(INT64_C( 8052472118589501494), INT64_C(-9124867990213523455),
                            INT64_C( 8236047422487698023), INT64_C( 5945811080038499438),
                            INT64_C( 6518955141271267147), INT64_C( 2853045589014014838),
                            INT64_C( 2581043198697401036), INT64_C( 3427873189039658818)) },
    { simde_mm512_set_epi64(INT64_C(-3824716377022324243), INT64_C( 2673669294054155970),
                            INT64_C(-3539125802254885907), INT64_C(-2325147789690797111),
                            INT64_C(-8959458229056338743), INT64_C( 8174617038106487054),
                            INT64_C( 2599971863369021204), INT64_C(-4237128038265732285)),
      simde_mm512_set_epi64(INT64_C(-3824716377022324243), INT64_C( 2673669294054155970),
                            INT64_C(-3539125802254885907), INT64_C(-2325147789690797111),
                            INT64_C(-8959458229056338743), INT64_C( 8174617038106487054),
                            INT64_C( 2599971863369021204), INT64_C(-4237128038265732285)) },
    { simde_mm512_set_epi64(INT64_C(  467193351903391999), INT64_C( 4360504692705357109),
                            INT64_C( 9013482442495432840), INT64_C(-2009950226490426066),
                            INT64_C(  483507342486254018), INT64_C(  183419223247078674),
                            INT64_C( 2287275646029180906), INT64_C(-3986540516598251529)),
      simde_mm512_set_epi64(INT64_C(  467193351903391999), INT64_C( 4360504692705357109),
                            INT64_C( 9013482442495432840), INT64_C(-2009950226490426066),
                            INT64_C(  483507342486254018), INT64_C(  183419223247078674),
                            INT64_C( 2287275646029180906), INT64_C(-3986540516598251529)) },
    { simde_mm512_set_epi64(INT64_C(-3077645007853633163), INT64_C(-1572779564739653348),
                            INT64_C(-9063178373150191806), INT64_C(-7245631614198864513),
                            INT64_C(-8737364776584512923), INT64_C(-2892458838054920588),
                            INT64_C(  471547968378055740), INT64_C(  775209234293426283)),
      simde_mm512_set_epi64(INT64_C(-3077645007853633163), INT64_C(-1572779564739653348),
                            INT64_C(-9063178373150191806), INT64_C(-7245631614198864513),
                            INT64_C(-8737364776584512923), INT64_C(-2892458838054920588),
                            INT64_C(  471547968378055740), INT64_C(  775209234293426283)) },
    { simde_mm512_set_epi64(INT64_C(-6041628496323364468), INT64_C(-9137870096902224131),
                            INT64_C(-1573113398564050163), INT64_C( 8854240990059664398),
                            INT64_C(-2186298020810705372), INT64_C(-2128579590201805820),
                            INT64_C( 2972790510556186969), INT64_C(-3293272700939969964)),
      simde_mm512_set_epi64(INT64_C(-6041628496323364468), INT64_C(-9137870096902224131),
                            INT64_C(-1573113398564050163), INT64_C( 8854240990059664398),
                            INT64_C(-2186298020810705372), INT64_C(-2128579590201805820),
                            INT64_C( 2972790510556186969), INT64_C(-3293272700939969964)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_castsi512_pd(test_vec[i].a));
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castsi512_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-2932283473482861325), INT64_C(-7551244776617231168),
                            INT64_C(-7807931705287408960), INT64_C(-5709657210396534887),
                            INT64_C(-5733822187822983321), INT64_C(-6449026762131886574),
                            INT64_C(-4469461462291751367), INT64_C(-1966061413581863839)),
      simde_mm512_set_epi64(INT64_C(-2932283473482861325), INT64_C(-7551244776617231168),
                            INT64_C(-7807931705287408960), INT64_C(-5709657210396534887),
                            INT64_C(-5733822187822983321), INT64_C(-6449026762131886574),
                            INT64_C(-4469461462291751367), INT64_C(-1966061413581863839)) },
    { simde_mm512_set_epi64(INT64_C( 1279499818540525091), INT64_C( 7861921123166401201),
                            INT64_C(-7776868620274070112), INT64_C( 1282257894426558327),
                            INT64_C( 7845161632627040500), INT64_C(-5240804517978550546),
                            INT64_C( 5302531798719229176), INT64_C(-5843726612220411524)),
      simde_mm512_set_epi64(INT64_C( 1279499818540525091), INT64_C( 7861921123166401201),
                            INT64_C(-7776868620274070112), INT64_C( 1282257894426558327),
                            INT64_C( 7845161632627040500), INT64_C(-5240804517978550546),
                            INT64_C( 5302531798719229176), INT64_C(-5843726612220411524)) },
    { simde_mm512_set_epi64(INT64_C( 8264653255264008622), INT64_C(  162552689407469387),
                            INT64_C(-6727945509653055258), INT64_C(-8051656683653588284),
                            INT64_C(-3673194952315675544), INT64_C(-6715795272606770523),
                            INT64_C( 4713570617590173150), INT64_C(-1460297597223426823)),
      simde_mm512_set_epi64(INT64_C( 8264653255264008622), INT64_C(  162552689407469387),
                            INT64_C(-6727945509653055258), INT64_C(-8051656683653588284),
                            INT64_C(-3673194952315675544), INT64_C(-6715795272606770523),
                            INT64_C( 4713570617590173150), INT64_C(-1460297597223426823)) },
    { simde_mm512_set_epi64(INT64_C( 8052472118589501494), INT64_C(-9124867990213523455),
                            INT64_C( 8236047422487698023), INT64_C( 5945811080038499438),
                            INT64_C( 6518955141271267147), INT64_C( 2853045589014014838),
                            INT64_C( 2581043198697401036), INT64_C( 3427873189039658818)),
      simde_mm512_set_epi64(INT64_C( 8052472118589501494), INT64_C(-9124867990213523455),
                            INT64_C( 8236047422487698023), INT64_C( 5945811080038499438),
                            INT64_C( 6518955141271267147), INT64_C( 2853045589014014838),
                            INT64_C( 2581043198697401036), INT64_C( 3427873189039658818)) },
    { simde_mm512_set_epi64(INT64_C(-3824716377022324243), INT64_C( 2673669294054155970),
                            INT64_C(-3539125802254885907), INT64_C(-2325147789690797111),
                            INT64_C(-8959458229056338743), INT64_C( 8174617038106487054),
                            INT64_C( 2599971863369021204), INT64_C(-4237128038265732285)),
      simde_mm512_set_epi64(INT64_C(-3824716377022324243), INT64_C( 2673669294054155970),
                            INT64_C(-3539125802254885907), INT64_C(-2325147789690797111),
                            INT64_C(-8959458229056338743), INT64_C( 8174617038106487054),
                            INT64_C( 2599971863369021204), INT64_C(-4237128038265732285)) },
    { simde_mm512_set_epi64(INT64_C(  467193351903391999), INT64_C( 4360504692705357109),
                            INT64_C( 9013482442495432840), INT64_C(-2009950226490426066),
                            INT64_C(  483507342486254018), INT64_C(  183419223247078674),
                            INT64_C( 2287275646029180906), INT64_C(-3986540516598251529)),
      simde_mm512_set_epi64(INT64_C(  467193351903391999), INT64_C( 4360504692705357109),
                            INT64_C( 9013482442495432840), INT64_C(-2009950226490426066),
                            INT64_C(  483507342486254018), INT64_C(  183419223247078674),
                            INT64_C( 2287275646029180906), INT64_C(-3986540516598251529)) },
    { simde_mm512_set_epi64(INT64_C(-3077645007853633163), INT64_C(-1572779564739653348),
                            INT64_C(-9063178373150191806), INT64_C(-7245631614198864513),
                            INT64_C(-8737364776584512923), INT64_C(-2892458838054920588),
                            INT64_C(  471547968378055740), INT64_C(  775209234293426283)),
      simde_mm512_set_epi64(INT64_C(-3077645007853633163), INT64_C(-1572779564739653348),
                            INT64_C(-9063178373150191806), INT64_C(-7245631614198864513),
                            INT64_C(-8737364776584512923), INT64_C(-2892458838054920588),
                            INT64_C(  471547968378055740), INT64_C(  775209234293426283)) },
    { simde_mm512_set_epi64(INT64_C(-6041628496323364468), INT64_C(-9137870096902224131),
                            INT64_C(-1573113398564050163), INT64_C( 8854240990059664398),
                            INT64_C(-2186298020810705372), INT64_C(-2128579590201805820),
                            INT64_C( 2972790510556186969), INT64_C(-3293272700939969964)),
      simde_mm512_set_epi64(INT64_C(-6041628496323364468), INT64_C(-9137870096902224131),
                            INT64_C(-1573113398564050163), INT64_C( 8854240990059664398),
                            INT64_C(-2186298020810705372), INT64_C(-2128579590201805820),
                            INT64_C( 2972790510556186969), INT64_C(-3293272700939969964)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_castsi512_pd(test_vec[i].a));
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castsi128_si512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C(-1668834023), INT32_C(-1352312258), INT32_C(  556637397), INT32_C( -245835434)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(-1668834023), INT32_C(-1352312258), INT32_C(  556637397), INT32_C( -245835434)) },
    { simde_mm_set_epi32(INT32_C( 1624943139), INT32_C( -461116292), INT32_C(-2119913042), INT32_C( -900874486)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1624943139), INT32_C( -461116292), INT32_C(-2119913042), INT32_C( -900874486)) },
    { simde_mm_set_epi32(INT32_C( -483029339), INT32_C( 1367342572), INT32_C( 1717714414), INT32_C( -425758846)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -483029339), INT32_C( 1367342572), INT32_C( 1717714414), INT32_C( -425758846)) },
    { simde_mm_set_epi32(INT32_C( -911211301), INT32_C(  327710477), INT32_C( -375455356), INT32_C(-1057936365)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -911211301), INT32_C(  327710477), INT32_C( -375455356), INT32_C(-1057936365)) },
    { simde_mm_set_epi32(INT32_C(  275440871), INT32_C(  641107064), INT32_C( 1055988271), INT32_C(  449906430)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  275440871), INT32_C(  641107064), INT32_C( 1055988271), INT32_C(  449906430)) },
    { simde_mm_set_epi32(INT32_C( -222973014), INT32_C( -356035719), INT32_C( -568690768), INT32_C( 1550764475)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -222973014), INT32_C( -356035719), INT32_C( -568690768), INT32_C( 1550764475)) },
    { simde_mm_set_epi32(INT32_C( 1024653484), INT32_C(-1114765609), INT32_C(  501755487), INT32_C( 1011153625)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1024653484), INT32_C(-1114765609), INT32_C(  501755487), INT32_C( 1011153625)) },
    { simde_mm_set_epi32(INT32_C( 1112995048), INT32_C( 1174048979), INT32_C(  979704874), INT32_C( -581442043)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1112995048), INT32_C( 1174048979), INT32_C(  979704874), INT32_C( -581442043)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castsi128_si512(test_vec[i].a);
    r=simde_mm512_maskz_mov_epi32(UINT8_C(15), r);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castsi256_si512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C( -627018310), INT32_C( -732773372), INT32_C(-1935004141), INT32_C( 1864732488),
                            INT32_C(  140289699), INT32_C(-1570899663), INT32_C(-1630998993), INT32_C(  818347323)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -627018310), INT32_C( -732773372), INT32_C(-1935004141), INT32_C( 1864732488),
                            INT32_C(  140289699), INT32_C(-1570899663), INT32_C(-1630998993), INT32_C(  818347323)) },
    { simde_mm256_set_epi32(INT32_C( -208321892), INT32_C(-1091642607), INT32_C(  600758994), INT32_C(  251226426),
                            INT32_C( -263819406), INT32_C(-1881495622), INT32_C( 1833100173), INT32_C(-1639467184)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -208321892), INT32_C(-1091642607), INT32_C(  600758994), INT32_C(  251226426),
                            INT32_C( -263819406), INT32_C(-1881495622), INT32_C( 1833100173), INT32_C(-1639467184)) },
    { simde_mm256_set_epi32(INT32_C(-1754784817), INT32_C( -579924690), INT32_C(  661573432), INT32_C(-1181370591),
                            INT32_C(   60095626), INT32_C(  956226137), INT32_C( -117371256), INT32_C(-1481124874)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(-1754784817), INT32_C( -579924690), INT32_C(  661573432), INT32_C(-1181370591),
                            INT32_C(   60095626), INT32_C(  956226137), INT32_C( -117371256), INT32_C(-1481124874)) },
    { simde_mm256_set_epi32(INT32_C( 2059950650), INT32_C(  100086839), INT32_C( 1040622382), INT32_C(  188824341),
                            INT32_C( 1391404675), INT32_C(-2134538567), INT32_C( 1420659092), INT32_C( 1719736439)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2059950650), INT32_C(  100086839), INT32_C( 1040622382), INT32_C(  188824341),
                            INT32_C( 1391404675), INT32_C(-2134538567), INT32_C( 1420659092), INT32_C( 1719736439)) },
    { simde_mm256_set_epi32(INT32_C(  -72277944), INT32_C(    -175328), INT32_C( 1214469435), INT32_C( -481210186),
                            INT32_C( 1755907779), INT32_C( 1941862792), INT32_C(  -53966731), INT32_C(  177508362)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  -72277944), INT32_C(    -175328), INT32_C( 1214469435), INT32_C( -481210186),
                            INT32_C( 1755907779), INT32_C( 1941862792), INT32_C(  -53966731), INT32_C(  177508362)) },
    { simde_mm256_set_epi32(INT32_C( -579195101), INT32_C(  394175316), INT32_C( 2062674518), INT32_C(-2146728292),
                            INT32_C(   67273180), INT32_C(-1275684351), INT32_C(  245156584), INT32_C(  382170563)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -579195101), INT32_C(  394175316), INT32_C( 2062674518), INT32_C(-2146728292),
                            INT32_C(   67273180), INT32_C(-1275684351), INT32_C(  245156584), INT32_C(  382170563)) },
    { simde_mm256_set_epi32(INT32_C(-1725634605), INT32_C(-1992977299), INT32_C( -692965094), INT32_C( 1390281908),
                            INT32_C( 1056328531), INT32_C(  448471511), INT32_C(  676963201), INT32_C( -471721601)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(-1725634605), INT32_C(-1992977299), INT32_C( -692965094), INT32_C( 1390281908),
                            INT32_C( 1056328531), INT32_C(  448471511), INT32_C(  676963201), INT32_C( -471721601)) },
    { simde_mm256_set_epi32(INT32_C(  868040614), INT32_C( -210101311), INT32_C( 1711471891), INT32_C(  -87087404),
                            INT32_C(-1569971706), INT32_C(-1573190685), INT32_C( 1472943366), INT32_C(-1700287741)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  868040614), INT32_C( -210101311), INT32_C( 1711471891), INT32_C(  -87087404),
                            INT32_C(-1569971706), INT32_C(-1573190685), INT32_C( 1472943366), INT32_C(-1700287741)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castsi256_si512(test_vec[i].a);
    r=simde_mm512_maskz_mov_epi32(UINT8_C(255), r);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castps_si512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1318886849), INT32_C( -963615992), INT32_C( 1168255165), INT32_C(-1910220907),
                            INT32_C( -362848940), INT32_C( 1228894571), INT32_C( -604141281), INT32_C( -310556576),
                            INT32_C( -195291453), INT32_C(-1762187610), INT32_C(  785462248), INT32_C(-1654799886),
                            INT32_C(  443157727), INT32_C(  291457152), INT32_C( -673220379), INT32_C( 1588648097)),
      simde_mm512_set_epi32(INT32_C( 1318886849), INT32_C( -963615992), INT32_C( 1168255165), INT32_C(-1910220907),
                            INT32_C( -362848940), INT32_C( 1228894571), INT32_C( -604141281), INT32_C( -310556576),
                            INT32_C( -195291453), INT32_C(-1762187610), INT32_C(  785462248), INT32_C(-1654799886),
                            INT32_C(  443157727), INT32_C(  291457152), INT32_C( -673220379), INT32_C( 1588648097)) },
    { simde_mm512_set_epi32(INT32_C(  388365668), INT32_C( 1922479678), INT32_C(-1025869072), INT32_C( 1053113602),
                            INT32_C( -948060413), INT32_C( 1300260897), INT32_C(  723694906), INT32_C( 1579982945),
                            INT32_C(  551168292), INT32_C( 2117629385), INT32_C( 1278333888), INT32_C(-1219831740),
                            INT32_C( 1012175805), INT32_C(-2142193359), INT32_C(-1234937614), INT32_C(  296489222)),
      simde_mm512_set_epi32(INT32_C(  388365668), INT32_C( 1922479678), INT32_C(-1025869072), INT32_C( 1053113602),
                            INT32_C( -948060413), INT32_C( 1300260897), INT32_C(  723694906), INT32_C( 1579982945),
                            INT32_C(  551168292), INT32_C( 2117629385), INT32_C( 1278333888), INT32_C(-1219831740),
                            INT32_C( 1012175805), INT32_C(-2142193359), INT32_C(-1234937614), INT32_C(  296489222)) },
    { simde_mm512_set_epi32(INT32_C( 1558252976), INT32_C( -896391163), INT32_C( -282653026), INT32_C( -261252112),
                            INT32_C(-1886684518), INT32_C( -792082029), INT32_C(-1821569388), INT32_C(  452082945),
                            INT32_C( 1996286767), INT32_C( -816391022), INT32_C(   52318481), INT32_C( -732357064),
                            INT32_C(-1730294171), INT32_C( 1249707888), INT32_C( 1882801706), INT32_C( -348126448)),
      simde_mm512_set_epi32(INT32_C( 1558252976), INT32_C( -896391163), INT32_C( -282653026), INT32_C( -261252112),
                            INT32_C(-1886684518), INT32_C( -792082029), INT32_C(-1821569388), INT32_C(  452082945),
                            INT32_C( 1996286767), INT32_C( -816391022), INT32_C(   52318481), INT32_C( -732357064),
                            INT32_C(-1730294171), INT32_C( 1249707888), INT32_C( 1882801706), INT32_C( -348126448)) },
    { simde_mm512_set_epi32(INT32_C(-1306007403), INT32_C(  -14823844), INT32_C(   68140915), INT32_C(-1279691583),
                            INT32_C( -366833324), INT32_C( 1029809788), INT32_C(  398169753), INT32_C( 1644631001),
                            INT32_C(-1495643320), INT32_C( -349662299), INT32_C(  337000079), INT32_C(-1599869755),
                            INT32_C( -326525511), INT32_C( 2145777358), INT32_C( -671246537), INT32_C(-1483766126)),
      simde_mm512_set_epi32(INT32_C(-1306007403), INT32_C(  -14823844), INT32_C(   68140915), INT32_C(-1279691583),
                            INT32_C( -366833324), INT32_C( 1029809788), INT32_C(  398169753), INT32_C( 1644631001),
                            INT32_C(-1495643320), INT32_C( -349662299), INT32_C(  337000079), INT32_C(-1599869755),
                            INT32_C( -326525511), INT32_C( 2145777358), INT32_C( -671246537), INT32_C(-1483766126)) },
    { simde_mm512_set_epi32(INT32_C( 1306718049), INT32_C( -481786167), INT32_C( 1036658403), INT32_C(-1275842496),
                            INT32_C(-1384128337), INT32_C( 1110346848), INT32_C( 1633808121), INT32_C(  818679772),
                            INT32_C(  528120539), INT32_C( 1454269875), INT32_C(-1309905851), INT32_C(  195834431),
                            INT32_C( 1710057355), INT32_C(-1286641516), INT32_C( -825070396), INT32_C(-1314661204)),
      simde_mm512_set_epi32(INT32_C( 1306718049), INT32_C( -481786167), INT32_C( 1036658403), INT32_C(-1275842496),
                            INT32_C(-1384128337), INT32_C( 1110346848), INT32_C( 1633808121), INT32_C(  818679772),
                            INT32_C(  528120539), INT32_C( 1454269875), INT32_C(-1309905851), INT32_C(  195834431),
                            INT32_C( 1710057355), INT32_C(-1286641516), INT32_C( -825070396), INT32_C(-1314661204)) },
    { simde_mm512_set_epi32(INT32_C(  897666230), INT32_C(-1247200387), INT32_C(-1594960612), INT32_C(  784358708),
                            INT32_C( -639715103), INT32_C( 1722588202), INT32_C( -957042730), INT32_C( 1748493888),
                            INT32_C(-1424605467), INT32_C( 1715503797), INT32_C(-1877902145), INT32_C( 1433291829),
                            INT32_C(-1672795293), INT32_C( -125523250), INT32_C(  817060741), INT32_C(-1940108057)),
      simde_mm512_set_epi32(INT32_C(  897666230), INT32_C(-1247200387), INT32_C(-1594960612), INT32_C(  784358708),
                            INT32_C( -639715103), INT32_C( 1722588202), INT32_C( -957042730), INT32_C( 1748493888),
                            INT32_C(-1424605467), INT32_C( 1715503797), INT32_C(-1877902145), INT32_C( 1433291829),
                            INT32_C(-1672795293), INT32_C( -125523250), INT32_C(  817060741), INT32_C(-1940108057)) },
    { simde_mm512_set_epi32(INT32_C( -771897847), INT32_C(   51088448), INT32_C(  731748986), INT32_C(-2010019631),
                            INT32_C(-1805256328), INT32_C( -982668321), INT32_C(-1986983933), INT32_C( -806537017),
                            INT32_C( 2103732941), INT32_C(-1950119891), INT32_C( -262758582), INT32_C( -704905824),
                            INT32_C( -298698020), INT32_C( -290664422), INT32_C( -169474404), INT32_C( -381923585)),
      simde_mm512_set_epi32(INT32_C( -771897847), INT32_C(   51088448), INT32_C(  731748986), INT32_C(-2010019631),
                            INT32_C(-1805256328), INT32_C( -982668321), INT32_C(-1986983933), INT32_C( -806537017),
                            INT32_C( 2103732941), INT32_C(-1950119891), INT32_C( -262758582), INT32_C( -704905824),
                            INT32_C( -298698020), INT32_C( -290664422), INT32_C( -169474404), INT32_C( -381923585)) },
    { simde_mm512_set_epi32(INT32_C(-1424890095), INT32_C( -662922774), INT32_C( 1172732731), INT32_C(  220940559),
                            INT32_C(  736508018), INT32_C(-1201335155), INT32_C( -747976663), INT32_C( 1864271935),
                            INT32_C( 1893933430), INT32_C(  214467364), INT32_C(-1230640603), INT32_C(-1529352277),
                            INT32_C(  301681975), INT32_C(  -86355089), INT32_C( 1945634979), INT32_C( 1310017249)),
      simde_mm512_set_epi32(INT32_C(-1424890095), INT32_C( -662922774), INT32_C( 1172732731), INT32_C(  220940559),
                            INT32_C(  736508018), INT32_C(-1201335155), INT32_C( -747976663), INT32_C( 1864271935),
                            INT32_C( 1893933430), INT32_C(  214467364), INT32_C(-1230640603), INT32_C(-1529352277),
                            INT32_C(  301681975), INT32_C(  -86355089), INT32_C( 1945634979), INT32_C( 1310017249)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_castsi512_ps(test_vec[i].a));
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castps128_ps512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -124.41), SIMDE_FLOAT32_C(   994.42), SIMDE_FLOAT32_C(  -888.56), SIMDE_FLOAT32_C(  -241.67)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -124.41), SIMDE_FLOAT32_C(   994.42), SIMDE_FLOAT32_C(  -888.56), SIMDE_FLOAT32_C(  -241.67)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -472.63), SIMDE_FLOAT32_C(  -923.61), SIMDE_FLOAT32_C(  -185.31), SIMDE_FLOAT32_C(   531.85)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -472.63), SIMDE_FLOAT32_C(  -923.61), SIMDE_FLOAT32_C(  -185.31), SIMDE_FLOAT32_C(   531.85)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -636.73), SIMDE_FLOAT32_C(  -810.98), SIMDE_FLOAT32_C(  -952.52), SIMDE_FLOAT32_C(   896.08)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -636.73), SIMDE_FLOAT32_C(  -810.98), SIMDE_FLOAT32_C(  -952.52), SIMDE_FLOAT32_C(   896.08)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   142.17), SIMDE_FLOAT32_C(  -205.03), SIMDE_FLOAT32_C(  -269.17), SIMDE_FLOAT32_C(  -426.17)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   142.17), SIMDE_FLOAT32_C(  -205.03), SIMDE_FLOAT32_C(  -269.17), SIMDE_FLOAT32_C(  -426.17)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   149.14), SIMDE_FLOAT32_C(  -672.83), SIMDE_FLOAT32_C(  -162.76), SIMDE_FLOAT32_C(  -647.13)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   149.14), SIMDE_FLOAT32_C(  -672.83), SIMDE_FLOAT32_C(  -162.76), SIMDE_FLOAT32_C(  -647.13)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -915.60), SIMDE_FLOAT32_C(    75.79), SIMDE_FLOAT32_C(   831.71), SIMDE_FLOAT32_C(   144.92)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -915.60), SIMDE_FLOAT32_C(    75.79), SIMDE_FLOAT32_C(   831.71), SIMDE_FLOAT32_C(   144.92)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -748.69), SIMDE_FLOAT32_C(  -492.84), SIMDE_FLOAT32_C(  -132.48), SIMDE_FLOAT32_C(  -765.52)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -748.69), SIMDE_FLOAT32_C(  -492.84), SIMDE_FLOAT32_C(  -132.48), SIMDE_FLOAT32_C(  -765.52)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   346.51), SIMDE_FLOAT32_C(   297.54), SIMDE_FLOAT32_C(   196.93), SIMDE_FLOAT32_C(  -729.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   346.51), SIMDE_FLOAT32_C(   297.54), SIMDE_FLOAT32_C(   196.93), SIMDE_FLOAT32_C(  -729.98)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_castps128_ps512(test_vec[i].a);
    r=simde_mm512_maskz_mov_ps(UINT8_C(15), r);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castps256_ps512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m256 a;
    simde__m512 r;
  } test_vec[8] = {
     { simde_mm256_set_ps(SIMDE_FLOAT32_C(  -144.51), SIMDE_FLOAT32_C(   522.85),
                         SIMDE_FLOAT32_C(   259.94), SIMDE_FLOAT32_C(   889.02),
                         SIMDE_FLOAT32_C(    47.86), SIMDE_FLOAT32_C(   181.02),
                         SIMDE_FLOAT32_C(   389.89), SIMDE_FLOAT32_C(   -27.36)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -144.51), SIMDE_FLOAT32_C(   522.85), SIMDE_FLOAT32_C(   259.94), SIMDE_FLOAT32_C(   889.02),
                         SIMDE_FLOAT32_C(    47.86), SIMDE_FLOAT32_C(   181.02), SIMDE_FLOAT32_C(   389.89), SIMDE_FLOAT32_C(   -27.36)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(  -477.70), SIMDE_FLOAT32_C(  -298.40),
                         SIMDE_FLOAT32_C(  -730.62), SIMDE_FLOAT32_C(   379.73),
                         SIMDE_FLOAT32_C(   223.22), SIMDE_FLOAT32_C(  -917.34),
                         SIMDE_FLOAT32_C(  -268.46), SIMDE_FLOAT32_C(  -361.61)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -477.70), SIMDE_FLOAT32_C(  -298.40), SIMDE_FLOAT32_C(  -730.62), SIMDE_FLOAT32_C(   379.73),
                         SIMDE_FLOAT32_C(   223.22), SIMDE_FLOAT32_C(  -917.34), SIMDE_FLOAT32_C(  -268.46), SIMDE_FLOAT32_C(  -361.61)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   995.68), SIMDE_FLOAT32_C(   377.56),
                         SIMDE_FLOAT32_C(  -263.10), SIMDE_FLOAT32_C(   880.58),
                         SIMDE_FLOAT32_C(    61.47), SIMDE_FLOAT32_C(   993.95),
                         SIMDE_FLOAT32_C(   -45.95), SIMDE_FLOAT32_C(  -445.12)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   995.68), SIMDE_FLOAT32_C(   377.56), SIMDE_FLOAT32_C(  -263.10), SIMDE_FLOAT32_C(   880.58),
                         SIMDE_FLOAT32_C(    61.47), SIMDE_FLOAT32_C(   993.95), SIMDE_FLOAT32_C(   -45.95), SIMDE_FLOAT32_C(  -445.12)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   391.67), SIMDE_FLOAT32_C(  -695.88),
                         SIMDE_FLOAT32_C(   312.22), SIMDE_FLOAT32_C(   389.64),
                         SIMDE_FLOAT32_C(    81.09), SIMDE_FLOAT32_C(   745.89),
                         SIMDE_FLOAT32_C(  -178.17), SIMDE_FLOAT32_C(  -269.92)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   391.67), SIMDE_FLOAT32_C(  -695.88), SIMDE_FLOAT32_C(   312.22), SIMDE_FLOAT32_C(   389.64),
                         SIMDE_FLOAT32_C(    81.09), SIMDE_FLOAT32_C(   745.89), SIMDE_FLOAT32_C(  -178.17), SIMDE_FLOAT32_C(  -269.92)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   498.47), SIMDE_FLOAT32_C(  -985.97),
                         SIMDE_FLOAT32_C(   802.71), SIMDE_FLOAT32_C(   -28.60),
                         SIMDE_FLOAT32_C(  -781.89), SIMDE_FLOAT32_C(  -410.06),
                         SIMDE_FLOAT32_C(  -602.74), SIMDE_FLOAT32_C(  -353.26)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   498.47), SIMDE_FLOAT32_C(  -985.97), SIMDE_FLOAT32_C(   802.71), SIMDE_FLOAT32_C(   -28.60),
                         SIMDE_FLOAT32_C(  -781.89), SIMDE_FLOAT32_C(  -410.06), SIMDE_FLOAT32_C(  -602.74), SIMDE_FLOAT32_C(  -353.26)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   265.34), SIMDE_FLOAT32_C(  -587.26),
                         SIMDE_FLOAT32_C(  -801.19), SIMDE_FLOAT32_C(    -4.56),
                         SIMDE_FLOAT32_C(  -900.49), SIMDE_FLOAT32_C(   391.40),
                         SIMDE_FLOAT32_C(  -416.15), SIMDE_FLOAT32_C(  -286.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   265.34), SIMDE_FLOAT32_C(  -587.26), SIMDE_FLOAT32_C(  -801.19), SIMDE_FLOAT32_C(    -4.56),
                         SIMDE_FLOAT32_C(  -900.49), SIMDE_FLOAT32_C(   391.40), SIMDE_FLOAT32_C(  -416.15), SIMDE_FLOAT32_C(  -286.27)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   167.27), SIMDE_FLOAT32_C(  -483.42),
                         SIMDE_FLOAT32_C(   759.82), SIMDE_FLOAT32_C(  -560.80),
                         SIMDE_FLOAT32_C(  -968.24), SIMDE_FLOAT32_C(  -534.87),
                         SIMDE_FLOAT32_C(  -698.82), SIMDE_FLOAT32_C(  -382.24)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   167.27), SIMDE_FLOAT32_C(  -483.42), SIMDE_FLOAT32_C(   759.82), SIMDE_FLOAT32_C(  -560.80),
                         SIMDE_FLOAT32_C(  -968.24), SIMDE_FLOAT32_C(  -534.87), SIMDE_FLOAT32_C(  -698.82), SIMDE_FLOAT32_C(  -382.24)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(  -792.23), SIMDE_FLOAT32_C(   994.18),
                         SIMDE_FLOAT32_C(  -632.41), SIMDE_FLOAT32_C(   872.87),
                         SIMDE_FLOAT32_C(  -406.41), SIMDE_FLOAT32_C(   989.00),
                         SIMDE_FLOAT32_C(   781.89), SIMDE_FLOAT32_C(   609.26)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -792.23), SIMDE_FLOAT32_C(   994.18), SIMDE_FLOAT32_C(  -632.41), SIMDE_FLOAT32_C(   872.87),
                         SIMDE_FLOAT32_C(  -406.41), SIMDE_FLOAT32_C(   989.00), SIMDE_FLOAT32_C(   781.89), SIMDE_FLOAT32_C(   609.26)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_castps256_ps512(test_vec[i].a);
    r=simde_mm512_maskz_mov_ps(UINT8_C(255), r);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castsi512_si128(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1658575222), INT32_C(-1117261553), INT32_C(-1839997259), INT32_C( -299852262),
                            INT32_C( -856912374), INT32_C( 2142936567), INT32_C( -954684084), INT32_C( 1657017766),
                            INT32_C( -348123015), INT32_C( -101609698), INT32_C(  554693435), INT32_C(-1533582435),
                            INT32_C( -400642696), INT32_C( 1776297700), INT32_C( 1392524409), INT32_C(  398537104)),
      simde_mm_set_epi32(INT32_C( -400642696), INT32_C( 1776297700), INT32_C( 1392524409), INT32_C(  398537104)) },
    { simde_mm512_set_epi32(INT32_C(  776372316), INT32_C(  831674978), INT32_C(   55416914), INT32_C(  243273534),
                            INT32_C( -792877249), INT32_C( -771344813), INT32_C( -310546031), INT32_C(-2107743643),
                            INT32_C(-2024392408), INT32_C(-1627000847), INT32_C( 2079816508), INT32_C(  371701407),
                            INT32_C(-1936589526), INT32_C(-1276131185), INT32_C(-2091159003), INT32_C(  424652427)),
      simde_mm_set_epi32(INT32_C(-1936589526), INT32_C(-1276131185), INT32_C(-2091159003), INT32_C(  424652427)) },
    { simde_mm512_set_epi32(INT32_C( 1173201274), INT32_C( 2136038971), INT32_C( 1938580455), INT32_C( 1481503254),
                            INT32_C( -900987494), INT32_C(  381103974), INT32_C( -844512348), INT32_C(  585017509),
                            INT32_C(  711603246), INT32_C(-1519792959), INT32_C( 1933920466), INT32_C(-1698769431),
                            INT32_C(-1782976050), INT32_C(-1748575141), INT32_C(-1460360861), INT32_C( -334769088)),
      simde_mm_set_epi32(INT32_C(-1782976050), INT32_C(-1748575141), INT32_C(-1460360861), INT32_C( -334769088)) },
    { simde_mm512_set_epi32(INT32_C( 1804169520), INT32_C(-1929398298), INT32_C( 1527159056), INT32_C(  779300869),
                            INT32_C(-1718101462), INT32_C( -324959902), INT32_C(  356284621), INT32_C(  259973358),
                            INT32_C(  -85521098), INT32_C( 1509969945), INT32_C( -146300380), INT32_C(-1326146169),
                            INT32_C( -222527512), INT32_C( 2102498956), INT32_C(   21770579), INT32_C(  799947456)),
      simde_mm_set_epi32(INT32_C( -222527512), INT32_C( 2102498956), INT32_C(   21770579), INT32_C(  799947456)) },
    { simde_mm512_set_epi32(INT32_C(   21446459), INT32_C( -874117397), INT32_C(-1237626592), INT32_C(-1776987073),
                            INT32_C(-1868633440), INT32_C( 1756862535), INT32_C(  992623093), INT32_C( 2049114875),
                            INT32_C(-1780158792), INT32_C( -396995291), INT32_C( 1019180338), INT32_C(-1532238028),
                            INT32_C( 1367863842), INT32_C(-1092466184), INT32_C( 1828179807), INT32_C( -163821277)),
      simde_mm_set_epi32(INT32_C( 1367863842), INT32_C(-1092466184), INT32_C( 1828179807), INT32_C( -163821277)) },
    { simde_mm512_set_epi32(INT32_C(  430786946), INT32_C( -784387301), INT32_C(-1254472927), INT32_C(-1743072668),
                            INT32_C(  651286019), INT32_C(-1388487554), INT32_C(  757390683), INT32_C( 1947676255),
                            INT32_C(  969518281), INT32_C(  -72978072), INT32_C(  760758773), INT32_C( 1573392247),
                            INT32_C(  865448841), INT32_C( 1831180971), INT32_C(-1639739355), INT32_C(   75957511)),
      simde_mm_set_epi32(INT32_C(  865448841), INT32_C( 1831180971), INT32_C(-1639739355), INT32_C(   75957511)) },
    { simde_mm512_set_epi32(INT32_C( 2143921197), INT32_C( -566618658), INT32_C( -232811985), INT32_C( -646802339),
                            INT32_C(-1178364835), INT32_C( 1276977735), INT32_C(  767086016), INT32_C( -313594533),
                            INT32_C(-1508762742), INT32_C(   64124631), INT32_C(-1825241034), INT32_C(  688272909),
                            INT32_C(  217977439), INT32_C(  193086834), INT32_C( -334037547), INT32_C(  -96078706)),
      simde_mm_set_epi32(INT32_C(  217977439), INT32_C(  193086834), INT32_C( -334037547), INT32_C(  -96078706)) },
    { simde_mm512_set_epi32(INT32_C(-1446801805), INT32_C( 1701962052), INT32_C( 2110825600), INT32_C( 1003088425),
                            INT32_C(-1369466226), INT32_C( -588712121), INT32_C(-1911969632), INT32_C( 1166459977),
                            INT32_C(-1912318103), INT32_C( -539596217), INT32_C(-1472577858), INT32_C( -435893517),
                            INT32_C( 1340330175), INT32_C( 1762005905), INT32_C(-1553204174), INT32_C( -839776889)),
      simde_mm_set_epi32(INT32_C( 1340330175), INT32_C( 1762005905), INT32_C(-1553204174), INT32_C( -839776889)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_castsi512_si128(test_vec[i].a);
    simde_assert_m128i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_castsi512_si256(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1313023361), INT32_C( -284143420), INT32_C(  762022716), INT32_C( -109476439),
                            INT32_C( -377196873), INT32_C(  289021876), INT32_C(-1835156104), INT32_C(-2032178077),
                            INT32_C(-1821500948), INT32_C(  676317044), INT32_C( 1400280404), INT32_C(-1176819357),
                            INT32_C( -771867753), INT32_C(-1499090744), INT32_C(  845761822), INT32_C( 1051767340)),
      simde_mm256_set_epi32(INT32_C(-1821500948), INT32_C(  676317044), INT32_C( 1400280404), INT32_C(-1176819357),
                            INT32_C( -771867753), INT32_C(-1499090744), INT32_C(  845761822), INT32_C( 1051767340)) },
    { simde_mm512_set_epi32(INT32_C(  584849517), INT32_C( 1957310042), INT32_C( 1025237717), INT32_C( -771085485),
                            INT32_C(  578687068), INT32_C(   23449370), INT32_C(-1748604352), INT32_C( 1528872918),
                            INT32_C(  331610618), INT32_C( 1589449489), INT32_C(-1723663783), INT32_C( -274890341),
                            INT32_C( 1614271470), INT32_C(-1087921179), INT32_C(  318041946), INT32_C(-1241918784)),
      simde_mm256_set_epi32(INT32_C(  331610618), INT32_C( 1589449489), INT32_C(-1723663783), INT32_C( -274890341),
                            INT32_C( 1614271470), INT32_C(-1087921179), INT32_C(  318041946), INT32_C(-1241918784)) },
    { simde_mm512_set_epi32(INT32_C(-1852610712), INT32_C( 1069895831), INT32_C(-1772412775), INT32_C( 1674180593),
                            INT32_C( 1091612022), INT32_C( 2100501546), INT32_C(  179298125), INT32_C( -696375983),
                            INT32_C(  697782088), INT32_C(  360978520), INT32_C( -569372444), INT32_C(-1688535258),
                            INT32_C(  857372618), INT32_C( 1720533351), INT32_C(  174096645), INT32_C(  -27592678)),
      simde_mm256_set_epi32(INT32_C(  697782088), INT32_C(  360978520), INT32_C( -569372444), INT32_C(-1688535258),
                            INT32_C(  857372618), INT32_C( 1720533351), INT32_C(  174096645), INT32_C(  -27592678)) },
    { simde_mm512_set_epi32(INT32_C( 1588790835), INT32_C( 1641509494), INT32_C(-1255607211), INT32_C(  325546446),
                            INT32_C( -545478700), INT32_C( 1405876195), INT32_C(-1090548279), INT32_C(-1934286399),
                            INT32_C(-1145918593), INT32_C( -667441949), INT32_C(  804839434), INT32_C( 1326022329),
                            INT32_C(  307496614), INT32_C(-1108181850), INT32_C( 1980353111), INT32_C( 2112031083)),
      simde_mm256_set_epi32(INT32_C(-1145918593), INT32_C( -667441949), INT32_C(  804839434), INT32_C( 1326022329),
                            INT32_C(  307496614), INT32_C(-1108181850), INT32_C( 1980353111), INT32_C( 2112031083)) },
    { simde_mm512_set_epi32(INT32_C(-1042067236), INT32_C(-1304341967), INT32_C(  157202478), INT32_C(  335492980),
                            INT32_C( 1721471627), INT32_C( -970374929), INT32_C(-1215103856), INT32_C(  804236438),
                            INT32_C(  763203116), INT32_C( 1650455375), INT32_C(  191870396), INT32_C(-1509651897),
                            INT32_C( 1356692638), INT32_C(-1388072975), INT32_C(-1822839305), INT32_C(  511824893)),
      simde_mm256_set_epi32(INT32_C(  763203116), INT32_C( 1650455375), INT32_C(  191870396), INT32_C(-1509651897),
                            INT32_C( 1356692638), INT32_C(-1388072975), INT32_C(-1822839305), INT32_C(  511824893)) },
    { simde_mm512_set_epi32(INT32_C( -321038022), INT32_C( 1580565478), INT32_C( 1887212628), INT32_C( -321448210),
                            INT32_C( 1646995861), INT32_C(  223122732), INT32_C(-1757872907), INT32_C(-1745139814),
                            INT32_C( 1574697064), INT32_C( 1593370323), INT32_C( 1019611869), INT32_C( 1148772416),
                            INT32_C(   41212650), INT32_C( 1180707726), INT32_C(-1448405590), INT32_C( -480086797)),
      simde_mm256_set_epi32(INT32_C( 1574697064), INT32_C( 1593370323), INT32_C( 1019611869), INT32_C( 1148772416),
                            INT32_C(   41212650), INT32_C( 1180707726), INT32_C(-1448405590), INT32_C( -480086797)) },
    { simde_mm512_set_epi32(INT32_C(-2123925649), INT32_C(-1300783990), INT32_C( 1491522954), INT32_C(-1497225833),
                            INT32_C(  264593112), INT32_C(-1577295575), INT32_C( 1721102182), INT32_C( -406919611),
                            INT32_C( -248074414), INT32_C(-1644442090), INT32_C(-1586232899), INT32_C(  -62204050),
                            INT32_C(  356135618), INT32_C(-1239722051), INT32_C( 1765867002), INT32_C(-2007643147)),
      simde_mm256_set_epi32(INT32_C( -248074414), INT32_C(-1644442090), INT32_C(-1586232899), INT32_C(  -62204050),
                            INT32_C(  356135618), INT32_C(-1239722051), INT32_C( 1765867002), INT32_C(-2007643147)) },
    { simde_mm512_set_epi32(INT32_C( 1712658433), INT32_C( 1782026188), INT32_C( 1026238577), INT32_C(  999831740),
                            INT32_C( 1890262543), INT32_C( -282267234), INT32_C( -314081065), INT32_C( 1258071145),
                            INT32_C(  -96847849), INT32_C(  -33628993), INT32_C( -700755553), INT32_C( 1395877145),
                            INT32_C( 1943621426), INT32_C( 1606414493), INT32_C(  234520793), INT32_C( 1932887352)),
      simde_mm256_set_epi32(INT32_C(  -96847849), INT32_C(  -33628993), INT32_C( -700755553), INT32_C( 1395877145),
                            INT32_C( 1943621426), INT32_C( 1606414493), INT32_C(  234520793), INT32_C( 1932887352)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_castsi512_si256(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cmpgt_epi32_mask(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__mmask16 r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -126651070), INT32_C( 1757388710), INT32_C(  617530196), INT32_C(  407807901),
                            INT32_C( 1271989524), INT32_C( 1251214807), INT32_C(-1247045111), INT32_C(-1024057759),
                            INT32_C(   50729453), INT32_C(  464444874), INT32_C( 1840702207), INT32_C( 1916050591),
                            INT32_C(  484601458), INT32_C( -782065931), INT32_C(-1485735658), INT32_C(-1326388993)),
      simde_mm512_set_epi32(INT32_C(  111072774), INT32_C( 1757388710), INT32_C( 1496897687), INT32_C(  407807901),
                            INT32_C( 1271989524), INT32_C( 1496985365), INT32_C(-1247045111), INT32_C(-1225014979),
                            INT32_C(   50729453), INT32_C(  464444874), INT32_C(  924537351), INT32_C( 1916050591),
                            INT32_C(  484601458), INT32_C( -782065931), INT32_C(-1485735658), INT32_C(-1326388993)),
      UINT16_C(  288) },
    { simde_mm512_set_epi32(INT32_C( 2106044062), INT32_C( 1752498924), INT32_C(-1086695378), INT32_C(  627787891),
                            INT32_C(-1783053554), INT32_C(-1485517848), INT32_C( 1105114322), INT32_C(-1862707588),
                            INT32_C(  823946037), INT32_C(-2030244995), INT32_C( -219180660), INT32_C(  810910725),
                            INT32_C( -642105946), INT32_C(  760774613), INT32_C(  -62603432), INT32_C(-2064446807)),
      simde_mm512_set_epi32(INT32_C( 2106044062), INT32_C( 1752498924), INT32_C( -582421212), INT32_C( 1649238471),
                            INT32_C( 1446053889), INT32_C(-1485517848), INT32_C( 1105114322), INT32_C(-1862707588),
                            INT32_C( -846383385), INT32_C(-2030244995), INT32_C( -905258415), INT32_C(  810910725),
                            INT32_C(-1668595380), INT32_C( -760772652), INT32_C( 2145797270), INT32_C(   57887151)),
      UINT16_C(  172) },
    { simde_mm512_set_epi32(INT32_C(  948728954), INT32_C(  965445469), INT32_C( -298261731), INT32_C( 1889741023),
                            INT32_C(  101476677), INT32_C( -598834633), INT32_C( 1592735604), INT32_C(  428243294),
                            INT32_C(-2001034764), INT32_C( -639043872), INT32_C(  567427880), INT32_C(-1305749494),
                            INT32_C( -204185535), INT32_C( -550643286), INT32_C( -170363385), INT32_C( 1483518213)),
      simde_mm512_set_epi32(INT32_C(  948728954), INT32_C(  965445469), INT32_C(  364841947), INT32_C(-1221758106),
                            INT32_C(  101476677), INT32_C( -598834633), INT32_C( 1592735604), INT32_C(-1456245493),
                            INT32_C(-2001034764), INT32_C( -639043872), INT32_C(  567427880), INT32_C(-1305749494),
                            INT32_C( -204185535), INT32_C(  830345587), INT32_C( -170363385), INT32_C( -603563929)),
      UINT16_C( 4353) },
    { simde_mm512_set_epi32(INT32_C( -163413000), INT32_C( -831194762), INT32_C( -664019578), INT32_C( 2031024026),
                            INT32_C( 1912388774), INT32_C(  982200166), INT32_C(  596130243), INT32_C(  446035443),
                            INT32_C( 1373006598), INT32_C(-1540837035), INT32_C( 1581631435), INT32_C(-2083299381),
                            INT32_C( 1992847454), INT32_C(  448258110), INT32_C(  875345838), INT32_C( 1612926819)),
      simde_mm512_set_epi32(INT32_C(   81573630), INT32_C( -831194762), INT32_C( -664019578), INT32_C( 2031024026),
                            INT32_C( 1912388774), INT32_C(-2072470454), INT32_C(  596130243), INT32_C( 1961646011),
                            INT32_C( 1373006598), INT32_C( -197223193), INT32_C( 1581631435), INT32_C(-2083299381),
                            INT32_C( 1457480410), INT32_C( 1181119535), INT32_C( 1263228451), INT32_C(-1998542716)),
      UINT16_C( 1033) },
    { simde_mm512_set_epi32(INT32_C( 1436278246), INT32_C(   99684976), INT32_C( 1345577484), INT32_C(  166701508),
                            INT32_C( -780731111), INT32_C( -840749601), INT32_C( 1523342039), INT32_C( 1058674665),
                            INT32_C( -523908416), INT32_C( 1659465207), INT32_C(-1927062215), INT32_C(-1156760340),
                            INT32_C(  715569317), INT32_C(-1515814414), INT32_C( 1243253180), INT32_C( 2080215882)),
      simde_mm512_set_epi32(INT32_C(  432908742), INT32_C(   99684976), INT32_C(  -14330157), INT32_C(-1223154556),
                            INT32_C( -780731111), INT32_C(  696697372), INT32_C( 1523342039), INT32_C( 1058674665),
                            INT32_C( -523908416), INT32_C( 1659465207), INT32_C(-1927062215), INT32_C(-1156760340),
                            INT32_C( -171262349), INT32_C(-1515814414), INT32_C(-1234169573), INT32_C(-1847568101)),
      UINT16_C(45067) },
    { simde_mm512_set_epi32(INT32_C( 1399825551), INT32_C(-1064541474), INT32_C( 2112452992), INT32_C(  575137303),
                            INT32_C( -979898374), INT32_C(-1476679333), INT32_C( 1320423852), INT32_C( 1767893242),
                            INT32_C( -389599783), INT32_C(-1459729991), INT32_C(  995424065), INT32_C( -522129019),
                            INT32_C( -466751981), INT32_C( 1371238810), INT32_C( 1006677155), INT32_C( 1609037982)),
      simde_mm512_set_epi32(INT32_C( 1399825551), INT32_C(-1064541474), INT32_C( 2112452992), INT32_C(  134645750),
                            INT32_C(  500192289), INT32_C( 1600988950), INT32_C( 1320423852), INT32_C( 1198845893),
                            INT32_C( -389599783), INT32_C( 1504468794), INT32_C(  995424065), INT32_C(-2123865443),
                            INT32_C( -466751981), INT32_C(-1711282630), INT32_C( 1006677155), INT32_C( 1609037982)),
      UINT16_C( 4372) },
    { simde_mm512_set_epi32(INT32_C(-1862774816), INT32_C(   28374488), INT32_C(  250156705), INT32_C( -932694837),
                            INT32_C(-2079251566), INT32_C( -246439183), INT32_C( -875109534), INT32_C( 1740046060),
                            INT32_C( 1735819269), INT32_C( 1371885292), INT32_C( -914870851), INT32_C( -473073032),
                            INT32_C( -580976455), INT32_C( 1688786028), INT32_C(  637430498), INT32_C(-1740972685)),
      simde_mm512_set_epi32(INT32_C(-1862774816), INT32_C(   28374488), INT32_C(  580744870), INT32_C( -666445473),
                            INT32_C( -129274908), INT32_C( -928751425), INT32_C( -388443661), INT32_C( 1740046060),
                            INT32_C(-1909361652), INT32_C( 1371885292), INT32_C(  857928163), INT32_C(   37075976),
                            INT32_C( -580976455), INT32_C(-1545948444), INT32_C(  637430498), INT32_C(-1740972685)),
      UINT16_C( 1156) },
    { simde_mm512_set_epi32(INT32_C(-1890406982), INT32_C( 2110791016), INT32_C( 1083476771), INT32_C( -620691621),
                            INT32_C(  543588207), INT32_C( -227503647), INT32_C( -759273149), INT32_C(  775085710),
                            INT32_C( 1404885802), INT32_C(-1395233065), INT32_C(  832528180), INT32_C( 1065959566),
                            INT32_C(-2083201484), INT32_C(  937916550), INT32_C( -710457746), INT32_C( -246147415)),
      simde_mm512_set_epi32(INT32_C(   84669207), INT32_C(  470641840), INT32_C( 1083476771), INT32_C( -620691621),
                            INT32_C( 1099959895), INT32_C( -961354454), INT32_C(-1751384146), INT32_C(  775085710),
                            INT32_C( 1075765582), INT32_C(  834655006), INT32_C(  832528180), INT32_C( 1065959566),
                            INT32_C(  954342416), INT32_C(  937916550), INT32_C(-1946395018), INT32_C(  757651617)),
      UINT16_C(18050) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask16 r = simde_mm512_cmpgt_epi32_mask(test_vec[i].a, test_vec[i].b);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cmpgt_epi32_mask(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__mmask16 r;
  } test_vec[8] = {
       { UINT16_C(12249),
      simde_mm512_set_epi32(INT32_C(-1151856667), INT32_C(  -49918748), INT32_C(-1709830250), INT32_C( 1750293451),
                            INT32_C(-1728641738), INT32_C(   79295022), INT32_C(  308064941), INT32_C( 1216157597),
                            INT32_C(  320231148), INT32_C( -697875804), INT32_C(  990066955), INT32_C(-2091005609),
                            INT32_C( 1037816180), INT32_C( -748290940), INT32_C( 1166526776), INT32_C(-1428331975)),
      simde_mm512_set_epi32(INT32_C( 1846695950), INT32_C(  884046092), INT32_C( -392734818), INT32_C(-1618937497),
                            INT32_C(  670851975), INT32_C(-1269946840), INT32_C(-1917256160), INT32_C(  228203505),
                            INT32_C( 1263965918), INT32_C(-2053175323), INT32_C(-1206891799), INT32_C( -371464947),
                            INT32_C(  -99745007), INT32_C( -847464628), INT32_C(  -93605380), INT32_C(-1859612096)),
      UINT16_C( 1865) },
    { UINT16_C(47912),
      simde_mm512_set_epi32(INT32_C(  238725197), INT32_C( 1521340392), INT32_C(-2077410041), INT32_C( 2110592657),
                            INT32_C(  630925822), INT32_C(  689275449), INT32_C(-1970822997), INT32_C(-1836727953),
                            INT32_C(  237271984), INT32_C( -578417637), INT32_C( -768235708), INT32_C( 1362514984),
                            INT32_C( 2019276284), INT32_C(-1411239380), INT32_C( 2052618114), INT32_C(-1238213534)),
      simde_mm512_set_epi32(INT32_C( -669654436), INT32_C( -822780196), INT32_C(  205688995), INT32_C( 1680146061),
                            INT32_C(  393599682), INT32_C(-1451941808), INT32_C(  947305201), INT32_C(  -75999449),
                            INT32_C( -484166756), INT32_C(  833555969), INT32_C( -549302423), INT32_C( 1610578173),
                            INT32_C(-2070337581), INT32_C(  664412106), INT32_C(  255732930), INT32_C( 1319359183)),
      UINT16_C(38920) },
    { UINT16_C(35968),
      simde_mm512_set_epi32(INT32_C( -359339347), INT32_C( -666198998), INT32_C(  830421084), INT32_C(-2087460228),
                            INT32_C(-1475104408), INT32_C(  721289147), INT32_C( 1281161083), INT32_C(  852871551),
                            INT32_C(-1589170839), INT32_C( -746357372), INT32_C( -154169474), INT32_C( -148954645),
                            INT32_C(-1357418925), INT32_C(-2112116028), INT32_C(  174617048), INT32_C(   -4103177)),
      simde_mm512_set_epi32(INT32_C( -871886017), INT32_C(-1688556984), INT32_C(  524690471), INT32_C( -124192434),
                            INT32_C( 1423100629), INT32_C( -161239972), INT32_C( -396308843), INT32_C( 1070701282),
                            INT32_C(-1826016016), INT32_C(  425347357), INT32_C(  366897524), INT32_C( -401692546),
                            INT32_C( -812557761), INT32_C( 1614519786), INT32_C(-1648390428), INT32_C( 1830061179)),
      UINT16_C(33920) },
    { UINT16_C(16809),
      simde_mm512_set_epi32(INT32_C(-1280324837), INT32_C( -161347329), INT32_C(  735858798), INT32_C( -674499230),
                            INT32_C(  -21391979), INT32_C( -381504266), INT32_C( 1528317100), INT32_C(  430345703),
                            INT32_C( -679679907), INT32_C(  515475896), INT32_C( -731085275), INT32_C( 1308429395),
                            INT32_C(  582932299), INT32_C(  489160586), INT32_C( 1760701165), INT32_C(  118948129)),
      simde_mm512_set_epi32(INT32_C( 1240889055), INT32_C(  888122014), INT32_C(-1469013917), INT32_C( 1209972337),
                            INT32_C( -691361230), INT32_C(  361393556), INT32_C(-1167116782), INT32_C( 1467757879),
                            INT32_C( 2124803699), INT32_C( -154870634), INT32_C(-1603618479), INT32_C(-2032203238),
                            INT32_C( 2111773805), INT32_C( -496949166), INT32_C( 1844580247), INT32_C(-2053814402)),
      UINT16_C(   33) },
    { UINT16_C(44464),
      simde_mm512_set_epi32(INT32_C( 1072149321), INT32_C( 1813169024), INT32_C(-1284365076), INT32_C(-1623700182),
                            INT32_C(  530512850), INT32_C(  116537892), INT32_C(  258206492), INT32_C(  690441736),
                            INT32_C( 1005371642), INT32_C( 1116924342), INT32_C( 1297564984), INT32_C( -835039581),
                            INT32_C( 1286263864), INT32_C(-1749149234), INT32_C(  558298824), INT32_C( 1064688827)),
      simde_mm512_set_epi32(INT32_C( 1323805616), INT32_C(-1558886902), INT32_C( 1778691088), INT32_C(  108147743),
                            INT32_C( 1106435712), INT32_C( -967535450), INT32_C(  600280311), INT32_C(  109364043),
                            INT32_C(  423389578), INT32_C( 1225761441), INT32_C( -440804681), INT32_C( -707540326),
                            INT32_C(-1898655855), INT32_C(-1268681648), INT32_C(-1360056367), INT32_C( -275254487)),
      UINT16_C( 1440) },
    { UINT16_C(23993),
      simde_mm512_set_epi32(INT32_C(-2038065128), INT32_C( -446679229), INT32_C(   78082001), INT32_C(  379830516),
                            INT32_C(-1929569644), INT32_C( 1595859976), INT32_C(  320798226), INT32_C( -738570818),
                            INT32_C( -165441023), INT32_C( -172594873), INT32_C(  912601062), INT32_C(  -56802863),
                            INT32_C(  503255814), INT32_C( 2046199592), INT32_C( -622599746), INT32_C( 1337235103)),
      simde_mm512_set_epi32(INT32_C(-1519343201), INT32_C( -448055921), INT32_C(-1909251875), INT32_C( -347447915),
                            INT32_C(  397553753), INT32_C(  713040821), INT32_C(-1458903601), INT32_C(  -45886582),
                            INT32_C( 1230465483), INT32_C( -828483015), INT32_C( -699493978), INT32_C(-1811052070),
                            INT32_C( 1577065087), INT32_C( -109599940), INT32_C(-1093577090), INT32_C(-1788879767)),
      UINT16_C(21553) },
    { UINT16_C(10358),
      simde_mm512_set_epi32(INT32_C(  648390363), INT32_C(  -30837841), INT32_C(-1635592815), INT32_C( -694389961),
                            INT32_C( -883952626), INT32_C( -761345991), INT32_C(  346040825), INT32_C(-1780780575),
                            INT32_C( 1510717568), INT32_C(-1185143236), INT32_C( 2143540932), INT32_C(  880567806),
                            INT32_C(-1670993371), INT32_C(-1942419167), INT32_C(-1196759463), INT32_C( 1386099146)),
      simde_mm512_set_epi32(INT32_C(-1614031176), INT32_C(  414071648), INT32_C(-1152911954), INT32_C(  424701353),
                            INT32_C( 1739922394), INT32_C( -506382165), INT32_C(  257126844), INT32_C( 1724223193),
                            INT32_C( 1096709845), INT32_C(-1643231112), INT32_C(-1639890652), INT32_C( -403971200),
                            INT32_C( 1318667734), INT32_C(  206062573), INT32_C(  -18472190), INT32_C(   -1701112)),
      UINT16_C(  112) },
    { UINT16_C(35023),
      simde_mm512_set_epi32(INT32_C(  228305355), INT32_C(-1904004735), INT32_C(  118523411), INT32_C( 1661507666),
                            INT32_C(-1400326500), INT32_C(   63010183), INT32_C(   62197704), INT32_C( -635599967),
                            INT32_C( 1677709284), INT32_C(-1294080152), INT32_C( -900737233), INT32_C(-1991940005),
                            INT32_C( -240404149), INT32_C(-1448242105), INT32_C(-1972665039), INT32_C( 1511694245)),
      simde_mm512_set_epi32(INT32_C(-1506289043), INT32_C(   82234507), INT32_C( -557930538), INT32_C( -911612825),
                            INT32_C( 1352158017), INT32_C( -554125937), INT32_C( -727289650), INT32_C(-1102664191),
                            INT32_C( 1941639559), INT32_C(-2124299952), INT32_C( -385431179), INT32_C(  112242864),
                            INT32_C(  -66697069), INT32_C( 1379403470), INT32_C(-1996504296), INT32_C(  658235880)),
      UINT16_C(32835) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask16 r = simde_mm512_mask_cmpgt_epi32_mask(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cmpgt_epi64_mask(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__mmask8 r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-3344943500899736927), INT64_C( -508674271294480923),
                            INT64_C( 4367550852745697236), INT64_C(-1765523250257788813),
                            INT64_C(-6325172456788566604), INT64_C( 3340966423446181237),
                            INT64_C( 4899669249714194025), INT64_C(-3109134868060088778)),
      simde_mm512_set_epi64(INT64_C(-3344943500899736927), INT64_C( -115747659216396604),
                            INT64_C( 7173930031566073597), INT64_C(-1765523250257788813),
                            INT64_C(-4525526016698522721), INT64_C( 3340966423446181237),
                            INT64_C( 4899669249714194025), INT64_C(-3109134868060088778)),
      UINT8_C(  0) },
    { simde_mm512_set_epi64(INT64_C(  161300839730502102), INT64_C(-4154011080047619988),
                            INT64_C( 3510183776865199207), INT64_C( 5188039415407364902),
                            INT64_C(-8649541106015277925), INT64_C( 2036065042708484429),
                            INT64_C(-7714656548902029220), INT64_C(-9105101012109136347)),
      simde_mm512_set_epi64(INT64_C(  161300839730502102), INT64_C(-4154011080047619988),
                            INT64_C( 6120426103009778754), INT64_C( -248023738166197182),
                            INT64_C(-8649541106015277925), INT64_C( 2693047687180833180),
                            INT64_C(-7714656548902029220), INT64_C(-9105101012109136347)),
      UINT8_C( 16) },
    { simde_mm512_set_epi64(INT64_C(-2825253727352691686), INT64_C( 4405965118825132522),
                            INT64_C(-6791426899562410985), INT64_C(-4409900925389880930),
                            INT64_C( 1845845856613597884), INT64_C(-4842241423465696621),
                            INT64_C(  163081221433998591), INT64_C( 4482804709675222173)),
      simde_mm512_set_epi64(INT64_C( 3285810068291760082), INT64_C( 4405965118825132522),
                            INT64_C(-6791426899562410985), INT64_C(-4943963491966669187),
                            INT64_C( 1845845856613597884), INT64_C( 7384036385676540465),
                            INT64_C(-7313503223753260102), INT64_C( 5128036791088991318)),
      UINT8_C( 18) },
    { simde_mm512_set_epi64(INT64_C(-1633105180711142836), INT64_C(-4287337651200520652),
                            INT64_C( 8346707004388378871), INT64_C(-5848595418894109542),
                            INT64_C(-7300386321370732776), INT64_C( -648586863376006844),
                            INT64_C(-3473939784680689044), INT64_C(-1628617817613399979)),
      simde_mm512_set_epi64(INT64_C( 1934898870952452550), INT64_C(-4287337651200520652),
                            INT64_C( 1557479703737443505), INT64_C(-5848595418894109542),
                            INT64_C(-7179299072208562799), INT64_C( -648586863376006844),
                            INT64_C(-3473939784680689044), INT64_C( 2817575692173645704)),
      UINT8_C( 32) },
    { simde_mm512_set_epi64(INT64_C(-8952123954418726140), INT64_C( 5461301954902244462),
                            INT64_C(-5820184907423972656), INT64_C(  420402622060248705),
                            INT64_C(-1664441445637860283), INT64_C(-9088734991256809986),
                            INT64_C( 5606803261787264235), INT64_C( 3392608019150722653)),
      simde_mm512_set_epi64(INT64_C(-8952123954418726140), INT64_C(-6318099565586317695),
                            INT64_C(-5820184907423972656), INT64_C(  420402622060248705),
                            INT64_C(-1664441445637860283), INT64_C( 6565206217411025613),
                            INT64_C( 8598198622090956400), INT64_C(-7576266643160730964)),
      UINT8_C( 65) },
    { simde_mm512_set_epi64(INT64_C(-3313522622815895345), INT64_C(-6452175545498154090),
                            INT64_C( -937049212555566038), INT64_C(-4143019958444030865),
                            INT64_C(-3410665359562609619), INT64_C(  966786109195223540),
                            INT64_C( 7283097367839393163), INT64_C(-2640534975929709368)),
      simde_mm512_set_epi64(INT64_C(-3313522622815895345), INT64_C(-6452175545498154090),
                            INT64_C( 7057508826094118763), INT64_C(-2466255848420720587),
                            INT64_C(-3410665359562609619), INT64_C(-7091282311083875172),
                            INT64_C(-5778676633446214654), INT64_C(-2640534975929709368)),
      UINT8_C(  6) },
    { simde_mm512_set_epi64(INT64_C( 7946101066156420330), INT64_C(-1199223599247032864),
                            INT64_C(-1997073553979895023), INT64_C(-2305098272308636911),
                            INT64_C( -630363562210498119), INT64_C( 4426020973322885294),
                            INT64_C( 8782098874831326668), INT64_C(-6058337867533474769)),
      simde_mm512_set_epi64(INT64_C( 7946101066156420330), INT64_C(-1199223599247032864),
                            INT64_C(-1997073553979895023), INT64_C(-2305098272308636911),
                            INT64_C( -630363562210498119), INT64_C( 8629524505567702841),
                            INT64_C( 8782098874831326668), INT64_C( 2660246489815857132)),
      UINT8_C(  0) },
    { simde_mm512_set_epi64(INT64_C(-2815932903868980343), INT64_C(  791308056982133256),
                            INT64_C( 8277712790583824674), INT64_C(-3943050990178000322),
                            INT64_C(-2127265598488665647), INT64_C( 4379715049649431166),
                            INT64_C(-9154071905230416728), INT64_C(-2123362159730266714)),
      simde_mm512_set_epi64(INT64_C(-2815932903868980343), INT64_C(  791308056982133256),
                            INT64_C(-6685750631550937327), INT64_C( 1585978438239301211),
                            INT64_C( 3432556139556266760), INT64_C( 4379715049649431166),
                            INT64_C(-9154071905230416728), INT64_C(-1483875325616410698)),
      UINT8_C( 32) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask8 r = simde_mm512_cmpgt_epi64_mask(test_vec[i].a, test_vec[i].b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cmpgt_epi64_mask(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__mmask8 r;
  } test_vec[8] = {
    { UINT8_C( 16),
      simde_mm512_set_epi64(INT64_C( 2255026789087372129), INT64_C( 6954636019969939696),
                            INT64_C( 8135587588110756767), INT64_C(-6775895683000468083),
                            INT64_C( -900701109459786534), INT64_C(-7915280239032503360),
                            INT64_C(-1216817989725562739), INT64_C(-8432176550710264558)),
      simde_mm512_set_epi64(INT64_C( 2255026789087372129), INT64_C( 6954636019969939696),
                            INT64_C( 8135587588110756767), INT64_C(-6775895683000468083),
                            INT64_C( -900701109459786534), INT64_C(-3598046066747317833),
                            INT64_C(-1709941778912207388), INT64_C( 3169788859761289772)),
      UINT8_C(  0) },
    { UINT8_C(  6),
      simde_mm512_set_epi64(INT64_C(-2239217399172416071), INT64_C(-1788368623206610532),
                            INT64_C(-8621353419023287056), INT64_C( 2167646577764527484),
                            INT64_C( 7373916114077140533), INT64_C( 4679914315089846813),
                            INT64_C(-4785189148228967071), INT64_C(-2291884851836211058)),
      simde_mm512_set_epi64(INT64_C(-4674113889822188979), INT64_C( 6851357122574510655),
                            INT64_C(-8621353419023287056), INT64_C( 2167646577764527484),
                            INT64_C( 7373916114077140533), INT64_C(-2091630176064440564),
                            INT64_C(-4166997073722787613), INT64_C(-2291884851836211058)),
      UINT8_C(  4) },
    { UINT8_C(231),
      simde_mm512_set_epi64(INT64_C( -437845083503418422), INT64_C( 7030863114044209442),
                            INT64_C( 8238064857893579595), INT64_C( 3062172269146343786),
                            INT64_C( 7457006241836305381), INT64_C(-9078752323516671886),
                            INT64_C(-6382075143273833301), INT64_C( 3840898770164583597)),
      simde_mm512_set_epi64(INT64_C(-4268408126209392137), INT64_C( 7030863114044209442),
                            INT64_C(  663353489862938549), INT64_C( 3062172269146343786),
                            INT64_C( 7457006241836305381), INT64_C( 8174310593560152615),
                            INT64_C(-6382075143273833301), INT64_C(-4495103935185291795)),
      UINT8_C(161) },
    { UINT8_C( 60),
      simde_mm512_set_epi64(INT64_C( 3543184366849060052), INT64_C( 8101296544771348510),
                            INT64_C( 1359772700119148960), INT64_C(-8357828074665392254),
                            INT64_C(-5672294839872616078), INT64_C(-2918525673450782654),
                            INT64_C(-6303315662009814438), INT64_C( 4773615511108508590)),
      simde_mm512_set_epi64(INT64_C( 3543184366849060052), INT64_C(  286276641590586651),
                            INT64_C( 1359772700119148960), INT64_C(-3217204137928962858),
                            INT64_C(-5672294839872616078), INT64_C(-2918525673450782654),
                            INT64_C(-2554453706959743566), INT64_C(-6197005744039272430)),
      UINT8_C(  0) },
    { UINT8_C( 97),
      simde_mm512_set_epi64(INT64_C(-4278296701436995238), INT64_C( 3569507405853529045),
                            INT64_C(-3380367559374400304), INT64_C(-4948363566435325304),
                            INT64_C(-6678298576976263631), INT64_C( 8848650777417470336),
                            INT64_C( 6320411494008491541), INT64_C( 2280208700508329072)),
      simde_mm512_set_epi64(INT64_C(  326944370261152484), INT64_C( 3569507405853529045),
                            INT64_C(  715678757448860576), INT64_C(-4948363566435325304),
                            INT64_C(-6678298576976263631), INT64_C(-5367013526541491012),
                            INT64_C( 1008601224594483315), INT64_C( 2280208700508329072)),
      UINT8_C(  0) },
    { UINT8_C(153),
      simde_mm512_set_epi64(INT64_C( 8361426666750729591), INT64_C(-6668359429543518025),
                            INT64_C( 2952092805333509636), INT64_C( 8284871946243647248),
                            INT64_C(-8896262213455925533), INT64_C( 3194469353298560173),
                            INT64_C( 5466230282228711049), INT64_C(-1091365868294702661)),
      simde_mm512_set_epi64(INT64_C(-8667260419906723988), INT64_C(-6668359429543518025),
                            INT64_C( 2952092805333509636), INT64_C( 8284871946243647248),
                            INT64_C( 3185065043241333471), INT64_C( -134870333477219304),
                            INT64_C( 5466230282228711049), INT64_C(-4571723861926798973)),
      UINT8_C(129) },
    { UINT8_C( 60),
      simde_mm512_set_epi64(INT64_C(-5632979726637184794), INT64_C( 3790754159972080576),
                            INT64_C(-7842038005332057398), INT64_C(-1292705499011984897),
                            INT64_C( 7597886654367336733), INT64_C( 1457057381762531412),
                            INT64_C(-1572264173383359920), INT64_C(-8716209376375056305)),
      simde_mm512_set_epi64(INT64_C(-5632979726637184794), INT64_C( 3790754159972080576),
                            INT64_C( 1913605115921194336), INT64_C(-6143563121944184390),
                            INT64_C( 7597886654367336733), INT64_C( 1457057381762531412),
                            INT64_C( 7253226870637562008), INT64_C(-6283001429373579825)),
      UINT8_C( 16) },
    { UINT8_C( 88),
      simde_mm512_set_epi64(INT64_C(-2374777447002601129), INT64_C(-5785141086360428669),
                            INT64_C( 6450311718709789609), INT64_C( 4609381622161693926),
                            INT64_C( -638886780002324864), INT64_C(-5739159461288227194),
                            INT64_C(-4392084870376418631), INT64_C( 2798977638636065147)),
      simde_mm512_set_epi64(INT64_C(  753500986908300233), INT64_C(-5785141086360428669),
                            INT64_C( 6450311718709789609), INT64_C(-4648819914956469219),
                            INT64_C(-8767820380557260648), INT64_C(-5739159461288227194),
                            INT64_C( 2360822030941279123), INT64_C(-6092063218708168180)),
      UINT8_C( 24) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask8 r = simde_mm512_mask_cmpgt_epi64_mask(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cmpeq_epi32_mask(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__mmask16 r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1955445938), INT32_C( 1791143901), INT32_C(-1554982337), INT32_C(-1864115653),
                            INT32_C(-1774796435), INT32_C( 1168347531), INT32_C(  660969508), INT32_C( 1153796239),
                            INT32_C(  609464964), INT32_C( 1687040663), INT32_C( -477087011), INT32_C(  309017072),
                            INT32_C(-2144421691), INT32_C(  682838279), INT32_C( 1954361304), INT32_C( 1694661655)),
      simde_mm512_set_epi32(INT32_C( 1955445938), INT32_C( 1791143901), INT32_C(-1554982337), INT32_C(-1021004514),
                            INT32_C(-1774796435), INT32_C( 1168347531), INT32_C( 1662960443), INT32_C( 1153796239),
                            INT32_C(  609464964), INT32_C(  428287860), INT32_C(  954212709), INT32_C(  309017072),
                            INT32_C(-2144421691), INT32_C( 1263291650), INT32_C( 1954361304), INT32_C( 1694661655)),
      UINT16_C(60827) },
    { simde_mm512_set_epi32(INT32_C(-1966343749), INT32_C(  200215975), INT32_C(-1192030414), INT32_C(  122433675),
                            INT32_C( 2024442800), INT32_C(-1612503082), INT32_C( -352764842), INT32_C( -964919402),
                            INT32_C(  198779956), INT32_C(-1052322954), INT32_C(-2053476283), INT32_C( 1410443780),
                            INT32_C( -220052733), INT32_C( 1401656692), INT32_C(  344284159), INT32_C(  358488145)),
      simde_mm512_set_epi32(INT32_C(-1966343749), INT32_C(  200215975), INT32_C( 1606502639), INT32_C(  122433675),
                            INT32_C( 2024442800), INT32_C( 1676122341), INT32_C( 1013297923), INT32_C( 1764819720),
                            INT32_C(-1006160158), INT32_C(  500282446), INT32_C(-2053476283), INT32_C( 1410443780),
                            INT32_C( -891361692), INT32_C( 1401656692), INT32_C(-1666424273), INT32_C(  358488145)),
      UINT16_C(55349) },
    { simde_mm512_set_epi32(INT32_C( 2029514541), INT32_C(-1405121342), INT32_C( -922299669), INT32_C(-1157312309),
                            INT32_C(  348700274), INT32_C( 1888848128), INT32_C( -909115111), INT32_C(-1689020830),
                            INT32_C( -310128293), INT32_C(-2105034529), INT32_C( 1894199289), INT32_C( -520350707),
                            INT32_C( 2077151614), INT32_C(  947639177), INT32_C(  972348424), INT32_C(  688864219)),
      simde_mm512_set_epi32(INT32_C( 2029514541), INT32_C(-1405121342), INT32_C( -922299669), INT32_C(-1157312309),
                            INT32_C( -582496350), INT32_C(  321618555), INT32_C( -707023911), INT32_C(-1689020830),
                            INT32_C(-1223923200), INT32_C( -293312096), INT32_C( 1894199289), INT32_C(  -89856908),
                            INT32_C( 2077151614), INT32_C(  947639177), INT32_C(  352633301), INT32_C( -580477735)),
      UINT16_C(61740) },
    { simde_mm512_set_epi32(INT32_C(   63224893), INT32_C( -945243939), INT32_C( 1472177104), INT32_C(-1518639341),
                            INT32_C( 1244274955), INT32_C(-1053463974), INT32_C(  207788622), INT32_C( -375726536),
                            INT32_C( -219606791), INT32_C( -871332353), INT32_C( 2036105492), INT32_C( 1465626684),
                            INT32_C(  930923741), INT32_C(-1996202276), INT32_C(  336972805), INT32_C(-1729161210)),
      simde_mm512_set_epi32(INT32_C( 1739683658), INT32_C( -945243939), INT32_C( 1472177104), INT32_C( -959740920),
                            INT32_C( 1244274955), INT32_C(  236525659), INT32_C(  207788622), INT32_C( 1159372556),
                            INT32_C( -632841040), INT32_C( -871332353), INT32_C( 2036105492), INT32_C(-1821161401),
                            INT32_C(  626098659), INT32_C(-1099705903), INT32_C( 1892226777), INT32_C(  183269504)),
      UINT16_C(27232) },
    { simde_mm512_set_epi32(INT32_C( 1660264790), INT32_C( -188014963), INT32_C( 1960568786), INT32_C(  630575470),
                            INT32_C(-1560285386), INT32_C(-1080983958), INT32_C( -186614663), INT32_C(-1365084922),
                            INT32_C( 1687374482), INT32_C( 2091712477), INT32_C( 1770300152), INT32_C( 1222615684),
                            INT32_C(  987382002), INT32_C( -869689297), INT32_C( 1381156346), INT32_C(  352829646)),
      simde_mm512_set_epi32(INT32_C( 1495188549), INT32_C( -188014963), INT32_C( -879412194), INT32_C( 1172150075),
                            INT32_C( 1163780404), INT32_C(-1080983958), INT32_C( -186614663), INT32_C(-1365084922),
                            INT32_C( 1196042729), INT32_C( 2091712477), INT32_C( 1770300152), INT32_C( 1222615684),
                            INT32_C(  987382002), INT32_C( -453542339), INT32_C(-1460537486), INT32_C( 1311735715)),
      UINT16_C(18296) },
    { simde_mm512_set_epi32(INT32_C( -830898164), INT32_C( 2065530031), INT32_C( 1849339474), INT32_C( -161498764),
                            INT32_C(  726295410), INT32_C(-1366062470), INT32_C(  110025501), INT32_C(-2061598845),
                            INT32_C(-1911113344), INT32_C(-2008355607), INT32_C( 1140427951), INT32_C( 1963231912),
                            INT32_C( 1593065931), INT32_C( 1712671682), INT32_C(-2139143015), INT32_C(  330252777)),
      simde_mm512_set_epi32(INT32_C( -830898164), INT32_C(-2132024757), INT32_C( 1102342058), INT32_C( -161498764),
                            INT32_C(-2106128090), INT32_C(  648329890), INT32_C(-1284054768), INT32_C(-2061598845),
                            INT32_C(-1911113344), INT32_C(-2008355607), INT32_C( 1715485148), INT32_C(  155412419),
                            INT32_C( 1273550758), INT32_C( 1712671682), INT32_C(-1857983881), INT32_C( 1633779150)),
      UINT16_C(37316) },
    { simde_mm512_set_epi32(INT32_C( 1956746364), INT32_C( 1930323834), INT32_C(  923874794), INT32_C(  121318212),
                            INT32_C(-1375858452), INT32_C( -462992597), INT32_C( 1495829546), INT32_C(  697040437),
                            INT32_C(  727111035), INT32_C(-2061427382), INT32_C( -815432287), INT32_C(  913775211),
                            INT32_C(-1333809472), INT32_C(  114048073), INT32_C( 1312920985), INT32_C(-1819914035)),
      simde_mm512_set_epi32(INT32_C( 1659376087), INT32_C( 1930323834), INT32_C(  923874794), INT32_C(  121318212),
                            INT32_C(-1375858452), INT32_C(-1480844812), INT32_C(-1803673478), INT32_C(  697040437),
                            INT32_C( 1793922150), INT32_C(  391658500), INT32_C( -815432287), INT32_C(  913775211),
                            INT32_C(-1333809472), INT32_C( -169359358), INT32_C(  140424991), INT32_C(-1819914035)),
      UINT16_C(31033) },
    { simde_mm512_set_epi32(INT32_C( 1334496661), INT32_C(-1765072906), INT32_C(-1980138391), INT32_C(-1150536116),
                            INT32_C( -711226926), INT32_C( 1955166809), INT32_C( 1418224832), INT32_C( 1791996583),
                            INT32_C(-1305868646), INT32_C( -507537618), INT32_C(  272749509), INT32_C(-1826072492),
                            INT32_C( -629068596), INT32_C(-2142583585), INT32_C( 2048200365), INT32_C(-1377550438)),
      simde_mm512_set_epi32(INT32_C( 1334496661), INT32_C(-1765072906), INT32_C( -890751438), INT32_C(-1150536116),
                            INT32_C(  307879329), INT32_C( 1955166809), INT32_C(-1884386825), INT32_C( 1791996583),
                            INT32_C( 1128431085), INT32_C( -507537618), INT32_C(  272749509), INT32_C( 1579228324),
                            INT32_C( 1577134581), INT32_C(-2142583585), INT32_C( 1998674783), INT32_C(-1377550438)),
      UINT16_C(54629) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask16 r = simde_mm512_cmpeq_epi32_mask(test_vec[i].a, test_vec[i].b);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cmpeq_epi32_mask(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__mmask16 r;
  } test_vec[8] = {
    { UINT16_C(15798),
      simde_mm512_set_epi32(INT32_C(-1396783922), INT32_C( 2147469122), INT32_C(  245941047), INT32_C(-1608794680),
                            INT32_C( 1508622706), INT32_C( -820009589), INT32_C(-2056933337), INT32_C( 1399160559),
                            INT32_C( -781782717), INT32_C( -745938688), INT32_C( 1376412239), INT32_C(  358147007),
                            INT32_C(-2141927311), INT32_C(  -13921241), INT32_C(  993767039), INT32_C( 1399506469)),
      simde_mm512_set_epi32(INT32_C(-1396783922), INT32_C( 1246457300), INT32_C( 1479047358), INT32_C(  -46329110),
                            INT32_C( 1508622706), INT32_C( -820009589), INT32_C( 2080173402), INT32_C( 1937046285),
                            INT32_C( 1642337112), INT32_C(  717149471), INT32_C(  913865239), INT32_C(  358147007),
                            INT32_C( 1535690363), INT32_C(  -13921241), INT32_C(  993767039), INT32_C(  520030741)),
      UINT16_C( 3094) },
    { UINT16_C(11237),
      simde_mm512_set_epi32(INT32_C( -503773363), INT32_C(-1842967576), INT32_C(  214407733), INT32_C( 2136243689),
                            INT32_C(  619563347), INT32_C(-2046647578), INT32_C( -882198373), INT32_C( -832110086),
                            INT32_C(  828288790), INT32_C( 1819046419), INT32_C(  292064966), INT32_C( -747926472),
                            INT32_C( -881463995), INT32_C(-1469434386), INT32_C( -207196114), INT32_C(-1865668915)),
      simde_mm512_set_epi32(INT32_C(  261150369), INT32_C(-1842967576), INT32_C(-1148601855), INT32_C( 2136243689),
                            INT32_C(-2013121518), INT32_C(-2046647578), INT32_C(-1816537753), INT32_C( -832110086),
                            INT32_C(  828288790), INT32_C( 1801925150), INT32_C(  292064966), INT32_C(-1755078036),
                            INT32_C(-1190065823), INT32_C(  764421376), INT32_C( -207196114), INT32_C(-1865668915)),
      UINT16_C(  417) },
    { UINT16_C(   61),
      simde_mm512_set_epi32(INT32_C(  219790698), INT32_C(  346516102), INT32_C(-2082495343), INT32_C(  775700745),
                            INT32_C( -630025741), INT32_C( 1914162819), INT32_C( -226095226), INT32_C( -350619769),
                            INT32_C( 1825330894), INT32_C(  -79420124), INT32_C(  939662489), INT32_C(  667606641),
                            INT32_C(-1935137328), INT32_C(  247120548), INT32_C( -894861328), INT32_C(-1690188311)),
      simde_mm512_set_epi32(INT32_C(  400967600), INT32_C(  346516102), INT32_C(-2082495343), INT32_C(  775700745),
                            INT32_C( -630025741), INT32_C( 1914162819), INT32_C(-1771983812), INT32_C( -350619769),
                            INT32_C( 1825330894), INT32_C(-1887033010), INT32_C(  939662489), INT32_C( -440942769),
                            INT32_C(-1935137328), INT32_C(  247120548), INT32_C(-1352163534), INT32_C(-1690188311)),
      UINT16_C(   45) },
    { UINT16_C(40799),
      simde_mm512_set_epi32(INT32_C(-1103261115), INT32_C(-1530092257), INT32_C( -178076517), INT32_C( 1725180044),
                            INT32_C( -619562554), INT32_C(-2028225116), INT32_C( -516465044), INT32_C( -790492311),
                            INT32_C(  770588317), INT32_C(  966229539), INT32_C( 1696981823), INT32_C( 1846986452),
                            INT32_C(  201003704), INT32_C(  -88144749), INT32_C( -459260562), INT32_C(-1462493092)),
      simde_mm512_set_epi32(INT32_C(-1103261115), INT32_C(-1530092257), INT32_C( 1272329003), INT32_C( 1725180044),
                            INT32_C(-1714282865), INT32_C( 1964019033), INT32_C(   -6888948), INT32_C( -794446809),
                            INT32_C(  770588317), INT32_C(  966229539), INT32_C(-1587543669), INT32_C( 1846986452),
                            INT32_C(  201003704), INT32_C(  -88144749), INT32_C(-1549459108), INT32_C(-1462493092)),
      UINT16_C(36957) },
    { UINT16_C(18708),
      simde_mm512_set_epi32(INT32_C( 1654168369), INT32_C(-1358646009), INT32_C(  945188582), INT32_C( 1242452940),
                            INT32_C(-2068238117), INT32_C(  613827224), INT32_C( 1766050173), INT32_C(  788865946),
                            INT32_C( -226150288), INT32_C(   20626714), INT32_C(-1790747056), INT32_C(-1510999017),
                            INT32_C(-2059568770), INT32_C(  525242273), INT32_C(-1970979230), INT32_C( -983788353)),
      simde_mm512_set_epi32(INT32_C(-1802152524), INT32_C(-1358646009), INT32_C( 1400410557), INT32_C( 1242452940),
                            INT32_C(-2068238117), INT32_C(-1745049433), INT32_C(-1272787498), INT32_C(   36641197),
                            INT32_C( -226150288), INT32_C(-2087200149), INT32_C( -530182364), INT32_C(-1510999017),
                            INT32_C(-2082577633), INT32_C(  525242273), INT32_C(-1970979230), INT32_C( -983788353)),
      UINT16_C(18452) },
    { UINT16_C(48938),
      simde_mm512_set_epi32(INT32_C(-1802886705), INT32_C(  505130099), INT32_C( 1294359394), INT32_C(  564426410),
                            INT32_C( -813242663), INT32_C(-1097324530), INT32_C( 1599346411), INT32_C(-1815738445),
                            INT32_C( 2114996332), INT32_C( 2143192037), INT32_C(  342894910), INT32_C( 1933006347),
                            INT32_C(  215936041), INT32_C( 2138148935), INT32_C(-1975112588), INT32_C(-1313889253)),
      simde_mm512_set_epi32(INT32_C( 1272515820), INT32_C(-1571014987), INT32_C( 1294359394), INT32_C(  564426410),
                            INT32_C( -305474417), INT32_C(-2099686495), INT32_C(  217917259), INT32_C( 1770631752),
                            INT32_C( 2114996332), INT32_C( 2143192037), INT32_C(  -26985081), INT32_C(  603877714),
                            INT32_C( 1592556524), INT32_C(  420570241), INT32_C(-1975112588), INT32_C(-1313889253)),
      UINT16_C(12290) },
    { UINT16_C(14127),
      simde_mm512_set_epi32(INT32_C(  452796731), INT32_C( -256668338), INT32_C(-1710549095), INT32_C( 1982965424),
                            INT32_C( 1184306045), INT32_C( -221254467), INT32_C( 1420239721), INT32_C( 2028887361),
                            INT32_C(-1950932361), INT32_C( 1650853943), INT32_C(  239751123), INT32_C( 1018010808),
                            INT32_C( -248946240), INT32_C(  701510715), INT32_C(  824235240), INT32_C( 1829156606)),
      simde_mm512_set_epi32(INT32_C(  452796731), INT32_C( 1031814185), INT32_C(-1710549095), INT32_C(  406415467),
                            INT32_C( 1184306045), INT32_C( -221254467), INT32_C(  419739010), INT32_C( 1708161231),
                            INT32_C(-1950932361), INT32_C( 1650853943), INT32_C(  239751123), INT32_C( 1018010808),
                            INT32_C( -248946240), INT32_C(  701510715), INT32_C(-1571248435), INT32_C( 1829156606)),
      UINT16_C( 9261) },
    { UINT16_C(22801),
      simde_mm512_set_epi32(INT32_C( 1869800572), INT32_C(  184060195), INT32_C(   81710208), INT32_C( -451284065),
                            INT32_C(  397153235), INT32_C(  120564446), INT32_C(-2128920097), INT32_C( 1498011427),
                            INT32_C( -602736654), INT32_C( -931955343), INT32_C(  270436915), INT32_C( -984637478),
                            INT32_C( 2080482721), INT32_C( 1599947836), INT32_C(  374268618), INT32_C(  202341051)),
      simde_mm512_set_epi32(INT32_C( 1869800572), INT32_C(  350721255), INT32_C( 1725621650), INT32_C( 2020045509),
                            INT32_C(  397153235), INT32_C( 2059505832), INT32_C(-2128920097), INT32_C( 1498011427),
                            INT32_C(  884679844), INT32_C( -931955343), INT32_C(-1565261303), INT32_C( -984637478),
                            INT32_C( 1047792745), INT32_C(  969830078), INT32_C(  374268618), INT32_C(  202341051)),
      UINT16_C( 2321) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask16 r = simde_mm512_mask_cmpeq_epi32_mask(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_mmask16(HEDLEY_STATIC_CAST(uint16_t, r), ==, HEDLEY_STATIC_CAST(uint16_t, test_vec[i].r));
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cmpeq_epi64_mask(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__mmask8 r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 1145569124203592220), INT64_C( 8866992319046943109),
                            INT64_C( 1920152028348566704), INT64_C( 5434169962120345100),
                            INT64_C( 2279810443797316081), INT64_C( 8202334326145056493),
                            INT64_C(-3178237508368099649), INT64_C(  691539695110831332)),
      simde_mm512_set_epi64(INT64_C( 1145569124203592220), INT64_C( 7456258988741005047),
                            INT64_C( 1920152028348566704), INT64_C(-5531344059509930758),
                            INT64_C( 2279810443797316081), INT64_C( 4212838215119313762),
                            INT64_C(-3178237508368099649), INT64_C(  805234941373423218)),
      UINT8_C(170) },
    { simde_mm512_set_epi64(INT64_C(-9153030146845322943), INT64_C(-3269802522838954453),
                            INT64_C( 4057416377680772191), INT64_C(-4770808841142629569),
                            INT64_C(-3341852712217874901), INT64_C( 2807150139607841402),
                            INT64_C(-4019938614639236980), INT64_C(-5612844059017155926)),
      simde_mm512_set_epi64(INT64_C(-9153030146845322943), INT64_C(-3269802522838954453),
                            INT64_C( 4057416377680772191), INT64_C( -940603917401247915),
                            INT64_C(-3341852712217874901), INT64_C( 2807150139607841402),
                            INT64_C(-4019938614639236980), INT64_C(-5612844059017155926)),
      UINT8_C(239) },
    { simde_mm512_set_epi64(INT64_C(-6535160640888369935), INT64_C( 4320507985166557084),
                            INT64_C( 5472888856009482484), INT64_C(-3128657528300165720),
                            INT64_C( 7430914891859325397), INT64_C( -863913734109164798),
                            INT64_C(-7794735185814972844), INT64_C( 6163895309000776850)),
      simde_mm512_set_epi64(INT64_C(-4220461743444256802), INT64_C( 2616373937292152846),
                            INT64_C( 5472888856009482484), INT64_C(-3128657528300165720),
                            INT64_C(-4693544589206901195), INT64_C( -863913734109164798),
                            INT64_C(-7794735185814972844), INT64_C(-7914074467628930001)),
      UINT8_C( 54) },
    { simde_mm512_set_epi64(INT64_C(-2366835550617683621), INT64_C(-7526131353484035546),
                            INT64_C(  792273323175818479), INT64_C(-3293855504021481788),
                            INT64_C( 5766970185621377329), INT64_C( 8555682923177627503),
                            INT64_C( 6363802824474944219), INT64_C(-2294667848278645794)),
      simde_mm512_set_epi64(INT64_C(-2869910567061155737), INT64_C(-7526131353484035546),
                            INT64_C(  792273323175818479), INT64_C( 7002007300200137801),
                            INT64_C( 5766970185621377329), INT64_C( 8555682923177627503),
                            INT64_C( 6363802824474944219), INT64_C( 1235305386817146646)),
      UINT8_C(110) },
    { simde_mm512_set_epi64(INT64_C( 1781453983767744725), INT64_C( 8717105342239974860),
                            INT64_C(-6555437070120516632), INT64_C( -945434448901737124),
                            INT64_C( 2467182069214891728), INT64_C( 6380355612148977321),
                            INT64_C(-8948278762783625779), INT64_C( 4377995125356963906)),
      simde_mm512_set_epi64(INT64_C( 1781453983767744725), INT64_C( 8717105342239974860),
                            INT64_C(-1574265126641933862), INT64_C(-3609136820784425910),
                            INT64_C( 8413977304256335681), INT64_C(-4046135395677773903),
                            INT64_C(-8948278762783625779), INT64_C( 4377995125356963906)),
      UINT8_C(195) },
    { simde_mm512_set_epi64(INT64_C( 3163831140256245302), INT64_C(-7476767975543057611),
                            INT64_C( 4783231484571490676), INT64_C( 4356333140933542181),
                            INT64_C(-5734470250608567849), INT64_C( 5428089484060124354),
                            INT64_C( 6915844972668556704), INT64_C(-4128418267655054219)),
      simde_mm512_set_epi64(INT64_C( 3163831140256245302), INT64_C(-7945608864951271413),
                            INT64_C(-3728561425301803734), INT64_C( 6344562138072151085),
                            INT64_C(-5734470250608567849), INT64_C( 5868624435741359252),
                            INT64_C( 6915844972668556704), INT64_C(-4128418267655054219)),
      UINT8_C(139) },
    { simde_mm512_set_epi64(INT64_C(-8545152605640787948), INT64_C(-5234822620280611494),
                            INT64_C(-2932089901585751375), INT64_C( 4017618173912988951),
                            INT64_C(-8696436452927061736), INT64_C( 3602642025812661939),
                            INT64_C( 5777136991119584953), INT64_C(-5473038481952171581)),
      simde_mm512_set_epi64(INT64_C(-8545152605640787948), INT64_C(-7808252369899371159),
                            INT64_C(-2932089901585751375), INT64_C(  -26139149052321087),
                            INT64_C(-6986660196527912755), INT64_C( 3602642025812661939),
                            INT64_C( 5777136991119584953), INT64_C(-5473038481952171581)),
      UINT8_C(167) },
    { simde_mm512_set_epi64(INT64_C( 7780170108497689334), INT64_C(-8001282944915881932),
                            INT64_C(  382835809361431399), INT64_C(-5014881555296189468),
                            INT64_C(-1844642447215154571), INT64_C(-5452282829002750089),
                            INT64_C(-7793611854809744260), INT64_C(  938166230586687295)),
      simde_mm512_set_epi64(INT64_C( 7780170108497689334), INT64_C( 1739290942316187796),
                            INT64_C( 7815402837606564081), INT64_C(-5014881555296189468),
                            INT64_C(-1844642447215154571), INT64_C( -253128228754997390),
                            INT64_C(-7793611854809744260), INT64_C(  938166230586687295)),
      UINT8_C(155) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask8 r = simde_mm512_cmpeq_epi64_mask(test_vec[i].a, test_vec[i].b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cmpeq_epi64_mask(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__mmask8 r;
  } test_vec[8] = {
       { UINT8_C( 90),
      simde_mm512_set_epi64(INT64_C( 7722926897436765530), INT64_C( 7338279138551748064),
                            INT64_C( 8433308126101200079), INT64_C(-4390305748733976547),
                            INT64_C(-1482589068035252753), INT64_C(-5002011091694476743),
                            INT64_C( 5809674310022718254), INT64_C( 7620249298233351482)),
      simde_mm512_set_epi64(INT64_C( 7722926897436765530), INT64_C( 7338279138551748064),
                            INT64_C(-2656726859984743367), INT64_C(-4390305748733976547),
                            INT64_C(-1482589068035252753), INT64_C(-5269390469191050553),
                            INT64_C( 5809674310022718254), INT64_C( 7620249298233351482)),
      UINT8_C( 90) },
    { UINT8_C(178),
      simde_mm512_set_epi64(INT64_C(-8806453660480970182), INT64_C(-2722914847628644365),
                            INT64_C( 2171146127585219679), INT64_C(-7200523266853707115),
                            INT64_C( 8505301695237968355), INT64_C( 6373940775215479358),
                            INT64_C( 8209357864908427195), INT64_C( -995665125730760835)),
      simde_mm512_set_epi64(INT64_C(-8806453660480970182), INT64_C(-2722914847628644365),
                            INT64_C(-1180134256156200317), INT64_C(-7200523266853707115),
                            INT64_C(-1610604796376715795), INT64_C( 5419019224867820225),
                            INT64_C( 8209357864908427195), INT64_C( -995665125730760835)),
      UINT8_C(146) },
    { UINT8_C(171),
      simde_mm512_set_epi64(INT64_C(-6245801519083893310), INT64_C(-7866373458730819532),
                            INT64_C(-5627757407772356197), INT64_C(-2425546480980122794),
                            INT64_C(-8451301604567613199), INT64_C( 1369383717682198649),
                            INT64_C( -532343328754521574), INT64_C(-1062878680437210584)),
      simde_mm512_set_epi64(INT64_C(-9163399881020056955), INT64_C(-2992244142829238392),
                            INT64_C( -213476403626539965), INT64_C(-8591297333400286921),
                            INT64_C(-8451301604567613199), INT64_C(-8139768780594538635),
                            INT64_C(-4714070518405120331), INT64_C(-1062878680437210584)),
      UINT8_C(  9) },
    { UINT8_C( 28),
      simde_mm512_set_epi64(INT64_C( 7845956693704412298), INT64_C(-5781930833336581955),
                            INT64_C( 2851517750261041799), INT64_C(-5814293521236182366),
                            INT64_C( 2292150971239308783), INT64_C( 2594053186857735013),
                            INT64_C( 7307535341641173075), INT64_C(-4427478291595527940)),
      simde_mm512_set_epi64(INT64_C(  536264388241191871), INT64_C(-5781930833336581955),
                            INT64_C( 2851517750261041799), INT64_C( 1349842462394812975),
                            INT64_C( 2292150971239308783), INT64_C( 2594053186857735013),
                            INT64_C( 7307535341641173075), INT64_C(-4427478291595527940)),
      UINT8_C( 12) },
    { UINT8_C(248),
      simde_mm512_set_epi64(INT64_C( 4900816215694077255), INT64_C(-2732029741423656661),
                            INT64_C( 1082977534221618055), INT64_C(-3092044493389993636),
                            INT64_C(-4299277917890019767), INT64_C(-2055775203132417874),
                            INT64_C( -778633101599852237), INT64_C( -563223173848121636)),
      simde_mm512_set_epi64(INT64_C( 7049304296219110648), INT64_C(-2732029741423656661),
                            INT64_C( 7088083428992159722), INT64_C(-3092044493389993636),
                            INT64_C(-4299277917890019767), INT64_C( 4225506809727089751),
                            INT64_C( -778633101599852237), INT64_C( -563223173848121636)),
      UINT8_C( 88) },
    { UINT8_C(171),
      simde_mm512_set_epi64(INT64_C(-1412821155990992029), INT64_C( 4454576651901490962),
                            INT64_C(-7284760734604447652), INT64_C(-7443130466673006479),
                            INT64_C(  320054597637804434), INT64_C(-8860872372305530355),
                            INT64_C(-8428145646879978292), INT64_C(-6547252853189215611)),
      simde_mm512_set_epi64(INT64_C(-1412821155990992029), INT64_C(-2354123670646573707),
                            INT64_C( 4506838144989822528), INT64_C(-7443130466673006479),
                            INT64_C(-5147543239321546686), INT64_C(-8860872372305530355),
                            INT64_C(-8428145646879978292), INT64_C(-6547252853189215611)),
      UINT8_C(131) },
    { UINT8_C( 29),
      simde_mm512_set_epi64(INT64_C( 5675137803130124480), INT64_C( 1211541157654985046),
                            INT64_C( 8724633375562564314), INT64_C(-2760658800846254598),
                            INT64_C(-6714474269646576270), INT64_C( 3484180661422871715),
                            INT64_C( 1469796163712815354), INT64_C(-2336393240308600160)),
      simde_mm512_set_epi64(INT64_C( 5675137803130124480), INT64_C( 1211541157654985046),
                            INT64_C(-8867413355151838495), INT64_C(-8867147959443474315),
                            INT64_C(-6714474269646576270), INT64_C( 3484180661422871715),
                            INT64_C(-7735267815657951749), INT64_C(  413036036281601883)),
      UINT8_C( 12) },
    { UINT8_C(211),
      simde_mm512_set_epi64(INT64_C(-6713502673628263139), INT64_C( 1559753162601267291),
                            INT64_C( 5045660940436454371), INT64_C( 7013290440433503154),
                            INT64_C(-8475145246816690249), INT64_C(-6834826688677600633),
                            INT64_C(-2109099044497919348), INT64_C( 1351143524438105934)),
      simde_mm512_set_epi64(INT64_C( 5625319538109918668), INT64_C( 1559753162601267291),
                            INT64_C( 5045660940436454371), INT64_C(-4654386914804892920),
                            INT64_C( 2407237530895996207), INT64_C(-6834826688677600633),
                            INT64_C( 4684210505965066200), INT64_C( 1351143524438105934)),
      UINT8_C( 65) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask8 r = simde_mm512_mask_cmpeq_epi64_mask(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtepi8_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(   6), INT8_C(  33), INT8_C( 124), INT8_C(-128),
                        INT8_C(  38), INT8_C(  59), INT8_C( -37), INT8_C( 121),
                        INT8_C(  67), INT8_C( 112), INT8_C(  62), INT8_C(-113),
                        INT8_C(-121), INT8_C( -89), INT8_C(  63), INT8_C( -72)),
      simde_mm512_set_epi32(INT32_C(          6), INT32_C(         33), INT32_C(        124), INT32_C(       -128),
                            INT32_C(         38), INT32_C(         59), INT32_C(        -37), INT32_C(        121),
                            INT32_C(         67), INT32_C(        112), INT32_C(         62), INT32_C(       -113),
                            INT32_C(       -121), INT32_C(        -89), INT32_C(         63), INT32_C(        -72)) },
    { simde_mm_set_epi8(INT8_C( -94), INT8_C( 125), INT8_C( 105), INT8_C( -92),
                        INT8_C(-107), INT8_C( -69), INT8_C(  25), INT8_C(-125),
                        INT8_C( -11), INT8_C( -44), INT8_C(  60), INT8_C( -96),
                        INT8_C( 119), INT8_C( -40), INT8_C(  29), INT8_C(   9)),
      simde_mm512_set_epi32(INT32_C(        -94), INT32_C(        125), INT32_C(        105), INT32_C(        -92),
                            INT32_C(       -107), INT32_C(        -69), INT32_C(         25), INT32_C(       -125),
                            INT32_C(        -11), INT32_C(        -44), INT32_C(         60), INT32_C(        -96),
                            INT32_C(        119), INT32_C(        -40), INT32_C(         29), INT32_C(          9)) },
    { simde_mm_set_epi8(INT8_C(  47), INT8_C(  -9), INT8_C(-127), INT8_C(  66),
                        INT8_C( 126), INT8_C( -95), INT8_C(  99), INT8_C( -25),
                        INT8_C( -19), INT8_C(  84), INT8_C( -96), INT8_C( -77),
                        INT8_C( -34), INT8_C(-108), INT8_C( -33), INT8_C(  62)),
      simde_mm512_set_epi32(INT32_C(         47), INT32_C(         -9), INT32_C(       -127), INT32_C(         66),
                            INT32_C(        126), INT32_C(        -95), INT32_C(         99), INT32_C(        -25),
                            INT32_C(        -19), INT32_C(         84), INT32_C(        -96), INT32_C(        -77),
                            INT32_C(        -34), INT32_C(       -108), INT32_C(        -33), INT32_C(         62)) },
    { simde_mm_set_epi8(INT8_C(  93), INT8_C(-106), INT8_C(  61), INT8_C(  64),
                        INT8_C( -98), INT8_C(  31), INT8_C( -75), INT8_C( -23),
                        INT8_C( 113), INT8_C( -94), INT8_C( -70), INT8_C( -73),
                        INT8_C(  66), INT8_C( -84), INT8_C(  92), INT8_C(-113)),
      simde_mm512_set_epi32(INT32_C(         93), INT32_C(       -106), INT32_C(         61), INT32_C(         64),
                            INT32_C(        -98), INT32_C(         31), INT32_C(        -75), INT32_C(        -23),
                            INT32_C(        113), INT32_C(        -94), INT32_C(        -70), INT32_C(        -73),
                            INT32_C(         66), INT32_C(        -84), INT32_C(         92), INT32_C(       -113)) },
    { simde_mm_set_epi8(INT8_C( -92), INT8_C( -17), INT8_C( -42), INT8_C( 103),
                        INT8_C( 111), INT8_C( -23), INT8_C(  14), INT8_C(-122),
                        INT8_C( -61), INT8_C(   1), INT8_C(  16), INT8_C(-124),
                        INT8_C( -62), INT8_C(  59), INT8_C( 106), INT8_C(-105)),
      simde_mm512_set_epi32(INT32_C(        -92), INT32_C(        -17), INT32_C(        -42), INT32_C(        103),
                            INT32_C(        111), INT32_C(        -23), INT32_C(         14), INT32_C(       -122),
                            INT32_C(        -61), INT32_C(          1), INT32_C(         16), INT32_C(       -124),
                            INT32_C(        -62), INT32_C(         59), INT32_C(        106), INT32_C(       -105)) },
    { simde_mm_set_epi8(INT8_C(  71), INT8_C(  82), INT8_C( -83), INT8_C( 118),
                        INT8_C( 127), INT8_C(  -5), INT8_C(  43), INT8_C( -48),
                        INT8_C(  67), INT8_C(-117), INT8_C( -15), INT8_C( 105),
                        INT8_C( -88), INT8_C( 127), INT8_C(  85), INT8_C(-110)),
      simde_mm512_set_epi32(INT32_C(         71), INT32_C(         82), INT32_C(        -83), INT32_C(        118),
                            INT32_C(        127), INT32_C(         -5), INT32_C(         43), INT32_C(        -48),
                            INT32_C(         67), INT32_C(       -117), INT32_C(        -15), INT32_C(        105),
                            INT32_C(        -88), INT32_C(        127), INT32_C(         85), INT32_C(       -110)) },
    { simde_mm_set_epi8(INT8_C(   8), INT8_C(  35), INT8_C( -44), INT8_C(  18),
                        INT8_C(   3), INT8_C(  -7), INT8_C(  34), INT8_C(  98),
                        INT8_C(  43), INT8_C(  97), INT8_C(-127), INT8_C( 121),
                        INT8_C( -47), INT8_C( -94), INT8_C( -74), INT8_C( -59)),
      simde_mm512_set_epi32(INT32_C(          8), INT32_C(         35), INT32_C(        -44), INT32_C(         18),
                            INT32_C(          3), INT32_C(         -7), INT32_C(         34), INT32_C(         98),
                            INT32_C(         43), INT32_C(         97), INT32_C(       -127), INT32_C(        121),
                            INT32_C(        -47), INT32_C(        -94), INT32_C(        -74), INT32_C(        -59)) },
    { simde_mm_set_epi8(INT8_C(  79), INT8_C(  98), INT8_C(  70), INT8_C( -29),
                        INT8_C(  61), INT8_C( 100), INT8_C( -92), INT8_C(  10),
                        INT8_C(-107), INT8_C( -13), INT8_C(  83), INT8_C(-102),
                        INT8_C( -54), INT8_C(  80), INT8_C(  43), INT8_C(  56)),
      simde_mm512_set_epi32(INT32_C(         79), INT32_C(         98), INT32_C(         70), INT32_C(        -29),
                            INT32_C(         61), INT32_C(        100), INT32_C(        -92), INT32_C(         10),
                            INT32_C(       -107), INT32_C(        -13), INT32_C(         83), INT32_C(       -102),
                            INT32_C(        -54), INT32_C(         80), INT32_C(         43), INT32_C(         56)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_cvtepi8_epi32(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtepi8_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -948891544), INT32_C( 1432337912), INT32_C( -157601137), INT32_C(  557886841),
                            INT32_C(  405458672), INT32_C( 1108283966), INT32_C(  671693090), INT32_C( -656229219),
                            INT32_C( 1844651109), INT32_C(  722363475), INT32_C( 1034392721), INT32_C( -352112508),
                            INT32_C( -454251458), INT32_C( -470551525), INT32_C( -884105703), INT32_C( 1305270065)),
      UINT16_C(30779),
      simde_mm_set_epi8(INT8_C(-116), INT8_C(  40), INT8_C(  78), INT8_C(  27),
                        INT8_C( -49), INT8_C(  94), INT8_C(  11), INT8_C(  59),
                        INT8_C(-125), INT8_C(  49), INT8_C(  61), INT8_C(-117),
                        INT8_C( -52), INT8_C(-102), INT8_C(  15), INT8_C(  39)),
      simde_mm512_set_epi32(INT32_C( -948891544), INT32_C(         40), INT32_C(         78), INT32_C(         27),
                            INT32_C(        -49), INT32_C( 1108283966), INT32_C(  671693090), INT32_C( -656229219),
                            INT32_C( 1844651109), INT32_C(  722363475), INT32_C(         61), INT32_C(       -117),
                            INT32_C(        -52), INT32_C( -470551525), INT32_C(         15), INT32_C(         39)) },
    { simde_mm512_set_epi32(INT32_C(-1678459053), INT32_C( -751001872), INT32_C(  948848242), INT32_C(-1255138969),
                            INT32_C(  843494992), INT32_C( 1568180449), INT32_C( 1114219711), INT32_C(  -91183620),
                            INT32_C(  -74671399), INT32_C( -726320383), INT32_C(  339500370), INT32_C(  976271619),
                            INT32_C( -404720858), INT32_C(-1856521809), INT32_C(  491307301), INT32_C(  992626281)),
      UINT16_C(28152),
      simde_mm_set_epi8(INT8_C(  95), INT8_C( -75), INT8_C(  81), INT8_C(  24),
                        INT8_C(  84), INT8_C(   1), INT8_C(  61), INT8_C( -53),
                        INT8_C( -76), INT8_C(  20), INT8_C( -59), INT8_C(  -3),
                        INT8_C(  37), INT8_C(-113), INT8_C(-116), INT8_C(   6)),
      simde_mm512_set_epi32(INT32_C(-1678459053), INT32_C(        -75), INT32_C(         81), INT32_C(-1255138969),
                            INT32_C(         84), INT32_C(          1), INT32_C( 1114219711), INT32_C(        -53),
                            INT32_C(        -76), INT32_C(         20), INT32_C(        -59), INT32_C(         -3),
                            INT32_C(         37), INT32_C(-1856521809), INT32_C(  491307301), INT32_C(  992626281)) },
    { simde_mm512_set_epi32(INT32_C( 1482305192), INT32_C( 1996708987), INT32_C(  723535956), INT32_C(-1670036300),
                            INT32_C( -363335449), INT32_C(  -47930977), INT32_C( -403738658), INT32_C( 1182057510),
                            INT32_C(  831714400), INT32_C(-1744656678), INT32_C(-1981557460), INT32_C( -724476623),
                            INT32_C(  939131668), INT32_C(  353771966), INT32_C( 1625236055), INT32_C(-2058304981)),
      UINT16_C(35313),
      simde_mm_set_epi8(INT8_C(  81), INT8_C(  92), INT8_C( -16), INT8_C( -77),
                        INT8_C( -10), INT8_C(  31), INT8_C(  48), INT8_C(  46),
                        INT8_C( 122), INT8_C( -46), INT8_C( 100), INT8_C( -42),
                        INT8_C( -95), INT8_C( -43), INT8_C(  68), INT8_C(-104)),
      simde_mm512_set_epi32(INT32_C(         81), INT32_C( 1996708987), INT32_C(  723535956), INT32_C(-1670036300),
                            INT32_C(        -10), INT32_C(  -47930977), INT32_C( -403738658), INT32_C(         46),
                            INT32_C(        122), INT32_C(        -46), INT32_C(        100), INT32_C(        -42),
                            INT32_C(  939131668), INT32_C(  353771966), INT32_C( 1625236055), INT32_C(       -104)) },
    { simde_mm512_set_epi32(INT32_C( 1830936869), INT32_C(-1741742044), INT32_C(-1737557360), INT32_C( -273240771),
                            INT32_C(-1961147279), INT32_C( -229097469), INT32_C( -763894872), INT32_C( 1529380840),
                            INT32_C( -665264196), INT32_C(  639643484), INT32_C(  240439150), INT32_C( 1588063308),
                            INT32_C(  586997217), INT32_C( 1018879786), INT32_C( 1848501586), INT32_C( -979270594)),
      UINT16_C(33265),
      simde_mm_set_epi8(INT8_C(-123), INT8_C(  90), INT8_C(  -2), INT8_C(  97),
                        INT8_C(  68), INT8_C(  78), INT8_C(  77), INT8_C(  55),
                        INT8_C(-105), INT8_C(-116), INT8_C(  28), INT8_C( -62),
                        INT8_C( -88), INT8_C(  -8), INT8_C( 115), INT8_C(-111)),
      simde_mm512_set_epi32(INT32_C(       -123), INT32_C(-1741742044), INT32_C(-1737557360), INT32_C( -273240771),
                            INT32_C(-1961147279), INT32_C( -229097469), INT32_C( -763894872), INT32_C(         55),
                            INT32_C(       -105), INT32_C(       -116), INT32_C(         28), INT32_C(        -62),
                            INT32_C(  586997217), INT32_C( 1018879786), INT32_C( 1848501586), INT32_C(       -111)) },
    { simde_mm512_set_epi32(INT32_C( 1041391151), INT32_C( 1093398186), INT32_C(-1592148298), INT32_C( 1164412431),
                            INT32_C(-1767544990), INT32_C(  389847058), INT32_C(  388979962), INT32_C(  625503476),
                            INT32_C(-1034617064), INT32_C(  622175081), INT32_C(  228405175), INT32_C( 1183542116),
                            INT32_C( 1822156655), INT32_C(  232229273), INT32_C(-1158516102), INT32_C( 1768636180)),
      UINT16_C(48999),
      simde_mm_set_epi8(INT8_C(  97), INT8_C( -30), INT8_C( 124), INT8_C(  25),
                        INT8_C( -97), INT8_C(-128), INT8_C(  84), INT8_C( -30),
                        INT8_C( -42), INT8_C(  87), INT8_C(  15), INT8_C(-126),
                        INT8_C(  19), INT8_C( 104), INT8_C(  47), INT8_C( -47)),
      simde_mm512_set_epi32(INT32_C(         97), INT32_C( 1093398186), INT32_C(        124), INT32_C(         25),
                            INT32_C(        -97), INT32_C(       -128), INT32_C(         84), INT32_C(        -30),
                            INT32_C(-1034617064), INT32_C(         87), INT32_C(         15), INT32_C( 1183542116),
                            INT32_C( 1822156655), INT32_C(        104), INT32_C(         47), INT32_C(        -47)) },
    { simde_mm512_set_epi32(INT32_C(   95296887), INT32_C(-1039650207), INT32_C(  851568188), INT32_C(-1463556566),
                            INT32_C( -376745735), INT32_C(  437204996), INT32_C( -892096867), INT32_C(-1877695621),
                            INT32_C(  434885689), INT32_C(-1210422431), INT32_C(  933725665), INT32_C( -669851252),
                            INT32_C(  149185919), INT32_C(-1625981030), INT32_C( -573589938), INT32_C(  178183647)),
      UINT16_C(16017),
      simde_mm_set_epi8(INT8_C(-121), INT8_C( -49), INT8_C( 114), INT8_C(-112),
                        INT8_C( -43), INT8_C( -25), INT8_C( 119), INT8_C( 103),
                        INT8_C(  40), INT8_C(-112), INT8_C(  -8), INT8_C( -89),
                        INT8_C(  73), INT8_C( -54), INT8_C(  83), INT8_C( -40)),
      simde_mm512_set_epi32(INT32_C(   95296887), INT32_C(-1039650207), INT32_C(        114), INT32_C(       -112),
                            INT32_C(        -43), INT32_C(        -25), INT32_C(        119), INT32_C(-1877695621),
                            INT32_C(         40), INT32_C(-1210422431), INT32_C(  933725665), INT32_C(        -89),
                            INT32_C(  149185919), INT32_C(-1625981030), INT32_C( -573589938), INT32_C(        -40)) },
    { simde_mm512_set_epi32(INT32_C( -739285659), INT32_C(   81885987), INT32_C(  847528238), INT32_C(-2089319861),
                            INT32_C(-1629855246), INT32_C( 2048738507), INT32_C( 1195062920), INT32_C(  399815228),
                            INT32_C(   83479215), INT32_C(  685330423), INT32_C( -319423423), INT32_C(  771854533),
                            INT32_C(  -58748336), INT32_C(-1621219518), INT32_C(-1185941810), INT32_C(-2120224468)),
      UINT16_C(27674),
      simde_mm_set_epi8(INT8_C(  70), INT8_C( -19), INT8_C( -25), INT8_C( -59),
                        INT8_C(  64), INT8_C( 110), INT8_C( -46), INT8_C( -94),
                        INT8_C(  69), INT8_C(   2), INT8_C(  44), INT8_C( -87),
                        INT8_C( -72), INT8_C(-120), INT8_C( -53), INT8_C( -22)),
      simde_mm512_set_epi32(INT32_C( -739285659), INT32_C(        -19), INT32_C(        -25), INT32_C(-2089319861),
                            INT32_C(         64), INT32_C(        110), INT32_C( 1195062920), INT32_C(  399815228),
                            INT32_C(   83479215), INT32_C(  685330423), INT32_C( -319423423), INT32_C(        -87),
                            INT32_C(        -72), INT32_C(-1621219518), INT32_C(        -53), INT32_C(-2120224468)) },
    { simde_mm512_set_epi32(INT32_C(  812929568), INT32_C( -558329566), INT32_C( -820382393), INT32_C(-2033989853),
                            INT32_C(-1691272889), INT32_C(-1250227005), INT32_C(  926032561), INT32_C( -492473735),
                            INT32_C(  131989417), INT32_C( -707025949), INT32_C(  705079402), INT32_C( 2061631189),
                            INT32_C(  860537414), INT32_C(-1872373827), INT32_C(  166627023), INT32_C( 1034543202)),
      UINT16_C(10417),
      simde_mm_set_epi8(INT8_C(  61), INT8_C( 124), INT8_C(  65), INT8_C(-124),
                        INT8_C( -83), INT8_C(  86), INT8_C(  53), INT8_C(   9),
                        INT8_C( -15), INT8_C(-117), INT8_C( -19), INT8_C(  84),
                        INT8_C(-107), INT8_C( 116), INT8_C(  83), INT8_C( -43)),
      simde_mm512_set_epi32(INT32_C(  812929568), INT32_C( -558329566), INT32_C(         65), INT32_C(-2033989853),
                            INT32_C(        -83), INT32_C(-1250227005), INT32_C(  926032561), INT32_C( -492473735),
                            INT32_C(        -15), INT32_C( -707025949), INT32_C(        -19), INT32_C(         84),
                            INT32_C(  860537414), INT32_C(-1872373827), INT32_C(  166627023), INT32_C(        -43)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_cvtepi8_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtepi8_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(55089),
      simde_mm_set_epi8(INT8_C(  61), INT8_C( -89), INT8_C(-108), INT8_C(-111),
                        INT8_C( -21), INT8_C(   3), INT8_C(  48), INT8_C(-124),
                        INT8_C( -28), INT8_C( -20), INT8_C( -84), INT8_C(  62),
                        INT8_C( -29), INT8_C( -13), INT8_C( -12), INT8_C(  27)),
      simde_mm512_set_epi32(INT32_C(         61), INT32_C(        -89), INT32_C(          0), INT32_C(       -111),
                            INT32_C(          0), INT32_C(          3), INT32_C(         48), INT32_C(       -124),
                            INT32_C(          0), INT32_C(          0), INT32_C(        -84), INT32_C(         62),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(         27)) },
    { UINT16_C(25683),
      simde_mm_set_epi8(INT8_C(  24), INT8_C(  42), INT8_C( -50), INT8_C( -16),
                        INT8_C(  66), INT8_C(  15), INT8_C(  18), INT8_C(  62),
                        INT8_C(  40), INT8_C(   9), INT8_C(  57), INT8_C(  34),
                        INT8_C( -40), INT8_C( -30), INT8_C( -68), INT8_C( -99)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(         42), INT32_C(        -50), INT32_C(          0),
                            INT32_C(          0), INT32_C(         15), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          9), INT32_C(          0), INT32_C(         34),
                            INT32_C(          0), INT32_C(          0), INT32_C(        -68), INT32_C(        -99)) },
    { UINT16_C(44409),
      simde_mm_set_epi8(INT8_C( 121), INT8_C(  62), INT8_C( -84), INT8_C(  58),
                        INT8_C( -34), INT8_C(  31), INT8_C( 120), INT8_C(  59),
                        INT8_C( -57), INT8_C( 113), INT8_C(  16), INT8_C( 104),
                        INT8_C(  85), INT8_C(  95), INT8_C( -67), INT8_C(  -8)),
      simde_mm512_set_epi32(INT32_C(        121), INT32_C(          0), INT32_C(        -84), INT32_C(          0),
                            INT32_C(        -34), INT32_C(         31), INT32_C(          0), INT32_C(         59),
                            INT32_C(          0), INT32_C(        113), INT32_C(         16), INT32_C(        104),
                            INT32_C(         85), INT32_C(          0), INT32_C(          0), INT32_C(         -8)) },
    { UINT16_C( 3879),
      simde_mm_set_epi8(INT8_C(  29), INT8_C(  72), INT8_C( -63), INT8_C(  37),
                        INT8_C(  59), INT8_C(  42), INT8_C(  70), INT8_C( 105),
                        INT8_C(-116), INT8_C(  40), INT8_C(  78), INT8_C(  27),
                        INT8_C( -49), INT8_C(  94), INT8_C(  11), INT8_C(  59)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(         59), INT32_C(         42), INT32_C(         70), INT32_C(        105),
                            INT32_C(          0), INT32_C(          0), INT32_C(         78), INT32_C(          0),
                            INT32_C(          0), INT32_C(         94), INT32_C(         11), INT32_C(         59)) },
    { UINT16_C(47535),
      simde_mm_set_epi8(INT8_C(  -5), INT8_C(-116), INT8_C(-102), INT8_C( -39),
                        INT8_C( -44), INT8_C( -75), INT8_C(  59), INT8_C(   1),
                        INT8_C(  20), INT8_C(  60), INT8_C(  93), INT8_C(  82),
                        INT8_C(  58), INT8_C(  48), INT8_C( -71), INT8_C(   3)),
      simde_mm512_set_epi32(INT32_C(         -5), INT32_C(          0), INT32_C(       -102), INT32_C(        -39),
                            INT32_C(        -44), INT32_C(          0), INT32_C(          0), INT32_C(          1),
                            INT32_C(         20), INT32_C(          0), INT32_C(         93), INT32_C(          0),
                            INT32_C(         58), INT32_C(         48), INT32_C(        -71), INT32_C(          3)) },
    { UINT16_C(42492),
      simde_mm_set_epi8(INT8_C(  56), INT8_C(-114), INT8_C(  70), INT8_C( 114),
                        INT8_C( -75), INT8_C(  48), INT8_C(  25), INT8_C( 103),
                        INT8_C(  50), INT8_C(  70), INT8_C( -74), INT8_C(  80),
                        INT8_C(  93), INT8_C( 120), INT8_C(-120), INT8_C( -31)),
      simde_mm512_set_epi32(INT32_C(         56), INT32_C(          0), INT32_C(         70), INT32_C(          0),
                            INT32_C(          0), INT32_C(         48), INT32_C(          0), INT32_C(        103),
                            INT32_C(         50), INT32_C(         70), INT32_C(        -74), INT32_C(         80),
                            INT32_C(         93), INT32_C(        120), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(40688),
      simde_mm_set_epi8(INT8_C( -76), INT8_C(  20), INT8_C( -59), INT8_C(  -3),
                        INT8_C(  37), INT8_C(-113), INT8_C(-116), INT8_C(   6),
                        INT8_C(-128), INT8_C(   9), INT8_C( -96), INT8_C(  -4),
                        INT8_C( -38), INT8_C(-121), INT8_C( 109), INT8_C(  -8)),
      simde_mm512_set_epi32(INT32_C(        -76), INT32_C(          0), INT32_C(          0), INT32_C(         -3),
                            INT32_C(         37), INT32_C(       -113), INT32_C(       -116), INT32_C(          0),
                            INT32_C(       -128), INT32_C(          9), INT32_C(        -96), INT32_C(         -4),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(15819),
      simde_mm_set_epi8(INT8_C(  55), INT8_C(  -6), INT8_C(   3), INT8_C(  20),
                        INT8_C(  21), INT8_C(  22), INT8_C(  33), INT8_C( -66),
                        INT8_C(  96), INT8_C( -33), INT8_C(  34), INT8_C(  87),
                        INT8_C(-123), INT8_C(  80), INT8_C( -62), INT8_C(  43)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          3), INT32_C(         20),
                            INT32_C(         21), INT32_C(         22), INT32_C(          0), INT32_C(        -66),
                            INT32_C(         96), INT32_C(        -33), INT32_C(          0), INT32_C(          0),
                            INT32_C(       -123), INT32_C(          0), INT32_C(        -62), INT32_C(         43)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_cvtepi8_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtepi8_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(  40), INT8_C( -85), INT8_C(  94), INT8_C(  35),
                        INT8_C( -54), INT8_C( -71), INT8_C(-106), INT8_C(-127),
                        INT8_C(  11), INT8_C( 105), INT8_C(  37), INT8_C(-105),
                        INT8_C(   6), INT8_C( -65), INT8_C(  17), INT8_C(  29)),
      simde_mm512_set_epi64(INT64_C(                  11), INT64_C(                 105),
                            INT64_C(                  37), INT64_C(                -105),
                            INT64_C(                   6), INT64_C(                 -65),
                            INT64_C(                  17), INT64_C(                  29)) },
    { simde_mm_set_epi8(INT8_C(  65), INT8_C( -98), INT8_C( -80), INT8_C( -71),
                        INT8_C( -32), INT8_C(  61), INT8_C( -91), INT8_C(   2),
                        INT8_C(  62), INT8_C(  86), INT8_C( -39), INT8_C( -20),
                        INT8_C(  65), INT8_C( -87), INT8_C( 116), INT8_C(-112)),
      simde_mm512_set_epi64(INT64_C(                  62), INT64_C(                  86),
                            INT64_C(                 -39), INT64_C(                 -20),
                            INT64_C(                  65), INT64_C(                 -87),
                            INT64_C(                 116), INT64_C(                -112)) },
    { simde_mm_set_epi8(INT8_C( -35), INT8_C(  91), INT8_C(  95), INT8_C( -91),
                        INT8_C(-120), INT8_C( -72), INT8_C(   6), INT8_C( 127),
                        INT8_C( -64), INT8_C(  43), INT8_C( -93), INT8_C(   8),
                        INT8_C( 105), INT8_C( -16), INT8_C(  39), INT8_C( 125)),
      simde_mm512_set_epi64(INT64_C(                 -64), INT64_C(                  43),
                            INT64_C(                 -93), INT64_C(                   8),
                            INT64_C(                 105), INT64_C(                 -16),
                            INT64_C(                  39), INT64_C(                 125)) },
    { simde_mm_set_epi8(INT8_C( 108), INT8_C( 105), INT8_C(  98), INT8_C( -57),
                        INT8_C( -42), INT8_C( -18), INT8_C( -55), INT8_C(  -1),
                        INT8_C( -97), INT8_C( -26), INT8_C( -21), INT8_C(-119),
                        INT8_C(  95), INT8_C(  83), INT8_C( -95), INT8_C(  86)),
      simde_mm512_set_epi64(INT64_C(                 -97), INT64_C(                 -26),
                            INT64_C(                 -21), INT64_C(                -119),
                            INT64_C(                  95), INT64_C(                  83),
                            INT64_C(                 -95), INT64_C(                  86)) },
    { simde_mm_set_epi8(INT8_C(  63), INT8_C(-119), INT8_C(  65), INT8_C( 108),
                        INT8_C( -17), INT8_C( -16), INT8_C(  -4), INT8_C(  16),
                        INT8_C(-117), INT8_C( -62), INT8_C( -96), INT8_C(   5),
                        INT8_C( 116), INT8_C( -58), INT8_C( -56), INT8_C(-121)),
      simde_mm512_set_epi64(INT64_C(                -117), INT64_C(                 -62),
                            INT64_C(                 -96), INT64_C(                   5),
                            INT64_C(                 116), INT64_C(                 -58),
                            INT64_C(                 -56), INT64_C(                -121)) },
    { simde_mm_set_epi8(INT8_C( -69), INT8_C(  98), INT8_C(  52), INT8_C( -73),
                        INT8_C(  71), INT8_C(  44), INT8_C( -92), INT8_C(  54),
                        INT8_C( 126), INT8_C(  88), INT8_C(  32), INT8_C(-122),
                        INT8_C( -69), INT8_C( -55), INT8_C(-123), INT8_C(  79)),
      simde_mm512_set_epi64(INT64_C(                 126), INT64_C(                  88),
                            INT64_C(                  32), INT64_C(                -122),
                            INT64_C(                 -69), INT64_C(                 -55),
                            INT64_C(                -123), INT64_C(                  79)) },
    { simde_mm_set_epi8(INT8_C(-110), INT8_C( -17), INT8_C( -48), INT8_C( -64),
                        INT8_C(   9), INT8_C(  96), INT8_C( 113), INT8_C( -55),
                        INT8_C( -45), INT8_C(  -9), INT8_C( 104), INT8_C( -61),
                        INT8_C( 127), INT8_C( 121), INT8_C(  70), INT8_C( -22)),
      simde_mm512_set_epi64(INT64_C(                 -45), INT64_C(                  -9),
                            INT64_C(                 104), INT64_C(                 -61),
                            INT64_C(                 127), INT64_C(                 121),
                            INT64_C(                  70), INT64_C(                 -22)) },
    { simde_mm_set_epi8(INT8_C(  68), INT8_C( -24), INT8_C(  90), INT8_C( -28),
                        INT8_C(  55), INT8_C( -48), INT8_C(  13), INT8_C(  95),
                        INT8_C(  10), INT8_C( -72), INT8_C( 109), INT8_C( -27),
                        INT8_C(  94), INT8_C( 121), INT8_C(  33), INT8_C(  52)),
      simde_mm512_set_epi64(INT64_C(                  10), INT64_C(                 -72),
                            INT64_C(                 109), INT64_C(                 -27),
                            INT64_C(                  94), INT64_C(                 121),
                            INT64_C(                  33), INT64_C(                  52)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_cvtepi8_epi64(test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtepi8_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-4075458147498607112), INT64_C( -676891728669528711),
                            INT64_C( 1741431737227874878), INT64_C( 2884899858137922717),
                            INT64_C( 7922716186407494739), INT64_C( 4442682911858307204),
                            INT64_C(-1950995152445901797), INT64_C(-3797205079286819023)),
      UINT8_C( 59),
      simde_mm_set_epi8(INT8_C( -49), INT8_C(  94), INT8_C(  11), INT8_C(  59),
                        INT8_C(-125), INT8_C(  49), INT8_C(  61), INT8_C(-117),
                        INT8_C( -52), INT8_C(-102), INT8_C(  15), INT8_C(  39),
                        INT8_C( 121), INT8_C(  62), INT8_C( -84), INT8_C(  58)),
      simde_mm512_set_epi64(INT64_C(-4075458147498607112), INT64_C( -676891728669528711),
                            INT64_C(                  15), INT64_C(                  39),
                            INT64_C(                 121), INT64_C( 4442682911858307204),
                            INT64_C(                 -84), INT64_C(                  58)) },
    { simde_mm512_set_epi64(INT64_C(-3225528478525929870), INT64_C(-5390780822946662832),
                            INT64_C( 6735283743795815615), INT64_C( -391630661610595623),
                            INT64_C(-3119522291063693998), INT64_C( 4193054679508218662),
                            INT64_C(-7973700453474451163), INT64_C( 4263297416396557851)),
      UINT8_C( 83),
      simde_mm_set_epi8(INT8_C( -76), INT8_C(  20), INT8_C( -59), INT8_C(  -3),
                        INT8_C(  37), INT8_C(-113), INT8_C(-116), INT8_C(   6),
                        INT8_C(-128), INT8_C(   9), INT8_C( -96), INT8_C(  -4),
                        INT8_C( -38), INT8_C(-121), INT8_C( 109), INT8_C(  -8)),
      simde_mm512_set_epi64(INT64_C(-3225528478525929870), INT64_C(                   9),
                            INT64_C( 6735283743795815615), INT64_C(                  -4),
                            INT64_C(-3119522291063693998), INT64_C( 4193054679508218662),
                            INT64_C(                 109), INT64_C(                  -8)) },
    { simde_mm512_set_epi64(INT64_C( 3107563271125025972), INT64_C(-1560513866685439585),
                            INT64_C(-1734044331058871258), INT64_C( 3572186150162573018),
                            INT64_C(-8510724482274337487), INT64_C( 4033539801051701694),
                            INT64_C( 6980335706741719595), INT64_C( 6896507569318215115)),
      UINT8_C(123),
      simde_mm_set_epi8(INT8_C( -95), INT8_C( -43), INT8_C(  68), INT8_C(-104),
                        INT8_C(  48), INT8_C( 123), INT8_C( 124), INT8_C(   4),
                        INT8_C(  -4), INT8_C(  21), INT8_C(-119), INT8_C( -15),
                        INT8_C(  88), INT8_C(  90), INT8_C(  46), INT8_C( -88)),
      simde_mm512_set_epi64(INT64_C( 3107563271125025972), INT64_C(                  21),
                            INT64_C(                -119), INT64_C(                 -15),
                            INT64_C(                  88), INT64_C( 4033539801051701694),
                            INT64_C(                  46), INT64_C(                 -88)) },
    { simde_mm512_set_epi64(INT64_C(-1173560173045005199), INT64_C( -983966133420301400),
                            INT64_C( 6568640694558711740), INT64_C( 2747247845119938414),
                            INT64_C( 6820679972424572385), INT64_C( 4376055361273980242),
                            INT64_C(-4205935173799448397), INT64_C( -711797239914011434)),
      UINT8_C(144),
      simde_mm_set_epi8(INT8_C( -38), INT8_C( 103), INT8_C( 119), INT8_C(  -1),
                        INT8_C(  70), INT8_C(  39), INT8_C(-127), INT8_C( -15),
                        INT8_C( 109), INT8_C(  33), INT8_C( -31), INT8_C(  37),
                        INT8_C(-104), INT8_C(  47), INT8_C(  32), INT8_C(  36)),
      simde_mm512_set_epi64(INT64_C(                 109), INT64_C( -983966133420301400),
                            INT64_C( 6568640694558711740), INT64_C(                  37),
                            INT64_C( 6820679972424572385), INT64_C( 4376055361273980242),
                            INT64_C(-4205935173799448397), INT64_C( -711797239914011434)) },
    { simde_mm512_set_epi64(INT64_C(-7591547925868799982), INT64_C( 1670656216214826228),
                            INT64_C(-4443646453141363863), INT64_C(  980992758045698916),
                            INT64_C( 7826103241645984153), INT64_C(-4975788768210764012),
                            INT64_C(-8837471625049912009), INT64_C(-7526609254858853487)),
      UINT8_C( 15),
      simde_mm_set_epi8(INT8_C(-111), INT8_C( 121), INT8_C( -65), INT8_C( 103),
                        INT8_C(  62), INT8_C(  18), INT8_C(  94), INT8_C(  47),
                        INT8_C(  65), INT8_C(  43), INT8_C( -18), INT8_C( -86),
                        INT8_C( -95), INT8_C(  25), INT8_C( -66), INT8_C( -74)),
      simde_mm512_set_epi64(INT64_C(-7591547925868799982), INT64_C( 1670656216214826228),
                            INT64_C(-4443646453141363863), INT64_C(  980992758045698916),
                            INT64_C(                 -95), INT64_C(                  25),
                            INT64_C(                 -66), INT64_C(                 -74)) },
    { simde_mm512_set_epi64(INT64_C( 1877781162870681245), INT64_C(-8064641283602525127),
                            INT64_C(-5198724754556090911), INT64_C(-2876989220375468673),
                            INT64_C(-6983535344045017522), INT64_C(  765292938189241369),
                            INT64_C(-6953464491424673918), INT64_C( 1398420262929112913)),
      UINT8_C(249),
      simde_mm_set_epi8(INT8_C(   5), INT8_C( -82), INT8_C(  29), INT8_C( 119),
                        INT8_C( -62), INT8_C(   8), INT8_C(  50), INT8_C(  97),
                        INT8_C(  50), INT8_C( -63), INT8_C( -26), INT8_C(  60),
                        INT8_C( -88), INT8_C( -61), INT8_C( -26), INT8_C(  42)),
      simde_mm512_set_epi64(INT64_C(                  50), INT64_C(                 -63),
                            INT64_C(                 -26), INT64_C(                  60),
                            INT64_C(                 -88), INT64_C(  765292938189241369),
                            INT64_C(-6953464491424673918), INT64_C(                  42)) },
    { simde_mm512_set_epi64(INT64_C( 5132756158462079548), INT64_C(  358540499006083063),
                            INT64_C(-1371913154589519675), INT64_C( -252322179140671678),
                            INT64_C(-5093581286734302932), INT64_C(-8660577592020600985),
                            INT64_C( 2923109555544675288), INT64_C( 9023002677975531153)),
      UINT8_C(203),
      simde_mm_set_epi8(INT8_C(   4), INT8_C( -31), INT8_C( 123), INT8_C(  35),
                        INT8_C(  50), INT8_C(-124), INT8_C(  65), INT8_C(  46),
                        INT8_C(-125), INT8_C( 119), INT8_C(-126), INT8_C(  75),
                        INT8_C( -98), INT8_C( -38), INT8_C(  97), INT8_C( -14)),
      simde_mm512_set_epi64(INT64_C(                -125), INT64_C(                 119),
                            INT64_C(-1371913154589519675), INT64_C( -252322179140671678),
                            INT64_C(                 -98), INT64_C(-8660577592020600985),
                            INT64_C(                  97), INT64_C(                 -14)) },
    { simde_mm512_set_epi64(INT64_C(-2115158585831981143), INT64_C(-3036653327673284502),
                            INT64_C( 8854638534029132358), INT64_C(-8041784352684734769),
                            INT64_C( 4443329220079118277), INT64_C( 4642879860249865385),
                            INT64_C(-5149641967024239455), INT64_C(-8739960639361948315)),
      UINT8_C(177),
      simde_mm_set_epi8(INT8_C( -49), INT8_C(  25), INT8_C( -11), INT8_C(  71),
                        INT8_C(-122), INT8_C( -61), INT8_C( -57), INT8_C(  35),
                        INT8_C(-101), INT8_C(  49), INT8_C(  57), INT8_C(  71),
                        INT8_C( -75), INT8_C( 123), INT8_C(  12), INT8_C( -61)),
      simde_mm512_set_epi64(INT64_C(                -101), INT64_C(-3036653327673284502),
                            INT64_C(                  57), INT64_C(                  71),
                            INT64_C( 4443329220079118277), INT64_C( 4642879860249865385),
                            INT64_C(-5149641967024239455), INT64_C(                 -61)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_cvtepi8_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtepi8_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 49),
      simde_mm_set_epi8(INT8_C( -21), INT8_C(   3), INT8_C(  48), INT8_C(-124),
                        INT8_C( -28), INT8_C( -20), INT8_C( -84), INT8_C(  62),
                        INT8_C( -29), INT8_C( -13), INT8_C( -12), INT8_C(  27),
                        INT8_C( -53), INT8_C(  77), INT8_C( -98), INT8_C(  25)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                 -12), INT64_C(                  27),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                  25)) },
    { UINT8_C(145),
      simde_mm_set_epi8(INT8_C(  40), INT8_C(   9), INT8_C(  57), INT8_C(  34),
                        INT8_C( -40), INT8_C( -30), INT8_C( -68), INT8_C( -99),
                        INT8_C( 109), INT8_C( -13), INT8_C(  36), INT8_C( 101),
                        INT8_C(  43), INT8_C(  14), INT8_C( 100), INT8_C(  83)),
      simde_mm512_set_epi64(INT64_C(                 109), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                 101),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                  83)) },
    { UINT8_C( 62),
      simde_mm_set_epi8(INT8_C(  85), INT8_C(  95), INT8_C( -67), INT8_C(  -8),
                        INT8_C( -10), INT8_C(-101), INT8_C(  50), INT8_C(-113),
                        INT8_C(  33), INT8_C(  64), INT8_C( -83), INT8_C( 121),
                        INT8_C(  24), INT8_C(  42), INT8_C( -50), INT8_C( -16)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                 -83), INT64_C(                 121),
                            INT64_C(                  24), INT64_C(                  42),
                            INT64_C(                 -50), INT64_C(                   0)) },
    { UINT8_C(104),
      simde_mm_set_epi8(INT8_C(-125), INT8_C(  49), INT8_C(  61), INT8_C(-117),
                        INT8_C( -52), INT8_C(-102), INT8_C(  15), INT8_C(  39),
                        INT8_C( 121), INT8_C(  62), INT8_C( -84), INT8_C(  58),
                        INT8_C( -34), INT8_C(  31), INT8_C( 120), INT8_C(  59)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                  62),
                            INT64_C(                 -84), INT64_C(                   0),
                            INT64_C(                 -34), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C( 59),
      simde_mm_set_epi8(INT8_C(-111), INT8_C(  87), INT8_C( -71), INT8_C( -81),
                        INT8_C(  29), INT8_C(  72), INT8_C( -63), INT8_C(  37),
                        INT8_C(  59), INT8_C(  42), INT8_C(  70), INT8_C( 105),
                        INT8_C(-116), INT8_C(  40), INT8_C(  78), INT8_C(  27)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                  70), INT64_C(                 105),
                            INT64_C(                -116), INT64_C(                   0),
                            INT64_C(                  78), INT64_C(                  27)) },
    { UINT8_C( 38),
      simde_mm_set_epi8(INT8_C(  -5), INT8_C(-116), INT8_C(-102), INT8_C( -39),
                        INT8_C( -44), INT8_C( -75), INT8_C(  59), INT8_C(   1),
                        INT8_C(  20), INT8_C(  60), INT8_C(  93), INT8_C(  82),
                        INT8_C(  58), INT8_C(  48), INT8_C( -71), INT8_C(   3)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                  93), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                  48),
                            INT64_C(                 -71), INT64_C(                   0)) },
    { UINT8_C(252),
      simde_mm_set_epi8(INT8_C( -75), INT8_C(  48), INT8_C(  25), INT8_C( 103),
                        INT8_C(  50), INT8_C(  70), INT8_C( -74), INT8_C(  80),
                        INT8_C(  93), INT8_C( 120), INT8_C(-120), INT8_C( -31),
                        INT8_C(  66), INT8_C( 105), INT8_C( -92), INT8_C( -65)),
      simde_mm512_set_epi64(INT64_C(                  93), INT64_C(                 120),
                            INT64_C(                -120), INT64_C(                 -31),
                            INT64_C(                  66), INT64_C(                 105),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(114),
      simde_mm_set_epi8(INT8_C(-128), INT8_C(   9), INT8_C( -96), INT8_C(  -4),
                        INT8_C( -38), INT8_C(-121), INT8_C( 109), INT8_C(  -8),
                        INT8_C(-101), INT8_C( -12), INT8_C( -65), INT8_C(  83),
                        INT8_C( -45), INT8_C(  60), INT8_C( -98), INT8_C( -16)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                 -12),
                            INT64_C(                 -65), INT64_C(                  83),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                 -98), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_cvtepi8_epi64(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtepi32_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(         -2), INT32_C(         -4), INT32_C( -120451969), INT32_C(      14509),
                            INT32_C(      -1510), INT32_C(      98804), INT32_C(       1802), INT32_C(     -32352),
                            INT32_C(      14540), INT32_C(         -2), INT32_C(        222), INT32_C(        152),
                            INT32_C(     -48720), INT32_C(     250746), INT32_C(         15), INT32_C(         -2)),
      simde_mm_set_epi8(INT8_C(  -2), INT8_C(  -4), INT8_C( 127), INT8_C( -83),
                        INT8_C(  26), INT8_C( -12), INT8_C(  10), INT8_C( -96),
                        INT8_C( -52), INT8_C(  -2), INT8_C( -34), INT8_C(-104),
                        INT8_C( -80), INT8_C( 122), INT8_C(  15), INT8_C(  -2)) },
    { simde_mm512_set_epi32(INT32_C(   -2537924), INT32_C(          0), INT32_C(       3842), INT32_C(    -439330),
                            INT32_C(      39001), INT32_C(         -1), INT32_C(   32480192), INT32_C(          0),
                            INT32_C(          4), INT32_C(  -11661865), INT32_C(          0), INT32_C(          2),
                            INT32_C(      63322), INT32_C(        -33), INT32_C(      14448), INT32_C(          2)),
      simde_mm_set_epi8(INT8_C(  60), INT8_C(   0), INT8_C(   2), INT8_C( -34),
                        INT8_C(  89), INT8_C(  -1), INT8_C( -64), INT8_C(   0),
                        INT8_C(   4), INT8_C( -41), INT8_C(   0), INT8_C(   2),
                        INT8_C(  90), INT8_C( -33), INT8_C( 112), INT8_C(   2)) },
    { simde_mm512_set_epi32(INT32_C(        -18), INT32_C(   -2011647), INT32_C(   -7768922), INT32_C( -921783558),
                            INT32_C(   -1941821), INT32_C(          0), INT32_C(    -647690), INT32_C(      -5119),
                            INT32_C(         -1), INT32_C(        343), INT32_C(  113610714), INT32_C(          3),
                            INT32_C(      38353), INT32_C(        246), INT32_C(  -10559231), INT32_C(       8543)),
      simde_mm_set_epi8(INT8_C( -18), INT8_C(   1), INT8_C( -90), INT8_C(  -6),
                        INT8_C( -61), INT8_C(   0), INT8_C( -10), INT8_C(   1),
                        INT8_C(  -1), INT8_C(  87), INT8_C( -38), INT8_C(   3),
                        INT8_C( -47), INT8_C( -10), INT8_C(   1), INT8_C(  95)) },
    { simde_mm512_set_epi32(INT32_C( -177879544), INT32_C(         -4), INT32_C(         -1), INT32_C(      -1874),
                            INT32_C(         -4), INT32_C( -469598096), INT32_C(      -1647), INT32_C(     129553),
                            INT32_C(    9710669), INT32_C(          3), INT32_C(    1473856), INT32_C(  134714256),
                            INT32_C(   70527996), INT32_C(    2339642), INT32_C(     148218), INT32_C(  -32767248)),
      simde_mm_set_epi8(INT8_C(   8), INT8_C(  -4), INT8_C(  -1), INT8_C( -82),
                        INT8_C(  -4), INT8_C( 112), INT8_C(-111), INT8_C(  17),
                        INT8_C(  77), INT8_C(   3), INT8_C(  64), INT8_C(-112),
                        INT8_C(  -4), INT8_C(  58), INT8_C(  -6), INT8_C( -16)) },
    { simde_mm512_set_epi32(INT32_C(   -1418204), INT32_C(    -122943), INT32_C(     799821), INT32_C(        -51),
                            INT32_C(     207931), INT32_C(  -11778782), INT32_C(      66993), INT32_C(     -15812),
                            INT32_C(   55345677), INT32_C( -194873886), INT32_C(      -3955), INT32_C(        -22),
                            INT32_C(       1761), INT32_C(        751), INT32_C(         19), INT32_C(         -4)),
      simde_mm_set_epi8(INT8_C(  36), INT8_C( -63), INT8_C(  77), INT8_C( -51),
                        INT8_C(  59), INT8_C(  34), INT8_C( -79), INT8_C(  60),
                        INT8_C(  13), INT8_C( -30), INT8_C(-115), INT8_C( -22),
                        INT8_C( -31), INT8_C( -17), INT8_C(  19), INT8_C(  -4)) },
    { simde_mm512_set_epi32(INT32_C(        -17), INT32_C(        -26), INT32_C(     854285), INT32_C(         51),
                            INT32_C(  -60746537), INT32_C(    3687234), INT32_C(   52848365), INT32_C(   26958727),
                            INT32_C(          2), INT32_C(        104), INT32_C(    4725058), INT32_C(     -56297),
                            INT32_C(       5336), INT32_C(     443041), INT32_C(        -35), INT32_C(     229612)),
      simde_mm_set_epi8(INT8_C( -17), INT8_C( -26), INT8_C(  13), INT8_C(  51),
                        INT8_C( -41), INT8_C(  66), INT8_C( -19), INT8_C(-121),
                        INT8_C(   2), INT8_C( 104), INT8_C(  66), INT8_C(  23),
                        INT8_C( -40), INT8_C( -95), INT8_C( -35), INT8_C( -20)) },
    { simde_mm512_set_epi32(INT32_C(        -27), INT32_C(  127397292), INT32_C(   29325489), INT32_C(        691),
                            INT32_C(       -978), INT32_C(    -559751), INT32_C(   -3037707), INT32_C(     189833),
                            INT32_C(         11), INT32_C(   -4085970), INT32_C(      -3499), INT32_C(     -16323),
                            INT32_C(     732682), INT32_C(     108115), INT32_C(   29565452), INT32_C( -145574324)),
      simde_mm_set_epi8(INT8_C( -27), INT8_C( -84), INT8_C( -79), INT8_C( -77),
                        INT8_C(  46), INT8_C( 121), INT8_C( -11), INT8_C(-119),
                        INT8_C(  11), INT8_C(  46), INT8_C(  85), INT8_C(  61),
                        INT8_C(  10), INT8_C(  83), INT8_C(  12), INT8_C(  76)) },
    { simde_mm512_set_epi32(INT32_C(        -14), INT32_C(    6208981), INT32_C(  133763173), INT32_C(  -30227251),
                            INT32_C(  -17898651), INT32_C( -197203605), INT32_C(         -4), INT32_C(         13),
                            INT32_C(   -1312564), INT32_C(         -3), INT32_C(    5632807), INT32_C(       2549),
                            INT32_C(         -3), INT32_C(      -2772), INT32_C(      -1504), INT32_C(          1)),
      simde_mm_set_epi8(INT8_C( -14), INT8_C( -43), INT8_C( 101), INT8_C( -51),
                        INT8_C( 101), INT8_C( 107), INT8_C(  -4), INT8_C(  13),
                        INT8_C( -52), INT8_C(  -3), INT8_C(  39), INT8_C( -11),
                        INT8_C(  -3), INT8_C(  44), INT8_C(  32), INT8_C(   1)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtepi32_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtepi32_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(-115), INT8_C(  38), INT8_C( -52), INT8_C( -72),
                        INT8_C( -15), INT8_C( -73), INT8_C(  99), INT8_C(  21),
                        INT8_C(  43), INT8_C( -43), INT8_C( -79), INT8_C( -14),
                        INT8_C( -64), INT8_C(  13), INT8_C( -52), INT8_C(  21)),
      UINT16_C(59249),
      simde_mm512_set_epi32(INT32_C( -633053809), INT32_C( 1045231202), INT32_C( 1582941645), INT32_C(  118733775),
                            INT32_C( -376070904), INT32_C(-1346059078), INT32_C(-1785775774), INT32_C( 1344364263),
                            INT32_C(-1284394530), INT32_C(  845981746), INT32_C( 1995633378), INT32_C(   86515524),
                            INT32_C( 1684133172), INT32_C(  134426622), INT32_C( 1324231526), INT32_C(-1677357037)),
      simde_mm_set_epi8(INT8_C(-113), INT8_C(  98), INT8_C( -51), INT8_C( -72),
                        INT8_C( -15), INT8_C( -70), INT8_C(  98), INT8_C( -25),
                        INT8_C(  43), INT8_C(  50), INT8_C( -30), INT8_C(  68),
                        INT8_C( -64), INT8_C(  13), INT8_C( -52), INT8_C(  19)) },
    { simde_mm_set_epi8(INT8_C(  11), INT8_C( -83), INT8_C(-118), INT8_C( -45),
                        INT8_C( 127), INT8_C( -97), INT8_C(  58), INT8_C( -28),
                        INT8_C( 119), INT8_C( 125), INT8_C( 108), INT8_C( 121),
                        INT8_C( -56), INT8_C(  31), INT8_C( 101), INT8_C(  30)),
      UINT16_C(15281),
      simde_mm512_set_epi32(INT32_C(-1863151852), INT32_C( 1841626176), INT32_C(  128694508), INT32_C(-1952059086),
                            INT32_C(   35559102), INT32_C(  170089620), INT32_C(  592079099), INT32_C(  986767838),
                            INT32_C(-2078459331), INT32_C(-1259484799), INT32_C( -745194505), INT32_C( -806534005),
                            INT32_C(  178571728), INT32_C(  724303028), INT32_C(-1558698351), INT32_C( 1087124862)),
      simde_mm_set_epi8(INT8_C(  11), INT8_C( -83), INT8_C( -20), INT8_C(  50),
                        INT8_C( -66), INT8_C( -97), INT8_C(  -5), INT8_C( -34),
                        INT8_C(  61), INT8_C( 125), INT8_C(  -9), INT8_C(-117),
                        INT8_C( -56), INT8_C(  31), INT8_C( 101), INT8_C( 126)) },
    { simde_mm_set_epi8(INT8_C( -37), INT8_C(  17), INT8_C( -46), INT8_C(  62),
                        INT8_C(-122), INT8_C( -48), INT8_C(  95), INT8_C(  94),
                        INT8_C(-120), INT8_C(  70), INT8_C(  86), INT8_C( -34),
                        INT8_C( 103), INT8_C(-121), INT8_C(-125), INT8_C( -37)),
      UINT16_C(60508),
      simde_mm512_set_epi32(INT32_C(  785975787), INT32_C(-1640844720), INT32_C(  801487669), INT32_C( -230277812),
                            INT32_C( 1785836931), INT32_C(-1252913343), INT32_C( -593649464), INT32_C(  826028222),
                            INT32_C( 1348935036), INT32_C( 1700017282), INT32_C(  975497373), INT32_C(  680935328),
                            INT32_C(  926666753), INT32_C(-1896106060), INT32_C(  828740304), INT32_C(  854544945)),
      simde_mm_set_epi8(INT8_C( -21), INT8_C(  80), INT8_C(  53), INT8_C(  62),
                        INT8_C(-125), INT8_C(  65), INT8_C(  95), INT8_C(  94),
                        INT8_C(-120), INT8_C(-126), INT8_C(  86), INT8_C( -96),
                        INT8_C(   1), INT8_C( -76), INT8_C(-125), INT8_C( -37)) },
    { simde_mm_set_epi8(INT8_C(  10), INT8_C(  72), INT8_C(  14), INT8_C(  71),
                        INT8_C(  38), INT8_C( 127), INT8_C(  12), INT8_C( 109),
                        INT8_C(  37), INT8_C(   0), INT8_C(-100), INT8_C( 103),
                        INT8_C(  94), INT8_C(-124), INT8_C(  98), INT8_C( 121)),
      UINT16_C(27626),
      simde_mm512_set_epi32(INT32_C(-2129288013), INT32_C( -640424446), INT32_C( -475582201), INT32_C(-1084971703),
                            INT32_C(-1025326436), INT32_C(  476614498), INT32_C(-2026838640), INT32_C( 1547453534),
                            INT32_C(  608354747), INT32_C( -419005782), INT32_C( -987109050), INT32_C(-1602281775),
                            INT32_C(-1047324466), INT32_C( 1650730666), INT32_C(-1634562157), INT32_C( 1609673966)),
      simde_mm_set_epi8(INT8_C(  10), INT8_C(   2), INT8_C(   7), INT8_C(  71),
                        INT8_C(-100), INT8_C( 127), INT8_C(-112), INT8_C(  94),
                        INT8_C( -69), INT8_C( -86), INT8_C(  70), INT8_C( 103),
                        INT8_C( -50), INT8_C(-124), INT8_C(-109), INT8_C( 121)) },
    { simde_mm_set_epi8(INT8_C(-121), INT8_C(  49), INT8_C( -49), INT8_C(-115),
                        INT8_C(  46), INT8_C(  78), INT8_C( -24), INT8_C( 108),
                        INT8_C(   6), INT8_C( -70), INT8_C( -66), INT8_C( -81),
                        INT8_C( -83), INT8_C(-127), INT8_C( -30), INT8_C( -93)),
      UINT16_C(65313),
      simde_mm512_set_epi32(INT32_C( -866938802), INT32_C(-1684051229), INT32_C( 1750567699), INT32_C( -890678355),
                            INT32_C( 1248284911), INT32_C(  685959243), INT32_C(-1971795404), INT32_C( -380779864),
                            INT32_C(  768594520), INT32_C( 1934584708), INT32_C( 1771009293), INT32_C(  977292129),
                            INT32_C(  116806954), INT32_C( 1249795808), INT32_C( 1848907251), INT32_C(-1141949480)),
      simde_mm_set_epi8(INT8_C(  78), INT8_C( -29), INT8_C(  19), INT8_C( -83),
                        INT8_C( -17), INT8_C(  75), INT8_C(  52), INT8_C( -88),
                        INT8_C(   6), INT8_C( -70), INT8_C(  13), INT8_C( -81),
                        INT8_C( -83), INT8_C(-127), INT8_C( -30), INT8_C( -40)) },
    { simde_mm_set_epi8(INT8_C( -22), INT8_C( 123), INT8_C(-115), INT8_C(-106),
                        INT8_C( -17), INT8_C(  46), INT8_C(-125), INT8_C( -54),
                        INT8_C( 109), INT8_C( -45), INT8_C(  71), INT8_C(  92),
                        INT8_C(   4), INT8_C(  62), INT8_C( 125), INT8_C(  87)),
      UINT16_C(61443),
      simde_mm512_set_epi32(INT32_C(-1334588139), INT32_C(-1182979414), INT32_C(  652824334), INT32_C(-1886834903),
                            INT32_C(-2077275827), INT32_C(-2124149498), INT32_C( 1711430883), INT32_C( 1063836903),
                            INT32_C(  196832116), INT32_C( 1645984078), INT32_C( -706180494), INT32_C(-2114013608),
                            INT32_C( -409538379), INT32_C( -238667028), INT32_C( -203770260), INT32_C(  668929896)),
      simde_mm_set_epi8(INT8_C(  21), INT8_C( -86), INT8_C(  14), INT8_C(  41),
                        INT8_C( -17), INT8_C(  46), INT8_C(-125), INT8_C( -54),
                        INT8_C( 109), INT8_C( -45), INT8_C(  71), INT8_C(  92),
                        INT8_C(   4), INT8_C(  62), INT8_C( 108), INT8_C( 104)) },
    { simde_mm_set_epi8(INT8_C( -86), INT8_C(  70), INT8_C(  52), INT8_C( 118),
                        INT8_C(   9), INT8_C(  39), INT8_C( 114), INT8_C( -16),
                        INT8_C( 117), INT8_C(  23), INT8_C(  56), INT8_C( 116),
                        INT8_C(-103), INT8_C( -20), INT8_C( -54), INT8_C( -40)),
      UINT16_C(17932),
      simde_mm512_set_epi32(INT32_C(-1632449466), INT32_C( -201574937), INT32_C( -969151873), INT32_C( 1717523001),
                            INT32_C(  328447052), INT32_C( -807008823), INT32_C(-1137627317), INT32_C(  257941943),
                            INT32_C( -575132239), INT32_C(-1243310715), INT32_C( 1780403357), INT32_C(-1221616339),
                            INT32_C(-1111710057), INT32_C( -345601306), INT32_C( 1878290192), INT32_C( 1400816063)),
      simde_mm_set_epi8(INT8_C( -86), INT8_C( -25), INT8_C(  52), INT8_C( 118),
                        INT8_C(   9), INT8_C( -55), INT8_C(  75), INT8_C( -16),
                        INT8_C( 117), INT8_C(  23), INT8_C(  56), INT8_C( 116),
                        INT8_C(-105), INT8_C( -26), INT8_C( -54), INT8_C( -40)) },
    { simde_mm_set_epi8(INT8_C(-111), INT8_C(-107), INT8_C( -18), INT8_C( -11),
                        INT8_C( -19), INT8_C(  72), INT8_C( -61), INT8_C( -36),
                        INT8_C( 112), INT8_C( -89), INT8_C(  40), INT8_C( -85),
                        INT8_C( -85), INT8_C( -24), INT8_C(   1), INT8_C( 118)),
      UINT16_C( 5127),
      simde_mm512_set_epi32(INT32_C( 1198543284), INT32_C( -953548547), INT32_C(-1654400435), INT32_C(-1584606457),
                            INT32_C(-1412039433), INT32_C(  978591462), INT32_C(  932668344), INT32_C(    4811681),
                            INT32_C( 1470607276), INT32_C( -870107132), INT32_C(-1548582602), INT32_C(-1629317483),
                            INT32_C( 1240536520), INT32_C( 1712711286), INT32_C(  -33733649), INT32_C( -824647896)),
      simde_mm_set_epi8(INT8_C(-111), INT8_C(-107), INT8_C( -18), INT8_C(   7),
                        INT8_C( -19), INT8_C( -26), INT8_C( -61), INT8_C( -36),
                        INT8_C( 112), INT8_C( -89), INT8_C(  40), INT8_C( -85),
                        INT8_C( -85), INT8_C( 118), INT8_C( -17), INT8_C(  40)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtepi32_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtepi32_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT16_C(30170),
      simde_mm512_set_epi32(INT32_C(-1490536007), INT32_C( 1066535041), INT32_C( -550061645), INT32_C(-1270000019),
                            INT32_C( 1151371971), INT32_C(-1698501418), INT32_C(  316796751), INT32_C( 1981586156),
                            INT32_C(-1946547448), INT32_C( 1308377999), INT32_C( -844439772), INT32_C(   66448910),
                            INT32_C( -858584361), INT32_C(  937214420), INT32_C(-1606259759), INT32_C(-1809752141)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(-127), INT8_C( -77), INT8_C( 109),
                        INT8_C(   0), INT8_C( -42), INT8_C(   0), INT8_C( -20),
                        INT8_C(   8), INT8_C(-113), INT8_C(   0), INT8_C(  14),
                        INT8_C( -41), INT8_C(   0), INT8_C( -47), INT8_C(   0)) },
    { UINT16_C(62388),
      simde_mm512_set_epi32(INT32_C( -211999323), INT32_C(-1844444789), INT32_C(-1795946402), INT32_C( 1448672032),
                            INT32_C( -934434797), INT32_C(-2083148260), INT32_C(-1634027507), INT32_C(-1634157891),
                            INT32_C(  293018136), INT32_C( -844643479), INT32_C(-1645027193), INT32_C(-1269262821),
                            INT32_C(  471304631), INT32_C(-1979771737), INT32_C( 1002540732), INT32_C( -224819409)),
      simde_mm_set_epi8(INT8_C( -91), INT8_C(-117), INT8_C(  94), INT8_C(  32),
                        INT8_C(   0), INT8_C(   0), INT8_C(  13), INT8_C( -67),
                        INT8_C(  24), INT8_C(   0), INT8_C(-121), INT8_C(  27),
                        INT8_C(   0), INT8_C( -89), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(62171),
      simde_mm512_set_epi32(INT32_C(   52699594), INT32_C(  488650932), INT32_C( 1760732830), INT32_C( 2055656791),
                            INT32_C(  901653669), INT32_C(-1110809377), INT32_C( -921351138), INT32_C(-1112269588),
                            INT32_C( -380695850), INT32_C(-1321417051), INT32_C( -488851354), INT32_C( 1504345267),
                            INT32_C( -892751803), INT32_C( 1370474856), INT32_C(-1824202240), INT32_C( 1071081537)),
      simde_mm_set_epi8(INT8_C( -54), INT8_C( -76), INT8_C( -98), INT8_C(  87),
                        INT8_C(   0), INT8_C(   0), INT8_C(  30), INT8_C(   0),
                        INT8_C( -42), INT8_C( -91), INT8_C(   0), INT8_C( -77),
                        INT8_C(  69), INT8_C(   0), INT8_C(   0), INT8_C(  65)) },
    { UINT16_C(25207),
      simde_mm512_set_epi32(INT32_C(-1432829131), INT32_C(  749437314), INT32_C( 2054705796), INT32_C( -391985181),
                            INT32_C( 1552532172), INT32_C(-1395437371), INT32_C( 2050638296), INT32_C(  899300612),
                            INT32_C( -439013837), INT32_C( 1227975380), INT32_C( 1163662343), INT32_C( 1510671711),
                            INT32_C(  284422502), INT32_C( -115052064), INT32_C(-1371764485), INT32_C( 1680575393)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(-126), INT8_C(-124), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C( -40), INT8_C(   0),
                        INT8_C(   0), INT8_C( -44), INT8_C(   7), INT8_C(  95),
                        INT8_C(   0), INT8_C( -32), INT8_C(  -5), INT8_C( -95)) },
    { UINT16_C(24449),
      simde_mm512_set_epi32(INT32_C( 1421206868), INT32_C( 2098367545), INT32_C(-1824932601), INT32_C( -645288155),
                            INT32_C(-1709048480), INT32_C( -754994889), INT32_C( 1910543063), INT32_C( 1017349806),
                            INT32_C(-1269450519), INT32_C( -918081531), INT32_C( -891095185), INT32_C(  463720262),
                            INT32_C(-1217679150), INT32_C(   15502381), INT32_C( 1778046630), INT32_C( 1937097954)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(  57), INT8_C(   0), INT8_C(  37),
                        INT8_C(  96), INT8_C(  55), INT8_C( -41), INT8_C( -82),
                        INT8_C( -23), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -30)) },
    { UINT16_C(29387),
      simde_mm512_set_epi32(INT32_C( 1987003656), INT32_C( 1660112052), INT32_C(  352562636), INT32_C(-1628125955),
                            INT32_C( -306345670), INT32_C(-1188776876), INT32_C( 1380454403), INT32_C(-1671272217),
                            INT32_C(   81322544), INT32_C(-1759657246), INT32_C(  996228951), INT32_C(-1018298216),
                            INT32_C(  368633517), INT32_C(   88316290), INT32_C( 1098979766), INT32_C( -528467658)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C( -76), INT8_C( -52), INT8_C(  -3),
                        INT8_C(   0), INT8_C(   0), INT8_C(   3), INT8_C(   0),
                        INT8_C(  48), INT8_C( -30), INT8_C(   0), INT8_C(   0),
                        INT8_C( -83), INT8_C(   0), INT8_C( -74), INT8_C(  54)) },
    { UINT16_C( 4173),
      simde_mm512_set_epi32(INT32_C( -494557689), INT32_C( 1662324959), INT32_C( 1347000947), INT32_C( -162261965),
                            INT32_C(-1860865781), INT32_C(-1254000245), INT32_C(  -64373227), INT32_C( -128416414),
                            INT32_C(  679962295), INT32_C( -855169599), INT32_C(-1117560264), INT32_C(-1737150948),
                            INT32_C( -703356518), INT32_C(-1187698017), INT32_C(  283483033), INT32_C(  528640345)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  51),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( -63), INT8_C(   0), INT8_C(   0),
                        INT8_C(-102), INT8_C( -97), INT8_C(   0), INT8_C(  89)) },
    { UINT16_C(42283),
      simde_mm512_set_epi32(INT32_C( -988318647), INT32_C(-1013856804), INT32_C( -265791763), INT32_C(  863100255),
                            INT32_C( 1939230730), INT32_C( -592540076), INT32_C( -177529906), INT32_C(  212115003),
                            INT32_C( -240452355), INT32_C(-2076064537), INT32_C(-1929718803), INT32_C( -295261474),
                            INT32_C( -375082091), INT32_C(-1476895591), INT32_C(  282049218), INT32_C( 1800963183)),
      simde_mm_set_epi8(INT8_C(  73), INT8_C(   0), INT8_C( -19), INT8_C(   0),
                        INT8_C(   0), INT8_C(  84), INT8_C(   0), INT8_C(  59),
                        INT8_C(   0), INT8_C(   0), INT8_C( -19), INT8_C(   0),
                        INT8_C(-107), INT8_C(   0), INT8_C( -62), INT8_C( 111)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtepi32_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtepi32_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-1332198505), INT32_C(  940882866), INT32_C(-1991926811), INT32_C(  366788767),
                            INT32_C(-1143068695), INT32_C(-1032919886), INT32_C(-2128096317), INT32_C( -268982741),
                            INT32_C(-2068381358), INT32_C(  616587314), INT32_C( -881395152), INT32_C( -231234031),
                            INT32_C( -227030663), INT32_C(-2069337736), INT32_C( -657801206), INT32_C( -481592217)),
      simde_mm256_set_epi16(INT16_C( 17303), INT16_C(-17486), INT16_C(-25627), INT16_C(-16225),
                            INT16_C( 10217), INT16_C( -6990), INT16_C(-11325), INT16_C(-22997),
                            INT16_C(   338), INT16_C( 24626), INT16_C( -1488), INT16_C(-23023),
                            INT16_C(-13959), INT16_C( 27000), INT16_C(-16374), INT16_C( 31847)) },
    { simde_mm512_set_epi32(INT32_C(-2105440453), INT32_C( -920655637), INT32_C(-1308443450), INT32_C( 1530912720),
                            INT32_C(  327449420), INT32_C(-1587559441), INT32_C( 1636779725), INT32_C( 1337676952),
                            INT32_C(  437538013), INT32_C( -547064950), INT32_C(-1989988035), INT32_C( 1891070193),
                            INT32_C(  268816178), INT32_C(-2114746214), INT32_C(-1830004348), INT32_C( -326386089)),
      simde_mm256_set_epi16(INT16_C(-30917), INT16_C( -5909), INT16_C(-17210), INT16_C( -8240),
                            INT16_C( 31564), INT16_C(-15377), INT16_C( 18125), INT16_C( 21656),
                            INT16_C( 19677), INT16_C( 29578), INT16_C( 12605), INT16_C( 28913),
                            INT16_C(-12494), INT16_C(-30566), INT16_C( 22916), INT16_C(-16809)) },
    { simde_mm512_set_epi32(INT32_C(   65778579), INT32_C(-1223552012), INT32_C( 2126519569), INT32_C( -313585586),
                            INT32_C(  176014611), INT32_C( -579188851), INT32_C( 1255563377), INT32_C( 2103012622),
                            INT32_C(  632398769), INT32_C( -371050877), INT32_C( 1353038446), INT32_C(  825386533),
                            INT32_C(-1874850751), INT32_C(-1002389366), INT32_C(  190340527), INT32_C(-1797342038)),
      simde_mm256_set_epi16(INT16_C(-19565), INT16_C(  5108), INT16_C(  7441), INT16_C(  4174),
                            INT16_C(-15085), INT16_C( 18317), INT16_C( 24689), INT16_C( 27918),
                            INT16_C(-23631), INT16_C( 13955), INT16_C(-17810), INT16_C( 26149),
                            INT16_C(  3137), INT16_C(-16246), INT16_C( 23983), INT16_C(-17238)) },
    { simde_mm512_set_epi32(INT32_C( 1791900902), INT32_C(-1811007005), INT32_C(-1697094173), INT32_C(  380010296),
                            INT32_C(-1383570112), INT32_C( 1657316314), INT32_C(  607692203), INT32_C( 1114614249),
                            INT32_C(-2038136105), INT32_C( -866829407), INT32_C(  438125461), INT32_C(-1743999841),
                            INT32_C( 1594785360), INT32_C( 1281130339), INT32_C( -934532021), INT32_C(  860713686)),
      simde_mm256_set_epi16(INT16_C( 15590), INT16_C( 14819), INT16_C( 26083), INT16_C( 32568),
                            INT16_C( 25920), INT16_C(-23590), INT16_C(-23125), INT16_C(-22039),
                            INT16_C(-32041), INT16_C( 15265), INT16_C( 17301), INT16_C(-21345),
                            INT16_C( 32336), INT16_C( 32611), INT16_C( 11339), INT16_C( 29398)) },
    { simde_mm512_set_epi32(INT32_C( 2051345376), INT32_C( 1984352737), INT32_C( -556704403), INT32_C(-1837075026),
                            INT32_C(-1368462428), INT32_C(-1008395565), INT32_C( -364003348), INT32_C( 1194517560),
                            INT32_C( -724894707), INT32_C( 1417788368), INT32_C(  813255612), INT32_C(-1030021276),
                            INT32_C(  619302066), INT32_C( -403277800), INT32_C(-1431725263), INT32_C( -983683884)),
      simde_mm256_set_epi16(INT16_C(  3040), INT16_C(-11807), INT16_C( 23917), INT16_C( 30126),
                            INT16_C( -5212), INT16_C(  6867), INT16_C(-16404), INT16_C( -7112),
                            INT16_C( -1011), INT16_C(-17456), INT16_C( 19388), INT16_C(  8036),
                            INT16_C(-13134), INT16_C( 30744), INT16_C(-25807), INT16_C( 11476)) },
    { simde_mm512_set_epi32(INT32_C(-1122798844), INT32_C(-1762786870), INT32_C( -798997093), INT32_C(-1700624092),
                            INT32_C(  274386357), INT32_C(-1124458887), INT32_C(-1997652654), INT32_C( -871223206),
                            INT32_C(-1283482560), INT32_C(-1963416707), INT32_C( 1626011520), INT32_C(-1795267365),
                            INT32_C( 1814240343), INT32_C(  563972607), INT32_C(-1989715395), INT32_C( 1806536177)),
      simde_mm256_set_epi16(INT16_C( 29444), INT16_C(   458), INT16_C( 17819), INT16_C(-30428),
                            INT16_C(-12875), INT16_C(  7801), INT16_C( 15698), INT16_C( 12378),
                            INT16_C(-25536), INT16_C(-23683), INT16_C( -2176), INT16_C( 25819),
                            INT16_C(  7255), INT16_C(-30209), INT16_C( 23101), INT16_C(-29199)) },
    { simde_mm512_set_epi32(INT32_C(  630973837), INT32_C(  113536673), INT32_C( -518601343), INT32_C(-1286957458),
                            INT32_C( 1161343316), INT32_C(-1166902161), INT32_C( -814068470), INT32_C(  979467752),
                            INT32_C(-1209279266), INT32_C(  299486195), INT32_C(-1121450290), INT32_C(-1207879975),
                            INT32_C(-1157080602), INT32_C( 1406950891), INT32_C(-1364074288), INT32_C(  597374251)),
      simde_mm256_set_epi16(INT16_C( -6771), INT16_C( 28321), INT16_C(-14975), INT16_C(-27026),
                            INT16_C(-20140), INT16_C( 31855), INT16_C( 19722), INT16_C( 32232),
                            INT16_C( -8994), INT16_C(-13325), INT16_C(  1742), INT16_C( 14041),
                            INT16_C( 23014), INT16_C( 24043), INT16_C( -7984), INT16_C( 13611)) },
    { simde_mm512_set_epi32(INT32_C(-1559893444), INT32_C(-1654530433), INT32_C(  517502824), INT32_C( -829443487),
                            INT32_C( 1010937703), INT32_C(  846882690), INT32_C( -969138942), INT32_C( -844620432),
                            INT32_C(-1194851807), INT32_C( -113524179), INT32_C(-1017791786), INT32_C(  -30012988),
                            INT32_C(-1341538338), INT32_C( -898929280), INT32_C( 1716675353), INT32_C( -910502857)),
      simde_mm256_set_epi16(INT16_C( -5572), INT16_C( -8577), INT16_C( 30568), INT16_C(-19871),
                            INT16_C(-20633), INT16_C( 26498), INT16_C(  7426), INT16_C(  7536),
                            INT16_C(   545), INT16_C(-15827), INT16_C(-17706), INT16_C(  2500),
                            INT16_C(-16418), INT16_C( 28032), INT16_C( 25369), INT16_C(-11209)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_cvtepi32_epi16(test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtepi32_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m256i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(-24546), INT16_C( -2793), INT16_C( 30898), INT16_C(-16727),
                            INT16_C(  8944), INT16_C(-21339), INT16_C( 15242), INT16_C( 19130),
                            INT16_C(  6628), INT16_C(   311), INT16_C( 10740), INT16_C(-29133),
                            INT16_C(-29367), INT16_C(  5394), INT16_C( -3185), INT16_C(  6093)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C( -192240909), INT32_C( 1754400453), INT32_C( 2092363074), INT32_C( -216902206),
                            INT32_C( 1710225239), INT32_C(-1888469291), INT32_C( 1486952867), INT32_C( -799781754),
                            INT32_C(-1362821820), INT32_C(-1788704779), INT32_C(-1168048073), INT32_C(  865982914),
                            INT32_C(-1536735064), INT32_C( 1323733132), INT32_C(-1675904112), INT32_C( -962936689)),
      simde_mm256_set_epi16(INT16_C(-24546), INT16_C( -2793), INT16_C( 30898), INT16_C(-16727),
                            INT16_C(  8944), INT16_C(-21339), INT16_C( 15242), INT16_C( 19130),
                            INT16_C(  6628), INT16_C(   311), INT16_C( 10740), INT16_C(-29133),
                            INT16_C(-29367), INT16_C(  5394), INT16_C( -3185), INT16_C(  6093)) },
    { simde_mm256_set_epi16(INT16_C(   479), INT16_C( 20906), INT16_C( 31339), INT16_C(-25789),
                            INT16_C( 26024), INT16_C( 23257), INT16_C( 14038), INT16_C(-13152),
                            INT16_C(  6837), INT16_C( -4529), INT16_C(-16234), INT16_C(-29803),
                            INT16_C(   345), INT16_C(  3529), INT16_C( -4208), INT16_C(-14200)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(-1401550594), INT32_C( -392755156), INT32_C(-1796819596), INT32_C(-1026487242),
                            INT32_C(  635705585), INT32_C( 1947566486), INT32_C( 1579971968), INT32_C(  671552800),
                            INT32_C(  -53163704), INT32_C(   93283209), INT32_C( 1507494084), INT32_C( 2043988513),
                            INT32_C( 1862942047), INT32_C( -239955513), INT32_C( -505978631), INT32_C( 1742275249)),
      simde_mm256_set_epi16(INT16_C(   479), INT16_C( 20906), INT16_C( 31339), INT16_C(-25789),
                            INT16_C( 26024), INT16_C( 23257), INT16_C( 14038), INT16_C(-13152),
                            INT16_C(  6837), INT16_C( -4529), INT16_C(-16234), INT16_C(-29803),
                            INT16_C(   345), INT16_C(  3529), INT16_C( -4208), INT16_C(-14200)) },
    { simde_mm256_set_epi16(INT16_C( -3427), INT16_C( 17836), INT16_C(  7353), INT16_C(-29365),
                            INT16_C( -9227), INT16_C( 28467), INT16_C(-13660), INT16_C( 28564),
                            INT16_C(  3740), INT16_C( 26153), INT16_C(-27156), INT16_C(  8245),
                            INT16_C( 28488), INT16_C( -5847), INT16_C(-18032), INT16_C(-28087)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(  484462136), INT32_C(-1869216478), INT32_C(-1224423681), INT32_C(  259010490),
                            INT32_C(  271695042), INT32_C( -508100150), INT32_C(  326629519), INT32_C(-2075728301),
                            INT32_C( 1430269846), INT32_C( -557150729), INT32_C(-1603145004), INT32_C(-1926846046),
                            INT32_C( 1515815293), INT32_C(-1074983379), INT32_C(-1337974349), INT32_C( 1536893374)),
      simde_mm256_set_epi16(INT16_C( -3427), INT16_C( 17836), INT16_C(  7353), INT16_C(-29365),
                            INT16_C( -9227), INT16_C( 28467), INT16_C(-13660), INT16_C( 28564),
                            INT16_C(  3740), INT16_C( 26153), INT16_C(-27156), INT16_C(  8245),
                            INT16_C( 28488), INT16_C( -5847), INT16_C(-18032), INT16_C(-28087)) },
    { simde_mm256_set_epi16(INT16_C( 11942), INT16_C(  3200), INT16_C( 26207), INT16_C( 28891),
                            INT16_C(-21120), INT16_C( 23935), INT16_C( 23697), INT16_C( -6686),
                            INT16_C(  4973), INT16_C(-12874), INT16_C( 21523), INT16_C( 18403),
                            INT16_C(-30279), INT16_C( 22300), INT16_C(-10277), INT16_C( 31923)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(   99021516), INT32_C( -725988672), INT32_C(-1021115053), INT32_C( 1709403708),
                            INT32_C(   -4910667), INT32_C( 1604667568), INT32_C(-1909621033), INT32_C( 1075339790),
                            INT32_C( -834463428), INT32_C(-1922935907), INT32_C(  358822882), INT32_C(-1580688075),
                            INT32_C(  770718488), INT32_C( 1450946493), INT32_C(-1709682666), INT32_C( 1943195809)),
      simde_mm256_set_epi16(INT16_C( 11942), INT16_C(  3200), INT16_C( 26207), INT16_C( 28891),
                            INT16_C(-21120), INT16_C( 23935), INT16_C( 23697), INT16_C( -6686),
                            INT16_C(  4973), INT16_C(-12874), INT16_C( 21523), INT16_C( 18403),
                            INT16_C(-30279), INT16_C( 22300), INT16_C(-10277), INT16_C( 31923)) },
    { simde_mm256_set_epi16(INT16_C( 23594), INT16_C(-12459), INT16_C(-31007), INT16_C( 27189),
                            INT16_C(  9973), INT16_C(-17207), INT16_C( 29928), INT16_C(-22888),
                            INT16_C( -6347), INT16_C(-15381), INT16_C( 11924), INT16_C( 27730),
                            INT16_C( 32118), INT16_C(-13020), INT16_C(-15209), INT16_C(-12363)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(  663660472), INT32_C( 1693978555), INT32_C(-1956550652), INT32_C(  443334302),
                            INT32_C(-1616544448), INT32_C( 1550532992), INT32_C( -641247424), INT32_C( -628774246),
                            INT32_C( -100015796), INT32_C(-1083334846), INT32_C( 1289981083), INT32_C(-1401758632),
                            INT32_C( 1185649877), INT32_C(-1803875103), INT32_C(-1965714593), INT32_C(-1155378946)),
      simde_mm256_set_epi16(INT16_C( 23594), INT16_C(-12459), INT16_C(-31007), INT16_C( 27189),
                            INT16_C(  9973), INT16_C(-17207), INT16_C( 29928), INT16_C(-22888),
                            INT16_C( -6347), INT16_C(-15381), INT16_C( 11924), INT16_C( 27730),
                            INT16_C( 32118), INT16_C(-13020), INT16_C(-15209), INT16_C(-12363)) },
    { simde_mm256_set_epi16(INT16_C(-27164), INT16_C( 24479), INT16_C( -5741), INT16_C( 22801),
                            INT16_C(  4156), INT16_C( 16807), INT16_C(-14947), INT16_C(-16390),
                            INT16_C(-10040), INT16_C(-23097), INT16_C(-17671), INT16_C(  7407),
                            INT16_C( 10396), INT16_C( 18504), INT16_C(  -151), INT16_C( 25224)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(-1299509468), INT32_C(  625075243), INT32_C(  366130068), INT32_C(  879215932),
                            INT32_C(-1266430196), INT32_C(  614354711), INT32_C( -106069824), INT32_C( 1616883515),
                            INT32_C(   76033746), INT32_C( 1717724968), INT32_C(-1376284635), INT32_C( 1121289971),
                            INT32_C(-2088495749), INT32_C(  336749198), INT32_C( -155113738), INT32_C(-1830992577)),
      simde_mm256_set_epi16(INT16_C(-27164), INT16_C( 24479), INT16_C( -5741), INT16_C( 22801),
                            INT16_C(  4156), INT16_C( 16807), INT16_C(-14947), INT16_C(-16390),
                            INT16_C(-10040), INT16_C(-23097), INT16_C(-17671), INT16_C(  7407),
                            INT16_C( 10396), INT16_C( 18504), INT16_C(  -151), INT16_C( 25224)) },
    { simde_mm256_set_epi16(INT16_C( 12625), INT16_C(-27479), INT16_C(  8247), INT16_C(-17103),
                            INT16_C( 25662), INT16_C( 20506), INT16_C( 11066), INT16_C(  6659),
                            INT16_C( 26083), INT16_C( 21192), INT16_C( 16228), INT16_C(-11727),
                            INT16_C( 26864), INT16_C(-24696), INT16_C(-30487), INT16_C( 29529)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(   34084284), INT32_C(  409350881), INT32_C(-1729303858), INT32_C( -262004167),
                            INT32_C(  146178005), INT32_C(-2107356944), INT32_C(  395917776), INT32_C( 2107797030),
                            INT32_C( -440548267), INT32_C( -370689036), INT32_C(  806714643), INT32_C( 1905229024),
                            INT32_C( -272848474), INT32_C(-1969451680), INT32_C(-1920837116), INT32_C( 1649878440)),
      simde_mm256_set_epi16(INT16_C( 12625), INT16_C(-27479), INT16_C(  8247), INT16_C(-17103),
                            INT16_C( 25662), INT16_C( 20506), INT16_C( 11066), INT16_C(  6659),
                            INT16_C( 26083), INT16_C( 21192), INT16_C( 16228), INT16_C(-11727),
                            INT16_C( 26864), INT16_C(-24696), INT16_C(-30487), INT16_C( 29529)) },
    { simde_mm256_set_epi16(INT16_C(-32242), INT16_C(  1477), INT16_C( 29883), INT16_C(-20158),
                            INT16_C( 25891), INT16_C(  4875), INT16_C(-10189), INT16_C( -5617),
                            INT16_C(  6706), INT16_C( 24335), INT16_C(  9342), INT16_C(-13786),
                            INT16_C(  3512), INT16_C( 16237), INT16_C( -8714), INT16_C( 16153)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(  195376036), INT32_C( -495064786), INT32_C(  -53371424), INT32_C( 1311701561),
                            INT32_C(-2117744500), INT32_C(-1179511086), INT32_C(-1867959349), INT32_C(  475635859),
                            INT32_C(  904842846), INT32_C(  684217125), INT32_C( 1793883204), INT32_C(-1966946957),
                            INT32_C(-2097971976), INT32_C( -962037344), INT32_C(  848042064), INT32_C( -776821492)),
      simde_mm256_set_epi16(INT16_C(-32242), INT16_C(  1477), INT16_C( 29883), INT16_C(-20158),
                            INT16_C( 25891), INT16_C(  4875), INT16_C(-10189), INT16_C( -5617),
                            INT16_C(  6706), INT16_C( 24335), INT16_C(  9342), INT16_C(-13786),
                            INT16_C(  3512), INT16_C( 16237), INT16_C( -8714), INT16_C( 16153)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_mask_cvtepi32_epi16(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtepi32_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { UINT16_C(62448),
      simde_mm512_set_epi32(INT32_C( 1733443347), INT32_C( 1269945573), INT32_C( 1675340693), INT32_C( -372773929),
                            INT32_C( 2035462260), INT32_C(  166743347), INT32_C(-2017260412), INT32_C( -999942581),
                            INT32_C( 1336596551), INT32_C( 1310130923), INT32_C(-2130894500), INT32_C(   79178551),
                            INT32_C( -564203638), INT32_C( -457882615), INT32_C(  841119524), INT32_C(-1991554176)),
      simde_mm256_set_epi16(INT16_C( 16147), INT16_C(-11035), INT16_C(-21611), INT16_C( -5161),
                            INT16_C(     0), INT16_C(     0), INT16_C(  3204), INT16_C(  5707),
                            INT16_C(-10169), INT16_C(   747), INT16_C(  8540), INT16_C( 11063),
                            INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(44660),
      simde_mm512_set_epi32(INT32_C(-1218417254), INT32_C( 1692990026), INT32_C(-1386090603), INT32_C( 1397806664),
                            INT32_C( -575365446), INT32_C(-1171498312), INT32_C( 1924925658), INT32_C( -627760657),
                            INT32_C(-1356569050), INT32_C(-1868805570), INT32_C( -261243738), INT32_C( 1743833717),
                            INT32_C(-1242914880), INT32_C( 1254989546), INT32_C( 1994544999), INT32_C( 1112335649)),
      simde_mm256_set_epi16(INT16_C( 28058), INT16_C(     0), INT16_C( -4203), INT16_C(     0),
                            INT16_C(-24902), INT16_C( 23224), INT16_C(  2266), INT16_C(     0),
                            INT16_C(     0), INT16_C( 19006), INT16_C(-17242), INT16_C(-13707),
                            INT16_C(     0), INT16_C(-24854), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(31128),
      simde_mm512_set_epi32(INT32_C(   69575138), INT32_C( 1919952173), INT32_C( 1401785497), INT32_C( 1100669410),
                            INT32_C(-1630704561), INT32_C(-1855467473), INT32_C(   23912175), INT32_C(-1425116453),
                            INT32_C( 1420355947), INT32_C( 1598566097), INT32_C( 2042011572), INT32_C(  547568041),
                            INT32_C( 2024698419), INT32_C(  280574247), INT32_C( 1759265883), INT32_C(  938304495)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(  9517), INT16_C(-29543), INT16_C( -7710),
                            INT16_C( 27727), INT16_C(     0), INT16_C(     0), INT16_C( 29403),
                            INT16_C( -5781), INT16_C(     0), INT16_C(     0), INT16_C( 14761),
                            INT16_C( 29235), INT16_C(     0), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(19693),
      simde_mm512_set_epi32(INT32_C(-1905579249), INT32_C( 1462851665), INT32_C(-1254373510), INT32_C(  100048907),
                            INT32_C(  516278640), INT32_C(-1425769185), INT32_C(   -7461592), INT32_C( 1494577436),
                            INT32_C(  502504372), INT32_C( 1089107848), INT32_C(-1216995523), INT32_C(  461811592),
                            INT32_C( -365493071), INT32_C(  361540743), INT32_C(-2121833283), INT32_C(  157473702)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C( 22609), INT16_C(     0), INT16_C(     0),
                            INT16_C(-13968), INT16_C( 32031), INT16_C(     0), INT16_C(     0),
                            INT16_C(-25676), INT16_C( 30600), INT16_C(  7997), INT16_C(     0),
                            INT16_C(  1201), INT16_C(-21369), INT16_C(     0), INT16_C( -9306)) },
    { UINT16_C(38458),
      simde_mm512_set_epi32(INT32_C( 1275507295), INT32_C( 1283962881), INT32_C( 1982699930), INT32_C(-1010602497),
                            INT32_C(  376304743), INT32_C( 1749590467), INT32_C( 1162888686), INT32_C( 1275140125),
                            INT32_C(-1086608184), INT32_C(-1843574089), INT32_C( 1348635921), INT32_C(-1517373547),
                            INT32_C(  605500023), INT32_C(  634159381), INT32_C( 1101094426), INT32_C( 1261269548)),
      simde_mm256_set_epi16(INT16_C(-19873), INT16_C(     0), INT16_C(     0), INT16_C( 28159),
                            INT16_C(     0), INT16_C(-24125), INT16_C( 17902), INT16_C(     0),
                            INT16_C(     0), INT16_C(     0), INT16_C(-29423), INT16_C(-18539),
                            INT16_C( 12919), INT16_C(     0), INT16_C( 24090), INT16_C(     0)) },
    { UINT16_C(65367),
      simde_mm512_set_epi32(INT32_C( 1934570464), INT32_C( 1255340091), INT32_C(  198090707), INT32_C(-1547945733),
                            INT32_C( -765577646), INT32_C(  117063114), INT32_C(-1248867899), INT32_C( 1778355257),
                            INT32_C( -663218462), INT32_C( 1146911732), INT32_C(  833410001), INT32_C(  831168945),
                            INT32_C(  809078758), INT32_C( 2051643150), INT32_C( 1911602886), INT32_C( 2000896017)),
      simde_mm256_set_epi16(INT16_C( 13280), INT16_C( -1989), INT16_C(-24621), INT16_C( 14587),
                            INT16_C( 13906), INT16_C( 15818), INT16_C(-13883), INT16_C(-29639),
                            INT16_C(     0), INT16_C( 31732), INT16_C(     0), INT16_C(-24143),
                            INT16_C(     0), INT16_C(-26866), INT16_C(-16698), INT16_C( 16401)) },
    { UINT16_C(62783),
      simde_mm512_set_epi32(INT32_C( 1627565128), INT32_C(-1096299878), INT32_C(  406409532), INT32_C( -810620223),
                            INT32_C(-1765599290), INT32_C(  940004452), INT32_C(    4015514), INT32_C(  569486595),
                            INT32_C( 1803479161), INT32_C(  923486692), INT32_C(  979134652), INT32_C(  -47423892),
                            INT32_C(  359951146), INT32_C( 1353906358), INT32_C( -927313372), INT32_C( -280751214)),
      simde_mm256_set_epi16(INT16_C(-21432), INT16_C(-13670), INT16_C( 20796), INT16_C( -5439),
                            INT16_C(     0), INT16_C( 21604), INT16_C(     0), INT16_C(-21245),
                            INT16_C(     0), INT16_C(     0), INT16_C( 26812), INT16_C( 24172),
                            INT16_C( 27434), INT16_C( -1866), INT16_C( 21028), INT16_C(  5010)) },
    { UINT16_C(44985),
      simde_mm512_set_epi32(INT32_C(  592495083), INT32_C( 1070090610), INT32_C( 1700726886), INT32_C( -811687128),
                            INT32_C( 1766271711), INT32_C(  793626986), INT32_C(-1804033847), INT32_C(-1014866300),
                            INT32_C(-1598413920), INT32_C( 2119648754), INT32_C( 2102514823), INT32_C(   24213809),
                            INT32_C( 1119164534), INT32_C(  583559347), INT32_C(-2107244695), INT32_C( -582680808)),
      simde_mm256_set_epi16(INT16_C(-15893), INT16_C(     0), INT16_C(  2150), INT16_C(     0),
                            INT16_C( 10975), INT16_C(-13974), INT16_C(-24375), INT16_C( 24196),
                            INT16_C(  9120), INT16_C(     0), INT16_C(-11129), INT16_C( 31025),
                            INT16_C(  6262), INT16_C(     0), INT16_C(     0), INT16_C(  -232)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_maskz_cvtepi32_epi16(test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtepi64_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(              273955), INT64_C(               -4013),
                            INT64_C(            -7033556), INT64_C(      -1383025729160),
                            INT64_C(          -218214744), INT64_C(   -9402863842296753),
                            INT64_C(                   0), INT64_C(          -240066712)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  35), INT8_C(  83), INT8_C(  44), INT8_C( 120),
                        INT8_C( -88), INT8_C(  79), INT8_C(   0), INT8_C( 104)) },
    { simde_mm512_set_epi64(INT64_C(               50833), INT64_C(           484208564),
                            INT64_C(                -124), INT64_C(         -8907018595),
                            INT64_C(         -1053135968), INT64_C(    2128258677497261),
                            INT64_C(                   3), INT64_C(  263107913893504060)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-111), INT8_C( -76), INT8_C(-124), INT8_C( -99),
                        INT8_C( -96), INT8_C( -83), INT8_C(   3), INT8_C(  60)) },
    { simde_mm512_set_epi64(INT64_C(    6119961081599912), INT64_C(                 949),
                            INT64_C(  761714638418543261), INT64_C(         -7281666562),
                            INT64_C(         -3399190417), INT64_C(                 231),
                            INT64_C(-1217801394263696454), INT64_C(                  43)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -88), INT8_C( -75), INT8_C( -99), INT8_C(  -2),
                        INT8_C( 111), INT8_C( -25), INT8_C( -70), INT8_C(  43)) },
    { simde_mm512_set_epi64(INT64_C(     -74999030828832), INT64_C(       3805264232880),
                            INT64_C(              321542), INT64_C(        -18067967511),
                            INT64_C(      21602392348127), INT64_C(          -482358781),
                            INT64_C(              -12189), INT64_C(   16807900958735709)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -32), INT8_C( -80), INT8_C(   6), INT8_C( -23),
                        INT8_C( -33), INT8_C(   3), INT8_C(  99), INT8_C(  93)) },
    { simde_mm512_set_epi64(INT64_C(               11997), INT64_C(                 192),
                            INT64_C(        -32039837063), INT64_C(               39788),
                            INT64_C(           684970521), INT64_C(     361409660761858),
                            INT64_C(   27655177518327113), INT64_C(         -7050752136)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -35), INT8_C( -64), INT8_C( 121), INT8_C( 108),
                        INT8_C(  25), INT8_C(   2), INT8_C(  73), INT8_C( 120)) },
    { simde_mm512_set_epi64(INT64_C(    -131967536383739), INT64_C(                 -32),
                            INT64_C(                 -54), INT64_C(                -245),
                            INT64_C(              -42658), INT64_C(                   3),
                            INT64_C(         26934708458), INT64_C(               -6255)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   5), INT8_C( -32), INT8_C( -54), INT8_C(  11),
                        INT8_C(  94), INT8_C(   3), INT8_C( -22), INT8_C(-111)) },
    { simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-4069220464223700234),
                            INT64_C(                   0), INT64_C(-8071151262900075656),
                            INT64_C(                  -1), INT64_C(          4132460747),
                            INT64_C(                 -27), INT64_C(            -7116923)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( -10), INT8_C(   0), INT8_C( 120),
                        INT8_C(  -1), INT8_C( -53), INT8_C( -27), INT8_C(-123)) },
    { simde_mm512_set_epi64(INT64_C(         54634445793), INT64_C(         63569905620),
                            INT64_C(            20981054), INT64_C(             2614918),
                            INT64_C(         11917164823), INT64_C(   15401551242937960),
                            INT64_C(                  -1), INT64_C(        366397165244)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -31), INT8_C( -44), INT8_C(  62), INT8_C(-122),
                        INT8_C(  23), INT8_C( 104), INT8_C(  -1), INT8_C( -68)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtepi64_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtepi64_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( -29), INT8_C(  40), INT8_C(-125), INT8_C( 109),
                        INT8_C(  49), INT8_C(-123), INT8_C(  61), INT8_C(  93),
                        INT8_C( 119), INT8_C(  82), INT8_C( -47), INT8_C(-123),
                        INT8_C(  96), INT8_C( 115), INT8_C(  88), INT8_C( 113)),
      UINT8_C(223),
      simde_mm512_set_epi64(INT64_C( 6687541384263213551), INT64_C( 5724651445127999262),
                            INT64_C(  515917234952190154), INT64_C(-2078513286835465824),
                            INT64_C(-5678095839466519103), INT64_C(-5250598357465102176),
                            INT64_C( 5628372052312524003), INT64_C(-6534496578513864639)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -17), INT8_C(  30), INT8_C( -47), INT8_C( -96),
                        INT8_C( -63), INT8_C( -96), INT8_C( -29), INT8_C(  65)) },
    { simde_mm_set_epi8(INT8_C( -14), INT8_C(-119), INT8_C( -22), INT8_C(  58),
                        INT8_C(  54), INT8_C( 111), INT8_C(  42), INT8_C( -77),
                        INT8_C( -18), INT8_C( -70), INT8_C( -80), INT8_C(  46),
                        INT8_C( 106), INT8_C( -96), INT8_C(  15), INT8_C(  95)),
      UINT8_C( 47),
      simde_mm512_set_epi64(INT64_C(-7199926593972874140), INT64_C( 3582798934829117823),
                            INT64_C(-3995728052429757855), INT64_C( 5657566593790284700),
                            INT64_C( 6773484901462632925), INT64_C( 8241942592423347061),
                            INT64_C( 2229820159646878746), INT64_C( 3896347912103466998)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -18), INT8_C( -70), INT8_C(  97), INT8_C(  46),
                        INT8_C( -35), INT8_C( 117), INT8_C(  26), INT8_C( -10)) },
    { simde_mm_set_epi8(INT8_C( -79), INT8_C( -49), INT8_C(   7), INT8_C(-126),
                        INT8_C( -28), INT8_C( 123), INT8_C( -96), INT8_C( 124),
                        INT8_C(  91), INT8_C( -99), INT8_C( -80), INT8_C( -88),
                        INT8_C( 116), INT8_C( -89), INT8_C( 114), INT8_C(  45)),
      UINT8_C( 32),
      simde_mm512_set_epi64(INT64_C( 1872773192980738689), INT64_C(-1477397926013066747),
                            INT64_C(-6194316386860074693), INT64_C(-4822857680519278981),
                            INT64_C(-5283405123335993403), INT64_C(-6429150440483390464),
                            INT64_C(-4929379111042447395), INT64_C(-6405006500483831845)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  91), INT8_C( -99), INT8_C(  59), INT8_C( -88),
                        INT8_C( 116), INT8_C( -89), INT8_C( 114), INT8_C(  45)) },
    { simde_mm_set_epi8(INT8_C( 114), INT8_C( -31), INT8_C(  47), INT8_C(-111),
                        INT8_C(  33), INT8_C( -14), INT8_C( 122), INT8_C( 111),
                        INT8_C( -83), INT8_C(  91), INT8_C( -60), INT8_C(  45),
                        INT8_C(  37), INT8_C( -88), INT8_C( 104), INT8_C(  99)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C(-7653330178277967370), INT64_C(-5020495147285141275),
                            INT64_C( 6758764557968224023), INT64_C(-1851791773179197952),
                            INT64_C(-6103777279451617061), INT64_C(  206815346516353622),
                            INT64_C( 6813004940767614351), INT64_C(-7563904436720933700)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -10), INT8_C( -27), INT8_C(  23), INT8_C(   0),
                        INT8_C( -37), INT8_C( -88), INT8_C( 104), INT8_C( -68)) },
    { simde_mm_set_epi8(INT8_C( -67), INT8_C(  -1), INT8_C( -46), INT8_C( -84),
                        INT8_C(  39), INT8_C(  39), INT8_C(-119), INT8_C(  65),
                        INT8_C(  46), INT8_C( -71), INT8_C(  52), INT8_C( -94),
                        INT8_C( -35), INT8_C(  40), INT8_C(  43), INT8_C(  49)),
      UINT8_C(  7),
      simde_mm512_set_epi64(INT64_C(-3757474529630882229), INT64_C(-5520552524494149273),
                            INT64_C( 8098657770465110034), INT64_C(-3376247133280868335),
                            INT64_C(-6999648475839917999), INT64_C(-6263768072863701828),
                            INT64_C( 8573253492598977185), INT64_C(-5991441569081448700)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  46), INT8_C( -71), INT8_C(  52), INT8_C( -94),
                        INT8_C( -35), INT8_C( -68), INT8_C( -95), INT8_C(   4)) },
    { simde_mm_set_epi8(INT8_C(  -9), INT8_C(-118), INT8_C(  65), INT8_C( -57),
                        INT8_C(  27), INT8_C(  91), INT8_C( -23), INT8_C(  52),
                        INT8_C(   2), INT8_C( -28), INT8_C(  97), INT8_C( -21),
                        INT8_C(  27), INT8_C( -30), INT8_C( 122), INT8_C(-109)),
      UINT8_C(149),
      simde_mm512_set_epi64(INT64_C( 5458614153698880625), INT64_C( -778938361127636026),
                            INT64_C( 8050145995564739343), INT64_C( 1111000858517683962),
                            INT64_C( 3114364286144520444), INT64_C( 6655688875273101631),
                            INT64_C(-7614107199841735371), INT64_C( 2031290604527978571)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 113), INT8_C( -28), INT8_C(  97), INT8_C(  -6),
                        INT8_C(  27), INT8_C(  63), INT8_C( 122), INT8_C(  75)) },
    { simde_mm_set_epi8(INT8_C( -60), INT8_C(  62), INT8_C( 101), INT8_C( -30),
                        INT8_C( -93), INT8_C( -50), INT8_C(  37), INT8_C(  31),
                        INT8_C(  27), INT8_C( -48), INT8_C(  69), INT8_C(  44),
                        INT8_C( -43), INT8_C(  17), INT8_C( -35), INT8_C( 115)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C( 3482415301107112997), INT64_C(-1066915963483370707),
                            INT64_C(  975879680641503817), INT64_C( 1992911285604122710),
                            INT64_C(-3358914405152476651), INT64_C(-2231921767905544800),
                            INT64_C(-2421993418742613205), INT64_C( 7643326833070961923)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  37), INT8_C(  45), INT8_C(  73), INT8_C(  86),
                        INT8_C(  21), INT8_C(  17), INT8_C( -35), INT8_C(   3)) },
    { simde_mm_set_epi8(INT8_C(   6), INT8_C( -97), INT8_C(  51), INT8_C( 125),
                        INT8_C( -36), INT8_C(  69), INT8_C(-124), INT8_C(-125),
                        INT8_C(  22), INT8_C( 105), INT8_C( -62), INT8_C( -46),
                        INT8_C(  52), INT8_C(  71), INT8_C(  89), INT8_C(  35)),
      UINT8_C(239),
      simde_mm512_set_epi64(INT64_C( 1040334403932540361), INT64_C(-6095052890563426812),
                            INT64_C( 3754572277322521777), INT64_C(-6367682946633100387),
                            INT64_C( 1442431276846498456), INT64_C( 7207346769540727007),
                            INT64_C(  308866448566430857), INT64_C( -679722980524780734)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -55), INT8_C(   4), INT8_C( -79), INT8_C( -46),
                        INT8_C(-104), INT8_C( -33), INT8_C(-119), INT8_C(  66)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtepi64_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtepi64_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT8_C(113),
      simde_mm512_set_epi64(INT64_C(  515917234952190154), INT64_C(-2078513286835465824),
                            INT64_C(-5678095839466519103), INT64_C(-5250598357465102176),
                            INT64_C( 5628372052312524003), INT64_C(-6534496578513864639),
                            INT64_C(-8550316027439185043), INT64_C( 3568325751404155269)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( -96), INT8_C( -63), INT8_C( -96),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(-123)) },
    { UINT8_C( 30),
      simde_mm512_set_epi64(INT64_C( 6773484901462632925), INT64_C( 8241942592423347061),
                            INT64_C( 2229820159646878746), INT64_C( 3896347912103466998),
                            INT64_C(-5906398137596384710), INT64_C( 3922400752755716142),
                            INT64_C( 7683157866547440500), INT64_C( -739819664570184664)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -10),
                        INT8_C(  58), INT8_C(  46), INT8_C( 116), INT8_C(   0)) },
    { UINT8_C(156),
      simde_mm512_set_epi64(INT64_C(-4929379111042447395), INT64_C(-6405006500483831845),
                            INT64_C( 2215750266367444866), INT64_C(-1982814754953121624),
                            INT64_C( 8405812769728018664), INT64_C(-2496168041650804570),
                            INT64_C( 1451057929838678657), INT64_C(-5774632452127213865)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -35), INT8_C(   0), INT8_C(   0), INT8_C( -88),
                        INT8_C( -24), INT8_C( -90), INT8_C(   0), INT8_C(   0)) },
    { UINT8_C(  0),
      simde_mm512_set_epi64(INT64_C( 8742888725914136465), INT64_C( 2446152167684031533),
                            INT64_C( 2713533550337813835), INT64_C(-1117540861446505879),
                            INT64_C( -858039058779513328), INT64_C(-6425799683333831555),
                            INT64_C( -116444347105438968), INT64_C( 8292331728128119228)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT8_C(188),
      simde_mm512_set_epi64(INT64_C(-2510709300292622144), INT64_C( 9148774531133054531),
                            INT64_C( 7313609285459244174), INT64_C( 6211078615791902001),
                            INT64_C(-8527066713273202969), INT64_C( 5633907317463236982),
                            INT64_C( 4554513983759755237), INT64_C(-7822939952516719363)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -64), INT8_C(   0), INT8_C(-114), INT8_C(  49),
                        INT8_C( -25), INT8_C( 118), INT8_C(   0), INT8_C(   0)) },
    { UINT8_C(162),
      simde_mm512_set_epi64(INT64_C( 7323954049511077461), INT64_C( 7928490467780798750),
                            INT64_C(-3252469668491018036), INT64_C(  357209688360787999),
                            INT64_C(-1907656262787903328), INT64_C(-2036415786530496605),
                            INT64_C( 1036996715160227079), INT64_C(-4755851044646844095)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  85), INT8_C(   0), INT8_C( -52), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   7), INT8_C(   0)) },
    { UINT8_C(186),
      simde_mm512_set_epi64(INT64_C( -208925926019796527), INT64_C( 5822768373978481092),
                            INT64_C(-3878688622084013609), INT64_C(-3481162437083752438),
                            INT64_C(-1839007639418773611), INT64_C( -609602476141975244),
                            INT64_C(  208399145178921619), INT64_C(-3757474529630882229)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -47), INT8_C(   0), INT8_C( -41), INT8_C(  10),
                        INT8_C(-107), INT8_C(   0), INT8_C(-109), INT8_C(   0)) },
    { UINT8_C(100),
      simde_mm512_set_epi64(INT64_C( 3568880194756499921), INT64_C( 7170640089353194818),
                            INT64_C( 3704198613752651513), INT64_C(-4305892169634405089),
                            INT64_C( 2004177893035466099), INT64_C( 5458614153698880625),
                            INT64_C( -778938361127636026), INT64_C( 8050145995564739343)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(  66), INT8_C(  -7), INT8_C(   0),
                        INT8_C(   0), INT8_C( 113), INT8_C(   0), INT8_C(   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtepi64_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtepi64_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-7076619543590638545), INT64_C(-1547640767609811155),
                            INT64_C(-3757295450136412488), INT64_C( 7215428799648851281),
                            INT64_C( 4617967402188816469), INT64_C(-6436617106056712861),
                            INT64_C(-5810949596740302593), INT64_C(-6127930763767189365)),
      simde_mm_set_epi16(INT16_C( 27695), INT16_C( 20269), INT16_C( 32440), INT16_C( 25937),
                         INT16_C( 16469), INT16_C( 28003), INT16_C( 18687), INT16_C( -3957)) },
    { simde_mm512_set_epi64(INT64_C(-5381921416793630443), INT64_C( -135676082428955899),
                            INT64_C( 5552873466219372969), INT64_C( 6395720488802354930),
                            INT64_C( 7752411237949515847), INT64_C( 7000778850177503560),
                            INT64_C(-2818020559339799438), INT64_C(-2377752498793556325)),
      simde_mm_set_epi16(INT16_C( -9963), INT16_C( 13061), INT16_C(-24151), INT16_C(-30990),
                         INT16_C( 10311), INT16_C(-11960), INT16_C(-17294), INT16_C(-10597)) },
    { simde_mm512_set_epi64(INT64_C( 2102322971158047948), INT64_C(-4534059143765607616),
                            INT64_C(-8156186454528956670), INT64_C(-1110322353084003886),
                            INT64_C( 2722224959827855869), INT64_C(-8203319889631878316),
                            INT64_C( 3055433368131497869), INT64_C(-5348785296051142577)),
      simde_mm_set_epi16(INT16_C( -4916), INT16_C( 15168), INT16_C( -6398), INT16_C(-30254),
                         INT16_C(-13827), INT16_C( 25428), INT16_C( 28557), INT16_C(    79)) },
    { simde_mm512_set_epi64(INT64_C( 1873487555889602710), INT64_C(-5277671716035001257),
                            INT64_C(-1735965719996896598), INT64_C(-3435103322406338318),
                            INT64_C( 3655189638093457022), INT64_C( 2841010088304043730),
                            INT64_C( 5827184599862175253), INT64_C( -971579224938582414)),
      simde_mm_set_epi16(INT16_C(  8342), INT16_C(-31657), INT16_C( 27306), INT16_C(-16142),
                         INT16_C(-14722), INT16_C( -6446), INT16_C( 26133), INT16_C( -4494)) },
    { simde_mm512_set_epi64(INT64_C(-7259266739541946745), INT64_C(-8120641422716522112),
                            INT64_C( 3643097831469212363), INT64_C(-8930268807776982308),
                            INT64_C( 8468988530628615446), INT64_C( 8901878832922162004),
                            INT64_C( 3527022727868595852), INT64_C( 3854384042855807476)),
      simde_mm_set_epi16(INT16_C( 20103), INT16_C(-20096), INT16_C( -7477), INT16_C(-21796),
                         INT16_C( -8938), INT16_C( -3244), INT16_C( 16012), INT16_C(-17932)) },
    { simde_mm512_set_epi64(INT64_C( 7662135416143755844), INT64_C(-4685369386623572014),
                            INT64_C( 2116014329396479379), INT64_C(-3403348481015692825),
                            INT64_C( 6159842858110289023), INT64_C(  -32959868996567131),
                            INT64_C( 5549180600650677482), INT64_C( 3533643669382248608)),
      simde_mm_set_epi16(INT16_C( 14916), INT16_C( -6190), INT16_C( 17811), INT16_C(-15897),
                         INT16_C(  1151), INT16_C(-16475), INT16_C(-17174), INT16_C( 14496)) },
    { simde_mm512_set_epi64(INT64_C(-7461888410031105124), INT64_C(-5633250430711017511),
                            INT64_C( 2303316420630814828), INT64_C( 8431273119974737498),
                            INT64_C( 6792217696133583430), INT64_C(-1977946393686863567),
                            INT64_C(    7761100783468144), INT64_C( 1660710717369297622)),
      simde_mm_set_epi16(INT16_C( -9316), INT16_C( 20441), INT16_C(  8300), INT16_C(-14758),
                         INT16_C(  4678), INT16_C(-28367), INT16_C(-16784), INT16_C( 10966)) },
    { simde_mm512_set_epi64(INT64_C(  797114543045129949), INT64_C(-2395988650566541266),
                            INT64_C( 3385029089061196001), INT64_C(-8666680959957475234),
                            INT64_C(-2790900178180541201), INT64_C( 9019487904069949528),
                            INT64_C( 4196811514711324420), INT64_C( -189237280820252155)),
      simde_mm_set_epi16(INT16_C(-18723), INT16_C(  8238), INT16_C( 20705), INT16_C( -8098),
                         INT16_C( 29935), INT16_C(-26536), INT16_C( 14084), INT16_C(-19963)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtepi64_epi16(test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtepi64_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi16(INT16_C( -7384), INT16_C(-31891), INT16_C( 12677), INT16_C( 15709),
                         INT16_C( 30546), INT16_C(-11899), INT16_C( 24691), INT16_C( 22641)),
      UINT8_C(223),
      simde_mm512_set_epi64(INT64_C( 6687541384263213551), INT64_C( 5724651445127999262),
                            INT64_C(  515917234952190154), INT64_C(-2078513286835465824),
                            INT64_C(-5678095839466519103), INT64_C(-5250598357465102176),
                            INT64_C( 5628372052312524003), INT64_C(-6534496578513864639)),
      simde_mm_set_epi16(INT16_C(-24081), INT16_C( -4322), INT16_C( 12677), INT16_C(  4512),
                         INT16_C(-31295), INT16_C(  2208), INT16_C(  1251), INT16_C( -1983)) },
    { simde_mm_set_epi16(INT16_C( -3447), INT16_C( -5574), INT16_C( 13935), INT16_C( 10931),
                         INT16_C( -4422), INT16_C(-20434), INT16_C( 27296), INT16_C(  3935)),
      UINT8_C( 47),
      simde_mm512_set_epi64(INT64_C(-7199926593972874140), INT64_C( 3582798934829117823),
                            INT64_C(-3995728052429757855), INT64_C( 5657566593790284700),
                            INT64_C( 6773484901462632925), INT64_C( 8241942592423347061),
                            INT64_C( 2229820159646878746), INT64_C( 3896347912103466998)),
      simde_mm_set_epi16(INT16_C( -3447), INT16_C( -5574), INT16_C( 25185), INT16_C( 10931),
                         INT16_C( -5667), INT16_C( 17269), INT16_C(-13286), INT16_C(  6134)) },
    { simde_mm_set_epi16(INT16_C(-20017), INT16_C(  1922), INT16_C( -7045), INT16_C(-24452),
                         INT16_C( 23453), INT16_C(-20312), INT16_C( 29863), INT16_C( 29229)),
      UINT8_C( 32),
      simde_mm512_set_epi64(INT64_C( 1872773192980738689), INT64_C(-1477397926013066747),
                            INT64_C(-6194316386860074693), INT64_C(-4822857680519278981),
                            INT64_C(-5283405123335993403), INT64_C(-6429150440483390464),
                            INT64_C(-4929379111042447395), INT64_C(-6405006500483831845)),
      simde_mm_set_epi16(INT16_C(-20017), INT16_C(  1922), INT16_C( -1733), INT16_C(-24452),
                         INT16_C( 23453), INT16_C(-20312), INT16_C( 29863), INT16_C( 29229)) },
    { simde_mm_set_epi16(INT16_C( 29409), INT16_C( 12177), INT16_C(  8690), INT16_C( 31343),
                         INT16_C(-21157), INT16_C(-15315), INT16_C(  9640), INT16_C( 26723)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C(-7653330178277967370), INT64_C(-5020495147285141275),
                            INT64_C( 6758764557968224023), INT64_C(-1851791773179197952),
                            INT64_C(-6103777279451617061), INT64_C(  206815346516353622),
                            INT64_C( 6813004940767614351), INT64_C(-7563904436720933700)),
      simde_mm_set_epi16(INT16_C( -1546), INT16_C( 10469), INT16_C( 12055), INT16_C(-14848),
                         INT16_C(-22309), INT16_C(-15315), INT16_C(  9640), INT16_C( 25788)) },
    { simde_mm_set_epi16(INT16_C(-16897), INT16_C(-11604), INT16_C( 10023), INT16_C(-30399),
                         INT16_C( 11961), INT16_C( 13474), INT16_C( -8920), INT16_C( 11057)),
      UINT8_C(  7),
      simde_mm512_set_epi64(INT64_C(-3757474529630882229), INT64_C(-5520552524494149273),
                            INT64_C( 8098657770465110034), INT64_C(-3376247133280868335),
                            INT64_C(-6999648475839917999), INT64_C(-6263768072863701828),
                            INT64_C( 8573253492598977185), INT64_C(-5991441569081448700)),
      simde_mm_set_epi16(INT16_C(-16897), INT16_C(-11604), INT16_C( 10023), INT16_C(-30399),
                         INT16_C( 11961), INT16_C(-23364), INT16_C( 12961), INT16_C(  9988)) },
    { simde_mm_set_epi16(INT16_C( -2166), INT16_C( 16839), INT16_C(  7003), INT16_C( -5836),
                         INT16_C(   740), INT16_C( 25067), INT16_C(  7138), INT16_C( 31379)),
      UINT8_C(149),
      simde_mm512_set_epi64(INT64_C( 5458614153698880625), INT64_C( -778938361127636026),
                            INT64_C( 8050145995564739343), INT64_C( 1111000858517683962),
                            INT64_C( 3114364286144520444), INT64_C( 6655688875273101631),
                            INT64_C(-7614107199841735371), INT64_C( 2031290604527978571)),
      simde_mm_set_epi16(INT16_C( 11377), INT16_C( 16839), INT16_C(  7003), INT16_C(-16646),
                         INT16_C(   740), INT16_C(  8511), INT16_C(  7138), INT16_C(-30645)) },
    { simde_mm_set_epi16(INT16_C(-15298), INT16_C( 26082), INT16_C(-23602), INT16_C(  9503),
                         INT16_C(  7120), INT16_C( 17708), INT16_C(-10991), INT16_C( -8845)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C( 3482415301107112997), INT64_C(-1066915963483370707),
                            INT64_C(  975879680641503817), INT64_C( 1992911285604122710),
                            INT64_C(-3358914405152476651), INT64_C(-2231921767905544800),
                            INT64_C(-2421993418742613205), INT64_C( 7643326833070961923)),
      simde_mm_set_epi16(INT16_C( -9179), INT16_C( 19245), INT16_C(-28087), INT16_C(    86),
                         INT16_C(-19947), INT16_C( 17708), INT16_C(-10991), INT16_C( -2813)) },
    { simde_mm_set_epi16(INT16_C(  1695), INT16_C( 13181), INT16_C( -9147), INT16_C(-31613),
                         INT16_C(  5737), INT16_C(-15662), INT16_C( 13383), INT16_C( 22819)),
      UINT8_C(239),
      simde_mm512_set_epi64(INT64_C( 1040334403932540361), INT64_C(-6095052890563426812),
                            INT64_C( 3754572277322521777), INT64_C(-6367682946633100387),
                            INT64_C( 1442431276846498456), INT64_C( 7207346769540727007),
                            INT64_C(  308866448566430857), INT64_C( -679722980524780734)),
      simde_mm_set_epi16(INT16_C( 19913), INT16_C(-14844), INT16_C(-25423), INT16_C(-31613),
                         INT16_C( 17048), INT16_C(-16161), INT16_C(-21367), INT16_C(-27838)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtepi64_epi16(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtepi64_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT8_C(113),
      simde_mm512_set_epi64(INT64_C(  515917234952190154), INT64_C(-2078513286835465824),
                            INT64_C(-5678095839466519103), INT64_C(-5250598357465102176),
                            INT64_C( 5628372052312524003), INT64_C(-6534496578513864639),
                            INT64_C(-8550316027439185043), INT64_C( 3568325751404155269)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(  4512), INT16_C(-31295), INT16_C(  2208),
                         INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(-11899)) },
    { UINT8_C( 30),
      simde_mm512_set_epi64(INT64_C( 6773484901462632925), INT64_C( 8241942592423347061),
                            INT64_C( 2229820159646878746), INT64_C( 3896347912103466998),
                            INT64_C(-5906398137596384710), INT64_C( 3922400752755716142),
                            INT64_C( 7683157866547440500), INT64_C( -739819664570184664)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(  6134),
                         INT16_C( -5574), INT16_C(-20434), INT16_C( -5260), INT16_C(     0)) },
    { UINT8_C(156),
      simde_mm512_set_epi64(INT64_C(-4929379111042447395), INT64_C(-6405006500483831845),
                            INT64_C( 2215750266367444866), INT64_C(-1982814754953121624),
                            INT64_C( 8405812769728018664), INT64_C(-2496168041650804570),
                            INT64_C( 1451057929838678657), INT64_C(-5774632452127213865)),
      simde_mm_set_epi16(INT16_C(-30755), INT16_C(     0), INT16_C(     0), INT16_C(-20312),
                         INT16_C(-18200), INT16_C(-22362), INT16_C(     0), INT16_C(     0)) },
    { UINT8_C(  0),
      simde_mm512_set_epi64(INT64_C( 8742888725914136465), INT64_C( 2446152167684031533),
                            INT64_C( 2713533550337813835), INT64_C(-1117540861446505879),
                            INT64_C( -858039058779513328), INT64_C(-6425799683333831555),
                            INT64_C( -116444347105438968), INT64_C( 8292331728128119228)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(     0),
                         INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(     0)) },
    { UINT8_C(188),
      simde_mm512_set_epi64(INT64_C(-2510709300292622144), INT64_C( 9148774531133054531),
                            INT64_C( 7313609285459244174), INT64_C( 6211078615791902001),
                            INT64_C(-8527066713273202969), INT64_C( 5633907317463236982),
                            INT64_C( 4554513983759755237), INT64_C(-7822939952516719363)),
      simde_mm_set_epi16(INT16_C( -5952), INT16_C(     0), INT16_C( -2930), INT16_C(  7473),
                         INT16_C(  1767), INT16_C(-16010), INT16_C(     0), INT16_C(     0)) },
    { UINT8_C(162),
      simde_mm512_set_epi64(INT64_C( 7323954049511077461), INT64_C( 7928490467780798750),
                            INT64_C(-3252469668491018036), INT64_C(  357209688360787999),
                            INT64_C(-1907656262787903328), INT64_C(-2036415786530496605),
                            INT64_C( 1036996715160227079), INT64_C(-4755851044646844095)),
      simde_mm_set_epi16(INT16_C( 13909), INT16_C(     0), INT16_C( 16588), INT16_C(     0),
                         INT16_C(     0), INT16_C(     0), INT16_C( 19719), INT16_C(     0)) },
    { UINT8_C(186),
      simde_mm512_set_epi64(INT64_C( -208925926019796527), INT64_C( 5822768373978481092),
                            INT64_C(-3878688622084013609), INT64_C(-3481162437083752438),
                            INT64_C(-1839007639418773611), INT64_C( -609602476141975244),
                            INT64_C(  208399145178921619), INT64_C(-3757474529630882229)),
      simde_mm_set_epi16(INT16_C( 29137), INT16_C(     0), INT16_C( 16855), INT16_C(-26614),
                         INT16_C( -8299), INT16_C(     0), INT16_C( 31379), INT16_C(     0)) },
    { UINT8_C(100),
      simde_mm512_set_epi64(INT64_C( 3568880194756499921), INT64_C( 7170640089353194818),
                            INT64_C( 3704198613752651513), INT64_C(-4305892169634405089),
                            INT64_C( 2004177893035466099), INT64_C( 5458614153698880625),
                            INT64_C( -778938361127636026), INT64_C( 8050145995564739343)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(-29374), INT16_C(-24839), INT16_C(     0),
                         INT16_C(     0), INT16_C( 11377), INT16_C(     0), INT16_C(     0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtepi64_epi16(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtsepi32_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(  699026811), INT32_C( -375318237), INT32_C(  900674930), INT32_C( -532379219),
                            INT32_C(  259624037), INT32_C( -680802854), INT32_C( 1547540196), INT32_C( -396867814),
                            INT32_C( -680186334), INT32_C( -646546417), INT32_C( 1050185959), INT32_C(  210164141),
                            INT32_C( -363246440), INT32_C( -713541510), INT32_C(-2055712079), INT32_C( 1173262878)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C( 127)) },
    { simde_mm512_set_epi32(INT32_C( -577654055), INT32_C( 1305263210), INT32_C(-1667480872), INT32_C(-1872423369),
                            INT32_C( 1597611840), INT32_C( 1615412737), INT32_C( -741695305), INT32_C(-2095777141),
                            INT32_C(-1892042790), INT32_C(  882313442), INT32_C( -646755171), INT32_C(  911535864),
                            INT32_C( 1767815726), INT32_C( 1917649804), INT32_C(-2128748387), INT32_C( -486061354)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C( 1571463905), INT32_C(  -29910853), INT32_C( -395190633), INT32_C(-1244489996),
                            INT32_C( -209986706), INT32_C( 1448112069), INT32_C(  665030078), INT32_C( -422459827),
                            INT32_C(  944452940), INT32_C( -118153072), INT32_C(-1276500342), INT32_C( -169030554),
                            INT32_C( -426966016), INT32_C(-1701128651), INT32_C( 2062590869), INT32_C( -398665788)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C(-1419625975), INT32_C(-1694295907), INT32_C( 1865288725), INT32_C(-1595213023),
                            INT32_C( 1905686792), INT32_C(  988618194), INT32_C(  317245247), INT32_C( 1735742912),
                            INT32_C(-1998627679), INT32_C( 1116885338), INT32_C(-1515593887), INT32_C( 2092259348),
                            INT32_C(-1378434205), INT32_C(-1489378877), INT32_C(-2045535759), INT32_C( -659443643)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C(-2129053891), INT32_C( 1544504796), INT32_C( -507196434), INT32_C( 1930810889),
                            INT32_C(  483646719), INT32_C(-1089855428), INT32_C( 2145343900), INT32_C( -881615810),
                            INT32_C( 1970504702), INT32_C( 1583273127), INT32_C( 1163729707), INT32_C(-2126388173),
                            INT32_C( -806581107), INT32_C(-1698532218), INT32_C(-1392314532), INT32_C(-1512591869)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C( 1328977658), INT32_C(  842060052), INT32_C(-1244029168), INT32_C(  509188382),
                            INT32_C(  982259454), INT32_C(-1037867093), INT32_C(  686186344), INT32_C(  324188035),
                            INT32_C( 1498892321), INT32_C(  508595369), INT32_C( -602085962), INT32_C(-1205094693),
                            INT32_C( -758631374), INT32_C(-1994460640), INT32_C( 1443985396), INT32_C(-1807079691)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C( 1080310926), INT32_C(-1355901593), INT32_C(  256636159), INT32_C( 1796880446),
                            INT32_C(-1357394192), INT32_C( 1762335377), INT32_C( -735301194), INT32_C(  230956482),
                            INT32_C( 1868400485), INT32_C(  -33472709), INT32_C( 1961785078), INT32_C( 1853410726),
                            INT32_C(-1472588796), INT32_C( 1430279086), INT32_C( -445655125), INT32_C( 1987885572)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127)) },
    { simde_mm512_set_epi32(INT32_C(  270394269), INT32_C( -407998192), INT32_C(-1084410926), INT32_C(-1333891767),
                            INT32_C( 1497122895), INT32_C( -215247173), INT32_C(-1835983824), INT32_C(  -84201182),
                            INT32_C(-1074778144), INT32_C( 1293769723), INT32_C( 1950241896), INT32_C(-1359838077),
                            INT32_C(  595655233), INT32_C( 2095576879), INT32_C(-1995468186), INT32_C(-1602668019)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtsepi32_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtsepi32_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(   4), INT8_C( 110), INT8_C(   8), INT8_C( 124),
                        INT8_C( -85), INT8_C( -65), INT8_C( -62), INT8_C(-114),
                        INT8_C( 110), INT8_C(  33), INT8_C( -28), INT8_C( 106),
                        INT8_C( 107), INT8_C( 102), INT8_C(  89), INT8_C( 124)),
      UINT16_C(12710),
      simde_mm512_set_epi32(INT32_C(-1797606071), INT32_C(  433393018), INT32_C(  712958933), INT32_C( -930490644),
                            INT32_C(  760893683), INT32_C(-2027734617), INT32_C( 1683947105), INT32_C( -424320007),
                            INT32_C(  107722959), INT32_C(  -13745640), INT32_C(-1276316442), INT32_C(-1722135079),
                            INT32_C( 1405989540), INT32_C(-1209232739), INT32_C(  265793764), INT32_C(-1369604571)),
      simde_mm_set_epi8(INT8_C(   4), INT8_C( 110), INT8_C( 127), INT8_C(-128),
                        INT8_C( -85), INT8_C( -65), INT8_C( -62), INT8_C(-128),
                        INT8_C( 127), INT8_C(  33), INT8_C(-128), INT8_C( 106),
                        INT8_C( 107), INT8_C(-128), INT8_C( 127), INT8_C( 124)) },
    { simde_mm_set_epi8(INT8_C(-103), INT8_C(-117), INT8_C(  21), INT8_C( -66),
                        INT8_C(   4), INT8_C( 101), INT8_C(-112), INT8_C( -12),
                        INT8_C(  -5), INT8_C( -50), INT8_C(  26), INT8_C( -76),
                        INT8_C( -83), INT8_C( -93), INT8_C(  76), INT8_C( -53)),
      UINT16_C(12983),
      simde_mm512_set_epi32(INT32_C( -521071455), INT32_C( -376568333), INT32_C(-1920414563), INT32_C(-1600156196),
                            INT32_C( -699126682), INT32_C(  420932189), INT32_C(  368322579), INT32_C(  983236120),
                            INT32_C(  564176453), INT32_C( -258554655), INT32_C(-1141343253), INT32_C(  -63087550),
                            INT32_C( 1360677357), INT32_C(-1430308733), INT32_C( -736696329), INT32_C(  172239397)),
      simde_mm_set_epi8(INT8_C(-103), INT8_C(-117), INT8_C(-128), INT8_C(-128),
                        INT8_C(   4), INT8_C( 101), INT8_C( 127), INT8_C( -12),
                        INT8_C( 127), INT8_C( -50), INT8_C(-128), INT8_C(-128),
                        INT8_C( -83), INT8_C(-128), INT8_C(-128), INT8_C( 127)) },
    { simde_mm_set_epi8(INT8_C( 103), INT8_C( -24), INT8_C( -57), INT8_C( -16),
                        INT8_C( -92), INT8_C( -81), INT8_C(-124), INT8_C(  85),
                        INT8_C(  75), INT8_C(  77), INT8_C(  63), INT8_C(  75),
                        INT8_C( 108), INT8_C( -84), INT8_C( -66), INT8_C( -39)),
      UINT16_C(58066),
      simde_mm512_set_epi32(INT32_C(  123157162), INT32_C(  -87325004), INT32_C(  -74930983), INT32_C( 1838315906),
                            INT32_C(  -81455038), INT32_C( -220592487), INT32_C( 1565085824), INT32_C(   60698648),
                            INT32_C( 1617237598), INT32_C( 1724869908), INT32_C( 1766262170), INT32_C(-1382435444),
                            INT32_C(  -17747961), INT32_C( -126136440), INT32_C( -844801591), INT32_C(-1631191639)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C( -16),
                        INT8_C( -92), INT8_C( -81), INT8_C( 127), INT8_C(  85),
                        INT8_C( 127), INT8_C( 127), INT8_C(  63), INT8_C(-128),
                        INT8_C( 108), INT8_C( -84), INT8_C(-128), INT8_C( -39)) },
    { simde_mm_set_epi8(INT8_C(  26), INT8_C(-116), INT8_C( -21), INT8_C( -72),
                        INT8_C(  76), INT8_C( -52), INT8_C( -71), INT8_C( 116),
                        INT8_C(  41), INT8_C(-107), INT8_C( -34), INT8_C(  88),
                        INT8_C(  29), INT8_C( 112), INT8_C(  89), INT8_C( 127)),
      UINT16_C(63967),
      simde_mm512_set_epi32(INT32_C(-1851657392), INT32_C(  834971347), INT32_C(   96523257), INT32_C( 1786925287),
                            INT32_C( -210779015), INT32_C( 1982975326), INT32_C( 1450447838), INT32_C( 2089131393),
                            INT32_C(-1680682639), INT32_C( -651728462), INT32_C(-1976749038), INT32_C(-1119382179),
                            INT32_C(  674486709), INT32_C(  -51017687), INT32_C(  -85403955), INT32_C( -771014813)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( -52), INT8_C( -71), INT8_C( 127),
                        INT8_C(-128), INT8_C(-128), INT8_C( -34), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C( -70), INT8_C( -89), INT8_C(  47), INT8_C(  24),
                        INT8_C( 102), INT8_C(  -2), INT8_C(  77), INT8_C(  93),
                        INT8_C(   8), INT8_C(-121), INT8_C(-101), INT8_C(  37),
                        INT8_C(  18), INT8_C(-107), INT8_C(  62), INT8_C( -51)),
      UINT16_C(26024),
      simde_mm512_set_epi32(INT32_C( 1802290864), INT32_C(-1214702907), INT32_C(  821368379), INT32_C( 1594494278),
                            INT32_C(  958208413), INT32_C(-1231281503), INT32_C(  509306793), INT32_C( 1685930534),
                            INT32_C( -983619151), INT32_C(  293070963), INT32_C( 1991494863), INT32_C( 1867270897),
                            INT32_C(-1968548850), INT32_C(-1694953429), INT32_C(  342647810), INT32_C(  475783090)),
      simde_mm_set_epi8(INT8_C( -70), INT8_C(-128), INT8_C( 127), INT8_C(  24),
                        INT8_C( 102), INT8_C(-128), INT8_C(  77), INT8_C( 127),
                        INT8_C(-128), INT8_C(-121), INT8_C( 127), INT8_C(  37),
                        INT8_C(-128), INT8_C(-107), INT8_C(  62), INT8_C( -51)) },
    { simde_mm_set_epi8(INT8_C(  39), INT8_C( 109), INT8_C( 102), INT8_C( -61),
                        INT8_C( -24), INT8_C(  15), INT8_C( 101), INT8_C( -27),
                        INT8_C(  60), INT8_C(-116), INT8_C(  88), INT8_C(   9),
                        INT8_C(  98), INT8_C( -87), INT8_C( -42), INT8_C(  93)),
      UINT16_C(31919),
      simde_mm512_set_epi32(INT32_C(-2040024665), INT32_C(  961434729), INT32_C(-1271404275), INT32_C( 1259406574),
                            INT32_C(-1600364807), INT32_C( 1013354479), INT32_C(  720414379), INT32_C( -600382162),
                            INT32_C( 1552024491), INT32_C(-1608825765), INT32_C( 1791475411), INT32_C(  664867443),
                            INT32_C(  674611871), INT32_C( 2098484402), INT32_C(-1985583060), INT32_C(-2005775341)),
      simde_mm_set_epi8(INT8_C(  39), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C( 101), INT8_C( -27),
                        INT8_C( 127), INT8_C(-116), INT8_C( 127), INT8_C(   9),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C( -81), INT8_C(  41), INT8_C(  25), INT8_C(  -5),
                        INT8_C( -64), INT8_C(  70), INT8_C( 124), INT8_C( -34),
                        INT8_C(  -2), INT8_C(-101), INT8_C(  45), INT8_C( -46),
                        INT8_C(  80), INT8_C(  30), INT8_C( -59), INT8_C(-110)),
      UINT16_C(11111),
      simde_mm512_set_epi32(INT32_C( -897513851), INT32_C(  631520322), INT32_C(  634536018), INT32_C(-2012793591),
                            INT32_C( 1627606334), INT32_C(-1721387558), INT32_C(-1230866277), INT32_C(-1212419278),
                            INT32_C(  577821894), INT32_C( 1429584547), INT32_C(-2040986931), INT32_C( 2040118421),
                            INT32_C(  936070741), INT32_C(  920738578), INT32_C(-1305390023), INT32_C(-1363026646)),
      simde_mm_set_epi8(INT8_C( -81), INT8_C(  41), INT8_C( 127), INT8_C(  -5),
                        INT8_C( 127), INT8_C(  70), INT8_C(-128), INT8_C(-128),
                        INT8_C(  -2), INT8_C( 127), INT8_C(-128), INT8_C( -46),
                        INT8_C(  80), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C(-107), INT8_C( -52), INT8_C(  65), INT8_C(  94),
                        INT8_C( -32), INT8_C( -76), INT8_C(  70), INT8_C( -85),
                        INT8_C( -67), INT8_C(  72), INT8_C(  36), INT8_C( -31),
                        INT8_C(  47), INT8_C( 126), INT8_C(  14), INT8_C(-117)),
      UINT16_C(53720),
      simde_mm512_set_epi32(INT32_C( -215671891), INT32_C( 1165933857), INT32_C(  229060973), INT32_C(-1525363693),
                            INT32_C(-1670687960), INT32_C(  159553420), INT32_C( -802431529), INT32_C(-1862869598),
                            INT32_C(-1575511501), INT32_C( 1180622586), INT32_C( 1144679036), INT32_C( 1714786815),
                            INT32_C( 1463480444), INT32_C( 1916349718), INT32_C( -744963447), INT32_C(-1790859765)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C(  65), INT8_C(-128),
                        INT8_C( -32), INT8_C( -76), INT8_C(  70), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C(  36), INT8_C( 127),
                        INT8_C( 127), INT8_C( 126), INT8_C(  14), INT8_C(-117)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtsepi32_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtsepi32_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT16_C(22908),
      simde_mm512_set_epi32(INT32_C(  760893683), INT32_C(-2027734617), INT32_C( 1683947105), INT32_C( -424320007),
                            INT32_C(  107722959), INT32_C(  -13745640), INT32_C(-1276316442), INT32_C(-1722135079),
                            INT32_C( 1405989540), INT32_C(-1209232739), INT32_C(  265793764), INT32_C(-1369604571),
                            INT32_C( -598522698), INT32_C( 1529360806), INT32_C(   74319996), INT32_C(-1413496178)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(-128), INT8_C(   0), INT8_C(-128),
                        INT8_C( 127), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                        INT8_C(   0), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(55020),
      simde_mm512_set_epi32(INT32_C(  564176453), INT32_C( -258554655), INT32_C(-1141343253), INT32_C(  -63087550),
                            INT32_C( 1360677357), INT32_C(-1430308733), INT32_C( -736696329), INT32_C(  172239397),
                            INT32_C(  696452905), INT32_C( 1625371319), INT32_C(-1718938178), INT32_C(   73765108),
                            INT32_C(  -70378828), INT32_C(-1381806901), INT32_C(-1797606071), INT32_C(  433393018)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C(   0), INT8_C(-128),
                        INT8_C(   0), INT8_C(-128), INT8_C(-128), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(65048),
      simde_mm512_set_epi32(INT32_C(  -17747961), INT32_C( -126136440), INT32_C( -844801591), INT32_C(-1631191639),
                            INT32_C( 1959138923), INT32_C( 1255989970), INT32_C( 1743308784), INT32_C(-1532001195),
                            INT32_C( 1263353675), INT32_C( 1823260377), INT32_C( -521071455), INT32_C( -376568333),
                            INT32_C(-1920414563), INT32_C(-1600156196), INT32_C( -699126682), INT32_C(  420932189)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                        INT8_C(-128), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(46476),
      simde_mm512_set_epi32(INT32_C( 1491415646), INT32_C( -327353889), INT32_C(  445443000), INT32_C( 1288485236),
                            INT32_C(  697687640), INT32_C(  493902207), INT32_C(  123157162), INT32_C(  -87325004),
                            INT32_C(  -74930983), INT32_C( 1838315906), INT32_C(  -81455038), INT32_C( -220592487),
                            INT32_C( 1565085824), INT32_C(   60698648), INT32_C( 1617237598), INT32_C( 1724869908)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(   0), INT8_C( 127), INT8_C( 127),
                        INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C(-128),
                        INT8_C(-128), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(16227),
      simde_mm512_set_epi32(INT32_C(  143104805), INT32_C(  311770829), INT32_C(-1851657392), INT32_C(  834971347),
                            INT32_C(   96523257), INT32_C( 1786925287), INT32_C( -210779015), INT32_C( 1982975326),
                            INT32_C( 1450447838), INT32_C( 2089131393), INT32_C(-1680682639), INT32_C( -651728462),
                            INT32_C(-1976749038), INT32_C(-1119382179), INT32_C(  674486709), INT32_C(  -51017687)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C(   0), INT8_C( 127), INT8_C(-128), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C( 127), INT8_C(-128)) },
    { UINT16_C(19805),
      simde_mm512_set_epi32(INT32_C(  821368379), INT32_C( 1594494278), INT32_C(  958208413), INT32_C(-1231281503),
                            INT32_C(  509306793), INT32_C( 1685930534), INT32_C( -983619151), INT32_C(  293070963),
                            INT32_C( 1991494863), INT32_C( 1867270897), INT32_C(-1968548850), INT32_C(-1694953429),
                            INT32_C(  342647810), INT32_C(  475783090), INT32_C(-1901860575), INT32_C(-1695455832)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C(   0), INT8_C( 127),
                        INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C(   0), INT8_C(-128)) },
    { UINT16_C( 6853),
      simde_mm512_set_epi32(INT32_C(  720414379), INT32_C( -600382162), INT32_C( 1552024491), INT32_C(-1608825765),
                            INT32_C( 1791475411), INT32_C(  664867443), INT32_C(  674611871), INT32_C( 2098484402),
                            INT32_C(-1985583060), INT32_C(-2005775341), INT32_C(-1904713988), INT32_C(  -21726033),
                            INT32_C(  661481155), INT32_C( -401644059), INT32_C( 1015830537), INT32_C( 1655297629)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                        INT8_C( 127), INT8_C(   0), INT8_C( 127), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(-128), INT8_C(   0), INT8_C( 127)) },
    { UINT16_C(36847),
      simde_mm512_set_epi32(INT32_C(-2040986931), INT32_C( 2040118421), INT32_C(  936070741), INT32_C(  920738578),
                            INT32_C(-1305390023), INT32_C(-1363026646), INT32_C(  965440518), INT32_C(-1619252377),
                            INT32_C(-1356260869), INT32_C(-1069122338), INT32_C(  -23384622), INT32_C( 1344193938),
                            INT32_C(-2040024665), INT32_C(  961434729), INT32_C(-1271404275), INT32_C( 1259406574)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(   0),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtsepi32_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtepi64_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
      { simde_mm512_set_epi64(INT64_C( 3636972464222704255), INT64_C(-8637898597608066092),
                            INT64_C(-7117920839692631501), INT64_C(  779376322937895778),
                            INT64_C( 8625341086991316813), INT64_C( -619898035156678588),
                            INT64_C( 9102644088317736936), INT64_C( 6748691650780068291)),
      simde_mm256_set_epi32(INT32_C( -874573185), INT32_C(-1279659052), INT32_C( 1791312435), INT32_C( -351257758),
                            INT32_C(   78837581), INT32_C( 2038827076), INT32_C(  150796264), INT32_C( -547096125)) },
    { simde_mm512_set_epi64(INT64_C(-3188691098302098823), INT64_C(-2239829205413177691),
                            INT64_C(-3686324874761745207), INT64_C( 7163879657873596880),
                            INT64_C(-3722186173215781182), INT64_C( -688687161986027520),
                            INT64_C( -414049911340659363), INT64_C(-5760455419255020630)),
      simde_mm256_set_epi32(INT32_C(  725668473), INT32_C( 1842173605), INT32_C(-1323053879), INT32_C( -852237872),
                            INT32_C( 2107264706), INT32_C( -864849920), INT32_C(-1555952291), INT32_C(  -19618902)) },
    { simde_mm512_set_epi64(INT64_C(  954928457282272691), INT64_C( 6743095391785628647),
                            INT64_C(-4113183023138858226), INT64_C(-4216200208371002899),
                            INT64_C( 4862978039430736617), INT64_C(-8133245958833822305),
                            INT64_C(  422975927762384643), INT64_C(-2594689062389743569)),
      simde_mm256_set_epi32(INT32_C( 1513668019), INT32_C( 1240773607), INT32_C( 1101273870), INT32_C(-1239526931),
                            INT32_C(  -20223255), INT32_C( 1758733727), INT32_C(-2100201725), INT32_C(-2052390865)) },
    { simde_mm512_set_epi64(INT64_C( 1235418251988608652), INT64_C(-1697784138139039522),
                            INT64_C(-7844197282489602934), INT64_C( 8524715380160522696),
                            INT64_C( 2488078747749094790), INT64_C(  668997188312749400),
                            INT64_C( 5402489328274860019), INT64_C( 3255809389037965161)),
      simde_mm256_set_epi32(INT32_C( -892295540), INT32_C( 1424885982), INT32_C( -987796342), INT32_C( -305731128),
                            INT32_C( 1288213894), INT32_C( -182594216), INT32_C(-1565956109), INT32_C(-1443016855)) },
    { simde_mm512_set_epi64(INT64_C( 8211940214269115513), INT64_C(-1467524992070371089),
                            INT64_C( 7395547920392152513), INT64_C( 5720449131594366621),
                            INT64_C(-4149254972258420537), INT64_C(-2405712178673606191),
                            INT64_C(-3007967901025062583), INT64_C( 5561908781648105862)),
      simde_mm256_set_epi32(INT32_C(-1136558983), INT32_C(-2069677841), INT32_C( 2052744641), INT32_C( 1281343133),
                            INT32_C(  537098439), INT32_C(  -28040751), INT32_C(-2086686391), INT32_C(-1824337530)) },
    { simde_mm512_set_epi64(INT64_C(  380636487945477944), INT64_C( 4631524987567251326),
                            INT64_C( 2584519444747767308), INT64_C(  816258266807653524),
                            INT64_C(-1187714861805569615), INT64_C(-4379934582538880391),
                            INT64_C(-4279816909313388105), INT64_C(-2848166400408167703)),
      simde_mm256_set_epi32(INT32_C( 2089475896), INT32_C( -196436098), INT32_C( -616051188), INT32_C(-1509174124),
                            INT32_C( 1067481521), INT32_C(  -61719943), INT32_C( 1219121591), INT32_C(-1032332567)) },
    { simde_mm512_set_epi64(INT64_C(-7496005764999359846), INT64_C(-8729150684960000613),
                            INT64_C( -490843546754529526), INT64_C(-4780112060131952140),
                            INT64_C( 3884257663950773205), INT64_C(-6711214101553505106),
                            INT64_C(-7275075039360044698), INT64_C( -663449602597245794)),
      simde_mm256_set_epi32(INT32_C( -420556134), INT32_C( 2014967195), INT32_C(  325535498), INT32_C( 1296447988),
                            INT32_C(   65002453), INT32_C(  156212398), INT32_C( -190697114), INT32_C(  -19845986)) },
    { simde_mm512_set_epi64(INT64_C(-3417201024126416880), INT64_C(-6294431374261990633),
                            INT64_C(-8960361343509742295), INT64_C(-4865505548535460298),
                            INT64_C( 9141410768359557548), INT64_C(  154894143831828371),
                            INT64_C(-9081419987100634516), INT64_C(-5334041650922884862)),
      simde_mm256_set_epi32(INT32_C(  249038864), INT32_C(-1307948265), INT32_C(   83213609), INT32_C( 1859784246),
                            INT32_C(  436242860), INT32_C(  887252883), INT32_C(  482079340), INT32_C(  597324034)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_cvtepi64_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtepi64_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m256i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(  262030369), INT32_C(  472381389), INT32_C( 1820292769), INT32_C( 1183923792),
                            INT32_C(-1208341183), INT32_C(-1257970130), INT32_C(  847221831), INT32_C(  575663548)),
      UINT8_C( 10),
      simde_mm512_set_epi64(INT64_C(-7966868535937805932), INT64_C( 5299008656095408483),
                            INT64_C(-5590285311753850045), INT64_C( 7137964584137341662),
                            INT64_C(-6434834176816605387), INT64_C(-6333271890954486833),
                            INT64_C( 3282767373978865693), INT64_C( 4967684130096997469)),
      simde_mm256_set_epi32(INT32_C(  262030369), INT32_C(  472381389), INT32_C( 1820292769), INT32_C( 1183923792),
                            INT32_C( -833775819), INT32_C(-1257970130), INT32_C(-1505718243), INT32_C(  575663548)) },
    { simde_mm256_set_epi32(INT32_C(-1816200207), INT32_C(  105537387), INT32_C( 1059695632), INT32_C(  611774133),
                            INT32_C( 1637079250), INT32_C( -358018783), INT32_C( 1261913751), INT32_C( -501053537)),
      UINT8_C( 45),
      simde_mm512_set_epi64(INT64_C(-1802035461684441265), INT64_C( 1464385524221546353),
                            INT64_C( 2980222649966367123), INT64_C( 8099038506776411426),
                            INT64_C(  905145040348223898), INT64_C(-2812330085540872295),
                            INT64_C( 9037878101616659294), INT64_C(   45028518261476449)),
      simde_mm256_set_epi32(INT32_C(-1816200207), INT32_C(  105537387), INT32_C( -758017645), INT32_C(  611774133),
                            INT32_C( 1479121306), INT32_C(   70336409), INT32_C( 1261913751), INT32_C( -474166175)) },
    { simde_mm256_set_epi32(INT32_C( 1160949832), INT32_C(  790671825), INT32_C( 1609551524), INT32_C( 1396807876),
                            INT32_C(-1711874318), INT32_C(-1489515049), INT32_C( 1928745488), INT32_C( 1062430982)),
      UINT8_C( 64),
      simde_mm512_set_epi64(INT64_C(-9005930620427138564), INT64_C(-2261531819996514214),
                            INT64_C(-1908590373704801150), INT64_C(-8335280881430651030),
                            INT64_C(-7726592629601528093), INT64_C(  557495353260737980),
                            INT64_C( 8816092406723833767), INT64_C( 1087345366514775540)),
      simde_mm256_set_epi32(INT32_C( 1160949832), INT32_C(-1260554150), INT32_C( 1609551524), INT32_C( 1396807876),
                            INT32_C(-1711874318), INT32_C(-1489515049), INT32_C( 1928745488), INT32_C( 1062430982)) },
    { simde_mm256_set_epi32(INT32_C( -921525147), INT32_C( 1970381634), INT32_C(-1500476692), INT32_C(-1962686032),
                            INT32_C( -498932433), INT32_C( 1322625139), INT32_C( 2144788778), INT32_C(  686983052)),
      UINT8_C( 43),
      simde_mm512_set_epi64(INT64_C(  339079297327618442), INT64_C( -530434709005010771),
                            INT64_C( 3893900520561141682), INT64_C( 6261327879404285016),
                            INT64_C(-4792365276005063244), INT64_C(  139860979584315670),
                            INT64_C( 1098126303565167243), INT64_C(-6673986251972702615)),
      simde_mm256_set_epi32(INT32_C( -921525147), INT32_C( 1970381634), INT32_C(-2034145358), INT32_C(-1962686032),
                            INT32_C(  701310388), INT32_C( 1322625139), INT32_C(  310622859), INT32_C(  928222825)) },
    { simde_mm256_set_epi32(INT32_C( 1215918679), INT32_C( 1464765810), INT32_C( -508860334), INT32_C(  370237422),
                            INT32_C( 2136016247), INT32_C( -652356139), INT32_C(  878885802), INT32_C( 1025777926)),
      UINT8_C(125),
      simde_mm512_set_epi64(INT64_C( 1292112428628168131), INT64_C( 3996200026511053278),
                            INT64_C(-1183464275819018712), INT64_C( 7009328331734920985),
                            INT64_C( 1248957650094672934), INT64_C(-3647207441389629252),
                            INT64_C( 5572165573184689325), INT64_C( 4950883391543296907)),
      simde_mm256_set_epi32(INT32_C( 1215918679), INT32_C( 1469649374), INT32_C( -344898008), INT32_C(  254029593),
                            INT32_C(-1855566810), INT32_C( -506507076), INT32_C(  878885802), INT32_C(-1252932725)) },
    { simde_mm256_set_epi32(INT32_C(  352792784), INT32_C(-1695146469), INT32_C(-1794219354), INT32_C(  767902659),
                            INT32_C(-1756346957), INT32_C( 1345243832), INT32_C( -845357564), INT32_C(  382403555)),
      UINT8_C(222),
      simde_mm512_set_epi64(INT64_C( 8625819089946530671), INT64_C( 1260476417816797254),
                            INT64_C( 6695601616901705840), INT64_C(-3241707820207032037),
                            INT64_C( 2724457343546367531), INT64_C( -510428823621027398),
                            INT64_C(-1373332270819176988), INT64_C( 8119817203721462534)),
      simde_mm256_set_epi32(INT32_C(-1056189585), INT32_C(-1413865402), INT32_C(-1794219354), INT32_C(-1359726309),
                            INT32_C( -406249941), INT32_C( 1962064314), INT32_C( 1866337764), INT32_C(  382403555)) },
    { simde_mm256_set_epi32(INT32_C(  294146054), INT32_C( 1809922858), INT32_C(   26511597), INT32_C( -735526333),
                            INT32_C( -676610806), INT32_C( -291478152), INT32_C(-1583953722), INT32_C(-1538734134)),
      UINT8_C( 12),
      simde_mm512_set_epi64(INT64_C(  972095241651198809), INT64_C( 2070126427207517639),
                            INT64_C( 2191096336057470646), INT64_C(  728519017029928955),
                            INT64_C(-8870108936041189470), INT64_C(-3116006297117081948),
                            INT64_C( 1324483146372913362), INT64_C(-3117177062817132908)),
      simde_mm256_set_epi32(INT32_C(  294146054), INT32_C( 1809922858), INT32_C(   26511597), INT32_C( -735526333),
                            INT32_C(  608120738), INT32_C(  913620644), INT32_C(-1583953722), INT32_C(-1538734134)) },
    { simde_mm256_set_epi32(INT32_C( 1667809857), INT32_C( 1891439881), INT32_C(  486588760), INT32_C( 1276336231),
                            INT32_C(-1279253460), INT32_C(  -53345180), INT32_C( -873798020), INT32_C(  302688693)),
      UINT8_C(103),
      simde_mm512_set_epi64(INT64_C(-3227093025788146904), INT64_C(  779144960212675434),
                            INT64_C( -398606095670776166), INT64_C(-3328435871470452771),
                            INT64_C( 6940472738149975738), INT64_C( 2141675531954249559),
                            INT64_C(  283583591498109395), INT64_C(  106915262196291640)),
      simde_mm256_set_epi32(INT32_C( 1667809857), INT32_C(-1778177174), INT32_C(  775745178), INT32_C( 1276336231),
                            INT32_C(-1279253460), INT32_C(-1706598569), INT32_C(  593482195), INT32_C( 1346836536)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_mask_cvtepi64_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtepi64_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { UINT8_C(188),
      simde_mm512_set_epi64(INT64_C(-6434834176816605387), INT64_C(-6333271890954486833),
                            INT64_C( 3282767373978865693), INT64_C( 4967684130096997469),
                            INT64_C(-6489573770131520479), INT64_C( 2028862618814346913),
                            INT64_C( 5084913970682932545), INT64_C(-5402940566847646649)),
      simde_mm256_set_epi32(INT32_C( -833775819), INT32_C(          0), INT32_C(-1505718243), INT32_C( 2059854941),
                            INT32_C(  262030369), INT32_C( 1820292769), INT32_C(          0), INT32_C(          0)) },
    { UINT8_C(222),
      simde_mm512_set_epi64(INT64_C( 7219165944822561777), INT64_C(  453279626729991184),
                            INT64_C( 2627549895410833618), INT64_C(-1537678963076807017),
                            INT64_C(-2152008552520089786), INT64_C(-2364159369550373812),
                            INT64_C( 2326515355522650400), INT64_C(-6353467178539342600)),
      simde_mm256_set_epi32(INT32_C(-1816200207), INT32_C( 1059695632), INT32_C(          0), INT32_C( 1261913751),
                            INT32_C(-1854931130), INT32_C( 1233771596), INT32_C(-1301589728), INT32_C(          0)) },
    { UINT8_C( 97),
      simde_mm512_set_epi64(INT64_C( 4563106325822562891), INT64_C(-7952143840967488800),
                            INT64_C( 2986747956576379061), INT64_C(-3255660993180233572),
                            INT64_C(-9034716586009708386), INT64_C( 6352777639726979248),
                            INT64_C(  302092578477375039), INT64_C(-6186666752246548174)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(  340953824), INT32_C(  693887157), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(   10484018)) },
    { UINT8_C( 16),
      simde_mm512_set_epi64(INT64_C( 6284948560106856688), INT64_C(-3714600103307337967),
                            INT64_C(-1222298779301376683), INT64_C(-7941271492122704167),
                            INT64_C(-8122323343088436160), INT64_C( 4986241561527366097),
                            INT64_C( 6912971158203766980), INT64_C(-7352444207866851881)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(  253167321),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT8_C( 51),
      simde_mm512_set_epi64(INT64_C( 3986686679770014763), INT64_C(-3957920366836210878),
                            INT64_C(-6444498318217983568), INT64_C(-2142898481326086029),
                            INT64_C( 9211797659024787340), INT64_C(-9005930620427138564),
                            INT64_C(-2261531819996514214), INT64_C(-1908590373704801150)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(-1962686032), INT32_C( 1322625139),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1260554150), INT32_C( -778541950)) },
    { UINT8_C(221),
      simde_mm512_set_epi64(INT64_C( 3774785777534509318), INT64_C(  339079297327618442),
                            INT64_C( -530434709005010771), INT64_C( 3893900520561141682),
                            INT64_C( 6261327879404285016), INT64_C(-4792365276005063244),
                            INT64_C(  139860979584315670), INT64_C( 1098126303565167243)),
      simde_mm256_set_epi32(INT32_C( 1025777926), INT32_C(  199678346), INT32_C(          0), INT32_C(-2034145358),
                            INT32_C( 1243901016), INT32_C(  701310388), INT32_C(          0), INT32_C(  310622859)) },
    { UINT8_C(213),
      simde_mm512_set_epi64(INT64_C( 7009328331734920985), INT64_C( 1248957650094672934),
                            INT64_C(-3647207441389629252), INT64_C( 5572165573184689325),
                            INT64_C( 4950883391543296907), INT64_C(-3417026514343333289),
                            INT64_C( 6291121254035056722), INT64_C( 1590157621381367159)),
      simde_mm256_set_epi32(INT32_C(  254029593), INT32_C(-1855566810), INT32_C(          0), INT32_C( -212495187),
                            INT32_C(          0), INT32_C( 1215918679), INT32_C(          0), INT32_C( 2136016247)) },
    { UINT8_C( 40),
      simde_mm512_set_epi64(INT64_C( 8119817203721462534), INT64_C( 4481656528138875088),
                            INT64_C(-7280598643784129882), INT64_C( 3298116809455060403),
                            INT64_C( 5777778267035328004), INT64_C( 1642410762899980648),
                            INT64_C( 7944294506037598912), INT64_C( 6312096001936293250)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(-1794219354), INT32_C(          0),
                            INT32_C( -845357564), INT32_C(          0), INT32_C(          0), INT32_C(          0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_maskz_cvtepi64_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtsepi32_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -633826313), INT32_C( -624858207), INT32_C( 1490386470), INT32_C(-2098903851),
                            INT32_C(-1539984349), INT32_C( 1958536651), INT32_C(-1468703883), INT32_C( -330293651),
                            INT32_C( 2111698546), INT32_C(-1712476271), INT32_C( 1928035775), INT32_C(  815855626),
                            INT32_C(-1671100889), INT32_C(-1198252639), INT32_C(-1543874579), INT32_C(  564524240)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi32(INT32_C( 1797360798), INT32_C(  205569425), INT32_C( 1156136221), INT32_C( 1772840561),
                            INT32_C( 2126360842), INT32_C( 1592304637), INT32_C( 1062735606), INT32_C( -473740091),
                            INT32_C( 1443682851), INT32_C( -765250611), INT32_C( -738272039), INT32_C( 1732082686),
                            INT32_C( -881631866), INT32_C(  -99399017), INT32_C(  252546807), INT32_C(-1695736037)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768)) },
    { simde_mm512_set_epi32(INT32_C( -260564878), INT32_C(  427961207), INT32_C(-1056239301), INT32_C( 1225809734),
                            INT32_C(  213123881), INT32_C(  379025644), INT32_C( 1973103858), INT32_C(-2053346996),
                            INT32_C(-1954883006), INT32_C(  667947760), INT32_C(  184139306), INT32_C(  168675179),
                            INT32_C(-1651207657), INT32_C(-1932451475), INT32_C(  207773420), INT32_C(-1587464239)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768)) },
    { simde_mm512_set_epi32(INT32_C( -301891179), INT32_C( 1118988231), INT32_C(  -81115331), INT32_C(   95289597),
                            INT32_C(   79711365), INT32_C( -252265140), INT32_C(-1901421271), INT32_C(  -86214221),
                            INT32_C(  494361026), INT32_C( 1393707090), INT32_C( 1367330444), INT32_C( 1529415750),
                            INT32_C( -219576728), INT32_C(  785923707), INT32_C(-1615246269), INT32_C( -860551446)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768)) },
    { simde_mm512_set_epi32(INT32_C( -668182583), INT32_C(  778563571), INT32_C( 1803655402), INT32_C(  668307371),
                            INT32_C(  166146078), INT32_C( 2089905810), INT32_C(     540370), INT32_C( -610320208),
                            INT32_C( -644851790), INT32_C( 1731401690), INT32_C( 1162165084), INT32_C( -702531425),
                            INT32_C( 1412040872), INT32_C( -506509249), INT32_C(-1429255709), INT32_C( 1954724351)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi32(INT32_C(  895849603), INT32_C( 1425609255), INT32_C(  649927889), INT32_C( 1727611756),
                            INT32_C( 1297009704), INT32_C(-1218308449), INT32_C(-1305018130), INT32_C( 1967276695),
                            INT32_C( -413179024), INT32_C(-1726322338), INT32_C( 1840183236), INT32_C(  284256212),
                            INT32_C( 1788242258), INT32_C(  806002384), INT32_C( -365421949), INT32_C(-2081268803)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768)) },
    { simde_mm512_set_epi32(INT32_C( -850026299), INT32_C(  763115562), INT32_C( -698858672), INT32_C(-2063362395),
                            INT32_C(-1349430084), INT32_C(  963716730), INT32_C( 1798713564), INT32_C(-1893101926),
                            INT32_C(  662898969), INT32_C(  188387073), INT32_C( -332183490), INT32_C(-1234385790),
                            INT32_C(  839752488), INT32_C( 1421488669), INT32_C(  544371590), INT32_C(  832408505)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767)) },
    { simde_mm512_set_epi32(INT32_C( 1070222476), INT32_C( 1830989582), INT32_C(  732250943), INT32_C(-1460385002),
                            INT32_C(-1298211555), INT32_C(-1342658291), INT32_C(  646581591), INT32_C(-2104719353),
                            INT32_C( 1887679149), INT32_C( 1571180495), INT32_C( 1241438931), INT32_C(   97459949),
                            INT32_C( 1558770394), INT32_C(-1780666235), INT32_C( 1621650638), INT32_C( -918750867)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_cvtsepi32_epi16(test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtsepi32_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m256i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(  4055), INT16_C(-20252), INT16_C(-20899), INT16_C( 32293),
                            INT16_C( -9133), INT16_C( 17590), INT16_C( 23336), INT16_C( 12710),
                            INT16_C(  1134), INT16_C(  2172), INT16_C(-21569), INT16_C(-15730),
                            INT16_C( 28193), INT16_C( -7062), INT16_C( 27494), INT16_C( 22908)),
      UINT16_C(37533),
      simde_mm512_set_epi32(INT32_C(-1718938178), INT32_C(   73765108), INT32_C(  -70378828), INT32_C(-1381806901),
                            INT32_C(-1797606071), INT32_C(  433393018), INT32_C(  712958933), INT32_C( -930490644),
                            INT32_C(  760893683), INT32_C(-2027734617), INT32_C( 1683947105), INT32_C( -424320007),
                            INT32_C(  107722959), INT32_C(  -13745640), INT32_C(-1276316442), INT32_C(-1722135079)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-20252), INT16_C(-20899), INT16_C(-32768),
                            INT16_C( -9133), INT16_C( 17590), INT16_C( 32767), INT16_C( 12710),
                            INT16_C( 32767), INT16_C(  2172), INT16_C(-21569), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(-32768), INT16_C( 27494), INT16_C(-32768)) },
    { simde_mm256_set_epi16(INT16_C(-17416), INT16_C( 31723), INT16_C(  -963), INT16_C( 23618),
                            INT16_C( 20762), INT16_C( 18925), INT16_C(-21825), INT16_C( 14467),
                            INT16_C(-11242), INT16_C( -6153), INT16_C(  2628), INT16_C( 10789),
                            INT16_C( 10627), INT16_C(  1833), INT16_C( 24801), INT16_C( 12983)),
      UINT16_C(50401),
      simde_mm512_set_epi32(INT32_C( -844801591), INT32_C(-1631191639), INT32_C( 1959138923), INT32_C( 1255989970),
                            INT32_C( 1743308784), INT32_C(-1532001195), INT32_C( 1263353675), INT32_C( 1823260377),
                            INT32_C( -521071455), INT32_C( -376568333), INT32_C(-1920414563), INT32_C(-1600156196),
                            INT32_C( -699126682), INT32_C(  420932189), INT32_C(  368322579), INT32_C(  983236120)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C(  -963), INT16_C( 23618),
                            INT16_C( 20762), INT16_C(-32768), INT16_C(-21825), INT16_C( 14467),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C( 10789),
                            INT16_C( 10627), INT16_C(  1833), INT16_C( 24801), INT16_C( 32767)) },
    { simde_mm256_set_epi16(INT16_C( 23881), INT16_C( 20608), INT16_C(   926), INT16_C( 12312),
                            INT16_C( 24677), INT16_C(  5726), INT16_C( 26319), INT16_C( 27924),
                            INT16_C( 26951), INT16_C(  1434), INT16_C(-21095), INT16_C(-19060),
                            INT16_C(  -271), INT16_C( 12295), INT16_C( -1925), INT16_C( 20360)),
      UINT16_C( 1689),
      simde_mm512_set_epi32(INT32_C(-1976749038), INT32_C(-1119382179), INT32_C(  674486709), INT32_C(  -51017687),
                            INT32_C(  -85403955), INT32_C( -771014813), INT32_C( 1491415646), INT32_C( -327353889),
                            INT32_C(  445443000), INT32_C( 1288485236), INT32_C(  697687640), INT32_C(  493902207),
                            INT32_C(  123157162), INT32_C(  -87325004), INT32_C(  -74930983), INT32_C( 1838315906)),
      simde_mm256_set_epi16(INT16_C( 23881), INT16_C( 20608), INT16_C(   926), INT16_C( 12312),
                            INT16_C( 24677), INT16_C(-32768), INT16_C( 32767), INT16_C( 27924),
                            INT16_C( 32767), INT16_C(  1434), INT16_C(-21095), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 12295), INT16_C( -1925), INT16_C( 32767)) },
    { simde_mm256_set_epi16(INT16_C(  1472), INT16_C(-11271), INT16_C( 27266), INT16_C( 20711),
                            INT16_C( -3217), INT16_C(-15239), INT16_C( 30257), INT16_C(-12962),
                            INT16_C( 22132), INT16_C(  5086), INT16_C( 31877), INT16_C(-25215),
                            INT16_C(-25646), INT16_C(-11919), INT16_C( -9945), INT16_C( 27058)),
      UINT16_C(42707),
      simde_mm512_set_epi32(INT32_C(  509306793), INT32_C( 1685930534), INT32_C( -983619151), INT32_C(  293070963),
                            INT32_C( 1991494863), INT32_C( 1867270897), INT32_C(-1968548850), INT32_C(-1694953429),
                            INT32_C(  342647810), INT32_C(  475783090), INT32_C(-1901860575), INT32_C(-1695455832),
                            INT32_C(-1163448552), INT32_C( 1727941981), INT32_C(  143104805), INT32_C(  311770829)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C(-11271), INT16_C(-32768), INT16_C( 20711),
                            INT16_C( -3217), INT16_C( 32767), INT16_C(-32768), INT16_C(-12962),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 31877), INT16_C(-32768),
                            INT16_C(-25646), INT16_C(-11919), INT16_C( 32767), INT16_C( 32767)) },
    { simde_mm256_set_epi16(INT16_C( 15500), INT16_C( 22537), INT16_C( 25257), INT16_C(-10659),
                            INT16_C( 27500), INT16_C(-14672), INT16_C(-18535), INT16_C(  6853),
                            INT16_C( 12533), INT16_C(  5691), INT16_C( 24330), INT16_C(  3398),
                            INT16_C( 14621), INT16_C(  6557), INT16_C(-18788), INT16_C(  8865)),
      UINT16_C(26085),
      simde_mm512_set_epi32(INT32_C(-1271404275), INT32_C( 1259406574), INT32_C(-1600364807), INT32_C( 1013354479),
                            INT32_C(  720414379), INT32_C( -600382162), INT32_C( 1552024491), INT32_C(-1608825765),
                            INT32_C( 1791475411), INT32_C(  664867443), INT32_C(  674611871), INT32_C( 2098484402),
                            INT32_C(-1985583060), INT32_C(-2005775341), INT32_C(-1904713988), INT32_C(  -21726033)),
      simde_mm256_set_epi16(INT16_C( 15500), INT16_C( 32767), INT16_C(-32768), INT16_C(-10659),
                            INT16_C( 27500), INT16_C(-32768), INT16_C(-18535), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(  3398),
                            INT16_C( 14621), INT16_C(-32768), INT16_C(-18788), INT16_C(-32768)) },
    { simde_mm256_set_epi16(INT16_C( 14731), INT16_C( 29702), INT16_C(-24708), INT16_C( 11111),
                            INT16_C(-20695), INT16_C(  6651), INT16_C(-16314), INT16_C( 31966),
                            INT16_C(  -357), INT16_C( 11730), INT16_C( 20510), INT16_C(-14958),
                            INT16_C(-31129), INT16_C(-20057), INT16_C( 14670), INT16_C( 21609)),
      UINT16_C(56618),
      simde_mm512_set_epi32(INT32_C(-1119345439), INT32_C(  796790411), INT32_C( -897513851), INT32_C(  631520322),
                            INT32_C(  634536018), INT32_C(-2012793591), INT32_C( 1627606334), INT32_C(-1721387558),
                            INT32_C(-1230866277), INT32_C(-1212419278), INT32_C(  577821894), INT32_C( 1429584547),
                            INT32_C(-2040986931), INT32_C( 2040118421), INT32_C(  936070741), INT32_C(  920738578)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-24708), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-16314), INT16_C(-32768),
                            INT16_C(  -357), INT16_C( 11730), INT16_C( 32767), INT16_C(-14958),
                            INT16_C(-32768), INT16_C(-20057), INT16_C( 32767), INT16_C( 21609)) },
    { simde_mm256_set_epi16(INT16_C( 22330), INT16_C( -3972), INT16_C( 29241), INT16_C( 11542),
                            INT16_C(-11368), INT16_C(-15735), INT16_C(-27327), INT16_C(-23029),
                            INT16_C( -8383), INT16_C(-32273), INT16_C(  3895), INT16_C(-11816),
                            INT16_C(-27188), INT16_C( 16734), INT16_C( -8012), INT16_C( 18091)),
      UINT16_C(37375),
      simde_mm512_set_epi32(INT32_C( -221392236), INT32_C( -711063988), INT32_C( 2133711027), INT32_C(  509873922),
                            INT32_C( 1729615213), INT32_C( -831792372), INT32_C( -215671891), INT32_C( 1165933857),
                            INT32_C(  229060973), INT32_C(-1525363693), INT32_C(-1670687960), INT32_C(  159553420),
                            INT32_C( -802431529), INT32_C(-1862869598), INT32_C(-1575511501), INT32_C( 1180622586)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( -3972), INT16_C( 29241), INT16_C( 32767),
                            INT16_C(-11368), INT16_C(-15735), INT16_C(-27327), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm256_set_epi16(INT16_C(  -456), INT16_C( -6277), INT16_C(-13154), INT16_C( 20424),
                            INT16_C(  7701), INT16_C(  9153), INT16_C( 18208), INT16_C(  1720),
                            INT16_C(-14211), INT16_C( 26958), INT16_C( 27767), INT16_C(-15259),
                            INT16_C(-16868), INT16_C(-31112), INT16_C( -6815), INT16_C( 21503)),
      UINT16_C(29475),
      simde_mm512_set_epi32(INT32_C(  441964111), INT32_C(-1041336788), INT32_C( 1240459905), INT32_C( 1190379131),
                            INT32_C(  660048661), INT32_C(  992314379), INT32_C(   13012539), INT32_C(-1751743734),
                            INT32_C(-1323762798), INT32_C(  640110239), INT32_C( 1866432721), INT32_C(-1867947321),
                            INT32_C(-2066921506), INT32_C(  704072659), INT32_C(  929476148), INT32_C(-1455589491)),
      simde_mm256_set_epi16(INT16_C(  -456), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(  7701), INT16_C(  9153), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-14211), INT16_C( 26958), INT16_C( 32767), INT16_C(-15259),
                            INT16_C(-16868), INT16_C(-31112), INT16_C( 32767), INT16_C(-32768)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_mask_cvtsepi32_epi16(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtsepi32_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { UINT16_C(22908),
      simde_mm512_set_epi32(INT32_C(  760893683), INT32_C(-2027734617), INT32_C( 1683947105), INT32_C( -424320007),
                            INT32_C(  107722959), INT32_C(  -13745640), INT32_C(-1276316442), INT32_C(-1722135079),
                            INT32_C( 1405989540), INT32_C(-1209232739), INT32_C(  265793764), INT32_C(-1369604571),
                            INT32_C( -598522698), INT32_C( 1529360806), INT32_C(   74319996), INT32_C(-1413496178)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(-32768), INT16_C(     0), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(     0), INT16_C(     0), INT16_C(-32768),
                            INT16_C(     0), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(55020),
      simde_mm512_set_epi32(INT32_C(  564176453), INT32_C( -258554655), INT32_C(-1141343253), INT32_C(  -63087550),
                            INT32_C( 1360677357), INT32_C(-1430308733), INT32_C( -736696329), INT32_C(  172239397),
                            INT32_C(  696452905), INT32_C( 1625371319), INT32_C(-1718938178), INT32_C(   73765108),
                            INT32_C(  -70378828), INT32_C(-1381806901), INT32_C(-1797606071), INT32_C(  433393018)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C(-32768), INT16_C(     0), INT16_C(-32768),
                            INT16_C(     0), INT16_C(-32768), INT16_C(-32768), INT16_C(     0),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C(     0),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(65048),
      simde_mm512_set_epi32(INT32_C(  -17747961), INT32_C( -126136440), INT32_C( -844801591), INT32_C(-1631191639),
                            INT32_C( 1959138923), INT32_C( 1255989970), INT32_C( 1743308784), INT32_C(-1532001195),
                            INT32_C( 1263353675), INT32_C( 1823260377), INT32_C( -521071455), INT32_C( -376568333),
                            INT32_C(-1920414563), INT32_C(-1600156196), INT32_C( -699126682), INT32_C(  420932189)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(     0),
                            INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(     0), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(46476),
      simde_mm512_set_epi32(INT32_C( 1491415646), INT32_C( -327353889), INT32_C(  445443000), INT32_C( 1288485236),
                            INT32_C(  697687640), INT32_C(  493902207), INT32_C(  123157162), INT32_C(  -87325004),
                            INT32_C(  -74930983), INT32_C( 1838315906), INT32_C(  -81455038), INT32_C( -220592487),
                            INT32_C( 1565085824), INT32_C(   60698648), INT32_C( 1617237598), INT32_C( 1724869908)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C(     0), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(     0), INT16_C( 32767), INT16_C(     0), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(     0), INT16_C(     0), INT16_C(     0),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(16227),
      simde_mm512_set_epi32(INT32_C(  143104805), INT32_C(  311770829), INT32_C(-1851657392), INT32_C(  834971347),
                            INT32_C(   96523257), INT32_C( 1786925287), INT32_C( -210779015), INT32_C( 1982975326),
                            INT32_C( 1450447838), INT32_C( 2089131393), INT32_C(-1680682639), INT32_C( -651728462),
                            INT32_C(-1976749038), INT32_C(-1119382179), INT32_C(  674486709), INT32_C(  -51017687)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(-32768), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767),
                            INT16_C(     0), INT16_C( 32767), INT16_C(-32768), INT16_C(     0),
                            INT16_C(     0), INT16_C(     0), INT16_C( 32767), INT16_C(-32768)) },
    { UINT16_C(19805),
      simde_mm512_set_epi32(INT32_C(  821368379), INT32_C( 1594494278), INT32_C(  958208413), INT32_C(-1231281503),
                            INT32_C(  509306793), INT32_C( 1685930534), INT32_C( -983619151), INT32_C(  293070963),
                            INT32_C( 1991494863), INT32_C( 1867270897), INT32_C(-1968548850), INT32_C(-1694953429),
                            INT32_C(  342647810), INT32_C(  475783090), INT32_C(-1901860575), INT32_C(-1695455832)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C( 32767), INT16_C(     0), INT16_C(     0),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(     0), INT16_C( 32767),
                            INT16_C(     0), INT16_C( 32767), INT16_C(     0), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(     0), INT16_C(-32768)) },
    { UINT16_C( 6853),
      simde_mm512_set_epi32(INT32_C(  720414379), INT32_C( -600382162), INT32_C( 1552024491), INT32_C(-1608825765),
                            INT32_C( 1791475411), INT32_C(  664867443), INT32_C(  674611871), INT32_C( 2098484402),
                            INT32_C(-1985583060), INT32_C(-2005775341), INT32_C(-1904713988), INT32_C(  -21726033),
                            INT32_C(  661481155), INT32_C( -401644059), INT32_C( 1015830537), INT32_C( 1655297629)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(     0), INT16_C( 32767), INT16_C(     0),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(     0), INT16_C(     0),
                            INT16_C(     0), INT16_C(-32768), INT16_C(     0), INT16_C( 32767)) },
    { UINT16_C(36847),
      simde_mm512_set_epi32(INT32_C(-2040986931), INT32_C( 2040118421), INT32_C(  936070741), INT32_C(  920738578),
                            INT32_C(-1305390023), INT32_C(-1363026646), INT32_C(  965440518), INT32_C(-1619252377),
                            INT32_C(-1356260869), INT32_C(-1069122338), INT32_C(  -23384622), INT32_C( 1344193938),
                            INT32_C(-2040024665), INT32_C(  961434729), INT32_C(-1271404275), INT32_C( 1259406574)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(     0), INT16_C(     0), INT16_C(     0),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C(     0),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_maskz_cvtsepi32_epi16(test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtsepi64_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
     { simde_mm512_set_epi64(INT64_C( 5688878986790062607), INT64_C( -507006338933993777),
                            INT64_C( 2731700857838766689), INT64_C( 5038766546414012764),
                            INT64_C( 8031668245477288096), INT64_C( 8558843731862564067),
                            INT64_C( 1445459848667085535), INT64_C(  -83609451038325995)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(-128)) },
    { simde_mm512_set_epi64(INT64_C( 8541199063845314870), INT64_C( 6183904058691300593),
                            INT64_C( 4987917255037318931), INT64_C(-6461883516210235787),
                            INT64_C(-2290177652468594952), INT64_C( -292256871616436513),
                            INT64_C( 8655434451305349549), INT64_C( 2098144080369590904)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C( 127)) },
    { simde_mm512_set_epi64(INT64_C( 8921116119188857611), INT64_C(-7689618306950712122),
                            INT64_C( 1518593000980919604), INT64_C( 4285824591643310377),
                            INT64_C(-5934398452121416225), INT64_C( 3925581736765257012),
                            INT64_C(-1280589966661740296), INT64_C(-6610380819072272984)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi64(INT64_C(-7117386000751069153), INT64_C(-4042166984900833962),
                            INT64_C( 5586053217269340515), INT64_C(-7584975210127513398),
                            INT64_C( 3657404296828616096), INT64_C(-2770219948143520502),
                            INT64_C( 4495347402256698165), INT64_C( -398794429961863290)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128)) },
    { simde_mm512_set_epi64(INT64_C(-7309298589786727464), INT64_C( 6351016797136664888),
                            INT64_C( -796932205202576292), INT64_C( 5974868289408841942),
                            INT64_C(-5246045528720698174), INT64_C( 5897717133320969396),
                            INT64_C( 8855007060554502516), INT64_C( 1795078486534561817)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C( 127)) },
    { simde_mm512_set_epi64(INT64_C(-8235362769944622173), INT64_C(-1394979073669202107),
                            INT64_C(-1990341901794640861), INT64_C(-5381553921365904442),
                            INT64_C( 2258222102802041167), INT64_C( 8110919934642332525),
                            INT64_C( 3859146607814864086), INT64_C( 5989226152769229183)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C( 127)) },
    { simde_mm512_set_epi64(INT64_C( 7694422639508462949), INT64_C(-2072064725478552758),
                            INT64_C( 3296208651842008367), INT64_C(-5674066605955291159),
                            INT64_C( 5400731755389512823), INT64_C( 4001845542455980591),
                            INT64_C( 1263947732552642640), INT64_C( 1152590716152552703)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C( 127)) },
    { simde_mm512_set_epi64(INT64_C(-5372614012128921165), INT64_C(-2041172040005057172),
                            INT64_C(-8233280709947200170), INT64_C( 1449409854914395254),
                            INT64_C( 6224671476337807114), INT64_C( 4022680139615958951),
                            INT64_C( -277976409085756360), INT64_C(-4710983618327201696)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtsepi64_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtsepi64_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( -98), INT8_C(-118), INT8_C( -55), INT8_C(  84),
                        INT8_C( -38), INT8_C(-100), INT8_C(  69), INT8_C( -11),
                        INT8_C( -76), INT8_C( -44), INT8_C( 111), INT8_C(  57),
                        INT8_C(-108), INT8_C( 117), INT8_C( -94), INT8_C(  99)),
      UINT8_C( 89),
      simde_mm512_set_epi64(INT64_C(-6464916445333598184), INT64_C(-9074050598773963479),
                            INT64_C(-6773163771856001287), INT64_C( 8528895860955669022),
                            INT64_C(  743395091751495893), INT64_C(-5753646205421077345),
                            INT64_C( 6781795189594641427), INT64_C( 7308036127418939455)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -76), INT8_C(-128), INT8_C( 111), INT8_C( 127),
                        INT8_C( 127), INT8_C( 117), INT8_C( -94), INT8_C( 127)) },
    { simde_mm_set_epi8(INT8_C( -29), INT8_C( -88), INT8_C(  59), INT8_C( -11),
                        INT8_C(-100), INT8_C(-112), INT8_C(  21), INT8_C( -50),
                        INT8_C(  53), INT8_C( -68), INT8_C(-121), INT8_C(  94),
                        INT8_C(-100), INT8_C( -25), INT8_C( 102), INT8_C( 120)),
      UINT8_C(108),
      simde_mm512_set_epi64(INT64_C(-8224370586313845882), INT64_C(-1993121323347227003),
                            INT64_C( 8831696545371448920), INT64_C( 2185815767663877612),
                            INT64_C( -270570670623534210), INT64_C(-7835605414239696560),
                            INT64_C(-1637430132815230795), INT64_C( 7232010829548987022)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  53), INT8_C(-128), INT8_C( 127), INT8_C(  94),
                        INT8_C(-128), INT8_C(-128), INT8_C( 102), INT8_C( 120)) },
    { simde_mm_set_epi8(INT8_C( -93), INT8_C( -72), INT8_C(   3), INT8_C( -86),
                        INT8_C(-106), INT8_C(  54), INT8_C( -93), INT8_C(  77),
                        INT8_C( 111), INT8_C(-126), INT8_C(  23), INT8_C( -40),
                        INT8_C(  -4), INT8_C(  -3), INT8_C( -74), INT8_C(   6)),
      UINT8_C(212),
      simde_mm512_set_epi64(INT64_C(-3956629285800542954), INT64_C( 7768661847448918660),
                            INT64_C( 8414129281940348180), INT64_C(-3282225164507364365),
                            INT64_C( 3972788221445796445), INT64_C(-1066584669561441235),
                            INT64_C( 2182845185502488635), INT64_C(-2591044625591562731)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C( 127), INT8_C(  23), INT8_C(-128),
                        INT8_C(  -4), INT8_C(-128), INT8_C( -74), INT8_C(   6)) },
    { simde_mm_set_epi8(INT8_C(  95), INT8_C( 126), INT8_C( 102), INT8_C(  -2),
                        INT8_C(  37), INT8_C( -94), INT8_C(  95), INT8_C( -94),
                        INT8_C(-125), INT8_C(  -9), INT8_C( -95), INT8_C(  41),
                        INT8_C(  -4), INT8_C( -36), INT8_C( -88), INT8_C(  79)),
      UINT8_C( 80),
      simde_mm512_set_epi64(INT64_C( 5062048906197334415), INT64_C( -115992959878725320),
                            INT64_C( 1136786795492082678), INT64_C(-1948109654186813959),
                            INT64_C(  120681793319816518), INT64_C(-3982161075345546564),
                            INT64_C(-4623140196798581694), INT64_C(-1633613586256931500)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-125), INT8_C(-128), INT8_C( -95), INT8_C(-128),
                        INT8_C(  -4), INT8_C( -36), INT8_C( -88), INT8_C(  79)) },
    { simde_mm_set_epi8(INT8_C( -59), INT8_C( -75), INT8_C( -72), INT8_C( -42),
                        INT8_C( -55), INT8_C(-119), INT8_C( -60), INT8_C(  67),
                        INT8_C( 109), INT8_C( -73), INT8_C(  91), INT8_C(  51),
                        INT8_C(  78), INT8_C(  83), INT8_C(-115), INT8_C(  62)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C( 5851174439164188465), INT64_C(-6319779881941429071),
                            INT64_C( 1879763181654775792), INT64_C( 4044922446109520806),
                            INT64_C( 1345671618220988839), INT64_C(-8367274893382946539),
                            INT64_C( 4386554571925238747), INT64_C(-2885812676818812761)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C( 127), INT8_C(  83), INT8_C(-115), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C(  79), INT8_C(  39), INT8_C(  96), INT8_C(  42),
                        INT8_C( -85), INT8_C( 114), INT8_C(  97), INT8_C( -62),
                        INT8_C(   8), INT8_C(  38), INT8_C(  93), INT8_C( 127),
                        INT8_C(-112), INT8_C( 122), INT8_C(-125), INT8_C( -69)),
      UINT8_C( 46),
      simde_mm512_set_epi64(INT64_C(-5543414139022216695), INT64_C(-7351051862345725962),
                            INT64_C( 7765938089573807601), INT64_C(-8520500957815241015),
                            INT64_C(-4114690540063739335), INT64_C( 2127377446843006162),
                            INT64_C( 4629741254325605495), INT64_C( 6283226406328232321)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   8), INT8_C(  38), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C( -69)) },
    { simde_mm_set_epi8(INT8_C( -81), INT8_C(-127), INT8_C( -86), INT8_C(  87),
                        INT8_C(  54), INT8_C(  74), INT8_C(  -3), INT8_C(  90),
                        INT8_C( -47), INT8_C( -23), INT8_C(  73), INT8_C(  26),
                        INT8_C(  65), INT8_C(  65), INT8_C(  86), INT8_C(-128)),
      UINT8_C(244),
      simde_mm512_set_epi64(INT64_C( 3168330803816226018), INT64_C(-5665190690683619181),
                            INT64_C(-6479275574786056503), INT64_C(-4854337280374936929),
                            INT64_C( 9110007504327473973), INT64_C( 2306966881245130829),
                            INT64_C( 3371747741827789856), INT64_C( 4883552887250537067)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C(  65), INT8_C( 127), INT8_C(  86), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C( -36), INT8_C( -15), INT8_C(  94), INT8_C(  96),
                        INT8_C(  41), INT8_C(  71), INT8_C(-125), INT8_C(  50),
                        INT8_C(  -5), INT8_C( -42), INT8_C(   6), INT8_C(  31),
                        INT8_C( -82), INT8_C(  -3), INT8_C( -84), INT8_C( 118)),
      UINT8_C( 84),
      simde_mm512_set_epi64(INT64_C(-5802305007366524132), INT64_C( 6368102336095963395),
                            INT64_C(   26715019440636716), INT64_C( 3396530625665467524),
                            INT64_C( 4010180268457941346), INT64_C( 5473964637387956828),
                            INT64_C( 6527063328512873018), INT64_C( 7600981363267795639)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  -5), INT8_C( 127), INT8_C(   6), INT8_C( 127),
                        INT8_C( -82), INT8_C( 127), INT8_C( -84), INT8_C( 118)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtsepi64_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtsepi64_epi8(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT8_C( 99),
      simde_mm512_set_epi64(INT64_C(-6773163771856001287), INT64_C( 8528895860955669022),
                            INT64_C(  743395091751495893), INT64_C(-5753646205421077345),
                            INT64_C( 6781795189594641427), INT64_C( 7308036127418939455),
                            INT64_C(  616165601606027604), INT64_C(-2694201555471274183)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( 127), INT8_C( 127), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C( 127), INT8_C(-128)) },
    { UINT8_C( 41),
      simde_mm512_set_epi64(INT64_C( -270570670623534210), INT64_C(-7835605414239696560),
                            INT64_C(-1637430132815230795), INT64_C( 7232010829548987022),
                            INT64_C(  564423766678453237), INT64_C(-7165203031737464994),
                            INT64_C(-7140626015800786754), INT64_C(-2499691201975648498)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C(   0),
                        INT8_C( 127), INT8_C(   0), INT8_C(   0), INT8_C(-128)) },
    { UINT8_C(236),
      simde_mm512_set_epi64(INT64_C( 2182845185502488635), INT64_C(-2591044625591562731),
                            INT64_C(  634126652413576106), INT64_C(-7622725766299379752),
                            INT64_C( -216817067777774970), INT64_C(-4785193871492054345),
                            INT64_C( 7467476729839254839), INT64_C( 3956665538785416156)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(   0), INT8_C(   0)) },
    { UINT8_C( 45),
      simde_mm512_set_epi64(INT64_C(-8852586683955255554), INT64_C( 2711835077233844521),
                            INT64_C( -226120820645741920), INT64_C(-5768129162479938358),
                            INT64_C( 4995224830392729088), INT64_C(-9204129394426957414),
                            INT64_C( 6102592944456478166), INT64_C(-5538879581396485404)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C(   0), INT8_C(-128)) },
    { UINT8_C( 84),
      simde_mm512_set_epi64(INT64_C( 5644010056631452329), INT64_C(  687917764947732740),
                            INT64_C( 9077991551678197137), INT64_C( 5379835588808993943),
                            INT64_C( 5669915352590237555), INT64_C(-7084358722419653223),
                            INT64_C(-5646349838748004740), INT64_C( 5115442951238204697)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C( 127),
                        INT8_C(   0), INT8_C(-128), INT8_C(   0), INT8_C(   0)) },
    { UINT8_C( 51),
      simde_mm512_set_epi64(INT64_C( 6212879174325387943), INT64_C(-9195198218682338424),
                            INT64_C(-6314975051621021387), INT64_C(  316124606243108816),
                            INT64_C(-1018533604686287836), INT64_C( 7455766202425772200),
                            INT64_C( -151643922692032263), INT64_C(-4200247844809358269)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C( 127),
                        INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C(-128)) },
    { UINT8_C(174),
      simde_mm512_set_epi64(INT64_C(-4824473637125509439), INT64_C( 2222300162020472452),
                            INT64_C( 4485933577556469028), INT64_C( 3606420641901871392),
                            INT64_C( 7533512282749681966), INT64_C( 5703633189469315522),
                            INT64_C(  587259603884606395), INT64_C( 5851174439164188465)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(   0), INT8_C( 127), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(   0)) },
    { UINT8_C(133),
      simde_mm512_set_epi64(INT64_C(-2795345254402041164), INT64_C( -585762482098414214),
                            INT64_C( 1037739329164009972), INT64_C(-5800167553526727334),
                            INT64_C(-3321042873096251776), INT64_C(-5543414139022216695),
                            INT64_C(-7351051862345725962), INT64_C( 7765938089573807601)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(-128), INT8_C(   0), INT8_C( 127)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtsepi64_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtsepi64_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-2030970610590957423), INT64_C(-6407881172688895992),
                            INT64_C( 5763311992085393311), INT64_C( 5413217893862876377),
                            INT64_C( 2736248102150189299), INT64_C(-3905239043220002295),
                            INT64_C(-8525695447592722282), INT64_C( 8985359849763220184)),
      simde_mm_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767),
                         INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi64(INT64_C( 5528142976218439915), INT64_C( 2800801417967116565),
                            INT64_C(-5498223206200113790), INT64_C(-7735651953136898344),
                            INT64_C(  -45944962922223273), INT64_C( 7270573409050727432),
                            INT64_C(-3005715738589766193), INT64_C(    8104807772879505)),
      simde_mm_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768),
                         INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi64(INT64_C( 1914743572707752888), INT64_C(-1522220651072788958),
                            INT64_C(  774824395663784136), INT64_C(-5948238846445238616),
                            INT64_C(-6703970016956217448), INT64_C(-4949181030820609545),
                            INT64_C(-1484143836373726158), INT64_C( 7023453391392224586)),
      simde_mm_set_epi16(INT16_C( 32767), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                         INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi64(INT64_C(  877787331628902271), INT64_C(-4271522850214152795),
                            INT64_C(  -36139528410486948), INT64_C( 8221636509979881717),
                            INT64_C( 8317903343708448256), INT64_C(  273589563704309286),
                            INT64_C(-3793565254975032249), INT64_C(-8693302524009918115)),
      simde_mm_set_epi16(INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767),
                         INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768)) },
    { simde_mm512_set_epi64(INT64_C( 8982303740345517000), INT64_C( 3826887655534630950),
                            INT64_C( 3239767036578666476), INT64_C(-6247325025317220634),
                            INT64_C( 7671225941352774255), INT64_C(-4520433707995010156),
                            INT64_C( -202556859458160671), INT64_C( 8376976669674082741)),
      simde_mm_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                         INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi64(INT64_C(-6735519140496673702), INT64_C(-3340797945830539185),
                            INT64_C(-4234248954061729169), INT64_C(-8216832367433725114),
                            INT64_C( 7806307918239696158), INT64_C( 1170496629302395613),
                            INT64_C(-6389283806078878937), INT64_C( 6713220048127768064)),
      simde_mm_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C(-32768),
                         INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi64(INT64_C( 4236272701979881947), INT64_C( 3804598066273696798),
                            INT64_C(-3344748723935369850), INT64_C( 1825331027945283861),
                            INT64_C(  238286709106496427), INT64_C( 5804239832581888150),
                            INT64_C(  140213339004639903), INT64_C(-8021982786001039013)),
      simde_mm_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767),
                         INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768)) },
    { simde_mm512_set_epi64(INT64_C(-4062543779038977729), INT64_C( 2624543651353099153),
                            INT64_C( 8730480163779377063), INT64_C( 1087955593743156912),
                            INT64_C(-7352034341207303173), INT64_C( -816150536575471214),
                            INT64_C(  427511002321984596), INT64_C( 6609731989001691179)),
      simde_mm_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                         INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtsepi64_epi16(test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtsepi64_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m128i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi16(INT16_C(-24950), INT16_C(-13996), INT16_C( -9572), INT16_C( 17909),
                         INT16_C(-19244), INT16_C( 28473), INT16_C(-27531), INT16_C(-23965)),
      UINT8_C( 89),
      simde_mm512_set_epi64(INT64_C(-6464916445333598184), INT64_C(-9074050598773963479),
                            INT64_C(-6773163771856001287), INT64_C( 8528895860955669022),
                            INT64_C(  743395091751495893), INT64_C(-5753646205421077345),
                            INT64_C( 6781795189594641427), INT64_C( 7308036127418939455)),
      simde_mm_set_epi16(INT16_C(-24950), INT16_C(-32768), INT16_C( -9572), INT16_C( 32767),
                         INT16_C( 32767), INT16_C( 28473), INT16_C(-27531), INT16_C( 32767)) },
    { simde_mm_set_epi16(INT16_C( -7256), INT16_C( 15349), INT16_C(-25456), INT16_C(  5582),
                         INT16_C( 13756), INT16_C(-30882), INT16_C(-25369), INT16_C( 26232)),
      UINT8_C(108),
      simde_mm512_set_epi64(INT64_C(-8224370586313845882), INT64_C(-1993121323347227003),
                            INT64_C( 8831696545371448920), INT64_C( 2185815767663877612),
                            INT64_C( -270570670623534210), INT64_C(-7835605414239696560),
                            INT64_C(-1637430132815230795), INT64_C( 7232010829548987022)),
      simde_mm_set_epi16(INT16_C( -7256), INT16_C(-32768), INT16_C( 32767), INT16_C(  5582),
                         INT16_C(-32768), INT16_C(-32768), INT16_C(-25369), INT16_C( 26232)) },
    { simde_mm_set_epi16(INT16_C(-23624), INT16_C(   938), INT16_C(-27082), INT16_C(-23731),
                         INT16_C( 28546), INT16_C(  6104), INT16_C(  -771), INT16_C(-18938)),
      UINT8_C(212),
      simde_mm512_set_epi64(INT64_C(-3956629285800542954), INT64_C( 7768661847448918660),
                            INT64_C( 8414129281940348180), INT64_C(-3282225164507364365),
                            INT64_C( 3972788221445796445), INT64_C(-1066584669561441235),
                            INT64_C( 2182845185502488635), INT64_C(-2591044625591562731)),
      simde_mm_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-27082), INT16_C(-32768),
                         INT16_C( 28546), INT16_C(-32768), INT16_C(  -771), INT16_C(-18938)) },
    { simde_mm_set_epi16(INT16_C( 24446), INT16_C( 26366), INT16_C(  9634), INT16_C( 24482),
                         INT16_C(-31753), INT16_C(-24279), INT16_C(  -804), INT16_C(-22449)),
      UINT8_C( 80),
      simde_mm512_set_epi64(INT64_C( 5062048906197334415), INT64_C( -115992959878725320),
                            INT64_C( 1136786795492082678), INT64_C(-1948109654186813959),
                            INT64_C(  120681793319816518), INT64_C(-3982161075345546564),
                            INT64_C(-4623140196798581694), INT64_C(-1633613586256931500)),
      simde_mm_set_epi16(INT16_C( 24446), INT16_C(-32768), INT16_C(  9634), INT16_C(-32768),
                         INT16_C(-31753), INT16_C(-24279), INT16_C(  -804), INT16_C(-22449)) },
    { simde_mm_set_epi16(INT16_C(-14923), INT16_C(-18218), INT16_C(-13943), INT16_C(-15293),
                         INT16_C( 28087), INT16_C( 23347), INT16_C( 20051), INT16_C(-29378)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C( 5851174439164188465), INT64_C(-6319779881941429071),
                            INT64_C( 1879763181654775792), INT64_C( 4044922446109520806),
                            INT64_C( 1345671618220988839), INT64_C(-8367274893382946539),
                            INT64_C( 4386554571925238747), INT64_C(-2885812676818812761)),
      simde_mm_set_epi16(INT16_C( 32767), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767),
                         INT16_C( 32767), INT16_C( 23347), INT16_C( 20051), INT16_C(-32768)) },
    { simde_mm_set_epi16(INT16_C( 20263), INT16_C( 24618), INT16_C(-21646), INT16_C( 25026),
                         INT16_C(  2086), INT16_C( 23935), INT16_C(-28550), INT16_C(-31813)),
      UINT8_C( 46),
      simde_mm512_set_epi64(INT64_C(-5543414139022216695), INT64_C(-7351051862345725962),
                            INT64_C( 7765938089573807601), INT64_C(-8520500957815241015),
                            INT64_C(-4114690540063739335), INT64_C( 2127377446843006162),
                            INT64_C( 4629741254325605495), INT64_C( 6283226406328232321)),
      simde_mm_set_epi16(INT16_C( 20263), INT16_C( 24618), INT16_C( 32767), INT16_C( 25026),
                         INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C(-31813)) },
    { simde_mm_set_epi16(INT16_C(-20607), INT16_C(-21929), INT16_C( 13898), INT16_C(  -678),
                         INT16_C(-11799), INT16_C( 18714), INT16_C( 16705), INT16_C( 22144)),
      UINT8_C(244),
      simde_mm512_set_epi64(INT64_C( 3168330803816226018), INT64_C(-5665190690683619181),
                            INT64_C(-6479275574786056503), INT64_C(-4854337280374936929),
                            INT64_C( 9110007504327473973), INT64_C( 2306966881245130829),
                            INT64_C( 3371747741827789856), INT64_C( 4883552887250537067)),
      simde_mm_set_epi16(INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C(-32768),
                         INT16_C(-11799), INT16_C( 32767), INT16_C( 16705), INT16_C( 22144)) },
    { simde_mm_set_epi16(INT16_C( -8975), INT16_C( 24160), INT16_C( 10567), INT16_C(-31950),
                         INT16_C( -1066), INT16_C(  1567), INT16_C(-20739), INT16_C(-21386)),
      UINT8_C( 84),
      simde_mm512_set_epi64(INT64_C(-5802305007366524132), INT64_C( 6368102336095963395),
                            INT64_C(   26715019440636716), INT64_C( 3396530625665467524),
                            INT64_C( 4010180268457941346), INT64_C( 5473964637387956828),
                            INT64_C( 6527063328512873018), INT64_C( 7600981363267795639)),
      simde_mm_set_epi16(INT16_C( -8975), INT16_C( 32767), INT16_C( 10567), INT16_C( 32767),
                         INT16_C( -1066), INT16_C( 32767), INT16_C(-20739), INT16_C(-21386)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtsepi64_epi16(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtsepi64_epi16(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT8_C( 99),
      simde_mm512_set_epi64(INT64_C(-6773163771856001287), INT64_C( 8528895860955669022),
                            INT64_C(  743395091751495893), INT64_C(-5753646205421077345),
                            INT64_C( 6781795189594641427), INT64_C( 7308036127418939455),
                            INT64_C(  616165601606027604), INT64_C(-2694201555471274183)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C( 32767), INT16_C( 32767), INT16_C(     0),
                         INT16_C(     0), INT16_C(     0), INT16_C( 32767), INT16_C(-32768)) },
    { UINT8_C( 41),
      simde_mm512_set_epi64(INT64_C( -270570670623534210), INT64_C(-7835605414239696560),
                            INT64_C(-1637430132815230795), INT64_C( 7232010829548987022),
                            INT64_C(  564423766678453237), INT64_C(-7165203031737464994),
                            INT64_C(-7140626015800786754), INT64_C(-2499691201975648498)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(-32768), INT16_C(     0),
                         INT16_C( 32767), INT16_C(     0), INT16_C(     0), INT16_C(-32768)) },
    { UINT8_C(236),
      simde_mm512_set_epi64(INT64_C( 2182845185502488635), INT64_C(-2591044625591562731),
                            INT64_C(  634126652413576106), INT64_C(-7622725766299379752),
                            INT64_C( -216817067777774970), INT64_C(-4785193871492054345),
                            INT64_C( 7467476729839254839), INT64_C( 3956665538785416156)),
      simde_mm_set_epi16(INT16_C( 32767), INT16_C(-32768), INT16_C( 32767), INT16_C(     0),
                         INT16_C(-32768), INT16_C(-32768), INT16_C(     0), INT16_C(     0)) },
    { UINT8_C( 45),
      simde_mm512_set_epi64(INT64_C(-8852586683955255554), INT64_C( 2711835077233844521),
                            INT64_C( -226120820645741920), INT64_C(-5768129162479938358),
                            INT64_C( 4995224830392729088), INT64_C(-9204129394426957414),
                            INT64_C( 6102592944456478166), INT64_C(-5538879581396485404)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(-32768), INT16_C(     0),
                         INT16_C( 32767), INT16_C(-32768), INT16_C(     0), INT16_C(-32768)) },
    { UINT8_C( 84),
      simde_mm512_set_epi64(INT64_C( 5644010056631452329), INT64_C(  687917764947732740),
                            INT64_C( 9077991551678197137), INT64_C( 5379835588808993943),
                            INT64_C( 5669915352590237555), INT64_C(-7084358722419653223),
                            INT64_C(-5646349838748004740), INT64_C( 5115442951238204697)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C( 32767), INT16_C(     0), INT16_C( 32767),
                         INT16_C(     0), INT16_C(-32768), INT16_C(     0), INT16_C(     0)) },
    { UINT8_C( 51),
      simde_mm512_set_epi64(INT64_C( 6212879174325387943), INT64_C(-9195198218682338424),
                            INT64_C(-6314975051621021387), INT64_C(  316124606243108816),
                            INT64_C(-1018533604686287836), INT64_C( 7455766202425772200),
                            INT64_C( -151643922692032263), INT64_C(-4200247844809358269)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(-32768), INT16_C( 32767),
                         INT16_C(     0), INT16_C(     0), INT16_C(-32768), INT16_C(-32768)) },
    { UINT8_C(174),
      simde_mm512_set_epi64(INT64_C(-4824473637125509439), INT64_C( 2222300162020472452),
                            INT64_C( 4485933577556469028), INT64_C( 3606420641901871392),
                            INT64_C( 7533512282749681966), INT64_C( 5703633189469315522),
                            INT64_C(  587259603884606395), INT64_C( 5851174439164188465)),
      simde_mm_set_epi16(INT16_C(-32768), INT16_C(     0), INT16_C( 32767), INT16_C(     0),
                         INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(     0)) },
    { UINT8_C(133),
      simde_mm512_set_epi64(INT64_C(-2795345254402041164), INT64_C( -585762482098414214),
                            INT64_C( 1037739329164009972), INT64_C(-5800167553526727334),
                            INT64_C(-3321042873096251776), INT64_C(-5543414139022216695),
                            INT64_C(-7351051862345725962), INT64_C( 7765938089573807601)),
      simde_mm_set_epi16(INT16_C(-32768), INT16_C(     0), INT16_C(     0), INT16_C(     0),
                         INT16_C(     0), INT16_C(-32768), INT16_C(     0), INT16_C( 32767)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtsepi64_epi16(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_div_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   653.62), SIMDE_FLOAT32_C(   981.74), SIMDE_FLOAT32_C(   780.10), SIMDE_FLOAT32_C(    59.38),
                         SIMDE_FLOAT32_C(  -795.11), SIMDE_FLOAT32_C(   923.87), SIMDE_FLOAT32_C(  -270.01), SIMDE_FLOAT32_C(  -411.99),
                         SIMDE_FLOAT32_C(   -97.83), SIMDE_FLOAT32_C(  -393.82), SIMDE_FLOAT32_C(   934.81), SIMDE_FLOAT32_C(    74.53),
                         SIMDE_FLOAT32_C(   843.79), SIMDE_FLOAT32_C(   465.05), SIMDE_FLOAT32_C(   -42.07), SIMDE_FLOAT32_C(  -685.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   596.54), SIMDE_FLOAT32_C(  -116.40), SIMDE_FLOAT32_C(  -989.77), SIMDE_FLOAT32_C(  -794.40),
                         SIMDE_FLOAT32_C(   183.38), SIMDE_FLOAT32_C(  -185.75), SIMDE_FLOAT32_C(   429.70), SIMDE_FLOAT32_C(   664.04),
                         SIMDE_FLOAT32_C(   296.78), SIMDE_FLOAT32_C(  -698.78), SIMDE_FLOAT32_C(   908.33), SIMDE_FLOAT32_C(   181.85),
                         SIMDE_FLOAT32_C(  -397.89), SIMDE_FLOAT32_C(  -586.75), SIMDE_FLOAT32_C(   904.99), SIMDE_FLOAT32_C(  -321.15)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     1.10), SIMDE_FLOAT32_C(    -8.43), SIMDE_FLOAT32_C(    -0.79), SIMDE_FLOAT32_C(    -0.07),
                         SIMDE_FLOAT32_C(    -4.34), SIMDE_FLOAT32_C(    -4.97), SIMDE_FLOAT32_C(    -0.63), SIMDE_FLOAT32_C(    -0.62),
                         SIMDE_FLOAT32_C(    -0.33), SIMDE_FLOAT32_C(     0.56), SIMDE_FLOAT32_C(     1.03), SIMDE_FLOAT32_C(     0.41),
                         SIMDE_FLOAT32_C(    -2.12), SIMDE_FLOAT32_C(    -0.79), SIMDE_FLOAT32_C(    -0.05), SIMDE_FLOAT32_C(     2.14)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   729.63), SIMDE_FLOAT32_C(  -908.06), SIMDE_FLOAT32_C(  -769.77), SIMDE_FLOAT32_C(   -70.66),
                         SIMDE_FLOAT32_C(   482.71), SIMDE_FLOAT32_C(   244.66), SIMDE_FLOAT32_C(  -615.83), SIMDE_FLOAT32_C(   841.42),
                         SIMDE_FLOAT32_C(  -571.10), SIMDE_FLOAT32_C(   971.96), SIMDE_FLOAT32_C(   149.38), SIMDE_FLOAT32_C(   497.71),
                         SIMDE_FLOAT32_C(   988.69), SIMDE_FLOAT32_C(   479.68), SIMDE_FLOAT32_C(  -128.24), SIMDE_FLOAT32_C(   585.28)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   359.65), SIMDE_FLOAT32_C(  -730.08), SIMDE_FLOAT32_C(   977.98), SIMDE_FLOAT32_C(  -215.53),
                         SIMDE_FLOAT32_C(  -315.50), SIMDE_FLOAT32_C(    80.64), SIMDE_FLOAT32_C(  -996.10), SIMDE_FLOAT32_C(  -556.83),
                         SIMDE_FLOAT32_C(  -628.68), SIMDE_FLOAT32_C(   938.60), SIMDE_FLOAT32_C(  -147.98), SIMDE_FLOAT32_C(   378.31),
                         SIMDE_FLOAT32_C(   246.47), SIMDE_FLOAT32_C(   109.18), SIMDE_FLOAT32_C(  -575.64), SIMDE_FLOAT32_C(  -426.86)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     2.03), SIMDE_FLOAT32_C(     1.24), SIMDE_FLOAT32_C(    -0.79), SIMDE_FLOAT32_C(     0.33),
                         SIMDE_FLOAT32_C(    -1.53), SIMDE_FLOAT32_C(     3.03), SIMDE_FLOAT32_C(     0.62), SIMDE_FLOAT32_C(    -1.51),
                         SIMDE_FLOAT32_C(     0.91), SIMDE_FLOAT32_C(     1.04), SIMDE_FLOAT32_C(    -1.01), SIMDE_FLOAT32_C(     1.32),
                         SIMDE_FLOAT32_C(     4.01), SIMDE_FLOAT32_C(     4.39), SIMDE_FLOAT32_C(     0.22), SIMDE_FLOAT32_C(    -1.37)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -148.70), SIMDE_FLOAT32_C(  -327.17), SIMDE_FLOAT32_C(  -310.14), SIMDE_FLOAT32_C(  -718.80),
                         SIMDE_FLOAT32_C(   382.69), SIMDE_FLOAT32_C(  -181.61), SIMDE_FLOAT32_C(  -214.09), SIMDE_FLOAT32_C(    55.72),
                         SIMDE_FLOAT32_C(   438.03), SIMDE_FLOAT32_C(  -458.01), SIMDE_FLOAT32_C(   144.59), SIMDE_FLOAT32_C(   165.00),
                         SIMDE_FLOAT32_C(  -331.04), SIMDE_FLOAT32_C(   406.96), SIMDE_FLOAT32_C(  -326.43), SIMDE_FLOAT32_C(   373.82)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   791.83), SIMDE_FLOAT32_C(   191.69), SIMDE_FLOAT32_C(  -460.58), SIMDE_FLOAT32_C(  -915.08),
                         SIMDE_FLOAT32_C(  -877.38), SIMDE_FLOAT32_C(  -915.27), SIMDE_FLOAT32_C(   207.85), SIMDE_FLOAT32_C(   567.35),
                         SIMDE_FLOAT32_C(   304.30), SIMDE_FLOAT32_C(  -777.07), SIMDE_FLOAT32_C(  -683.73), SIMDE_FLOAT32_C(  -113.32),
                         SIMDE_FLOAT32_C(  -701.16), SIMDE_FLOAT32_C(  -942.92), SIMDE_FLOAT32_C(  -489.97), SIMDE_FLOAT32_C(   911.34)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    -0.19), SIMDE_FLOAT32_C(    -1.71), SIMDE_FLOAT32_C(     0.67), SIMDE_FLOAT32_C(     0.79),
                         SIMDE_FLOAT32_C(    -0.44), SIMDE_FLOAT32_C(     0.20), SIMDE_FLOAT32_C(    -1.03), SIMDE_FLOAT32_C(     0.10),
                         SIMDE_FLOAT32_C(     1.44), SIMDE_FLOAT32_C(     0.59), SIMDE_FLOAT32_C(    -0.21), SIMDE_FLOAT32_C(    -1.46),
                         SIMDE_FLOAT32_C(     0.47), SIMDE_FLOAT32_C(    -0.43), SIMDE_FLOAT32_C(     0.67), SIMDE_FLOAT32_C(     0.41)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -869.58), SIMDE_FLOAT32_C(   763.75), SIMDE_FLOAT32_C(  -558.93), SIMDE_FLOAT32_C(   756.19),
                         SIMDE_FLOAT32_C(   509.82), SIMDE_FLOAT32_C(  -855.71), SIMDE_FLOAT32_C(  -965.40), SIMDE_FLOAT32_C(  -279.29),
                         SIMDE_FLOAT32_C(  -798.08), SIMDE_FLOAT32_C(   256.40), SIMDE_FLOAT32_C(   739.89), SIMDE_FLOAT32_C(  -903.46),
                         SIMDE_FLOAT32_C(  -771.75), SIMDE_FLOAT32_C(   -54.77), SIMDE_FLOAT32_C(   397.04), SIMDE_FLOAT32_C(   925.94)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -355.51), SIMDE_FLOAT32_C(   136.73), SIMDE_FLOAT32_C(   586.70), SIMDE_FLOAT32_C(   712.56),
                         SIMDE_FLOAT32_C(   135.88), SIMDE_FLOAT32_C(  -693.91), SIMDE_FLOAT32_C(  -131.33), SIMDE_FLOAT32_C(  -933.79),
                         SIMDE_FLOAT32_C(   864.29), SIMDE_FLOAT32_C(  -834.00), SIMDE_FLOAT32_C(   475.52), SIMDE_FLOAT32_C(   502.31),
                         SIMDE_FLOAT32_C(  -746.87), SIMDE_FLOAT32_C(  -364.10), SIMDE_FLOAT32_C(  -995.18), SIMDE_FLOAT32_C(   683.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     2.45), SIMDE_FLOAT32_C(     5.59), SIMDE_FLOAT32_C(    -0.95), SIMDE_FLOAT32_C(     1.06),
                         SIMDE_FLOAT32_C(     3.75), SIMDE_FLOAT32_C(     1.23), SIMDE_FLOAT32_C(     7.35), SIMDE_FLOAT32_C(     0.30),
                         SIMDE_FLOAT32_C(    -0.92), SIMDE_FLOAT32_C(    -0.31), SIMDE_FLOAT32_C(     1.56), SIMDE_FLOAT32_C(    -1.80),
                         SIMDE_FLOAT32_C(     1.03), SIMDE_FLOAT32_C(     0.15), SIMDE_FLOAT32_C(    -0.40), SIMDE_FLOAT32_C(     1.35)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   119.21), SIMDE_FLOAT32_C(   360.54), SIMDE_FLOAT32_C(   885.26), SIMDE_FLOAT32_C(  -618.98),
                         SIMDE_FLOAT32_C(    -8.97), SIMDE_FLOAT32_C(  -881.58), SIMDE_FLOAT32_C(   -89.25), SIMDE_FLOAT32_C(  -937.64),
                         SIMDE_FLOAT32_C(  -660.18), SIMDE_FLOAT32_C(  -649.17), SIMDE_FLOAT32_C(  -279.52), SIMDE_FLOAT32_C(   812.95),
                         SIMDE_FLOAT32_C(  -471.80), SIMDE_FLOAT32_C(   805.98), SIMDE_FLOAT32_C(   532.44), SIMDE_FLOAT32_C(   126.30)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   944.81), SIMDE_FLOAT32_C(   946.29), SIMDE_FLOAT32_C(   161.37), SIMDE_FLOAT32_C(  -637.11),
                         SIMDE_FLOAT32_C(    16.54), SIMDE_FLOAT32_C(   417.79), SIMDE_FLOAT32_C(   257.34), SIMDE_FLOAT32_C(  -857.05),
                         SIMDE_FLOAT32_C(   770.17), SIMDE_FLOAT32_C(  -559.67), SIMDE_FLOAT32_C(  -862.75), SIMDE_FLOAT32_C(  -541.96),
                         SIMDE_FLOAT32_C(   412.30), SIMDE_FLOAT32_C(  -147.64), SIMDE_FLOAT32_C(   553.94), SIMDE_FLOAT32_C(  -736.63)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.13), SIMDE_FLOAT32_C(     0.38), SIMDE_FLOAT32_C(     5.49), SIMDE_FLOAT32_C(     0.97),
                         SIMDE_FLOAT32_C(    -0.54), SIMDE_FLOAT32_C(    -2.11), SIMDE_FLOAT32_C(    -0.35), SIMDE_FLOAT32_C(     1.09),
                         SIMDE_FLOAT32_C(    -0.86), SIMDE_FLOAT32_C(     1.16), SIMDE_FLOAT32_C(     0.32), SIMDE_FLOAT32_C(    -1.50),
                         SIMDE_FLOAT32_C(    -1.14), SIMDE_FLOAT32_C(    -5.46), SIMDE_FLOAT32_C(     0.96), SIMDE_FLOAT32_C(    -0.17)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -81.24), SIMDE_FLOAT32_C(  -934.88), SIMDE_FLOAT32_C(   -84.21), SIMDE_FLOAT32_C(  -265.16),
                         SIMDE_FLOAT32_C(  -978.34), SIMDE_FLOAT32_C(  -425.47), SIMDE_FLOAT32_C(   792.31), SIMDE_FLOAT32_C(  -306.03),
                         SIMDE_FLOAT32_C(   911.07), SIMDE_FLOAT32_C(   992.01), SIMDE_FLOAT32_C(   172.45), SIMDE_FLOAT32_C(  -135.31),
                         SIMDE_FLOAT32_C(   652.11), SIMDE_FLOAT32_C(  -529.15), SIMDE_FLOAT32_C(    -0.58), SIMDE_FLOAT32_C(   883.05)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -110.89), SIMDE_FLOAT32_C(  -325.07), SIMDE_FLOAT32_C(   834.96), SIMDE_FLOAT32_C(  -681.06),
                         SIMDE_FLOAT32_C(  -877.63), SIMDE_FLOAT32_C(  -653.45), SIMDE_FLOAT32_C(    40.48), SIMDE_FLOAT32_C(  -644.02),
                         SIMDE_FLOAT32_C(  -687.76), SIMDE_FLOAT32_C(  -660.68), SIMDE_FLOAT32_C(   802.46), SIMDE_FLOAT32_C(  -477.95),
                         SIMDE_FLOAT32_C(  -125.80), SIMDE_FLOAT32_C(  -475.50), SIMDE_FLOAT32_C(  -806.50), SIMDE_FLOAT32_C(  -778.62)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.73), SIMDE_FLOAT32_C(     2.88), SIMDE_FLOAT32_C(    -0.10), SIMDE_FLOAT32_C(     0.39),
                         SIMDE_FLOAT32_C(     1.11), SIMDE_FLOAT32_C(     0.65), SIMDE_FLOAT32_C(    19.57), SIMDE_FLOAT32_C(     0.48),
                         SIMDE_FLOAT32_C(    -1.32), SIMDE_FLOAT32_C(    -1.50), SIMDE_FLOAT32_C(     0.21), SIMDE_FLOAT32_C(     0.28),
                         SIMDE_FLOAT32_C(    -5.18), SIMDE_FLOAT32_C(     1.11), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.13)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -387.95), SIMDE_FLOAT32_C(   255.55), SIMDE_FLOAT32_C(   948.01), SIMDE_FLOAT32_C(   520.84),
                         SIMDE_FLOAT32_C(   310.00), SIMDE_FLOAT32_C(  -412.39), SIMDE_FLOAT32_C(   412.17), SIMDE_FLOAT32_C(  -913.22),
                         SIMDE_FLOAT32_C(   810.06), SIMDE_FLOAT32_C(  -696.65), SIMDE_FLOAT32_C(   807.84), SIMDE_FLOAT32_C(    63.85),
                         SIMDE_FLOAT32_C(    -2.75), SIMDE_FLOAT32_C(  -763.61), SIMDE_FLOAT32_C(  -850.85), SIMDE_FLOAT32_C(   913.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -915.78), SIMDE_FLOAT32_C(   471.39), SIMDE_FLOAT32_C(  -324.79), SIMDE_FLOAT32_C(  -855.69),
                         SIMDE_FLOAT32_C(   966.81), SIMDE_FLOAT32_C(   668.44), SIMDE_FLOAT32_C(   925.33), SIMDE_FLOAT32_C(   564.88),
                         SIMDE_FLOAT32_C(  -130.24), SIMDE_FLOAT32_C(   360.71), SIMDE_FLOAT32_C(   966.21), SIMDE_FLOAT32_C(  -919.67),
                         SIMDE_FLOAT32_C(   198.47), SIMDE_FLOAT32_C(  -796.49), SIMDE_FLOAT32_C(   428.08), SIMDE_FLOAT32_C(   264.02)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.42), SIMDE_FLOAT32_C(     0.54), SIMDE_FLOAT32_C(    -2.92), SIMDE_FLOAT32_C(    -0.61),
                         SIMDE_FLOAT32_C(     0.32), SIMDE_FLOAT32_C(    -0.62), SIMDE_FLOAT32_C(     0.45), SIMDE_FLOAT32_C(    -1.62),
                         SIMDE_FLOAT32_C(    -6.22), SIMDE_FLOAT32_C(    -1.93), SIMDE_FLOAT32_C(     0.84), SIMDE_FLOAT32_C(    -0.07),
                         SIMDE_FLOAT32_C(    -0.01), SIMDE_FLOAT32_C(     0.96), SIMDE_FLOAT32_C(    -1.99), SIMDE_FLOAT32_C(     3.46)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   534.55), SIMDE_FLOAT32_C(  -263.46), SIMDE_FLOAT32_C(  -958.21), SIMDE_FLOAT32_C(   927.39),
                         SIMDE_FLOAT32_C(   830.49), SIMDE_FLOAT32_C(  -394.19), SIMDE_FLOAT32_C(  -755.65), SIMDE_FLOAT32_C(  -594.24),
                         SIMDE_FLOAT32_C(  -371.00), SIMDE_FLOAT32_C(   623.04), SIMDE_FLOAT32_C(   879.76), SIMDE_FLOAT32_C(   838.28),
                         SIMDE_FLOAT32_C(  -100.77), SIMDE_FLOAT32_C(  -708.14), SIMDE_FLOAT32_C(  -206.06), SIMDE_FLOAT32_C(  -203.03)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    65.94), SIMDE_FLOAT32_C(   158.39), SIMDE_FLOAT32_C(   532.17), SIMDE_FLOAT32_C(    -1.61),
                         SIMDE_FLOAT32_C(  -802.21), SIMDE_FLOAT32_C(  -782.13), SIMDE_FLOAT32_C(   831.96), SIMDE_FLOAT32_C(  -692.14),
                         SIMDE_FLOAT32_C(   581.38), SIMDE_FLOAT32_C(   943.65), SIMDE_FLOAT32_C(   585.87), SIMDE_FLOAT32_C(   329.94),
                         SIMDE_FLOAT32_C(  -747.39), SIMDE_FLOAT32_C(   976.32), SIMDE_FLOAT32_C(   362.23), SIMDE_FLOAT32_C(  -137.03)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     8.11), SIMDE_FLOAT32_C(    -1.66), SIMDE_FLOAT32_C(    -1.80), SIMDE_FLOAT32_C(  -576.02),
                         SIMDE_FLOAT32_C(    -1.04), SIMDE_FLOAT32_C(     0.50), SIMDE_FLOAT32_C(    -0.91), SIMDE_FLOAT32_C(     0.86),
                         SIMDE_FLOAT32_C(    -0.64), SIMDE_FLOAT32_C(     0.66), SIMDE_FLOAT32_C(     1.50), SIMDE_FLOAT32_C(     2.54),
                         SIMDE_FLOAT32_C(     0.13), SIMDE_FLOAT32_C(    -0.73), SIMDE_FLOAT32_C(    -0.57), SIMDE_FLOAT32_C(     1.48)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_div_ps(test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_cvtsepi64_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(   -1385298216494496), INT64_C(       -996445051636),
                            INT64_C(             3207004), INT64_C(    -651667828674554),
                            INT64_C(   -3476602607657036), INT64_C(            -4627787),
                            INT64_C(   -2474879574406724), INT64_C(  508842849663014820)),
      simde_mm256_set_epi32(          INT32_MIN ,           INT32_MIN , INT32_C(    3207004),           INT32_MIN ,
                                      INT32_MIN , INT32_C(   -4627787),           INT32_MIN ,           INT32_MAX ) },
    { simde_mm512_set_epi64(INT64_C(                  -2), INT64_C(               -7621),
                            INT64_C(        296832081577), INT64_C(              -10040),
                            INT64_C(               10421), INT64_C(           316486368),
                            INT64_C(                -279), INT64_C(          1776399948)),
      simde_mm256_set_epi32(INT32_C(         -2), INT32_C(      -7621),           INT32_MAX , INT32_C(     -10040),
                            INT32_C(      10421), INT32_C(  316486368), INT32_C(       -279), INT32_C( 1776399948)) },
    { simde_mm512_set_epi64(INT64_C(   69786007435882969), INT64_C(-6327619736082157669),
                            INT64_C(      24285899089645), INT64_C(           -54009961),
                            INT64_C(        216203910725), INT64_C(               34664),
                            INT64_C(              760577), INT64_C(         52093180175)),
      simde_mm256_set_epi32(          INT32_MAX ,           INT32_MIN ,           INT32_MAX , INT32_C(  -54009961),
                                      INT32_MAX , INT32_C(      34664), INT32_C(     760577),           INT32_MAX ) },
    { simde_mm512_set_epi64(INT64_C(       -212538005711), INT64_C(                1541),
                            INT64_C(                  -1), INT64_C(  700703313895845440),
                            INT64_C(               -2820), INT64_C(  -13888321195009228),
                            INT64_C( -125614148245680479), INT64_C(                -251)),
      simde_mm256_set_epi32(          INT32_MIN , INT32_C(       1541), INT32_C(         -1),           INT32_MAX ,
                            INT32_C(      -2820),           INT32_MIN ,           INT32_MIN , INT32_C(       -251)) },
    { simde_mm512_set_epi64(INT64_C(          8722215763), INT64_C(   69834045587926754),
                            INT64_C(      14995086304428), INT64_C(               32908),
                            INT64_C(                  36), INT64_C(                   3),
                            INT64_C(           480756836), INT64_C(                  20)),
      simde_mm256_set_epi32(          INT32_MAX ,           INT32_MAX ,           INT32_MAX , INT32_C(      32908),
                            INT32_C(         36), INT32_C(          3), INT32_C(  480756836), INT32_C(         20)) },
    { simde_mm512_set_epi64(INT64_C(              393479), INT64_C(           -33215582),
                            INT64_C(           220007333), INT64_C(                  93),
                            INT64_C(            53663901), INT64_C(     456460283212472),
                            INT64_C(-4461760402649505374), INT64_C(       1569855624019)),
      simde_mm256_set_epi32(INT32_C(     393479), INT32_C(  -33215582), INT32_C(  220007333), INT32_C(         93),
                            INT32_C(   53663901),           INT32_MAX ,           INT32_MIN ,           INT32_MAX ) },
    { simde_mm512_set_epi64(INT64_C(             -360269), INT64_C(             -838041),
                            INT64_C(      68701792440150), INT64_C(      -8279642051532),
                            INT64_C(         12194321408), INT64_C(  -16610408706990781),
                            INT64_C(         31156899843), INT64_C(      -1087719579897)),
      simde_mm256_set_epi32(INT32_C(    -360269), INT32_C(    -838041),           INT32_MAX ,           INT32_MIN ,
                                      INT32_MAX ,           INT32_MIN ,           INT32_MAX ,           INT32_MIN ) },
    { simde_mm512_set_epi64(INT64_C(          -583546092), INT64_C(         -1051671055),
                            INT64_C(         -4042501098), INT64_C(        -81260673017),
                            INT64_C(               16230), INT64_C(                  -3),
                            INT64_C(      15889333151091), INT64_C(       -747559120874)),
      simde_mm256_set_epi32(INT32_C( -583546092), INT32_C(-1051671055),           INT32_MIN ,           INT32_MIN ,
                            INT32_C(      16230), INT32_C(         -3),           INT32_MAX ,           INT32_MIN ) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_cvtsepi64_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_cvtsepi64_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m256i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C( -783872818), INT32_C(  224826276), INT32_C(  833953142), INT32_C(  704393899),
                            INT32_C(-1282792525), INT32_C(-1682931810), INT32_C( 1715663188), INT32_C(-1736532826)),
      UINT8_C(184),
      simde_mm512_set_epi64(INT64_C( -304424330925858839), INT64_C( 6920061078002113788),
                            INT64_C( 5392365268250659625), INT64_C( 7451707948021880068),
                            INT64_C(-8094918606566868518), INT64_C( 4039990350021559518),
                            INT64_C(-5079608809355355713), INT64_C( 4544449552448075830)),
      simde_mm256_set_epi32(INT32_MIN, INT32_C(  224826276), INT32_C( 2147483647), INT32_C( 2147483647),
                            INT32_MIN, INT32_C(-1682931810), INT32_C( 1715663188), INT32_C(-1736532826)) },
    { simde_mm256_set_epi32(INT32_C(  228147153), INT32_C( 1769984359), INT32_C( -876802121), INT32_C( -647616802),
                            INT32_C(-1031550247), INT32_C( -304505632), INT32_C(  452449791), INT32_C( -324230599)),
      UINT8_C( 65),
      simde_mm512_set_epi64(INT64_C(    7961945492670935), INT64_C(-8501526825185861704),
                            INT64_C( 8872615053841923423), INT64_C( -146553742277366704),
                            INT64_C(  634942473771684266), INT64_C(-8149949883580988621),
                            INT64_C(-5593558253107684500), INT64_C( -775182801274770337)),
      simde_mm256_set_epi32(INT32_C(  228147153), INT32_MIN, INT32_C( -876802121), INT32_C( -647616802),
                            INT32_C(-1031550247), INT32_C( -304505632), INT32_C(  452449791), INT32_MIN) },
    { simde_mm256_set_epi32(INT32_C( 1131322125), INT32_C(-1676568554), INT32_C( -296320699), INT32_C( 1382446137),
                            INT32_C( 1414899713), INT32_C(  334199932), INT32_C( -144765655), INT32_C( 1792521151)),
      UINT8_C( 87),
      simde_mm512_set_epi64(INT64_C( 5926848964672818106), INT64_C( 5125881974217215893),
                            INT64_C(-4592732829056100767), INT64_C(-5617139660166741281),
                            INT64_C( 8408189234922807969), INT64_C(   46860512120483389),
                            INT64_C( 4524750203780244200), INT64_C(-5543529261566757021)),
      simde_mm256_set_epi32(INT32_C( 1131322125), INT32_C( 2147483647), INT32_C( -296320699), INT32_MIN,
                            INT32_C( 1414899713), INT32_C( 2147483647), INT32_C( 2147483647), INT32_MIN) },
    { simde_mm256_set_epi32(INT32_C(  492196155), INT32_C(  311401987), INT32_C(-1826918634), INT32_C(  466373339),
                            INT32_C(  793993811), INT32_C( -275847262), INT32_C( 1108600799), INT32_C( -126394960)),
      UINT8_C(202),
      simde_mm512_set_epi64(INT64_C( 5858518275788586214), INT64_C( 6325882691723539358),
                            INT64_C( 1467602667059784657), INT64_C(-5469777716636073985),
                            INT64_C(-4718566456661627980), INT64_C(-8823362601370356535),
                            INT64_C( 3910769586217516674), INT64_C( 5379736184341507060)),
      simde_mm256_set_epi32(INT32_C( 2147483647), INT32_C( 2147483647), INT32_C(-1826918634), INT32_C(  466373339),
                            INT32_MIN, INT32_C( -275847262), INT32_C( 2147483647), INT32_C( -126394960)) },
    { simde_mm256_set_epi32(INT32_C( 1821029326), INT32_C( 1839096524), INT32_C(  594162196), INT32_C(-2140380523),
                            INT32_C(  445328546), INT32_C( 1418264985), INT32_C(  986213431), INT32_C(-1974658128)),
      UINT8_C(130),
      simde_mm512_set_epi64(INT64_C( 4654693694425511327), INT64_C( 4987555405601157538),
                            INT64_C(-3482406609442521151), INT64_C(  -74847652664381928),
                            INT64_C( 8566618987618331801), INT64_C( 7168824261247067964),
                            INT64_C( 3636052218697288232), INT64_C( 1769491692739488645)),
      simde_mm256_set_epi32(INT32_C( 2147483647), INT32_C( 1839096524), INT32_C(  594162196), INT32_C(-2140380523),
                            INT32_C(  445328546), INT32_C( 1418264985), INT32_C( 2147483647), INT32_C(-1974658128)) },
    { simde_mm256_set_epi32(INT32_C( -386374366), INT32_C( 1264203001), INT32_C(  448591089), INT32_C( -477219931),
                            INT32_C(  412806942), INT32_C( 1851486104), INT32_C(-1806146323), INT32_C(  -50243203)),
      UINT8_C(  4),
      simde_mm512_set_epi64(INT64_C( 2504249540805731307), INT64_C(-4838934383064989013),
                            INT64_C( 8244741120151641494), INT64_C( 4465455276942587795),
                            INT64_C(-6984757728940989206), INT64_C(-6611841259510994786),
                            INT64_C(-5737746940667504182), INT64_C( 5990707151189672678)),
      simde_mm256_set_epi32(INT32_C( -386374366), INT32_C( 1264203001), INT32_C(  448591089), INT32_C( -477219931),
                            INT32_C(  412806942), INT32_MIN, INT32_C(-1806146323), INT32_C(  -50243203)) },
    { simde_mm256_set_epi32(INT32_C(  694739216), INT32_C( -856286446), INT32_C( 1100751549), INT32_C(-1569791268),
                            INT32_C(-1217857754), INT32_C(-1399806754), INT32_C(  531097456), INT32_C(-1371358872)),
      UINT8_C( 39),
      simde_mm512_set_epi64(INT64_C( 5846906676369557746), INT64_C(-2200894904770223764),
                            INT64_C( 3050515010749934860), INT64_C( 1771796209347881841),
                            INT64_C(-8449543381956761958), INT64_C(-8910958770806120087),
                            INT64_C( 4697259477864506225), INT64_C( 1343234160387478022)),
      simde_mm256_set_epi32(INT32_C(  694739216), INT32_C( -856286446), INT32_C( 2147483647), INT32_C(-1569791268),
                            INT32_C(-1217857754), INT32_MIN, INT32_C( 2147483647), INT32_C( 2147483647)) },
    { simde_mm256_set_epi32(INT32_C(-1373417385), INT32_C(  748657731), INT32_C( 1483755877), INT32_C( 2104018565),
                            INT32_C(-1239060882), INT32_C( 1131976241), INT32_C( 1213166902), INT32_C( 1657511546)),
      UINT8_C( 72),
      simde_mm512_set_epi64(INT64_C( 8062252864411540983), INT64_C(-8797914700722318653),
                            INT64_C(-7262681553986948486), INT64_C( 5851217154060748384),
                            INT64_C(-8009878144581101693), INT64_C(-5600336949614751156),
                            INT64_C( 6240654256081358479), INT64_C(-7094305606247183129)),
      simde_mm256_set_epi32(INT32_C(-1373417385), INT32_MIN, INT32_C( 1483755877), INT32_C( 2104018565),
                            INT32_MIN, INT32_C( 1131976241), INT32_C( 1213166902), INT32_C( 1657511546)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_mask_cvtsepi64_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_cvtsepi64_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { UINT8_C(166),
      simde_mm512_set_epi64(INT64_C(-8094918606566868518), INT64_C( 4039990350021559518),
                            INT64_C(-5079608809355355713), INT64_C( 4544449552448075830),
                            INT64_C(  214030627736325326), INT64_C(  965621503535422838),
                            INT64_C( 3025348762719101875), INT64_C(-7228137083632422572)),
      simde_mm256_set_epi32(INT32_MIN, INT32_C(          0), INT32_MIN, INT32_C(          0),
                            INT32_C(          0), INT32_C( 2147483647), INT32_C( 2147483647), INT32_C(          0)) },
    { UINT8_C(  4),
      simde_mm512_set_epi64(INT64_C(-2040127053262929967), INT64_C( 7602024939754688439),
                            INT64_C(-2781492981666690343), INT64_C(-1307841730435361281),
                            INT64_C(-1392559814843402321), INT64_C( 1598109267473792738),
                            INT64_C(-1562013513342941276), INT64_C( 4124183631221082341)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 2147483647), INT32_C(          0), INT32_C(          0)) },
    { UINT8_C( 95),
      simde_mm512_set_epi64(INT64_C( 7698819720935131480), INT64_C(-1959125385312173467),
                            INT64_C( 4636349956089830497), INT64_C(-2231536501614225883),
                            INT64_C(-2662282344960899888), INT64_C(-2740420949015425086),
                            INT64_C(-2599910470561320857), INT64_C(-7129238424086773927)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_MIN, INT32_C(          0), INT32_MIN,
                            INT32_MIN, INT32_MIN, INT32_MIN, INT32_MIN) },
    { UINT8_C( 41),
      simde_mm512_set_epi64(INT64_C( 7926549609163252623), INT64_C(-8658624789933032605),
                            INT64_C( 3670203061911626838), INT64_C( 5886946783247035777),
                            INT64_C(-1561491900908529833), INT64_C( 4858991530734622742),
                            INT64_C(-1272687709950413767), INT64_C( 6076947994788985980)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C( 2147483647), INT32_C(          0),
                            INT32_MIN, INT32_C(          0), INT32_C(          0), INT32_C( 2147483647)) },
    { UINT8_C( 61),
      simde_mm512_set_epi64(INT64_C( 9006543894860367562), INT64_C( 2113966389253348867),
                            INT64_C(-7846555785016620325), INT64_C( 3410177455490525090),
                            INT64_C( 4761404180193041840), INT64_C( 5926848964672818106),
                            INT64_C( 5125881974217215893), INT64_C(-4592732829056100767)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_MIN, INT32_C( 2147483647),
                            INT32_C( 2147483647), INT32_C( 2147483647), INT32_C(          0), INT32_MIN) },
    { UINT8_C(142),
      simde_mm512_set_epi64(INT64_C( 4235754435341261744), INT64_C( 5858518275788586214),
                            INT64_C( 6325882691723539358), INT64_C( 1467602667059784657),
                            INT64_C(-5469777716636073985), INT64_C(-4718566456661627980),
                            INT64_C(-8823362601370356535), INT64_C( 3910769586217516674)),
      simde_mm256_set_epi32(INT32_C( 2147483647), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_MIN, INT32_MIN, INT32_MIN, INT32_C(          0)) },
    { UINT8_C(153),
      simde_mm512_set_epi64(INT64_C(  -74847652664381928), INT64_C( 8566618987618331801),
                            INT64_C( 7168824261247067964), INT64_C( 3636052218697288232),
                            INT64_C( 1769491692739488645), INT64_C(-4519209933294881842),
                            INT64_C( 7898859425361441300), INT64_C(-9192864346835047262)),
      simde_mm256_set_epi32(INT32_MIN, INT32_C(          0), INT32_C(          0), INT32_C( 2147483647),
                            INT32_C( 2147483647), INT32_C(          0), INT32_C(          0), INT32_MIN) },
    { UINT8_C(193),
      simde_mm512_set_epi64(INT64_C( 5990707151189672678), INT64_C( 2863208564165993762),
                            INT64_C( 5429710545248646385), INT64_C(-2049643996231569634),
                            INT64_C( 7952072268167275757), INT64_C( -215792912647533760),
                            INT64_C(-2126906703336744137), INT64_C( 6372507310579515811)),
      simde_mm256_set_epi32(INT32_C( 2147483647), INT32_C( 2147483647), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C( 2147483647)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_maskz_cvtsepi64_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_div_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -745.89), SIMDE_FLOAT32_C(   663.97), SIMDE_FLOAT32_C(   886.69), SIMDE_FLOAT32_C(  -271.39),
                         SIMDE_FLOAT32_C(   845.36), SIMDE_FLOAT32_C(  -391.34), SIMDE_FLOAT32_C(  -606.86), SIMDE_FLOAT32_C(   818.59),
                         SIMDE_FLOAT32_C(   953.36), SIMDE_FLOAT32_C(   863.40), SIMDE_FLOAT32_C(   241.85), SIMDE_FLOAT32_C(  -815.86),
                         SIMDE_FLOAT32_C(   460.12), SIMDE_FLOAT32_C(  -674.64), SIMDE_FLOAT32_C(   868.62), SIMDE_FLOAT32_C(  -710.40)),
      UINT16_C( 9207),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -956.83), SIMDE_FLOAT32_C(  -855.01), SIMDE_FLOAT32_C(  -219.41), SIMDE_FLOAT32_C(    94.89),
                         SIMDE_FLOAT32_C(  -270.85), SIMDE_FLOAT32_C(   356.85), SIMDE_FLOAT32_C(   872.24), SIMDE_FLOAT32_C(   100.53),
                         SIMDE_FLOAT32_C(   234.39), SIMDE_FLOAT32_C(  -639.13), SIMDE_FLOAT32_C(   981.49), SIMDE_FLOAT32_C(   706.62),
                         SIMDE_FLOAT32_C(  -983.90), SIMDE_FLOAT32_C(   124.15), SIMDE_FLOAT32_C(  -820.87), SIMDE_FLOAT32_C(  -946.81)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -17.46), SIMDE_FLOAT32_C(   371.67), SIMDE_FLOAT32_C(   390.77), SIMDE_FLOAT32_C(  -132.40),
                         SIMDE_FLOAT32_C(   276.69), SIMDE_FLOAT32_C(  -338.80), SIMDE_FLOAT32_C(   359.09), SIMDE_FLOAT32_C(  -631.66),
                         SIMDE_FLOAT32_C(  -455.96), SIMDE_FLOAT32_C(    16.63), SIMDE_FLOAT32_C(   194.96), SIMDE_FLOAT32_C(  -407.18),
                         SIMDE_FLOAT32_C(  -447.59), SIMDE_FLOAT32_C(  -276.48), SIMDE_FLOAT32_C(   631.98), SIMDE_FLOAT32_C(   430.67)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -745.89), SIMDE_FLOAT32_C(   663.97), SIMDE_FLOAT32_C(    -0.56), SIMDE_FLOAT32_C(  -271.39),
                         SIMDE_FLOAT32_C(   845.36), SIMDE_FLOAT32_C(  -391.34), SIMDE_FLOAT32_C(     2.43), SIMDE_FLOAT32_C(    -0.16),
                         SIMDE_FLOAT32_C(    -0.51), SIMDE_FLOAT32_C(   -38.43), SIMDE_FLOAT32_C(     5.03), SIMDE_FLOAT32_C(    -1.74),
                         SIMDE_FLOAT32_C(   460.12), SIMDE_FLOAT32_C(    -0.45), SIMDE_FLOAT32_C(    -1.30), SIMDE_FLOAT32_C(    -2.20)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   769.85), SIMDE_FLOAT32_C(   -75.51), SIMDE_FLOAT32_C(   417.80), SIMDE_FLOAT32_C(   327.87),
                         SIMDE_FLOAT32_C(   287.24), SIMDE_FLOAT32_C(  -627.46), SIMDE_FLOAT32_C(   540.48), SIMDE_FLOAT32_C(  -625.88),
                         SIMDE_FLOAT32_C(  -108.88), SIMDE_FLOAT32_C(   663.67), SIMDE_FLOAT32_C(  -412.74), SIMDE_FLOAT32_C(  -226.36),
                         SIMDE_FLOAT32_C(   757.77), SIMDE_FLOAT32_C(  -897.40), SIMDE_FLOAT32_C(    27.15), SIMDE_FLOAT32_C(  -443.34)),
      UINT16_C(26651),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -896.67), SIMDE_FLOAT32_C(  -181.49), SIMDE_FLOAT32_C(  -338.89), SIMDE_FLOAT32_C(   -19.28),
                         SIMDE_FLOAT32_C(   886.35), SIMDE_FLOAT32_C(  -662.07), SIMDE_FLOAT32_C(   925.60), SIMDE_FLOAT32_C(   651.41),
                         SIMDE_FLOAT32_C(   597.16), SIMDE_FLOAT32_C(    67.32), SIMDE_FLOAT32_C(  -911.68), SIMDE_FLOAT32_C(   202.35),
                         SIMDE_FLOAT32_C(   208.06), SIMDE_FLOAT32_C(   747.61), SIMDE_FLOAT32_C(    81.71), SIMDE_FLOAT32_C(    40.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   429.04), SIMDE_FLOAT32_C(    13.69), SIMDE_FLOAT32_C(   491.03), SIMDE_FLOAT32_C(   366.42),
                         SIMDE_FLOAT32_C(  -264.60), SIMDE_FLOAT32_C(   201.75), SIMDE_FLOAT32_C(  -598.58), SIMDE_FLOAT32_C(  -939.94),
                         SIMDE_FLOAT32_C(   118.06), SIMDE_FLOAT32_C(   355.92), SIMDE_FLOAT32_C(   978.59), SIMDE_FLOAT32_C(  -224.11),
                         SIMDE_FLOAT32_C(   -71.37), SIMDE_FLOAT32_C(   333.99), SIMDE_FLOAT32_C(  -515.40), SIMDE_FLOAT32_C(   -38.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   769.85), SIMDE_FLOAT32_C(   -13.26), SIMDE_FLOAT32_C(    -0.69), SIMDE_FLOAT32_C(   327.87),
                         SIMDE_FLOAT32_C(    -3.35), SIMDE_FLOAT32_C(  -627.46), SIMDE_FLOAT32_C(   540.48), SIMDE_FLOAT32_C(  -625.88),
                         SIMDE_FLOAT32_C(  -108.88), SIMDE_FLOAT32_C(   663.67), SIMDE_FLOAT32_C(  -412.74), SIMDE_FLOAT32_C(    -0.90),
                         SIMDE_FLOAT32_C(    -2.92), SIMDE_FLOAT32_C(  -897.40), SIMDE_FLOAT32_C(    -0.16), SIMDE_FLOAT32_C(    -1.07)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -301.18), SIMDE_FLOAT32_C(  -952.56), SIMDE_FLOAT32_C(   361.18), SIMDE_FLOAT32_C(    53.08),
                         SIMDE_FLOAT32_C(   179.94), SIMDE_FLOAT32_C(  -914.68), SIMDE_FLOAT32_C(  -695.32), SIMDE_FLOAT32_C(  -492.39),
                         SIMDE_FLOAT32_C(   -86.02), SIMDE_FLOAT32_C(   123.88), SIMDE_FLOAT32_C(   274.86), SIMDE_FLOAT32_C(   554.74),
                         SIMDE_FLOAT32_C(  -845.80), SIMDE_FLOAT32_C(  -156.28), SIMDE_FLOAT32_C(  -737.31), SIMDE_FLOAT32_C(   590.88)),
      UINT16_C(31164),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   694.79), SIMDE_FLOAT32_C(   687.99), SIMDE_FLOAT32_C(  -648.58), SIMDE_FLOAT32_C(  -272.14),
                         SIMDE_FLOAT32_C(   -52.85), SIMDE_FLOAT32_C(  -298.63), SIMDE_FLOAT32_C(   917.21), SIMDE_FLOAT32_C(  -876.76),
                         SIMDE_FLOAT32_C(   677.44), SIMDE_FLOAT32_C(  -857.42), SIMDE_FLOAT32_C(   -56.60), SIMDE_FLOAT32_C(   488.58),
                         SIMDE_FLOAT32_C(   876.79), SIMDE_FLOAT32_C(  -578.18), SIMDE_FLOAT32_C(  -335.03), SIMDE_FLOAT32_C(   980.62)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   247.15), SIMDE_FLOAT32_C(   126.77), SIMDE_FLOAT32_C(   867.78), SIMDE_FLOAT32_C(  -450.16),
                         SIMDE_FLOAT32_C(    94.73), SIMDE_FLOAT32_C(  -587.88), SIMDE_FLOAT32_C(   776.58), SIMDE_FLOAT32_C(  -595.96),
                         SIMDE_FLOAT32_C(   345.82), SIMDE_FLOAT32_C(  -768.91), SIMDE_FLOAT32_C(   -31.17), SIMDE_FLOAT32_C(    -4.10),
                         SIMDE_FLOAT32_C(  -234.58), SIMDE_FLOAT32_C(   278.53), SIMDE_FLOAT32_C(  -336.24), SIMDE_FLOAT32_C(  -974.01)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -301.18), SIMDE_FLOAT32_C(     5.43), SIMDE_FLOAT32_C(    -0.75), SIMDE_FLOAT32_C(     0.60),
                         SIMDE_FLOAT32_C(    -0.56), SIMDE_FLOAT32_C(  -914.68), SIMDE_FLOAT32_C(  -695.32), SIMDE_FLOAT32_C(     1.47),
                         SIMDE_FLOAT32_C(     1.96), SIMDE_FLOAT32_C(   123.88), SIMDE_FLOAT32_C(     1.82), SIMDE_FLOAT32_C(  -119.17),
                         SIMDE_FLOAT32_C(    -3.74), SIMDE_FLOAT32_C(    -2.08), SIMDE_FLOAT32_C(  -737.31), SIMDE_FLOAT32_C(   590.88)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -412.81), SIMDE_FLOAT32_C(  -265.44), SIMDE_FLOAT32_C(  -550.71), SIMDE_FLOAT32_C(  -725.27),
                         SIMDE_FLOAT32_C(  -302.34), SIMDE_FLOAT32_C(  -375.85), SIMDE_FLOAT32_C(   423.25), SIMDE_FLOAT32_C(   778.83),
                         SIMDE_FLOAT32_C(   480.23), SIMDE_FLOAT32_C(  -401.59), SIMDE_FLOAT32_C(   489.09), SIMDE_FLOAT32_C(   775.60),
                         SIMDE_FLOAT32_C(  -569.06), SIMDE_FLOAT32_C(  -632.55), SIMDE_FLOAT32_C(  -156.10), SIMDE_FLOAT32_C(   658.93)),
      UINT16_C( 3671),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -474.43), SIMDE_FLOAT32_C(  -465.91), SIMDE_FLOAT32_C(   545.15), SIMDE_FLOAT32_C(   262.71),
                         SIMDE_FLOAT32_C(   599.41), SIMDE_FLOAT32_C(  -408.02), SIMDE_FLOAT32_C(  -664.44), SIMDE_FLOAT32_C(   446.87),
                         SIMDE_FLOAT32_C(   816.32), SIMDE_FLOAT32_C(   622.16), SIMDE_FLOAT32_C(    40.97), SIMDE_FLOAT32_C(  -230.30),
                         SIMDE_FLOAT32_C(   122.84), SIMDE_FLOAT32_C(   457.98), SIMDE_FLOAT32_C(  -118.87), SIMDE_FLOAT32_C(  -211.46)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   271.75), SIMDE_FLOAT32_C(   598.59), SIMDE_FLOAT32_C(  -130.09), SIMDE_FLOAT32_C(   474.47),
                         SIMDE_FLOAT32_C(   -94.60), SIMDE_FLOAT32_C(   846.28), SIMDE_FLOAT32_C(   108.99), SIMDE_FLOAT32_C(  -793.22),
                         SIMDE_FLOAT32_C(   -12.05), SIMDE_FLOAT32_C(  -325.70), SIMDE_FLOAT32_C(  -510.95), SIMDE_FLOAT32_C(   213.60),
                         SIMDE_FLOAT32_C(  -818.29), SIMDE_FLOAT32_C(  -431.12), SIMDE_FLOAT32_C(  -186.49), SIMDE_FLOAT32_C(    53.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -412.81), SIMDE_FLOAT32_C(  -265.44), SIMDE_FLOAT32_C(  -550.71), SIMDE_FLOAT32_C(  -725.27),
                         SIMDE_FLOAT32_C(    -6.34), SIMDE_FLOAT32_C(    -0.48), SIMDE_FLOAT32_C(    -6.10), SIMDE_FLOAT32_C(   778.83),
                         SIMDE_FLOAT32_C(   480.23), SIMDE_FLOAT32_C(    -1.91), SIMDE_FLOAT32_C(   489.09), SIMDE_FLOAT32_C(    -1.08),
                         SIMDE_FLOAT32_C(  -569.06), SIMDE_FLOAT32_C(    -1.06), SIMDE_FLOAT32_C(     0.64), SIMDE_FLOAT32_C(    -3.97)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -254.94), SIMDE_FLOAT32_C(   234.45), SIMDE_FLOAT32_C(   235.56), SIMDE_FLOAT32_C(   930.35),
                         SIMDE_FLOAT32_C(   137.93), SIMDE_FLOAT32_C(   979.46), SIMDE_FLOAT32_C(   688.15), SIMDE_FLOAT32_C(   707.95),
                         SIMDE_FLOAT32_C(    35.42), SIMDE_FLOAT32_C(   748.55), SIMDE_FLOAT32_C(   649.98), SIMDE_FLOAT32_C(   702.04),
                         SIMDE_FLOAT32_C(   443.56), SIMDE_FLOAT32_C(  -944.39), SIMDE_FLOAT32_C(   717.51), SIMDE_FLOAT32_C(   716.62)),
      UINT16_C(24144),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -982.71), SIMDE_FLOAT32_C(   639.75), SIMDE_FLOAT32_C(   842.03), SIMDE_FLOAT32_C(   717.68),
                         SIMDE_FLOAT32_C(   294.25), SIMDE_FLOAT32_C(  -411.52), SIMDE_FLOAT32_C(   632.28), SIMDE_FLOAT32_C(   531.91),
                         SIMDE_FLOAT32_C(  -198.66), SIMDE_FLOAT32_C(   722.92), SIMDE_FLOAT32_C(  -890.25), SIMDE_FLOAT32_C(   -36.77),
                         SIMDE_FLOAT32_C(  -651.17), SIMDE_FLOAT32_C(   559.24), SIMDE_FLOAT32_C(   496.39), SIMDE_FLOAT32_C(  -143.68)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -779.19), SIMDE_FLOAT32_C(   587.24), SIMDE_FLOAT32_C(   850.25), SIMDE_FLOAT32_C(   172.75),
                         SIMDE_FLOAT32_C(   237.73), SIMDE_FLOAT32_C(   792.79), SIMDE_FLOAT32_C(  -225.26), SIMDE_FLOAT32_C(   810.16),
                         SIMDE_FLOAT32_C(   235.61), SIMDE_FLOAT32_C(   123.68), SIMDE_FLOAT32_C(  -869.51), SIMDE_FLOAT32_C(   811.23),
                         SIMDE_FLOAT32_C(   292.28), SIMDE_FLOAT32_C(   158.60), SIMDE_FLOAT32_C(  -861.10), SIMDE_FLOAT32_C(   297.31)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -254.94), SIMDE_FLOAT32_C(     1.09), SIMDE_FLOAT32_C(   235.56), SIMDE_FLOAT32_C(     4.15),
                         SIMDE_FLOAT32_C(     1.24), SIMDE_FLOAT32_C(    -0.52), SIMDE_FLOAT32_C(    -2.81), SIMDE_FLOAT32_C(   707.95),
                         SIMDE_FLOAT32_C(    35.42), SIMDE_FLOAT32_C(     5.85), SIMDE_FLOAT32_C(   649.98), SIMDE_FLOAT32_C(    -0.05),
                         SIMDE_FLOAT32_C(   443.56), SIMDE_FLOAT32_C(  -944.39), SIMDE_FLOAT32_C(   717.51), SIMDE_FLOAT32_C(   716.62)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   711.46), SIMDE_FLOAT32_C(  -417.42), SIMDE_FLOAT32_C(  -736.14), SIMDE_FLOAT32_C(  -654.73),
                         SIMDE_FLOAT32_C(  -297.59), SIMDE_FLOAT32_C(   899.88), SIMDE_FLOAT32_C(   819.21), SIMDE_FLOAT32_C(  -451.55),
                         SIMDE_FLOAT32_C(   831.09), SIMDE_FLOAT32_C(   694.55), SIMDE_FLOAT32_C(  -231.88), SIMDE_FLOAT32_C(  -711.25),
                         SIMDE_FLOAT32_C(  -213.96), SIMDE_FLOAT32_C(  -411.84), SIMDE_FLOAT32_C(  -325.79), SIMDE_FLOAT32_C(  -424.22)),
      UINT16_C( 4465),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   985.56), SIMDE_FLOAT32_C(  -969.44), SIMDE_FLOAT32_C(   -91.63), SIMDE_FLOAT32_C(  -416.19),
                         SIMDE_FLOAT32_C(   716.00), SIMDE_FLOAT32_C(   579.33), SIMDE_FLOAT32_C(   678.78), SIMDE_FLOAT32_C(   650.46),
                         SIMDE_FLOAT32_C(  -988.30), SIMDE_FLOAT32_C(   206.47), SIMDE_FLOAT32_C(   214.00), SIMDE_FLOAT32_C(  -226.18),
                         SIMDE_FLOAT32_C(  -410.63), SIMDE_FLOAT32_C(  -238.02), SIMDE_FLOAT32_C(   520.82), SIMDE_FLOAT32_C(  -882.63)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   373.48), SIMDE_FLOAT32_C(  -376.14), SIMDE_FLOAT32_C(   103.99), SIMDE_FLOAT32_C(   900.82),
                         SIMDE_FLOAT32_C(   827.14), SIMDE_FLOAT32_C(   -50.15), SIMDE_FLOAT32_C(   675.06), SIMDE_FLOAT32_C(   239.90),
                         SIMDE_FLOAT32_C(   531.97), SIMDE_FLOAT32_C(    52.69), SIMDE_FLOAT32_C(  -376.06), SIMDE_FLOAT32_C(  -290.42),
                         SIMDE_FLOAT32_C(  -325.12), SIMDE_FLOAT32_C(  -471.17), SIMDE_FLOAT32_C(  -511.21), SIMDE_FLOAT32_C(   -90.11)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   711.46), SIMDE_FLOAT32_C(  -417.42), SIMDE_FLOAT32_C(  -736.14), SIMDE_FLOAT32_C(    -0.46),
                         SIMDE_FLOAT32_C(  -297.59), SIMDE_FLOAT32_C(   899.88), SIMDE_FLOAT32_C(   819.21), SIMDE_FLOAT32_C(     2.71),
                         SIMDE_FLOAT32_C(   831.09), SIMDE_FLOAT32_C(     3.92), SIMDE_FLOAT32_C(    -0.57), SIMDE_FLOAT32_C(     0.78),
                         SIMDE_FLOAT32_C(  -213.96), SIMDE_FLOAT32_C(  -411.84), SIMDE_FLOAT32_C(  -325.79), SIMDE_FLOAT32_C(     9.80)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -788.99), SIMDE_FLOAT32_C(   888.94), SIMDE_FLOAT32_C(   861.99), SIMDE_FLOAT32_C(  -655.94),
                         SIMDE_FLOAT32_C(  -815.78), SIMDE_FLOAT32_C(   460.30), SIMDE_FLOAT32_C(  -596.09), SIMDE_FLOAT32_C(   480.08),
                         SIMDE_FLOAT32_C(  -800.23), SIMDE_FLOAT32_C(  -511.53), SIMDE_FLOAT32_C(   235.71), SIMDE_FLOAT32_C(   833.52),
                         SIMDE_FLOAT32_C(   343.49), SIMDE_FLOAT32_C(   413.97), SIMDE_FLOAT32_C(   264.73), SIMDE_FLOAT32_C(   769.22)),
      UINT16_C(57880),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -782.73), SIMDE_FLOAT32_C(   -41.33), SIMDE_FLOAT32_C(   183.64), SIMDE_FLOAT32_C(     0.86),
                         SIMDE_FLOAT32_C(  -449.70), SIMDE_FLOAT32_C(   153.64), SIMDE_FLOAT32_C(   543.55), SIMDE_FLOAT32_C(  -321.17),
                         SIMDE_FLOAT32_C(   944.46), SIMDE_FLOAT32_C(  -863.15), SIMDE_FLOAT32_C(   155.57), SIMDE_FLOAT32_C(   671.09),
                         SIMDE_FLOAT32_C(   138.46), SIMDE_FLOAT32_C(   937.90), SIMDE_FLOAT32_C(   367.36), SIMDE_FLOAT32_C(  -187.79)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -177.92), SIMDE_FLOAT32_C(   382.46), SIMDE_FLOAT32_C(  -344.53), SIMDE_FLOAT32_C(   306.51),
                         SIMDE_FLOAT32_C(   804.79), SIMDE_FLOAT32_C(    74.50), SIMDE_FLOAT32_C(  -171.92), SIMDE_FLOAT32_C(  -865.07),
                         SIMDE_FLOAT32_C(   788.06), SIMDE_FLOAT32_C(  -723.82), SIMDE_FLOAT32_C(    43.98), SIMDE_FLOAT32_C(  -303.25),
                         SIMDE_FLOAT32_C(  -511.21), SIMDE_FLOAT32_C(   460.56), SIMDE_FLOAT32_C(   217.57), SIMDE_FLOAT32_C(  -900.02)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     4.40), SIMDE_FLOAT32_C(    -0.11), SIMDE_FLOAT32_C(    -0.53), SIMDE_FLOAT32_C(  -655.94),
                         SIMDE_FLOAT32_C(  -815.78), SIMDE_FLOAT32_C(   460.30), SIMDE_FLOAT32_C(    -3.16), SIMDE_FLOAT32_C(   480.08),
                         SIMDE_FLOAT32_C(  -800.23), SIMDE_FLOAT32_C(  -511.53), SIMDE_FLOAT32_C(   235.71), SIMDE_FLOAT32_C(    -2.21),
                         SIMDE_FLOAT32_C(    -0.27), SIMDE_FLOAT32_C(   413.97), SIMDE_FLOAT32_C(   264.73), SIMDE_FLOAT32_C(   769.22)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    54.65), SIMDE_FLOAT32_C(   276.51), SIMDE_FLOAT32_C(   227.98), SIMDE_FLOAT32_C(  -128.31),
                         SIMDE_FLOAT32_C(  -191.48), SIMDE_FLOAT32_C(   348.63), SIMDE_FLOAT32_C(   444.48), SIMDE_FLOAT32_C(   206.11),
                         SIMDE_FLOAT32_C(  -692.44), SIMDE_FLOAT32_C(  -865.72), SIMDE_FLOAT32_C(   763.64), SIMDE_FLOAT32_C(  -849.66),
                         SIMDE_FLOAT32_C(   804.26), SIMDE_FLOAT32_C(   570.08), SIMDE_FLOAT32_C(   125.91), SIMDE_FLOAT32_C(   149.60)),
      UINT16_C(24771),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   747.34), SIMDE_FLOAT32_C(   607.83), SIMDE_FLOAT32_C(    25.24), SIMDE_FLOAT32_C(  -542.52),
                         SIMDE_FLOAT32_C(   568.70), SIMDE_FLOAT32_C(   899.42), SIMDE_FLOAT32_C(   120.86), SIMDE_FLOAT32_C(  -424.59),
                         SIMDE_FLOAT32_C(   377.13), SIMDE_FLOAT32_C(   761.91), SIMDE_FLOAT32_C(  -902.23), SIMDE_FLOAT32_C(  -759.84),
                         SIMDE_FLOAT32_C(   430.99), SIMDE_FLOAT32_C(   555.32), SIMDE_FLOAT32_C(  -397.14), SIMDE_FLOAT32_C(   608.52)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -615.94), SIMDE_FLOAT32_C(   113.16), SIMDE_FLOAT32_C(    26.04), SIMDE_FLOAT32_C(  -142.02),
                         SIMDE_FLOAT32_C(   273.49), SIMDE_FLOAT32_C(   374.88), SIMDE_FLOAT32_C(   453.99), SIMDE_FLOAT32_C(  -241.36),
                         SIMDE_FLOAT32_C(   181.97), SIMDE_FLOAT32_C(   143.35), SIMDE_FLOAT32_C(   400.04), SIMDE_FLOAT32_C(   610.27),
                         SIMDE_FLOAT32_C(  -726.06), SIMDE_FLOAT32_C(  -819.96), SIMDE_FLOAT32_C(   674.91), SIMDE_FLOAT32_C(   406.86)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    54.65), SIMDE_FLOAT32_C(     5.37), SIMDE_FLOAT32_C(     0.97), SIMDE_FLOAT32_C(  -128.31),
                         SIMDE_FLOAT32_C(  -191.48), SIMDE_FLOAT32_C(   348.63), SIMDE_FLOAT32_C(   444.48), SIMDE_FLOAT32_C(   206.11),
                         SIMDE_FLOAT32_C(     2.07), SIMDE_FLOAT32_C(     5.32), SIMDE_FLOAT32_C(   763.64), SIMDE_FLOAT32_C(  -849.66),
                         SIMDE_FLOAT32_C(   804.26), SIMDE_FLOAT32_C(   570.08), SIMDE_FLOAT32_C(    -0.59), SIMDE_FLOAT32_C(     1.50)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_div_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_div_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(32824),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   745.69), SIMDE_FLOAT32_C(  -258.59), SIMDE_FLOAT32_C(  -549.06), SIMDE_FLOAT32_C(   646.98),
                         SIMDE_FLOAT32_C(   925.86), SIMDE_FLOAT32_C(   378.90), SIMDE_FLOAT32_C(  -524.10), SIMDE_FLOAT32_C(  -563.31),
                         SIMDE_FLOAT32_C(   112.08), SIMDE_FLOAT32_C(   712.48), SIMDE_FLOAT32_C(  -754.71), SIMDE_FLOAT32_C(   256.61),
                         SIMDE_FLOAT32_C(   768.73), SIMDE_FLOAT32_C(   227.99), SIMDE_FLOAT32_C(   174.97), SIMDE_FLOAT32_C(   338.39)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   841.82), SIMDE_FLOAT32_C(  -330.14), SIMDE_FLOAT32_C(  -498.05), SIMDE_FLOAT32_C(  -706.46),
                         SIMDE_FLOAT32_C(  -284.71), SIMDE_FLOAT32_C(  -940.98), SIMDE_FLOAT32_C(  -491.84), SIMDE_FLOAT32_C(    52.49),
                         SIMDE_FLOAT32_C(   759.92), SIMDE_FLOAT32_C(   629.58), SIMDE_FLOAT32_C(    23.76), SIMDE_FLOAT32_C(   980.95),
                         SIMDE_FLOAT32_C(   224.97), SIMDE_FLOAT32_C(   818.07), SIMDE_FLOAT32_C(  -531.75), SIMDE_FLOAT32_C(  -531.67)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.89), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -31.76), SIMDE_FLOAT32_C(     0.26),
                         SIMDE_FLOAT32_C(     3.42), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C( 4283),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   838.22), SIMDE_FLOAT32_C(   464.78), SIMDE_FLOAT32_C(  -248.37), SIMDE_FLOAT32_C(    28.49),
                         SIMDE_FLOAT32_C(  -176.67), SIMDE_FLOAT32_C(  -468.39), SIMDE_FLOAT32_C(  -893.30), SIMDE_FLOAT32_C(   771.96),
                         SIMDE_FLOAT32_C(  -167.30), SIMDE_FLOAT32_C(  -738.71), SIMDE_FLOAT32_C(  -816.67), SIMDE_FLOAT32_C(    43.31),
                         SIMDE_FLOAT32_C(   -98.40), SIMDE_FLOAT32_C(   217.89), SIMDE_FLOAT32_C(   626.98), SIMDE_FLOAT32_C(  -409.09)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -156.59), SIMDE_FLOAT32_C(   153.69), SIMDE_FLOAT32_C(   895.38), SIMDE_FLOAT32_C(  -242.63),
                         SIMDE_FLOAT32_C(   994.17), SIMDE_FLOAT32_C(  -265.23), SIMDE_FLOAT32_C(   -57.91), SIMDE_FLOAT32_C(  -586.11),
                         SIMDE_FLOAT32_C(  -443.71), SIMDE_FLOAT32_C(  -786.78), SIMDE_FLOAT32_C(   -92.41), SIMDE_FLOAT32_C(  -378.62),
                         SIMDE_FLOAT32_C(   632.49), SIMDE_FLOAT32_C(  -867.20), SIMDE_FLOAT32_C(   977.79), SIMDE_FLOAT32_C(  -788.71)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.12),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.38), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     8.84), SIMDE_FLOAT32_C(    -0.11),
                         SIMDE_FLOAT32_C(    -0.16), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.64), SIMDE_FLOAT32_C(     0.52)) },
    { UINT16_C(27708),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -309.30), SIMDE_FLOAT32_C(  -478.69), SIMDE_FLOAT32_C(  -499.66), SIMDE_FLOAT32_C(  -834.97),
                         SIMDE_FLOAT32_C(  -926.76), SIMDE_FLOAT32_C(   306.74), SIMDE_FLOAT32_C(   350.68), SIMDE_FLOAT32_C(   698.74),
                         SIMDE_FLOAT32_C(  -748.23), SIMDE_FLOAT32_C(   960.31), SIMDE_FLOAT32_C(   -52.56), SIMDE_FLOAT32_C(   -18.49),
                         SIMDE_FLOAT32_C(  -174.79), SIMDE_FLOAT32_C(  -875.70), SIMDE_FLOAT32_C(   270.45), SIMDE_FLOAT32_C(   571.57)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -891.46), SIMDE_FLOAT32_C(   298.88), SIMDE_FLOAT32_C(   907.92), SIMDE_FLOAT32_C(   585.94),
                         SIMDE_FLOAT32_C(   976.00), SIMDE_FLOAT32_C(   860.60), SIMDE_FLOAT32_C(  -807.57), SIMDE_FLOAT32_C(  -501.53),
                         SIMDE_FLOAT32_C(   887.26), SIMDE_FLOAT32_C(  -380.63), SIMDE_FLOAT32_C(   603.15), SIMDE_FLOAT32_C(   906.17),
                         SIMDE_FLOAT32_C(  -446.90), SIMDE_FLOAT32_C(   518.96), SIMDE_FLOAT32_C(   325.09), SIMDE_FLOAT32_C(   394.29)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.60), SIMDE_FLOAT32_C(    -0.55), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -0.95), SIMDE_FLOAT32_C(     0.36), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.09), SIMDE_FLOAT32_C(    -0.02),
                         SIMDE_FLOAT32_C(     0.39), SIMDE_FLOAT32_C(    -1.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(21979),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -722.04), SIMDE_FLOAT32_C(  -251.19), SIMDE_FLOAT32_C(   885.20), SIMDE_FLOAT32_C(  -718.95),
                         SIMDE_FLOAT32_C(  -995.18), SIMDE_FLOAT32_C(   316.41), SIMDE_FLOAT32_C(   425.49), SIMDE_FLOAT32_C(  -889.60),
                         SIMDE_FLOAT32_C(  -764.37), SIMDE_FLOAT32_C(  -698.84), SIMDE_FLOAT32_C(   111.54), SIMDE_FLOAT32_C(   627.05),
                         SIMDE_FLOAT32_C(   619.20), SIMDE_FLOAT32_C(   107.79), SIMDE_FLOAT32_C(   830.07), SIMDE_FLOAT32_C(  -991.50)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    -9.32), SIMDE_FLOAT32_C(   588.15), SIMDE_FLOAT32_C(   740.36), SIMDE_FLOAT32_C(   589.64),
                         SIMDE_FLOAT32_C(  -146.10), SIMDE_FLOAT32_C(   771.62), SIMDE_FLOAT32_C(  -975.31), SIMDE_FLOAT32_C(   550.04),
                         SIMDE_FLOAT32_C(   902.97), SIMDE_FLOAT32_C(  -970.67), SIMDE_FLOAT32_C(  -396.71), SIMDE_FLOAT32_C(   740.42),
                         SIMDE_FLOAT32_C(  -740.07), SIMDE_FLOAT32_C(   691.95), SIMDE_FLOAT32_C(  -434.89), SIMDE_FLOAT32_C(   270.74)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.43), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.22),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.41), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.62),
                         SIMDE_FLOAT32_C(    -0.85), SIMDE_FLOAT32_C(     0.72), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.85),
                         SIMDE_FLOAT32_C(    -0.84), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.91), SIMDE_FLOAT32_C(    -3.66)) },
    { UINT16_C( 1193),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   347.59), SIMDE_FLOAT32_C(   162.65), SIMDE_FLOAT32_C(   724.10), SIMDE_FLOAT32_C(   124.00),
                         SIMDE_FLOAT32_C(  -823.97), SIMDE_FLOAT32_C(  -185.15), SIMDE_FLOAT32_C(    33.85), SIMDE_FLOAT32_C(  -430.54),
                         SIMDE_FLOAT32_C(  -534.02), SIMDE_FLOAT32_C(   815.29), SIMDE_FLOAT32_C(   942.25), SIMDE_FLOAT32_C(  -825.08),
                         SIMDE_FLOAT32_C(   638.03), SIMDE_FLOAT32_C(   599.07), SIMDE_FLOAT32_C(   164.45), SIMDE_FLOAT32_C(   429.94)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -817.35), SIMDE_FLOAT32_C(  -889.89), SIMDE_FLOAT32_C(   528.79), SIMDE_FLOAT32_C(  -600.85),
                         SIMDE_FLOAT32_C(  -168.12), SIMDE_FLOAT32_C(  -798.12), SIMDE_FLOAT32_C(  -637.75), SIMDE_FLOAT32_C(  -580.73),
                         SIMDE_FLOAT32_C(   697.23), SIMDE_FLOAT32_C(   654.25), SIMDE_FLOAT32_C(  -236.09), SIMDE_FLOAT32_C(   234.13),
                         SIMDE_FLOAT32_C(  -696.60), SIMDE_FLOAT32_C(  -486.03), SIMDE_FLOAT32_C(    69.79), SIMDE_FLOAT32_C(   435.18)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.23), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -0.77), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -3.99), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -0.92), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.99)) },
    { UINT16_C(47777),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    96.65), SIMDE_FLOAT32_C(   -38.76), SIMDE_FLOAT32_C(   585.22), SIMDE_FLOAT32_C(  -683.52),
                         SIMDE_FLOAT32_C(   268.64), SIMDE_FLOAT32_C(  -393.28), SIMDE_FLOAT32_C(   102.94), SIMDE_FLOAT32_C(   786.82),
                         SIMDE_FLOAT32_C(   138.90), SIMDE_FLOAT32_C(   225.78), SIMDE_FLOAT32_C(   449.88), SIMDE_FLOAT32_C(   347.32),
                         SIMDE_FLOAT32_C(    33.80), SIMDE_FLOAT32_C(  -559.14), SIMDE_FLOAT32_C(  -159.05), SIMDE_FLOAT32_C(  -491.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -55.95), SIMDE_FLOAT32_C(  -837.50), SIMDE_FLOAT32_C(  -575.23), SIMDE_FLOAT32_C(   248.03),
                         SIMDE_FLOAT32_C(   907.04), SIMDE_FLOAT32_C(   -74.96), SIMDE_FLOAT32_C(  -821.80), SIMDE_FLOAT32_C(  -847.93),
                         SIMDE_FLOAT32_C(  -925.94), SIMDE_FLOAT32_C(   664.01), SIMDE_FLOAT32_C(  -745.59), SIMDE_FLOAT32_C(  -301.31),
                         SIMDE_FLOAT32_C(   146.53), SIMDE_FLOAT32_C(  -440.81), SIMDE_FLOAT32_C(   427.27), SIMDE_FLOAT32_C(  -219.59)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    -1.73), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.02), SIMDE_FLOAT32_C(    -2.76),
                         SIMDE_FLOAT32_C(     0.30), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.13), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -0.15), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.60), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     2.24)) },
    { UINT16_C(50336),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -724.66), SIMDE_FLOAT32_C(  -778.29), SIMDE_FLOAT32_C(  -888.11), SIMDE_FLOAT32_C(  -623.31),
                         SIMDE_FLOAT32_C(   617.08), SIMDE_FLOAT32_C(    42.91), SIMDE_FLOAT32_C(   907.40), SIMDE_FLOAT32_C(  -402.88),
                         SIMDE_FLOAT32_C(  -278.23), SIMDE_FLOAT32_C(  -640.08), SIMDE_FLOAT32_C(   108.85), SIMDE_FLOAT32_C(  -527.72),
                         SIMDE_FLOAT32_C(  -791.82), SIMDE_FLOAT32_C(  -207.31), SIMDE_FLOAT32_C(  -642.88), SIMDE_FLOAT32_C(   536.44)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   766.31), SIMDE_FLOAT32_C(   592.01), SIMDE_FLOAT32_C(   324.90), SIMDE_FLOAT32_C(    55.55),
                         SIMDE_FLOAT32_C(   -34.13), SIMDE_FLOAT32_C(  -588.88), SIMDE_FLOAT32_C(   991.78), SIMDE_FLOAT32_C(  -468.91),
                         SIMDE_FLOAT32_C(    78.86), SIMDE_FLOAT32_C(    18.25), SIMDE_FLOAT32_C(   295.51), SIMDE_FLOAT32_C(  -293.26),
                         SIMDE_FLOAT32_C(  -877.24), SIMDE_FLOAT32_C(   952.33), SIMDE_FLOAT32_C(  -274.18), SIMDE_FLOAT32_C(   654.17)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    -0.95), SIMDE_FLOAT32_C(    -1.31), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.07), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -3.53), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.37), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(  740),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -222.38), SIMDE_FLOAT32_C(  -847.72), SIMDE_FLOAT32_C(  -497.04), SIMDE_FLOAT32_C(   862.38),
                         SIMDE_FLOAT32_C(  -840.40), SIMDE_FLOAT32_C(   998.10), SIMDE_FLOAT32_C(  -257.93), SIMDE_FLOAT32_C(  -204.46),
                         SIMDE_FLOAT32_C(  -373.11), SIMDE_FLOAT32_C(  -912.42), SIMDE_FLOAT32_C(   207.13), SIMDE_FLOAT32_C(   784.69),
                         SIMDE_FLOAT32_C(    82.66), SIMDE_FLOAT32_C(   123.09), SIMDE_FLOAT32_C(  -384.17), SIMDE_FLOAT32_C(  -845.08)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   839.49), SIMDE_FLOAT32_C(  -285.39), SIMDE_FLOAT32_C(  -548.76), SIMDE_FLOAT32_C(   -35.10),
                         SIMDE_FLOAT32_C(  -295.04), SIMDE_FLOAT32_C(   738.77), SIMDE_FLOAT32_C(   340.04), SIMDE_FLOAT32_C(  -585.87),
                         SIMDE_FLOAT32_C(  -711.46), SIMDE_FLOAT32_C(   926.37), SIMDE_FLOAT32_C(   696.23), SIMDE_FLOAT32_C(   766.17),
                         SIMDE_FLOAT32_C(  -330.24), SIMDE_FLOAT32_C(   369.18), SIMDE_FLOAT32_C(  -498.71), SIMDE_FLOAT32_C(  -288.61)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.76), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.52), SIMDE_FLOAT32_C(    -0.98), SIMDE_FLOAT32_C(     0.30), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.33), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_div_ps(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_div_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  -97.83), SIMDE_FLOAT64_C( -393.82),
                         SIMDE_FLOAT64_C(  934.81), SIMDE_FLOAT64_C(   74.53),
                         SIMDE_FLOAT64_C(  843.79), SIMDE_FLOAT64_C(  465.05),
                         SIMDE_FLOAT64_C(  -42.07), SIMDE_FLOAT64_C( -685.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  296.78), SIMDE_FLOAT64_C( -698.78),
                         SIMDE_FLOAT64_C(  908.33), SIMDE_FLOAT64_C(  181.85),
                         SIMDE_FLOAT64_C( -397.89), SIMDE_FLOAT64_C( -586.75),
                         SIMDE_FLOAT64_C(  904.99), SIMDE_FLOAT64_C( -321.15)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.33), SIMDE_FLOAT64_C(    0.56),
                         SIMDE_FLOAT64_C(    1.03), SIMDE_FLOAT64_C(    0.41),
                         SIMDE_FLOAT64_C(   -2.12), SIMDE_FLOAT64_C(   -0.79),
                         SIMDE_FLOAT64_C(   -0.05), SIMDE_FLOAT64_C(    2.14)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  653.62), SIMDE_FLOAT64_C(  981.74),
                         SIMDE_FLOAT64_C(  780.10), SIMDE_FLOAT64_C(   59.38),
                         SIMDE_FLOAT64_C( -795.11), SIMDE_FLOAT64_C(  923.87),
                         SIMDE_FLOAT64_C( -270.01), SIMDE_FLOAT64_C( -411.99)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  596.54), SIMDE_FLOAT64_C( -116.40),
                         SIMDE_FLOAT64_C( -989.77), SIMDE_FLOAT64_C( -794.40),
                         SIMDE_FLOAT64_C(  183.38), SIMDE_FLOAT64_C( -185.75),
                         SIMDE_FLOAT64_C(  429.70), SIMDE_FLOAT64_C(  664.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    1.10), SIMDE_FLOAT64_C(   -8.43),
                         SIMDE_FLOAT64_C(   -0.79), SIMDE_FLOAT64_C(   -0.07),
                         SIMDE_FLOAT64_C(   -4.34), SIMDE_FLOAT64_C(   -4.97),
                         SIMDE_FLOAT64_C(   -0.63), SIMDE_FLOAT64_C(   -0.62)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -571.10), SIMDE_FLOAT64_C(  971.96),
                         SIMDE_FLOAT64_C(  149.38), SIMDE_FLOAT64_C(  497.71),
                         SIMDE_FLOAT64_C(  988.69), SIMDE_FLOAT64_C(  479.68),
                         SIMDE_FLOAT64_C( -128.24), SIMDE_FLOAT64_C(  585.28)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -628.68), SIMDE_FLOAT64_C(  938.60),
                         SIMDE_FLOAT64_C( -147.98), SIMDE_FLOAT64_C(  378.31),
                         SIMDE_FLOAT64_C(  246.47), SIMDE_FLOAT64_C(  109.18),
                         SIMDE_FLOAT64_C( -575.64), SIMDE_FLOAT64_C( -426.86)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.91), SIMDE_FLOAT64_C(    1.04),
                         SIMDE_FLOAT64_C(   -1.01), SIMDE_FLOAT64_C(    1.32),
                         SIMDE_FLOAT64_C(    4.01), SIMDE_FLOAT64_C(    4.39),
                         SIMDE_FLOAT64_C(    0.22), SIMDE_FLOAT64_C(   -1.37)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  729.63), SIMDE_FLOAT64_C( -908.06),
                         SIMDE_FLOAT64_C( -769.77), SIMDE_FLOAT64_C(  -70.66),
                         SIMDE_FLOAT64_C(  482.71), SIMDE_FLOAT64_C(  244.66),
                         SIMDE_FLOAT64_C( -615.83), SIMDE_FLOAT64_C(  841.42)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  359.65), SIMDE_FLOAT64_C( -730.08),
                         SIMDE_FLOAT64_C(  977.98), SIMDE_FLOAT64_C( -215.53),
                         SIMDE_FLOAT64_C( -315.50), SIMDE_FLOAT64_C(   80.64),
                         SIMDE_FLOAT64_C( -996.10), SIMDE_FLOAT64_C( -556.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    2.03), SIMDE_FLOAT64_C(    1.24),
                         SIMDE_FLOAT64_C(   -0.79), SIMDE_FLOAT64_C(    0.33),
                         SIMDE_FLOAT64_C(   -1.53), SIMDE_FLOAT64_C(    3.03),
                         SIMDE_FLOAT64_C(    0.62), SIMDE_FLOAT64_C(   -1.51)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  438.03), SIMDE_FLOAT64_C( -458.01),
                         SIMDE_FLOAT64_C(  144.59), SIMDE_FLOAT64_C(  165.00),
                         SIMDE_FLOAT64_C( -331.04), SIMDE_FLOAT64_C(  406.96),
                         SIMDE_FLOAT64_C( -326.43), SIMDE_FLOAT64_C(  373.82)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  304.30), SIMDE_FLOAT64_C( -777.07),
                         SIMDE_FLOAT64_C( -683.73), SIMDE_FLOAT64_C( -113.32),
                         SIMDE_FLOAT64_C( -701.16), SIMDE_FLOAT64_C( -942.92),
                         SIMDE_FLOAT64_C( -489.97), SIMDE_FLOAT64_C(  911.34)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    1.44), SIMDE_FLOAT64_C(    0.59),
                         SIMDE_FLOAT64_C(   -0.21), SIMDE_FLOAT64_C(   -1.46),
                         SIMDE_FLOAT64_C(    0.47), SIMDE_FLOAT64_C(   -0.43),
                         SIMDE_FLOAT64_C(    0.67), SIMDE_FLOAT64_C(    0.41)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -148.70), SIMDE_FLOAT64_C( -327.17),
                         SIMDE_FLOAT64_C( -310.14), SIMDE_FLOAT64_C( -718.80),
                         SIMDE_FLOAT64_C(  382.69), SIMDE_FLOAT64_C( -181.61),
                         SIMDE_FLOAT64_C( -214.09), SIMDE_FLOAT64_C(   55.72)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  791.83), SIMDE_FLOAT64_C(  191.69),
                         SIMDE_FLOAT64_C( -460.58), SIMDE_FLOAT64_C( -915.08),
                         SIMDE_FLOAT64_C( -877.38), SIMDE_FLOAT64_C( -915.27),
                         SIMDE_FLOAT64_C(  207.85), SIMDE_FLOAT64_C(  567.35)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.19), SIMDE_FLOAT64_C(   -1.71),
                         SIMDE_FLOAT64_C(    0.67), SIMDE_FLOAT64_C(    0.79),
                         SIMDE_FLOAT64_C(   -0.44), SIMDE_FLOAT64_C(    0.20),
                         SIMDE_FLOAT64_C(   -1.03), SIMDE_FLOAT64_C(    0.10)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -798.08), SIMDE_FLOAT64_C(  256.40),
                         SIMDE_FLOAT64_C(  739.89), SIMDE_FLOAT64_C( -903.46),
                         SIMDE_FLOAT64_C( -771.75), SIMDE_FLOAT64_C(  -54.77),
                         SIMDE_FLOAT64_C(  397.04), SIMDE_FLOAT64_C(  925.94)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  864.29), SIMDE_FLOAT64_C( -834.00),
                         SIMDE_FLOAT64_C(  475.52), SIMDE_FLOAT64_C(  502.31),
                         SIMDE_FLOAT64_C( -746.87), SIMDE_FLOAT64_C( -364.10),
                         SIMDE_FLOAT64_C( -995.18), SIMDE_FLOAT64_C(  683.54)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.92), SIMDE_FLOAT64_C(   -0.31),
                         SIMDE_FLOAT64_C(    1.56), SIMDE_FLOAT64_C(   -1.80),
                         SIMDE_FLOAT64_C(    1.03), SIMDE_FLOAT64_C(    0.15),
                         SIMDE_FLOAT64_C(   -0.40), SIMDE_FLOAT64_C(    1.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -869.58), SIMDE_FLOAT64_C(  763.75),
                         SIMDE_FLOAT64_C( -558.93), SIMDE_FLOAT64_C(  756.19),
                         SIMDE_FLOAT64_C(  509.82), SIMDE_FLOAT64_C( -855.71),
                         SIMDE_FLOAT64_C( -965.40), SIMDE_FLOAT64_C( -279.29)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -355.51), SIMDE_FLOAT64_C(  136.73),
                         SIMDE_FLOAT64_C(  586.70), SIMDE_FLOAT64_C(  712.56),
                         SIMDE_FLOAT64_C(  135.88), SIMDE_FLOAT64_C( -693.91),
                         SIMDE_FLOAT64_C( -131.33), SIMDE_FLOAT64_C( -933.79)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    2.45), SIMDE_FLOAT64_C(    5.59),
                         SIMDE_FLOAT64_C(   -0.95), SIMDE_FLOAT64_C(    1.06),
                         SIMDE_FLOAT64_C(    3.75), SIMDE_FLOAT64_C(    1.23),
                         SIMDE_FLOAT64_C(    7.35), SIMDE_FLOAT64_C(    0.30)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_div_pd(test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_div_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -962.94), SIMDE_FLOAT64_C(  989.45),
                         SIMDE_FLOAT64_C( -190.71), SIMDE_FLOAT64_C(  -80.90),
                         SIMDE_FLOAT64_C( -820.03), SIMDE_FLOAT64_C(  710.84),
                         SIMDE_FLOAT64_C(  742.77), SIMDE_FLOAT64_C( -124.19)),
      UINT8_C( 62),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  764.73), SIMDE_FLOAT64_C( -738.72),
                         SIMDE_FLOAT64_C(  462.89), SIMDE_FLOAT64_C( -909.36),
                         SIMDE_FLOAT64_C(  920.77), SIMDE_FLOAT64_C(  830.94),
                         SIMDE_FLOAT64_C( -436.90), SIMDE_FLOAT64_C( -984.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  318.55), SIMDE_FLOAT64_C( -958.54),
                         SIMDE_FLOAT64_C( -878.41), SIMDE_FLOAT64_C(  198.47),
                         SIMDE_FLOAT64_C(  585.51), SIMDE_FLOAT64_C(  -97.52),
                         SIMDE_FLOAT64_C( -112.08), SIMDE_FLOAT64_C( -145.20)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -962.94), SIMDE_FLOAT64_C(  989.45),
                         SIMDE_FLOAT64_C(   -0.53), SIMDE_FLOAT64_C(   -4.58),
                         SIMDE_FLOAT64_C(    1.57), SIMDE_FLOAT64_C(   -8.52),
                         SIMDE_FLOAT64_C(    3.90), SIMDE_FLOAT64_C( -124.19)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  267.17), SIMDE_FLOAT64_C( -878.34),
                         SIMDE_FLOAT64_C(  132.07), SIMDE_FLOAT64_C(  827.87),
                         SIMDE_FLOAT64_C(  178.51), SIMDE_FLOAT64_C(  362.39),
                         SIMDE_FLOAT64_C(  200.13), SIMDE_FLOAT64_C( -407.98)),
      UINT8_C( 51),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -126.54), SIMDE_FLOAT64_C( -164.31),
                         SIMDE_FLOAT64_C( -971.32), SIMDE_FLOAT64_C(  611.23),
                         SIMDE_FLOAT64_C(  591.83), SIMDE_FLOAT64_C(  793.58),
                         SIMDE_FLOAT64_C(  171.77), SIMDE_FLOAT64_C(  109.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  225.35), SIMDE_FLOAT64_C( -734.84),
                         SIMDE_FLOAT64_C(  728.29), SIMDE_FLOAT64_C( -721.11),
                         SIMDE_FLOAT64_C( -448.10), SIMDE_FLOAT64_C(  310.61),
                         SIMDE_FLOAT64_C( -362.27), SIMDE_FLOAT64_C( -413.07)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  267.17), SIMDE_FLOAT64_C( -878.34),
                         SIMDE_FLOAT64_C(   -1.33), SIMDE_FLOAT64_C(   -0.85),
                         SIMDE_FLOAT64_C(  178.51), SIMDE_FLOAT64_C(  362.39),
                         SIMDE_FLOAT64_C(   -0.47), SIMDE_FLOAT64_C(   -0.27)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  439.30), SIMDE_FLOAT64_C(  935.39),
                         SIMDE_FLOAT64_C(   20.00), SIMDE_FLOAT64_C( -941.65),
                         SIMDE_FLOAT64_C(  988.79), SIMDE_FLOAT64_C(  773.96),
                         SIMDE_FLOAT64_C( -788.78), SIMDE_FLOAT64_C( -311.91)),
      UINT8_C(178),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -374.30), SIMDE_FLOAT64_C(  599.21),
                         SIMDE_FLOAT64_C(  966.83), SIMDE_FLOAT64_C(  775.18),
                         SIMDE_FLOAT64_C(  846.32), SIMDE_FLOAT64_C(  124.04),
                         SIMDE_FLOAT64_C( -883.36), SIMDE_FLOAT64_C( -405.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  467.70), SIMDE_FLOAT64_C( -626.02),
                         SIMDE_FLOAT64_C(  355.93), SIMDE_FLOAT64_C(  294.34),
                         SIMDE_FLOAT64_C( -575.79), SIMDE_FLOAT64_C( -504.82),
                         SIMDE_FLOAT64_C(  854.52), SIMDE_FLOAT64_C( -173.82)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.80), SIMDE_FLOAT64_C(  935.39),
                         SIMDE_FLOAT64_C(    2.72), SIMDE_FLOAT64_C(    2.63),
                         SIMDE_FLOAT64_C(  988.79), SIMDE_FLOAT64_C(  773.96),
                         SIMDE_FLOAT64_C(   -1.03), SIMDE_FLOAT64_C( -311.91)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -274.81), SIMDE_FLOAT64_C(  196.78),
                         SIMDE_FLOAT64_C( -805.22), SIMDE_FLOAT64_C(  855.89),
                         SIMDE_FLOAT64_C( -996.67), SIMDE_FLOAT64_C(  424.78),
                         SIMDE_FLOAT64_C(  489.73), SIMDE_FLOAT64_C(  635.35)),
      UINT8_C( 38),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   79.19), SIMDE_FLOAT64_C( -114.25),
                         SIMDE_FLOAT64_C(  983.59), SIMDE_FLOAT64_C(  645.66),
                         SIMDE_FLOAT64_C(  982.80), SIMDE_FLOAT64_C( -683.73),
                         SIMDE_FLOAT64_C(  259.13), SIMDE_FLOAT64_C(  186.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  665.49), SIMDE_FLOAT64_C( -110.92),
                         SIMDE_FLOAT64_C(  978.65), SIMDE_FLOAT64_C(  104.45),
                         SIMDE_FLOAT64_C(  903.68), SIMDE_FLOAT64_C( -580.74),
                         SIMDE_FLOAT64_C(  776.44), SIMDE_FLOAT64_C(  571.14)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -274.81), SIMDE_FLOAT64_C(  196.78),
                         SIMDE_FLOAT64_C(    1.01), SIMDE_FLOAT64_C(  855.89),
                         SIMDE_FLOAT64_C( -996.67), SIMDE_FLOAT64_C(    1.18),
                         SIMDE_FLOAT64_C(    0.33), SIMDE_FLOAT64_C(  635.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  105.93), SIMDE_FLOAT64_C( -486.73),
                         SIMDE_FLOAT64_C(  293.04), SIMDE_FLOAT64_C(  328.58),
                         SIMDE_FLOAT64_C( -725.03), SIMDE_FLOAT64_C(    3.53),
                         SIMDE_FLOAT64_C(  663.75), SIMDE_FLOAT64_C(  -59.32)),
      UINT8_C( 67),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  917.98), SIMDE_FLOAT64_C( -430.92),
                         SIMDE_FLOAT64_C(  839.77), SIMDE_FLOAT64_C( -412.68),
                         SIMDE_FLOAT64_C( -397.37), SIMDE_FLOAT64_C( -947.31),
                         SIMDE_FLOAT64_C(  584.59), SIMDE_FLOAT64_C( -352.12)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  781.61), SIMDE_FLOAT64_C(  978.32),
                         SIMDE_FLOAT64_C(  374.68), SIMDE_FLOAT64_C( -857.00),
                         SIMDE_FLOAT64_C(  821.72), SIMDE_FLOAT64_C(  -88.08),
                         SIMDE_FLOAT64_C(  243.00), SIMDE_FLOAT64_C( -640.77)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  105.93), SIMDE_FLOAT64_C(   -0.44),
                         SIMDE_FLOAT64_C(  293.04), SIMDE_FLOAT64_C(  328.58),
                         SIMDE_FLOAT64_C( -725.03), SIMDE_FLOAT64_C(    3.53),
                         SIMDE_FLOAT64_C(    2.41), SIMDE_FLOAT64_C(    0.55)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -591.91), SIMDE_FLOAT64_C(  615.29),
                         SIMDE_FLOAT64_C( -726.07), SIMDE_FLOAT64_C(  857.36),
                         SIMDE_FLOAT64_C(  636.31), SIMDE_FLOAT64_C(  104.40),
                         SIMDE_FLOAT64_C( -167.77), SIMDE_FLOAT64_C( -372.65)),
      UINT8_C( 15),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  411.16), SIMDE_FLOAT64_C(  928.95),
                         SIMDE_FLOAT64_C(  110.13), SIMDE_FLOAT64_C(  933.76),
                         SIMDE_FLOAT64_C(  836.76), SIMDE_FLOAT64_C(  628.60),
                         SIMDE_FLOAT64_C( -586.52), SIMDE_FLOAT64_C(  293.24)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -457.28), SIMDE_FLOAT64_C(  705.56),
                         SIMDE_FLOAT64_C( -798.08), SIMDE_FLOAT64_C(  773.61),
                         SIMDE_FLOAT64_C( -590.48), SIMDE_FLOAT64_C( -291.69),
                         SIMDE_FLOAT64_C(  654.27), SIMDE_FLOAT64_C( -537.59)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -591.91), SIMDE_FLOAT64_C(  615.29),
                         SIMDE_FLOAT64_C( -726.07), SIMDE_FLOAT64_C(  857.36),
                         SIMDE_FLOAT64_C(   -1.42), SIMDE_FLOAT64_C(   -2.16),
                         SIMDE_FLOAT64_C(   -0.90), SIMDE_FLOAT64_C(   -0.55)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  270.92), SIMDE_FLOAT64_C( -517.94),
                         SIMDE_FLOAT64_C(   36.22), SIMDE_FLOAT64_C(  204.54),
                         SIMDE_FLOAT64_C(  579.30), SIMDE_FLOAT64_C(  257.34),
                         SIMDE_FLOAT64_C( -998.24), SIMDE_FLOAT64_C( -146.41)),
      UINT8_C(152),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  268.93), SIMDE_FLOAT64_C( -893.46),
                         SIMDE_FLOAT64_C( -476.89), SIMDE_FLOAT64_C( -696.00),
                         SIMDE_FLOAT64_C( -817.69), SIMDE_FLOAT64_C(  127.75),
                         SIMDE_FLOAT64_C( -366.34), SIMDE_FLOAT64_C( -437.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -372.16), SIMDE_FLOAT64_C(  900.88),
                         SIMDE_FLOAT64_C( -550.65), SIMDE_FLOAT64_C(  567.85),
                         SIMDE_FLOAT64_C(  968.56), SIMDE_FLOAT64_C( -695.12),
                         SIMDE_FLOAT64_C(  555.56), SIMDE_FLOAT64_C(  952.92)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.72), SIMDE_FLOAT64_C( -517.94),
                         SIMDE_FLOAT64_C(   36.22), SIMDE_FLOAT64_C(   -1.23),
                         SIMDE_FLOAT64_C(   -0.84), SIMDE_FLOAT64_C(  257.34),
                         SIMDE_FLOAT64_C( -998.24), SIMDE_FLOAT64_C( -146.41)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -468.36), SIMDE_FLOAT64_C(  186.16),
                         SIMDE_FLOAT64_C( -910.43), SIMDE_FLOAT64_C( -280.07),
                         SIMDE_FLOAT64_C(  -96.94), SIMDE_FLOAT64_C(  387.95),
                         SIMDE_FLOAT64_C(  198.14), SIMDE_FLOAT64_C( -504.51)),
      UINT8_C( 21),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  573.90), SIMDE_FLOAT64_C(  496.67),
                         SIMDE_FLOAT64_C( -823.61), SIMDE_FLOAT64_C(  204.56),
                         SIMDE_FLOAT64_C( -856.87), SIMDE_FLOAT64_C( -449.28),
                         SIMDE_FLOAT64_C(    9.73), SIMDE_FLOAT64_C( -739.12)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -471.24), SIMDE_FLOAT64_C( -711.43),
                         SIMDE_FLOAT64_C( -281.50), SIMDE_FLOAT64_C(  493.76),
                         SIMDE_FLOAT64_C(  103.01), SIMDE_FLOAT64_C( -996.35),
                         SIMDE_FLOAT64_C(  670.04), SIMDE_FLOAT64_C( -895.53)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -468.36), SIMDE_FLOAT64_C(  186.16),
                         SIMDE_FLOAT64_C( -910.43), SIMDE_FLOAT64_C(    0.41),
                         SIMDE_FLOAT64_C(  -96.94), SIMDE_FLOAT64_C(    0.45),
                         SIMDE_FLOAT64_C(  198.14), SIMDE_FLOAT64_C(    0.83)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_div_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_div_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C(113),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  112.08), SIMDE_FLOAT64_C(  712.48),
                         SIMDE_FLOAT64_C( -754.71), SIMDE_FLOAT64_C(  256.61),
                         SIMDE_FLOAT64_C(  768.73), SIMDE_FLOAT64_C(  227.99),
                         SIMDE_FLOAT64_C(  174.97), SIMDE_FLOAT64_C(  338.39)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  759.92), SIMDE_FLOAT64_C(  629.58),
                         SIMDE_FLOAT64_C(   23.76), SIMDE_FLOAT64_C(  980.95),
                         SIMDE_FLOAT64_C(  224.97), SIMDE_FLOAT64_C(  818.07),
                         SIMDE_FLOAT64_C( -531.75), SIMDE_FLOAT64_C( -531.67)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    1.13),
                         SIMDE_FLOAT64_C(  -31.76), SIMDE_FLOAT64_C(    0.26),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -0.64)) },
    { UINT8_C( 88),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  841.82), SIMDE_FLOAT64_C( -330.14),
                         SIMDE_FLOAT64_C( -498.05), SIMDE_FLOAT64_C( -706.46),
                         SIMDE_FLOAT64_C( -284.71), SIMDE_FLOAT64_C( -940.98),
                         SIMDE_FLOAT64_C( -491.84), SIMDE_FLOAT64_C(   52.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  259.38), SIMDE_FLOAT64_C(  745.69),
                         SIMDE_FLOAT64_C( -258.59), SIMDE_FLOAT64_C( -549.06),
                         SIMDE_FLOAT64_C(  646.98), SIMDE_FLOAT64_C(  925.86),
                         SIMDE_FLOAT64_C(  378.90), SIMDE_FLOAT64_C( -524.10)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -0.44),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    1.29),
                         SIMDE_FLOAT64_C(   -0.44), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(184),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -167.30), SIMDE_FLOAT64_C( -738.71),
                         SIMDE_FLOAT64_C( -816.67), SIMDE_FLOAT64_C(   43.31),
                         SIMDE_FLOAT64_C(  -98.40), SIMDE_FLOAT64_C(  217.89),
                         SIMDE_FLOAT64_C(  626.98), SIMDE_FLOAT64_C( -409.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -443.71), SIMDE_FLOAT64_C( -786.78),
                         SIMDE_FLOAT64_C(  -92.41), SIMDE_FLOAT64_C( -378.62),
                         SIMDE_FLOAT64_C(  632.49), SIMDE_FLOAT64_C( -867.20),
                         SIMDE_FLOAT64_C(  977.79), SIMDE_FLOAT64_C( -788.71)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.38), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    8.84), SIMDE_FLOAT64_C(   -0.11),
                         SIMDE_FLOAT64_C(   -0.16), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(119),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -156.59), SIMDE_FLOAT64_C(  153.69),
                         SIMDE_FLOAT64_C(  895.38), SIMDE_FLOAT64_C( -242.63),
                         SIMDE_FLOAT64_C(  994.17), SIMDE_FLOAT64_C( -265.23),
                         SIMDE_FLOAT64_C(  -57.91), SIMDE_FLOAT64_C( -586.11)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -851.62), SIMDE_FLOAT64_C(  838.22),
                         SIMDE_FLOAT64_C(  464.78), SIMDE_FLOAT64_C( -248.37),
                         SIMDE_FLOAT64_C(   28.49), SIMDE_FLOAT64_C( -176.67),
                         SIMDE_FLOAT64_C( -468.39), SIMDE_FLOAT64_C( -893.30)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.18),
                         SIMDE_FLOAT64_C(    1.93), SIMDE_FLOAT64_C(    0.98),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    1.50),
                         SIMDE_FLOAT64_C(    0.12), SIMDE_FLOAT64_C(    0.66)) },
    { UINT8_C(181),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -748.23), SIMDE_FLOAT64_C(  960.31),
                         SIMDE_FLOAT64_C(  -52.56), SIMDE_FLOAT64_C(  -18.49),
                         SIMDE_FLOAT64_C( -174.79), SIMDE_FLOAT64_C( -875.70),
                         SIMDE_FLOAT64_C(  270.45), SIMDE_FLOAT64_C(  571.57)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  887.26), SIMDE_FLOAT64_C( -380.63),
                         SIMDE_FLOAT64_C(  603.15), SIMDE_FLOAT64_C(  906.17),
                         SIMDE_FLOAT64_C( -446.90), SIMDE_FLOAT64_C(  518.96),
                         SIMDE_FLOAT64_C(  325.09), SIMDE_FLOAT64_C(  394.29)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.84), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   -0.09), SIMDE_FLOAT64_C(   -0.02),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -1.69),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    1.45)) },
    { UINT8_C(108),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -891.46), SIMDE_FLOAT64_C(  298.88),
                         SIMDE_FLOAT64_C(  907.92), SIMDE_FLOAT64_C(  585.94),
                         SIMDE_FLOAT64_C(  976.00), SIMDE_FLOAT64_C(  860.60),
                         SIMDE_FLOAT64_C( -807.57), SIMDE_FLOAT64_C( -501.53)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -634.78), SIMDE_FLOAT64_C( -309.30),
                         SIMDE_FLOAT64_C( -478.69), SIMDE_FLOAT64_C( -499.66),
                         SIMDE_FLOAT64_C( -834.97), SIMDE_FLOAT64_C( -926.76),
                         SIMDE_FLOAT64_C(  306.74), SIMDE_FLOAT64_C(  350.68)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -0.97),
                         SIMDE_FLOAT64_C(   -1.90), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   -1.17), SIMDE_FLOAT64_C(   -0.93),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(  5),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -764.37), SIMDE_FLOAT64_C( -698.84),
                         SIMDE_FLOAT64_C(  111.54), SIMDE_FLOAT64_C(  627.05),
                         SIMDE_FLOAT64_C(  619.20), SIMDE_FLOAT64_C(  107.79),
                         SIMDE_FLOAT64_C(  830.07), SIMDE_FLOAT64_C( -991.50)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  902.97), SIMDE_FLOAT64_C( -970.67),
                         SIMDE_FLOAT64_C( -396.71), SIMDE_FLOAT64_C(  740.42),
                         SIMDE_FLOAT64_C( -740.07), SIMDE_FLOAT64_C(  691.95),
                         SIMDE_FLOAT64_C( -434.89), SIMDE_FLOAT64_C(  270.74)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.16),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -3.66)) },
    { UINT8_C( 94),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -9.32), SIMDE_FLOAT64_C(  588.15),
                         SIMDE_FLOAT64_C(  740.36), SIMDE_FLOAT64_C(  589.64),
                         SIMDE_FLOAT64_C( -146.10), SIMDE_FLOAT64_C(  771.62),
                         SIMDE_FLOAT64_C( -975.31), SIMDE_FLOAT64_C(  550.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  380.47), SIMDE_FLOAT64_C( -722.04),
                         SIMDE_FLOAT64_C( -251.19), SIMDE_FLOAT64_C(  885.20),
                         SIMDE_FLOAT64_C( -718.95), SIMDE_FLOAT64_C( -995.18),
                         SIMDE_FLOAT64_C(  316.41), SIMDE_FLOAT64_C(  425.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -0.81),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.67),
                         SIMDE_FLOAT64_C(    0.20), SIMDE_FLOAT64_C(   -0.78),
                         SIMDE_FLOAT64_C(   -3.08), SIMDE_FLOAT64_C(    0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_div_pd(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_load_si512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu32(UINT32_C(2307486612), UINT32_C( 772959324), UINT32_C(2088243446), UINT32_C(3897025989),
                              UINT32_C( 869133120), UINT32_C(2804250721), UINT32_C(1097182028), UINT32_C( 897796365),
                              UINT32_C(3076780984), UINT32_C(1156581381), UINT32_C( 175207521), UINT32_C(3060063945),
                              UINT32_C(1233244582), UINT32_C(3594112000), UINT32_C(2979101794), UINT32_C(1885968130)),
      simde_x_mm512_set_epu32(UINT32_C(2307486612), UINT32_C( 772959324), UINT32_C(2088243446), UINT32_C(3897025989),
                              UINT32_C( 869133120), UINT32_C(2804250721), UINT32_C(1097182028), UINT32_C( 897796365),
                              UINT32_C(3076780984), UINT32_C(1156581381), UINT32_C( 175207521), UINT32_C(3060063945),
                              UINT32_C(1233244582), UINT32_C(3594112000), UINT32_C(2979101794), UINT32_C(1885968130)) },
    { simde_x_mm512_set_epu32(UINT32_C(1723534008), UINT32_C(4247516214), UINT32_C(2059209086), UINT32_C( 125977829),
                              UINT32_C(3701590438), UINT32_C( 787371643), UINT32_C(4154718194), UINT32_C(2924919521),
                              UINT32_C(2042687607), UINT32_C(  97954970), UINT32_C( 784634460), UINT32_C(1187123278),
                              UINT32_C(2177208278), UINT32_C( 582111455), UINT32_C(2249543224), UINT32_C(2999976342)),
      simde_x_mm512_set_epu32(UINT32_C(1723534008), UINT32_C(4247516214), UINT32_C(2059209086), UINT32_C( 125977829),
                              UINT32_C(3701590438), UINT32_C( 787371643), UINT32_C(4154718194), UINT32_C(2924919521),
                              UINT32_C(2042687607), UINT32_C(  97954970), UINT32_C( 784634460), UINT32_C(1187123278),
                              UINT32_C(2177208278), UINT32_C( 582111455), UINT32_C(2249543224), UINT32_C(2999976342)) },
    { simde_x_mm512_set_epu32(UINT32_C(1489097294), UINT32_C(1317978507), UINT32_C(3317911074), UINT32_C(3811250336),
                              UINT32_C( 726117554), UINT32_C(1141945936), UINT32_C(2967228592), UINT32_C(1057056249),
                              UINT32_C( 814464662), UINT32_C( 112704332), UINT32_C(3052661000), UINT32_C(1384458807),
                              UINT32_C( 962739128), UINT32_C(3280757151), UINT32_C(2593496887), UINT32_C(1987869085)),
      simde_x_mm512_set_epu32(UINT32_C(1489097294), UINT32_C(1317978507), UINT32_C(3317911074), UINT32_C(3811250336),
                              UINT32_C( 726117554), UINT32_C(1141945936), UINT32_C(2967228592), UINT32_C(1057056249),
                              UINT32_C( 814464662), UINT32_C( 112704332), UINT32_C(3052661000), UINT32_C(1384458807),
                              UINT32_C( 962739128), UINT32_C(3280757151), UINT32_C(2593496887), UINT32_C(1987869085)) },
    { simde_x_mm512_set_epu32(UINT32_C(2685429527), UINT32_C(3030233284), UINT32_C(3144837738), UINT32_C(2079867326),
                              UINT32_C(2718871212), UINT32_C( 638009587), UINT32_C(1746704273), UINT32_C(1785787430),
                              UINT32_C(1376438250), UINT32_C(3110594325), UINT32_C(2802966903), UINT32_C(1623533439),
                              UINT32_C( 818134571), UINT32_C(1616341471), UINT32_C(1617845622), UINT32_C(1998224490)),
      simde_x_mm512_set_epu32(UINT32_C(2685429527), UINT32_C(3030233284), UINT32_C(3144837738), UINT32_C(2079867326),
                              UINT32_C(2718871212), UINT32_C( 638009587), UINT32_C(1746704273), UINT32_C(1785787430),
                              UINT32_C(1376438250), UINT32_C(3110594325), UINT32_C(2802966903), UINT32_C(1623533439),
                              UINT32_C( 818134571), UINT32_C(1616341471), UINT32_C(1617845622), UINT32_C(1998224490)) },
    { simde_x_mm512_set_epu32(UINT32_C( 799998643), UINT32_C(2158675766), UINT32_C(2639170838), UINT32_C(2026768033),
                              UINT32_C( 417038853), UINT32_C( 987028378), UINT32_C(1999157218), UINT32_C(2908291470),
                              UINT32_C(4158892285), UINT32_C(2617218564), UINT32_C(3118768983), UINT32_C(3550554212),
                              UINT32_C(1938726035), UINT32_C( 744056698), UINT32_C( 793532882), UINT32_C(1206093923)),
      simde_x_mm512_set_epu32(UINT32_C( 799998643), UINT32_C(2158675766), UINT32_C(2639170838), UINT32_C(2026768033),
                              UINT32_C( 417038853), UINT32_C( 987028378), UINT32_C(1999157218), UINT32_C(2908291470),
                              UINT32_C(4158892285), UINT32_C(2617218564), UINT32_C(3118768983), UINT32_C(3550554212),
                              UINT32_C(1938726035), UINT32_C( 744056698), UINT32_C( 793532882), UINT32_C(1206093923)) },
    { simde_x_mm512_set_epu32(UINT32_C(4149762290), UINT32_C(2200927650), UINT32_C( 469169161), UINT32_C(2649357489),
                              UINT32_C( 501467543), UINT32_C(2196017318), UINT32_C( 902670401), UINT32_C(1185049393),
                              UINT32_C(1462585034), UINT32_C(3206403756), UINT32_C(4183616002), UINT32_C(3461640287),
                              UINT32_C( 417381550), UINT32_C(1851953642), UINT32_C(  74253140), UINT32_C(2242551698)),
      simde_x_mm512_set_epu32(UINT32_C(4149762290), UINT32_C(2200927650), UINT32_C( 469169161), UINT32_C(2649357489),
                              UINT32_C( 501467543), UINT32_C(2196017318), UINT32_C( 902670401), UINT32_C(1185049393),
                              UINT32_C(1462585034), UINT32_C(3206403756), UINT32_C(4183616002), UINT32_C(3461640287),
                              UINT32_C( 417381550), UINT32_C(1851953642), UINT32_C(  74253140), UINT32_C(2242551698)) },
    { simde_x_mm512_set_epu32(UINT32_C(3572878479), UINT32_C( 918008698), UINT32_C(3285117640), UINT32_C( 321650853),
                              UINT32_C(3573674805), UINT32_C(2577794583), UINT32_C( 366403780), UINT32_C(1431583390),
                              UINT32_C( 904162832), UINT32_C(1335714059), UINT32_C(3804806590), UINT32_C( 712879911),
                              UINT32_C(2152630393), UINT32_C(2807501401), UINT32_C(1455098423), UINT32_C(1316688242)),
      simde_x_mm512_set_epu32(UINT32_C(3572878479), UINT32_C( 918008698), UINT32_C(3285117640), UINT32_C( 321650853),
                              UINT32_C(3573674805), UINT32_C(2577794583), UINT32_C( 366403780), UINT32_C(1431583390),
                              UINT32_C( 904162832), UINT32_C(1335714059), UINT32_C(3804806590), UINT32_C( 712879911),
                              UINT32_C(2152630393), UINT32_C(2807501401), UINT32_C(1455098423), UINT32_C(1316688242)) },
    { simde_x_mm512_set_epu32(UINT32_C(2289360766), UINT32_C( 432894560), UINT32_C(2761652233), UINT32_C(2119096941),
                              UINT32_C(2008109945), UINT32_C(1806126906), UINT32_C( 883816031), UINT32_C( 954297540),
                              UINT32_C(  58235371), UINT32_C(3225322036), UINT32_C(3536085950), UINT32_C(3389687377),
                              UINT32_C(1236355728), UINT32_C(2830432354), UINT32_C(3830173342), UINT32_C(1246461045)),
      simde_x_mm512_set_epu32(UINT32_C(2289360766), UINT32_C( 432894560), UINT32_C(2761652233), UINT32_C(2119096941),
                              UINT32_C(2008109945), UINT32_C(1806126906), UINT32_C( 883816031), UINT32_C( 954297540),
                              UINT32_C(  58235371), UINT32_C(3225322036), UINT32_C(3536085950), UINT32_C(3389687377),
                              UINT32_C(1236355728), UINT32_C(2830432354), UINT32_C(3830173342), UINT32_C(1246461045)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_load_si512(&(test_vec[i].a));
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_loadu_si512(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu32(UINT32_C(2465927924), UINT32_C(3593197775), UINT32_C( 612910812), UINT32_C(3812769805),
                              UINT32_C(4149829677), UINT32_C(3483799324), UINT32_C(1459962882), UINT32_C(4149819515),
                              UINT32_C(2650201844), UINT32_C( 758753621), UINT32_C(1440172455), UINT32_C(1093653043),
                              UINT32_C(4135659774), UINT32_C(2249292246), UINT32_C(3926208727), UINT32_C( 363227362)),
      simde_x_mm512_set_epu32(UINT32_C(2465927924), UINT32_C(3593197775), UINT32_C( 612910812), UINT32_C(3812769805),
                              UINT32_C(4149829677), UINT32_C(3483799324), UINT32_C(1459962882), UINT32_C(4149819515),
                              UINT32_C(2650201844), UINT32_C( 758753621), UINT32_C(1440172455), UINT32_C(1093653043),
                              UINT32_C(4135659774), UINT32_C(2249292246), UINT32_C(3926208727), UINT32_C( 363227362)) },
    { simde_x_mm512_set_epu32(UINT32_C( 468967701), UINT32_C(1464888328), UINT32_C(2623912787), UINT32_C(3584306317),
                              UINT32_C(3441172772), UINT32_C(1957813224), UINT32_C(3956090282), UINT32_C(2819645236),
                              UINT32_C(2119397630), UINT32_C(3325357179), UINT32_C( 910080153), UINT32_C(3698201489),
                              UINT32_C(3945376801), UINT32_C(2699586726), UINT32_C(1169343086), UINT32_C(2983594096)),
      simde_x_mm512_set_epu32(UINT32_C( 468967701), UINT32_C(1464888328), UINT32_C(2623912787), UINT32_C(3584306317),
                              UINT32_C(3441172772), UINT32_C(1957813224), UINT32_C(3956090282), UINT32_C(2819645236),
                              UINT32_C(2119397630), UINT32_C(3325357179), UINT32_C( 910080153), UINT32_C(3698201489),
                              UINT32_C(3945376801), UINT32_C(2699586726), UINT32_C(1169343086), UINT32_C(2983594096)) },
    { simde_x_mm512_set_epu32(UINT32_C(3220925730), UINT32_C(4163700514), UINT32_C( 208162340), UINT32_C(  72282893),
                              UINT32_C(2784701415), UINT32_C(2960668076), UINT32_C(2280551509), UINT32_C( 511971347),
                              UINT32_C(3142311802), UINT32_C(3582165504), UINT32_C(3533175269), UINT32_C(3138584679),
                              UINT32_C(3117232701), UINT32_C(1582887517), UINT32_C(2957127939), UINT32_C(3388466484)),
      simde_x_mm512_set_epu32(UINT32_C(3220925730), UINT32_C(4163700514), UINT32_C( 208162340), UINT32_C(  72282893),
                              UINT32_C(2784701415), UINT32_C(2960668076), UINT32_C(2280551509), UINT32_C( 511971347),
                              UINT32_C(3142311802), UINT32_C(3582165504), UINT32_C(3533175269), UINT32_C(3138584679),
                              UINT32_C(3117232701), UINT32_C(1582887517), UINT32_C(2957127939), UINT32_C(3388466484)) },
    { simde_x_mm512_set_epu32(UINT32_C(2382371522), UINT32_C(  66180421), UINT32_C(3915007092), UINT32_C(3548556152),
                              UINT32_C(3063171483), UINT32_C( 175336822), UINT32_C(2621074902), UINT32_C(2785523281),
                              UINT32_C(3351907467), UINT32_C(3611626580), UINT32_C(3274777282), UINT32_C(2819588991),
                              UINT32_C(4142757399), UINT32_C(3841212820), UINT32_C(1375549108), UINT32_C(3217099434)),
      simde_x_mm512_set_epu32(UINT32_C(2382371522), UINT32_C(  66180421), UINT32_C(3915007092), UINT32_C(3548556152),
                              UINT32_C(3063171483), UINT32_C( 175336822), UINT32_C(2621074902), UINT32_C(2785523281),
                              UINT32_C(3351907467), UINT32_C(3611626580), UINT32_C(3274777282), UINT32_C(2819588991),
                              UINT32_C(4142757399), UINT32_C(3841212820), UINT32_C(1375549108), UINT32_C(3217099434)) },
    { simde_x_mm512_set_epu32(UINT32_C(1625945136), UINT32_C(  82950125), UINT32_C(3598722192), UINT32_C(2456005821),
                              UINT32_C(3054050921), UINT32_C(3350002014), UINT32_C(1546778759), UINT32_C(3175686900),
                              UINT32_C(3418645543), UINT32_C(1247476579), UINT32_C(2559569107), UINT32_C(3884223622),
                              UINT32_C(2206347705), UINT32_C(1195297710), UINT32_C(4206427691), UINT32_C(2187435296)),
      simde_x_mm512_set_epu32(UINT32_C(1625945136), UINT32_C(  82950125), UINT32_C(3598722192), UINT32_C(2456005821),
                              UINT32_C(3054050921), UINT32_C(3350002014), UINT32_C(1546778759), UINT32_C(3175686900),
                              UINT32_C(3418645543), UINT32_C(1247476579), UINT32_C(2559569107), UINT32_C(3884223622),
                              UINT32_C(2206347705), UINT32_C(1195297710), UINT32_C(4206427691), UINT32_C(2187435296)) },
    { simde_x_mm512_set_epu32(UINT32_C(3055114510), UINT32_C( 314498376), UINT32_C( 259740532), UINT32_C(2845634146),
                              UINT32_C(3528445754), UINT32_C(1438308061), UINT32_C(1618483487), UINT32_C(4280155704),
                              UINT32_C(4191548278), UINT32_C( 955760205), UINT32_C(3071952989), UINT32_C(3353486020),
                              UINT32_C(3091053226), UINT32_C(2241572393), UINT32_C(3491849165), UINT32_C(2750648051)),
      simde_x_mm512_set_epu32(UINT32_C(3055114510), UINT32_C( 314498376), UINT32_C( 259740532), UINT32_C(2845634146),
                              UINT32_C(3528445754), UINT32_C(1438308061), UINT32_C(1618483487), UINT32_C(4280155704),
                              UINT32_C(4191548278), UINT32_C( 955760205), UINT32_C(3071952989), UINT32_C(3353486020),
                              UINT32_C(3091053226), UINT32_C(2241572393), UINT32_C(3491849165), UINT32_C(2750648051)) },
    { simde_x_mm512_set_epu32(UINT32_C(2791699552), UINT32_C(1697626027), UINT32_C(3068022880), UINT32_C( 492436222),
                              UINT32_C(2413088982), UINT32_C(1530446668), UINT32_C(1370127960), UINT32_C(2402932897),
                              UINT32_C(4061542194), UINT32_C( 154485056), UINT32_C(3577835063), UINT32_C(3500138573),
                              UINT32_C(  48074834), UINT32_C(1773313389), UINT32_C(3571862316), UINT32_C(1059958902)),
      simde_x_mm512_set_epu32(UINT32_C(2791699552), UINT32_C(1697626027), UINT32_C(3068022880), UINT32_C( 492436222),
                              UINT32_C(2413088982), UINT32_C(1530446668), UINT32_C(1370127960), UINT32_C(2402932897),
                              UINT32_C(4061542194), UINT32_C( 154485056), UINT32_C(3577835063), UINT32_C(3500138573),
                              UINT32_C(  48074834), UINT32_C(1773313389), UINT32_C(3571862316), UINT32_C(1059958902)) },
    { simde_x_mm512_set_epu32(UINT32_C(1166001194), UINT32_C( 115042765), UINT32_C( 557502548), UINT32_C(2408114255),
                              UINT32_C( 870354895), UINT32_C( 955362708), UINT32_C(1149136654), UINT32_C(1920883489),
                              UINT32_C(3238897491), UINT32_C(1952390233), UINT32_C( 223001918), UINT32_C( 310736118),
                              UINT32_C(2747509005), UINT32_C( 134376306), UINT32_C(1234549716), UINT32_C( 594304164)),
      simde_x_mm512_set_epu32(UINT32_C(1166001194), UINT32_C( 115042765), UINT32_C( 557502548), UINT32_C(2408114255),
                              UINT32_C( 870354895), UINT32_C( 955362708), UINT32_C(1149136654), UINT32_C(1920883489),
                              UINT32_C(3238897491), UINT32_C(1952390233), UINT32_C( 223001918), UINT32_C( 310736118),
                              UINT32_C(2747509005), UINT32_C( 134376306), UINT32_C(1234549716), UINT32_C( 594304164)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_loadu_si512(&(test_vec[i].a));
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_mov_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-1748841636), INT32_C(  600342911), INT32_C( 1346502861), INT32_C(-1119296012),
                            INT32_C(  542725165), INT32_C(  811581991), INT32_C(-1753809264), INT32_C(-2095888677),
                            INT32_C(   21844621), INT32_C( -668859652), INT32_C(  304402382), INT32_C( 1173008100),
                            INT32_C( -727866068), INT32_C(     599757), INT32_C( 1586862788), INT32_C(-1998308703)),
      UINT16_C(60467),
      simde_mm512_set_epi32(INT32_C(  646279344), INT32_C(-1381858570), INT32_C( 1528846110), INT32_C( -686931066),
                            INT32_C( 1116827472), INT32_C(-1909789352), INT32_C( 1965198777), INT32_C(  743332806),
                            INT32_C( -485827488), INT32_C(-1346955971), INT32_C(  625279893), INT32_C(  390045432),
                            INT32_C( 1242706406), INT32_C( -451702195), INT32_C( -797642518), INT32_C( 1682577743)),
      simde_mm512_set_epi32(INT32_C(  646279344), INT32_C(-1381858570), INT32_C( 1528846110), INT32_C(-1119296012),
                            INT32_C( 1116827472), INT32_C(-1909789352), INT32_C(-1753809264), INT32_C(-2095888677),
                            INT32_C(   21844621), INT32_C( -668859652), INT32_C(  625279893), INT32_C(  390045432),
                            INT32_C( -727866068), INT32_C(     599757), INT32_C( -797642518), INT32_C( 1682577743)) },
    { simde_mm512_set_epi32(INT32_C(  478337815), INT32_C( -537978403), INT32_C(-1351889488), INT32_C( 1090048308),
                            INT32_C( 1261235095), INT32_C(-1289893124), INT32_C( -387446550), INT32_C(-1938729505),
                            INT32_C(-1389958008), INT32_C( 1730413171), INT32_C( 1087827160), INT32_C( 1023459790),
                            INT32_C(-1481706049), INT32_C(  911835427), INT32_C(-1124770978), INT32_C( 1212776438)),
      UINT16_C(65510),
      simde_mm512_set_epi32(INT32_C( -876793269), INT32_C(  354193822), INT32_C(-1942817736), INT32_C(   48337666),
                            INT32_C(-1069034730), INT32_C( -258187388), INT32_C(-1735480646), INT32_C( 1239662333),
                            INT32_C(-1087348321), INT32_C(  777072035), INT32_C( -223191004), INT32_C( -671373205),
                            INT32_C( -333775053), INT32_C( 1946636837), INT32_C(  875386084), INT32_C(   41135181)),
      simde_mm512_set_epi32(INT32_C( -876793269), INT32_C(  354193822), INT32_C(-1942817736), INT32_C(   48337666),
                            INT32_C(-1069034730), INT32_C( -258187388), INT32_C(-1735480646), INT32_C( 1239662333),
                            INT32_C(-1087348321), INT32_C(  777072035), INT32_C( -223191004), INT32_C( 1023459790),
                            INT32_C(-1481706049), INT32_C( 1946636837), INT32_C(  875386084), INT32_C( 1212776438)) },
    { simde_mm512_set_epi32(INT32_C(  739047763), INT32_C( 1498945773), INT32_C( 1776295699), INT32_C( 1298376143),
                            INT32_C(-1413206606), INT32_C(-1101195004), INT32_C( 1096357047), INT32_C( 1201409099),
                            INT32_C(-1184934080), INT32_C(-1142871559), INT32_C(-1331799428), INT32_C( 2127606263),
                            INT32_C( 1810587941), INT32_C(-1568035201), INT32_C(-1514801640), INT32_C( 1754146272)),
      UINT16_C(17782),
      simde_mm512_set_epi32(INT32_C(-1637684250), INT32_C( 1624419961), INT32_C(-1721698305), INT32_C( 1216991175),
                            INT32_C( 1086797293), INT32_C( -544515074), INT32_C(-1866991972), INT32_C( 1497966040),
                            INT32_C(  183681068), INT32_C( 1846911046), INT32_C(  396433769), INT32_C( 1567943719),
                            INT32_C( 1544652060), INT32_C( 1999507462), INT32_C( -389522003), INT32_C(  660842170)),
      simde_mm512_set_epi32(INT32_C(  739047763), INT32_C( 1624419961), INT32_C( 1776295699), INT32_C( 1298376143),
                            INT32_C(-1413206606), INT32_C( -544515074), INT32_C( 1096357047), INT32_C( 1497966040),
                            INT32_C(-1184934080), INT32_C( 1846911046), INT32_C(  396433769), INT32_C( 1567943719),
                            INT32_C( 1810587941), INT32_C( 1999507462), INT32_C( -389522003), INT32_C( 1754146272)) },
    { simde_mm512_set_epi32(INT32_C(-1787060903), INT32_C( 1591528199), INT32_C( 1360730903), INT32_C( -392663993),
                            INT32_C( 1833403381), INT32_C(  667948495), INT32_C(-1351186880), INT32_C(-1869951013),
                            INT32_C(-1764668962), INT32_C( 1727501907), INT32_C(-1699520398), INT32_C(-2078068732),
                            INT32_C(-1191187391), INT32_C(  809086335), INT32_C( -915516374), INT32_C( 2044786719)),
      UINT16_C(19153),
      simde_mm512_set_epi32(INT32_C(-1124863619), INT32_C( -733840886), INT32_C(  225375619), INT32_C( 2033345748),
                            INT32_C(   62836182), INT32_C(-1797131359), INT32_C( -791707937), INT32_C(-1161020437),
                            INT32_C( 1933148289), INT32_C(-1354039663), INT32_C(  533923030), INT32_C(  457770626),
                            INT32_C(-2130199261), INT32_C( -201626469), INT32_C( 1603256738), INT32_C(  385840376)),
      simde_mm512_set_epi32(INT32_C(-1787060903), INT32_C( -733840886), INT32_C( 1360730903), INT32_C( -392663993),
                            INT32_C(   62836182), INT32_C(  667948495), INT32_C( -791707937), INT32_C(-1869951013),
                            INT32_C( 1933148289), INT32_C(-1354039663), INT32_C(-1699520398), INT32_C(  457770626),
                            INT32_C(-1191187391), INT32_C(  809086335), INT32_C( -915516374), INT32_C(  385840376)) },
    { simde_mm512_set_epi32(INT32_C(-1844996035), INT32_C( -483918772), INT32_C(-1530619556), INT32_C( -447486042),
                            INT32_C( -153016391), INT32_C( 1772993408), INT32_C(-1557466731), INT32_C( 1884729185),
                            INT32_C(-1170473640), INT32_C( -231873321), INT32_C( 1063107119), INT32_C( 1409583343),
                            INT32_C(  131479252), INT32_C(-1464445699), INT32_C(-1859507666), INT32_C( 1142318206)),
      UINT16_C(39686),
      simde_mm512_set_epi32(INT32_C(-1710909147), INT32_C( 1655743921), INT32_C(-1520991125), INT32_C(-1200934587),
                            INT32_C( -721899112), INT32_C( 1216881740), INT32_C( -481496777), INT32_C( -893026644),
                            INT32_C(-2035526652), INT32_C( -294630589), INT32_C(-1446210787), INT32_C( -547573265),
                            INT32_C( 1911285838), INT32_C(-1067024301), INT32_C(-1545394687), INT32_C( 1507767747)),
      simde_mm512_set_epi32(INT32_C(-1710909147), INT32_C( -483918772), INT32_C(-1530619556), INT32_C(-1200934587),
                            INT32_C( -721899112), INT32_C( 1772993408), INT32_C( -481496777), INT32_C( -893026644),
                            INT32_C(-1170473640), INT32_C( -231873321), INT32_C( 1063107119), INT32_C( 1409583343),
                            INT32_C(  131479252), INT32_C(-1067024301), INT32_C(-1545394687), INT32_C( 1142318206)) },
    { simde_mm512_set_epi32(INT32_C( 2003854537), INT32_C(  316518418), INT32_C(-2128378506), INT32_C( -814023178),
                            INT32_C( 2134095257), INT32_C( -273917753), INT32_C(  269941696), INT32_C(-1761573676),
                            INT32_C( -504711162), INT32_C( 1086943646), INT32_C( -304633534), INT32_C( -905159738),
                            INT32_C(-1025692186), INT32_C(-2082862175), INT32_C(-1626855678), INT32_C(-1231176910)),
      UINT16_C(13329),
      simde_mm512_set_epi32(INT32_C(  838273890), INT32_C( 1209103370), INT32_C(  947433971), INT32_C(   91213725),
                            INT32_C(  749577280), INT32_C(  157602752), INT32_C( 2125537515), INT32_C( -782796801),
                            INT32_C( -120430288), INT32_C( -810448185), INT32_C( -659512402), INT32_C(  419195007),
                            INT32_C( -830103963), INT32_C( -756234442), INT32_C(  376291679), INT32_C( -610488282)),
      simde_mm512_set_epi32(INT32_C( 2003854537), INT32_C(  316518418), INT32_C(  947433971), INT32_C(   91213725),
                            INT32_C( 2134095257), INT32_C(  157602752), INT32_C(  269941696), INT32_C(-1761573676),
                            INT32_C( -504711162), INT32_C( 1086943646), INT32_C( -304633534), INT32_C(  419195007),
                            INT32_C(-1025692186), INT32_C(-2082862175), INT32_C(-1626855678), INT32_C( -610488282)) },
    { simde_mm512_set_epi32(INT32_C( -974755823), INT32_C(  -98121742), INT32_C( 1561555936), INT32_C(-1281058782),
                            INT32_C(-2008886211), INT32_C( 1568326299), INT32_C( 1232828554), INT32_C(  127919997),
                            INT32_C( 1015818460), INT32_C( -681833659), INT32_C(  340145717), INT32_C( 1048452961),
                            INT32_C(  749206991), INT32_C( 1290937767), INT32_C(-1150545818), INT32_C(  -48881570)),
      UINT16_C(55435),
      simde_mm512_set_epi32(INT32_C( 1177945769), INT32_C(-1878447950), INT32_C( -271391312), INT32_C(-2014500164),
                            INT32_C(-2080120479), INT32_C( 1195569010), INT32_C(-1583493780), INT32_C( 1466155853),
                            INT32_C( -735473338), INT32_C( 1922464741), INT32_C( -224185100), INT32_C( -929578437),
                            INT32_C(  831459587), INT32_C(-1105963780), INT32_C(-1360707796), INT32_C( -211781248)),
      simde_mm512_set_epi32(INT32_C( 1177945769), INT32_C(-1878447950), INT32_C( 1561555936), INT32_C(-2014500164),
                            INT32_C(-2080120479), INT32_C( 1568326299), INT32_C( 1232828554), INT32_C(  127919997),
                            INT32_C( -735473338), INT32_C( -681833659), INT32_C(  340145717), INT32_C( 1048452961),
                            INT32_C(  831459587), INT32_C( 1290937767), INT32_C(-1360707796), INT32_C( -211781248)) },
    { simde_mm512_set_epi32(INT32_C( 1583932216), INT32_C(-1528139164), INT32_C(  665399981), INT32_C(  718332631),
                            INT32_C( -984331868), INT32_C(-1317077859), INT32_C(-1440392153), INT32_C(-1978382578),
                            INT32_C(  828185710), INT32_C( 1905160582), INT32_C(  120938992), INT32_C( 1613459128),
                            INT32_C( -812252493), INT32_C(-1503952372), INT32_C(  231875300), INT32_C( -885498028)),
      UINT16_C(45743),
      simde_mm512_set_epi32(INT32_C(-1033540577), INT32_C( -995705628), INT32_C(-2098565905), INT32_C(-1609941379),
                            INT32_C(  451122481), INT32_C(  898911803), INT32_C( -918933314), INT32_C( 1301755496),
                            INT32_C(  654535343), INT32_C( 1915381036), INT32_C( -595265918), INT32_C( -204141630),
                            INT32_C(-1824782722), INT32_C(-1457642917), INT32_C(-1358921472), INT32_C( 1013008616)),
      simde_mm512_set_epi32(INT32_C(-1033540577), INT32_C(-1528139164), INT32_C(-2098565905), INT32_C(-1609941379),
                            INT32_C( -984331868), INT32_C(-1317077859), INT32_C( -918933314), INT32_C(-1978382578),
                            INT32_C(  654535343), INT32_C( 1905160582), INT32_C( -595265918), INT32_C( 1613459128),
                            INT32_C(-1824782722), INT32_C(-1457642917), INT32_C(-1358921472), INT32_C( 1013008616)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_mov_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_mov_epi64(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 8729250599109288206), INT64_C(  925123000700261284),
                            INT64_C( -996462675499144949), INT64_C(-5486361937319788764),
                            INT64_C(-1619246833501834651), INT64_C(-1914665916415518359),
                            INT64_C( 4596079613709719053), INT64_C(-1669293344454375632)),
      UINT8_C(136),
      simde_mm512_set_epi64(INT64_C(-2718786087636304341), INT64_C( 6271007050593066413),
                            INT64_C( 7325428114350079264), INT64_C( 8373606416957659495),
                            INT64_C( 8585702140748752091), INT64_C(-6106352141912550191),
                            INT64_C(-7415158757307660945), INT64_C(-4168322686232168747)),
      simde_mm512_set_epi64(INT64_C(-2718786087636304341), INT64_C(  925123000700261284),
                            INT64_C( -996462675499144949), INT64_C(-5486361937319788764),
                            INT64_C( 8585702140748752091), INT64_C(-1914665916415518359),
                            INT64_C( 4596079613709719053), INT64_C(-1669293344454375632)) },
    { simde_mm512_set_epi64(INT64_C( 8240025841211248490), INT64_C( 2437990159450284908),
                            INT64_C( 2201815834941113848), INT64_C( 7879550161691977002),
                            INT64_C( 3825487759520775297), INT64_C( 6674403996216424931),
                            INT64_C(-5802137669857725171), INT64_C( 5686996017309487110)),
      UINT8_C(227),
      simde_mm512_set_epi64(INT64_C(  120730317606372397), INT64_C(-1410770079656234556),
                            INT64_C( 4532617684378198659), INT64_C( 9004023903916376139),
                            INT64_C( 7206885247739448460), INT64_C(-6411218032719574536),
                            INT64_C( -962636034832057562), INT64_C(-6211267245753502041)),
      simde_mm512_set_epi64(INT64_C(  120730317606372397), INT64_C(-1410770079656234556),
                            INT64_C( 4532617684378198659), INT64_C( 7879550161691977002),
                            INT64_C( 3825487759520775297), INT64_C( 6674403996216424931),
                            INT64_C( -962636034832057562), INT64_C(-6211267245753502041)) },
    { simde_mm512_set_epi64(INT64_C( 4674722797399239366), INT64_C( 2000178744548395677),
                            INT64_C(-3230169679464817239), INT64_C( 6675942378016655726),
                            INT64_C(-4074632284771109640), INT64_C(-1969073951075376054),
                            INT64_C(-7309602967246577272), INT64_C( 6746883208360816464)),
      UINT8_C(189),
      simde_mm512_set_epi64(INT64_C( 7111791735729821232), INT64_C(-6377956101145598745),
                            INT64_C(-4955467359912007508), INT64_C( -340840922408165844),
                            INT64_C( 3280430708356940081), INT64_C(  400669322893233577),
                            INT64_C( 6742772793155919855), INT64_C(-1365845768056837484)),
      simde_mm512_set_epi64(INT64_C( 7111791735729821232), INT64_C( 2000178744548395677),
                            INT64_C(-4955467359912007508), INT64_C( -340840922408165844),
                            INT64_C( 3280430708356940081), INT64_C(  400669322893233577),
                            INT64_C(-7309602967246577272), INT64_C(-1365845768056837484)) },
    { simde_mm512_set_epi64(INT64_C(-5185665192936807952), INT64_C( 2873887117219468065),
                            INT64_C(  944218707053685182), INT64_C(-6471325153303919649),
                            INT64_C(-1551809186210791512), INT64_C( 8676397618641344048),
                            INT64_C(-1480083839359048471), INT64_C(-2573286236881012052)),
      UINT8_C(135),
      simde_mm512_set_epi64(INT64_C( 4851071406626175825), INT64_C( 2006733877612279017),
                            INT64_C( 9148059701805005067), INT64_C( 3484083856858518164),
                            INT64_C( -542612751996632572), INT64_C( 6154040976669554118),
                            INT64_C( 4310055852136225460), INT64_C( 6666177398356729891)),
      simde_mm512_set_epi64(INT64_C( 4851071406626175825), INT64_C( 2873887117219468065),
                            INT64_C(  944218707053685182), INT64_C(-6471325153303919649),
                            INT64_C(-1551809186210791512), INT64_C( 6154040976669554118),
                            INT64_C( 4310055852136225460), INT64_C( 6666177398356729891)) },
    { simde_mm512_set_epi64(INT64_C(-6362423492218583699), INT64_C( 4052676248150053459),
                            INT64_C(-1785632160509127109), INT64_C( 4504790352522402260),
                            INT64_C(  214305831990150369), INT64_C( 4122674741194642780),
                            INT64_C(-9061446978520477770), INT64_C( -925260945734331795)),
      UINT8_C( 88),
      simde_mm512_set_epi64(INT64_C( 7816755513219693536), INT64_C(-8078701368125426812),
                            INT64_C( 5999276564615449517), INT64_C(-3747208296317683129),
                            INT64_C(-3767121149493822975), INT64_C( 3269862772677933078),
                            INT64_C(-1274534447611012205), INT64_C(  367478185734650139)),
      simde_mm512_set_epi64(INT64_C(-6362423492218583699), INT64_C(-8078701368125426812),
                            INT64_C(-1785632160509127109), INT64_C(-3747208296317683129),
                            INT64_C(-3767121149493822975), INT64_C( 4122674741194642780),
                            INT64_C(-9061446978520477770), INT64_C( -925260945734331795)) },
    { simde_mm512_set_epi64(INT64_C(-6749425177074609965), INT64_C( 8453995530571484051),
                            INT64_C(-7619559937003101591), INT64_C( 3005943923235484348),
                            INT64_C( 4327678115781969631), INT64_C( 5990841649027118513),
                            INT64_C(-1241607161778990291), INT64_C(  -91855491071654622)),
      UINT8_C( 22),
      simde_mm512_set_epi64(INT64_C( 4461859928182214174), INT64_C(-5186049742858346871),
                            INT64_C( -636993447067685727), INT64_C( 8339698509359201789),
                            INT64_C( 4598711567911914631), INT64_C( 7428996315725576873),
                            INT64_C( 6513452752711502515), INT64_C(-6603414145042292282)),
      simde_mm512_set_epi64(INT64_C(-6749425177074609965), INT64_C( 8453995530571484051),
                            INT64_C(-7619559937003101591), INT64_C( 8339698509359201789),
                            INT64_C( 4327678115781969631), INT64_C( 7428996315725576873),
                            INT64_C( 6513452752711502515), INT64_C(  -91855491071654622)) },
    { simde_mm512_set_epi64(INT64_C(-7023609179598013523), INT64_C(-2166290313032224989),
                            INT64_C(-1367963225958164233), INT64_C(-9082538196892642083),
                            INT64_C(-7482977792619995502), INT64_C( 4800709110944492165),
                            INT64_C( 3082355013095664677), INT64_C(-4286500001112695437)),
      UINT8_C( 42),
      simde_mm512_set_epi64(INT64_C(  522664068472938939), INT64_C(-5622535385140832229),
                            INT64_C( 4829749372798053845), INT64_C( -330958976268778895),
                            INT64_C(-2657198631452288613), INT64_C(-3805394135151266272),
                            INT64_C( 4429043998616724751), INT64_C( 4131511442627175760)),
      simde_mm512_set_epi64(INT64_C(-7023609179598013523), INT64_C(-2166290313032224989),
                            INT64_C( 4829749372798053845), INT64_C(-9082538196892642083),
                            INT64_C(-2657198631452288613), INT64_C( 4800709110944492165),
                            INT64_C( 4429043998616724751), INT64_C(-4286500001112695437)) },
    { simde_mm512_set_epi64(INT64_C(-5567656428388000347), INT64_C( -971128712423557311),
                            INT64_C( 3761317547504069574), INT64_C( 6096071933426825544),
                            INT64_C( 3108166743366703612), INT64_C(-3435283790563075237),
                            INT64_C( 3598996591046999900), INT64_C( 2520744130071328064)),
      UINT8_C( 13),
      simde_mm512_set_epi64(INT64_C( -976144998301952820), INT64_C( 5304141922221069696),
                            INT64_C( 1153833608356774417), INT64_C(-5431879705444140176),
                            INT64_C(-4200442870371425874), INT64_C( 9118970466689378415),
                            INT64_C( 7182201605874776129), INT64_C( 6344954152679193639)),
      simde_mm512_set_epi64(INT64_C(-5567656428388000347), INT64_C( -971128712423557311),
                            INT64_C( 3761317547504069574), INT64_C( 6096071933426825544),
                            INT64_C(-4200442870371425874), INT64_C( 9118970466689378415),
                            INT64_C( 3598996591046999900), INT64_C( 6344954152679193639)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_mov_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_mov_ps(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -278.44), SIMDE_FLOAT32_C(   958.04), SIMDE_FLOAT32_C(  -686.18), SIMDE_FLOAT32_C(  -120.52),
                         SIMDE_FLOAT32_C(   759.91), SIMDE_FLOAT32_C(   470.87), SIMDE_FLOAT32_C(  -723.57), SIMDE_FLOAT32_C(   170.04),
                         SIMDE_FLOAT32_C(   559.73), SIMDE_FLOAT32_C(   984.13), SIMDE_FLOAT32_C(   -84.72), SIMDE_FLOAT32_C(  -543.95),
                         SIMDE_FLOAT32_C(   998.02), SIMDE_FLOAT32_C(  -559.31), SIMDE_FLOAT32_C(   134.12), SIMDE_FLOAT32_C(  -230.64)),
      UINT16_C(    0),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -161.72), SIMDE_FLOAT32_C(   540.27), SIMDE_FLOAT32_C(  -745.55), SIMDE_FLOAT32_C(   623.14),
                         SIMDE_FLOAT32_C(  -272.95), SIMDE_FLOAT32_C(   176.76), SIMDE_FLOAT32_C(  -957.12), SIMDE_FLOAT32_C(  -720.97),
                         SIMDE_FLOAT32_C(  -491.62), SIMDE_FLOAT32_C(   442.72), SIMDE_FLOAT32_C(    94.42), SIMDE_FLOAT32_C(  -425.44),
                         SIMDE_FLOAT32_C(   378.60), SIMDE_FLOAT32_C(  -248.93), SIMDE_FLOAT32_C(   638.30), SIMDE_FLOAT32_C(  -857.32)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -278.44), SIMDE_FLOAT32_C(   958.04), SIMDE_FLOAT32_C(  -686.18), SIMDE_FLOAT32_C(  -120.52),
                         SIMDE_FLOAT32_C(   759.91), SIMDE_FLOAT32_C(   470.87), SIMDE_FLOAT32_C(  -723.57), SIMDE_FLOAT32_C(   170.04),
                         SIMDE_FLOAT32_C(   559.73), SIMDE_FLOAT32_C(   984.13), SIMDE_FLOAT32_C(   -84.72), SIMDE_FLOAT32_C(  -543.95),
                         SIMDE_FLOAT32_C(   998.02), SIMDE_FLOAT32_C(  -559.31), SIMDE_FLOAT32_C(   134.12), SIMDE_FLOAT32_C(  -230.64)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -455.21), SIMDE_FLOAT32_C(  -180.02), SIMDE_FLOAT32_C(  -110.74), SIMDE_FLOAT32_C(  -586.50),
                         SIMDE_FLOAT32_C(    -9.89), SIMDE_FLOAT32_C(  -597.54), SIMDE_FLOAT32_C(   553.79), SIMDE_FLOAT32_C(   611.64),
                         SIMDE_FLOAT32_C(   717.03), SIMDE_FLOAT32_C(  -381.85), SIMDE_FLOAT32_C(   862.32), SIMDE_FLOAT32_C(   302.29),
                         SIMDE_FLOAT32_C(   146.86), SIMDE_FLOAT32_C(  -693.40), SIMDE_FLOAT32_C(  -247.57), SIMDE_FLOAT32_C(  -469.49)),
      UINT16_C(    0),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   842.67), SIMDE_FLOAT32_C(  -856.89), SIMDE_FLOAT32_C(  -490.76), SIMDE_FLOAT32_C(   922.81),
                         SIMDE_FLOAT32_C(   -69.36), SIMDE_FLOAT32_C(   380.23), SIMDE_FLOAT32_C(  -846.01), SIMDE_FLOAT32_C(  -485.23),
                         SIMDE_FLOAT32_C(  -171.14), SIMDE_FLOAT32_C(   602.88), SIMDE_FLOAT32_C(  -717.33), SIMDE_FLOAT32_C(   336.05),
                         SIMDE_FLOAT32_C(  -432.71), SIMDE_FLOAT32_C(  -881.01), SIMDE_FLOAT32_C(  -255.82), SIMDE_FLOAT32_C(   168.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -455.21), SIMDE_FLOAT32_C(  -180.02), SIMDE_FLOAT32_C(  -110.74), SIMDE_FLOAT32_C(  -586.50),
                         SIMDE_FLOAT32_C(    -9.89), SIMDE_FLOAT32_C(  -597.54), SIMDE_FLOAT32_C(   553.79), SIMDE_FLOAT32_C(   611.64),
                         SIMDE_FLOAT32_C(   717.03), SIMDE_FLOAT32_C(  -381.85), SIMDE_FLOAT32_C(   862.32), SIMDE_FLOAT32_C(   302.29),
                         SIMDE_FLOAT32_C(   146.86), SIMDE_FLOAT32_C(  -693.40), SIMDE_FLOAT32_C(  -247.57), SIMDE_FLOAT32_C(  -469.49)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -694.40), SIMDE_FLOAT32_C(  -404.01), SIMDE_FLOAT32_C(   766.51), SIMDE_FLOAT32_C(  -392.19),
                         SIMDE_FLOAT32_C(  -908.15), SIMDE_FLOAT32_C(  -690.12), SIMDE_FLOAT32_C(  -262.73), SIMDE_FLOAT32_C(  -353.25),
                         SIMDE_FLOAT32_C(  -451.03), SIMDE_FLOAT32_C(   -88.58), SIMDE_FLOAT32_C(   658.99), SIMDE_FLOAT32_C(  -961.05),
                         SIMDE_FLOAT32_C(  -743.39), SIMDE_FLOAT32_C(   747.85), SIMDE_FLOAT32_C(  -989.89), SIMDE_FLOAT32_C(   -48.62)),
      UINT16_C(    0),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -585.79), SIMDE_FLOAT32_C(  -884.44), SIMDE_FLOAT32_C(  -722.53), SIMDE_FLOAT32_C(   296.99),
                         SIMDE_FLOAT32_C(   791.87), SIMDE_FLOAT32_C(   514.23), SIMDE_FLOAT32_C(   110.66), SIMDE_FLOAT32_C(  -891.24),
                         SIMDE_FLOAT32_C(  -893.87), SIMDE_FLOAT32_C(   597.88), SIMDE_FLOAT32_C(  -561.25), SIMDE_FLOAT32_C(  -182.63),
                         SIMDE_FLOAT32_C(   -91.96), SIMDE_FLOAT32_C(   272.32), SIMDE_FLOAT32_C(   -87.60), SIMDE_FLOAT32_C(    34.84)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -694.40), SIMDE_FLOAT32_C(  -404.01), SIMDE_FLOAT32_C(   766.51), SIMDE_FLOAT32_C(  -392.19),
                         SIMDE_FLOAT32_C(  -908.15), SIMDE_FLOAT32_C(  -690.12), SIMDE_FLOAT32_C(  -262.73), SIMDE_FLOAT32_C(  -353.25),
                         SIMDE_FLOAT32_C(  -451.03), SIMDE_FLOAT32_C(   -88.58), SIMDE_FLOAT32_C(   658.99), SIMDE_FLOAT32_C(  -961.05),
                         SIMDE_FLOAT32_C(  -743.39), SIMDE_FLOAT32_C(   747.85), SIMDE_FLOAT32_C(  -989.89), SIMDE_FLOAT32_C(   -48.62)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   706.89), SIMDE_FLOAT32_C(   473.35), SIMDE_FLOAT32_C(   525.10), SIMDE_FLOAT32_C(    58.51),
                         SIMDE_FLOAT32_C(  -849.29), SIMDE_FLOAT32_C(   830.92), SIMDE_FLOAT32_C(   666.67), SIMDE_FLOAT32_C(   510.60),
                         SIMDE_FLOAT32_C(   494.95), SIMDE_FLOAT32_C(  -644.02), SIMDE_FLOAT32_C(   666.48), SIMDE_FLOAT32_C(   728.99),
                         SIMDE_FLOAT32_C(    57.50), SIMDE_FLOAT32_C(  -509.99), SIMDE_FLOAT32_C(   -86.32), SIMDE_FLOAT32_C(   945.97)),
      UINT16_C(    0),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   396.65), SIMDE_FLOAT32_C(  -337.05), SIMDE_FLOAT32_C(    13.39), SIMDE_FLOAT32_C(   374.11),
                         SIMDE_FLOAT32_C(   941.83), SIMDE_FLOAT32_C(   -80.39), SIMDE_FLOAT32_C(  -533.82), SIMDE_FLOAT32_C(   -81.97),
                         SIMDE_FLOAT32_C(   -76.37), SIMDE_FLOAT32_C(  -466.22), SIMDE_FLOAT32_C(  -527.13), SIMDE_FLOAT32_C(   285.31),
                         SIMDE_FLOAT32_C(  -159.19), SIMDE_FLOAT32_C(  -769.18), SIMDE_FLOAT32_C(   908.64), SIMDE_FLOAT32_C(  -647.66)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   706.89), SIMDE_FLOAT32_C(   473.35), SIMDE_FLOAT32_C(   525.10), SIMDE_FLOAT32_C(    58.51),
                         SIMDE_FLOAT32_C(  -849.29), SIMDE_FLOAT32_C(   830.92), SIMDE_FLOAT32_C(   666.67), SIMDE_FLOAT32_C(   510.60),
                         SIMDE_FLOAT32_C(   494.95), SIMDE_FLOAT32_C(  -644.02), SIMDE_FLOAT32_C(   666.48), SIMDE_FLOAT32_C(   728.99),
                         SIMDE_FLOAT32_C(    57.50), SIMDE_FLOAT32_C(  -509.99), SIMDE_FLOAT32_C(   -86.32), SIMDE_FLOAT32_C(   945.97)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   305.38), SIMDE_FLOAT32_C(   354.29), SIMDE_FLOAT32_C(   625.78), SIMDE_FLOAT32_C(   840.33),
                         SIMDE_FLOAT32_C(   398.08), SIMDE_FLOAT32_C(  -775.15), SIMDE_FLOAT32_C(  -749.75), SIMDE_FLOAT32_C(  -579.50),
                         SIMDE_FLOAT32_C(   326.67), SIMDE_FLOAT32_C(  -369.97), SIMDE_FLOAT32_C(  -888.36), SIMDE_FLOAT32_C(  -369.43),
                         SIMDE_FLOAT32_C(   587.01), SIMDE_FLOAT32_C(  -977.20), SIMDE_FLOAT32_C(  -154.58), SIMDE_FLOAT32_C(  -264.71)),
      UINT16_C(    0),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   472.46), SIMDE_FLOAT32_C(  -814.28), SIMDE_FLOAT32_C(   331.94), SIMDE_FLOAT32_C(   -36.35),
                         SIMDE_FLOAT32_C(   -98.00), SIMDE_FLOAT32_C(   862.68), SIMDE_FLOAT32_C(  -130.24), SIMDE_FLOAT32_C(    65.39),
                         SIMDE_FLOAT32_C(  -826.35), SIMDE_FLOAT32_C(    92.38), SIMDE_FLOAT32_C(  -698.83), SIMDE_FLOAT32_C(   457.07),
                         SIMDE_FLOAT32_C(  -472.97), SIMDE_FLOAT32_C(  -117.57), SIMDE_FLOAT32_C(  -498.77), SIMDE_FLOAT32_C(   798.69)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   305.38), SIMDE_FLOAT32_C(   354.29), SIMDE_FLOAT32_C(   625.78), SIMDE_FLOAT32_C(   840.33),
                         SIMDE_FLOAT32_C(   398.08), SIMDE_FLOAT32_C(  -775.15), SIMDE_FLOAT32_C(  -749.75), SIMDE_FLOAT32_C(  -579.50),
                         SIMDE_FLOAT32_C(   326.67), SIMDE_FLOAT32_C(  -369.97), SIMDE_FLOAT32_C(  -888.36), SIMDE_FLOAT32_C(  -369.43),
                         SIMDE_FLOAT32_C(   587.01), SIMDE_FLOAT32_C(  -977.20), SIMDE_FLOAT32_C(  -154.58), SIMDE_FLOAT32_C(  -264.71)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   845.06), SIMDE_FLOAT32_C(  -527.19), SIMDE_FLOAT32_C(  -753.05), SIMDE_FLOAT32_C(  -867.95),
                         SIMDE_FLOAT32_C(   -98.38), SIMDE_FLOAT32_C(   -90.28), SIMDE_FLOAT32_C(   321.06), SIMDE_FLOAT32_C(  -308.74),
                         SIMDE_FLOAT32_C(   969.13), SIMDE_FLOAT32_C(  -263.02), SIMDE_FLOAT32_C(  -517.54), SIMDE_FLOAT32_C(   566.67),
                         SIMDE_FLOAT32_C(  -321.03), SIMDE_FLOAT32_C(   -19.45), SIMDE_FLOAT32_C(  -773.18), SIMDE_FLOAT32_C(  -562.24)),
      UINT16_C(    0),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -313.43), SIMDE_FLOAT32_C(  -900.90), SIMDE_FLOAT32_C(  -480.72), SIMDE_FLOAT32_C(   288.15),
                         SIMDE_FLOAT32_C(   603.38), SIMDE_FLOAT32_C(   964.29), SIMDE_FLOAT32_C(   140.98), SIMDE_FLOAT32_C(   269.46),
                         SIMDE_FLOAT32_C(   960.77), SIMDE_FLOAT32_C(  -220.33), SIMDE_FLOAT32_C(   524.23), SIMDE_FLOAT32_C(  -633.14),
                         SIMDE_FLOAT32_C(  -680.30), SIMDE_FLOAT32_C(   880.56), SIMDE_FLOAT32_C(   661.76), SIMDE_FLOAT32_C(  -794.03)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   845.06), SIMDE_FLOAT32_C(  -527.19), SIMDE_FLOAT32_C(  -753.05), SIMDE_FLOAT32_C(  -867.95),
                         SIMDE_FLOAT32_C(   -98.38), SIMDE_FLOAT32_C(   -90.28), SIMDE_FLOAT32_C(   321.06), SIMDE_FLOAT32_C(  -308.74),
                         SIMDE_FLOAT32_C(   969.13), SIMDE_FLOAT32_C(  -263.02), SIMDE_FLOAT32_C(  -517.54), SIMDE_FLOAT32_C(   566.67),
                         SIMDE_FLOAT32_C(  -321.03), SIMDE_FLOAT32_C(   -19.45), SIMDE_FLOAT32_C(  -773.18), SIMDE_FLOAT32_C(  -562.24)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -595.71), SIMDE_FLOAT32_C(   923.49), SIMDE_FLOAT32_C(  -968.66), SIMDE_FLOAT32_C(   136.30),
                         SIMDE_FLOAT32_C(   658.04), SIMDE_FLOAT32_C(    31.08), SIMDE_FLOAT32_C(   664.79), SIMDE_FLOAT32_C(   525.95),
                         SIMDE_FLOAT32_C(   643.61), SIMDE_FLOAT32_C(  -559.86), SIMDE_FLOAT32_C(  -291.18), SIMDE_FLOAT32_C(    35.13),
                         SIMDE_FLOAT32_C(  -188.19), SIMDE_FLOAT32_C(   767.03), SIMDE_FLOAT32_C(  -828.01), SIMDE_FLOAT32_C(   801.09)),
      UINT16_C(    0),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -750.17), SIMDE_FLOAT32_C(   128.67), SIMDE_FLOAT32_C(   441.75), SIMDE_FLOAT32_C(   625.42),
                         SIMDE_FLOAT32_C(   865.73), SIMDE_FLOAT32_C(  -522.43), SIMDE_FLOAT32_C(   871.78), SIMDE_FLOAT32_C(   736.62),
                         SIMDE_FLOAT32_C(   -52.49), SIMDE_FLOAT32_C(  -188.89), SIMDE_FLOAT32_C(   163.52), SIMDE_FLOAT32_C(   743.65),
                         SIMDE_FLOAT32_C(  -912.98), SIMDE_FLOAT32_C(  -904.70), SIMDE_FLOAT32_C(   973.06), SIMDE_FLOAT32_C(  -214.13)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -595.71), SIMDE_FLOAT32_C(   923.49), SIMDE_FLOAT32_C(  -968.66), SIMDE_FLOAT32_C(   136.30),
                         SIMDE_FLOAT32_C(   658.04), SIMDE_FLOAT32_C(    31.08), SIMDE_FLOAT32_C(   664.79), SIMDE_FLOAT32_C(   525.95),
                         SIMDE_FLOAT32_C(   643.61), SIMDE_FLOAT32_C(  -559.86), SIMDE_FLOAT32_C(  -291.18), SIMDE_FLOAT32_C(    35.13),
                         SIMDE_FLOAT32_C(  -188.19), SIMDE_FLOAT32_C(   767.03), SIMDE_FLOAT32_C(  -828.01), SIMDE_FLOAT32_C(   801.09)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -351.95), SIMDE_FLOAT32_C(   902.78), SIMDE_FLOAT32_C(  -172.20), SIMDE_FLOAT32_C(   540.77),
                         SIMDE_FLOAT32_C(  -431.24), SIMDE_FLOAT32_C(   243.87), SIMDE_FLOAT32_C(   216.07), SIMDE_FLOAT32_C(   747.45),
                         SIMDE_FLOAT32_C(  -864.81), SIMDE_FLOAT32_C(  -982.67), SIMDE_FLOAT32_C(  -710.14), SIMDE_FLOAT32_C(  -539.39),
                         SIMDE_FLOAT32_C(  -100.27), SIMDE_FLOAT32_C(  -988.79), SIMDE_FLOAT32_C(  -220.83), SIMDE_FLOAT32_C(   489.72)),
      UINT16_C(    0),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   688.70), SIMDE_FLOAT32_C(  -942.30), SIMDE_FLOAT32_C(  -353.35), SIMDE_FLOAT32_C(  -645.42),
                         SIMDE_FLOAT32_C(   206.41), SIMDE_FLOAT32_C(   546.87), SIMDE_FLOAT32_C(  -878.90), SIMDE_FLOAT32_C(   614.84),
                         SIMDE_FLOAT32_C(   757.82), SIMDE_FLOAT32_C(   388.29), SIMDE_FLOAT32_C(  -767.39), SIMDE_FLOAT32_C(   567.68),
                         SIMDE_FLOAT32_C(   464.76), SIMDE_FLOAT32_C(  -828.44), SIMDE_FLOAT32_C(   843.54), SIMDE_FLOAT32_C(   504.38)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -351.95), SIMDE_FLOAT32_C(   902.78), SIMDE_FLOAT32_C(  -172.20), SIMDE_FLOAT32_C(   540.77),
                         SIMDE_FLOAT32_C(  -431.24), SIMDE_FLOAT32_C(   243.87), SIMDE_FLOAT32_C(   216.07), SIMDE_FLOAT32_C(   747.45),
                         SIMDE_FLOAT32_C(  -864.81), SIMDE_FLOAT32_C(  -982.67), SIMDE_FLOAT32_C(  -710.14), SIMDE_FLOAT32_C(  -539.39),
                         SIMDE_FLOAT32_C(  -100.27), SIMDE_FLOAT32_C(  -988.79), SIMDE_FLOAT32_C(  -220.83), SIMDE_FLOAT32_C(   489.72)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_mov_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_mask_mov_pd(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -997.43), SIMDE_FLOAT64_C(  -24.75),
                         SIMDE_FLOAT64_C(  811.92), SIMDE_FLOAT64_C(  716.01),
                         SIMDE_FLOAT64_C( -286.81), SIMDE_FLOAT64_C(  360.81),
                         SIMDE_FLOAT64_C( -618.94), SIMDE_FLOAT64_C(  103.41)),
      UINT8_C( 17),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  779.73), SIMDE_FLOAT64_C(  -71.34),
                         SIMDE_FLOAT64_C(   74.67), SIMDE_FLOAT64_C(  569.44),
                         SIMDE_FLOAT64_C(  765.94), SIMDE_FLOAT64_C(  114.94),
                         SIMDE_FLOAT64_C(   85.69), SIMDE_FLOAT64_C(  982.40)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -997.43), SIMDE_FLOAT64_C(  -24.75),
                         SIMDE_FLOAT64_C(  811.92), SIMDE_FLOAT64_C(  569.44),
                         SIMDE_FLOAT64_C( -286.81), SIMDE_FLOAT64_C(  360.81),
                         SIMDE_FLOAT64_C( -618.94), SIMDE_FLOAT64_C(  982.40)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -989.28), SIMDE_FLOAT64_C( -906.64),
                         SIMDE_FLOAT64_C( -211.36), SIMDE_FLOAT64_C( -108.84),
                         SIMDE_FLOAT64_C(  211.05), SIMDE_FLOAT64_C( -602.13),
                         SIMDE_FLOAT64_C(   19.95), SIMDE_FLOAT64_C( -745.56)),
      UINT8_C(115),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -995.20), SIMDE_FLOAT64_C(   66.82),
                         SIMDE_FLOAT64_C(  747.55), SIMDE_FLOAT64_C(  590.56),
                         SIMDE_FLOAT64_C(  522.53), SIMDE_FLOAT64_C(  340.37),
                         SIMDE_FLOAT64_C( -323.43), SIMDE_FLOAT64_C( -598.33)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -989.28), SIMDE_FLOAT64_C(   66.82),
                         SIMDE_FLOAT64_C(  747.55), SIMDE_FLOAT64_C(  590.56),
                         SIMDE_FLOAT64_C(  211.05), SIMDE_FLOAT64_C( -602.13),
                         SIMDE_FLOAT64_C( -323.43), SIMDE_FLOAT64_C( -598.33)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  435.94), SIMDE_FLOAT64_C( -117.09),
                         SIMDE_FLOAT64_C( -343.63), SIMDE_FLOAT64_C( -686.94),
                         SIMDE_FLOAT64_C( -632.13), SIMDE_FLOAT64_C(  520.11),
                         SIMDE_FLOAT64_C(  584.62), SIMDE_FLOAT64_C(  269.90)),
      UINT8_C(142),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -307.53), SIMDE_FLOAT64_C(  533.35),
                         SIMDE_FLOAT64_C( -283.32), SIMDE_FLOAT64_C(  860.26),
                         SIMDE_FLOAT64_C( -955.05), SIMDE_FLOAT64_C( -767.10),
                         SIMDE_FLOAT64_C( -553.49), SIMDE_FLOAT64_C(  540.17)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -307.53), SIMDE_FLOAT64_C( -117.09),
                         SIMDE_FLOAT64_C( -343.63), SIMDE_FLOAT64_C( -686.94),
                         SIMDE_FLOAT64_C( -955.05), SIMDE_FLOAT64_C( -767.10),
                         SIMDE_FLOAT64_C( -553.49), SIMDE_FLOAT64_C(  269.90)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  591.80), SIMDE_FLOAT64_C( -733.65),
                         SIMDE_FLOAT64_C(  371.96), SIMDE_FLOAT64_C( -998.26),
                         SIMDE_FLOAT64_C(   61.01), SIMDE_FLOAT64_C( -918.19),
                         SIMDE_FLOAT64_C( -797.48), SIMDE_FLOAT64_C(   81.07)),
      UINT8_C(155),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  378.17), SIMDE_FLOAT64_C(  574.36),
                         SIMDE_FLOAT64_C(  687.12), SIMDE_FLOAT64_C( -618.22),
                         SIMDE_FLOAT64_C(  388.77), SIMDE_FLOAT64_C( -731.92),
                         SIMDE_FLOAT64_C(  958.30), SIMDE_FLOAT64_C(   51.30)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  378.17), SIMDE_FLOAT64_C( -733.65),
                         SIMDE_FLOAT64_C(  371.96), SIMDE_FLOAT64_C( -618.22),
                         SIMDE_FLOAT64_C(  388.77), SIMDE_FLOAT64_C( -918.19),
                         SIMDE_FLOAT64_C(  958.30), SIMDE_FLOAT64_C(   51.30)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  721.16), SIMDE_FLOAT64_C(   21.28),
                         SIMDE_FLOAT64_C( -269.14), SIMDE_FLOAT64_C( -241.41),
                         SIMDE_FLOAT64_C( -307.10), SIMDE_FLOAT64_C(   78.73),
                         SIMDE_FLOAT64_C(  336.91), SIMDE_FLOAT64_C( -793.36)),
      UINT8_C(174),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  944.42), SIMDE_FLOAT64_C(  986.58),
                         SIMDE_FLOAT64_C( -765.43), SIMDE_FLOAT64_C(  392.41),
                         SIMDE_FLOAT64_C(  229.44), SIMDE_FLOAT64_C(   52.87),
                         SIMDE_FLOAT64_C( -238.79), SIMDE_FLOAT64_C(  440.21)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  944.42), SIMDE_FLOAT64_C(   21.28),
                         SIMDE_FLOAT64_C( -765.43), SIMDE_FLOAT64_C( -241.41),
                         SIMDE_FLOAT64_C(  229.44), SIMDE_FLOAT64_C(   52.87),
                         SIMDE_FLOAT64_C( -238.79), SIMDE_FLOAT64_C( -793.36)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  456.66), SIMDE_FLOAT64_C( -366.58),
                         SIMDE_FLOAT64_C(  715.22), SIMDE_FLOAT64_C(  -16.79),
                         SIMDE_FLOAT64_C( -320.68), SIMDE_FLOAT64_C(  273.81),
                         SIMDE_FLOAT64_C( -581.56), SIMDE_FLOAT64_C(  277.97)),
      UINT8_C(205),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    2.81), SIMDE_FLOAT64_C(  801.66),
                         SIMDE_FLOAT64_C(  310.16), SIMDE_FLOAT64_C(  634.68),
                         SIMDE_FLOAT64_C( -889.89), SIMDE_FLOAT64_C( -998.37),
                         SIMDE_FLOAT64_C( -493.27), SIMDE_FLOAT64_C(  120.40)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    2.81), SIMDE_FLOAT64_C(  801.66),
                         SIMDE_FLOAT64_C(  715.22), SIMDE_FLOAT64_C(  -16.79),
                         SIMDE_FLOAT64_C( -889.89), SIMDE_FLOAT64_C( -998.37),
                         SIMDE_FLOAT64_C( -581.56), SIMDE_FLOAT64_C(  120.40)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   44.26), SIMDE_FLOAT64_C(  891.25),
                         SIMDE_FLOAT64_C(  290.62), SIMDE_FLOAT64_C(  -70.18),
                         SIMDE_FLOAT64_C(   -3.54), SIMDE_FLOAT64_C(  783.54),
                         SIMDE_FLOAT64_C( -718.82), SIMDE_FLOAT64_C(  922.75)),
      UINT8_C( 72),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -286.94), SIMDE_FLOAT64_C( -573.68),
                         SIMDE_FLOAT64_C( -931.52), SIMDE_FLOAT64_C(  249.22),
                         SIMDE_FLOAT64_C(  735.88), SIMDE_FLOAT64_C(  653.72),
                         SIMDE_FLOAT64_C(  732.59), SIMDE_FLOAT64_C(  161.45)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   44.26), SIMDE_FLOAT64_C( -573.68),
                         SIMDE_FLOAT64_C(  290.62), SIMDE_FLOAT64_C(  -70.18),
                         SIMDE_FLOAT64_C(  735.88), SIMDE_FLOAT64_C(  783.54),
                         SIMDE_FLOAT64_C( -718.82), SIMDE_FLOAT64_C(  922.75)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  729.70), SIMDE_FLOAT64_C( -950.99),
                         SIMDE_FLOAT64_C(  115.61), SIMDE_FLOAT64_C( -132.19),
                         SIMDE_FLOAT64_C(  834.99), SIMDE_FLOAT64_C(  471.53),
                         SIMDE_FLOAT64_C(   54.12), SIMDE_FLOAT64_C(  238.73)),
      UINT8_C(209),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -345.93), SIMDE_FLOAT64_C(  598.65),
                         SIMDE_FLOAT64_C(  954.89), SIMDE_FLOAT64_C( -441.90),
                         SIMDE_FLOAT64_C(  845.52), SIMDE_FLOAT64_C( -659.44),
                         SIMDE_FLOAT64_C( -844.59), SIMDE_FLOAT64_C(  331.33)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -345.93), SIMDE_FLOAT64_C(  598.65),
                         SIMDE_FLOAT64_C(  115.61), SIMDE_FLOAT64_C( -441.90),
                         SIMDE_FLOAT64_C(  834.99), SIMDE_FLOAT64_C(  471.53),
                         SIMDE_FLOAT64_C(   54.12), SIMDE_FLOAT64_C(  331.33)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_mov_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return MUNIT_OK;
}

static MunitResult
test_simde_mm512_maskz_mov_epi32(const MunitParameter params[], void* data) {
  (void) params;
  (void) data;

  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(23562),
      simde_mm512_set_epi32(INT32_C(  413218138), INT32_C(-2056039012), INT32_C(  359898417), INT32_C(  503742711),
                            INT32_C( -964140572), INT32_C( 1845540628), INT32_C( 1555270769), INT32_C(  276306907),
                            INT32_C(  923961977), INT32_C( 2070870327), INT32_C( -106769082), INT32_C(   21505510),
                            INT32_C(-1894191102), INT32_C(  -61868066), INT32_C(-1022555483), INT32_C(  842262872)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(-2056039012), INT32_C(          0), INT32_C(  503742711),
                            INT32_C( -964140572), INT32_C( 1845540628), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(-1894191102), INT32_C(          0), INT32_C(-1022555483), INT32_C(          0)) },
    { UINT16_C(36203),
      simde_mm512_set_epi32(INT32_C(-1836353351), INT32_C(-1955161161), INT32_C( 1387065895), INT32_C(  829477081),
                            INT32_C( 1194773762), INT32_C( 1305535140), INT32_C(  692999175), INT32_C(-1162293370),
                            INT32_C( 1281198604), INT32_C( -270591140), INT32_C(   23870431), INT32_C(-1469107120),
                            INT32_C( 1859513610), INT32_C(-1425966851), INT32_C(  381161214), INT32_C(  706499700)),
      simde_mm512_set_epi32(INT32_C(-1836353351), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1194773762), INT32_C( 1305535140), INT32_C(          0), INT32_C(-1162293370),
                            INT32_C(          0), INT32_C( -270591140), INT32_C(   23870431), INT32_C(          0),
                            INT32_C( 1859513610), INT32_C(          0), INT32_C(  381161214), INT32_C(  706499700)) },
    { UINT16_C(61846),
      simde_mm512_set_epi32(INT32_C(-1731705333), INT32_C( 1975072423), INT32_C( -536413935), INT32_C( 1477835290),
                            INT32_C( 1453154713), INT32_C( -133697355), INT32_C( 1038848393), INT32_C(  897042603),
                            INT32_C( 1474696001), INT32_C( 1692444627), INT32_C(-1157569404), INT32_C(-1969459150),
                            INT32_C(  120064093), INT32_C(-1121934893), INT32_C( 1895180026), INT32_C( 1628067999)),
      simde_mm512_set_epi32(INT32_C(-1731705333), INT32_C( 1975072423), INT32_C( -536413935), INT32_C( 1477835290),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(  897042603),
                            INT32_C( 1474696001), INT32_C(          0), INT32_C(          0), INT32_C(-1969459150),
                            INT32_C(          0), INT32_C(-1121934893), INT32_C( 1895180026), INT32_C(          0)) },
    { UINT16_C( 9005),
      simde_mm512_set_epi32(INT32_C(  317112464), INT32_C(  741023218), INT32_C(-1717304973), INT32_C( 1768422162),
                            INT32_C(-1938535542), INT32_C( -593182598), INT32_C( -560734377), INT32_C(-1833964883),
                            INT32_C(-2069017846), INT32_C( 1509337971), INT32_C(-1663080670), INT32_C( -363349477),
                            INT32_C( -761414190), INT32_C( 1575734613), INT32_C(  758160476), INT32_C(  434110055)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(-1717304973), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( -560734377), INT32_C(-1833964883),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1663080670), INT32_C(          0),
                            INT32_C( -761414190), INT32_C( 1575734613), INT32_C(          0), INT32_C(  434110055)) },
    { UINT16_C(16381),
      simde_mm512_set_epi32(INT32_C(-1241873035), INT32_C(-1720080742), INT32_C( 1575508697), INT32_C(  644418481),
                            INT32_C( -191348066), INT32_C( 1363259829), INT32_C( -969945370), INT32_C(-1662256156),
                            INT32_C( -483657475), INT32_C( 1693775573), INT32_C( -588936550), INT32_C( -831491481),
              