/*--------------------------------------------------------------------*//*:Ignore this sentence.
Copyright (C) 1999, 2001 SIL International. All rights reserved.

Distributable under the terms of either the Common Public License or the
GNU Lesser General Public License, as specified in the LICENSING.txt file.

File: FileInput.cpp
Responsibility: Sharon Correll
Last reviewed: Not yet.

Description:
    Contains the functions for reading from a TT file.  (These are functions, not methods
	associated with a class.)
----------------------------------------------------------------------------------------------*/

//:>********************************************************************************************
//:>	Include files
//:>********************************************************************************************
#include "Main.h"
#ifndef _MSC_VER
#include "config.h"
#endif

#pragma hdrstop
#undef THIS_FILE
DEFINE_THIS_FILE

//:End Ignore

//:>********************************************************************************************
//:>	Forward declarations
//:>********************************************************************************************

//:>********************************************************************************************
//:>	Local Constants and static variables
//:>********************************************************************************************

namespace gr
{

//:>********************************************************************************************
//:>	Methods of GrFileIStream
//:>********************************************************************************************

/*----------------------------------------------------------------------------------------------
	Constructor.
----------------------------------------------------------------------------------------------*/
GrFileIStream::GrFileIStream()
{
}

/*----------------------------------------------------------------------------------------------
	Destructor.
----------------------------------------------------------------------------------------------*/
GrFileIStream::~GrFileIStream()
{
	Close();
}

/*----------------------------------------------------------------------------------------------
	Initialize the stream.
----------------------------------------------------------------------------------------------*/
#ifdef GR_FW
bool GrFileIStream::Open(std::wstring stuFileName, std::ios::openmode kMode)
#else
bool GrFileIStream::Open(const char *pcFileName, std::ios::openmode kMode)
#endif
{
#ifdef GR_FW
	char rgchs[256];
	size_t cchs = 256;
	char * pchs = rgchs;
	char * prgchsBig = NULL;
	if (stuFileName.size() > cchs/2)
	{
		prgchsBig = new char[stuFileName.size() + 1];
		pchs = prgchsBig;
		cchs = stuFileName.size();
	}
	size_t cchw = Platform_UnicodeToANSI(stuFileName.data(), stuFileName.size(), pchs, cchs);
	*(pchs + cchw) = 0;
	open(pchs, kMode | std::ios::in | std::ios::binary);
	bool fRet = !fail();
	delete[] prgchsBig;
	return fRet;
#else
	open(pcFileName, kMode | std::ios::in | std::ios::binary);
	bool fRet = !fail();
	return fRet;
#endif
}

/*----------------------------------------------------------------------------------------------
	Close the stream.
----------------------------------------------------------------------------------------------*/
void GrFileIStream::Close()
{
	close();
}

/*----------------------------------------------------------------------------------------------
	Read a byte from the stream.
----------------------------------------------------------------------------------------------*/
byte GrFileIStream::ReadByteFromFont()
{
	long lPosPre = tellg();
	byte bInput;
	read((char *)&bInput, isizeof(byte));
	if ((unsigned long)tellg() != lPosPre + isizeof(byte))
		THROW(kresReadFault);
	return bInput;
}

/*----------------------------------------------------------------------------------------------
	Read a short (signed 16-bit) word from the stream. Switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
short GrFileIStream::ReadShortFromFont()
{
	long lPosPre = tellg();
	utf16 chwInput;
	read((char *)&chwInput, isizeof(short));
	short snInput = lsbf(chwInput);
	if ((unsigned long)tellg() != lPosPre + isizeof(short))
		THROW(kresReadFault);
	return snInput;
}

/*----------------------------------------------------------------------------------------------
	Read a wide character (unsigned 16-bit word) from the stream.
	Switch the bytes from big-endian to little-endian format.
----------------------------------------------------------------------------------------------*/
utf16 GrFileIStream::ReadUShortFromFont()
{
	long lPosPre = tellg();
	utf16 chwInput;
	read((char *)&chwInput, isizeof(short));
	chwInput = lsbf(chwInput);
	if ((unsigned long)tellg() != lPosPre + isizeof(short))
		THROW(kresReadFault);
	return chwInput;
}

/*----------------------------------------------------------------------------------------------
	Read a standard (32-bit) word from the stream. Switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
int GrFileIStream::ReadIntFromFont()
{
	long lPosPre = tellg();
	int nInput;
	read((char *)&nInput, isizeof(int));
	nInput = lsbf(nInput);
	if ((unsigned long)tellg() != lPosPre + isizeof(int))
		THROW(kresReadFault);
	return nInput;
}

/*----------------------------------------------------------------------------------------------
	Read a block of data from the stream. DON'T switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
void GrFileIStream::ReadBlockFromFont(void * pvInput, int cb)
{
	long lPosPre = tellg();
	read((char *)pvInput, cb);
	if ((long)tellg() != lPosPre + cb)
		THROW(kresReadFault);
}

/*----------------------------------------------------------------------------------------------
	Get the absolute position of the font-file stream (relative to the beginning of
	the file).
----------------------------------------------------------------------------------------------*/
void GrFileIStream::GetPositionInFont(long * plPos)
{
	std::streampos strmp = tellg();
	*plPos = (int)strmp;
}

/*----------------------------------------------------------------------------------------------
	Set the position of the font-file stream to the given absolute position (relative
	to the beginning of the file).
----------------------------------------------------------------------------------------------*/
void GrFileIStream::SetPositionInFont(long lPos)
{
	seekg(lPos);
	if ((long)tellg() != lPos)
		THROW(kresReadFault);
}


//:>********************************************************************************************
//:>	Methods of GrBufferIStream
//:>********************************************************************************************

/*----------------------------------------------------------------------------------------------
	Constructor.namespace gr
{


----------------------------------------------------------------------------------------------*/
GrBufferIStream::GrBufferIStream()
{
	m_pbStart = NULL;
	m_pbNext = NULL;
	m_pbLim = NULL;
}

/*----------------------------------------------------------------------------------------------
	Destructor.
----------------------------------------------------------------------------------------------*/
GrBufferIStream::~GrBufferIStream()
{
	Close();
}

/*----------------------------------------------------------------------------------------------
	Initialize the stream.
----------------------------------------------------------------------------------------------*/
#ifdef GR_FW
bool GrBufferIStream::Open(std::wstring stuFileName, int kMode)
#else
bool GrBufferIStream::Open(const char * pcFileName, std::ios::openmode kMode)
#endif
{
	Assert(false); // use OpenBuffer
	return false;
}

/*----------------------------------------------------------------------------------------------
	Initialize the stream to a buffer.
----------------------------------------------------------------------------------------------*/
bool GrBufferIStream::OpenBuffer(byte * pbBuffer, int cb)
{
	Assert(m_pbStart == NULL);
	Assert(m_pbNext == NULL);
	Assert(m_pbLim == NULL);

	m_pbStart = pbBuffer;
	m_pbNext = pbBuffer;
	if (cb >= 0)
		m_pbLim = m_pbStart + cb;
	// otherwise we don't know the length

	return true;
}

/*----------------------------------------------------------------------------------------------
	Close the stream.
----------------------------------------------------------------------------------------------*/
void GrBufferIStream::Close()
{
	m_pbStart = NULL;
	m_pbNext = NULL;
	m_pbLim = NULL;
}

/*----------------------------------------------------------------------------------------------
	Read a byte from the stream.
----------------------------------------------------------------------------------------------*/
byte GrBufferIStream::ReadByteFromFont()
{
	byte bInput = *m_pbNext;
	m_pbNext += isizeof(byte);
	if (m_pbLim && m_pbNext > m_pbLim)
		THROW(kresReadFault);
	return bInput;
}

/*----------------------------------------------------------------------------------------------
	Read a short (signed 16-bit) word from the stream. Switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
short GrBufferIStream::ReadShortFromFont()
{
	short snInput = *(short *)m_pbNext;
	m_pbNext += isizeof(short);
	if (m_pbLim && m_pbNext > m_pbLim)
		THROW(kresReadFault);
	snInput = lsbf(snInput);
	return snInput;
}

/*----------------------------------------------------------------------------------------------
	Read a wide character (unsigned 16-bit word) from the stream.
	Switch the bytes from big-endian to little-endian format.
----------------------------------------------------------------------------------------------*/
utf16 GrBufferIStream::ReadUShortFromFont()
{
	utf16 chwInput = *(utf16 *)m_pbNext;
	m_pbNext += isizeof(utf16);
	if (m_pbLim && m_pbNext > m_pbLim)
		THROW(kresReadFault);
	chwInput = lsbf(chwInput);
	return chwInput;
}

/*----------------------------------------------------------------------------------------------
	Read a standard (32-bit) word from the stream. Switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
int GrBufferIStream::ReadIntFromFont()
{
	int nInput = *(int *)m_pbNext;
	m_pbNext += isizeof(int);
	if (m_pbLim && m_pbNext > m_pbLim)
		THROW(kresReadFault);
	nInput = lsbf(nInput);
	return nInput;
}

/*----------------------------------------------------------------------------------------------
	Read a block of data from the stream. DON'T switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
void GrBufferIStream::ReadBlockFromFont(void * pvInput, int cb)
{
	memcpy(pvInput, m_pbNext, cb);
	m_pbNext += cb;
	if (m_pbLim && m_pbNext > m_pbLim)
		THROW(kresReadFault);
}

/*----------------------------------------------------------------------------------------------
	Get the absolute position of the font-file stream (relative to the beginning of
	the file). For buffers, we just return the byte position in the buffer.
----------------------------------------------------------------------------------------------*/
void GrBufferIStream::GetPositionInFont(long * plPos)
{
	*plPos = (m_pbNext - m_pbStart);
}

/*----------------------------------------------------------------------------------------------
	Set the position of the font-file stream to the given absolute position (relative
	to the beginning of the file). For buffers, assume the position is relative to the
	beginning of the buffer.
----------------------------------------------------------------------------------------------*/
void GrBufferIStream::SetPositionInFont(long lPos)
{
	m_pbNext = m_pbStart + lPos;
	if (m_pbLim && m_pbNext > m_pbLim)
		THROW(kresReadFault);
}


//:>********************************************************************************************
//:>	Swap byte order.
//:>********************************************************************************************
int swapb(int nArg)
{
#if WORDS_BIGENDIAN
return nArg;
#else
	int b1, b2, b3, b4;
	b1 = ((nArg & 0xFF000000) >> 24)     & 0x000000FF;	// remove sign extension
	b2 = ((nArg & 0x00FF0000) >> 8);  // & 0x0000FF00;
	b3 = ((nArg & 0x0000FF00) << 8);  // & 0x00FF0000;
	b4 = ((nArg & 0x000000FF) << 24); // & 0xFF000000;
	int nRet = b1 | b2 | b3 | b4;
	return nRet;
#endif
}

utf16 swapb(utf16 chwArg)
{
#if WORDS_BIGENDIAN
return chwArg;
#else
	utf16 b1, b2;
	b1 = ((chwArg & 0xFF00) >> 8)     & 0x00FF;		// remove sign extension
	b2 = ((chwArg & 0x00FF) << 8); // & 0xFF00;
	utf16 chwRet = b1 | b2;
	return chwRet;
#endif
}

short swapb(short snArg)
{
#if WORDS_BIGENDIAN
return snArg;
#else
	short b1, b2;
	b1 = ((snArg & 0xFF00) >> 8)     & 0x00FF;		// remove sign extension
	b2 = ((snArg & 0x00FF) << 8); // & 0xFF00;
	short snRet = b1 | b2;
	return snRet;
#endif
}

} //namespace gr



//:>********************************************************************************************
//:>	Methods of GrComFileIStream - not used
//:>********************************************************************************************

#if 0

/*----------------------------------------------------------------------------------------------
	Constructor.
----------------------------------------------------------------------------------------------*/
GrComFileIStream::GrComFileIStream()
{
}

/*----------------------------------------------------------------------------------------------
	Destructor.
----------------------------------------------------------------------------------------------*/
GrComFileIStream::~GrComFileIStream()
{
	Close();
	m_qstrm = NULL;
}

/*----------------------------------------------------------------------------------------------
	Initialize the stream.
----------------------------------------------------------------------------------------------*/
bool GrComFileIStream::Open(std::wstring stuFileName, int kMode)
{
	try 
	{
		FileStream::Create(strFileName.c_str(), kMode | STGM_READ, &m_qstrm);
	}
	catch (...)
	{
		return false;
	}
	return true;
}

/*----------------------------------------------------------------------------------------------
	Close the stream.
----------------------------------------------------------------------------------------------*/
void GrComFileIStream::Close()
{
	// There is no close method on IFileStream. The destructor takes care of closing the stream.
}

/*----------------------------------------------------------------------------------------------
	Read a byte from the stream.
----------------------------------------------------------------------------------------------*/
byte GrComFileIStream::ReadByteFromFont()
{
	int bInput;
	ULONG cbRead;
	HRESULT hr = m_qstrm->Read(&bInput, 1, &cbRead);
	if (FAILED(hr))
		ThrowHr(WarnHr(hr));
	else if (cbRead != 1)
		ThrowHr(WarnHr(STG_E_READFAULT));
	
	return bInput;
}

/*----------------------------------------------------------------------------------------------
	Read a short (signed 16-bit) word from the stream. Switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
short GrComFileIStream::ReadShortFromFont()
{
	utf16 chwInput;
	ULONG cbRead;
	HRESULT hr = m_qstrm->Read(&chwInput, 2, &cbRead);
	if (FAILED(hr))
		ThrowHr(WarnHr(hr));
	else if (cbRead != 2)
		ThrowHr(WarnHr(STG_E_READFAULT));
	short snInput = lsbf(chwInput);
	return snInput;
}

/*----------------------------------------------------------------------------------------------
	Read a wide character (unsigned 16-bit word) from the stream.
	Switch the bytes from big-endian to little-endian format.
----------------------------------------------------------------------------------------------*/
utf16 GrComFileIStream::ReadUShortFromFont()
{
	utf16 chwInput;
	ULONG cbRead;
	HRESULT hr = m_qstrm->Read(&chwInput, 2, &cbRead);
	if (FAILED(hr))
		ThrowHr(WarnHr(hr));
	else if (cbRead != 2)
		ThrowHr(WarnHr(STG_E_READFAULT));
	chwInput = lsbf(chwInput);
	return chwInput;
}

/*----------------------------------------------------------------------------------------------
	Read a standard (32-bit) word from the stream. Switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
int GrComFileIStream::ReadIntFromFont()
{
	int nInput;
	ULONG cbRead;
	HRESULT hr = m_qstrm->Read(&nInput, 4, &cbRead);
	if (FAILED(hr))
		ThrowHr(WarnHr(hr));
	else if (cbRead != 4)
		ThrowHr(WarnHr(STG_E_READFAULT));
	nInput = lsbf(nInput);
	return nInput;
}

/*----------------------------------------------------------------------------------------------
	Read a block of data from the stream. DON'T switch the bytes from big-endian 
	to little-endian format.
----------------------------------------------------------------------------------------------*/
void GrComFileIStream::ReadBlockFromFont(void * pvInput, int cb)
{
	ULONG cbRead;
	HRESULT hr = m_qstrm->Read(pvInput, cb, &cbRead);
	if (FAILED(hr))
		ThrowHr(WarnHr(hr));
	else if ((ULONG)cb != cbRead)
		ThrowHr(WarnHr(STG_E_READFAULT));
}

/*----------------------------------------------------------------------------------------------
	Get the absolute position of the font-file stream (relative to the beginning of
	the file).
----------------------------------------------------------------------------------------------*/
void GrComFileIStream::GetPositionInFont(long * plPos)
{
	LARGE_INTEGER libSeekPos;
	ULARGE_INTEGER libNewPos;
	libSeekPos.QuadPart = 0;
	HRESULT hr = m_qstrm->Seek(libSeekPos, STREAM_SEEK_CUR, &libNewPos);
	if (FAILED(hr))
		ThrowHr(WarnHr(hr));
	*plPos = (long)libNewPos.QuadPart;
}

/*----------------------------------------------------------------------------------------------
	Set the position of the font-file stream to the given absolute position (relative
	to the beginning of the file).
----------------------------------------------------------------------------------------------*/
void GrComFileIStream::SetPositionInFont(long lPos)
{
	LARGE_INTEGER libSeekPos;
	ULARGE_INTEGER libNewPos;
	libSeekPos.QuadPart = lPos;
	HRESULT hr = m_qstrm->Seek(libSeekPos, STREAM_SEEK_SET, &libNewPos);
	if (FAILED(hr))
		ThrowHr(WarnHr(hr));
}

#endif // 0
