/*--------------------------------------------------------------------*//*:Ignore this sentence.
Copyright (C) 1999, 2001 SIL International. All rights reserved.

Distributable under the terms of either the Common Public License or the
GNU Lesser General Public License, as specified in the LICENSING.txt file.

File: Font.h
Responsibility: Sharon Correll
Last reviewed: Not yet.

Description:
    A Font is an object that represents a font-family + bold + italic setting, that contains
	Graphite tables.
----------------------------------------------------------------------------------------------*/
#ifdef _MSC_VER
#pragma once
#endif
#ifndef FONT_INCLUDED
#define FONT_INCLUDED

//:End Ignore

namespace gr
{

class Font;
class FontFace;
class FeatureSettingIterator;
class Segment;
class LayoutEnvironment;
class ITextSource;

typedef struct tagFontProps
{
	unsigned long clrFore;
	unsigned long clrBack;
	bool fBold;
	bool fItalic;
	float pixHeight;
	wchar_t szFaceName[ 32 ];
} FontProps;


/*----------------------------------------------------------------------------------------------
	A FontError object is an exception that is thrown when there is an error in initializing
	a Graphite font.
----------------------------------------------------------------------------------------------*/
struct FontException
{
	FontErrorCode errorCode;
	int version;
	int subVersion;
};

/*----------------------------------------------------------------------------------------------
	Iterator to provide access to a font's features.
----------------------------------------------------------------------------------------------*/
class FeatureIterator
{
	friend class Font;
	friend class FeatureSettingIterator;

public:
	FeatureIterator() // needed for creating std::pair of these, and for default FSI
	{
		m_pfont = NULL;
		m_ifeat = 0;
		m_cfeat = 0;
	}

protected:
	FeatureIterator(Font * pfont, int ifeat, size_t cfeat)
	{
		m_pfont = pfont;
		m_ifeat = ifeat;
		m_cfeat = cfeat;
	}
public:
	featid operator*();
	FeatureIterator operator++();
	FeatureIterator operator+=(int n);
	bool operator==(FeatureIterator &);
	bool operator!=(FeatureIterator &);
	int operator-(FeatureIterator &);

protected:
	Font * m_pfont;
	size_t m_ifeat;	// feature being pointed at
	size_t m_cfeat;	// number of features for this font

	FeatureSettingIterator BeginSetting();
	FeatureSettingIterator EndSetting();
};

/*----------------------------------------------------------------------------------------------
	Iterator to provide access to the defined values for a single font feature.
----------------------------------------------------------------------------------------------*/
class FeatureSettingIterator
{
	friend class Font;
	friend class FeatureIterator;

public:
	FeatureSettingIterator() // needed for creating std::pair of these, I think
	{
		m_ifset = 0;
		m_cfset = 0;
	}
protected:
	FeatureSettingIterator(FeatureIterator fit, int ifset, size_t cfset)
	{
		m_fit = fit;
		m_ifset = ifset;
		m_cfset = cfset;
	}
public:
	int operator*();
	FeatureSettingIterator operator++();
	FeatureSettingIterator operator +=(int n);
	bool operator==(FeatureSettingIterator &);
	bool operator!=(FeatureSettingIterator &);
	int operator-(FeatureSettingIterator);

protected:
	FeatureIterator m_fit;
	size_t m_ifset;	// which setting being pointed at
	size_t m_cfset;	// number of settings for this feature
};

/*----------------------------------------------------------------------------------------------
	Abstract superclass for Graphite fonts. A font represents a face name, size, and bold and
	italic styles.
----------------------------------------------------------------------------------------------*/
class Font {

	friend class FeatureIterator;
	friend class FeatureSettingIterator;

public:
	Font()
	{ }
	virtual ~Font();

	virtual Font * copyThis() = 0;

	virtual float ascent() = 0;
	virtual float descent() = 0;
	virtual bool bold() = 0;
	virtual bool italic() = 0;
	virtual float height() = 0;
	virtual float fakeItalicRatio()
	{
		return 0; // no support for fake italic
	}

	virtual void * getTable(fontTableId32 tableID, size_t * pcbSize) = 0;
	virtual unsigned int getDPIx() = 0;
	virtual unsigned int getDPIy() = 0;
	virtual void getFontMetrics(float * pAscent, float * pDescent = NULL,
		float * pEmSquare = NULL) = 0;
	virtual void getGlyphPoint(gid16 glyphID, unsigned int pointNum, gr::Point & pointReturn) = 0;
	virtual void getGlyphMetrics(gid16 glyphID, gr::Rect & boundingBox, gr::Point & advances) = 0;

	static void SetFlushMode(int);
	static int GetFlushMode();

	// obsolete:
	//virtual FontErrorCode isValidForGraphite(int * pnVersion = NULL, int * pnSubVersion = NULL) = 0;

	// Features:
	std::pair<FeatureIterator, FeatureIterator> getFeatures();
	FeatureIterator featureWithID(featid id);
	bool getFeatureLabel(FeatureIterator, lgid language, utf16 * label);
	FeatureSettingIterator getDefaultFeatureValue(FeatureIterator);
	std::pair<FeatureSettingIterator, FeatureSettingIterator> getFeatureSettings(FeatureIterator);
	bool getFeatureSettingLabel(FeatureSettingIterator, lgid language, utf16 * label);

public:
	// For use in segment creation:
	void RenderLineFillSegment(Segment * pseg, ITextSource * pts, LayoutEnvironment & layout,
		toffset ichStart, toffset ichStop, float xsMaxWidth, bool fBacktracking);
	void RenderRangeSegment(Segment * pseg, ITextSource * pts, LayoutEnvironment & layout,
		toffset ichStart, toffset ichEnd);
	void RenderJustifiedSegment(Segment * pseg, ITextSource * pts, LayoutEnvironment & layout,
		toffset ichStart, toffset ichEnd, float xsCurrentWidth, float xsDesiredWidth);

protected:
	FontFace * m_pfface;	// set up with Graphite tables

	// Feature access:
	FeatureIterator BeginFeature();
	FeatureIterator EndFeature();

	size_t NumberOfFeatures();
	featid FeatureID(size_t ifeat);
	size_t FeatureWithID(featid id);
	bool GetFeatureLabel(size_t ifeat, lgid language, utf16 * label);
	int GetFeatureDefault(size_t ifeat);
	size_t NumberOfSettings(size_t ifeat);
	int GetFeatureSettingValue(size_t ifeat, size_t ifset);
	bool GetFeatureSettingLabel(size_t ifeat, size_t ifset, lgid language, utf16 * label);

	void UniqueCacheInfo(std::wstring & stuFace, bool & fBold, bool & fItalic);
	static bool FontHasGraphiteTables(Font & font);
};

} // namespace gr


#endif // !FONT_INCLUDED
