/* $Id: tftp_fileops.c,v 1.5 2003/05/30 00:42:38 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "getfile.h"
#include "dns.h"
#include "tftp.h"
#include "libc.h"

typedef struct {
	char     *filename;
	uint32_t  server_ip;
	int32_t   offset;
	uint16_t  server_port;
} tftp_ops_data_t;


/*
 * Expects a string in this format:
 * 
 *  tftp:<server name or ip>:filename\0
 * 
 */
static void *tftp_op_open(char *loc)
{
	char delims[] = ": \t";
	tftp_ops_data_t *ret;
	char *tok;
	if (lib_strncmp(loc, "tftp:", 5)) {
		return 0;
	}
	loc += 5;
	
	ret = lib_malloc(sizeof(tftp_ops_data_t));
	if (!ret) {
		lib_die("Out of memory");
	}
	
	ret->offset = 0;
	
	tok = lib_strtok(loc, delims);
	if (!tok) {
		goto fail;
	}
	if (lib_isdigit(*tok)) {
		/* Consider this to be an ip */
		ret->server_ip = lib_parse_ip(tok);
	} else {
		ret->server_ip = dns_resolve_name(tok);
	}
	if (!ret->server_ip) {
		lib_printf("Couldn't resolve %s as a server identifier\n", tok);
		goto fail;
	} 	
	tok = lib_strtok(0, delims);
	if (!tok) {
		goto fail;
	}
	ret->filename = lib_strdup(tok);
	if (!ret->filename) {
		lib_die("Out of memory");
	}
	ret->server_port = TFTP_PORT;
	goto success;
 fail:
	lib_free(ret);
	ret = 0;
 success:
	return ret;
}

static int32_t tftp_op_seek(void *private, int32_t ofs, file_handler_whence_t whence)
{
	tftp_ops_data_t *data = (tftp_ops_data_t *)private;
	switch (whence) {
	case GF_SEEK_SET:
		data->offset = ofs;
		break;
	case GF_SEEK_CUR:
		data->offset += ofs;
		break;
	default:
		lib_die("Out of memory");
	}
	return data->offset;
}

static int32_t tftp_op_read(void *private, void *buf, int32_t amount)
{
	int32_t retval;
	tftp_ops_data_t *data = (tftp_ops_data_t *)private;
	retval = tftp_get_chunk(data->server_ip, data->server_port, data->filename, buf, amount, data->offset);
	if (retval > 0) {
		data->offset += retval;
	}
	return retval;
}

static void tftp_op_close(void *private)
{
	tftp_ops_data_t *data = (tftp_ops_data_t *)private;
	if (data) {
		lib_free(data->filename);
		lib_free(data);
	}
}


file_ops_t tftp_ops = { 
	tftp_op_open,
	tftp_op_seek,
	tftp_op_read,
	tftp_op_close
};
