/* $Id: tftp.c,v 1.8 2003/05/30 00:42:40 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
#include "tftp.h"
#include "mbuf.h"
#include "dns.h"
#include "udp.h"
#include "libc.h"
#include "misc.h"
#include "global.h"
#include "ip.h"
#include "eth.h"


/* 
 * Data structures
 */

#define TFTP_OP_RRQ      1
#define TFTP_OP_WRQ      2
#define TFTP_OP_DATA     3
#define TFTP_OP_ACK      4
#define TFTP_OP_ERR      5

#define TFTP_TIMEOUT 3000000

typedef struct tftp_transfer_node_s {
	unsigned int done_flag;
	unsigned int err_flag;
	unsigned int eof;
	unsigned int last_block_num;
	uint32_t server_ip;
	uint32_t src_ip;
	uint32_t offset;
	uint32_t amount;
	uint32_t transferred;
	uint16_t server_port;
	uint16_t src_port;
	uint8_t *buf;
	char *filename;
	mbuf_t *mbuf;
	struct tftp_transfer_node_s *next;
} tftp_transfer_t;


static tftp_transfer_t *transfer = 0;


/* This number has no special meaning */
static uint16_t outgoing_port = 1583;


/* 
 * Static functions
 */



/* 
 * Found an error.  Bail from the reception 
 */
static void tftp_error(tftp_transfer_t *data, char *msg)
{
	data->done_flag = 1;
	data->err_flag = 1;
	lib_printf("TFTP: %s\n", msg);
}

static void tftp_ack_block(tftp_transfer_t * data)
{
	uint8_t *ptr;
	mbuf_init(transfer->mbuf, 0);
	if (mbuf_append_space(transfer->mbuf, 4) != MBUF_SUCCESS) {
		lib_die("Out of memory");
	}
	ptr = transfer->mbuf->front_ptr;
	set_uint16_field(ptr, TFTP_OP_ACK);
	ptr += 2;
	set_uint16_field(ptr, data->last_block_num);
	udp_send_packet(transfer->mbuf, data->src_ip, data->server_ip, IP_FLAGS_NO_FRAGMENT, data->src_port, data->server_port);
}


/*
 * Handler registered to handle udp packets to the port used to make the tftp request.
 * 
 * This handler handles 2 types of tftp packets: data and errors.
 * 
 * For data packets, the data recieved is stashed into the buffer, and the
 * data is ack'd
 * 
 * Error packets or malformed data packets abort the transfer
 *
 * The handler makes no effort to validate that blocks are recieved sequentially, nor
 * that all packets are recieved. 
 * 
 * Returns 1 if the packet given is an error or data tftp packet, 0 otherwise
 */
static int tftp_handler(mbuf_t *mbuf, void *data_ptr, uint32_t src_ip, uint32_t dest_ip, uint16_t src_port, uint16_t dest_port)
{
	int retval = 0;
	uint8_t *ptr;
	uint16_t opcode;
	tftp_transfer_t *data = (tftp_transfer_t *)data_ptr;
	if (mbuf->size < 4) {
		goto out;
	}
	ptr = mbuf->front_ptr;
	opcode = get_uint16_field(ptr);
	ptr += 2;
	if (opcode == TFTP_OP_DATA) {
		uint16_t block_num;
		uint32_t block_ofs;
		int block_size;
		retval = 1;
		transfer->server_port = src_port;
		if (mbuf->size < 516) {
			data->done_flag = 1;
			data->eof = 1;
			data->err_flag = 0;
		} else if (mbuf->size > 516) {
			tftp_error(data, "Malformed packet");
			goto out;
		}
		block_num = get_uint16_field(ptr);
		ptr += 2;
		block_ofs = (block_num - 1) * 512;
		block_size = mbuf->end_ptr - ptr;
		
		if ((block_ofs + block_size) > data->offset) {
			/* 
			 * Painful.  This should handle all cases correctly:
			 * - block is entirely within the desired range
			 * - desired range is entirely within block
			 * - block contains the start of desired range
			 * - block contains the end of desired range
			 */
			
			unsigned int floor, dest_ofs = 0, src_ofs = 0, amount = block_size;
			
			if (block_ofs > data->offset) {
				dest_ofs = block_ofs - data->offset;
			}
			
			if (data->offset > block_ofs) {
				src_ofs = data->offset - block_ofs;
				floor = data->offset;
			} else {
				floor = block_ofs;
			}
			
			if ((block_ofs + block_size) > (data->offset + data->amount)) {
				amount = (data->offset + data->amount) - floor;
				data->done_flag = 1;
				data->err_flag = 0;
			}
			lib_memcpy(data->buf + dest_ofs, ptr + src_ofs, amount);
			data->transferred = dest_ofs + amount;
		}
		
		/* 
		 * If we're done with the transfer, act like we didn't get this
		 * block.  This will let us re-ack the last block to get this
		 * one resent if there's a need in the next transfer to re-read
		 * from this block.
		 */
		if (!data->done_flag) {
			data->last_block_num = block_num;
			tftp_ack_block(data);
			eth_reset_timeout(TFTP_TIMEOUT);
		} 
	} else if (opcode == TFTP_OP_ERR) {
		uint8_t *tmp;
		retval = 1;
		ptr += 2;
		
		/* Make sure the error message exists in
		   valid space */
		if (!mbuf_range_ok(mbuf, ptr, 0)) {
			goto out;
		}
		for (tmp = ptr; *tmp; tmp++) {
			if (tmp == mbuf->end_ptr) {
				tftp_error(data, "Malformed packet");
				goto out;
			}
		}
		
		tftp_error(data, ptr);
	}
 out:	       
	return retval;
}


static void tftp_open_connection(uint32_t server, uint16_t server_port, char *filename)
{
	uint8_t *ptr;
	int fn_len;
	mbuf_t *tmp;
	if (!transfer) {
		transfer = lib_malloc(sizeof(tftp_transfer_t));
		if (!transfer) {
			lib_die("Out of memory");
		}
		transfer->mbuf = lib_malloc(sizeof(mbuf_t));
		if (!transfer->mbuf) {
			lib_die("Out of memory");
		}
	} else {
		udp_del_handler(tftp_handler, transfer, transfer->server_ip, transfer->src_ip, 0, transfer->src_port);
		lib_free(transfer->filename);
	}

	tmp = transfer->mbuf;
	lib_bzero(transfer, sizeof(tftp_transfer_t));
	transfer->mbuf = tmp;

	transfer->server_ip = server;
	transfer->server_port = server_port;
	transfer->filename = lib_strdup(filename);
	transfer->src_port = outgoing_port;
	transfer->src_ip = net_cfg.ip_addr;
	
	outgoing_port++;
	mbuf_init(transfer->mbuf, 0);
	
	fn_len = lib_strlen(filename);
	
	if (mbuf_append_space(transfer->mbuf, 9 + fn_len) != MBUF_SUCCESS) {
		lib_die("Out of memory");
	}

	ptr = transfer->mbuf->front_ptr;
	set_uint16_field(ptr, TFTP_OP_RRQ);
	ptr += 2;
	lib_strcpy(ptr, filename);
	ptr += fn_len + 1;
	lib_strcpy(ptr, "octet");
	
	udp_add_handler(tftp_handler, transfer, transfer->server_ip, transfer->src_ip, 0, transfer->src_port);
	
	udp_send_packet(transfer->mbuf, net_cfg.ip_addr, transfer->server_ip, IP_FLAGS_NO_FRAGMENT, 
			transfer->src_port, transfer->server_port);
	
	/* That's it.  The server should start spewing packets our way soon...*/
}

/*
 * Exported interface.  See header for semantics 
 */
int32_t tftp_get_chunk(uint32_t server, uint16_t server_port, char *filename, uint8_t *buffer, unsigned int amount, unsigned int offset)
{
	int32_t retval;
	int reopens = 2;
	int retries = 2;
	/*
	 * Check if we're already transferring from this file in a way that we can reuse.  If not,
	 * open a new connection 
	 */
	if (!transfer
	    || (server != transfer->server_ip)
	    || lib_strcmp(filename, transfer->filename)
	    || (offset < 512)
	    || (transfer->last_block_num <= 1)
	    || (offset < (transfer->last_block_num - 1) * 512)) {
		/*
		 * Different connection, or, in the first block, 
		 * which makes it impossible to ack the block before this one,
		 * or in an area of the file we've already read and don't
		 * have buffered.
		 */
		tftp_open_connection(server, server_port, filename);
	} else {
		/*
		 * If we're trying to go to the same block as we last
		 * recieved or on to something beyond the current block,
		 * re-ack the last block
		 */
		tftp_ack_block(transfer);
	}
	
	transfer->buf = buffer;
	transfer->done_flag = 0;
	transfer->err_flag = 0;
	transfer->offset = offset;
	transfer->amount = amount;
	transfer->transferred = 0;
	
	
	do {
		if (!retries) {
			retries = 2;
			lib_printf("TFTP: Reopening connection\n");
			tftp_open_connection(server, server_port, filename);
		}
		do {
			eth_recv_loop(TFTP_TIMEOUT, &(transfer->done_flag));
			if (transfer->done_flag) {
				retries = 0;
				reopens = 0;
			} else if (transfer->last_block_num) {
				tftp_ack_block(transfer);
			} else {
				/* If we haven't gotten anything yet, just
				   reopen */
				retries = 0;
			}
		} while (retries--);
	} while (reopens--);
	
	if (!transfer->done_flag) {
		lib_printf("TFTP: Transfer timed out\n");
		retval = -1;
	} else if (transfer->err_flag) {
		retval = -1;
	} else {
		retval = transfer->transferred;
	}
	
	/* 
	 *  If we finished the file, free up resources
	 */
	if (transfer->eof) {
		udp_del_handler(tftp_handler, transfer, transfer->server_ip, transfer->src_ip, 0, transfer->src_port);
		lib_free(transfer->mbuf);
		lib_free(transfer->filename);
		lib_free(transfer);
		transfer = 0;
	}
	
	return retval;
}
