/* qshutdown, a program to shutdown the shutdown/reboot/suspend/hibernate
 * Copyright (C) 2009 2010 2011 Christian Metscher <hakaishi@web.de>

 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gui.h"
#include "power.h"
#include "preferences.h"
#include "info.h"
#include "calendar.h"
#include "passwd.h"
#include "editor.h"

bool timeRunning;
bool lockMyScreen;
extern QString fonts; //variable declared in preferences.cpp
extern int fontS1;    // ""
extern int fontS2;    // ""
extern int fontS3;    // ""
extern QDate setCalendarDate; //variable declared in calendar.cpp
bool lockAll = false;

Gui::Gui(QWidget *parent): QWidget(parent){

	setupUi(this);

     finishedExecuted = false;
     timeRunning = false;
     logFileSize = 1.5;

     datetime = QDateTime::currentDateTime();
     elapsedTime.start();

     setWindowFlags(Qt::WindowStaysOnTopHint);

   //Preferences
     pref = new Preferences(this);

   //CalendarBox
     calendarBox = new Calendar(this);

   //InfoMessageBox
     infoBox = new Info(this);

   //ChangePassword
     checkPassword = new PassWord(this);

   //Editor
     editor = new Editor(this);

   //Changing SizePolicy for Spacers
     horizontalSpacer1->changeSize(40,45,QSizePolicy::Expanding,QSizePolicy::Expanding);
     horizontalSpacer3->changeSize(85,13,QSizePolicy::Maximum,QSizePolicy::Expanding);
     horizontalSpacer4->changeSize(85,13,QSizePolicy::Maximum,QSizePolicy::Expanding);
     horizontalSpacer5->changeSize(85,13,QSizePolicy::Maximum,QSizePolicy::Expanding);

   //Versioning
     QFile versionFile(":version");
     versionFile.open(QIODevice::ReadOnly | QIODevice::Text);
     QTextStream in(&versionFile);
     QString string = in.readLine();
     version->setText(string);
     versionFile.close();

   //TrayIcon
     icon = QIcon(":red_glasses");
     TIcon = new QSystemTrayIcon(this);
     TIcon->setIcon(icon);
     TIcon->show();

   //TrayIconContextMenu
     menu = new QMenu(this);
     reset_action = new QAction(tr("R&eset \t Ctrl+E"),this);
     quit_action = new QAction(tr("&Quit \t Ctrl+Q"),this);
     info_action = new QAction(tr("&Info \t Ctrl+I"),this);
     log_action = new QAction(tr("&Logfile \t Ctrl+L"),this);
     log_action->setCheckable(true);
     pref_action = new QAction(tr("&Preferences \t Ctrl+P"),this);

     power_actions = new QActionGroup(this);
     shutdown_action = new QAction(tr("&Shutdown \t Ctrl+S"),this);
     shutdown_action->setCheckable(true);
     reboot_action = new QAction(tr("&Reboot \t Ctrl+R"),this);
     reboot_action->setCheckable(true);
     suspend_action = new QAction(tr("S&uspend \t Ctrl+U"),this);
     suspend_action->setCheckable(true);
     hibernate_action = new QAction(tr("&Hibernate \t Ctrl+H"),this);
     hibernate_action->setCheckable(true);
     power_actions->addAction(shutdown_action);
     power_actions->addAction(reboot_action);
     power_actions->addAction(suspend_action);
     power_actions->addAction(hibernate_action);

     menu->addAction(info_action);
     menu->addAction(reset_action);
     menu->addAction(pref_action);
     menu->addAction(log_action);
     menu->addSeparator();
     menu->addAction(shutdown_action);
     menu->addAction(reboot_action);
     menu->addAction(suspend_action);
     menu->addAction(hibernate_action);
     reset_action->setVisible(false);
     menu->addSeparator();
     menu->addAction(quit_action);
     
     TIcon->setContextMenu(menu);  //setting contextmenu for the systray

   //PushButton Minimize
     minim = new QPushButton(this); //to minimize
     minim->setText(tr("&Minimize!"));
     minim->setHidden(true); //not yet visible
     minim->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
     gridLayout5->addWidget(minim,0,1,3,1);

   //LCD-Anzeige
     lcd = new QLCDNumber(4, this); //display for countdown
     lcd->setSegmentStyle(QLCDNumber::Flat);
     lcd->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
     gridLayout4->addWidget(lcd,0,1,1,1);

   //Fonts
     font1 = new QFont;
     font1->setBold(true);

     font2 = new QFont;
     font2->setBold(true);

     font3 = new QFont;

   //Timer
     timer = new QTimer(this); //for updating countdown display and check if time is over
     ti = new QTimer(this); //for displaying the window, till the OK-Button is clicked
     ti->start(30000); //gives every 30 seconds a timeout()-signal

   //Connect signals with slots (actions with funktions)
     connect(radio1, SIGNAL(toggled(bool)), timeEdit, SLOT(setEnabled(bool)));
     connect(radio2, SIGNAL(toggled(bool)), spin, SLOT(setEnabled(bool)));
     connect(comboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(power_action(int)));
     connect(power_actions, SIGNAL(triggered(QAction*)), this, SLOT(power(QAction*)));
     connect(ti, SIGNAL(timeout()), this, SLOT(showW())); //set window state for ti
     connect(ok, SIGNAL(clicked(bool)), this, SLOT(set())); //starts timers and time calculations
     connect(timer, SIGNAL(timeout()), this, SLOT(updateT()));    //runs updateT() every 10 seconds
     connect(now, SIGNAL(clicked(bool)), this, SLOT(finished()));//closes the program & starts the shutdown immediately
     connect(minim, SIGNAL(clicked(bool)), this, SLOT(showMinimized())); //minimizes window
     connect(TIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
       this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason))); //handles systray-symbol
     connect(pref_action, SIGNAL(triggered()), pref, SLOT(show()));
     connect(pref, SIGNAL(starting()), ti, SLOT(stop()));
     connect(pref, SIGNAL(finishing()), ti, SLOT(start())); //start the Qtimer ti if timeRunning == false
     connect(pref, SIGNAL(changeFont()), this, SLOT(getFonts()));
     connect(reset_action, SIGNAL(triggered()), this, SLOT(reset()));
     connect(info_action, SIGNAL(triggered()), this, SLOT(setInfoText()));
     connect(info_action, SIGNAL(triggered()), infoBox, SLOT(show()));
     connect(infoBox, SIGNAL(starting()), ti, SLOT(stop()));
     connect(infoBox, SIGNAL(finishing()), ti, SLOT(start()));
     connect(toolButton, SIGNAL(clicked(bool)), this, SLOT(showCalendarBox()));
     connect(calendarBox, SIGNAL(aDateWasSet()), this, SLOT(setDate()));
     connect(qApp, SIGNAL(aboutToQuit()), this, SLOT(beforeQuit())); //save window size / logs and unregister qshutdown
     connect(quit_action, SIGNAL(triggered()), qApp, SLOT(quit())); //contextmenu "Quit" for the systray quits Programm
     connect(checkPassword, SIGNAL(success()), editor, SLOT(show())); //if correct password was entered open editor
     connect(editor, SIGNAL(saved()), this, SLOT(updateLock()));      //update to see if Lock_all was (de)activated
     connect(pref, SIGNAL(editConf()), checkPassword, SLOT(show()));  //ask for password when button was pressed at preferences
}

Gui::~Gui(){ delete font1; delete font2; delete font3; }

void Gui::setInfoText(){ infoBox->setHtml(information); }

void Gui::setDate(){ toolButton->setText(setCalendarDate.toString(Qt::SystemLocaleShortDate)); }

void Gui::center(){
     QDesktopWidget *desktop = qApp->desktop();
     QPoint pos((desktop->width() - frameGeometry().width()) / 2,
                (desktop->height() - frameGeometry().height()) / 2);
     move(pos);
}

void Gui::iconActivated(QSystemTrayIcon::ActivationReason reason){
     if(reason == QSystemTrayIcon::Trigger){
       if(isHidden()||isMinimized())  //in case thant the window is minimized or hidden
         showNormal();
       else
         hide();
     }
}

void Gui::power_action(int action){
     QList<QAction*> actions = power_actions->actions();
     actions[action]->setChecked(true);
     switch(action){
       case 0:
         targetTime->setText(tr("Shutdown-time:"));
         minutes->setText(tr("Minutes till shutdown:"));
         break;
       case 1:
         targetTime->setText(tr("Reboot-time:"));
         minutes->setText(tr("Minutes till reboot:"));
         break;
       case 2:
         targetTime->setText(tr("Suspend-time:"));
         minutes->setText(tr("Minutes till suspend:"));
         break;
       case 3:
         targetTime->setText(tr("Hibernate-time:"));
         minutes->setText(tr("Minutes till hibernate:"));
         break;
       default:;
     }
}

void Gui::power(QAction *action){
     if(action == shutdown_action)
       comboBox->setCurrentIndex(0);
     if(action == reboot_action)
       comboBox->setCurrentIndex(1);
     if(action == suspend_action)
       comboBox->setCurrentIndex(2);
     if(action == hibernate_action)
       comboBox->setCurrentIndex(3);
}

void Gui::showW(){
     if(warnings->isChecked()) //warnings is checked
       showRunningProgram();
}

void Gui::showRunningProgram(){
     center();
     if(isMinimized() || isHidden())
       showNormal();
     else
       activateWindow();
}

void Gui::keyPressEvent(QKeyEvent *kEvent){
     if(kEvent->modifiers() == Qt::ControlModifier){
       if(!timeRunning && !lockAll){
         if(kEvent->key() == Qt::Key_S){
           shutdown_action->setChecked(true);
           comboBox->setCurrentIndex(0);
         }
         if(kEvent->key() == Qt::Key_R){
           reboot_action->setChecked(true);
           comboBox->setCurrentIndex(1);
         }
         if(kEvent->key() == Qt::Key_U){
           suspend_action->setChecked(true);
           comboBox->setCurrentIndex(2);
         }
         if(kEvent->key() == Qt::Key_H){
           hibernate_action->setChecked(true);
           comboBox->setCurrentIndex(3);
         }
       }
       if(!lockAll){
         if(kEvent->key() == Qt::Key_E && timeRunning)
           reset();
         if(kEvent->key() == Qt::Key_L){
           if(log_action->isChecked())
             log_action->setChecked(false);
           else
             log_action->setChecked(true);
         }
         if(kEvent->key() == Qt::Key_Q)
           qApp->quit();
       }
       if(kEvent->key() == Qt::Key_I){
         setInfoText();
         infoBox->show();
       }
       if(kEvent->key() == Qt::Key_P)
         pref->show();
     }
     if(kEvent->modifiers() == Qt::ShiftModifier){
       if(kEvent->key() == Qt::Key_E)
         checkPassword->show(); //ask for password to edit qshutdown.conf
     }
}

void Gui::updateT(){
     if(finishedExecuted)
       reset();

     QString tip1, tip2;

     if(shutdown_action->isChecked())
       tip1 = (tr("shutdown in "));
     if(reboot_action->isChecked())
       tip1 = (tr("reboot in "));
     if(suspend_action->isChecked())
       tip1 = (tr("suspend in "));
     if(hibernate_action->isChecked())
       tip1 = (tr("hibernate in "));

     if(QDate::currentDate().daysTo(setCalendarDate) >= 1)
       toolButton->setText(setCalendarDate.toString(Qt::SystemLocaleShortDate));

     if(QDate::currentDate().daysTo(setCalendarDate) > 1){ //if the date difference between today and the selected day
                                                           //in the calendar is greater than one
     //if more than one year
       if(QDate::currentDate().daysTo(setCalendarDate) >= 365*2){
         tip2 = (QString::number(QDate::currentDate().daysTo(setCalendarDate)/365) + " " + tr("years"));
         lcdL->setText(tr("years"));
         lcd->display(QDate::currentDate().daysTo(setCalendarDate)/365);
       }
       if(QDate::currentDate().daysTo(setCalendarDate) >= 365 && QDate::currentDate().daysTo(setCalendarDate) < 365*2){
         tip2 = (">= 1 " + tr("year"));
         lcdL->setText(tr("year"));
         lcd->display(1);
       }
     //if more than one month
       if(QDate::currentDate().daysTo(setCalendarDate) >= 30*2 && QDate::currentDate().daysTo(setCalendarDate) < 356){
         tip2 = (QString::number(QDate::currentDate().daysTo(setCalendarDate)/30) + " " + tr("months"));
         lcdL->setText(tr("months"));
         lcd->display(QDate::currentDate().daysTo(setCalendarDate)/30);
       }
       if(QDate::currentDate().daysTo(setCalendarDate) >= 30 && QDate::currentDate().daysTo(setCalendarDate) < 30*2){
         tip2 = (">= 1 " + tr("month"));
         lcdL->setText(tr("month"));
         lcd->display(1);
       }
     //if less than 30 days
       if(QDate::currentDate().daysTo(setCalendarDate) < 30){
         tip2 = (QString::number(QDate::currentDate().daysTo(setCalendarDate)) + " " + tr("days"));
         lcdL->setText(tr("days"));
         lcd->display(QDate::currentDate().daysTo(setCalendarDate));
       }
     }
     if(QDate::currentDate().daysTo(setCalendarDate) == 1){ //if there is one more day to go

       Time();

       if(i>=86400){ //if one day and some time to go
         tip2 = (">= 1 " + tr("day"));
         lcdL->setText(tr("day"));
         lcd->display(1);
       }
       if(i<86400 && i>=7200){ //if there is less than one day show hours
         tip2 = (QString::number(i/3600) + " " + tr("hours"));
         lcdL->setText(tr("hours"));
         lcd->display(i/3600);
       }
       if(i<7200 && i>=3600){
         tip2 = ("1 " + tr("hour"));
         lcdL->setText(tr("hour"));
         lcd->display(1);
       }
       if(i<3600 && i>=120){ //if less than one hour
         tip2 = (QString::number(i/60) + " " + tr("minutes"));
         lcdL->setText(tr("minutes"));
         lcd->display(i/60);
       }
       if(i<60){
         tip2 = ("1 " + tr("minute"));
         lcdL->setText(tr("minute"));
         lcd->display(1);
       }
     }
     if(QDate::currentDate()==setCalendarDate || setCalendarDate.isNull()){ //if there was no date
                                             //set in the calendar or the set day is the current day
       Time();

       if(warnings->isChecked() && (((i<=100) && (i>95)) || ((i<=40) && (i>35))) && !ti->isActive())
         ti->start(30000);

       if(i >= 3600){ //Display hours and minutes
         tip2 = (QString::number(i/3600) + " " + tr("hours") + " + "
                + QString::number(i/60 - (i/3600)*60) + " " + tr("minutes"));
         lcdL->setText(tr("hours"));
         lcd->display(i/3600);
       }
       if(i<3600 && i>=120){ //Display only minutes
         tip2 = (QString::number(i/60) + " " + tr("minutes"));
         lcdL->setText(tr("minutes"));
         lcd->display(i/60);
       }
       if(i<120 && i>=60){
         tip2 = ("1 " + tr("minute"));
         lcdL->setText(tr("minute"));
         lcd->display(1);
       }
       if(i<60){ //Display only seconds
         tip2 = (QString::number(i) + " " + tr("seconds"));
         lcdL->setText(tr("seconds"));
         lcd->display(i);
       }

       //this will ensure that the shutdown-type will be executed in case a few seconds were skipped
       if(i<=86390)
         n = 10; //any number to keep i in check
       if((i==0) || ((i>n) && (i>86390)))
         finished(); //execute shutdown-type
     }
     setWindowTitle(tip1 + tip2);
     TIcon->setToolTip(tip1 + tip2);
}

void Gui::set(){
     timeRunning = true;

     ti->stop();
     timer->start(1000);

     localTime = QTime::currentTime(); //the time now
     if(radio2->isChecked()) //if minute-countdown
       futureTime = localTime.addSecs(spin->value()*60); //add spinbox-value to the current time
     else
       futureTime = timeEdit->time(); //else the future time is the time of the timeEdit
     n = 86400; // any number bigger than i - just for initializing

     if(lock->isChecked()){       //when OK-button is clicked and lock is checked
       QList<QWidget*> list;
       list << spin << radio1 << radio2 << lock << timeEdit << comboBox << targetTime
            << minutes;
       foreach(QWidget * ptr, list)
         ptr->setDisabled(true);
       power_actions->setDisabled(true);
       ok->hide();
       minim->show();
       minim->setFocus();
       minim->setAutoDefault(true);         //minimize-button will accept the return key
     }
     showMinimized();
     if(!lockAll)
       reset_action->setVisible(true);
}

void Gui::Time(){
     localTime = QTime::currentTime();
     i = localTime.secsTo(futureTime); //the difference of the localTime and the future time
     if(radio2->isChecked() && (QDate::currentDate().daysTo(setCalendarDate) == 1))
       i = ((23-localTime.hour())*3600 + (59-localTime.minute())*60 + spin->value()*60);
     if(radio1->isChecked() && (QDate::currentDate().daysTo(setCalendarDate) == 1))
       i = ((23-localTime.hour())*3600 + (59-localTime.minute())*60 + timeEdit->time().hour()*3600
           + timeEdit->time().minute()*60);
     if(i<0)                           //if the date is on the next day
       i += 86400; //add 1 day in seconds
}

void Gui::saveWindowSize(){
   #ifdef Q_OS_WIN32
     QString file(QDesktopServices::storageLocation(QDesktopServices::DataLocation) + "/qshutdown/qshutdown.conf");
     QSettings settings(file, QSettings::NativeFormat);
   #else //!Q_OS_WIN32
     QString file(QDir::homePath() + "/.qshutdown/qshutdown.conf");
     QSettings settings(file, QSettings::NativeFormat);
   #endif //Q_OS_WIN32
     settings.setValue("MainWindowSize/size", size());
}

void Gui::finished(){
     finishedExecuted = true;
     switch(comboBox->currentIndex()){
       case 0:
         saveWindowSize();
         Power::shutdown();
         break;
       case 1:
         saveWindowSize();
         Power::reboot();
         break;
       case 2:
         Power::suspend();
         break;
       case 3:
         Power::hibernate();
         break;
       default:;
     }
}

void Gui::closeEvent(QCloseEvent* window_close){
     hide();
     QWidget::closeEvent(window_close);
}

void Gui::beforeQuit(){
     saveWindowSize();
     if(log_action->isChecked()){ //if logfile is set in the icon contextmenu
     #ifdef Q_OS_WIN32
       QFile logfile(QDesktopServices::storageLocation(QDesktopServices::DataLocation) + "/qshutdown/log.txt");
     #else //!Q_OS_WIN32
       QFile logfile(QDir::homePath() + "/.qshutdown/log.txt");
     #endif //Q_OS_WIN32
       if(!logfile.open(QIODevice::ReadWrite | QIODevice::Text)){
         qCritical("E: Can not open log.txt!");
         return;
       }

/******* writing log to log.txt *******/
       QTextStream out(&logfile);
       while(!logfile.atEnd())
         out.readLine();
       out << "[" << datetime.toString("yyyy.MM.dd hh:mm") << "] "
           << 1/(60000.0/elapsedTime.elapsed()) << " Minutes uptime\n";
       logfile.close();

/******* if log.txt is bigger than set in preferences, delete first line *******/
       bool isBigger;
       if(logfile.size()>logFileSize*1000)
         isBigger = true;
       else
         isBigger = false;
       while(isBigger){
         logfile.open(QIODevice::ReadWrite | QIODevice::Text);
         QTextStream out(&logfile);
         out.readLine();
         QString content = out.readAll();
         logfile.resize(0);
         out << content;
         logfile.close();
         if(logfile.size()<=logFileSize*1000)
           isBigger = false;
       }
     }

  #ifndef Q_OS_WIN32
     QDBusConnection::sessionBus().unregisterObject(OBJECT_NAME, QDBusConnection::UnregisterNode);
     QDBusConnection::sessionBus().unregisterService(SERVICE_NAME);
  #endif //Q_OS_WIN32
}

void Gui::loadSettings(){
/***************** create file and it's entries *****************/
#ifdef Q_OS_WIN32
     QString file(QDesktopServices::storageLocation(QDesktopServices::DataLocation) + "/qshutdown/qshutdown.conf");
     QSettings settings(file, QSettings::NativeFormat);
#else //!Q_OS_WIN32
     QString file(QDir::homePath() + "/.qshutdown/qshutdown.conf");
     QSettings settings(file, QSettings::NativeFormat);
#endif //Q_OS_WIN32
     if(!settings.isWritable())
       qCritical() << "W: qshutdown.conf is not writeable!";

     if(!QFile::exists(file) || !settings.contains("Lock_all"))
       settings.setValue("Lock_all", false);
     if(!settings.contains("Time/time_hour"))
       settings.setValue("Time/time_hour", 22);
     if(!settings.contains("Time/time_minute"))
       settings.setValue("Time/time_minute", 00);
     if(!settings.contains("Time/countdown_minutes"))
       settings.setValue("Time/countdown_minutes", 60);
     if(!settings.contains("Time/countdown_at_startup"))
       settings.setValue("Time/countdown_at_startup", false);
     if(!settings.contains("Hide_at_startup"))
       settings.setValue("Hide_at_startup", false);
     if(!settings.contains("MainWindowSize/size"))
       settings.setValue("MainWindowSize/size", QSize(430, 325));
     if(!settings.contains("Fonts/font_type"))
       settings.setValue("Fonts/font_type", "Times New Roman");
     if(!settings.contains("Fonts/font1"))
       settings.setValue("Fonts/font1", 13);
     if(!settings.contains("Fonts/font2"))
       settings.setValue("Fonts/font2", 18);
     if(!settings.contains("Fonts/font3"))
       settings.setValue("Fonts/font3", 11);
     if(!settings.contains("CheckBoxes/target_time"))
       settings.setValue("CheckBoxes/target_time", false);
     if(!settings.contains("CheckBoxes/countdown"))
       settings.setValue("CheckBoxes/countdown", true);
     if(!settings.contains("heckBoxes/lock"))
       settings.setValue("CheckBoxes/lock", true);
     if(!settings.contains("CheckBoxes/warnings"))
       settings.setValue("CheckBoxes/warnings", true);
     if(!settings.contains("Power/comboBox"))
       settings.setValue("Power/comboBox", 0);
     if(!settings.contains("Logfile/logging"))
       settings.setValue("Logfile/logging", false);
     if(!settings.contains("Logfile/size"))
       settings.setValue("Logfile/size", 1.5);
     if(!settings.contains("Lock_screen"))
       settings.setValue("Lock_screen", true);
     if(!settings.contains("Autostart"))
       settings.setValue("Autostart", false);

/***************** read files entries *****************/
     timeEdit->setTime(QTime(settings.value("Time/time_hour").toInt(),settings.value("Time/time_minute").toInt()));
     spin->setValue(settings.value("Time/countdown_minutes").toInt());
     resize(settings.value("MainWindowSize/size").toSize());
     font1->setFamily(settings.value("Fonts/font_type").toString());
     font2->setFamily(settings.value("Fonts/font_type").toString());
     font3->setFamily(settings.value("Fonts/font_type").toString());
     font1->setPointSize(settings.value("Fonts/font1").toInt());
     font2->setPointSize(settings.value("Fonts/font2").toInt());
     font3->setPointSize(settings.value("Fonts/font3").toInt());
     radio1->setChecked(settings.value("CheckBoxes/target_time").toBool());
     radio2->setChecked(settings.value("CheckBoxes/countdown").toBool());
     lock->setChecked(settings.value("CheckBoxes/lock").toBool());
     comboBox->setCurrentIndex(settings.value("Power/comboBox").toInt());
     warnings->setChecked(settings.value("CheckBoxes/warnings").toBool());
     log_action->setChecked(settings.value("Logfile/logging").toBool());
     logFileSize = settings.value("Logfile/size").toDouble();

     if(settings.value("Time/countdown_at_startup").toBool()){
       set();
       if(settings.value("Hide_at_startup").toBool())
         QTimer::singleShot(1500, this, SLOT(hide()));
     }

     if(settings.value("Lock_all").toBool()){
       lockAll = true;
       lockEverything(true);
     }
     else
       lockAll = false;

     QList<QWidget*> list;
     list << lcdL << comboBox << targetTime
          << minutes << spin << timeEdit;
     foreach(QWidget * ptr, list)
       ptr->setFont(*font1);
     ok->setFont(*font2);
     minim->setFont(*font2);
     warnings->setFont(*font3);
     lock->setFont(*font3);
     now->setFont(*font3);

     switch(settings.value("Power/comboBox").toInt()){
       case 0:
         targetTime->setText(tr("Shutdown-time:"));
         minutes->setText(tr("Minutes till shutdown:"));
         shutdown_action->setChecked(true);
         break;
       case 1:
         targetTime->setText(tr("Reboot-time:"));
         minutes->setText(tr("Minutes till reboot:"));
         reboot_action->setChecked(true);
         break;
       case 2:
         targetTime->setText(tr("Suspend-time:"));
         minutes->setText(tr("Minutes till suspend:"));
         suspend_action->setChecked(true);
         break;
       case 3:
         targetTime->setText(tr("Hibernate-time:"));
         minutes->setText(tr("Minutes till hibernate:"));
         hibernate_action->setChecked(true);
         break;
       default:;
     }

     if(settings.value("Lock_screen").toBool())
       lockMyScreen = true;
     else
       lockMyScreen = false;
}

void Gui::lockEverything(bool actual){
     if(timeRunning){
       reset_action->setVisible(true);
       reset_action->setDisabled(actual);
       quit_action->setDisabled(actual);
       log_action->setDisabled(actual);
       toolButton->setDisabled(actual);
     }
     else if(!timeRunning){
       if(actual)
         lock->setChecked(true);
       if(radio1->isChecked())
         timeEdit->setDisabled(actual);
       else
         spin->setDisabled(actual);
       reset_action->setDisabled(actual);
       quit_action->setDisabled(actual);
       log_action->setDisabled(actual);
       power_actions->setDisabled(actual);
       reset_action->setVisible(false);
       QList<QWidget*> widgetList;
       widgetList << radio1 << radio2 << lock << comboBox << targetTime
                  << minutes << toolButton;
       foreach(QWidget * widgetPtr, widgetList)
         widgetPtr->setDisabled(actual);
     }
     QList<QWidget*> widgetList;
     widgetList << pref->comboBox << pref->timeEdit << pref->spin
                << pref->radio1 << pref->radio2
                << pref->autostart << pref->lock << pref->countdown
                << pref->log << pref->reset;
     foreach(QWidget * widgetPtr, widgetList)
       widgetPtr->setDisabled(actual);
     if(pref->log->isEnabled() && pref->log->isChecked())
       pref->spinBox->setDisabled(false);
     else
       pref->spinBox->setDisabled(true);
}
void Gui::reset(){
     setWindowTitle("'qshutdown'");
     toolButton->setText("...");
     lcd->display(0);
     TIcon->setToolTip(NULL);
     lcdL->setText(tr("minutes"));
     setCalendarDate = QDate();
     if(finishedExecuted)
       showNormal();
     timer->stop();
     if(!ti->isActive())
       ti->start(30000);
     timeRunning = false;
     finishedExecuted = false;
     minim->hide();
     ok->show();
     if(!lockAll)
       lockEverything(false);
     else loadSettings();
}

void Gui::getFonts(){
     font1->setFamily(fonts);
     font1->setPointSize(fontS1);
     font2->setFamily(fonts);
     font2->setPointSize(fontS2);
     font3->setFamily(fonts);
     font3->setPointSize(fontS3);

     QList<QWidget*> list;
     list << lcdL << comboBox << targetTime
          << minutes << spin << timeEdit;
     foreach(QWidget * ptr, list)
       ptr->setFont(*font1);
     ok->setFont(*font2);
     minim->setFont(*font2);
     warnings->setFont(*font3);
     lock->setFont(*font3);
     now->setFont(*font3);

     resize(minimumSize());
}

void Gui::showCalendarBox(){
     if(setCalendarDate.isNull())
       calendarBox->calendarWidget->setSelectedDate(QDate::currentDate());
     else
       calendarBox->calendarWidget->setSelectedDate(setCalendarDate);
     if(timeRunning && lock->isChecked())
       calendarBox->calendarWidget->blockSignals(true);
     else
       calendarBox->calendarWidget->blockSignals(false);
     calendarBox->show();
}

void Gui::updateLock(){
    #ifdef Q_OS_WIN32
       QString file(QDesktopServices::storageLocation(QDesktopServices::DataLocation) + "/qshutdown/qshutdown.conf");
    #else //!Q_OS_WIN32
       QString file(QDir::homePath() + "/.qshutdown/qshutdown.conf");
    #endif //Q_OS_WIN32
       QSettings settings(file, QSettings::NativeFormat);
       if(settings.value("Lock_all").toBool()){
         lockAll = true;
         lockEverything(true);
       }
       else{
         lockAll = false;
         lockEverything(false);
       }
}
