/*
 *  Part of the shrinkta program, a dvd copy tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include <bonobo.h>
#include <gnome.h>

#include <dvd.h>

#include "preferences.h"
#include "app.h"
#include "dialogs.h"
#include "pipe.h"
#include "shrink.h"

static void
child_setup_func		(gpointer	 data)
{
	PipeHandle *handle;

	handle = (PipeHandle *) data;
}

gboolean
shrink_video		(const gchar	*path,
			 guint		 size_mb,
			 gfloat		 ratio,
			 progress_func	 func)
{
	PipeHandle shrink_handle;
	PipeHandle file_handle;
	gchar *file_path, *dir;
	gchar *tmpargs;
	gchar **args;
	gboolean ok;
	
	/* move large movie first */
	func ("Shrinking video file", 0.0);
	dir = g_path_get_dirname (path);
	file_path = g_strdup_printf ("%s/video_shrink.m2v", dir);
	g_free (dir);
	
	/* open files */
	shrink_handle.read_fd = open (path, O_RDONLY);
	if (shrink_handle.read_fd < 1) {
		g_warning ("Unable to open read file %s", path);
		g_free (file_path);
		return FALSE;
	}
	file_handle.write_fd = open (file_path, O_WRONLY | O_CREAT | O_TRUNC, 0664);
	if (file_handle.write_fd < 1) {
		g_warning ("Unable to open file %s", file_path);
		g_free (file_path);
		return FALSE;
	}
	
	/* setup tcrequant */
	shrink_handle.read_mbs = size_mb;
	shrink_handle.func = func; 
	g_message ("Read mbytes = %u", shrink_handle.read_mbs);
	g_message ("Ratio is %.2F", (gfloat) ratio);
	tmpargs = g_strdup_printf ("tcrequant:-f%.2F", (gfloat) ratio);
	g_message ("Command is %s", tmpargs);
	args = g_strsplit_set (tmpargs, ":", -1);
	g_free (tmpargs);

	ok = g_spawn_async_with_pipes (NULL,
				       args,
				       NULL,
				       G_SPAWN_SEARCH_PATH,
				       child_setup_func,
				       &shrink_handle,
				       &shrink_handle.child,
				       &shrink_handle.write_fd,
				       &file_handle.read_fd,
				       NULL, NULL);
	if (ok == FALSE) {
		g_warning ("can't spawn tcrequant process");
		close (shrink_handle.read_fd);
		close (shrink_handle.write_fd);
		g_strfreev (args);
		g_free (file_path);
		return FALSE;
	}
	
	/* now have process running - pipe file into process */
	shrink_handle.thread = g_thread_create (pipe_thread_func,
						&shrink_handle,
						TRUE,
						NULL);
	/* pipe into file thread */
	file_handle.thread = g_thread_create (pipe_file_thread_func,
					      &file_handle,
					      TRUE,
					      NULL);
	
	g_message ("join shrink thread");
	g_thread_join (shrink_handle.thread);
	g_message ("close shrink fds");
	close (shrink_handle.read_fd);
	close (shrink_handle.write_fd);
	g_message ("join file thread");
	g_thread_join (file_handle.thread);
	g_message ("close file fds");
	close (file_handle.read_fd);
	close (file_handle.write_fd);
	g_message ("free args");
	g_strfreev (args);
	g_message ("deleting old video file");
	func ("Deleting old video file...", 0.0);
	if (unlink (path) == -1) {
		g_warning ("Unable to remove large video file");
	}
	if (rename (file_path, path) == -1) {
		g_free (file_path);
		g_warning ("unable to rename file");
		return FALSE;
	}
	g_free (file_path);
	g_message ("video shrink done");
	
	return TRUE;
}
