/*
 *  Part of the shrinkta program, a dvd copy tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <inttypes.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <bonobo.h>
#include <gnome.h>

#include <gconf/gconf-client.h>
#include <nautilus-burn-drive-selection.h>

#include <dvd.h>
#include "preferences.h"

static GObjectClass	*preferences_parent_class = NULL;
static void		preferences_class_init		(PreferencesClass	*class);
static void		preferences_instance_init	(GTypeInstance	*instance,
							 gpointer	 g_class);
static void		preferences_dispose		(GObject	*object);

enum {
	PREFERENCES_LAST_SIGNAL
};

guint preferences_signals[PREFERENCES_LAST_SIGNAL];

static void
preferences_class_init	(PreferencesClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	preferences_parent_class = g_type_class_peek_parent (class);
	object_class->dispose = preferences_dispose;
}

static void
preferences_instance_init(GTypeInstance	*instance,
			 gpointer	 g_class)
{
	Preferences *prefs;
	
	prefs = PREFERENCES (instance);
	prefs->mutex = g_mutex_new ();
}

static void
preferences_dispose	(GObject	*object)
{
	Preferences *prefs;
	GConfClient *gconf_client;
	
	prefs = PREFERENCES (object);
	
	/* save prefs */
	g_mutex_lock (prefs->mutex);
	gconf_client = gconf_client_get_default ();
	if (gconf_client == NULL) {
		g_warning ("Unable to contact gconf");
	} else {
		gconf_client_set_string (gconf_client, "/apps/shrinkta/read_drive", prefs->drive_path, NULL);
		gconf_client_set_string (gconf_client, "/apps/shrinkta/image_folder", prefs->img_folder, NULL);
		gconf_client_set_string (gconf_client, "/apps/shrinkta/temp_folder", prefs->tmp_folder, NULL);
		gconf_client_set_int (gconf_client, "/apps/shrinkta/media_type", prefs->media, NULL);
		gconf_client_set_bool (gconf_client, "/apps/shrinkta/region_free", prefs->region_free, NULL);
		gconf_client_set_int (gconf_client, "/apps/shrinkta/preferred_audio", prefs->audio_format, NULL);
		gconf_client_set_bool (gconf_client, "/apps/shrinkta/include_dts", prefs->include_dts, NULL);
		gconf_client_set_int (gconf_client, "/apps/shrinkta/audio_language", prefs->language, NULL);
		gconf_client_set_bool (gconf_client, "/apps/shrinkta/allow_undefined_audio", prefs->allow_undefined, NULL);
		gconf_client_set_float (gconf_client, "/apps/shrinkta/maximum_video_shrink", prefs->max_shrink, NULL);
		gconf_client_set_bool (gconf_client, "/apps/shrinkta/drop_second_audio", prefs->drop_dts, NULL);
		gconf_client_set_string (gconf_client, "/apps/shrinkta/burn_command", prefs->burn_command, NULL);
		g_object_unref (G_OBJECT (gconf_client));
	}
	g_mutex_unlock (prefs->mutex);
	g_mutex_free (prefs->mutex);
	g_message ("prefs dispose");
	G_OBJECT_CLASS (preferences_parent_class)->dispose (G_OBJECT (prefs));
}

/* --- prototypes --- */
GType
preferences_get_type	(void)
{
	static GType prefs_type = 0;

	if (prefs_type == 0) {
		GTypeInfo prefs_info = {
			sizeof (PreferencesClass),
			NULL,
			NULL,
			(GClassInitFunc) preferences_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (Preferences),
			0, /* n_preallocs */
			(GInstanceInitFunc) preferences_instance_init
	    	};
		prefs_type = g_type_register_static (G_TYPE_OBJECT,
						     "Preferences",
						     &prefs_info, 0);
	}
	return prefs_type;
}

Preferences
*preferences_new		(void)
{
	Preferences *prefs;
	GConfClient *gconf_client;
	gchar *program_path;
	
	gconf_client = gconf_client_get_default ();
	if (gconf_client == NULL) {
		g_warning ("Unable to contact gconf");
		return NULL;
	}
	prefs = g_object_new (preferences_get_type (), NULL);
	g_mutex_lock (prefs->mutex);
	prefs->drive_path = gconf_client_get_string (gconf_client, "/apps/shrinkta/read_drive", NULL);
	if (prefs->drive_path == NULL) {
		/* set defaults */
		prefs->drive_path	= NULL; /** ToDo - search for suitable optical drive */
		prefs->img_folder	= g_strdup (g_get_home_dir ());
		prefs->tmp_folder	= g_strdup ("/tmp");
		prefs->media		= DVD_MEDIA_SL;
		prefs->region_free	= TRUE;
		prefs->audio_format	= DVD_AUDIO_FORMAT_AC3;
		prefs->include_dts	= TRUE;
		prefs->language		= DVD_LANG_ENGLISH;
		prefs->allow_undefined	= TRUE;
		prefs->max_shrink	= 1.4;
		prefs->drop_dts		= TRUE;
	} else {
		prefs->img_folder	= gconf_client_get_string (gconf_client, "/apps/shrinkta/image_folder", NULL);
		prefs->tmp_folder	= gconf_client_get_string (gconf_client, "/apps/shrinkta/temp_folder", NULL);
		if (prefs->drive_path[0] != '/') {
			/* first release! */
			g_free (prefs->drive_path);
			prefs->drive_path = NULL;
			prefs->media = DVD_MEDIA_SL;
		} else {
			prefs->media	= gconf_client_get_int (gconf_client, "/apps/shrinkta/media_type", NULL);
		}
		prefs->region_free	= gconf_client_get_bool (gconf_client, "/apps/shrinkta/region_free", NULL);
		prefs->audio_format	= gconf_client_get_int (gconf_client, "/apps/shrinkta/preferred_audio", NULL);
		prefs->include_dts	= gconf_client_get_bool (gconf_client, "/apps/shrinkta/include_dts", NULL);
		prefs->language		= gconf_client_get_int (gconf_client, "/apps/shrinkta/audio_language", NULL);
		prefs->allow_undefined	= gconf_client_get_bool (gconf_client, "/apps/shrinkta/allow_undefined_audio", NULL);
		prefs->max_shrink	= gconf_client_get_float (gconf_client, "/apps/shrinkta/maximum_video_shrink", NULL);
		prefs->drop_dts		= gconf_client_get_bool (gconf_client, "/apps/shrinkta/drop_second_audio", NULL);
	}
	/* added 04/2008 */
	prefs->burn_command = gconf_client_get_string (gconf_client, "/apps/shrinkta/burn_command", NULL);
	
	/* used to be able to set audio to DTS but now LPCM is supported */
	/* DVD should have first audio track as AC3 or LPCM anyways? */
	if (prefs->audio_format == DVD_AUDIO_FORMAT_DTS) {
		prefs->audio_format = DVD_AUDIO_FORMAT_LPCM;
	}
	
	program_path = g_find_program_in_path ("nautilus-cd-burner");
	if (program_path != NULL) {
		prefs->have_nautilus_cd_burner = TRUE;
		g_free (program_path);
	} else {
		prefs->have_nautilus_cd_burner = FALSE;
	}
	program_path = g_find_program_in_path ("brasero");
	if (program_path != NULL) {
		prefs->have_brasero = TRUE;
		g_free (program_path);
	} else {
		prefs->have_brasero = FALSE;
	}
	if ((prefs->have_nautilus_cd_burner == FALSE) && (prefs->have_brasero == FALSE)) {
		g_warning ("No available burning program");
	} else if (prefs->have_brasero == FALSE) {
		if (prefs->burn_command == NULL) {
			prefs->burn_command = g_strdup (BURNER_COMMAND_NAUTILUS_CD_BURNER);
		} else if (strcmp (prefs->burn_command, BURNER_COMMAND_NAUTILUS_CD_BURNER) == 0) {
			g_message ("burner program nautilus_cd_burner exists");
		} else {
			g_message ("brasero not available - automatically using nautilus cd burner");
			g_free (prefs->burn_command);
			prefs->burn_command = g_strdup (BURNER_COMMAND_NAUTILUS_CD_BURNER);
		}
	} else if (prefs->have_nautilus_cd_burner == FALSE) {
		if (prefs->burn_command == NULL) {
			prefs->burn_command = g_strdup (BURNER_COMMAND_BRASERO);
		} else if (strcmp (prefs->burn_command, BURNER_COMMAND_BRASERO) == 0) {
			g_message ("burner program brasero exists");
		} else {
			g_message ("nautilus cd burner not available - automatically using brasero");
			prefs->burn_command = g_strdup (BURNER_COMMAND_BRASERO);
		}
	} else {
		if (prefs->burn_command == NULL) {
			prefs->burn_command = g_strdup (BURNER_COMMAND_NAUTILUS_CD_BURNER);
		} else if (strcmp (prefs->burn_command, BURNER_COMMAND_NAUTILUS_CD_BURNER) == 0) {
			g_message ("burner program nautilus_cd_burner exists");
		} else if (strcmp (prefs->burn_command, BURNER_COMMAND_BRASERO) == 0) {
			g_message ("burner program brasero exists");
		} else {
			g_warning ("burner program error \"%s\" - using nautilus cd burner", prefs->burn_command);
			g_free (prefs->burn_command);
			prefs->burn_command = g_strdup (BURNER_COMMAND_NAUTILUS_CD_BURNER);
		}
	}
	
	g_object_unref (G_OBJECT (gconf_client));
	g_mutex_unlock (prefs->mutex);
	return prefs;
}

gchar
*preferences_get_drive_path	(Preferences	*prefs)
{
	gchar *drive_path;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	drive_path = g_strdup (prefs->drive_path);
	g_mutex_unlock (prefs->mutex);
	return drive_path;
}

gboolean
preferences_set_drive_path	(Preferences	*prefs,
				 const gchar	*drive_path)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	g_free (prefs->drive_path);
	prefs->drive_path = g_strdup (drive_path);
	g_mutex_unlock (prefs->mutex);
	return TRUE;
}

gchar
*preferences_get_burn_command	(Preferences	*prefs)
{
	gchar *burn_command;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	burn_command = g_strdup (prefs->burn_command);
	g_mutex_unlock (prefs->mutex);
	return burn_command;
}

gboolean
preferences_set_burn_command	(Preferences	*prefs,
				 const gchar	*burn_command)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	g_free (prefs->burn_command);
	prefs->burn_command = g_strdup (burn_command);
	g_mutex_unlock (prefs->mutex);
	return TRUE;
}

gchar
*preferences_get_image_folder	(Preferences	*prefs)
{
	gchar *folder;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	folder = g_strdup (prefs->img_folder);
	g_mutex_unlock (prefs->mutex);
	return folder;
}

gboolean
preferences_set_image_folder	(Preferences	*prefs,
				 const gchar	*image_folder)
{
	g_assert (prefs != NULL);

	g_mutex_lock (prefs->mutex);
	g_free (prefs->img_folder);
	prefs->img_folder = g_strdup (image_folder);
	g_mutex_unlock (prefs->mutex);
	return TRUE;
}

gchar
*preferences_get_temp_folder	(Preferences	*prefs)
{
	gchar *folder;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	folder = g_strdup (prefs->tmp_folder);
	g_mutex_unlock (prefs->mutex);
	return folder;
}

gboolean
preferences_set_temp_folder	(Preferences	*prefs,
				 const gchar	*temp_folder)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	g_free (prefs->tmp_folder);
	prefs->tmp_folder = g_strdup (temp_folder);
	g_mutex_unlock (prefs->mutex);
	return TRUE;
}

DvdMedia
preferences_get_media		(Preferences	*prefs)
{
	DvdMedia media;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	media = prefs->media;
	g_mutex_unlock (prefs->mutex);
	return media;
}

void
preferences_set_media		(Preferences	*prefs,
				 DvdMedia	 media)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	prefs->media = media;
	g_mutex_unlock (prefs->mutex);
}

gboolean
preferences_get_region_free	(Preferences	*prefs)
{
	gboolean region_free;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	region_free = prefs->region_free;
	g_mutex_unlock (prefs->mutex);
	return region_free;
}

void
preferences_set_region_free	(Preferences	*prefs,
				 gboolean	 region_free)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	prefs->region_free = region_free;
	g_mutex_unlock (prefs->mutex);
}

DvdAudioFormat
preferences_get_audio_format	(Preferences	*prefs)
{
	DvdAudioFormat audio_format;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	audio_format = prefs->audio_format;
	g_mutex_unlock (prefs->mutex);
	return audio_format;
}

void
preferences_set_audio_format	(Preferences	*prefs,
				 DvdAudioFormat	 audio_format)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	prefs->audio_format = audio_format;
	g_mutex_unlock (prefs->mutex);
}

gboolean
preferences_get_include_dts	(Preferences	*prefs)
{
	gboolean include_dts;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	include_dts = prefs->include_dts;
	g_mutex_unlock (prefs->mutex);
	return include_dts;
}

void
preferences_set_include_dts	(Preferences	*prefs,
				 gboolean	 include_dts)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	prefs->include_dts = include_dts;
	g_mutex_unlock (prefs->mutex);
}

DvdLanguage
preferences_get_language	(Preferences	*prefs)
{
	DvdLanguage language;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	language = prefs->language;
	g_mutex_unlock (prefs->mutex);
	return language;
}

void
preferences_set_language	(Preferences	*prefs,
				 DvdLanguage	 language)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	prefs->language = language;
	g_mutex_unlock (prefs->mutex);
}

gboolean
preferences_get_use_undefined	(Preferences	*prefs)
{
	gboolean use_undefined;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	use_undefined = prefs->allow_undefined;
	g_mutex_unlock (prefs->mutex);
	return use_undefined;
}

void
preferences_set_use_undefined	(Preferences	*prefs,
				 gboolean	 use_undefined)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	prefs->allow_undefined = use_undefined;
	g_mutex_unlock (prefs->mutex);
}

gfloat
preferences_get_maximum_shrink	(Preferences	*prefs)
{
	gfloat maximum_shrink;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	maximum_shrink = prefs->max_shrink;
	g_mutex_unlock (prefs->mutex);
	return maximum_shrink;
}

void
preferences_set_maximum_shrink	(Preferences	*prefs,
				 gfloat		 maximum_shrink)
{
	g_assert (prefs != NULL);
	g_assert (maximum_shrink >= 0);
	
	g_mutex_lock (prefs->mutex);
	prefs->max_shrink = maximum_shrink;
	g_mutex_unlock (prefs->mutex);
}

gboolean
preferences_get_drop_dts	(Preferences	*prefs)
{
	gboolean drop_dts;
	
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	drop_dts = prefs->drop_dts;
	g_mutex_unlock (prefs->mutex);
	return drop_dts;
}

void
preferences_set_drop_dts	(Preferences	*prefs,
				 gboolean	 drop_dts)
{
	g_assert (prefs != NULL);
	
	g_mutex_lock (prefs->mutex);
	prefs->drop_dts = drop_dts;
	g_mutex_unlock (prefs->mutex);
}

