/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <config.h>
#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#else
#include <stdint.h>
#endif
#include <dvdread/ifo_types.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <dvdread/ifo_read.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvd.h>
#include "dvd-marshal.h"
#include "pes.h"

static GObjectClass *dvd_demux_parent_class = NULL;
static void     dvd_demux_class_init	(DvdDemuxClass	*class);
static void     dvd_demux_instance_init	(GTypeInstance	*instance,
					 gpointer	 g_class);
static void     dvd_demux_dispose	(GObject	*object);

enum {
	DEMUX_OUTPUT_DATA,
	DEMUX_LAST_SIGNAL
};

guint dvd_demux_signals[DEMUX_LAST_SIGNAL];

static void
dvd_demux_class_init	(DvdDemuxClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	dvd_demux_signals[DEMUX_OUTPUT_DATA] =
		g_signal_new ("output_data",
			      G_TYPE_FROM_CLASS(class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (DvdDemuxClass, output_data),
			      NULL, NULL,
			      dvd_marshal_VOID__INT_INT_INT_POINTER_UINT_UINT64,
			      G_TYPE_NONE,
			      6,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_INT,
			      G_TYPE_POINTER,
			      G_TYPE_UINT,
			      G_TYPE_UINT64);
	dvd_demux_parent_class = g_type_class_ref (G_TYPE_OBJECT);
	object_class->dispose = dvd_demux_dispose;
}

static void
dvd_demux_instance_init(GTypeInstance	*instance,
			 gpointer	 g_class)
{
	DvdDemux *demux;
	
	demux = DVD_DEMUX (instance);
}

static void
dvd_demux_dispose	(GObject	*object)
{
	DvdDemux *demux;
	
	demux = DVD_DEMUX (object);
	
	dvd_demux_reset (demux);
	G_OBJECT_CLASS (dvd_demux_parent_class)->dispose (G_OBJECT (demux));
}

/**
 * dvd_demux_get_type
 * @return The GType for the DvdDemux class.
 */
GType
dvd_demux_get_type	(void)
{
	static GType demux_type = 0;

	if (demux_type == 0) {
		GTypeInfo demux_info = {
			sizeof (DvdDemuxClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_demux_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdDemux),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_demux_instance_init
	    	};
		demux_type = g_type_register_static (G_TYPE_OBJECT,
						     "DvdDemux",
						     &demux_info, 0);
	}
	return demux_type;
}

DvdDemux
*dvd_demux_new		(void)
{
	DvdDemux *demux;
	
	demux = g_object_new (dvd_demux_get_type (), NULL);
	
	return demux;
}

static void
decoder_output_frame_cb		(DvdDecoder	*decoder,
				 DvdStream	 stream_type,
				 gint		 track,
				 gint		 bytes,
				 guint8		*frame_buffer,
				 guint32	 pts,
				 guint64	 frame_clocks,
				 DvdDemux	*demux)
{
	g_signal_emit (G_OBJECT (demux),
		       dvd_demux_signals[DEMUX_OUTPUT_DATA], 0,
		       stream_type,
		       track,
		       bytes,
		       frame_buffer,
		       pts,
		       frame_clocks);
}


gboolean
dvd_demux_vob_blocks		(DvdDemux	*demux,
				 guint8		*buffer,
				 guint		 blocks)
{
	guint block;
	static guint64 last_scr = 0;
	static guint64 all_scr = 0;
	
	
	/*g_message (" ");
	g_message ("===================================================================");
	g_message (" ");*/
	for (block = 0;
	     block < blocks;
	     block++) {
		DvdPackHeader pack_hdr;
		guint8 stream_id;
		DvdStream stream_type = DVD_STREAM_SDDS_AUDIO;
		guint8 *end = NULL;
		guint8 *ptr = NULL;
		guint8 track = 0;
		guint8 decoder_id = 0;
		DvdPESHeader stream_hdr;
		DvdPS1Header ps1_hdr;
		DvdLpcmHeader lpcm_hdr;
		
		ptr = buffer + block * 2048;
		end = ptr + 2048;
		
		if (dvd_pes_read_pack_header (&pack_hdr, &ptr) == FALSE) {
			g_warning ("no pack header");
			return FALSE;
		}
		
		stream_id = dvd_pes_find_header (&ptr, end);
		g_assert (stream_id != DVD_PES_START_CODE_NOT_FOUND); 
		if (dvd_pes_read_stream_header (&stream_hdr, &ptr) == FALSE) {
			g_message ("Not a pes header");
			return FALSE;
		}
		/*g_message ("packet scr = %llu diff = %llu pts = %llu payload bytes = %d",
			 pack_hdr.scr,
			 (guint64) pack_hdr.scr - last_scr, 
			 (guint64) stream_hdr.pts,
			 end - ptr - 1 + 24);
		g_message ("payload SCR should be %f", (27000000.0 / (1260000.0 / (end - ptr - 1 + 24))));*/
		all_scr += pack_hdr.scr - last_scr;
		last_scr = pack_hdr.scr;
		if (block == 0) {
			/*g_warning ("scr for last sequence=%llu should be 16200000 for 15 frames at 25 per second", all_scr);*/
			all_scr = 0;
		}

		if ((stream_hdr.stream_id == DVD_PES_START_CODE_PADDING) ||
		    (stream_hdr.stream_id == DVD_PES_START_CODE_PRIV_STREAM2)) {
			g_warning ("stream id %d", stream_hdr.stream_id);
			continue;
		}
		if (stream_hdr.mpeg_id == 1) {
			/* mpeg1 */
			g_warning ("mpeg1 not supported yet stream id 0x%x", stream_id);
			return FALSE;
		} else {
			/* mpeg2 */
			/* Assuming the following works */
			/* stream_hdr.packet_length is length of whole header extension and packet data */
			end = ptr + stream_hdr.packet_length;
			
			/* take off 3 header extension bytes */
			end -= 3;
			
			/* take the header extension field data length off */
			end -= stream_hdr.hdr_dlength;
			
			/*if (buffer + (block * 2048) + 2048 > end + 3) {
				g_message ("%d stuffing bytes", (buffer + (block * 2048) + 2048) - end);
				
			} 
			g_message ("0x%x 0x%x 0x%x 0x%x", end[0], end[1], end[2], end[3]);*/
			
		
		}
		switch (stream_id) {
		case 0xbd:
			/* Private stream 1 (non MPEG audio, subpictures) */
			if (dvd_pes_read_ps1_header (&ps1_hdr, &ptr) == FALSE) {
				g_warning ("Unable to read ps1 header");
				continue;
			}
			stream_type = ps1_hdr.stream_type;
			track = ps1_hdr.track;
			if (stream_type == DVD_STREAM_SUB_PICTURE) {
				decoder_id = track + 8;
				if (demux->decoder[decoder_id] == NULL) {
					/* move to start of first frame in block */
					ptr += ps1_hdr.pts_offset;
				}
				/* fail quietly */
				/*g_warning ("SUB STREAM");*/
				continue;
			} else if (stream_type == DVD_STREAM_AC3_AUDIO) {
				/*g_message ("ac3 audio");*/
				decoder_id = track;
				if (demux->decoder[decoder_id] == NULL) {
					/* move to start of first frame in block */
					ptr += ps1_hdr.pts_offset;
				}
			} else if (stream_type == DVD_STREAM_LPCM_AUDIO) {
				/*g_message ("lpcm audio");*/
				decoder_id = track;
				dvd_pes_read_lpcm_header (&lpcm_hdr, &ptr);
				if (demux->decoder[decoder_id] == NULL) {
					/* move to start of first frame in block */
					ptr += (ps1_hdr.pts_offset - 3);
				}
			} else {
				decoder_id = track;
				if (demux->decoder[decoder_id] == NULL) {
					/* move to start of first frame in block */
					ptr += ps1_hdr.pts_offset;
				}
			}
			
			break;
		case 0xc0:
		case 0xc1:
		case 0xc2:
		case 0xc3:
		case 0xc4:
		case 0xc5:
		case 0xc6:
		case 0xc7:
			/* MPEG-1 or MPEG-2 audio stream */
			/* MPEG audio ends at 0xdf - but unsupported for dvd */
			/* (DVD supports only 8 audio tracks) */
			track = stream_id - 0xc0;
			decoder_id = track;
			if (stream_hdr.mpeg_id == 1) {
				stream_type = DVD_STREAM_MPEG1_AUDIO;
			} else {
				stream_type = DVD_STREAM_MPEG2_AUDIO;
			}
			/* fail quietly */
			/*g_message ("mpeg audio");*/
			continue;
		case 0xe0:
			/* MPEG-1 or MPEG-2 video stream */
			decoder_id = 8;
			track = 0;
			/*g_message ("video");*/
			if (stream_hdr.mpeg_id == 1) {
				stream_type = DVD_STREAM_MPEG1_VIDEO;
			} else {
				stream_type = DVD_STREAM_MPEG2_VIDEO;
			}
			break;
		default:
			g_warning ("unknown packet");
			break;
		}
		if (demux->decoder[decoder_id] == NULL) {
			demux->decoder[decoder_id] = dvd_decoder_new (stream_type);
			g_signal_connect (demux->decoder[decoder_id],
					 "output_frame",
					 G_CALLBACK (decoder_output_frame_cb),
					 (gpointer) demux);
			dvd_decoder_set_track (demux->decoder[decoder_id], track);
			if (stream_type == DVD_STREAM_LPCM_AUDIO) {
				dvd_decoder_lpcm_set_quantization (DVD_DECODER_LPCM (demux->decoder[decoder_id]), (DvdAudioQuant) lpcm_hdr.word_size);
				dvd_decoder_lpcm_set_samplerate (DVD_DECODER_LPCM (demux->decoder[decoder_id]), (DvdAudioSamp) lpcm_hdr.samples);
				dvd_decoder_lpcm_set_clocks_per_frame (DVD_DECODER_LPCM (demux->decoder[decoder_id]), 150);
				dvd_decoder_lpcm_set_channels (DVD_DECODER_LPCM (demux->decoder[decoder_id]), (guint8) lpcm_hdr.channels + 1);
				g_message ("LPCM gain values %d %d", lpcm_hdr.dr_x, lpcm_hdr.dr_y);
			}
		}
		if (stream_hdr.pts_dts > 0) {
			dvd_decoder_write (demux->decoder[decoder_id], ptr, end - ptr, stream_hdr.pts);
		} else {
			dvd_decoder_write (demux->decoder[decoder_id], ptr, end - ptr, 0);
		}
	}
	
	return TRUE;
}

void
dvd_demux_reset		(DvdDemux	*demux)
{
	
	guint i;
	
	for (i = 0;
	     i < DVD_DEMUX_DECODERS;
	     i++) {
		if (demux->decoder[i] != NULL) {
			g_object_unref (G_OBJECT (demux->decoder[i]));
			demux->decoder[i] = NULL;
		}
	}
}

