
//////////////////////////////////////////////////////////////////////////
//
//   Slib, Sfront's SAOL library
//   This file: The SSM (Scaled Smoothed MIDInames) library
//   Copyright (C) 2000  Regents of the University of California
//
// This library file is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// The GNU C Library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with the GNU C Library; see the file COPYING.LIB.  If not,
// write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.  
//
//   Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
//
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//                                                
//
// This library is intended to replace direct MIDIctrl[], MIDIbend, 
// and MIDItouch calls in SAOL programs. It defines a set of SSM
// symbols with the following properties:
//
//  *  Can be used like standard names in expressions
//  *  Linear [0,1], [-1, 1], or binary 0/1 scaling as required
//  *  Non-binary names as smoothed at the k-rate, using the port()
//     opcode, at a default timeconstant of 4/krate. This rate
//     may be overriden by setting the SSM_SMOOTHRATE symbol to a time
//     value (in seconds).
//  *  Scaled, non-smoothed value for all symbols available as SMname.
//     For binary names, SMname defined identically to SSMname.
//  *  If SSM_HIGHRES defined, also uses LSB (default is MSB only)
//
// 
//       Name        Scaling     Function           uses SAOL stname
//
//  SSMattack        [0,1]       Sound Attack        MIDIctrl[73]
//  SSMbalance      [-1,+1]      Stereo Balance      MIDIctrl[8, 40]
//  SSMbend         [-1,+1]      Pitch Bend Wheel    MIDIbend
//  SSMbreath        [0,1]       Breath Controller   MIDIctrl[2, 34]
//  SSMbright        [0,1]       Sound Brightness    MIDIctrl[74]
//  SSMbutton1       0/1         G.P Button 1        MIDIctrl[80]
//  SSMbutton2       0/1         G.P Button 2        MIDIctrl[81]
//  SSMbutton3       0/1         G.P Button 3        MIDIctrl[82]
//  SSMbutton4       0/1         G.P Button 4        MIDIctrl[83]
//  SSMchorus        [0,1]       Chorus Level        MIDIctrl[93]
//  SSMdataslider    [0,1]       Data Entry Slider   MIDIctrl[6, 38]
//  SSMdetune        [0,1]       Detuning Amount     MIDIctrl[94]
//  SSMeffect        [0,1]       Effects Level       MIDIctrl[91]
//  SSMeffect1       [0,1]       Effect Control 1    MIDIctrl[12,44]
//  SSMeffect2       [0,1]       Effect Control 2    MIDIctrl[13,45]
//  SSMexpress       [0,1]       Expression          MIDIctrl[11,43]
//  SSMfoot          [0,1]       Foot Controller     MIDIctrl[2, 34]
//  SSMhold           0/1        Hold Pedal          MIDIctrl[64]
//  SSMhold2          0/1        Hold 2 Pedal        MIDIctrl[69]
//  SSMlegato         0/1        Legato Pedal        MIDIctrl[68]
//  SSMlocal          0/1        Local Kbd Off/On    MIDIctrl[122]
//  SSMmodwheel      [0,1]       Modulation Wheel    MIDIctrl[1, 33]   
//  SSMpan          [-1,+1]      Stereo Panning      MIDIctrl[10,42]
//  SSMphasor        [0,1]       Phasor Level        MIDIctrl[95]
//  SSMporta          0/1        Portamento On/Off   MIDIctrl[65]
//  SSMportatime     [0,1]       Portamento Time     MIDIctrl[5, 36]
//  SSMrelease       [0,1]       Sound Release       MIDIctrl[72]
//  SSMslider1       [0,1]       G. P. Slider 1      MIDIctrl[16]
//  SSMslider2       [0,1]       G. P. Slider 2      MIDIctrl[17]
//  SSMslider3       [0,1]       G. P. Slider 3      MIDIctrl[18]
//  SSMslider4       [0,1]       G. P. Slider 4      MIDIctrl[19]
//  SSMsoft           0/1        Soft Pedal          MIDIctrl[67]
//  SSMsound6        [0,1]       Sound Control 6     MIDIctrl[75]
//  SSMsound7        [0,1]       Sound Control 7     MIDIctrl[76]
//  SSMsound8        [0,1]       Sound Control 8     MIDIctrl[77]
//  SSMsound9        [0,1]       Sound Control 9     MIDIctrl[78]
//  SSMsound10       [0,1]       Sound Control 10    MIDIctrl[79]
//  SSMsust           0/1        Sustenuto Pedal     MIDIctrl[66]
//  SSMtimbre        [0,1]       Sound Timbre        MIDIctrl[71]
//  SSMtouch         [0,1]       Aftertouch          MIDItouch
//  SSMtremelo       [0,1]       Tremelo Level       MIDIctrl[92]
//  SSMvar           [0,1]       Sound Variation     MIDIctrl[70]
//  SSMvolume        [0,1]       Channel Volume      MIDIctrl[7, 39]
//
//
//  Note that decoder support is needed to handle Registered and
//  Non-Registered Parameters, since these are event-based. So, 
//  no definitions for Data Entry Button +/- and Parameter Number
//  controllers appear above. I did include the Data Entry Slider,
//  since its possible to use these values in a non-event-based way.
//
//
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//                                                
// library definitions begin here
//

#ifndef SLIB_SSM
#define SLIB_SSM

// where MIDI_ constants are found

#include <Slib/std.hs>

#ifndef SSM_SMOOTHRATE
#define SSM_SMOOTHRATE (4/k_rate)
#endif 

#define  SMattack (MIDI_SCALE*MIDIctrl[73])
#define SSMattack (port(SMattack, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define  SMbalance (MIDI_BIGSSCALE*(MIDI_MSBSHIFT*MIDIctrl[8] + MIDIctrl[40] - MIDI_BIGNULL))
#else
#define  SMbalance (MIDI_SSCALE*(MIDIctrl[8] - MIDI_NULL))
#endif
#define SSMbalance (port(SMbalance, SSM_SMOOTHRATE))

#define  SMbend (MIDI_BIGSSCALE*(MIDIbend - MIDI_BIGNULL))
#define SSMbend (port(SMbend, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define  SMbreath (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[2] + MIDIctrl[34]))
#else
#define  SMbreath (MIDI_SCALE*MIDIctrl[2])
#endif
#define SSMbreath (port(SMbreath, SSM_SMOOTHRATE))

#define  SMbright  (MIDI_SCALE*MIDIctrl[74])
#define SSMbright (port(SMbright, SSM_SMOOTHRATE))
#define  SMbutton1 ((MIDIctrl[80] > MIDI_OFF) ? 1 : 0)
#define SSMbutton1 SMbutton1
#define  SMbutton2 ((MIDIctrl[81] > MIDI_OFF) ? 1 : 0)
#define SSMbutton2 SMbutton2
#define  SMbutton3 ((MIDIctrl[82] > MIDI_OFF) ? 1 : 0)
#define SSMbutton3 SMbutton3
#define  SMbutton4 ((MIDIctrl[83] > MIDI_OFF) ? 1 : 0)
#define SSMbutton4 SMbutton4
#define  SMchorus  (MIDI_SCALE*MIDIctrl[93])
#define SSMchorus (port(SMchorus, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define SMdataslider (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[6] + MIDIctrl[38]))
#else
#define SMdataslider (MIDI_SCALE*MIDIctrl[6])
#endif
#define SSMdataslider (port(SMdataslider, SSM_SMOOTHRATE))

#define  SMdetune (MIDI_SCALE*MIDIctrl[94])
#define SSMdetune (port(SMdetune, SSM_SMOOTHRATE))
#define  SMeffect (MIDI_SCALE*MIDIctrl[91])
#define SSMeffect (port(SMeffect, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define  SMeffect1 (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[12] + MIDIctrl[44]))
#else
#define  SMeffect1 (MIDI_SCALE*MIDIctrl[12])
#endif
#define SSMeffect1 (port(SMeffect1, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define  SMeffect2 (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[13] + MIDIctrl[45]))
#else
#define  SMeffect2 (MIDI_SCALE*MIDIctrl[13])
#endif
#define SSMeffect2 (port(SMeffect2, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define  SMexpress (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[11] + MIDIctrl[43]))
#else
#define  SMexpress (MIDI_SCALE*MIDIctrl[11])
#endif
#define SSMexpress (port(SMexpress, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define  SMfoot (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[4] + MIDIctrl[36]))
#else
#define  SMfoot (MIDI_SCALE*MIDIctrl[4])
#endif
#define SSMfoot (port(SMfoot, SSM_SMOOTHRATE))

#define  SMhold   ((MIDIctrl[64] > MIDI_OFF)  ? 1 : 0)
#define SSMhold   SMhold
#define  SMhold2  ((MIDIctrl[69] > MIDI_OFF)  ? 1 : 0)
#define SSMhold2  SMhold2
#define  SMlegato ((MIDIctrl[68] > MIDI_OFF)  ? 1 : 0)
#define SSMlegato SMlegato
#define  SMlocal  ((MIDIctrl[122] > MIDI_OFF) ? 1 : 0)
#define SSMlocal  SMlocal

#ifdef SSM_HIGHRES
#define  SMmodwheel (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[1] + MIDIctrl[33]))
#else
#define  SMmodwheel (MIDI_SCALE*MIDIctrl[1])
#endif
#define SSMmodwheel (port(SMmodwheel, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define  SMpan (MIDI_BIGSSCALE*(MIDI_MSBSHIFT*MIDIctrl[10] + MIDIctrl[42] - MIDI_BIGNULL))
#else
#define  SMpan (MIDI_SSCALE*(MIDIctrl[10] - MIDI_NULL))
#endif
#define SSMpan (port(SMpan, SSM_SMOOTHRATE))

#define  SMphasor (MIDI_SCALE*MIDIctrl[95])
#define SSMphasor (port(SMphasor, SSM_SMOOTHRATE))
#define  SMporta  ((MIDIctrl[65] > MIDI_OFF) ? 1 : 0)
#define SSMporta  SMporta

#ifdef SSM_HIGHRES
#define  SMportatime (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[5] + MIDIctrl[36]))
#else
#define  SMportatime (MIDI_SCALE*MIDIctrl[5])
#endif
#define SSMportatime (port(SMportatime, SSM_SMOOTHRATE))

#define  SMrelease (MIDI_SCALE*MIDIctrl[72])
#define SSMrelease (port(SMrelease, SSM_SMOOTHRATE))
#define  SMslider1 (MIDI_SCALE*MIDIctrl[16])
#define SSMslider1 (port(SMslider1, SSM_SMOOTHRATE))
#define  SMslider2 (MIDI_SCALE*MIDIctrl[17])
#define SSMslider2 (port(SMslider2, SSM_SMOOTHRATE))
#define  SMslider3 (MIDI_SCALE*MIDIctrl[18])
#define SSMslider3 (port(SMslider3, SSM_SMOOTHRATE))
#define  SMslider4 (MIDI_SCALE*MIDIctrl[19])
#define SSMslider4 (port(SMslider4, SSM_SMOOTHRATE))
#define  SMsoft    ((MIDIctrl[67] > MIDI_OFF) ? 1 : 0)
#define SSMsoft    SMsoft
#define  SMsound6  (MIDI_SCALE*MIDIctrl[75])
#define SSMsound6  (port(SMsound6, SSM_SMOOTHRATE))
#define  SMsound7  (MIDI_SCALE*MIDIctrl[76])
#define SSMsound7  (port(SMsound7, SSM_SMOOTHRATE))
#define  SMsound8  (MIDI_SCALE*MIDIctrl[77])
#define SSMsound8  (port(SMsound8, SSM_SMOOTHRATE))
#define  SMsound9  (MIDI_SCALE*MIDIctrl[78])
#define SSMsound9  (port(SMsound9, SSM_SMOOTHRATE))
#define  SMsound10 (MIDI_SCALE*MIDIctrl[79])
#define SSMsound10 (port(SMsound10, SSM_SMOOTHRATE))
#define  SMsust    ((MIDIctrl[66] > MIDI_OFF) ? 1 : 0)
#define SSMsust    SMsust
#define  SMtimbre  (MIDI_SCALE*MIDIctrl[71])
#define SSMtimbre  (port(SMtimbre, SSM_SMOOTHRATE))
#define  SMtremelo (MIDI_SCALE*MIDIctrl[92])
#define SSMtremelo (port(SMtremelo, SSM_SMOOTHRATE))
#define  SMtouch   (MIDI_BIGSCALE*MIDItouch)
#define SSMtouch   (port(SMtouch, SSM_SMOOTHRATE))
#define  SMvar     (MIDI_SCALE*MIDIctrl[70])
#define SSMvar     (port(SMvar, SSM_SMOOTHRATE))

#ifdef SSM_HIGHRES
#define  SMvolume (MIDI_BIGSCALE*(MIDI_MSBSHIFT*MIDIctrl[7] + MIDIctrl[39]))
#else
#define  SMvolume (MIDI_SCALE*MIDIctrl[7])
#endif
#define SSMvolume (port(SMvolume, SSM_SMOOTHRATE))

#endif // SLIB_SSM 
