/*
 *  Copyright (C) 2004-2012 Savoir-Faire Linux Inc.
 *  Author: Alexandre Bourget <alexandre.bourget@savoirfairelinux.com>
 *  Author: Yan Morin <yan.morin@savoirfairelinux.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA.
 *
 *  Additional permission under GNU GPL version 3 section 7:
 *
 *  If you modify this program, or any covered work, by linking or
 *  combining it with the OpenSSL project's OpenSSL library (or a
 *  modified version of that library), containing parts covered by the
 *  terms of the OpenSSL or SSLeay licenses, Savoir-Faire Linux Inc.
 *  grants you additional permission to convey the resulting work.
 *  Corresponding Source for a non-source form of such a combination
 *  shall include the source code for the parts of OpenSSL used as well
 *  as that of the covered work.
 */
#include "iaxvoiplink.h"
#include <unistd.h>
#include <cmath>
#include <algorithm>

#include "iaxcall.h"
#include "eventthread.h"
#include "im/instant_messaging.h"
#include "iaxaccount.h"
#include "logger.h"
#include "manager.h"
#include "hooks/urlhook.h"
#include "audio/audiolayer.h"
#include "audio/samplerateconverter.h"
#include "array_size.h"
#include "scoped_lock.h"
#include "map_utils.h"

AccountMap IAXVoIPLink::iaxAccountMap_;
IAXCallMap IAXVoIPLink::iaxCallMap_;
// has same effect as pthread_mutex_init with default args, but for a statically
// allocated mutex
pthread_mutex_t IAXVoIPLink::iaxCallMapMutex_ = PTHREAD_MUTEX_INITIALIZER;

IAXVoIPLink::IAXVoIPLink(const std::string& accountID) :
    regSession_(NULL)
    , nextRefreshStamp_(0)
    , mutexIAX_()
    , decData_()
    , resampledData_()
    , encodedData_()
    , converter_(44100)
    , initDone_(false)
    , accountID_(accountID)
    , evThread_(this)
{
    pthread_mutex_init(&mutexIAX_, NULL);
    srand(time(NULL));    // to get random number for RANDOM_PORT
}


IAXVoIPLink::~IAXVoIPLink()
{
    handlingEvents_ = false;
    regSession_ = NULL; // shall not delete it // XXX: but why?
    terminate();

    // This is our last account
    if (iaxAccountMap_.size() == 1)
        clearIaxCallMap();
    pthread_mutex_destroy(&mutexIAX_);
}

void
IAXVoIPLink::init()
{
    if (initDone_)
        return;

    for (int port = IAX_DEFAULT_PORTNO, nbTry = 0; nbTry < 3 ; port = rand() % 64000 + 1024, nbTry++) {
        if (iax_init(port) >= 0) {
            handlingEvents_ = true;
            evThread_.start();
            initDone_ = true;
            break;
        }
    }
}

void
IAXVoIPLink::terminate()
{
    if (!initDone_)
        return;

    sfl::ScopedLock m(iaxCallMapMutex_);

    for (IAXCallMap::iterator iter = iaxCallMap_.begin(); iter != iaxCallMap_.end(); ++iter) {
        IAXCall *call = static_cast<IAXCall*>(iter->second);

        if (call) {
            sfl::ScopedLock lock(mutexIAX_);
            iax_hangup(call->session, const_cast<char*>("Dumped Call"));
            delete call;
        }
    }

    iaxCallMap_.clear();

    initDone_ = false;
}

bool
IAXVoIPLink::getEvent()
{
    iax_event *event = NULL;

    {
        sfl::ScopedLock lock(mutexIAX_);
        event = iax_get_event(0);
    }

    while (event != NULL) {

        // If we received an 'ACK', libiax2 tells apps to ignore them.
        if (event->etype == IAX_EVENT_NULL) {
            sfl::ScopedLock lock(mutexIAX_);
            iax_event_free(event);
            event = iax_get_event(0);
            continue;
        }

        IAXCall *call = iaxFindCallBySession(event->session);

        if (call) {
            iaxHandleCallEvent(event, call);
        } else if (event->session && event->session == regSession_) {
            // This is a registration session, deal with it
            iaxHandleRegReply(event);
        } else {
            // We've got an event before it's associated with any call
            iaxHandlePrecallEvent(event);
        }

        {
            sfl::ScopedLock lock(mutexIAX_);
            iax_event_free(event);
            event = iax_get_event(0);
        }
    }

    if (nextRefreshStamp_ && nextRefreshStamp_ < time(NULL))
        sendRegister(Manager::instance().getIaxAccount(accountID_));

    sendAudioFromMic();

    // thread wait 3 millisecond
    usleep(3000);
    return handlingEvents_;
}

std::vector<std::string>
IAXVoIPLink::getCallIDs()
{
    std::vector<std::string> v;
    sfl::ScopedLock m(iaxCallMapMutex_);

    map_utils::vectorFromMapKeys(iaxCallMap_, v);
    return v;
}

void
IAXVoIPLink::sendAudioFromMic()
{
    for (IAXCallMap::const_iterator iter = iaxCallMap_.begin(); iter != iaxCallMap_.end() ; ++iter) {
        IAXCall *currentCall = static_cast<IAXCall*>(iter->second);

        if (!currentCall or currentCall->getState() != Call::ACTIVE)
            continue;

        int codecType = currentCall->getAudioCodec();
        sfl::AudioCodec *audioCodec = static_cast<sfl::AudioCodec *>(Manager::instance().audioCodecFactory.getCodec(codecType));

        if (!audioCodec)
            continue;

        Manager::instance().getMainBuffer().setInternalSamplingRate(audioCodec->getClockRate());

        unsigned int mainBufferSampleRate = Manager::instance().getMainBuffer().getInternalSamplingRate();

        // we have to get 20ms of data from the mic *20/1000 = /50
        // rate/50 shall be lower than IAX__20S_48KHZ_MAX
        const size_t bytesNeeded = mainBufferSampleRate * 20 / 1000 * sizeof(SFLDataFormat);

        if (Manager::instance().getMainBuffer().availableForGet(currentCall->getCallId()) < bytesNeeded)
            continue;

        // Get bytes from micRingBuffer to data_from_mic
        int bytes = Manager::instance().getMainBuffer().getData(decData_, bytesNeeded, currentCall->getCallId());
        int samples = bytes / sizeof(SFLDataFormat);

        int compSize;
        unsigned int audioRate = audioCodec->getClockRate();
        int outSamples;
        SFLDataFormat *in;

        if (audioRate != mainBufferSampleRate) {
            converter_.resample(decData_, resampledData_, ARRAYSIZE(resampledData_),
                                audioRate, mainBufferSampleRate, samples);
            in = resampledData_;
            outSamples = 0;
        } else {
            outSamples = samples;
            in = decData_;
        }

        compSize = audioCodec->encode(encodedData_, in, DEC_BUFFER_SIZE);

        if (currentCall->session and bytes > 0) {
            sfl::ScopedLock m(mutexIAX_);

            if (iax_send_voice(currentCall->session, currentCall->format, encodedData_, compSize, outSamples) == -1)
                ERROR("IAX: Error sending voice data.");
        }
    }
}


IAXCall*
IAXVoIPLink::getIAXCall(const std::string& id)
{
    return getIaxCall(id);
}

void
IAXVoIPLink::sendRegister(Account *a)
{
    IAXAccount *account = static_cast<IAXAccount*>(a);

    if (account->getHostname().empty())
        throw VoipLinkException("Account hostname is empty");

    if (account->getUsername().empty())
        throw VoipLinkException("Account username is empty");

    sfl::ScopedLock m(mutexIAX_);

    if (regSession_)
        iax_destroy(regSession_);

    regSession_ = iax_session_new();

    if (regSession_) {
        iax_register(regSession_, account->getHostname().data(), account->getUsername().data(), account->getPassword().data(), 120);
        nextRefreshStamp_ = time(NULL) + 10;
        account->setRegistrationState(TRYING);
    }
}

void
IAXVoIPLink::sendUnregister(Account *a)
{
    if (regSession_) {
        sfl::ScopedLock m(mutexIAX_);
        iax_destroy(regSession_);
        regSession_ = NULL;
    }

    nextRefreshStamp_ = 0;

    static_cast<IAXAccount*>(a)->setRegistrationState(UNREGISTERED);
}

Call*
IAXVoIPLink::newOutgoingCall(const std::string& id, const std::string& toUrl, const std::string &account_id)
{
    IAXCall* call = new IAXCall(id, Call::OUTGOING, account_id);

    call->setPeerNumber(toUrl);
    call->initRecFilename(toUrl);

    iaxOutgoingInvite(call);
    call->setConnectionState(Call::PROGRESSING);
    call->setState(Call::ACTIVE);
    addIaxCall(call);

    return call;
}


void
IAXVoIPLink::answer(Call *call)
{
    Manager::instance().addStream(call->getCallId());

    {
        sfl::ScopedLock lock(mutexIAX_);
        call->answer();
    }

    call->setState(Call::ACTIVE);
    call->setConnectionState(Call::CONNECTED);

    Manager::instance().getMainBuffer().flushAllBuffers();
}

void
IAXVoIPLink::hangup(const std::string& id, int reason UNUSED)
{
    IAXCall* call = getIAXCall(id);

    if (call == NULL)
        throw VoipLinkException("Could not find call");

    Manager::instance().getMainBuffer().unBindAll(call->getCallId());

    {
        sfl::ScopedLock lock(mutexIAX_);
        iax_hangup(call->session, (char*) "Dumped Call");
    }

    call->session = NULL;

    removeIaxCall(id);
}


void
IAXVoIPLink::peerHungup(const std::string& id)
{
    IAXCall* call = getIAXCall(id);

    if (call == NULL)
        throw VoipLinkException("Could not find call");

    Manager::instance().getMainBuffer().unBindAll(call->getCallId());

    call->session = NULL;

    removeIaxCall(id);
}



void
IAXVoIPLink::onhold(const std::string& id)
{
    IAXCall* call = getIAXCall(id);

    if (call == NULL)
        throw VoipLinkException("Call does not exist");

    Manager::instance().getMainBuffer().unBindAll(call->getCallId());

    {
        sfl::ScopedLock lock(mutexIAX_);
        iax_quelch_moh(call->session, true);
    }

    call->setState(Call::HOLD);
}

void
IAXVoIPLink::offhold(const std::string& id)
{
    IAXCall* call = getIAXCall(id);

    if (call == NULL)
        throw VoipLinkException("Call does not exist");

    Manager::instance().addStream(call->getCallId());

    {
        sfl::ScopedLock lock(mutexIAX_);
        iax_unquelch(call->session);
    }

    Manager::instance().startAudioDriverStream();
    call->setState(Call::ACTIVE);
}

void
IAXVoIPLink::transfer(const std::string& id, const std::string& to)
{
    IAXCall* call = getIAXCall(id);

    if (!call)
        return;

    char callto[to.length() +1];
    strcpy(callto, to.c_str());

    {
        sfl::ScopedLock lock(mutexIAX_);
        iax_transfer(call->session, callto);
    }
}

bool
IAXVoIPLink::attendedTransfer(const std::string& /*transferID*/, const std::string& /*targetID*/)
{
    return false; // TODO
}

void
IAXVoIPLink::refuse(const std::string& id)
{
    IAXCall* call = getIAXCall(id);

    if (call) {
        {
            sfl::ScopedLock lock(mutexIAX_);
            iax_reject(call->session, (char*) "Call rejected manually.");
        }

        removeIaxCall(id);
    }
}


void
IAXVoIPLink::carryingDTMFdigits(const std::string& id, char code)
{
    IAXCall* call = getIAXCall(id);

    if (call) {
        sfl::ScopedLock lock(mutexIAX_);
        iax_send_dtmf(call->session, code);
    }
}

#if HAVE_INSTANT_MESSAGING
void
IAXVoIPLink::sendTextMessage(const std::string& callID,
                             const std::string& message,
                             const std::string& /*from*/)
{
    IAXCall* call = getIAXCall(callID);

    if (call) {
        sfl::ScopedLock lock(mutexIAX_);
        sfl::InstantMessaging::send_iax_message(call->session, callID, message.c_str());
    }
}
#endif

void
IAXVoIPLink::clearIaxCallMap()
{
    sfl::ScopedLock m(iaxCallMapMutex_);

    for (IAXCallMap::const_iterator iter = iaxCallMap_.begin();
            iter != iaxCallMap_.end(); ++iter)
        delete iter->second;

    iaxCallMap_.clear();

}

void
IAXVoIPLink::addIaxCall(IAXCall* call)
{
    if (call and getIaxCall(call->getCallId()) == NULL) {
        sfl::ScopedLock m(iaxCallMapMutex_);
        iaxCallMap_[call->getCallId()] = call;
    }
}

void
IAXVoIPLink::removeIaxCall(const std::string& id)
{
    sfl::ScopedLock m(iaxCallMapMutex_);

    DEBUG("Removing call %s from list", id.c_str());

    delete iaxCallMap_[id];
    iaxCallMap_.erase(id);
}

IAXCall*
IAXVoIPLink::getIaxCall(const std::string& id)
{
    IAXCallMap::iterator iter = iaxCallMap_.find(id);

    if (iter != iaxCallMap_.end())

        return iter->second;
    else
        return NULL;
}

std::string
IAXVoIPLink::getCurrentVideoCodecName(Call * /*call*/) const
{
    // FIXME: Video not supported for IAX yet
    return "";
}

std::string
IAXVoIPLink::getCurrentAudioCodecNames(Call *c) const
{
    IAXCall *call = static_cast<IAXCall*>(c);
    sfl::AudioCodec *audioCodec = Manager::instance().audioCodecFactory.getCodec(call->getAudioCodec());
    return audioCodec ? audioCodec->getMimeSubtype() : "";
}

void
IAXVoIPLink::iaxOutgoingInvite(IAXCall* call)
{
    sfl::ScopedLock m(mutexIAX_);

    call->session = iax_session_new();

    IAXAccount *account = Manager::instance().getIaxAccount(accountID_);
    std::string username(account->getUsername());
    std::string strNum(username + ":" + account->getPassword() + "@" + account->getHostname() + "/" + call->getPeerNumber());

    /** @todo Make preference dynamic, and configurable */
    int audio_format_preferred = call->getFirstMatchingFormat(call->getSupportedFormat(accountID_), accountID_);
    int audio_format_capability = call->getSupportedFormat(accountID_);

    iax_call(call->session, username.c_str(), username.c_str(), strNum.c_str(),
             NULL, 0, audio_format_preferred, audio_format_capability);
}


IAXCall*
IAXVoIPLink::iaxFindCallBySession(iax_session* session)
{
    sfl::ScopedLock m(iaxCallMapMutex_);

    for (IAXCallMap::const_iterator iter = iaxCallMap_.begin(); iter != iaxCallMap_.end(); ++iter) {
        IAXCall* call = static_cast<IAXCall*>(iter->second);

        if (call and call->session == session)
            return call;
    }

    return NULL;
}

void
IAXVoIPLink::iaxHandleCallEvent(iax_event* event, IAXCall* call)
{
    std::string id = call->getCallId();

    switch (event->etype) {
        case IAX_EVENT_HANGUP:
            Manager::instance().peerHungupCall(id);

            removeIaxCall(id);
            break;

        case IAX_EVENT_REJECT:
            call->setConnectionState(Call::CONNECTED);
            call->setState(Call::ERROR);
            Manager::instance().callFailure(id);
            removeIaxCall(id);
            break;

        case IAX_EVENT_ACCEPT:

            if (event->ies.format)
                call->format = event->ies.format;

            break;

        case IAX_EVENT_ANSWER:
        case IAX_EVENT_TRANSFER:

            if (call->getConnectionState() == Call::CONNECTED)
                break;

            Manager::instance().addStream(call->getCallId());

            call->setConnectionState(Call::CONNECTED);
            call->setState(Call::ACTIVE);

            if (event->ies.format)
                call->format = event->ies.format;

            Manager::instance().peerAnsweredCall(id);

            Manager::instance().startAudioDriverStream();
            Manager::instance().getMainBuffer().flushAllBuffers();

            break;

        case IAX_EVENT_BUSY:
            call->setConnectionState(Call::CONNECTED);
            call->setState(Call::BUSY);
            Manager::instance().callBusy(id);
            removeIaxCall(id);
            break;

        case IAX_EVENT_VOICE:
            iaxHandleVoiceEvent(event, call);
            break;

        case IAX_EVENT_TEXT:
#if HAVE_INSTANT_MESSAGING
            Manager::instance().incomingMessage(call->getCallId(), call->getPeerNumber(), std::string((const char*) event->data));
#endif
            break;

        case IAX_EVENT_RINGA:
            call->setConnectionState(Call::RINGING);
            Manager::instance().peerRingingCall(call->getCallId());
            break;

        case IAX_IE_MSGCOUNT:
        case IAX_EVENT_TIMEOUT:
        case IAX_EVENT_PONG:
        default:
            break;

        case IAX_EVENT_URL:

            if (Manager::instance().getConfigString("Hooks", "Hooks.iax2_enabled") == "1")
                UrlHook::runAction(Manager::instance().getConfigString("Hooks", "Hooks.url_command"), (char*) event->data);

            break;
    }
}


/* Handle audio event, VOICE packet received */
void IAXVoIPLink::iaxHandleVoiceEvent(iax_event* event, IAXCall* call)
{
    // Skip this empty packet.
    if (!event->datalen)
        return;

    sfl::AudioCodec *audioCodec = static_cast<sfl::AudioCodec *>(Manager::instance().audioCodecFactory.getCodec(call->getAudioCodec()));

    if (!audioCodec)
        return;

    Manager::instance().getMainBuffer().setInternalSamplingRate(audioCodec->getClockRate());
    unsigned int mainBufferSampleRate = Manager::instance().getMainBuffer().getInternalSamplingRate();

    if (event->subclass)
        call->format = event->subclass;

    unsigned char *data = (unsigned char*) event->data;
    unsigned int size = event->datalen;

    unsigned int max = audioCodec->getClockRate() * 20 / 1000;

    if (size > max)
        size = max;

    int samples = audioCodec->decode(decData_, data , size);
    int outSize = samples * sizeof(SFLDataFormat);
    SFLDataFormat *out = decData_;
    unsigned int audioRate = audioCodec->getClockRate();

    if (audioRate != mainBufferSampleRate) {
        outSize = (double)outSize * (mainBufferSampleRate / audioRate);
        converter_.resample(decData_, resampledData_, ARRAYSIZE(resampledData_),
                            mainBufferSampleRate, audioRate, samples);
        out = resampledData_;
    }

    Manager::instance().getMainBuffer().putData(out, outSize, call->getCallId());
}

/**
 * Handle the registration process
 */
void IAXVoIPLink::iaxHandleRegReply(iax_event* event)
{
    IAXAccount *account = Manager::instance().getIaxAccount(accountID_);

    if (event->etype != IAX_EVENT_REGREJ && event->etype != IAX_EVENT_REGACK)
        return;

    {
        sfl::ScopedLock m(mutexIAX_);
        iax_destroy(regSession_);
        regSession_ = NULL;
    }

    account->setRegistrationState((event->etype == IAX_EVENT_REGREJ) ? ERROR_AUTH : REGISTERED);

    if (event->etype == IAX_EVENT_REGACK)
        nextRefreshStamp_ = time(NULL) + (event->ies.refresh ? event->ies.refresh : 60);
}

void IAXVoIPLink::iaxHandlePrecallEvent(iax_event* event)
{
    IAXCall *call;
    std::string id;
    int format;

    switch (event->etype) {
        case IAX_EVENT_CONNECT:
            id = Manager::instance().getNewCallID();

            call = new IAXCall(id, Call::INCOMING, accountID_);
            call->session = event->session;
            call->setConnectionState(Call::PROGRESSING);

            if (event->ies.calling_number)
                call->setPeerNumber(event->ies.calling_number);

            if (event->ies.calling_name)
                call->setDisplayName(std::string(event->ies.calling_name));

            // if peerNumber exist append it to the name string
            if (event->ies.calling_number)
                call->initRecFilename(std::string(event->ies.calling_number));
            Manager::instance().incomingCall(*call, accountID_);

            format = call->getFirstMatchingFormat(event->ies.format, accountID_);

            if (!format)
                format = call->getFirstMatchingFormat(event->ies.capability, accountID_);

            iax_accept(event->session, format);
            iax_ring_announce(event->session);
            addIaxCall(call);
            call->format = format;

            break;

        case IAX_EVENT_HANGUP:
            call = iaxFindCallBySession(event->session);

            if (call) {
                id = call->getCallId();
                Manager::instance().peerHungupCall(id);
                removeIaxCall(id);
            }
            break;

        case IAX_EVENT_TIMEOUT: // timeout for an unknown session
        case IAX_IE_MSGCOUNT:
        case IAX_EVENT_REGACK:
        case IAX_EVENT_REGREJ:
        case IAX_EVENT_REGREQ:
            // Received when someone wants to register to us!?!
            // Asterisk receives and answers to that, not us, we're a phone.
        default:
            break;
    }
}

void
IAXVoIPLink::unloadAccountMap()
{
    std::for_each(iaxAccountMap_.begin(), iaxAccountMap_.end(), unloadAccount);
    iaxAccountMap_.clear();
}
