/*
 *  Copyright (C) 2004-2012 Savoir-Faire Linux Inc.
 *  Author: Pierre-Luc Bacon <pierre-luc.bacon@savoirfairelinux.com>
 *  Author: Alexandre Bourget <alexandre.bourget@savoirfairelinux.com>
 *  Author: Laurielle Lea <laurielle.lea@savoirfairelinux.com>
 *  Author: Emmanuel Milou <emmanuel.milou@savoirfairelinux.com>
 *  Author: Yan Morin <yan.morin@savoirfairelinux.com>
 *  Author: Alexandre Savard <alexandre.savard@savoirfairelinux.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA.
 *
 *  Additional permission under GNU GPL version 3 section 7:
 *
 *  If you modify this program, or any covered work, by linking or
 *  combining it with the OpenSSL project's OpenSSL library (or a
 *  modified version of that library), containing parts covered by the
 *  terms of the OpenSSL or SSLeay licenses, Savoir-Faire Linux Inc.
 *  grants you additional permission to convey the resulting work.
 *  Corresponding Source for a non-source form of such a combination
 *  shall include the source code for the parts of OpenSSL used as well
 *  as that of the covered work.
 */
#ifndef AUDIO_RTP_SESSION_H_
#define AUDIO_RTP_SESSION_H_

#include "audio_rtp_record_handler.h"
#include <ccrtp/rtp.h>
#include <ccrtp/formats.h>
#include "noncopyable.h"

class SIPCall;

namespace sfl {

class AudioCodec;

class AudioRtpSession : public AudioRtpRecordHandler {
    public:
        /**
        * Constructor
        * @param sipcall The pointer on the SIP call
        */
        AudioRtpSession(SIPCall &sipcall, ost::RTPDataQueue &queue);
        virtual ~AudioRtpSession();

        void updateSessionMedia(const std::vector<AudioCodec*> &audioCodecs);

        void startRtpThreads(const std::vector<AudioCodec*> &audioCodecs);

        /**
         * Used mostly when receiving a reinvite
         */
        void updateDestinationIpAddress();

        virtual int getIncrementForDTMF() const;

        virtual void setLocalMasterKey(const std::vector<uint8>& key) = 0;

        virtual void setLocalMasterSalt(const std::vector<uint8>& key) = 0;

        virtual std::vector<uint8> getLocalMasterKey() const = 0;

        virtual std::vector<uint8> getLocalMasterSalt() const = 0;

        virtual std::vector<long>
        getSocketDescriptors() const = 0;

    private:
        bool isStarted_;

        void prepareRtpReceiveThread(const std::vector<AudioCodec*> &audioCodecs);
        /**
         * Set the audio codec for this RTP session
         */
        void setSessionMedia(const std::vector<AudioCodec*> &codec);

    protected:
        bool onRTPPacketRecv(ost::IncomingRTPPkt&);

        ost::RTPDataQueue &queue_;
        SIPCall &call_;

        /**
         * Timestamp for this session
         */
        int timestamp_;

        /**
         * Timestamp incrementation value based on codec period length (framesize)
         * except for G722 which require a 8 kHz incrementation.
         */
        int timestampIncrement_;

        /**
         * Rate at which the transport layer handle packets, should be
         * synchronized with codec requirements.
         */
        unsigned int transportRate_;

    private:

        NON_COPYABLE(AudioRtpSession);
        virtual void startReceiveThread() = 0;
        void startSendThread();

        /**
         * Send DTMF over RTP (RFC2833). The timestamp and sequence number must be
         * incremented as if it was microphone audio. This function change the payload type of the rtp session,
         * send the appropriate DTMF digit using this payload, discard coresponding data from mainbuffer and get
         * back the codec payload for further audio processing.
         */
        void sendDtmfEvent();

        /**
         * Send encoded data to peer
         */
        virtual void sendMicData();


        class AudioRtpSendThread {
            public:
                AudioRtpSendThread(AudioRtpSession &session);
                ~AudioRtpSendThread();
                void start();
                bool running_;

            private:
                static void *runCallback(void *data);
                void run();
                NON_COPYABLE(AudioRtpSendThread);
                AudioRtpSession &rtpSession_;
                pthread_t thread_;
                ost::TimerPort timer_;
        };

        /**
         * Set RTP Sockets send/receive timeouts
         */
        void setSessionTimeouts();

        /**
         * Retreive destination address for this session. Stored in CALL
         */
        void setDestinationIpAddress();

        /**
         * Receive data from peer
         */
        void receiveSpeakerData();

        // Main destination address for this rtp session.
        // Stored in case or reINVITE, which may require to forget
        // this destination and update a new one.
        ost::InetHostAddress remote_ip_;

        // Main destination port for this rtp session.
        // Stored in case reINVITE, which may require to forget
        // this destination and update a new one
        unsigned short remote_port_;

        /**
         * Timestamp reset frequency specified in number of packet sent
         */
        short timestampCount_;

        AudioRtpSendThread rtpSendThread_;
};

}
#endif // AUDIO_RTP_SESSION_H__

